function [ PP_VarResults, PV_VarResults, PA_VarResults, InfoVarsComputed ] = optionsVariablesComp( PP, PV, PA, handles )
% =====================
% FUNCTION DESCRIPTION:
% =====================
% The function "optionsVariablesComp.m" contains the definition of all 
% self-defined variables. To implement new variables follow these two steps:
% 
%
% 1. Create function that computes the new variable:
% ==================================================
%       - Create a function at the end of this function that receives ONE
%         time-series COLUMN VECTOR form (Tx1 where T is the overall time)!
%       - The function must then have ONE output called "Result" (The result can
%         be of any type: skalar, vector, matrix, cell, struct, etc.
%
%       (HINT: See function template at end of function)
% 
% 
% 2. Specify variable information:
% ================================
%       Include the 4 mandatory variable specifications in InfoVarsComputed:
%           - InfoVarsComputed(#).VarName = 'NameNewVar';           % Specify variable name
%           - InfoVarsComputed(#).VarUnit = 'NameNewUnits';         % Specify units
%           - InfoVarsComputed(#).VarFunctionHandle = @NewFunction; % handle to function
%           - InfoVarsComputed(#).ComputeVarYesOrNo = 1;            % 1: Will be computed, 0: will not be computed
% 
%        (HINT: Follow template in lines 78-81 or edit the four cells
%         "allVarNames", "allVarUnits", "allVarFunctionHandles", "allComputeVarsYesOrNo")
% 
% 
% REMARK 1: 
% After specifying a new variable it will always be automatically 
% computed if enabled (via fourth variable information) and if the checkbox
% "Compute selfdefined variables" is activated in the interface.
% 
% REMARK 2: 
% The user can disable any of the defined variables changing the
% respective value "1" in the variable cell "allComputeVarsYesOrNo" to "0".
% 
% REMARK 3: 
% All variables are computed on PP, PV, AND PA. The user can
% enable/disable PP, PV or PA variable computations by changing the values
% of the respective compPPVars = 1, compPVVars = 1, compPAVars = 1 to "0".


% To optimize computation by deactivating unwanted variables select 0
% instead of one
compPPVars = 1; % 1: All "enabled" variables will be computed on all PP_k (0: PP variable computation disabled)
compPVVars = 1; % 1: All "enabled" variables will be computed on all PV_k (0: PV variable computation disabled)
compPAVars = 1; % 1: All "enabled" variables will be computed on all PA_k (0: PA variable computation disabled)

% % Add all four details (name, unit, function handle, computation decision) or
% % use template. NOTE: Function must be manually added at the end of this function (see pre-implementations or tamplate)! 
allVarNames = {'Mean', 'std', 'MeanPos', 'MeanNeg', 'stdPos', 'stdNeg', ...
    'NoZC', 'NoP', 'meanTbZC', 'meanTbP', 'stdTbZC', 'stdTbP', 'ratioZCP', ...
    'instantaneousPower', 'averagePower', 'RMS', ...
    'instantaneousActivity','averageActivity'};
allVarUnits = {'[AU]', '[AU]', '[AU]', '[AU]', '[AU]', '[AU]', ...
    '[#]', '[#]', '[s]', '[s]','[s]', '[s]', '[-]', ...
    '[AU]', '[AU]', '[AU]', ...
    '[AU]','[AU]'};
allVarFunctionHandles = {@meanNew, @stdNew, @MeanPos, @MeanNeg, @stdPos, @stdNeg, ...
    @NoZC, @NoP, @meanTbZC, @meanTbP,  @stdTbZC, @stdTbP, @ratioZCP, ...
    @instantaneousPower, @averagePower, @rmsNew, ...
    @instantaneousActivity, @averageActivity};
allComputeVarsYesOrNo = {0,1,0,0,1,1, ...
    1,1,1,0,1,0,0, ...
    0,1,1, ...
    0,1};

if sum(length(allVarNames) ~= [length(allVarFunctionHandles), length(allComputeVarsYesOrNo), length(allVarUnits)]) > 0
    msgbox('Error in function "optionsVariablesComp".  No. "variable names", "function handles" or "computation (yes/no) decisions" do not coincide', 'Error', 'error')
    error('Error in function "optionsVariablesComp".  No. "variable names", "function handles" or "computation (yes/no) decisions" do not coincide')
end



InfoVarsComputed = struct('VarName', allVarNames,'VarUnit', allVarUnits, 'VarFunctionHandle', allVarFunctionHandles, 'ComputeVarYesOrNo', allComputeVarsYesOrNo);

% % Variable template: Define function at the end of the function usign template (or: add all four details of the new variable above following the predefined examples)
% InfoVarsComputed(17).VarName = 'NameNewVar';
% InfoVarsComputed(17).VarUnit = 'NameNewUnits';
% InfoVarsComputed(17).VarFunctionHandle = @NewFunction;
% InfoVarsComputed(17).ComputeVarYesOrNo = 1; % 1: Will be computed, 0: will not be computed



PP_VarResults = cell(1,length(allVarNames));
PV_VarResults = cell(1,length(allVarNames));
PA_VarResults = cell(1,length(allVarNames));


for VarNr = 1:length(InfoVarsComputed)
    if InfoVarsComputed(VarNr).ComputeVarYesOrNo == 1
        if compPPVars == 1
            PP_VarResults{VarNr} = InfoVarsComputed(VarNr).VarFunctionHandle(PP, handles);
        end
        if compPVVars == 1
            PV_VarResults{VarNr} = InfoVarsComputed(VarNr).VarFunctionHandle(PV, handles);
        end
        if compPAVars == 1
            PA_VarResults{VarNr} = InfoVarsComputed(VarNr).VarFunctionHandle(PA, handles);
        end
    else
        PV_VarResults{VarNr} = 0;
    end
end


% Handles to functions that are not available later cannot be saved! If
% functions were available outside (saved a separate .m) the following line 
% would be obsolete.
InfoVarsComputed = rmfield(InfoVarsComputed, 'VarFunctionHandle');



%% Functions
    function [Result] = meanNew(timeSeries, ~)   
        Result = mean(timeSeries);
    end
    function [Result] = stdNew(timeSeries, ~)   
        Result = std(timeSeries);
    end
    function [Result] = MeanPos(timeSeries, ~)   
        Result = mean(timeSeries(timeSeries >= 0));
    end
    function [Result] = MeanNeg(timeSeries, ~)
        Result = mean(timeSeries(timeSeries <= 0));
    end
    function [Result] = stdPos(timeSeries, ~)
        Result = std(timeSeries(timeSeries >= 0));
    end
    function [Result] = stdNeg(timeSeries, ~)
        Result = std(timeSeries(timeSeries <= 0));
    end
    function [Result] = NoZC(timeSeries, ~)        
        y_sign          = sign(timeSeries);
        locations_ZC    = find( y_sign(1: end-1) ~= y_sign(2:end) ); % Zero values could theoretically lead to a minor alteration of the NoZC
        Result          = length(locations_ZC);
    end
    function [Result] = NoP(timeSeries, ~)
        [~, peak_pos]   = findpeaks(timeSeries);
        [~, peak_neg]   = findpeaks(-timeSeries);
        locations_extrp = sort([peak_pos; peak_neg]);
        Result          = length(locations_extrp);
    end
    function [Result] = meanTbZC(timeSeries, handles)
        y_sign              = sign(timeSeries);
        locations_ZC        = find( y_sign(1: end-1) ~= y_sign(2:end) );
        frames_between_ZC   = locations_ZC(2:end)-locations_ZC(1:end-1);
        Result              = mean(frames_between_ZC/handles.sample_frequency);
    end
    function [Result] = meanTbP(timeSeries, handles)
        [~, peak_pos]           = findpeaks(timeSeries);
        [~, peak_neg]           = findpeaks(-timeSeries);
        locations_extrp         = sort([peak_pos; peak_neg]);
        frames_between_extrema  = locations_extrp(2:end)-locations_extrp(1:end-1);
        Result                  = mean(frames_between_extrema/handles.sample_frequency);
    end
    function [Result] = stdTbZC(timeSeries, handles)
        y_sign              = sign(timeSeries);
        locations_ZC        = find( y_sign(1: end-1) ~= y_sign(2:end) );
        frames_between_ZC   = locations_ZC(2:end)-locations_ZC(1:end-1);
        Result              = std(frames_between_ZC/handles.sample_frequency);
    end
    function [Result] = stdTbP(timeSeries, handles)
        [~, peak_pos]           = findpeaks(timeSeries);
        [~, peak_neg]           = findpeaks(-timeSeries);
        locations_extrp         = sort([peak_pos; peak_neg]);
        frames_between_extrema  = locations_extrp(2:end)-locations_extrp(1:end-1);
        Result                  = std(frames_between_extrema/handles.sample_frequency);
    end
    function [Result] = ratioZCP(timeSeries, ~)
        y_sign          = sign(timeSeries);
        [~, peak_pos]   = findpeaks(timeSeries);
        [~, peak_neg]   = findpeaks(-timeSeries);

        locations_ZC    = find( y_sign(1: end-1) ~= y_sign(2:end) ); % Zero values could theoretically lead to a minor alteration of the NoZC
        locations_extrp = sort([peak_pos; peak_neg]);
        Result          = length(locations_ZC)/length(locations_extrp);
    end
    function [Result] = instantaneousPower(timeSeries, ~) % "Power" due to voltage computations P proportional to V^2
        Result = timeSeries.^2;
    end
    function [Result] = averagePower(timeSeries, ~) % "Power" due to voltage computations P proportional to V^2
        Result = mean(timeSeries.^2);
    end
    function [Result] = rmsNew(timeSeries, ~) % "Power" due to voltage computations P proportional to V^2
        Result = sqrt(mean(timeSeries.^2));
    end
    function [Result] = instantaneousActivity(timeSeries, ~) % "Activity" due to absolute values (no distortion through squares! Squares weigh heigh values more than low ones!)
        Result = abs(timeSeries);
    end
    function [Result] = averageActivity(timeSeries, ~) % "Activity" due to absolute values (no distortion through squares! Squares weigh heigh values more than low ones!)
        Result = mean(abs(timeSeries));
    end

% % % Function template: Replace handles by "~" if not needed. User can editthis example.
% %     function [Result] = NewFunction(timeSeries, handles)
% %         Result = mean(abs(timeSeries));   % "timeSeries" will be a Tx1 column vector (T = Number of time frames)!
% %     end

end

