//rlp_math.cpp, Copyright (c) 2004-2024 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#define SWAP(a,b) {double temp=(a);(a)=(b);(b)=temp;}
#define _PREC 1.0e-12

extern def_vars defs;

static char *MRQ_error = 0L;
static double sqrt2pi = sqrt(_PI*2.0);
extern void *prog_bar_ptr;
extern long prog_bar_max, *prog_bar_current;
extern DWORD prog_bar_color;
extern int prog_bar_mode;

//fitting curve to data with the simplex algorithm
//Ref: Caceci M.S., Cacheris W.P. (1984) Fitting curves to data: The simplex algorithm is the answer.
//     Byte January 1984, 340-362
class simplex {
public:
	double mean[20];	//the final result

	simplex(DataObj *d, double **s_data, int nd, double *s_par, unsigned char **s_pnam, int np,
		char *formula, double vg, int maxfit, double *chi_2, bool bPgr);
	~simplex();
	double f(double *x, double *d);
	void sum_of_residuals(double *x);
	void new_vertex();
	void order();
	int DoFit();

private:
	int m;			//numbers of parameters to fit
	int nvpp;		//total number of vars per data point
	int n;
	double alfa;		//reflection coefficient, >0
	double beta;		//contaction coefficient, 0 to 1
	double gamma;		//expansion coefficient, > 1
	int lw;			//width of line in data fields + 1
	bool bProgress;		//show progress bar
	int index;		//0..255
	unsigned char *formula, tmp_formula[800], **p_nam;
	DataObj *dobj;
	bool done;			//convergence
	int *h, *l;			//[1..n], number high/low paramts
	int np;				//number of data points
	long maxiter;		//max number iterations
	long niter;			//number of iterations
	double *next;		//new vertex to be tested
	double *center;		//center of hyperplane described by all
	//   vertexes of the simplex excluding the worst
	double *error, *maxerr;		//maximum error accepted
	double *p, *q;		//to compute first simplex
	double *step;		//input starting steps
	double **simp;		//the simplex
	double **data;
	double *chi2;
};

simplex::simplex(DataObj *d, double **s_data, int nd, double *s_par, unsigned char **s_pnam, int npar,
	char *expr, double vg, int maxfit, double *chi_2, bool bPgr)
{
	int i;

	maxiter = maxfit;		niter = 0;		 bProgress = bPgr;
	formula = rlp_strdup((unsigned char*)expr);
	maxerr = (double*)malloc(sizeof(double)*(npar + 2));
	step = (double*)malloc(sizeof(double)*(npar + 2));
	l = (int*)malloc(sizeof(int)*(npar + 2));
	h = (int*)malloc(sizeof(int)*(npar + 2));
	p = (double*)malloc(sizeof(double)*(npar + 2));
	q = (double*)malloc(sizeof(double)*(npar + 2));
	error = (double*)malloc(sizeof(double)*(npar + 2));
	m = npar;	n = m + 1;	 np = nd;
	center = (double*)malloc(sizeof(double)*(npar + 2));
	next = (double*)malloc(sizeof(double)*(npar + 2));
	simp = (double**)malloc((m + 2) * sizeof(double*));
	for (i = 1; i < m + 2; i++){
		simp[i] = (double*)malloc(sizeof(double)*(npar + 2));
		}
	p_nam = s_pnam;		dobj = d;
	for (i = npar; i > 0; i--) {
		simp[1][npar-i+1] = s_par[i];
		step[npar - i + 1] = s_par[i] / 5.0;
		maxerr[npar - i + 1] = vg; 
		}
	if (bProgress) {
		prog_bar_mode = 0;
		prog_bar_max = maxfit;
		prog_bar_current = &niter;
		prog_bar_color = 0x00ff0000;
		}
	maxerr[npar + 1] = vg;
	data = s_data;		nvpp = 2;
	alfa = 1.0;			beta = 0.5;
	gamma = 2.0;		lw = 5;
	chi2 = chi_2;
}

simplex::~simplex()
{
	int i;

	if (formula) free(formula);
	formula = 0L;				if (maxerr) free(maxerr);
	maxerr = 0L;				if (step) free(step);
	step = 0L;					if (center) free(center);
	center = 0L;				if (next) free(next);
	next = 0L;					if (l) free(l);	
	l = 0L;						if (h) free(h);	
	h = 0L;						if (p) free(p);	
	p = 0L;						if (q) free(q);
	q = 0L;						if (error) free(error);
	error = 0L;
	for (i = 1; i < m + 2; i++){
		if (simp[i]) free(simp[i]);
		simp[i] = 0L;
		}
	free(simp);
}

//the function to fit
double
simplex::f(double *x, double *d)	// x(1..m) the parameters, d has the data
{
	double ret;
	int i, spos;
	anyResult *res;

	for (spos = 0, i = m; i; i--) {
#ifdef USE_WIN_SECURE
		spos += sprintf_s((char*)(tmp_formula + spos), 800 - spos - 20, "%s=%g;", p_nam[i], x[m-i+1]);
#else
		spos += sprintf((char*)(tmp_formula + spos), "%s=%g;", p_nam[i], x[m - i + 1]);
#endif
		}
#ifdef USE_WIN_SECURE
	sprintf_s((char*)(tmp_formula + spos), 800 - spos - rlp_strlen(formula) - 10, "x=%g;z=%g;\n%s", d[1], d[3], formula);
#else
	sprintf((char*)(tmp_formula + spos), "x=%g;z=%g;\n%s", d[1], d[3], formula);
#endif
	res = do_formula(dobj, (char*)tmp_formula);
	if (res->type == ET_VALUE) 	ret = res->value;
	else ret = 0.0;
	return ret;
}

//computes the sum of the squares of the residuals
//   x(1..m) passes the parameters. Result returned in x(n)
void
simplex::sum_of_residuals(double *x)
{
	int i;
	double tmp;

	x[n] = 0.0;
	for (i = 1; i <= np; i++) {
		tmp = f(x, data[i]) - data[i][2];
		x[n] += (tmp * tmp);
		}
	*chi2 = x[n];
}

//next in place of the worst vertex
void
simplex::new_vertex()
{
	int i;

	for (i = 1; i <= n; i++) {
		simp[h[n]][i] = next[i];
		}
}

//gives high/low in each parameter in simp. caution: not initialized
void
simplex::order()
{
	int i, j;

	for (j = 1; j <= n; j++) {
		for (i = 1; i <= n; i++) {
			if (simp[i][j] < simp[l[j]][j]) l[j] = i;
			if (simp[i][j] > simp[h[j]][j]) h[j] = i;
			}
		}
}

//start fitting the data to the function
int
simplex::DoFit()
{
	int i, j;

	for (i = 1; i <= n; i++) {	//preset, again preset at lower code
		l[i] = 1;	h[i] = 1;
		}
	sum_of_residuals(simp[1]);	//first vertex;
	for (i = 1; i <= m; i++) {	//compute offset of the vertexes
		p[i] = step[i] * (sqrt((double)n) + m - 1) / (m * _SQRT2);
		q[i] = step[i] * (sqrt((double)n) - 1) / (m * _SQRT2);
		}
	for (i = 2; i <= n; i++) {	//all vertexes of the starting simplex
		for (j = 1; j <= m; j++) {
			simp[i][j] = simp[1][j] + q[j];
			}
		simp[i][i - 1] = simp[1][i - 1] + p[i - 1];
		sum_of_residuals(simp[i]);
		}
	for (i = 1; i <= n; i++) {	//preset
		l[i] = 1;	h[i] = 1;
		}
	order();
	niter = 0;			//no iterations yet
	if (bProgress) {
		prog_bar_max = maxiter;
		prog_bar_current = &niter;
		}
	do {
		done = true;		//wish it where ...
		niter++;
		for (i = 1; i <= n; i++) center[i] = 0.0;
		for (i = 1; i <= n; i++) {	//compute centroid
			if (i != h[n]) {	//excluding the worst
				for (j = 1; j <= m; j++) {
					center[j] = center[j] + simp[i][j];
					}
				}
			}
		for (i = 1; i <= n; i++) {	//first attempt to reflect
			center[i] = center[i] / m;
			next[i] = (1.0 + alfa) * center[i] - alfa * simp[h[n]][i];
			//next vertext is specular reflection of the worst
			}
		sum_of_residuals(next);
		if (next[n] <= simp[l[n]][n]) {	//better than the best?
			new_vertex();		//accepted !
			for (i = 1; i <= m; i++){ 	//and expanded
				next[i] = gamma * simp[h[n]][i] + (1.0 - gamma) * center[i];
				}
			sum_of_residuals(next);
			if (next[n] <= simp[l[n]][n]) new_vertex(); // expansion accepted
			}
		else {
			if (next[n] <= simp[h[n]][n]) new_vertex(); //better than worst
			else {			//worse than worst
				for (i = 1; i <= m; i++)	//then: contract
					next[i] = beta * simp[h[n]][i] + (1.0 - beta) * center[i];
				sum_of_residuals(next);
				if (next[n] <= simp[h[n]][n]) new_vertex();//contraction accepted
				else { 		//if still bad
					for (i = 1; i <= n; i++) {	//shrink all bad vertexes
						for (j = 1; j <= m; j++) simp[i][j] =
							(simp[i][j] + simp[l[n]][j]) * beta;
						sum_of_residuals(simp[i]);
						}
					}
				}
			}
		order();
		for (j = 1; j <= n; j++) { //check for convergence
			error[j] = (simp[h[j]][j] - simp[l[j]][j]) / simp[h[j]][j];
			if (error[j] > maxerr[j]) {
				done = false;
				}
			}
		if (NoWaitDlgLoop()) NoWaitDlgLoop();
	} while (!done && niter <= maxiter);
	for (i = 1; i <= n && i < 20; i++) {	//average each parameter
		mean[i] = 0.0;
		for (j = 1; j <= n; j++) mean[i] += simp[j][i];
		mean[i] /= (double) n;
		}
	if (bProgress) {
		prog_bar_max = 0;
		}
	return niter-1;
}

long ExecSimplex(DataObj *d, double **s_data, int nd, double *s_par, unsigned char **s_pnam, int np,
	char *formula, double vg, int maxfit, double *chi_2, bool bProgress)
{
	static long niter;
	int i;
	simplex *spx;

	spx = new simplex(d, s_data, nd, s_par, s_pnam, np, formula, vg, maxfit, chi_2, bProgress);
	niter = spx->DoFit();
	if (bProgress) {
		prog_bar_current = &niter;			//point 'prog_bar_current' to permanent location
		}
	for (i = 1; i <= np && i < 20; i++) s_par[i] = spx->mean[np-i+1];
	delete spx;
	return niter;
}

//---------------------------------------------------------------------------
//The routine gaussj solves linear equations by Gauss-Jordan elimination
bool gaussj_1(double **a, int n, double **b, int m)
{
	int *indxc, *indxr, *ipiv;
	int i, icol = 0, irow = 0, j, k, l, ll;
	double big, dum, pivinv;

	indxc = (int*)calloc((n + 1), sizeof(int));
	indxr = (int*)calloc((n + 1), sizeof(int));
	ipiv = (int*)calloc((n + 1), sizeof(int));

	for (i = 1; i <= n; i++) {				//This is the main loop over the
		big = 0.0;							//    columns to be reduced
		for (j = 1; j <= n; j++)			//This is the outer loop of the search
			if (ipiv[j] != 1)				//    for a pivot element
				for (k = 1; k <= n; k++) {
					if (ipiv[k] == 0) {
						if (fabs(a[j][k]) >= big) {
							big = fabs(a[j][k]);
							irow = j;
							icol = k;
							}
						}
					else if (ipiv[k] > 1) {
						MRQ_error = (char*)"Singular Matrix (1)";
						free(ipiv);		free(indxr);	free(indxc);
						return false;
						}
					}
		++(ipiv[icol]);
		//We now have the pivot element, so we interchange rows, if needed,
		// to put the pivot element on the diagonal.
		if (irow != icol) {
			for (l = 1; l <= n; l++) SWAP(a[irow][l], a[icol][l]);
			for (l = 1; l <= m; l++) SWAP(b[irow][l], b[icol][l]);
			}
		indxr[i] = irow;
		indxc[i] = icol;
		if (a[icol][icol] < 1.0e-30) {
			MRQ_error = (char*)"Singular Matrix (2)";
			free(ipiv);		free(indxr);	free(indxc);
			return false;
			}
		pivinv = 1.0 / a[icol][icol];
		a[icol][icol] = 1.0;
		for (l = 1; l <= n; l++) a[icol][l] *= pivinv;
		for (l = 1; l <= m; l++) b[icol][l] *= pivinv;
		for (ll = 1; ll <= n; ll++)
			if (ll != icol) { 							//Next, we reduce the rows
				dum = a[ll][icol];
				a[ll][icol] = 0.0;
				for (l = 1; l <= n; l++) a[ll][l] -= a[icol][l] * dum;
				for (l = 1; l <= m; l++) b[ll][l] -= b[icol][l] * dum;
				}
		}											// This is the end of the main loop
	for (l = n; l >= 1; l--) {						//   over columns of the reduction.
		if (indxr[l] != indxc[l]) 					//   Unscramble the solution
			for (k = 1; k <= n; k++) SWAP(a[k][indxr[l]], a[k][indxc[l]]);
		}											//And we are done.
	free(ipiv);		free(indxr);	free(indxc);
	return true;
}

/*
//DEBUG version
bool test_gj(double **a, int n, double **b, int m)
{
	int i;
	bool bRet;

	for (i = 0; i < n; i++) a[i] -= 1;
	a -= 1;
	for (i = 0; i < m; i++) b[i] -= 1;
	b -= 1;
	bRet = gaussj_1(a, n, b, m);
	b += 1;
	for (i = 0; i < m; i++) b[i] += 1;
	a += 1;
	for (i = 0; i < n; i++) a[i] += 1;
	return bRet;
}
*/

bool gaussj(double **a, int n, double **b, int m)
{
	int *indxc, *indxr, *ipiv;
	int i, icol = 0, irow = 0, j, k, l, ll;
	double big, dum, pivinv;

	indxc = (int*)calloc((n+1), sizeof(int));
	indxr = (int*)calloc((n+1), sizeof(int));
	ipiv = (int*)calloc((n+1), sizeof(int));
	for (i = 0; i < n; i++) {				//This is the main loop over the
		big = 0.0;							//    columns to be reduced
		for(j = 0; j < n; j ++)				//This is the outer loop of the search
			if(ipiv[j] != 1)				//    for a pivot element
				for(k = 0; k < n; k ++) {
					if (ipiv[k] == 0) {
						if(fabs(a[j][k]) >= big) {
							big = fabs(a[j][k]);
							irow = j;				icol = k;
							}
						}
					else if(ipiv[k] > 1) {
						MRQ_error = (char*)"Singular Matrix (1)";
						free(ipiv);		free(indxr);	free(indxc);
						return false;
						}
				}
		++(ipiv[icol]);
		//We now have the pivot element, so we interchange rows, if needed,
		// to put the pivot element on the diagonal.
		if(irow != icol) {
			for(l = 0; l < n; l++) SWAP(a[irow][l], a[icol][l]);
			for(l = 0; l < m; l++) SWAP(b[irow][l], b[icol][l]);
			}
		indxr[i] = irow;		indxc[i] = icol;
		if(a[icol][icol] < 1.0e-30) {
			MRQ_error = (char*)"Singular Matrix (2)";
			free(ipiv);		free(indxr);	free(indxc);
			return false;
			}
		pivinv = 1.0/a[icol][icol];
		a[icol][icol] = 1.0;
		for(l = 0; l < n; l++) a[icol][l] *= pivinv;
		for(l = 0; l < m; l++) b[icol][l] *= pivinv;
		for(ll = 0; ll <  n; ll++)
			if(ll != icol) { 							//Next, we reduce the rows
				dum = a[ll][icol];
				a[ll][icol] = 0.0;
				for(l = 0; l < n; l++) a[ll][l] -= a[icol][l]*dum;
				for(l = 0; l < m; l++) b[ll][l] -= b[icol][l]*dum;
				}
		}											// This is the end of the main loop
	for (l = n-1; l >= 0; l--) {						//   over columns of the reduction.
		if(indxr[l] != indxc[l]) 					//   Unscramble the solution
			for(k = 0; k < n; k++) SWAP (a[k][indxr[l]], a[k][indxc[l]]);
		}											//And we are done.
	free(ipiv);		free(indxr);	free(indxc);
	return true;
}

//---------------------------------------------------------------------------
//The routine mrqcof is called by mrqmin to evaluate the linearized fitting
// matrix alpha and vector beta
void mrqcof(double x[], double y[], double z[], int ndata, double **a, int ma,
	int lista[], int mfit, double **alpha, double beta[], double *chisq,
	void (*funcs)(double, double, double **, double *, double *, int))
{
	int k, j, i;
	double ymod, wt, dy;
	double *dyda, sig2i;

	dyda = (double*)malloc(ma*sizeof(double));
	for(j = 0; j < mfit; j++) {					//Initialize (symmetric) alpha, beta
		for(k = 0; k <= j; k++) alpha[j][k] = 0.0;
		beta[j] = 0.0;
		}
	*chisq = 0.0;
	for (i = 0; i < ndata; i++) {		 		//Summation loop over all data
		(*funcs)(x[i], z ? z[i] : 0.0, a, &ymod, dyda, ma);
		if(ymod != 0.0) dy = y[i]-ymod;			//functions = 0.0 if out of range
		else dy = 0.0;			
		if(z) sig2i = 1.0 / (z[i] * z[i]);
		else sig2i = 1.0;
		for(j = 0; j < mfit; j++) {
			wt = dyda[lista[j]] * sig2i;
			for (k = 0; k <= j; k++){
				alpha[j][k] += wt*dyda[lista[k]];
				}
			beta[j] += dy*wt;
			}
		(*chisq) += dy*dy; 							//And find X^2 if function o.k.
		}
	for (j = 1; j < mfit; j++) {					//Fill the symmetric side
		for (k = 0; k < j; k++) {
			alpha[k][j] = alpha[j][k];
			}
		}
	free(dyda);
}

//---------------------------------------------------------------------------
//The routine mrqmin performs one iteration of Marquart's method for nonlinear
// parameter estimation
bool mrqmin(double *x, double *y, double *z, int ndata, double **a, int ma,
	int *lista, int mfit, double **covar, double **alpha, double *chisq,
	void (*funcs)(double, double, double **, double *, double *, int), double *alamda)
{
	int k, kk, j, ihit;
	static double *da, *atry, *beta, ochisq;
	static double **oneda, **atryref;

	if (*alamda < 0.0) {								//Initialization
		MRQ_error = 0L;
		oneda = (double**)calloc(mfit + 1, sizeof(double*));
		for (k = 0; k < mfit; k++) oneda[k] = (double*)calloc(mfit + 1, sizeof(double*));
		atry = (double *)calloc(ma + 1, sizeof(double));
		atryref = (double**)calloc(ma + 1, sizeof(double*));
		for (j = 0; j < ma; j++) {
			atryref[j] = &atry[j];
			}
		da = (double*)calloc(ma+1, sizeof(double));
		beta = (double*)calloc(ma+1, sizeof(double));
		kk = mfit;
		for(j = 0; j < ma; j++) { 						//Does lista contain a proper
			ihit = 0;									//   permutation of the
			for(k = 0; k < mfit; k++)					//   coefficients ?
				if(lista[k] == j) ihit++;
			if(ihit == 0)
				lista[kk++] = j;
			else if (ihit >1) ErrorBox((char*)"Bad LISTA permutations in MRQMIN-1");
			}
		if(kk != ma) ErrorBox((char*)"Bad LISTA permutations in MRQMIN-2");
		*alamda = 0.001;
		mrqcof(x, y, z, ndata, a, ma, lista, mfit, alpha, beta, chisq, funcs);
		ochisq=(*chisq);
		}
	for (j = 0; j < mfit; j++) {						//Alter linearized fitting matrix
		for(k = 0; k < mfit; k++) covar[j][k] = alpha[j][k];	// by augmenting
		covar[j][j] = alpha[j][j]*(1.0+(*alamda));		// diagaonal elements
		oneda[j][0] = beta[j];
		}
	if (!gaussj(covar, mfit, oneda, 1)) return false;	//Matrix solution ?
	for(j = 0; j < mfit; j++) da[j] = oneda[j][0];
	if(*alamda == 0.0) {								//Once converged evaluate
														//  covariance matrix with
		free(beta);										//  alamda = 0.
		free(da);
		free(atry);
		free(atryref);
		for (j = 0; j <= mfit; j++) {
			if (oneda[j]) free(oneda[j]);
			}
		free(oneda);
		return true;
		}
	for(j = 0; j < ma; j++) atry[j] = *a[j];
	for(j = 0; j < mfit; j++)							//Did the trial succeed ?
		atry[lista[j]] = *a[lista[j]] + da[j];
	mrqcof(x, y, z, ndata, atryref, ma, lista, mfit, covar, da, chisq, funcs);
	if(*chisq < ochisq) {								//Success, accept the new solution
		*alamda *= 0.1;
		ochisq=(*chisq);
		for(j = 0; j < mfit; j++) {
			for(k = 0; k < mfit; k++) alpha[j][k] = covar[j][k];
			beta[j] = da[j];
			*a[lista[j]] = atry[lista[j]];
			}
		}
	else {												//Failure, increase almda and
		*alamda *= 10.0;								//    return.
		*chisq = ochisq;
		}
	return true;
}

bool Check_MRQerror()
{
	bool bRet = MRQ_error != 0L;
	if(bRet) ErrorBox(MRQ_error);
	MRQ_error = 0L;
	return bRet;
}

//---------------------------------------------------------------------------
//matinv(a, n, d) inverts real square matix of size n x n and returns 
//     determinant in d.
//Ref.: S.B. Levitt and K.J. Johnson (1980) in: "Numerical Methods in Chemistry".
//     K.J. Johnson ed.; Marcel Dekker, Inc. New York and Basel; p216ff
//     ISBN 0-8247-6818-3
bool matinv(double **a, int n, double *d)
{
	int i, j, k, l, l1, irow = 0, icol = 0, jrow = 0, jcol = 0, **ipv, nswap;
	double max, tmp, pivot;

	if(!(ipv = (int**) malloc(n * sizeof(int*)))) return false;
	for (j = 0; j < n; j++) ipv[j] = (int*)calloc(4, sizeof(int));
	*d = 1.0;
	// main loop, eliminate one row at a time
	for(i = 0; i < n; i++) {
		// search remaining matrix for maximum element (pivot);
		// test for a singular coefficient matrix
		for(j = 0, max = 0.0; j < n; j++) if(ipv[j][2] != 1) {
			for(k = 0; k < n; k++) if (max < (tmp = fabs(a[j][k]))){
				irow = j;	icol = k;	max = tmp;
				}
			}
		if(max <= 1.0e-35) {
			*d = 0.0;	return false;
			}
		ipv[icol][2] = 1;	ipv[i][0] = irow;	ipv[i][1] = icol;
		// interchange rows (if necessary) to put pivot element on diagonal
		if(irow != icol) for(l = 0; l < n; l++) {
			tmp = a[irow][l];	a[irow][l] = a[icol][l];	a[icol][l] = tmp;
			}
		// update determinant and normalize pivot row
		pivot = a[icol][icol];		*d *= pivot;	a[icol][icol] = 1.0;
		for(l = 0; l < n; l++) a[icol][l] /= pivot;
		// eliminate icolumn retaining inverse elements
		for(l1 = 0; l1 < n; l1++) if(l1 != icol) {
			tmp = a[l1][icol];	a[l1][icol] = 0.0;
			for(l = 0; l < n; l++) a[l1][l] -= (a[icol][l] * tmp);
			}
		}
	for(i = nswap = 0, l = n-1; i < n; i++, l--) if(ipv[l][0] != ipv[l][1]){
		// interchange columns and modify determinant
		jrow = ipv[l][0];	jcol = ipv[l][1];	nswap++;
		for(k = 0; k < n; k++) {
			tmp = a[k][jrow];	a[k][jrow] =a[k][jcol];		a[k][jcol] = tmp;
			}
		*d *= ((nswap & 0x01) ? -1.0 : 1.0);
		}
	for (j = 0; j < n; j++) if(ipv[j]) free(ipv[j]);
	free(ipv);
	return true;
}

//---------------------------------------------------------------------------
bool hdiag(double **h, int n, double **u)
{
	int i, j, k, l, nr, nmi1, ipl1, ipiv, jpiv, *iq;
	double *x, xmax, tmp, tmp1, hdimin, tang, cosine, sine, hii;
	double rap = 7.45058060e-9;
	double hdtest = 1.7e38;
	bool bvalid;

	if(n < 2) return false;
	if(!(x = (double*)malloc(n * sizeof(double)))) return false;
	if(!(iq = (int*)malloc(n * sizeof(int)))) return false;
	//initialize u
	if(u) for(i = 0; i < n; i++) for(j = 0; j < n; j++) {
		u[i][j] = i != j ? 0.0 : 1.0;
		}
	//scan for largest off-diagonal element in each row
	//   x[i] contains largest element in ith row
	//   iq[i] holds second subscript defining position of element
	for(i = nr = 0, nmi1 = n-1; i < n; i++) {
		for (j = ipl1 = i+1, x[i] = 0.0; j < n; j++) {
			if((tmp = fabs(h[i][j])) >= x[i]){
				x[i] = tmp;	iq[i] = j;
				}
			}
		}
	//find the maximum of x[i]'s for pivot element
	//test for end of problem
	for(l = 0 ; l < 100; l++) {
		for(i = 1, ipiv = 0, jpiv = iq[0], xmax = x[0]; i < n; i++) {
			if(xmax < x[i]) {
				xmax = x[ipiv = i];	jpiv = iq[i];
				}
			}
		//is max. x[i] equal to zero? if less than hdtest, revise hdtest
		if(xmax <= 0.0) {
			free(iq);	free(x);	return false;
			}
		if(hdtest < 0.0) hdimin = fabs(h[0][0]);
		if(xmax <= hdtest || hdtest < 0.0) {
			for(i = 1, hdimin = fabs(h[0][0]); i < n; i++) {
				if((tmp = fabs(h[i][i])) < hdimin) hdimin = tmp;
				}
			hdtest = hdimin * rap;
			if(hdtest >= xmax) {
				free(iq);	free(x);	return true;
				}
			}
		nr ++;		
		//compute tangent, sin and cosine, h[i][i], h[j][j]
		tang = (tmp = h[ipiv][ipiv] - h[jpiv][jpiv]) >= 0.0 ? 2.0 : -2.0;
		tang *= h[ipiv][jpiv];
		tmp1 = sqrt(tmp * tmp + 4.0 * h[ipiv][jpiv] * h[ipiv][jpiv]);
		tang /= (fabs(tmp) +tmp1);
//		tang /= (fabs(tmp) + sqrt(tmp) * sqrt(tmp) + 4.0 * h[ipiv][jpiv] * h[ipiv][jpiv]);


		cosine = 1.0 / sqrt(1.0+tang * tang);		sine = tang * cosine;
		hii = h[ipiv][ipiv];
		h[ipiv][ipiv] = cosine * cosine * (hii + tang * (2.0 * h[ipiv][jpiv] + tang * h[jpiv][jpiv]));
		h[jpiv][jpiv] = cosine * cosine * (h[jpiv][jpiv] - tang * (2.0 * h[ipiv][jpiv] - tang * hii));
		h[ipiv][jpiv] = 0.0;
		//pseudo rank the eigenvalues
		//adjust sine and cos for computation of h[i][k] and u[i][j]
		if(h[ipiv][ipiv] < h[jpiv][jpiv]) {
			tmp  = h[ipiv][ipiv];	h[ipiv][ipiv] = h[jpiv][jpiv];	h[jpiv][jpiv] = tmp;
			//recompute sine and cosine
			tmp = sine < 0.0 ? cosine : -cosine;	cosine = fabs(sine);	sine = tmp;
			}
		//inspect the iq's between i+1 and n-1 to determine
		//   whether a new maximum value should be computed since
		//   the present maximum is in the i or j row
		for(i = 0; i < nmi1; i++) {
			bvalid = (i != ipiv);
			if(i > ipiv && i == jpiv) bvalid = false;
			if((i < ipiv || bvalid) && iq[i] == ipiv) bvalid = true;
			else if(i < ipiv && iq[i] != jpiv) bvalid = false;
			if(bvalid) {
				k = iq[i];		tmp = h[i][k];		h[i][k] = 0.0;
				h[i][k] = 0.0;		ipl1 = i+1;		x[i] = 0.0;
				//search in depleted row for new maximum
				for(j = ipl1; j < n; j++) {
					if(x[i] <= fabs(h[i][j])) {
 						x[i] = fabs(h[i][j]);		iq[i] = j;
					}
					h[i][k] = tmp;
					}
				}
			}
		x[ipiv] = x[jpiv] = 0.0;
		//change the order elements of h
		for(i = 0; i < n; i++) {
			if(i < ipiv) {
				tmp = h[i][ipiv];
				h[i][ipiv] = cosine * tmp + sine * h[i][jpiv];
				if(x[i] < fabs(h[i][ipiv])) {
					x[i] = fabs(h[i][ipiv]);	iq[i] = ipiv;
					}
				h[i][jpiv] = -sine * tmp + cosine * h[i][jpiv];
				if(x[i] < fabs(h[i][jpiv])) {
					x[i] = fabs(h[i][jpiv]);	iq[i] = jpiv;
					}
				}
			else if(i > ipiv) {
				if(i < jpiv) {
					tmp = h[ipiv][i];
					h[ipiv][i] = cosine * tmp + sine * h[i][jpiv];
					if(x[ipiv] < fabs(h[ipiv][i])) {
						x[ipiv] = fabs(h[ipiv][i]);	iq[ipiv] = i;
						}
					h[i][jpiv] = -sine * tmp + cosine * h[i][jpiv];
					if(x[i] < fabs(h[i][jpiv])) {
						x[i] = fabs(h[i][jpiv]);	iq[i] = jpiv;
						}
					}
				else if(i > jpiv) {
					tmp = h[ipiv][i];
					h[ipiv][i] = cosine * tmp + sine * h[jpiv][i];
					if(x[ipiv] < fabs(h[ipiv][i])) {
						x[ipiv] = fabs(h[ipiv][i]);	iq[ipiv] = i;
						}
					h[jpiv][i] = -sine * tmp + cosine * h[jpiv][i];
					if(x[jpiv] < fabs(h[jpiv][i])) {
						x[jpiv] = fabs(h[jpiv][i]);	iq[jpiv] = i;
						}
					}
				}
			}
		if(u) {
			for(i = 0; i < n; i++) {
				if(ipiv == jpiv) {
					i = i;	//DEBUG
					}
				tmp = u[i][ipiv];
				u[i][ipiv] = cosine * tmp + sine * u[i][jpiv];
				u[i][jpiv] = -sine * tmp +cosine * u[i][jpiv];
				}
			}
		}
	free(iq);	free(x);
	return false;
}

//---------------------------------------------------------------------------
//Use heap sort to sort elements of an float array
//Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1988/1989)
//      Numerical Recipes in C, Cambridge University Press, ISBN 0-521-35465-X
//      p. 245ff

void SortArray(long n, double *vals)
{
	long l, j, ir, i;
	double rra, *ra = vals-1;

	if(n < 2 || !vals) return;
	l=(n >> 1) + 1;				ir = n;
	for( ; ; ) {
		if(l > 1) rra = ra[--l];
		else {
			rra = ra[ir];		ra[ir] = ra[1];
			if(--ir == 1) {
				ra[1] = rra;	return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j] < ra[j+1]) ++j;
			if (rra < ra[j]) {
				ra[i] = ra[j];	j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i] = rra;
		}
}

//sorts array v1 making the corresponding rearrangement of v2

/*
void BubSrt2(long n, double *v1, double *v2)		// use this Bubble Sort to check SortArray2()
{
	bool bSorted = true;
	long i, it = 0;
	double tmp;

	do {
		bSorted = true;
		for (i = 0; i < (n - 1); i++) {
			if (v1[i] > v1[i + 1]) {
				tmp = v1[i];	v1[i] = v1[i + 1];	v1[i + 1] = tmp;
				tmp = v2[i];	v2[i] = v2[i + 1];	v2[i + 1] = tmp;
				bSorted = false;
				}
			}
		it++;
		} while (!bSorted);
	it = it;					//DEBUG: number of iterations needed
}
*/

void SortArray2(long n, double *v1, double *v2)
{
	long l, j, ir, i;
	double rra, rrb, *ra = v1-1, *rb = v2-1;

	if(n < 2 || !v1 || !v2) return;
//	BubSrt2(n, v1, v2);
//	return;
	l=(n >> 1) + 1;				ir = n;
	for( ; ; ) {
		if(l > 1) {
			rra = ra[--l];		rrb = rb[l];
			}
		else {
			rra = ra[ir];		rrb = rb[ir];
			ra[ir] = ra[1];		rb[ir] = rb[1];
			if(--ir == 1) {
				ra[1] = rra;	rb[1] = rrb;
				return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j] < ra[j+1]) ++j;
			if (rra < ra[j]) {
				ra[i] = ra[j];	rb[i] = rb[j];
				j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i] = rra;			rb[i] = rrb;
		}
}

//Use heap sort to sort elements of an xy array
void SortFpArray(long n, lfPOINT *vals)
{
	long l, j, ir, i;
	lfPOINT rra, *ra = vals-1;

	if(n < 2) return;
	l=(n >> 1) + 1;					ir = n;
	for( ; ; ) {
		if(l > 1) {
			rra.fx = ra[--l].fx; rra.fy = ra[l].fy;
			}
		else {
			rra.fx = ra[ir].fx;		rra.fy = ra[ir].fy;
			ra[ir].fx = ra[1].fx;	ra[ir].fy = ra[1].fy;	
			if(--ir == 1) {
				ra[1].fx = rra.fx;	ra[1].fy = rra.fy;
				return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j].fx < ra[j+1].fx) ++j;
			if (rra.fx < ra[j].fx) {
				ra[i].fx = ra[j].fx;	ra[i].fy = ra[j].fy;
				j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i].fx = rra.fx;				ra[i].fy = rra.fy;
		}
}

//randomize array
double *randarr(double *v0, int n, long *seed)
{
	double r, *v, *v_tmp;
	int i, j, l;

	if(!(v = (double*)malloc(n *sizeof(double)))) return 0L;
	if(!(v_tmp = (double*)memdup(v0, n *sizeof(double),0))) return 0L;
	for(l = n, i = 0; i < n; ) {
		r = ran2(seed);			j = (int)(r *((double)l));
		if(j < l) {
			v[i++] = v_tmp[j];
			if(j < l)memcpy(v_tmp+j, v_tmp+j+1, (l-j)*sizeof(double));
			l--;
			}
		}
	return v;
}

//resample array
double *resample(double *v0, int n, long *seed)
{
	double r, *v;
	int i, j;

	if(!(v = (double*)malloc(n *sizeof(double)))) return 0L;
	for(i = 0; i < n; ) {
		r = ran2(seed);			j = (int)(r *((double)n));
		if(j < n) v[i++] = v0[j];
		}
	return v;
}

//---------------------------------------------------------------------------
// Cubic Spline Interpolation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 96 ff.
void spline(lfPOINT *v, int n, double *y2)
{
	int i, k;
	double p, qn, sig, un, *u;

	u = (double *)malloc(n * sizeof(double));
	y2[0] = u[0] = 0.0;
	for(i = 1; i < (n-1); i++) {
		sig = (v[i].fx-v[i-1].fx)/(v[i+1].fx-v[i-1].fx);
		p = sig*y2[i-1]+2.0;			y2[i]=(sig-1.0)/p;
		u[i]=(v[i+1].fy-v[i].fy)/(v[i+1].fx-v[i].fx)-(v[i].fy-v[i-1].fy)/(v[i].fx-v[i-1].fx);
		u[i]=(6.0*u[i]/(v[i+1].fx-v[i-1].fx)-sig*u[i-1])/p;
		}
	qn = un = 0.0;
	y2[n-1] = (un - qn * u[n-2])/(qn*y2[n-2]+1.0);
	for(k = n-2; k >= 0; k--) {
		y2[k] = y2[k]*y2[k+1]+u[k];
		}
	free(u);
}

//---------------------------------------------------------------------------
// The Gamma Function: return the ln(G(xx)) for xx > 0
// Ref: B.W. Brown, J. Lovato, K. Russel (1994)
//    DCDFLIB.C, Library of C Routinesfor Cumulative Distribution Functions,
//    Inverses, and other Parameters.

double devlpl(double a[], int n, double x)
{
	double term;
	int i;

	for(term = a[n-1], i= n-2; i>=0; i--) term = a[i] + term * x;
	return term;
}


double gammln(double x)
{
	static double coef[] = {0.83333333333333023564e-1,-0.27777777768818808e-2, 
	0.79365006754279e-3, -0.594997310889e-3, 0.8065880899e-3};
static double scoefd[] = {0.62003838007126989331e2, 0.9822521104713994894e1,
	-0.8906016659497461257e1, 0.1000000000000000000e1};
static double scoefn[] = {0.62003838007127258804e2, 0.36036772530024836321e2,
	0.20782472531792126786e2, 0.6338067999387272343e1,0.215994312846059073e1,
	0.3980671310203570498e0, 0.1093115956710439502e0,0.92381945590275995e-2,
	0.29737866448101651e-2};
	double offset, prod, xx;
	int i,n;

    if(x < 6.0) {
		prod = 1.0e0;	    xx = x;
		while(xx > 3.0) {
			xx -= 1.0;			prod *= xx;
			}
		if(x <= 2.0) while(xx < 2.0) {
			prod /= xx;			xx += 1.0;
			}
		// compute rational approximation to gamma(x)
		return log(devlpl(scoefn, 9, xx-2.0) / devlpl(scoefd, 4, xx-2.0) * prod);
		}
	else {
		offset = 0.91893853320467274178;	// hln2pi
		// if necessary make x at least 12 and carry correction in offset
		n = 13.0 >= x ? (int)(12.0 - x) : 0;
		if(n) xx = x;
		else {
			for(i=1, prod = 1.0; i<= n; i++) prod *= (x+(double)(i-1));
			offset -= log(prod);			xx = x+(double)n;
			}
		// compute power series
		return devlpl(coef, 5, 1.0/(xx*xx)) / xx + (offset+(xx-0.5)*log(xx)-xx);
		}
}

//---------------------------------------------------------------------------
// Special Functions
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 166 ff.

//The Factorial Function: return n!
double factrl(int n)
{
	static int ntop = 4;
	static double a[33]={1.0, 1.0, 2.0, 6.0, 24.0};
	int j;

	if(n < 0) return 0.0;		//error: no factorial for negative numbers
	if(n > 32) return exp(gammln(n+1.0));
	while(ntop < n) {			//fill in table up to desired value
		j = ntop++;		a[ntop]=a[j] * ntop;
		}
	return a[n];
}

//returns the incomplete gamma function evaluated by its series representation
void gser(double *gamser, double a, double x, double *gln)
{
	int n;
	double sum, del, ap;

	*gln = gammln(a);
	if(x <= 0) {
		*gamser = 0.0;			return;
		}
	else {
		ap = a;					del = sum = 1.0/a;
		for(n = 1; n <= 100; n++) {
			ap += 1.0;			del *= x/ap;		sum += del;
			if(fabs(del) <= fabs(sum) * _PREC) {
				*gamser = sum * exp(-x + a * log(x)-(*gln));
				return;
				}
			}
		// maximum number of iterations exceeded
		*gamser = sum * exp(-x + a * log(x)-(*gln));
		}

}

//returns the incomplete gamma function evaluated by its continued fraction representation
void gcf(double *gammcf, double a, double x, double *gln)
{
	int n;
	double gold=0.0, g, fac=1.0, b1=1.0, b0=0.0, anf, ana, an, a1, a0=1.0;

	*gln=gammln(a);		a1=x;
	for(n=1; n <= 100; n++) {
		an = (double)n;			ana = an -a;		a0 = (a1 + a0 * ana) * fac;
		b0 = (b1 + b0 * ana) *fac;					anf = an * fac;
		a1 = x * a0 + anf * a1;						b1 = x * b0 + anf * b1;
		if(a1) {
			fac = 1.0 / a1;							g = b1 * fac;
			if(fabs((g-gold)/g) <= _PREC) {
				*gammcf = exp(-x + a * log(x) -(*gln)) * g;
				return;
				}
			gold = g;
			}
		}
	// maximum number of iterations exceeded
	*gammcf = exp(-x + a * log(x) -(*gln)) * gold;
}

//returns the incomplete gamma function P(a,x)
double gammp(double a, double x)
{
	double gamser, gammcf, gln;

	if(x < 0.0 || a <= 0.0) return 0.0;
	if(x < (a+1.0)) {
		gser(&gamser, a, x, &gln);			return gamser;
		}
	else {
		gcf(&gammcf, a, x, &gln);			return 1.0-gammcf;
		}
	return 0.0;
}

//returns the complementary incomplete gamma function Q(a,x)
double gammq(double a, double x)
{
	double gamser, gammcf, gln;

	if(x < 0.0 || a <= 0.0) return 0.0;
	if(x < (a+1.0)) {
		gser(&gamser, a, x, &gln);			return 1.0-gamser;
		}
	else {
		gcf(&gammcf, a, x, &gln);			return gammcf;
		}
	return 0.0;
}

//continued fraction for incomplete beta function, used by betai()
double betacf(double a, double b, double x)
{
	double qap, qam, qab, em, tem, d, bz, bm = 1.0, bp, bpp, az = 1.0, am = 1.0, ap, app, aold;
	int m;

	qab = a+b;		qap = a+1.0;		qam = a-1.0;	bz = 1.0-qab*x/qap;
	for(m = 1; m <= 100; m++) {
		em = (double)m;			tem = em+em;
		d = em*(b-em)*x/((qam+tem)*(a+tem));
		ap = az + d * am;		bp = bz + d *bm;
		d = -(a+em)*(qab+em)*x/((qap+tem)*(a+tem));
		app = ap + d * az;		bpp = bp + d * bz;
		aold = az;				am = ap/bpp;
		bm = bp/bpp;			az = app/bpp;
		bz = 1.0;
		if(fabs(az-aold) <= (_PREC * fabs(az))) return az;	//success: return
		}
	return az;												//fail: iterations exceeded
}

//The incomplete beta function Ix(a,b) for 0 <= x <= 1
double betai(double a, double b, double x)
{
	double bt;

	if(x < 0.0 || x > 1.0) return 0.0;		//range !
	if(x == 0.0 || x == 1.0) bt = 0.0;
	else
		bt = exp(gammln(a+b)-gammln(a)-gammln(b)+a*log(x)+b*log(1.0-x));
	if(x < (a+1.0)/(a+b+2.0)) return bt * betacf(a, b, x)/a;
	else return 1.0 - bt * betacf(b, a, 1.0 - x)/b;
}

//The following relations are obviously based on:
//  Abramowitz, M. & Stegun I.A. (1964): Hanbook of Mathematical Functions.
//    Applied Mathematics Series, vol. 55 (Washington: National Bureau
//    of Standards).

//the binomial coefficient
double bincof(double n, double k)
{
	if(n<0 || k<0 || k > n) return 0.0;
	return exp(gammln(n+1.0) - gammln(k+1.0) - gammln(n-k+1.0));
}

//the cumulative binomial distribution
double binomdistf(double k, double n, double p)
{
	if(k > n || n < 0.0 || p < 0.0 || p >1.0) return 0.0;
	return betai(n-k, k+1, p);
}

//the beta function
double betaf(double z, double w)
{
	return exp(gammln(z)+gammln(w)-gammln(z+w));
}

//the error function: not all compilers have a built in erf()
double errf(double x)
{
	return x < 0.0 ? -gammp(0.5, x*x) : gammp(0.5, x*x);
}

//the complementary error function
double  errfc(double x)
{
//	return x < 0.0 ? 2.0 - gammq(0.5, x*x) : gammq(0.5, x*x);
	return x < 0.0 ? 1.0 + gammp(0.5, x*x) : gammq(0.5, x*x);
}

//cumulative normal distribution
double norm_dist(double x, double m, double s)
{
	return 0.5 + errf((x - m)/(s * _SQRT2))/2.0;
}

//normal distribution
double norm_freq(double x, double m, double s)
{
	double ex;

	ex = (x-m)/s;	ex = exp(-0.5*ex*ex);
	return ex/(s*sqrt2pi);
}

//cumulative exponential distribution
double exp_dist(double x, double l, double)
{
	if(x >= 0.0 && l > 0.0) return 1.0-exp(-x*l);
	else return 0.0;
}

//inverse exponential distribution
double exp_inv(double p, double l, double)
{
	if(p >= 1.0) return HUGE_VAL;
	if(l <= 0.0) return 0.0;
	return -log(1.0-p)/l;
}

//exponential distribution
double exp_freq(double x, double l, double)
{
	if(x >= 0.0 && l > 0.0) return l*exp(-x*l);
	else return fabs(l);
}

//cumulative lognormal distribution
double lognorm_dist(double x, double m, double s)
{
	return 0.5 + errf((log(x) - m)/(s * _SQRT2))/2.0;
}

//lognormal distribution
double lognorm_freq(double x, double m, double s)
{
	double tmp;

	if(x > 0.0 && m > 0.0 && s > 0.0) {
		tmp = (log(x)-m)/s;
		return exp(-0.5*tmp*tmp)/(x*s*sqrt2pi);
		}
	return 0.0;
}

//chi square distribution
double chi_dist(double x, double df, double)
{
	if(x <= 0.0) return 1.0;
	return gammq(df/2.0, x/2.0);
}

double chi_freq(double x, double df)
{
	if(x < 0.0 || df <= 0.0) return 0.0;
	if(x < 1.0e-32) x = 1.0e-32;
//formula by Wikipedia 
//	return exp(log(2.0)*(1.0-df/2.0)+log(x)*(df-1.0)+x*x/-2.0-gammln(df/2.0));
//formula by StatSoft's STATISTICA documentation
	return exp(-x/2.0+log(x)*(df/2.0-1.0)-log(2.0)*df/2.0-gammln(df/2.0));
}

//t-distribution
double t_dist(double t, double df, double)
{
	return betai(df/2.0, 0.5, (df/(df+t*t)));
}

double t_freq(double t, double df)
{
	double a, b, c, d;
 
	a = gammln((df+1.0)/2.0);		b = log(sqrt(df * _PI));
	c = gammln(df/2.0);				d = log(1.0+t*t/df) * (df+1)/2.0;
	return exp(a-b-c-d);
}

//poisson distribution
double pois_dist(double x, double m, double)
{
	return gammq(x+1.0, m);
}

//f-distribution
double f_dist(double f, double df1, double df2)
{
	return f > 0.0 ? betai(df2/2.0, df1/2.0, df2/(df2+df1*f)): 1.0;
}

double f_freq(double x, double df1, double df2)
{
	double a, b, c, d;

	a = gammln((df1+df2)/2.0);		b = gammln(df1/2.0) + gammln(df2/2.0);
	c = log(df1/df2) * df1/2.0 + log(x) * (df1/2.0-1.0);
	d = log(1+(df1/df2)*x) * (-(df1+df2)/2.0);
	return exp(a-b+c+d);
}

//---------------------------------------------------------------------------
// The Weibull distribution
//---------------------------------------------------------------------------
double weib_dist(double x, double shape, double scale)
{
	double dn=1.0, sum, term, tmp;

	if(shape <= 0.0 || scale <= 0.0) return HUGE_VAL;
	if(x <= 0.0) return 0.0;
	term = -pow(x/scale, shape);		tmp = fabs(term);
	if(tmp < 2.22e-16) return tmp;
	if (tmp > 0.697) return -exp(term)+1.0;
	x = sum = term;
	do {				//do taylor series
		dn += 1.0 ;		term *= x/dn;		sum += term;
		}while (fabs(term) > fabs(sum) * 2.22e-16) ;
	return -sum;
}

double weib_freq(double x, double shape, double scale)
{
	double tmp1, tmp2;

	if (shape <= 0.0 || scale <= 0.0) return HUGE_VAL;
	if (x < 0) return 0.0;
	if(x > -HUGE_VAL && x < HUGE_VAL) {
		if(x == 0.0 && shape < 1.0) return HUGE_VAL;
		tmp1 = pow(x / scale, shape - 1.0);
		tmp2 = tmp1 * (x / scale);
		return shape * tmp1 * exp(-tmp2) / scale;
		}
	return HUGE_VAL;
}

//---------------------------------------------------------------------------
// The geometric distribution
//---------------------------------------------------------------------------
double geom_freq(double x, double p)
{ 
    if (p <= 0 || p > 1 || x < 0.0) return HUGE_VAL;
	x = floor(x + 1.0e-16);
	return pow(1.0 - p, x) * p;
}

double geom_dist(double x, double p)
{
	double sum, x1;

	for(x1 = sum = 0.0; x1 <= x; sum += geom_freq(x1, p), x1 += 1.0);
	return sum;
}

//---------------------------------------------------------------------------
// The hypergeometric distribution
//---------------------------------------------------------------------------
double hyper_freq(double k, double n0, double m, double n1)
{
	double pr;

	if(k < 0.0 || m < 0.0 || n1 < 0.0 || n1 > n0+m) return HUGE_VAL;
	k = floor(k + 1.0e-16);		n0 = floor(n0 + 1.0e-16);
	m = floor(m + 1.0e-16);		n1 = floor(n1 + 1.0e-16);

	pr = gammln(m+1.0) - gammln(k+1.0) - gammln(m-k+1.0)
		+ gammln(n0-m+1.0) - gammln(n1-k+1.0) - gammln(n0-m-n1+k+1.0)
		- gammln(n0+1.0) + gammln(n1+1.0) + gammln(n0-n1+1.0);
	return exp(pr);
}

double hyper_dist(double k, double n0, double m, double n1)
{
	double sum, x1;

	for(x1 = sum = 0.0; x1 <= k; sum += hyper_freq(x1, n0, m, n1), x1 += 1.0);
	return sum;
}

//---------------------------------------------------------------------------
// The Cauchy (Lorentz) distribution
//---------------------------------------------------------------------------
double cauch_dist(double x, double loc, double scale)
{
	double y;

	if(scale < 0.0) return HUGE_VAL;
	x = (x - loc) / scale;
	if(x > -HUGE_VAL && x < HUGE_VAL) {
		if (fabs(x) > 1.0) {
			y = atan(1.0/x)/_PI;		return (x > 0) ? 1.0-y : -y;
			} 
		else return 0.5 + atan(x)/_PI;
		}
	return HUGE_VAL;
}

double cauch_freq(double x, double loc, double scale)
{
	double y;

	if(scale < 0.0) return HUGE_VAL;
	if(x > -HUGE_VAL && x < HUGE_VAL) {
		y = (x - loc) / scale;
		return 1.0 / (_PI * scale * (1.0 + y*y));
		}
	return HUGE_VAL;
}

//---------------------------------------------------------------------------
// The Logistic distribution
//---------------------------------------------------------------------------
double logis_dist(double x, double loc, double scale)
{
	if(scale < 0.0) return HUGE_VAL;
	x = exp(-(x - loc) / scale);
	if(x > -HUGE_VAL && x < HUGE_VAL) {
		return 1.0/(1.0 + x);
		}
	return HUGE_VAL;
}

double logis_freq(double x, double loc, double scale)
{
	double e, f;

	x = fabs((x - loc) / scale);
	if(x > -HUGE_VAL && x < HUGE_VAL) {
		e = exp(-x);     f = 1.0 + e;	
		return  e / (scale * f*f);
		}
	return HUGE_VAL;
}

//---------------------------------------------------------------------------
// Shapiro-Wilk W test and its significance level
// Algorithm AS 394, 1995, Appl. Statist. 44(4), 547-551
//
static int do_swilk(double (*func)(double, double, double), double p1, double p2, 
	double *x, int n, int n1, int n2, double *a, double *w, double *pw)
{

//initialized data
const static double z90 = 1.2816;		//tinv(0.2, inf)
const static double z95 = 1.6449;		//tinv(0.1, inf)
const static double z99 = 2.3263;		//tinv(.05, inf)
const static double zm = 1.7509;		//(z90 + z95 + z99)/3
const static double zss = 0.56268;
const static double bf1 = 0.8378;
const static double xx90 = 0.556;
const static double xx95 = 0.622;
const static double sqrth = 0.70711;	//sqrt(0.5)
const static double smal = 1.0e-19;		//small value
const static double pi6 = 1.909859;
const static double stqr = 1.047198;	//pi / 3

//polynomial coefficients
static double g[2] = {-2.273, 0.459};
static double c1[6] = {0.0, 0.221157, -0.147981, -2.07119, 4.434685, -2.706056};
static double c2[6] = {0.0, 0.042981, -0.293762, -1.752461, 5.682633, -3.582633};
static double c3[4] = {0.544, -0.39978, 0.025054, -6.714e-4};
static double c4[4] = {1.3822, -0.77857, 0.062767, -0.0020322};
static double c5[4] = {-1.5861, -0.31082, -0.083751, 0.0038915};
static double c6[3] = {-0.4803, -0.082676, 0.0030302};
static double c7[2] = {0.164, 0.533};
static double c8[2] = {0.1736, 0.315};
static double c9[2] = {0.256, -0.00635};

	//local variables
	int i, j, ncens, i1, nn2;
	double zbar, ssassx, summ2, ssumm2, gamma, delta, range;
	double a1, a2, an, bf, ld, m, s, sa, xi, sx, xx, y, w1;
	double fac, asa, an25, ssa, z90f, sax, zfm, z95f, zsd, z99f, rsn, ssx, xsx;

	//parameter adjustment
	--a;

	*pw = 1.0;
	if(*w >= 0.0) *w = 1.0;
	an = (double)(n);			nn2 = n>>1;
	if(n2 < nn2) return 3;
	if(n < 3) return 1;
	// calculate coefficients a[]
	if(true) {
		if(n == 3) a[1] = sqrth;
		else {
			for(i = 1, summ2 = 0.0, an25 = an + 0.25; i <= n2; ++i) {
				a[i] = distinv(func, p1, p2, (i-0.375)/an25, 0);
				summ2 += (a[i] * a[i]);
				}
			summ2 *= 2.0;			ssumm2 = sqrt(summ2);
			rsn = 1.0 / sqrt(an);	a1 = devlpl(c1, 6, rsn) -a[1]/ssumm2;
			//normalize a[]
			if(n > 5) {
				i1 = 3;
				a2 = -a[2] / ssumm2 + devlpl(c2, 6, rsn);
				fac = sqrt((summ2 - 2.0*a[1]*a[1] - 2.0*a[2]*a[2])
					/ (1.0 - 2.0*a1*a1 - 2.0*a2*a2));
				a[2] = a2;
				}
			else {
				i1 = 2;
				fac = sqrt((summ2 -2.0*a[1]*a[1]) / (1.0 - 2.0*a1*a1));
				}
			a[1] = a1;
			for(i = i1; i <= nn2; ++i) a[i] /= -fac;
			} 
		}
	if(n1 < 3) return 1;
	ncens = n - n1;
	if(ncens < 0 || (ncens > 0 && n < 20)) return 4;
	delta = (double)ncens / an;
	if(delta > 0.8) return 5;
	//if w input as negative, calculate significance level of -w
	if(*w < 0.0) { 
		w1 = 1.0 + *w;
		goto sw_prob;
		}
	//check for zero range
	if((range = x[n1-1] -x[0]) < smal) return 6;
	//check for sort order
	xx = x[0]/range;	sx = xx;	sa = -a[1];		j = n -1;
	for(i = 1; i < n1; --j) {
		xi = x[i] / range;			sx += xi;			++i;
		if(i != j) sa += i > j ? a[i < j ? i : j] : -a[i < j ? i : j];
		xx = xi;
		}
	//calculate w statistic as squared correlation between data and coefficients
	sa /= n1;		sx /= n1;		ssa = ssx = sax = 0.0;		j = n -1;
	for(i = 0; i < n1; ++i, --j) {
		if(i > j) asa = a[1+j] - sa;
		else if(i < j) asa = -a[1+i] - sa;
		else asa = -sa;
		xsx = x[i] / range - sx;		ssa += asa * asa;
		ssx += xsx * xsx;				sax += asa * xsx;
		}
	ssassx = sqrt(ssa * ssx);
	w1 = (ssassx - sax) * (ssassx + sax) / (ssa * ssx);
sw_prob:
	*w = 1.0 - w1;			//reduce rounding errors
	if(n == 3) {
		*pw = pi6 * (asin(sqrt(*w)) - stqr);
		return 0;
		}
	y = log(w1);
	xx = log(an);
	if(n <= 11) {
		gamma = devlpl(g, 2, an);
		if(y >= gamma) {
			*pw = smal;		return 0;
			}
		y = -log(gamma - y);		m = devlpl(c3, 4, an);
		s = exp(devlpl(c4, 4, an));
		}
	else {					//n >= 12
		m = devlpl(c5, 4, xx);		s = exp(devlpl(c6, 3, xx));
		}
	//Censoring by proportion  NCENS/N
	if(ncens > 0) {
		ld = -log(delta);			bf = 1.0 + xx * bf1;
		z90f = z90 + bf * pow(devlpl(c7, 2, pow(xx90, xx)), ld);
		z95f = z95 + bf * pow(devlpl(c8, 2, pow(xx95, xx)), ld);
		z99f = z99 + bf * pow(devlpl(c9, 2, xx), ld);
		//Regress z90f ... z99f on normal deviates z90 ... z99
		//   to get pseudo-mean and pseudo-sd of z as the slope and intercept 
		zfm = (z90f + z95f + z99f)/3.0;
		zsd = (z90 * (z90f - zfm) + z95 * (z95f - zfm) + z99 * (z99f - zfm)) / zss;
		zbar = zfm - zsd * zm;		m += zbar * s;		s *= zsd;
		}
	*pw = 1.0 - norm_dist(y, m, s);
	return 0;
}

void swilk1(int n, double *v0, double (*func)(double, double, double), double p1, double p2, 
	bool bsorted, double *w, double *p)
{
	double *v, *a;

	if(!n || !w || !p) return;
	*w = *p = 1.0;
	a = (double*)malloc(n *sizeof(double));
	if(!a) return;
	if(!bsorted && (v = (double*)memdup(v0, n*sizeof(double), 0)))SortArray(n, v);
	else if(bsorted) v = v0;
	else return;
	if(do_swilk(func, p1, p2, v, n, n, n>>1, a, w, p)){
		//an error occured
		*w = *p = -1.0;
		}
	free(a);	if(v != v0) free(v);
}

//Kolmogorov-Smirnov's test and distribution of D
// (1) Miller L. (1956) Journal of the American Statistical Association.  51: 111-121
// (2) Mises R. (1964) Mathematical Theory of Probability and Statistics (New York: Academic Press)
//     Chapters IX(C) and IX(E)
// (3) Press W.H., Flannery B.P.,Teukolsky S.A., Vetterling W.T. (1988/1989)
//     Numerical Recipes in C, Cambridge University Press, ISBN 0-521-35465-X, pp. 490 ff.
//
double ks_dist(int n, double d)
{
	double j, jn, sum, las, q, r, s, dn = (double)n;

	las = floor(dn - dn * d);
	for (j = sum = 0.0; j <= las; j += 1.0) {
		jn = j / dn;							q = gammln(dn+1) - gammln(j+1) - gammln(dn-j+1.0);
		r = (dn - j) * log( 1 - d - jn );		s = (j - 1.0) * log( d + jn );
		sum += exp(q + r + s);
		}
	return(d*sum);
}

void KolSmir(int n, double *v0, double (*func)(double, double, double), double p1, double p2, 
	bool bsorted, double *d, double *p)
{
	int i;
	double *v, *dev, *x, ff, dt, dt1, dt2;
	double dn = (double)n, f0 = 0.0;

	if(!n || !d || !p) return;
	*d = *p = 0.0;
	dev = (double*)malloc(n*sizeof(double));
	if(!dev) return;
	x = (double*)malloc(n*sizeof(double));
	if(!x){
		free(dev);						return;
		}
	if(!bsorted && (v = (double*)memdup(v0, n*sizeof(double), 0)))SortArray(n, v);
	else if(bsorted) v = v0;
	else return;
	for(i = 0, *d = 0.0; i < n; i++) {
		x[i] = (double)(i+1)/dn;		ff = (*func)(v[i], p1, p2); 
		dt1 = fabs(f0-ff);				dt2 = fabs(dev[i] = (f0 = x[i])-ff);
		dt = dt1 > dt2 ? dt1 : dt2;		if(dt > *d) *d = dt;
		}
	free(dev);	free(x);
	*p = ks_dist(n, *d);
	if(v != v0) free(v);
}

//---------------------------------------------------------------------------
// Inverse of statitistical functions:
// funcd supplies the function value fn and the derivative df of the function sf at x
void funcd(double x, double *fn, double *df, double (*sf)(double, double, double), 
		   double df1, double df2, double p)
{
	double y1, y2;

	*fn = (sf)(x, df1, df2);
	if(sf == norm_dist) *df = norm_freq(x, df1,df2);
	else if(sf == chi_dist) *df = -chi_freq(x, df1);
	else if(sf == t_dist) *df = -2.0 * t_freq(x, df1);
	else if(sf == f_dist) *df = -1.0 * f_freq(x, df1, df2);
	else if(sf == lognorm_dist) *df = lognorm_freq(x, df1, df2);
	else if(sf == weib_dist) *df = weib_freq(x, df1, df2);
	else if(sf == cauch_dist) *df = cauch_freq(x, df1, df2);
	else if(sf == logis_dist) *df = logis_freq(x, df1, df2);
	else {		//numerical differentiation
		y1 = (sf)(x * 0.995, df1, df2);		y2 = (sf)(x * 1.005, df1, df2);
		*df = (y2-y1)*100.0/x;
		}
	*fn = *fn - p;
}

//distinv does actual Newton-Raphson root finding
double distinv(double (*sf)(double, double, double), double df1, double df2, double p, double x0)
{
	int i, j;
	double df, df0, adf, dx, f, rtn;

	for(j = 0, rtn = dx = x0; j < 200; j++) {
		for(i = 0, df0 = 0.0; i < 20; i++) {
			funcd(rtn, &f, &df, sf, df1, df2, p);
			if((adf=fabs(df)) > 1.0e-12 || df0 > adf) break;
			rtn += (dx = dx/2.0);				df0 = adf;
			if(i >= 19) return HUGE_VAL;
			}
		dx = f/df*(0.01*(double)(100-j));		rtn -= dx;
		if(fabs(dx) < _PREC && j > 3)return rtn; 
		}
	return HUGE_VAL;
}

//---------------------------------------------------------------------------
//some statistical basics
//do quartiles, median of data
void d_quartile(int n, double *v, double *q1, double *q2, double *q3)
{
	int n2, n3;
	double f1, f2;

	if(!v || n<2) return;
	SortArray(n, v);			n2 = n >> 1;
	if(q1) {
		n3 = n2 >> 1;
		switch(n%4) {
		case 3:		n3 ++;		f1 = 2.0;		f2 = 2.0;		break;
		case 2:		n3 ++;		f1 = 3.0;		f2 = 1.0;		break;
		case 1:		n3 ++;		f1 = 4.0;		f2 = 0.0;		break;
		default:	f1 = 1.0;	f2 = 3.0;						break;
			}
		*q1 = (f1*v[n3-1] + f2*v[n3])/4.0;
		}
	if(q2) {
		if(n & 1) *q2 = v[n2];
		else *q2 = (v[n2-1] + v[n2])/2.0;
		}
	if(q3) {
		n3 = n2 >> 1;
		switch(n%4) {
		case 3:		n3++;		f1 = 2.0;		f2 = 2.0;	break;
		case 2:		f1 = 3.0;	f2 = 1.0;					break;
		case 1:		f1 = 4.0;	f2 = 0.0;					break;
		default:	f1 = 1.0;	f2 = 3.0;					break;
			}
		n3 += n2;
		*q3 = (f2*v[n3-1] + f1*v[n3])/4.0;
		}
}

// statistical basics partly based on
// Davies, J. and Gogh, B. (2000), GSL-1.7 - The GNU scientific library
//
//do variance
double d_variance(int n, double *v, double *mean, double *ss)
{
	int i;
	double d, m, va, e;

	for(i = 0, m = 0.0, d = 1.0; i < n; i++, d += 1.0) {
		m += (v[i] - m)/d;
		}
	if (mean) *mean = m;
	for(i = 0, va = 0.0, d = 1.0; i < n; i++, d += 1.0) {
		e = v[i] - m;		va += (e * e - va)/d;
		}
	if (ss) *ss = va * (double)n;
	return va * ((double)n/((double)(n-1)));
}

//do arithmethic mean
double d_amean(int n, double *v)
{
	int i;
	double d, mean;

	for(i = 0, mean = 0.0, d = 1.0; i < n; i++, d += 1.0) {
		mean += (v[i] - mean)/d;
		}
	return mean;
}


//do geometric mean
double d_gmean(int n, double *v)
{
	int i;
	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		if(v[i] <= 0.0) return 0.0;
		sum += log(v[i]);
		}
	return exp(sum/n);
}

//do harmonic mean
double d_hmean(int n, double *v)
{
	int i;

	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		if(v[i] == 0.0) return 0.0;
		sum += 1.0/(v[i]);
		}
	return (n/sum);
}

//kurtosis
double d_kurt(int n, double *v)
{
	double sum, avg, sd, tmp, dn = n;
	int i;

	for(i = 0, sum = 0.0; i < n; i++) sum += v[i];
	avg = sum / dn;		tmp = v[i] - avg;
	for(i = 0, sum = 0.0; i < n; i++) sum += (tmp * tmp);
	sd = sqrt(sum / (dn - 1.0));	tmp = v[i] - avg;
	for(i = 0, sum=0.0; i < n; i++) sum += ((tmp/sd)*tmp*tmp*tmp);
	sum *= ((dn*(dn+1.0))/((dn-1.0)*(dn-2.0)*(dn-3.0)));
	tmp = (3.0 * (dn-1.0) * (dn-1.0))/((dn-2.0)*(dn-3.0));
	return sum - tmp;
}

//skewness
double d_skew(int n, double *v)
{
	double sum, avg, sd, tmp, dn = n;
	int i;

	for(i = 0, avg = 0.0; i < n; i++) avg += ((v[i]-avg)/((double)(i+1)));
	tmp = v[i] - avg;
	for(i = 0, sum = 0.0; i < n; i++) sum += (tmp * tmp);
	sd = sqrt(sum / (dn - 1.0));
	for (i = 0, sum = 0.0; i < n; i++) {
		tmp = v[i] - avg;
		sum += (( tmp/ sd)*tmp*tmp);
		}
	return sum * dn/((dn-1.0)*(dn-2.0));
}

//---------------------------------------------------------------------------
// Create a frequency distribution by counting the elements which may be 
// assigned to a bin
double d_classes(DataObj *d, double start, double step, double *v, int nv, char *range)
{
	int nc, *f;
	long i, j, r, c;
	AccRange *ar;

	if(!range || !nv || !v || step <= 0.0 || !(ar = new AccRange(range))) return 0.0;
	if(!(nc = ar->CountItems()) || !ar->GetFirst(&c, &r) || !(f=(int*)calloc(nc, sizeof(int)))) {
		delete ar;				return 0.0;
		}
	for(i = 0; i < nv; i++) {
		j = (int)(floor((v[i] - start)/step));
		if(j < 0) j = 0;
		if(j >= nc) j = (nc-1);
		f[j]++;
		}
	for( ; nc > 0 && !(f[nc-1]); nc--);
	for(i = 0; ar->GetNext(&c, &r) && i < nc; i++) {
		d->SetValue(r, c, (double)f[i]);
		}
	free(f);					return ((double)nv);
}

//---------------------------------------------------------------------------
// Pearsons linear correlation
// (1) W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 503 ff.
// (2) B. Gough (2000), linear.c, gsl-1.7 the GNU scientific library
double d_pearson(double *x, double *y, int n, char *dest, DataObj *data, double *ra)
{
	long j, r, c;
	double yt, xt, t, df, res[4];
	double syy=0.0, sxy=0.0, sxx=0.0, ay=0.0, ax=0.0;
	AccRange *rD;


	for(j = 0;	j < n; j++) {				// find means
		ax += (x[j] - ax) / (j+1);			ay += (y[j] - ay) / (j+1);
		}
	for(j = 0; j < n; j++) {				// correlation
		xt = x[j] - ax;						yt = y[j] - ay;
		sxx += (xt*xt-sxx) / (j+1);			syy += (yt*yt-syy) / (j+1);
		sxy += (xt*yt-sxy) / (j+1);
		}
	res[0] = sxy/sqrt(sxx*syy);				//pearsons r
	if(dest || ra) {
		res[1] = 0.5 * log((1.0+res[0]+_PREC)/(1.0-res[0]+_PREC));	//Fishers z-transform
		df = n-2;
		t = res[0]*sqrt(df/((1.0-res[0]+_PREC)*(1.0+res[0]+_PREC)));	//Student's t
		res[2] = betai(0.5*df, 0.5, df/(df+t*t));					//probability
		res[3] = n;
		}
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 4 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if (ra){
		memcpy(ra, res, 4 * sizeof(double));
		}
	return res[0];
}

//---------------------------------------------------------------------------
// Given an array w, rank returns the rank of v1 in v
// if v1 is not found in v 0 is returned
double d_rank(int n, double *v, double v1)
{
	double *sv;
	int i, j;

	if(!n || !v) return 0.0;
	if(n < 2) return 1.0;
	sv = (double*)memdup(v, n * sizeof(double), 0);
	if(!sv) return 0.0;
	SortArray(n, sv);
	for(i = j = 0; i < n; i++) {
		if(v1 == sv[i]) {
			for( ;(i+j)<n; j++) if(sv[i+j] > v1) break;
			free(sv);				return (double)i + 1.0 + (((double)j-1.0)/2.0);
			}
		}
	free(sv);						return 0.0;
}

//---------------------------------------------------------------------------
// Spearman rank-order correlation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Recipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 507 ff.

//Given a sorted array w, crank replaces the elements by their rank
void crank(int n, double *w0, double *s)
{
	int j=1, ji, jt;
	double t, rank, *w = w0-1;

	*s = 0.0;
	while (j < n) {
		if(w[j+1] != w[j]) {
			w[j] = j;		++j;
			}
		else {
			for(jt = j+1; jt <= n; jt++) if(w[jt] != w[j]) break;
			rank = 0.5 * (j+jt-1);
			for(ji = j; ji <= (jt-1); ji++) w[ji] = rank;
			t = jt -j;		*s += t*t*t -t;				j = jt;
			}
		}
	if(j == n) w[n] = n;
}

//the actual rank correlation
double d_spearman(double *sx, double *sy, int n, char *dest, DataObj *data, double *ra)
{
	long j, r, c;
	double *x, *y, vard, t, sg, sf, fac, en3n, en, df, aved, tmp;
	double res[6];
	AccRange *rD;

	if(!(x = (double*)memdup(sx, n*sizeof(double), 0)) 
		|| !(y = (double*)memdup(sy, n*sizeof(double), 0)))return 0.0;
	SortArray2(n, x, y);			crank(n, x, &sf);
	SortArray2(n, y, x);			crank(n, y, &sg);
	for(j = 0, res[0] = 0.0; j < n; j++){
		tmp = x[j] - y[j];
		res[0] += (tmp*tmp);
		}
	en = n;						en3n = en*en*en -en;
	aved = en3n/6.0 - (sf+sg)/12.0;
	fac = (1.0-sf/en3n)*(1.0-sg/en3n);
	tmp = (en + 1.0);
	vard = ((en-1.0)*en*en*tmp*tmp/36.0)*fac;
	res[1] = (res[0]-aved)/sqrt(vard);
	res[2] = errfc(fabs(res[1])/_SQRT2);
	res[3] = (1.0-(6.0/en3n)*(res[0]+0.5*(sf+sg)))/fac;
	t = res[3]*sqrt((en-2.0)/((res[3]+1.0)*(1.0-res[3])));
	df = en-2.0;	res[5] = (double)n;
    res[4] = betai(0.5*df, 0.5, df/(df+t*t));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 6 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if(ra) {
		memcpy(ra, res, 6 * sizeof(double));
		}
	free(x);						free(y);
	return res[3];
}

//---------------------------------------------------------------------------
// Kendal's non-parametric correlation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Recipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 510 ff.

double d_kendall(double *x, double *y, int n, char *dest, DataObj *data, double *ra)
{
	int j, k, n1, n2, is;
	long r, c;
	double aa, a1, a2, sv, res[4];
	AccRange *rD;

	for (j = n1 = n2 = is = 0; j < (n-1); j++) {
		for(k = j+1; k < n; k++) {
			a1 = x[j] - x[k];		a2 = y[j] - y[k];		aa = a1*a2;
			if(aa != 0.0) {
				n1++;				n2++;
				if (aa > 0.0) is++;
				else is--;
				}
			else {
				if(a1 != 0.0) n1++;	
				if(a2 != 0.0) n2++;
				}
			}
		}
	res[0] = ((double)is)/(sqrt((double)n1) * sqrt((double)n2));
	sv = (4.0 * ((double)n) + 10.0)/(9.0*((double)n)*((double)(n-1)));
	res[1] = res[0]/sqrt(sv);	res[2] = errfc(fabs(res[1])/_SQRT2);
	res[3] = n;			
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 4 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if (ra){
		memcpy(ra, res, 4 * sizeof(double));
		}
	return res[0];
}


//linear regression
double d_regression(double *x, double *y, int n, char *dest, DataObj *data, double ra[])
// parameters: x[n] the source data x
//             y[n] the source data y
//             n the number of data pairs
//             dest range of spreadsheet to write results, can be NULL
//             data a handle to the spreadsheet, can be NULL
//             ra[10] receives the regression results,  can be NULL
//             the return value is n
{
	double sx, sy, dx, dy, sxy, sxx, syy, sdy, df;
	double res[10];		// slope, intercept, mean x, mean y, SE of slope, 
						//   variance(x), variance(y), variance(fit), F of regression, significance
	int i, j;
	long r, c;
	AccRange *rD;

	if(n < 2) return 0.0;
	for(i = 0, 	sx = sy = 0.0; i < n; i++) {
		sx += x[i];			sy += y[i];
		}
	res[2] = sx /n;			res[3] = sy/n;
	sxy = sxx = syy = 0.0;
	for(i = 0; i < n; i++) {
		dx = x[i]-res[2];	dy = y[i]-res[3];
		sxx += (dx*dx);		syy += (dy*dy);		sxy += (dx*dy);
		}
	res[0] = sxy / sxx;		res[1] = res[3] - res[0] * res[2];
	for(i = 0, sdy = 0.0; i < n; i++) {
		dy = y[i] - (res[1] + x[i] *res[0]);
		sdy += (dy * dy);
		}
	sdy = sdy/(n-2);		res[4] = sqrt(sdy/sxx);		df = (n-2);
	res[5] = sxx/(n-1);		res[6] = syy/(n-1);			res[7] = sdy;
	res[8] = sxy/sdy*sxy/sxx;
	res[9] = betai(df/2.0, 0.5, df/(df+res[8]));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 10 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if (ra)	memcpy(ra, res, 10 * sizeof(double));
	return n;
}

//covariance
double d_covar(double *x, double *y, int n, char *, DataObj *)
{
	int i;
	double sx, sy, dx, dy, sxy;

	if(n < 2) return 0.0;
	for(i = 0, 	sx = sy = 0.0; i < n; i++) {
		sx += x[i];			sy += y[i];
		}
	sx /= n;		sy /= n;		sxy = 0.0;
	for(i = 0; i < n; i++) {
		dx = x[i]-sx;		dy = y[i]-sy;
		sxy += (dx*dy - sxy) / (i+1);
		}
	return sxy;
}

//Mann-Whitney U Test
double d_utest(double *x, double *y, int n1, int n2, char *dest, DataObj *data, double *ra)
{
	double *da, *ta, u1, u2, su, su1, ts, dn1 = n1, dn2 = n2;
	double res[9];
	AccRange *rD;
	int i, j, n;
	long r, c;

	if(!x || !y || n1 < 2 || n2 < 2) return 0.0;
	da = (double*)malloc((n = (n1+n2)) * sizeof(double));
	ta = (double*)malloc(n * sizeof(double));
	if(!da || !ta) {
		if(da) free(da);
		if(ta) free(ta); 
		return 0.0;
		}
	for(i = 0; i < n1; i++) {
		da[i] = x[i];		ta[i] = 1.0;
		}
	for(j = 0; j < n2; j++) {
		da[i] = y[j];		ta[i++] = 2.0;
		}
	SortArray2(n, da, ta);	crank(n, da, &ts);
	for(i = 0, res[0] = res[1] = 0.0; i < n; i++) {
		if(ta[i] == 1.0) res[0] += da[i];
		else res[1] += da[i];
		}
	free(da);										free(ta);
	u1 = (dn1*dn2 + (dn1*(dn1+1))/2.0) - res[0];	u2 = (dn1*dn2 + ((dn2+1)*dn2)/2.0) - res[1];
	su = sqrt((dn1*dn2*(dn1+dn2+1))/12.0);			res[2] = u2 > u1 ? u2 : u1;
	su1 = ((dn1*dn2)/((dn1+dn2)*(dn1+dn2-1))) * (((dn1+dn2)*(dn1+dn2)*(dn1+dn2)-(dn1+dn2)-ts)/12.0);
	su1 = sqrt(su1);
	res[3] = (res[2] - (n1*n2)/2.0)/su;			res[6] = errfc(res[3]/_SQRT2);
	res[4] = n1;								res[5] = n2;
	res[7] = (res[2] - (n1*n2)/2.0)/su1;		res[8] = errfc(res[7]/_SQRT2);
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(i = 0; i < 9 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if (ra)	memcpy(ra, res, 9 * sizeof(double));
	return res[8];
}

//t-test
double d_ttest(double *x, double *y, int n1, int n2, char *dest, DataObj *data, double *results)
{
	int i;
	long r, c;
	double sx, sy, mx, my, d, df, p;
	double res[9];			// mean1, SD1, n1, mean2, SD2, n2, p if variances equal,
	AccRange *rD;			//    corrected df, corrected p

	d_variance(n1, x, &mx, &sx);		d_variance(n2, y, &my, &sy);
	d = ((sx+sy)/(n1+n2-2)) * ((double)(n1+n2)/(double)(n1*n2));
	d = (mx-my)/sqrt(d);	//Student's t

	//Welch's correction for differences in variance
	df = (sx/(double)n1)*(sx/(double)n1)/(double)(n1+1)+(sy/(double)n2)*(sy/(double)n2)/(double)(n2+1);
	df = (sx/(double)n1+sy/(double)n2)*(sx/(double)n1+sy/(double)n2)/df;
	df -= 2.0;		df = floor(df);

//	an alternative formula for correction
//	p = (sx/(double)n1)*(sx/(double)n1)/(double)(n1-1) + (sy/(double)n2)*(sy/(double)n2)/(double)(n2-1);
//	df = (sx/(double)n1 + sy/(double)n2) * (sx/(double)n1 + sy/(double)n2) / p;

	p = betai(df/2.0, 0.5, (df/(df+d*d)));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[0] = mx;	res[1] = sqrt(sx/(double)(n1-1));	res[2] = n1;
		res[3] = my;	res[4] = sqrt(sy/(double)(n2-1));	res[5] = n2;
		res[7] = df;	df = (n1-1) + (n2-1);	res[6] = betai(df/2.0, 0.5, (df/(df+d*d)));
		res[8] = p;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 9 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if(results) {
		results[0] = mx;	results[1] = sqrt(sx/(double)(n1-1));	results[2] = n1;
		results[3] = my;	results[4] = sqrt(sy/(double)(n2-1));	results[5] = n2;
		results[7] = df;	df = (n1-1) + (n2-1);	results[6] = betai(df/2.0, 0.5, (df/(df+d*d)));
		results[8] = p;		results[9] = d;
		}
	return p;
}

//t-test for paired samples
double d_ttest2(double *x, double *y, int n, char *dest, DataObj *data, double *ra)
{
	double sx, sy, mx, my, df, cov, sd, t, p;
	long i, r, c;
	double res[6];			// mean1, SD1, mean2, SD2, n, p 
	AccRange *rD;

	d_variance(n, x, &mx, &sx);		d_variance(n, y, &my, &sy);
	sx = d_variance(n, x, &mx);		sy = d_variance(n, y, &my);
	cov = d_covar(x, y, n, 0L, 0L) * ((double)n/(double)(n-1));
	sd = sqrt((sx+sy-2*cov)/n);
	t = (mx-my)/sd;					df = (n-1);
	p = betai(0.5*df, 0.5, df/(df+t*t));
	res[0] = mx;	res[1] = sqrt(sx);	res[5] = p;
	res[2] = my;	res[3] = sqrt(sy);	res[4] = n;
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(i = 0; i < 6 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if (ra)	memcpy(ra, res, 6 * sizeof(double));
	return p;
}

//f-test
double d_ftest(double *x, double *y, int n1, int n2, char *dest, DataObj *data, double *ra)
{
	int i;
	long r, c;
	double sx, sy, mx, my, d, df1, df2, p;
	double res[6];			// mean1, SD1, n1, mean2, SD2, n2
	AccRange *rD;

	for(i=0, sx = 0.0; i < n1; sx += x[i], i++);
	mx = sx/n1;			for(i=0, sy = 0.0; i < n2; sy += y[i], i++);	
	my = sy/n2;
	for (i = 0, sx = 0.0; i < n1; i++){
		d = x[i] - mx;		sx += (d*d);
		}
	sx /= (n1-1);		
	for (i = 0, sy = 0.0; i < n2; i++) {
		d = y[i] - my;		sy += (d*d);
		}
	sy /= (n2-1);
	if(sx > sy) {
		d = sx/sy;		df1 = n1-1;		df2 = n2-1;
		}
	else {
		d = sy/sx;		df1 = n2-1;		df2 = n1-1;
		}
	p = 2.0 * betai(df2/2.0, df1/2.0, df2/(df2+df1*d));
	if(p > 1.0) p = 2.0-p;
	res[0] = mx;	res[1] = sqrt(sx);	res[2] = n1;
	res[3] = my;	res[4] = sqrt(sy);	res[5] = n2;
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(i = 0; i < 6 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	if (ra)	memcpy(ra, res, 6 * sizeof(double));
	return p;
}
//---------------------------------------------------------------------------
// Simple one way anova
//---------------------------------------------------------------------------
bool do_anova1(int n, int *nv, double **vals, double **res_tab, double *gm, double **means, double **ss)
{
	int i, j, ntot;
	double tmp, *csums, *css, ssa, ssw, sst, mtot, d;

	if(!(csums = (double*)calloc(n+1, sizeof(double)))
		|| !(css = (double*)calloc(n+1, sizeof(double)))) return false;

	for(i = ntot = 0, mtot = 0.0, d = 1.0; i< n; i++){
		for(j = 0, csums[i] = 0.0, tmp = 1.0; j < nv[i]; j++, d+=1.0, tmp +=1.0) {
			mtot += (vals[i][j] - mtot)/d;		
			csums[i] += (vals[i][j] -csums[i])/tmp;
			}
		ntot += nv[i];
		}
	for(i = 0; i < n; i++) {
		for(j = 0, css[i] = 0.0; j < nv[i]; j++) {
			tmp = vals[i][j] - csums[i];	css[i] += (tmp*tmp);
			}
		}
	for(i = 0, ssa = ssw = sst = 0.0;  i < n; i++) {
		tmp =(csums[i] - mtot);		ssa += (tmp*tmp) * ((double)nv[i]);
		ssw += css[i];
		}
	sst = ssa + ssw;
	res_tab[0][0] = n - 1;				res_tab[1][0] = ntot - n;
	res_tab[2][0] = ntot -1;			res_tab[0][1] = ssa;
	res_tab[1][1] = ssw;				res_tab[2][1] = sst;
	res_tab[0][2] = ssa/res_tab[0][0];	res_tab[1][2] = ssw/res_tab[1][0];
	res_tab[0][3] = res_tab[0][2]/res_tab[1][2];
	res_tab[0][4] = f_dist(res_tab[0][3], res_tab[0][0], res_tab[1][0]);
	if(gm) *gm = mtot;
	if(means) *means = csums;			else free(csums);
	if(ss) *ss = css;					else free(css);
	return true;
}

//---------------------------------------------------------------------------
// Bartlett's Test for homogeneity of variances
// RR Sokal & FJ Rohlf: Biometry, 3rd ed., pp. 398 ff.
//---------------------------------------------------------------------------
bool bartlett(int n, int *nc, double *ss, double *chi2)
{
	int i, sdf, df;
	double mss, mlss, *lnss, cf;

	if(!n || !nc || !ss || !chi2) return false;
	if(!(lnss = (double*)malloc(n * sizeof(double))))return false;
	for(i = sdf = 0, mss = mlss = cf = 0.0; i < n; i++) {
		sdf += (df = nc[i]-1);				lnss[i] = log(ss[i]);
		mss += (ss[i] * ((double)df));		mlss += (lnss[i] * ((double)df)); 
		cf += (1.0/((double)df));
		}
	*chi2 = ((double)sdf) * log(mss/((double)sdf)) - mlss;
	cf -= (1.0/((double)sdf));				cf = 1.0 + cf/(3.0 * ((double)(n-1)));
	*chi2 /= cf;
	// P = chi_dist(*chi2, n-1, 0);
	free(lnss);		return true;
}
//---------------------------------------------------------------------------
// Leven's Test for homogeneity of variances
//---------------------------------------------------------------------------
bool levene(int type, int n, int *nv, double *means, double **vals, double *F, double *P)
{
	int i, j;
	bool bRet = false;
	double cm, **res_tab=NULL, **cols=NULL;
	
	if(!n || !nv || !means || !vals) return false;
	//setup matrix for results
	if((res_tab = (double**)calloc(3, sizeof(double*)))
		&& (res_tab[0] = (double*) malloc(5*sizeof(double)))
		&& (res_tab[1] = (double*) malloc(5*sizeof(double)))
		&& (res_tab[2] = (double*) malloc(5*sizeof(double)))
		&& (cols = (double**)calloc(n+1, sizeof(double*)))) bRet = true;
	//allocate mem for data
	for(i = 0; bRet && i<n; i++) {
		if(!(cols[i]=(double*)malloc((nv[i]+1)*sizeof(double)))) bRet = false;
		}
	//data are absolute differences to mean ...
	for(i = 0, cm = 0.0; bRet && i < n; i++) {
		switch(type) {
			case 1:			//use means
				cm = means[i];								break;
			case 2:			//use medians
				d_quartile(nv[i], vals[i], 0L, &cm, 0L);	break;
			}
		for(j = 0; j < nv[i]; j++) {
			cols[i][j] = vals[i][j] > cm ? vals[i][j] - cm : cm - vals[i][j];
			}
		}
	//Levene's test statistic is based on ANOVA of the differences
	if(bRet && (bRet = do_anova1(n, nv, cols, res_tab, 0L, 0L, 0L))){
		if(F) *F = res_tab[0][3];
		if(P) *P = res_tab[0][4];
		}
	//clean up
	if(bRet) {
		for(i = 0; i < n; i++) if(cols[i]) free(cols[i]);
		for(i = 0; i < 3; i++) if(res_tab[i]) free(res_tab[i]);
		free(cols);								free(res_tab);	
		}
	return bRet;
}

//---------------------------------------------------------------------------
// Modules from the R-project
//
//---------------------------------------------------------------------------
#define M_1_SQRT_2PI	0.398942280401432677939946059934	/* 1/sqrt(2pi) */
/*
 *  Copyright (C) 1998       Ross Ihaka
 *  Copyright (C) 2000--2005 The R Development Core Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *  DESCRIPTION
 *    Computes the probability that the maximum of rr studentized
 *    ranges, each based on cc means and with df degrees of freedom
 *    for the standard error, is less than q.
 *    The algorithm is based on that of the reference.
 *
 *  REFERENCE
 *    Copenhaver, Margaret Diponzio & Holland, Burt S.
 *    Multiple comparisons of simple effects in
 *    the two-way analysis of variance with fixed effects.
 *    Journal of Statistical Computation and Simulation,
 *    Vol.30, pp.1-15, 1988.
 */

double wprob(double w, double rr, double cc)
{
/*  wprob() :

	This function calculates probability integral of Hartley's
	form of the range.

	w     = value of range
	rr    = no. of rows or groups
	cc    = no. of columns or treatments
	ir    = error flag = 1 if pr_w probability > 1
	pr_w = returned probability integral from (0, w)

	program will not terminate if ir is raised.

	bb = upper limit of legendre integration
	iMax = maximum acceptable value of integral
	nleg = order of legendre quadrature
	ihalf = int ((nleg + 1) / 2)
	wlar = value of range above which wincr1 intervals are used to
	       calculate second part of integral,
	       else wincr2 intervals are used.
	C1, C2, C3 = values which are used as cutoffs for terminating
           or modifying a calculation.
	xleg = legendre 12-point nodes
	aleg = legendre 12-point coefficients
 */
#define nleg	12
#define ihalf	6

    /* looks like this is suboptimal for double precision.
       (see how C1-C3 are used) <MM> */
    /* const double iMax  = 1.; not used if = 1*/
    const static double C1 = -30.0, C2 = -50.0, C3 = 60.;
    const static double bb = 8.0, wlar = 3.0, wincr1 = 2.0, wincr2 = 3.;
    const static double xleg[ihalf] = {	0.981560634246719250690549090149,
	0.904117256370474856678465866119,	0.769902674194304687036893833213,
	0.587317954286617447296702418941,	0.367831498998180193752691536644,
	0.125233408511468915472441369464};
    const static double aleg[ihalf] = {	0.047175336386511827194615961485,
	0.106939325995318430960254718194,	0.160078328543346226334652529543,
	0.203167426723065921749064455810,	0.233492536538354808760849898925,
	0.249147045813402785000562436043};
    double a, ac, pr_w, b, binc, blb, bub, c, cc1, einsum, elsum,
		pminus, pplus, qexpo, qsqz, rinsum, wi, wincr, xx;
    int j, jj;

    qsqz = w * 0.5;

    // if w >= 16 then the integral lower bound (occurs for c=20)
    // is 0.99999999999995 so return a value of 1
	if (qsqz >= bb)	return 1.0;

	// find (f(w/2) - 1) ^ cc
    // (first term in integral of hartley's form). 
	pr_w = 2.0 * norm_dist(qsqz, 0.0, 1.0) -1.0;
    // if pr_w ^ cc < 2e-22 then set pr_w = 0 
    if (pr_w >= exp(C2 / cc)) pr_w = pow(pr_w, cc);
    else pr_w = 0.0;
    // if w is large then the second component of the
    // integral is small, so fewer intervals are needed.
    if (w > wlar) wincr = wincr1;
    else wincr = wincr2;

    /* find the integral of second term of hartley's form */
    /* for the integral of the range for equal-length */
    /* intervals using legendre quadrature.  limits of */
    /* integration are from (w/2, 8).  two or three */
    /* equal-length intervals are used. */
    /* blb and bub are lower and upper limits of integration. */
    blb = qsqz;			    binc = (bb - qsqz) / wincr;
    bub = blb + binc;	    einsum = 0.0;

    // integrate over each interval
    cc1 = cc - 1.0;
    for (wi = 1; wi <= wincr; wi++) {
		elsum = 0.0;		a = 0.5 * (bub + blb);
		// legendre quadrature with order = nleg
		b = 0.5 * (bub - blb);
		for (jj = 1; jj <= nleg; jj++) {
			if (ihalf < jj) {
				j = (nleg - jj) + 1;		xx = xleg[j-1];
				}
			else {
				j = jj;						xx = -xleg[j-1];
				}
			c = b * xx;					    ac = a + c;
			// if exp(-qexpo/2) < 9e-14, then doesn't contribute to integral
			if ((qexpo = ac * ac) > C3) break;
			pplus = 2.0 * norm_dist(ac, 0.0, 1.0);    pminus= 2.0 * norm_dist(ac, w, 1.0);
			// if rinsum ^ (cc-1) < 9e-14, then doesn't contribute to integral
			rinsum = (pplus * 0.5) - (pminus * 0.5);
			if (rinsum >= exp(C1 / cc1)) {
				rinsum = (aleg[j-1] * exp(-(0.5 * qexpo))) * pow(rinsum, cc1);
				elsum += rinsum;
				}
			}
		elsum *= (((2.0 * b) * cc) * M_1_SQRT_2PI);
		einsum += elsum;		blb = bub;			bub += binc;
		}
	// if pr_w ^ rr < 9e-14, then return 0 */
	pr_w = einsum + pr_w;
	if (pr_w <= exp(C1 / rr))return 0.;
    pr_w = pow(pr_w, rr);
 	return pr_w < 1.0 ? pr_w : 1.0;
}

//double ptukey(double q, double rr, double cc, double df, int lower_tail, int log_p)
double ptukey(double q, double rr, double cc, double df, int, int)
{
/* 	q = value of studentized range
	rr = no. of rows or groups
	cc = no. of columns or treatments
	df = degrees of freedom of error term
	ir[0] = error flag = 1 if wprob probability > 1
	ir[1] = error flag = 1 if qprob probability > 1

	All references in wprob to Abramowitz and Stegun
	are from the following reference:
		Abramowitz, Milton and Stegun, Irene A.
		Handbook of Mathematical Functions.
		New York:  Dover publications, Inc. (1970).
	All constants taken from this text are given to 25 significant digits.

	nlegq = order of legendre quadrature
	ihalfq = int ((nlegq + 1) / 2)
	eps = max. allowable value of integral
	eps1 & eps2 = values below which there is no contribution to integral.

	d.f. <= dhaf:	integral is divided into ulen1 length intervals.  else
	d.f. <= dquar:	integral is divided into ulen2 length intervals.  else
	d.f. <= deigh:	integral is divided into ulen3 length intervals.  else
	d.f. <= dlarg:	integral is divided into ulen4 length intervals.

	d.f. > dlarg:	the range is used to calculate integral.

	xlegq = legendre 16-point nodes
	alegq = legendre 16-point coefficients

	The coefficients and nodes for the legendre quadrature used in
	qprob and wprob were calculated using the algorithms found in:
		Stroud, A. H. and Secrest, D.,	Gaussian Quadrature Formulas.
		Englewood Cliffs, New Jersey:  Prentice-Hall, Inc, 1966.

	All values matched the tables (provided in same reference)
	to 30 significant digits.

	f(x) = .5 + erf(x / sqrt(2)) / 2      for x > 0
	f(x) = erfc( -x / sqrt(2)) / 2	      for x < 0
	where f(x) is standard normal c. d. f.

	if degrees of freedom large, approximate integral with range distribution.
 */
#define nlegq	16
#define ihalfq	8

/*  const double eps = 1.0; not used if = 1 */
    const static double eps1 = -30.0, eps2 = 1.0e-14;
    const static double dhaf  = 100.0, dquar = 800.0, deigh = 5000.0, dlarg = 25000.0;
    const static double ulen1 = 1.0, ulen2 = 0.5, ulen3 = 0.25, ulen4 = 0.125;
    const static double xlegq[ihalfq] = { 0.989400934991649932596154173450,
	0.944575023073232576077988415535, 0.865631202387831743880467897712,
	0.755404408355003033895101194847, 0.617876244402643748446671764049,
	0.458016777657227386342419442984, 0.281603550779258913230460501460,
	0.950125098376374401853193354250e-1};
    const static double alegq[ihalfq] = {0.271524594117540948517805724560e-1,
	0.622535239386478928628438369944e-1, 0.951585116824927848099251076022e-1,
	0.124628971255533872052476282192, 0.149595988816576732081501730547,
	0.169156519395002538189312079030, 0.182603415044923588866763667969,
	0.189450610455068496285396723208};
    double ans, f2, f21, f2lf, ff4, otsum, qsqz, rotsum, t1, twa1, ulen, wprb;
    int i, j, jj;

    if (df > dlarg)	return wprob(q, rr, cc);
    f2 = df * 0.5;								// calculate leading constant
    f2lf = ((f2 * log(df)) - (df * log(2.0))) - gammln(f2);
    f21 = f2 - 1.0;
    // integral is divided into unit, half-unit, quarter-unit, or eighth-unit length intervals 
	//    depending on the value of the degrees of freedom.
    ff4 = df * 0.25;
    if	    (df <= dhaf)	ulen = ulen1;
    else if (df <= dquar)	ulen = ulen2;
    else if (df <= deigh)	ulen = ulen3;
    else					ulen = ulen4;
    f2lf += log(ulen);
    for (i = 1, ans = 0.0; i <= 50; i++) {		// integrate over each subinterval
		otsum = 0.0;
		// legendre quadrature with order = nlegq, nodes (stored in xlegq) are symmetric around zero.
		twa1 = (2 * i - 1) * ulen;
		for (jj = 1; jj <= nlegq; jj++) {
			if (ihalfq < jj) {
				j = jj - ihalfq - 1;
				t1 = (f2lf + (f21 * log(twa1 + (xlegq[j] * ulen)))) - (((xlegq[j] * ulen) + twa1) * ff4);
				} 
			else {
				j = jj - 1;
				t1 = (f2lf + (f21 * log(twa1 - (xlegq[j] * ulen)))) + (((xlegq[j] * ulen) - twa1) * ff4);
				}
			if (t1 >= eps1) {			// if exp(t1) < 9e-14, then doesn't contribute to integral 
				if (ihalfq < jj) qsqz = q * sqrt(((xlegq[j] * ulen) + twa1) * 0.5);
				else qsqz = q * sqrt(((-(xlegq[j] * ulen)) + twa1) * 0.5);
				wprb = wprob(qsqz, rr, cc);		// call wprob to find integral of range portion
				rotsum = (wprb * alegq[j]) * exp(t1);			otsum += rotsum;
				}
			}									// end legendre integral for interval i
		// If integral for interval i < 1e-14, then stop. However, in order to avoid small area 
		//    under left tail, at least  1 / ulen  intervals are calculated.
		if (i * ulen >= 1.0 && otsum <= eps2) break;
		ans += otsum;							//end of interval i 
		}
	return ans > 1.0 ? 1.0 : ans;
 }

 /*
 *  Copyright (C) 1998 	     Ross Ihaka
 *  Copyright (C) 2000--2005 The R Development Core Team
 *  based in part on AS70 (C) 1974 Royal Statistical Society
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *  DESCRIPTION
 *	Computes the quantiles of the maximum of rr studentized
 *	ranges, each based on cc means and with df degrees of freedom
 *	for the standard error, is less than q.
 *	The algorithm is based on that of the reference.
 *
 *  REFERENCE
 *	Copenhaver, Margaret Diponzio & Holland, Burt S., Multiple comparisons of simple
 *	effects in the two-way analysis of variance with fixed effects.
 *	Journal of Statistical Computation and Simulation, Vol.30, pp.1-15, 1988.
 */

/* qinv() :
 *	this function finds percentage point of the studentized range
 *	which is used as initial estimate for the secant method.
 *	function is adapted from portion of algorithm as 70
 *	from applied statistics (1974) ,vol. 23, no. 1
 *	by odeh, r. e. and evans, j. o.
 *	  p = percentage point
 *	  c = no. of columns or treatments
 *	  v = degrees of freedom
 *	  qinv = returned initial estimate
 *	vmax is cutoff above which degrees of freedom
 *	is treated as infinity.
 */

static double qinv(double p, double c, double v)
{
    const static double p0 = 0.322232421088, q0 = 0.993484626060e-01;
    const static double p1 = -1.0, q1 = 0.588581570495;
    const static double p2 = -0.342242088547, q2 = 0.531103462366;
    const static double p3 = -0.204231210125, q3 = 0.103537752850;
    const static double p4 = -0.453642210148e-04, q4 = 0.38560700634e-02;
    const static double c1 = 0.8832, c2 = 0.2368, c3 = 1.214, c4 = 1.208, c5 = 1.4142;
    const static double vmax = 120.0;
    double ps, q, t, yi;

    ps = 0.5 - 0.5 * p;
    yi = sqrt (log (1.0 / (ps * ps)));
    t = yi + (((( yi * p4 + p3) * yi + p2) * yi + p1) * yi + p0)
	   / (((( yi * q4 + q3) * yi + q2) * yi + q1) * yi + q0);
    if (v < vmax) t += (t * t * t + t) / v / 4.0;
    q = c1 - c2 * t;
    if (v < vmax) q += -c3 / v + c4 * t / v;
    return t * (q * log (c - 1.0) + c5);
}

/*
 *  Copenhaver, Margaret Diponzio & Holland, Burt S.
 *  Multiple comparisons of simple effects in
 *  the two-way analysis of variance with fixed effects.
 *  Journal of Statistical Computation and Simulation,
 *  Vol.30, pp.1-15, 1988.
 *
 *  Uses the secant method to find critical values.
 *
 *  p = confidence level (1 - alpha)
 *  rr = no. of rows or groups
 *  cc = no. of columns or treatments
 *  df = degrees of freedom of error term
 *
 */
double qtukey(double p, double rr, double cc, double df, int /*lower_tail*/, int /*log_p*/)
{
    const int maxiter = 50;
    double ans = HUGE_VAL, valx0, valx1, x0 = 0.0, x1 = 0.0;
    int iter;

    // df must be > 1 ; there must be at least two values 
	if(p >= 1.0 || df < 2 || rr < 1 || cc < 2) return HUGE_VAL;
	if(p < 0.0) p = 0.0;
    x0 = qinv(p, cc, df);									// Initial value
    valx0 = ptukey(x0, rr, cc, df, true, false) - p;		// Find prob(value < x0)
    // Find the second iterate and prob(value < x1). If the first iterate has probability value 
    // exceeding p then second iterate is 1 less than first iterate; otherwise it is 1 greater.
	x1 = x0 > 1.0 ? x0 - 1.0 : 0.0;
	x1 = valx0 > 0.0 ? x1 : (x0 + 1.0);
    valx1 = ptukey(x1, rr, cc, df, true, false) - p;
    for(iter=1; iter < maxiter ; iter++) {					// Iterate
		ans = x1 - ((valx1 * (x1 - x0)) / (valx1 - valx0));
		valx0 = valx1;		x0 = x1;
		if (ans < 0.0) {									// New iterate must be >= 0
			ans = 0.0;			valx1 = -p;
			}
		valx1 = ptukey(ans, rr, cc, df, true, false) - p;	//  Find prob(value < new iterate)
		x1 = ans;
		if (fabs(x1 - x0) < _PREC)	return ans;				// Convergence ?
		}
    //The process did not converge in 'maxiter' iterations 
    return ans;
}
//---------------------------------------------------------------------------
// END Modules from the R-project


//---------------------------------------------------------------------------
// Calendar, Date- and Time functions
// The following characters are used as format specifiers in a format string,
//    all other characters are either ignored or copyied to the output
//
//    Y   four digits year               y    two digits year
//    X   month's full name              x    three character month name
//    Z   two digits day of month        z    same as Z but no leading zero
//    V   two digit month number         v    number of month
//    W   single letter month
//    D   full name of day               d    three characters for day name
//    E   two digits weekday             e    one or two digits weekday
//    F   single character day name
//    H   two digits for hours           h    hours with no leading zero
//    M   two digits for minutes         m    minutes with no leading zero
//    S   two digits for seconds         s    seconds with no leading zero
//    T   two digits seconds, two dec.   t    same as T but no leading zero
//    U   full precision seconds

static char *dt_month[] = {(char*)"January", (char*)"February", (char*)"March", (char*)"April", 
	(char*)"May", (char*)"June", (char*)"July", (char*)"August", (char*)"September", (char*)"October",
	(char*)"November", (char*)"December"};

static char *dt_months[] = {(char*)"Jan", (char*)"Feb", (char*)"Mar", (char*)"Apr", (char*)"May", (char*)"Jun",
	(char*)"Jul", (char*)"Aug", (char*)"Sep", (char*)"Oct", (char*)"Nov", (char*)"Dec"};

static int dt_monthl[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

static char *dt_day[] = {(char*)"Sunday", (char*)"Monday", (char*)"Tuesday", (char*)"Wednesday",
	(char*)"Thursday", (char*)"Friday", (char*)"Saturday"};

static char *dt_days[] = {(char*)"Sun", (char*)"Mon", (char*)"Tue", (char*)"Wed", (char*)"Thu",
	(char*)"Fri", (char*)"Sat"};

static bool leapyear(int year) {
	return ((year % 4 == 0 && year % 100 != 0) || year % 400 == 0);
}

int year2aday(int y)
{
	int aday, y1;

	y1 = y - 1900;
	if (y1 < 0) return 0;
	aday = y1 * 365;
	aday += ((y1-1) >> 2 );
	aday -= (y1 / 100);
	aday += ((y/400)-4);
	return aday;
}

static void set_dow(rlp_datetime *dt)
{
	dt->dow = (dt->aday %7)+1;
}

void add_date(rlp_datetime *base, rlp_datetime *inc)
{
	int i, dom;

	if(base) {
		if(base->month < 1) base->month = 1;
		if(inc) {
			base->seconds += inc->seconds;
			if(base->seconds >= 60.0) {
				base->minutes++;		base->seconds -= 60.0;
				}
			base->minutes += inc->minutes;
			if(base->minutes >= 60) {
				base->hours++;			base->minutes -= 60;
				}
			base->hours += inc->hours;
			if(base->hours >= 24) {
				base->dom++;			base->hours -= 24;
				}
			base->year += inc->year;	base->dom += inc->dom;
			base->month += inc->month;
			}
		dom = dt_monthl[base->month-1];
		if(leapyear(base->year) && base->month == 2) dom = 29;
		if(base->dom > dom) {
			base->month++;			base->dom -= dom;
			}
		if(base->month > 12) {
			base->year++;			base->month -= 12;
			}
		base->aday = year2aday(base->year);
		for(i = base->doy = 0; i < (base->month-1); i++) {
			dom = dt_monthl[i];
			if(i == 1 && leapyear(base->year)) dom = 29;
			base->doy += dom;
			}
		base->doy += base->dom;
		base->aday += base->doy;	set_dow(base);
		}
}

static int parse_date (rlp_datetime *dt, char *src, char *fmt)
{
	int i, j, k;
	char tmp_str[10];

	if(!src || !src[0] || !fmt || !fmt[0]) return 0;
	if(*src == '\'') src++;
	for(i = j = 0; fmt[i] && src[j]; i++) {
		switch (fmt[i]) {
		case 'Y':		case 'y':			// year is numeric
			if(j && (src[j] == '-' || src[j] == '/' || src[j] == '.')) j++;
#ifdef USE_WIN_SECURE
			if(sscanf_s(src+j, "%d", &dt->year)) {
#else
			if(sscanf(src+j, "%d", &dt->year)) {
#endif
				if(dt->year < 0) return 0;
				while(isdigit(src[j])) j++;
				if(dt->year<60) dt->year += 2000;
				else if(dt->year <99) dt->year += 1900;
				}
			else return 0;
			break;
		case 'X':		case 'x':			// month can be text
			if(j && (src[j] == '-' || src[j] == '/' || src[j] == '.')) j++;
			tmp_str[0] = toupper(src[j]);
			tmp_str[1] = tolower(src[j+1]);
			tmp_str[2] = tolower(src[j+2]);
			tmp_str[3] = 0;
			for(k = dt->month = 0; k < 12; k++) {
				if(0 == strcmp(tmp_str,dt_months[k])) {
					dt->month = k+1;			break;
					}
				}
			if(dt->month) while(isalpha(src[j])) j++;
			else return 0;
			break;
		case 'V':		case 'v':			//    or numeric
			if(j && (src[j] == '-' || src[j] == '/' || src[j] == '.')) j++;
#ifdef USE_WIN_SECURE
			if(sscanf_s(src+j, "%d", &dt->month)) {
#else
			if(sscanf(src+j, "%d", &dt->month)) {
#endif
				if(dt->month <= 0 || dt->month > 12) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'Z':		case 'z':			// day of month is numeric
			if(j && (src[j] == '-' || src[j] == '/' || src[j] == '.')) j++;
#ifdef USE_WIN_SECURE
			if(sscanf_s(src+j, "%d", &dt->dom)) {
#else
			if(sscanf(src+j, "%d", &dt->dom)) {
#endif
				if(dt->dom <= 0 || dt->dom > 31) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'H':		case 'h':			// hours are numeric
#ifdef USE_WIN_SECURE
			if(sscanf_s(src+j, "%2d", &dt->hours)) {
#else
			if(sscanf(src+j, "%2d", &dt->hours)) {
#endif
				if(dt->hours < 0 || dt->hours > 23) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'M':		case 'm':			// minutes are numeric
			if(j && (src[j] == ' ' || src[j] == ':')) j++;
#ifdef USE_WIN_SECURE
			if(sscanf_s(src+j, "%2d", &dt->minutes)) {
#else
			if(sscanf(src+j, "%2d", &dt->minutes)) {
#endif
				if(dt->minutes < 0 || dt->minutes >= 60) return 0;
				j++;				if(isdigit(src[j])) j++;
				}
			else return 0;
			break;
		case 'S':		case 's':			// seconds are numeric
		case 'T':		case 't':
			if(j && (src[j] == ' ' || src[j] == ':')) j++;
#ifdef USE_WIN_SECURE
			if(sscanf_s(src+j, "%lf", &dt->seconds)) {
#else
			if(sscanf(src+j, "%lf", &dt->seconds)) {
#endif
				if(dt->seconds < 0.0 || dt->seconds >= 60.0) return 0;
				while(isdigit(src[j]) || src[j] == '.') j++;
				}
			else return 0;
			dt->seconds += 1.0e-12;
			break;
		default:
			if(fmt[i] && fmt[i] == src[j]) j++;
			}
		}
	if(dt->year && dt->month && dt->dom) {
		for(dt->doy = 0, i = dt->month-2; i >= 0; i--) {
			if(i == 1) dt->doy += leapyear(dt->year) ? 29 : 28;
			else dt->doy += dt_monthl[i]; 
			}
		dt->doy += dt->dom;
		if(dt->year >= 1900) dt->aday = year2aday(dt->year);
		dt->aday += dt->doy;
		}
	return j;
}

char *date2text(rlp_datetime *dt, char *fmt)
{
	static char res[80];
	int i, pos;
	double secs;

	res[0] = 0;
	if(!fmt || !fmt[0] || !dt) return res;
	set_dow(dt);
	secs = dt->seconds;
	if (secs > 59.4999) secs = 59.4999;
	for(pos = i = 0; fmt[i] && pos < 70; i++) {
#ifdef USE_WIN_SECURE
		switch(fmt[i]) {
		case 'Y':
			if(dt->year) pos+=sprintf_s(res+pos, 80-pos, "%4d", dt->year);
			else pos += sprintf_s(res+pos, 80-pos, "####");
			break;
		case 'y':
			if(dt->year) pos+=sprintf_s(res+pos, 80-pos, "%02d", (dt->year %100));
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'Z':
			if(dt->dom) pos+=sprintf_s(res+pos, 80-pos, "%02d", dt->dom);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'z':
			if(dt->dom) pos+=sprintf_s(res+pos, 80-pos, "%d", dt->dom);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'X':
			if(dt->month >0 && dt->month < 13) pos+=sprintf_s(res+pos, 80-pos, "%s", dt_month[dt->month-1]);
			else pos += sprintf_s(res+pos, 80-pos, "###");
			break;
		case 'x':
			if(dt->month >0 && dt->month < 13) pos+=sprintf_s(res+pos, 80-pos, "%s", dt_months[dt->month-1]);
			else pos += sprintf_s(res+pos, 80-pos, "###");
			break;
		case 'V':
			if(dt->month >0 && dt->month < 13) pos+=sprintf_s(res+pos, 80-pos, "%02d", dt->month);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'v':
			if(dt->month >0 && dt->month < 13) pos+=sprintf_s(res+pos, 80-pos, "%d", dt->month);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'W':
			if(dt->month >0 && dt->month < 13) pos+=sprintf_s(res+pos, 80-pos, "%c", dt_month[dt->month-1][0]);
			else pos += sprintf_s(res+pos, 80-pos, "#");
			break;
		case 'D':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf_s(res+pos, 80-pos, "%s", dt_day[dt->dow-1]);
			else pos += sprintf_s(res+pos, 80-pos, "###");
			break;
		case 'd':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf_s(res+pos, 80-pos, "%s", dt_days[dt->dow-1]);
			else pos += sprintf_s(res+pos, 80-pos, "###");
			break;
		case 'E':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf_s(res+pos, 80-pos, "%02d", dt->dow);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'e':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf_s(res+pos, 80-pos, "%d", dt->dow);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'F':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf_s(res+pos, 80-pos, "%c", dt_day[dt->dow-1][0]);
			else pos += sprintf_s(res+pos, 80-pos, "#");
			break;
		case 'H':
			if(dt->hours >=0 && dt->hours < 24) pos+=sprintf_s(res+pos, 80-pos, "%02d", dt->hours);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'h':
			if(dt->hours >=0 && dt->hours < 24) pos+=sprintf_s(res+pos, 80-pos, "%d", dt->hours);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'M':
			if(dt->minutes >=0 && dt->minutes < 60) pos+=sprintf_s(res+pos, 80-pos, "%02d", dt->minutes);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'm':
			if(dt->minutes >=0 && dt->minutes < 60) pos+=sprintf_s(res+pos, 80-pos, "%d", dt->minutes);
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'S':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf_s(res+pos, 80-pos, "%02d", (int)iround(secs));
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 's':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf_s(res+pos, 80-pos, "%d", (int)iround(secs));
			else pos += sprintf_s(res+pos, 80-pos, "##");
			break;
		case 'T':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf_s(res+pos, 80-pos, "%02.2lf", dt->seconds);
			else pos += sprintf_s(res+pos, 80-pos, "##.##");
			break;
		case 't':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf_s(res+pos, 80-pos, "%.2lf", dt->seconds);
			else pos += sprintf_s(res+pos, 80-pos, "##.##");
			break;
		default:
			pos += sprintf_s(res+pos, 80-pos, "%c", fmt[i]);
			break;
			}
#else
		switch(fmt[i]) {
		case 'Y':
			if(dt->year) pos+=sprintf(res+pos, "%4d", dt->year);
			else pos += sprintf(res+pos, "####");
			break;
		case 'y':
			if(dt->year) pos+=sprintf(res+pos, "%02d", (dt->year %100));
			else pos += sprintf(res+pos, "##");
			break;
		case 'Z':
			if(dt->dom) pos+=sprintf(res+pos, "%02d", dt->dom);
			else pos += sprintf(res+pos, "##");
			break;
		case 'z':
			if(dt->dom) pos+=sprintf(res+pos, "%d", dt->dom);
			else pos += sprintf(res+pos, "##");
			break;
		case 'X':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%s", dt_month[dt->month-1]);
			else pos += sprintf(res+pos, "###");
			break;
		case 'x':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%s", dt_months[dt->month-1]);
			else pos += sprintf(res+pos, "###");
			break;
		case 'V':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%02d", dt->month);
			else pos += sprintf(res+pos, "##");
			break;
		case 'v':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%d", dt->month);
			else pos += sprintf(res+pos, "##");
			break;
		case 'W':
			if(dt->month >0 && dt->month < 13) pos+=sprintf(res+pos, "%c", dt_month[dt->month-1][0]);
			else pos += sprintf(res+pos, "#");
			break;
		case 'D':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%s", dt_day[dt->dow-1]);
			else pos += sprintf(res+pos, "###");
			break;
		case 'd':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%s", dt_days[dt->dow-1]);
			else pos += sprintf(res+pos, "###");
			break;
		case 'E':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%02d", dt->dow);
			else pos += sprintf(res+pos, "##");
			break;
		case 'e':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%d", dt->dow);
			else pos += sprintf(res+pos, "##");
			break;
		case 'F':
			if(dt->dow >0 && dt->dow < 8) pos+=sprintf(res+pos, "%c", dt_day[dt->dow-1][0]);
			else pos += sprintf(res+pos, "#");
			break;
		case 'H':
			if(dt->hours >=0 && dt->hours < 24) pos+=sprintf(res+pos, "%02d", dt->hours);
			else pos += sprintf(res+pos, "##");
			break;
		case 'h':
			if(dt->hours >=0 && dt->hours < 24) pos+=sprintf(res+pos, "%d", dt->hours);
			else pos += sprintf(res+pos, "##");
			break;
		case 'M':
			if(dt->minutes >=0 && dt->minutes < 60) pos+=sprintf(res+pos, "%02d", dt->minutes);
			else pos += sprintf(res+pos, "##");
			break;
		case 'm':
			if(dt->minutes >=0 && dt->minutes < 60) pos+=sprintf(res+pos, "%d", dt->minutes);
			else pos += sprintf(res+pos, "##");
			break;
		case 'S':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%02d", (int)iround(secs));
			else pos += sprintf(res+pos, "##");
			break;
		case 's':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%d", (int)iround(secs));
			else pos += sprintf(res+pos, "##");
			break;
		case 'T':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%02.2lf", dt->seconds);
			else pos += sprintf(res+pos, "##.##");
			break;
		case 't':
			if(dt->seconds >=0.0 && dt->seconds < 60.0) pos+=sprintf(res+pos, "%.2lf", dt->seconds);
			else pos += sprintf(res+pos, "##.##");
			break;
		default:
			pos += sprintf(res+pos, "%c", fmt[i]);
			break;
			}
#endif
	}
	res[pos] = 0;
	return res;
}

double date2value(rlp_datetime *dt)
{
	double res;

	if(!dt) return 0.0;

	res = dt->seconds/60.0 + (double)dt->minutes;
	res = res/60.0 + (double)dt->hours;
	res = res/24.0 + (double)dt->aday;
	return res;
}

void parse_datevalue(rlp_datetime *dt, double dv)
{
	int i, j, d;

	if(!dt || dv < 0.0) return;
	if(dv > 1.0) {
		dt->aday = (int)floor(dv);
		dt->year = (int)(dv/365.2425);
		d = (int)floor(dv);
		do {
			dt->doy = d - 365*dt->year;
			dt->doy -= ((dt->year-1)>>2);
			dt->doy += ((dt->year)/100);
			dt->doy -= ((dt->year+300)/400);
			if(dt->doy < 1) dt->year--;
			}while(dt->doy < 1);
		dt->year += 1900;
		for(i = dt->month = 0, d = dt->doy; i < 12 && d > 0; i++) {
			if(i == 1 && d > (j = (leapyear(dt->year)) ? 29 : 28)) d -= j;
			else if(i != 1 && d > dt_monthl[i]) d -= dt_monthl[i];
			else break;
			}
		dt->month = i+1;				dt->dom = d;
		}
	else {
		dt->aday = dt->year = dt->doy = dt->dom = dt->month = 0;
		}
	dv -= floor(dv);				dv *= 24.0;
	dt->hours = (int)floor(dv);		dv -= floor(dv);
	dv *= 60.0;						dt->minutes = (int)floor(dv); 
	dv -= floor(dv);				dt->seconds = dv *60.0 + 1.0e-12; 
	if(dt->seconds > 59.9999) {
		dt->seconds = 0.0;			dt->minutes++;
		if(dt->minutes == 60) {
			dt->hours++;			dt->minutes = 0;
			}
		}
}

static char *dt_popfmt[] = {(char*)"Z.V.Y H:M:S", (char*)"Z/V/Y H:M:S", (char*)"Z-V-Y H:M:S",
	(char*)"Z.X.Y H:M:S", (char*)"Y.V.Z H:M:S", (char*)"Y-X-Z H:M:S", (char*)"H:M:S", 0L};

bool date_value(char *desc, char *fmt, double *value)
{
	int i;
	rlp_datetime dt;

	dt.year = dt.aday = dt.doy = dt.month = dt.dom = dt.dow = dt.hours = dt.minutes = 0;
	dt.seconds = 0.0;
	if(!value || !desc || !desc[0]) return false;
	if(fmt && fmt[0]) {
		if(parse_date(&dt, desc, fmt)) {
			*value = date2value(&dt);	return true;
			}
		}
	else {
		if(parse_date(&dt, desc, defs.fmt_datetime)) {
			*value = date2value(&dt);	return true;
			}
		}
	for(i=0; dt_popfmt[i]; i++) {
		if(parse_date(&dt, desc, dt_popfmt[i])) {
			*value = date2value(&dt);	return true;
			}
		}
	return false;
}

char *value_date(double dv, char *fmt)
{
	rlp_datetime dt;

	parse_datevalue(&dt, dv);
	return date2text(&dt, fmt ? fmt : defs.fmt_date);
}

double now_today()
{
	double res = 0.0;
	time_t ti = time(0L);
#ifdef USE_WIN_SECURE
	char dtbuff[80];

	ctime_s(dtbuff, 80, &ti);
	date_value(dtbuff+4, (char*)"x z H:M:S Y", &res);
#else
	date_value(ctime(&ti)+4, (char*)"x z H:M:S Y", &res);
#endif
	return res;
}

void split_date(double dv, int *y, int *mo, int *dom, int *dow, int *doy, int *h, int *m, double *s)
{
	rlp_datetime dt;

	parse_datevalue(&dt, dv);
	set_dow(&dt);
	if(y) *y = dt.year;
	if(mo) *mo = dt.month;
	if(dom) *dom = dt.dom;
	if(dow) *dow = dt.dow;
	if(doy) *doy = dt.doy;	
	if(h) *h = dt.hours;
	if(m) *m = dt.minutes;	
	if(s) *s = dt.seconds;
}

//Calculate the intersection point between two lines
//   The lines are defined by {(x1,y1)(x2,y2)} and {(x3,y3)(x4,y4)}.
//   Coordinates of the intersection point are returned in x and y.
//   
// Ref: Mukesh Prasad (1991) Intersection of Line Segments.
//      In: Graphics Gems II: 7-9 Ed.: J. Arvo, Academic Press, Inc.
#define SAME_SIGNS(a, b) (((long)((unsigned long) a ^ (unsigned long) b)) >=0)
bool line_intersect(double x1, double y1, double x2, double y2, double x3, double y3,
	double x4, double y4, double *x, double *y, bool bBrack)
{
	double a1, a2, b1, b2, c1, c2;				//coefficients of line equations
	double r1, r2, r3, r4;						//'sign' values
	double denom/*, offset, num*/;				//intermediate values

	a1 = y2 - y1;						b1 = x1 - x2;
	c1 = x2 * y1 - x1 * y2;
	r3 = a1 * x3 + b1 * y3 + c1;
	r4 = a1 * x4 + b1 * y4 + c1;
	if (bBrack && r3 != 0 && r4 != 0 && SAME_SIGNS(r3, r4)) return false;
	// check here if lines intersect at all!
	a2 = y4 - y3;						b2 = x3 - x4;
	c2 = x4 * y3 - x3 * y4;
	r1 = a2 * x1 + b2 * y1 + c2;
	r2 = a2 * x2 + b2 * y2 + c2;
	if (bBrack && r1 != 0 && r2 != 0 && SAME_SIGNS(r1, r2)) return false;
	// check here again if lines intersect at all
	denom = a1 * b2 - a2 * b1;			//if denom == 0 :lines are collinear
	if (fabs(denom) < 1.0e-16) return false;
	*x = (b1 * c2 - b2 * c1) / denom;	*y = (a2 * c1 - a1 * c2) / denom;
	return true;
}

//---------------------------------------------------------------------------
// Use the Delauney triangulation to create a 3D mesh of dispersed data
//
fPOINT3D super_dim[4];

//test if point has reference to supertriangles during triangulation
bool bToBaseRect(double x, double y, double z)
{
	int i, j;

	for (i = j = 0; i < 4; i++) {
		if (x == super_dim[i].fx && y == super_dim[i].fy && z == super_dim[i].fz) {
			j++;
			if(j > 1) return true;
			}
		}
	return false;
}

//triangulate spreadsheet data
Triangle* Triangulate1(char *xr, char *yr, char *zr, DataObj *data)
{
	AccRange *rX, *rY, *rZ;
	long i, j, n;
	long rx, cx, ry, cy, rz, cz;
	double zMin = HUGE_VAL, zMax = -HUGE_VAL, dx, dy, dz;
	fPOINT3D *da;
	fRECT lim;
	Triangle *trl, *trn, *trnn,  *trc;
	Triangulate *tria=NULL;

	lim.Xmax = -HUGE_VAL;		lim.Ymax = -HUGE_VAL;
	lim.Xmin = HUGE_VAL;		lim.Ymin = HUGE_VAL;
	rX = 0L; rY = 0L; rZ = 0L; trl = 0L; trn = 0L;
	if((rX = new AccRange(xr)) && (rY = new AccRange(yr)) && (rZ = new AccRange(zr))
		&& rX->GetFirst(&cx, &rx) && rY->GetFirst(&cy, &ry) && rZ->GetFirst(&cz, &rz)
		&& (n = rX->CountItems()) && (da = (fPOINT3D*)malloc(n * sizeof(fPOINT3D)))
		&& (trl = new Triangle()) && (trn = new Triangle())) {
		//get minima and maxima
		for(i = j = 0; i < n; i++) {
			if(rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && rZ->GetNext(&cz, &rz)) {
				if(data->GetValue(rx, cx, &da[j].fx) && data->GetValue(ry, cy, &da[j].fy) &&
				data->GetValue(rz, cz, &da[j].fz))	j++;
				}
			}
		if(j<3) {
			free(da); delete rX;	delete rY;	delete rZ;	return trl;
			}
		for(i = 0; i < j; i++) {
			if(i) {
				if(da[i].fx < lim.Xmin) lim.Xmin = da[i].fx;
				if(da[i].fx > lim.Xmax) lim.Xmax = da[i].fx;
				if(da[i].fy < lim.Ymin) lim.Ymin = da[i].fy;
				if(da[i].fy > lim.Ymax) lim.Ymax = da[i].fy;
				if(da[i].fz < zMin) zMin = da[i].fz;
				if (da[i].fz > zMax) zMax = da[i].fz;
				}
			else {
				lim.Xmax = lim.Xmin = da[i].fx;		lim.Ymax = lim.Ymin = da[i].fy;		zMin = zMax = da[i].fz;
				}
			}
		//setup two super triangles
		dx = (lim.Xmax - lim.Xmin) / 100.0; 		dy = (lim.Ymax - lim.Ymin) / 100.0;
		dz = (zMax - zMin) / 100.0;
		super_dim[0].fx = super_dim[2].fx = lim.Xmin - dx;		super_dim[0].fy = lim.Ymin - dy;
		super_dim[1].fx = lim.Xmax + dx;		super_dim[1].fy = super_dim[2].fy = lim.Ymax + dy;
		super_dim[3].fx = lim.Xmax + dx;		super_dim[3].fy = lim.Ymin - dy;
		super_dim[0].fz = super_dim[1].fz = super_dim[2].fz = super_dim[3].fz = zMin-dz;
		trl->pt[0].fz = trl->pt[1].fz = trl->pt[2].fz = super_dim[0].fz;
		trn->pt[0].fz = trn->pt[1].fz = trn->pt[2].fz = super_dim[0].fz;
		trl->pt[0].fx = trn->pt[0].fx = trl->pt[2].fx = super_dim[0].fx;
		trl->pt[0].fy = trn->pt[0].fy = trn->pt[1].fy = super_dim[0].fy;
		trl->pt[1].fx = trn->pt[2].fx = trn->pt[1].fx = super_dim[1].fx;
		trl->pt[1].fy = trn->pt[2].fy = trl->pt[2].fy = super_dim[1].fy;
		trl->flags = trn->flags = 0x01;							//these triangles reference the super triangles
		trl->SetRect();			trn->SetRect();
		trl->next = trn;		trn->next = 0L;
		//do triangulation
		tria = new Triangulate(trl);
		for (i = 0; i < n; i++) 			tria->AddVertex(&da[i]);
		free(da);
		}
	//remove references to superrectangle 
	trl = tria->trl;
	if (true) {
		trn = trl->next;
		while (trn && trl && (trl->flags & 0x01)) {
			trn = trl->next;		delete trl;				trl = trn->next;
			}
		trc = trl;
		if(trc)trn = trc->next;
		if (trc && trl && trn) do {
			trnn = trn->next;
			if (trn->flags & 0x01) {
				trc->next = trnn;		delete trn;			trn = trnn;
				}
			else {
				trc = trn;			trn = trc->next;
				}
			} while (trn);
		}
	if (rX) delete rX;
	if (rY) delete rY;	
	if (rZ) delete rZ;
	delete tria;		return trl;
}

//Compute the plane equation of an arbitrary 3D polygon
//Ref.: Tamperi F. (1992) Newell's Method for Computing the Plane Equation of a Polygon.
//In: Graphic Gems III, pp. 231-232 and 517-518; Ed.: D. Kirk
//Academic Press, Inc. 1992
//void PlaneEquation(fPOINT3D *verts, int nverts, double result[])
void PlaneEquation(fPOINT3D *verts, int nverts, double *)
{
	int i;
	fPOINT3D normal, refpt, *u, *v;
	double len;
//	double result[4];		//holds the equation

	normal.fx = normal.fy = normal.fz = 0.0;
	refpt.fx = refpt.fy = refpt.fz = 0.0;
	for (i = 0; i < nverts; i++) {
		u = &verts[i];		v = &verts[(i + 1) % nverts];
		normal.fx += (u->fy - v->fy) * (u->fz + v->fz);
		normal.fy += (u->fz - v->fz) * (u->fx + v->fx);
		normal.fz += (u->fx - v->fx) * (u->fy + v->fy);
		refpt.fx += u->fx;		refpt.fy += u->fy;		refpt.fz += u->fz;
		}
	len = sqrt(normal.fx * normal.fx + normal.fy * normal.fy + normal.fz * normal.fz);
//	result[0] = normal.fx / len;	result[1] = normal.fy / len;	result[2] = normal.fz / len;
	len *= nverts;
//	result[3] = -(refpt.fx * normal.fx + refpt.fy * normal.fy + refpt.fz * normal.fz);
#ifdef _DEBUG
#ifdef _WINDOWS
//debug show results
	char tmp_txt[100];
	OutputDebugStringA("\nPlane Equation\n   Plane Data:");
	for (i = 0; i < nverts; i++) {
		sprintf_s(tmp_txt, 90, "  (%d) {%lf, %lf, %lf}\n", i + 1, verts[i].fx, verts[i].fy, verts[i].fz);
		OutputDebugStringA(tmp_txt);
		}
//	OutputDebugString("The parameters of the equation are:\n");
//	sprintf_s(tmp_txt, 90, "   %g  %g  %g  %g\n", result[0], result[1], result[2], result[3]);
#endif
#endif
}