//PropertyDlg.cpp, Copyright (c) 2000-2025 R.Lackner
//Property dialogs for graphic objects
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "TheDialog.h"
#include "rlp_strings.h"

extern tag_Units Units[];
extern char TmpTxt[];
extern TextDEF DlgText;
extern def_vars defs;
extern int dlgtxtheight;
extern Axis **CurrAxes;
extern long NumCurrAxes;
extern UndoObj Undo;
extern int AxisTempl3D;
extern void *prog_bar_ptr;


int ODtickstyle;

//prototypes: WinSpec.cpp ...
void *CreateDlgWnd(char *title, int x, int y, int width, int height, tag_DlgObj *d, DWORD flags);

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Symbol properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *SymDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,145,10,60,12\n"
	".,.,,,PUSHBUTTON,-28,145,25,60,12\n"
	".,.,,,PUSHBUTTON,-2,145,40,60,12\n"
	".,50,5,CHECKED | ISPARENT,GROUP,,,,,\n"
	".,+,100,ISPARENT | CHECKED,SHEET,2,5,10,130,113\n"
	".,.,200,TOUCHEXIT | ISPARENT,SHEET,3,5,10,130,113\n"
	".,,300,ISPARENT,SHEET,4,5,10,130,113\n"
	"50,,,TOUCHEXIT,SYMBUTT,0,155,75,40,40\n"
	"100,+,,,RTEXT,5,5,25,45,8\n"
	".,.,,TOUCHEXIT,INCDECVAL1,9,55,25,33,10\n"
	".,.,,,LTEXT,-3,89,25,20,8\n"
	".,.,,,RTEXT,-29,5,37,45,8\n"
	".,.,,TOUCHEXIT,INCDECVAL1,10,55,37,33,10\n"
	".,.,,,LTEXT,-3,89,37,20,8\n"
	".,.,,,RTEXT,-30,5,49,45,8\n"
	".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,11,55,49,25,10\n"
	".,.,,,RTEXT,12,5,61,45,8\n"
	".,401,,TOUCHEXIT | OWNDIALOG,COLBUTT,13,55,61,25,10\n"
	"200,204,201,CHECKED | ISPARENT,GROUPBOX,14,12,28,50,39\n"
	".,+,,HICOL | TOUCHEXIT,RADIO1,15,15,33,45,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,16,15,43,45,8\n"
	".,,,HICOL | TOUCHEXIT,RADIO1,17,15,53,43,8\n"
	".,250,205,CHECKED | ISPARENT,GROUPBOX,18,72,28,57,39\n"
	"205,+,,HICOL | TOUCHEXIT,CHECKBOX,19,75,33,25,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,20,75,43,25,8\n"
	".,,,HICOL | TOUCHEXIT,CHECKBOX,21,75,53,25,8\n"
	"250,+,,HICOL | TOUCHEXIT | CHECKED,RADIO1,22,10,75,45,8\n"
	".,.,,,EDTEXT,23,60,75,68,10\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,24,10,92,60,8\n"
	".,,,,RANGEINPUT,25,20,102,100,10\n"
	"300,+,,,RTEXT,-12,5,30,45,8\n"
	".,.,,,EDVAL1,6,55,30,65,10\n"
	".,.,,,RTEXT,-13,5,50,45,8\n"
	".,.,,,EDVAL1,7,55,50,65,10\n"
	".,.,,,RTEXT,-14,5,70,45,8\n"
	".,,,LASTOBJ,EDVAL1,8,65,70,45,10";

bool
Symbol::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_SIZECOL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TEXT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	double fx = 0.0, fy = 0.0, fz = 0.0;
	Symbol *PrevSym = 0L;
	unsigned char text1[40], text2[100];
	void *dyndata[] = {(void*)SDLG_SYM_APPLY, (void*)tab1,
		(void*)tab2, (void*)tab3, (void*)SDLG_SYM_SIZE, (void*)&fx, (void*)&fy, (void*) &fz, 
		(void*)&size, (void*)&SymLine.width, (void *)&SymLine.color, (void*)SDLG_SYM_FILLC,
		(void *)&SymFill.color, (void*)SDLG_SYM_FONT, (void*)SDLG_SYM_HELV, (void*)SDLG_SYM_TIMES,
		(void*)SDLG_SYM_COUR, (void*)SDLG_SYM_STXT, (void*)SDLG_SYM_BOLD, (void*)SDLG_SYM_ITALIC,
		(void*)SDLG_SYM_UNDERL, (void*)SDLG_SYM_FIXED, (void*)text1,
		(void*)SDLG_SYM_FROMSS, (void*)text2};
	DlgInfo *SymDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int k, ix = 0, iy = 0, res, n_syms;
	long width, height;
	long tmpType;
	DWORD tmpCol, undo_flags = 0L;
	double tmpVal, o_size, n_size, o_lwidth, n_lwidth;
	TextDEF textdef;
	bool bContinue = false;
	anyOutput *cdisp = Undo.Cdisp();
	lfPOINT o_pos, n_pos;
	fPOINT3D o_pos3d, n_pos3d;
	static const int syms[] = {SYM_CIRCLE, SYM_CIRCLEF, SYM_CIRCLEC, SYM_RECT, SYM_RECTF, SYM_RECTC, 
		SYM_TRIAU, SYM_TRIAUF, SYM_TRIAUC, SYM_TRIAUL, SYM_TRIAUR, SYM_TRIAD, SYM_TRIADF, SYM_TRIADC,
		SYM_TRIADL, SYM_TRIADR, SYM_DIAMOND, SYM_DIAMONDF, SYM_DIAMONDC, SYM_5GON, SYM_5GONF, SYM_5GONC,
		SYM_4STAR, SYM_4STARF, SYM_5STAR, SYM_5STARF, SYM_6STAR, SYM_6STARF, SYM_1QUAD, SYM_2QUAD,
		SYM_3QUAD, SYM_PLUS, SYM_CROSS, SYM_STAR, SYM_HLINE, SYM_VLINE, SYM_TEXT};

	fx = GetSize(SIZE_XPOS);  fy = GetSize(SIZE_YPOS);	fz = GetSize(SIZE_ZPOS);
	if(!(SymDlg = CompileDialog(SymDlg_DlgTmpl, dyndata)))return false;
	if(!Command(CMD_GETTEXT, (void*)text1, 0L)) rlp_strcpy(text1, 40, (char*)SDLG_SYM_DEFTXT);
#ifdef USE_WIN_SECURE
	if(parent && data && data->GetSize(&width, &height)) sprintf_s((char*)text2, 100, "b1:b%ld", height);
#else
	if(parent && data && data->GetSize(&width, &height)) sprintf((char*)text2, "b1:b%ld", height);
#endif
	else rlp_strcpy(text2, 100, (char*)"(not available)");
	n_syms = sizeof(syms)/sizeof(int);
	for(k = 1; !(SymDlg[k-1].flags & LASTOBJ); k++);
	if(!parent) n_syms--;
	if(!(SymDlg = (DlgInfo *)realloc(SymDlg, (k+n_syms)*sizeof(DlgInfo)))) return false;
	SymDlg[k-1].flags &= (~LASTOBJ);
	for(i = 1, iy = 66; i <= n_syms; i++, ix += 10) {
		if((i%11) == 1) {
			iy += 10;		ix = 15;
			}
		SymDlg[k].id = 400 + i;					SymDlg[k].next = 400 + i + 1;
		SymDlg[k].first = 0;					SymDlg[k].flags = TOUCHEXIT;
		SymDlg[k].type = SYMRADIO;				SymDlg[k].ptype = (void*)&syms[i-1];
		SymDlg[k].x = ix;						SymDlg[k].y = iy;
		if(type == syms[i-1]) SymDlg[k].flags |= CHECKED;
		SymDlg[k].w = SymDlg[k].h = 10;			k++;
		}
	SymDlg[k-1].flags |= LASTOBJ;				if(parent) SymDlg[k-1].w = 30;
	if (parent && parent->Id == GO_MYREGR && !data) {
		SymDlg[5].flags |= HIDDEN;				SymDlg[6].flags |= HIDDEN;
		SymDlg[1].flags |= HIDDEN;				SymDlg[2].y = 25;
		}
	else if (parent && parent->Id != GO_NORMQUANT) {
		SymDlg[0].ptype = dyndata[0];
		}
	else {
		SymDlg[5].flags |= HIDDEN;				SymDlg[6].flags |= HIDDEN;
		SymDlg[1].flags |= HIDDEN;				SymDlg[2].y = 25;
		SymDlg[0].w = SymDlg[2].w = 45;			SymDlg[7].x = 145;
		}
	if(Id == GO_SYMBOL) {						//Hide z-value
		SymDlg[34].flags |= HIDDEN;
		SymDlg[35].flags |= HIDDEN;
		}
	if((PrevSym = new Symbol(0L, data, 0.0f, 0.0f, type))){
		PrevSym->SetColor(COL_SYM_LINE, SymLine.color);
		PrevSym->SetColor(COL_SYM_FILL, SymFill.color);
		PrevSym->SetSize(SIZE_SYMBOL, size);
		PrevSym->SetSize(SIZE_SYM_LINE, SymLine.width);
		PrevSym->Command(CMD_SETTEXT, (void*)text1, 0L);
		PrevSym->Command(CMD_GETTEXTDEF, &textdef, 0L);
		PrevSym->Command(CMD_SET_DATAOBJ, (void*)data, 0L);
		if(Command(CMD_GETTEXTDEF, &textdef, 0L))
			PrevSym->Command(CMD_SETTEXTDEF, &textdef, 0L);
		PrevSym->idx = idx;
		SymDlg[7].ptype = (void*)&PrevSym;
		}
	if(PrevSym && (Dlg = new DlgRoot(SymDlg, data))) {
		Dlg->TextFont(201, FONT_HELVETICA);
		Dlg->TextFont(202, FONT_TIMES);
		Dlg->TextFont(203, FONT_COURIER);
		Dlg->TextStyle(205, TXS_BOLD);
		Dlg->TextStyle(206, TXS_ITALIC);
		Dlg->TextStyle(207, TXS_UNDERLINE);
		switch(textdef.Font) {
		case FONT_TIMES:	Dlg->SetCheck(202, 0L, true);	break;
		case FONT_COURIER:	Dlg->SetCheck(203, 0L, true);	break;
		default:			Dlg->SetCheck(201, 0L, true);	break;
			}
		if(textdef.Style & TXS_BOLD) Dlg->SetCheck(205, 0L, true);
		if(textdef.Style & TXS_ITALIC) Dlg->SetCheck(206, 0L, true);
		if(textdef.Style & TXS_UNDERLINE) Dlg->SetCheck(207, 0L, true);
		if (parent && parent->Id == GO_CONTOUR)Dlg->ShowItem(7, false);
		}
	else return false;
	if(parent && parent->name) {
		width = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_SYM_HD1);
		rlp_strcpy(TmpTxt+width, TMP_TXT_SIZE-width, parent->name);
		width =10;
		}
	else {
		rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_SYM_HD2);
		}
	if(!(hDlg = CreateDlgWnd(TmpTxt, 50, 50, parent ? 450 : 410, 307, Dlg, 0x4L)))return false;
	tmpCol = 0x00c0c0c0L;	o_size = size;	o_lwidth = SymLine.width;
	Dlg->GetValue(101, &o_size);		Dlg->GetValue(104, &o_lwidth);
	n_size = o_size;					n_lwidth = o_lwidth;
	o_pos.fx = n_pos3d.fx = fx;			o_pos.fy = n_pos3d.fy = fy;
	o_pos3d.fz = n_pos3d.fz = fz;
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res){
		case 0:
			if(bContinue) res = -1;
			break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);
			if(parent && PrevSym->type == SYM_TEXT && Dlg->GetCheck(250)){
				Dlg->GetText(251, text1, 40);
				if(PrevSym->Command(CMD_GETTEXT, (void *)text2, 0L) && strcmp((char*)text1, (char*)text2)) {
					PrevSym->Command(CMD_SETTEXT, (void *)text1, 0L);
					Dlg->DoPlot(NULL);
					res = -1;
					}
				}
			else if(parent && PrevSym->type == SYM_TEXT && Dlg->GetCheck(252)) {
				if(Dlg->GetCheck(252) && Dlg->GetText(253, text2, 100))	
					PrevSym->Command(CMD_RANGETEXT, &text2, 0L);
				}
			Dlg->GetValue(101, &n_size);		Dlg->GetValue(104, &n_lwidth);
		case 3:
			break;
		case 6:									//the text sheets
			if(parent)Dlg->SetCheck(400+n_syms, 0L, true);
			if(PrevSym->type != SYM_TEXT) {
				PrevSym->type = SYM_TEXT;		Dlg->DoPlot(0L);
				}
			res = -1;		bContinue = true;
			break;
		case 201:	case 202:	case 203:				//fonts and styles
		case 205:	case 206:	case 207:
			if(PrevSym->Command(CMD_GETTEXTDEF, &textdef, 0L)) {
				if(Dlg->GetCheck(202)) textdef.Font = FONT_TIMES;
				else if(Dlg->GetCheck(203)) textdef.Font = FONT_COURIER;
				else textdef.Font = FONT_HELVETICA;
				textdef.Style = TXS_NORMAL;
				if(Dlg->GetCheck(205)) textdef.Style |= TXS_BOLD;
				if(Dlg->GetCheck(206)) textdef.Style |= TXS_ITALIC;
				if(Dlg->GetCheck(207)) textdef.Style |= TXS_UNDERLINE;
				PrevSym->Command(CMD_SETTEXTDEF, &textdef, 0L);
				Dlg->DoPlot(0L);
				}
			res = -1;
			break;
		default:										//symbol selection ?
			if(res > 400 && res <= (400+n_syms)) tmpType = syms[res-401];
			else break;
			PrevSym->type = tmpType;
		case 107:										//line color button
			if(res == 107 && Dlg->GetColor(107, &tmpCol))
				PrevSym->SetColor(COL_SYM_LINE, tmpCol);
		case 109:										//fill color button
			if(res == 109 && Dlg->GetColor(109, &tmpCol)) 
				PrevSym->SetColor(COL_SYM_FILL, tmpCol);
		case 101:										//symbol size changed
		case 104:										//line width changed
		case 50:										//preview button
			if(Dlg->GetValue(101, &tmpVal))	PrevSym->SetSize(SIZE_SYMBOL, tmpVal);
			if(Dlg->GetValue(104, &tmpVal))	PrevSym->SetSize(SIZE_SYM_LINE, tmpVal);
			if(PrevSym->type == SYM_TEXT) {
				if(Dlg->GetCheck(250) && Dlg->GetText(251,text1,40))PrevSym->Command(CMD_SETTEXT, text1, 0L);
				else if(Dlg->GetCheck(252) && Dlg->GetText(253, text2,100))	
					PrevSym->Command(CMD_RANGETEXT, text2, 0L);
				}
			Dlg->DoPlot(0L);
			res = -1;
			break;
		case 252:										//use spreadsheet text
			if(!data) Dlg->SetCheck(250, 0L, true);
		case 250:										//use fixed text
			if(Dlg->GetCheck(250) && Dlg->GetText(251,text1,40))PrevSym->Command(CMD_SETTEXT, text1, 0L);
			else if(Dlg->GetCheck(252) && Dlg->GetText(253, text2, 100))	
				PrevSym->Command(CMD_RANGETEXT, text2, 0L);
			Dlg->DoPlot(NULL);
			res = -1;
			break;
			}
		}while (res < 0);
	switch (res) {
	case 1:								//accept values for symbol
		Undo.Separator();
		undo_flags = CheckNewFloat(&size, o_size, n_size, parent, undo_flags);
		undo_flags = CheckNewFloat(&SymLine.width, o_lwidth, n_lwidth, parent, undo_flags);
		if(Dlg->GetValue(301, &tmpVal)) n_pos.fx = n_pos3d.fx = tmpVal;
		if(Dlg->GetValue(303, &tmpVal)) n_pos.fy = n_pos3d.fy = tmpVal;
		if(Dlg->GetValue(305, &tmpVal)) n_pos3d.fz = tmpVal;
		undo_flags = CheckNewLFPoint(&fPos, &o_pos, &n_pos, parent, undo_flags);
		if(Id == GO_SYMTERN) {
			undo_flags = CheckNewLFPoint3D(&((SymTernary*)this)->fPos3D, &o_pos3d, &n_pos3d, parent, undo_flags);
			}
		if(Dlg->GetColor(107, &tmpCol)) undo_flags = 
			CheckNewDword(&SymLine.color, SymLine.color, tmpCol, parent, undo_flags);
		if(Dlg->GetColor(109, &tmpCol)) undo_flags = 
			CheckNewDword(&SymFill.color, SymFill.color, tmpCol, parent, undo_flags);
		undo_flags = CheckNewLong(&type, type, PrevSym->type, parent, undo_flags);
		if(type == SYM_TEXT && PrevSym->Command(CMD_GETTEXTDEF, &textdef, 0L)){
			if(SymTxt && cmpTextDEF(SymTxt, &textdef)){
				Undo.TextDef(parent, SymTxt, undo_flags);	undo_flags |= UNDO_CONTINUE;
				}
			if(PrevSym->Command(CMD_GETTEXT, text1, 0L)) Command(CMD_SETTEXT, text1, 0L);
			Command(CMD_SETTEXTDEF, &textdef, 0L);
			}
		break;
	case 2:											//accept values for all symbols of plot
		parent->Command(CMD_SAVE_SYMBOLS, 0L, 0L);
		undo_flags |= UNDO_CONTINUE;
		parent->SetSize(SIZE_SYMBOL, n_size);
		parent->SetSize(SIZE_SYM_LINE, n_lwidth);
		if(Dlg->GetColor(107, &tmpCol))	parent->SetColor(COL_SYM_LINE, tmpCol);
		if(Dlg->GetColor(109, &tmpCol))	parent->SetColor(COL_SYM_FILL, tmpCol);
		parent->Command(CMD_SYM_TYPE, (void*)(& PrevSym->type), 0L);
		if(PrevSym->type == SYM_TEXT) {
			if(Dlg->GetCheck(250) && PrevSym->Command(CMD_GETTEXT, text1, 0L))
				parent->Command(CMD_SYMTEXT_UNDO, text1, 0L);
			else if(Dlg->GetCheck(252) && Dlg->GetText(253, text2, 100))	
				parent->Command(CMD_SYM_RANGETEXT, text2, 0L);
			if(PrevSym->Command(CMD_GETTEXTDEF, &textdef, 0L))
				parent->Command(CMD_SYMTEXTDEF, &textdef, 0L);
			}
		break;
		}
	CloseDlgWnd(hDlg);		delete Dlg;			free(SymDlg);
	free(tab1);				free(tab2);			free(tab3);
	delete PrevSym;			return undo_flags != 0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bubble properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *BubDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,1,130,10,60,12\n"
	".,.,,,PUSHBUTTON,-28,130,25,60,12\n"
	".,.,,,PUSHBUTTON,-2,130,40,60,12\n"
	".,,5,CHECKED | ISPARENT,GROUP,,,,,\n"
	"5,+,100,ISPARENT | CHECKED,SHEET,2,5,10,120,100\n"
	".,.,200,ISPARENT,SHEET,3,5,10,120,100\n"
	".,,300,ISPARENT,SHEET,4,5,10,120,100\n"
	"100,109,,NOSELECT,ODB,5,18,57,90,50\n"
	"109,+,,ISRADIO,ODB,6,30,30,20,20\n"
	".,.,,ISRADIO,ODB,6,50,30,20,20\n"
	".,.,,ISRADIO,ODB,6,70,30,20,20\n"
	".,,,ISRADIO,ODB,6,90,30,20,20\n"
	"200,+,,,LTEXT,7,10,30,110,8\n"
	".,.,210,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,,,LTEXT,8,10,64,110,8\n"
	".,,220,ISPARENT | CHECKED,GROUP,,,,,\n"
	"210,+,,,RADIO1,-3,40,38,45,8\n"
	".,.,,,RADIO1,9,40,46,45,8\n"
	".,,,,RADIO1,10,40,54,45,8\n"
	"220,+,,,RADIO1,11,40,72,45,8\n"
	".,.,,,RADIO1,12,40,80,45,8\n"
	".,,,,RADIO1,13,40,88,45,8\n"
	"300,+,,,RTEXT,-12,10,40,45,8\n"
	".,.,,,EDVAL1,14,60,40,35,10\n"
	".,.,,,RTEXT,-13,10,60,45,8\n"
	".,.,,,EDVAL1,15,60,60,35,10\n"
	".,.,,,RTEXT,16,10,80,45,8\n"
	".,,,LASTOBJ,EDVAL1,17,60,80,35,10";

bool
Bubble::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_SHAPE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCALE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	void *dyndata[] = {(void*)SDLG_BUB_APPLY, (void*)tab1, (void*)tab2,
		(void*)tab3, (void*)&OD_filldef, (void *)OD_BubbleTempl, (void*)SDLG_BUB_SIZEDEF,
		(void*)SDLG_BUB_PROP, (void*)SDLG_BUB_SCALEX, (void*)SDLG_BUB_SCALEY,
		(void*)SDLG_BUB_DIAM, (void*)SDLG_BUB_CIRC, (void*)SDLG_BUB_AREA, (void*)&fPos.fx,
		(void*)&fPos.fy, (void*)SDLG_BUB_SIZE, (void*)&fs};
	DlgInfo *BubDlg = CompileDialog(BubDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmpType;
	lfPOINT o_pos, n_pos;
	LineDEF newLine, newFillLine;
	static FillDEF newFill;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	double o_size, n_size;
	bool bRet = false;

	if(!parent) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&BubbleLine, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&BubbleFill, 0);
	Dlg = new DlgRoot(BubDlg, data);
	switch(type & 0x00f) {
	case BUBBLE_SQUARE:		Dlg->SetCheck(110, 0L, true);		break;
	case BUBBLE_UPTRIA:		Dlg->SetCheck(111, 0L, true);		break;
	case BUBBLE_DOWNTRIA:	Dlg->SetCheck(112, 0L, true);		break;
	default:				Dlg->SetCheck(109, 0L, true);		break;
		}
	switch(type & 0x0f0) {
	case BUBBLE_XAXIS:		Dlg->SetCheck(211, 0L, true);		break;
	case BUBBLE_YAXIS:		Dlg->SetCheck(212, 0L, true);		break;
	default:				Dlg->SetCheck(210, 0L, true);		break;
		}
	switch(type & 0xf00) {
	case BUBBLE_CIRCUM:		Dlg->SetCheck(221, 0L, true);		break;
	case BUBBLE_AREA:		Dlg->SetCheck(222, 0L, true);		break;
	default:				Dlg->SetCheck(220, 0L, true);		break;
		}
	if(!Dlg->GetValue(301, &o_pos.fx))	o_pos.fx = fPos.fx;
	if(!Dlg->GetValue(303, &o_pos.fy))	o_pos.fy = fPos.fy;
	if(!Dlg->GetValue(305, &o_size))	o_size = fs;
	n_pos.fx = o_pos.fx;	n_pos.fy = o_pos.fy;	n_size = o_size;
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BUB_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BUB_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 410, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 1:			//accept for current bubble only
		case 2:			//accept for plot
			Undo.SetDisp(cdisp);
			OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
			memcpy(&newFillLine, &BubbleFillLine, sizeof(LineDEF));
			if(newFill.hatch) memcpy(&newFillLine, newFill.hatch, sizeof(LineDEF));
			if(Dlg->GetCheck(110)) tmpType = BUBBLE_SQUARE;
			else if(Dlg->GetCheck(111)) tmpType = BUBBLE_UPTRIA;
			else if(Dlg->GetCheck(112)) tmpType = BUBBLE_DOWNTRIA;
			else tmpType = BUBBLE_CIRCLE;
			if(Dlg->GetCheck(211)) tmpType |= BUBBLE_XAXIS;
			else if(Dlg->GetCheck(212)) tmpType |= BUBBLE_YAXIS;
			if(Dlg->GetCheck(221)) tmpType |= BUBBLE_CIRCUM;
			else if(Dlg->GetCheck(222)) tmpType |= BUBBLE_AREA;
			break;
			}
		}while (res < 0);
	switch (res) {
	case 1:				//new setting for current bubble only
		Dlg->GetValue(301, &n_pos.fx);		Dlg->GetValue(303, &n_pos.fy);
		undo_flags = CheckNewLFPoint(&fPos, &o_pos, &n_pos, parent, undo_flags);
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		Dlg->GetValue(305, &n_size);
		undo_flags = CheckNewFloat(&fs, o_size, n_size, parent, undo_flags);
		if(cmpLineDEF(&BubbleLine, &newLine)) {
			Undo.Line(parent, &BubbleLine, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&BubbleLine, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&BubbleFillLine, &newFillLine)) {
			Undo.Line(parent, &BubbleFillLine, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&BubbleFillLine, &newFillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&BubbleFill, &newFill)) {
			Undo.Fill(parent, &BubbleFill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&BubbleFill, &newFill, sizeof(FillDEF));
			}
		BubbleFill.hatch = &BubbleFillLine;
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		break;
	case 2:				//accept settings for plot
		parent->Command(CMD_SAVE_SYMBOLS, 0L, 0L);
		parent->Command(CMD_BUBBLE_TYPE, (void*)(& tmpType), 0L);
		parent->Command(CMD_BUBBLE_ATTRIB, (void*)(& tmpType), 0L);
		parent->Command(CMD_BUBBLE_FILL, (void*)&newFill, 0L);
		parent->Command(CMD_BUBBLE_LINE, (void*)&newLine, 0L);
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);			delete Dlg;				free(BubDlg);
	free(tab1);					free(tab2);				free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bar properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *BarDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,1,130,10,55,12\n"
	".,.,,,PUSHBUTTON,-28,130,25,55,12\n"
	".,.,,,PUSHBUTTON,-2,130,40,55,12\n"
	".,.,,HIDDEN,PUSHBUTTON,-1,130,10,55,12\n"
	".,.,10,CHECKED | ISPARENT,GROUP,,,,,\n"
	".,,120,HIDDEN | CHECKED | ISPARENT,GROUP,,,,,\n"
	"10,+,100,TOUCHEXIT | ISPARENT | CHECKED,SHEET,2,5,10,120,125\n"
	".,.,200,ISPARENT,SHEET,3,5,10,120,125\n"
	".,.,300,ISPARENT,SHEET,4,5,10,120,125\n"
	".,,,TOUCHEXIT,BARBUTT,22,135,80,45,50\n"
	"100,109,,TOUCHEXIT | NOSELECT,ODB,5,18,25,90,50\n"
	"109,+,,,LTEXT,6,10,73,45,8\n"
	".,.,,TOUCHEXIT | HICOL,RADIO1,7,20,82,25,8\n"
	".,.,,TOUCHEXIT,INCDECVAL1,8,60,82,33,10\n"
	".,.,,,LTEXT,-3,95,82,20,8\n"
	".,.,,TOUCHEXIT | HICOL,RADIO1,9,20,93,25,8\n"
	".,.,,TOUCHEXIT,INCDECVAL1,10,60,93,33,10\n"
	".,,,,LTEXT,-10,95,93,10,8\n"
	"120,+,,,LTEXT,23,10,108,45,8\n"
	".,.,,HICOL,RADIO1,24,20,119,25,8\n"
	".,.,,HICOL,RADIO1,25,50,119,25,8\n"
	".,,,HICOL,RADIO1,26,80,119,25,8\n"
	"200,+,,HICOL | TOUCHEXIT,RADIO2,11,20,30,45,8\n"
	".,205,202,CHECKED | ISPARENT,GROUP,,,,,\n"
	".,+,,HICOL | TOUCHEXIT,RADIO1,12,30,40,35,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,-22,30,48,35,8\n"
	".,,,HICOL | TOUCHEXIT,RADIO1,13,30,56,35,8\n"
	"205,+,,,EDVAL1,14,65,56,35,10\n"
	".,.,,HICOL | TOUCHEXIT,RADIO2,15,20,70,45,8\n"
	".,211,208,CHECKED | ISPARENT,GROUP,,,,,\n"
	"208,+,,HICOL | TOUCHEXIT,RADIO1,16,30,80,35,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,-21,30,88,35,8\n"
	".,,,HICOL | TOUCHEXIT,RADIO1,17,30,96,35,8\n"
	"211,+,,,EDVAL1,18,65,96,35,10\n"
	".,,,HICOL,CHECKBOX,19,20,113,50,8\n"
	"300,+,,,RTEXT,-12,10,50,35,8\n"
	".,.,,,EDVAL1,20,50,50,60,10\n"
	".,.,,,RTEXT,-13,10,75,35,8\n"
	".,,,LASTOBJ,EDVAL1,21,50,75,60,10";
bool
Bar::PropertyDlg()
{
	int i;
	double bw1, bw2;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_SIZECOL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_BASELINE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	Bar *PrevBar = new Bar(0L, data, 0.0, 0.0, BAR_VERTB);
	void *dyndata[] = {(void*)SDLG_BAR_TOBAR, (void*)tab1,
		(void*)tab2, (void*)tab3, (void*)OD_filldef, (void*)SDLG_BAR_WIDTH , (void*)SDLG_BAR_FIXED,
		(void*)&bw1, (void*)SDLG_BAR_REL, (void*)&bw2, (void*)SDLG_BAR_VERT,
		(void*)SDLG_BAR_BOTBASE, (void*)SDLG_BAR_USERY, (void*)&BarBase.fy,
		(void*)SDLG_BAR_HOR, (void*)SDLG_BAR_LEFTBASE, (void*)SDLG_BAR_USERX,
		(void*)&BarBase.fx, (void*)SDLG_BAR_CENT, (void*)&fPos.fx, (void*)&fPos.fy, (void*)&PrevBar,
		(void*)SDLG_BAR_ALIGN, (void*)SDLG_BAR_LEFT, (void*)SDLG_BAR_CENTER, (void*)SDLG_BAR_RIGHT};
	DlgInfo *BarDlg;
	DlgRoot *Dlg;
	void *hDlg;
	double n_size;
	int cb, res;
	long tmpType = type;
	bool bRet = false, isVertical = true;
	LineDEF newLine, newFillLine, tmpLine;
	static FillDEF newFill, tmpFill;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	lfPOINT o_bl, n_bl, o_pos, n_pos;

	if(!(BarDlg = CompileDialog(BarDlg_DlgTmpl, dyndata))) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&BarLine, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&BarFill, 0);
	if (!parent) {							//prepare for being part of a dialog
		BarDlg[7].flags |= HIDDEN;			BarDlg[8].flags |= HIDDEN;
		BarDlg[0].flags |= HIDDEN;			BarDlg[1].flags |= HIDDEN;
		BarDlg[2].y = BarDlg[1].y;			BarDlg[3].flags &= ~HIDDEN;
		}
	PrevBar->Command(CMD_PREVSYM, this, 0L);
	if (type & BAR_RELWIDTH) {
		bw1 = DefSize(SIZE_BAR);	bw2 = size;
		}
	else {
		bw1 = size;					bw2 = 50.0;
		}
	Dlg = new DlgRoot(BarDlg, data);
	switch (type & 0xff) {
	case BAR_VERTB:		case BAR_VERTT:		case BAR_VERTU:
		Dlg->SetCheck(200, 0L, true);
		Dlg->SetCheck(208, 0L, true);
		isVertical = true;
		switch(type & 0xff) {
		case BAR_VERTT:	Dlg->SetCheck(203, 0L, true);	break;
		case BAR_VERTU:	Dlg->SetCheck(204, 0L, true);	break;
		default:		Dlg->SetCheck(202, 0L, true);	break;
			}
		Dlg->ShowItem(6, true);
		break;
	case BAR_HORL:		case BAR_HORR:		case BAR_HORU:
		Dlg->SetCheck(206, 0L, true);
		Dlg->SetCheck(202, 0L, true);
		isVertical = false;
		switch(type & 0xff) {
		case BAR_HORR:	Dlg->SetCheck(209, 0L, true);	break;
		case BAR_HORU:	Dlg->SetCheck(210, 0L, true);	break;
		default:		Dlg->SetCheck(208, 0L, true);	break;
			}
		break;
		}
	if (type & BAR_ALIGN_LEFT) Dlg->SetCheck(121, 0L, true);
	else if (type & BAR_ALIGN_RIGHT) Dlg->SetCheck(123, 0L, true);
	else Dlg->SetCheck(122, 0L, true);
	if (type & BAR_RELWIDTH) Dlg->SetCheck(113, 0L, true);
	else Dlg->SetCheck(110, 0L, true);
	if(type & BAR_CENTERED) Dlg->SetCheck(212, 0L, true);
	if(!Dlg->GetValue(211, &o_bl.fx))	o_bl.fx = BarBase.fx;
	if(!Dlg->GetValue(205, &o_bl.fy))	o_bl.fy = BarBase.fy;
	if(!Dlg->GetValue(301, &o_pos.fx))	o_pos.fx = fPos.fx;
	if(!Dlg->GetValue(303, &o_pos.fy))	o_pos.fy = fPos.fy;
	n_bl.fx = o_bl.fx;			n_bl.fy = o_bl.fy;
	n_pos.fx = o_pos.fx;		n_pos.fy = o_pos.fy;
	n_size = size;
	if (parent && parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BAR_HD1);
		rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BAR_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 400, 330, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 10:
			Dlg->SetCheck(6, 0L, true);		Dlg->DoPlot(0L);
			Dlg->ShowItem(6, isVertical);
			res = -1;	break;
		case 13:						//the bar preview button
		case 110:	case 111:	case 113:	case 114:
			if (res == 111) {
				Dlg->SetCheck(110, 0L, true);				PrevBar->type &= ~BAR_RELWIDTH;
				Dlg->GetValue(111, &n_size);				PrevBar->SetSize(SIZE_BAR, n_size);
				}
			if (res == 114) {
				Dlg->SetCheck(113, 0L, true);				PrevBar->type |= BAR_RELWIDTH;
				Dlg->GetValue(114, &n_size);				PrevBar->SetSize(SIZE_BAR, n_size);
				}
			OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&tmpFill, 0);
			PrevBar->Command(CMD_BAR_FILL, &tmpFill, 0L);
			OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&tmpLine, 0);
			PrevBar->SetSize(SIZE_BAR_LINE, tmpLine.width);		PrevBar->SetColor(COL_BAR_LINE, tmpLine.color);
			if (Dlg->GetCheck(113)) Dlg->GetValue(114, &n_size);
			else Dlg->GetValue(111, &n_size);
			PrevBar->SetSize(SIZE_BAR, n_size);
			Dlg->DoPlot(0L);
			res = -1;	break;
		case 100:					//the shape fill sub-dialog
			OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&tmpFill, 0);
			PrevBar->Command(CMD_BAR_FILL, &tmpFill, 0L);
			OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&tmpLine, 0);
			PrevBar->SetSize(SIZE_BAR_LINE, tmpLine.width);		PrevBar->SetColor(COL_BAR_LINE, tmpLine.color);
			if (Dlg->GetCheck(113)) Dlg->GetValue(114, &n_size);
			else Dlg->GetValue(111, &n_size);
			PrevBar->SetSize(SIZE_BAR, n_size);
			Dlg->DoPlot(0L);
			res = -1;	break;
		case 202:	case 203:	case 204:
			Dlg->SetCheck(200, NULL, true);
			tmpType = res == 202 ? BAR_VERTB : res == 203 ? BAR_VERTT : BAR_VERTU;
			res = -1;				//continue on radiobutton
			break;
		case 208:	case 209:	case 210:
			Dlg->SetCheck(206, NULL, true);
			tmpType = res == 208 ? BAR_HORL : res == 209 ? BAR_HORR : BAR_HORU;
			res = -1;				//continue on radiobutton
			break;
		case 200:					//group of vertical bars
			if(Dlg->GetCheck(203)) tmpType = BAR_VERTT;
			else if(Dlg->GetCheck(204)) tmpType = BAR_VERTU;
			else tmpType = BAR_VERTB;
			res = -1;
			break;
		case 206:					//group of horizontal bars
			if(Dlg->GetCheck(209)) tmpType = BAR_HORR;
			else if(Dlg->GetCheck(210)) tmpType = BAR_HORU;
			else tmpType = BAR_HORL;
			res = -1;
			break;
		case 1:		case 2:		case 4:
			Undo.SetDisp(cdisp);
			OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
			memcpy(&newFillLine, &HatchLine, sizeof(LineDEF));
			if(newFill.hatch) memcpy(&newFillLine, newFill.hatch, sizeof(LineDEF));
			if(Dlg->GetCheck(113)) {
				tmpType |= BAR_RELWIDTH;			Dlg->GetValue(114, &n_size);
				}
			else {
				tmpType &= ~BAR_RELWIDTH;			Dlg->GetValue(111, &n_size);
				}
			if (isVertical){
				tmpType &= ~(BAR_ALIGN_LEFT | BAR_ALIGN_RIGHT);
				if (Dlg->GetCheck(121)) tmpType |= BAR_ALIGN_LEFT;
				else if (Dlg->GetCheck(123)) tmpType |= BAR_ALIGN_RIGHT;
				}
			if(Dlg->GetCheck(212))tmpType |= BAR_CENTERED; 
			else tmpType &= ~BAR_CENTERED;
			Dlg->GetValue(211, &n_bl.fx);			Dlg->GetValue(205, &n_bl.fy);
			Dlg->GetValue(301, &n_pos.fx);			Dlg->GetValue(303, &n_pos.fy);
			break;
			}
		}while (res < 0);
	if(parent)switch (res) {
	case 1:				//new setting for current bar only
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		undo_flags = CheckNewLFPoint(&BarBase, &o_bl, &n_bl, parent, undo_flags);
		undo_flags = CheckNewLFPoint(&fPos, &o_pos, &n_pos, parent, undo_flags);
		if(parent && undo_flags & UNDO_CONTINUE) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		undo_flags = CheckNewFloat(&size, size, n_size, parent, undo_flags);
		if(cmpLineDEF(&BarLine, &newLine)) {
			Undo.Line(parent, &BarLine, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&BarLine, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&HatchLine, &newFillLine)) {
			Undo.Line(parent, &HatchLine, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&HatchLine, &newFillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&BarFill, &newFill)) {
			Undo.Fill(parent, &BarFill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&BarFill, &newFill, sizeof(FillDEF));
			}
		BarFill.hatch = &HatchLine;
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		break;
	case 2:				//new settings to all bars of plot
		parent->Command(CMD_SAVE_BARS, 0L, 0L);
		if(parent->Id == GO_PLOTSCATT && parent->parent 
			&& parent->parent->Id == GO_STACKBAR){ 
			parent->parent->SetSize(SIZE_BAR, n_size);
			parent->parent->Command(CMD_BAR_TYPE, (void *)(& tmpType), 0L);
			}
		else {
			parent->SetSize(SIZE_BAR, n_size);
			parent->Command(CMD_BAR_TYPE, (void *)(& tmpType), 0L);
			}
		parent->SetColor(COL_BAR_LINE, newLine.color);
		parent->SetSize(SIZE_BAR_LINE, newLine.width);
		parent->SetSize(SIZE_YBASE, n_bl.fy);
		parent->SetSize(SIZE_XBASE, n_bl.fx);
		parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		parent->Command(CMD_BAR_TYPE, (void *)(& tmpType), 0L);
		parent->Command(CMD_BAR_FILL, (void *)&newFill, 0L);
		bRet = true;
		break;
		}
	else if (!parent && (res == 1 || res == 4)) {
		memcpy(&BarLine, &newLine, sizeof(LineDEF));
		memcpy(&HatchLine, &newFillLine, sizeof(LineDEF));
		memcpy(&BarFill, &newFill, sizeof(FillDEF));
		BarFill.hatch = &HatchLine;		type = tmpType;
		size = n_size;
		bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(BarDlg);
	free(tab1);				free(tab2);		free(tab3);
	delete PrevBar;
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Data line properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *LineDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,180,10,60,12\n"
".,.,,,PUSHBUTTON,-2,180,25,60,12\n"
".,.,,HIDDEN,PUSHBUTTON,10,180,110,60,12\n"		//DEBUG
".,,8,ISPARENT | CHECKED,GROUP,,,,,\n"
"8,+,100,ISPARENT | CHECKED,SHEET,1,5,10,165,120\n"
".,.,200,ISPARENT,SHEET,2,5,10,165,120\n"
".,,300,ISPARENT | HIDDEN,SHEET,3,5,10,165,120\n"
"100,,,NOSELECT,ODB,6,18,38,130,100\n"
"200,+,,,LTEXT,-27,8,32,130,100\n"
".,.,,EXRADIO,ODB,7,12,45,,\n"
".,.,,EXRADIO,ODB,7,37,45,,\n"
".,.,,EXRADIO,ODB,7,62,45,,\n"
".,.,,EXRADIO,ODB,7,87,45,,\n"
".,.,,EXRADIO,ODB,7,112,45,,\n"
".,.,,EXRADIO,ODB,7,37,70,,\n"
".,.,,EXRADIO,ODB,7,62,70,,\n"
".,.,,EXRADIO,ODB,7,87,70,,\n"
".,.,,EXRADIO,ODB,7,112,70,,\n"
".,.,,EXRADIO,ODB,7,37,95,,\n"
".,.,,EXRADIO,ODB,7,62,95,,\n"
".,,,EXRADIO,ODB,7,87,95,,\n"
"300,+,,,LTEXT,-31,15,30,80,9\n"
".,.,,,EDTEXT,4,15,40,139,10\n"
".,.,,,LTEXT,-32,15,55,80,9\n"
".,.,,,EDTEXT,5,15,65,139,10\n"
".,.,,,LTEXT,8,15,80,60,8\n"
".,.,,,EDTEXT,9,50,80,105,10\n"
".,.,,,ICON,11,10,95,10,10\n"
".,.,,,LTEXT,8,35,95,30,6\n"
".,.,,,LTEXT,12,35,101,60,8\n"
".,.,,,LTEXT,13,35,107,60,8\n"
".,,,HREF | LASTOBJ,LTEXT,14,35,113,60,8";

bool
DataLine::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	int icon = ICO_INFO;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)ssXref, (void*)ssYref,
		(void*)OD_linedef, (void*)(OD_LineStyleTempl), (void*)SDLG_DLINE_COND, (void*)cond,
		(void*)SDLG_DLINE_COPY, (void*)&icon, (void*)SDLG_DLINE_COND1,
		(void*)SDLG_DLINE_COND2, (void*)SDLG_DLINE_COND3 };
	DlgInfo *LineDlg = CompileDialog(LineDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmpType = type;
	DWORD undo_flags = 0L;
	LineDEF newLine;
	bool bRet = false;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent) return false;
	if(parent->Id == GO_FUNCTION) return parent->PropertyDlg();
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
	if(!(Dlg = new DlgRoot(LineDlg, data)))return false;
	Dlg->SetCheck(201 + (type & 0x0f), 0L, true);
	if (parent->Id == GO_TERNLINE)Dlg->ShowItem(9, false);
	if(ssXref && ssYref) Dlg->ShowItem(10, true);
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DLINE_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DLINE_HD2);
#ifdef _WINDOWS
	for (i = 307; i <= 310; i++) Dlg->TextSize(i, 12);
#else
	for (i = 307; i <= 310; i++) Dlg->TextSize(i, 10);
#endif
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 510, 320, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 201:	case 202:	case 203:	case 204:	case 205:	case 206:
		case 207:	case 208:	case 209:	case 210:	case 211:	case 212:
			if((tmpType & 0x0f) == (res-201)) res = 1;
			else {
				tmpType = res - 201;
				res = -1;
				}
			break;
		case 3:			//copy data
			if (parent) parent->Command(CMD_COPY, (void*)this, 0L);
			res = -1;
			break;
			}
		}while (res < 0);
	if(res == 1){						//OK pressed
		Undo.SetDisp(cdisp);
		if(ssXref && ssYref) {
			TmpTxt[0] = 0;		Dlg->GetText(301, (unsigned char*)TmpTxt, TMP_TXT_SIZE); 
			undo_flags = CheckNewString((unsigned char**)&ssXref, (unsigned char*)ssXref, (unsigned char*)TmpTxt, this, undo_flags);
			TmpTxt[0] = 0;	Dlg->GetText(303, (unsigned char*)TmpTxt, TMP_TXT_SIZE);
			undo_flags = CheckNewString((unsigned char**)&ssYref, (unsigned char*)ssYref, (unsigned char*)TmpTxt, this, undo_flags);
			TmpTxt[0] = 0;	Dlg->GetText(305, (unsigned char*)TmpTxt, TMP_TXT_SIZE);
			if (TmpTxt[0]) undo_flags = CheckNewString((unsigned char**)&cond, (unsigned char*)cond, (unsigned char*)TmpTxt, this, undo_flags);
			else undo_flags = CheckNewString((unsigned char**)&cond, (unsigned char*)cond, (unsigned char*)0L, this, undo_flags);
			if(undo_flags & UNDO_CONTINUE) {
				Command(CMD_UPDATE, 0L, cdisp);			parent->Command(CMD_MRK_DIRTY, 0L, cdisp);
				}
			}
		OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		if(cmpLineDEF(&LineDef, &newLine)) {
			Undo.Line(parent, &LineDef, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&LineDef, &newLine, sizeof(LineDEF));
			}
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		if (bRet &&parent->Id == GO_TERNLINE){
			((TernaryXYZ*)(parent->parent))->Command(CMD_SET_LINE, &newLine, 0L);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(LineDlg);
	free(tab1);			free(tab2);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Data polygon properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *PolygonDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,130,25,45,12\n"
	".,,4,CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED, SHEET,1, 5,10,120,85\n"
	".,,200,ISPARENT,SHEET,2,5,10,120,85\n"
	"100,204,,NOSELECT,ODB,3,23,30,90,50\n"
	"200,+,,,LTEXT,-27,10,27,130,100\n"
	".,.,,EXRADIO,ODB,4,10,40,25,25\n"
	".,.,,EXRADIO,ODB,4,37,40,25,25\n"
	".,213,,EXRADIO,ODB,4,64,40,25,25\n"
	".,,,HICOL | TOUCHEXIT,CHECKBOX,5,15,82,62,8\n"
	"213,204,,LASTOBJ | EXRADIO,ODB,4,91,40,25,25";
bool
DataPolygon::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_POLGON);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)OD_filldef, (void*)OD_PolygonStyleTempl,
		(void*)SDLG_DPGON_TOBL};
	DlgInfo *PolygonDlg;
	DlgRoot *Dlg;
	void *hDlg;
	LineDEF newLine, newFillLine;
	FillDEF newFill;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	int cb, res;
	long tmpType = type;
	bool bRet = false;

	if(!data || !parent) return false;
	if(!(PolygonDlg = CompileDialog(PolygonDlg_DlgTmpl, dyndata))) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&pgFill, 0);
	Dlg = new DlgRoot(PolygonDlg, data);
	if (Id == GO_ERRORPOLYGON) {
		Dlg->ShowItem(204, false);
		type &= ~0x100;
		}
	else Dlg->SetCheck(204, 0L, type & 0x100 ? true : false);
	if (parent->Id == GO_TERNLINE) {
		Dlg->ShowItem(5, false);
		}
	switch (type & 0x0f) {
		case 0:		default:
			Dlg->SetCheck(201, 0L, true);
			break;
		case 1:		case 2:		case 12:
			Dlg->SetCheck(201+type, 0L, true);
			break;
		}
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, Id == GO_ERRORPOLYGON ? (char*)SDLG_DPGON_HD5 : (char*)SDLG_DPGON_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else if(parent->Id == GO_TICK) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DPGON_HD2);
		if(parent->Command(CMD_GETTEXT, TmpTxt+cb, 0L)) {
			cb = rlp_strlen(TmpTxt); TmpTxt[cb++] = '"';	TmpTxt[cb++] = ' ';
			}
		else cb--;
		cb += rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE, (char*)SDLG_DPGON_HD3);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DPGON_HD4);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 380, 250, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 201:	case 202:	case 203:	case 213:
			if((tmpType & 0x0f) == (res-201)) res = 1;
			else {
				tmpType &= ~0x0f;	tmpType |= (res-201);
				res = -1;
				}
			break;
		case 204:				//expand polygon to baseline
			if (Dlg->GetCheck(204)) tmpType |= 0x100;
			else tmpType &= ~(0x100);
			res = -1;					break;
			}
		}while (res < 0);
	if(res == 1){						//OK pressed
		Undo.SetDisp(cdisp);
		if (Dlg->GetCheck(204)) tmpType |= 0x100;
		else tmpType &= ~(0x100);
		if (tmpType != type && Dlg->GetCheck(204)) {
			Undo.DataMem(this, (void**)&Values, nPnt * sizeof(lfPOINT), &nPnt, 0L);
			undo_flags |= UNDO_CONTINUE;
			}
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
		memcpy(&newFillLine, &pgFillLine, sizeof(LineDEF));
		if(newFill.hatch) memcpy(&newFillLine, newFill.hatch, sizeof(LineDEF));
		if(cmpLineDEF(&LineDef, &newLine)) {
			Undo.Line(parent, &LineDef, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&LineDef, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&pgFillLine, &newFillLine)) {
			Undo.Line(parent, &pgFillLine, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&pgFillLine, &newFillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&pgFill, &newFill)) {
			Undo.Fill(parent, &pgFill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&pgFill, &newFill, sizeof(FillDEF));
			}
		pgFill.hatch = &pgFillLine;
		if ((type & 0x100) != (tmpType & 0x100)) {		//remove points completing the polygon
			nPnt = src_nPnt;
			}
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		if (undo_flags & UNDO_CONTINUE) bRet = true;
		if (bRet &&parent->Id == GO_TERNLINE){
			((TernaryXYZ*)(parent->parent))->Command(CMD_PG_FILL, &newFill, 0L);
			((TernaryXYZ*)(parent->parent))->Command(CMD_SET_LINE, &newLine, 0L);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;		free(PolygonDlg);
	free(tab1);		free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Error polygon properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool
ErrorPolygon::PropertyDlg()
{
	return DataPolygon::PropertyDlg();
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Regression line properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *RegLine_DlgTmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,-1,155,10,45,12\n"
		".,.,,,PUSHBUTTON,-2,155,25,45,12\n"
		".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
		"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,144,130\n"
		".,.,200,ISPARENT,SHEET,2,5,10,144,130\n"
		".,,300,ISPARENT,SHEET,3,5,10,144,130\n"
		"100,,,NOSELECT,ODBUTTON,4,12,38,130,100\n"
		"200,+,,HICOL | TOUCHEXIT,RADIO1,5,15,25,130,8\n"
		".,.,,,LTEXT,6,25,33,120,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,7,15,45,130,8\n"
		".,.,,,LTEXT,8,25,53,120,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,9,15,65,130,8\n"
		".,.,,,LTEXT,10,25,73,120,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,11,15,85,130,8\n"
		".,.,,,LTEXT,12,25,93,120,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,13,15,105,130,8\n"
		".,.,,,LTEXT,14,25,113,60,8\n"
		".,.,,,RTEXT,15,92,113,10,8\n"
		".,.,,,EDVAL1,16,104,113,40,10\n"
		".,.,,,RTEXT,17,92,125,10,8\n"
		".,,,,EDVAL1,18,104,125,40,10\n"
		"300,+,,,LTEXT,19,15,30,100,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,20,15,45,100,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,21,15,60,100,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,22,15,75,100,8\n"
		".,.,,,RTEXT,-4,10,89,15,8\n"
		".,.,,,EDVAL1,23,27,89,40,10\n"
		".,.,,,LTEXT,-7,70,89,5,8\n"
		".,.,,,EDVAL1,24,76,89,40,10\n"
		".,.,,,LTEXT,-34,119,89,15,8\n"
		".,.,,,RTEXT,-5,10,101,15,8\n"
		".,.,,,EDVAL1,25,27,101,40,10\n"
		".,.,,,LTEXT,-7,70,101,5,8\n"
		".,.,,,EDVAL1,26,76,101,40,10\n"
		".,,,LASTOBJ,LTEXT,-34,119,101,15,8";

bool
RegLine::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_MODEL);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_CLIP);
	char text1[60], text2[60], text3[60], text4[60], text5[60];
	DlgRoot *Dlg;
	DlgInfo *RegLineDlg;
	void *hDlg;
	int cb, res;
	long tmpType;
	bool bRet = false;
	LineDEF newLine;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	lfPOINT o_l5, n_l5;
	fRECT o_clip, n_clip;
	char *tx, *ty;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)OD_linedef, (void*)SDLG_REGR_YDEPX,
		(void*)text1, (void*)SDLG_REGR_XDEPY,  (void*)text2, (void*)SDLG_REGR_MIX, (void*)text3,
		(void*)SDLG_REGR_ZERO,  (void*)text4, (void*)SDLG_REGR_MANU, (void*)text5,
		(void*)SDLG_REGR_AEQ, (void*)&l5.fx, (void*)SDLG_REGR_BEQ, (void*)&l5.fy, (void*)SDLG_REGR_LIM,
		(void*)SDLG_REGR_MINMAX, (void*)SDLG_REGR_FRAME, (void*)SDLG_REGR_UDEF,
		(void*)&uclip.Xmin, (void*)&uclip.Xmax, (void*)&uclip.Ymin, (void*)&uclip.Ymax};

	if(!(RegLineDlg = CompileDialog(RegLine_DlgTmpl, dyndata))) return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
	switch(type &0x700) {
	case 0x100:		tx = (char*)"log(x)";	break;
	case 0x200:		tx = (char*)"(1/x)";	break;
	case 0x300:		tx = (char*)"sqrt(x)"; break;
	default:		tx = (char*)"x";		break;
		}
	switch(type &0x7000) {
	case 0x1000:	ty = (char*)"log(y)";	break;
	case 0x2000:	ty = (char*)"(1/y)";	break;
	case 0x3000:	ty = (char*)"sqrt(y)"; break;
	default:		ty = (char*)"y";		break;
		}
#ifdef USE_WIN_SECURE
	sprintf_s(text1, 60, "%s = %.3lf + %.3lf * %s   (n = %ld)", ty, l1.fx, l1.fy, tx, nPoints);
	sprintf_s(text2, 60, "%s = %.3lf + %.3lf * %s", ty, l2.fx, l2.fy, tx);
	sprintf_s(text3, 60, "%s = %.3lf + %.3lf * %s", ty, l3.fx, l3.fy, tx);
	sprintf_s(text4, 60, "%s = %.3lf + %.3lf * %s", ty, l4.fx, l4.fy, tx);
	sprintf_s(text5, 60, "%s = a + b * %s", ty, tx);
#else
	sprintf(text1, "%s = %.3lf + %.3lf * %s   (n = %ld)", ty, l1.fx, l1.fy, tx, nPoints);
	sprintf(text2, "%s = %.3lf + %.3lf * %s", ty, l2.fx, l2.fy, tx);
	sprintf(text3, "%s = %.3lf + %.3lf * %s", ty, l3.fx, l3.fy, tx);
	sprintf(text4, "%s = %.3lf + %.3lf * %s", ty, l4.fx, l4.fy, tx);
	sprintf(text5, "%s = a + b * %s", ty, tx);
#endif
	if(!(Dlg = new DlgRoot(RegLineDlg, data))) return false;
	Dlg->Activate(211, 0);	Dlg->Activate(213, 0);
	switch(type & 0x07) {
	case 1:		Dlg->SetCheck(202, 0L, true);	break;
	case 2:		Dlg->SetCheck(204, 0L, true);	break;
	case 3:		Dlg->SetCheck(206, 0L, true);	break;
	case 4:		
		Dlg->SetCheck(208, 0L, true);
		Dlg->Activate(211, 1);	Dlg->Activate(213, 1);
		break;
	default:	Dlg->SetCheck(200, 0L, true);	break;
		}
	switch(type & 0x70) {
	case 0x10:	Dlg->SetCheck(302, 0L, true);	break;
	case 0x20:	Dlg->SetCheck(303, 0L, true);	break;
	default:	Dlg->SetCheck(301, 0L, true);	break;
		}
	if(0x20 == (type & 0x70)) {
		Dlg->Activate(305, 1);	Dlg->Activate(307, 1);
		Dlg->Activate(310, 1);	Dlg->Activate(312, 1);
		}
	else {
		Dlg->Activate(305, 0);	Dlg->Activate(307, 0);
		Dlg->Activate(310, 0);	Dlg->Activate(312, 0);
		}
	if(!Dlg->GetValue(211, &o_l5.fx))	o_l5.fx = l5.fx;
	if(!Dlg->GetValue(213, &o_l5.fy))	o_l5.fy = l5.fy;
	n_l5.fx = o_l5.fx;			n_l5.fy = o_l5.fy;
	if(!Dlg->GetValue(305, &o_clip.Xmin)) o_clip.Xmin = uclip.Xmin;
	if(!Dlg->GetValue(307, &o_clip.Xmax)) o_clip.Xmax = uclip.Xmax;
	if(!Dlg->GetValue(310, &o_clip.Ymin)) o_clip.Ymin = uclip.Ymin;
	if(!Dlg->GetValue(312, &o_clip.Ymax)) o_clip.Ymax = uclip.Ymax;
	memcpy(&n_clip, &o_clip, sizeof(fRECT));
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_REGR_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_REGR_HD2);
	hDlg = CreateDlgWnd(cp ? TmpTxt : 
		(char*)SDLG_REGR_HD3, 50, 50, 430, 340, Dlg, 0x4L);
	if(!cp) Dlg->SetCheck(5, 0L, true);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 200:	case 202:	case 204:	case 206:
			Dlg->Activate(211, 0);	Dlg->Activate(213, 0);
			res = -1;
			break;
		case 208:
			Dlg->Activate(211, 1);	Dlg->Activate(213, 1);
			res = -1;
			break;
		case 301:	case 302:
			Dlg->Activate(305, 0);	Dlg->Activate(307, 0);
			Dlg->Activate(310, 0);	Dlg->Activate(312, 0);
			res = -1;
			break;
		case 303:
			Dlg->Activate(305, 1);	Dlg->Activate(307, 1);
			Dlg->Activate(310, 1);	Dlg->Activate(312, 1);
			res = -1;
			break;
			}
		}while (res < 0);
	if(res == 1){						//OK pressed
		Undo.SetDisp(cdisp);
		OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		if(cmpLineDEF(&LineDef, &newLine)) {
			Undo.Line(parent, &LineDef, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&LineDef, &newLine, sizeof(LineDEF));
			}
		tmpType = type & (~0x77);
		if(Dlg->GetCheck(202)) tmpType |= 1;
		else if(Dlg->GetCheck(204)) tmpType |= 2;
		else if(Dlg->GetCheck(206)) tmpType |= 3;
		else if(Dlg->GetCheck(208)) tmpType |= 4;
		if(Dlg->GetCheck(302)) tmpType |= 0x10;
		else if(Dlg->GetCheck(303)) tmpType |= 0x20;
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		Dlg->GetValue(211, &n_l5.fx);			Dlg->GetValue(213, &n_l5.fy);
		undo_flags = CheckNewLFPoint(&l5, &o_l5, &n_l5, parent, undo_flags);
		Dlg->GetValue(305, &n_clip.Xmin);		Dlg->GetValue(307, &n_clip.Xmax);
		Dlg->GetValue(310, &n_clip.Ymin);		Dlg->GetValue(312, &n_clip.Ymax);
		undo_flags = CheckNewFloat(&uclip.Xmin, o_clip.Xmin, n_clip.Xmin, parent, undo_flags);
		undo_flags = CheckNewFloat(&uclip.Xmax, o_clip.Xmax, n_clip.Xmax, parent, undo_flags);
		undo_flags = CheckNewFloat(&uclip.Ymin, o_clip.Ymin, n_clip.Ymin, parent, undo_flags);
		undo_flags = CheckNewFloat(&uclip.Ymax, o_clip.Ymax, n_clip.Ymax, parent, undo_flags);
		if(!cp || (undo_flags & UNDO_CONTINUE)) bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(RegLineDlg);
	free(tab1);				free(tab2);		free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// SDellipse properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* SdEllipseDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,150,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,150,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,139,140\n"
	".,,200,ISPARENT,SHEET,2,5,10,139,140\n"
	"100,,,NOSELECT,ODB,3,10,48,130,100\n"
	"200,+,,HICOL,CHECKBOX,4,20,26,80,9\n"
	".,.,250,CHECKED, GROUPBOX,5,10,45,129,100\n"
	"250,+,,,LTEXT,6,25,82,60,8\n"
	".,.,,,RTEXT,7,20,92,60,8\n"
	".,.,,,LTEXT,0,82,92,30,8\n"
	".,.,,,RTEXT,8,20,100,60,8\n"
	".,.,,,LTEXT,0,82,100,30,8\n"
	".,.,,,LTEXT,9,25,112,60,8\n"
	".,.,,,RTEXT,10,20,122,60,8\n"
	".,.,,,LTEXT,0,82,122,30,8\n"
	".,.,,,RTEXT,11,20,130,60,8\n"
	".,.,,,LTEXT,0,82,130,30,8\n"
	".,.,,,LTEXT,12,25,52,30,8\n"
	".,.,,HICOL,RADIO1,13,50,52,30,8\n"
	".,.,,HICOL,RADIO1,14,50,61,30,8\n"
	".,,,HICOL | LASTOBJ,RADIO1,15,50,70,30,8";
bool
SDellipse::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAIL);
	void *dyndata[] = {(void*)tab1, (void*) tab2, (void*)OD_linedef, (void*)SDLG_SDELLY_REGLINE,
		(void*)SDLG_SDELLY_ELLY, (void*)SDLG_SDELLY_CENTER, (void*)SDLG_SDELLY_XEQ, (void*)SDLG_SDELLY_YEQ,
		(void*)SDLG_SDELLY_STDEV, (void*)SDLG_SDELLY_MAXIS, (void*)SDLG_SDELLY_MIXIS, (void*)SDLG_SDELLY_SIZE,
		(void*)SDLG_SDELLY_1SD, (void*)SDLG_SDELLY_2SD, (void*)SDLG_SDELLY_3SD};
	DlgInfo *ellDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmpType;
	LineDEF newLine;
	bool bRet = false;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent) return false;
	if(!(ellDlg = CompileDialog(SdEllipseDlg_Tmpl, dyndata))) return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
	if(!(Dlg = new DlgRoot(ellDlg, data))) return false;
	if(!(type & 0x10000)) Dlg->SetCheck(200, 0L, true);
	WriteNatFloatToBuff(TmpTxt, mx);		Dlg->SetText(252, (unsigned char*)(TmpTxt+1));
	WriteNatFloatToBuff(TmpTxt, my);		Dlg->SetText(254, (unsigned char*)(TmpTxt+1));
	rlp_strcpy(TmpTxt, 4, (char*)"+/-");
	WriteNatFloatToBuff(TmpTxt+3, sd1);		Dlg->SetText(259, (unsigned char*)TmpTxt);
	WriteNatFloatToBuff(TmpTxt+3, sd2);		Dlg->SetText(257, (unsigned char*)TmpTxt);
	switch(type & 0x60000) {
		case 0x20000:	Dlg->SetCheck(262, 0L, true);	break;
		case 0x40000:	Dlg->SetCheck(263, 0L, true);	break;
		default:		Dlg->SetCheck(261, 0L, true);	break;
		}
	tmpType = type;
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_SDELLY_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_SDELLY_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 410, 340, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		}while (res < 0);
	if(res == 1){						//OK pressed
		Undo.SetDisp(cdisp);
		OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		if(cmpLineDEF(&LineDef, &newLine)) {
			Undo.Line(parent, &LineDef, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&LineDef, &newLine, sizeof(LineDEF));
			}
		if(Dlg->GetCheck(200)) tmpType &= ~0x10000;
		else tmpType |= 0x10000;
		tmpType &= ~0x60000;
		if(Dlg->GetCheck(262)) tmpType |= 0x20000;
		else if(Dlg->GetCheck(263)) tmpType |= 0x40000;
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;			free(ellDlg);
	free(tab1);				free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Normal error bar properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* ErrDlg_Tmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,1,120,10,70,12\n"
	"2,3,,,PUSHBUTTON,-28,120,25,70,12\n"
	"3,4,,,PUSHBUTTON,-2,120,40,70,12\n"
	"4,,5,ISPARENT | CHECKED,GROUP,,,,,\n"
	"5,6,100,ISPARENT | CHECKED,SHEET,2,5,10,110,100\n"
	"6,7,200,ISPARENT,SHEET,3,5,10,110,100\n"
	"7,8,300,TOUCHEXIT | ISPARENT,SHEET,4,5,10,110,100\n"
	"8,9,,DEFAULT | HIDDEN,PUSHBUTTON,-1,120,10,70,12\n"
	"9,10,,HIDDEN,PUSHBUTTON,-2,120,25,70,12\n"
	"10,,,TOUCHEXIT,ERRPREV,16,130,70,40,40\n"
	"100,+,,,RTEXT,5,25,40,28,8\n"
	".,.,,TOUCHEXIT,INCDECVAL1,6,56,40,33,10\n"
	".,.,,,LTEXT,-3,91,40,20,8\n"
	".,.,,,RTEXT,-29,25,55,28,8\n"
	".,.,,TOUCHEXIT,INCDECVAL1,7,56,55,33,10\n"
	".,.,,,LTEXT,-3,91,55,20,8\n"
	".,.,,,RTEXT,-30,25,70,28,8\n"
	".,,,OWNDIALOG | TOUCHEXIT,COLBUTT,8,56,70,25,10\n"
	"200,+,500,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,,,,LTEXT,-27,10,30,70,8\n"
	"300,+,,,RTEXT,-12,15,35,28,8\n"
	".,.,,,EDVAL1,9,46,35,55,10\n"
	".,.,,,RTEXT,-13,15,48,28,8\n"
	".,.,,,EDVAL1,10,46,48,55,10\n"
	".,.,,,RTEXT,11,15,61,28,8\n"
	".,.,,,EDVAL1,12,46,61,55,10\n"
	".,.,,,LTEXT,13,10,80,70,8\n"
	".,.,,,EDTEXT,14,10,90,100,10\n"
	".,,,HICOL,CHECKBOX,17,10,67,90,8\n"
	"500,+,,EXRADIO,ODB,15,22,45,,\n"
	".,.,,EXRADIO,ODB,15,47,45,,\n"
	".,.,,EXRADIO,ODB,15,72,45,,\n"
	".,.,,EXRADIO,ODB,15,22,70,,\n"
	".,.,,EXRADIO,ODB,15,47,70,,\n"
	".,,,LASTOBJ | EXRADIO,ODB,15,72,70,,";
	
bool
ErrorBar::PropertyDlg()
{
	int i;
	ErrorBar *prevsym = new ErrorBar(0L, data, 0.0, 0.0, 0.0, 0);
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_ERRBAR);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TYPE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	void *dyndata[] = { (void*)SDLG_EBAR_TOERR, (void*)tab1, (void*)tab2, (void*)tab3,
		(void*)SDLG_EBAR_CWIDTH, (void*)&SizeBar, (void*)&ErrLine.width, (void *)&ErrLine.color,
		(void*)&fPos.fx, (void*)&fPos.fy, (void*)SDLG_EBAR_ERR, (void*)&ferr, (void*)SDLG_EBAR_DESC,
		(void*)name, (void*)(OD_ErrBarTempl), (void*)&prevsym, (void*)SDLG_EBAR_CENT};
	DlgInfo *ErrDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmpType = type;
	double n_sb, o_sb, n_lw, o_lw, n_err, o_err;
	lfPOINT n_pos, o_pos;
	DWORD n_col, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	bool bRet = false;
	unsigned char desc[80];

	if (!(ErrDlg = CompileDialog(ErrDlg_Tmpl, dyndata))) return false;
	prevsym->Command(CMD_PREVSYM, this, 0L);
	desc[0] = 0;
	if (!(Dlg = new DlgRoot(ErrDlg, data)))return false;
	if (!parent) {						//adjust ErrorBar dialog to dialog item
		Dlg->ShowItem(7, false);		Dlg->ShowItem(1, false);
		Dlg->ShowItem(8, true);			Dlg->ShowItem(2, false);
		Dlg->ShowItem(9, true);			Dlg->ShowItem(3, false);
	}
	Dlg->SetCheck(500 + (type & 0x7), 0L, true);
	if (type & 0x100){
		Dlg->ShowItem(308, true);	Dlg->SetCheck(308, 0L, (type & 0x200) == 0x200);
		}
	else Dlg->ShowItem(308, false);
	if (!(Dlg->GetValue(101, &o_sb))) o_sb = SizeBar;
	if (!(Dlg->GetValue(104, &o_lw))) o_lw = ErrLine.width;
	if (!(Dlg->GetValue(305, &o_err))) o_err = ferr;
	if (!(Dlg->GetValue(301, &o_pos.fx))) o_pos.fx = fPos.fx;
	if (!(Dlg->GetValue(303, &o_pos.fy))) o_pos.fy = fPos.fy;
	n_sb = o_sb;	n_lw = o_lw;	n_err = o_err; n_pos.fx = o_pos.fx;	n_pos.fy = o_pos.fy;
	if (parent && parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_EBAR_HD1);
		rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, parent->name);
	}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_EBAR_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 410, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 500:	case 501:	case 502:
		case 503:	case 504:	case 505:
			tmpType = (tmpType & 0x300) + (res - 500);
			res = -1;
			prevsym->type = tmpType;		Dlg->DoPlot(0L);
			break;
		case 7:								//edit tab
			Dlg->Activate(307, 1);	res = -1;	break;
		case 1:								//accept for this object
		case 2:								//   or all objects of plot
			if (type & 0x100) {
				if (Dlg->GetCheck(308)) tmpType |= 0x200;
				else (tmpType &= (~0x200));
				}
			desc[0] = 0;
			Undo.SetDisp(cdisp);			Dlg->GetText(307, (unsigned char*)desc, 80);
			Dlg->GetValue(101, &n_sb);		Dlg->GetValue(104, &n_lw);
			Dlg->GetColor(107, &n_col);		Dlg->GetValue(305, &n_err);
			Dlg->GetValue(301, &n_pos.fx);		Dlg->GetValue(303, &n_pos.fy);
			break;
		case 10:
		case 101:	case 104:	case 107:	//these items also redraw the preview button
			Dlg->GetValue(101, &n_sb);		Dlg->GetValue(104, &n_lw);
			Dlg->GetColor(107, &n_col);		prevsym->SetSize(SIZE_ERRBAR_LINE, n_lw);
			prevsym->SetSize(SIZE_ERRBAR, n_sb);
			prevsym->SetColor(COL_ERROR_LINE, n_col);
			Dlg->DoPlot(0L);				res = -1;
			break;
		}
	} while (res < 0);
	if (parent) switch (res) {
	case 1:				//new setting for current error bar only
		undo_flags = CheckNewFloat(&ferr, o_err, n_err, parent, undo_flags);
		undo_flags = CheckNewLFPoint(&fPos, &o_pos, &n_pos, parent, undo_flags);
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		if (undo_flags & UNDO_CONTINUE) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		undo_flags = CheckNewString((unsigned char**)&name, (unsigned char*)name, desc, this, undo_flags);
		undo_flags = CheckNewFloat(&SizeBar, o_sb, n_sb, parent, undo_flags);
		undo_flags = CheckNewFloat(&ErrLine.width, o_lw, n_lw, parent, undo_flags);
		undo_flags = CheckNewDword(&ErrLine.color, ErrLine.color, n_col, parent, undo_flags);
		if (undo_flags & UNDO_CONTINUE) bRet = true;
		break;
	case 2:				//new settings to all error bars of plot
		parent->Command(CMD_SAVE_ERRS, 0L, 0L);
		if (desc[0] || name) parent->Command(CMD_ERRDESC, desc, 0L);
		parent->SetSize(SIZE_ERRBAR, n_sb);
		parent->SetSize(SIZE_ERRBAR_LINE, n_lw);
		parent->SetColor(COL_ERROR_LINE, n_col);
		if (type != tmpType) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		bRet = parent->Command(CMD_ERR_TYPE, (void*)(&tmpType), 0L);
		break;
		}
	else switch (res) {
	case 1:		case 8:				//no parent: its part of a dialog, preview OK
		Dlg->GetValue(101, &SizeBar);			Dlg->GetValue(104, &ErrLine.width);
		Dlg->GetColor(107, &ErrLine.color);		type = tmpType;
//		ferr = n_err;	SizeBar = n_sb;		ErrLine.width = n_lw;
//		ErrLine.color = n_col;	type = tmpType;
		bRet = true;
		break;
	case 9:				//no parent: its part of a dialog, preview Cancel
		bRet = false;
		break;
		}
	CloseDlgWnd(hDlg);	delete Dlg;			free(ErrDlg);
	free(tab1);			free(tab2);			free(tab3);
	delete prevsym;
	return bRet;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// 3D error bars properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* Err3DDlg_Tmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,1,120,10,70,12\n"
	"2,3,,,PUSHBUTTON,-28,120,25,70,12\n"
	"3,4,,,PUSHBUTTON,-2,120,40,70,12\n"
	"4,,5,ISPARENT | CHECKED,GROUP,,,,,\n"
	"5,6,100,ISPARENT | CHECKED,SHEET,2,5,10,110,105\n"
	"6,7,200,ISPARENT,SHEET,3,5,10,110,105\n"
	"7,700,300,TOUCHEXIT | ISPARENT,SHEET,4,5,10,110,105\n"
	"100,+,,,RTEXT,5,25,40,28,8\n"
	".,.,,,INCDECVAL1,6,56,40,33,10\n"
	".,.,,,LTEXT,-3,91,40,20,8\n"
	".,.,,,RTEXT,-29,25,55,28,8\n"
	".,.,,,INCDECVAL1,7,56,55,33,10\n"
	".,.,,,LTEXT,-3,91,55,20,8\n"
	".,.,,,RTEXT,-30,25,70,28,8\n"
	".,,,OWNDIALOG,COLBUTT,8,56,70,25,10\n"
	"200,+,600,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,,,,LTEXT,-27,10,25,70,8\n"
	"300,+,,,RTEXT,-12,15,30,28,8\n"
	".,.,,,EDVAL1,9,46,30,55, 10\n"
	".,.,,,RTEXT,-13,15,45,28,8\n"
	".,.,,,EDVAL1,10,46,45,55,10\n"
	".,.,,,RTEXT,-14,15,60,28,8\n"
	".,.,,,EDVAL1,11,46,60,55,10\n"
	".,.,,,RTEXT,12,15,75,28,8\n"
	".,.,,,EDVAL1,13,46,75,55,10\n"
	".,.,,,LTEXT,14,10,90,70,8\n"
	".,,,,EDTEXT,15,10,100,100,10\n"
	"600,+,,EXRADIO,ODB,16,22,35,,\n"
	".,.,,EXRADIO,ODB,16,47,35,,\n"
	".,.,,EXRADIO,ODB,16,72,35,,\n"
	".,.,,EXRADIO,ODB,16,22,60,,\n"
	".,.,,EXRADIO,ODB,16,47,60,,\n"
	".,.,,EXRADIO,ODB,16,72,60,,\n"
	".,.,,EXRADIO,ODB,16,22,85,,\n"
	".,.,,EXRADIO,ODB,16,47,85,,\n"
	".,,,EXRADIO,ODB,16,72,85,,\n"
	"700,,,LASTOBJ | NOSELECT,ODB,17,140,65,45,45";
	
bool
ErrBar3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_ERRBAR);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TYPE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	void *dyndata[] = {(void*)SDLG_EBAR_TOERR, (void*)tab1, (void*)tab2, (void*)tab3,
		(void*)SDLG_EBAR_CWIDTH, (void*)&SizeBar, (void*)&ErrLine.width, (void *)&ErrLine.color,
		(void*)&fPos.fx, (void*)&fPos.fy, (void*)&fPos.fz, (void*)SDLG_EBAR_ERR, (void*)&ferr,
		(void*)SDLG_EBAR_DESC, (void*)name, (void*)(OD_ErrBarTempl), (void*)(OD_AxisDesc3D)};
	DlgInfo *ErrDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmpType = type;
	double n_sb, o_sb, n_lw, o_lw, n_err, o_err;
	fPOINT3D n_pos, o_pos;
	DWORD n_col, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	bool bRet = false;
	unsigned char desc[80];

	if(!parent) return false;
	if(!(ErrDlg = CompileDialog(Err3DDlg_Tmpl, dyndata))) return false;
	desc[0] = 0;
	if(!(Dlg = new DlgRoot(ErrDlg, data)))return false;
	Dlg->SetCheck(600 + (type & 0x0f), 0L, true);
	if(!(Dlg->GetValue(101, &o_sb))) o_sb = SizeBar;
	if(!(Dlg->GetValue(104, &o_lw))) o_lw = ErrLine.width;
	if(!(Dlg->GetValue(307, &o_err))) o_err = ferr;
	if(!(Dlg->GetValue(301, &o_pos.fx))) o_pos.fx = fPos.fx;
	if(!(Dlg->GetValue(303, &o_pos.fy))) o_pos.fy = fPos.fy;
	if(!(Dlg->GetValue(305, &o_pos.fy))) o_pos.fz = fPos.fz;
	n_sb = o_sb;	n_lw = o_lw;	n_err = o_err; n_pos.fx = o_pos.fx;	n_pos.fy = o_pos.fy;
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_EBAR3D_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_EBAR3D_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 400, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:								//accept for this object
		case 2:								//   or all objects of plot
			desc[0] = 0;
			Undo.SetDisp(cdisp);			Dlg->GetText(309, desc, 80);
			Dlg->GetValue(101, &n_sb);		Dlg->GetValue(104, &n_lw);
			Dlg->GetColor(107, &n_col);		Dlg->GetValue(307, &n_err);
			Dlg->GetValue(301, &n_pos.fx);		Dlg->GetValue(303, &n_pos.fy);
			Dlg->GetValue(305, &n_pos.fz);
			break;
		case 7:								//edit tab
			Dlg->Activate(307, 1);	res = -1;	break;
		case 600:	case 601:	case 602:	case 603:	case 604:
		case 605:	case 606:	case 607:	case 608:
			tmpType = res-600;			res = -1;	break;
			}
		}while (res <0);
	switch (res) {
	case 1:				//new setting for current error bar only
		undo_flags = CheckNewFloat(&ferr, o_err, n_err, parent, undo_flags);
		undo_flags = CheckNewLFPoint3D(&fPos, &o_pos, &n_pos, parent, undo_flags);
		undo_flags = CheckNewLong(&type, type, tmpType, parent, undo_flags);
		if(undo_flags & UNDO_CONTINUE) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		undo_flags = CheckNewString((unsigned char**)&name, (unsigned char*)name, desc, this, undo_flags);
		undo_flags = CheckNewFloat(&SizeBar, o_sb, n_sb, parent, undo_flags);
		undo_flags = CheckNewFloat(&ErrLine.width, o_lw, n_lw, parent, undo_flags);
		undo_flags = CheckNewDword(&ErrLine.color, ErrLine.color, n_col, parent, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		break;
	case 2:				//new settings to all error bars of plot
		parent->Command(CMD_SAVE_ERRS, 0L, 0L);
		if(desc[0] || name) parent->Command(CMD_ERRDESC, desc, 0L);
		parent->SetSize(SIZE_ERRBAR, n_sb);
		parent->SetSize(SIZE_ERRBAR_LINE, n_lw);
		parent->SetColor(COL_ERROR_LINE, n_col);
		if(type != tmpType) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		bRet = parent->Command(CMD_ERR_TYPE, (void*)(& tmpType), 0L);
		break;
		}
	CloseDlgWnd(hDlg);	delete Dlg;			free(ErrDlg);
	free(tab1);			free(tab2);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Arrow properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* ArrowDlg_Dlg_Tmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,-1,115,10,62,12\n"
		".,.,,,PUSHBUTTON,-28,115,25,62,12\n"
		".,.,,,PUSHBUTTON,-2,115,25,62,12\n"
		".,50,5,ISPARENT | CHECKED,GROUP,,,,,\n"
		".,+,100,ISPARENT | CHECKED,SHEET,1,5,10,102,100\n"
		".,., 200,ISPARENT,SHEET,2,5,10,102,100\n"
		".,., 300,ISPARENT,SHEET,3,5,10,102,100\n"
		".,,,TOUCHEXIT,ARROWPREV,22,115,90,62,20\n"
		"50,, 600,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
		"100,+,,, RTEXT,4,15,40,28,8\n"
		".,.,,TOUCHEXIT,INCDECVAL1,5,46,40,30,10\n"
		".,.,,,LTEXT,-3,78,40,20,8\n"
		".,.,,,RTEXT,6,15,52,28,8\n"
		".,.,,TOUCHEXIT,INCDECVAL1,7,46,52,30,10\n"
		".,.,,,LTEXT,-3, 78, 52, 20, 8\n"
		".,.,,,RTEXT,8,15,70,28,8\n"
		".,.,,TOUCHEXIT,INCDECVAL1,9,46,70,30,10\n"
		".,.,,,LTEXT,-3,78,70,20,8\n"
		".,.,,,RTEXT,-11,15,82,28,8\n"
		".,,,TOUCHEXIT | OWNDIALOG,COLBUTT,10,46,82,25,10\n"
		"200,+,,HICOL | TOUCHEXIT,RADIO1,11,15,35,60,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,12,15,50,60,8\n"
		".,.,,HICOL | TOUCHEXIT,RADIO1,13,15,65,60,8\n"
		".,,,HICOL,CHECKBOX,23,15,90,60,8\n"
		"300,+,,,RTEXT,-12,10,30,28,8\n"
		".,.,,,EDVAL1,14,46,30,35,10\n"
		".,.,,,RTEXT,-13,10,42,28,8\n"
		".,.,,,EDVAL1,15,46,42,35,10\n"
		".,.,,,RTEXT,16,10,60,28,8\n"
		".,.,,,EDVAL1,17,46,60,35,10\n"
		".,.,,,RTEXT,-5,10,72,28,8\n"
		".,.,,,EDVAL1,18,46,72,35,10\n"
		".,.,,HICOL,CHECKBOX,19,16,85,70,8\n"
		".,.,,,LTEXT,20,25,93,65,8\n"
		".,.,,,LTEXT,-34,84,30,20,8\n"
		".,.,,,LTEXT,-34,84,42,20,8\n"
		".,.,,,LTEXT,-34,84,60,20,8\n"
		".,,,,LTEXT,-34,84,72,20,8\n"
		"600,+,,TOUCHEXIT,ODB,21,128,47,15,15\n"
		".,.,,TOUCHEXIT,ODB,21,154,47,15,15\n"
		".,.,,TOUCHEXIT,ODB,21,154,62,15,15\n"
		".,,,LASTOBJ | TOUCHEXIT,ODB,21,128,62,15,15";

bool
Arrow::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_ARROW);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TYPE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	Arrow *PrevArr = new Arrow(0L, data, 0L, 0L, type);
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_ARROW_CWIDTH,
		(void*)&cw, (void*)SDLG_ARROW_CLENGTH, (void*)&cl, (void*)SDLG_ARROW_LWIDTH,
		(void*)&LineDef.width, (void *)&LineDef.color, (void*)SDLG_ARROW_LONLY, 
		(void*)SDLG_ARROW_WLINE, (void*)SDLG_ARROW_FILLED, (void*)&pos2.fx, (void*)&pos2.fy,
		(void*)SDLG_ARROW_ORGX, (void*)&pos1.fx, (void*)&pos1.fy, (void*)SDLG_ARROW_COMORG,
		(void*)SDLG_ARROW_TOALL, (void*)OD_DrawOrder, (void*)PrevArr, (void*)SDLG_ARROW_SYMSPACE };
	DlgInfo *ArrowDlg; 
	DlgRoot *Dlg;
	void *hDlg;
	lfPOINT o_pos1, o_pos2, n_pos1, n_pos2;
	double o_cw, o_cl, n_cw, n_cl, o_lw, n_lw;
	int cb, res, undo_level = Undo.level();
	long tmptype = type;
	bool bRet = false;
	DWORD o_col, n_col, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();

	if(!(ArrowDlg = CompileDialog(ArrowDlg_Dlg_Tmpl, dyndata))) return false;
	if (!parent){		//its a dialog!
		}
	else if(!(type & ARROW_UNITS)) {
		ArrowDlg[0].ptype = (void*)SDLG_ARROW_TOARR;
		ArrowDlg[2].y = 40;
		}
	else {								//change '[data]' to current units 
		for(i = 34; i < 38; i++) ArrowDlg[i].ptype = (void*)Units[defs.units()].display;
		}
	if(!(Dlg = new DlgRoot(ArrowDlg, data))) return false;
	PrevArr->SetSize(SIZE_ARROW_CAPWIDTH, cw);
	PrevArr->SetSize(SIZE_ARROW_CAPLENGTH, cl);
	PrevArr->SetSize(SIZE_ARROW_LINE, LineDef.width);
	PrevArr->SetColor(COL_ARROW, LineDef.color);
	PrevArr->type = type;
	if (parent && parent->Id == GO_TERNARYXYZ) Dlg->ShowItem(7, false);
	Dlg->GetValue(301, &o_pos2.fx);		Dlg->GetValue(303, &o_pos2.fy);
	Dlg->GetValue(305, &o_pos1.fx);		Dlg->GetValue(307, &o_pos1.fy);
	Dlg->GetValue(101, &o_cw);			Dlg->GetValue(104, &o_cl);
	Dlg->GetValue(107, &o_lw);			Dlg->GetColor(110, &o_col);
	if (type & ARROW_UNITS) Dlg->ShowItem(203, false);
	else if (type & ARROW_SYMSPACE) Dlg->SetCheck(203, 0L, true);
	if(!parent)Dlg->ShowItem(7, false);
	if(parent && (parent->Id ==  GO_GRAPH || parent->Id == GO_PAGE)) Dlg->ShowItem(50, true);
	switch(type & 0xff) {
	case ARROW_LINE:		Dlg->SetCheck(201, 0L, true);		break;
	case ARROW_TRIANGLE:	Dlg->SetCheck(202, 0L, true);		break;
	default:				Dlg->SetCheck(200, 0L, true);		break;
		}
	if(!parent || (tmptype & ARROW_UNITS)){
		Dlg->ShowItem(2, false);
		Dlg->ShowItem(308, false);		Dlg->ShowItem(309, false);
		}
	if(parent && parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_ARROW_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_ARROW_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 390, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();			res = Dlg->GetResult();
		switch (res) {
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			if(parent) res = ExecDrawOrderButt(parent, this, res);
			}
		switch (res) {
		case 200:	case 201:	case 202:				//arrow styles
		case 101:	case 104:	case 107:	case 110:	//arrow adjustments
		case 8:		//Arrow Preview Button
			if (res == 200 || res == 201 || res == 202) {
				if (res == 200) tmptype = ARROW_NOCAP;
				else if (res == 201) tmptype = ARROW_LINE;
				else if (res == 202) tmptype = ARROW_TRIANGLE;
				}
			Dlg->GetValue(101, &n_cw);			Dlg->GetValue(104, &n_cl);
			Dlg->GetValue(107, &n_lw);			Dlg->GetColor(110, &n_col);
			PrevArr->SetSize(SIZE_ARROW_CAPWIDTH, n_cw);
			PrevArr->SetSize(SIZE_ARROW_CAPLENGTH, n_cl);
			PrevArr->SetSize(SIZE_ARROW_LINE, n_lw);
			PrevArr->SetColor(COL_ARROW, n_col);
			if (Dlg->GetCheck(203)) tmptype |= ARROW_SYMSPACE;
			else tmptype &= ~ARROW_SYMSPACE;
			PrevArr->type = tmptype;
			Dlg->DoPlot(0L);
			res = -1;
			break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);
			Dlg->GetValue(301, &n_pos2.fx);		Dlg->GetValue(303, &n_pos2.fy);
			Dlg->GetValue(305, &n_pos1.fx);		Dlg->GetValue(307, &n_pos1.fy);
			Dlg->GetValue(101, &n_cw);			Dlg->GetValue(104, &n_cl);
			Dlg->GetValue(107, &n_lw);			Dlg->GetColor(110, &n_col);
			if (Dlg->GetCheck(203)) tmptype |= ARROW_SYMSPACE;
			else tmptype &= ~ARROW_SYMSPACE;
			if (!parent){	//its a dialog item
				cl = n_cl;		cw = n_cw;		LineDef.width = n_lw;		LineDef.color = n_col;
				type = (tmptype & 0x0f);
				res = 5;	//skip further processing
				}
			PrevArr->type = tmptype;
			type = tmptype;
			break;
			}
		}while (res <0);
	switch (res) {
	case 1:				//new setting for current arrow
		if(n_pos1.fx != o_pos1.fx || n_pos1.fy != o_pos1.fy ||
			n_pos2.fx != o_pos2.fx || n_pos2.fy != o_pos2.fy){
			Command(CMD_SAVEPOS, 0L, 0L);				undo_flags |= UNDO_CONTINUE;
			memcpy(&pos1, &n_pos1, sizeof(lfPOINT));	memcpy(&pos2, &n_pos2, sizeof(lfPOINT));
			if(parent && !(type & ARROW_UNITS)) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
			}
		if((type & 0x2ff) != (tmptype & 0x2ff)){
			Undo.ValLong(this, &type, undo_flags);
			type &= ~0x2ff;	type |= (tmptype & 0x2ff);	undo_flags |= UNDO_CONTINUE;
			}
		undo_flags = CheckNewFloat(&cw, o_cw, n_cw, this, undo_flags);
		undo_flags = CheckNewFloat(&cl, o_cl, n_cl, this, undo_flags);
		undo_flags = CheckNewFloat(&LineDef.width, o_lw, n_lw, this, undo_flags);
		undo_flags = CheckNewDword(&LineDef.color, o_col, n_col, this, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bModified = true;
		bRet = true;
		break;
	case 2:				//new settings to all arrows of plot
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			parent->Command(CMD_SAVE_ARROWS, 0L, 0L);
			if(Dlg->GetCheck(308)) parent->Command(CMD_ARROW_ORG, &n_pos1, 0L);
			parent->SetSize(SIZE_ARROW_LINE, n_lw);
			parent->SetSize(SIZE_ARROW_CAPWIDTH, n_cw);
			parent->SetSize(SIZE_ARROW_CAPLENGTH, n_cl);
			parent->SetColor(COL_ARROW, n_col);
			parent->Command(CMD_ARROW_TYPE, &tmptype, 0L);
			bRet = true;
			}
		break;
	case 3:								//Cancel
		Undo.SetDisp(cdisp);
		if(Undo.level() > undo_level) {	//restore plot order
			while (Undo.level() > undo_level)	Undo.Restore(false, 0L);
			bRet = true;
			}
		break;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(ArrowDlg);		delete PrevArr;
	free(tab1);			free(tab2);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Box properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* BoxDlg_Dlg_Tmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,3,130,10,55,12\n"
		".,.,,,PUSHBUTTON,-28,130,25,55,12\n"
		".,.,,,PUSHBUTTON,-2,130,40,55,12\n"
		".,,5,CHECKED | ISPARENT,GROUP,,,,,\n"
		"5,+,100,ISPARENT | CHECKED,SHEET,1,5,10,120,115\n"
		".,,300,ISPARENT,SHEET,2,5,10,120,115\n"
		"100,105,,NOSELECT,ODB,4,8,30,90,50\n"
		"105,,109,CHECKED | ISPARENT,GROUP,,,,,\n"
		"109,+,,,LTEXT,5,10,80,45,8\n"
		".,.,,HICOL,RADIO1,6,20,92,25,8\n"
		".,.,,,EDTEXT,7,60,92,25,10\n"
		".,.,,,LTEXT,-3,87,92,20,8\n"
		".,.,,HICOL,RADIO1,8,20,104,25,8\n"
		".,.,,,EDTEXT,9,60,104,25,10\n"
		".,,,,LTEXT,-10,87,104,10,8\n"
		"300,+,,,RTEXT,10,15,40,28,8\n"
		".,.,,,EDVAL1,11,46,40,35,10\n"
		".,.,,,RTEXT,-5,15,52,28,8\n"
		".,.,,,EDVAL1,12,46,52,35,10\n"
		".,.,,,RTEXT,13,15,70,28,8\n"
		".,.,,,EDVAL1,14,46,70,35,10\n"
		".,.,,,RTEXT,-5,15,82,28,8\n"
		".,.,,,EDVAL1,15,46,82,35,10\n"
		".,.,,HIDDEN,RTEXT,16,15,100,28,8\n"
		".,,,HIDDEN | LASTOBJ,EDVAL1,17,46,100,35,10";

bool
Box::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_SIZECOL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	char *sTxt1 = (char*)malloc(50);
	char *sTxt2 = (char*)malloc(50);
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)SDLG_BOX_TOBOX,  (void*)OD_filldef,
		(void*)SDLG_BOX_BWIDTH, (void*)SDLG_BOX_FIX, (void*)(sTxt1+1), (void*)SDLG_BOX_REL,
		(void*)(sTxt2+1), (void*)SDLG_BOX_PT1X, (void*)&pos1.fx, (void*)&pos1.fy,
		(void*)SDLG_BOX_PT2X, (void*)&pos2.fx, (void*)&pos2.fy, (void*)SDLG_BOX_WIDTH, (void*)&size};
	DlgRoot *Dlg;
	DlgInfo *BoxDlg; 
	void *hDlg;
	int cb, res;
	long tmpType = type;
	static FillDEF newFill;
	LineDEF newLine, newHatchLine;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	lfPOINT n_pos;
	double tmpVal, o_size, n_size;
	bool bRet = false;
	GraphObj *ppar = parent;

	if(!parent) return false;
	if (type & BAR_RELWIDTH) {
		WriteNatFloatToBuff(sTxt2, size);
		WriteNatFloatToBuff(sTxt1, DefSize(SIZE_BAR));
		}
	else {
		WriteNatFloatToBuff(sTxt1, size);
		rlp_strcpy(sTxt2, 20, (char*)" 50");
		}
	if (!(BoxDlg = CompileDialog(BoxDlg_Dlg_Tmpl, dyndata))) return false;
	memcpy(&newHatchLine, &Hatchline, sizeof(LineDEF));
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&Outline, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
	Dlg = new DlgRoot(BoxDlg, data);
	if(type & BAR_WIDTHDATA) {
		Dlg->ShowItem(105, false);
		Dlg->ShowItem(308, true);		Dlg->ShowItem(309, true);
		}
	if(type & BAR_RELWIDTH) Dlg->SetCheck(113, 0L, true);
	else Dlg->SetCheck(110, 0L, true);
	Dlg->GetValue(309, &o_size);
	if (parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BOX_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BOX_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 410, 310, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:								//accept for this object
		case 2:								//   or all objects of plot
			Undo.SetDisp(cdisp);
			OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
			if(newFill.hatch) memcpy(&newHatchLine, newFill.hatch, sizeof(LineDEF));
			newFill.hatch = &newHatchLine;
			if(type & BAR_WIDTHDATA) Dlg->GetValue(309, &n_size);
			else {
				if(Dlg->GetCheck(113)) {
					tmpType |= BAR_RELWIDTH;
					Dlg->GetValue(114, &n_size);
					}
				else {
					tmpType &= ~BAR_RELWIDTH;
					Dlg->GetValue(111, &n_size);
					}
				}
			break;
			}
		}while (res < 0);
	switch (res) {
	case 1:				//new setting for current box
		if(cmpLineDEF(&Outline, &newLine)) {
			Undo.Line(parent, &Outline, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&Outline, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&Hatchline, &newHatchLine)) {
			Undo.Line(parent, &Hatchline, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&Hatchline, &newHatchLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&Fill, &newFill)) {
			Undo.Fill(parent, &Fill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&Fill, &newFill, sizeof(FillDEF));
			}
		Fill.hatch = &Hatchline;
		if(Dlg->GetValue(301, &tmpVal)) n_pos.fx = tmpVal;
		else n_pos.fx = pos1.fx;
		if(Dlg->GetValue(303, &tmpVal)) n_pos.fy = tmpVal;
		else n_pos.fy = pos1.fy;
		undo_flags = CheckNewLFPoint(&pos1, &pos1, &n_pos, parent, undo_flags);
		if(Dlg->GetValue(305, &tmpVal)) n_pos.fx = tmpVal;
		else n_pos.fx = pos2.fx;
		if(Dlg->GetValue(307, &tmpVal)) n_pos.fy = tmpVal;
		else n_pos.fy = pos2.fy;
		undo_flags = CheckNewLFPoint(&pos2, &pos2, &n_pos, parent, undo_flags);
		undo_flags = CheckNewFloat(&size, o_size, n_size, parent, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		break;
	case 2:				//new settings to all boxes of plot
		Undo.ValLong(this, &type, 0L);			//dummy: all following have UNDO_CONTINUE
		undo_flags |= UNDO_CONTINUE;
		if(parent->parent && parent->parent->Id == GO_STACKBAR) ppar = parent->parent;
		Undo.ValLong(this, &type, 0L);			//dummy: all following have UNDO_CONTINUE
		ppar->Command(CMD_SAVE_BARS_CONT, 0L, 0L);
		ppar->Command(CMD_BOX_TYPE, (void*)(& tmpType), 0L);
		ppar->SetSize(SIZE_BOX_LINE, newLine.width);
		ppar->SetSize(SIZE_BOX, n_size);
		parent->SetColor(COL_BOX_LINE, newLine.color);
		parent->Command(CMD_BOX_FILL, &newFill, 0L);
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);					delete Dlg;
	free(BoxDlg);						free(tab1);	
	free(tab2);		free(sTxt1);		free(sTxt2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Whisker properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *WhiskerDlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,1,110,10,64,12\n"
	".,.,,,PUSHBUTTON,-28,110,25,64,12\n"
	".,.,,,PUSHBUTTON,-2,110,40,64,12\n"
	".,,5,ISPARENT | CHECKED,GROUP,,,,,\n"
	"5,+,100,ISPARENT | CHECKED,SHEET,2, 5,10,100,100\n"
	".,.,500,ISPARENT,SHEET,3,5,10,100,100\n"
	".,,300,ISPARENT,SHEET,4,5,10,100,100\n"
	"100,+,,,RTEXT,5,15,40,28,8\n"
	".,.,,,INCDECVAL1,6,46,40,33,10\n"
	".,.,,,LTEXT,-3,81,40,20,8\n"
	".,.,,,RTEXT,7,15,55,28,8\n"
	".,.,,,INCDECVAL1,8,46,55,33,10\n"
	".,.,,,LTEXT,-3,81,55,20,8\n"
	".,.,,,RTEXT,9,15,70,28,8\n"
	".,,,OWNDIALOG,COLBUTT,10,46,70,25,10\n"
	"300,+,,,RTEXT,11,15,30,28,8\n"
	".,.,,,EDVAL1,12,46,30,35,10\n"
	".,.,,,RTEXT,-5,15,42,28,8\n"
	".,.,,,EDVAL1,13,46,42,35,10\n"
	".,.,,,RTEXT,14,15,55,28,8\n"
	".,.,,,EDVAL1,15,46,55,35,10\n"
	".,.,,,RTEXT,-5,15,67,28,8\n"
	".,.,,,EDVAL1,16,46,67,35,10\n"
	".,.,,,LTEXT,17,10,85,70,8\n"
	".,,,,EDTEXT,18,10,95,80,10\n"
	"500,+,,EXRADIO,ODB,19,32,40,18,18\n"
	".,.,,EXRADIO,ODB,19,14,40,18,18\n"
	".,.,,EXRADIO,ODB,19,50,40,18,18\n"
	".,.,,EXRADIO,ODB,19,68,40,18,18\n"
	".,,,LASTOBJ,LTEXT,-27,14,30,30,9";

bool
Whisker::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_WHISKER);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	void *dyndata[] = {(void*)SDLG_WHISKER_TOOBJ, (void*)tab1,
		(void*)tab3, (void*)tab2, (void*)SDLG_EBAR_CWIDTH, (void*)&size, (void*)SDLG_WHISKER_LWIDTH,
		(void*)&LineDef.width, (void*)SDLG_WHISKER_LCOL, (void *)&LineDef.color, (void*)SDLG_WHISKER_P1X,
		(void*)&pos1.fx, (void*)&pos1.fy, (void*)SDLG_WHISKER_P2X, (void*)&pos2.fx, (void*)&pos2.fy,
		(void*)SDLG_EBAR_DESC, (void*)name, (void*)(OD_WhiskerTempl)};
	DlgInfo *ErrDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmp_type;
	DWORD undo_flags = 0L, n_col = LineDef.color;
	anyOutput *cdisp = Undo.Cdisp();
	lfPOINT n_pos;
	double tmpVal, o_size, n_size, o_lw, n_lw;
	bool bRet = false;
	unsigned char desc[80];

	if(!parent) return false;
	if(!(ErrDlg = CompileDialog(WhiskerDlgTmpl, dyndata))) return false;
	desc[0] = 0;	tmp_type = type;
	Dlg = new DlgRoot(ErrDlg, data);
	Dlg->SetCheck(500 + (tmp_type & 0x03), 0L, true);
	Dlg->GetValue(101, &o_size);		Dlg->GetValue(104, &o_lw);
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_WHISKER_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_WHISKER_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 380, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 500:	case 501:	case 502:	case 503:
			tmp_type &= ~0x0f;	tmp_type |= (res-500);	res = -1;
			break;
		case 1:								//accept for this object
		case 2:								//   or all objects of plot
			Undo.SetDisp(cdisp);			Dlg->GetValue(101, &n_size);
			Dlg->GetValue(104, &n_lw);		Dlg->GetColor(107, &n_col);
			desc[0] = 0;					Dlg->GetText(309, desc, 80);
			break;
			}
		}while (res <0);
	switch (res) {
	case 1:				//new setting for current whisker
		undo_flags = CheckNewString((unsigned char**)&name, (unsigned char*)name, desc, this, undo_flags);
		if(Dlg->GetValue(301, &tmpVal)) n_pos.fx = tmpVal;
		else n_pos.fx = pos1.fx;
		if(Dlg->GetValue(303, &tmpVal)) n_pos.fy = tmpVal;
		else n_pos.fy = pos1.fy;
		undo_flags = CheckNewLFPoint(&pos1, &pos1, &n_pos, parent, undo_flags);
		if(Dlg->GetValue(305, &tmpVal)) n_pos.fx = tmpVal;
		else n_pos.fx = pos2.fx;
		if(Dlg->GetValue(307, &tmpVal)) n_pos.fy = tmpVal;
		else n_pos.fy = pos2.fy;
		undo_flags = CheckNewLFPoint(&pos2, &pos2, &n_pos, parent, undo_flags);
		undo_flags = CheckNewFloat(&size, o_size, n_size, parent, undo_flags);
		undo_flags = CheckNewLong(&type, type, tmp_type, parent, undo_flags);
		undo_flags = CheckNewFloat(&LineDef.width, o_lw, n_lw, parent, undo_flags);
		undo_flags = CheckNewDword(&LineDef.color, LineDef.color, n_col, parent, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		break;
	case 2:				//new settings to all whiskers of plot
		parent->Command(CMD_SAVE_ERRS, 0L, 0L);
		if(desc[0] || name) parent->Command(CMD_ERRDESC, desc, 0L);
		parent->SetSize(SIZE_WHISKER, n_size);
		parent->SetSize(SIZE_WHISKER_LINE, n_lw);
		parent->SetColor(COL_WHISKER, n_col);
		parent->Command(CMD_WHISKER_STYLE, &tmp_type, 0L);
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(ErrDlg);
	free(tab1);			free(tab2);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Drop line properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *DropLineDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,3,150,10,50,12\n"
	".,.,,,PUSHBUTTON,-28,150,25,50,12\n"
	".,.,,,PUSHBUTTON,-2,150,40,50,12\n"
	".,300,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,139,120\n"
	".,,200,ISPARENT,SHEET,2,5,10,139,120\n"
	"100,,,NOSELECT,ODB,4,10,38,130,100\n"
	"200,+,,,RTEXT,-12,35,40,28,8\n"
	".,.,,,EDVAL1,5,76,40,35,10\n"
	".,.,,,RTEXT,-13,35,52,28,8\n"
	".,.,,,EDVAL1,6,76,52,35,10\n"
	".,,250,CHECKED | ISPARENT,GROUPBOX,7,10,70,129,50\n"
	"250,+,,,RTEXT,8,15,80,28,8\n"
	".,.,,HICOL,CHECKBOX,-20,46,80,35,8\n"
	".,.,,HICOL,CHECKBOX,-21,46,90,35,8\n"
	".,.,,HICOL,CHECKBOX,-25,46,100,35,8\n"
	".,.,,HICOL,CHECKBOX,-22,86,80,35,8\n"
	".,.,,HICOL,CHECKBOX,-23,86,90,35,8\n"
	".,,,HICOL | LASTOBJ,CHECKBOX,-24,86,100,35,8";

bool
DropLine::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	DlgInfo *LineDlg;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)SDLG_DROPL_TOLINE,
		(void*)OD_linedef, (void*)&fPos.fx, (void*)&fPos.fy, (void*)SDLG_DROPL_SHAPE,
		(void*)SDLG_DROPL_LINETO};
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmptype;
	bool bRet = false;
	LineDEF newLine;
	DWORD undo_flags = 0L;
	anyOutput * cdisp = Undo.Cdisp();
	lfPOINT o_pos, n_pos;

	if(!parent) return false;
	if(!(LineDlg = CompileDialog(DropLineDlg_DlgTmpl, dyndata))) return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
	Dlg = new DlgRoot(LineDlg, data);
	Dlg->SetCheck(251, 0L, type & DL_LEFT ? true : false);
	Dlg->SetCheck(252, 0L, type & DL_RIGHT ? true : false);
	Dlg->SetCheck(253, 0L, type & DL_YAXIS ? true : false);
	Dlg->SetCheck(254, 0L, type & DL_TOP ? true : false);
	Dlg->SetCheck(255, 0L, type & DL_BOTTOM ? true : false);
	Dlg->SetCheck(256, 0L, type & DL_XAXIS ? true : false);
	Dlg->GetValue(201, &o_pos.fx);		Dlg->GetValue(203, &o_pos.fy);
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DROPL_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DROPL_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 425, 320, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:							//this line
		case 2:							//all lines of plot
			Undo.SetDisp(cdisp);		tmptype = 0;
			if(Dlg->GetCheck(251)) tmptype |= DL_LEFT;
			if(Dlg->GetCheck(252)) tmptype |= DL_RIGHT;
			if(Dlg->GetCheck(253)) tmptype |= DL_YAXIS;
			if(Dlg->GetCheck(254)) tmptype |= DL_TOP;
			if(Dlg->GetCheck(255)) tmptype |= DL_BOTTOM;
			if(Dlg->GetCheck(256)) tmptype |= DL_XAXIS;
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			break;
			}
		}while (res < 0);
	if(res == 1){						//Apply to line
		if(cmpLineDEF(&LineDef, &newLine)) {
			Undo.Line(this, &LineDef, undo_flags);
			memcpy(&LineDef, &newLine, sizeof(LineDEF));
			undo_flags |= UNDO_CONTINUE;
			}
		Dlg->GetValue(201, &n_pos.fx);		Dlg->GetValue(203, &n_pos.fy);
		undo_flags = CheckNewLFPoint(&fPos, &o_pos, &n_pos, this, undo_flags);
		undo_flags = CheckNewLong(&type, type, tmptype, this, undo_flags);
		if (undo_flags & UNDO_CONTINUE) bModified = true;
		bRet = true;
		}
	else if(res == 2) {					//Apply to plot
		parent->Command(CMD_SAVE_DROPLINES, 0L, 0L);
		parent->Command(CMD_DL_LINE, (void*)&newLine, 0L);
		parent->Command(CMD_DL_TYPE, (void*)(&tmptype), 0L);
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(LineDlg);
	free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Drop line properties: DropLine 3D
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *DropLine3D_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,3,150,10,50,12\n"
	".,.,,,PUSHBUTTON,-28,150,25,50,12\n"
	".,.,,,PUSHBUTTON,-2,150,40,50,12\n"
	".,300,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,139,120\n"
	".,,200,ISPARENT,SHEET,2,5,10,139,120\n"
	"100,,0,NOSELECT,ODBUTTON,4,10,38,130,100\n"
	"200,+,,,RTEXT,-12,35,30,28,8\n"
	".,.,,,EDVAL1,5,76,30,35,10\n"
	".,.,,,RTEXT,-13,35,42,28,8\n"
	".,.,,,EDVAL1,6,76,42,35,10\n"
	".,.,,,RTEXT,-14,35,54,28,8\n"
	".,240,,,EDVAL1,7,76,54,35,10\n"
	"240,.,250,CHECKED | ISPARENT,GROUPBOX,8,10,70,129,50\n"
	"250,+,,,RTEXT,9,15,80,28,8\n"
	".,.,,HICOL,CHECKBOX,-23,46,80,35,8\n"
	".,.,,HICOL,CHECKBOX,-22,86,80,35,8\n"
	".,.,,HICOL,CHECKBOX,10,46,90,35,8\n"
	".,.,,HICOL,CHECKBOX,11,86,90,35,8\n"
	".,.,,HICOL,CHECKBOX,-20,46,100,35,8\n"
	".,,,HICOL | LASTOBJ,CHECKBOX,-21,86,100,35,8";

bool
DropLine3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	DlgInfo *LineDlg;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)SDLG_DROPL_TOLINE,
		(void*)OD_linedef, (void*)&fPos.fx, (void*)&fPos.fy, (void*)&fPos.fz,
		(void*)SDLG_DROPL_SHAPE, (void*)SDLG_DROPL_LINETO, (void*)SDLG_DROPL3D_BACK,
		(void*)SDLG_DROPL3D_FRONT};
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	long tmptype;
	bool bRet = false;
	LineDEF newLine;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	fPOINT3D o_pos, n_pos;

	if(!parent) return false;
	if(!(LineDlg = CompileDialog(DropLine3D_DlgTmpl, dyndata))) return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	Dlg = new DlgRoot(LineDlg, data);
	for(i = 0; i < 6; i++) {
		Dlg->SetCheck(251+i, 0L, (type & (1<<i)) ? true : false);
		}
	Dlg->GetValue(201, &o_pos.fx);		Dlg->GetValue(203, &o_pos.fy);
	Dlg->GetValue(205, &o_pos.fz);
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DROPL_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_DROPL_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 415, 300, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:							//this line
		case 2:							//all lines of plot
			Undo.SetDisp(cdisp);
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			for(i = tmptype = 0; i < 6; i++) {
				tmptype |= Dlg->GetCheck(251+i) ? 1<<i : 0;
				}
			break;
			}
		}while (res < 0);
	if(res == 1){						//Apply to line
		if(cmpLineDEF(&Line, &newLine)) {
			Undo.Line(this, &Line, undo_flags);
			memcpy(&Line, &newLine, sizeof(LineDEF));
			undo_flags |= UNDO_CONTINUE;
			}
		Dlg->GetValue(201, &n_pos.fx);		Dlg->GetValue(203, &n_pos.fy);
		Dlg->GetValue(205, &n_pos.fz);
		if(n_pos.fx != o_pos.fx || n_pos.fy != o_pos.fy || n_pos.fz != o_pos.fz) {
			Undo.ValLFP3D(this, &fPos, undo_flags);
			memcpy(&fPos, &n_pos, sizeof(fPOINT3D));
			undo_flags |= UNDO_CONTINUE;
			}
		undo_flags = CheckNewLong(&type, type, tmptype, this, undo_flags);
		if (undo_flags & UNDO_CONTINUE) bModified = true;
		bRet = true;
		}
	else if(res == 2) {					//Apply to plot
		parent->Command(CMD_SAVE_DROPLINES, 0L, 0L);
		parent->Command(CMD_DL_LINE, (void*)&newLine, 0L);
		parent->Command(CMD_DL_TYPE, (void*)(&tmptype), 0L);
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(LineDlg);
	free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// sphere (ball) properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *BallDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,3,110,10,50,12\n"
	".,.,,,PUSHBUTTON,-28,110,25,50,12\n"
	".,.,,,PUSHBUTTON,-2,110,40,50,12\n"
	".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,100,80\n"
	".,, 200, ISPARENT,SHEET,2,5,10,100,80\n"
	"100,+,,,RTEXT,4,15,35,28,8\n"
	".,.,,,INCDECVAL1,5,46,35,33,10\n"
	".,.,,,LTEXT,0,80,35,15,8\n"
	".,.,,,RTEXT,6,15,47,28,8\n"
	".,.,,,INCDECVAL1,7,46,47,33,10\n"
	".,.,,,LTEXT,-3,80,47,15,8\n"
	".,.,,,RTEXT,8,15,59,28,8\n"
	".,.,,OWNDIALOG,COLBUTT,9,46,59,25,10\n"
	".,.,,,RTEXT,10,15,71,28,8\n"
	".,,,OWNDIALOG,SHADE3D,11,46,71,25,10\n"
	"200,+,,,RTEXT,-12,15,35,28,8\n"
	".,.,,,EDVAL1,12,46,35,35,10\n"
	".,.,,,RTEXT,-13,15,47,28,8\n"
	".,.,,,EDVAL1,13,46,47,35,10\n"
	".,.,,,RTEXT,-14,15,59,28,8\n"
	".,.,,LASTOBJ,EDVAL1,14,46,59,35,10";

bool
Sphere::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_BALL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	static FillDEF newFill;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)SDLG_BALL_TOBALL,
		(void*)SDLG_BALL_SIZE, (void*)&size, (void*)SDLG_BALL_LWIDTH, (void*)&Line.width,
		(void*)SDLG_BALL_LCOLOR, (void *)&Line.color, (void*)SDLG_BALL_FCOLOR, 
		(void*)&newFill, (void*)&fPos.fx, (void*)&fPos.fy, (void*)&fPos.fz};
	DlgInfo *BallDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	bool bRet = false, bContinue = false;
	fPOINT3D n_pos, o_pos;
	DWORD new_lcolor = Line.color, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	double o_size, n_size, o_lsize, n_lsize;

	if(!parent) return false;
	if(!(BallDlg = CompileDialog(BallDlg_DlgTmpl, dyndata))) return false;
	switch(type) {
		case 1: BallDlg[8].ptype = (void*)SDLG_BALL_XDATA;		break;
		case 2: BallDlg[8].ptype = (void*)SDLG_BALL_YDATA;		break;
		case 3: BallDlg[8].ptype = (void*)SDLG_BALL_ZDATA;		break;
		default: BallDlg[8].ptype = (void*)Units[defs.units()].display;
			break;
		}
	memcpy(&newFill, &Fill, sizeof(FillDEF));
	Dlg = new DlgRoot(BallDlg, data);
	Dlg->GetValue(201, &o_pos.fx);			Dlg->GetValue(203, &o_pos.fy);
	Dlg->GetValue(205, &o_pos.fz);			Dlg->GetValue(101, &o_size);
	Dlg->GetValue(104, &o_lsize);
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BALL_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BALL_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 355, 240, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			bContinue = false;
			break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);			Dlg->GetColor(107, &new_lcolor);
			Dlg->GetValue(101, &n_size);	Dlg->GetValue(104, &n_lsize);
			break;
		case 109:
			Dlg->DoPlot(0L);				bContinue = true;
			break;
			}
		}while (res < 0);
	if(res == 1){
		Dlg->GetValue(201, &n_pos.fx);		Dlg->GetValue(203, &n_pos.fy);
		Dlg->GetValue(205, &n_pos.fz);
		if(n_pos.fx != o_pos.fx || n_pos.fy != o_pos.fy || n_pos.fz != o_pos.fz) {
			Undo.ValLFP3D(this, &fPos, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&fPos, &n_pos, sizeof(fPOINT3D));	parent->Command(CMD_MRK_DIRTY, 0L, 0L);
			}
		undo_flags = CheckNewFloat(&size, o_size, n_size, this, undo_flags);
		undo_flags = CheckNewFloat(&Line.width, o_lsize, n_lsize, this, undo_flags);
		undo_flags = CheckNewDword(&Line.color, Line.color, new_lcolor, this, undo_flags);
		undo_flags = CheckNewDword(&Fill.color, Fill.color, newFill.color, this, undo_flags);
		undo_flags = CheckNewDword(&Fill.color2, Fill.color2, newFill.color2, this, undo_flags);
		undo_flags = CheckNewInt(&Fill.type, Fill.type, newFill.type, this, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = bModified = true;
		}
	else if(res == 2) {
		parent->Command(CMD_SAVE_SYMBOLS, 0L, 0L);
		parent->Command(CMD_SYM_FILL, &newFill, 0L);
		parent->SetSize(SIZE_SYMBOL, n_size);
		parent->SetSize(SIZE_SYM_LINE, n_lsize);
		parent->SetColor(COL_SYM_LINE, new_lcolor);
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;					free(BallDlg);
	free(tab1);					free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// properties of 3D plane
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *Plane3D_DlgTmpl = (char*)
	"1,+,,DEFAULT, PUSHBUTTON,1,115,10,55,12\n"
	".,.,,,PUSHBUTTON,-28,115,25,55,12\n"
	".,.,,,PUSHBUTTON,-2,115,40,55,12\n"
	".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,,100,ISPARENT | CHECKED,SHEET,2,5,10,105,85\n"
	"100,+,,,RTEXT,3,15,30,36,8\n"
	".,.,,,INCDECVAL1,4,53,30,33,10\n"
	".,.,,,LTEXT,-3,89,30,15,8\n"
	".,.,,,RTEXT,5,15,42,36,8\n"
	".,.,,OWNDIALOG,COLBUTT,6,53,42,25,10\n"
	".,.,,,RTEXT,7,15,54,36,8\n"
	".,200,,OWNDIALOG,SHADEPLANE,8,53,54,25,10\n"
	"200,,201,CHECKED,GROUP,,,,,\n"
	"201,+,,,RTEXT,9,15,66,36,8\n"
	".,.,,,INCDECVAL1,10,53,66,33,10\n"
	".,.,,,LTEXT,-10,86,66,5,8\n"
	".,.,,,RTEXT,11,15,78,36,8\n"
	".,.,,,EDVAL1,12,53,78,25,10\n"
	".,,,LASTOBJ,LTEXT,13,80,78,40,8";

bool
Plane3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_PLANE);
	double rb_width = 0.0, rb_z = 0.0;
	static FillDEF newFill;
	LineDEF newLine;
	void *dyndata[] = {(void*)SDLG_PLANE3D_TOPLANE, (void*)tab1, (void*)SDLG_PLANE3D_LWIDTH,
		(void*)&Line.width, (void*)SDLG_PLANE3D_LCOLOR, (void *)&Line.color,
		(void*)SDLG_PLANE3D_FCOLOR, (void*)&newFill, (void*)SDLG_PLANE3D_RWIDTH,
		(void*)&rb_width, (void*)SDLG_PLANE3D_RPOS, (void*)&rb_z,  (void*)SDLG_PLANE3D_ZDATA};
	DlgInfo *PlaneDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	bool bRet = false;
	DWORD new_lcolor = Line.color, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	double o_lsize, n_lsize, o_rbw, n_rbw, o_rbz, n_rbz;

	if(!parent) return false;
	if(!(PlaneDlg = CompileDialog(Plane3D_DlgTmpl, dyndata))) return false;
	if (parent->Id == GO_POLYGON3D) PlaneDlg[2].y = 25;
	if(parent->Id == GO_GRID3D) return parent->PropertyDlg();
	memcpy(&newFill, &Fill, sizeof(FillDEF));
	if(parent->Id == GO_RIBBON && parent->type == 1) {
		rb_width = parent->GetSize(SIZE_CELLWIDTH) *100.0;
		rb_z = parent->GetSize(SIZE_ZPOS);
		}
	Dlg = new DlgRoot(PlaneDlg, data);
	if (parent->Id == GO_POLYGON3D) {
		Dlg->ShowItem(2, false);		Dlg->ShowItem(200, false);
		}
	Dlg->GetValue(101, &o_lsize);		Dlg->GetValue(202, &o_rbw);
	Dlg->GetValue(205, &o_rbz);
	if (parent && ((parent->Id == GO_RIBBON && parent->type > 1) || parent->Id == GO_GRID3D))
		Dlg->ShowItem(200, false);	
	if(parent->Id == GO_RIBBON && parent->type >2) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_PLANE3D_HD3);
	else if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_PLANE3D_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_PLANE3D_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 375, 256, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:		case 2:
			Undo.SetDisp(cdisp);				Dlg->GetColor(104, &new_lcolor);
			Dlg->GetValue(101, &n_lsize);		Dlg->GetValue(202, &n_rbw);
			Dlg->GetValue(205, &n_rbz);			newLine.color = new_lcolor;
			newLine.width = n_lsize;
			break;
			}
		}while (res < 0);
	if(res == 1 && parent->Id != GO_POLYGON3D){
		undo_flags = CheckNewFloat(&Line.width, o_lsize, n_lsize, this, undo_flags);
		undo_flags = CheckNewDword(&Line.color, Line.color, new_lcolor, this, undo_flags);
		undo_flags = CheckNewDword(&Fill.color, Fill.color, newFill.color, this, undo_flags);
		undo_flags = CheckNewDword(&Fill.color2, Fill.color2, newFill.color2, this, undo_flags);
		undo_flags = CheckNewInt(&Fill.type, Fill.type, newFill.type, this, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bRet = true;
		}
	if (res == 1 && parent->Id == GO_POLYGON3D){
		if (Fill.color != newFill.color || Fill.color2 != newFill.color2 || Fill.type != newFill.type ||
			Fill.scale != newFill.scale || Line.color != new_lcolor || o_lsize != n_lsize) {
			parent->Command(CMD_SAVE_PG, 0L, 0L);
			parent->Command(CMD_PG_FILL, &newFill, 0L);
			parent->Command(CMD_SET_LINE, &newLine, 0L);
			}
		bRet = true;
		}
	else if(res == 2) {
		parent->Command(CMD_SAVE_SYMBOLS, 0L, 0L);
		parent->Command(CMD_SYM_FILL, &newFill, 0L);
		parent->SetSize(SIZE_SYM_LINE, n_lsize);
		parent->SetColor(COL_POLYLINE, new_lcolor);
		if(parent->Id == GO_RIBBON && parent->type == 1) {
				parent->SetSize(SIZE_CELLWIDTH, n_rbw/100.0);
				parent->SetSize(SIZE_ZPOS, n_rbz);
				}
		bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;
	free(PlaneDlg);			free(tab1);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// properties of 3D column
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *ColumnDlg_DlgTmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,3,130,10,65,12\n"
		".,.,,,PUSHBUTTON,-28,130,25,65,12\n"
		".,.,,,PUSHBUTTON,-2,130,40,65,12\n"
		".,,5,CHECKED | ISPARENT,GROUP,,,,,\n"
		".,7,100,ISPARENT | CHECKED,SHEET,1,5,10,120,110\n"
		"7,,300,ISPARENT,SHEET,2,5,10,120,110\n"
		"100,+,,,RTEXT,4,18,30,40,8\n"
		".,.,,,EDVAL1,5,60,30,25,10\n"
		".,.,,,LTEXT,-3,87,30,20,8\n"
		".,.,,, RTEXT,6,18,42,40,8\n"
		".,.,,OWNDIALOG,COLBUTT,7,60,42,25,10\n"
		".,.,,,RTEXT,8,18,54,40,8\n"
		".,.,,OWNDIALOG,SHADE3D,9, 60, 54, 25, 10\n"
		".,.,,,RTEXT,10,18,74,40,8\n"
		".,.,,,EDVAL1,11, 60, 74, 25, 10\n"
		".,.,,,LTEXT,-3,87,74,20,8\n"
		".,.,,,RTEXT,12,18,86,40,8\n"
		".,.,,,EDVAL1,13,60,86,25,10\n"
		".,,,,LTEXT,-3,87,86,20,8\n"
		"300,+,,,RTEXT,-12,10,30,35,8\n"
		".,.,,,EDVAL1,14,50,30,30,10\n"
		".,.,,,RTEXT,15,10,75,35,8\n"
		".,.,,,EDVAL1,16,50,75,30,10\n"
		".,.,,,RTEXT,-14,10,42,35,8\n"
		".,.,,,EDVAL1,17,50,42,30,10\n"
		".,.,,,RTEXT,18,10,87,35,8\n"
		".,.,,,EDVAL1,19,50,87,30,10\n"
		".,.,,,LTEXT,-34,84,30,20,8\n"
		".,.,,,LTEXT,-34,84,75,20,8\n"
		".,.,,,LTEXT,-34,84,42,20,8\n"
		".,,,LASTOBJ,LTEXT,-34,84,87,20,8";

bool
Brick::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_SIZECOL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	static FillDEF newFill;
	DlgInfo *ColumnDlg;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)SDLG_BRICK_TOCOLUMN,
		(void*)SDLG_BRICK_LWIDTH, (void*)&Line.width, (void*)SDLG_BRICK_LCOLOR,
		(void *)&Line.color, (void*)SDLG_BRICK_FCOLOR, (void*)&newFill,
		(void*)SDLG_BRICK_CWIDTH, (void*)&width, (void*)SDLG_BRICK_CDEPTH, (void*)&depth,
		(void*)&fPos.fx, (void*)SDLG_BRICK_YBASE, (void*)&fPos.fy, (void*)&fPos.fz,
		(void*)SDLG_BRICK_HEIGHT, (void*)&height};
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	bool bRet = false;
	DWORD col1 = Line.color, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	double o_lw = Line.width, n_lw, o_height = height, n_height, o_width = width, n_width;
	double o_depth = depth, n_depth;
	fPOINT3D o_pos, n_pos;

	if(!parent) return false;
	if(!(ColumnDlg = CompileDialog(ColumnDlg_DlgTmpl, dyndata))) return false;
	memcpy(&newFill, &Fill, sizeof(FillDEF));
	Dlg = new DlgRoot(ColumnDlg, data);
	Dlg->GetValue(101, &o_lw);		Dlg->GetValue(301, &o_pos.fx);
	Dlg->GetValue(303, &o_pos.fy);	Dlg->GetValue(305, &o_pos.fz);
	Dlg->GetValue(307, &o_height);	Dlg->GetValue(108, &o_width);
	Dlg->GetValue(111, &o_depth);
	memcpy(&n_pos, &o_pos, sizeof(fPOINT3D));
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BRICK_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_BRICK_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 420, 300, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:		case 2:
			Undo.SetDisp(cdisp);			Dlg->GetColor(104, &col1);
			Dlg->GetValue(101, &n_lw);		Dlg->GetValue(301, &n_pos.fx);
			Dlg->GetValue(303, &n_pos.fy);	Dlg->GetValue(305, &n_pos.fz);
			Dlg->GetValue(307, &n_height);	Dlg->GetValue(108, &n_width);
			Dlg->GetValue(111, &n_depth);
			break;
			}
		}while (res < 0);
	switch (res) {
	case 1:				//new setting for current column only
		undo_flags = CheckNewFloat(&Line.width, o_lw, n_lw, this, undo_flags);
		undo_flags = CheckNewDword(&Line.color, Line.color, col1, this, undo_flags);
		undo_flags = CheckNewDword(&Fill.color, Fill.color, newFill.color, this, undo_flags);
		undo_flags = CheckNewDword(&Fill.color2, Fill.color2, newFill.color2, this, undo_flags);
		undo_flags = CheckNewInt(&Fill.type, Fill.type, newFill.type, this, undo_flags);
		if(o_pos.fx != n_pos.fx || o_pos.fy != n_pos.fy || o_pos.fz != n_pos.fz) {
			Undo.ValLFP3D(this, &fPos, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&fPos, &n_pos, sizeof(fPOINT3D));	parent->Command(CMD_MRK_DIRTY, 0L, 0L);
			}
		undo_flags = CheckNewFloat(&height, o_height, n_height, this, undo_flags);
		if(o_height != n_height) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		undo_flags = CheckNewFloat(&width, o_width, n_width, this, undo_flags);
		undo_flags = CheckNewFloat(&depth, o_depth, n_depth, this, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bModified = bRet = true;
		break;
	case 2:
		parent->Command(CMD_SAVE_BARS, 0L, 0L);
		parent->Command(CMD_MRK_DIRTY, 0L, 0L);
		parent->SetSize(SIZE_BAR_BASE, n_pos.fy);
		parent->SetSize(SIZE_BAR_LINE, n_lw);
		parent->SetSize(SIZE_BAR, n_width);
		parent->SetSize(SIZE_BAR_DEPTH, n_depth);
		parent->SetColor(COL_BAR_LINE, col1);
		parent->Command(CMD_BAR_FILL, &newFill, 0L);
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(ColumnDlg);
	free(tab1),			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Properties of arrow in 3D space
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *Arrow3D_DlgTmpl = (char*)
	"1,2,,DEFAULT, PUSHBUTTON,1,110,10,70,12\n"
	"2,3,,,PUSHBUTTON,-28,110,25,70,12\n"
	"3,4,,,PUSHBUTTON,-2,110,40,70,12\n"
	"4,,5,ISPARENT | CHECKED,GROUP,,,,,\n"
	"5,6,100,ISPARENT | CHECKED,SHEET,2,5,10,100,100\n"
	"6,7,200,ISPARENT,SHEET,3,5,10,100,100\n"
	"7,,300,ISPARENT,SHEET,4,5,10,100,100\n"
	"100,+,,,RTEXT,5,15,40,28,8\n"
	".,.,,,INCDECVAL1,6,46,40,33,10\n"
	".,.,,,LTEXT,-3,81,40,20,8\n"
	".,.,,,RTEXT,7,15,52,28,8\n"
	".,.,,,INCDECVAL1,8,46,52,33,10\n"
	".,.,,,LTEXT,-3,81,52,20,8\n"
	".,.,,,RTEXT,9,15,70,28,8\n"
	".,.,,,INCDECVAL1,10,46,70,33,10\n"
	".,.,,,LTEXT,-3,81,70,20,8\n"
	".,.,,,RTEXT,-11,15,82,28,8\n"
	".,,,OWNDIALOG,COLBUTT,11,46,82,25,10\n"
	"200,+,,TOUCHEXIT | HICOL,RADIO1,12,15,40,60,8\n"
	".,.,,TOUCHEXIT | HICOL,RADIO1,13,15,55,60,8\n"
	".,,,TOUCHEXIT | HICOL,RADIO1,14,15,70,60,8\n"
	"300,+,,,RTEXT,-12,10,25,28,8\n"
	".,.,,,EDVAL1,15,46,25,35,10\n"
	".,.,,,RTEXT,-13,10,36,28,8\n"
	".,.,,,EDVAL1,16,46,36,35,10\n"
	".,.,,,RTEXT,-14,10,47,28,8\n"
	".,.,,,EDVAL1,17,46,47,35,10\n"
	".,.,,,RTEXT,18,10,60,28,8\n"
	".,.,,,EDVAL1,19,46,60,35,10\n"
	".,.,,,RTEXT,-5,10,71,28,8\n"
	".,.,,,EDVAL1,20,46,71,35,10\n"
	".,.,,,RTEXT,-6,10,82,28,8\n"
	".,.,,,EDVAL1,21,46,82,35,10\n"
	".,,,LASTOBJ | HICOL,CHECKBOX,22,16,95,70,8";

bool
Arrow3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_ARROW);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TYPE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	void *dyndata[] = {(void*)SDLG_ARROW_TOARR, (void*)tab1,
		(void*)tab2, (void*)tab3, (void*)SDLG_ARROW_CWIDTH, (void*)&cw, (void*)SDLG_ARROW_CLENGTH,
		(void*)&cl, (void*)SDLG_ARROW_LWIDTH, (void*)&Line.width, (void *)&Line.color,
		(void*)SDLG_ARROW_LONLY, (void*)SDLG_ARROW_WLINE, (void*)SDLG_ARROW_FILLED,
		(void*)&fPos2.fx, (void*)&fPos2.fy, (void*)&fPos2.fz, (void*)SDLG_ARROW_ORGX,
		(void*)&fPos1.fx, (void*)&fPos1.fy, (void*)&fPos1.fz, (void*)SDLG_ARR3D_TOALL};
	DlgInfo *ArrowDlg = CompileDialog(Arrow3D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	fPOINT3D o_pos1, o_pos2, n_pos1, n_pos2;
	double o_cw, o_cl, n_cw, n_cl, o_lw, n_lw;
	int cb, res;
	long tmptype = type;
	bool bRet = false;
	DWORD o_col, n_col, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent || !ArrowDlg) return false;
	if(!(Dlg = new DlgRoot(ArrowDlg, data))) return false;
	Dlg->GetValue(301, &o_pos2.fx);		Dlg->GetValue(303, &o_pos2.fy);
	Dlg->GetValue(305, &o_pos2.fz);		Dlg->GetValue(307, &o_pos1.fx);
	Dlg->GetValue(309, &o_pos1.fy);		Dlg->GetValue(311, &o_pos1.fz);
	Dlg->GetValue(101, &o_cw);		Dlg->GetValue(104, &o_cl);
	Dlg->GetValue(107, &o_lw);		Dlg->GetColor(110, &o_col);
	switch(type & 0xff) {
	case ARROW_LINE:		Dlg->SetCheck(201, 0L, true);		break;
	case ARROW_TRIANGLE:	Dlg->SetCheck(202, 0L, true);		break;
	default:				Dlg->SetCheck(200, 0L, true);		break;
		}
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_ARROW_HD1 );
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_ARROW_HD2);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 380, 260, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 200:	tmptype = ARROW_NOCAP;		res = -1;	break;
		case 201:	tmptype = ARROW_LINE;		res = -1;	break;
		case 202:	tmptype = ARROW_TRIANGLE;	res = -1;	break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);
			Dlg->GetValue(301, &n_pos2.fx);		Dlg->GetValue(303, &n_pos2.fy);
			Dlg->GetValue(305, &n_pos2.fz);		Dlg->GetValue(307, &n_pos1.fx);
			Dlg->GetValue(309, &n_pos1.fy);		Dlg->GetValue(311, &n_pos1.fz);
			Dlg->GetValue(101, &n_cw);			Dlg->GetValue(104, &n_cl);
			Dlg->GetValue(107, &n_lw);			Dlg->GetColor(110, &n_col);
			break;
			}
		}while (res <0);
	switch (res) {
	case 1:				//new setting for current arrow
		if(n_pos1.fx != o_pos1.fx || n_pos1.fy != o_pos1.fy ||
			n_pos1.fz != o_pos1.fz){
			Undo.ValLFP3D(this, &fPos1, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&fPos1, &n_pos1, sizeof(fPOINT3D));
			parent->Command(CMD_MRK_DIRTY, 0L, 0L);
			}
		if(n_pos2.fx != o_pos2.fx || n_pos2.fy != o_pos2.fy ||
			n_pos2.fz != o_pos2.fz){
			Undo.ValLFP3D(this, &fPos2, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&fPos2, &n_pos2, sizeof(fPOINT3D));
			parent->Command(CMD_MRK_DIRTY, 0L, 0L);
			}
		if((type & 0xff) != (tmptype & 0xff)){
			Undo.ValLong(this, &type, undo_flags);
			type &= ~0xff;	type |= (tmptype & 0xff);	undo_flags |= UNDO_CONTINUE;
			}
		undo_flags = CheckNewFloat(&cw, o_cw, n_cw, this, undo_flags);
		undo_flags = CheckNewFloat(&cl, o_cl, n_cl, this, undo_flags);
		undo_flags = CheckNewFloat(&Line.width, o_lw, n_lw, this, undo_flags);
		undo_flags = CheckNewDword(&Line.color, o_col, n_col, this, undo_flags);
		if(undo_flags & UNDO_CONTINUE) bModified = true;
		bRet = true;
		break;
	case 2:				//new settings to all arrows of plot
		parent->Command(CMD_SAVE_ARROWS, 0L, 0L);
		if(Dlg->GetCheck(312)) parent->Command(CMD_ARROW_ORG3D, &n_pos1, 0L);
		parent->SetSize(SIZE_ARROW_LINE, n_lw);
		parent->SetSize(SIZE_ARROW_CAPWIDTH, n_cw);
		parent->SetSize(SIZE_ARROW_CAPLENGTH, n_cl);
		parent->SetColor(COL_ARROW, n_col);
		parent->Command(CMD_ARROW_TYPE, &tmptype, 0L);
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);	delete Dlg;		free(ArrowDlg);
	free(tab1);		free(tab2);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// properties of 3D data line
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *Line3D_DlgTmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,-1,152,10,45,12\n"
		".,.,,,PUSHBUTTON,-2,152,25,45,12\n"
		".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
		"4,,100,ISPARENT | CHECKED,SHEET,1,5,10,139,130\n"
		"100,,,LASTOBJ | NOSELECT,ODBUTTON,2,10,30,130,90";

bool
Line3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	void *dyndata[] = {(void*)tab1, (void*)OD_linedef};
	DlgInfo *LineDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res;
	bool bRet = false;
	LineDEF newLine;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent) return false;
	if(parent->Id == GO_GRID3D) return parent->PropertyDlg();
	if(!(LineDlg = CompileDialog(Line3D_DlgTmpl, dyndata))) return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	if(!(Dlg = new DlgRoot(LineDlg, data)))return false;
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)"Line of ");
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)"Line properties");
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 420, 339, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
	}while (res < 0);
	switch(res) {
	case 1:							//OK pressed
		Undo.SetDisp(cdisp);
		OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		if(cmpLineDEF(&Line, &newLine)) {
			if(parent->Id == GO_GRID3D) {
				parent->Command(CMD_SET_LINE, &newLine, 0L);
				}
			else {
				Undo.Line(this, &Line, 0L);
				memcpy(&Line, &newLine, sizeof(LineDEF));
				bModified = true;
				}
			}
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;		free(LineDlg);		free(tab1);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// label (text) properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *LabelDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,170,10,55,12\n"
	".,.,,,PUSHBUTTON,-2,170,40,55,12\n"
	"3,50,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT,SHEET,1,5,10,159,100\n"
	".,.,200,ISPARENT | CHECKED,SHEET,2,5,10,159,100\n"
	".,.,300,ISPARENT,SHEET,3,5,10,159,100\n"
	".,.,,HIDDEN,PUSHBUTTON,37,170,68,55,12\n"
	".,.,,HIDDEN,PUSHBUTTON,38,170,83,55,12\n"
	".,,,,PUSHBUTTON,39,170,98,55,12\n"
	"10,50,,,PUSHBUTTON,-28,170,25,55,12\n"
	"50,,600,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
	"100,+,,,RTEXT,4,30,33,45,8\n"
	".,.,,,INCDECVAL1,5,80,33,33,10\n"
	".,.,,,LTEXT,-3,115,33,20,8\n"
	".,.,,,RTEXT,-11,30,45,45,8\n"
	".,107,,OWNDIALOG,COLBUTT,6,80,45,25,10\n"
	"105,+,,,LTEXT,7,10,83,25,8\n"
	".,,,TOUCHEXIT,EDTEXT,8,10,95,149,10\n"
	"107,+,,,RTEXT,9,30,57,45,8\n"
	".,.,,,EDVAL1,10,80,57,25,10\n"
	".,150,,,LTEXT,11,107,57,20,8\n"
	"150,154,151,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
	"151,+,,,RTEXT,12,30,69,45,8\n"
	".,.,,,INCDECVAL1,13,80,69,33,10\n"
	".,,,,LTEXT,-10,115,69,20,8\n"
	"154,105,,HICOL,CHECKBOX,14,80,83,60,9\n"
	"200,244,221,CHECKED | ISPARENT,GROUPBOX,15,17,28,60,50\n"
	"221,+,,HICOL | TOUCHEXIT,RADIO1,16,20,35,45,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,17,20,45,45,8\n"
	".,.,,HICOL | TOUCHEXIT, RADIO1,18,20,55,45,8\n"
	".,,,HICOL | TOUCHEXIT, RADIO1,19,20,65,45,8\n"
	"244,105,245,CHECKED | ISPARENT, GROUPBOX,20,87,28,67,60\n"
	"245,+,,HICOL | TOUCHEXIT,CHECKBOX,21,90,35,25,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,22,90,45,25,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,23,90,55,25,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,24,90,65,25,8\n"
	".,,,HICOL | TOUCHEXIT,CHECKBOX,25,90,75,25,8\n"
	"300,+,,,LTEXT,26,10,25,30,8\n"
	".,.,,,RTEXT,-4,5,37,15,8\n"
	".,.,,,EDVAL1,27,22,37,45,10\n"
	".,.,,,LTEXT,,69,37,10,8\n"
	".,.,,,RTEXT,-5,85,37,10,8\n"
	".,.,,,EDVAL1,28,97,37,45,10\n"
	".,.,,,LTEXT,,144,37,10,8\n"
	".,.,,,LTEXT,29,10,52,50,8\n"
	".,.,,,RTEXT,30,5,64,15,8\n"
	".,.,,,EDVAL1,31,22,64,45,10\n"
	".,.,,,LTEXT,-3,69,64,10,8\n"
	".,.,,,RTEXT,32,85,64,10,8\n"
	".,.,,,EDVAL1,33,97,64,45,10\n"
	".,.,,,LTEXT,-3,144,64,10,8\n"
	".,.,,,LTEXT,34,10,85,80,8\n"
	".,,,,TXTHSP,35,97,78,45,25\n"
	"600,+,,TOUCHEXIT,ODB,36,180,45,15,15\n"
	".,., 0,TOUCHEXIT,ODB,36,180,60,15,15\n"
	".,., 0,TOUCHEXIT,ODB,36,200,45,15,15\n"
	".,, 0,LASTOBJ | TOUCHEXIT,ODB,36,200,60,15,15";

bool
Label::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LABEL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TYPE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	bool isDataLabel = ((parent) && (parent->Id == GO_PLOTSCATT || parent->Id == GO_XYSTAT || parent->Id == GO_BOXPLOT
		|| parent->Id == GO_CONTOUR));
	double lspc = 100.0;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_LABEL_SIZE, (void*)&TextDef.fSize,
		 (void *)&TextDef.ColTxt, (void*)SDLG_LABEL_TEXT, (void*)TextDef.text, (void*)SDLG_LABEL_ROT,
		 (void*)&TextDef.RotBL, (void *)SDLG_LABEL_DEG, (void *)SDLG_LABEL_LSPC, (void*)&lspc,
		 (void*) SDLG_LABEL_MOVE, (void*)SDLG_LABEL_FONT, (void*)SDLG_LABEL_HELV, (void*)SDLG_LABEL_TIMES,
		 (void*)SDLG_LABEL_COURIER, (void*)SDLG_LABEL_GREEK, (void*)SDLG_LABEL_STYLE, (void*)SDLG_LABEL_BOLD,
		 (void*)SDLG_LABEL_ITALIC, (void*)SDLG_LABEL_UNDERL, (void*)SDLG_LABEL_SUPER, (void*)SDLG_LABEL_SUB,
		 (void*)SDLG_LABEL_ANCHOR, (void*)&fPos.fx, (void*)&fPos.fy, (void*)SDLG_LABEL_ADIST, 
		 (void*)SDLG_LABEL_DX, (void*)&fDist.fx, (void*)SDLG_LABEL_DY, (void*)&fDist.fy, (void*)SDLG_LABEL_HSP,
		 (void*)&TextDef.Align, (void*)OD_DrawOrder, (void*)SDLG_LABEL_AXIS, (void*)SDLG_LABEL_TICKPROP, (void*)SDLG_LABEL_DELOBJ };
	DlgInfo *LabelDlg = CompileDialog(LabelDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, c_style, c_font;
	bool RetVal = false, check, bContinue = false;
	double tmp;
	DWORD undo_flags = 0x0;
	anyOutput *cdisp = Undo.Cdisp();
	lfPOINT o_pos, o_dist, n_pos, n_dist;
	TextDEF OldTxtDef, NewTxtDef;
	Axis *MyAxis, *pa;
	fmtText *fmt = 0L;

	if(!parent || !LabelDlg) return false;
	bBusy = true;
	if(isDataLabel) {
		LabelDlg[1].y = 40;		//'cancel' button
		LabelDlg[2].next = 10;	// display 'Apply to PLOT' button'
		}
	else {
		LabelDlg[1].y =	25;
		LabelDlg[2].next = 50;
		}
	if(parent->Id == GO_MLABEL) {
		lspc = 100.0 * parent->GetSize(SIZE_LSPC);
		}
	if (lspc < 50.0) lspc = 100.0;
	if(parent->Id == GO_AXIS) MyAxis = (Axis*)parent;
	else if(parent->Id == GO_TICK && parent->parent && parent->parent->Id == GO_AXIS) MyAxis = (Axis*)parent->parent;
	else MyAxis = NULL;
	if(parent->Id != GO_TICK) LabelDlg[6].y += 15;
	if((Dlg = new DlgRoot(LabelDlg, data))) {
		if(MyAxis) Dlg->ShowItem(7, true);
		if(parent->Id == GO_TICK) Dlg->ShowItem(8, true);
		if(parent->Id == GO_MLABEL) Dlg->ShowItem(150, true);
		Dlg->TextFont(221, FONT_HELVETICA);		Dlg->TextFont(222, FONT_TIMES);
		Dlg->TextFont(223, FONT_COURIER);		Dlg->TextFont(224, FONT_GREEK);
		Dlg->TextStyle(205, TXS_BOLD);			Dlg->TextStyle(206, TXS_ITALIC);
		Dlg->TextStyle(207, TXS_UNDERLINE);		Dlg->TextStyle(207, TXS_SUPER);
		Dlg->TextStyle(207, TXS_SUB);
		if(moveable){
			Dlg->SetCheck(154, 0L, true);
			}
		switch(TextDef.Font) {
		case FONT_TIMES:	Dlg->SetCheck(222, 0L, true);	break;
		case FONT_COURIER:	Dlg->SetCheck(223, 0L, true);	break;
		case FONT_GREEK:	Dlg->SetCheck(224, 0L, true);	break;
		default:			Dlg->SetCheck(221, 0L, true);	break;
			}
		if(TextDef.Style & TXS_BOLD) Dlg->SetCheck(245, 0L, true);
		if(TextDef.Style & TXS_ITALIC) Dlg->SetCheck(246, 0L, true);
		if(TextDef.Style & TXS_UNDERLINE) Dlg->SetCheck(247, 0L, true);
		if(TextDef.Style & TXS_SUPER) Dlg->SetCheck(248, 0L, true);
		if(TextDef.Style & TXS_SUB) Dlg->SetCheck(249, 0L, true);
		if (parent && parent->Id == GO_CONTOUR) Dlg->ShowItem(6, false);
		}
	else return false;
	if(parent->Id ==  GO_GRAPH || parent->Id == GO_PAGE){
		Dlg->ShowItem(50, true);			//the  'Draw Order Group'
		Dlg->ShowItem(154, true);			//moveable: it can be dragged with the mouse
		}
	else {
		Dlg->ShowItem(50, false);
		Dlg->ShowItem(154, false);	
		}
	switch(flags & 0x03) {
	case LB_X_DATA:
		Dlg->SetText(303, (unsigned char*)"[data]");
		break;
	case LB_X_PARENT:
		Dlg->SetText(303, (unsigned char*)" ");
		TmpTxt[0] = 0;
		if(parent->Id == GO_MLABEL) {
			if(parent->parent->Id == GO_AXIS) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PAXIS);
			else if(parent->parent->Id == GO_TICK) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PTICK);
			}
		else {
			if(parent->Id == GO_AXIS) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PAXIS);
			else if(parent->Id == GO_TICK) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PTICK);
			}
		if(TmpTxt[0]) {
			Dlg->SetText(302, (unsigned char*)TmpTxt);			Dlg->Activate(302, 0);
			}
		break;
	default:
		Dlg->SetText(303, (unsigned char*)Units[defs.units()].display);
		break;
		}
	switch(flags & 0x30) {
	case LB_Y_DATA:
		Dlg->SetText(306, (unsigned char*)SDLG_COM_DATAREF);
		break;
	case LB_Y_PARENT:
		Dlg->SetText(306, (unsigned char*)" ");					TmpTxt[0] = 0;
		if(parent->Id == GO_MLABEL) {
			if(parent->parent->Id == GO_AXIS) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PAXIS);
			if(parent->parent->Id == GO_TICK) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PTICK);
			}
		else {
			if(parent->Id == GO_AXIS) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PAXIS);
			if(parent->Id == GO_TICK) rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_LABEL_PTICK);
			}
		if(TmpTxt[0]) {
			Dlg->SetText(305, (unsigned char*)TmpTxt);			Dlg->Activate(305, 0);
			}
		break;
	default:
		Dlg->SetText(306, (unsigned char*) Units[defs.units()].display);
		break;
		}
	memcpy(&OldTxtDef, &TextDef, sizeof(TextDEF));	OldTxtDef.text = 0L;
	Dlg->GetValue(101, &OldTxtDef.fSize);			Dlg->GetValue(108, &OldTxtDef.RotBL);
	memcpy(&NewTxtDef, &OldTxtDef, sizeof(TextDEF));
	o_pos.fx = fPos.fx;	o_pos.fy = fPos.fy;	o_dist.fx = fDist.fx;	o_dist.fy = fDist.fy;
	Dlg->GetValue(302, &o_pos.fx);					Dlg->GetValue(305, &o_pos.fy);
	Dlg->GetValue(309, &o_dist.fx);					Dlg->GetValue(312, &o_dist.fy);
	n_pos.fx = o_pos.fx;	n_pos.fy = o_pos.fy;	n_dist.fx = o_dist.fx;	n_dist.fy = o_dist.fy;
	hDlg = CreateDlgWnd((char*)SDLG_LABEL_HD1, 50, 50, 480, 278, Dlg, 0x4L);
	do{
		LoopDlgWnd();			res = Dlg->GetResult();
		switch (res) {
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);
			}
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			break;
		case 7:					//Axis properties
			if(MyAxis)MyAxis->PropertyDlg();
			bContinue = true;	res = -1;
			break;
		case 8:					//Tick properties
			if (parent && parent->Id == GO_TICK)parent->PropertyDlg();
			bContinue = true;	res = -1;
			break;
		case 9:					//Delete object
			parent->Command(CMD_DELOBJ, this, cdisp);
			res = 2;			break;
		case 10:				case 1:
			if (res == 10) {
				Undo.SetDisp(cdisp);
				parent->Command(CMD_SAVE_LABELS, 0L, 0L);
				undo_flags |= UNDO_CONTINUE;
				}
			Dlg->GetValue(101, &NewTxtDef.fSize);	Dlg->GetColor(104, &NewTxtDef.ColTxt);
			Dlg->GetValue(108, &NewTxtDef.RotBL);	Dlg->GetInt(315, &NewTxtDef.Align);
			Dlg->GetValue(302, &n_pos.fx);			Dlg->GetValue(305, &n_pos.fy);
			Dlg->GetValue(309, &n_dist.fx);			Dlg->GetValue(312, &n_dist.fy);
			break;
		case 106:													//text modified
			if(!(Dlg->GetText(res, (unsigned char*)TmpTxt, TMP_TXT_SIZE))) TmpTxt[0] = 0;
			else {
				if(!fmt) fmt = new fmtText(0L, 0, 0, (unsigned char*)TmpTxt);
				else fmt->SetText(0L, (unsigned char*)TmpTxt, 0L, 0L, false);
				}
			if(fmt && TmpTxt[0] && Dlg->GetInt(106, &i)) {
				c_style = NewTxtDef.Style;	c_font = NewTxtDef.Font;
				fmt->StyleAt(i, &NewTxtDef, &c_style, &c_font);
				Dlg->SetCheck(245, 0L, TXS_BOLD == (c_style & TXS_BOLD));
				Dlg->SetCheck(246, 0L, TXS_ITALIC == (c_style & TXS_ITALIC));
				Dlg->SetCheck(247, 0L, TXS_UNDERLINE == (c_style & TXS_UNDERLINE));
				Dlg->SetCheck(248, 0L, TXS_SUPER == (c_style & TXS_SUPER));
				Dlg->SetCheck(249, 0L, TXS_SUB == (c_style & TXS_SUB));
				switch(c_font) {
				case FONT_HELVETICA:	Dlg->SetCheck(221, 0L, true);	break;
				case FONT_TIMES:		Dlg->SetCheck(222, 0L, true);	break;
				case FONT_COURIER:		Dlg->SetCheck(223, 0L, true);	break;
				case FONT_GREEK:		Dlg->SetCheck(224, 0L, true);	break;
					}
				}
			res = -1;
			break;
		case 221:	case 222:	case 223:	case 224:				//fonts
			switch (res){
			case 221:		res = FONT_HELVETICA;		break;
			case 222:		res = FONT_TIMES;			break;
			case 223:		res = FONT_COURIER;			break;
			case 224:		res = FONT_GREEK;			break;
				}
			if(!Dlg->ItemCmd(106, CMD_SETFONT, &res)) NewTxtDef.Font = res;
			res = -1;
			break;
		case 245:	case 246:	case 247:	case 248:	case 249:	//styles
			check = Dlg->GetCheck(res);
			switch (res){
			case 245:		res = TXS_BOLD;			break;
			case 246:		res = TXS_ITALIC;		break;
			case 247:		res = TXS_UNDERLINE;	break;
			case 248:		res = TXS_SUPER;		break;
			case 249:		res = TXS_SUB;			break;
				}
			if(!check) {
				res = ~res;
				if(!Dlg->ItemCmd(106, CMD_SETSTYLE, &res)) NewTxtDef.Style &= res;
				}
			else if(!Dlg->ItemCmd(106, CMD_SETSTYLE, &res)) NewTxtDef.Style |= res;
			res = -1;
			break;
			}
		}while (res < 0);
	if(res == 1 || res == 10) {
		Undo.SetDisp(cdisp);
		if(parent->Id == GO_TICK && parent->parent && parent->parent->Id == GO_AXIS)
			pa = (Axis*)parent->parent;
		else if(parent->Id == GO_MLABEL && parent->parent->Id == GO_TICK 
			&& parent->parent->parent && parent->parent->parent->Id == GO_AXIS)
			pa = (Axis*)parent->parent->parent;
		else pa = 0L;
		if(pa && (cmpTextDEF(&OldTxtDef, &NewTxtDef) || o_dist.fx != n_dist.fx ||
			o_dist.fy != n_dist.fy)){
			pa->Command(CMD_SAVE_TICKS, 0L, 0L);			undo_flags |= UNDO_CONTINUE;
			}
		i = Dlg->GetCheck(154) ? 1 : 0;
		if(i != moveable) {
			Undo.ValInt(parent, &moveable, undo_flags);
			moveable = i;									undo_flags |= UNDO_CONTINUE;
			}
		TmpTxt[0] = 0;		Dlg->GetText(106, (unsigned char*)TmpTxt, TMP_TXT_SIZE);
		if(res == 1) undo_flags = CheckNewString(&TextDef.text, TextDef.text, (unsigned char*)TmpTxt, this, undo_flags);
		if(cmpTextDEF(&OldTxtDef, &NewTxtDef)){
			if(NewTxtDef.ColTxt != TextDef.ColTxt) bBGvalid = false;
			if (pa) pa->Command(CMD_TLB_TXTDEF, &NewTxtDef, 0L);
			else if(res == 10 || parent->Id == GO_MLABEL) {
				if(parent->Command(CMD_SETTEXTDEF, &NewTxtDef, 0L))RetVal = true;
				}
			else {
				Undo.TextDef(this, &TextDef, undo_flags);
				NewTxtDef.text = TextDef.text;	memcpy(&TextDef, &NewTxtDef, sizeof(TextDEF));	
				undo_flags |= UNDO_CONTINUE;
				}
			}
		if(n_pos.fx != o_pos.fx || n_pos.fy != o_pos.fy) {
			if(parent->Id == GO_MLABEL) {
				if(parent->SetSize(SIZE_XPOS, n_pos.fx) || 
					parent->SetSize(SIZE_YPOS, n_pos.fy)) RetVal = true;
				}
			else {
				Undo.SaveLFP(this, &fPos, undo_flags);		undo_flags |= UNDO_CONTINUE;
				fPos.fx = n_pos.fx;		fPos.fy = n_pos.fy;
				}
			}
		if(n_dist.fx != o_dist.fx || n_dist.fy != o_dist.fy) {
			if(pa) {
				pa->SetSize(SIZE_TLB_XDIST, n_dist.fx);		pa->SetSize(SIZE_TLB_YDIST, n_dist.fy);
				}
			else if(res == 10 || parent->Id == GO_MLABEL) {
				if(n_dist.fx != o_dist.fx && parent->SetSize(SIZE_LB_XDIST, n_dist.fx)) RetVal = true;
				if(n_dist.fy != o_dist.fy && parent->SetSize(SIZE_LB_YDIST, n_dist.fy)) RetVal = true;
				}
			else {
				Undo.SaveLFP(this, &fDist, undo_flags);		undo_flags |= UNDO_CONTINUE;
				fDist.fx = n_dist.fx;	fDist.fy = n_dist.fy;
				}
			}
		if(parent->Id == GO_MLABEL && Dlg->GetValue(152, &tmp) && tmp != lspc) {
			parent->SetSize(SIZE_LSPC, tmp/100.0);	RetVal = true;
			}
		if(undo_flags & UNDO_CONTINUE) RetVal = true;
		}
	CloseDlgWnd(hDlg);
	if(fmt) delete(fmt);
	delete Dlg;			free(LabelDlg);			bBusy = false;
	free(tab1);			free(tab2);				free(tab3);
	return RetVal;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Text frame properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *TextFrmDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,130,25,45,12\n"
	".,50,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,119,100\n"
	".,.,200,ISPARENT,SHEET,2,5,10,119,100\n"
	".,,300,ISPARENT,SHEET,3,5,10,119,100\n"
	"50,.,600,ISPARENT | CHECKED,GROUP,,,,,\n"
	"100,+,,,RTEXT,4,10,33,45,8\n"
	".,.,,,INCDECVAL1,5, 60, 33, 33, 10\n"
	".,.,,,LTEXT,-3,95,33,20,8\n"
	".,.,,,RTEXT,6,10,45,45,8\n"
	".,150,,OWNDIALOG,COLBUTT,7,60,45,25,10\n"
	"150,,151,ISPARENT | CHECKED,GROUP,,,,,\n"
	"151,+,,,RTEXT,8,10,57,45,8\n"
	".,.,,,INCDECVAL1,9,60,57,33,10\n"
	".,,,,LTEXT,-10,95,57,20,8\n"
	"200,,,NOSELECT,ODB,10,20,30,90,50\n"
	"300,+,,,LTEXT,12,10,30,90,8\n"
	".,.,,,RTEXT,13,10,45,45,8\n"
	".,.,,,EDVAL1,14,57,45,35,10\n"
	".,.,,,LTEXT,-3,92,45,10,8\n"
	".,.,,,RTEXT,15,10,57,45,8\n"
	".,.,,,EDVAL1,16,57,57,35,10\n"
	".,.,,,LTEXT,-3,92,57,10,8\n"
	".,.,,,RTEXT,17,10,74,45,8\n"
	".,.,,,EDVAL1,18,57,74,35,10\n"
	".,.,,,LTEXT,-3,92,74,10,8\n"
	".,.,,,RTEXT,15,10,86,45,8\n"
	".,.,,,EDVAL1,19,57,86,35,10\n"
	".,,,,LTEXT,-3,92,86,10,8\n"
	"600,+,,TOUCHEXIT,ODB,11,145,45,15,15\n"
	".,.,,TOUCHEXIT,ODB,11,145,60,15,15\n"
	".,., 0,TOUCHEXIT,ODB,11,145,75,15,15\n"
	".,, 0,LASTOBJ | TOUCHEXIT,ODB,11,145,90,15,15";

bool
TextFrame::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_TEXT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_FRAME);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	double lspc2, lspc1 = lspc2 = lspc * 100.0;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_TXTFRM_SIZE, (void*)&TextDef.fSize,
		(void*)SDLG_TXTFRM_COL, (void *)&TextDef.ColTxt, (void *)SDLG_TXTFRM_LSPAC, (void*)&lspc1,
		(void*)OD_filldef, (void*)OD_DrawOrder, (void*)SDLG_TXTFRM_POS, (void*)SDLG_TXTFRM_PT1_X , (void*)&pos1.fx,
		(void*)SDLG_TXTFRM_PT_Y, (void*)&pos1.fy, (void*)SDLG_TXTFRM_PT2_X, (void*)&pos2.fx, (void*)&pos2.fy };
	DlgInfo *TextFrmDlg = CompileDialog(TextFrmDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	anyOutput *cdisp = Undo.Cdisp();
	int res;
	bool bRet = false;
	LineDEF newLine, newFillLine;
	static FillDEF newFill;
	DWORD undo_flags = 0L;
	TextDEF OldTxtDef, NewTxtDef;
	lfPOINT lp1, lp2, cp1, cp2;

	if(!parent || !data || !TextFrmDlg) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
	if(!(Dlg = new DlgRoot(TextFrmDlg, data)))return false;
	memcpy(&OldTxtDef, &TextDef, sizeof(TextDEF));	OldTxtDef.text = 0L;
	Dlg->GetValue(101, &OldTxtDef.fSize);			memcpy(&NewTxtDef, &OldTxtDef, sizeof(TextDEF));
	Dlg->GetValue(152, &lspc1);
	Dlg->GetValue(302, &cp1.fx);		Dlg->GetValue(305, &cp1.fy);
	Dlg->GetValue(308, &cp2.fx);		Dlg->GetValue(311, &cp2.fy);
	hDlg = CreateDlgWnd((char*)SDLG_TXTFRM_HD1, 50, 50, 380, 268, Dlg, 0x4L);
	do{
		LoopDlgWnd();			res = Dlg->GetResult();
		switch (res) {
		case 0:
			res = -1;
			break;
		case 1:
			Dlg->GetValue(101, &NewTxtDef.fSize);	Dlg->GetColor(104, &NewTxtDef.ColTxt);
			Dlg->GetValue(152, &lspc2);			lspc1 /= 100.0;			lspc2 /= 100.0;
			Dlg->GetValue(302, &lp1.fx);		Dlg->GetValue(305, &lp1.fy);
			Dlg->GetValue(308, &lp2.fx);		Dlg->GetValue(311, &lp2.fy);	
			break;
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);
			break;
			}
		}while (res < 0);
	if(res == 1){						//OK pressed
		Undo.SetDisp(cdisp);
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
		memcpy(&newFillLine, &FillLine, sizeof(LineDEF));
		if(newFill.hatch) memcpy(&newFillLine, newFill.hatch, sizeof(LineDEF));
		if(cmpTextDEF(&OldTxtDef, &NewTxtDef)){
			Undo.TextDef(this, &TextDef, undo_flags);
			memcpy(&TextDef, &NewTxtDef, sizeof(TextDEF));	
			undo_flags |= UNDO_CONTINUE;	TextDef.text = 0L;
			}
		undo_flags = CheckNewFloat(&lspc, lspc1, lspc2, parent, undo_flags);
		undo_flags = CheckNewFloat(&pos1.fx, cp1.fx, lp1.fx, parent, undo_flags);
		undo_flags = CheckNewFloat(&pos1.fy, cp1.fy, lp1.fy, parent, undo_flags);
		undo_flags = CheckNewFloat(&pos2.fx, cp2.fx, lp2.fx, parent, undo_flags);
		undo_flags = CheckNewFloat(&pos2.fy, cp2.fy, lp2.fy, parent, undo_flags);
		if (cmpLineDEF(&Line, &newLine)) {
			Undo.Line(parent, &Line, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&Line, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&FillLine, &newFillLine)) {
			Undo.Line(parent, &FillLine, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&FillLine, &newFillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&Fill, &newFill)) {
			Undo.Fill(parent, &Fill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&Fill, &newFill, sizeof(FillDEF));
			}
		Fill.hatch = &FillLine;
		if(undo_flags & UNDO_CONTINUE){
			bRet = bModified = true;					lines2text();
			Undo.TextBuffer(this, &csize, &cpos, &text, undo_flags, cdisp);
			if(lines) {
				for(i = 0; i < nlines; i++) if(lines[i]) free(lines[i]);
				free(lines);			lines = 0L;
				}
			HideTextCursor();			cur_pos.x = cur_pos.y = 0;
			}
		}
	CloseDlgWnd(hDlg);	delete Dlg;				free(TextFrmDlg);
	free(tab1);			free(tab2);				free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// segment properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *SegDlg_DlgTmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,4,130,10,65,12\n"
		".,.,,,PUSHBUTTON,-28,130,25,65,12\n"
		".,.,,,PUSHBUTTON,-2,130,40,65,12\n"
		".,,5,CHECKED | ISPARENT,GROUP,,,,,\n"
		"5,+,100,ISPARENT | CHECKED,SHEET,1,5,10,120,115\n"
		".,.,200,ISPARENT,SHEET,2,5,10,120,115\n"
		".,,150,ISPARENT,SHEET,3,5,10,120,115\n"
		"100,109,,NOSELECT,ODB,5,25,35,90,50\n"
		"109,+,,,RTEXT,6,10,90,55,8\n"
		".,.,,,EDVAL1,7,67,90,25,10\n"
		".,.,,,LTEXT,-3,94,90,20,8\n"
		".,,,HICOL,CHECKBOX,8,12,107,100,8\n"
		"150,+,,,LTEXT,21,10,30,20,8\n"
		".,,,,EDTEXT,22,10,45,110,10\n"
		"200,+,,,RTEXT,9,10,30,40,8\n"
		".,.,,,EDVAL1,10,55,30,40,10\n"
		".,.,,,LTEXT,-3,98,30,20,8\n"
		".,.,,,RTEXT,-5,10,42,40,8\n"
		".,.,,,EDVAL1,11,55,42,40,10\n"
		".,.,,,LTEXT,-3,98,42,20,8\n"
		".,.,,,RTEXT,12,10,57,40,8\n"
		".,.,,,EDVAL1,13,55,57,40,10\n"
		".,.,,,LTEXT,14,98,57,20,8\n"
		".,.,,,RTEXT,15,10,69,40,8\n"
		".,.,,,EDVAL1,16,55,69,40,10\n"
		".,.,,,LTEXT,14,98,69,20,8\n"
		".,.,,,RTEXT,17,10,84,40,8\n"
		".,.,,,EDVAL1,18,55,84,40,10\n"
		".,.,,,LTEXT,-3,98,84,20,8\n"
		".,.,,,RTEXT,19,10,96,40,8\n"
		".,.,,,EDVAL1,20,55,96,40,10\n"
		".,,,LASTOBJ,LTEXT,-3,98,96,20,8";

bool
segment::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_SIZECOL);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DESC);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_SEGMENT_TOSEGMENT, (void*)OD_filldef,
		(void*)SDLG_SEGMENT_EXPLODE, (void*)&shift, (void*)SDLG_SEGMENT_MOVEABLE,
		(void*)SDLG_SEGMENT_CENTX, (void*)&fCent.fx, (void*)&fCent.fy, (void*)SDLG_SEGMENT_STARTANG,
		(void*)&angle1, (void*)SDLG_SEGMENT_DEG, (void*)SDLG_SEGMENT_STOP, (void*)&angle2,
		(void*)SDLG_SEGMENT_RAD1, (void*)&radius2, (void*)SDLG_SEGMENT_RAD2, (void*)&radius1,
		(void*)SDLG_SEGMENT_NAM, (void*)name};
	DlgInfo *SegDlg = CompileDialog(SegDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, new_mov;
	double old_r1, old_r2, old_a1, old_a2, old_shift;
	double new_r1, new_r2, new_a1, new_a2, new_shift = shift;
	lfPOINT old_cent, new_cent;
	bool bRet = false;
	LineDEF newLine, newFillLine;
	static FillDEF newFill;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent || !SegDlg) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&segLine, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&segFill, 0);
	if(!(Dlg = new DlgRoot(SegDlg, data)))return false;
	Dlg->GetValue(216, &old_r1);		new_r1 = old_r1;
	Dlg->GetValue(213, &old_r2);		new_r2 = old_r2;
	Dlg->GetValue(201, &old_cent.fx);	new_cent.fx = old_cent.fx;
	Dlg->GetValue(204, &old_cent.fy);	new_cent.fy = old_cent.fy;
	Dlg->GetValue(207, &old_a1);		new_a1 = old_a1;
	Dlg->GetValue(210, &old_a2);		new_a2 = old_a2;
	Dlg->GetValue(110, &old_shift);		new_shift = old_shift;
	if(moveable) Dlg->SetCheck(112, 0L, true);
	hDlg = CreateDlgWnd((char*)SDLG_SEGMENT_HD1, 50, 50, 420, 310, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 1:		case 2:
			Undo.SetDisp(cdisp);
			OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
			memcpy(&newFillLine, &segFillLine, sizeof(LineDEF));
			if(newFill.hatch) memcpy(&newFillLine, newFill.hatch, sizeof(LineDEF));
			Dlg->GetValue(201, &new_cent.fx);			Dlg->GetValue(204, &new_cent.fy);
			Dlg->GetValue(207, &new_a1);				Dlg->GetValue(210, &new_a2);
			Dlg->GetValue(216, &new_r1);				Dlg->GetValue(213, &new_r2);
			Dlg->GetValue(110, &new_shift);
			new_mov = Dlg->GetCheck(112) ? 1 : 0;
			break;
			}
		}while (res < 0);
	switch (res) {
	case 1:				//new setting for current segment only
		if (Dlg->GetText(151, (unsigned char*)TmpTxt, 40) && TmpTxt[0]){
			if (name && (strcmp(TmpTxt, name))) {
				Undo.String(this, &name, 0L);
				free(name);						undo_flags |= UNDO_CONTINUE;
				name = rlp_strdup(TmpTxt);
				}
			else if (!name){
				Undo.String(this, &name, 0L);	undo_flags |= UNDO_CONTINUE;
				name = rlp_strdup(TmpTxt);
				}
			}
		undo_flags = CheckNewLFPoint(&fCent, &old_cent, &new_cent, parent, undo_flags);
		undo_flags = CheckNewFloat(&angle1, old_a1, new_a1, parent, undo_flags);
		undo_flags = CheckNewFloat(&angle2, old_a2, new_a2, parent, undo_flags);
		undo_flags = CheckNewFloat(&radius1, old_r1, new_r1, parent, undo_flags);
		undo_flags = CheckNewFloat(&radius2, old_r2, new_r2, parent, undo_flags);
		undo_flags = CheckNewFloat(&shift, old_shift, new_shift, parent, undo_flags);
		if(cmpLineDEF(&segLine, &newLine)) {
			Undo.Line(parent, &segLine, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&segLine, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&segFillLine, &newFillLine)) {
			Undo.Line(parent, &segFillLine, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&segFillLine, &newFillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&segFill, &newFill)) {
			Undo.Fill(parent, &segFill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&segFill, &newFill, sizeof(FillDEF));
			}
		segFill.hatch = &segFillLine;
		undo_flags = CheckNewInt(&moveable, moveable, new_mov, parent, undo_flags);
		bRet = bModified = ((undo_flags & UNDO_CONTINUE) == UNDO_CONTINUE);
		break;
	case 2:				//new settings to all segments of chart
		parent->Command(CMD_SAVE_SYMBOLS, 0L, 0L);
//		parent->SetSize(SIZE_YPOS, new_cent.fy);
		parent->SetSize(SIZE_RADIUS1, new_r1);		parent->SetSize(SIZE_RADIUS2, new_r2);
		parent->Command(CMD_SHIFT_OUT, (void*)&new_shift, 0L);
		parent->Command(CMD_SEG_LINE, (void*)&newLine, 0L);
//		parent->Command(CMD_SEG_FILL, (void*)&newFill, 0L);
		parent->Command(CMD_SEG_MOVEABLE, (void *)&new_mov, 0L);
		bRet = true;
		break;
		}
	CloseDlgWnd(hDlg);	delete Dlg;				free(SegDlg);
	free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// polyline properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *PolyLineDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,150,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,150,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,50,100,ISPARENT | CHECKED,SHEET,1,5,10,139,130\n"
	"50,,600,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
	"100,+,,NOSELECT,ODB,2,10,30,130,90\n"
	".,,,HICOL,CHECKBOX,3,10,125,60,9\n"
	"600,+,,TOUCHEXIT,ODB,4,165,60,15,15\n"
	".,.,,TOUCHEXIT,ODB,4,165,75,15,15\n"
	".,.,,TOUCHEXIT,ODB,4,165,90,15,15\n"
	".,.,,LASTOBJ | TOUCHEXIT,ODB,4,165,105,15,15";

bool
polyline::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	void *dyndata[] = { (void*)tab1, (void*)OD_linedef, (void*)SDLG_POLYLINE_DEFAULT,
		(void*)OD_DrawOrder};
	DlgInfo *LineDlg = CompileDialog(PolyLineDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, undo_level = Undo.level();
	anyOutput *cdisp = Undo.Cdisp();
	bool bRet = false;
	LineDEF newLine;

	if(!parent || !LineDlg) return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&pgLine, 0);
	if(!(Dlg = new DlgRoot(LineDlg, data)))return false;
	if(parent->Id ==  GO_GRAPH || parent->Id == GO_PAGE) Dlg->ShowItem(50, true);
	hDlg = CreateDlgWnd((char*)SDLG_POLYLINE_HD1, 50, 50, 430, 344, Dlg, 0x4L);
	do{
		LoopDlgWnd();			res = Dlg->GetResult();
		switch (res) {
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);	break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);	break;
			}
	}while (res < 0);
	switch(res) {
	case 1:							//OK pressed
		OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		if(cmpLineDEF(&pgLine, &newLine)) {
			Undo.Line(this, &pgLine, 0L);
			memcpy(&pgLine, &newLine, sizeof(LineDEF));
			bModified = true;
			}
		if(Dlg->GetCheck(101)) defs.plLineDEF(&pgLine);
		bRet = true;
		break;
	case 2:							//Cancel
		if (Undo.level() > undo_level) {	//restore plot order
			while (Undo.level() > undo_level)	Undo.Restore(false, 0L);
			bRet = true;
			}
		break;
		}
	CloseDlgWnd(hDlg);		delete Dlg;
	free(LineDlg);			free(tab1);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// polygon properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *PolygDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,107,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,107,25,45,12\n"
	".,50,100,ISPARENT | CHECKED,SHEET,1,5,10,95,85\n"
	"50,, 600,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
	"100,+,0,NOSELECT,ODB,2,8,30,90,50\n"
	".,,,HICOL,CHECKBOX,3,10,82,60,9\n"
	"600,+,,TOUCHEXIT,ODB,4,107,52,15,15\n"
	".,., 0,TOUCHEXIT,ODB,4,127,52,15,15\n"
	".,., 0,TOUCHEXIT,ODB,4,127,67,15,15\n"
	".,., 0,LASTOBJ | TOUCHEXIT,ODB,4,107,67,15,15";

bool
polygon::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_POLGON);
	void *dyndata[] = { (void*)tab1, (void*)OD_filldef, (void*)SDLG_POLYGON_DEFAULT,
		(void*)OD_DrawOrder};
	DlgInfo *PolygDlg = CompileDialog(PolygDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	LineDEF newLine, newFillLine;
	static FillDEF newFill;
	DWORD undo_flags = 0L;
	int res, undo_level = Undo.level();
	anyOutput *cdisp = Undo.Cdisp();
	bool bRet = false;

	if(!parent || !PolygDlg) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&pgLine, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&pgFill, 0);
	Dlg = new DlgRoot(PolygDlg, data);
	if(parent->Id ==  GO_GRAPH || parent->Id == GO_PAGE) Dlg->ShowItem(50, true);
	hDlg = CreateDlgWnd((char*)SDLG_POLYGON_HD1, 50, 50, 330, 250, Dlg, 0x4L);
	do{
		LoopDlgWnd();			res = Dlg->GetResult();
		switch(res) {
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);		break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);	break;
			}
		}while (res < 0);
	switch (res) {
	case 1:						//OK pressed
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&newFill, 0);
		memcpy(&newFillLine, &pgFillLine, sizeof(LineDEF));
		if(newFill.hatch) memcpy(&newFillLine, newFill.hatch, sizeof(LineDEF));
		if(cmpLineDEF(&pgLine, &newLine)) {
			Undo.Line(this, &pgLine, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&pgLine, &newLine, sizeof(LineDEF));
			}
		if(newFill.type && cmpLineDEF(&pgFillLine, &newFillLine)) {
			Undo.Line(this, &pgFillLine, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&pgFillLine, &newFillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&pgFill, &newFill)) {
			Undo.Fill(this, &pgFill, undo_flags);		undo_flags |= UNDO_CONTINUE;
			memcpy(&pgFill, &newFill, sizeof(FillDEF));
			}
		pgFill.hatch = &pgFillLine;
		if(undo_flags & UNDO_CONTINUE) bModified = true;
		if(Dlg->GetCheck(101)){
			defs.pgLineDEF(&pgLine);			defs.pgFillDEF(&pgFill);
			}
		bRet = true;
		break;
	case 2:							//Cancel
		if (Undo.level() > undo_level) {	//restore plot order
			while (Undo.level() > undo_level)	Undo.Restore(false, 0L);
			bRet = true;
			}
		break;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	free(PolygDlg);			free(tab1);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// rectangle, round rectangle and ellipse properties dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *RecDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,117,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,117,25,45,12\n"
	".,,10,CHECKED | ISPARENT,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,100,97\n"
	".,50,200,ISPARENT,SHEET,2,5,10,100,97\n"
	"50,,600,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
	"100,+,0,NOSELECT,ODB,3,8,30,90,50\n"
	".,120,110,ISPARENT | CHECKED,GROUP,,,,,\n"
	"110,+,,,RTEXT,4,0,79,48,8\n"
	".,.,,,EDVAL1,5,50,79,25,10\n"
	".,,,,LTEXT,-3,77,79,10,8\n"
	"120,,,HICOL,CHECKBOX,6,10,92,60,9\n"
	"200,+,,,RTEXT,7,5,30,40,8\n"
	".,.,,,EDVAL1,8,47,30,38,10\n"
	".,.,,,LTEXT,-3,87,30,10,8\n"
	".,.,,,RTEXT,-5,5,42,40,8\n"
	".,.,,,EDVAL1,9,47,42,38,10\n"
	".,.,,,LTEXT,-3,87,42,10,8\n"
	".,.,,,RTEXT,10,5,55,40,8\n"
	".,.,,,EDVAL1,11,47,55,38,10\n"
	".,.,,,LTEXT,-3,87,55,10,8\n"
	".,.,,,RTEXT,-5,5,67,40,8\n"
	".,.,,,EDVAL1,12,47,67,38,10\n"
	".,,,,LTEXT,-3,87,67,10,8\n"
	"600,+,,TOUCHEXIT,ODB,13,115,64,15,15\n"
	".,.,,TOUCHEXIT,ODB,13,139,64,15,15\n"
	".,.,,TOUCHEXIT, ODB,13,139,79,15,15\n"
	".,,,LASTOBJ | TOUCHEXIT,ODB,13,115,79,15,15";

bool
rectangle::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = type == 1 ? MakeTab(0, 10, &i, (char*)SDLG_TAB_ELLIPSE) : 
		MakeTab(0, 10, &i, (char*)SDLG_TAB_RECTANGLE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)OD_filldef, (void*)"edge radius",
		(void*)&rad, (void*)"use this style as default", (void*)"top left x", (void*)&fp1.fx,
		(void*)&fp1.fy, (void*)"lower right x", (void*)&fp2.fx, (void*)&fp2.fy,
		(void*)OD_DrawOrder};
	DlgInfo *RecDlg = CompileDialog(RecDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, undo_level = Undo.level();
	lfPOINT old_fp1, new_fp1, old_fp2, new_fp2;
	LineDEF old_Line, new_Line, old_FillLine, new_FillLine;
	static FillDEF old_Fill, new_Fill;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	bool bRet = false;

	if(!parent || !RecDlg) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
	if(!(Dlg = new DlgRoot(RecDlg, data)))return false;
	Dlg->GetValue(201, &old_fp1.fx);		Dlg->GetValue(204, &old_fp1.fy);
	Dlg->GetValue(207, &old_fp2.fx);		Dlg->GetValue(210, &old_fp2.fy);
	if(type != 2) Dlg->ShowItem(101, false);
	if(parent->Id ==  GO_GRAPH || parent->Id == GO_PAGE) Dlg->ShowItem(50, true);
	OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&old_Line, 0);
	OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&old_Fill, 0);
	if(old_Fill.hatch) memcpy(&old_FillLine, old_Fill.hatch, sizeof(LineDEF));
	old_Fill.hatch = &old_FillLine;
	hDlg = CreateDlgWnd(type == 1 ? (char*)SDLG_REC_HD1 :
		type == 2 ? (char*)SDLG_REC_HD2 : (char*)SDLG_REC_HD3,
		50, 50, 360, 275, Dlg, 0x4L);
	do{
		LoopDlgWnd();			res = Dlg->GetResult();
		switch(res) {
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);		break;
		case 1:		case 2:
			Undo.SetDisp(cdisp);	break;
			}
		}while (res < 0);
	switch (res) {
	case 1:						//OK pressed
		Dlg->GetValue(201, &new_fp1.fx);		Dlg->GetValue(204, &new_fp1.fy);
		Dlg->GetValue(207, &new_fp2.fx);		Dlg->GetValue(210, &new_fp2.fy);
		undo_flags = CheckNewLFPoint(&fp1, &old_fp1, &new_fp1, this, undo_flags);
		undo_flags = CheckNewLFPoint(&fp2, &old_fp2, &new_fp2, this, undo_flags);
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&new_Line, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&new_Fill, 0);
		if(new_Fill.hatch) memcpy(&new_FillLine, new_Fill.hatch, sizeof(LineDEF));
		new_Fill.hatch = &new_FillLine;
		if(cmpLineDEF(&old_Line, &new_Line)) {
			Undo.Line(this, &Line, undo_flags);
			undo_flags |= UNDO_CONTINUE;
			memcpy(&Line, &new_Line, sizeof(LineDEF));
			}
		if(new_Fill.type && cmpLineDEF(&old_FillLine, &new_FillLine)) {
			Undo.Line(this, &FillLine, undo_flags);
			undo_flags |= UNDO_CONTINUE;
			memcpy(&FillLine, &new_FillLine, sizeof(LineDEF));
			}
		if(cmpFillDEF(&Fill, &new_Fill)) {
			Undo.Fill(this, &Fill, undo_flags);
			undo_flags |= UNDO_CONTINUE;
			memcpy(&Fill, &new_Fill, sizeof(FillDEF));
			}
		Fill.hatch = &FillLine;
		if(type == 2) Dlg->GetValue(111, &rad);
		else rad = 0.0;
		if(Dlg->GetCheck(120)){
			defs.pgLineDEF(&Line);			defs.pgFillDEF(&Fill);
			if(type == 2)defs.rrectRad(rad);
			}
		if(undo_flags) bModified = true;
		bRet = true;
		break;
	case 2:							//Cancel
		if (Undo.level() > undo_level) {	//restore plot order
			while (Undo.level() > undo_level)	Undo.Restore(false, 0L);
			bRet = true;
			}
		break;
		}
	CloseDlgWnd(hDlg);			delete Dlg;
	free(RecDlg);				free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a bar chart: note this is a PlotScatt function
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *BarChartDlg_DlgTmpl = (char*)
		"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
		".,.,,,PUSHBUTTON,-2,130,25,45,12\n"
		".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
		"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,120,110\n"
		".,10,200,ISPARENT,SHEET,2,5,10,120,110\n"
		"10,,,CHECKED,CHECKPIN,,5,,12,8\n"
		"100,+,,,LTEXT,3,10,25,60,8\n"
		".,.,,,RANGEINPUT,-15,10,38,110,10\n"
		".,,110,ISPARENT | CHECKED,GROUPBOX,4,10,55,110,61\n" 
		"110,,,NOSELECT,ODB,5,25,62,90,50\n"
		"200,+,,,RTEXT,6,10,35,38,8\n"
		".,.,,,EDVAL1,7,58,35,35,10\n"
		".,.,,,RTEXT,8,10,50,38,8\n"
		".,.,,,EDVAL1,9,58,50,35,10\n"
		".,.,,,RTEXT,10,10,65,38,8\n"
		".,.,,,EDVAL1,11,58,65,35,10\n"
		".,,,LASTOBJ,LTEXT,-10,95,65,8,8";

bool
PlotScatt::CreateBarChart()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAIL);
	double start = 1.0, step = 1.0, bw = 60.0;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_BARCHART_RANGE,
		(void*)SDLG_BARCHART_STYLE, (void*)OD_filldef, (void*)SDLG_BARCHART_START,
		(void*)&start, (void*)SDLG_BARCHART_STEP, (void*)&step,
		(void*)SDLG_BARCHART_WIDTH, (void*)&bw };
	DlgInfo *BarDlg = CompileDialog(BarChartDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	bool bRet = false;
	long k, l;
	int n, ic, res;
	double x, y;
	AccRange *rY = 0L;
	LineDEF Line;
	static FillDEF Fill; 

	if(!parent || !data || !BarDlg) return false;
	memcpy(&Line, defs.GetOutLine(), sizeof(LineDEF));
	memcpy(&Fill, defs.GetFill(defs.units()), sizeof(FillDEF));
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
	UseRangeMark(data, 1, TmpTxt);
	if(!(Dlg = new DlgRoot(BarDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_BARCHART_HD1, 50, 50, 380, 300, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:
			if(Dlg->GetCheck(10)) res = -1;
			break;
		case 1:
			if(rY) delete rY;
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)){
				rY = new AccRange(TmpTxt);
				yRange = rlp_strdup(TmpTxt);
				}
			else {
				rY = 0L;		yRange = 0L;
				}
			if(!(n = rY ? rY->CountItems() : 0)) {
				res = -1;
				ErrorBox((char*)SCMS_BAD_DATA_RANGE);
				}
			break;
			}
	} while(res < 0);
	if(res == 1 && n && rY) {
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&Line, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
		Command(CMD_FLUSH, 0L, 0L);
		nPoints = nBars = n;
		DefName((char*)SDLG_PLOTSCATT_ISBAR, 0L, rY);
		Bars = (Bar**)calloc((nBars = nPoints)+2, sizeof(Bar*));
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
		Dlg->GetValue(201, &start);	Dlg->GetValue(203, &step);
		if(step < 0.001) step = 1.0;
		Dlg->GetValue(205, &bw);
		if(bw < 5) bw = 60;
		ic = 0;
		x = start;
		if(Bars) do {
			if(data->GetValue(l, k, &y)){
				Bars[ic] = new Bar(this, data, x, y, BAR_VERTB | BAR_RELWIDTH, -1, -1, k, l);
				CheckBounds(x, y);
				x += step;
				ic++;
				}
			}while(rY->GetNext(&k, &l));
		if(ic){
			bRet = true;
			Command(CMD_BAR_FILL, &Fill, 0L);
			SetColor(COL_BAR_LINE, Line.color);
			SetSize(SIZE_BAR_LINE, Line.width);
			SetSize(SIZE_BAR, bw);
			BarDist.fx = step;
			}
		}
	if(rY) delete rY;
	CloseDlgWnd(hDlg);
	delete Dlg;
	free(BarDlg);				free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Scatter plot properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *PlotScatt_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,180,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,180,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,TOUCHEXIT | ISPARENT | CHECKED,SHEET,1,5,10,165,132\n"
	".,.,200,TOUCHEXIT | ISPARENT,SHEET,2,5,10,165,132\n"
	".,7,300,TOUCHEXIT | ISPARENT,SHEET,3,5,10,165,132\n"
	".,10,400,TOUCHEXIT | ISPARENT,SHEET,4,5,10,165,132\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,25,60,8\n"
	".,.,,,RANGEINPUT,5,20,35,130,10\n"
	".,.,,,LTEXT,-32,10,50,60,8\n"
	".,.,,,RANGEINPUT,6,20,60,130,10\n"
	".,.,,,LTEXT,39,10,75,60,8\n"
	".,.,,,EDTEXT,40,45,75,105,10\n"
	".,.,,,ICON,7,10,95,10,10\n"
	".,.,,,LTEXT,8,35,90,30,6\n"
	".,.,,,LTEXT,9,35,96,30,6\n"
	".,.,,,LTEXT,10,35,102,30,6\n"
	".,.,,,LTEXT,39,35,111,30,6\n"
	".,.,,,LTEXT,41,35,117,30,6\n"
	".,.,,,LTEXT,42,35,123,30,6\n"
	".,,,,LTEXT,43,35,129,30,6\n"
	"200,+,,HICOL,CHECKBOX,11,25,30,60,8\n"
	".,.,250,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,255,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,,HICOL,CHECKBOX,12,25,105,60,8\n"
	".,.,,HICOL,CHECKBOX,13,25,120,60,8\n"
	".,.,,TOUCHEXIT | OWNDIALOG,LINEBUTT,26,105,45,30,9\n"
	".,.,,TOUCHEXIT | OWNDIALOG,BARBUTT,27,108,60,24,22\n"
	".,.,,TOUCHEXIT | OWNDIALOG,SYMBUTT,28,105,30,30,10\n"
	".,.,,TOUCHEXIT | OWNDIALOG,ARROWPREV,44,105,105,30,10\n"
	".,,,TOUCHEXIT | OWNDIALOG,FILLBUTTON,45,105,90,30,10\n"
	"250,+,,HICOL | ISRADIO,CHECKBOX,14,25,45,60,8\n"
	".,,,HICOL | ISRADIO,CHECKBOX,15,25,90,60,8\n"
	"255,+,,HICOL | ISRADIO,CHECKBOX,16,25,60,60,8\n"
	".,,,HICOL | ISRADIO,CHECKBOX,17,25,75,60,8\n"
	"300,+,500,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,,HICOL,CHECKBOX,18,15,28,50,8\n"
	".,.,,,LTEXT,19,35,50,20,8\n"
	".,.,,,LTEXT,20,15,109,60,8\n"
	".,.,,TOUCHEXIT,RANGEINPUT,21,20,120,130,10\n"
	".,.,,HICOL,CHECKBOX,46,15,84,50,8\n"
	".,.,,,LTEXT,47,30,94,30,8\n"
	".,,,TOUCHEXIT | OWNDIALOG,FILLBUTTON,48,95,94,30,10\n"
	"400,+,,HICOL,CHECKBOX,22,15,28,50,8\n"
	".,.,,,LTEXT,23,15,40,60,8\n"
	".,.,,TOUCHEXIT,RANGEINPUT,24,20,51,130,10\n"
	".,.,,,LTEXT,29,15,66,60,8\n"
	".,.,,TOUCHEXIT,EDTEXT,30,55,66,30,10\n"
	".,.,,,LTEXT,31,89,66,60,8\n"
	".,.,,,LTEXT,32,13,80,50,8\n"
	".,.,,,RTEXT,33,13,92,15,8\n"
	".,.,,TOUCHEXIT,EDVAL1,34,30,92,35,10\n"
	".,.,,,LTEXT,-3,68,92,10,8\n"
	".,.,,,RTEXT,35,13,106,15,8\n"
	".,.,,TOUCHEXIT,EDVAL1,36,30,106,35,10\n"
	".,.,,,LTEXT,-3,68,106,10,8\n"
	".,.,,,LTEXT,37,100,80,50,8\n"
	".,,,,TXTHSP,38,100,92,45,25\n"
	"500,+,.,TOUCHEXIT|CHECKED|ISRADIO,ODBUTTON,25,60,40,20,20\n"
	".,.,.,TOUCHEXIT | ISRADIO,ODBUTTON,25,80,40,20,20\n"
	".,.,.,TOUCHEXIT | ISRADIO,ODBUTTON,25,100,40,20,20\n"
	".,.,.,TOUCHEXIT | ISRADIO,ODBUTTON,25,60,60,20,20\n"
	".,.,.,TOUCHEXIT | ISRADIO,ODBUTTON,25,80,60,20,20\n"
	".,,.,LASTOBJ | TOUCHEXIT | ISRADIO,ODBUTTON,25,100,60,20,20";

bool
PlotScatt::PropertyDlg()
{
	int id;
	TabSHEET *tab1 = MakeTab(0, 10, &id, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(id, 10, &id, (char*)SDLG_TAB_LAYOUT);
	TabSHEET *tab3 = MakeTab(id, 10, &id, (char*)SDLG_TAB_ERRORS);
	TabSHEET *tab4 = MakeTab(id, 10, &id, (char*)SDLG_TAB_LABELS);
	char text1[100], text2[100], text3[100], text4[100], text5[50], deftxt[20];
	LineDEF Line;
	Symbol *PrevSym = new Symbol(0L, data, 0.0, 0.0, 0);
	Arrow *PrevArr = new Arrow(0L, data, 0L, 0L, ARROW_LINE);
	Bar *PrevBar = new Bar(0L, data, 0.0, 0.0, BAR_RELWIDTH | BAR_VERTB);
	LineDEF PG_FillLine = { defs.GetSize(SIZE_HAIRLINE), 6.0f, 0x0, 0x0 };
	static FillDEF PG_Fill = { FILL_NONE, 0x00cbcbcbL, 1.0f, &PG_FillLine, 0x00cbcbcbL };
	LineDEF ERRPG_FillLine = { defs.GetSize(SIZE_HAIRLINE), 6.0f, 0x0, 0x0 };
	static FillDEF ERRPG_Fill = { FILL_NONE, 0x00cbcbcbL, 1.0f, &ERRPG_FillLine, 0x00cbcbcbL };
	memcpy(&Line, defs.GetLine(), sizeof(LineDEF));
	ERRPG_Fill.color = ERRPG_Fill.color2 = 0xa4808080;
	ERRPG_Fill.hatch = &ERRPG_FillLine;	
	int icon = ICO_INFO;
	TextDEF lbdef = {defs.Color(COL_TEXT), defs.Color(COL_BG), DefSize(SIZE_TEXT), 0.0f, 0.0f, 0,
		TXA_HLEFT | TXA_VBOTTOM, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, (unsigned char*)TmpTxt};
	lbDist.fx = defs.GetSize(SIZE_SYMBOL) / 2.0; 		lbDist.fy = -lbDist.fx;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3,(void*)tab4, (void*)text1,
		(void*)text2, (void*)&icon, (void*)SDLG_PLOTSCATT_INFO1, (void*)SDLG_PLOTSCATT_INFO2,
		(void*)SDLG_PLOTSCATT_INFO3, (void*)SDLG_PLOTSCATT_DOSYM, (void*)SDLG_PLOTSCATT_DOARR,
		(void*)SDLG_PLOTSCATT_DODROP, (void*)SDLG_PLOTSCATT_DOLIN, (void*)SDLG_PLOTSCATT_DOPG,
		(void*)SDLG_PLOTSCATT_DOVBAR, (void*)SDLG_PLOTSCATT_DOHBAR, (void*)SDLG_PLOTSCATT_ERRS,
		(void*)SDLG_PLOTSCATT_STYLE, (void*)SDLG_PLOTSCATT_ERRANGE, (void*)text3, (void*)SDLG_PLOTSCATT_ADDLAB,
		(void*)SDLG_PLOTSCATT_LABRANGE, (void*)text4, (void*)(OD_ErrBarTempl), (void*)&Line, (void*)&PrevBar,
		(void*)&PrevSym, (void*)SDLG_PLOTSCATT_DEFTXT, (void*)SDLG_PLOTSCATT_NA, (void*)SDLG_PLOTSCATT_EMPTY,
		(void*)SDLG_PLOTSCATT_DIST, (void*)SDLG_PLOTSCATT_DX, (void*)&lbDist.fx, (void*)SDLG_PLOTSCATT_DY,
		(void*)&lbDist.fy, (void*)SDLG_PLOTSCATT_ALIGN, (void*)&(lbdef.Align), (void*)SDLG_PLOTSCATT_COND,
		(void*)&text5, (void*)SDLG_PLOTSCATT_INFO4, (void*)SDLG_PLOTSCATT_INFO5, (void*)SDLG_PLOTSCATT_INFO6,
		(void*)PrevArr, (void*)&PG_Fill, (char*)SDLG_PLOTSCATT_ERRPG, (char*)SDLG_PLOTSCATT_ERRPGSTYLE, (void*)&ERRPG_Fill };
	DlgInfo *XYDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int i1, j1, k1, l1, ic = 0;
	long i, j, m, n, o, p, c, k, l;
	int ErrType = 0, res, BarType, nVals, nTxt, nTime;
	double x = 0.0, y = 0.0, e = 0.0, ebw;
	lfPOINT fp1, fp2;
	bool bRet = false, bLayout = false, bContinue = false, bValid;
	anyResult xRes, yRes, *cRes;
	AccRange *rX, *rY, *rE, *rL;

	if(!parent || !data) return false;
	if(!(XYDlg = CompileDialog(PlotScatt_Tmpl, dyndata))) return false;
	if(Id == GO_BARCHART) return CreateBarChart();
	UseRangeMark(data, 1, text1, text2, text3, text4);
	rX = rY = rE = rL = 0L;
	text5[0] = 0;
	if(!(Dlg = new DlgRoot(XYDlg, data)))return false;
#ifdef _WINDOWS
	for(i = 106; i <= 113; i++) Dlg->TextSize(i, 12);
#else
	for(i = 106; i <= 113; i++) Dlg->TextSize(i, 10);
#endif
	if(DefSel & 0x01)Dlg->SetCheck(200, 0L, true);
	if(DefSel & 0x02)Dlg->SetCheck(250, 0L, true);
	if(DefSel & 0x04)Dlg->SetCheck(255, 0L, true);
	if(DefSel & 0x08)Dlg->SetCheck(256, 0L, true);
	hDlg = CreateDlgWnd((char*)SDLG_PLOTSCATT_HD1, 50, 50, 480, 345, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;									break;
		case 304:
			Dlg->SetCheck(301, 0L, true);	//editing the error bar tab
			res = -1;						break;
		case 307:
			Dlg->SetCheck(305, 0L, true);
			res = -1;						break;
		case 402:	case 404:	case 408:	case 411:	case 414:
			Dlg->SetCheck(400, 0L, true);   //editing the data label tab
			res = -1;											break;
		case 500:	case 501:	case 502:
		case 503:	case 504:	case 505:
			ErrType = res-500;
			Dlg->SetCheck(301, 0L, true);
			res = -1;											break;
		case 205:
			Dlg->SetCheck(250, 0L, true);
			res = -1; bContinue = false;	//line properties
			break;
		case 206:
			if (!Dlg->GetCheck(255) && !Dlg->GetCheck(256)) Dlg->SetCheck(255, 0L, true);
			res = -1; bContinue = false;						break;
		case 207:							//symbol properties
			Dlg->SetCheck(200, 0L, true);
			res = -1; bContinue = false;						break;
		case 208:							//arrow properties
			Dlg->SetCheck(203, 0L, true);
			res = -1; bContinue = false;						break;
		case 209:
			Dlg->SetCheck(251, 0L, true);
			res = -1; bContinue = false;	//polygon fill properties
			break;
		case 4:								// the data tab sheet
			Dlg->Activate(101, 1);	res = -1;					break;
		case 5:								// the layout tab sheet
			bLayout = true;				res = -1;				break;
		case 6:								// the error tab sheet
			Dlg->Activate(304, 1);	res = -1;					break;
		case 7:								// the label tab sheet
			Dlg->Activate(402, 1);	res = -1;					break;
		case 1:								// OK
			if(rX) delete rX;
			if(rY) delete rY;
			if(rE) delete rE;
			rX = 0L; rY = 0L; rE = 0L;						// check x-range
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rX = new AccRange(TmpTxt);
			if(!(n = rX ? rX->CountItems() : 0)) {
				Dlg->SetCheck(4, 0L, true);
				res = -1;
				bContinue = true;
				ErrorBox((char*)SCMS_BAD_XRANGE);
				}
			else {							// check y-range
				if(Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rY = new AccRange(TmpTxt);
				if(n != (rY ? rY->CountItems() : 0)) {
					Dlg->SetCheck(4, 0L, true);
					res = -1;
					bContinue = true;
					ErrorBox((char*)SCMS_BAD_YRANGE);
					}
				}
			//check for error bar
			if(res >0 && (Dlg->GetCheck(301) || Dlg->GetCheck(305))) {
				if(Dlg->GetText(304, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rE = new AccRange(TmpTxt);
				if (n != (rE ? rE->CountItems() : 0)) {
					Dlg->SetCheck(6, 0L, true);
					res = -1;
					bContinue = true;
					ErrorBox((char*)SCMS_BAD_ERRANGE);
					if(rE) delete (rE);
					rE = 0L;
					}
				}
			//check for data labels
			if(res >0 && Dlg->GetCheck(400)) {
				if(!(Dlg->GetText(404, (unsigned char*)deftxt, 19))) deftxt[0] = 0;
				Dlg->GetInt(414, &lbdef.Align);
				Dlg->GetValue(408, &lbDist.fx);		Dlg->GetValue(411, &lbDist.fy);
				if(Dlg->GetText(402, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rL = new AccRange(TmpTxt);
				if(n != (rL ? rL->CountItems() : 0)) {
					Dlg->SetCheck(7, 0L, true);
					res = -1;							bContinue = true;
					ErrorBox((char*)SCMS_BAD_LBRANGE);
					if(rL) delete (rL);
					rL = 0L;
					}
				}
			//check if something left to do
			if(res > 0 && !rE && !rL && !Dlg->GetCheck(200) && !Dlg->GetCheck(250) && 
				!Dlg->GetCheck(251) && !Dlg->GetCheck(255) && !Dlg->GetCheck(256) && 
				!Dlg->GetCheck(203) && !Dlg->GetCheck(301) && !Dlg->GetCheck(307)) {
				Dlg->SetCheck(5, 0L, true);
				res = -1;								bContinue = true;
				ErrorBox((char*)SCMS_NOPLOTTING);
				}
			//the layout menu must have been visited
			if(res > 0 && !bLayout){
				Dlg->SetCheck(5, 0L, bLayout = true);	res = -1;
				}
			if(Dlg->GetText(105, (unsigned char*)text5, 49) && text5[0]){
				cRes = do_formula(data, text5);
				if(cRes->type != ET_BOOL) {
					Dlg->SetCheck(5, 0L, true);
					ErrorBox((char*)SDLG_PLOTSCATT_EXPRERR);
					Dlg->SetCheck(4, 0L, true);			res = -1;
					}
				}
			break;
			}
		}while (res <0);
	if(res == 1 && n && rX && rY){				//OK pressed
		Command(CMD_FLUSH, 0L, 0L);
		nPoints = n;				y_info = rlp_strdup(rY->RangeDesc(data, 1));
		ebw = defs.GetSize(SIZE_ERRBAR)*9.0 / ((double)nPoints);
		if (Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) xRange = rlp_strdup(TmpTxt);
		if(Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) yRange = rlp_strdup(TmpTxt);
		if (Dlg->GetCheck(305)) {
			Dlg->GetText(304, (unsigned char*)TmpTxt, TMP_TXT_SIZE);
			ErrRange = rlp_strdup(TmpTxt);
			ErrPg = new ErrorPolygon(this, data, xRange, yRange, ErrRange, (char*)"Error Polygon");
			ErrPg->Command(CMD_PG_FILL, &ERRPG_Fill, 0L);
			}
		Bounds.Xmin = HUGE_VAL; Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = -HUGE_VAL; Bounds.Ymax = -HUGE_VAL;
		DefName((char*)SDLG_PLOTSCATT_ISXY, rX, rY);
		//analyse data types
		if(rX->DataTypes(data, &nVals, &nTxt, &nTime) && !x_tv){
			if(!nVals && nTxt > 1 && nTxt > nTime) x_tv = new TextValue();
			else if(!nVals && nTime > 1 && nTime > nTxt) x_dtype = ET_DATETIME;
			}
		if(rY->DataTypes(data, &nVals, &nTxt, &nTime) && !y_tv){
			if(!nVals && nTxt > 1 && nTxt > nTime) y_tv = new TextValue();
			else if(!nVals && nTime > 1 && nTime > nTxt) y_dtype = ET_DATETIME;
			}
		//Create graphic objects
		rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
		rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
		i1 = i;	j1 = j;	k1 = k;	l1 = l;
		ic = c = 0;
		if(Dlg->GetCheck(200)) Symbols = (Symbol**)calloc(nPoints+2, sizeof(Symbol*));
		if((BarType = Dlg->GetCheck(255) ? BAR_VERTB : Dlg->GetCheck(256) ? BAR_HORL : 0))
			Bars = (Bar**)calloc(nPoints+2, sizeof(Bar*));
		BarType |= BAR_RELWIDTH;
		if(Dlg->GetCheck(204)) DropLines = (DropLine**)calloc(nPoints+2, sizeof(DropLine*));
		if(Dlg->GetCheck(203)) Arrows = (Arrow**)calloc(nPoints+2, sizeof(Arrow*));
		if(Dlg->GetCheck(301) && rE) {					//error bars ?
			Errors = (ErrorBar**)calloc(nPoints+2, sizeof(ErrorBar*));
			if(Dlg->GetText(304, (unsigned char*)TmpTxt, TMP_TXT_SIZE)){
				ErrRange = rlp_strdup(TmpTxt);
				rE->GetFirst(&m, &n);	rE->GetNext(&m, &n);
				if (Bars && Dlg->GetCheck(255)) ErrType |= 0x300;
				}
			else {
				rE = 0L;		ErrRange = 0L;
				}
			}
		if(Dlg->GetCheck(400) && rL) {					//labels ?
			Labels = (Label**)calloc(nPoints+2, sizeof(Label*));
			if(Dlg->GetText(402, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) LbRange = rlp_strdup(TmpTxt);
			rL->GetFirst(&o, &p);	rL->GetNext(&o, &p);
			}
		bContinue = false;
		do {
			bValid = false;
			if(data->GetResult(&xRes, j, i, false) && data->GetResult(&yRes, l, k, false) 
				&& xRes.type != ET_UNKNOWN && yRes.type != ET_UNKNOWN) {
				bValid = true;
				if(x_tv) {
					if (xRes.type == ET_TEXT){
						if (y_tv || bContinue) x = x_tv->GetValue(xRes.text);
						else bValid = false;
						bContinue = true;
						}
					else bValid = false;
					}
				else if(x_dtype == ET_DATETIME) {
					if(xRes.type == ET_DATE || xRes.type == ET_TIME  || xRes.type == ET_DATETIME) x = xRes.value;
					else bValid = false;
					}
				else {
					if(xRes.type == ET_VALUE) x = xRes.value;
					else bValid = false;
					}
				if(y_tv) {
					if(yRes.type == ET_TEXT) y = y_tv->GetValue(yRes.text);
					else bValid = false;
					}
				else if(y_dtype == ET_DATETIME) {
					if(yRes.type == ET_DATE || yRes.type == ET_TIME  || yRes.type == ET_DATETIME) y = yRes.value;
					else bValid = false;
					}
				else {
					if(yRes.type == ET_VALUE ) y = yRes.value;
					else bValid = false;
					}
				}
			if(bValid && text5[0]) {
#ifdef USE_WIN_SECURE
				sprintf_s((char*)TmpTxt, 100, "x=%g;y=%g;%s", x, y, text5);
#else
				sprintf((char*)TmpTxt, "x=%g;y=%g;%s", x, y, text5);
#endif
				cRes = do_formula(data, TmpTxt);
				if(cRes->type == ET_BOOL) bValid = (cRes->value != 0.0);
				}
			if (bValid){
				if (Symbols && (Symbols[ic] = new Symbol(this, data, x, y, DefSym, i, j, k, l))){
					Symbols[ic]->Command(CMD_PREVSYM, PrevSym, 0L);
					Symbols[ic]->idx = c;
					}
				if (Bars && (Bars[ic] = new Bar(this, data, x, y, BarType, i, j, k, l))){
					Bars[ic]->Command(CMD_PREVSYM, PrevBar, 0L);
					Bars[ic]->type = (BarType & ~BAR_RELWIDTH) | (PrevBar->type & BAR_RELWIDTH);
					}
				if(DropLines) DropLines[ic] = new DropLine(this, data, x, y, 
					DL_YAXIS | DL_XAXIS, i, j, k, l);
				if(Arrows) {
					if(ic){
						fp1.fx = fp2.fx;	fp1.fy = fp2.fy;
						}
					else {
						fp1.fx = x;			fp1.fy = y;
						}
					fp2.fx = x;			fp2.fy = y;
					Arrows[ic] = new Arrow(this, data, ic ? &fp1 : &fp2, &fp2, (PrevArr->type) & 0x2ff,
						i1, j1, k1, l1, i, j, k, l);
					//the first arrow has zero length
					//all other arrows conncect to the following point
					i1 = i;	j1 = j;	k1 = k;	l1 = l;
					}
				if(Labels && rL) {
					if(data->GetText(p, o, TmpTxt, TMP_TXT_SIZE)) {
						lbdef.text = (unsigned char*)TmpTxt;
						}
					else lbdef.text = (unsigned char*)deftxt;
					if(lbdef.text && lbdef.text[0]){
						Labels[ic] = new Label(this, data, x, y, &lbdef, 
							LB_X_DATA | LB_Y_DATA, 0L, i, j, k, l, o, p);
						}
					rL->GetNext(&o, &p);
					}
				if(Errors && rE){
					if(data->GetValue(n, m, &e))
						Errors[ic]= new ErrorBar(this, data, x, y, e, ErrType,
							i, j, k, l, m, n);
					Errors[ic]->SetSize(SIZE_ERRBAR, ebw);
					rE->GetNext(&m, &n);
					}
				else CheckBounds(x, y);
				ic++;
				}
			else {
				if(Labels && rL) rL->GetNext(&o, &p);
				if(Errors && rE) rE->GetNext(&m, &n);
				}
			c++;
			}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
		if (Arrows) {
			nArrow = nPoints;
			SetSize(SIZE_ARROW_LINE, PrevArr->GetSize(SIZE_ARROW_LINE));
			SetSize(SIZE_ARROW_CAPWIDTH, PrevArr->GetSize(SIZE_ARROW_CAPWIDTH));
			SetSize(SIZE_ARROW_CAPLENGTH, PrevArr->GetSize(SIZE_ARROW_CAPLENGTH));
			SetColor(COL_ARROW, PrevArr->GetColor(COL_ARROW));
			}
		if(ic) bRet = true;
		if (Dlg->GetCheck(250) && nPoints >1) {
			TheLine = new DataLine(this, data, xRange, yRange, 0L, text5[0] ? text5 : 0L);
			TheLine->Command(CMD_SET_LINE, (void*)&Line, 0L);
			}
		else if (Dlg->GetCheck(251) && nPoints > 2){
			TheLine = new DataPolygon(this, data, xRange, yRange);
			TheLine->Command(CMD_PG_FILL, (void*)&PG_Fill, 0L);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rX) delete rX;	
	if(rY) delete rY;	
	if(rE) delete rE;	
	if(rL) delete rL;
	if (ErrPg) ErrPg->Update();
	if(ic) {
		nPoints = ic;
		if(Bars) nBars = nPoints;
		if(Errors) nErrs = nPoints;	
		if(Labels) nLabel = nPoints;
		if(DropLines) nDrops = nPoints;
		if(Arrows) nArrow = nPoints;
		}
	delete PrevSym;		PrevSym = 0L;	delete PrevArr;		PrevArr = 0L;
	delete PrevBar;		PrevBar = 0L;
	free(XYDlg);	free(tab1);		free(tab2);		free(tab3);		free(tab4);
	return (dirty = bRet);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// calculate means and error to create a xy-plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *StatDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
	".,10,,,PUSHBUTTON,-2,130,25,45,12\n"
	"10,100,,CHECKED,CHECKPIN,,5,0,12,8\n"
	"100,+,,,LTEXT,1,10,10,90,9\n"
	".,.,,,LTEXT,2,10,20,60,9\n"
	".,.,,,RANGEINPUT,3,10,30,100,10\n"
	".,.,,,LTEXT,4,10,45,90,9\n"
	".,200,,,RANGEINPUT,5,10,55,100,10\n"
	"200,300,201,ISPARENT | CHECKED,GROUPBOX,6,10,75,165,50\n"
	"201,+,,CHECKED | HICOL,CHECKBOX,7,15,80,50,9\n"
	".,.,,CHECKED | HICOL,CHECKBOX,8,15,90,50,9\n"
	".,.,,HICOL,CHECKBOX,9,15,100,50,9\n"
	".,.,,,LTEXT,10,65,90,30,9\n"
	".,.,,CHECKED | HICOL,RADIO1,11,95,80,50,9\n"
	".,.,,HICOL,RADIO1,12,95,90,50,9\n"
	".,.,,HICOL,RADIO1,13,95,100,50,9\n"
	".,,,HICOL,RADIO1,14,95,110,50,9\n"
	"300,400,301,ISPARENT | CHECKED, GROUPBOX,15,10,130,165,40\n"
	"301,+,,ISRADIO | CHECKED | HICOL,CHECKBOX,16,15,135,70,9\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,17,15,145,70,9\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,18,95,135,70,9\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,19,95,145,70,9\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,,15,155,70,9\n"
	".,.,,,EDVAL1,20,28,154,15,10\n"
	".,,,,LTEXT,21,45,155,70,9\n"
	"400,,401,ISPARENT | CHECKED,GROUPBOX,22,10,175,165,30\n"
	".,.,,ISRADIO | CHECKED | HICOL,CHECKBOX,23,15,180,70,9\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,24,95,180,70,9\n"
	".,.,,,LTEXT,25,15,190,24,9\n"
	".,,,LASTOBJ,EDTEXT,26,40,189,30,10";

bool
xyStat::PropertyDlg()
{
	unsigned char text1[100], text2[100];
	void *dyndata[] = { (void*)SDLG_XYSTAT_GROUP, (void*)SDLG_XYSTAT_XDTA, (void*)text1,
		(void*)SDLG_XYSTAT_YDTA, (void*)text2, (void*)SDLG_XYSTAT_MEANS, (void*)SDLG_XYSTAT_DOLINE,
		(void*)SDLG_XYSTAT_DOSYM, (void*)SDLG_XYSTAT_DOBAR, (void*)SDLG_XYSTAT_USE, (void*)SDLG_XYSTAT_AMEAN,
		(void*)SDLG_XYSTAT_GMEAN, (void*)SDLG_XYSTAT_HMEAN, (void*)SDLG_XYSTAT_MEDIAN,
		(void*)SDLG_XYSTAT_EBARS, (void*)SDLG_XYSTAT_SDEV, (void*)SDLG_XYSTAT_SEM, (void*)SDLG_XYSTAT_PERC,
		(void*)SDLG_XYSTAT_MIMA, (void*)&ci, (void*)SDLG_XYSTAT_CONF, (void*)SDLG_XYSTAT_NC,
		(void*)SDLG_XYSTAT_TOPE, (void*)SDLG_XYSTAT_TOPM, (void*)SDLG_XYSTAT_PREF, (void*)SDLG_XYSTAT_NPREF };
	DlgInfo *StatDlg = CompileDialog(StatDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	bool bRet = false;
	int i, res, cb_mdesc;
	long width, height;
	double x, y, e, f, dx, dy;
	lfPOINT fp1, fp2;
	char errdesc[40], *mdesc;
	TextDEF lbdef = {defs.Color(COL_TEXT), defs.Color(COL_BG), DefSize(SIZE_TEXT), 0.0f, 0.0f, 0,
		TXA_HLEFT | TXA_VBOTTOM, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, (unsigned char*)TmpTxt};
	if(!parent || !data) return false;
	UseRangeMark(data, 1, (char*)text1, (char*)text2);
	if(!(Dlg = new DlgRoot(StatDlg, data)))return false;
	text1[0] = text2[0] = 0;
	hDlg = CreateDlgWnd((char*)SDLG_XYSTAT_HD1, 50, 50, 390, 470, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:
			if(Dlg->GetCheck(10)) res=-1;
			break;
		case 1:
			if(!(Dlg->GetText(102, text1, 100) && Dlg->GetText(104, text2, 100) && text1[0] && text2[0])) res = 2;
			break;
			}
		}while (res <0);
	if(res == 1) {
		xRange = rlp_strdup((char*)text1);
		yRange = rlp_strdup((char*)text2);
		type = 0;
		if(Dlg->GetCheck(201)) type |= 0x0001;
		if(Dlg->GetCheck(202)) type |= 0x0002;
		if(Dlg->GetCheck(203)) type |= 0x0004;
		if(Dlg->GetCheck(205)) type |= 0x0010;
		if(Dlg->GetCheck(206)) type |= 0x0020;
		if(Dlg->GetCheck(207)) type |= 0x0040;
		if(Dlg->GetCheck(208)) type |= 0x0080;
		if(Dlg->GetCheck(301)) type |= 0x0100;
		if(Dlg->GetCheck(302)) type |= 0x0200;
		if(Dlg->GetCheck(303)) type |= 0x0400;
		if(Dlg->GetCheck(304)) type |= 0x0800;
		if(Dlg->GetCheck(305)) type |= 0x1000;
		if(Dlg->GetCheck(401)) type |= 0x2000;
		if(Dlg->GetCheck(402)) type |= 0x4000;
		Dlg->GetValue(306, &ci);					TmpTxt[0] = 0;
		if(Dlg->GetText(404, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0]) 
			case_prefix = rlp_strdup(TmpTxt);
		CreateData();
		if(type && curr_data) {
			switch (type & 0x00f0) {
				case 0x0010:		mdesc = (char*)"Mean";					break;
				case 0x0020:		mdesc = (char*)"Geometric mean";		break;
				case 0x0040:		mdesc = (char*)"Harmonic mean";			break;
				case 0x0080:		mdesc = (char*)"Median";				break;
				default:			mdesc = (char*)"n.a.";					break;
				}
			cb_mdesc = rlp_strlen(mdesc);
			curr_data->GetSize(&width, &height);		nPoints = height;
#ifdef USE_WIN_SECURE
			sprintf_s((char*)text1, 100, "a1:a%ld", height);	sprintf_s((char*)text2, 100, "b1:b%ld", height);
#else
			sprintf((char*)text1, "a1:a%ld", height);			sprintf((char*)text2, "b1:b%ld", height);
#endif
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			if(type & 0x0001) {
				if(nPoints >1 && (TheLine = new DataLine(this, curr_data, (char*)text1, (char*)text2)) &&
					(TheLine->name = (char*)malloc(cb_mdesc+2))) rlp_strcpy(TheLine->name, cb_mdesc+2, mdesc);
				}
			if((type & 0x0002) && (Symbols = (Symbol**)calloc(nPoints+2, sizeof(Symbol*)))) {
				for(i = 0; i < height; i++) {
					if(curr_data->GetValue(i, 0, &x) &&	curr_data->GetValue(i, 1, &y)
						&& (Symbols[i] = new Symbol(this, curr_data, x, y, DefSym, 0, i, 1, i))){
						Symbols[i]->idx = i;
						Symbols[i]->name = rlp_strdup(mdesc);
						}
					}
				}
			if((type & 0x0004) && (Bars = (Bar**)calloc(nPoints+2, sizeof(Bar*)))) {
				for(i = 0; i < height; i++) {
					if(curr_data->GetValue(i, 0, &x) &&	curr_data->GetValue(i, 1, &y)
						&& (Bars[i] = new Bar(this, curr_data, x, y, BAR_VERTB | BAR_RELWIDTH, 0, i, 1, i))){
						Bars[i]->name = rlp_strdup(mdesc);
						}
					}
				}
			if(type & 0x1f00) {
				Errors = (ErrorBar**)calloc(nPoints+2, sizeof(ErrorBar*));
				switch(type & 0x1f00) {
				case 0x0100:	rlp_strcpy(errdesc, 40, (char*)"Std. Dev.");			break;
				case 0x0200:	rlp_strcpy(errdesc, 40, (char*)"Std. Err.");			break;
				case 0x0400:	rlp_strcpy(errdesc, 40, (char*)"25, 75% Perc.");		break;
				case 0x0800:	rlp_strcpy(errdesc, 40, (char*)"Min./Max.");			break;
#ifdef USE_WIN_SECURE
				case 0x1000:	sprintf_s(errdesc, 40, "'%g%% CI", ci);			break;
#else
				case 0x1000:	sprintf(errdesc, "'%g%% CI", ci);				break;
#endif
				default:		rlp_strcpy(errdesc, 40, (char*)"error");
					}
				}
			if((type & 0x1300) && Errors) {
				for(i = 0; i < height; i++) {
					if(curr_data->GetValue(i, 0, &x) &&	curr_data->GetValue(i, 1, &y)
						&& curr_data->GetValue(i, 2, &e)) {
						Errors[i]= new ErrorBar(this, curr_data, x, y, e, 0, 0, i, 1, i, 2, i);
						if(Errors[i]) Errors[i]->Command(CMD_ERRDESC, errdesc, 0L);
						}
					}
				}
			if((type & 0x0c00) && Errors) {
				for(i = 0; i < height; i++) {
					if(curr_data->GetValue(i, 0, &x) && curr_data->GetValue(i, 2, &e) && curr_data->GetValue(i, 3, &f)){
						fp1.fx = fp2.fx = x;	fp1.fy = e;		fp2.fy = f;
						Errors[i]= (ErrorBar*)new Whisker(this, curr_data, fp1, fp2, 0, 0, i, 2, i, 0, i, 3, i);
						if(Errors[i]) Errors[i]->Command(CMD_ERRDESC, errdesc, 0L);
						}
					}
				}
			if((type & 0x6000) && (Labels = (Label**)calloc(nPoints+2, sizeof(Label*)))) {
				dy = -0.4 * DefSize(SIZE_SYMBOL);
				if(type & 0x2000){
					lbdef.Align = TXA_HCENTER | TXA_VBOTTOM;
					dx = 0.0;
					}
				else {
					lbdef.Align = TXA_HLEFT | TXA_VBOTTOM;
					dx = -dy;
					}
				for(i = 0; i < height; i++) {
					if(curr_data->GetValue(i, 0, &x) && curr_data->GetText(i, 5, TmpTxt, TMP_TXT_SIZE, false)){
						if((type & 0x2000) && curr_data->GetValue(i, 4, &y)) 
							Labels[i] = new Label(this, curr_data, x, y, &lbdef, 
							LB_X_DATA | LB_Y_DATA, 0L, 0, i, 4, i, 5, i);
						else if((type & 0x4000) && curr_data->GetValue(i, 1, &y)) 
							Labels[i] = new Label(this, curr_data, x, y, &lbdef, 
							LB_X_DATA | LB_Y_DATA, 0L, 0, i, 1, i, 5, i);
						if(Labels[i]){
							Labels[i]->SetSize(SIZE_LB_YDIST, dy);
							Labels[i]->SetSize(SIZE_LB_XDIST, dx);
							}
						}
					}
				}
			Command(CMD_AUTOSCALE, 0L, 0L);
			bRet = true;
			}
		if(Bars) nBars = nPoints;
		if(Labels) nLabel = nPoints;
		if(Errors) nErrs = nPoints;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(StatDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Frequency distribution
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *FreqDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,130,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,120,153\n"
	".,10,200,ISPARENT,SHEET,2,5,10,120,153\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,3,10,25,60,8\n"
	".,.,,,RANGEINPUT,-15,10,38,110,10\n"
	".,.,120,ISPARENT | CHECKED,GROUPBOX,4,10,55,110,42\n"
	".,,150,ISPARENT | CHECKED,GROUPBOX,5,10,102,110,57\n"
	"120,+,,CHECKED | HICOL,RADIO1,6,15,60,30,9\n"
	".,.,,,EDVAL1,7,47,60,15,10\n"
	".,.,,,LTEXT,8,64,60,35,8\n"
	".,.,,HICOL,RADIO1,9,15,72,45,9\n"
	".,.,,,EDTEXT,-16,65,72,50,10\n"
	".,.,,,RTEXT,10,15,84,47,8\n"
	".,,,,EDTEXT,-16,65,84,50,10\n"
	"150,+,,ISRADIO | CHECKED | HICOL,CHECKBOX,13,15,107,30,8\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,14,15,117,30,8\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,15,65,107,30,8\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,16,65,117,30,8\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,17,15,127,30,8\n"
	".,.,,ISRADIO | HICOL,CHECKBOX,18,15,137,30,8\n"
	".,,,ISRADIO | HICOL,CHECKBOX,19,15,147,30,8\n"
	"200,+,210,ISPARENT | CHECKED,GROUPBOX,11,10,27,110,61\n"
	".,,220,ISPARENT | CHECKED,GROUPBOX,20,10,100,110,31\n"
	"210,,,NOSELECT,ODB,12,25,34,90,50\n"
	"220,+,,,RTEXT,21,20,110,30,8\n"
	".,,,LASTOBJ | TOUCHEXIT | OWNDIALOG,LINEBUTT,22,55,110,55,12";

bool
FreqDist::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_FDIST_RANGE,
		(void*)SDLG_FDIST_CLASS, (void*)SDLG_FDIST_DODIST, (void*)SDLG_FDIST_CREA, (void*)&step,
		(void*)SDLG_FDIST_CLABAR, (void*)SDLG_FDIST_CLSIZE, (void*)SDLG_FDIST_START,
		(void*)SDLG_FDIST_BSTYLE, (void*)OD_filldef, (void*)SDLG_FDIST_NORM, (void*)SDLG_FDIST_LNORM,
		(void*)SDLG_FDIST_BINO, (void*)SDLG_FDIST_POISS, (void*)SDLG_FDIST_EXP, (void*)SDLG_FDIST_RECT,
		(void*)SDLG_FDIST_CHI2, (void*)SDLG_FDIST_FSTYLE, (void*)SDLG_FDIST_LINE, (void*)&FuncLine };
	DlgInfo *FreqDlg;
	DlgRoot *Dlg;
	void *hDlg;
	bool bRet = false;
	int res;
	long r, c;
	double tmp;
	AccRange *aR;

	if(!parent || !data) return false;
	if(!(FreqDlg = CompileDialog(FreqDlg_Tmpl, dyndata))) return false;
	step = 7;	TmpTxt[100] = 0;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&BarLine, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&BarFill, 0);
	UseRangeMark(data, 1, TmpTxt);
	if(!(Dlg = new DlgRoot(FreqDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_FDIST_HD1, 50, 50, 380, 385, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:
			if(Dlg->GetCheck(10)) res=-1;
			break;
		case 1:
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) ssRef = rlp_strdup(TmpTxt);
			if(Dlg->GetCheck(150)) type = 1;
			else if(Dlg->GetCheck(151)) type = 2;
			else if(Dlg->GetCheck(154)) type = 3;
			else if(Dlg->GetCheck(155)) type = 4;
			else if(Dlg->GetCheck(156)) type = 5;
			else if(Dlg->GetCheck(152)) type = 10;
			else if(Dlg->GetCheck(153)) type = 11;
			else type = 0;
			break;
		case 221:
			res = -1;					//function line properties
			break;
		}
		}while (res <0);
	if(res==1 && (plots = (GraphObj**)calloc((nPlots=3)+2, sizeof(GraphObj*)))) {
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&BarLine, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&BarFill, 0);
		if(BarFill.hatch) memcpy(&HatchLine, BarFill.hatch, sizeof(LineDEF));
		BarFill.hatch = &HatchLine;
		if(Dlg->GetCheck(123) && Dlg->GetValue(124, &step) && Dlg->GetValue(126, &start)) ProcData(-2);
		else {
			Dlg->GetValue(121, &step);		ProcData(-1);
			}
		if ((y_info = (char*)malloc(25 * sizeof(char)))) rlp_strcpy(y_info, 25, (char*)SDLG_FDIST_NOBS);
		if((x_info = (char*)malloc(25*sizeof(char)))){
			rlp_strcpy(x_info, 24, (char*)SDLG_FDIST_CATS);
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && (aR = new AccRange(TmpTxt))) {
				if(aR->GetFirst(&c, &r) && !data->GetValue(r, c, &tmp) && data->GetText(r, c, TmpTxt, 150, false))
					rlp_strcpy(x_info, 25, TmpTxt);
				delete aR;
				}
			}
		if(plots[0]) bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(FreqDlg);
	free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Regression properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* RegDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,140,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,140,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,130,85\n"
	".,10,200,ISPARENT,SHEET,2,5,10,130,85\n"
	"10,,,CHECKED,CHECKPIN,,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,22,60,8\n"
	".,.,,,RANGEINPUT,-16,20,32,100,10\n"
	".,.,,,LTEXT,-32,10,45,60,8\n"
	".,.,,,RANGEINPUT,-17,20,55,100,10\n"
	".,.,,HICOL | CHECKED,CHECKBOX,3,10,70,100,8\n"
	".,.,,HICOL,CHECKBOX,4,10,80,100,8\n" 
	".,,,TOUCHEXIT | OWNDIALOG,SYMBUTT,15,95,70,30,10\n"
	"200,210,201,CHECKED | ISPARENT, GROUPBOX,5,10,30,58,56\n"
	"201,+,,HICOL | CHECKED, RADIO1,6,20,40,30,8\n"
	".,.,,HICOL,RADIO1,7,20,50,30,8\n"
	".,.,,HICOL,RADIO1,8,20,60,30,8\n"
	".,,,HICOL,RADIO1,9,20,70,30,8\n"
	"210,,211,CHECKED | ISPARENT,GROUPBOX,10,72,30,58,56\n"
	".,+,,HICOL | CHECKED,RADIO1,11,82,40,30,8\n"
	".,.,,HICOL,RADIO1,12,82,50,30,8\n"
	".,.,,HICOL,RADIO1,13,82,60,30,8\n"
	".,,,HICOL | LASTOBJ,RADIO1,14,82,70,30,8";

bool
Regression::PropertyDlg()
{
	int i1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_TRANSFORM);
	Symbol *PrevSym = new Symbol(0L, data, 0.0, 0.0, 0);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_REGRA_SYMS, (void*)SDLG_REGRA_SDEL,
		(void*)SDLG_REGRA_XVAL, (void*)SDLG_REGRA_X, (void*)SDLG_REGRA_LOGX, (void*)SDLG_REGRA_RECX,
		(void*)SDLG_REGRA_SQRTX, (void*)SDLG_REGRA_YVAL, (void*)SDLG_REGRA_Y, (void*)SDLG_REGRA_LOGY,
		(void*)SDLG_REGRA_RECY, (void*)SDLG_REGRA_SQRTY, (void*)&PrevSym };
	DlgInfo *RegDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int c, ic, res, n;
	long i, j, k, l;
	double x, y;
	AccRange *rX, *rY;
	bool bRet = false, bContinue = false, dValid;
	lfPOINT *values = 0L;

	if(!parent || !data) return false;
	if(!(RegDlg = CompileDialog(RegDlg_Tmpl, dyndata))) return false;
	rX = rY = 0L;
	UseRangeMark(data, 1, TmpTxt+100, TmpTxt+200);
	if(!(Dlg = new DlgRoot(RegDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_REGRA_HD1, 50, 50, 400, 250, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 106:					//preview symbol
			res = -1;		bContinue = false;
			break;
		case 1:								// OK
			if(rX) delete rX;
			if(rY) delete rY;
			rX = rY = 0L;						// check x-range
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rX = new AccRange(TmpTxt);
			if(!(n = rX ? rX->CountItems() : 0)) {
				Dlg->SetCheck(4, 0L, true);
				res = -1;
				bContinue = true;
				ErrorBox((char*)SCMS_BAD_XRANGE);
				}
			else {							// check y-range
				if(Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rY = new AccRange(TmpTxt);
				if(n != (rY ? rY->CountItems() : 0)) {
					res = -1;
					bContinue = true;
					ErrorBox((char*)SCMS_BAD_YRANGE);
					}
				}
			}
		}while (res <0);
	if(res==1 && n && rX && rY && (values =(lfPOINT*)calloc((nPoints=n)+2, sizeof(lfPOINT)))){				//OK pressed
		Command(CMD_FLUSH, 0L, 0L);
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) xRange = rlp_strdup(TmpTxt);
		if(Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) yRange = rlp_strdup(TmpTxt);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		DefName((char*)"Regression ", rX, rY);
		rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
		rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
		if(Dlg->GetCheck(202)) type = 0x100;
		else if(Dlg->GetCheck(203)) type = 0x200;
		else if(Dlg->GetCheck(204)) type = 0x300;
		if(Dlg->GetCheck(212)) type |= 0x1000;
		else if(Dlg->GetCheck(213)) type |= 0x2000;
		else if(Dlg->GetCheck(214)) type |= 0x3000;
		ic = c = 0;
		if(Dlg->GetCheck(104)) Symbols = (Symbol**)calloc(nPoints+2, sizeof(Symbol*));
		do {
			if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
				dValid = true;
				switch(type & 0x700) {
				case 0x100:					//logarithmic x
					if((dValid = x > defs.min4log)) values[ic].fx = log10(x);
					break;
				case 0x200:					//reciprocal x
					if((dValid = fabs(x) >defs.min4log)) values[ic].fx = 1.0/x;
					break;
				case 0x300:					//square root x
					if((dValid = fabs(x) >defs.min4log)) values[ic].fx = sqrt(x);
					break;
				default:	values[ic].fx = x;	break;		//linear x
					}
				if(dValid) switch(type & 0x7000) {
				case 0x1000:				//logarithmic y
					if((dValid = y > defs.min4log)) values[ic].fy = log10(y);
					break;
				case 0x2000:				//reciprocal y
					if((dValid = fabs(y) > defs.min4log)) values[ic].fy = 1.0/y;
					break;
				case 0x3000:				//square root y
					if((dValid = fabs(y) > defs.min4log)) values[ic].fy = sqrt(y);
					break;
				default:	values[ic].fy = y;	break;		//linear y
					}
				if(dValid && Symbols && (Symbols[ic] = new Symbol(this, data, x, y, 
					SYM_CIRCLE, i, j, k, l))){
					Symbols[ic]->idx = c;
					}
				if(dValid) {
					CheckBounds(x, y);
					ic++;
					}
				}
			c++;
			}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
		if(ic) {
			if(Dlg->GetCheck(105) && (sde = new SDellipse(this, data, values, ic, type | 0x20002))&&
				(bRet= sde->Command(CMD_INIT, 0L, 0L)))sde->Command(CMD_BOUNDS, &Bounds, 0L);
			else if((rLine = new RegLine(this, data, values, ic, type)) && 
				(bRet= rLine->PropertyDlg()))rLine->Command(CMD_BOUNDS, &Bounds, 0L);
			}
		if (Symbols) {
			SetSize(SIZE_SYMBOL, PrevSym->GetSize(SIZE_SYMBOL));			SetSize(SIZE_SYM_LINE, PrevSym->GetSize(SIZE_SYM_LINE));
			SetColor(COL_SYM_LINE, PrevSym->GetColor(COL_SYM_LINE));		SetColor(COL_SYM_FILL, PrevSym->GetColor(COL_SYM_FILL));
			Command(CMD_SYM_TYPE, (void*)(&PrevSym->type), 0L);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rX) delete rX;
	if(rY) delete rY;
	free(RegDlg);		if(values) free(values);
	free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// myRegression properties dialog: x, many y
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *myRegDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,158,10,45,12\n"
".,.,,,PUSHBUTTON,-2,158,25,45,12\n"
".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,140,112\n"
".,20,200,ISPARENT | TOUCHEXIT,SHEET,2,5,10,140,112\n"
"20,,,CHECKED,CHECKPIN,,5,0,12,8\n"
"100,+,,,LTEXT,-35,15,30,60,8\n"
".,152,,,RANGEINPUT,3,25,40,100,10\n"
"152,+,,ISPARENT | CHECKED,GROUPBOX,-36,12,60,128,50\n"
".,.,,,LTEXT,0,25,65,60,8\n"
".,.,,,RANGEINPUT,3,25,75,100,10\n"
".,.,,,PUSHBUTTON,-8,95,87,30,12\n"
".,,,,PUSHBUTTON,-9,60,87,35,12\n"
"200,250,210,ISPARENT | CHECKED,GROUPBOX,4,12,30,128,32\n"
"210,+,,CHECKED| HICOL,CHECKBOX,5,25,35,100,10\n"
".,.,,HICOL,CHECKBOX,13,25,47,20,10\n"
".,.,,,EDVAL1,14,53,47,15,10\n"
".,.,,,LTEXT,15,70,47,30,10\n"
".,,,OWNDIALOG,LINEBUTT,16,105,47,30,10\n"
"250,,260,ISPARENT | CHECKED,GROUPBOX,6,12,70,128,45\n"
"260,+,,HICOL | ISRADIO,CHECKBOX,7,25,75,100,10\n"
".,.,,HICOL | ISRADIO | CHECKED,CHECKBOX,8,25,87,100,10\n"
".,270,,HICOL | ISRADIO,CHECKBOX,9,25,99,100,10\n"
"270,+,,TOUCHEXIT | OWNDIALOG,SYMBUTT,10,105,75,30,10\n"
".,.,,TOUCHEXIT | OWNDIALOG,ERRPREV,11,105,87,30,20\n"
".,,,TOUCHEXIT | OWNDIALOG | LASTOBJ,LINEBUTT,12,105,35,30,10";

bool
myRegression::PropertyDlg()
{
	int i1, j1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_DETAILS);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)TmpTxt, (void*)SDLG_MYREGR_REGR, (void*)SDLG_MYREGR_RLINE,
		(void*)SDLG_MYREGR_DATA, (void*)SDLG_MYREGR_ALLVAL, (void*)SDLG_MYREGR_MEANSD, (void*)SDLG_MYREGR_MEANSERR,
		(void*)&PrevSym, (void*)&PrevErr, (void*)&PrevLine, (void*)SDLG_MYREGR_CI1, (void*)&ci, (void*)SDLG_MYREGR_CI2,
		(void*)&PrevCiLine};
	DlgInfo *myRegDlg;
	DlgRoot *Dlg;
	void *hDlg;
	char **rd = 0L;
	AccRange *rX = 0L, *rY = 0L;
	bool bContinue, updateYR = true, RetVal = false;
	int  maxYR = 0, currYR = 0, res, nx = 0, ny = 0, n;
	long i, j;
	double tmp;
	bool bVisited = false;

	if (!parent || !data) return false;
	name = rlp_strdup((char*)"Regression");
	if (!UseRangeMark(data, 2, TmpTxt, TmpTxt + 100, TmpTxt + 200, TmpTxt + 300, TmpTxt + 400,
		TmpTxt + 500, TmpTxt + 600, TmpTxt + 700, TmpTxt + 800, TmpTxt + 900, TmpTxt + 1000)) return false;
	if (!(myRegDlg = CompileDialog(myRegDlg_DlgTmpl, dyndata))) return false;
	if (TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(20, sizeof(char*)))) {
		for (i = 100, j = 0; i <= 1000; i += 100) if (TmpTxt[i]){
			rd[j++] = rlp_strdup(TmpTxt + i);			 maxYR = j - 1;
			rd[j] = 0L;
			}
		}
	if (!rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	if (!(Dlg = new DlgRoot(myRegDlg, data))) return false;
	if (rd && rd[currYR] && *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd((char*)SDLG_MYREGR_HD1, 50, 50, 440, 300, Dlg, 0x4L);
	do {
		if (updateYR) {
			if (currYR >0) {
				Dlg->ShowItem(156, 1);					Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);				Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_STB_RFMT, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt, SDLG_STB_RFMT, currYR + 1, maxYR + 1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if (bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 270:					//preview symbol
			res = -1;				bContinue = false;
			break;
		case 271:					//preview error bar
			res = -1;				bContinue = false;
			break;
		case 272:
			res = -1;				bContinue = false;
			break;
		case 11:					case 1:
			if (res == 11) {		//the details tab must be visited
				bVisited = true;	res = -1;
				}
			if (!bVisited) {
				Dlg->SetCheck(11, 0L, 1);
				bVisited = true;	 res = -1;		break;
				}
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		case 155:		case 156:
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		}
	} while (res < 0);
	if (res == 1){					//OK pressed
		if (rX) x_info = rlp_strdup(rX->RangeDesc(data, 1));
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		for (i1 = j1 = n = 0; i1 <= maxYR; i1++) {
			if (rd[i1]) {
				if (n) TmpTxt[n++] = '|';
				n += rlp_strcpy(TmpTxt + n, 100, rd[i1]);
				}
			}
		TmpTxt[n++] = '\0';			if(TmpTxt[0]) yRange = rlp_strdup(TmpTxt);
		if (Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) xRange = rlp_strdup(TmpTxt);
		type = 0;
		if (Dlg->GetCheck(210)) type |= 0x001;
		if (Dlg->GetCheck(260))	type |= 0x010;
		else if (Dlg->GetCheck(261)) type |= 0x20;
		else if (Dlg->GetCheck(262)) type |= 0x30;
		if (Dlg->GetCheck(211)) {
			if (Dlg->GetValue(212, &tmp)) {
				if (tmp >= 50.0 && tmp < 100) ci = tmp;
				}
			type |= 0x80;
			}
		RetVal = Recalc();			//do calculations
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if (rX) delete rX;
	if (rd) {
		for (i = 0; i < maxYR; i++)	if (rd[i]) free(rd[i]);
		free(rd);
		}
	free(myRegDlg);		free(tab1);			free(tab2);
	return RetVal;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// myBarPlot properties dialog: x, many y
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *myBarDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,158,10,45,12\n"
".,.,,,PUSHBUTTON,-2,158,25,45,12\n"
".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,140,142\n"
".,20,200,ISPARENT | TOUCHEXIT,SHEET,2,5,10,140,142\n"
"20,,,CHECKED,CHECKPIN,,5,0,12,8\n"
"100,+,,,LTEXT,-35,15,30,60,8\n"
".,152,,,RANGEINPUT,3,25,45,100,10\n"
"152,+,,ISPARENT | CHECKED,GROUPBOX,-36,12,70,128,55\n"
".,.,,,LTEXT,0,25,77,60,8\n"
".,.,,,RANGEINPUT,3,25,90,100,10\n"
".,.,,,PUSHBUTTON,-8,95,107,30,12\n"
".,,,,PUSHBUTTON,-9,60,107,35,12\n"
"200,250,270,ISPARENT | CHECKED,GROUPBOX,4,12,76,128,72\n"
"250,+,260,ISPARENT | CHECKED,GROUPBOX,6,12,30,128,40\n"
".,.,,HICOL | CHECKED,CHECKBOX,14,25,82,100,10\n"
".,.,,HICOL,CHECKBOX,15,25,94,100,10\n"
".,.,,HICOL | CHECKED,CHECKBOX,16,25,106,100,10\n"
".,.,,HICOL,CHECKBOX,17,25,130,100,10\n"
".,,,HICOL,CHECKBOX,18,25,118,100,10\n"
"260,+,,HICOL | ISRADIO,CHECKBOX,7,25,35,100,10\n"
".,.,,HICOL | ISRADIO | CHECKED,CHECKBOX,8,25,45,100,10\n"
".,,,HICOL | ISRADIO,CHECKBOX,9,25,55,100,10\n"
"270,+,,ODEXIT,SYMBUTT,10,100,94,30,10\n"
".,.,,ODEXIT,ERRPREV,11,100,106,30,10\n"
".,.,,ODEXIT,BARBUTT,12,100,82,30,10\n"
".,.,,ODEXIT,FILLBUTTON,19,100,118,30,10\n"
".,,,ODEXIT | LASTOBJ,LINEBUTT,13,100,130,30,10";

bool
myBarPlot::PropertyDlg()
{
	int i1, j1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_DETAILS);
	LineDEF PG_FillLine = { defs.GetSize(SIZE_HAIRLINE), 6.0f, 0x0, 0x0 };
	static FillDEF PG_Fill = { FILL_NONE, 0xa4808080, 1.0f, &PG_FillLine, 0xa4808080 };
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)TmpTxt, (void*)SDLG_MYBAR_LAYOUT, (void*)NULL,
		(void*)SDLG_MYBAR_DATA, (void*)SDLG_MYBAR_MEANS, (void*)SDLG_MYBAR_MEANSD, (void*)SDLG_MYBAR_MEANSERR,
		(void*)&PrevSym, (void*)&PrevErr, (void*)&PrevBar, (void*)&PrevLine, (void*)SDLG_MYBAR_VBARS, (void*)SDLG_MYBAR_SYMS,
		(void*)SDLG_MYBAR_ERRS, (void*)SDLG_MYBAR_LINE, (void*)SDLG_MYBAR_ERRPG, (void*)&PG_Fill };

	DlgInfo *myBarDlg;
	DlgRoot *Dlg;
	void *hDlg;
	char **rd = 0L;
	AccRange *rX = 0L, *rY = 0L;
	bool bContinue, updateYR = true, RetVal = false;
	int  maxYR = 0, currYR = 0, res, nx = 0, ny = 0, n;
	long i, j;
	int nVals, nTxt, nTime;
	bool bVisited = false;

	if (!parent || !data) return false;
	name = rlp_strdup((char*)"Bar Chart");
	if (!UseRangeMark(data, 2, TmpTxt, TmpTxt + 100, TmpTxt + 200, TmpTxt + 300, TmpTxt + 400,
		TmpTxt + 500, TmpTxt + 600, TmpTxt + 700, TmpTxt + 800, TmpTxt + 900, TmpTxt + 1000)) return false;
	if (!(myBarDlg = CompileDialog(myBarDlg_DlgTmpl, dyndata))) return false;
	if (TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(20, sizeof(char*)))) {
		for (i = 100, j = 0; i <= 1000; i += 100) if (TmpTxt[i]){
			rd[j++] = rlp_strdup(TmpTxt + i);		 maxYR = j - 1;
			rd[j] = 0L;
			}
		}
	if (!rd && !(rd = (char**)calloc(maxYR + 2, sizeof(char*))))return false;
	if (!(Dlg = new DlgRoot(myBarDlg, data))) return false;
	if (rd && rd[currYR] && *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd((char*)SDLG_MYBAR_HD, 50, 50, 440, 364, Dlg, 0x4L);
	do {
		if (updateYR) {
			if (currYR >0) {
				Dlg->ShowItem(156, 1);				Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);				Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_STB_RFMT, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt, SDLG_STB_RFMT, currYR + 1, maxYR + 1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if (bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 11:		case 1:
			if (res == 11) {					//the details tab must be visited
				bVisited = true;				res = -1;
				}
			if (!bVisited) {
				Dlg->SetCheck(11, 0L, 1);
				bVisited = true;	 res = -1;		break;
				}
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		case 155:		case 156:
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		case 270:	case 271:	case 272:	case 273:	case 274:		//the preview buttons
			res = -1;								break;
			}
		} while (res < 0);
	if (res == 1){					//OK pressed
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		for (i1 = j1 = n = 0; i1 <= maxYR; i1++) {
			if (n) TmpTxt[n++] = '|';
#ifdef USE_WIN_SECURE
			n += sprintf_s(TmpTxt + n, TMP_TXT_SIZE - n - 2, "%s", rd[i1]);
#else
			n += sprintf(TmpTxt + n, "%s", rd[i1]);
#endif
			}
		yRange = rlp_strdup(TmpTxt);
		if (Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) xRange = rlp_strdup(TmpTxt);
		if(!rX) rX = new AccRange(xRange);
		if(rX->DataTypes(data, &nVals, &nTxt, &nTime)){
			if (!nVals && nTxt > 1 && nTxt > nTime) x_tv = new TextValue();
			else if (!nVals && nTime > 1 && nTime > nTxt) x_dtype = ET_DATETIME;
			}
		type = 0;
		if (Dlg->GetCheck(261)) type |= 0x001;
		else if (Dlg->GetCheck(262)) type |= 0x002;
		if (Dlg->GetCheck(251)) type |= 0x010;
		if (Dlg->GetCheck(252)) type |= 0x020;
		if (Dlg->GetCheck(253)) type |= 0x040;
		if (Dlg->GetCheck(254)) type |= 0x080;
		if (Dlg->GetCheck(255)) {
			type |= 0x100;
			ErrPg = new ErrorPolygon(this, 0L, 0, 0L, 0, 0L);
			ErrPg->Command(CMD_PG_FILL, &PG_Fill, 0L);
			}
		RetVal = Recalc();			//do calculations
		}

	CloseDlgWnd(hDlg);
	delete Dlg;
	if (rX) delete rX;
	if (rd) {
		for (i = 0; i < maxYR; i++)	if (rd[i]) free(rd[i]);
		free(rd);
		}
	free(myBarDlg);		free(tab1);			free(tab2);
	return RetVal;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bubble plot properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *BubPlotDlg_Tmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,-1,148,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,148,25,45,12\n"
	"3,500,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,5,100,ISPARENT | CHECKED,SHEET,1,5,10,135,120\n"
	"5,6,200,ISPARENT | TOUCHEXIT, SHEET,2,5,10,135,120\n"
	"6,10,300,ISPARENT, SHEET,3,5,10,135,120\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,40,60,8\n"
	".,.,,,RANGEINPUT,-15,20,50,100,10\n"
	".,.,,,LTEXT,-32,10,65,60,8\n"
	".,.,,,RANGEINPUT,-16,20,75,100,10\n"
	".,.,,,LTEXT,4,10,90,60,8\n"
	".,,,,RANGEINPUT,-17,20,100,100,10\n"
	"200,+,,,LTEXT,-27,10,30,110,8\n"
	".,.,,ISRADIO | TOUCHEXIT | CHECKED,ODB,5,30,40,20,20\n"
	".,.,,ISRADIO | TOUCHEXIT,ODB,5,50,40,20,20\n"
	".,.,,ISRADIO | TOUCHEXIT,ODB,5,70,40,20,20\n"
	".,.,,ISRADIO | TOUCHEXIT,ODB,5,90,40,20,20\n"
	".,.,,,LTEXT,6,10,63,45,8\n"
	".,.,,,RTEXT,-11,12,77,20,8\n"
	".,.,,OWNDIALOG,COLBUTT,7,34,77,25,10\n"
	".,.,,,RTEXT,8,72,77,20,8\n"
	".,.,,,EDVAL1,9,93,77,25,10\n"
	".,.,,,LTEXT,-3,119,77,15,8\n"
	".,.,,,LTEXT,10,10,94,45,8\n"
	".,.,,,RTEXT,-11,12,105,20,8\n"
	".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,11,34,105,25,10\n"
	".,.,,,RTEXT,12,72,105,20,8\n"
	".,,,TOUCHEXIT | OWNDIALOG,FILLBUTTON,13,93,105,25,10\n"
	"300,+,,,LTEXT,14,20,30,110,8\n"
	".,.,400,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,,,LTEXT,15,20,75,110,8\n"
	".,,410,ISPARENT | CHECKED,GROUP,,,,,\n"
	"400,+,,HICOL | CHECKED,RADIO1,-3,40,40,45,8\n"
	".,.,,HICOL,RADIO1,16,40,50,45,8\n"
	".,,,HICOL,RADIO1,17,40,60,45,8\n"
	"410,+,,HICOL | CHECKED,RADIO1,18,40,85,45,8\n"
	".,.,,HICOL,RADIO1,19,40,95,45,8\n"
	".,,,HICOL | LASTOBJ,RADIO1,20,40,105,45,8";
bool
BubblePlot::PropertyDlg()
{
	int i1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_LAYOUT);
	TabSHEET *tab3 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_SCALE);
	static FillDEF ShowFill;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_BPLT_SRANG, (void*)OD_BubbleTempl,
		(void*)SDLG_BPLT_OLIN, (void *)&BubbleLine.color, (void*)SDLG_BPLT_LW, (void*)&BubbleLine.width, (void*)SDLG_BPLT_FILL,
		(void *)&BubbleFill.color, (void*)SDLG_BPLT_PAT, (void *)&ShowFill, (void*)SDLG_BUB_SIZEDEF, (void*)SDLG_BUB_PROP,
		(void*)SDLG_BUB_SCALEX, (void*)SDLG_BUB_SCALEY, (void*)SDLG_BUB_DIAM, (void*)SDLG_BUB_CIRC, (void*)SDLG_BUB_AREA };
	DlgInfo *PlotDlg;
	DlgRoot *Dlg;
	void *hDlg;
	long i, j, k, l, m, n;
	int ic, res, BubbleType, curr_sel = 201;
	double x, y, s;
	double tmp;
	bool bRetVal = false, bLayout = false, bContinue = false;
	AccRange *rX, *rY, *rS;
	LineDEF ShowFillLine ={0.2f, 1.0f, 0x0L, 0x0L};

	if(!parent || !data) return false;
	if(!(PlotDlg = CompileDialog(BubPlotDlg_Tmpl, dyndata))) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100, TmpTxt+200);
	memcpy(&ShowFill, &BubbleFill, sizeof(FillDEF));
	if(BubbleFill.hatch) memcpy(&ShowFillLine, BubbleFill.hatch, sizeof(LineDEF));
	ShowFill.hatch = &ShowFillLine;
	if(!(Dlg = new DlgRoot(PlotDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_BPLT_HD1, 50, 50, 415, 320, Dlg, 0x4L);
	rX = rY = rS = 0L;
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if (bContinue) res = -1;
			else if (Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 213:					//fillcolor changed
			Dlg->GetColor(213, &ShowFill.color);
			Dlg->DoPlot(NULL);
			res = -1;
			break;
		case 215:					//copy color from pattern dialog
			Dlg->SetColor(213, ShowFill.color);
			res = -1;
			break;
		case 5:
			bLayout = true;				res = -1;				break;
		case 201:	case 202:	case 203:	case 204:	//style templates
		case 1:						//OK button
			if (res == curr_sel) res = 1;
			else if(res > 200){
				curr_sel = res;		res = -1;	break;
				}
			if (!bLayout) {			//the layout tab must have been visited
				Dlg->SetCheck(5, 0L, bLayout = true);	res = -1;
				}
			else if (Dlg->GetText(101, (unsigned char *)TmpTxt, 100) && Dlg->GetText(103, (unsigned char*)(TmpTxt + 100), 100) &&
				Dlg->GetText(105, (unsigned char*)(TmpTxt + 200), 100) && (rX = new AccRange(TmpTxt)) &&
				(rY = new AccRange(TmpTxt + 100)) && (rS = new AccRange(TmpTxt + 200))) {
				if ((i = rX->CountItems()) == rY->CountItems() && i == rS->CountItems()){
					// OK pressed and ranges checked: exit loop and process data
					}
				else {
					if (rX) delete (rX);
					if (rY) delete (rY);
					if (rS) delete (rS);
					rX = 0L; rY = 0L; rS = 0L;
					ErrorBox((char*)SCMS_BAD_RANGE);
					Dlg->SetCheck(4, 0L, bContinue = true);
					res = -1;
					}
				}
			else res = -1;				//continue with dialog if error
			}
		}while (res <0);

	if(res ==1 && rX && rY && rS && (nPoints = rX->CountItems()) && 
		(Bubbles = (Bubble**)calloc(nPoints+2, sizeof(Bubble*)))) {
		//accept settings and create bubbles for plot
		if(Dlg->GetCheck(202)) BubbleType = BUBBLE_SQUARE;
		else if(Dlg->GetCheck(203)) BubbleType = BUBBLE_UPTRIA;
		else if(Dlg->GetCheck(204)) BubbleType = BUBBLE_DOWNTRIA;
		else BubbleType = BUBBLE_CIRCLE;
		if(Dlg->GetCheck(401)) BubbleType |= BUBBLE_XAXIS;
		else if(Dlg->GetCheck(402)) BubbleType |= BUBBLE_YAXIS;
		if(Dlg->GetCheck(411)) BubbleType |= BUBBLE_CIRCUM;
		else if(Dlg->GetCheck(412)) BubbleType |= BUBBLE_AREA;
		if(Dlg->GetValue(209, &tmp)) BubbleLine.width = (float)tmp;
		Dlg->GetColor(207, &BubbleLine.color);		Dlg->GetColor(213, &BubbleFill.color);
		rX->GetFirst(&i, &j);		rY->GetFirst(&k, &l);	rS->GetFirst(&m, &n);
		rX->GetNext(&i, &j);		rY->GetNext(&k, &l);	rS->GetNext(&m, &n);
		ic = 0;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		if(Bubbles) do {
			if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(n, m, &s)){
				CheckBounds(x, y);
				Bubbles[ic++] = new Bubble(this, data, x, y, s, BubbleType, &ShowFill, 
					&BubbleLine, i, j, k, l, m, n);
				}
			}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rS->GetNext(&m, &n));
		bRetVal = ic >0;
		}
	CloseDlgWnd(hDlg);
	if(rX) delete (rX);
	if(rY) delete (rY);
	if(rS) delete (rS);
	delete Dlg;			free(PlotDlg);
	free(tab1);			free(tab2);				free(tab3);
	return bRetVal;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Polar plot properties dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *AddPolDlg_Tmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,-1,140,14,45,12\n"
	"2,3,,,PUSHBUTTON,-2,140,29,45,12\n"
	"3,10,200,ISPARENT | CHECKED,GROUPBOX,1,5,14,131,96\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"200,+,,CHECKED | EXRADIO,ODB,2,10,24,20,20\n"
	".,.,,EXRADIO,ODB,2,30,24,20,20\n"
	".,.,,EXRADIO,ODB,2,50,24,20,20\n"
	".,.,,EXRADIO,ODB,2,70,24,20,20\n"
	".,210,,EXRADIO,ODB,2,90,24,20,20\n"
	"210,+,,,LTEXT,3,10,50,50,8\n"
	".,.,,,RANGEINPUT,-15,20,62,100,10\n"
	".,.,,,LTEXT,4,10,75,50,8\n"
	".,,,LASTOBJ,RANGEINPUT,-16,20,87,100,10";

bool
PolarPlot::AddPlot()
{
	void *dyndata[] = { (void*)SDLG_POLPLT_SELECT, (void*)OD_PolarTempl,
		(void*)SDLG_POLPLT_XRNG, (void*)SDLG_POLPLT_YRNG };
	DlgInfo *PolDlg;
	DlgRoot *Dlg;
	void *hDlg;
	long i, j, k, l;
	int ic, n, res, cType = 200;
	bool bRet = false, bContinue = false;
	double x, y;
	AccRange *rX = 0L, *rY = 0L;
	Symbol **Symbols = 0L;
	DataLine *TheLine = 0L;
	Plot **tmpPlots;
	Function *func;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent || !data) return false;
	if(!(PolDlg = CompileDialog(AddPolDlg_Tmpl, dyndata))) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100);
	if(!(Dlg = new DlgRoot(PolDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_POLPLT_HD3, 50, 50, 408, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 200: case 201:	case 202:	case 203:	case 204:
			if(res == 204) {
				Dlg->Activate(211, 0);				Dlg->Activate(213, 0);
				}
			else if(cType == 204) {
				Dlg->Activate(211, 1);				Dlg->Activate(213, 1);
				}
			if(res == cType) res = 1;
			else {
				cType = res;
				res = -1;
				}
			break;
			}
		}while (res <0);
	if(res == 1 && Dlg->GetText(211, (unsigned char*)TmpTxt, 100) && Dlg->GetText(213, (unsigned char*)(TmpTxt+100), 100) && 
		(rX = new AccRange(TmpTxt)) && (rY = new AccRange(TmpTxt+100)) &&
		(n = rX ? rX->CountItems() : 0) && 
		(tmpPlots = (Plot**)realloc(Plots, (nPlots+2)*sizeof(Plot*)))) {
		Undo.SetDisp(cdisp);
		Plots = tmpPlots;
		if(Dlg->GetCheck(200) || Dlg->GetCheck(201)) 
			Symbols = (Symbol**) calloc(n+2, sizeof(Symbol*));
		if(Dlg->GetCheck(201) || Dlg->GetCheck(202)) 
			TheLine = new DataLine(this, data, TmpTxt, TmpTxt+100);
		else if(Dlg->GetCheck(203))
			TheLine = new DataPolygon(this, data, TmpTxt, TmpTxt+100);
		else if(Dlg->GetCheck(204)) {
			func = new Function(this, data, (char*)"Function");
			if(func){
				bRet = func->PropertyDlg();
				if(bRet){
					Undo.SetGO(this, (GraphObj**) &Plots[nPlots++], func, 0L);
					memcpy(&Bounds, &Plots[nPlots-1]->Bounds, sizeof(fRECT));
					}
				else DeleteGO(func);
				}
			}
		rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
		rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
		ic = 0;
		if(Symbols || TheLine) {
			do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
					CheckBounds(y, y);
					if(Symbols)Symbols[ic++] = new Symbol(this,data,x,y,SYM_CIRCLE,i,j,k,l);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			Undo.SetGO(this, (GraphObj**) &Plots[nPlots], 
				new PlotScatt(this, data, ic, Symbols, TheLine), 0L);
			Plots[nPlots]->DefName((char*)SDLG_PLOTSCATT_ISXY, rX, rY);
			nPlots++;
			bRet = true;
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(PolDlg);
	if(rX) delete rX;
	if(rY) delete rY;
	return bRet;
}

static char *PPDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,140,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,140,25,45,12\n"
	".,,100,ISPARENT | CHECKED, SHEET,1,5,10,90,90\n"
	"100,+,,NOSELECT,ODBUTTON,2,8,30,90,50\n"
	".,,, LASTOBJ || HICOL,CHECKBOX,3,15,85,60,9";

bool
PolarPlot::Config()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_PLOT);
	void *dyndata[] = { (void*)tab1, (void*)OD_filldef, (void*)SDLG_POLPLT_DISPA };
	DlgInfo *PPDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int res;
	bool bRet = false;
	LineDEF OutLine;

	if (!(PPDlg = CompileDialog(PPDlg_DlgTmpl, dyndata))) return false;
	memcpy(&OutLine, defs.GetOutLine(), sizeof(LineDEF));
	if(Axes && Axes[0]) {
		OutLine.color = Axes[0]->GetColor(COL_AXIS);
		OutLine.width = Axes[0]->GetSize(SIZE_AXIS_LINE);
		}
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)&OutLine, 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
	Dlg = new DlgRoot(PPDlg, data);
	if(!(type & 0x01))Dlg->SetCheck(101, 0L, true);
	hDlg = CreateDlgWnd((char*)SDLG_POLPLT_HD2, 50, 50, 400, 260, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		}while (res < 0);
	if(res == 1){						//OK pressed
		if(Dlg->GetCheck(101)) type &= ~0x01;
		else type |= 0x01;
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&OutLine, 0);
		if(Axes && Axes[0]) {
			Axes[0]->SetColor(COL_AXIS, OutLine.color);
			Axes[0]->SetSize(SIZE_AXIS_LINE, OutLine.width);
			}
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&Fill, 0);
		if(Fill.hatch) memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;		free(tab1);		free(PPDlg);
	return bRet;
}

static char *PolDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,140,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,140,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,5,100,ISPARENT | TOUCHEXIT,SHEET,1,5,10,131,100\n"
	"5,10,200,ISPARENT | TOUCHEXIT | CHECKED,SHEET,2,5,10,131,100\n"
	"10,,,CHECKED,CHECKPIN,,5,,12,8\n"
	"100,+,,,LTEXT,3,10,25,60,8\n"
	".,.,,,RTEXT,4,5,37,25,8\n"
	".,.,,,EDVAL1,5,30,37,30,10\n"
	".,.,,,RTEXT,6,60,37,25,8\n"
	".,.,,,EDVAL1,7,85,37,30,10\n"
	".,.,,,RTEXT,8,10,49,50,8\n"
	".,.,,,EDVAL1,9,62,49,30,10\n"
	".,.,,,LTEXT,10,10,65,40,8\n"
	".,.,,,RTEXT,11,5,77,25,8\n"
	".,.,,,EDVAL1,12,30,77,30,10\n"
	".,.,,,RTEXT,13,60,77,25,8\n"
	".,.,,,EDVAL1,14,85,77,30,10\n"
	".,.,,,LTEXT,-3,117,77,15,8\n" 
	".,.,,,EDVAL1,15,62,89,30,10\n"
	".,.,,,LTEXT,-3,94,89,15,8\n"
	".,,,,RTEXT,16,10,89,50,8\n"
	"200,+,,CHECKED | TOUCHEXIT | ISRADIO,ODB,17,10,25,20,20\n"
	".,.,,TOUCHEXIT | ISRADIO,ODB,17,30,25,20,20\n"
	".,.,,TOUCHEXIT | ISRADIO,ODB,17,50,25,20,20\n"
	".,.,,TOUCHEXIT | ISRADIO,ODB,17,70,25,20,20\n"
	".,.,,TOUCHEXIT | ISRADIO,ODB,17,90,25,20,20\n"
	".,.,,,LTEXT,18,10,55,50,8\n"
	".,.,,,RANGEINPUT,19,20,67,100,10\n"
	".,.,,,LTEXT,20,10,80,50,8\n"
	".,,,LASTOBJ,RANGEINPUT,21,20,92,100,10";

bool
PolarPlot::PropertyDlg()
{
	int i1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_COORD);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_TYPE);
	unsigned char text1[100], text2[100];
	double lox = 0.0, hix = 360.0, fcx =10.0, fcy = 20.0, frad=40.0;
	DlgInfo *PolDlg;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_POLPLT_RANGE,
		(void*)SDLG_POLPLT_MIN, (void*)&lox, (void*)SDLG_POLPLT_MAX, (void*)&hix, (void*)SDLG_POLPLT_OFFS,
		(void*)&offs, (void*)SDLG_POLPLT_CENT, (void*)SDLG_POLPLT_CX, (void*)&fcx, (void*)SDLG_POLPLT_CY,
		(void*)&fcy, (void*)&frad, (void*)SDLG_POLPLT_RAD, (void*)OD_PolarTempl,
		(void*)SDLG_POLPLT_XRNG, (void*)text1, (void*)SDLG_POLPLT_YRNG, (void*)text2 };
	DlgRoot *Dlg;
	void *hDlg;
	int res, n, ic, cType = 200;
	long i, j, k, l;
	double x, y;
	bool bRet = false, bType = false;
	AccRange *rX = 0L, *rY = 0L;
	Symbol **Symbols = 0L;
	DataLine *TheLine = 0L;
	TextDEF tlbdef;
	AxisDEF ang_axis, rad_axis;

	if(!parent || !data) return false;
	if(!(PolDlg = CompileDialog(PolDlg_DlgTmpl, dyndata))) return false;
	if(Plots) return Config();
	frad = (parent->GetSize(SIZE_DRECT_BOTTOM) - parent->GetSize(SIZE_DRECT_TOP))/2.0f;
	fcx = parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_DRECT_LEFT)*1.5 + frad;
	fcy = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_DRECT_TOP) + frad;
	UseRangeMark(data, 1, (char*)text1, (char*)text2);
	tlbdef.ColTxt = defs.Color(COL_AXIS);
	tlbdef.ColBg = 0x00ffffffL;
	tlbdef.RotBL = tlbdef.RotCHAR = 0.0f;
	tlbdef.fSize = DefSize(SIZE_TICK_LABELS);
	tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
	tlbdef.Style = TXS_NORMAL;
	tlbdef.Mode = TXM_TRANSPARENT;
	tlbdef.Font = FONT_HELVETICA;
	tlbdef.text = 0L;
	if(!(Dlg = new DlgRoot(PolDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_POLPLT_HD1, 50, 50, 408, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(Dlg->GetCheck(10)) res = -1;
			break;
		case 5:		case 4:
			bType = true;
			res = -1;
			break;
		case 1:
			if(!bType) {		//the 'Coordinates' sheet must have been visited
				bType = true;
				Dlg->SetCheck(4, 0L, true);
				res = -1;
				}
			break;
		case 200: case 201:	case 202:	case 203:	case 204:
			if(res == 204) {
				Dlg->Activate(211, 0);			Dlg->Activate(213, 0);
				}
			else if(cType == 204) {
				Dlg->Activate(211, 1);			Dlg->Activate(213, 1);
				}
			if(res == cType) res = 1;
			else {
				cType = res;
				res = -1;
				}
			break;
			}
		}while (res <0);
	if(res == 1) {
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		//set axis information in ang_axis and rad_axis
		ang_axis.owner = rad_axis.owner = 0L;
		ang_axis.breaks = rad_axis.breaks = 0L;
		ang_axis.nBreaks = rad_axis.nBreaks = 0;
		ang_axis.flags = AXIS_ANGULAR;	
		rad_axis.flags = AXIS_RADIAL | AXIS_DEFRECT;
		Dlg->GetValue(109, &ang_axis.Center.fx);	
		rad_axis.Center.fx = ang_axis.Center.fx;
		Dlg->GetValue(111, &ang_axis.Center.fy);	
		rad_axis.Center.fy = ang_axis.Center.fy;
		Dlg->GetValue(113, &ang_axis.Radius);
		rad_axis.Radius = ang_axis.Radius;
		Dlg->GetValue(102, &ang_axis.min);
		Dlg->GetValue(104, &ang_axis.max);
		Dlg->GetValue(106, &offs);
		ang_axis.loc[0].fy = ang_axis.loc[1].fy = ang_axis.Center.fy + ang_axis.Radius;
		ang_axis.loc[0].fx = ang_axis.Center.fx - ang_axis.Radius;
		ang_axis.loc[1].fx = ang_axis.Center.fx + ang_axis.Radius;
		rad_axis.loc[0].fx = rad_axis.loc[1].fx = 
			parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_DRECT_LEFT);
		rad_axis.loc[0].fy = rad_axis.Center.fy - rad_axis.Radius;
		rad_axis.loc[1].fy = rad_axis.Center.fy;
		if(Dlg->GetText(206, text1, 100) && Dlg->GetText(208, text2, 100) && 
			(rX = new AccRange((char*)text1)) && (rY = new AccRange((char*)text2)) &&
			(n = rX ? rX->CountItems() : 0) && (Plots = (Plot**)calloc(2, sizeof(Plot*)))) {
			if(Dlg->GetCheck(200) || Dlg->GetCheck(201)) 
				Symbols = (Symbol**) calloc(n+2, sizeof(Symbol*));
			if(Dlg->GetCheck(201) || Dlg->GetCheck(202)) 
				TheLine = new DataLine(this, data, (char*)text1, (char*)text2);
			else if(Dlg->GetCheck(203))
				TheLine = new DataPolygon(this, data, (char*)text1, (char*)text2);
			else if(Dlg->GetCheck(204)) {
				Plots[nPlots] = new Function(this, data, (char*)"Function");
				if (Plots[nPlots]){
					nPlots++;
					bRet = Plots[nPlots - 1]->PropertyDlg();
					if (bRet)
						memcpy(&Bounds, &Plots[nPlots-1]->Bounds, sizeof(fRECT));
					else {
						DeleteGO(Plots[nPlots-1]);
						Plots[nPlots-1] = 0L;
						}
					}
				}
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			ic = 0;
			if(Symbols || TheLine) do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
					CheckBounds(y, y);
					if(Symbols) Symbols[ic++] = new Symbol(this, data, x, y, SYM_CIRCLE, i, j, k, l);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			rad_axis.min = Bounds.Ymin;			rad_axis.max = Bounds.Ymax;
			NiceAxis(&rad_axis, 4);
			ang_axis.Start = ang_axis.Step = 0.0;
			if (Symbols || TheLine) {
				Plots[nPlots] = new PlotScatt(this, data, ic, Symbols, TheLine);
				Plots[nPlots]->DefName((char*)SDLG_PLOTSCATT_ISXY, rX, rY);
				nPlots++;
				}
			if(Plots[0] && (Axes = (GraphObj**)calloc(4, sizeof(Axis*)))){
				Axes[0] = new Axis(this, data, &ang_axis, ang_axis.flags);
				Axes[1] = new Axis(this, data, &rad_axis, 
					rad_axis.flags | AXIS_AUTOTICK | AXIS_NEGTICKS);
				Axes[1]->SetSize(SIZE_LB_XDIST, 
					NiceValue(-DefSize(SIZE_AXIS_TICKS)*6.0)); 
				Axes[1]->SetSize(SIZE_TLB_XDIST, 
					NiceValue(-DefSize(SIZE_AXIS_TICKS)*2.0)); 
				Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
				nAxes = 2;
				bRet = true;
				}
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rX) delete rX;
	if(rY) delete rY;
	free(PolDlg);		free(tab1);			free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Ternary (Triangle) plot properties dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *TernaryAddPlot_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,128,10,45,12\n"
	".,10,,,PUSHBUTTON,-2,128,25,45,12\n"
	"10,100,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,13,60,8\n"
	".,.,,,RANGEINPUT,-15,20,25,100,10\n"
	".,.,,,LTEXT,-32,10,38,60,8\n"
	".,.,,,RANGEINPUT,-16,20,50,100,10\n"
	".,200,,,LTEXT,1,10,70,60,8\n"
	"200,+,,CHECKED,CHECKBOX,2,35,70,60,8\n"
	".,,250,ISPARENT | CHECKED,GROUP,,,,,\n"
	"250,+,,ISRADIO,CHECKBOX,3,35,82,60,8\n"
	".,,,ISRADIO | LASTOBJ,CHECKBOX,4,35,95,60,8";
	
bool
TernaryPlot::AddPlot()
{
	void *dyndata[] = { (void*)SDLG_TERN_LAYOUT, (void*)SDLG_TERN_SYM, (void*)SDLG_TERN_LIN,
		(void*)SDLG_TERN_PG };
	DlgRoot *Dlg;
	void *hDlg;
	long i, j, k, l;
	int res, ic, n;
	double x, y;
	bool bContinue = false, bRet = false;
	AccRange *rX = 0L, *rY = 0L;
	DlgInfo *TernaryAddDlg;
	Symbol **Symbols = 0L;
	DataLine *TheLine = 0L;
	GraphObj *newplot;
	anyOutput *cdisp = Undo.Cdisp();


	if(!(TernaryAddDlg = CompileDialog(TernaryAddPlot_DlgTmpl, dyndata))) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100);
	if(!(Dlg = new DlgRoot(TernaryAddDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_TERN_HD3, 50, 50, 380, 270, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case 1:
			break;
		case -1:
			bContinue = false;
			break;
			}
		}while (res < 0);
	if(res == 1) {
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, 100) && Dlg->GetText(103, (unsigned char*)(TmpTxt+100), 100) && 
			(rX = new AccRange(TmpTxt)) && (rY = new AccRange(TmpTxt+100)) &&
			(n = rX ? rX->CountItems() : 0)) {
			if(Dlg->GetCheck(200)) Symbols = (Symbol**) calloc(n+2, sizeof(Symbol*));
			if(Dlg->GetCheck(250)) TheLine = new DataLine(this, data, TmpTxt, TmpTxt+100, (char*)"Teranry Line");
			else if(Dlg->GetCheck(251)) TheLine = (DataLine*) new DataPolygon(this, data, TmpTxt, TmpTxt+100, (char*)"Ternary Polygon");
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			ic = 0;
			if(Symbols) do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
					if(Symbols) Symbols[ic++] = new Symbol(this, data, x, y, 
						SYM_CIRCLE, i, j, k, l);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			if(Symbols || TheLine) {
				Undo.SetDisp(cdisp);
				Plots = (Plot**)realloc(Plots, (nPlots+2) * sizeof(Plot*));
				newplot = (GraphObj*)new PlotScatt(this, data, ic, Symbols, TheLine);
				Undo.SetGO(this, (GraphObj**)Plots+nPlots, newplot, 0L);
				Undo.ValLong(this, &nPlots, UNDO_CONTINUE);
				nPlots++;		Plots[nPlots] = 0L;		bRet = true;
				}
			else InfoBox((char*)SCMS_NOPLOT);
			}
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(TernaryAddDlg);
	if(rX) delete rX;
	if(rY) delete rY;
	return bRet;
}

bool
TernaryPlot::Config()
{
	InfoBox((char*)SDLG_TERN_HD2);
	return false;
}

static char *TernaryPlot_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,148,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,148,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,130,80\n"
	".,10,200,ISPARENT | TOUCHEXIT,SHEET,2,5,10,130,80\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,30,60,8\n"
	".,.,,,RANGEINPUT,-15,20,40,100,10\n"
	".,.,,,LTEXT,-32,10,55,60,8\n"
	".,,,,RANGEINPUT,-16,20,65,100,10\n"
	"200,+,,HICOL | CHECKED,CHECKBOX,3,25,35,60,8\n"
	".,,250,ISPARENT | CHECKED,GROUP,,,,,\n"
	"250,+,,HICOL | ISRADIO,CHECKBOX,4,25,50,60,8\n"
	".,,,HICOL | ISRADIO | LASTOBJ,CHECKBOX,5,25,65,60,8";

bool
TernaryPlot::PropertyDlg()
{
	int id;
	TabSHEET *tab1 = MakeTab(0, 10, &id, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(id, 10, &id, (char*)SDLG_TAB_STYLE);
	DlgInfo *TernaryDlg;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_TERN_SYM, (void*)SDLG_TERN_LIN,
		(void*)SDLG_TERN_PG};
	DlgRoot *Dlg;
	void *hDlg;
	int res;
	long i, j, k, l, ic, n;
	bool bContinue = false, bRet = false, bLayout = false;
	AccRange *rX = NULL, *rY = NULL;
	lfPOINT cent;
	double rad_l, rad_s, x, y;
	AxisDEF axis0, axis1, axis2;
	Symbol **Symbols = 0L;
	DataLine *TheLine = 0L;
	TextDEF tlbdef0 = {0x0, 0x00ffffff, 0.0, 0.0, 0.0, 0, TXA_VTOP | TXA_HCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	TextDEF tlbdef1 = {0x0, 0x00ffffff, 0.0, -60.0, 0.0, 0, TXA_VBOTTOM | TXA_HCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	TextDEF tlbdef2 = {0x0, 0x00ffffff, 0.0, 60.0, 0.0, 0, TXA_VBOTTOM | TXA_HCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};

	if(!parent || !data) return false;
	if(!(TernaryDlg = CompileDialog(TernaryPlot_DlgTmpl, dyndata))) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100);
	if(!(Dlg = new DlgRoot(TernaryDlg, data)))return false;
	rad_s = (parent->GetSize(SIZE_DRECT_BOTTOM) - parent->GetSize(SIZE_DRECT_TOP))*0.4;
	rad_l = rad_s / 0.732051;
	cent.fx = (parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_GRECT_RIGHT))*0.5;
	cent.fy = (parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_GRECT_BOTTOM))*0.5;
	tlbdef0.ColTxt = tlbdef1.ColTxt = tlbdef2.ColTxt = defs.Color(COL_AXIS);
	tlbdef0.fSize = tlbdef1.fSize = tlbdef2.fSize = DefSize(SIZE_TICK_LABELS);
	hDlg = CreateDlgWnd((char*)SDLG_TERN_HD1, 50, 50, 420, 250, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case 1:
			//the layout menu must have been visited
			if(!bLayout) {
				Dlg->SetCheck(5, 0L, bLayout = true);
				res = -1;
				}
			break;
		case -1:
			bContinue = false;
			break;
		case 5:
			bLayout = true;
			res = -1;
			break;
			}
		}while (res < 0);
	if(res == 1) {
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
#ifdef _WINDOWS
		memset(&axis0, 0, sizeof(AxisDEF));		memset(&axis1, 0, sizeof(AxisDEF));
		memset(&axis2, 0, sizeof(AxisDEF));		bRet = true;
#else
		bzero(&axis0, sizeof(AxisDEF));			bzero(&axis1, sizeof(AxisDEF)); 
		bzero(&axis2, sizeof(AxisDEF)); 		bRet = true;
#endif
		axis0.flags = AXIS_NEGTICKS | AXIS_DEFRECT | AXIS_TRIA;
		axis1.flags = AXIS_POSTICKS | AXIS_DEFRECT | AXIS_TRIA;
		axis2.flags = AXIS_POSTICKS | AXIS_DEFRECT | AXIS_TRIA;
		axis0.loc[0].fx = cent.fx-rad_l;	axis0.loc[1].fx = cent.fx+rad_l;
		axis0.loc[0].fy = cent.fy+rad_s;	axis0.loc[1].fy = cent.fy+rad_s;
		axis1.loc[1].fx = cent.fx+rad_l;	axis1.loc[0].fx = cent.fx;
		axis1.loc[1].fy = cent.fy+rad_s;	axis1.loc[0].fy = cent.fy-rad_l;
		axis2.loc[1].fx = cent.fx;		axis2.loc[0].fx = cent.fx-rad_l;
		axis2.loc[1].fy = cent.fy-rad_l;	axis2.loc[0].fy = cent.fy+rad_s;
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, 100) && Dlg->GetText(103, (unsigned char*)(TmpTxt+100), 100) && 
			(rX = new AccRange(TmpTxt)) && (rY = new AccRange(TmpTxt+100)) &&
			(n = rX ? rX->CountItems() : 0) && (Plots = (Plot**)calloc(4, sizeof(Plot*)))) {
			if(Dlg->GetCheck(200)) Symbols = (Symbol**) calloc(n+2, sizeof(Symbol*));
			if(Dlg->GetCheck(250)) TheLine = new DataLine(this, data, TmpTxt, TmpTxt+100, (char*)"Ternary Line");
			else if(Dlg->GetCheck(251)) TheLine = (DataLine*) new DataPolygon(this, data, TmpTxt, TmpTxt+100, (char*)"Ternary Polygon");
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			ic = 0;
			if(Symbols) do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
					CheckBounds(x, y);
					if(Symbols) Symbols[ic++] = new Symbol(this, data, x, y, 
						SYM_CIRCLE, i, j, k, l);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			if(Symbols || TheLine) 
				Plots[nPlots++] = new PlotScatt(this, data, ic, Symbols, TheLine);
			else InfoBox((char*)SCMS_NOPLOTTING);
			}
		axis0.min = axis1.min = axis2.min = 0.0;
		axis0.max = axis1.max = axis2.max = 100.0;
		axis0.Start = axis1.Start = axis2.Start = 0.0;
		axis0.Step = axis1.Step = axis2.Step = 10.0;
		axis0.owner = axis1.owner = axis2.owner = 0L;	//must be NULL, otherwise Axes attempts
														//   to take ownership of a temporary pointer
		axis0.breaks = axis1.breaks = axis2.breaks = 0L;
		axis0.nBreaks = axis1.nBreaks = axis2.nBreaks = 0;
		if(bRet && (Axes = (GraphObj**)calloc(4, sizeof(Axis*))) &&
			(Axes[0] = new Axis(this, data, &axis0, axis0.flags)) &&
			(Axes[1] = new Axis(this, data, &axis1, axis1.flags)) &&
			(Axes[2] = new Axis(this, data, &axis2, axis2.flags))) {
			Axes[0]->type = 5;	Axes[1]->type = 6;	Axes[2]->type = 7;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef0, 0L);
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef1, 0L);
			Axes[2]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef2, 0L);
			Axes[0]->SetSize(SIZE_TLB_YDIST, DefSize(SIZE_AXIS_TICKS)*2.0);
			Axes[1]->SetSize(SIZE_TLB_XDIST, DefSize(SIZE_AXIS_TICKS)*1.73205);
			Axes[2]->SetSize(SIZE_TLB_XDIST, -DefSize(SIZE_AXIS_TICKS)*1.73205);
			Axes[1]->SetSize(SIZE_TLB_YDIST, -DefSize(SIZE_AXIS_TICKS));
			Axes[2]->SetSize(SIZE_TLB_YDIST, -DefSize(SIZE_AXIS_TICKS));
			nAxes = 3;
			}
		else bRet = false;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(TernaryDlg);
	if(rX) delete rX;
	if(rY) delete rY;
	free(tab1);				free(tab2);
	return bRet;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Ternary XYZ (Triangle) plot properties dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *TernaryXYZ_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,148,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,148,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,130,100\n"
	".,10,200,ISPARENT | TOUCHEXIT,SHEET,2,5,10,130,100\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,25,60,8\n"
	".,.,,,RANGEINPUT,-15,20,35,100,10\n"
	".,.,,,LTEXT,-32,10,50,60,8\n"
	".,.,,,RANGEINPUT,-16,20,60,100,10\n"
	".,.,,,LTEXT,-33,10,75,60,8\n"
	".,,,,RANGEINPUT,-17,20,85,100,10\n"
	"200,+,,HICOL | CHECKED,CHECKBOX,3,25,30,60,8\n"
	".,.,250,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,,HICOL | CHECKED,CHECKBOX,6,25,95,60,8\n"
	".,.,,TOUCHEXIT | OWNDIALOG,SYMBUTT,7,85,30,35,10\n"
	".,.,,OWNDIALOG,LINEBUTT,8,85,95,35,10\n"
	".,.,,HICOL,CHECKBOX,9,25,75,60,8\n"
	".,,,OWNDIALOG,ARROWPREV,10,85,75,35,10\n"
	"250,+,,HICOL | ISRADIO,CHECKBOX,4,25,45,60,8\n"
	".,,,HICOL | ISRADIO | LASTOBJ,CHECKBOX,5,25,60,60,8";

bool
TernaryXYZ::AddPlot()
{
	int id;
	TabSHEET *tab1 = MakeTab(0, 10, &id, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(id, 10, &id, (char*)SDLG_TAB_STYLE);
	DlgInfo *TernaryDlg;
	Symbol *PrevSym = new Symbol(0L, data, 0.0, 0.0, 0);
	Arrow *PrevArr = new Arrow(0L, data, 0L, 0L, ARROW_LINE);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_TERN_SYM, (void*)SDLG_TERN_LIN, (void*)SDLG_TERN_PG,
		(void*)SDLG_TERN_GRID, (void*)&PrevSym, (void*)NULL, (void*)SDLG_TERN_ARROWS, (void*)PrevArr };
	DlgRoot *Dlg;
	void *hDlg;
	int res, n;
	long i, j, k, l, i1 = 0, j1 = 0, k1 = 0, l1 = 0, ic, m1, m2, m11 = 0, m22 = 0;
	bool bContinue = false, bRet = false, bLayout = false;
	AccRange *rX = NULL, *rY = NULL, *rZ = NULL;
	double x, y, z;
	Symbol **NewSymbols = NULL;
	long nNewSymbols;
	Arrow **NewArrows = NULL;
	long nNewArrows;
	TernaryXYZ *NewPlot = 0L;
	DataLine *NewLine = 0L;
	fPOINT3D fp1, fp2;

	if(!parent || !data) return false;
	if(!(TernaryDlg = CompileDialog(TernaryXYZ_DlgTmpl, dyndata))) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100, TmpTxt+200);
	TernaryDlg[14].flags |= HIDDEN;						//disable draw_grid-check_box
	if(!(Dlg = new DlgRoot(TernaryDlg, data)))return false;
	Dlg->ShowItem(204, false);
	hDlg = CreateDlgWnd((char*)SDLG_TERXYZ_HD3, 50, 50, 420, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case 1:
			//the layout menu must have been visited
			if(!bLayout) {
				Dlg->SetCheck(5, 0L, bLayout = true);
				res = -1;
				}
			break;
		case -1:
			bContinue = false;
			break;
		case 5:
			bLayout = true;
			res = -1;
			break;
		case 203:
			res = -1; bContinue = false;	//symbol properties
			break;
			}
		}while (res < 0);
	if(res == 1) {
		//create the additional plot
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, 100) && Dlg->GetText(103, (unsigned char*)(TmpTxt+100), 100) && 
			(rX = new AccRange(TmpTxt)) && (rY = new AccRange(TmpTxt+100)) && (rZ = new AccRange(TmpTxt+200)) &&
			(n = rX ? rX->CountItems() : 0)) {
			if(Dlg->GetCheck(200)) NewSymbols = (Symbol**) calloc(n+2, sizeof(Symbol*));
			if (Dlg->GetCheck(205)) NewArrows = (Arrow**)calloc(n + 2, sizeof(ArrTernary*));
			if (Dlg->GetCheck(250)) {
				NewLine = new LineTernary(this, data, TmpTxt, TmpTxt + 100, TmpTxt + 200);
				NewLine->Command(CMD_SET_LINE, &Outline, 0L);
				}
			else if (Dlg->GetCheck(251)) {
				NewLine = new LineTernary(this, data, TmpTxt, TmpTxt + 100, TmpTxt + 200);
				NewLine->Command(CMD_SET_LINE, &Outline, 0L);
				NewLine->Command(CMD_PG_FILL, &Fill, 0L);
				}
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);	rZ->GetFirst(&m1, &m2);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);	rZ->GetNext(&m1, &m2);
			ic = 0;
			if(NewSymbols) do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(m2, m1, &z) ){
					if(NewSymbols) NewSymbols[ic++] = new SymTernary(this, data, x, y, z,
						SYM_CIRCLE, i, j, k, l, m1, m2);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m1, &m2));
			nNewSymbols = ic;
			if (NewArrows) do {
				if (data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(m2, m1, &z)){
					CheckBounds(x, y);
					if (ic){
						fp1.fx = fp2.fx;	fp1.fy = fp2.fy;	fp1.fz = fp2.fz;
						}
					else {
						fp1.fx = x;			fp1.fy = y;			fp1.fz = z;
						i1 = i;	j1 = j;	k1 = k;	l1 = l; m11 = m1; m22 = m2;
						}
					fp2.fx = x;				fp2.fy = y;			fp2.fz = z;
					NewArrows[ic] = new ArrTernary(this, data, ic ? &fp1 : &fp2, &fp2, PrevArr->type,
						i1, j1, k1, l1, m11, m22, i, j, k, l, m1, m2);
					//the first arrow has zero length, all other arrows conncect to the following point
					i1 = i;	j1 = j;	k1 = k;	l1 = l; m11 = m1; m22 = m2;
					ic++;
					}
				} while (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m1, &m2));
			nNewArrows = ic;
			if (NewLine || (NewSymbols && nNewSymbols)|| (NewArrows && nNewArrows)){
				//NewPlot takes ownership of symbols and DataLine/DataPolygon
				NewPlot = new TernaryXYZ(this, data, NewSymbols, nNewSymbols);
				NewPlot->Arrows = NewArrows;		NewPlot->nArrow = nNewArrows;
				NewPlot->SetSize(SIZE_SYMBOL, PrevSym->GetSize(SIZE_SYMBOL));		NewPlot->SetSize(SIZE_SYM_LINE, PrevSym->GetSize(SIZE_SYM_LINE));
				NewPlot->SetColor(COL_SYM_LINE, PrevSym->GetColor(COL_SYM_LINE));	NewPlot->SetColor(COL_SYM_FILL, PrevSym->GetColor(COL_SYM_FILL));
				NewPlot->Command(CMD_SYM_TYPE, (void*)(& PrevSym->type), 0L);
				NewPlot->TheLine = NewLine;
				if (Dlg->GetCheck(251)) NewPlot->doPG = 1;
				if (NewArrows) {
					NewPlot->SetSize(SIZE_ARROW_LINE, PrevArr->GetSize(SIZE_ARROW_LINE));
					NewPlot->SetSize(SIZE_ARROW_CAPWIDTH, PrevArr->GetSize(SIZE_ARROW_CAPWIDTH));
					NewPlot->SetSize(SIZE_ARROW_CAPLENGTH, PrevArr->GetSize(SIZE_ARROW_CAPLENGTH));
					NewPlot->SetColor(COL_ARROW, PrevArr->GetColor(COL_ARROW));
					}
				Command(CMD_DROP_PLOT, NewPlot, 0L);
				bRet = true;
				}
			if(!bRet) InfoBox((char*)SCMS_NOPLOTTING);
			}
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(TernaryDlg);
	delete PrevSym;			PrevSym = 0L;
	delete PrevArr;			PrevArr = 0L;
	if(rX) delete rX;
	if(rY) delete rY;	
	if(rZ) delete rZ;
	free(tab1);				free(tab2);
	return bRet;
}

bool
TernaryXYZ::Config()
{
	InfoBox((char*)SDLG_TERXYZ_HD2);
	return false;
}

bool
TernaryXYZ::PropertyDlg()
{
	int id;
	TabSHEET *tab1 = MakeTab(0, 10, &id, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(id, 10, &id, (char*)SDLG_TAB_STYLE);
	DlgInfo *TernaryDlg;
	Symbol *PrevSym = new Symbol(0L, data, 0.0, 0.0, 0);
	Arrow *PrevArr = new Arrow(0L, data, 0L, 0L, ARROW_LINE);
	LineDEF gridline;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_TERN_SYM, (void*)SDLG_TERN_LIN, (void*)SDLG_TERN_PG,
		(void*)SDLG_TERN_GRID, (void*)&PrevSym, (void*)&gridline, (void*)SDLG_TERN_ARROWS, (void*)PrevArr };
	DlgRoot *Dlg;
	void *hDlg;
	long i, j, k, l, i1, j1, k1, l1, ic, m1, m2, m11, m22;
	int res, n;
	bool bContinue = false, bRet = false, bLayout = false;
	AccRange *rX = NULL, *rY = NULL, *rZ = NULL;
	double x, y, z;
	AxisDEF axis0, axis1, axis2;
	TextDEF tlbdef0 = {0x0, 0x00ffffff, 0.0, 0.0, 0.0, 0, TXA_VTOP | TXA_HCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	TextDEF tlbdef1 = {0x0, 0x00ffffff, 0.0, -60.0, 0.0, 0, TXA_VBOTTOM | TXA_HCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	TextDEF tlbdef2 = {0x0, 0x00ffffff, 0.0, 60.0, 0.0, 0, TXA_VBOTTOM | TXA_HCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	TextDEF label_def = { defs.Color(COL_AXIS), defs.Color(COL_BG), defs.GetSize(SIZE_TICK_LABELS)*1.2, 0.0, 0.0,
		0, 0, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L };
	Label *label = 0L;
	unsigned char ax_desc1[20], ax_desc2[20], ax_desc3[20];
	fPOINT3D fp1, fp2;

	if(!parent || !data) return false;
	memcpy(&gridline, defs.GetGridline(), sizeof(LineDEF));
	if(!(TernaryDlg = CompileDialog(TernaryXYZ_DlgTmpl, dyndata))) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100, TmpTxt+200);
	if(!(Dlg = new DlgRoot(TernaryDlg, data)))return false;
	rad_s = (parent->GetSize(SIZE_DRECT_BOTTOM) - parent->GetSize(SIZE_DRECT_TOP))*0.4;
	rad_l = rad_s / 0.732051;
	cent.fx = (parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_GRECT_RIGHT))*0.5;
	cent.fy = (parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_GRECT_BOTTOM))*0.5;
	tlbdef0.ColTxt = tlbdef1.ColTxt = tlbdef2.ColTxt = defs.Color(COL_AXIS);
	tlbdef0.fSize = tlbdef1.fSize = tlbdef2.fSize = DefSize(SIZE_TICK_LABELS);
	hDlg = CreateDlgWnd((char*)SDLG_TERXYZ_HD1, 50, 50, 420, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case 1:
			//the layout menu must have been visited
			if(!bLayout) {
				Dlg->SetCheck(5, 0L, bLayout = true);
				res = -1;
				}
			break;
		case -1:
			bContinue = false;
			break;
		case 5:
			bLayout = true;
			res = -1;
			break;
		case 203:
			res = -1; bContinue = false;	//symbol properties
			break;
			}
		}while (res < 0);
	if(res == 1) {
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
#ifdef _WINDOWS
		memset(&axis0, 0, sizeof(AxisDEF));		memset(&axis1, 0, sizeof(AxisDEF));
		memset(&axis2, 0, sizeof(AxisDEF));		bRet = true;
#else
		bzero(&axis0, sizeof(AxisDEF));			bzero(&axis1, sizeof(AxisDEF)); 
		bzero(&axis2, sizeof(AxisDEF)); 		bRet = true;
#endif
		axis0.flags = AXIS_NEGTICKS | AXIS_DEFRECT;
		axis1.flags = AXIS_POSTICKS | AXIS_DEFRECT;
		axis2.flags = AXIS_POSTICKS | AXIS_DEFRECT;
		axis0.loc[0].fx = cent.fx-rad_l;	axis0.loc[1].fx = cent.fx+rad_l;
		axis0.loc[0].fy = cent.fy+rad_s;	axis0.loc[1].fy = cent.fy+rad_s;
		axis1.loc[1].fx = cent.fx+rad_l;	axis1.loc[0].fx = cent.fx;
		axis1.loc[1].fy = cent.fy+rad_s;	axis1.loc[0].fy = cent.fy-rad_l;
		axis2.loc[1].fx = cent.fx;			axis2.loc[0].fx = cent.fx-rad_l;
		axis2.loc[1].fy = cent.fy-rad_l;	axis2.loc[0].fy = cent.fy+rad_s;
		if(Dlg->GetCheck(202)) {
			axis0.flags |= AXIS_GRIDLINE;	axis1.flags |= AXIS_GRIDLINE;
			axis2.flags |= AXIS_GRIDLINE;
			}
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, 100) && Dlg->GetText(103, (unsigned char*)(TmpTxt+100), 100) && 
			(rX = new AccRange(TmpTxt)) && (rY = new AccRange(TmpTxt+100)) && (rZ = new AccRange(TmpTxt+200)) &&
			(n = rX ? rX->CountItems() : 0) && (Plots = (Plot**)calloc(4, sizeof(Plot*)))) {
			rlp_strcpy(ax_desc1, 19, rX->RangeDesc(data, 1));
			rlp_strcpy(ax_desc2, 19, rY->RangeDesc(data, 1));
			rlp_strcpy(ax_desc3, 19, rZ->RangeDesc(data, 1));
			if (Dlg->GetCheck(200)) Symbols = (Symbol**)calloc(n + 2, sizeof(Symbol*));
			if (Dlg->GetCheck(205)) Arrows = (Arrow**)calloc(n + 2, sizeof(ArrTernary*));
			if (Dlg->GetCheck(250)) {
				TheLine = new LineTernary(this, data, TmpTxt, TmpTxt + 100, TmpTxt + 200);
				TheLine->Command(CMD_SET_LINE, &Outline, 0L);
				}
			else if (Dlg->GetCheck(251)) {
				doPG = 1;
				TheLine = new LineTernary(this, data, TmpTxt, TmpTxt + 100, TmpTxt + 200);
				TheLine->Command(CMD_SET_LINE, &Outline, 0L);
				TheLine->Command(CMD_PG_FILL, &Fill, 0L);
				}
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);	rZ->GetFirst(&m1, &m2);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);	rZ->GetNext(&m1, &m2);
			ic = 0;
			if(Symbols) do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(m2, m1, &z) ){
					CheckBounds(x, y);
					if(Symbols) Symbols[ic++] = new SymTernary(this, data, x, y, z,
						SYM_CIRCLE, i, j, k, l, m1, m2);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m1, &m2));
			nPoints = ic;
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);	rZ->GetFirst(&m1, &m2);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);	rZ->GetNext(&m1, &m2);
			ic = i1 = j1 = k1 = m11 = m22 = 0;
			if (Arrows) do {
				if (data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(m2, m1, &z)){
					CheckBounds(x, y);
					if (ic){
						fp1.fx = fp2.fx;	fp1.fy = fp2.fy;	fp1.fz = fp2.fz;
					}
					else {
						fp1.fx = x;			fp1.fy = y;			fp1.fz = z;
						i1 = i;	j1 = j;	k1 = k;	l1 = l; m11 = m1; m22 = m2;
						}
					fp2.fx = x;				fp2.fy = y;			fp2.fz = z;
					Arrows[ic] = new ArrTernary(this, data, ic ? &fp1 : &fp2, &fp2, PrevArr->type,
						i1, j1, k1, l1, m11, m22, i, j, k, l, m1, m2);
					//the first arrow has zero length, all other arrows conncect to the following point
					i1 = i;	j1 = j;	k1 = k;	l1 = l; m11 = m1; m22 = m2;
					ic++;
					}
				} while (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m1, &m2));
			nArrow = ic;
			if (Symbols || Arrows || TheLine) bRet = true;
			else InfoBox((char*)SCMS_NOPLOTTING);
			if(Symbols) {
				SetSize(SIZE_SYMBOL, PrevSym->GetSize(SIZE_SYMBOL));			SetSize(SIZE_SYM_LINE, PrevSym->GetSize(SIZE_SYM_LINE));
				SetColor(COL_SYM_LINE, PrevSym->GetColor(COL_SYM_LINE));		SetColor(COL_SYM_FILL, PrevSym->GetColor(COL_SYM_FILL));
				Command(CMD_SYM_TYPE, (void*)(& PrevSym->type), 0L);
				}
			if (Arrows) {
				SetSize(SIZE_ARROW_LINE, PrevArr->GetSize(SIZE_ARROW_LINE));
				SetSize(SIZE_ARROW_CAPWIDTH, PrevArr->GetSize(SIZE_ARROW_CAPWIDTH));
				SetSize(SIZE_ARROW_CAPLENGTH, PrevArr->GetSize(SIZE_ARROW_CAPLENGTH));
				SetColor(COL_ARROW, PrevArr->GetColor(COL_ARROW));
				}
			}
		axis0.min = axis1.min = axis2.min = 0.0;
		axis0.max = axis1.max = axis2.max = 100.0;
		axis0.Start = axis1.Start = axis2.Start = 0.0;
		axis0.Step = axis1.Step = axis2.Step = 10.0;
		axis0.owner = axis1.owner = axis2.owner = 0L;	//must be NULL, otherwise Axes attempts
														//   to take ownership of a temporary pointer
		axis0.breaks = axis1.breaks = axis2.breaks = 0L;
		axis0.nBreaks = axis1.nBreaks = axis2.nBreaks = 0;
		if(bRet && (Axes = (GraphObj**)calloc(4, sizeof(Axis*))) &&
			(Axes[0] = new Axis(this, data, &axis0, axis0.flags)) &&
			(Axes[1] = new Axis(this, data, &axis1, axis1.flags)) &&
			(Axes[2] = new Axis(this, data, &axis2, axis2.flags))) {
			Axes[0]->type = 5;	Axes[1]->type = 6;	Axes[2]->type = 7;
			Axes[0]->Command(CMD_SET_GRIDLINE, (void*)&gridline, 0L);
			label_def.text = ax_desc1;			label_def.Align = TXA_VCENTER | TXA_HLEFT;
			label = new Label(Axes[0], data, axis0.loc[1].fx,
				axis0.loc[1].fy + DefSize(SIZE_AXIS_TICKS), &label_def, 0, 0L);
			if (label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if (label) DeleteGO(label);
			Axes[0]->SetSize(SIZE_LB_XDIST, defs.GetSize(SIZE_AXIS_TICKS)*2.0);
			Axes[0]->SetSize(SIZE_LB_YDIST, defs.GetSize(SIZE_AXIS_TICKS));
			Axes[1]->Command(CMD_SET_GRIDLINE, (void*)&gridline, 0L);
			label_def.text = ax_desc2;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, cent.fx,
				cent.fy - rad_l + DefSize(SIZE_AXIS_TICKS), &label_def, 0, 0L);
			if (label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if (label) DeleteGO(label);
			Axes[1]->SetSize(SIZE_LB_YDIST, -defs.GetSize(SIZE_AXIS_TICKS) * 4);
			Axes[2]->Command(CMD_SET_GRIDLINE, (void*)&gridline, 0L);
			label_def.text = ax_desc3;			label_def.Align = TXA_VCENTER | TXA_HRIGHT;
			label = new Label(Axes[2], data, axis2.loc[0].fx,
				axis2.loc[0].fy + DefSize(SIZE_AXIS_TICKS), &label_def, 0L, 0L);
			if (label && Axes[2]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if (label) DeleteGO(label);
			Axes[2]->SetSize(SIZE_LB_XDIST, -defs.GetSize(SIZE_AXIS_TICKS)*2.0);
			Axes[2]->SetSize(SIZE_LB_YDIST, defs.GetSize(SIZE_AXIS_TICKS));
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef0, 0L);
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef1, 0L);
			Axes[2]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef2, 0L);
			Axes[0]->SetSize(SIZE_TLB_YDIST, DefSize(SIZE_AXIS_TICKS)*2.0);
			Axes[1]->SetSize(SIZE_TLB_XDIST, DefSize(SIZE_AXIS_TICKS)*1.73205);
			Axes[2]->SetSize(SIZE_TLB_XDIST, -DefSize(SIZE_AXIS_TICKS)*1.73205);
			Axes[1]->SetSize(SIZE_TLB_YDIST, -DefSize(SIZE_AXIS_TICKS));
			Axes[2]->SetSize(SIZE_TLB_YDIST, -DefSize(SIZE_AXIS_TICKS));
			nAxes = 3;
			}
		else bRet = false;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(TernaryDlg);
	delete PrevSym;			PrevSym = 0L;
	delete PrevArr;			PrevArr = 0L;
	if(rX) delete rX;
	if(rY) delete rY;
	if(rZ) delete rZ;
	free(tab1);				free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Box plot properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *BoxPlotDlg_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
	".,10,,,PUSHBUTTON,-2,130,25,45,12\n"
	"10,50,,CHECKED,CHECKPIN,,5,,12,8\n"
	"50,60,51,ISPARENT | CHECKED, GROUP,,,,,\n"
	"51,+,,,LTEXT,1,10,12,40,9\n"
	".,.,,TOUCHEXIT,RADIO2,2,60,12,50,9\n"
	".,,,TOUCHEXIT,RADIO2,3,60,22,60,9\n"
	"60,+,100,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,,200,ISPARENT | CHECKED,GROUP,,,,,\n"
	"100,102,,,LTEXT,4,10,39,140,9\n"
	"102,+,,, RANGEINPUT,-16,10,49,165,10\n"
	".,.,,,LTEXT,-32,10,60,90,9\n"
	".,150,,,RANGEINPUT,-17,10,70,165,10\n"
	"150,160,151,ISPARENT | CHECKED,GROUPBOX,5,10,87,165,45\n"
	"151,+,,HICOL,CHECKBOX,6,15,92,50,9\n"
	".,.,,HICOL | CHECKED, CHECKBOX,7,15,101,50,9\n"
	".,.,,,LTEXT,8,65,101,30,9\n"
	".,.,,HICOL,RADIO1,9,95,90,50,9\n"
	".,.,,HICOL,RADIO1,10,95,99,50,9\n"
	".,.,,HICOL,RADIO1,11,95,108,50,9\n"
	".,,,HICOL | CHECKED,RADIO1,12,95,117,50,9\n"
	"160,170,161, ISPARENT | CHECKED, GROUPBOX,13,10,137,165,38\n"
	"161,+,,HICOL | ISRADIO,CHECKBOX,14,15,142,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,15,15,151,70,9\n"
	".,.,,HICOL | ISRADIO | CHECKED,CHECKBOX,16,95,142,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,17,95,151,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,18,15,161,70,9\n"
	".,.,,,EDVAL1,19,28,160,15,10\n"
	".,,,,LTEXT,20,45,161,70,9\n"
	"170,400,171,ISPARENT | CHECKED,GROUPBOX,21,10,180,165,38\n"
	"171,+,,HICOL | ISRADIO,CHECKBOX,14,15,185,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,15,15,194,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,16,95,185,70,9\n"
	".,.,,HICOL | ISRADIO | CHECKED,CHECKBOX,17,95,194,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,18,15,204,70,9\n"
	".,.,,,EDVAL1,22,28,203,15,10\n"
	".,,,,LTEXT,20,45,203,70,9\n"
	"200,202,,,LTEXT,23,10,39,140,9\n"
	"202,250,,,RANGEINPUT,-16,10,49, 165,10\n"
	"250,260,251,ISPARENT | CHECKED,GROUPBOX,,10,68,165,30\n"
	"251,+,,HICOL,CHECKBOX,24,15,63,50,9\n"
	".,.,,,LTEXT,25,15,73,80,9\n"
	".,,,,RANGEINPUT,-17,15,83,155,10\n"
	"260,270,261,ISPARENT | CHECKED,GROUPBOX,,10,106,165,30\n"
	"261,+,,HICOL | CHECKED,CHECKBOX,26,15,101,50,9\n"
	".,.,,,LTEXT,27,15,111,80,9\n"
	".,,,,RANGEINPUT,-17,15,121,155,10\n"
	"270,280,271,ISPARENT | CHECKED,GROUPBOX,,10,144,165,50\n"
	"271,+,,HICOL | CHECKED,CHECKBOX,28,15,139,50,9\n"
	".,.,,,LTEXT,29,15,149,80,9\n"
	".,.,,,RANGEINPUT,-18,15,159,155,10\n"
	".,.,,,LTEXT,30,15,169,80,9\n"
	".,,,,RANGEINPUT,-19,15,179,155,10\n"
	"280,,281,ISPARENT | CHECKED,GROUPBOX,,10,202,165,50\n"
	"281,+,0,HICOL | CHECKED | TOUCHEXIT,CHECKBOX,21,15,197,50,9\n"
	".,.,,,LTEXT,29,15,207,80,9\n"
	".,.,,,RANGEINPUT,31,15,217,155,10\n"
	".,.,,,LTEXT,30,15,227,80,9\n"
	".,300,,,RANGEINPUT,32,15,237,155,10\n"
	"300,+,,ISPARENT | CHECKED,GROUPBOX,,10,260,165,50\n"
	".,.,0,HICOL | TOUCHEXIT,CHECKBOX,38,15,255,50,9\n"
	".,.,,,LTEXT,29,15,265,80,9\n"
	".,.,,,RANGEINPUT,31,15,275,155,10\n"
	".,.,,,LTEXT,30,15,285,80,9\n"
	".,,,,RANGEINPUT,32,15,295,155,10\n"
	"400,,401,ISPARENT | CHECKED,GROUPBOX,33,10,223,165,30\n"
	"401,+,, HICOL | ISRADIO | CHECKED,CHECKBOX,34,15,228,70,9\n"
	".,.,,HICOL | ISRADIO,CHECKBOX,35,95,228,70,9\n"
	".,.,,,LTEXT,36,15,238,24,9\n"
	".,,,LASTOBJ,EDTEXT,37,40,237,30,10";

static int boxplot_mode_sel = 52;
bool
BoxPlot::PropertyDlg()
{
	void *dyndata[] = { (void*)SDLG_BOXPLT_DSC, (void*)SDLG_BOXPLT_UVAL, (void*)SDLG_BOXPLT_SVAL,
		(void*)SDLG_BOXPLT_XVAL, (void*)SDLG_BOXPLT_DMEAN, (void*)SDLG_BOXPLT_DLINE, (void*)SDLG_BOXPLT_DSYMS,
		(void*)SDLG_BOXPLT_USING, (void*)SDLG_BOXPLT_AMEAN, (void*)SDLG_BOXPLT_GMEAN, (void*)SDLG_BOXPLT_HMEAN,
		(void*)SDLG_BOXPLT_MEDIAN, (void*)SDLG_BOXPLT_BOXES, (void*)SDLG_BOXPLT_SDEV, (void*)SDLG_BOXPLT_SERR,
		(void*)SDLG_BOXPLT_PERC, (void*)SDLG_BOXPLT_MIMA, (void*)SDLG_BOXPLT_SPC, (void*)&ci_box,
		(void*)SDLG_BOXPLT_COFI, (void*)SDLG_BOXPLT_ERRS, (void*)&ci_err, (void*)SDLG_BOXPLT_COMX,
		(void*)SDLG_BOXPLT_DOLIN, (void*)SDLG_BOXPLT_LINVAL, (void*)SDLG_BOXPLT_DOSYMS,
		(void*)SDLG_BOXPLT_RNGSYMS, (void*)SDLG_BOXPLT_DOBOX, (void*)SDLG_BOXPLT_RNGHI, (void*)SDLG_BOXPLT_RNGLO,
		(void*)(TmpTxt + 500), (void*)(TmpTxt + 600), (void*)SDLG_BOXPLT_NCASE, (void*)SDLG_BOXPLT_TOPE,
		(void*)SDLG_BOXPLT_TOPM, (void*)SDLG_BOXPLT_PREF, (void*)SDLG_BOXPLT_NEQ, (void*)SDLG_BOXPLOT_ERRPG };

	DlgInfo *PlotDlg = CompileDialog(BoxPlotDlg_DlgTmpl, dyndata);

	DlgRoot *Dlg;
	void *hDlg;
	bool bRet = false;
	long i, j, k, k1, l, l1;
	int n, ic, c, res, nVals, nTxt, nTime;
	long width, height;
	double x, y1, y2, dx, dy;
	char errdesc[40], boxdesc[40], symdesc[40];
	lfPOINT fp1, fp2;
	TextDEF lbdef = {defs.Color(COL_TEXT), defs.Color(COL_BG), DefSize(SIZE_TEXT), 0.0f, 0.0f, 0,
		TXA_HLEFT | TXA_VBOTTOM, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, (unsigned char*)TmpTxt};
	AccRange *rX = 0L, *rY1 = 0L, *rY2 = 0L;

	if(!parent || !data) return false;
	name = rlp_strdup((char*)"Box Plot");
	n = UseRangeMark(data, 1, TmpTxt+100, TmpTxt+200, TmpTxt+300, TmpTxt+400, TmpTxt+500, TmpTxt+600);
	ci_box = ci_err = 95.0;
	if(!(Dlg = new DlgRoot(PlotDlg, data)))return false;
	TmpTxt[0] = TmpTxt[100] = 0;
	//restore previous style
	if(boxplot_mode_sel == 53) {
		Dlg->ShowItem(61, false);			Dlg->SetCheck(53, 0L, true);
		}
	else {
		Dlg->SetCheck(52, 0L, true);		Dlg->ShowItem(60, false);
		}
	if (n == 2) {
		Dlg->SetCheck(53, 0L, true);		Dlg->SetCheck(52, 0L, false);
		Dlg->ShowItem(60, true);			Dlg->ShowItem(61, false);
		}
	hDlg = CreateDlgWnd((char*)SDLG_BOXPLT_HD1, 100, -10, 390, 680, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if (Dlg->GetCheck(10)) res = -1;
			break;
		case 52:	case 53:
			boxplot_mode_sel = res;
			if (res == 53) {
				Dlg->ShowItem(60, true);	Dlg->ShowItem(61, false);
			}
			else {
				Dlg->ShowItem(60, false);	Dlg->ShowItem(61, true);
			}
			Dlg->Command(CMD_REDRAW, 0L, 0L);	res = -1;
			break;
		case 281:			//draw whiskers
			Dlg->SetCheck(301, 0L, false);		res = -1;
			break;
		case 301:			//draw error polygon
			Dlg->SetCheck(281, 0L, false);		res = -1;
			break;
			}
		}while (res <0);
	if(res == 1) {
		type = 0;				dirty = true;
		if(Dlg->GetCheck(52) && Dlg->GetText(202, (unsigned char*)(TmpTxt+100), 50) && TmpTxt[100] &&(rX = new AccRange(TmpTxt+100))) {
			xRange = rlp_strdup(TmpTxt+100);
			n = rX->CountItems();	nPoints = n;
			//analyse data types
			if (rX->DataTypes(data, &nVals, &nTxt, &nTime) && !x_tv){
				if (!nVals && nTxt > 1 && nTxt > nTime) x_tv = new TextValue();
				else if (!nVals && nTime > 1 && nTime > nTxt) x_dtype = ET_DATETIME;
				}
//			if (rY->DataTypes(data, &nVals, &nTxt, &nTime) && !y_tv){
//				if (!nVals && nTxt > 1 && nTxt > nTime) y_tv = new TextValue();
//				else if (!nVals && nTime > 1 && nTime > nTxt) y_dtype = ET_DATETIME;
//				}
			//  data line
			if(n > 1 && Dlg->GetCheck(251) && Dlg->GetText(253, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
				TheLine = new DataLine(this, data, TmpTxt+100, TmpTxt);
				bRet = true;
				}
			// error polygon
			if (n >1 && Dlg->GetCheck(301) && xRange && xRange[0] && Dlg->GetText(303, (unsigned char*)(TmpTxt + 100), TMP_TXT_SIZE - 100) &&
				Dlg->GetText(305, (unsigned char*)(TmpTxt + 200), TMP_TXT_SIZE - 200)){
				ErrPg = new ErrorPolygon(this, data, xRange, TmpTxt+200, TmpTxt+100);
				if (!ErrPg->LoadValues()) {
					DeleteGO(ErrPg);	ErrPg = 0L;
					}
				else bRet = true;
				}
			// symbols
			if(n > 0 && Dlg->GetCheck(261) && Dlg->GetText(263, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0] 
				&& (Symbols = (Symbol**)calloc(n+2, sizeof(Symbol*)))
				&& (rY1 = new AccRange(TmpTxt))) {
				yRange = rlp_strdup(TmpTxt);
				rX->GetFirst(&i, &j);	rY1->GetFirst(&k, &l);
				rX->GetNext(&i, &j);	rY1->GetNext(&k, &l);
				ic = c = 0;
				do {
					if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y1)) {
						Symbols[ic] = new Symbol(this, data, x, y1, SYM_PLUS, i, j, k, l);
						if (Symbols[ic]) Symbols[ic++]->idx = c;
						}
					c++;
					}while(rX->GetNext(&i, &j) && rY1->GetNext(&k, &l));
				delete rY1;		rY1 = 0L;
				if(ic) bRet = true;
				}
			// boxes
			if(n > 0 && Dlg->GetCheck(271) && Dlg->GetText(273, (unsigned char*)(TmpTxt+300), 50) && Dlg->GetText(275, (unsigned char*)(TmpTxt+400), 50)
				&& (Boxes = (Box**)calloc(n+2, sizeof(Box*)))
				&& (rY1 = new AccRange(TmpTxt+300)) && (rY2 = new AccRange(TmpTxt+400))) {
				rX->GetFirst(&i, &j);	rY1->GetFirst(&k, &l);	rY2->GetFirst(&k1, &l1);
				rX->GetNext(&i, &j);	rY1->GetNext(&k, &l);	rY2->GetNext(&k1, &l1);
				ic = 0;
				do {
					if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y1) && data->GetValue(l1, k1, &y2)) {
						fp1.fy = y1;	fp2.fy = y2;		fp1.fx = fp2.fx = x;
						Boxes[ic] = new Box(this, data, fp1, fp2, BAR_RELWIDTH, i, j, k, l, i, j, k1, l1);
						if(Boxes[ic]) Boxes[ic++]->SetSize(SIZE_BOX, 60.0);
						}
					}while(rX->GetNext(&i, &j) && rY1->GetNext(&k, &l) && rY2->GetNext(&k1, &l1));
				delete rY1;		rY1 = 0L;		delete rY2;		rY2 = 0L;
				if(ic) bRet = true;
				}
			// whiskers
			if(n > 0 && Dlg->GetCheck(281) && Dlg->GetText(283, (unsigned char*)(TmpTxt+300), 50) && Dlg->GetText(285, (unsigned char*)(TmpTxt+400), 50)
				&& (Whiskers = (Whisker**)calloc(n+2, sizeof(Whisker*)))
				&& (rY1 = new AccRange(TmpTxt+300)) && (rY2 = new AccRange(TmpTxt+400))) {
				rX->GetFirst(&i, &j);	rY1->GetFirst(&k, &l);	rY2->GetFirst(&k1, &l1);
				rX->GetNext(&i, &j);	rY1->GetNext(&k, &l);	rY2->GetNext(&k1, &l1);
				ic = 0;
				do {
					if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y1) && data->GetValue(l1, k1, &y2)) {
						fp1.fy = y1;	fp2.fy = y2;		fp1.fx = fp2.fx = x;
						Whiskers[ic++] = new Whisker(this, data, fp1, fp2, 0, i, j, k, l, i, j, k1, l1);
						}
					}while(rX->GetNext(&i, &j) && rY1->GetNext(&k, &l) && rY2->GetNext(&k1, &l1));
				delete rY1;		rY1 = 0L;		delete rY2;		rY2 = 0L;
				if(ic) bRet = true;
				}
			if (bRet) Command(CMD_AUTOSCALE, 0L, 0L);
			}
		else if(Dlg->GetText(102, (unsigned char*)(TmpTxt+100), 50) && TmpTxt[100] && Dlg->GetText(104, (unsigned char*)(TmpTxt+200), 50) && TmpTxt[200]){
			xRange = rlp_strdup(TmpTxt+100);			yRange = rlp_strdup(TmpTxt+200);
			if((rX = new AccRange(xRange)) && (rY1 = new AccRange(yRange))) {
				x_info = rlp_strdup(rX->RangeDesc(data, 2));
				y_info = rlp_strdup(rY1->RangeDesc(data, 2));
				delete rX;		delete rY1;		rX = rY1 = 0L;
				}
			if(Dlg->GetCheck(154)) type |= 0x0001;
			if(Dlg->GetCheck(155)) type |= 0x0002;
			if(Dlg->GetCheck(156)) type |= 0x0003;
			if(Dlg->GetCheck(157)) type |= 0x0004;
			if(Dlg->GetCheck(161)) type |= 0x0010;
			if(Dlg->GetCheck(162)) type |= 0x0020;
			if(Dlg->GetCheck(163)) type |= 0x0030;
			if(Dlg->GetCheck(164)) type |= 0x0040;
			if(Dlg->GetCheck(165)) type |= 0x0050;
			if(Dlg->GetCheck(171)) type |= 0x0100;
			if(Dlg->GetCheck(172)) type |= 0x0200;
			if(Dlg->GetCheck(173)) type |= 0x0300;
			if(Dlg->GetCheck(174)) type |= 0x0400;
			if(Dlg->GetCheck(175)) type |= 0x0500;
			if(Dlg->GetCheck(151)) type |= 0x1000;
			if(Dlg->GetCheck(152)) type |= 0x2000;
			if(Dlg->GetCheck(401)) type |= 0x4000;
			if(Dlg->GetCheck(402)) type |= 0x8000;
			Dlg->GetValue(166, &ci_box);				Dlg->GetValue(176, &ci_err);
			if(Dlg->GetText(404, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) case_prefix = rlp_strdup(TmpTxt);
			CreateData();
			if(curr_data && type) {
				curr_data->GetSize(&width, &height);
#ifdef USE_WIN_SECURE
				sprintf_s(TmpTxt+100, 50, "a1:a%ld", height);	sprintf_s(TmpTxt+200, 50, "b1:b%ld", height);
#else	
				sprintf(TmpTxt+100, "a1:a%ld", height);			sprintf(TmpTxt+200, "b1:b%ld", height);
#endif
				nPoints = height;
				if(nPoints > 1 && (type & 0x1000)) {
					TheLine = new DataLine(this, curr_data, TmpTxt+100, TmpTxt+200);
					bRet = true;
					}
				if(nPoints > 0 && (type & 0x2000) && (Symbols = (Symbol**)calloc(nPoints+2, sizeof(Symbol*)))) {
					switch(type & 0x000f) {
					case 0x0001:	rlp_strcpy(symdesc, 40, (char*)"Mean");				break;
					case 0x0002:	rlp_strcpy(symdesc, 40, (char*)"Geometric mean");		break;
					case 0x0003:	rlp_strcpy(symdesc, 40, (char*)"Harmonic mean");		break;
					case 0x0004:	rlp_strcpy(symdesc, 40, (char*)"Median");				break;
					default:		rlp_strcpy(symdesc, 40, (char*)"n.a.");				break;
						}
					for(i = 0; i < height; i++) {
						if(curr_data->GetValue(i, 0, &x) &&	curr_data->GetValue(i, 1, &y1)
							&& (Symbols[i] = new Symbol(this, curr_data, x, y1, SYM_PLUS, 0, i, 1, i))){
							Symbols[i]->idx = i;
							Symbols[i]->name = rlp_strdup(symdesc);
							}
						}
					bRet = true;
					}
				if(nPoints > 0 && (type & 0x00f0) && (Boxes = (Box**)calloc(nPoints+2, sizeof(Box*)))) {
					switch(type & 0x00f0) {
					case 0x0010:	rlp_strcpy(boxdesc, 40, (char*)"Std. Dev.");			break;
					case 0x0020:	rlp_strcpy(boxdesc, 40, (char*)"Std. Err.");			break;
					case 0x0030:	rlp_strcpy(boxdesc, 40, (char*)"25, 75% Perc.");		break;
					case 0x0040:	rlp_strcpy(boxdesc, 40, (char*)"Min./Max.");			break;
#ifdef USE_WIN_SECURE
					case 0x0500:	sprintf_s(boxdesc, 40, "'%g%% CI", ci_err);	break;
#else
					case 0x0500:	sprintf(boxdesc, "'%g%% CI", ci_err);			break;
#endif
					default:		rlp_strcpy(boxdesc, 40, (char*)"n.a.");
						}
					for(i = 0; i < height; i++) {
						if(curr_data->GetValue(i, 0, &x) &&	curr_data->GetValue(i, 2, &y1)
							&&	curr_data->GetValue(i, 3, &y2)) {
							fp1.fy = y1;	fp2.fy = y2;		fp1.fx = fp2.fx = x;
							Boxes[i] = new Box(this, curr_data, fp1, fp2, BAR_RELWIDTH, 0, i, 2, i, 0, i, 3, i);
							if(Boxes[i]){
								Boxes[i]->SetSize(SIZE_BOX, 60.0);
								Boxes[i]->name = rlp_strdup(boxdesc);
								}
							}
						}
					bRet = true;
					}
				if(nPoints > 0 && (type & 0x0f00) && (Whiskers = (Whisker**)calloc(nPoints+2, sizeof(Whisker*)))) {
					switch(type & 0x0f00) {
					case 0x0100:	rlp_strcpy(errdesc, 40, (char*)"Std. Dev.");			break;
					case 0x0200:	rlp_strcpy(errdesc, 40, (char*)"Std. Err.");			break;
					case 0x0300:	rlp_strcpy(errdesc, 40, (char*)"25, 75% Perc.");		break;
					case 0x0400:	rlp_strcpy(errdesc, 40, (char*)"Min./Max.");			break;
#ifdef USE_WIN_SECURE
					case 0x0500:	sprintf_s(errdesc, 40, "'%g%% CI", ci_err);		break;
#else
					case 0x0500:	sprintf(errdesc, "'%g%% CI", ci_err);			break;
#endif
					default:		rlp_strcpy(errdesc, 40, (char*)"error");
						}
					for(i = 0; i < height; i++) {
						if(curr_data->GetValue(i, 0, &x) &&	curr_data->GetValue(i, 4, &y1)
							&&	curr_data->GetValue(i, 5, &y2)) {
							fp1.fy = y1;	fp2.fy = y2;		fp1.fx = fp2.fx = x;
							Whiskers[i] = new Whisker(this, curr_data, fp1, fp2, 0, 0, i, 4, i, 0, i, 5, i);
							if(Whiskers[i]) Whiskers[i]->Command(CMD_ERRDESC, errdesc, 0L);
							}
						}
					bRet = true;
					}
				if(nPoints > 0 && (type & 0xc000) && (Labels = (Label**)calloc(nPoints+2, sizeof(Label*)))) {
					dy = -0.4 * DefSize(SIZE_SYMBOL);
					if(type & 0x4000){
						lbdef.Align = TXA_HCENTER | TXA_VBOTTOM;
						dx = 0.0;
						}
					else {
						lbdef.Align = TXA_HLEFT | TXA_VBOTTOM;
						dx = -dy;
						}
					for(i = 0; i < height; i++) {
						if(curr_data->GetValue(i, 0, &x) && curr_data->GetText(i, 7, TmpTxt, TMP_TXT_SIZE)){
							if(curr_data->GetValue(i, 6, &y1))Labels[i] = new Label(this, curr_data, x, y1, &lbdef, 
								LB_X_DATA | LB_Y_DATA, 0L, 0, i, 6, i, 7, i);
							if(Labels[i]){
								Labels[i]->SetSize(SIZE_LB_YDIST, dy);	Labels[i]->SetSize(SIZE_LB_XDIST, dx);
								}
							}
						}
					bRet = true;
					}
				}
			}
		}
	if(bRet) {
		if	(Boxes) nBoxes = nPoints;	
		if (Whiskers) nWhiskers = nPoints;
		if (Labels) nLabel = nPoints;
		dirty = true;
		Command(CMD_AUTOSCALE, 0L, 0L);
		}
	CloseDlgWnd(hDlg);		delete Dlg;		
	if (rX) delete rX;	
	free(PlotDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create density distribution plot 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *DensDisp_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1, 148, 10, 45, 12\n"
".,.,,,PUSHBUTTON,-2,148,25,45,12\n"
".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,130,100\n"
".,10,200,TOUCHEXIT | ISPARENT,SHEET,2,5,10,130,100\n"
"10,,,CHECKED,CHECKPIN,,5,0,12,8\n"
"100,+,,,LTEXT,3,10,30,60,8\n"
".,.,,,RANGEINPUT,4,20,40,100,10\n"
".,.,,,LTEXT,5,10,55,60,8\n"
".,.,,,RANGEINPUT,6,20,65,100,10\n"
".,,,HICOL,CHECKBOX,7,10,90,100,8\n"
"200,+,,NOSELECT,ODBUTTON,8,25,30,90,45\n"
".,.,,TOUCHEXIT | HICOL | CHECKED,RADIO1,9,25,80,60,8\n"
".,.,,TOUCHEXIT | HICOL,RADIO1,0,25,88,60,8\n"
".,,,TOUCHEXIT | HICOL | LASTOBJ,RADIO1,0,25,96,60,9";

bool
DensDisp::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	char text1[100], text2[100];
	DlgInfo *PlotDlg;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_DENSDISP_RANGE1, (void*)text1,
		(void*)SDLG_DENSDISP_RANGE2, (void*)text2, (void*)SDLG_DENSDISP_VPROF, (void*)OD_filldef,
		(void*)SDLG_DENSDISP_SYMBAR};
	DlgRoot *Dlg;
	void *hDlg;
	int n, res, align = 0;
	bool bRet = false, bContinue = false, bVert;
	AccRange *rX = 0L, *rY = 0L;

	if(!parent || !data) return false;
	if (!(PlotDlg = CompileDialog(DensDisp_DlgTmpl, dyndata))) return false;
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)defs.GetOutLine(), 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)defs.GetFill(defs.units()), 0);
	UseRangeMark(data, 1, text1, text2);
	if(!(Dlg = new DlgRoot(PlotDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_DENSDISP_HD1, 50, 50, 420, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 5:
			if(Dlg->GetCheck(104)) {
				Dlg->SetText(202, (unsigned char*)SDLG_DENSDISP_RIGHT);	
				Dlg->SetText(203, (unsigned char*)SDLG_DENSDISP_LEFT);
				}
			else {
				Dlg->SetText(202, (unsigned char*)SDLG_DENSDISP_UP);
				Dlg->SetText(203, (unsigned char*)SDLG_DENSDISP_DOWN);
				}
			res = -1;
			break;
		case 201:	case 202:	case 203:
			align = res - 201;
			res = -1;
			break;
		case 1:
			if(rX) delete rX;
			if(rY) delete rY;
			rX = 0L;			rY = 0L;
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rX = new AccRange(TmpTxt);
			n = rX ? rX->CountItems() : 0;
			if(!n) {
				ErrorBox((char*)SCMS_RANGE_DIR);
				bContinue = true;				res = -1;
				}
			if(n && Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && (rY = new AccRange(TmpTxt))){
				if(n != rY->CountItems()) {
					ErrorBox((char*)SCMS_BAD_RANGE);
					bContinue = true;			res = -1;
					}
				}
			}
		}while (res < 0);
	if(res == 1 && n && rX && rY) {
		if(Dlg->GetCheck(104)) {
			y_info = rlp_strdup(rX->RangeDesc(data, 0));
			x_info = rlp_strdup(rY->RangeDesc(data, 0));
			}
		else {
			x_info = rlp_strdup(rX->RangeDesc(data, 0));
			y_info = rlp_strdup(rY->RangeDesc(data, 0));
			}
		type = (bVert = Dlg->GetCheck(104)) ? align | 0x10 : align;
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) xRange = rlp_strdup(TmpTxt);
		if(Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) yRange = rlp_strdup(TmpTxt);
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&DefLine, 0);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&DefFill, 0);
		if(DefFill.hatch) memcpy(&DefFillLine, DefFill.hatch, sizeof(LineDEF));
		DefFill.hatch = &DefFillLine;
		DoUpdate();
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rX) delete rX;
	if(rY) delete rY;
	free(tab1);				free(tab2);				free(PlotDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create stacked bar or stacked polygon 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *StackBar_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,158,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,158,25,45,12\n"
	".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED, SHEET,1,5,10,140,100\n"
	".,.,200,ISPARENT,SHEET,2,5,10,140,100\n"
	".,20,300,ISPARENT,SHEET,3,5,10,140,100\n"
	"20,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,4,15,30,60,8\n"
	".,152,,,RANGEINPUT,5,25,40,100,10\n"
	"152,+,,ISPARENT | CHECKED,GROUPBOX,6,12,60,128,45\n"
	".,.,,,LTEXT,0,25,65,60,8\n"
	".,.,,,RANGEINPUT,5,25,75,100,10\n"
	".,.,,,PUSHBUTTON,-8,95,87,30,12\n"
	".,,,,PUSHBUTTON,-9,60,87,35,12\n"
	"200,+,,HICOL| CHECKED,RADIO1,7,25,35,60,8\n"
	".,.,,HICOL,RADIO1,8,25,50,60,8\n"
	".,.,,,RTEXT,9,31,65,38,8\n"
	".,.,,,EDVAL1,10,70,65,30,10\n"
	".,,,HICOL,CHECKBOX,11,25,90,60,8\n"
	"300,,,LASTOBJ | NOSELECT,ODB,12,20,35,80,60";

bool
StackBar::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_STB_COMX,
		(void*)TmpTxt, (void*)SDLG_STB_MBY, (void*)SDLG_STB_YADD,
		(void*)SDLG_STB_YSUB, (void*)SDLG_STB_STV, (void*)&StartVal,
		(void*)SDLG_STB_HPLOT, (void*)(OD_scheme) };
	DlgInfo *StackBarDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int sc, j, res, currYR = 0, maxYR = 0, nx = 0, ny, c_num, c_txt, c_datetime;
	bool updateYR = true, bContinue = false, bRet = false;
	char **rd = 0L, *rname;
	AccRange *rX = 0L, *rY = 0L;
	TextValue *tv = 0L;

	if(!parent || !data) return false;
	if(!UseRangeMark(data, 2, TmpTxt, TmpTxt+100, TmpTxt+200, TmpTxt+300, TmpTxt+400,
		TmpTxt+500, TmpTxt+600, TmpTxt+700, TmpTxt+800, TmpTxt+900, TmpTxt+1000)) return false;
	if(!(StackBarDlg = CompileDialog(StackBar_DlgTmpl, dyndata))) return false;
	if(TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(20, sizeof(char*)))) {
		for (i = 100, j = 0; i <= 1000; i += 100) if (TmpTxt[i]) {
			rd[j++] = rlp_strdup(TmpTxt + i);					maxYR = j - 1;
			rd[j] = 0L;
			}
		}
	if (!rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	if(!(Dlg = new DlgRoot(StackBarDlg, data))) return false;
	if(rd && rd[currYR] &&  *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd(Id == GO_STACKBAR ? (char*)SDLG_STB_HD1 : (char*)SDLG_STB_HD2, 50, 50, 440, 280, Dlg, 0x4L);
	do {
		if(updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(156, 1);				Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);				Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_STB_RFMT, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt,SDLG_STB_RFMT, currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		ny = 0;
		if(rX) delete rX;
		rX = 0L;
		switch(res) {
		case 0:
			if(bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 1:		case 155:		case 156:
			if (res == 1) Dlg->GetValue(203, &StartVal);
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
			}
		}while (res < 0);
	if(res == 1 && nx && rX && rd && rd[0] && rd[0][0]){	//accept settings and create plot
		maxYR++;
		for(i = j = 0; i < maxYR; i++) {
			if(i) TmpTxt[j++] = '&';
			j+= rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, rd[i]);
			}
		rX->DataTypes(data, &c_num, &c_txt, &c_datetime);
		if(!c_num && (c_txt + c_datetime) > 0 ) tv = new TextValue();
		if(Dlg->GetCheck(204)) {
			y_info = rlp_strdup(rX->RangeDesc(data, 3));	y_tv = tv;
			}
		else {
			x_info = rlp_strdup(rX->RangeDesc(data, 3));	x_tv = tv;
			}
		ssYrange = rlp_strdup(TmpTxt);
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) ssXrange = rlp_strdup(TmpTxt);
		cum_data_mode = Dlg->GetCheck(200) ? 1 : 2;
		if(Id == GO_STACKPG) cum_data_mode += 2;
		CumData = CreaCumData(ssXrange, ssYrange, cum_data_mode, StartVal);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		//do stacked bar
		if(rY) delete rY;
		rY = 0L;
		if(Id == GO_STACKBAR){
			numPlots = maxYR;
			Boxes = (BoxPlot**)calloc(numPlots + 2, sizeof(BoxPlot*));
			if(Boxes) for(i = sc = 0; i < (maxYR) && rd[i] && *rd[i]; i++) {
				rY = new AccRange(rd[i]);
				rname = rlp_strdup(rY->RangeDesc(data, 3));
				Boxes[i] = new BoxPlot(0L, CumData, Dlg->GetCheck(204) ? 2 : 1, 0, i + 1, i + 2, rname);
				if (Boxes[i]){
					Boxes[i]->Command(CMD_UPDATE, 0L, 0L);	Boxes[i]->parent = this;
					Boxes[i]->Command(CMD_AUTOSCALE, 0L, 0L);
					Boxes[i]->Command(CMD_BOX_FILL, GetSchemeFill(&sc), 0L);
					Boxes[i]->SetSize(SIZE_BOX, 60.0);
					if(rname) free(rname);
					delete rY;	rY = 0L;
					}
				}
			}
		//do stacked polygon
		else if(Id == GO_STACKPG){
			numPG = maxYR;
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, 20, "a1:a%d", nx*2);
#else
			sprintf(TmpTxt, "a1:a%d", nx*2);
#endif
			Polygons = (DataPolygon**)calloc(numPG + 2, sizeof(DataPolygon*));
			if(Polygons) for(i=sc=0; i < maxYR && rd[i] && *rd[i];i++){
#ifdef USE_WIN_SECURE
				sprintf_s(TmpTxt+20, 20, "%c1:%c%d", 'c'+i, 'c'+i, nx*2);
#else
				sprintf(TmpTxt+20, "%c1:%c%d", 'c'+i, 'c'+i, nx*2);
#endif
				rY = new AccRange(rd[i]);				rname = rlp_strdup(rY->RangeDesc(data, 3));
				if(Dlg->GetCheck(204)) Polygons[i]=new DataPolygon(this,CumData,TmpTxt+20,TmpTxt, rname);
				else Polygons[i] = new DataPolygon(this, CumData, TmpTxt, TmpTxt+20, rname);
				if(Polygons[i]) {
					Polygons[i]->Command(CMD_AUTOSCALE, 0L, 0L);
					Polygons[i]->Command(CMD_PG_FILL, GetSchemeFill(&sc), 0L);
					}
				if(rname) free(rname);
				delete rY;					rY = 0L;
				}
			}
		if(Bounds.Xmax >= Bounds.Xmin && Bounds.Ymax >= Bounds.Ymin) bRet = true;
		else bRet = false;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rd) {
		for (i = 0; i <= maxYR; i++) {
			if (rd[i]) free(rd[i]);
			}
		free(rd);
		}
	if(rX) delete rX;
	if(rY) delete rY;
	free(StackBarDlg);
	free(tab1);				free(tab2);				free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create a phenology plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *Phenology_DlgTmpl = (char*)
"1,2,,DEFAULT,PUSHBUTTON,-1,158,10,45,12\n"
"2,3,,,PUSHBUTTON,-2,158,25,45,12\n"
"3,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
"10,11,100,ISPARENT | CHECKED, SHEET,1,5,10,140,100\n"
"11,12,200,TOUCHEXIT | ISPARENT,SHEET,2,5,10,140,100\n"
"12,20,300,ISPARENT,SHEET,3,5,10,140,100\n"
"20,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
"100,101,,,LTEXT,4,15,30,60,8\n"
"101,152,,,RANGEINPUT,5,25,40,100,10\n"
"152,153,,ISPARENT | CHECKED,GROUPBOX,6,12,60,128,45\n"
"153,154,,,LTEXT,0,25,65,60,8\n"
"154,155,,,RANGEINPUT,5,25,75,100,10\n"
"155,156,,,PUSHBUTTON,-8,95,87,30,12\n"
"156,,,,PUSHBUTTON,-9,60,87,35,12\n"
"200,+,,CHECKED | HICOL,CHECKBOX,7,25,30,60,8\n"
".,.,250,ISPARENT | CHECKED,GROUP,,,,,\n"
".,.,,TOUCHEXIT | HICOL,CHECKBOX,11,25,75,60,8\n"
".,.,,ODEXIT,BARBUTT,18,95,75,30,10\n"
".,206,255,ISPARENT | CHECKED,GROUP,,,,,\n"
"206,+,,ODEXIT,SYMBUTT,12,95,30,30,10\n"
".,.,,ODEXIT,LINEBUTT,13,95,45,30,10\n"
".,,,ODEXIT,FILLBUTTON,17,95,60,30,10\n"
"250,+,,HICOL | ISRADIO,CHECKBOX,9,25,45,60,8\n"
".,,,HICOL | ISRADIO,CHECKBOX,10,25,60,60,8\n"
"255,,,HIDDEN | HICOL | ISRADIO,CHECKBOX,8,25,90,60,8\n"
"300,+,,,LTEXT,14,20,35,80,60\n"
".,.,,,INCDECVAL1,15,50,50,30,10\n"
".,.,,,LTEXT,-3,85,50,40,8\n"
".,,,LASTOBJ | HICOL | CHECKED,CHECKBOX,16,20,80,60,10";

bool
Phenology::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_LAYOUT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	Symbol *PrevSym = new Symbol(0L, data, 0.0, 0.0, 0);
	LineDEF Line, PG_FillLine = { defs.GetSize(SIZE_HAIRLINE), 6.0, 0x0, 0x0 };
	static FillDEF PG_Fill = { FILL_NONE, 0x00cbcbcbL, 1.0f, &PG_FillLine, 0x00cbcbcbL };
	double PlotInc;
	Bar *PrevBar = new Bar(0L, data, 0.0, 0.0, BAR_RELWIDTH | BAR_VERTB);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_PHEN_XRANG,
		(void*)TmpTxt, (void*)SDLG_PHEN_YRANG, (void*)SDLG_PLOTSCATT_DOSYM,
		(void*)SDLG_PLOTSCATT_DODROP, (void*)SDLG_PLOTSCATT_DOLIN, (void*)SDLG_PLOTSCATT_DOPG,
		(void*)SDLG_PLOTSCATT_DOVBAR, (void*)&PrevSym, (void*)&Line, (void*)SDLG_PHEN_PLOTINC,
		(void*)&PlotInc, (void*)SDLG_PHEN_INDAXES, (void*)&PG_Fill, (void*)&PrevBar};
	DlgInfo *PhenologyDlg;
	DlgRoot *Dlg;
	Symbol **Symbols = 0L;
	Bar **Bars = 0L;
	void *hDlg;
	int cb, j, n, res, currYR = 0, maxYR = 0, nx = 0, ny, c_num, c_txt, c_datetime;
	bool updateYR = true, bContinue = false, bRet = false, bVisited = false;
	char **rd = 0L;
	AccRange *rX = 0L, *rY = 0L;
	long xc, xr, yc, yr, nSym, nBar;
	double tmp, drec_max, grec_max, x, y;
	AxisDEF axis;
	Axis *the_new;
	TextDEF tlbdef, label_def;
	DataLine *TheLine = NULL;
	Label *label;


	if (!parent || parent->Id != GO_GRAPH || !data) return false;
	PlotInc = NiceValue((parent->GetSize(SIZE_DRECT_TOP) + parent->GetSize(SIZE_DRECT_BOTTOM)) / 3.0);
	drec_max = parent->GetSize(SIZE_DRECT_TOP);
	grec_max = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_GRECT_BOTTOM) - parent->GetSize(SIZE_DRECT_BOTTOM);
	tlbdef.ColTxt = defs.Color(COL_AXIS);					tlbdef.ColBg = 0x00ffffffL;
	tlbdef.RotBL = tlbdef.RotCHAR = 0.0;					tlbdef.iSize = 0;
	tlbdef.fSize = DefSize(SIZE_TICK_LABELS);				tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
	tlbdef.Style = TXS_NORMAL;								tlbdef.Mode = TXM_TRANSPARENT;
	tlbdef.Font = FONT_HELVETICA;							tlbdef.text = 0L;
	axis.breaks = NULL;				axis.nBreaks = 0;		axis.owner = NULL;
	axis.Center.fx = axis.Center.fy = axis.Radius = 0.0;
	axis.flags = AXIS_NEGTICKS | AXIS_AUTOTICK | AXIS_DEFRECT;
	memcpy(&Line, defs.GetLine(), sizeof(LineDEF));
	name = rlp_strdup((char*)"Phenology Plot");
	if (!UseRangeMark(data, 2, TmpTxt, TmpTxt + 100, TmpTxt + 200, TmpTxt + 300, TmpTxt + 400,
		TmpTxt + 500, TmpTxt + 600, TmpTxt + 700, TmpTxt + 800, TmpTxt + 900, TmpTxt + 1000)) return false;
	ssXrange = rlp_strdup(TmpTxt);
	if (!(PhenologyDlg = CompileDialog(Phenology_DlgTmpl, dyndata))) return false;
	if (TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(12, sizeof(char*)))) {
		for (i = 100, j = 0; i <= 1000; i += 100) if (TmpTxt[i])
			rd[j++] = rlp_strdup(TmpTxt + i);
		maxYR = j - 1;
		}
	if (!rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	if (!(Dlg = new DlgRoot(PhenologyDlg, data))) return false;
	if (rd && rd[currYR] && *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd((char*)SDLG_PHEN_HD1, 50, 50, 440, 280, Dlg, 0x4L);
	do {
		if (updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(156, 1);				Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);				Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			cb = sprintf_s(TmpTxt, TMP_TXT_SIZE, "y-values # %d/%d", currYR+1, maxYR+1);
#else
			cb = sprintf(TmpTxt,"y-values # %d/%d", currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		ny = 0;
		if (rX) delete rX;
		rX = 0L;
		switch (res) {
		case 0:
			if (bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:			bContinue = false;						break;
		case 1:
			if (!bVisited) {
				Dlg->SetCheck(11, 0L, 1);				bVisited = true;	 res = -1;		break;
			}
		//fall through
		case 155:		case 156:
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR, &rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		case 11:					//the layout tab must be visited
			bVisited = true;		res = -1;				break;
		case 202:							//create bars
			res = -1;				break;
		case 203:							//bar preview button
			Dlg->SetCheck(202, 0L, true);
			res = -1;				break;
		case 206:
			Dlg->SetCheck(200, 0L, true);
			res = -1; bContinue = false;	//symbol properties
			break;
		case 207:
			Dlg->SetCheck(250, 0L, true);
			res = -1; bContinue = false;	//line properties
			break;
		case 208:
			Dlg->SetCheck(251, 0L, true);
			res = -1; bContinue = false;	//polygon fill properties
			break;
			}
		} while (res < 0);
	if (res == 1 && nx && rX && rd && rd[0] && rd[0][0]){	//accept settings and create plot
		maxYR++;
		for (i = j = n = 0; i <= maxYR ; i++) {
			if (rd[i]) {
				if (n) TmpTxt[n++] = '|';
				n += rlp_strcpy(TmpTxt + n, 100, rd[i]);
				}
			}
		TmpTxt[n++] = '\0';		if (TmpTxt[0]) ssYrange = rlp_strdup(TmpTxt);
		Plots = (Plot**)calloc(maxYR + 2, sizeof(Plot*));		nPlots = 0;
		//get minima and maxima for x and y values first
		Bounds.Xmin = HUGE_VAL;		Bounds.Xmax = -HUGE_VAL;	Bounds.Ymin = HUGE_VAL;		Bounds.Ymax = -HUGE_VAL;
		if (Dlg->GetValue(301, &tmp) && tmp > PlotInc / 10.0 && tmp < PlotInc*10.0) PlotInc = tmp;
		rX->DataTypes(data, &c_num, &c_txt, &c_datetime);
		if (rY) delete rY;
		rY = 0L;
		for (i = 0; i < maxYR; i++){
			rY = new AccRange(rd[i]);	rY->GetFirst(&yc, &yr);
			rX->GetFirst(&xc, &xr);
			for (j = 0; j < nx; j++) {
				rX->GetNext(&xc, &xr);
				if (data->GetValue(xr, xc, &x)) {
					rY->GetNext(&yc, &yr);
					if (data->GetValue(yr, yc, &y)) {
						if (x > Bounds.Xmax) Bounds.Xmax = x;
						if (x < Bounds.Xmin) Bounds.Xmin = x;
						if (y > Bounds.Ymax) Bounds.Ymax = y;
						if (y < Bounds.Ymin) Bounds.Ymin = y;
						}
					}
				else rY->GetNext(&yr, &yc);
				}
			}
		((Graph*)parent)->SetSize(SIZE_BOUNDS_XMIN, Bounds.Xmin);		((Graph*)parent)->SetSize(SIZE_BOUNDS_XMAX, Bounds.Xmax);
		((Graph*)parent)->SetSize(SIZE_BOUNDS_YMIN, Bounds.Ymin);		((Graph*)parent)->SetSize(SIZE_BOUNDS_YMAX, Bounds.Ymax);
		x_info = rlp_strdup(rX->RangeDesc(data, 3));
		if(c_datetime > c_num/2)((Graph*)parent)->CreateAxes(21, (maxYR<<1) + 5);  //date / time axis ?
		else ((Graph*)parent)->CreateAxes(20, (maxYR << 1) + 5);
		((Graph*)parent)->Axes[0]->Command(CMD_SETTEXT, x_info, 0L);						//x-axis label
		((Graph*)parent)->Axes[1]->hidden = 1;												//hide master y-Axis
		//now create the plots
		if (rY) delete rY;
		rY = 0L;
		for (i = 0; i < maxYR; i++){
			if(Dlg->GetCheck(200)) Symbols = (Symbol**)calloc(nx + 2, sizeof(Symbol*));		//create symbols
			else Symbols = NULL;
			nSym = 0;
			if (Dlg->GetCheck(202)) Bars = (Bar**)calloc(nx + 2, sizeof(Bar*));				//create bars
			else Bars = NULL;
			nBar = 0;
			if (Dlg->GetCheck(250)) {
				TheLine = new DataLine(this, data, ssXrange, rd[i]);	//create a data line
				TheLine->Command(CMD_SET_LINE, (void*)&Line, 0L);
				}
			else if (Dlg->GetCheck(251)) {								//create a data polygon
				TheLine = new DataPolygon(this, data, ssXrange, rd[i]);
				TheLine->Command(CMD_SET_LINE, (void*)&Line, 0L);
				TheLine->Command(CMD_PG_FILL, (void*)&PG_Fill, 0L);
				TheLine->SetSize(SIZE_DATA_LINE, PG_FillLine.width);
				TheLine->type |= 0x100;									//epand polygons to baseline
				}
			rY = new AccRange(rd[i]);	
			rY->GetFirst(&yc, &yr);			rX->GetFirst(&xc, &xr);
			for (j = 0; j < nx; j++) {
				rX->GetNext(&xc, &xr);
				if (data->GetValue(xr, xc, &x)) {
					rY->GetNext(&yc, &yr);
					if (Symbols && data->GetValue(yr, yc, &y)) {
						Symbols[nSym] = new Symbol(this, data, x, y, SYM_CIRCLE, xc, xr, yc, yr);
						Symbols[nSym]->Command(CMD_PREVSYM, PrevSym, 0L);
						nSym++;
						}
					if (Bars && data->GetValue(yr, yc, &y)){
						Bars[nBar] = new Bar(this, data, x, y, BAR_RELWIDTH | BAR_VERTB, xc, xr, yc, yr);
						Bars[nBar]->Command(CMD_PREVSYM, PrevBar, 0L);
						nBar++;
						}
					}
				else {
					rY->GetNext(&yr, &yc);			//no valid x
					}
				}
			if (!nSym && Symbols) {
					free(Symbols);		Symbols = 0L;
					}
			if ((nSym && Symbols) || (nBar && Bars) || TheLine) {					//a valid plot needs his own axes
					Plots[nPlots] = new PlotScatt(this, data, nSym, Symbols, TheLine, nBar, Bars);
#ifdef USE_WIN_SECURE
					cb = sprintf_s(TmpTxt, 40, "Plot %s vs. ", rY->RangeDesc(data, 3));
#else
					cb = sprintf(TmpTxt, "Plot %s vs. ", rY->RangeDesc(data, 3));
#endif
					rlp_strcpy(TmpTxt + cb, 40-cb, rX->RangeDesc(data, 3));
					Plots[nPlots]->name = rlp_strdup(TmpTxt);
					if (i < (maxYR -1) && Dlg->GetCheck(303)) {						//create x-axes for every plot
						axis.loc[0].fy = axis.loc[1].fy = parent->GetSize(SIZE_GRECT_TOP) +
							parent->GetSize(SIZE_DRECT_TOP) + (PlotInc * (double)(i+1)) - PlotInc*0.2;
						axis.loc[0].fx = parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_DRECT_LEFT);
						axis.loc[1].fx = parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_DRECT_RIGHT);
						axis.loc[0].fz = axis.loc[1].fz = 0.0;
						axis.min = Bounds.Xmin;			axis.max = Bounds.Xmax;
						NiceAxis(&axis, 4);
						the_new = new Axis(parent, data, &axis, AXIS_MINORTICK | AXIS_NEGTICKS | AXIS_AUTOTICK | AXIS_DEFRECT);
						if (the_new){
							tlbdef.Align = TXA_VTOP | TXA_HCENTER;
							the_new->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
							the_new->SetSize(SIZE_TLB_YDIST, ((Graph*)parent)->Axes[0]->GetSize(SIZE_TLB_YDIST));
#ifdef USE_WIN_SECURE
							sprintf_s(TmpTxt, 40, "%s: X-axis", rY->RangeDesc(data, 3));
#else
							sprintf(TmpTxt, "%s: X-axis", rY->RangeDesc(data, 3));
#endif
							the_new->name = rlp_strdup(TmpTxt);
							((Graph*)parent)->Axes[((Graph*)parent)->NumAxes] = the_new;
							Plots[nPlots]->use_yaxis = ((Graph*)parent)->NumAxes;
							((Graph*)parent)->NumAxes++;
							}
						}
					tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
					axis.loc[0].fx = axis.loc[1].fx = parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_DRECT_LEFT);
					drec_max += PlotInc;			grec_max += PlotInc;
					axis.loc[0].fy = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_DRECT_TOP);
					axis.loc[0].fy += (PlotInc * (double)i);
					axis.loc[1].fy = axis.loc[0].fy + PlotInc * 0.8;
					axis.loc[0].fz = axis.loc[1].fz = 0.0;
					axis.min = Bounds.Ymin;			axis.max = Bounds.Ymax;
					NiceAxis(&axis, 4);
					the_new = new Axis(parent, data, &axis, axis.flags);
					if (the_new){
						the_new->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
						the_new->Command(CMD_PREVSYM, ((Graph*)parent)->Axes[1], 0L);		//Axis[1] is the master y-axis
#ifdef USE_WIN_SECURE
						sprintf_s(TmpTxt, 40, "%s: Y-axis", rY->RangeDesc(data, 3));
#else
						sprintf(TmpTxt, "%s: Y-axis", rY->RangeDesc(data, 3));
#endif
						label_def.ColTxt = defs.Color(COL_AXIS);				label_def.ColBg = 0x00ffffffL;
						label_def.fSize = DefSize(SIZE_TICK_LABELS)*1.2;		label_def.RotBL = label_def.RotCHAR = 0.0;
						label_def.iSize = 0;									label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
						label_def.Mode = TXM_TRANSPARENT;						label_def.Style = TXS_NORMAL;
						label_def.Font = FONT_HELVETICA;						label_def.RotBL = 90.0;
						label_def.text = rlp_strdup((unsigned char*)rY->RangeDesc(data, 3));
						label = new Label(the_new, data, axis.loc[0].fx - DefSize(SIZE_AXIS_TICKS)*6.0,
							(axis.loc[0].fy +axis.loc[1].fy) / 2.0, &label_def, LB_X_PARENT, 0L);
						if (label && the_new->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
						else if (label) DeleteGO(label);
						the_new->name = rlp_strdup(TmpTxt);
						((Graph*)parent)->Axes[((Graph*)parent)->NumAxes] = the_new;
						Plots[nPlots]->use_yaxis = ((Graph*)parent)->NumAxes;
						((Graph*)parent)->NumAxes++;
						}
					nPlots++;		bRet = true;
					}
				delete rY;		rY = 0L;
				}
		}
	if (bRet) {
		parent->SetSize(SIZE_DRECT_BOTTOM, drec_max);				parent->SetSize(SIZE_GRECT_BOTTOM, grec_max);
		((Graph*)parent)->Axes[0]->axis->flags &= ~AXIS_BOTTOM;		((Graph*)parent)->Axes[1]->axis->flags &= ~AXIS_LEFT;
		((Graph*)parent)->Axes[0]->axis->loc[0].fy = ((Graph*)parent)->Axes[0]->axis->loc[1].fy = drec_max - PlotInc * 0.2;
		((Graph*)parent)->Axes[1]->axis->loc[1].fy = drec_max;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if (rd) {
		for (i = 0; i < maxYR; i++)	if (rd[i]) free(rd[i]);
		free(rd);
		}
	if (rX) delete rX;
	if (rY) delete rY;
	free(PhenologyDlg);
	delete PrevSym;			PrevSym = 0L;
	free(tab1);			free(tab2);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create grouped bars chart
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *GBDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,140,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,140,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,99,ISPARENT | CHECKED | TOUCHEXIT,SHEET,1,5,10,128,100\n"
	".,.,200,ISPARENT,SHEET,2,5,10,128,100\n"
	".,.,300,ISPARENT,SHEET,3,5,10,128,100\n"
	".,10,250,ISPARENT | TOUCHEXIT,SHEET,16,5,10,128,100\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"99,100,110,ISPARENT | CHECKED,GROUP,,,,,\n"
	"100,,160,ISPARENT | CHECKED,GROUP,,,,,\n"
	"101,+,,,LTEXT,0,25,39,60,8\n"
	".,.,,,RANGEINPUT,-15,25,49,100,10\n"
	".,.,,,LTEXT,0,25,61,60,8\n"
	".,.,,,RANGEINPUT,-16,25,71,100,10\n"
	".,.,,,PUSHBUTTON,-8,95,87,30,12\n"
	".,,,,PUSHBUTTON,-9,60,87,35,12\n"
	"110,+,,,LTEXT,4,10,25,60,8\n"
	".,150,,,LTEXT,5,10,33,60,8\n"
	"150,,153,ISPARENT | CHECKED,GROUPBOX,6,10,50,118,55\n"
	"160,101,,ISPARENT | CHECKED,GROUPBOX,19,10,30,118,75\n"
	"153,+,,,LTEXT,0,15,60,60,8\n"
	".,.,,,RANGEINPUT,-1,15,70,108,10\n"
	".,.,,,PUSHBUTTON,-8,93,87,30,12\n"
	".,,,,PUSHBUTTON,-9,58,87,35,12\n"
	"200,+,,,RTEXT,7,10,35,38,8\n"
	".,.,,,EDVAL1,8,58,35,35,10\n"
	".,.,,,RTEXT,9,10,50,38,8\n"
	".,.,,,EDVAL1,10,58,50,35,10\n"
	".,.,,,RTEXT,11,10,65,38,8\n"
	".,.,,,EDVAL1,12,58,65,35,10\n"
	".,.,,,LTEXT,-10,95,65,8,8\n"
	".,.,,,RTEXT,13,10,80,38,8\n"
	".,.,,,EDVAL1,14,58,80,35,10\n"
	".,,,,LTEXT,-10,95,80,8,8\n"
	"250,+,, TOUCHEXIT | CHECKED | HICOL,RADIO1,17,20,35,70,8\n"
	".,.,,TOUCHEXIT | HICOL,RADIO1,18,20,50,70,8\n"
	".,,,,RANGEINPUT,0,15,65,108,10\n"
	"300,,,LASTOBJ | NOSELECT,ODB,15,24,30,90,60";

bool
GroupBars::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	TabSHEET *tab4 = MakeTab(i, 10, &i, (char*)SDLG_TAB_LABEL);
	double start = 1.0, step = 1.0, bw = 100.0, gg = 100.0;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_GBAR_VALUES,
		(void*)SDLG_GBAR_RANGES, (void*)SDLG_GBAR_YVALS, (void*)SDLG_GBAR_START,
		(void*)&start, (void*)SDLG_GBAR_GSTEP, (void*)&step, (void*)SDLG_GBAR_BWIDTH, (void*)&bw,
		(void*)SDLG_GBAR_GGAP, (void*)&gg, (void*)(OD_scheme), (void*)tab4, (void*)SDLG_GBAR_GLABEL,
		(void*)SDLG_GBAR_SSRNG, (void*)SDLG_GBAR_YERNG };
	DlgInfo *GBDlg;
	DlgRoot *Dlg;
	void *hDlg;
	bool bRet = false, updateYR = true, bContinue = false, bError;
	char **rd=0L, **rdx=0L, **rdy=0L, *desc;
	Bar **bars = 0L;
	ErrorBar **errs = 0L;
	AccRange *rX = 0L, *rY = 0L;
	anyResult ares;
	int j, ic, res, ny, s1 = 1, s2 = 1, sc = 0, currYR = 0, maxYR = 0;
	long ix, iy, ex, ey;
	double x, y, xinc, e, ebw;
	
	if(!parent || !data) return false;
	if(!(UseRangeMark(data, 2, TmpTxt, TmpTxt+100, TmpTxt+200, TmpTxt+300, TmpTxt+400,
		TmpTxt+500, TmpTxt+600, TmpTxt+700, TmpTxt+800, TmpTxt+900, TmpTxt+1000))) return false;
	if(!(GBDlg = CompileDialog(GBDlg_Tmpl, dyndata)))return false;
	if(mode == 0 && TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(12, sizeof(char*)))) {
		for (i = 0, j = 0; i <= 1000; i += 100) {
			if (TmpTxt[i]) rd[j++] = rlp_strdup(TmpTxt + i);
			}
			maxYR = j-1;
		}
	else if(mode == 1 && TmpTxt[0] && TmpTxt[100] && (rdx = (char**)calloc(12, sizeof(char*))) 
		&& (rdy = (char**)calloc(12, sizeof(char*)))) {
		for(i=j=0; i <= 1000; i +=200) if(TmpTxt[i]) {
			rdx[j] = rlp_strdup(TmpTxt+i);			rdy[j] = rlp_strdup(TmpTxt+i+100);
			maxYR = j++;
			}
		}
	Id = GO_STACKBAR;
	if(mode == 0 && !rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	if(mode == 1 && !rdx && !rdy && !(rdx = (char**)calloc(1, sizeof(char*))) 
		&& !(rdy = (char**)calloc(1, sizeof(char*))))return false;
	if(!(Dlg = new DlgRoot(GBDlg, data)))return false;
	if(mode == 1) {
		Dlg->ShowItem(99, false);		Dlg->ShowItem(100, true);
		}
	else {
		Dlg->ShowItem(99, true);		Dlg->ShowItem(100, false);
		}
	if(rd && rd[currYR] &&  *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	else Dlg->SetText(154, (unsigned char*)"");
	hDlg = CreateDlgWnd(mode == 0 ? (char*)SDLG_GBAR_HD1 : (char*)SDLG_GBAR_HD2, 50, 50, 410, 280, Dlg, 0x4L);
	do {
		if(updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(156, true);		Dlg->ShowItem(106, true);
				}
			else {
				Dlg->ShowItem(156, false);		Dlg->ShowItem(106, false);		
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, 20, "y-values # %d/%d", currYR+1, maxYR+1);
			sprintf_s(TmpTxt+100, 20, "errors # %d/%d", currYR+1, maxYR+1);
#else
			sprintf(TmpTxt,"y-values # %d/%d", currYR+1, maxYR+1);
			sprintf(TmpTxt+100,"errors # %d/%d", currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			if(mode == 0) 	Dlg->SetText(153, (unsigned char*)TmpTxt);
			else {
				Dlg->SetText(101, (unsigned char*)TmpTxt);		Dlg->SetText(103, (unsigned char*)(TmpTxt+100));
				}
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue || Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 250:		case 251:
		case 7:
			Dlg->Activate(252, 1);
			res = -1;						break;
		case 4:
			Dlg->Activate(mode ? 102:154, 1);
			res = -1;						break;
		case 1:								//OK button
		case 105:							//next button
			if (res == 1){
				Dlg->GetValue(201, &start);			Dlg->GetValue(203, &step);
				Dlg->GetValue(205, &bw);			Dlg->GetValue(208, &gg);
				res = com_StackDlg(res, Dlg, 0L, 0L, &rd, &currYR,
					&rY, &bContinue, &ny, &maxYR, &updateYR);
				if (!mode) break;
				}
			bError=false;	s1 = s2 = 0;
			if(Dlg->GetText(102, (unsigned char*)TmpTxt, 100)) {
				if((rX = new AccRange(TmpTxt))) {
					s1 = rX->CountItems();
					if(s1 < 2) {
						ErrorBox((char*)"y-range not valid");
						bContinue=bError=true;
						Dlg->Activate(102, 1);
						}
					delete rX;
					}
				else bError = true;
				}
			else bError = true;
			if((Dlg->GetText(104, (unsigned char*)(TmpTxt+100), 100)) && !bError) {
				if((rY = new AccRange(TmpTxt+100))) {
					s2 = rY->CountItems();
					if(s2 < 2) {
						ErrorBox((char*)SCMS_BAD_ERROR_RANGE);
						bContinue=bError=true;
						Dlg->Activate(104, 1);
						}
					delete rY;
					}
				else bError = true;
				}
			else {
				Dlg->Activate(104, 1);
				bError = true;
				}
			if(!s1 || !s2) bError = true;
			rX = rY = 0L;
			if(!bError && s1!=s2) {
				ErrorBox((char*)SCMS_YE_RNG_DIFFER);
				bContinue=bError=true;
				}
			if(!bError) {
				if((currYR+1) > maxYR) {
					rdx = (char**)realloc(rdx, sizeof(char*)*(currYR+2));
					rdy = (char**)realloc(rdy, sizeof(char*)*(currYR+2));
					rdx[currYR] = rdx[currYR+1] = rdy[currYR] = rdy[currYR+1] = 0L;
					maxYR = currYR+1;
					}
				if(rdx[currYR]) free(rdx[currYR]);		//store x-range
				rdx[currYR] = rlp_strdup(TmpTxt);
				if(rdy[currYR]) free(rdy[currYR]);		//store y range
				rdy[currYR] = rlp_strdup(TmpTxt+100);
				updateYR = true;						currYR++;
				Dlg->SetText(102, (unsigned char*)rdx[currYR]);			Dlg->SetText(104, (unsigned char*)rdy[currYR]);
				Dlg->Activate(102, 1);				
				if(res != 1) res = -1;
				}
			else if(res != 1){
				bContinue = true;
				res = -1;
				}
			break;
		case 106:										//prev button
			if(Dlg->GetText(102, (unsigned char*)TmpTxt, 100) && Dlg->GetText(104, (unsigned char*)(TmpTxt+100), 100)){
				if(rdx[currYR]) free(rdx[currYR]);
				if(rdy[currYR]) free(rdy[currYR]);
				rdx[currYR] = rlp_strdup(TmpTxt);
				rdy[currYR] = rlp_strdup(TmpTxt+100);
				}
			else if(currYR == maxYR) maxYR--;
			currYR--;	
			Dlg->SetText(102, (unsigned char*)rdx[currYR]);
			Dlg->SetText(104, (unsigned char*)rdy[currYR]);				Dlg->Activate(102, 1);
			updateYR = true;
			res = -1;
			break;
		case 155:		case 156:
			res = com_StackDlg(res, Dlg, 0L, 0L, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
			}
		} while(res < 0);
	if(res == 1 && ((rd && rd[0] && rd[0][0])||(rdx && rdy && rdx[0] && rdy[0] && rdx[0][0] && rdy[0][0]))){
		//accept settings and create plots
		if((mode == 0 && rd[maxYR]) || (mode == 1 && rdx[maxYR])) maxYR++;
		ebw = defs.GetSize(SIZE_ERRBAR)*9.0 / ((double)(s1*maxYR));
		if(Dlg->GetCheck(251) && Dlg->GetText(252, (unsigned char*)TmpTxt, 100) && (rX = new AccRange(TmpTxt))
			&& rX->GetFirst(&ix, &iy)){
			x_tv = new TextValue();				rX->GetNext(&ix, &iy);
			i = 1;								start = step = 1.0;
			do {
				if(data->GetResult(&ares, iy, ix, false)) switch(ares.type) {
					case ET_TEXT:
						x_tv->GetValue(ares.text);			break;
					case ET_VALUE:		case ET_DATE:		case ET_TIME:
					case ET_DATETIME:	case ET_BOOL:
						TranslateResult(&ares);
						x_tv->GetValue(ares.text);			break;
					}
				i++;
				}while(rX->GetNext(&ix, &iy));
			}
		if(rX) delete rX;
		rX = 0L;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		if(!(xyPlots = (PlotScatt**)calloc(maxYR+2, sizeof(PlotScatt*)))){
			CloseDlgWnd(hDlg);
			delete Dlg;
			if(rd) {
				for (i = 0; i < maxYR; i++)	if(rd[i]) free(rd[i]);
				free(rd);
				}
			return false;
			}
		xinc = fabs(step / ((double)maxYR + gg/100.0));
		start -= (xinc * ((double)maxYR-1))/2.0;
		for(i = 0, desc = 0L; i < maxYR; i++) {
			x = start + xinc * (double)i;
			if(mode == 0 && rd) {
				if(rd[i] && (rY = new AccRange(rd[i]))) ny = rY->CountItems();
				else {
					rY = 0L;	ny = 0;
					}
				}
			else if(mode == 1 && rdx && rdy) {
				if(rdx[i] && (rY = new AccRange(rdx[i]))) ny = rY->CountItems();
				else {
					rY = 0L;	ny = 0;
					}
				}
			if(rY) {
				desc = rY->RangeDesc(data, 1);
				rY->GetFirst(&ix, &iy);
				if(mode == 1 && (rX = new AccRange(rdy[i]))) {
					errs = (ErrorBar**)calloc(ny+2, sizeof(ErrorBar*)); 
					rX->GetFirst(&ex, &ey);
					}
				rY->GetNext(&ix, &iy);
				if(ny && rY && (bars = (Bar **)calloc(ny+2, sizeof(Bar*)))){
					for(ic = 0; ic < ny; ic++) {
						if((bContinue = data->GetValue(iy, ix, &y))){
							bars[ic] = new Bar(0L, data, x, y, BAR_VERTB | BAR_RELWIDTH,
								-1, -1, ix, iy, desc);
							CheckBounds(x, y);
							}
						if(errs && rX && (rX->GetNext(&ex, &ey)) && (data->GetValue(ey, ex, &e))) {
							if(bContinue && (errs[ic] = new ErrorBar(0L, data, x, y, e, 0, -1, -1, ix, iy, ex, ey)))
								errs[ic]->SetSize(SIZE_ERRBAR, ebw);
							CheckBounds(x, y+e);	CheckBounds(x, y-e);
							}
						x += step;
						rY->GetNext(&ix, &iy);
						}
					xyPlots[numXY++] = new PlotScatt(this, data, ic, bars, errs);
					if(xyPlots[i]) {
						xyPlots[i]->SetSize(SIZE_BARMINX, xinc);
						xyPlots[i]->SetSize(SIZE_BAR, bw);
						xyPlots[i]->Command(CMD_BAR_FILL, GetSchemeFill(&sc), 0L);
						}
					for(ic = 0; ic < ny; ic++) if(bars[ic]) delete(bars[ic]);
					if(errs) for(ic = 0; ic < ny; ic++) if(errs[ic]) delete(errs[ic]);
					free(bars);		if(errs) free(errs);
					bRet = true;
					}
				delete(rY);		rY = 0L;
				}
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rd) {
		for (i = 0; i < maxYR; i++)	if(rd[i]) free(rd[i]);
		free(rd);
		}
	if(rdx) {
		for (i = 0; i < maxYR; i++)	if(rdx[i]) free(rdx[i]);
		free(rdx);
		}
	if(rdy) {
		for (i = 0; i < maxYR; i++)	if(rdy[i]) free(rdy[i]);
		free(rdy);
		}
	if(rY) delete rY;
	if(rX) delete rX;
	free(GBDlg);	free(tab1);		free(tab2);		free(tab3);		free(tab4);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create a waterfall graph
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

static char *WaterfDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,158,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,158,25,45,12\n"
	"3,,10,ISPARENT | CHECKED,GROUP,0,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,140,100\n"
	".,.,200,ISPARENT | TOUCHEXIT,SHEET,2,5,10,140,100\n"
	".,20,300,ISPARENT,SHEET,3,5,10,140,100\n"
	"20,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,15,30,60,8\n"
	".,152,,,RANGEINPUT,-15,25,40,100,10\n"
	"152,+,,ISPARENT | CHECKED,GROUPBOX,4,12,60,128,45\n"
	".,.,,,LTEXT,0,25,65,60,8\n"
	".,.,,,RANGEINPUT,-15,25,75,100,10\n"
	".,.,,,PUSHBUTTON,-8,95,87,30,12\n"
	".,,,,PUSHBUTTON,-9,60,87,35,12\n"
	"200,+,,,LTEXT,5,20,35,80,8\n"
	".,.,,,RTEXT,6,28,45,13,8\n"
	".,.,,,EDVAL1,7,43,45,25,10\n"
	".,.,,,RTEXT,8,73,45,13,8\n"
	".,.,,,EDVAL1,9,88,45,25,10\n"
	".,210,,,LTEXT,-3,115,45,15,8\n"
	"210,220,,HICOL | CHECKED,CHECKBOX,10,25,65,20,8\n"
	"220,,,HICOL,CHECKBOX,11,25,80,20,8\n"
	"300,+,,HICOL | CHECKED, RADIO1,12,20,35,80,9\n"
	".,.,,ODEXIT,COLBUTT,13,105,35,20,10\n"
	".,.,,HICOL,RADIO1,14,20,55,80,9\n"
	".,.,,ODEXIT,COLBUTT,15,25,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,16,37,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,17,49,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,18,61,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,19,73,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,20,85,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,21,97,70,10,10\n"
	".,,,ODEXIT,COLBUTT,22,109,70,10,10\n"
	"500,,,LASTOBJ,NONE,0,,,,";

bool
Waterfall::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	static DWORD colarr[] = {0x00000000L, 0x00ff0000L, 0x0000ff00L, 0x000000ffL,
		0x00ff00ff, 0x00ffff00L, 0x0000ffff, 0x00c0c0c0};
	static DWORD defcol = 0x0L;
	unsigned char text1[100];
	DlgInfo *StackBarDlg;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_WFPLOT_YRNGS,
		(void*)SDLG_WFPLOT_XYDISP, (void*)SDLG_WFPLOT_DX, (void*)&dspm.fx, (void*)SDLG_WFPLOT_DY,
		(void*)&dspm.fy, (void*)SDLG_WFPLOT_DLINES, (void*)SDLG_WFPLOT_DBARS, (void*)SDLG_WFPLOT_COL_ALL,
		(void*)&defcol, (void*)SDLG_WFPLOT_COL_INC, (void*)&colarr[0], (void*)&colarr[1], (void*)&colarr[2],
		(void*)&colarr[3], (void*)&colarr[4], (void*)&colarr[5], (void*)&colarr[6], (void*)&colarr[7]};
	DlgRoot *Dlg;
	void *hDlg;
	int j, res, currYR=0, maxYR=0, nx=0, ny;
	long ic, xc, xr, yc, yr;
	char **rd = 0L;
	double x,y;
	bool updateYR = true, bContinue = false, bRet = false, bUseSch = true;
	bool bVisited = false;
	AccRange *rX = 0L, *rY = 0L;
	Bar **Bars;

	if(!parent || !data) return false;
	if(!(UseRangeMark(data, 2, TmpTxt, TmpTxt+100, TmpTxt+200, TmpTxt+300, TmpTxt+400,
		TmpTxt+500, TmpTxt+600, TmpTxt+700, TmpTxt+800, TmpTxt+900, TmpTxt+1000))) return false;
	if(TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(12, sizeof(char*)))) {
		for (i = 100, j = 0; i <= 1000; i += 100) {
			if (TmpTxt[i]) rd[j++] = rlp_strdup(TmpTxt + i);
			}
			maxYR = j-1;
		}
	if(!rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	switch(defs.units()) {
		case 1:		dspm.fx = 0.5;		break;
		case 2:		dspm.fx = 0.2;		break;
		default:	dspm.fx = 5.0;		break;
		}
	dspm.fy = -dspm.fx;
	if (!(StackBarDlg = CompileDialog(WaterfDlg_Tmpl, dyndata)))return false;
	if(!(Dlg = new DlgRoot(StackBarDlg, data))) return false;
	if(rd && rd[currYR] &&  *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd((char*)SDLG_WFPLOT_HD1, 50, 50, 440, 280, Dlg, 0x4L);
	do {
		if(updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(156, 1);				Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);				Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, 20, SDLG_WFPLOT_YFMT, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt,SDLG_WFPLOT_YFMT, currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		ny = 0;
		if(rX) delete rX;
		rX = 0L;
		switch(res) {
		case 0:
			if(bContinue || Dlg->GetCheck(20)) res = -1;
			if(!bVisited) res = -1;
			break;
		case -1:
			bContinue = false;			break;
		case 1:
			if(!bVisited) {
				Dlg->SetCheck(11, 0L, true);
				res = -1;				bVisited =true;
				break;
				}
			for(i = 0; i < 8; i++) Dlg->GetColor(303+i, &colarr[i]);
			Dlg->GetColor(301, &defcol);
			bUseSch = Dlg->GetCheck(302);
			Dlg->GetValue(202, &dspm.fx);	Dlg->GetValue(204, &dspm.fy);
			//execute com_StackDlg for <OK>
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		case 155:		case 156:
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			break;
		case 11:
			bVisited = true;			res = -1;
			break;
		case 301:
			Dlg->SetCheck(300, 0L, true);
			res = -1;	break;
		case 303:	case 304:	case 305:	case 306:
		case 307:	case 308:	case 309:	case 310:
			Dlg->SetCheck(302, 0L, true);
			res = -1;	break;
			}
		}while (res < 0);
	if(res == 1 && nx && rX && rd && rd[0] && rd[0][0]) {
		maxYR++;
		numPL = maxYR;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		Dlg->GetText(101, text1, 100);
		if(Dlg->GetCheck(210) && (Lines=(DataLine**)calloc(numPL+2,sizeof(DataLine*))))for(i=0;i<maxYR;i++){
			if(rd[i] && rd[i][0]) Lines[i] = new DataLine(this, data, (char*)text1, rd[i]);
			if(Lines[i]) {
				if(bUseSch) Lines[i]->SetColor(COL_DATA_LINE, colarr[(i & 0x07)]);
				else Lines[i]->SetColor(COL_DATA_LINE, defcol);
				//give lines a name
				if(rY) Lines[i]->name = rlp_strdup(rY->RangeDesc(data, 0));
				bRet = true;
				}
			}
		if(Dlg->GetCheck(220) && (xyPlots=(PlotScatt**)calloc(numPL+2, sizeof(PlotScatt*))))for(i=numXY=0; i<maxYR; i++){
			if(rX) delete rX;
			if(rY) delete rY;
			rX = new AccRange((char*)text1);		rY = new AccRange(rd[i]);
			ic = rY->CountItems();					Bars = (Bar**)calloc(ic+2, sizeof(Bar*));
			rX->GetFirst(&xc, &xr);					rY->GetFirst(&yc, &yr);
			rX->GetNext(&xc, &xr);					rY->GetNext(&yc, &yr);
			ic = 0;
			do {
				if(data->GetValue(xr, xc, &x) && data->GetValue(yr, yc, &y)) {
					Bars[ic++] = new Bar(this, data, x, y, BAR_VERTB | BAR_RELWIDTH, xc, xr, yc, yr);
					}
				} while(rX->GetNext(&xc, &xr) && rY->GetNext(&yc, &yr));
			if(ic) {
				xyPlots[numXY] = new PlotScatt(this, data, ic, Bars, 0L);
				if(bUseSch) xyPlots[numXY]->SetColor(COL_BAR_LINE, colarr[(i & 0x07)]);
				else xyPlots[numXY]->SetColor(COL_BAR_LINE, defcol);
				xyPlots[numXY]->SetSize(SIZE_BAR, 30.0);
				//give the plots a name
				if(xyPlots[numXY]->name) free(xyPlots[numXY]->name);
				xyPlots[numXY]->name = (char*)malloc(40 *sizeof(char));
#ifdef USE_WIN_SECURE
				sprintf_s(xyPlots[numXY]->name, 18 , "Bar Chart %ld", numXY+1);
#else
				sprintf(xyPlots[numXY]->name, "Bar Chart %ld", numXY+1);
#endif
				xyPlots[numXY]->data_desc = rlp_strdup(rY->RangeDesc(data, 3));
				numXY++;
				}
			free(Bars);						bRet = true;
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rd) {
		for (i = 0; i < maxYR; i++){
			if (rd[i]) free(rd[i]);
			}
		free(rd);
		}
	if(rX) delete rX;
	if(rY) delete rY;
	if(bRet) Command(CMD_AUTOSCALE, 0L, 0L);
	free(StackBarDlg);
	free(tab1);			free(tab2);				free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create a multi data line plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *MultiLineDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,158,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,158,25,45,12\n"
	".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,140,100\n"
	".,.,300,ISPARENT,SHEET,2,5,10,140,100\n"
	".,20,200,ISPARENT,SHEET,15,5,10,140,100\n"
	"20,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,ISPARENT | CHECKED,GROUPBOX,3,12,30,128,75\n"
	".,.,,,LTEXT,0,25,39,60,8\n"
	".,.,,,RANGEINPUT,-15,25,49,100,10\n"
	".,.,,,LTEXT,0,25,61,60,8\n"
	".,.,,,RANGEINPUT,-16,25,71,100,10\n"
	".,.,,,PUSHBUTTON,-8,95,87,30,12\n"
	".,.,,,PUSHBUTTON,-9,60,87,35,12\n"
	".,.,,OWNDIALOG,COLBUTT,4,25,87,20,12\n"
	".,,,,LTEXT,12,47,90,30,9\n"
	"200,+,,CHECKED | HICOL,CHECKBOX,16,25,35,80,9\n"
	".,.,,ODEXIT,SYMBUTT,17,25,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,18,37,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,19,49,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,20,61,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,21,73,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,22,85,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,23,97,70,10,10\n"
	".,.,,ODEXIT,SYMBUTT,24,109,70,10,10\n"
	".,,,CHECKED | HICOL,CHECKBOX,25,25,50,80,9\n"
	"300,+,,TOUCHEXIT | HICOL,RADIO1,13,20,35,80,9\n"
	".,.,,ODEXIT,COLBUTT,4,105,35,20,10\n"
	".,.,,CHECKED | TOUCHEXIT | HICOL, RADIO1,14,20,55,80,9\n"
	".,.,,ODEXIT,COLBUTT,4,25,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,5,37,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,6,49,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,7,61,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,8,73,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,9,85,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,10,97,70,10,10\n"
	".,,,LASTOBJ | ODEXIT,COLBUTT,11,109,70,10,10\n";

bool
MultiLines::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SYMBOLS);
	static DWORD colarr[] = {0x00000000L, 0x00ff0000L, 0x0000ff00L, 0x000000ffL,
		0x00ff00ff, 0x00ffff00L, 0x0000ffff, 0x00c0c0c0};
	Symbol *syms[8], **lsyms;
	static DWORD defcol = 0x0L;
	unsigned char x_txt[100], y_txt[100];
	DlgInfo *StackBarDlg;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_MLPLOT_XYRNG,
		(void*)&colarr[0], (void*)&colarr[1], (void*)&colarr[2], (void*)&colarr[3],
		(void*)&colarr[4], (void*)&colarr[5], (void*)&colarr[6], (void*)&colarr[7],
		(void*)SDLG_MLPLOT_LICOL, (void*)SDLG_MLPLOT_COMCOL, (void*)SDLG_MLPLOT_INCCOL,
		(void*)tab3, (void*)SDLG_MLPLOT_DOSYM, (void*)&syms[0], (void*)&syms[1], (void*)&syms[2],
		(void*)&syms[3], (void*)&syms[4], (void*)&syms[5], (void*)&syms[6], (void*)&syms[7],
		(void*)SDLG_MLPLOT_COLSYM };
	DlgRoot *Dlg;
	void *hDlg;
	unsigned char **rdx=0L, **rdy=0L;
	DWORD *rdc = 0L, curr_col;
	int j, nd, res, currYR = 0, maxYR = 0, s1, s2;
	long rx, ry, cx, cy;
	double symsize, x, y;
	bool updateYR = true, bContinue = false, bError, bRet = false;
	AccRange *rX = 0L, *rY = 0L;
	DataLine *dl;

	if(!parent || !data) return false;
	symsize = NiceValue(defs.GetSize(SIZE_SYMBOL)*.8);
	for (i = 0; i < 8; i++) {
		syms[i] = new Symbol(0L, data, 0.0, 0.0, i);
		if (syms[i]) {
			if (i != 2 && i != 3)syms[i]->SetSize(SIZE_SYMBOL, symsize);
			}
		}
	if(!(StackBarDlg = CompileDialog(MultiLineDlg_Tmpl, dyndata)))return false;
	if(!UseRangeMark(data, 2, TmpTxt, TmpTxt+100, TmpTxt+200, TmpTxt+300, TmpTxt+400,
		TmpTxt+500, TmpTxt+600, TmpTxt+700, TmpTxt+800, TmpTxt+900, TmpTxt+1000)) return false;
	if(TmpTxt[0] && TmpTxt[100] && (rdx = (unsigned char**)calloc(12, sizeof(unsigned char*))) 
		&& (rdy = (unsigned char**)calloc(12, sizeof(unsigned char*))) && (rdc = (DWORD*)malloc(12*sizeof(DWORD)))) {
		for(i=100, j= 0; i <= 1000; i +=100) if(TmpTxt[i]) {
			rdx[j] = rlp_strdup((unsigned char*)TmpTxt);			rdc[j] = colarr[j%8];	
			rdy[j] = rlp_strdup(((unsigned char*)TmpTxt)+i);		maxYR = j++;
			}
		}
	if(!(Dlg = new DlgRoot(StackBarDlg, data))) return false;
	hDlg = CreateDlgWnd((char*)SDLG_MLPLOT_HD1, 50, 50, 440, 280, Dlg, 0x4L);
	do {
		if(updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(106, true);	Dlg->ShowItem(108, false);
				}
			else {
				Dlg->ShowItem(106, false);	Dlg->ShowItem(108, true);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_MLPLOT_XFMT, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt, SDLG_MLPLOT_XFMT, currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(101, (unsigned char*)TmpTxt);
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_MLPLOT_YFMT, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt,SDLG_MLPLOT_YFMT, currYR+1, maxYR+1);
#endif
			Dlg->SetText(103, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:
			if(bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 1:
		case 105:										//next button
			bError=false;	s1 = s2 = 0;
			if((Dlg->GetText(102, x_txt, 100))) {
				rX = new AccRange((char*)x_txt);
				if(rX) {
					s1 = rX->CountItems();
					if(s1 < 2) {
						ErrorBox((char*)SCMS_BAD_XRANGE);
						bContinue=bError=true;
						Dlg->Activate(102, 1);
						}
					delete rX;
					}
				else bError = true;
				}
			else bError = true;
			if((Dlg->GetText(104, y_txt, 100)) && !bError) {
				if((rY = new AccRange((char*)y_txt))) {
					s2 = rY->CountItems();
					if(s2 < 2) {
						ErrorBox((char*)SCMS_BAD_YRANGE);
						bContinue=bError=true;
						Dlg->Activate(104, 1);
						}
					delete rY;
					}
				else bError = true;
				}
			else {
				Dlg->Activate(104, 1);
				bError = true;
				}
			if(!s1 || !s2) bError = true;
			rX = rY = 0L;
			if(!bError && s1!=s2) {
				ErrorBox((char*)SCMS_BAD_RANGE);
				bContinue=bError=true;
				}
			if(!bError) {
				if((currYR+1) > maxYR) {
					rdx = (unsigned char**)realloc(rdx, sizeof(unsigned char*)*(currYR+2));
					rdy = (unsigned char**)realloc(rdy, sizeof(unsigned char*)*(currYR+2));
					rdc = (DWORD*)realloc(rdc, sizeof(DWORD)*(currYR+2));
					rdx[currYR] = rdx[currYR+1] = rdy[currYR] = rdy[currYR+1] = 0L;
					maxYR = currYR+1;
					rdc[currYR] = rdc[currYR+1] = Dlg->GetCheck(302) ? colarr[maxYR & 0x07] : defcol;
					}
				if(rdx[currYR]) free(rdx[currYR]);		//store x-range
				rdx[currYR] = rlp_strdup(x_txt);
				if(rdy[currYR]) free(rdy[currYR]);		//store y range
				rdy[currYR] = rlp_strdup(y_txt);
				Dlg->GetColor(107, &curr_col);			rdc[currYR] = curr_col;
				updateYR = true;						currYR++;
				Dlg->SetColor(107, rdc[currYR]);		Dlg->SetText(102, rdx[currYR]);
				Dlg->SetText(104, rdy[currYR]);			Dlg->Activate(102, 1);				
				if(res != 1) res = -1;
				}
			else if(res != 1){
				bContinue = true;						res = -1;
				}
			break;
		case 106:										//prev button
			if((Dlg->GetText(102, x_txt, 100)) && (Dlg->GetText(104, y_txt, 100))){
				if(rdx[currYR]) free(rdx[currYR]);
				if(rdy[currYR]) free(rdy[currYR]);
				rdx[currYR] = rlp_strdup(x_txt);
				rdy[currYR] = rlp_strdup(y_txt);
				Dlg->GetColor(107, &curr_col);			rdc[currYR] = curr_col;
				}
			else if(currYR == maxYR) maxYR--;
			currYR--;
			Dlg->SetColor(107, rdc[currYR]);			Dlg->SetText(102, (unsigned char*)rdx[currYR]);
			Dlg->SetText(104, rdy[currYR]);				Dlg->Activate(102, 1);
			updateYR = true;
			res = -1;
			break;
		case 300:	case 301:
		case 201:	case 202:	case 203:	case 204:
		case 205:	case 206:	case 207:	case 208:
			if (res == 300)	Dlg->SetColor(107, defcol);
			if (res == 301) {
				Dlg->SetCheck(300, 0L, true);		Dlg->GetColor(res, &defcol);
				}
			res = -1;								break;
		case 302:
			Dlg->SetColor(107, colarr[currYR & 0x07]);
			bContinue = true;
			res = -1;	break;
		case 303:	case 304:	case 305:	case 306:
		case 307:	case 308:	case 309:	case 310:
			Dlg->SetCheck(302, 0L, true);
			i = res-303;
			if(rdx && rdy && i <= maxYR && rdc[i] == colarr[i]) {
				Dlg->GetColor(res, &colarr[i]);		rdc[i] = colarr[i];
				Dlg->SetColor(107, rdc[currYR]);
				}
			else {
				Dlg->GetColor(res, &colarr[i]);
				}
			res = -1;	break;
			}
		}while (res < 0);
	if(res == 1 && rdx && rdy && maxYR) {
		maxYR++;		rX = rY = 0L;
		xyPlots = (PlotScatt**)calloc(maxYR + 2, sizeof(PlotScatt*));
		if (xyPlots) for (i = numXY = 0; i < maxYR; i++){
			if(rdx[i] && rdy[i] && rdx[i][0] && rdy[i][0]) {
				if(Dlg->GetCheck(200) && (rX = new AccRange((char*)rdx[i])) && (rY = new AccRange((char*)rdy[i]))) {
					lsyms = (Symbol**)calloc(rX->CountItems()+2, sizeof(Symbol*));
					symsize = syms[i &0x07]->GetSize(SIZE_SYMBOL);
					for(nd = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry); rX->GetNext(&cx, &rx), rY->GetNext(&cy, &ry); ) {
						if(data->GetValue(rx, cx, &x) && data->GetValue(ry, cy, &y)) {
							lsyms[nd] = new Symbol(0L, data, x, y, syms[i &0x07]->type, cx, rx, cy, ry);
							if(Dlg->GetCheck(209)) lsyms[nd]->SetColor(COL_SYM_LINE, Dlg->GetCheck(300) ? defcol : rdc[i & 0x07]);
							else {
								lsyms[nd]->SetColor(COL_SYM_LINE, syms[i &0x07]->GetColor(COL_SYM_LINE));
								lsyms[nd]->SetColor(COL_SYM_FILL, syms[i &0x07]->GetColor(COL_SYM_FILL));
								}
							lsyms[nd]->SetSize(SIZE_SYMBOL, symsize);
							nd++;
							}
						}
					}
				else {
					nd = 0;		lsyms = 0L;
					}
				dl = new DataLine(this, data, (char*)rdx[i], (char*)rdy[i]);
				if(dl) {
					dl->SetColor(COL_DATA_LINE, Dlg->GetCheck(300) ? defcol : rdc[i & 0xf]);
					xyPlots[numXY] = new PlotScatt(this, data, nd, lsyms, dl);
					if (xyPlots[numXY]) {
						if(rX && rY) xyPlots[numXY]->DefName((char*)"Data Line: ", rX, rY);
						else if(rY) xyPlots[numXY]->data_desc = rlp_strdup(rY->RangeDesc(data, 1));
						numXY++;
						}
					else delete dl;
					}
				if(rX)delete rX;
				if(rY)delete rY;
				rX = 0L; rY = 0L;
				}
			}
		if(numXY) bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rdx) {
		for (i = 0; i < maxYR; i++)	if(rdx[i]) free(rdx[i]);
		free(rdx);
		}
	if(rdy) {
		for (i = 0; i < maxYR; i++)	if(rdy[i]) free(rdy[i]);
		free(rdy);
		}
	for(i = 0; i < 8; i++) if(syms[i]) delete syms[i];
	free(StackBarDlg);		if(rdc) free(rdc);
	if(bRet) {
		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
		Command(CMD_AUTOSCALE, 0L, 0L);
		}
	free(tab1);		free(tab2);		free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Pie and ring chart properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *PieDlgTmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,-1,175,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,175,25,45,12\n"
	"3,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,5,100,ISPARENT | CHECKED,SHEET,1,5,10,160,103\n"
	"5,6,200,ISPARENT,SHEET,2,5,10,160,103\n"
	"6,7,300,ISPARENT,SHEET,3,5,10,160,103\n"
	"7,10,350,ISPARENT,SHEET,4,5,10,160,103\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,5,10,25,60,8\n"
	".,105,,,RANGEINPUT,-15,15,35,140,10\n"
	"105,.,500,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,600,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,,,EDVAL1,7,58,59,30,10\n"
	".,,,,LTEXT,-3,89,59,15,8\n" 
	"200,+,,,LTEXT,8,15,30,60,8\n"
	".,.,,,RTEXT,-4,2,42,20,8\n"
	".,204,,,EDVAL1,9,23,42,30,10\n"
	"204,.,,,RTEXT,-5,47,42,20,8\n"
	".,.,,,EDVAL1,10,68,42,30,10\n"
	".,.,,,LTEXT,-3,99,42,15,8\n" 
	".,.,,,RTEXT,11,27,58,20,8\n"
	".,.,,,EDVAL1,12,48,58,30,10\n"
	".,210,,,LTEXT,13,79,58,15,8\n" 
	"210,.,400,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,.,410,ISPARENT | CHECKED,GROUP,,,,,\n"
	".,,,,LTEXT,-27,15,80,30,8\n"
	"300,,,NOSELECT,ODB,15,40,35,80,60\n"
	"350,+,,HICOL,CHECKBOX,14,15,30,60,8\n"
	".,,,TOUCHEXIT,RANGEINPUT,-17,14,50,140,10\n"
	"400,+,,EXRADIO | CHECKED,ODB,16,65,75,30,30\n"
	".,,,EXRADIO,ODB,16,115,75,30,30\n"
	"410,+,,EXRADIO | CHECKED,ODB,16,55,75,30,30\n"
	".,,,EXRADIO,ODB,16,85,75,30,30\n"
	"500,+,,HICOL | CHECKED,RADIO1,17,10,59,20,8\n"
	".,.,,HICOL,RADIO1,18,10,71,40,8\n"
	".,.,,TOUCHEXIT,RANGEINPUT,-16,15,82,140,10\n"
	".,.,,,LTEXT,20,15,96,10,8\n"
	".,.,,,EDVAL1,21,42,96,25,10\n"
	".,,,,LTEXT,22,70,96,15,8\n"
	"600,+,,,RTEXT,23,8,59,45,8\n"
	".,.,,,RTEXT,24,8,74,45,8\n"
	".,.,,,EDVAL1,25,58,74,30,10\n"
	".,,,LASTOBJ,LTEXT,-3,89,74,15,8";
bool
PieChart::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	TabSHEET *tab4 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DESC);
	double fcx =10.0, fcy = 20.0, frad=40.0, firad = 30.0;
	char txt1[100], txt2[100];
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)tab4, (void*)SDLG_PIE_RNG,
		(void*)0L, (void*)&frad, (void*)SDLG_PIE_CENT, (void*)&fcx, (void*)&fcy,
		(void*)SDLG_PIE_START, (void*)&CtDef.fx, (void*)SDLG_PIE_DEG, (void*)SDLG_PIE_USENAME, (void*)(OD_scheme),
		(void*)(OD_PieTempl), (void*)SDLG_PIE_FIXR, (void*)SDLG_PIE_SSR,
		(void*)0L, (void*)SDLG_PIE_XFAC, (void*)&FacRad, (void*)&txt2, (void*)SDLG_PIE_ORAD,
		(void*)SDLG_PIE_IRAD, (void*)&firad };
	DlgInfo *PieDlg = CompileDialog(PieDlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	long ix, iy, rix, riy, nix, niy;
	int ny = 0, res, cf, cb;
	bool bRet = false, bContinue = false;
	double sum = 0.0, dang1, dang2;
	double fv;
	lfPOINT fpCent;
	AccRange *rY = 0L, *rR = 0L, *rN = 0L;

	if(!parent || !data) return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100, TmpTxt+200);
	cb = rlp_strcpy(txt2, 80, (char*)"= [");	cb += rlp_strcpy(txt2+cb, 80-cb, Units[defs.units()].display);
	rlp_strcpy(txt2+cb, 80-cb, (char*)"]");
	frad = (parent->GetSize(SIZE_DRECT_BOTTOM) - parent->GetSize(SIZE_DRECT_TOP))/2.0;
	fcx = parent->GetSize(SIZE_GRECT_LEFT) + (parent->GetSize(SIZE_DRECT_LEFT))/2.0 + frad;
	fcy = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_DRECT_TOP) + frad;
	firad = frad-frad/10.0f;
	if(!(Dlg = new DlgRoot(PieDlg, data)))return false;
	if(Id == GO_PIECHART) {
		Dlg->ShowItem(105, true);		Dlg->ShowItem(106, false);
		Dlg->ShowItem(210, true);		Dlg->ShowItem(211, false);
		}
	else {
		Dlg->ShowItem(105, false);		Dlg->ShowItem(106, true);
		Dlg->ShowItem(210, false);		Dlg->ShowItem(211, true);
		}
	hDlg = CreateDlgWnd(Id == GO_PIECHART ? (char*)SDLG_PIE_HD1 :
		(char*)SDLG_PIE_HD2, 50, 50, 475, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		//rY is defined by OK. If other buttons use rY reset to 0L!
		switch(res) {
		case 0:							//lost focus ?
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 351:
			Dlg->SetCheck(350, NULL, true);
			res = -1;
			break;
		case 400:	case 410:
			Dlg->SetText(208, (unsigned char*)"90");		Dlg->DoPlot(0L);
			CtDef.fy = 360.0;					res = -1;
			break;
		case 401:	case 411:
			Dlg->SetText(208, (unsigned char*)"180");		Dlg->DoPlot(0L);
			CtDef.fy = 180.0;								res = -1;
			break;
		case 1:		case 502:
			if (res == 502) {
				Dlg->SetCheck(501, NULL, true);				res = -1;
				}
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0] && (rY = new AccRange(TmpTxt))) 
				ny = rY->CountItems();
			else ny = 0;
			Dlg->GetValue(208, &CtDef.fx);		Dlg->GetValue(202, &fcx);
			Dlg->GetValue(205, &fcy);			Dlg->GetValue(107, &frad);
			Dlg->GetValue(602, &firad);			Dlg->GetValue(504, &FacRad);
			if(Dlg->GetCheck(501) && ny && Dlg->GetText(502, (unsigned char*)txt1, 100) && 
				(rR = new AccRange(txt1))){
				if(rR->CountItems() != ny) {
					delete rR;
					delete rY;
					rR = 0L;	 rY = 0L;
					ErrorBox((char*)SCMS_RNG_RAD_DIFFER);
					bContinue = true;
					res = -1;
					}
				}
			if (Dlg->GetCheck(350) && ny && Dlg->GetText(351, (unsigned char*)txt1, 100) &&
				(rY && (rN = new AccRange(txt1)))){
				if (rN->CountItems() != ny) {
					delete rN;	delete rR;	delete rY;
					rR = 0L;	 rY = 0L;	rN = 0L;
					ErrorBox((char*)SCMS_RNG_SEG_DIFFER);
					bContinue = true;
					res = -1;
					}
				}
			break;
			}
		}while (res < 0);

	if(res == 1 && rY && ny >1 && (Segments = (segment **)calloc(ny+2, sizeof(segment*)))) {
		nPts = ny;
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) ssRefA = rlp_strdup(TmpTxt);
		if(rR && Dlg->GetText(502, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) ssRefR = rlp_strdup(TmpTxt);
		Bounds.Xmax = Bounds.Ymax = 100.0;		Bounds.Xmin = Bounds.Ymin = -100.0;
		fpCent.fx = fcx;					fpCent.fy = fcy;
		rY->GetFirst(&ix, &iy);				rY->GetNext(&ix, &iy);
		for(i = 0; i < ny; i++){
			if(data->GetValue(iy, ix, &fv)) sum += fv;
			rY->GetNext(&ix, &iy);
			}
		sum /= CtDef.fy;
		dang1 = dang2 = CtDef.fx;
		rY->GetFirst(&ix, &iy);				rY->GetNext(&ix, &iy);
		if(rR) {
			rR->GetFirst(&rix, &riy);		rR->GetNext(&rix, &riy);
			}
		if (rN) {
			rN->GetFirst(&nix, &niy);		rN->GetNext(&nix, &niy);
			}
		for (i = cf = 0; i < ny; i++){
			if(data->GetValue(iy, ix, &fv)) {
				dang2 -= (double)fv / sum;
				if(dang2 < 0.0) dang2 += 360.0;
				if(rR && data->GetValue(riy, rix, &frad)) frad *= FacRad;
				if(rN) data->GetText(niy, nix, txt1, 100);
				Segments[i] = new segment(this, data, &fpCent, Id == GO_PIECHART ? 0.0 : firad, frad,
					dang1, dang2, (rN && txt1[0]) ? txt1 : NULL);
				if(Segments[i])Segments[i]->Command(CMD_SEG_FILL, GetSchemeFill(&cf), 0L);
				dang1 = dang2;
				}
			rY->GetNext(&ix, &iy);
			if(rR) rR->GetNext(&rix, &riy);
			if (rN) rN->GetNext(&nix, &niy);
			}
		bRet = true;
		}
	if(rY) delete rY;
	if(rR) delete rR;
	if(rN) delete rN;
	CloseDlgWnd(hDlg);		delete Dlg;				free(PieDlg);
	free(tab1);				free(tab2);				free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a star chart
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *StarDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,130,10,45,12\n"
".,.,,,PUSHBUTTON,-2,130,25,45,12\n"
".,,4,ISPARENT | CHECKED,GROUP,0,,,,\n"
"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,120,103\n"
"., 10,200,ISPARENT,SHEET,2,5,10,120,103\n"
"10,,,CHECKED, CHECKPIN,0,5,0,12,8\n"
"100,+,,,LTEXT,3,10,30,60,8\n"
".,.,,,RANGEINPUT,4,15,45,100,10\n"
".,.,,,RTEXT,5,17,57,30,8\n"
".,.,,,EDVAL1,6,48,57,30,10\n"
".,.,,,LTEXT,7,79,57,15,8\n"
".,.,,,RTEXT,8,17,72,30,8\n"
".,.,,,EDVAL1,9,48,72,30,10\n"
".,.,,,LTEXT,10,79,72,15,8\n"
".,.,,HICOL | CHECKED,CHECKBOX,11,25,87,20,8\n"
".,,,HICOL | CHECKED,CHECKBOX,12,25,97,20,8\n"
"200,+,,HICOL,CHECKBOX,13,15,28,50,8\n"
".,.,,,RANGEINPUT,14,15,51,100,10\n"
".,.,,, LTEXT,15,15,40,60,8\n"
".,.,,, RTEXT,16,10,70,40,8\n"
".,.,,,EDVAL1,17,52,70,30,10\n"
".,,,LASTOBJ,LTEXT,-3,85,70,10,8";

bool
StarChart::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_LABELS);
	double sa = 90.0, factor = 1.0, lbdist = NiceValue(DefSize(SIZE_TEXT)*1.2);
	char txt1[80], txt2[80];
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_STAR_RNG, (void*)txt1,
		(void*)SDLG_STAR_XFAC, (void*)&factor, (void*)&txt2, (void*)SDLG_STAR_SANG,
		(void*)&sa, (void*)SDLG_STAR_DEG, (void*)SDLG_STAR_DOPG, (void*)SDLG_STAR_DRAY,
		(void*)SDLG_STAR_DLABEL, (void*)txt1, (void*)SDLG_STAR_LBRNG, (void*)SDLG_STAR_LBDIST,
		(void*)&lbdist};
	DlgInfo *StarDlg = CompileDialog(StarDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	long ix, iy;
	int res, ny;
	long width, height;
	bool bRet = false, bContinue = false;
	AccRange *rY = 0L, *rL = 0L;
	Label *lb;
	lfPOINT *fp = 0L, *fpt = 0L, fl[2];
	double fx, fy, tmpval, frad;
	double sia = 0.0, csia = 1.0;
	polyline *plo;
	TextDEF td = {0x00000000L, 0x00ffffffL, DefSize(SIZE_TEXT), 0.0, 0.0, 0,
		TXA_HCENTER | TXA_VCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L}; 

	if(!parent || !data) return false;
	data->GetSize(&width, &height);
#ifdef USE_WIN_SECURE
	sprintf_s(txt1, 80,"a1:a%ld", height);	sprintf_s(txt2, 80, "= [%s]", Units[defs.units()].display);
#else
	sprintf(txt1, "a1:a%ld", height);		sprintf(txt2, "= [%s]", Units[defs.units()].display);
#endif
	if(parent) {
		frad = (parent->GetSize(SIZE_DRECT_BOTTOM) - parent->GetSize(SIZE_DRECT_TOP))/2.0f;
		fPos.fx = parent->GetSize(SIZE_GRECT_LEFT) + parent->GetSize(SIZE_DRECT_LEFT) + frad;
		fPos.fy = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_DRECT_TOP) + frad;
		}
	if(!(Dlg = new DlgRoot(StarDlg, data)))return false;
#ifdef USE_WIN_SECURE
	sprintf_s(TmpTxt, 80, "%g", lbdist);
#else
	sprintf(TmpTxt, "%g", lbdist);
#endif
	Dlg->SetText(204, (unsigned char*)TmpTxt);
	hDlg = CreateDlgWnd((char*)SDLG_STAR_HD1, 50, 50, 390, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:							//lost focus ?
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 1:
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0] && (rY = new AccRange(TmpTxt))) 
				ny = rY->CountItems();
			else ny = 0;
			Dlg->GetValue(106, &sa);			Dlg->GetValue(103, &factor);
			Dlg->GetValue(204, &lbdist);
			break;
			}
		}while (res < 0);
	if(res == 1 && rY && ny >1 && (fp = (lfPOINT*)calloc(ny+2, sizeof(lfPOINT))) &&
		(fpt = (lfPOINT*)calloc(ny+2, sizeof(lfPOINT)))){
		Bounds.Xmax = Bounds.Ymax = 100.0;		Bounds.Xmin = Bounds.Ymin = -100.0;
		rY->GetFirst(&ix, &iy);
		for(i = 0; i < ny; i++){
			rY->GetNext(&ix, &iy);
			if(data->GetValue(iy, ix, &tmpval)){
				tmpval *= factor;
				sia = sin(sa * 0.01745329252);			csia = cos(sa * 0.01745329252);
				fx = (tmpval * csia);					fy = (-tmpval * sia);
				}
			else fx = fy = 0.0f;
			fp[i].fx = fpt[i].fx = fx;			fp[i].fy = fpt[i].fy = fy;
			fpt[i].fx += lbdist *csia;			fpt[i].fy += (-lbdist * sia);
			sa -= 360.0/ny;
			}
		fp[i].fx = fp[0].fx;		fp[i].fy = fp[0].fy;
		if(Dlg->GetCheck(108)){
			if((plo = new polygon(this, data, fp, ny+1))) {		//ny+1 to close the shape!
				if(!(bRet = Command(CMD_DROP_OBJECT, (void*)plo, 0L))) delete plo;
				}
			}
		if(Dlg->GetCheck(109)){
			fl[0].fx = fl[0].fy = 0.0f;
			for(i = 0; i < ny; i++) {
				fl[1].fx = fp[i].fx;		fl[1].fy = fp[i].fy;
				if((plo = new polyline(this, data, fl, 2))) {
					if(Command(CMD_DROP_OBJECT, (void*)plo, 0L)) bRet = true;
					else delete plo;
					}
				}
			}
		if(Dlg->GetCheck(200) && Dlg->GetText(201, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0] && (rL = new AccRange(TmpTxt))){
			rL->GetFirst(&ix, &iy);
			td.text = (unsigned char*)TmpTxt;
			for(i = 0; i < ny; i++) {
				rL->GetNext(&ix, &iy);
				if(data->GetText(iy, ix, TmpTxt, TMP_TXT_SIZE)){
					if((lb = new Label(this, data, fpt[i].fx, fpt[i].fy, &td, 0L, 0L))) {
						if(Command(CMD_DROP_OBJECT, (void*)lb, 0L)) bRet = true;
						else delete lb;
						}
					}
				}
			}
		}
	if(rY) delete rY;
	if(rL) delete rL;
	if(fp) free(fp);
	if(fpt) free(fpt);
	CloseDlgWnd(hDlg);
	free(StarDlg);				free(tab1);				free(tab2);
	delete Dlg;					return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Grid3D represents a surface in space
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *GridDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,155,10,50,12\n"
".,.,,,PUSHBUTTON,-2,155,25,50,12\n"
".,,300,ISPARENT | CHECKED,GROUP,0,,,,\n"
"300,+,500, HIDDEN | CHECKED,GROUPBOX,1,10,10,140,100\n"
"., 305, 400, HIDDEN | CHECKED,GROUPBOX,2,10,10,140,100\n"
"305,+,,TOUCHEXIT,RADIO1,1,155,45,50,10\n"
".,,,TOUCHEXIT,RADIO1,2,155,57,50,10\n"
"400,+,,,RTEXT,3,38,20,40,8\n"
".,.,,,EDVAL1,4,80,20,25,10\n"
".,.,,,LTEXT,-3,107,20,20,8\n"
".,.,,,RTEXT,5,38,35,40,8\n"
".,.,,OWNDIALOG,COLBUTT,6,80,35,25,10\n"
".,.,,,RTEXT,7,38,50,40,8\n"
".,.,,OWNDIALOG,SHADEPLANE,8,80,50,25,10\n"
".,.,,,ICON,10,20,70,20,20\n"
".,.,,,LTEXT,11,45,72,100,6\n"
".,,,,LTEXT,12,45,78,100,6\n"
"500,,,LASTOBJ | NOSELECT,ODBUTTON,9,15,15,130,100";

bool
Grid3D::PropertyDlg()
{
	return Configure();
}

bool
Grid3D::Configure()
{
	static FillDEF newFill;
	int icon = ICO_INFO;
	void *dyndata[] = { (void*)SDLG_G3D_GLINE, (void*)SDLG_G3D_SURF, (void*)SDLG_G3D_GLW,
		(void*)&Line.width, (void*)SDLG_G3D_GLC, (void*)&Line.color, (void*)SDLG_G3D_PLC,
		(void*)&newFill, (void*)OD_linedef, (void*)&icon, (void*)SDLG_G3D_INF1, (void*)SDLG_G3D_INF2};
	DlgInfo *GridDlg = CompileDialog(GridDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int i, res, cb, new_type, undo_level = Undo.level();
	bool bRet = false;
	double tmp;
	DWORD new_col;
	LineDEF newLine;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent) return false;
	memcpy(&newFill, &Fill, sizeof(FillDEF));
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	if(!(Dlg = new DlgRoot(GridDlg, data))) return false;
	if(!type) {
		Dlg->ShowItem(300, true);	Dlg->SetCheck(305, 0L, true);
		}
	else {
		Dlg->ShowItem(301, true);	Dlg->SetCheck(306, 0L, true);
		}
	if(parent->name) {
		cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_G3D_HD1);
		rlp_strcpy(TmpTxt+cb, TMP_TXT_SIZE-cb, parent->name);
		}
	else rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_G3D_HD2);
#ifdef _WINDOWS
	for (i = 408; i <= 409; i++) Dlg->TextSize(i, 12);
#else
	for (i = 408; i <= 409; i++) Dlg->TextSize(i, 10);
#endif
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 446, 280, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 305:		case 306:
			if(Dlg->GetCheck(305)) {
				Dlg->ShowItem(300, true);	Dlg->ShowItem(301, false);
				}
			else {
				Dlg->ShowItem(300, false);	Dlg->ShowItem(301, true);
				}
			Dlg->Command(CMD_REDRAW, 0L, 0L);
			res = -1;
			break;
			}
		}while (res < 0);
		if (cdisp) {
			Undo.SetDisp(cdisp);
			while (Undo.level() > undo_level)	Undo.Pop(cdisp);
			}
	if(res == 1) {
		if(Dlg->GetCheck(305) && type == 0) {
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			if(cmpLineDEF(&Line, &newLine)) {
				Command(CMD_SET_LINE, &newLine, 0L);
				bRet = true;
				}
			}
		else if(Dlg->GetCheck(306) && type == 1) {
			Dlg->GetValue(401, &tmp);			Dlg->GetColor(404, &new_col);
			if(planes && (cmpFillDEF(&Fill, &newFill) || tmp != Line.width || new_col != Line.color)) {
				Command(CMD_SAVE_SYMBOLS, 0L, 0L);
				Command(CMD_SYM_FILL, &newFill, 0L);
				if(tmp != Line.width) SetSize(SIZE_SYM_LINE, tmp);
				if(new_col != Line.color) SetColor(COL_POLYLINE, new_col);
				bRet = true;
				}
			}
		else {
			Undo.ValLong(parent, &type, 0L);
			Undo.Line(this, &Line, UNDO_CONTINUE);	Undo.Fill(this, &Fill, UNDO_CONTINUE);
			if(Dlg->GetCheck(305)) {
				new_type = 0;
				OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&Line, 0);
				}
			else {
				new_type = 1;
				memcpy(&Fill, &newFill, sizeof(FillDEF));
				Dlg->GetValue(401, &Line.width);
				Dlg->GetColor(404, &Line.color);
				Line.pattern = 0L;
				Line.patlength = 1;
				}
			if(planes && nPlanes) Undo.DropListGO(parent, (GraphObj***)&planes, &nPlanes, UNDO_CONTINUE);
			if(lines && nLines) Undo.DropListGO(parent, (GraphObj***)&lines, &nLines, UNDO_CONTINUE);
			Undo.VoidPtr(parent, (void**)&planes, 0L, 0L, UNDO_CONTINUE);
			Undo.VoidPtr(parent, (void**)&lines, 0L, 0L, UNDO_CONTINUE);
			type = new_type;
			CreateObs(true);
			bRet = true;
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;				free(GridDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Scatt3D is a layer representing most simple 3D plots
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *Dlg3DTmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,-1,162,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,162,25,45,12\n"
	"3,50,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,5,100,ISPARENT | CHECKED,SHEET,1,5,10,151,100\n"
	"5,6,200,TOUCHEXIT | ISPARENT,SHEET,2,5,10,151,100\n"
	"6,7,400,ISPARENT,SHEET,3,5,10,151,100\n"
	"7,10,300,ISPARENT,SHEET,18,5,10,151,100\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"50,60,,,ODB,4,164,65,45,45\n"
//	"50,60,,NOSELECT,ODB,4,144,65,45,45\n"
	"60,+,,,ICON,5,10,114,20,20\n"
	".,.,,,LTEXT,6,35,116,100,6\n"
	".,,,,LTEXT,7,35,122,100,6\n"
	"100,+,,,LTEXT,-31,10,30,60,8\n"
	".,.,,,RANGEINPUT,8,20,40,120,10\n"
	".,.,,,LTEXT,-32,10,55,60,8\n"
	".,.,,,RANGEINPUT,9,20,65,120,10\n"
	".,.,,,LTEXT,-33,10,80,60,8\n"
	".,,,,RANGEINPUT,10,20,90,120,10\n"
	"200,+,,,LTEXT,-27,25,29,60,8\n"
	".,.,,HICOL,CHECKBOX,11,40,50,60,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,12,40,60,60,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,13,40,40,60,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,14,40,70,60,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,15,40,80,60,8\n"
	".,,,HICOL | TOUCHEXIT,CHECKBOX,22,40,90,60,8\n"
	"300,+,,,LTEXT,19,10,50,60,8\n"
	".,.,,TOUCHEXIT,RANGEINPUT,20,20,60,120,10\n"
	".,,,HICOL,CHECKBOX,21,10,30,60,8\n"
	"400,410,,,LTEXT,16,20,30,60,8\n"
	"410,+,,EXRADIO,ODB,17,20,42,,\n"
	".,.,,EXRADIO,ODB,17,45,42,,\n"
	".,,,LASTOBJ | EXRADIO,ODB,17,70,42,,";

static bool bAxis_3d_arr = false;

bool
Scatt3D::PropertyDlg()
{
	int i1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_LAYOUT);
	TabSHEET *tab3 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_AXES);
	TabSHEET *tab4 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_ERRBARS);
	unsigned char text1[100], text2[100], text3[100], text4[100];
	int icon = ICO_INFO;
	void *dyndata[] = {(void*)tab1, (void*)tab2, (void*)tab3, (void*)(OD_AxisDesc3D), (void*)&icon,
		(void*)SDLG_PLOT3D_ROT1, (void*)SDLG_PLOT3D_ROT2, (void*)text1, (void*)text2, (void*)text3,
		(void*)SDLG_PLOT3D_SEL1, (void*)SDLG_PLOT3D_SEL2, (void*)SDLG_PLOT3D_SEL3,
		(void*)SDLG_PLOT3D_SEL4, (void*)SDLG_PLOT3D_SEL5, (void*)SDLG_PLOT3D_TEMPL,
		(void*)(OD_AxisTempl3D), (void*)tab4, (void*)SDLG_PLOT3D_ERRRNG, (void*)text4,
		(void*)SDLG_PLOT3D_DOERRS, (void*)SDLG_PLOT3D_SEL6 };
	DlgInfo *Dlg3D = CompileDialog(Dlg3DTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, n1 = 0, n2 = 0, n3 = 0, ne, ic = 0;
	long i, j, k, l, m, n, o, p;
	int i2, j2, k2, l2, m2, cb;
	double x, y, z, e, bar_w, bar_d, rad;
	bool bRet = false, bLayout = false, bContinue = false;
	AccRange *rX, *rY, *rZ, *rE;
	fPOINT3D pos1, pos2;

	if(!data || !parent)return false;
	rX = rY = rZ = rE = 0L;
	UseRangeMark(data, 1, (char*)text1, (char*)text2, (char*)text3, (char*)text4);
	if(!(Dlg = new DlgRoot(Dlg3D, data)))return false;
	Dlg->SetCheck(410 + AxisTempl3D, 0L, true);
	Dlg->SetCheck(50, 0L, bAxis_3d_arr);
	for(i = 0; i < 5; i++) Dlg->SetCheck(201+i, 0L, (c_flags & (1<<i))!=0);
	if(c_flags == 0x2000 || c_flags == 0x4000) {
		Dlg->ShowItem(5, false);		Dlg->ShowItem(6, false);	Dlg->ShowItem(7, false);
		}
	else Dlg->ShowItem(6, (c_flags & 0x1000) == 0x1000);
	rX = rY = rZ = 0L;					rad = DefSize(SIZE_SYMBOL);
#ifdef _WINDOWS
	for(i = 61; i <= 62; i++) Dlg->TextSize(i, 12);
#else
	for(i = 61; i <= 62; i++) Dlg->TextSize(i, 10);
#endif
	hDlg = CreateDlgWnd(c_flags == 0x2000 ? (char*)SDLG_PLOT3D_HD3 :
		c_flags == 0x4000 ? (char*)SDLG_PLOT3D_HD2 : (char*)SDLG_PLOT3D_HD1, 50, 50, 448, 325, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 202:
			if (Dlg->GetCheck(202)) {
				Dlg->SetCheck(204, 0L, false);		res = -1;
				}
			break;
		case 204:
			if (Dlg->GetCheck(204)) {
				Dlg->SetCheck(202, 0L, false);		res = -1;
				}
			break;
		case 203:
			if (Dlg->GetCheck(203)) {
				Dlg->SetCheck(205, 0L, false);	Dlg->SetCheck(206, 0L, false);
				res = -1;
				}
			break;
		case 205:
			if (Dlg->GetCheck(205)) {
				Dlg->SetCheck(203, 0L, false);	Dlg->SetCheck(206, 0L, false);
				res = -1;
				}
			break;
		case 206:
			if (Dlg->GetCheck(206)) {
				Dlg->SetCheck(205, 0L, false);	Dlg->SetCheck(203, 0L, false);
				res = -1;
				}
			Dlg->SetCheck(203, 0L, false);		res = -1;
			break;
		case 410:	case 411:	case 412:	//axis templates
			AxisTempl3D = res-410;
			res = -1;			break;
		case 301:
			Dlg->SetCheck(302, 0L, true);		res = -1;			break;
		case 50:		case 1:
			if (res == 50) {
				if (Dlg->GetCheck(50)) Dlg->SetCheck(50, 0L, false);
				else Dlg->SetCheck(50, 0L, true);
				res = -1;
				}
			if(rX) delete rX;
			if(rY) delete rY;
			if(rZ) delete rZ;	
			if(rE) delete rE;
			rX = 0L; rY = 0L; rZ = 0L; rE = 0L;		n1 = 0; n2 = 0; n3 = 0; ne = 0;
			if(Dlg->GetText(101, text1, 100) && (rX = new AccRange((char*)text1))) n1 = rX->CountItems();
			if (Dlg->GetCheck(50)) {		//axis arrangement
				bAxis_3d_arr = true;
				if (Dlg->GetText(103, text2, 100) && (rZ = new AccRange((char*)text2))) n3 = rZ->CountItems();
				if (Dlg->GetText(105, text3, 100) && (rY = new AccRange((char*)text3))) n2 = rY->CountItems();
				}
			else {
				bAxis_3d_arr = false;
				if (Dlg->GetText(103, text2, 100) && (rY = new AccRange((char*)text2))) n2 = rY->CountItems();
				if (Dlg->GetText(105, text3, 100) && (rZ = new AccRange((char*)text3))) n3 = rZ->CountItems();
				}
			if(Dlg->GetCheck(302) && Dlg->GetText(301, text4, 100) && (rE = new AccRange((char*)text4)))
				ne = rE->CountItems();
			if(n1 && n2 && n3){
				if(c_flags == 0x2000 || c_flags == 0x4000) {
					//no more but a ribbon or surface
					}
				else if(n1 == n2 && n2 == n3 && (rE == 0L || ne == n2)) {
					//o.k., three ranges of equal size have been defined
					c_flags = 0;
					for(i = 0; i < 6; i++) if(Dlg->GetCheck(201+i)) c_flags |= (1<<i);
					}
				else {
					InfoBox((char*)SCMS_BAD_RANGE);
					res = -1;	bContinue = true;
					}
				}
			else {
				cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_PLOT3D_BADR);
				if (!n1) cb += rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, (char*)SDLG_PLOT3D_BADRX);
				if (!n2) cb += rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, (char*)SDLG_PLOT3D_BADRY);
				if (!n3) cb += rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, (char*)SDLG_PLOT3D_BADRZ);
				rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, (char*)SDLG_PLOT3D_RNGMISS);
				InfoBox(TmpTxt);
				res = -1;	bContinue = true;
				}
			//the layout menu must have been visited
			if (res > 0 && !bLayout && c_flags != 0x2000 && c_flags != 0x4000){
				Dlg->SetCheck(5, 0L, bLayout = true);	res = -1;
				}
			break;
		case 5:								// the layout tab sheet
			bLayout = true;				res = -1;				break;
			}
		}while (res <0);
	if(res == 1 && rX && rY && rZ) {
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		bar_w = bar_d = (DefSize(SIZE_BAR)/2.0);
		if(c_flags & 0x01) (Balls = (Sphere**)calloc((nBalls = n1)+2, sizeof(Sphere*)));
		if(c_flags & 0x02) (Columns = (Brick**)calloc((nColumns = n1)+2, sizeof(Brick*)));
		if (Dlg->GetCheck(50)) {		//axis arrangement
			if (c_flags & 0x04) Line = new Line3D(this, data, (char*)text1, (char*)text3, (char*)text2);
			else if (c_flags & 0x20) Line = new Polygon3D(this, data, (char*)text1, (char*)text3, (char*)text2);
			}
		else {
			if (c_flags & 0x04) Line = new Line3D(this, data, (char*)text1, (char*)text2, (char*)text3);
			else if (c_flags & 0x20) Line = new Polygon3D(this, data, (char*)text1, (char*)text2, (char*)text3);
			}
		if (c_flags & 0x08) (DropLines = (DropLine3D**)calloc((nDropLines = n1) + 2, sizeof(DropLine3D*)));
		if(c_flags & 0x010) (Arrows = (Arrow3D**)calloc((nArrows = n1)+2, sizeof(Arrow3D*)));
		if(rE) (Errors = (ErrBar3D**)calloc((nErrors = n1)+2, sizeof(ErrBar3D*)));
		rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
		rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
		rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
		if(rE)rE->GetFirst(&o, &p);
		if(rE)rE->GetNext(&o, &p);
		i2 = i;	j2 = j;	k2 = k;	l2 = l;	m2 = m;	n2 = n;
		if(c_flags == 0x2000){
			Bounds.Ymin = 0.0;
			if (Dlg->GetCheck(50)) {		//axis arrangement
				rib = new Ribbon(this, data, 2, (char*)text1, (char*)text3, (char*)text2);
				}
			else {
				rib = new Ribbon(this, data, 2, (char*)text1, (char*)text2, (char*)text3);
				}
			if (!(rib->HasPlanes())) {
				delete rib;		rib = 0L;
				}
			}
		else if (c_flags == 0x4000){
			if (Dlg->GetCheck(50)) {		//axis arrangement
				rib = new Ribbon(this, data, 3, (char*)text1, (char*)text3, (char*)text2);
				}
			else {
				rib = new Ribbon(this, data, 3, (char*)text1, (char*)text2, (char*)text3);
				}
			if (!(rib->HasPlanes())) {
				delete rib;		rib = 0L;
				}
			}
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		do {
			if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && 
				data->GetValue(n, m, &z)){
				if(ic) {
					pos1.fx = pos2.fx;		pos1.fy = pos2.fy;	pos1.fz = pos2.fz;
					}
				else {
					pos1.fx = x;			pos1.fy = y;		pos1.fz = z;
					}
				pos2.fx = x;	pos2.fy = y;	pos2.fz = z;
				CheckBounds3D(x, y, z);
				if(!bRet) memcpy(&pos1, &pos2, sizeof(fPOINT3D));
				if(Balls) Balls[ic] = new Sphere(this, data, 0, x, y, z, rad, i, j, k, l, m, n);
				if(Columns)Columns[ic] = new Brick(this, data, x, 0.0, z, 
					bar_d, bar_w, y, 0x800L, i, j, -1, -1, m, n, -1, -1, -1, -1, k, l);
				if(DropLines) DropLines[ic] = new DropLine3D(this, data, &pos2, i, j, k, l, m, n);
				if(Arrows) Arrows[ic] = new Arrow3D(this, data, &pos1, &pos2, 
					i, j, k, l, m, n, i2, j2, k2, l2, m2, n2);
				if(rE && Errors && data->GetValue(p, o, &e)){
					Errors[ic] = new ErrBar3D(this, data, x, y, z, e, 0, i, j, k, l, m, n, o, p);
					rE->GetNext(&o, &p);
					}
				bRet = true;
				}
			i2 = i;	j2 = j;	k2 = k;	l2 = l;	m2 = m;	n2 = n;
			ic++;
			}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
			if (!bRet) InfoBox((char*)SCMS_RNG_XYZ_EMPTY);
		bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;			free(Dlg3D);
	if(rX) delete rX;
	if(rY) delete rY;	
	if(rZ) delete rZ;	
	if(rE) delete rE;
	free(tab1);				free(tab2);			free(tab3);			free(tab4);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// user defined function properties dialog
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

static char *FuncDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,160,10,32,12\n"
".,.,,,PUSHBUTTON,-2,160,25,32,12\n"
"3,10,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,5,100,ISPARENT, SHEET,1,5,10,149,110\n"
"5,, 500,ISPARENT | CHECKED,SHEET,2,5,10,149,110\n"
"10,600,,CHECKED,CHECKPIN,0,5,0,12,8\n"
"100,+,,,LTEXT,3,10,24,100,8\n"
".,.,,,RTEXT,4,10,40,28,8\n"
".,.,,,EDVAL1,5,38,40,25,10\n"
".,.,,, RTEXT,6,61,40,17,8\n"
".,.,,,EDVAL1,7,78,40,25,10\n"
".,.,,,RTEXT,8,102,40,17,8\n"
".,.,,,EDVAL1,9,119,40,25,10\n"
".,200,,,RTEXT,10,22,54,122,40\n"
"200,,,,TEXTBOX,11,22,54,122,40\n"
"500,,,NOSELECT,ODBUTTON,12,15,25,130,100\n"
"600,+,,TOUCHEXIT,ODBUTTON,13,170,45,15,15\n"
".,.,,TOUCHEXIT,ODBUTTON,13,170,60,15,15\n"
".,.,,TOUCHEXIT,ODBUTTON,13,170,75,15,15\n"
".,,,LASTOBJ | TOUCHEXIT,ODBUTTON,13,170,90,15,15";

bool
Function::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_FUNCTION);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_LINE);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_FUNC2D_TIT, (void*)SDLG_FUNC2D_BEGX,
		(void*)&x1, (void*)SDLG_FUNC2D_ENDX, (void*)&x2, (void*)SDLG_FUNC2D_STEPX,
		(void*)&xstep, (void*)SDLG_FUNC2D_YEQ, (void*)cmdxy, (void*)OD_linedef, (void*)OD_DrawOrder };
	DlgInfo *FuncDlg = CompileDialog(FuncDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, undo_level = Undo.level();
	bool bRet = false, bNew = (dl == 0L);
	DWORD undo_flags = 0L;
	LineDEF newLine;
	double o_x1, n_x1, o_x2, n_x2, o_xstep, n_xstep;
	anyOutput *cdisp = Undo.Cdisp();

	if(!parent) return false;
	if(parent->Id == GO_FITFUNC) return parent->PropertyDlg();
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	if(!(Dlg = new DlgRoot(FuncDlg, data))) return false;
	if(!bNew) Dlg->ShowItem(10, false);
	Dlg->GetValue(102, &o_x1);		n_x1 = o_x1;
	Dlg->GetValue(104, &o_x2);		n_x2 = o_x2;
	Dlg->GetValue(106, &o_xstep);	n_xstep = o_xstep;
	if (parent->Id == GO_FREQDIST || parent->Id == GO_MYREGR) {
		Dlg->ShowItem(600, false);		Dlg->ShowItem(601, false);
		Dlg->ShowItem(602, false);		Dlg->ShowItem(603, false);
		}
	hDlg = CreateDlgWnd((char*)SDLG_FUNC2D_HD1, 50, 50, 410, 295, Dlg, 0x4L);
	if(bNew) Dlg->SetCheck(4, 0L, true);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(Dlg->GetCheck(10)) res = -1;
			break;
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);
			}
		}while (res < 0);
	Undo.SetDisp(cdisp);
	if (res == 2) while (Undo.level() > undo_level)	Undo.Restore(true, cdisp);
	else if(res == 1){					//OK pressed
		if(bNew) {						//create function
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&Line, 0);
			Dlg->GetValue(102, &x1);		Dlg->GetValue(104, &x2);
			Dlg->GetValue(106, &xstep);		Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE);
			cmdxy = rlp_strdup(TmpTxt);
			ReshapeFormula(&cmdxy);			bRet = Update(0L, 0L);
			}
		else {							//edit existing function
			Dlg->GetValue(102, &n_x1);		Dlg->GetValue(104, &n_x2);
			Dlg->GetValue(106, &n_xstep);
			undo_flags = CheckNewFloat(&x1, o_x1, n_x1, this, undo_flags);
			undo_flags = CheckNewFloat(&x2, o_x2, n_x2, this, undo_flags);
			undo_flags = CheckNewFloat(&xstep, o_xstep, n_xstep, this, undo_flags);
			if(Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE))
				undo_flags = CheckNewString((unsigned char**)&cmdxy, (unsigned char*)cmdxy, (unsigned char*)TmpTxt, this, undo_flags);
			if(undo_flags & UNDO_CONTINUE){
				Update(0L, UNDO_CONTINUE);		Command(CMD_MRK_DIRTY, 0L, 0L);
				}
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			if(cmpLineDEF(&Line, &newLine)) {
				Undo.Line(parent, &Line, undo_flags);	undo_flags |= UNDO_CONTINUE;
				memcpy(&Line, &newLine, sizeof(LineDEF));
				if (parent->Id == GO_FREQDIST) parent->Command(CMD_SET_LINE, &Line, 0L);
				}
			bRet = (undo_flags & UNDO_CONTINUE) != 0;
			}
		}
	CloseDlgWnd(hDlg);		delete Dlg;
	free(tab1);				free(tab2);
	free(FuncDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// fit function by nonlinear regression
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *FitFuncDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,180,10,42,12\n"
".,.,,,PUSHBUTTON,-2,180,25,42,12\n"
".,10,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,+,400,ISPARENT | CHECKED,SHEET,1,5,10,169,180\n"
".,.,100,ISPARENT,SHEET,2,5,10,169,180\n"
".,.,500,ISPARENT,SHEET,3,5,10,169,180\n"
".,,,,PUSHBUTTON,4,180,178,42,12\n"
"10,+,,CHECKED,CHECKPIN,0,5,0,12,8\n"
".,.,,,PROGBAR,,16,132,128,8\n"
".,,,HIDDEN,PUSHBUTTON,29,180,163,42,12\n"
"100,+,,,LTEXT,5,10,24,100,8\n"
".,.,,,LTEXT,6,10,34,100,8\n"
".,150,,,TEXTBOX,7,22,44,142,25\n"
"150,+,,,LTEXT,8,10,72,10,8\n"
".,.,,,RTEXT,9,10,86,10,8\n"
".,.,,,RTEXT,10,20,118,26,8\n"
".,.,,,EDVAL1,11,46,118,35,10\n"
".,.,,,RTEXT,12, 72, 118, 47, 8\n"
".,200,,,EDVAL1,13,119,118,25,10\n"
"200,+,,,TEXTBOX,14,22,84,142,30\n"
".,.,,,RTEXT,25,20,162,22,8\n"
".,.,,HICOL | TOUCHEXIT,RADIO1,26,50,162,60,8\n"
".,.,,HICOL | TOUCHEXIT,RADIO1,27,50,172,60,8\n"
".,.,,HIDDEN,LTEXT,4,10,152,22,8\n"
".,,,HIDDEN,LTEXT,4,10,142,22,8\n"
"400,+,,,LTEXT,-31,10,40,60,8\n"
".,.,,,RANGEINPUT,15,20,50,140,10\n"
".,.,,,LTEXT,-32,10,65,60,8\n"
".,.,,,RANGEINPUT,16,20,75,140,10\n"
".,.,,HICOL | CHECKED,CHECKBOX,17,20,105,60,8\n"
".,.,,HIDDEN,LTEXT,0,20,115,60,8\n"
".,,,ODEXIT,SYMBUTT,28,90,105,30,10\n"
"500,550,501,CHECKED,GROUP,0,,,,\n"
"501,+,,,RTEXT,18,10,30,28,8\n"
".,.,,,EDVAL1,19,38,30,25,10\n"
".,.,,,RTEXT,20,61,30,17,8\n"
".,.,,,EDVAL1,21,78,30,25,10\n"
".,.,,,RTEXT,22,102,30,17,8\n"
".,,,,EDVAL1,23,119,30,25,10\n"
"550,,,LASTOBJ | NOSELECT,ODBUTTON,24,25,45,140,100";

bool
FitFunc::PropertyDlg()
{
	int i1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_FUNCTION);
	TabSHEET *tab3 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_LINE);
	unsigned char text1[100], text2[100];
	double iter;
	bool bNew = (dl == 0L);
	Symbol *PrevSym = new Symbol(0L, data, 0.0, 0.0, 0);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_FITFUNC_FIT,
		(void*)SDLG_FITFUNC_FITFUNC, (void*)SDLG_FITFUNC_PARAM, (void*)parxy, (void*)SDLG_FITFUNC_DEF,
		(void*)SDLG_FITFUNC_YEQ, (void*)SDLG_FITFUNC_CONV, (void*)&conv, (void*)SDLG_FITFUNC_ITER,
		(void*)&iter, (void*)cmdxy, (void*)text1, (void*)text2, (void*)SDLG_FITFUNC_DOSYMS,
		(void*)SDLG_FITFUNC_XSTART, (void*)&x1, (void*)SDLG_FITFUNC_XEND, (void*)&x2,
		(void*)SDLG_FITFUNC_XSTEP, (void*)&xstep, (void*)OD_linedef, (void*)SDLG_FITFUNC_METH,
		(void*)SDLG_FITFUNC_LEVMARC, (void*)SDLG_FITFUNC_SIMPLEX, (void*)&PrevSym, (void*)SDLG_FITFUNC_BOOT };
	DlgInfo *FuncDlg = CompileDialog(FitFuncDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, undo_level = Undo.level(), cs;
	long i, j, k, l;
	size_t cb;
	bool bRet = false, bContinue = false, bDone = false;
	DWORD undo_flags = 0L;
	LineDEF newLine;
	double tmp, tmpy, o_x1, n_x1, o_x2, n_x2, o_xstep, n_xstep;
	AccRange *rX, *rY;
	unsigned char *o_cmdxy, *o_parxy, *tmp_char;
	anyOutput *cdisp = Undo.Cdisp();
	anyResult *ares;

	if(!parent || !data) return false;
	if(!(o_cmdxy = rlp_strdup((unsigned char*)cmdxy))) return false;
	if(!(o_parxy = rlp_strdup((unsigned char*)parxy))) return false;
	if (bNew) {
		for (i = 0; i < 100 && FuncDlg[i].id != 550; i++);
		if (i < 90) FuncDlg[i].y = 35;
		}
	UseRangeMark(data, 1, (char*)text1, (char*)text2);
	iter = (double)maxiter;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	if(!(Dlg = new DlgRoot(FuncDlg, data))) return false;
	if(!bNew){
		Dlg->ShowItem(10, 0);		Dlg->Activate(401, 0);
		Dlg->Activate(403, 0);		Dlg->SetCheck(6, 0L, 1);
		Dlg->SetCheck(4, 0L, 0);	Dlg->ShowItem(404, 0);
		Dlg->ShowItem(406, 0);		bDone = true;
		if(chi2 > 0.0) {
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BTFUNC_CHI2, chi2, nVals);
#else
			sprintf(TmpTxt, SDLG_BTFUNC_CHI2, chi2, nVals);
#endif
			Dlg->SetText(204,(unsigned char*)TmpTxt);			Dlg->ShowItem(204, true);
			}
		}
	else {
		Dlg->ShowItem(500, false);
		}
#ifdef USE_WIN_SECURE
	sprintf_s(TmpTxt, TMP_TXT_SIZE, "%g", conv);
#else
	sprintf(TmpTxt, "%g", conv);
#endif
	Dlg->SetText(153, (unsigned char*)TmpTxt);
	Dlg->GetValue(502, &o_x1);		n_x1 = o_x1;
	Dlg->GetValue(504, &o_x2);		n_x2 = o_x2;
	Dlg->GetValue(506, &o_xstep);	n_xstep = o_xstep;
	if (bNew){
		Dlg->SetCheck(4, 0L, true);
		}
	if (method == 1) Dlg->SetCheck(202, 0L, true);
	else {
		Dlg->SetCheck(203, 0L, true);	method = 2;
		}
	hDlg = CreateDlgWnd((char*)SDLG_FITFUNC_HD1, 50, 50, 480, 440, Dlg, 0x4L);
	if(bNew) Dlg->SetCheck(4, 0L, true);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:							//focus out
			if(Dlg->GetCheck(10)) res = -1;
			if(bContinue) res = -1;
			bContinue = false;
			break;
		case 12:						//request bootstrap
			rep_BT_Function(parent, data, text1, text2, 0L, (unsigned char *)parxy, (unsigned char *)cmdxy, (int)iter, method, conv);
			res = -1;
			break;
		case 406:						//symbol button
			bContinue = true;			res = -1;
			break;
		case 1:							//OK button
			if(!bNew){
				if(Dlg->GetText(102, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if((parxy = (char*)realloc(parxy, (cb = strlen(TmpTxt)+2)))) rlp_strcpy(parxy, (int)cb, TmpTxt);
					}
				if(Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if((cmdxy = (char*)realloc(cmdxy, (cb = strlen(TmpTxt)+2)))) rlp_strcpy(cmdxy, (int)cb, TmpTxt);
					}
				ReshapeFormula(&parxy);		ReshapeFormula(&cmdxy);
				dirty = true;
				break;
				}
			if (bDone) break;
			res = 7;						//must show dialog before exit
			//fall through
		case 7:								//Start: do nonlinear regression
			Undo.SetDisp(cdisp);
			if(Dlg->GetCheck(5)) {			//  the function tab must be shown
				if(Dlg->CurrDisp) Dlg->CurrDisp->MouseCursor(MC_WAIT, true);
				Dlg->GetText(401, text1, 100);		Dlg->GetText(403, text2, 100);
				if(Dlg->GetText(102, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if((parxy = (char*)realloc(parxy, (cb = strlen(TmpTxt)+2)))) rlp_strcpy(parxy, (int)cb, TmpTxt);
					}
				if(Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if((cmdxy = (char*)realloc(cmdxy, (cb = strlen(TmpTxt)+2)))) rlp_strcpy(cmdxy, (int)cb, TmpTxt);
					}
				Dlg->GetValue(153, &conv);	Dlg->GetValue(155, &iter);
				ReshapeFormula(&parxy);		ReshapeFormula(&cmdxy);
				do_formula(data, 0L);		//clear any error condition
				ares = do_formula(data, parxy);
				if(ares->type != ET_VALUE) {
					ErrorBox((char*)SCMS_ERR_PARAM);
					bContinue = true;	res = -1;
					break;
					}
				ares = do_formula(data, cmdxy);
				if(ares->type != ET_VALUE) {
					ErrorBox((char*) SCMS_ERR_FORMULA);
					bContinue = true;	res = -1;
					break;
					}
				i = rlp_strcpy(TmpTxt, TMP_TXT_SIZE-2, parxy);	i += rlp_strcpy(TmpTxt+i, TMP_TXT_SIZE-i, (char*)";x=1;");
				rlp_strcpy(TmpTxt+i, TMP_TXT_SIZE-i, cmdxy);	yywarn(0L, true);
				ares = do_formula(data, TmpTxt);
				if((tmp_char = (unsigned char*)yywarn(0L, false))) {
					ErrorBox((char*)tmp_char);
					bContinue = true;	res = -1;
					break;
					}
				//fit function using thes simplex algorithm
				if (method == 2){
					Dlg->Activate(11, 3);			//activate the progress bar
					i = do_simplex(data, (char*)text1, (char*)text2, 0L, &parxy, cmdxy, conv, (int)iter, &nVals, &chi2, true);
					}
				//fit function using thes levenberg marquart algorithm
				else {
					i = do_fitfunc(data, (char*)text1, (char*)text2, 0L, &parxy, cmdxy, conv, (int)iter, &nVals, &chi2, 0L, 0L, 0L, 0, 0L, 0L, 0L);
					}
				if (i < iter) bDone = true;
				if (i < (iter - 1)) {
					Dlg->ShowItem(12, true);			//enable bootstrap
					Dlg->SetText(102, (unsigned char*)parxy);
#ifdef USE_WIN_SECURE
					sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BTFUNC_CHI2, chi2, nVals);
#else
					sprintf(TmpTxt, SDLG_BTFUNC_CHI2, chi2, nVals);
#endif
					Dlg->SetText(204, (unsigned char*)TmpTxt);	Dlg->ShowItem(204, true);
					}
#ifdef USE_WIN_SECURE
				sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BTFUNC_END, method == 2 ? SDLG_FITFUNC_SIMPLEX : SDLG_FITFUNC_LEVMARC, i);
#else
				sprintf(TmpTxt, SDLG_BTFUNC_END, method == 2 ? SDLG_FITFUNC_SIMPLEX : SDLG_FITFUNC_LEVMARC, i);
#endif
				Dlg->SetText(205, (unsigned char*)TmpTxt);	Dlg->ShowItem(205, true);
				Dlg->DoPlot(0L);
				bContinue = true;
				if(res == 7) res = -1;
				if(Dlg->CurrDisp) Dlg->CurrDisp->MouseCursor(MC_ARROW, true);
				}
			else {							//diplay function tab first
				Dlg->SetCheck(5, 0L, true);
				res = -1;
				}
			break;
		case 202:
			method = 1;		res = -1;	break;			//select Levenberg  Marquart
		case 203:
			method = 2;		res = -1;	break;			//select Simplex
			}
		}while (res < 0);
		if (cdisp) {
			Undo.SetDisp(cdisp);
			while (Undo.level() > undo_level)	Undo.Pop(cdisp);
			}
	if(res == 1){						//OK pressed
		//get ranges for x- and y data (again).
		if(Dlg->GetText(401, text1, 100) && (ssXref = (char*)realloc(ssXref, (cb = strlen((char*)text1)+2))))
			rlp_strcpy(ssXref, (int)cb, text1);
		if(Dlg->GetText(403, text2, 100) && (ssYref = (char*)realloc(ssYref, (cb = strlen((char*)text2)+2))))
			rlp_strcpy(ssYref, (int)cb, text2);
		if(bNew) {						//create function
			if(!(rX = new AccRange((char*)text1)) || ! (rY = new AccRange((char*)text2))) return false;
			i = rX->CountItems();	maxiter = int(iter);
			if(Dlg->GetCheck(404)) Symbols = (Symbol**)calloc(i+2, sizeof(Symbol*));
			x_info = rlp_strdup(rX->RangeDesc(data, 1));
			y_info = rlp_strdup(rY->RangeDesc(data, 1));
			x1 = HUGE_VAL;	x2 = -HUGE_VAL;	cs = 0;
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			do {
				if(data->GetValue(j, i, &tmp) && data->GetValue(l, k, &tmpy)){
					if(tmp < x1) x1 = tmp;
					if(tmp > x2) x2 = tmp;
					if(Symbols) Symbols[cs++] = new Symbol(this, data, tmp, tmpy, SYM_CIRCLE, i, j, k, l);
					nPoints = cs;					CheckBounds(tmp, tmpy);
					}
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			delete rX;	delete rY;
			if(x1 >= x2 || !(dl = new Function(this, data, (char*)"Fitted function"))){
				CloseDlgWnd(hDlg);
				delete Dlg;
				return false;
				}
			xstep = (x2 - x1)/100.0;
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&Line, 0);
			dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
			dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
			dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
			dl->Command(CMD_AUTOSCALE, 0L, 0L);		CheckBounds(dl->Bounds.Xmin, dl->Bounds.Ymin);
			if (Symbols) {
				SetSize(SIZE_SYMBOL, PrevSym->GetSize(SIZE_SYMBOL));			SetSize(SIZE_SYM_LINE, PrevSym->GetSize(SIZE_SYM_LINE));
				SetColor(COL_SYM_LINE, PrevSym->GetColor(COL_SYM_LINE));		SetColor(COL_SYM_FILL, PrevSym->GetColor(COL_SYM_FILL));
				Command(CMD_SYM_TYPE, (void*)(&PrevSym->type), 0L);
				}
			CheckBounds(dl->Bounds.Xmax, dl->Bounds.Ymax);
			Command(CMD_ENDDIALOG, 0L, 0L);			bRet = true;
			}
		else {							//edit existing function
			Dlg->GetValue(502, &n_x1);		Dlg->GetValue(504, &n_x2);
			Dlg->GetValue(506, &n_xstep);
			undo_flags = CheckNewFloat(&x1, o_x1, n_x1, this, undo_flags);
			undo_flags = CheckNewFloat(&x2, o_x2, n_x2, this, undo_flags);
			undo_flags = CheckNewFloat(&xstep, o_xstep, n_xstep, this, undo_flags);
			if(undo_flags){
				dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
				dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
				dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
				dl->Update(0L, UNDO_CONTINUE);
				}
			undo_flags = CheckNewString((unsigned char**)&parxy, (unsigned char*)o_parxy, (unsigned char*)parxy, this, undo_flags);
			undo_flags = CheckNewString((unsigned char**)&cmdxy, (unsigned char*)o_cmdxy, (unsigned char*)cmdxy, this, undo_flags);
			if(undo_flags & UNDO_CONTINUE) {
				Undo.ValInt(parent, (int*)&dirty, undo_flags);
				Command(CMD_MRK_DIRTY, 0L, 0L);
				}
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			if(cmpLineDEF(&Line, &newLine)) {
				Undo.Line(parent, &Line, undo_flags);	undo_flags |= UNDO_CONTINUE;
				memcpy(&Line, &newLine, sizeof(LineDEF));
				}
			bRet = (undo_flags & UNDO_CONTINUE) != 0;
			Command(CMD_ENDDIALOG, 0L, 0L);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;						delete PrevSym;
	if(o_parxy) free(o_parxy);
	if(o_cmdxy) free(o_cmdxy);
	free(tab1);		free(tab2);		free(tab3);			free(FuncDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a new normal quantile plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *NormQuantDlg_Tmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,148,10,45,12\n"
".,.,,,PUSHBUTTON,-2,148,25,45,12\n"
".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,10,100,ISPARENT | CHECKED,SHEET,1,5,10,130,80\n"
"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
"100,+,,,LTEXT,2,10,30,60,8\n"
".,.,,,RANGEINPUT,-15,20,45,100,10\n"
".,.,,,LTEXT,3,40,60,60,8\n"
".,,,LASTOBJ | OWNDIALOG,SYMBUTT,4,80,60,20,10";

bool
NormQuant::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	DlgInfo *QuantDlg;
	void *dyndata[] = { (void*)tab1, (void*)SDLG_NQUANT_RANGE, (void*)SDLG_NQUANT_SYMS, (void*)&sy };
	DlgRoot *Dlg;
	void *hDlg;
	int res;
	char *mrk;
	bool bContinue = false, bRet = false;

	if(!parent || !data) return false;
	if(!(QuantDlg = CompileDialog(NormQuantDlg_Tmpl, dyndata))) return false;
	if (!sy) {
		sy = new Symbol(this, data, 0.0, 0.0, SYM_CIRCLE);
		sy->SetSize(SIZE_SYMBOL, NiceValue(defs.GetSize(SIZE_SYMBOL)*0.75));
		}
	if(data->Command(CMD_GETMARK, &mrk, 0L))rlp_strcpy(TmpTxt, TMP_TXT_SIZE, mrk);
	else UseRangeMark(data, 1, TmpTxt);
	if(!(Dlg = new DlgRoot(QuantDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_NQUANT_HD1, 50, 50, 420, 240, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
			}
		}while (res < 0);
	if(res == 1 && Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)){
		ssRef = rlp_strdup(TmpTxt);
		bRet = ProcessData();
		}
	CloseDlgWnd(hDlg);	delete Dlg;		free(QuantDlg);		free(tab1);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Contour Plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *ContourPlot_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,142,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,142,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,131,100\n"
	".,.,200,ISPARENT,SHEET,2,5,10,131,100\n"
	".,10,300,TOUCHEXIT | ISPARENT,SHEET,10,5,10,131,100\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-31,10,30,60,8\n"
	".,.,,,RANGEINPUT,-15,20,40,100,10\n"
	".,.,,,LTEXT,-32,10,55,60,8\n"
	".,.,,,RANGEINPUT,-16,20,65,100,10\n"
	".,.,,,LTEXT,-33,10,80,60,8\n"
	".,,,,RANGEINPUT,-17,20,90,100,10\n"
	"200,,201,CHECKED,GROUPBOX,3,10,30,121,70\n"
	".,+,,HICOL | CHECKED,RADIO1,4,20,38,60,9\n"
	".,.,,HICOL,RADIO1,5,20,48,60,9\n"
	".,.,,HICOL,RADIO1,6,20,58,60,9\n"
	".,.,,HICOL,RADIO1,7,20,68,60,9\n"
	".,.,,,EDVAL1,8,50,78,30,10\n"
	".,,,,RTEXT,9,15,79,33,8\n"
	"300,310,301,CHECKED,GROUPBOX,11,10,30,121,55\n"
	".,+,,HICOL | CHECKED,RADIO1,12,20,38,60,9\n"
	".,.,,HICOL,RADIO1,13,20,48,60,9\n"
	".,.,,HICOL,RADIO1,14,20,58,60,9\n"
	".,,,HICOL,RADIO1,15,20,68,60,9\n"
	"310,,,HICOL | LASTOBJ,CHECKBOX,16,20,90,60,9";

bool
ContourPlot::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SYMBOLS);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_CONTOUR_SRECT, (void*)SDLG_CONTOUR_ZMIN,
		(void*)SDLG_CONTOUR_ZMAX, (void*)SDLG_CONTOUR_ZMEAN, (void*)SDLG_CONTOUR_ZUSER, (void*)&sr_zval, 
		(void*)SDLG_CONTOUR_ZEQ, (void*)tab3, (void*)SDLG_CONTOUR_DOSYM, (void*)SDLG_CONTOUR_NOSYM, 
		(void*)SDLG_CONTOUR_DOMIN, (void*)SDLG_CONTOUR_DOMAX, (void*)SDLG_CONTOUR_DOSRC, 
		(void*)SDLG_CONTOUR_SLABEL };
	DlgInfo *DlgInf;
	DlgRoot *Dlg;
	void *hDlg;
	int res;
	bool bRet = false, bContinue = false, bSymbols = false;

	if(!data || !parent)return false;
	if(!(DlgInf = CompileDialog(ContourPlot_Tmpl, dyndata)))return false;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100, TmpTxt+200);
	if(!(Dlg = new DlgRoot(DlgInf, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_CONTOUR_HD1, 50, 50, 408, 280, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 1:
			if (res > 0 && !bSymbols){
				Dlg->SetCheck(6, 0L, bSymbols = true);
				res = -1;		break;
				}
			res = -1;		bContinue = false;
			if(Dlg->GetCheck(202))flags = 0x01;
			else if(Dlg->GetCheck(203))flags = 0x02;
			else if(Dlg->GetCheck(204)){
				flags = 0x03;	Dlg->GetValue(205, &sr_zval);
				}
			else flags = 0x00;
			if(Dlg->GetCheck(302)) flags |= 0x10;
			else if(Dlg->GetCheck(303)) flags |= 0x20;
			else if(Dlg->GetCheck(304)) flags |= 0x30;
			if(Dlg->GetCheck(310)) flags |= 0x40;
			if(Dlg->GetText(101, (unsigned char*)TmpTxt, 100) && Dlg->GetText(103, (unsigned char*)(TmpTxt+100), 100) 
				&& Dlg->GetText(105, (unsigned char*)(TmpTxt+200), 100)){
				if(LoadData(TmpTxt, TmpTxt+100, TmpTxt+200)) {
					ssRefX = rlp_strdup(TmpTxt);				ssRefY =rlp_strdup(TmpTxt+100);
					ssRefZ = rlp_strdup(TmpTxt+200);
					res = 1;		bRet = true;
					}
				}
			if(res != 1) {
				bContinue = true;
				ErrorBox((char*)SCMS_NODATA4PLOT);
				}
			break;
		case 6:							//the symbols tab
			bSymbols = true;
			res = -1;
			}
		}while (res <0);
	CloseDlgWnd(hDlg);		delete Dlg;			free(DlgInf);
	free(tab1);			free(tab2);				free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Add axis to three dimensional graph
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *AddAxis3D_Tmpl = (char*)
		"1,+,,DEFAULT, PUSHBUTTON,-1,148,10,45,12\n"
		".,.,,,PUSHBUTTON,-2,148,25,45,12\n"
		".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
		"4,+,50,TOUCHEXIT | ISPARENT,SHEET,1,5,10,130,148\n"
		".,.,200,ISPARENT | CHECKED,SHEET,2,5,10,130,148\n"
		".,20,300,ISPARENT,SHEET,3,5,10,130,148\n"
		"20,,,NOSELECT,ODB,22,142,65,45,45\n"
		"50,+,100,ISPARENT | CHECKED,GROUPBOX,4,10,30,120,36\n"
		".,120,,HICOL,CHECKBOX,5,17,37,80,8\n"
		"100,+,,,RTEXT,6,10,51,35,8\n"
		".,.,,,EDVAL1,7,48,51,32,10\n"
		".,.,,,CTEXT,8,81,51,11,8\n"
		".,,,,EDVAL1,9,93,51,32,10\n"
		"120,,121,ISPARENT | CHECKED,GROUPBOX,10,10,72,120,20\n"
		"121,+,,,RTEXT,11,10,77,25,8\n"
		".,.,,,EDVAL1,12,37,77,25,10\n"
		".,.,,,LTEXT,-3,63,77,10,8\n"
		".,.,,,RTEXT,-11,73,77,25,8\n"
		".,130,,OWNDIALOG,COLBUTT,14,100,77,25,10\n"
		"130,+,,ISPARENT | CHECKED,GROUPBOX,15,10,98,120,20\n"
		".,,,,EDTEXT,0,15,103,110,10\n"
		"200,+,,,LTEXT,16,10,23,70,9\n"
		".,.,,CHECKED | EXRADIO,ODB,17,20,35,,\n"
		".,.,,EXRADIO,ODB,17,45,35,,\n"
		".,.,,EXRADIO,ODB,17,70,35,,\n"
		".,.,,EXRADIO,ODB,17,95,35,,\n"
		".,.,,EXRADIO,ODB,17,20,60,,\n"
		".,.,,EXRADIO,ODB,17,45,60,,\n"
		".,.,,EXRADIO,ODB,17,70,60,,\n"
		".,.,,EXRADIO,ODB,17,95,60,,\n"
		".,.,,EXRADIO,ODB,17,20,85,,\n"
		".,.,,EXRADIO,ODB,17,45,85,,\n"
		".,.,,EXRADIO,ODB,17,70,85,,\n"
		".,.,,EXRADIO,ODB,17,95,85,,\n"
		".,.,,,LTEXT,-12,20,120,70,9\n"
		".,.,,,LTEXT,-13,20,132,70,9\n"
		".,.,,,LTEXT,-14,20,144,70,9\n"
		".,.,250,ISPARENT | CHECKED,GROUP,,,,,\n"
		".,.,260,HIDDEN | ISPARENT | CHECKED,GROUP,,,,,\n"
		".,,270,HIDDEN |ISPARENT | CHECKED,GROUP,,,,,\n"
		"250,+,,,LTEXT,19,10,110,70,9\n"
		".,.,,,EDVAL1,0,58,120,32,10\n"
		".,.,,,LTEXT,-3,92,120,20,9\n"
		".,.,,,EDVAL1,0,43,132,32,10\n"
		".,.,,,CTEXT,-7,75,132,7,9\n"
		".,.,,,EDVAL1,0,82,132,32,10\n"
		".,.,,,LTEXT,-3,116,132,20,9\n"
		".,268,,,EDVAL1,0,58,144,32,10\n"
		"260,+,,,LTEXT,20,10,110,70,9\n"
		".,.,,,EDVAL1,0,43,120,32,10\n"
		".,.,,,CTEXT,-7,75,120,7,9\n"
		".,.,,,EDVAL1,0,82,120,32,10\n"
		".,.,,,LTEXT,-3,116,120,20,9\n"
		".,.,,,EDVAL1,0,58,132,32,10\n"
		".,.,,,LTEXT,-3,92,132,20,9\n"
		".,.,,,EDVAL1,0,58,144,32,10\n"
		"268,,,,LTEXT,-3,92,144,20,9\n"
		"270,+,,,LTEXT,21,10,110,70,9\n"
		".,.,,,EDVAL1,0,58,120,32,10\n"
		".,.,,,LTEXT,-3,92,120,20,9\n"
		".,.,,,EDVAL1,0,58,132,32,10\n"
		".,.,,,LTEXT,-3,92,132,20,9\n"
		".,.,,,EDVAL1,0,43,144,32,10\n"
		".,.,,,CTEXT,-7,75,144,7,9\n"
		".,.,,,EDVAL1,0,82,144,32,10\n"
		".,,,,LTEXT,-3,116,144,20,9\n"
		"300,,,LASTOBJ | NOSELECT,ODB,18,15,30,110,140";
bool
Plot3D::AddAxis()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_AXES);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_PLOT);
	AxisDEF axis, ax_def[12], *caxdef;
	double sizAxLine = DefSize(SIZE_AXIS_LINE);
	DWORD colAxis = 0x0;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_AXIS3D_SCALE, (void*)SDLG_AXIS3D_AUTOS,
		(void*)SDLG_AXIS3D_FROM, (void*)&axis.min, (void*)SDLG_AXIS3D_TO, (void*)&axis.max, (void*)SDLG_AXIS3D_LINE,
		(void*)SDLG_AXIS3D_LIW, (void*)&sizAxLine, (void*)0L, (void *)&colAxis, (void*)SDLG_AXIS3D_LABEL,
		(void*)SDLG_AXIS3D_TEMPL, (void*)(OD_NewAxisTempl3D), (void*)OD_axisplot, (void*)SDLG_AXIS3D_YAT,
		(void*)SDLG_AXIS3D_XAT, (void*)SDLG_AXIS3D_ZAT, (void*)(OD_AxisDesc3D)};
	DlgInfo *NewAxisDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int j, res, currTempl = 201, ax_type = 2, tick_type = 2;
	double tlb_dx, tlb_dy, lb_x, lb_y;
	TextDEF label_def, tlbdef;
	anyOutput *cdisp = Undo.Cdisp();
	Axis *the_new, **tmpAxes;
	bool bRet = false;
	char **names;
	GraphObj **somePlots;
	Label *label;

	if(!Axes || nAxes < 3 || !(NewAxisDlg = CompileDialog(AddAxis3D_Tmpl, dyndata))) return false;
	lb_y = 0.0;		lb_x = DefSize(SIZE_AXIS_TICKS)*4.0;
	tlb_dy = 0.0;	tlb_dx = -DefSize(SIZE_AXIS_TICKS)*2.0;
	tlbdef.ColTxt = colAxis;				tlbdef.ColBg = 0x00ffffffL;
	tlbdef.RotBL = tlbdef.RotCHAR = 0.0f;	tlbdef.fSize = DefSize(SIZE_TICK_LABELS);
	tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
	tlbdef.Style = TXS_NORMAL;				tlbdef.Mode = TXM_TRANSPARENT;
	tlbdef.Font = FONT_HELVETICA;			tlbdef.text = 0L;
	if(!(names = (char**)calloc(nscp+2, sizeof(char*))))return false;
	if(!(somePlots = (GraphObj**)calloc(nscp+2, sizeof(GraphObj*))))return false;
	for(i = 0; i < 12; i++) {
		ax_def[i].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK;
		ax_def[i].owner = 0L;		ax_def[i].breaks = 0L;
		ax_def[i].Center.fx = ax_def[i].Center.fy = ax_def[i].Radius = 0.0;
		ax_def[i].nBreaks = 0L;
		}
	//y-axes
	ax_def[0].min = ax_def[1].min = ax_def[2].min = ax_def[3].min = (caxdef = Axes[1]->GetAxis())->min;
	ax_def[0].max = ax_def[1].max = ax_def[2].max = ax_def[3].max = caxdef->max;
	ax_def[0].Start = ax_def[1].Start = ax_def[2].Start = ax_def[3].Start = caxdef->Start;
	ax_def[0].Step = ax_def[1].Step = ax_def[2].Step = ax_def[3].Step = caxdef->Step;
	ax_def[0].loc[0].fy  = ax_def[1].loc[0].fy = ax_def[2].loc[0].fy = ax_def[3].loc[0].fy = caxdef->loc[0].fy;
	ax_def[0].loc[1].fy  = ax_def[1].loc[1].fy = ax_def[2].loc[1].fy = ax_def[3].loc[1].fy = caxdef->loc[1].fy;
	ax_def[0].loc[0].fx = ax_def[0].loc[1].fx = ax_def[3].loc[0].fx = ax_def[3].loc[1].fx = cu1.fx;
	ax_def[1].loc[0].fx = ax_def[1].loc[1].fx = ax_def[2].loc[0].fx = ax_def[2].loc[1].fx = cu2.fx;
	ax_def[0].loc[0].fz = ax_def[0].loc[1].fz = ax_def[1].loc[0].fz = ax_def[1].loc[1].fz = cu2.fz;
	ax_def[2].loc[0].fz = ax_def[2].loc[1].fz = ax_def[3].loc[0].fz = ax_def[3].loc[1].fz = cu1.fz;
	ax_def[1].flags = ax_def[2].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_POSTICKS;
	ax_def[0].flags = ax_def[3].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_NEGTICKS;
	//x-axes
	ax_def[4].min = ax_def[6].min = ax_def[8].min = ax_def[10].min = (caxdef = Axes[0]->GetAxis())->min;
	ax_def[4].max = ax_def[6].max = ax_def[8].max = ax_def[10].max = caxdef->max;
	ax_def[4].Start = ax_def[6].Start = ax_def[8].Start = ax_def[10].Start = caxdef->Start;
	ax_def[4].Step = ax_def[6].Step = ax_def[8].Step = ax_def[10].Step = caxdef->Step;
	ax_def[4].loc[0].fx  = ax_def[6].loc[0].fx = ax_def[8].loc[0].fx = ax_def[10].loc[0].fx = caxdef->loc[0].fx;
	ax_def[4].loc[1].fx  = ax_def[6].loc[1].fx = ax_def[8].loc[1].fx = ax_def[10].loc[1].fx = caxdef->loc[1].fx;
	ax_def[4].loc[0].fy = ax_def[4].loc[1].fy = ax_def[6].loc[0].fy = ax_def[6].loc[1].fy = cu1.fy;
	ax_def[8].loc[0].fy = ax_def[8].loc[1].fy = ax_def[10].loc[0].fy = ax_def[10].loc[1].fy = cu2.fy;
	ax_def[4].loc[0].fz = ax_def[4].loc[1].fz = ax_def[8].loc[0].fz = ax_def[8].loc[1].fz = cu2.fz;
	ax_def[6].loc[0].fz = ax_def[6].loc[1].fz = ax_def[10].loc[0].fz = ax_def[10].loc[1].fz = cu1.fz;
	ax_def[4].flags = ax_def[6].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_NEGTICKS;
	ax_def[8].flags = ax_def[10].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_POSTICKS;
	//z-axes
	ax_def[5].min = ax_def[7].min = ax_def[9].min = ax_def[11].min = (caxdef = Axes[2]->GetAxis())->min;
	ax_def[5].max = ax_def[7].max = ax_def[9].max = ax_def[11].max = caxdef->max;
	ax_def[5].Start = ax_def[7].Start = ax_def[9].Start = ax_def[11].Start = caxdef->Start;
	ax_def[5].Step = ax_def[7].Step = ax_def[9].Step = ax_def[11].Step = caxdef->Step;
	ax_def[5].loc[0].fz  = ax_def[7].loc[0].fz = ax_def[9].loc[0].fz = ax_def[11].loc[0].fz = caxdef->loc[0].fz;
	ax_def[5].loc[1].fz  = ax_def[7].loc[1].fz = ax_def[9].loc[1].fz = ax_def[11].loc[1].fz = caxdef->loc[1].fz;
	ax_def[5].loc[0].fx = ax_def[5].loc[1].fx = ax_def[9].loc[0].fx = ax_def[9].loc[1].fx = cu2.fx;
	ax_def[7].loc[0].fx = ax_def[7].loc[1].fx = ax_def[11].loc[0].fx = ax_def[11].loc[1].fx = cu1.fx;
	ax_def[5].loc[0].fy = ax_def[5].loc[1].fy = ax_def[7].loc[0].fy = ax_def[7].loc[1].fy = cu1.fy;
	ax_def[9].loc[0].fy = ax_def[9].loc[1].fy = ax_def[11].loc[0].fy = ax_def[11].loc[1].fy = cu2.fy;
	ax_def[5].flags = ax_def[9].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_POSTICKS;
	ax_def[7].flags = ax_def[11].flags = AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_NEGTICKS;
	//the default axis is the first
	memcpy(&axis, &ax_def[0], sizeof(AxisDEF));
	if ((names[0] = (char*)malloc(10))) rlp_strcpy(names[0], 10, (char*)SDLG_AXIS3D_NONE);
	for(i = 0, j = 1; i < nscp; i++) {
		if(Sc_Plots[i] && Sc_Plots[i]->name){
			names[j] = rlp_strdup(Sc_Plots[i]->name);
			somePlots[j++] = Sc_Plots[i];
			}
		}
	OD_axisplot(OD_ACCEPT, 0L, 0L, 0L, names, 0);
	if(!(Dlg = new DlgRoot(NewAxisDlg, data)))return false;
	Dlg->SetValue(251, ax_def[0].loc[0].fx);		Dlg->SetValue(253, ax_def[0].loc[0].fy);
	Dlg->SetValue(255, ax_def[0].loc[1].fy);		Dlg->SetValue(257, ax_def[0].loc[0].fz);
	if(!nscp){						//must be root plot3d to link to plot
		Dlg->ShowItem(6, false);
		}
	hDlg = CreateDlgWnd((char*)SDLG_AXIS3D_HD1, 50, 50, 420, 380, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res){
		case 4:											//the axis sheet
			res = -1;
			break;
		//axis templates
		case 201:	case 202:	case 203:	case 204:	case 205:	case 206:
		case 207:	case 208:	case 209:	case 210:	case 211:	case 212:
			i = currTempl-201;
			switch(currTempl){
				case 201:	case 202:	case 203:	case 204:		//previous is y-template
					Dlg->GetValue(251, &axis.loc[0].fx);	Dlg->GetValue(253, &axis.loc[0].fy);
					Dlg->GetValue(255, &axis.loc[1].fy);	Dlg->GetValue(257, &axis.loc[0].fz);
					axis.loc[1].fx = axis.loc[0].fx;		axis.loc[1].fz = axis.loc[0].fz;
					memcpy(&ax_def[i], &axis, sizeof(AxisDEF));
					break;
				case 205:	case 207:	case 209:	case 211:		//previous is x-template
					Dlg->GetValue(261, &axis.loc[0].fx);	Dlg->GetValue(263, &axis.loc[1].fx);
					Dlg->GetValue(265, &axis.loc[0].fy);	Dlg->GetValue(267, &axis.loc[0].fz);
					axis.loc[1].fy = axis.loc[0].fy;		axis.loc[1].fz = axis.loc[0].fz;
					memcpy(&ax_def[i], &axis, sizeof(AxisDEF));
					break;
				case 206:	case 208:	case 210:	case 212:		//previous is z-template
					Dlg->GetValue(271, &axis.loc[0].fx);	Dlg->GetValue(273, &axis.loc[0].fy);
					Dlg->GetValue(275, &axis.loc[0].fz);	Dlg->GetValue(277, &axis.loc[1].fz);
					axis.loc[1].fx = axis.loc[0].fx;		axis.loc[1].fy = axis.loc[0].fy;
					memcpy(&ax_def[i], &axis, sizeof(AxisDEF));
					break;
				}
			i = res-201;
			switch (res) {
			case 201:	case 202:	case 203:	case 204:			//y-template
				Dlg->ShowItem(216, true);		Dlg->ShowItem(217, false);
				Dlg->ShowItem(218, false);
				Dlg->SetValue(251, ax_def[i].loc[0].fx);	Dlg->SetValue(253, ax_def[i].loc[0].fy);
				Dlg->SetValue(255, ax_def[i].loc[1].fy);	Dlg->SetValue(257, ax_def[i].loc[0].fz);
				ax_type = tick_type = 2;	tlb_dy = 0.0;
				if(res == 202 || res == 203){
					tlb_dx = DefSize(SIZE_AXIS_TICKS)*2.0;
					tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
					}
				else {
					tlb_dx = -DefSize(SIZE_AXIS_TICKS)*2.0;
					tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
					}
				break;
			case 205:	case 207:	case 209:	case 211:			//x-template
				Dlg->ShowItem(216, false);		Dlg->ShowItem(217, true);
				Dlg->ShowItem(218, false);
				Dlg->SetValue(261, ax_def[i].loc[0].fx);	Dlg->SetValue(263, ax_def[i].loc[1].fx);
				Dlg->SetValue(265, ax_def[i].loc[0].fy);	Dlg->SetValue(267, ax_def[i].loc[0].fz);
				ax_type = 1;	tick_type = 3; tlb_dx = 0;
				if(res == 205 || res == 207){
					tlb_dy = DefSize(SIZE_AXIS_TICKS)*2.0;
					tlbdef.Align = TXA_VTOP | TXA_HCENTER;
					}
				else {
					tlb_dy = -DefSize(SIZE_AXIS_TICKS)*2.0;
					tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
					}
				break;
			case 206:	case 208:	case 210:	case 212:			//z-template
				Dlg->ShowItem(216, false);		Dlg->ShowItem(217, false);
				Dlg->ShowItem(218, true);
				Dlg->SetValue(271, ax_def[i].loc[0].fx);	Dlg->SetValue(273, ax_def[i].loc[0].fy);
				Dlg->SetValue(275, ax_def[i].loc[0].fz);	Dlg->SetValue(277, ax_def[i].loc[1].fz);
				ax_type = 3;	tick_type = 2;	tlb_dy = 0;
				if(res == 206 || res == 210){
					tlb_dx = DefSize(SIZE_AXIS_TICKS)*2.0;
					tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
					}
				else {
					tlb_dx = -DefSize(SIZE_AXIS_TICKS)*2.0;
					tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
					}
				break;
				}
			memcpy(&axis, &ax_def[res-201], sizeof(AxisDEF));
			currTempl = res;
			Dlg->DoPlot(0L);
			res = -1;
			break;
			}
		}while (res < 0);
	if(res == 1) {
		Undo.SetDisp(cdisp);
		Dlg->GetValue(122, &sizAxLine);				Dlg->GetColor(125, &colAxis);
		Dlg->GetValue(101, &axis.min);				Dlg->GetValue(103, &axis.max);
		axis.Start = axis.min;						axis.Center.fx = axis.Center.fy = 0.0;
		axis.nBreaks = 0;			axis.breaks = 0L;		axis.Radius = 0.0;	
		tlbdef.ColTxt =	colAxis;
		label_def.ColTxt = colAxis;					label_def.ColBg = 0x00ffffffL;
		label_def.fSize = DefSize(SIZE_TICK_LABELS)*1.2f;	label_def.RotBL = 0.0f;
		label_def.RotCHAR = 0.0f;								label_def.iSize = 0;
		label_def.Align = TXA_VTOP | TXA_HCENTER;		label_def.Mode = TXM_TRANSPARENT;
		label_def.Style = TXS_NORMAL;	label_def.Font = FONT_HELVETICA;
		if(Dlg->GetCheck(51)) axis.flags |= AXIS_AUTOSCALE;
		the_new = new Axis(this, data, &axis, axis.flags);
		if (the_new){
			the_new->SetSize(SIZE_TLB_YDIST, tlb_dy);	the_new->SetSize(SIZE_TLB_XDIST, tlb_dx); 
			the_new->SetSize(SIZE_AXIS_LINE, sizAxLine);
			the_new->SetColor(COL_AXIS, colAxis);
			the_new->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			the_new->SetSize(SIZE_LB_XDIST, lb_x);		the_new->SetSize(SIZE_LB_YDIST, lb_y);
			the_new->type = ax_type;
			the_new->Command(CMD_TICK_TYPE, &tick_type, 0L);
			if(Dlg->GetText(131, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) label_def.text = (unsigned char*)TmpTxt;
			else label_def.text = 0L;
			if((label = new Label(Axes[0], data, (axis.loc[0].fx + axis.loc[1].fx)/2.0,
				(axis.loc[0].fy + axis.loc[1].fy)/2.0, &label_def, 
				label_def.RotBL < 45.0 ? LB_Y_PARENT : LB_X_PARENT, 0L))){
				label->SetSize(SIZE_LB_XDIST, lb_x);	label->SetSize(SIZE_LB_YDIST, lb_y); 
				if(the_new->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
				else DeleteGO(label);
				}
			for(i = 0; i < nAxes && Axes[i]; i++);
			if(i < nAxes) {
				Undo.SetGO(this, (GraphObj**)(&Axes[i]), the_new, 0L);
				bRet = true;
				}
			else {
				tmpAxes = (Axis**)calloc(nAxes + 2, sizeof(Axis*));
				if (tmpAxes){
					memcpy(tmpAxes, Axes, nAxes * sizeof(Axis*));
					Undo.ListGOmoved((GraphObj**)Axes, (GraphObj**)tmpAxes, nAxes);
					Undo.SetGO(this, (GraphObj**)(&tmpAxes[nAxes]), the_new, 0L);
					free(Axes);			Axes = tmpAxes;
					i = nAxes++;		bRet = true;
					}
				}
			CurrAxes = Axes;	NumCurrAxes = (int)nAxes;
			if(bRet) {
				OD_axisplot(OD_ACCEPT, 0L, 0L, (anyOutput*) &res, 0L, 0);
				if(res && i) somePlots[res]->Command(CMD_USEAXIS, &i, 0L);
				}
			}
		}
	CloseDlgWnd(hDlg);		delete Dlg;			free(NewAxisDlg);
	if(names) {
		for(j = 0; names[j]; j++) if(names[j]) free(names[j]);
		free(names);
		}
	if(somePlots) free(somePlots);
	free(tab1);		free(tab2);		free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Add 3D plot to graph
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *AddPlot3Dtmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,-1,150,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,150,25,45,12\n"
	"3,,560,ISPARENT | CHECKED,GROUPBOX,1,5,10,140,70\n"
	"560,+,,EXRADIO | CHECKED,ODB,2,12,20,,\n"
	".,.,,EXRADIO,ODB,2,37,20,,\n"
	".,.,,EXRADIO,ODB,2,62,20,,\n"
	".,.,,EXRADIO,ODB,2,87,20,,\n"
	".,.,,EXRADIO,ODB,2,112,20,,\n"
	".,.,,EXRADIO,ODB,2,12,45,,\n"
	".,.,,EXRADIO,ODB,2,37,45,,\n"
	".,,,LASTOBJ | EXRADIO,ODB,2,62,45,,";

bool
Plot3D::AddPlot(int)
{
	void *dyndata[] = { (void *)SDLG_ADDPLOT_3D_TEMPL, (void*)(OD_PlotTempl) };
	DlgInfo *PlotsDlg = CompileDialog(AddPlot3Dtmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, cSel = 560;
	bool bRet = false;
	Plot *p;

	if(!(Dlg = new DlgRoot(PlotsDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_ADDPLOT_3D_HD1, 50, 50, 420, 224, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 560:	case 561:	case 562:	case 563:	case 564:
		case 565:	case 566:	case 567:
			if(res == cSel) res = 1;
			else {
				cSel = res;		res = -1;
				}
			break;
			}
		}while (res < 0);
	if(res == 1){						//OK pressed
		switch (cSel) {
		case 560:		p = new Scatt3D(this, data, 0x01);			break;
		case 561:		p = new Scatt3D(this, data, 0x02);			break;
		case 562:		p = new Scatt3D(this, data, 0x04);			break;
		case 563:		p = new BubblePlot3D(this, data);			break;
		case 564:		p = new Scatt3D(this, data, 0x2000);		break;
		case 565:		p = new Func3D(this, data);					break;
		case 566:		p = new FitFunc3D(this, data);				break;
		case 567:		p = new Scatt3D(this, data, 0x4000);		break;
		default:		p = 0L;										break;
			}
		CloseDlgWnd(hDlg);
		if(p && p->PropertyDlg()) {
			if(!(bRet = Command(CMD_DROP_PLOT, p, (anyOutput *)NULL))) delete p;
			}
		else if(p) delete p;
		}
	else CloseDlgWnd(hDlg);
	delete Dlg;		free(PlotsDlg);
	return bRet;
}

//create rotation matrix from vector
static void vec2mat(double rotX, double rotY, double rotZ, double rot_m[3][3])
{
	double x, y, z, vl, sx, csx, sy, csy, sz, csz;
	double mr[3][3], ma[3][3], mb[3][3];

	//normalize vector(x,y,z)
	vl = sqrt((rotX * rotX) + (rotY * rotY) + (rotZ * rotZ));
	x = rotX / vl;		y = rotY / vl;		z = rotZ / vl;
	//set up new rotation matrix
	sx = sin(_PI *x / 2.0);					csx = cos(_PI *x / 2.0);
	sy = sin(_PI *y / 2.0);					csy = cos(_PI *y / 2.0);
	sz = sin(_PI *z / 2.0);					csz = cos(_PI *z / 2.0);
	ma[0][0] = ma[1][1] = csz;				ma[0][1] = -sz;		//rotate z
	ma[1][0] = sz;							ma[2][2] = 1.0;
	ma[2][0] = ma[2][1] = ma[0][2] = ma[1][2] = 0.0;
	mb[0][0] = mb[2][2] = csy;				mb[0][2] = sz;		//rotate y
	mb[2][0] = -sy;							mb[1][1] = 1.0;
	mb[0][1] = mb[1][0] = mb[1][2] = mb[2][1] = 0.0;
	MatMul(ma, mb, mr);
	ma[1][1] = ma[2][2] = csx;				ma[1][2] = -sx;		//rotate x
	ma[2][1] = sx;							ma[0][0] = 1.0;
	ma[0][1] = ma[0][2] = ma[1][0] = ma[2][0] = 0.0;
	MatMul(mr, ma, rot_m);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Properties of 3D graph
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *Plot3D_DlgTmpl = (char*)
"1,+,,DEFAULT, PUSHBUTTON,-1,163,10,45,12\n"
".,.,,,PUSHBUTTON,-2,163,25,45,12\n"
".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,145,95\n"
".,60,200,ISPARENT,SHEET,2,5,10,145,95\n"
"60,+,,,ICON,8,10,79,20,20\n"
".,.,,,LTEXT,9,35,81,100,6\n"
".,.,,,LTEXT,10,35,87,100,6\n"
".,,,HIDDEN,LTEXT,15,30,85,150,8\n"
"100,+,,,RTEXT,4,18,30,45,8\n"
".,.,,,EDVAL1,5,65,30,35,10\n"
".,.,,,LTEXT,3,102,30,20,8\n"
".,.,,,RTEXT,-5,18,45,45,8\n"
".,.,,,EDVAL1,6,65,45,35,10\n"
".,.,,,LTEXT,3,102,45,20,8\n"
".,.,,,RTEXT,-6,18,60,45,8\n"
".,.,,,EDVAL1,7,65,60,35,10\n"
".,,,,LTEXT,3,102,60,20,8\n"
"200,+,,,RTEXT,11,18,30,45,8\n"
".,.,,,EDVAL1,12,65,30,35,10\n"
".,.,,,LTEXT,3,102,30,20,8\n"
".,.,,,RTEXT,13,18,45,45,8\n"
".,.,,,EDVAL1,14,65,45,35,10\n"
".,,,LASTOBJ,LTEXT,3,102,45,20,8";

bool
Plot3D::Configure()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DETAILS);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_LIGHT);
	double xd, yd, zd, lsxd, lsyd;
	int icon = ICO_INFO;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)SDLG_PLOT_3D_DEG, (void*)SDLG_PLOT_3D_XROT,
		(void*)&xd, (void*)&yd, (void*)&zd, (void*)&icon, (void*)SDLG_PLOT3D_ROT1, (void*)SDLG_PLOT3D_ROT2,
		(void*)SDLG_PLOT_3D_AZI, (void*)&lsxd, (void*)SDLG_PLOT_3D_ELE, (void*)&lsyd, (void*)SDLG_PLOT3D_VEC };

	double rotM[3][3];			//rotation matrix
	double csphi, siphi, phi, x, y, z;
	double vl, ox, oy, oz, nx, ny, nz, olsx, olsy, nlsx, nlsy;
	DWORD undo_flags = 0;
	int res;
	bool bRet = false;
	anyOutput *o = getDisp();
	DlgInfo *Plot3D_Dlg = CompileDialog(Plot3D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;

	//set up rotation matrix from quaternion
	//see: Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
	//M.E. Pique: Rotation Tools
	// ISBN 0-12-286165-5, p. 466
	if (parent && parent->Id == GO_GRAPH) o = ((Graph*)parent)->getDisp();
	if (!o) return false;
	rotM[0][0] = RotDef[5] * RotDef[0] * RotDef[0] + RotDef[4];
	rotM[0][1] = RotDef[5] * RotDef[0] * RotDef[1] + RotDef[3] * RotDef[2];
	rotM[0][2] = RotDef[5] * RotDef[0] * RotDef[2] - RotDef[3] * RotDef[1];
	rotM[1][0] = RotDef[5] * RotDef[0] * RotDef[1] - RotDef[3] * RotDef[2];
	rotM[1][1] = RotDef[5] * RotDef[1] * RotDef[1] + RotDef[4];
	rotM[1][2] = RotDef[5] * RotDef[1] * RotDef[2] + RotDef[3] * RotDef[0];
	rotM[2][0] = RotDef[5] * RotDef[0] * RotDef[2] + RotDef[3] * RotDef[1];
	rotM[2][1] = RotDef[5] * RotDef[1] * RotDef[2] - RotDef[3] * RotDef[0];
	rotM[2][2] = RotDef[5] * RotDef[2] * RotDef[2] + RotDef[4];
	csphi = (rotM[0][0] + rotM[1][1] + rotM[2][2] - 1.0) / 2.0;
	phi = acos(csphi);		siphi = sin(phi);
	x = (rotM[1][2] - rotM[2][1]) / (2.0 * siphi);
	y = (rotM[2][0] - rotM[0][2]) / (2.0 * siphi);
	z = (rotM[0][1] - rotM[1][0]) / (2.0 * siphi);
	xd = asin(x) / 0.01745329252;			yd = asin(y) / 0.01745329252;
	zd = asin(z) / 0.01745329252;			lsxd = light_source.fx;
	lsyd = light_source.fy;
	//do dialog
	if (!(Dlg = new DlgRoot(Plot3D_Dlg, data))) return false;
#ifdef _WINDOWS
	for (i = 61; i <= 62; i++) Dlg->TextSize(i, 12);
#else
	for (i = 61; i <= 62; i++) Dlg->TextSize(i, 10);
#endif
	Dlg->GetValue(101, &ox);		Dlg->GetValue(104, &oy);
	Dlg->GetValue(107, &oz);		Dlg->GetValue(201, &olsx);
	Dlg->GetValue(204, &olsy);
	Dlg->SetColor(63, 0x000000ff);
	hDlg = CreateDlgWnd((char*)SDLG_PLOT_3D_HD, 50, 50, 450, 270, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:
			Dlg->GetValue(101, &nx);			Dlg->GetValue(104, &ny);
			Dlg->GetValue(107, &nz);
			x = nx;		y = ny;		z = nz;
			vl = sqrt((x * x) + (y * y) + (z * z));
			if (vl < 0.001) {
				Dlg->ShowItem(61, false);		Dlg->ShowItem(62, false);
				Dlg->ShowItem(63, true);		Dlg->SetCheck(10, 0L, true);
				Dlg->DoPlot(0L);
				res = -1;
				}
			break;
			}
		} while (res < 0);

	if (res == 1) {
		Undo.SetDisp(o);
		if (nx != ox || ny != oy || nz != oz) {		//rotation modified
			vec2mat(nx, ny, nz, rotM);				//create new rotation matrix
			//create new quaternion in RotDef from rotation matrix
			Undo.RotDef(this, &RotDef, undo_flags);		undo_flags |= UNDO_CONTINUE;
			RotDef[4] = (rotM[0][0] + rotM[1][1] + rotM[2][2] - 1) / 2.0;
			RotDef[3] = sqrt(1.0 - RotDef[4] * RotDef[4]);
			RotDef[0] = (rotM[1][2] - rotM[2][1]) / (2.0 * RotDef[3]);
			RotDef[1] = (rotM[2][0] - rotM[0][2]) / (2.0 * RotDef[3]);
			RotDef[2] = (rotM[0][1] - rotM[1][0]) / (2.0 * RotDef[3]);
			RotDef[5] = 1.0 - RotDef[4];
			bRet = true;
			}
		Dlg->GetValue(201, &nlsx);			Dlg->GetValue(204, &nlsy);
		if (nlsx != olsx || nlsy != olsy) {		//light source modified
			Undo.SaveLFP(this, &light_source, undo_flags);
			light_source.fx = nlsx;			light_source.fy = nlsy;
			bRet = true;
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;		free(Plot3D_Dlg);	free(tab1);		free(tab2);	
	return bRet;
}

bool
Plot3D::PropertyDlg()
{
	Plot *p;
	bool bRet = false;

	if(plots) {
		//plots already created - jump to configuration dialog
		return Configure();
		}
	if((p = new Scatt3D(this, data, crea_flags)) && p->PropertyDlg()) {
		if(!(bRet = Command(CMD_DROP_PLOT, p, (anyOutput *)NULL))) DeleteGO(p);
		}
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a 2.5 dimensional bar chart
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//NOTE: This dialog template is also used for 2.5D ribbon plot
static char *Base25D_DlgTmpl = (char*)
	"1,+,,DEFAULT, PUSHBUTTON,-1,158,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,158,25,45,12\n"
	".,,10,ISPARENT | CHECKED,GROUP,,,,,\n"
	"10,+,100,ISPARENT | CHECKED,SHEET,1,5,10,140,100\n"
	".,.,200,ISPARENT,SHEET,2,5,10,140,100\n"
	".,.,300,ISPARENT,SHEET,3,5,10,140,100\n"
	".,20,,HIDDEN,LTEXT,20,15,115,100,8\n"
	"20,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,+,,,LTEXT,-35,15,30,60,8\n"
	".,152,,,RANGEINPUT,-15,25,40,100,10\n"
	"152,+,,ISPARENT | CHECKED,GROUPBOX,-36,12,60,128,45\n"
	".,.,,,LTEXT,0,25,65,60,8\n"
	".,.,,,RANGEINPUT,-16,25,75,100,10\n"
	".,.,,,PUSHBUTTON,-8,95,87,30,12\n"
	".,,,,PUSHBUTTON,-9,60,87,35,12\n"
	"200,+,,,LTEXT,4,20,35,80,8\n"
	".,.,,,RTEXT,5,48,45,13,8\n"
	".,.,,,EDVAL1,6,65,45,25,10\n"
	".,.,,,RTEXT,7,48,57,13,8\n"
	".,,,,EDVAL1,8,65,57,25,10\n"
	"300,+,,HICOL,RADIO1,9,15,35,80,9\n"
	".,.,,ODEXIT,COLBUTT,10,110,35,20,10\n"
	".,.,,CHECKED | HICOL,RADIO1,11,15,55,80,9\n"
	".,.,,ODEXIT,COLBUTT,12,25,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,13,37,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,14,49,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,15,61,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,16,73,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,17,85,70,10,10\n"
	".,.,,ODEXIT,COLBUTT,18,97,70,10,10\n"
	".,,,LASTOBJ | ODEXIT,COLBUTT,19,109,70,10,10";

bool
Chart25D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	static DWORD colarr[] = {0x000080ffL, 0x00ff8000L, 0x0000ff00L, 0x000000ffL,
		0x00ff00ff, 0x00ffff00L, 0x0000ffff, 0x00c0c0c0};
	static DWORD defcol = 0x00ffffffL;
	double start_z = 1.0;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_BAR25D_DIST, (void*)SDLG_BAR25D_STZ,
		(void*)&start_z, (void*)SDLG_BAR25D_STEP, (void*)&dspm.fz, (void*)SDLG_BAR25D_CCOL,
		(void*)&defcol, (void*)SDLG_BAR25D_CINC, (void*)&colarr[0], (void*)&colarr[1],
		(void*)&colarr[2], (void*)&colarr[3], (void*)&colarr[4], (void*)&colarr[5], (void*)&colarr[6],
		(void*)&colarr[7], (void*)SCMS_BAD_DATA_RANGE };
	DlgInfo *Bar3D_Dlg = CompileDialog(Base25D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int j, ic, res, currYR = 0, maxYR = 0, ny, nx = 0;
	long rx, cx, ry, cy, oax;
	char **rd = 0L;
	double fx, fy, fz, fsz;
	bool updateYR = true, bContinue = false, bRet = false, bUseSch = true;
	AccRange *rX = 0L, *rY = 0L;
	Brick **cols;
	Scatt3D *plot;
	AxisDEF *ax;

	if(!parent || !data) return false;
	if(plots) {
		//Plots alredy defined: jump to config dialog
		return false;
		}
	if (!UseRangeMark(data, 2, TmpTxt, TmpTxt + 100, TmpTxt + 200, TmpTxt + 300, TmpTxt + 400,
		TmpTxt + 500, TmpTxt + 600, TmpTxt + 700, TmpTxt + 800, TmpTxt + 900, TmpTxt + 1000)){
//		return false;
		}
	if(TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(12, sizeof(char*)))) {
		for(i=100, j= 0; i <= 1000; i +=100){ 
			if(TmpTxt[i]) rd[j++] = rlp_strdup(TmpTxt+i);
			}
		if(j) maxYR = j-1;
		}
	if(!rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	if(!(Dlg = new DlgRoot(Bar3D_Dlg, data))) return false;
	Dlg->SetColor(12, 0x000000ff);					Dlg->SetColor(13, 0x000000ff);
	if(rd && rd[currYR] &&  *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd((char*)SDLG_BAR25D_HD1, 50, 50, 450, 300, Dlg, 0x4L);
	do {
		if(updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(156, 1);				Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);				Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BAR25D_YOFY, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt,SDLG_BAR25D_YOFY, currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		ny = 0;
		if(rX) delete rX;
		rX = 0L;
		switch(res) {
		case 0:
			if(bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:
			bContinue = false;			break;
		case 1:		case 155:		case 156:
			if (res == 1) {
				for (i = 0; i < 8; i++) Dlg->GetColor(303 + i, &colarr[i]);
				Dlg->GetColor(301, &defcol);
				bUseSch = Dlg->GetCheck(302);
				Dlg->GetValue(202, &start_z);	Dlg->GetValue(204, &dspm.fz);
				//execute com_StackDlg for <OK>
				}
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR,
				&rY, &bContinue, &ny, &maxYR, &updateYR);
			if (!nx) {
				Dlg->ShowItem(13, true);			Dlg->DoPlot(0L);
				res = -1;
				}
			break;
		case 301:
			Dlg->SetCheck(300, 0L, true);
			res = -1;	break;
		case 303:	case 304:	case 305:	case 306:
		case 307:	case 308:	case 309:	case 310:
			Dlg->SetCheck(302, 0L, true);
			res = -1;	break;
			}
		}while (res < 0);
	if(res == 1 && nx && rX && rd && rd[0] && rd[0][0]) {
		if(rd[maxYR]) maxYR++;
		fsz = DefSize(SIZE_BAR)/2.0;	fz = start_z;
		oax = AxisTempl3D;	AxisTempl3D = 1;	CreateAxes();
		if(Axes && nAxes > 2 && Axes[2] && (ax = Axes[2]->GetAxis())){
			ax->flags = AXIS_3D | AXIS_INVERT | AXIS_DEFRECT;
			ax->min = start_z-dspm.fz;
			ax->max = start_z+dspm.fz*maxYR;
			if(Axes[1] && (ax = Axes[1]->GetAxis())){
				ax->flags |= AXIS_GRIDLINE;
				i = 0x0c;
				Axes[1]->Command(CMD_SET_GRIDTYPE, &i, 0L);
				}
			AxisTempl3D = oax;
			}
		if((plots = (GraphObj**)calloc(maxYR+2, sizeof(GraphObj*)))) {
			for(i = 0; i < maxYR; i++, fz += dspm.fz) {
				if(rd[i] && (cols = (Brick**)calloc(nx+2, sizeof(Brick*))) && (rY = new AccRange(rd[i]))) {
					ic = 0;
					if(rX->GetFirst(&cx, &rx) && rX->GetNext(&cx, &rx) &&
						rY->GetFirst(&cy, &ry) && rY->GetNext(&cy, &ry)) {
						do {
							if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy)){
								cols[ic] = new Brick(this, data, fx, 0.0, fz, 
									fsz, fsz, fy, 0x800L, cx, rx, -1, -1, -1, -1,
									-1, -1, -1, -1, cy, ry);
								}
							ic++;
							}while(rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry));
						if(ic) bRet = true;
						}
					plot = new Scatt3D(this, data, cols, ic);
					if(plot){
						if(bUseSch) plot->SetColor(COL_BAR_FILL, colarr[(i & 0x07)]);
						else plot->SetColor(COL_BAR_FILL, defcol);
						plots[nPlots++] = plot;
						}
					if(rY) {
						plot->data_desc = rlp_strdup(rY->RangeDesc(data, 1));
						delete(rY);		rY = 0L;
						}
					}
				}
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rd) {
		for (i = 0; i < maxYR; i++){
			if (rd[i]) free(rd[i]);
			}
		free(rd);
		}
	if(rX) delete rX;
	if(rY) delete rY;
	free(Bar3D_Dlg);
	free(tab1);		free(tab2);			free(tab3);
	if(bRet) Command(CMD_MRK_DIRTY, 0L, 0L);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a 2.5 dimensional ribbon chart
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool
Ribbon25D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DETAILS);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SCHEME);
	static DWORD colarr[] = {0x000080ffL, 0x00ff8000L, 0x0000ff00L, 0x000000ffL,
		0x00ff00ff, 0x00ffff00L, 0x0000ffff, 0x00c0c0c0};
	static DWORD defcol = 0x00ffffffL;
	double start_z = 1.0;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_BAR25D_DIST, (void*)SDLG_BAR25D_STZ,
		(void*)&start_z, (void*)SDLG_BAR25D_STEP, (void*)&dspm.fz, (void*)SDLG_RIB25D_CCOL,
		(void*)&defcol, (void*)SDLG_BAR25D_CINC, (void*)&colarr[0], (void*)&colarr[1],
		(void*)&colarr[2], (void*)&colarr[3], (void*)&colarr[4], (void*)&colarr[5], (void*)&colarr[6],
		(void*)&colarr[7], (void*)SCMS_BAD_DATA_RANGE };
	DlgInfo *Bar3D_Dlg = CompileDialog(Base25D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int j, res, currYR=0, maxYR=0, nx=0, ny, oax;
	char **rd = 0L, xrange[100];
	double fz;
	bool updateYR = true, bContinue = false, bRet = false, bUseSch = true;
	AccRange *rX = 0L, *rY = 0L;
	AxisDEF *ax;
	Ribbon *plot;

	if(!parent || !data) return false;
	if(plots) {
		//Plots alredy defined: jump to config dialog
		return false;
		}
	if (!UseRangeMark(data, 2, TmpTxt, TmpTxt + 100, TmpTxt + 200, TmpTxt + 300, TmpTxt + 400,
		TmpTxt + 500, TmpTxt + 600, TmpTxt + 700, TmpTxt + 800, TmpTxt + 900, TmpTxt + 1000)){
//		return false;
		}
	if(TmpTxt[0] && TmpTxt[100] && (rd = (char**)calloc(12, sizeof(char*)))) {
		for(i=100, j= 0; i <= 1000; i +=100){ 
			if(TmpTxt[i]) rd[j++] = rlp_strdup(TmpTxt+i);
			}
		if(j) maxYR = j-1;
		}
	if(!rd && !(rd = (char**)calloc(maxYR+2, sizeof(char*))))return false;
	if(!(Dlg = new DlgRoot(Bar3D_Dlg, data)))return false;
	Dlg->SetColor(13, 0x000000ff);
	if(rd && rd[currYR] &&  *(rd[currYR])) Dlg->SetText(154, (unsigned char*)rd[currYR]);
	hDlg = CreateDlgWnd((char*)SDLG_RIB25D_HD1, 50, 50, 450, 300, Dlg, 0x4L);
	do {
		if(updateYR) {
			if(currYR >0) {
				Dlg->ShowItem(156, 1);		Dlg->Activate(101, 0);
				}
			else {
				Dlg->ShowItem(156, 0);		Dlg->Activate(101, 1);
				}
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BAR25D_YOFY, currYR + 1, maxYR + 1);
#else
			sprintf(TmpTxt, SDLG_BAR25D_YOFY, currYR+1, maxYR+1);
#endif
			//SetText will also cause a redraw of the whole dialog
			Dlg->SetText(153, (unsigned char*)TmpTxt);
			updateYR = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		ny = 0;		if(rX) delete rX;		rX = 0L;
		switch(res) {
		case 0:
			if(bContinue || Dlg->GetCheck(20)) res = -1;
			break;
		case -1:
			bContinue = false;			break;
		case 1:
		case 155:		case 156:
			if (res == 1){
				for (i = 0; i < 8; i++) Dlg->GetColor(303 + i, &colarr[i]);
				Dlg->GetColor(301, &defcol);	bUseSch = Dlg->GetCheck(302);
				Dlg->GetValue(202, &start_z);	Dlg->GetValue(204, &dspm.fz);
				//execute com_StackDlg for <OK>
				}
			res = com_StackDlg(res, Dlg, &rX, &nx, &rd, &currYR, &rY, &bContinue, &ny, &maxYR, &updateYR);
			if (!nx) {
				Dlg->ShowItem(13, true);			Dlg->DoPlot(0L);
				res = -1;
				}
			break;
		case 301:
			Dlg->SetCheck(300, 0L, true);
			res = -1;	break;
		case 303:	case 304:	case 305:	case 306:
		case 307:	case 308:	case 309:	case 310:
			Dlg->SetCheck(302, 0L, true);
			res = -1;	break;
			}
		}while (res < 0);
	if(res == 1 && nx && rX && rd && rd[0] && rd[0][0]) {
		if(rd[maxYR]) maxYR++;
		fz = start_z;
		Dlg->GetText(101, (unsigned char*)(TmpTxt+100), 100);
		oax = AxisTempl3D;	AxisTempl3D = 1;	CreateAxes();
		if(Axes && nAxes > 2 && Axes[2] && (ax = Axes[2]->GetAxis())){
			ax->flags = AXIS_3D | AXIS_INVERT | AXIS_DEFRECT;
			ax->min = start_z-dspm.fz;
			ax->max = start_z+dspm.fz*maxYR;
			if(Axes[1] && (ax = Axes[1]->GetAxis())){
				ax->flags |= AXIS_GRIDLINE;	
				i = 0x0c;
				Axes[1]->Command(CMD_SET_GRIDTYPE, &i, 0L);
				}
			AxisTempl3D = oax;
			}
		plots = (GraphObj**)calloc(maxYR + 2, sizeof(GraphObj*));
		if (plots) {
			Dlg->GetText(101, (unsigned char*)xrange, 100);
			for(i = 0; i < maxYR; i++, fz += dspm.fz) {
				plot = new Ribbon(this, data, fz, dspm.fz, xrange, rd[i]);
				if(plot){
					if(bUseSch) plot->SetColor(COL_POLYGON, colarr[(i & 0x07)]);
					else plot->SetColor(COL_POLYGON, defcol);
					plots[nPlots++] = plot;
					}
				}
			}
		Command(CMD_MRK_DIRTY, 0L, 0L);		Command(CMD_AUTOSCALE, 0L, 0L);
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rd) {
		for (i = 0; i < maxYR; i++)	if(rd[i]) free(rd[i]);
		free(rd);
		}
	if(rX) delete rX;
	if(rY) delete rY;
	free(Bar3D_Dlg);
	free(tab1);		free(tab2);			free(tab3);
	if(bRet) Command(CMD_MRK_DIRTY, 0L, 0L);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a 3 dimensional bubble plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//this template uses some definitions of the 'Bubble' and 'BubblePlot'objects
static char *BubDlg3D_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,152,10,45,12\n"
	".,.,,, PUSHBUTTON,-2,152,25,45,12\n"
	".,50,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,TOUCHEXIT | ISPARENT | CHECKED,SHEET,1,5,10,131,142\n"
	".,10,400,TOUCHEXIT | ISPARENT,SHEET,2,5,10,131,142\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"50,,,NOSELECT,ODBUTTON,3,147,65,45,45\n"
	"100,+,,,LTEXT,-31,10,25,60,8\n"
	".,.,,,RANGEINPUT,4,20,35,100,10\n"
	".,.,,,LTEXT,-32,10,48,60,8\n"
	".,.,,,RANGEINPUT,5,20,58,100,10\n"
	".,.,,,LTEXT,-33,10,71,60,8\n"
	".,.,,,RANGEINPUT,6,20,81,100,10\n"
	".,,150,ISPARENT | CHECKED,GROUPBOX,7,8,98,125,50\n"
	"150,+,,,LTEXT,8,12,102,60,8\n"
	".,.,,,RANGEINPUT,9,20,112,100,10\n"
	".,.,,,LTEXT,10,12,125,20,8\n"
	".,.,,TOUCHEXIT | CHECKED | HICOL,RADIO1,-3,38,125,20, 8\n"
	".,.,,TOUCHEXIT | HICOL,RADIO1,11,70,125,20,8\n"
	".,.,,TOUCHEXIT | HICOL,RADIO1,12,20,135,20,8\n"
	".,,,TOUCHEXIT | HICOL,RADIO1,13,70,135,20,8\n"
	"400,410,,,LTEXT,14,20,30,60,8\n"
	"410,+,,TOUCHEXIT | ISRADIO,ODBUTTON,15,20,42,25,25\n"
	".,.,,TOUCHEXIT | ISRADIO,ODBUTTON,15,45,42,25,25\n"
	".,,,LASTOBJ | TOUCHEXIT | ISRADIO,ODBUTTON,15,70,42,25,25";

bool
BubblePlot3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_AXES);
	char text1[100], text2[100], text3[100], text4[100];
	void *dyndata[] = { (void*)tab1, (void*)tab3, (void*)(OD_AxisDesc3D), (void*)&text1,
		(void*)&text2, (void*)&text3, (void*)SDLG_BUB_DIAM, (void*)SDLG_BPLT_SRANG,
		(void*)&text4, (void*)SDLG_BUB3DPLOT_SCALE, (void*)SDLG_BUB3DPLOT_SCALEX, (void*)SDLG_BUB3DPLOT_SCALEY,
		(void*)SDLG_BUB3DPLOT_SCALEZ, (void*)SDLG_BUB3DPLOT_TEMPL, (void*)(OD_AxisTempl3D) };
	DlgInfo *BubDlg3D = CompileDialog(BubDlg3D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, count;
	long cx, rx, cy, ry, cz, rz, cr, rr;
	int s_type = 5;
	bool bRet = false;
	double fx, fy, fz, fr;
	Sphere **Balls;
	AccRange *rX, *rY, *rZ, *rR;
	Scatt3D *sc_plot;

	if(!data || !parent)return false;
	UseRangeMark(data, 1, text1, text2, text3, text4);
	if(!(Dlg = new DlgRoot(BubDlg3D, data)))return false;
	rX = rY = rZ = rR = 0L;
	Dlg->SetCheck(410 + AxisTempl3D, 0L, true);
	hDlg = CreateDlgWnd((char*)SDLG_BUB3DPLOT_HD1, 50, 50, 418, 360, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:
			if(Dlg->GetCheck(10)) res = -1;
			break;
		case 4:		case 5:					//the tab sheets
			res = -1;
			break;
		case 153:	
			s_type = 5;						//absolute size, but use 5 to distinguish
			res = -1;						//  from symbol
			break;
		case 154:	case 155:	case 156:
			s_type = res - 153;
			res = -1;
			break;
		case 410:	case 411:	case 412:	//axis templates
			AxisTempl3D = res-410;
			res = -1;
			break;
			}
		}while (res <0);
	if(res == 1) {
		if(Dlg->GetText(101, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rX = new AccRange(TmpTxt);
		if(Dlg->GetText(103, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rY = new AccRange(TmpTxt);
		if(Dlg->GetText(105, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rZ = new AccRange(TmpTxt);
		if(Dlg->GetText(151, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) rR = new AccRange(TmpTxt);
		if(rX && rY && rZ && rR && (count = rX->CountItems()) 
			&& (Balls = (Sphere**)calloc(count+2, sizeof(Sphere*)))) {
			rX->GetFirst(&cx, &rx);		rY->GetFirst(&cy, &ry);
			rZ->GetFirst(&cz, &rz);		rR->GetFirst(&cr, &rr);
			for(i = 0; i < count; i++){
				if(rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry)
					&& rZ->GetNext(&cz, &rz) && rR->GetNext(&cr, &rr)
					&& data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy)
					&& data->GetValue(rz, cz, &fz) && data->GetValue(rr, cr, &fr)) {
					Balls[i] = new Sphere(this, data, s_type, fx, fy, fz, fr, cx, rx,
						cy, ry, cz, rz, cr, rr);
					}
				}
			sc_plot = new Scatt3D(this, data, Balls, count);
			if(parent->Id == GO_PLOT3D || parent->Id == GO_FUNC3D || parent->Id == GO_FITFUNC3D) {
				if(!(parent->Command(CMD_DROP_PLOT, sc_plot, 0L))) delete(sc_plot);
				bRet = true;
				}
			else if(!(bRet = Command(CMD_DROP_PLOT, sc_plot, 0L))) delete(sc_plot);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(rX) delete rX;
	if(rY) delete rY;
	if(rZ) delete rZ;
	if(rR) delete rR;
	free(BubDlg3D);		 free(tab1);			free(tab3);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a 3D function plot
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *FuncDlg3D_DlgTmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON,-1,160,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,160,25,45,12\n"
	".,10,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,100,ISPARENT,SHEET,1,5,10,149,134\n"
	".,50,300,ISPARENT | CHECKED | TOUCHEXIT,SHEET,2,5,10,149,134\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"50,,,NOSELECT,ODBUTTON,3,160,85,45,45\n"
	"100,+,,,LTEXT,4,10,30,100,8\n"
	".,.,,,RTEXT,5,10,50,28,8\n"
	".,.,,,EDVAL1,6,38,50,25,10\n"
	".,.,,,RTEXT,7,61,50,17,8\n"
	".,.,,,EDVAL1,8,78,50,25,10\n"
	".,.,,,RTEXT,9,102,50,17,8\n"
	".,.,,,EDVAL1,10,119,50,25,10\n"
	".,.,,,RTEXT,11,10,62,28,8\n"
	".,.,,,EDVAL1,12,38,62,25,10\n"
	".,.,,,EDVAL1,13,78,62,25,10\n"
	".,150,,,EDVAL1,14,119,62,25,10\n"
	"150,200,,,RTEXT,15,10,91,10,\n"
	"200,,,,TEXTBOX,16,22,89,122,40\n"
	"300,+,500,CHECKED,GROUPBOX,17,10,40,140,100\n"
	".,305, 400, HIDDEN | CHECKED,GROUPBOX,18,10,40,140,100\n"
	"305,+, 0, HICOL | CHECKED | TOUCHEXIT,RADIO1,19,15,25,50,10\n"
	".,, 0, HICOL| TOUCHEXIT, RADIO1,20,85,25,50,10\n"
	"400,+,,,RTEXT,21,38,50,40,8\n"
	".,.,,,EDVAL1,22,80,50,25,10\n"
	".,.,,,LTEXT,-3,107,50,20,8\n"
	".,.,,,RTEXT,23,38,62,40,8\n"
	".,.,,OWNDIALOG,COLBUTT,24,80,62,25,10\n"
	".,.,,,RTEXT,25,38,74,40,8\n"
	".,,,OWNDIALOG | TOUCHEXIT,SHADEPLANE,26,80,74,25,10\n"
	"500,,,LASTOBJ | NOSELECT,ODBUTTON,27,15,45,130,100";
		
bool
Func3D::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_FUNCTION);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	static FillDEF newFill;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)(OD_AxisDesc3D), (void*)SDLG_FUNC3DPLOT_UDF,
		(void*)SDLG_FUNC3DPLOT_STX, (void*)&x1, (void*)SDLG_FUNC3DPLOT_LASTX, (void*)&x2,
		(void*)SDLG_FUNC3DPLOT_STEP, (void*)&xstep, (void*)SDLG_FUNC3DPLOT_ZEQ, (void*)&z1,
		(void*)&z2, (void*)&zstep, (void*)SDLG_FUNC3DPLOT_YEQ, (void*)cmdxy, (void*)SDLG_FUNC3DPLOT_GRIDB,
		(void*)SDLG_FUNC3DPLOT_SURFB, (void*)SDLG_FUNC3DPLOT_GRIDR, (void*)SDLG_FUNC3DPLOT_SURFR,
		(void*)SDLG_FUNC3DPLOT_GLW, (void*)&Line.width, (void*)SDLG_FUNC3DPLOT_GLC, (void *)&Line.color,
		(void*)SDLG_FUNC3DPLOT_PLC, (void*)&newFill, (void*)OD_linedef };
	DlgInfo *FuncDlg = CompileDialog(FuncDlg3D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res;
	bool bRet = false, bNew = true;
	DWORD undo_flags = 0L;
	static LineDEF newLine;
	double o_x1, n_x1, o_x2, n_x2, o_xstep, n_xstep;
	double o_z1, o_z2, o_zstep;
	bool bVisited = false;

	if(!parent) return false;
//	if(parent->Id == GO_FITFUNC) return parent->PropertyDlg();
	memcpy(&newFill, &Fill, sizeof(FillDEF));
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	if(!(Dlg = new DlgRoot(FuncDlg, data))) return false;
	if(!bNew) Dlg->ShowItem(10, false);
	Dlg->GetValue(102, &o_x1);		n_x1 = o_x1;
	Dlg->GetValue(104, &o_x2);		n_x2 = o_x2;
	Dlg->GetValue(106, &o_xstep);	n_xstep = o_xstep;
	Dlg->GetValue(108, &o_z1);
	Dlg->GetValue(109, &o_z2);
	Dlg->GetValue(110, &o_zstep);
	hDlg = CreateDlgWnd((char*)SDLG_FUNC3DPLOT_HD1, 50, 50, 446, 345, Dlg, 0x4L);
	if(bNew) Dlg->SetCheck(4, 0L, true);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 305:		case 306:
			if(Dlg->GetCheck(305)) {
				Dlg->ShowItem(300, true);	Dlg->ShowItem(301, false);
				}
			else {
				Dlg->ShowItem(300, false);	Dlg->ShowItem(301, true);
				}
			Dlg->DoPlot(0L);
			res = -1;
			break;
		case 406:
//			Dlg->ItemCmd(res, CMD_REDRAW, 0L);
			res = -1;
			break;
		case 5:
			bVisited = true;		//the style tab must be visited
			res = -1;
			break;
		case 1:
			if (!bVisited) {
				Dlg->SetCheck(5, 0L, 1);
				bVisited = true;	 res = -1;
				}
			break;
		case 0:
			if(Dlg->GetCheck(10)) res = -1;
			break;
			}
		}while (res < 0);
//		if (cdisp) {
//			Undo.SetDisp(cdisp);
//			while (*Undo.pcb > undo_level)	Undo.Pop(cdisp);
//			}
	if(res == 1){						//OK pressed
		if(bNew) {						//create function
			if(Dlg->GetCheck(305)) {
				type = 0;
				OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&Line, 0);
				}
			else {
				type = 1;
				memcpy(&Fill, &newFill, sizeof(FillDEF));
				Dlg->GetValue(401, &Line.width);
				Dlg->GetColor(404, &Line.color);
				Line.pattern = 0L;
				Line.patlength = 1;
				}
			Dlg->GetValue(102, &x1);		Dlg->GetValue(104, &x2);
			Dlg->GetValue(106, &xstep);
			Dlg->GetValue(108, &z1);		Dlg->GetValue(109, &z2);
			Dlg->GetValue(110, &zstep);		type = Dlg->GetCheck(305) ? 0 : 1;
			if(Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
				if(cmdxy) free(cmdxy);
				cmdxy = rlp_strdup(TmpTxt);
				ReshapeFormula(&cmdxy);		bRet = Update();
				}
			if (gob) {
				Command(CMD_AUTOSCALE, 0L, 0L);
				if (Bounds.Xmax <= Bounds.Xmin || zBounds.fx >= zBounds.fy) {
					ErrorBox((char*)SCMS_BADFORMULA);
					bRet = false;
					}
				}
			else bRet = false;
			}
		else {							//edit existing function
			Dlg->GetValue(102, &n_x1);		Dlg->GetValue(104, &n_x2);
			Dlg->GetValue(106, &n_xstep);
			undo_flags = CheckNewFloat(&x1, o_x1, n_x1, this, undo_flags);
			undo_flags = CheckNewFloat(&x2, o_x2, n_x2, this, undo_flags);
			undo_flags = CheckNewFloat(&xstep, o_xstep, n_xstep, this, undo_flags);
			TmpTxt[0] = 0;	Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE);
			undo_flags = CheckNewString((unsigned char**)&cmdxy, (unsigned char*)cmdxy, (unsigned char*)TmpTxt, this, undo_flags);
//			if(undo_flags & UNDO_CONTINUE) Update(0L, UNDO_CONTINUE);
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			if(cmpLineDEF(&Line, &newLine)) {
				Undo.Line(parent, &Line, undo_flags);	undo_flags |= UNDO_CONTINUE;
				memcpy(&Line, &newLine, sizeof(LineDEF));
				}
			bRet = (undo_flags & UNDO_CONTINUE) != 0;
			}
		}
	CloseDlgWnd(hDlg);		delete Dlg;
	free(FuncDlg);			free(tab1);				free(tab2);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Fit a 3D function to data
static char *FitFuncDlg3D_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,170,10,45,12\n"
".,.,,,PUSHBUTTON,-2,170,25,45,12\n"
".,10,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,+,400,ISPARENT | CHECKED,SHEET,1,5,10,159,154\n"
".,.,100,ISPARENT,SHEET,2,5,10,159,154\n"
".,.,300,ISPARENT,SHEET,3,5,10,159,154\n"
".,,,,PUSHBUTTON,4,170,152,45,12\n"
"10,50,,CHECKED,CHECKPIN,0,5,0,12,8\n"
"50,,,NOSELECT,ODBUTTON,5,170,85,45,45\n"
"100,+,,,LTEXT,6,10,24,100,8\n"
".,.,,,LTEXT,7,10,34,100,8\n"
".,150,,,TEXTBOX,8,22,44,132,30\n"
"150,+,,,LTEXT,9,10,77,10,8\n"
".,.,,,RTEXT,10,10,91,10,8\n"
".,.,,,RTEXT,11,20,123,26,8\n"
".,.,,,EDVAL1,12,46,123,35,10\n"
".,.,,,RTEXT,13,82,123,47,8\n"
".,.,,,EDVAL1,14,129,123,25,10\n"
".,.,,HIDDEN,LTEXT,0,10,147,22,8\n"
".,200,,HIDDEN,LTEXT,0,10,137,22,8\n"
"200,,,,TEXTBOX,15,22,89,132,30\n"
"300,+,550,CHECKED,GROUPBOX,16,10,40,140,100\n"
".,305,350,HIDDEN | CHECKED,GROUPBOX,17,10,40,150,100\n"
"305,+,,HICOL | CHECKED | TOUCHEXIT,RADIO1,18,15,25,50,10\n"
".,,,HICOL | TOUCHEXIT,RADIO1,19,85,25,50,10\n"
"350,+,,,RTEXT,20,38,50,40,8\n"
".,.,,,EDVAL1,21,80,50,25,10\n"
".,.,,,LTEXT,-3,107,50,20,8\n"
".,.,,,RTEXT,22,38,62,40,8\n"
".,.,,OWNDIALOG,COLBUTT,23,80,62,25,10\n"
".,.,,,RTEXT,24,38,74,40,8\n"
".,,,OWNDIALOG,SHADE3D,25,80,74,25,10\n"
"400,+,,,LTEXT,-31,10,30,60,8\n"
".,.,,,RANGEINPUT,26,20,40,120,10\n"
".,.,,,LTEXT,-32,10,55,60,8\n"
".,.,,,RANGEINPUT,27,20,65,120,10\n"
".,.,,,LTEXT,-33,10,80,60,8\n"
".,.,,,RANGEINPUT,28,20,90,120,10\n"
".,.,,HICOL | CHECKED,CHECKBOX,29,20,110,60,8\n"
".,,,HIDDEN,LTEXT,0,20,110,60,8\n"
"500,550,501,CHECKED,GROUP,,,,,\n"
"501,+,,,RTEXT,30,10,30,28,8\n"
".,.,,,EDVAL1,31,38,30,25,10\n"
".,.,,,RTEXT,32,61,30,17,8\n"
".,.,,,EDVAL1,33,78,30,25,10\n"
".,.,,,RTEXT,34,102,30,17,8\n"
".,,,,EDVAL1,35,119,30,25,10\n"
"550,,,LASTOBJ | NOSELECT,ODBUTTON,36,15,45,130,100";

bool
FitFunc3D::PropertyDlg()
{
	int i1;
	TabSHEET *tab1 = MakeTab(0, 10, &i1, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_FUNCTION);
	TabSHEET *tab3 = MakeTab(i1, 10, &i1, (char*)SDLG_TAB_STYLE);
	unsigned char text1[100], text2[100], text3[100];
	static FillDEF newFill;
	double iter;
	bool bNew = (gob == 0L);

	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_FITFUNC_FIT,
		(void*)(OD_AxisDesc3D), (void*)SDLG_FITFUNC_FITFUNC, (void*)SDLG_FITFUNC_PARAM,
		(void*)param, (void*)SDLG_FITFUNC3D_DEF, (void*)SDLG_FUNC3DPLOT_YEQ,
		(void*)SDLG_FITFUNC_CONV, (void*)&conv, (void*)SDLG_FITFUNC_ITER, (void*)&iter,
		(void*)cmdxy, (void*)SDLG_FUNC3DPLOT_GRIDB, (void*)SDLG_FUNC3DPLOT_SURFB,
		(void*)SDLG_FUNC3DPLOT_GRIDR, (void*)SDLG_FUNC3DPLOT_SURFR, (void*)SDLG_FUNC3DPLOT_GLW,
		(void*)&Line.width, (void*)SDLG_FUNC3DPLOT_GLC, (void *)&Line.color,
		(void*)SDLG_FUNC3DPLOT_PLC, (void*)&newFill, (void*)text1, (void*)text2, (void*)text3,
		(void*)SDLG_FITFUNC3D_SYMS, (void*)SDLG_FITFUNC3D_PLTX, (void*)&x1, (void*)SDLG_FUNC3DPLOT_LASTX,
		(void*)&x2, (void*)SDLG_FUNC3DPLOT_STEP, (void*)&xstep, (void*)OD_linedef };
	DlgInfo *FuncDlg = CompileDialog(FitFuncDlg3D_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, undo_level = Undo.level();
	long i, j, k, l, m, n, ns = 0;
	bool bRet = false, bContinue = false;
	LineDEF newLine;
	double x, y, z, o_x1, o_x2, o_xstep, rad;
	AccRange *rX, *rY, *rZ;
	unsigned char *o_cmdxy, *o_param, *tmp_char;
	anyOutput *cdisp = Undo.Cdisp();
	anyResult *ares;
	Sphere **Balls;

	if(!parent || !data) return false;
	UseRangeMark(data, 1, (char*)text1, (char*)text2, (char*)text3);
	if(!(o_cmdxy = rlp_strdup((unsigned char*)cmdxy)))return false;
	if(!(o_param = rlp_strdup((unsigned char*)param)))return false;
	rX = rY = rZ = 0L;			iter = (double)maxiter;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&Line, 0);
	memcpy(&newFill, &Fill, sizeof(FillDEF));
	memcpy(&newLine, &Line, sizeof(LineDEF));
	if(!(Dlg = new DlgRoot(FuncDlg, data))) return false;
	if(!bNew){
		Dlg->ShowItem(10, 0);		Dlg->Activate(401, 0);
		Dlg->Activate(403, 0);		Dlg->Activate(405,0);		
		Dlg->SetCheck(6, 0L, 1);
		Dlg->SetCheck(4, 0L, 0);	Dlg->ShowItem(404, 0);
		if(chi2 > 0.0) {
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_FITFUNC3D_CHIFMT, chi2);
#else
			sprintf(TmpTxt, SDLG_FITFUNC3D_CHIFMT, chi2);
#endif
			Dlg->SetText(405,(unsigned char*)TmpTxt);	Dlg->ShowItem(405, true);
			}
		}
	else Dlg->ShowItem(500, false);
#ifdef USE_WIN_SECURE
	sprintf_s(TmpTxt, TMP_TXT_SIZE, "%g", conv);
#else
	sprintf(TmpTxt, "%g", conv);
#endif
	Dlg->SetText(153, (unsigned char*)TmpTxt);
	Dlg->GetValue(502, &o_x1);
	Dlg->GetValue(504, &o_x2);
	Dlg->GetValue(506, &o_xstep);
	hDlg = CreateDlgWnd((char*)SDLG_FITFUNC3D_HD1, 50, 50, 466, 390, Dlg, 0x4L);
	if(bNew) Dlg->SetCheck(4, 0L, true);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(Dlg->GetCheck(10)) res = -1;
			if(bContinue) res = -1;
			bContinue = false;
			break;
		case 305:		case 306:
			if(Dlg->GetCheck(305)) {
				Dlg->ShowItem(300, true);	Dlg->ShowItem(301, false);
				}
			else {
				Dlg->ShowItem(300, false);	Dlg->ShowItem(301, true);
				}
			Dlg->DoPlot(0L);
			res = -1;
			break;
		case 1:		case 7:								//Start: do nonlinear regression
			if ((res == 1) && !bNew){
				if (Dlg->GetText(102, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if (param) free(param);
					param = rlp_strdup(TmpTxt);
					}
				if (Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if (cmdxy) free(cmdxy);
					cmdxy = rlp_strdup(TmpTxt);
					}
				ReshapeFormula(&param);		ReshapeFormula(&cmdxy);
				dirty = true;
				break;
				}
			Undo.SetDisp(cdisp);
			if(!Dlg->GetText(401, text1, 100) || !Dlg->GetText(403, text2, 100) || !Dlg->GetText(405, text3, 100)) {
				Dlg->SetCheck(4, 0L, true);			bContinue = true;
				InfoBox((char*)SCMS_BAD_DATA_RANGE);
				res = -1;
				}
			else if(Dlg->GetCheck(5)) {		//  the function tab must be shown
				if(Dlg->CurrDisp) Dlg->CurrDisp->MouseCursor(MC_WAIT, true);
				if(Dlg->GetText(102, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if(param) free(param);	
					param = rlp_strdup(TmpTxt);
					}
				if(Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
					if(cmdxy) free(cmdxy);	
					cmdxy = rlp_strdup(TmpTxt);
					}
				Dlg->GetValue(153, &conv);	Dlg->GetValue(155, &iter);
				ReshapeFormula(&param);		ReshapeFormula(&cmdxy);
				do_formula(data, 0L);		//clear any error condition
				ares = do_formula(data, param);
				if(ares->type != ET_VALUE) {
					ErrorBox((char*)SCMS_ERR_PARAM);
					bContinue = true;	res = -1;
					break;
					}
				ares = do_formula(data, cmdxy);
				if(ares->type != ET_VALUE) {
					ErrorBox((char*)SCMS_ERR_FORMULA);
					bContinue = true;	res = -1;
					break;
					}
				i = rlp_strcpy(TmpTxt, TMP_TXT_SIZE-2, param);	i += rlp_strcpy(TmpTxt+i, TMP_TXT_SIZE-i, (char*)";x=1;z=1;");
				rlp_strcpy(TmpTxt+i, TMP_TXT_SIZE-i, cmdxy);	yywarn(0L, true);
				ares = do_formula(data, TmpTxt);
				if((tmp_char = (unsigned char*)yywarn(0L, false))) {
					ErrorBox((char*)tmp_char);
					bContinue = true;	res = -1;
					break;
					}
				i = do_fitfunc(data, (char*)text1, (char*)text2, (char*)text3, &param, cmdxy, conv, (int)iter, &nVals, &chi2, 0L, 0L, 0L, 0, 0L, 0L, 0L);
				if (i < (iter - 1)) {
					Dlg->SetText(102, (unsigned char*)param);
#ifdef USE_WIN_SECURE
					sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BTFUNC_CHI2, chi2, nVals);
#else
					sprintf(TmpTxt, SDLG_BTFUNC_CHI2, chi2, nVals);
#endif
					Dlg->SetText(156, (unsigned char*)TmpTxt);	Dlg->ShowItem(156, true);

					}
#ifdef USE_WIN_SECURE
				sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_BTFUNC_END, SDLG_FITFUNC_LEVMARC, i);
#else
				sprintf(TmpTxt, SDLG_BTFUNC_END, SDLG_FITFUNC_LEVMARC, i);
#endif
				Dlg->SetText(157, (unsigned char*)TmpTxt);	Dlg->ShowItem(157, true);

				bContinue = true;
				if(res == 7) res = -1;
				if(Dlg->CurrDisp) Dlg->CurrDisp->MouseCursor(MC_ARROW, true);
				}
			else {							//diplay function tab first
				Dlg->SetCheck(5, 0L, true);
				res = -1;
				}
			break;
			}
		}while (res < 0);
	if (cdisp) {
		Undo.SetDisp(cdisp);
		while (Undo.level() > undo_level)	Undo.Pop(cdisp);
		}
	if(res == 1 && (rX=new AccRange((char*)text1)) && (rY=new AccRange((char*)text2)) && (rZ=new AccRange((char*)text3))){
		//OK pressed
		if(bNew) {						//create function
			if(Dlg->GetCheck(305)) {
				type = 0;
				OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&Line, 0);
				}
			else {
				type = 1;
				memcpy(&Fill, &newFill, sizeof(FillDEF));
				Dlg->GetValue(401, &Line.width);
				Dlg->GetColor(404, &Line.color);
				Line.pattern = 0L;			Line.patlength = 1;
				}
			rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
			rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
			rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
			do {
				if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(n, m, &z))
					CheckBounds3D(x,y,z);
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
			x1 = xBounds.fx;		x2 = xBounds.fy;		xstep = (x2-x1)/10.0;
			z1 = zBounds.fx;		z2 = zBounds.fy;		zstep = (z2-z1)/10.0;
			if(Dlg->GetText(102, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
				if(param) free(param);	
				param = rlp_strdup(TmpTxt);
				}
			if(Dlg->GetText(200, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) {
				if(cmdxy) free(cmdxy);	
				cmdxy = rlp_strdup(TmpTxt);
				}
			ReshapeFormula(&param);		ReshapeFormula(&cmdxy);
			if((bRet = Update()) && gob && nPlots == 1 && (Balls=(Sphere**)calloc(rX->CountItems()+2, sizeof(Sphere*)))) {
				rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
				rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
				rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
				rad = DefSize(SIZE_SYMBOL);
				do {
					if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && data->GetValue(n, m, &z)) {
						Balls[ns++] = new Sphere(this, data, 0, x, y, z, rad, i, j, k, l, m, n);
						}
					}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
				if(ns) {
					plots[1] = (GraphObj*) new Scatt3D(this, data, Balls, ns);
					nPlots = 2;
					}
				else free(Balls);
				}
			if(bRet)Command(CMD_ENDDIALOG, 0L, 0L);
			}
		else {							//edit existing function
			Dlg->GetValue(102, &x1);		Dlg->GetValue(104, &x2);
			Dlg->GetValue(106, &xstep);
			Dlg->GetValue(108, &z1);		Dlg->GetValue(109, &z2);
			Dlg->GetValue(110, &zstep);		type = Dlg->GetCheck(305) ? 0 : 1;
			InfoBox((char*)"Not Implemented");
			}
		}
	if(rX) delete(rX);
	if(rY) delete(rY);
	if(rZ) delete(rZ);
	CloseDlgWnd(hDlg);
	delete Dlg;
	free(tab1);		free(tab2);			free(tab3);
	free(FuncDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Grid line properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *GridLineDlg_Tmpl = (char*)
	"1,+,,DEFAULT,PUSHBUTTON, 1,150,10,50,12\n"
	".,.,,,PUSHBUTTON,2,150,25,50,12\n"
	".,.,,,PUSHBUTTON,7,150,40,50,12\n"
	".,10,,,PUSHBUTTON,-2,150,55,50,12\n"
	"10,,100,ISPARENT | CHECKED,GROUPBOX,3,5,10,139,123\n"
	"100,+,,NOSELECT,ODB,4,10,18,130,100\n"
	".,.,112,HIDDEN | CHECKED,GROUP,,,,,\n"
	".,.,115,HIDDEN | CHECKED,GROUP,,,,,\n"
	".,.,120,HIDDEN | CHECKED,GROUP,,,,,\n"
	".,.,125,HIDDEN | CHECKED,GROUP,,,,,\n"
	".,,130,HIDDEN | CHECKED,GROUP,,,,,\n"
	"111,,,,RTEXT,5,15,117,23,8\n"
	"112,+,,HICOL,CHECKBOX,-20,41,117,25,8\n"
	".,.,,HICOL,CHECKBOX,-21,105,117,25,8\n"
	".,111,,HICOL,CHECKBOX,-25,70,117,25,8\n"
	"115,+,,HICOL,CHECKBOX,-22,41,117,25,8\n"
	".,.,,HICOL,CHECKBOX,-23,105,117,25,8\n"
	".,111,,HICOL,CHECKBOX,-24,70,117,25,8\n"
	"120,+,,HICOL,CHECKBOX,-24,55,117,25,8\n"
	".,135,,HICOL,CHECKBOX,-25,90,117,25,8\n"
	"125,+,,HICOL,CHECKBOX,-24,55,117,25,8\n"
	".,135,,HICOL,CHECKBOX,-26,90,117,25,8\n"
	"130,+,,HICOL,CHECKBOX,-25,55,117,25,8\n"
	".,135,,HICOL,CHECKBOX,-26,90,117,25,8\n"
	"135,,,HICOL | LASTOBJ,LTEXT,6,15,117,55,8";
bool
GridLine::PropertyDlg()
{
	void *dyndata[] = { (void*)SDLG_GRIDL_TOLINE, (void*)SDLG_GRIDL_TOAXIS,
		(void*)SDLG_GRIDL_GLINE, (void*)OD_linedef, (void*)SDLG_GRIDL_LINETO,
		(void*)SDLG_GRIDL_LINEALIGN, (void*)"Apply to Plot"};
	DlgInfo *LineDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int res;
	long tmptype;
	bool bRet = false;
	DWORD undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	LineDEF newLine;

	if(!parent || !parent->parent) return false;
	if(!(LineDlg = CompileDialog(GridLineDlg_Tmpl, dyndata)))return false;
	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
	if(!(Dlg = new DlgRoot(LineDlg, data)))return false;
	if ((flags & AXIS_ANGULAR) || (flags & AXIS_RADIAL)) {
		//no checkboxes, changed sizes
		}
	else if (parent->parent->parent->Id == GO_TERNARYXYZ) {
		//no checkboxes
		}
	else if(flags &AXIS_3D) {
		Dlg->SetCheck(120, 0L, type & 0x01 ? true : false);
		Dlg->SetCheck(121, 0L, type & 0x02 ? true : false);
		Dlg->SetCheck(125, 0L, type & 0x04 ? true : false);
		Dlg->SetCheck(126, 0L, type & 0x08 ? true : false);
		Dlg->SetCheck(130, 0L, type & 0x10 ? true : false);
		Dlg->SetCheck(131, 0L, type & 0x20 ? true : false);
		switch(parent->parent->type) {
		case 1:	Dlg->ShowItem(105, true);	break;
		case 2:	Dlg->ShowItem(104, true);	break;
		case 3: Dlg->ShowItem(103, true);	break;
			}
		}
	else {
		Dlg->SetCheck(112, 0L, type & DL_LEFT ? true : false);
		Dlg->SetCheck(113, 0L, type & DL_RIGHT ? true : false);
		Dlg->SetCheck(114, 0L, type & DL_YAXIS ? true : false);
		Dlg->SetCheck(115, 0L, type & DL_TOP ? true : false);
		Dlg->SetCheck(116, 0L, type & DL_BOTTOM ? true : false);
		Dlg->SetCheck(117, 0L, type & DL_XAXIS ? true : false);
		if(type & 0x07) Dlg->ShowItem(101, true);
		else Dlg->ShowItem(102, true);
		}
	hDlg = CreateDlgWnd((char*)SDLG_GRIDL_HD1, 50, 50, 435, 320, Dlg, 0x4L);
	do{	
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 1:							//this line
		case 2:							//all lines of axis
		case 3:							//all lines of plot
			if ((res == 3 && (flags & AXIS_ANGULAR)) || (flags & AXIS_RADIAL)) res = 2;
			Undo.SetDisp(cdisp);
			if(flags &AXIS_3D) {
				tmptype = 0;
				if(Dlg->GetCheck(120)) tmptype |= 0x01;
				if(Dlg->GetCheck(121)) tmptype |= 0x02;
				if(Dlg->GetCheck(125)) tmptype |= 0x04;
				if(Dlg->GetCheck(126)) tmptype |= 0x08;
				if(Dlg->GetCheck(130)) tmptype |= 0x10;
				if(Dlg->GetCheck(131)) tmptype |= 0x20;
				}
			else {
				tmptype = (type & ~0xff);
				if(Dlg->GetCheck(112)) tmptype |= DL_LEFT;
				if(Dlg->GetCheck(113)) tmptype |= DL_RIGHT;
				if(Dlg->GetCheck(114)) tmptype |= DL_YAXIS;
				if(Dlg->GetCheck(115)) tmptype |= DL_TOP;
				if(Dlg->GetCheck(116)) tmptype |= DL_BOTTOM;
				if(Dlg->GetCheck(117)) tmptype |= DL_XAXIS;
				}
			OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&newLine, 0);
			break;
			}
		}while (res < 0);
	if(res == 1){			//Apply to line
		undo_flags = CheckNewLong(&type, type, tmptype, parent, undo_flags);
		if(cmpLineDEF(&LineDef, &newLine)) {
			Undo.Line(parent, &LineDef, undo_flags);	undo_flags |= UNDO_CONTINUE;
			memcpy(&LineDef, &newLine, sizeof(LineDEF));
			}
		if(undo_flags & UNDO_CONTINUE) bRet = bModified = true;
		}
	else if(res == 2) {		//Apply to axis
		if(parent->Id == GO_TICK && parent->parent->Id == GO_AXIS &&
			(tmptype != type || cmpLineDEF(&LineDef, &newLine))) {
			parent->parent->Command(CMD_SAVE_TICKS, 0L, 0L);
			if(cmpLineDEF(&LineDef, &newLine)) parent->parent->Command(CMD_SET_GRIDLINE, (void*)&newLine, 0L);
			if(tmptype != type) parent->parent->Command(CMD_SET_GRIDTYPE, (void*)(& tmptype), 0L);
			bRet = true;
			}
		}
	else if (res == 3) {	// Apply to graph or plot
		if (parent->Id == GO_TICK && parent->parent->Id == GO_AXIS){
			parent->parent->parent->Command(CMD_SAVE_TICKS, 0L, 0L);
			parent->parent->parent->Command(CMD_SET_GRIDLINE, (void*)&newLine, 0L);
			bRet = true;
			}
		}
	CloseDlgWnd(hDlg);	delete Dlg;		free(LineDlg);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Tick properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *TickDlg_Tmpl = (char*)
	"1,2,,,PUSHBUTTON,1,120,10,60,12\n"
	"2,3,,DEFAULT, PUSHBUTTON,2,120,25,60,12\n"
	"3,4,,,PUSHBUTTON,-2,120,40,60,12\n"
	"4,,5,ISPARENT | CHECKED,GROUP,,,,,\n"
	"5,6,100,ISPARENT | CHECKED,SHEET,3,5,10,110,100\n"
	"6,7,200,ISPARENT,SHEET,4,5,10,110,100\n"
	"7,,300,ISPARENT,SHEET,5,5,10,110,100\n"
	"100,101,,,RTEXT,6,10,30,35,8\n"
	"101,102,,,EDVAL1,7,50,30,25,10\n"
	"102,103,,,LTEXT,-3,77,30,20,8\n"
	"103,104,,HICOL,CHECKBOX,8,18,45,30, 8\n"
	"104,105,,,LTEXT,9,18,60,20,8\n"
	"105,106,,HICOL,RADIO1,0,40,70,35,8\n"
	"106,107,,HICOL,RADIO1,0,40,60,35,8\n"
	"107,108,,HICOL,RADIO1,10,40,80,35,8\n"
	"108,,,HICOL,CHECKBOX,11,18,95,30,8\n"
	"200,201,,,RTEXT,12,10,35,23,8\n"
	"201,202,,,EDVAL1,13,40,35,65,10\n"
	"202,203,,,LTEXT,14,15,55,20,8\n"
	"203,,,,EDTEXT,0,15,67,90,10\n"
	"300,301,,,LTEXT,15,15,30,70,8\n"
	"301,302,,TOUCHEXIT | HICOL,RADIO1,16,15,42,70,8\n"
	"302,303,,TOUCHEXIT | HICOL,RADIO1,17,15,52,70,8\n"
	"303,304,,TOUCHEXIT | HICOL,RADIO1,-24,15,74,70,8\n"
	"304,305,,TOUCHEXIT | HICOL,RADIO1,-25,15,84,70,8\n"
	"305,306,,TOUCHEXIT | HICOL,RADIO1,-26,15,94,70,8\n"
	"306,307,,, EDVAL1,18,28,62,35,10\n"
	"307,,,LASTOBJ,LTEXT,19,65,62,20,8";

bool
Tick::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_TICK);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_EDIT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DIRECTION);
	double tick_rot;
	void *dyndata[] = { (void*)SDLG_TICK_TOTICK, (void*)SDLG_TICK_TOAXIS, (void*)tab1, (void*)tab2,
		(void*)tab3, (void*)SDLG_TICK_SIZE, (void*)&size, (void*)SDLG_TICK_ISMAJOR, (void*)SDLG_TICK_TYPE,
		(void*)SDLG_TICK_SYM, (void*)SDLG_TICK_DOGRID, (void*)SDLG_TICK_VALUE, (void*)&value,
		(void*)SDLG_TICK_LABEL, (void*)SDLG_TICK_DIREC, (void*)SDLG_TICK_PERPEND, (void*)SDLG_TICK_ANGLE,
		(void*)&tick_rot, (void*)SDLG_TICK_DEG };
	DlgInfo *TickDlg;
	DlgRoot *Dlg;
	void *hDlg;
	char old_value[80];
	double new_size, old_size, new_angle, old_angle;
	int res;
	long tmp_type = type;
	bool bRet = false;
	DWORD new_flags = flags, undo_flags = 0;
	anyOutput *cdisp = Undo.Cdisp();
	char *old_label = 0L;
	TextDEF *td;

	if(!parent || parent->Id != GO_AXIS) return false;
	if(!(TickDlg = CompileDialog(TickDlg_Tmpl, dyndata))) return false;
	switch(type & 0x0f) {
	case 1:		tick_rot = angle;					break;
	default:	tick_rot = trig2deg(lsi, lcsi);		break;
		}
	Dlg = new DlgRoot(TickDlg, data);
	if ((type & 0x0f) == 1)	Dlg->SetCheck(301 + (type & 0x07), 0L, true);
	if(flags & AXIS_ANGULAR) {
		Dlg->SetText(105, (unsigned char*)SDLG_TICK_OUT);		Dlg->SetText(106, (unsigned char*)SDLG_TICK_IN);
		Dlg->ShowItem(303, false);								Dlg->ShowItem(304, false);
		Dlg->ShowItem(305, false);
		}
	else if(flags & AXIS_3D) {
		Dlg->SetText(105, (unsigned char*)SDLG_TICK_POS);		Dlg->SetText(106, (unsigned char*)SDLG_TICK_NEG);
		if(parent->Id == GO_AXIS) {
			//disable tick direction onto the axis
			switch(parent->type) {
			case 1:	Dlg->Activate(303, 0);	break;
			case 2:	Dlg->Activate(304, 0);	break;
			case 3:	Dlg->Activate(305, 0);	break;
				}
			}
		}
	else {
		Dlg->ShowItem(303, false);	Dlg->ShowItem(304, false);
		Dlg->ShowItem(305, false);
		if(abs(pts[1].x - pts[0].x) > abs(pts[1].y - pts[0].y)){
			Dlg->SetText(105, (unsigned char*)SDLG_TICK_RIGHT);		Dlg->SetText(106, (unsigned char*)SDLG_TICK_LEFT);
			}
		else {
			Dlg->SetText(105, (unsigned char*)SDLG_TICK_UP);		Dlg->SetText(106, (unsigned char*)SDLG_TICK_DOWN);
			}
		}
	if(label) {
		if(label->Command(CMD_GETTEXT, &TmpTxt, 0L) && TmpTxt[0] && 
			(old_label = rlp_strdup(TmpTxt)))
			Dlg->SetText(203, (unsigned char*)old_label);
		if(label->Id != GO_LABEL) Dlg->Activate(203, 0);
		}
	switch(flags &0x03) {
	case AXIS_NEGTICKS:		Dlg->SetCheck(106, 0L, true);		break;
	case AXIS_SYMTICKS:		Dlg->SetCheck(107, 0L, true);		break;
	default:				Dlg->SetCheck(105, 0L, true);		break;
		}
	if((flags & AXIS_GRIDLINE) || (Grid)) Dlg->SetCheck(108, 0L, true); 
	if(!(flags & AXIS_MINORTICK)) Dlg->SetCheck(103, 0L, true);
	if(Dlg->GetValue(101, &old_size)) new_size = old_size;
	else new_size = old_size = size;
	if(Dlg->GetValue(306, &old_angle)) new_angle = old_angle;
	else new_angle = old_angle = tick_rot;
	if(flags & AXIS_DATETIME) Dlg->SetText(201, (unsigned char*)NiceTime(value));
	if(!(Dlg->GetText(201, (unsigned char*)old_value, sizeof(old_value)))) {
		old_value[0] = 0;
		}
	hDlg = CreateDlgWnd((char*)SDLG_TICK_HD1, 50, 50, 385, 275, Dlg, 0x4L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 1:		case 2:
			Undo.SetDisp(cdisp);				new_flags &= ~0x7;
			if(Dlg->GetCheck(105)) new_flags |= AXIS_POSTICKS;
			else if(Dlg->GetCheck(106)) new_flags |= AXIS_NEGTICKS;
			else new_flags |= AXIS_SYMTICKS;
			if(Dlg->GetCheck(108)) new_flags |= AXIS_GRIDLINE;
			else if(Grid){
				Undo.DeleteGO((GraphObj**)&Grid, undo_flags, 0L);
				new_flags &= ~AXIS_GRIDLINE;	undo_flags |= UNDO_CONTINUE; 
				}
			Dlg->GetValue(101, &new_size);		Dlg->GetValue(306, &new_angle);
			break;
		case 301:	case 302:	case 303:	case 304:	case 305:
			tmp_type = (tmp_type & ~0x0f) | (res -301);
			res = -1;
			break;
			}
		}while (res <0);
	if(res == 1) {
		if(Dlg->GetCheck(103)) new_flags &= ~AXIS_MINORTICK;
		else new_flags |= AXIS_MINORTICK;
		undo_flags = CheckNewDword(&flags, flags, new_flags, parent, undo_flags);
		undo_flags = CheckNewLong(&type, type, tmp_type, parent, undo_flags);
		undo_flags = CheckNewFloat(&size, old_size, new_size, parent, undo_flags);
		undo_flags = CheckNewFloat(&angle, old_angle, new_angle, parent, undo_flags);
		if(Dlg->GetText(201, (unsigned char*)TmpTxt, 80) && strcmp(TmpTxt, old_value)) {
			Undo.ValFloat(parent, &value, undo_flags);		undo_flags |= UNDO_CONTINUE;
			if(flags & AXIS_DATETIME) date_value(TmpTxt, 0L, &value);
			else Dlg->GetValue(201, &value);
			}
		if(label && label->Id == GO_LABEL) {
			td = ((Label*)label)->GetTextDef();
			if(!(Dlg->GetText(203, (unsigned char*)TmpTxt, TMP_TXT_SIZE))) TmpTxt[0] = 0;
			undo_flags = CheckNewString((unsigned char**)(&td->text), (unsigned char*)(td->text), (unsigned char*)TmpTxt, this, undo_flags);
			if (undo_flags) label->Command(CMD_SETTEXT, TmpTxt, 0L);
			}
		else if(!label){
			if (!(Dlg->GetText(203, (unsigned char*)TmpTxt, TMP_TXT_SIZE))) TmpTxt[0] = 0;
			Command(CMD_SETTEXT, TmpTxt, 0L);
			}
		if (undo_flags & UNDO_CONTINUE) bRet = true;
		}
	else if(res == 2) {
		parent->Command(CMD_SAVE_TICKS, 0L, 0L);
		if (Dlg->GetCheck(103)) new_flags &= ~AXIS_MINORTICK;
		else new_flags |= AXIS_MINORTICK;
		parent->Command(CMD_SET_TICKSTYLE, &new_flags, 0L);
		parent->SetSize(SIZE_AXIS_TICKS, new_size);
		parent->Command(CMD_TICK_TYPE, &tmp_type, 0L);
		parent->SetSize(SIZE_TICK_ANGLE, new_angle);
		parent->Command(CMD_REDRAW, 0L, 0L);
		bRet = true;
		}
	CloseDlgWnd(hDlg);		delete Dlg;		free(TickDlg);
	free(tab1);		free(tab2);			free(tab3);
	if (old_label) free(old_label);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Axis properties dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *ssTickTmpl = (char*)
	"1,+,,DEFAULT, PUSHBUTTON,-1,113,10,45,12\n"
	".,.,,, PUSHBUTTON,-2,113,25,45,12\n"
	".,.,,, LTEXT,1,5,10,100,9\n"
	".,.,,TOUCHEXIT,RANGEINPUT,-15,10,20,90,10\n"
	".,.,,, LTEXT,2,5, 32, 100, 9\n"
	".,.,,TOUCHEXIT,RANGEINPUT,-16,10,42,90,10\n"
	".,.,,,LTEXT,3, 5, 54, 80, 8\n"
	".,,,LASTOBJ | TOUCHEXIT,RANGEINPUT,-17,10,64,90,10";

bool
Axis::ssTicks()
{
	void *dyndata[] = { (void*)SDLG_SSTICK_RNG1, (void*)SDLG_SSTICK_RNG2,
		(void*)"minor tick VALUES:"};
	DlgInfo *TickDlg = CompileDialog(ssTickTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, n, n1, n2, n3;
	bool bRet = false, bContinue = true;
	AccRange *rT, *rL, *rMT;

	if(!data) return false;
	n = n1 = n2 = n3 = 0;			rT = rL = rMT = 0L;
	TmpTxt[0] = TmpTxt[100] = TmpTxt[200] = 0;
	if (ssMATval) rlp_strcpy(TmpTxt, 90, ssMATval);
	if (ssMATlbl) rlp_strcpy(TmpTxt + 100, 90, ssMATlbl);
	if (ssMITval) rlp_strcpy(TmpTxt + 200, 90, ssMITval);
	if (!ssMATval) {
		if(!UseRangeMark(data, 1, TmpTxt, TmpTxt + 100)) return false;
		}
	if (!(Dlg = new DlgRoot(TickDlg, data)))return false;
	hDlg = CreateDlgWnd((char*)SDLG_SSTICK_HD1, 50, 50, 350, 210, Dlg, 0x4L);
	Dlg->Activate(4, 1);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:								// focus lost
			if(bContinue) res = -1;
			break;
		case 4:		case 6:		case 8:
			bContinue = true;
			res = -1;			break;
		case 1:
			if(rT) delete rT;
			if(rL) delete rL;
			if(rMT) delete rMT;
			n = n1 = n2 = n3 = 0;
			rT = rL = rMT = 0L;
			if(Dlg->GetText(4, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && (rT=new AccRange(TmpTxt)) && (n1=rT->CountItems())){
				if(Dlg->GetText(6, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && (rL = new AccRange(TmpTxt)) && 
					(n2 = rL->CountItems()) && n1 != n2){
					ErrorBox((char*)SDLG_SSTICK_ERR);
					res = -1;		bContinue = true;
					break;
					}
				}
			if(Dlg->GetText(8, (unsigned char*)TmpTxt,TMP_TXT_SIZE) && (rMT=new AccRange(TmpTxt)) && (n3=rMT->CountItems())){
				//minor ticks are valid
				}
			if(!(n = n1 + n3)) {
				ErrorBox((char*)SCMS_BAD_RANGE);
				res = -1;			bContinue = true;
				}
			}
		}while (res < 0);
	if(res == 1 && n) {
		if(n1) {
			if(ssMATval) free(ssMATval);
			if(ssMATlbl) free(ssMATlbl);
			if(ssMITval) free(ssMITval);
			ssMATval = 0L; ssMATlbl = 0L; ssMITval = 0L;
			if(Dlg->GetText(4, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) 
				ssMATval = rlp_strdup(TmpTxt);
			if(Dlg->GetText(6, (unsigned char*)TmpTxt, TMP_TXT_SIZE))
				ssMATlbl = rlp_strdup(TmpTxt);
			if(Dlg->GetText(8, (unsigned char*)TmpTxt, TMP_TXT_SIZE))
				ssMITval = rlp_strdup(TmpTxt);
			UpdateTicks();
			}
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;			free(TickDlg);
	if(rT) delete rT;
	if(rL) delete rL;
	if(rMT) delete rMT;
	return bRet;
}

static char *AxisPropDlg_Tmpl = (char*)
	"1,2,,DEFAULT,PUSHBUTTON,-1,186,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,186,25,45,12\n"
	"3,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,5,50,ISPARENT | CHECKED,SHEET,1,5,10,168,165\n"
	"5,6,200,ISPARENT,SHEET,2,5,10,168,165\n"
	"6,7,300,ISPARENT,SHEET,3,5,10,168,165\n"
	"7,8,400,ISPARENT,SHEET,4,5,10,168,165\n"
	"8,9,500,HIDDEN | ISPARENT,SHEET,5,5,10,168,165\n"
	"9,,550,HIDDEN | ISPARENT,SHEET,6,5,10,168,165\n"
	"50,+,100,ISPARENT | CHECKED,GROUPBOX,7,10,30,158,36\n"
	".,104,,HICOL | TOUCHEXIT,CHECKBOX,8,17,37,80,8\n"
	"100,+,,,RTEXT,9,10,51,35,8\n"
	".,.,,,EDVAL1,10,48,51,51,10\n"
	".,.,,,CTEXT,11,100,51,11,8\n"
	".,,,,EDVAL1,12,112,51,51,10\n"
	".,120,105,ISPARENT | CHECKED,GROUPBOX,13,10,72,158,45\n"
	".,106,,HICOL | TOUCHEXIT | ISRADIO,CHECKBOX,-20,54,77,20,8\n"
	".,.,,HICOL | TOUCHEXIT | ISRADIO,CHECKBOX,-21,84,77,20,8\n"
	".,.,,HICOL | TOUCHEXIT | ISRADIO,CHECKBOX,-22,54,77,20,8\n"
	".,.,,HICOL | TOUCHEXIT | ISRADIO,CHECKBOX,-23,84,77,20,8\n"
	".,.,,,RTEXT,-4,10,-50,15,8\n"
	".,.,,,EDVAL1,14,27,-50,45,10\n"
	".,.,,,LTEXT,-7,75,-50,5,8\n"
	".,.,,,EDVAL1,15,81,-50,45,10\n"
	".,.,,,LTEXT,0,129,-50,15,8\n"
	".,.,,,RTEXT,-5,10,-38,15,8\n"
	".,.,,,EDVAL1,16,27,-38,45,10\n"
	".,.,,,LTEXT,-7,75,-38,5,8\n"
	".,.,,,EDVAL1,17,81,-38,45,10\n"
	".,150,,,LTEXT,0,129,-38,15,8\n"
	"150,+,,,RTEXT,-6,10,-26,15,8\n"
	".,.,,,EDVAL1,18,27,-26,45,10\n"
	".,.,,,LTEXT,-7,75,-26,5,8\n"
	".,.,,,EDVAL1,19,81,-26,45,10\n"
	".,,,,LTEXT,0,129,-26,15,8\n"
	"120,180, 121, ISPARENT | CHECKED,GROUPBOX,20,10,123,158,20\n"
	".,+,,,RTEXT,21,20,128,25,8\n"
	".,.,,,EDVAL1,22,47,128,25,10\n"
	".,.,,,LTEXT,-3,73,128, 10,8\n"
	".,.,,,RTEXT,-11,102,128,25,8\n"
	".,130,,OWNDIALOG | TOUCHEXIT,COLBUTT,23,129,128,25,10\n"
	"130,,131,ISPARENT | CHECKED,GROUPBOX,24,10,149,158,20\n"
	"131,,,,EDTEXT,0,15,154,148,10\n"
	"180,,181,HIDDEN | ISPARENT | CHECKED,GROUPBOX,13,10,72,158,45\n"
	"181,+,,,RTEXT,25,10,-62,50,8\n"
	".,.,,,EDVAL1,26,62,-62,45,10\n"
	".,.,,,LTEXT,-3,109,-62,15,8\n"
	".,.,,,RTEXT,-5,10,-50,50,8\n"
	".,.,,,EDVAL1,27,62,-50,45,10\n"
	".,.,,,LTEXT,-3,109,-50,15,8\n"
	".,.,,,RTEXT,28,10,-38,50,8\n"
	".,.,,,EDVAL1,29,62,-38,45,10\n"
	".,,,,LTEXT,-3,109,-38,15,8\n"
	"200,+,,HICOL | TOUCHEXIT,RADIO1,30,20,32,40,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,31,20,42,40,8\n"
	".,204,,HICOL | TOUCHEXIT,RADIO1,32,20,54,40,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,33,20,71,40,8\n"
	".,203,250,CHECKED | ISPARENT,GROUPBOX,0,15,75,148,50\n"
	".,206,,,PUSHBUTTON,34,30,137,90,12\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,0,20,137,8,8\n"
	".,.,,HICOL | TOUCHEXIT,CHECKBOX,69,20,155,75,8\n"
	".,,,TOUCHEXIT,EDTEXT,70,110,155,40,10\n"
	"250,+,,,RTEXT,35,35,83,45,8\n"
	".,.,,TOUCHEXIT,EDVAL1,36,87,83,65,10\n"
	".,.,,,RTEXT,37,35,95,45,8\n"
	".,.,,TOUCHEXIT,EDVAL1,38,87,95,65,10\n"
	".,.,,,RTEXT,39,25,107,75,8\n"
	".,,,TOUCHEXIT,EDTEXT,40,107,107,45,10\n"
	"300,+,,,LTEXT,41,20,30,110,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,42,40,45,70,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,43,40,57,70,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,44,40,69,70,8\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,45,40,81,70,8\n"
	".,,,HICOL,CHECKBOX,46,20,115,80,12\n"
	"400,450,,,LTEXT,-27,10,30,110,8\n"
	".,402,,ISRADIO,ODB,47,49,40,,\n"
	".,.,,ISRADIO,ODB,47,74,40,,\n"
	".,.,,ISRADIO,ODB,47,99,40,,\n"
	".,.,,ISRADIO,ODB,47,124,40,,\n"
	".,.,,,RTEXT,48,25,75,48,8\n"
	".,.,,,EDVAL1,49,75,75,29,10\n"
	".,.,,,LTEXT,-3,107,75,10,8\n"
	".,.,,,RTEXT,50,25,87,48,8\n"
	".,.,,,EDVAL1,51,75,87,29,10\n"
	".,,,,LTEXT,-3,107,87,10,8\n"
	"450,+,,ISPARENT | CHECKED,GROUPBOX,52,10,110,158,50\n"
	".,.,,,LTEXT,0,30,115,60,8\n"
	".,.,,,PUSHBUTTON,-8,108,142,41,12\n"
	".,.,,,PUSHBUTTON,-9,67,142,41,12\n"
	".,.,,,RTEXT,53,25,125,20,8\n"
	".,.,,,EDTEXT,0,47,125,41,10\n"
	".,.,,,RTEXT,54,92,125,9,8\n"
	".,.,,,EDTEXT,0,103,125,41,10\n"
	".,401,,,PUSHBUTTON,55,26,142,41,12\n"
	"500,,,NOSELECT,ODB,56,25,30,140,140\n"
	"550,,551,ISPARENT | CHECKED,GROUPBOX,57,15,30,148,123\n"
	".,+,,HICOL | TOUCHEXIT,RADIO1,58,30,38,75,9\n"
	".,.,,ODEXIT,COLBUTT,59,112,37,25,10\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,60,30,52,80,9\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,61,30,76,80,9\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,62,30,88,80,9\n"
	".,.,,HICOL | TOUCHEXIT,RADIO1,63,30,64,80,9\n"
	".,.,,OWNDIALOG | TOUCHEXIT,COLBUTT,64,50,100,25,10\n"
	".,.,,,CTEXT,54,76,100,18,9\n"
	".,.,,ODEXIT,COLBUTT,65,95,100,25,10\n"
	".,.,,HICOL,CHECKBOX,66,30,115,80,9\n"
	".,.,,,RTEXT,67,30,128,45,9\n"
	".,.,,,INCDECVAL1,68,77,128,33,10\n"
	".,,,LASTOBJ,LTEXT,-10,111,128,10,9";

bool
Axis::PropertyDlg()
{
	int i, j;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_AXIS);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TICKS);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_TRANSFORMS);
	j = i;
	TabSHEET *tab4 = MakeTab(i, 10, &i, (char*)SDLG_TAB_BREAKS);
	TabSHEET *tab5 = MakeTab(i, 10, &i, (char*)SDLG_TAB_PLOTS);
	TabSHEET *tab6 = MakeTab(j, 10, &i, (char*)SDLG_TAB_GRADIENT);

	int v1 = (axis->flags & AXIS_3D) ? 77 : (axis->flags & AXIS_RADIAL) ? 83 : 89; 
	double transp = (double)iround((double)((gTrans>>24) & 0xff)/2.55);
	void *dyndata[] ={(void*)tab1, (void*)tab2, (void*)tab3, (void*)tab4, (void*)tab5, (void*)tab6,
		(void*)SDLG_AXIS_SCALE, (void*)SDLG_AXIS_AUTOSC, (void*)SDLG_AXIS_FROM, (void*)&axis->min, (void*)SDLG_AXIS_TO,
		(void*)&axis->max, (void*)SDLG_AXIS_PLC, (void*)&axis->loc[0].fx, (void*)&axis->loc[1].fx,
		(void*)&axis->loc[0].fy, (void*)&axis->loc[1].fy, (void*)&axis->loc[0].fz, (void*)&axis->loc[1].fz,
		(void*)SDLG_AXIS_LINE, (void*)SDLG_AXIS_LWIDTH, (void*)&sizAxLine, (void*)&colAxis, (void*)SDLG_AXIS_LBL,
		(void*)SDLG_AXIS_CX, (void*)&axis->Center.fx, (void*)&axis->Center.fy, (void*)SDLG_AXIS_RAD,
		(void*)&axis->Radius, (void*)SDLG_AXIS_NOTI, (void*)SDLG_AXIS_AUTI, (void*)SDLG_AXIS_UCTI,
		(void*)SDLG_AXIS_MANTI, (void*)SDLG_AXIS_SSVAL, (void*)SDLG_AXIS_STVAL, (void*)&axis->Start,
		(void*)SDLG_AXIS_INTV, (void*)&axis->Step, (void*)SDLG_AXIS_MINT, (void*)SDLG_AXIS_ZERO, (void*)SDLG_AXIS_TFORM,
		(void*)SDLG_AXIS_TNONE, (void*)SDLG_AXIS_TLOG, (void*)SDLG_AXIS_TRECI, (void*)SDLG_AXIS_TSQR,
		(axis->loc[0].fx == axis->loc[1].fx) ? (void*)SDLG_AXIS_LVTOP : (void*)SDLG_AXIS_LVRIGHT,
		(void*)(OD_BreakTempl), (void*)SDLG_AXIS_BGAP, (void*)&brkgap, (void*)SDLG_AXIS_SSIZE, (void*)&brksymsize,
		(void*)SDLG_AXIS_BDATA, (void*)SDLG_AXIS_BFROM, (void*)SDLG_AXIS_BTO, (void*)SDLG_AXIS_DEL, (void*)OD_axisplot,
		(void*)SDLG_AXIS_CGRAD, (void*)SDLG_AXIS_CFILL, (void *)&gCol_0, (void*)SDLG_AXIS_RBOW_I,
		(void*)SDLG_AXIS_XRBOW, (void*)SDLG_AXIS_CORGR, (void*)SDLG_AXIS_RBOW_II, (void *)&gCol_1, (void *)&gCol_2,
		(void*)SDLG_AXIS_INVGR, (void*)SDLG_AXIS_TRANS, (void*)&transp, (void*)SDLG_AXIS_TLBF, (void*)TLBformat };
	DlgInfo *AxisPropDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int res, nbrk, cbrk, ttmpl, n_gradient;
	double tmp, tmp2, old_x1, old_x2, old_y1, old_y2;
	bool bRet = false, upd_brk = true, bNewScale = false;
	bool bContinue = false, bUpdPG = false;
	double use_step = 10.0, use_minmax[] = {0.0, 100.0};
	DWORD new_color, undo_flags = 0L;
	anyOutput *cdisp = Undo.Cdisp();
	int undo_pos = Undo.level();
	lfPOINT *brks = 0L, *tmpbrks = 0L;
	unsigned char *old_Label = 0L, *type_txt;
	TextDEF label_def, *lb_def;
	unsigned char **names;
	GraphObj **somePlots = 0L, **scp = 0L;
	AxisDEF old_a, new_a;
	void *sv_ptr;
	anyResult any_res;

	nbrk = cbrk = 0;
	if(!parent) return false;
	if(!(AxisPropDlg = CompileDialog(AxisPropDlg_Tmpl, dyndata))) return false;
	n_gradient = grad_type;
	//adjust dialog to 3D plot, polar plot ...
	for(i = 0, v1+= 50; !(AxisPropDlg[i].flags & LASTOBJ); i++) {
		if(AxisPropDlg[i].y < 0) AxisPropDlg[i].y += v1;
		}
	if(parent->Id == GO_GRAPH && (res=((Graph*)parent)->nscp)){
		scp = ((Graph*)parent)->Sc_Plots;
		CurrAxes = ((Graph*)parent)->Axes;
		if(!scp || !(names = (unsigned char**)calloc(res+2, sizeof(unsigned char*))) ||
			!(somePlots = (GraphObj**)calloc(res+2, sizeof(GraphObj*))))
			return false;
		names[0] = (unsigned char*)malloc(15 * sizeof(unsigned char));
		if (names[0]) rlp_strcpy(names[0], 15, (char*)SDLG_AXIS_UNCH);
		for(i = 0, j = 1; i < res; i++) {
			if(scp[i] && scp[i]->name){
				names[j] = rlp_strdup((unsigned char*)scp[i]->name);
				somePlots[j++] = scp[i];
				}
			}
		}
	else if(IsPlot3D(parent) && (res=((Plot3D*)parent)->nscp)){
		scp = ((Plot3D*)parent)->Sc_Plots;
		CurrAxes = ((Plot3D*)parent)->Axes;		NumCurrAxes = (int)(((Plot3D*)parent)->nAxes);
		if(!scp || !(names = (unsigned char**)calloc(res+2, sizeof(unsigned char*))) ||
			!(somePlots = (GraphObj**)calloc(res+2, sizeof(GraphObj*))))
			return false;
		names[0] = (unsigned char*)malloc(15 * sizeof(unsigned char));
		if (names[0]) rlp_strcpy(names[0], 15, (char*)SDLG_AXIS_UNCH);
		for(i = 0, j = 1; i < res; i++) {
			if(scp[i] && scp[i]->name){
				names[j] = rlp_strdup((unsigned char*)scp[i]->name);
				somePlots[j++] = scp[i];
				}
			}
		}
	else {
		names = (unsigned char**)calloc(2, sizeof(unsigned char*));	names[0] = (unsigned char*)malloc(10*sizeof(unsigned char));
		rlp_strcpy(names[0], 10, (unsigned char*)SDLG_AXIS_NA);
		}
	OD_axisplot(OD_ACCEPT, 0L, 0L, 0L, names, 0);
	if(!(Dlg = new DlgRoot(AxisPropDlg, data)))return false;
	if(names && somePlots) Dlg->ShowItem(8, true);		//show tab
	if(axis->breaks && axis->nBreaks) {
		brks = (lfPOINT*)calloc(axis->nBreaks + 2, sizeof(lfPOINT));
		if (!brks) return false;
		memcpy(brks, axis->breaks, axis->nBreaks*sizeof(lfPOINT));
		nbrk = axis->nBreaks-1;
		WriteNatFloatToBuff(TmpTxt, brks[cbrk].fx);
		Dlg->SetText(455, (unsigned char*)TmpTxt+1);
		WriteNatFloatToBuff(TmpTxt, brks[cbrk].fy);
		Dlg->SetText(457, (unsigned char*)(TmpTxt+1));
		}
	switch(brksym) {
	case 2:		Dlg->SetCheck(402, 0L, true);	break;
	case 3:		Dlg->SetCheck(403, 0L, true);	break;
	case 4:		Dlg->SetCheck(404, 0L, true);	break;
	default:	Dlg->SetCheck(401, 0L, true);	break;
		}
	if(!(axis->flags & 0x03)) Dlg->SetCheck(ttmpl = 200, 0L, true);
	else if(axis->flags & AXIS_AUTOTICK)Dlg->SetCheck(ttmpl = 201, 0L, true);
	else if(Ticks) Dlg->SetCheck(ttmpl = 202, 0L, true);
	else Dlg->SetCheck(ttmpl = 201, 0L, true);
	Dlg->SetCheck(305, 0L, (AXIS_INVERT == (axis->flags & AXIS_INVERT)));
	if(axis->flags & AXIS_AUTOSCALE) {
		Dlg->SetCheck(51, 0L, true);
		Dlg->Activate(101, 0);		Dlg->Activate(103, 0);
		}
	//check transforms
	switch(axis->flags & 0x7000L) {
	case AXIS_LINEAR:	Dlg->SetCheck(301, 0L, true);	break;
	case AXIS_LOG:		Dlg->SetCheck(302, 0L, true);	break;
	case AXIS_RECI:		Dlg->SetCheck(303, 0L, true);	break;
	case AXIS_SQR:		Dlg->SetCheck(304, 0L, true);	break;
		}
	if(axis->flags & AXIS_3D) {
		Dlg->ShowItem(105, false);	Dlg->ShowItem(106, false);
		Dlg->ShowItem(107, false);	Dlg->ShowItem(108, false);
		}
	else if(axis->flags & AXIS_ANGULAR) {
		Dlg->ShowItem(104, false);	Dlg->ShowItem(130, false);
		Dlg->ShowItem(180, true);	Dlg->ShowItem(7, false);
		}
	else {
		if(axis->flags & AXIS_RADIAL) {
			Dlg->ShowItem(105, false);	Dlg->ShowItem(106, false);
			Dlg->ShowItem(107, false);	Dlg->ShowItem(108, false);
			}
		Dlg->ShowItem(150, false);	Dlg->ShowItem(151, false);
		Dlg->ShowItem(152, false);	Dlg->ShowItem(153, false);
		Dlg->ShowItem(154, false);
		if(axis->loc[0].fx != axis->loc[1].fx) {
			Dlg->ShowItem(105, false);	Dlg->ShowItem(106, false);
			}
		if(axis->loc[0].fy != axis->loc[1].fy) {
			Dlg->ShowItem(107, false);	Dlg->ShowItem(108, false);
			}
		}
	//align to frame ?
	switch(axis->flags & 0x70) {
		case AXIS_LEFT: Dlg->SetCheck(105, 0L, true);	break;
		case AXIS_RIGHT: Dlg->SetCheck(106, 0L, true);	break;
		case AXIS_TOP: Dlg->SetCheck(107, 0L, true);	break;
		case AXIS_BOTTOM: Dlg->SetCheck(108, 0L, true);	break;
		}
	if (axis->flags & AXIS_X_DATA) Dlg->SetText(113, (unsigned char*)SDLG_AXIS_DATA);
	else Dlg->SetText(113, (unsigned char*)Units[defs.units()].display);
	if (axis->flags & AXIS_Y_DATA) Dlg->SetText(118, (unsigned char*)SDLG_AXIS_DATA);
	else Dlg->SetText(118, (unsigned char*)Units[defs.units()].display);
	if (axis->flags & AXIS_Z_DATA) Dlg->SetText(118, (unsigned char*)SDLG_AXIS_DATA);
	else Dlg->SetText(154, (unsigned char*)Units[defs.units()].display);
	//any label ?
	if(axisLabel){
		TmpTxt[0] = 0;
		axisLabel->Command(CMD_GETTEXT, TmpTxt, 0L);
		Dlg->SetText(131, (unsigned char*)TmpTxt);
		old_Label = rlp_strdup((unsigned char*)TmpTxt);
		if(axisLabel->Id == GO_MLABEL) Dlg->Activate(131, 0);
		}
	else Dlg->SetText(131, 0L);
	//remember: any updated values ?
	Dlg->GetValue(110, &old_x1);		Dlg->GetValue(112, &old_x2);
	Dlg->GetValue(115, &old_y1);		Dlg->GetValue(117, &old_y2);
	if(!name) switch(type) {
	case 1:		type_txt = (unsigned char*)(&SDLG_AXIS_XAX);		break;
	case 2:		type_txt = (unsigned char*)(&SDLG_AXIS_YAX);		break;
	case 3:		type_txt = (unsigned char*)(&SDLG_AXIS_ZAX);		break;
	case 4:		type_txt = (unsigned char*)(&SDLG_AXIS_GAX);		break;
	default:	type_txt = (unsigned char*)(&SDLG_AXIS_AX);			break;
		}
	else type_txt = (unsigned char*) name;
	//angular radial axis specials
	if(axis->flags & AXIS_ANGULAR){
		Dlg->SetText(305, (unsigned char*)SDLG_AXIS_DCW);
		type_txt = (unsigned char*)(&SDLG_AXIS_AAX);
		}
	if (axis->flags & AXIS_RADIAL) type_txt = (unsigned char*)(&SDLG_AXIS_RAX);
	//save old axis definition
	memcpy(&old_a, axis, sizeof(AxisDEF));
	Dlg->GetValue(101, &old_a.min);		Dlg->GetValue(103, &old_a.max);
	Dlg->GetValue(110, &old_a.loc[0].fx);	Dlg->GetValue(112, &old_a.loc[1].fx);
	Dlg->GetValue(115, &old_a.loc[0].fy);	Dlg->GetValue(117, &old_a.loc[1].fy);
	Dlg->GetValue(151, &old_a.loc[0].fz);	Dlg->GetValue(153, &old_a.loc[1].fz);
	Dlg->GetValue(182, &old_a.Center.fx);	Dlg->GetValue(185, &old_a.Center.fy);
	Dlg->GetValue(188, &old_a.Radius);
	memcpy(&new_a, &old_a, sizeof(AxisDEF));
	i = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, type_txt);
	rlp_strcpy(TmpTxt + i, TMP_TXT_SIZE - i, (char*)SDLG_AXIS_PROP);
	if((type &0x0f) == 4) {
		Dlg->ShowItem(7, false);	Dlg->ShowItem(8, false);
		Dlg->ShowItem(9, true);		upd_brk = false;
		switch(grad_type & 0x0f) {
		case 0:
			Dlg->SetCheck(551, 0L, true);				break;
		case 1:		default:
			grad_type = n_gradient = 1;
			Dlg->SetCheck(553, 0L, true);				break;
		case 2:		case 3:		case 4:
			Dlg->SetCheck(552+(grad_type & 0x0f), 0L, true);	break;
			}
		if(grad_type & 0x10) Dlg->SetCheck(560, 0L, true);
		Dlg->ItemCmd(562, CMD_STEP, (void*)&use_step);
		Dlg->ItemCmd(562, CMD_MINMAX, (void*)&use_minmax);
		}
	Dlg->Activate(208, 0);					//tick label format
	if (axis->flags & AXIS_DATETIME) {		//process Date- Time axis
		if (axis->min > 600.0 || axis->max > 600.0){
			if ((axis->min - floor(axis->min)) < 0.0001 && (axis->max - floor(axis->max)) < 0.001)
				i = ET_DATE;
			else i = ET_DATETIME;
			}
		else if (axis->max < 2.0) i = ET_TIME;
		any_res.text = 0;					any_res.value = axis->min;
		any_res.type = i;					TranslateResult(&any_res);
		Dlg->SetText(101, (unsigned char*)any_res.text);
		any_res.text = 0;					any_res.value = axis->max;
		any_res.type = i;					TranslateResult(&any_res);
		Dlg->SetText(103, (unsigned char*)any_res.text);
		any_res.text = 0;					any_res.value = axis->Start;
		any_res.type = i;					TranslateResult(&any_res);
		Dlg->SetText(251, (unsigned char*)any_res.text);
		}
	Dlg->Activate(251, 0);	Dlg->Activate(253, 0);	Dlg->Activate(255, 0);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 496, 410, Dlg, 0x4L);
	switch(axis->flags & 0x70) {
		case AXIS_LEFT: 
		case AXIS_RIGHT:
			Dlg->Activate(110, 0);			Dlg->Activate(112, 0);
			break;
		case AXIS_TOP:
		case AXIS_BOTTOM:
			Dlg->Activate(115, 0);			Dlg->Activate(117, 0);
			break;
			}
	do{
		if(upd_brk) {
			Dlg->ShowItem(453, cbrk > 0);
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SDLG_AXIS_BFMT, cbrk + 1, nbrk + 1);
#else
			sprintf(TmpTxt,SDLG_AXIS_BFMT, cbrk+1, nbrk+1);
#endif
			Dlg->SetText(451, (unsigned char*)TmpTxt);
			upd_brk = false;
			}
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:							//lost focus ?
			if(bContinue) res = -1;
			ShowDlgWnd(hDlg);
			break;
		case -1:
			bContinue = false;
			break;
		case 51:
			if(Dlg->GetCheck(51)) {
				Dlg->Activate(101, 0);		Dlg->Activate(103, 0);
				}
			else {
				Dlg->Activate(101, 1);		Dlg->Activate(103, 1);
				}
			res = -1;
			break;
		case 105:	case 106:	case 107:	case 108:		//axis left | right | top | bottom
			if(Dlg->GetCheck(105) || Dlg->GetCheck(106)) {
				Dlg->Activate(110, 0);			Dlg->Activate(112, 0);
				}
			else {
				Dlg->Activate(110, 1);			Dlg->Activate(112, 1);
				}
			if(Dlg->GetCheck(107) || Dlg->GetCheck(108)) {
				Dlg->Activate(115, 0);			Dlg->Activate(117, 0);
				}
			else {
				Dlg->Activate(115, 1);			Dlg->Activate(117, 1);
				}
			res = -1;			break;
		case 125:											//the color button
			if (Dlg->GetColor(125, &new_color) && new_color != colAxis) {
				CheckNewDword(&colAxis, colAxis, new_color, this, undo_pos < Undo.level() ? UNDO_CONTINUE : 0);
				if ((axis->flags & AXIS_ANGULAR) || (axis->flags & AXIS_RADIAL)) {
					Undo.ValDword(this, &GridLine.color, UNDO_CONTINUE);
					GridLine.color = new_color;
					}
				if (Ticks || axisLabel) {
					SavVarInit(200 * NumTicks);
					if (axisLabel){
						axisLabel->FileIO(SAVE_VARS);		axisLabel->SetColor(COL_TEXT, colAxis);
						}
					if (Ticks) for (i = 0; i < NumTicks; i++)  if (Ticks[i]){
						Ticks[i]->FileIO(SAVE_VARS);		Ticks[i]->SetColor(COL_AXIS, colAxis);
						}
					sv_ptr = SavVarFetch();
					Undo.SavVarBlock(this, &sv_ptr, UNDO_CONTINUE);
					}
				parent->Command(CMD_REDRAW, 0L, cdisp);
				}
			res = -1;			break;
		case 301:	case 302:	case 303:	case 304:		//transform radiobuttons
			new_a.flags &= ~0x7000L;
			if(res == 302) new_a.flags |= AXIS_LOG;
			else if(res == 303) new_a.flags |= AXIS_RECI;
			else if(res == 304) new_a.flags |= AXIS_SQR;
			res = -1;
			break;
		case 200:		case 201:		case 202:
			if (res == 200 && Ticks && NumTicks){					//axis no ticks
				Undo.AxisDef(this, axis, 0L);
				axis->flags &= ~(0x03);				axis->flags &= ~(AXIS_AUTOTICK | AXIS_MINORTICK);
				Undo.DeleteGO(&axisLabel, UNDO_CONTINUE, cdisp);
				Undo.DropListGO(this, (GraphObj***)(&Ticks), &NumTicks, undo_flags);
				undo_flags |= UNDO_CONTINUE;
				parent->Command(CMD_REDRAW, 0L, cdisp);
				}
			if (res == 201) {										//axis automatic ticks
				bNewScale = true;
				Undo.AxisDef(this, axis, 0L);
				axis->flags |= AXIS_AUTOTICK;			axis->flags &= ~AXIS_MINORTICK;
				if (Ticks && NumTicks) {
					if ((bNewScale = YesNoBox((char*)SCMS_RECALC_TICKS))){
						Undo.DropListGO(this, (GraphObj***)(&Ticks), &NumTicks, undo_flags);
						undo_flags |= UNDO_CONTINUE;
						}
					}
				if (bNewScale) {
					parent->Command(CMD_REDRAW, 0L, cdisp);			//will recalc ticks
					}
				}
			if (res == 202) {}										//no change
			ttmpl = res;
			Dlg->Activate(251, 0);	Dlg->Activate(253, 0);	Dlg->Activate(255, 0);
			res = -1;		break;
		case 251:	case 253:	case 255:
			Dlg->SetCheck(203, 0L, true);
			Dlg->Activate(251, 1);	Dlg->Activate(253, 1);	Dlg->Activate(255, 1);
			Dlg->Activate(res, 1);
			res = -1;
			break;
		case 203:
			ttmpl = res;
			Dlg->Activate(251, 1);	Dlg->Activate(253, 1);	Dlg->Activate(255, 1);
			res = -1;
			break;
		case 205:	case 206:
			Dlg->SetCheck(206, 0L, true);
			if (ssTicks()) {
				Dlg->SetCheck(ttmpl = 202, 0L, true);	Dlg->Activate(251, 0);
				Dlg->Activate(253, 0);	Dlg->Activate(255, 0);
				undo_flags |= UNDO_CONTINUE;	res = 1;
				}
			else {
				Dlg->SetCheck(ttmpl, 0L, true);
				res = -1;
				}
			bContinue = true;
			break;
		case 207:	case 208:				//tick label format
			if(Dlg->GetCheck(207)) {
				Dlg->Activate(208, 1);
				}
			else {
				Dlg->Activate(208, 0);
				HideTextCursor();
				}
			res = -1;
			break;
		case 452:			//next break
			if(Dlg->GetValue(455, &tmp) && Dlg->GetValue(457, &tmp2)){
				if(!brks && (brks = (lfPOINT*)calloc(2, sizeof(lfPOINT)))) {
					brks[0].fx = tmp;		brks[0].fy = tmp2;
					cbrk = nbrk = 0;
					}
				if(!brks) return false;		//mem allocation error
				if(cbrk && cbrk >= nbrk) {
					tmpbrks = (lfPOINT*)realloc(brks, (cbrk + 2)*sizeof(lfPOINT));
					if (tmpbrks){
						brks = tmpbrks;
						brks[cbrk].fx = tmp;		brks[cbrk].fy = tmp2;
						brks[cbrk+1].fx = brks[cbrk+1].fy = 0.0;
						}
					else return false;		//mem allocation error
					}
				else {
					brks[cbrk].fx = tmp;		brks[cbrk].fy = tmp2;
					}
				cbrk++;
				}
			if(cbrk>nbrk){
				nbrk = cbrk;
				Dlg->SetText(455,0L);		Dlg->SetText(457,0L);
				}
			else if(nbrk){
				if(brks[cbrk].fx == brks[cbrk].fy && cbrk == nbrk) {
					Dlg->SetText(455,0L);		Dlg->SetText(457,0L);
					}
				else {
					WriteNatFloatToBuff(TmpTxt, brks[cbrk].fx);
					Dlg->SetText(455, (unsigned char*)(TmpTxt+1));
					WriteNatFloatToBuff(TmpTxt, brks[cbrk].fy);
					Dlg->SetText(457, (unsigned char*)(TmpTxt+1));
					}
				}
			upd_brk = true;
			res= -1;
			break;
		case 453:			//previous break
			if(cbrk >0){
				if(Dlg->GetValue(455, &tmp) && Dlg->GetValue(457, &tmp2)){
					brks[cbrk].fx = tmp;		brks[cbrk].fy = tmp2;
					}
				else if(cbrk == nbrk)nbrk--;
				cbrk--;
				WriteNatFloatToBuff(TmpTxt, brks[cbrk].fx);
				Dlg->SetText(455, (unsigned char*)(TmpTxt+1));
				WriteNatFloatToBuff(TmpTxt, brks[cbrk].fy);
				Dlg->SetText(457, (unsigned char*)(TmpTxt+1));
				}
			upd_brk = true;
			res= -1;
			break;
		case 458:			//delete break;
			if(brks && nbrk > cbrk) {
				for(i = cbrk; i < nbrk; i++) {
					brks[i].fx = brks[i+1].fx;
					brks[i].fy = brks[i+1].fy;
					}
				nbrk--;
				if(brks[cbrk].fx == brks[cbrk].fy && cbrk == nbrk) {
					Dlg->SetText(455,0L);		Dlg->SetText(457,0L);
					}
				else {
					WriteNatFloatToBuff(TmpTxt, brks[cbrk].fx);
					Dlg->SetText(455, (unsigned char*)(TmpTxt+1));
					WriteNatFloatToBuff(TmpTxt, brks[cbrk].fy);
					Dlg->SetText(457, (unsigned char*)(TmpTxt + 1));
					}
				}
			else {
				Dlg->SetText(455, 0L);		Dlg->SetText(457, 0L);
			}
			upd_brk = true;
			res = -1;			break;
		case 551:
			n_gradient = 0;
			res = -1;		break;
		case 552:					//gradient color button 0
			Dlg->SetCheck(551, 0L, true);	n_gradient = 0;		res = -1;
			break;
		case 553:	case 554:	case 555:	case 556:
			n_gradient = (res - 552);
			res = -1;		break;
		case 557:	case 559:			//gradient color button 1 and 2
			Dlg->SetCheck(555, 0L, true);	n_gradient = 3;		res = -1;
			break;
			}
		}while (res < 0);
		Undo.SetDisp(cdisp);
		switch (res) {
		case 1:									//OK pressed
			bModified = true;
			if (Dlg->GetValue(455, &tmp) && Dlg->GetValue(457, &tmp2)){
				if (!brks && (brks = (lfPOINT*)calloc(2, sizeof(lfPOINT)))) {
					brks[0].fx = tmp;	brks[0].fy = tmp2;	cbrk = nbrk = 1;
				}
				else if (brks) {
					brks[cbrk].fx = tmp;	brks[cbrk].fy = tmp2;	nbrk++;
				}
			}
			Dlg->GetValue(101, &new_a.min);		Dlg->GetValue(103, &new_a.max);
			Dlg->GetValue(110, &new_a.loc[0].fx);	Dlg->GetValue(112, &new_a.loc[1].fx);
			Dlg->GetValue(115, &new_a.loc[0].fy);	Dlg->GetValue(117, &new_a.loc[1].fy);
			Dlg->GetValue(151, &new_a.loc[0].fz);	Dlg->GetValue(153, &new_a.loc[1].fz);
			Dlg->GetValue(182, &new_a.Center.fx);	Dlg->GetValue(185, &new_a.Center.fy);
			Dlg->GetValue(188, &new_a.Radius);
			new_a.breaks = brks;					new_a.nBreaks = nbrk;
			if (new_a.breaks) SortAxisBreaks(&new_a);
			brks = 0L;	nbrk = 0;
			if (Dlg->GetCheck(51)) {
				if (!(new_a.flags & AXIS_AUTOSCALE)) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
				new_a.flags |= AXIS_AUTOSCALE;
				}
			else new_a.flags &= ~AXIS_AUTOSCALE;
			if (Dlg->GetCheck(201)) new_a.flags |= AXIS_AUTOTICK;
			else new_a.flags &= ~AXIS_AUTOTICK;
			if (Dlg->GetCheck(200)) new_a.flags &= ~0x03;
			if (Dlg->GetCheck(305)) new_a.flags |= AXIS_INVERT;
			else new_a.flags &= ~AXIS_INVERT;
			new_a.flags &= ~0x70;
			if (Dlg->GetCheck(105)) new_a.flags |= AXIS_LEFT;
			else if (Dlg->GetCheck(106)) new_a.flags |= AXIS_RIGHT;
			else if (Dlg->GetCheck(107)) new_a.flags |= AXIS_TOP;
			else if (Dlg->GetCheck(108)) new_a.flags |= AXIS_BOTTOM;
			if ((new_a.flags & AXIS_LOG) == AXIS_LOG && new_a.min < defs.min4log) {
				switch (type & 0x0f) {
				case 1:
					new_a.min = parent->GetSize(SIZE_BOUNDS_XMIN);	break;
				case 2:
					new_a.min = parent->GetSize(SIZE_BOUNDS_YMIN);	break;
				case 3:
					new_a.min = parent->GetSize(SIZE_BOUNDS_ZMIN);	break;
				}
			}
			if (cmpAxisDEF(&old_a, &new_a)) {
				if (axis->flags != new_a.flags) parent->Command(CMD_MRK_DIRTY, 0L, 0L);
				Undo.AxisDef(this, axis, undo_flags);
				if (new_a.min != axis->min || new_a.max != axis->max) bNewScale = true;
				memcpy(axis, &new_a, sizeof(AxisDEF));	undo_flags |= UNDO_CONTINUE;
				axis->Start = axis->min;
				}
			else if (new_a.breaks) free(new_a.breaks);
			if (axis->nBreaks && Dlg->GetValue(406, &tmp))
				undo_flags = CheckNewFloat(&brkgap, brkgap, tmp, this, undo_flags);
			if (axis->nBreaks && Dlg->GetValue(409, &tmp))
				undo_flags = CheckNewFloat(&brksymsize, brksymsize, tmp, this, undo_flags);
			if (Dlg->GetCheck(402)) i = 2;
			else if (Dlg->GetCheck(403)) i = 3;
			else if (Dlg->GetCheck(404)) i = 4;
			else i = 0;
			if (axis->nBreaks) undo_flags = CheckNewInt(&brksym, brksym, i, this, undo_flags);
			if ((new_a.flags & AXIS_AUTOTICK) && (((new_a.max - new_a.min) / axis->Step < 3.0) 
				|| ((new_a.max - new_a.min) / axis->Step > 10.0))){
				Undo.DropListGO(this, (GraphObj ***)&Ticks, &NumTicks, undo_flags);
				Undo.ValFloat(this, &axis->Step, UNDO_CONTINUE);
				axis->Step = 0.0;
				undo_flags |= UNDO_CONTINUE;
				}
			if (Dlg->GetCheck(203)) {			//set ticks manually
				if (!Dlg->GetValue(255, &tmp)) tmp = 0.0;
				i = (int)tmp;			axis->flags &= AXIS_MINORTICK;
				if (!Dlg->GetValue(251, &tmp)) tmp = axis->Start;
				if (!Dlg->GetValue(253, &tmp2)) tmp2 = axis->Step;
				if (!(axis->flags & 0x03)){
					if (axis->flags & AXIS_ANGULAR) axis->flags |= AXIS_POSTICKS;
					else if (parent->Id == GO_GRAPH){
						switch ((((Graph*)parent)->tickstyle) & 0x07) {
						case 1:			//ticks inside
							if ((axis->flags & 0x70) == AXIS_RIGHT) axis->flags |= AXIS_NEGTICKS;
							else if ((axis->flags & 0x70) == AXIS_LEFT) axis->flags |= AXIS_POSTICKS;
							else if ((axis->flags & 0x70) == AXIS_TOP) axis->flags |= AXIS_NEGTICKS;
							else if ((axis->flags & 0x70) == AXIS_BOTTOM) axis->flags |= AXIS_POSTICKS;
							else axis->flags |= AXIS_SYMTICKS;
							break;
						case 2:			//ticks centered
							axis->flags |= AXIS_SYMTICKS;
							break;
						default:
						case 0:			//outside
							if ((axis->flags & 0x70) == AXIS_RIGHT) axis->flags |= AXIS_POSTICKS;
							else if ((axis->flags & 0x70) == AXIS_LEFT) axis->flags |= AXIS_NEGTICKS;
							else if ((axis->flags & 0x70) == AXIS_TOP) axis->flags |= AXIS_POSTICKS;
							else if ((axis->flags & 0x70) == AXIS_BOTTOM) axis->flags |= AXIS_NEGTICKS;
							else axis->flags |= AXIS_SYMTICKS;
							break;
							}
						}
					else axis->flags |= AXIS_SYMTICKS;
					}
			axis->flags &= ~AXIS_AUTOTICK;
			ManuTicks(tmp, tmp2, i, axis->flags);
			}
		if(!(new_a.flags & 0x03) && (new_a.flags & AXIS_AUTOTICK)){
			if(new_a.flags & AXIS_ANGULAR) axis->flags |= AXIS_POSTICKS;
			else if (parent->Id == GO_GRAPH){
				switch ((((Graph*)parent)->tickstyle) & 0x07) {
				case 1:			//ticks inside
					if ((axis->flags & 0x70) == AXIS_RIGHT) axis->flags |= AXIS_NEGTICKS;
					else if ((axis->flags & 0x70) == AXIS_LEFT) axis->flags |= AXIS_POSTICKS;
					else if ((axis->flags & 0x70) == AXIS_TOP) axis->flags |= AXIS_NEGTICKS;
					else if ((axis->flags & 0x70) == AXIS_BOTTOM) axis->flags |= AXIS_POSTICKS;
					else axis->flags |= AXIS_SYMTICKS;
					break;
				case 2:			//ticks centered
					axis->flags |= AXIS_SYMTICKS;
					break;
				default:
				case 0:			//outside
					if ((axis->flags & 0x70) == AXIS_RIGHT) axis->flags |= AXIS_POSTICKS;
					else if ((axis->flags & 0x70) == AXIS_LEFT) axis->flags |= AXIS_NEGTICKS;
					else if ((axis->flags & 0x70) == AXIS_TOP) axis->flags |= AXIS_POSTICKS;
					else if ((axis->flags & 0x70) == AXIS_BOTTOM) axis->flags |= AXIS_NEGTICKS;
					else axis->flags |= AXIS_SYMTICKS;
					break;
					}
				}
			else axis->flags |= AXIS_SYMTICKS;
			}
		if(undo_flags & UNDO_CONTINUE) {
			if((axis->flags & AXIS_RADIAL)||(axis->flags & AXIS_ANGULAR))
				parent->Command(CMD_AXIS, this, 0L);
			if(parent && parent->Id == GO_CONTOUR) parent->Command(CMD_RECALC, 0L, 0L);
			}
		if(names && CurrAxes && somePlots) {			//apply axis to plot ?
			for(i = 0; CurrAxes[i] != this; i++);		//find index 
			OD_axisplot(OD_ACCEPT, 0L, 0L, (anyOutput*) &res, 0L, 0);
			if(somePlots[res] && somePlots[res]->Command(CMD_USEAXIS, &i, 0L))
				undo_flags |= UNDO_CONTINUE;
			}
	if(Dlg->GetValue(122, &tmp)) 
			undo_flags = CheckNewFloat(&sizAxLine, sizAxLine, tmp, this, undo_flags);
		if(Dlg->GetText(131, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0]) {
			if(old_Label && strcmp((char*)old_Label,TmpTxt) && axisLabel->Id == GO_LABEL){
				lb_def = ((Label*)axisLabel)->GetTextDef();
				undo_flags = CheckNewString((unsigned char**)(&lb_def->text), (unsigned char*)(old_Label), (unsigned char*)TmpTxt, this, undo_flags);
				}
			else if(!axisLabel && (axis->flags & 0x03)) {
				label_def.ColTxt = colAxis;				label_def.ColBg = 0x00ffffffL;
				label_def.fSize = DefSize(SIZE_TICK_LABELS)*1.2;
				label_def.RotBL = fabs(si)>0.80 ? 90.0 : 0.0;
				label_def.RotCHAR = 0.0f;				label_def.iSize = 0;
				label_def.Align = TXA_VCENTER | TXA_HCENTER;
				label_def.Mode = TXM_TRANSPARENT;		label_def.Style = TXS_NORMAL;
				label_def.Font = FONT_HELVETICA;		label_def.text = (unsigned char*)TmpTxt;
				Undo.SetGO(this, &axisLabel, new Label(this, data, 0, 0, &label_def,
					LB_Y_PARENT | LB_X_PARENT, 0L), undo_flags);
				undo_flags |= UNDO_CONTINUE;
				}
			}
		else if(axisLabel) {
			Undo.DeleteGO(&axisLabel, undo_flags, 0L);	undo_flags |= UNDO_CONTINUE;
			}
		if((type & 0x0f) == 4) {
			if(Dlg->GetColor(552, &new_color) && gCol_0 != new_color){
				Undo.ValDword(this, &gCol_0, undo_flags);
				gCol_0 = new_color;
				undo_flags |= UNDO_CONTINUE;	bUpdPG = true;
				}
			if(Dlg->GetColor(557, &new_color) && gCol_1 != new_color){
				Undo.ValDword(this, &gCol_1, undo_flags);
				gCol_1 = new_color;
				undo_flags |= UNDO_CONTINUE;	bUpdPG = true;
				}
			if(Dlg->GetColor(559, &new_color) && gCol_2 != new_color){
				Undo.ValDword(this, &gCol_2, undo_flags);
				gCol_2 = new_color;
				undo_flags |= UNDO_CONTINUE;	bUpdPG = true;
				}
			Dlg->GetValue(562, &transp);
			new_color = ((((int)(transp*2.55))<<24)&0xff000000);
			if(gTrans != new_color){
				Undo.ValDword(this, &gTrans, undo_flags);
				gTrans = new_color;
				undo_flags |= UNDO_CONTINUE;	bUpdPG = true;
				}
			if(Dlg->GetCheck(560)) n_gradient |= 0x10;
			else n_gradient &= ~0x10;
			if(n_gradient != grad_type) {
				Undo.ValInt(this, &grad_type, undo_flags);
				grad_type = n_gradient;
				undo_flags |= UNDO_CONTINUE;	bUpdPG = true;
				}
			if(bUpdPG){
				Command(CMD_UPDPG, 0L, 0L);		bRet= true;
				}
			}
		if(Dlg->GetCheck(207) && Dlg->GetText(208, (unsigned char*)TmpTxt, TMP_TXT_SIZE) && TmpTxt[0]) {
			if(!tmpTLBformat) tmpTLBformat = (char*)malloc(50);
			rlp_strcpy(tmpTLBformat, 49, TmpTxt);
			Undo.VoidPtr(this, (void**)&TLBformat, TLBformat, 0L, undo_flags);
 			undo_flags |= UNDO_CONTINUE;		TLBformat = tmpTLBformat;
			if(Ticks && NumTicks) for (i = 0; i < NumTicks; i++) {
				if (Ticks[i]) Ticks[i]->Command(CMD_USE_FORMAT, TLBformat, 0L);
				}
			}
		else if (axis->flags & AXIS_AUTOTICK && !Dlg->GetCheck(51) && bNewScale) {
			TLBformat = GetNumFormat(floor(log10((axis->max -axis->min)/5.0)));
			if (Ticks && NumTicks) for (i = 0; i < NumTicks; i++) {
				if (Ticks[i]) Ticks[i]->Command(CMD_USE_FORMAT, TLBformat, 0L);
				}
			}
		if(undo_flags & UNDO_CONTINUE) bRet= true;
		break;
	case 2:											//cancel button pressed
		if (undo_pos < Undo.level()) {
			while (undo_pos < Undo.level()) {
				Undo.Restore(false, cdisp);
				}
			parent->Command(CMD_REDRAW, 0L, cdisp);
			}
		break;
		}
	if(brks && nbrk) free(brks);
	if(old_Label) free(old_Label);
	CloseDlgWnd(hDlg);
	delete Dlg;
	if(names) {
		for(j = 0; names[j]; j++) if(names[j]) free(names[j]);
		free(names);
		}
	if(somePlots) free(somePlots);
	free(AxisPropDlg);
	free(tab1);		free(tab2);			free(tab3);
	free(tab4);		free(tab5);			free(tab6);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Graph dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *AddPlotTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,175,10,45,12\n"
".,.,,,PUSHBUTTON,-2,175,25,45,12\n"
".,.,,,LTEXT,1,7,5,60,8\n"
".,.,,ISPARENT | CHECKED,GROUPBOX,4,5,88,160,35\n"
".,,520,ISPARENT | CHECKED,GROUPBOX,3,5,20,160,62\n"
"520,+,,EXRADIO | CHECKED,ODB,2,10,27,,\n"
".,523,,EXRADIO,ODB,2,35,27,,\n"
"523,+,,EXRADIO,ODB,2,60,27,,\n"
".,.,,EXRADIO,ODB,2,85,27,,\n"
".,.,,EXRADIO,ODB,2,110,27,,\n"
".,528,,EXRADIO,ODB,2,135,27,,\n"
"528,+,,EXRADIO,ODB,2,10,52,,\n"
".,.,,EXRADIO,ODB,2,35,52,,\n"
".,.,,EXRADIO,ODB,2,60,52,,\n"
".,.,,EXRADIO,ODB,2,85,52,,\n"
".,540,,EXRADIO,ODB,2,110,52,,\n"
"540,+,,EXRADIO,ODB,2,10,95,,\n"
".,546,,EXRADIO,ODB,2,35,95,,\n"
"546,+,,EXRADIO,ODB,2,60,95,,\n"
".,,,LASTOBJ | EXRADIO,ODB,2,85,95,,";

bool
Graph::AddPlot(int)
{
	void *dyndata[] = { (void *)SDLG_GRAFADD_TMPL, (void*)OD_PlotTempl, (void*)SDLG_GRAFADD_XY,
		(void*)SDLG_GRAFADD_XYY };
	DlgInfo *GraphDlg = CompileDialog(AddPlotTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int i, res, cSel = 520;
	bool bRet = false;
	Plot *p;

	switch(type) {
	case GT_STANDARD:
		break;
	case GT_POLARPLOT:
		for(i = 0; i < NumPlots; i++) {
			if(Plots[i] && Plots[i]->Id == GO_POLARPLOT) {
				if (((PolarPlot*)Plots[i])->AddPlot()) 
					return Command(CMD_REDRAW, 0L, 0L);
				}
			}
		return false;
	case GT_TERNARY:
		for(i = 0; i < NumPlots; i++) {
			if(Plots[i] && Plots[i]->Id == GO_TERNARY) {
				if (((TernaryPlot*)Plots[i])->AddPlot()) 
					return Command(CMD_REDRAW, 0L, 0L);
				}
			}
		return false;
	case GT_TERNARYXYZ:
		for(i = 0; i < NumPlots; i++) {
			if(Plots[i] && Plots[i]->Id == GO_TERNARYXYZ) {
				if (((TernaryXYZ*)Plots[i])->AddPlot()) 
					return Command(CMD_REDRAW, 0L, 0L);
				}
			}
		return false;
	default:
		InfoBox((char*)SCMS_NOADDPLOT);
		return false;
		}
	Dlg = new DlgRoot(GraphDlg, data);
	hDlg = CreateDlgWnd((char*)SDLG_GRAFADD_HD1, 50, 50, 470, 306, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 520:	case 521:	case 522:	case 523:	case 524:
		case 525:	case 526:	case 528:	case 529:	case 530:
		case 531:	case 532:	case 540:	case 541:
		case 546:	case 547:
			if(res == cSel) res = 1;
			else {
				cSel = res;				res = -1;
				}
			break;
			}
		}while (res < 0);
	if(res == 1){						//OK pressed
		if(Dlg->GetCheck(524)) p = new BubblePlot(this, data);
		else if(Dlg->GetCheck(525)) p = new BoxPlot(this, data);
		else if(Dlg->GetCheck(521)) p = new PlotScatt(this, data, 0x03);
		else if(Dlg->GetCheck(522)) p = new PlotScatt(this, data, 0x08);
		else if(Dlg->GetCheck(523)) p = new PlotScatt(this, data, 0x04);
		else if(Dlg->GetCheck(526)) p = new Regression(this, data);
		else if(Dlg->GetCheck(528)) p = new DensDisp(this, data);
		else if(Dlg->GetCheck(529)) p = new Function(this, data, (char*)"Function");
		else if(Dlg->GetCheck(530)) p = new FitFunc(this, data);
		else if(Dlg->GetCheck(531)) p = new MultiLines(this, data);
		else if(Dlg->GetCheck(532)) p = new xyStat(this, data);
		else if(Dlg->GetCheck(540)) p = new StackBar(this, data);
		else if(Dlg->GetCheck(541)) p = new StackPG(this, data);
		else if (Dlg->GetCheck(546)) p = new myRegression(this, data);
		else if(Dlg->GetCheck(547)) p = new myBarPlot(this, data);
		else p = new PlotScatt(this, data, 0x01);
		CloseDlgWnd(hDlg);
		for (i = 0; i < NumPlots; i++) {
			if (Plots[i]->Id > GO_PLOT && Plots[i]->Id < GO_GRAPH) {
				if (((Plot*)Plots[i])->x_tv) p->x_tv = (((Plot*)Plots[i])->x_tv)->Copy();
				if (((Plot*)Plots[i])->y_tv) p->y_tv = (((Plot*)Plots[i])->y_tv)->Copy();
				break;
				}
			}
		if (p && p->PropertyDlg()) {
			if(!Command(CMD_DROP_PLOT, p, (anyOutput *)NULL)) delete p;
			else bRet = true;
			}
		else if(p) delete p;
		}
	else CloseDlgWnd(hDlg);
	delete Dlg;		free(GraphDlg);
	return bRet;
}

static char *GraphDlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,190,10,45,12\n"
".,.,,,PUSHBUTTON,-2,190,25,45,12\n"
".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,+,100,ISPARENT | CHECKED,SHEET,1,5,10,177,122\n"
".,.,200,ISPARENT,SHEET,2,5,10,177,122\n"
".,.,300,ISPARENT,SHEET,3,5,10,177,122\n"
".,,600,ISPARENT,SHEET,30,5,10,177,122\n"
"100,+,,,LTEXT,4,10,25,60,8\n"
".,.,500,TOUCHEXIT | ISPARENT,SHEET,5,10,37,167,90\n"
".,.,520,TOUCHEXIT | ISPARENT,SHEET,6,10,37,167,90\n"
".,.,540,TOUCHEXIT | ISPARENT,SHEET,7,10,37,167,90\n"
".,,560,TOUCHEXIT | ISPARENT,SHEET,8,10,37,167,90\n"
"200,+,,,LTEXT,9,17,35,60,8\n"
".,.,,,RTEXT,10,13,47,58,8\n"
".,.,,,EDVAL1,11,72,47,30,10\n"
".,.,,,RTEXT,-5,105,47,10,8\n"
".,.,,,EDVAL1,12,117,47,30,10\n"
".,.,,,LTEXT,-3,150,47,20,8\n"
".,.,,,RTEXT,13,13,59,58,8\n"
".,.,,,EDVAL1,14,72,59,30,10\n"
".,.,,,RTEXT,-5,105,59,10,8\n"
".,.,,,EDVAL1,15,117,59,30,10\n"
".,.,,,LTEXT,-3,150,59,20,8\n"
".,.,,,LTEXT,16,17,84,60,8\n"
".,.,,,RTEXT,10,13,96,58,8\n"
".,.,,,EDVAL1,17,72,96,30,10\n"
".,.,,,RTEXT,-5,105,96,10,8\n"
".,.,,,EDVAL1,18,117,96,30,10\n"
".,.,,,LTEXT,-3,150,96,20,8\n"
".,.,,,RTEXT,13,13,108,58,8\n"
".,.,,,EDVAL1,19,72,108,30,10\n"
".,.,,,RTEXT,-5,105,108,10,8\n"
".,.,,,EDVAL1,20,117,108,30,10\n"
".,,,,LTEXT,-3,150,108,20,8\n"
"300,+,,,LTEXT,21,20,30,60,8\n"
".,400,310,CHECKED | ISPARENT,GROUP,,,,,\n"
"310,+,,EXRADIO,ODB,22,25,42,,\n"
".,.,,EXRADIO,ODB,22,52,42,,\n"
".,.,,EXRADIO,ODB,22,79,42,,\n"
".,.,,EXRADIO,ODB,22,106,42,,\n"
".,.,,EXRADIO,ODB,22,133,42,,\n"
".,.,,HICOL | CHECKED | TOUCHEXIT,RADIO1,23,12,85,40,8\n"
".,.,,HICOL | TOUCHEXIT,RADIO1,24,12,95,40,8\n"
".,.,,HICOL | TOUCHEXIT,RADIO1,25,12,105,40,8\n"
".,.,,HICOL | TOUCHEXIT,CHECKBOX,26,80,85,40,8\n"
".,,,HICOL | TOUCHEXIT,CHECKBOX,27,80,95,40,8\n"
"400,,410,HIDDEN | CHECKED | ISPARENT,GROUP,,,,,\n"
"410,+,,EXRADIO,ODB,28,25,42,,\n"
".,.,,EXRADIO,ODB,28,52,42,,\n"
".,,,EXRADIO, ODB,28,79,42,,\n"
"500,+,,EXRADIO,ODB,29,25,60,,\n"
".,.,,EXRADIO,ODB,29,52,60,,\n"
".,.,,EXRADIO,ODB,29,79,60,,\n"
".,.,,EXRADIO,ODB,29,25,85,,\n"
".,.,,EXRADIO,ODB,29,52,85,,\n"
".,.,,EXRADIO,ODB,29,106,85,,\n"
".,.,,EXRADIO,ODB,29,133,85,,\n"
".,,,EXRADIO,ODB,29,79,85,,\n"
"520,+,,EXRADIO,ODB,29,25,50,,\n"
".,.,,EXRADIO,ODB,29,52,50,,\n"
".,.,,EXRADIO,ODB,29,79,50,,\n"
".,.,,EXRADIO,ODB,29,106,50,,\n"
".,.,,EXRADIO,ODB,29,133,50,,\n"
".,.,,EXRADIO,ODB,29,25,75,,\n"
".,.,,EXRADIO,ODB,29,52,75,,\n"
".,.,,EXRADIO,ODB,29,79,75,,\n"
".,.,,EXRADIO,ODB,29,106,75,,\n"
".,.,,EXRADIO,ODB,29,133,75,,\n"
".,.,,EXRADIO,ODB,29,25,100,,\n"
".,.,,EXRADIO,ODB,29,52,100,,\n"
".,,,EXRADIO,ODB,29,79,100,,\n"				//Debug: No 2D Ternary plot defined
".,,,EXRADIO,ODB,29,106,100,,\n"
"540,+,,EXRADIO,ODB,29,25,60,,\n"
".,.,,EXRADIO,ODB,29,52,60,,\n"
".,.,,EXRADIO,ODB,29,79,60,,\n"
".,.,,EXRADIO,ODB,29,106,60,,\n"
".,.,,EXRADIO,ODB,29,133,60,,\n"
".,.,,EXRADIO,ODB,29,79,90,,\n"
".,.,,EXRADIO,ODB,29,52,90,,\n"
".,,,EXRADIO,ODB,29,25,90,,\n"
"560,+,,EXRADIO,ODB,29,25,60,,\n"
".,.,,EXRADIO,ODB,29,52,60,,\n"
".,.,,EXRADIO,ODB,29,79,60,,\n"
".,.,,EXRADIO,ODB,29,106,60,,\n"
".,.,,EXRADIO,ODB,29,133,60,,\n"
".,.,,EXRADIO,ODB,29,25,85,,\n"
".,.,,EXRADIO,ODB,29,52,85,,\n"
".,.,,EXRADIO,ODB,29,79,85,,\n"
".,.,,EXRADIO,ODB,29,106,85,,\n"
".,,,EXRADIO,ODB,29,133,85,,\n"
"600,+,,HICOL,CHECKBOX,31,17,35,40,8\n"
".,.,,,LTEXT,32,30,55,30,8\n"
".,.,,TOUCHEXIT,EDVAL1,33,72,55,30,10\n"
".,,,LASTOBJ,LTEXT,-3,105,55,20,8";

static int selSheet = 102, selPlt = 520, selAxis = 310;
bool
Graph::PropertyDlg()
{
	int i, res;
	double grid_spacing;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_DINPUT);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_PLACEMENT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_AXES);
	TabSHEET *tab4 = MakeTab(i, 10, &i, (char*)SDLG_TAB_GRID);
	TabSHEET *tab_A = MakeTab(0, 10, &i, (char*)SDLG_TAB_ONLYY);
	TabSHEET *tab_B = MakeTab(i, 10, &i, (char*)SDLG_TAB_XYVAL);
	TabSHEET *tab_C = MakeTab(i, 10, &i, (char*)SDLG_TAB_MANYY);
	TabSHEET *tab_D = MakeTab(i, 10, &i, (char*)SDLG_TAB_XYZ);
	switch (defs.units()) {
	case 1:		grid_spacing = 0.5;		break;
	case 2:		grid_spacing = 0.25;	break;
	default:	grid_spacing = 5.0;		break;
		}
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_GRAPH_SEL, (void*)tab_A, (void*)tab_B,
		(void*)tab_C, (void*)tab_D, (void*)SDLG_GRAPH_BOUNDS, (void*)SDLG_GRAPH_ULEFT, (void*)&GRect.Xmin,
		(void*)&GRect.Ymin, (void*)SDLG_GRAPH_LRIGHT, (void*)&GRect.Xmax, (void*)&GRect.Ymax, (void*)SDLG_GRAPH_PRECT,
		(void*)&DRect.Xmin, (void*)&DRect.Ymin, (void*)&DRect.Xmax, (void*)&DRect.Ymax, (void*)SDLG_GRAPH_TEMPL,
		(void*)(OD_AxisTempl), (void*)SDLG_GRAPH_TOUT, (void*)SDLG_GRAPH_TINS, (void*)SDLG_GRAPH_TSYM,
		(void*)SDLG_GRAPH_HGRID, (void*)SDLG_GRAPH_VGRID, (void*)(OD_AxisTempl3D), (void*)(OD_PlotTempl), (void*)tab4,
		(void*)SDLG_GRAPH_GRID, (void*)SDLG_GRAPH_GRIDSPACING, (void*)&grid_spacing };
	DlgInfo *GraphDlg = CompileDialog(GraphDlgTmpl, dyndata);
	DlgRoot *Dlg;
	GraphObj *p;
	void *hDlg;
	bool bRet, bContinue;
	fRECT rc1, rc2;

	ODtickstyle = tickstyle;
	AxisTempl = 0;
	if(!parent) return false;
	Dlg = new DlgRoot(GraphDlg, data);
	Dlg->SetCheck(410 + AxisTempl3D, 0L, true);
	if(parent->Id != GO_PAGE) {
		Dlg->Activate(202, 0);	Dlg->Activate(204, 0);
		}
	//restore previous settitings
	switch(selSheet) {
		case 101:
			if(selPlt >= 500 && selPlt <=507) Dlg->SetCheck(selPlt, 0L, true);
			else Dlg->SetCheck(500, 0L, true);
			Dlg->SetCheck(520, 0L, true);		Dlg->SetCheck(540, 0L, true);
			Dlg->SetCheck(560, 0L, true);		break;
		default:
			if(selPlt >= 520 && selPlt <=533) Dlg->SetCheck(selPlt, 0L, true);
			else Dlg->SetCheck(520, 0L, true);
			selSheet = 102;
			Dlg->SetCheck(500, 0L, true);		Dlg->SetCheck(540, 0L, true);
			Dlg->SetCheck(560, 0L, true);		break;
		case 103:
			if(selPlt >= 540 && selPlt <=545) Dlg->SetCheck(selPlt, 0L, true);
			else Dlg->SetCheck(540, 0L, true);
			Dlg->SetCheck(520, 0L, true);		Dlg->SetCheck(500, 0L, true);
			Dlg->SetCheck(560, 0L, true);		break;
		case 104:
			if(selPlt >= 560 && selPlt <=569) Dlg->SetCheck(selPlt, 0L, true);
			else Dlg->SetCheck(560, 0L, true);
			Dlg->ShowItem(301, false);	Dlg->ShowItem(400, true);
			Dlg->SetCheck(520, 0L, true);		Dlg->SetCheck(540, 0L, true);
			Dlg->SetCheck(500, 0L, true);		break;
		}
	Dlg->SetCheck(selSheet, 0L, true);
	if (selAxis >= 310 && selAxis <= 314) {
		AxisTempl = selAxis - 310;
		Dlg->SetCheck(selAxis, 0L, true);
		}
	else Dlg->SetCheck(310, 0L, true);
	if (selAxis >= 410 && selAxis <= 412) {
		AxisTempl = selAxis - 410;
		Dlg->SetCheck(selAxis, 0L, true);
		}
	else Dlg->SetCheck(410, 0L, true);
	//display the dialog
	hDlg = CreateDlgWnd((char*)SDLG_GRAPH_HD1, 50, 50, 500, 320, Dlg, 0x4L);
	bContinue = false;
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		bRet = false;
		switch(res) {
		case 0:
			if(bContinue) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 101:	//only y data
			for(i = 500; i <= 506; i++) if(Dlg->GetCheck(i))selPlt = i;
			Dlg->ShowItem(301, true);	Dlg->ShowItem(400, false);
			selSheet = res;				res = -1;
			break;
		case 102:	//xy data
			for(i = 520; i <= 533; i++) if(Dlg->GetCheck(i))selPlt = i;
			Dlg->ShowItem(301, true);	Dlg->ShowItem(400, false);
			selSheet = res;				res = -1;
			break;
		case 103:	//x many y data
			for(i = 540; i <= 544; i++) if(Dlg->GetCheck(i))selPlt = i;
			Dlg->ShowItem(301, true);	Dlg->ShowItem(400, false);
			selSheet = res;				res = -1;
			break;
		case 104:	//xyz data
			for(i = 560; i <= 567; i++) if(Dlg->GetCheck(i))selPlt = i;
			Dlg->ShowItem(301, false);	Dlg->ShowItem(400, true);
			selSheet = res;				res = -1;
			break;
		case 310:	case 311:	case 312:	case 313:	case 314:
			AxisTempl = res-310;
			if(res == selAxis) Dlg->SetCheck(4, 0L, true);
			selAxis = res;
			res = -1;
			break;
		case 315:	case 316:	case 317:	//tick style
		case 318:	case 319:				//horizontal or vertical grid
			if (res == 315 || res == 316 || res == 317) {
				tickstyle = res - 315;
				}
			tickstyle &= ~0x300;
			if(Dlg->GetCheck(318)) tickstyle |= 0x200;
			if(Dlg->GetCheck(319)) tickstyle |= 0x100;
			ODtickstyle = tickstyle;
			Dlg->DoPlot(0L);
			res = -1;
			break;
		case 410:	case 411:	case 412:	//axis templates
			AxisTempl3D = res-410;
			res = -1;
			break;
		case 500:	case 501:	case 502:	case 503:	case 504:	case 505:	case 506:	case 507:
		case 520:	case 521:	case 522:	case 523:	case 524:	case 525:	case 526:	case 527:
		case 528:	case 529:	case 530:	case 531:	case 532:	case 533:
		case 540:	case 541:	case 542:	case 543:	case 544:	case 545:	case 546:	case 547:
		case 560:	case 561:	case 562:	case 563:	case 564:	case 565:	case 566:	case 567:
		case 568:	case 569:
		case 1:
			if (res > 500){
				if (res != selPlt) {
					selPlt = res;
					res = -1;
					break;
				}
				//double click means select: continue as if OK
				res = 1;
				}
			if(!Dlg->GetCheck(4)) {
				Dlg->SetCheck(4, 0L, true);
				res = -1;
				break;
				}
			memcpy(&rc1, &GRect, sizeof(fRECT));
			memcpy(&rc2, &DRect, sizeof(fRECT));
			Dlg->GetValue(202, &rc1.Xmin);			Dlg->GetValue(204, &rc1.Ymin);
			Dlg->GetValue(207, &rc1.Xmax);			Dlg->GetValue(209, &rc1.Ymax);
			Dlg->GetValue(213, &rc2.Xmin);			Dlg->GetValue(215, &rc2.Ymin);
			Dlg->GetValue(218, &rc2.Xmax);			Dlg->GetValue(220, &rc2.Ymax);
			if(rc1.Xmin < 0.0 || rc1.Xmax < 0.0 || rc1.Ymin < 0.0 || 
				rc1.Ymax < 0.0 || rc2.Xmin < 0.0 || rc2.Xmax < 0.0 || 
				rc2.Ymin < 0.0 || rc2.Ymax < 0.0) {
				ErrorBox((char*)SCMS_PLOT_PLACE);
				res = -1;
				bContinue = true;
				Dlg->SetCheck(5, 0L, true);
				break;
				}
			if(rc2.Xmin > (rc1.Xmax-rc1.Xmin) || rc2.Ymax > (rc1.Ymax-rc1.Ymin) 
				|| (rc2.Xmax-rc2.Xmin) > (rc1.Xmax-rc1.Xmin)
				|| (rc2.Ymax-rc2.Ymin) > (rc1.Ymax-rc1.Ymin)){
				ErrorBox((char*)SCMS_PLOT_RECT);
				res = -1;
				bContinue = true;
				Dlg->SetCheck(5, 0L, true);
				break;
				}
			memcpy(&GRect, &rc1, sizeof(fRECT));
			memcpy(&DRect, &rc2, sizeof(fRECT));
			p = 0L;
			if(Dlg->GetCheck(101)) {
				if(Dlg->GetCheck(500))p = new PieChart(this, data);
				else if(Dlg->GetCheck(501))p = new RingChart(this, data);
				else if(Dlg->GetCheck(502))p = new StarChart(this, data);
				else if(Dlg->GetCheck(503))p = new BarChart(this, data);
				else if(Dlg->GetCheck(504))p = new GroupBars(this, data, 0);
				else if(Dlg->GetCheck(505))p = new FreqDist(this, data);
				else if(Dlg->GetCheck(506))p = new NormQuant(this, data, 0L);
				else if(Dlg->GetCheck(507))p = new GroupBars(this, data, 1);
				}
			else if(Dlg->GetCheck(102)){
				if(Dlg->GetCheck(524)) p = new BubblePlot(this, data);
				else if(Dlg->GetCheck(525)) p = new BoxPlot(this, data);
				else if(Dlg->GetCheck(521)) p = new PlotScatt(this, data, 0x03);
				else if(Dlg->GetCheck(522)) p = new PlotScatt(this, data, 0x08);
				else if(Dlg->GetCheck(523)) p = new PlotScatt(this, data, 0x04);
				else if(Dlg->GetCheck(526)) p = new Regression(this, data);
				else if(Dlg->GetCheck(527)) p = new PolarPlot(this, data);
				else if(Dlg->GetCheck(528)) p = new DensDisp(this, data);
				else if(Dlg->GetCheck(529)) p = new Function(this, data, (char*)"Function");
				else if(Dlg->GetCheck(530)) p = new FitFunc(this, data);
				else if(Dlg->GetCheck(531)) p = new MultiLines(this, data);
				else if(Dlg->GetCheck(532)) p = new xyStat(this, data);
				else if(Dlg->GetCheck(533)) p = new TernaryPlot(this, data);
				else p = new PlotScatt(this, data, 0x01);
				}
			else if(Dlg->GetCheck(103)) {
				if(Dlg->GetCheck(540)) p = new StackBar(this, data);
				else if(Dlg->GetCheck(542)) p = new Waterfall(this, data);
				else if(Dlg->GetCheck(543)) p = new Chart25D(this, data, 0L);
				else if(Dlg->GetCheck(544)) p = new Ribbon25D(this, data, 0L);
				else if (Dlg->GetCheck(546)) p = new myRegression(this, data);
				else if (Dlg->GetCheck(545)) {
					AxisTempl = 20;  p = new Phenology(this, data);
					}
				else if (Dlg->GetCheck(547)) p = new myBarPlot(this, data);
				else p = new StackPG(this, data);
				}
			else if(Dlg->GetCheck(104)) {
				if(Dlg->GetCheck(560)) p = new Plot3D(this, data, 0x1001);
				else if(Dlg->GetCheck(561)) p = new Plot3D(this, data, 0x1002);
				else if(Dlg->GetCheck(562)) p = new Plot3D(this, data, 0x1004);
				else if(Dlg->GetCheck(563)) p = new BubblePlot3D(this, data);
				else if(Dlg->GetCheck(564)) p = new Plot3D(this, data, 0x2000);
				else if(Dlg->GetCheck(565)) p = new Func3D(this, data);
				else if(Dlg->GetCheck(566)) p = new FitFunc3D(this, data);
				else if(Dlg->GetCheck(567)) p = new Plot3D(this, data, 0x4000);
				else if (Dlg->GetCheck(568)) p = new ContourPlot(this, data);
				else if(Dlg->GetCheck(569)) p = new TernaryXYZ(this, data);
				}
			if(p && p->PropertyDlg()) {
				if(!Command(CMD_DROP_PLOT, p, 0L)) DeleteGO(p);
				else bRet = true;
				}
			else if(p) {
				AxisTempl = 0;
				Dlg->SetCheck(410 + AxisTempl3D, 0L, true);
				DeleteGO(p);		Dlg->DoPlot(0L);			ShowDlgWnd(hDlg);
				}
			if(!bRet) {
				res = -1;
				//we might have lost the focus
				bContinue = true;
				}
			if (bRet && Dlg->GetCheck(600) && !theGrid){
				theGrid = new Grid(this);				theGrid->hidden = false;
				Dlg->GetValue(602, &grid_spacing);		theGrid->SetSize(SIZE_MINE, grid_spacing);
				}
			break;
			case 602:			//grid spacing selected
				Dlg->SetCheck(600, 0L, true);			res = -1;
				break;
				}
		}while(res <0);
	Command(CMD_SET_DATAOBJ, (void*)data, 0L);
	CloseDlgWnd(hDlg);		delete Dlg;			free(GraphDlg);
	free(tab1);				free(tab2);			free(tab3);
	free(tab4);				free(tab_A);		free(tab_B);
	free(tab_C);			free(tab_D);
	return bRet;
}
static char *GraphConfDlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,190,10,45,12\n"
".,.,,,PUSHBUTTON,-2,190,25,45,12\n"
".,.,,,PUSHBUTTON,1,190,73,45,12\n"
".,.,,,PUSHBUTTON,2,190,88,45,12\n"
".,.,,,PUSHBUTTON,3,190,103,45,12\n"
".,.,,,PUSHBUTTON,4,190,118,45,12\n"
".,13,,HIDDEN,CHECKBOX,5,5,135,160,10\n"
"13,50,14,ISPARENT | CHECKED,GROUP,,,,,\n"
"14,+,100,TOUCHEXIT | ISPARENT,SHEET,6,5,10,175,120\n"
".,.,200,TOUCHEXIT | ISPARENT,SHEET,7,5,10,175,120\n"
".,,700,TOUCHEXIT | ISPARENT,SHEET,31,5,10,175,120\n"
"50,,600,ISPARENT | CHECKED | HIDDEN,GROUP,,,,,\n"
"100,+,,,LTEXT,8,40,30,60,8\n"
".,.,,,RTEXT,9,35,42,58,8\n"
".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,10,94,42,25,10\n"
".,.,,,RTEXT,11,35,54,58,8\n"
".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,12,94,54,25,10\n"
".,.,,,LTEXT,13,40,68,60,8\n"
".,.,,,RTEXT,9,35,80,58,8\n"
".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,14,94,80,25,10\n"
".,.,,,LTEXT,15,40,94,60,8\n"
".,.,,,RTEXT,16,35,106,58,8\n"
".,,,TOUCHEXIT | OWNDIALOG,COLBUTT,17,94,106,25,10\n"
"200,+,,,LTEXT,18,20,35,60,8\n"
".,.,,,RTEXT, 19,5,47,68,8\n"
".,.,,,EDVAL1,20,74,47,30,10\n"
".,.,,,RTEXT,-5,110,47,10,8\n"
".,.,,,EDVAL1,21,122,47,30,10\n"
".,.,,,LTEXT,-3,155,47,20,8\n"
".,.,,,RTEXT,22,5,59,68,8\n"
".,.,,,EDVAL1,23,74,59,30,10\n"
".,.,,,RTEXT,-5,110,59,10,8\n"
".,.,,,EDVAL1,24,122,59,30,10\n"
".,.,,,LTEXT,-3,155,59,20,8\n"
".,.,,,LTEXT,25,20,84,60,8\n"
".,.,,,RTEXT,19,5,96,68,8\n"
".,.,,,EDVAL1,26,74,96,30,10\n"
".,.,,,RTEXT,-5,110,96,10,8\n"
".,.,,,EDVAL1,27,122,96,30,10\n"
".,.,,,LTEXT,-3,155,96,20,8\n"
".,.,,,RTEXT,22,5,108,68,8\n"
".,.,,,EDVAL1,28,74,108,30,10\n"
".,.,,,RTEXT,-5,110,108,10,8\n"
".,.,,,EDVAL1,29,122,108,30,10\n"
".,,,,LTEXT,-3,155,108,20,8\n"
"600,+,,TOUCHEXIT,ODBUTTON,30,190,40,15,15\n"
".,.,,TOUCHEXIT,ODBUTTON,30,210,40,15,15\n"
".,.,,TOUCHEXIT,ODBUTTON,30,210,55,15,15\n"
".,,,TOUCHEXIT,ODBUTTON,30,190,55,15,15\n"
"700,+,,HICOL,CHECKBOX,32,17,35,40,8\n"
".,.,,,RTEXT,33,27,55,40,8\n"
".,.,,TOUCHEXIT,EDVAL1,34,72,55,30,10\n"
".,.,,,LTEXT,-3,105,55,20,8\n"
".,.,,,RTEXT,35,27,70,40,8\n"
".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,36,72,70,30,10\n"
".,,,HICOL | TOUCHEXIT | LASTOBJ,CHECKBOX,37,17,90,40,8";

bool
Graph::Configure()
{
	int i;
	double grid_spacing;
	DWORD grid_color;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_COLORS);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_PLACEMENT);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_GRID);
	if (theGrid) {
		grid_spacing = theGrid->GetSize(SIZE_MINE);
		}	
	else switch (defs.units()) {
	case 1:		grid_spacing = 0.5;		break;
	case 2:		grid_spacing = 0.25;	break;
	default:	grid_spacing = 5.0;		break;
		}
	if (theGrid) grid_color = theGrid->GetColor(0);
	else grid_color = 0x00cbcbcb;
	void *dyndata[] = { (void*)SDLG_CGRA_ADDPLT, (void*)SDLG_CGRA_ADDAX, (void*)SDLG_CGRA_ADDLEG,
		(void*)SDLG_CGRA_LAYERS, (void*)SDLG_CGRA_CLIP, (void*)tab1, (void*)tab2, (void*)SDLG_CGRA_BRC,
		(void*)SDLG_CGRA_FILLC, (void *)&ColGR, (void*)SDLG_CGRA_OUTLC, (void *)&ColGRL, (void*)SDLG_CGRA_PRC,
		(void *)&ColDR, (void*)SDLG_CGRA_AXTILA, (void*)SDLG_CGRA_AXCOL, (void *)&ColAX, (void*)SDLG_GRAPH_BOUNDS,
		(void*)SDLG_GRAPH_ULEFT, (void*)&GRect.Xmin, (void*)&GRect.Ymin, (void*)SDLG_GRAPH_LRIGHT, (void*)&GRect.Xmax,
		(void*)&GRect.Ymax, (void*)SDLG_GRAPH_PRECT, (void*)&DRect.Xmin, (void*)&DRect.Ymin, (void*)&DRect.Xmax,
		(void*)&DRect.Ymax, (void*)OD_DrawOrder, (void*)tab3, (void*)SDLG_GRAPH_GRID, (void*)SDLG_GRAPH_GRIDSPACING,
		(void*)&grid_spacing, (void*)SDLG_GRAPH_GRIDCOLOR, (void*)&grid_color, (void*)SDLG_GRAPH_SNAP};
	DlgInfo *GraphDlg = CompileDialog(GraphConfDlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, undo_level = Undo.level();
	static int last_tab = 14;
	DWORD undo_flags = 0, tmpcol;
	anyOutput *cdisp = Undo.Cdisp();
	bool bRet = false, bContinue = false;
	fRECT o_gr, n_gr, o_dr, n_dr;

	if(!(Dlg = new DlgRoot(GraphDlg, data)))return false;
	Dlg->GetValue(202, &o_gr.Xmin);			Dlg->GetValue(204, &o_gr.Ymin);
	Dlg->GetValue(207, &o_gr.Xmax);			Dlg->GetValue(209, &o_gr.Ymax);
	Dlg->GetValue(213, &o_dr.Xmin);			Dlg->GetValue(215, &o_dr.Ymin);
	Dlg->GetValue(218, &o_dr.Xmax);			Dlg->GetValue(220, &o_dr.Ymax);
	if (theGrid){
		if (theGrid->hidden) Dlg->SetCheck(700, 0L, false);
		else Dlg->SetCheck(700, 0L, true);
		if (theGrid->type & 0x01) Dlg->SetCheck(706, 0L, true);
		else Dlg->SetCheck(706, 0L, false);
		}
	if (type == GT_STANDARD)Dlg->ShowItem(7, true);
	if (bClip)Dlg->SetCheck(7, NULL, true);
	Dlg->SetCheck(last_tab, 0L, true);
	if(parent && (parent->Id == GO_PAGE || parent->Id == GO_GRAPH)) Dlg->ShowItem(50, true);
	i = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, name ? name : (char*)SDLG_CGRA_HD1);
	rlp_strcpy(TmpTxt + i, TMP_TXT_SIZE - i, (char*)SDLG_CGRA_HD2);
	if (theGrid && !theGrid->hidden)Dlg->SetCheck(700, 0L, true);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 500, 350, Dlg, 0x4L);
	do{
		LoopDlgWnd();			res = Dlg->GetResult();
		switch (res) {
		case 600:	case 601:	case 602:	case 603:
			Undo.SetDisp(cdisp);
			res = ExecDrawOrderButt(parent, this, res);
			}
		switch(res) {
		case 0:
			if(bContinue) res = -1;
			break;
		case -1:
			bContinue = false;			break;
		case 102:	case 104:	case 107:	case 110:
			res = -1;					break;
		case 14:	case 15:	case 16:			//the tab sheets
			last_tab = res;				res = -1;			break;
		case 1:
			Dlg->GetValue(202, &n_gr.Xmin);			Dlg->GetValue(204, &n_gr.Ymin);
			Dlg->GetValue(207, &n_gr.Xmax);			Dlg->GetValue(209, &n_gr.Ymax);
			Dlg->GetValue(213, &n_dr.Xmin);			Dlg->GetValue(215, &n_dr.Ymin);
			Dlg->GetValue(218, &n_dr.Xmax);			Dlg->GetValue(220, &n_dr.Ymax);
			if (Dlg->GetCheck(7))bClip = 1;
			else bClip = 0;
			if(n_gr.Xmin < 0.0 || n_gr.Xmax < 0.0 || n_gr.Ymin < 0.0 || 
				n_gr.Ymax < 0.0 || n_dr.Xmin < 0.0 || n_dr.Xmax < 0.0 || 
				n_dr.Ymin < 0.0 || n_dr.Ymax < 0.0) {
				ErrorBox((char*)SCMS_PLOT_PLACE);
				res = -1;							bContinue = true;
				Dlg->SetCheck(5, 0L, true);
				break;
				}
			if(n_dr.Xmin > (n_gr.Xmax-n_gr.Xmin) || n_dr.Ymax > (n_gr.Ymax-n_gr.Ymin) 
				|| (n_dr.Xmax-n_dr.Xmin) > (n_gr.Xmax-n_gr.Xmin)
				|| (n_dr.Ymax-n_dr.Ymin) > (n_gr.Ymax-n_gr.Ymin)){
				ErrorBox((char*)SCMS_PLOT_RECT);
				res = -1;							bContinue = true;
				Dlg->SetCheck(5, 0L, true);
				break;
				}
			bRet = true;
			break;
		case 4:
			if(res == 4 && (bRet = Command(CMD_ADDAXIS, 0L, 0L))) break;
			bContinue = true;						res = -1;
			break;
		case 5:
			Command(CMD_LEGEND, 0L, 0L);
			break;
		case 6:
			Command(CMD_LAYERS, 0L, 0L);
			bContinue = true;						res = -1;
			break;
		case 702:			//grid spacing selected
			Dlg->SetCheck(700, 0L, true);			res = -1;
			break;
		case 705:			//grid color selected
			Dlg->SetCheck(700, 0L, true);			res = -1;
			break;
		case 706:			//snap to grid
			if (Dlg->GetCheck(706))Dlg->SetCheck(700, 0L, true);
			res = -1;
			break;
			}
		}while(res <0);
	if(res == 1 && bRet) {
		Undo.SetDisp(cdisp);
		if(n_gr.Xmin != o_gr.Xmin || n_gr.Xmax != o_gr.Xmax ||
			n_gr.Ymin != o_gr.Ymin || n_gr.Ymax != o_gr.Ymax ||
			n_dr.Xmin != o_dr.Xmin || n_dr.Xmax != o_dr.Xmax ||
			n_dr.Ymin != o_dr.Ymin || n_dr.Ymax != o_dr.Ymax){
			Command(CMD_SAVEPOS, 0L, 0L);
			memcpy(&GRect, &n_gr, sizeof(fRECT));
			memcpy(&DRect, &n_dr, sizeof(fRECT));
			undo_flags |= UNDO_CONTINUE;
			}
		if(Dlg->GetColor(102, &tmpcol))
			undo_flags = CheckNewDword(&ColGR, ColGR, tmpcol, this, undo_flags);
		if(Dlg->GetColor(104, &tmpcol))
			undo_flags = CheckNewDword(&ColGRL, ColGRL, tmpcol, this, undo_flags);
		if(Dlg->GetColor(107, &tmpcol))
			undo_flags = CheckNewDword(&ColDR, ColDR, tmpcol, this, undo_flags);
		if(Dlg->GetColor(110, &tmpcol) && tmpcol != ColAX) {
			undo_flags = CheckNewDword(&ColAX, ColAX, tmpcol, this, undo_flags);
			if(Axes) for(i = 0; i < NumAxes; i++)
				if(Axes[i]) Axes[i]->SetColor(COL_AXIS | UNDO_STORESET, ColAX);
			if(Plots && NumPlots && Plots[0] && (Plots[0]->Id == GO_PLOT3D || Plots[0]->Id == GO_FUNC3D)) 
				Plots[0]->SetColor(COL_AXIS | UNDO_STORESET, ColAX);
			}
		if (Dlg->GetCheck(700)) {			//draw grid ?
			if (theGrid) {
				theGrid->hidden = false;
				if(Dlg->GetValue(702, &grid_spacing))	theGrid->SetSize(SIZE_MINE, grid_spacing);
				if (Dlg->GetColor(705, &grid_color))	theGrid->SetColor(0, grid_color);
				}
			else {
				theGrid = new Grid(this);				theGrid->hidden = false;
				if(Dlg->GetValue(702, &grid_spacing))	theGrid->SetSize(SIZE_MINE, grid_spacing);
				if(Dlg->GetColor(705, &grid_color))	theGrid->SetColor(0, grid_color);
				}
			}
		else {								//hide existing grid
			if (theGrid) theGrid->hidden = true;
			}
		if (theGrid){
			if (Dlg->GetCheck(706)) Command(CMD_DOSNAP, (void*)1, 0L);
			else Command(CMD_DOSNAP, (void*)0, 0L);
			}
		CloseDlgWnd(hDlg);	delete Dlg;
		}
	else if(res == 2) {
		Undo.SetDisp(cdisp);
		if (Undo.level() > undo_level) {	//restore plot order
			while (Undo.level() > undo_level) Undo.Restore(false, 0L);
			bRet = true;
			}
		CloseDlgWnd(hDlg);	delete Dlg;
		}
	else if(res == 3) {
		CloseDlgWnd(hDlg);	delete Dlg;
		bRet = Command(CMD_ADDPLOT, 0L, 0L);
		}
	else {
		CloseDlgWnd(hDlg);	delete Dlg;
		}
	free(GraphDlg);		 free(tab1);		free(tab2);		free(tab3);
	return bRet;
}

static char *AddAxisTmpl = (char*)
	"1,+,,DEFAULT, PUSHBUTTON,-1,148,10,45,12\n"
	".,.,,,PUSHBUTTON,-2,148,25,45,12\n"
	".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
	"4,+,50,TOUCHEXIT | ISPARENT,SHEET,1,5,10,130,130\n"
	".,.,200,ISPARENT | CHECKED,SHEET,2,5,10,130,130\n"
	".,,300,ISPARENT,SHEET,3,5,10,130,130\n"
	"50,+,100,ISPARENT | CHECKED,GROUPBOX,4,10,30,120,36\n"
	".,120,,HICOL,CHECKBOX,5,17,37,80,8\n"
	"100,+,,,RTEXT,6,10,51,35,8\n"
	".,.,,,EDVAL1,7,48,51,32,10\n"
	".,.,,,CTEXT,8,81,51,11,8\n"
	".,,,,EDVAL1,9,93,51,32,10\n"
	"120,,121,ISPARENT | CHECKED,GROUPBOX,10,10,72,120,20\n"
	"121,+,,,RTEXT,11,10,77,25,8\n"
	".,.,,,EDVAL1,12,37,77,25,10\n"
	".,.,,,LTEXT,-3,63,77,10,8\n"
	".,.,,,RTEXT,-11,73,77,25,8\n"
	".,130,,OWNDIALOG,COLBUTT,14,100,77,25,10\n"
	"130,+,,ISPARENT | CHECKED,GROUPBOX,15,10,98,120,20\n"
	".,,,,EDTEXT,0,15,103,110,10\n"
	"200,+,,,LTEXT,16,10,30,70,9\n"
	".,.,,CHECKED | EXRADIO,ODB,17,20,42,,\n"
	".,.,,EXRADIO,ODB,17,45,42,,\n"
	".,.,,EXRADIO,ODB,17,70,42,,\n"
	".,.,,EXRADIO,ODB,17,95,42,,\n"
	".,.,,EXRADIO,ODB,17,20,67,,\n"
	".,.,,EXRADIO,ODB,17,45,67,,\n"
	".,.,,EXRADIO,ODB,17,70,67,,\n"
	".,.,,EXRADIO,ODB,17,95,67,,\n"
	".,,220,ISPARENT | CHECKED, GROUPBOX,18,10,97,120,35\n"
	"220,221,,,RTEXT,-4,10,105,15,8\n"
	"221,222,,,EDVAL1,19,27,105,35,10\n"
	"222,223,,,LTEXT,-7,65,105,5,8\n"
	"223,224,,,EDVAL1,20,71,105,35,10\n"
	"224,225,,,LTEXT,-3,109,105,15,8\n"
	"225,226,,,RTEXT,-5,10,117,15,8\n"
	"226,227,,,EDVAL1,21,27,117,35,10\n"
	"227,228,,,LTEXT,-7,65,117,5,8\n"
	"228,229,,,EDVAL1,22,71,117,35,10\n"
	"229,,,,LTEXT,-3,109,117,15,8\n"
	"300,,,LASTOBJ | NOSELECT,ODB,23,15,30,110,140";

bool
Graph::AddAxis(fRECT *frame)
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_AXIS);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_STYLE);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_PLOTS);
	AxisDEF axis;
	double sizAxLine = DefSize(SIZE_AXIS_LINE);
	DWORD colAxis = ColAX;
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)SDLG_ADDA_SCALE, (void*)SDLG_ADDA_ASCALE,
		(void*)SDLG_ADDA_FROM, (void*)&y_axis.min, (void*)SDLG_ADDA_TO, (void*)&y_axis.max, (void*)SDLG_ADDA_LIN,
		(void*)SDLG_ADDA_LWIDTH, (void*)&sizAxLine, (void*)0L, (void *)&colAxis, (void*)SDLG_ADDA_LBL,
		(void*)SDLG_ADDA_TMPL, (void*)(OD_NewAxisTempl), (void*)SDLG_ADDA_PLC, (void*)&axis.loc[0].fx,
		(void*)&axis.loc[1].fx, (void*)&axis.loc[0].fy, (void*)&axis.loc[1].fy,  (void*)OD_axisplot};
	DlgInfo *NewAxisDlg;
	DlgRoot *Dlg;
	void *hDlg;
	int j, res, currTempl = 201;
	double vx1, vx2, vy1, vy2, hx1, hx2, hy1, hy2;
	double tlb_dist, tlb_dx, tlb_dy, lb_x, lb_y;
	TextDEF label_def, tlbdef;
	DWORD flags;
	anyOutput *cdisp = Undo.Cdisp();
	Axis *the_new, **tmpAxes;
	bool bAxis = false, bRet = false;
	char **names;
	GraphObj **somePlots;
	Label *label;

	if(!(NewAxisDlg = CompileDialog(AddAxisTmpl, dyndata)))return false;
	if(!(names = (char**)calloc(nscp+2, sizeof(char*))))return false;
	if(!(somePlots = (GraphObj**)calloc(nscp+2, sizeof(GraphObj*))))return false;
	if(!Axes) Axes = (Axis**)calloc(2, sizeof(Axis*));
	if ((names[0] = (char*)malloc(10))) rlp_strcpy(names[0], 10, (char*)SDLG_ADDA_NONE);
	for(i = 0, j = 1; i < nscp; i++) {
		if(Sc_Plots[i] && Sc_Plots[i]->name){
			names[j] = rlp_strdup(Sc_Plots[i]->name);
			somePlots[j++] = Sc_Plots[i];
			}
		}
	OD_axisplot(OD_ACCEPT, 0L, 0L, 0L, names, 0);
	if (frame) {
		axis.loc[0].fx = axis.loc[1].fx = vx1 = vx2 = frame->Xmax;
		axis.loc[0].fy = vy1 = frame->Ymin;
		axis.loc[1].fy = vy2 = frame->Ymax;
		}
	else {
		axis.loc[0].fx = axis.loc[1].fx = vx1 = vx2 = (DRect.Xmin + DRect.Xmax) / 2.0;
		axis.loc[0].fy = vy1 = DRect.Ymin;
		axis.loc[1].fy = vy2 = DRect.Ymax;
		}
	axis.min = y_axis.min;		axis.max = y_axis.max;
	hy1 = hy2 = (DRect.Ymax + DRect.Ymin)/2.0;
	hx1 = DRect.Xmin;			hx2 = DRect.Xmax;
	if(!(Dlg = new DlgRoot(NewAxisDlg, data)))return false;
	if(type != 1 || !nscp){		//must be standard graph to link to plot
		Dlg->ShowItem(6, false);
		}
	hDlg = CreateDlgWnd((char*)SDLG_ADDA_HD1, 50, 50, 420, 338, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res){
		case 4:											//the axis sheet
			res = -1;
			bAxis = true;
			break;
		case 201:	case 202:	case 203:	case 204:	//axis templates
		case 205:	case 206:	case 207:	case 208:
			if (currTempl == res) {
				res = 1;		break;
				}
			if(currTempl > 204) {
				Dlg->GetValue(221, &hx1);				Dlg->GetValue(223, &hx2);
				Dlg->GetValue(226, &hy1);				Dlg->GetValue(228, &hy2);
				}
			else {
				Dlg->GetValue(221, &vx1);				Dlg->GetValue(223, &vx2);
				Dlg->GetValue(226, &vy1);				Dlg->GetValue(228, &vy2);
				}
			if(res > 204) {
				Dlg->SetValue(221, hx1);				Dlg->SetValue(223, hx2);
				Dlg->SetValue(226, hy1);				Dlg->SetValue(228, hy2);
				if(! bAxis) {
					Dlg->SetValue(101, x_axis.min);		Dlg->SetValue(103, x_axis.max);
					}
				}
			else {
				Dlg->SetValue(221, vx1);				Dlg->SetValue(223, vx2);
				Dlg->SetValue(226, vy1);				Dlg->SetValue(228, vy2);
				if(! bAxis) {
					Dlg->SetValue(101, y_axis.min);		Dlg->SetValue(103, y_axis.max);
					}
				}
			currTempl = res;
			Dlg->DoPlot(0L);
			res = -1;
			break;
			}
		}while (res < 0);
	if(res == 1) {
		Undo.SetDisp(cdisp);
		Dlg->GetValue(122, &sizAxLine);				Dlg->GetColor(125, &colAxis);
		Dlg->GetValue(221, &axis.loc[0].fx);		Dlg->GetValue(223, &axis.loc[1].fx);
		Dlg->GetValue(226, &axis.loc[0].fy);		Dlg->GetValue(228, &axis.loc[1].fy);
		axis.loc[0].fz = axis.loc[1].fz = 0.0;		axis.owner = 0L;
		Dlg->GetValue(101, &axis.min);				Dlg->GetValue(103, &axis.max);
		axis.Start = axis.min;				axis.Center.fx = axis.Center.fy = 0.0;
		axis.nBreaks = 0;			axis.breaks = 0L;		axis.Radius = 0.0;	
		tlb_dist = NiceValue(DefSize(SIZE_AXIS_TICKS));
		tlb_dx = tlb_dy = 0.0;
		tlbdef.ColTxt = colAxis;				tlbdef.ColBg = 0x00ffffffL;
		tlbdef.RotBL = tlbdef.RotCHAR = 0.0f;	tlbdef.fSize = DefSize(SIZE_TICK_LABELS);
		tlbdef.Align = TXA_VCENTER | TXA_HCENTER;
		tlbdef.Style = TXS_NORMAL;				tlbdef.Mode = TXM_TRANSPARENT;
		tlbdef.Font = FONT_HELVETICA;			tlbdef.text = 0L;
		label_def.ColTxt = colAxis;				label_def.ColBg = 0x00ffffffL;
		label_def.fSize = DefSize(SIZE_TICK_LABELS)*1.2f;	label_def.RotBL = 0.0f;
		label_def.RotCHAR = 0.0f;								label_def.iSize = 0;
		label_def.Align = TXA_VTOP | TXA_HCENTER;		label_def.Mode = TXM_TRANSPARENT;
		label_def.Style = TXS_NORMAL;	label_def.Font = FONT_HELVETICA;
		switch (currTempl) {
		default:	flags = AXIS_NEGTICKS;	axis.Step = y_axis.Step;
			tlb_dx = -tlb_dist * 2.0;	tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			lb_x = -tlb_dist * 6.0;		lb_y = 0;	label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label_def.RotBL = 90.0;
			break;
		case 202:	flags = AXIS_POSTICKS;	axis.Step = y_axis.Step;
			tlb_dx = -tlb_dist;			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			lb_x = -tlb_dist * 4.5;		lb_y = 0;	label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label_def.RotBL = 90.0;
			break;
		case 203:	flags = AXIS_POSTICKS;	axis.Step = y_axis.Step;
			tlb_dx = tlb_dist * 2.0;	tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
			lb_x = tlb_dist * 6;		lb_y = 0;	label_def.Align = TXA_VTOP | TXA_HCENTER;
			label_def.RotBL = 90.0;
			break;
		case 204:	flags = AXIS_NEGTICKS;	axis.Step = y_axis.Step;
			tlb_dx = tlb_dist;			tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
			lb_x = tlb_dist * 4.5;		lb_y = 0;	label_def.Align = TXA_VTOP | TXA_HCENTER;
			label_def.RotBL = 90.0;
			break;
		case 205:	flags = AXIS_NEGTICKS;	axis.Step = x_axis.Step;
			tlb_dy = tlb_dist * 2.0;	tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			lb_x = 0;					lb_y = tlb_dist * 4.0;
			break;
		case 206:	flags = AXIS_POSTICKS;	axis.Step = x_axis.Step;
			tlb_dy = tlb_dist;			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			lb_x = 0;					lb_y = tlb_dist * 3.0;
			break;
		case 207:	flags = AXIS_POSTICKS;	axis.Step = x_axis.Step;
			tlb_dy = -tlb_dist * 2.0;	tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			lb_x = 0;	lb_y = -tlb_dist * 4.0;	label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			break;
		case 208:	flags = AXIS_NEGTICKS;	axis.Step = x_axis.Step;
			tlb_dy = -tlb_dist;			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			lb_x = 0;	lb_y = -tlb_dist * 3.0;	label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			break;
			}
		flags |= AXIS_AUTOTICK;			flags |= AXIS_DEFRECT;
		if(Dlg->GetCheck(51)) flags |= AXIS_AUTOSCALE;
		if((the_new = new Axis(this, data, &axis, flags))){
			the_new->SetSize(SIZE_TLB_YDIST, tlb_dy);	the_new->SetSize(SIZE_TLB_XDIST, tlb_dx); 
			the_new->SetSize(SIZE_AXIS_LINE, sizAxLine);
			the_new->SetColor(COL_AXIS, colAxis);
			the_new->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			the_new->SetSize(SIZE_LB_XDIST, lb_x);		the_new->SetSize(SIZE_LB_YDIST, lb_y); 
			the_new->moveable = 1;
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, 100, (char*)"new Axis[%ld]", NumAxes +1);
#else
			sprintf(TmpTxt, (char*)"new Axis[%ld]", NumAxes +1);
#endif
			the_new->name = rlp_strdup(TmpTxt);
			if(Dlg->GetText(131, (unsigned char*)TmpTxt, TMP_TXT_SIZE)) label_def.text = (unsigned char*)TmpTxt;
			else label_def.text = 0L;
			label = new Label(Axes[0], data, (axis.loc[0].fx + axis.loc[1].fx) / 2.0, (axis.loc[0].fy + axis.loc[1].fy) / 2.0, &label_def,
				label_def.RotBL < 45.0 ? LB_Y_PARENT : LB_X_PARENT, 0L);
			if(label){
				label->SetSize(SIZE_LB_XDIST, lb_x);	label->SetSize(SIZE_LB_YDIST, lb_y); 
				if(the_new->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
				else DeleteGO(label);
				}
			for(i = 0; i < NumAxes && Axes[i]; i++);
			if(i < NumAxes) {
				Undo.SetGO(this, (GraphObj**)(&Axes[i]), the_new, 0L);
				bRet = true;
				}
			else {
				if((tmpAxes = (Axis**)calloc(NumAxes+2, sizeof(Axis*)))){
					memcpy(tmpAxes, Axes, NumAxes * sizeof(Axis*));
					Undo.ListGOmoved((GraphObj**)Axes, (GraphObj**)tmpAxes, NumAxes);
					Undo.SetGO(this, (GraphObj**)(&tmpAxes[NumAxes]), the_new, 0L);
					free(Axes);			Axes = tmpAxes;
					i = NumAxes++;		bRet = true;
					}
				else delete (the_new);	//very unlikely memory allocation error
				}
			CurrAxes = Axes;
			if(bRet) {
				OD_axisplot(OD_ACCEPT, 0L, 0L, (anyOutput*) &res, 0L, 0);
				if(res && i) somePlots[res]->Command(CMD_USEAXIS, &i, 0L);
				}
			}
		}
	CloseDlgWnd(hDlg);		delete Dlg;			free(NewAxisDlg);
	if(names) {
		for(j = 0; names[j]; j++) if(names[j]) free(names[j]);
		free(names);
		}
	free(tab1);		free(tab2);		free(tab3);
	if(somePlots) free(somePlots);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Page properties
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *PageDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,135,10,45,12\n"
".,.,,,PUSHBUTTON,-2,135,25,45,12\n"
".,,4,ISPARENT | CHECKED,GROUP,,,,,\n"
"4,+,100,ISPARENT | TOUCHEXIT,SHEET,1,5,25,120,120\n"
".,.,200,ISPARENT | TOUCHEXIT,SHEET,4,5,25,120,120\n"
".,.,,HIDDEN,PUSHBUTTON,2,135,135,45,12\n"
".,10,,HIDDEN,PUSHBUTTON,11,135,118,45,12\n"
"10,,,,CHECKPIN,,5,5,12,8\n"
"100,,,NOSELECT,ODBUTTON,3,15,45,110,160\n"
"200,+,,HICOL,CHECKBOX,5,17,50,40,8\n"
".,.,,,RTEXT,6,27,70,40,8\n"
".,.,,TOUCHEXIT,EDVAL1,7,72,70,30,10\n"
".,.,,,LTEXT,-3,105,70,20,8\n"
".,.,,,RTEXT,8,27,85,40,8\n"
".,.,,TOUCHEXIT | OWNDIALOG,COLBUTT,9,72,85,30,10\n"
".,,,LASTOBJ | HICOL | TOUCHEXIT,CHECKBOX,10,17,105,40,8";

bool
Page::Configure()
{
	int i;
	double grid_spacing;
	DWORD grid_color;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_PAPER_SIZE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_GRID);
	if (theGrid) {
		grid_spacing = theGrid->GetSize(SIZE_MINE);
		}
	else switch (defs.units()) {
	case 1:		grid_spacing = 1.0;		break;
	case 2:		grid_spacing = 0.5;	break;
	default:	grid_spacing = 10.0;		break;
		}
	if (theGrid) grid_color = theGrid->GetColor(0);
	else grid_color = 0x00cbcbcb;
	void *dyndata[] = { (void*)tab1, (void*)SDLG_PAGE_LAYER, (void*)OD_paperdef, (void*)tab2, (void*)SDLG_GRAPH_GRID,
		(void*)SDLG_GRAPH_GRIDSPACING, (void*)&grid_spacing, (void*)SDLG_GRAPH_GRIDCOLOR, (void*)&grid_color, (void*)SDLG_GRAPH_SNAP,
		(void*)SDLG_GRAPH_ADDGRAPH };
	DlgInfo *PageDlg = CompileDialog(PageDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int res, cb;
	static int last_tab = 4;
	bool bRet = false, bContinue = false;

	FindPaper(GRect.Xmax - GRect.Xmin, GRect.Ymax -GRect.Ymin, .0001);
	if(!(Dlg = new DlgRoot(PageDlg, data)))return false;
	if (OwnDisp) Dlg->ShowItem(7, true);
	else last_tab = 4;
	if (Command(CMD_HASSTACK, 0L, 0L)) Dlg->ShowItem(6, true);
	if (theGrid){
		if (theGrid->hidden) Dlg->SetCheck(200, 0L, false);
		else Dlg->SetCheck(200, 0L, true);
		if (theGrid->type & 0x01) Dlg->SetCheck(206, 0L, true);
		else Dlg->SetCheck(206, 0L, false);
		}
	if(name)cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, name);
	else cb = rlp_strcpy(TmpTxt, TMP_TXT_SIZE, (char*)SDLG_PAGE_HD1);
	rlp_strcpy(TmpTxt + cb, TMP_TXT_SIZE - cb, (char*)SDLG_PAGE_HD2);
	Dlg->SetCheck(last_tab, 0L, true);
	hDlg = CreateDlgWnd(TmpTxt, 50, 50, 400, 350, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:							//loose focus
			if (Dlg->GetCheck(10)) res = -1;
			if (bContinue) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 4:			case 5:			//the tab sheets
			last_tab = res;	res = -1;
			break;
		case 6:
			Command(CMD_LAYERS, 0L, 0L);
			bContinue = true;
			res = -1;		break;
		case 7:
			Command(CMD_NEWGRAPH, 0L, 0L);
			bContinue = true;
			res = -1;		break;
		case 202:			//grid spacing selected
			Dlg->SetCheck(200, 0L, true);			res = -1;
			break;
		case 205:			//grid color selected
			Dlg->SetCheck(200, 0L, true);			res = -1;
			break;
		case 206:			//snap to grid
			if(Dlg->GetCheck(206)) Dlg->SetCheck(200, 0L, true);
			res = -1;
			break;
		}
		}while(res <0);
	if(res == 1) {
		OD_paperdef(OD_ACCEPT, 0L, 0L, 0L, 0L, 0);
		GRect.Xmin = GRect.Ymin = 0.0;
		GetPaper(&GRect.Xmax, &GRect.Ymax);
		DoPlot(CurrDisp);
		if (Dlg->GetCheck(200)) {			//draw grid ?
			if (theGrid) {
				theGrid->hidden = false;
				if (Dlg->GetValue(202, &grid_spacing))	theGrid->SetSize(SIZE_MINE, grid_spacing);
				if (Dlg->GetColor(205, &grid_color))	theGrid->SetColor(0, grid_color);
				}
			else {
				theGrid = new Grid(this);				theGrid->hidden = false;
				if (Dlg->GetValue(202, &grid_spacing))	theGrid->SetSize(SIZE_MINE, grid_spacing);
				if (Dlg->GetColor(205, &grid_color))	theGrid->SetColor(0, grid_color);
				}
			}
		else {								//hide existing grid
				if (theGrid) theGrid->hidden = true;
				}
		if (theGrid){
			if (Dlg->GetCheck(206)) Command(CMD_DOSNAP, (void*)1, 0L);
			else Command(CMD_DOSNAP, (void*)0, 0L);
			}
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;		free(PageDlg);		 free(tab1);
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Edit global defaults
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* DefsDlg_DlgTmpl = (char*)
"1,+,,DEFAULT,PUSHBUTTON,-1,210,10,40,12\n"
".,.,,,PUSHBUTTON,-2,210,25,40,12\n"
".,,4,ISPARENT | CHECKED,GROUP,,,,,3\n"
"4,+,100,TOUCHEXIT | ISPARENT | CHECKED,SHEET,1,5,10,190,130\n"
".,.,200,TOUCHEXIT | ISPARENT,SHEET,2,5,10,190,130\n"
".,.,300,TOUCHEXIT | ISPARENT,SHEET,3,5,10,190,130\n"
".,.,400,TOUCHEXIT | ISPARENT,SHEET,4,5,10,190,130\n"
".,., 350,TOUCHEXIT | ISPARENT,SHEET,5,5,10,190,130\n"
".,, 450,TOUCHEXIT | ISPARENT,SHEET,41,5,10,190,130\n"
"100,,,NOSELECT,ODBUTTON,6,30,38,130,100\n"
"200,,,NOSELECT,ODBUTTON,7,45,50,90,50\n"
"300,+,,,RTEXT,8,20,30,38,8\n"
".,.,,,EDVAL1,9,60,30,20,10\n"
".,.,,,LTEXT,10,82,30,20,8\n"
".,.,,,LTEXT,33,20,53,120,8\n"
".,.,,,RTEXT,34,20,68,50,8\n"
".,.,,,EDVAL1,35,72,68,20,10\n"
".,.,,,LTEXT,10,94,68,20,8\n"
".,.,,,RTEXT,36,20,80,50,8\n"
".,.,,,EDVAL1,37,72,80,20,10\n"
".,.,,,LTEXT,10,94,80,20,8\n"
".,.,,,LTEXT,38,20,110,120,8\n"
".,,,,LTEXT,39,25,122,120,8\n"
"350,+,,,LTEXT,11,10,30,120,8\n"
".,.,,,LTEXT,12,10,43,70,8\n"
".,.,,,EDTEXT,13,10,53,60,10\n"
".,.,,,LTEXT,14,80,54,40,10\n"
".,.,,,LTEXT,15,10,65,70, 8\n"
".,.,,,EDTEXT,16,10,75,60,10\n"
".,.,,,LTEXT,17,80,76,40,10\n"
".,.,,,LTEXT,18,10,87,70,8\n"
".,.,,,EDTEXT,19,10,97,60,10\n"
".,.,,,LTEXT,20,80,98,40,10\n"
".,.,,,LTEXT,21,10,119,140,8\n"
".,.,,HREF | TOUCHEXIT,LTEXT,22,10,127,140,8\n"
".,,,,PUSHBUTTON,23,130,107,40,12\n"
"400,+,,,LTEXT,24,20,30,100,8\n"
".,.,,,RTEXT,25,45,45,40,8\n"
".,.,,,EDTEXT,26,90,45,10,10\n"
".,.,,,RTEXT,27,45,57,40,8\n"
".,.,,,EDTEXT,28,90,57,10,10\n"
".,.,,,RTEXT,29,25,75,47,8\n"
".,.,420,ISPARENT | CHECKED,GROUP,,,,,\n"
".,,,,PUSHBUTTON,40,140,110,40,12\n"
"420,+,,HICOL,RADIO1,30,75,75,20,8\n"
".,.,,HICOL,RADIO1,31,75,83,20,8\n"
".,.,,HICOL,RADIO1,32,75,91,20,8\n"
"450,+,,,LTEXT,42,15,25,120,8\n"
".,.,,,LTEXT,43,20,37,140,8\n"
".,.,,,LTEXT,44,25,46,140,8\n"
".,.,,,LTEXT,45,20,58,140,8\n"
".,.,,,LTEXT,46,25,67,140,8\n"
".,.,,,LTEXT,47,20,79,140,8\n"
".,.,,,LTEXT,48,25,88,140,8\n"
".,.,,,LTEXT,49,20,100,140,8\n"
".,.,,,LTEXT,50,25,109,140,8\n"
".,.,,,PUSHBUTTON,51,135,36,50,12\n"
".,.,,,PUSHBUTTON,51,135,57,50,12\n"
".,.,,,PUSHBUTTON,51,135,78,50,12\n"
".,,,LASTOBJ,PUSHBUTTON,51,135,99,50,12";

bool
def_vars::PropertyDlg()
{
	int i;
	TabSHEET *tab1 = MakeTab(0, 10, &i, (char*)SDLG_TAB_LINE);
	TabSHEET *tab2 = MakeTab(i, 10, &i, (char*)SDLG_TAB_SHAPES);
	TabSHEET *tab3 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DIALOGS);
	TabSHEET *tab4 = MakeTab(i, 10, &i, (char*)SDLG_TAB_INTERNAT);
	TabSHEET *tab5 = MakeTab(i, 10, &i, (char*)SDLG_TAB_DATETIME);
	TabSHEET* tab6 = MakeTab(i, 10, &i, (char*)SDLG_DEFS_FONTS);
	double ts =  dlgtxtheight;
	double dwidth = defs.DlgSizeAdj.x, dheight = defs.DlgSizeAdj.y;
	time_t ti = time(0L);
	unsigned char dt_info[50], date_info[50], datetime_info[50], time_info[50];
	void *dyndata[] = { (void*)tab1, (void*)tab2, (void*)tab3, (void*)tab4, (void*)tab5, (void*)OD_linedef,
		(void*)OD_filldef, (void*)SDLG_DEFS_TSIZE, (void*)&ts, (void*)SDLG_DEFS_PIX, (void*)dt_info,
		(void*)SDLG_DEFS_DFMT, (void*)defs.fmt_date, (void*)date_info, (void*)SDLG_DEFS_DTFMT,
		(void*)defs.fmt_datetime, (void*)datetime_info, (void*)SDLG_DEFS_TFMT, (void*)defs.fmt_time,
		(void*)time_info, (void*)SDLG_DEFS_FMT_INFO, (void*)SHREF_DTIME, (void*)SDLG_DEFS_TEST,
		(void*)SDLG_DEFS_INTL, (void*)SDLG_DEFS_DECP, (void*)DecPoint, (void*)SDLG_DEFS_CSEP,
		(void*)ColSep, (void*)SDLG_DEFS_UNITS, (void*)Units[0].display, (void*)Units[1].display,
		(void*)Units[2].display, (void*)SDLG_DEFS_ADJ, (void*)SDLG_DEFS_XINC, (void*)&dwidth,
		(void*)SDLG_DEFS_YINC, (void*)&dheight, (void*)SDLG_DEFS_FILE, (void*)IniFile,
		(void*)SDLG_DEFS_PAGE, (void*)tab6, (void*) SDLG_DEFS_FHD, (void*)SDLG_DEFS_SANS,
		(void*)defs.font_sans, (void*)SDLG_DEFS_SER, (void*)defs.font_ser, (void*)SDLG_DEFS_FIX,
		(void*)defs.font_fix, (void*)SDLG_DEFS_GREEK, defs.font_greek, (void*)SDLG_DEFS_SELF };
	DlgInfo *DefsDlg = CompileDialog(DefsDlg_DlgTmpl, dyndata);
	DlgRoot *Dlg;
	void *hDlg;
	int cb, res, tmpUnits = dUnits;
	bool bRet = false, bContinue = false;
	double dt;
	LineDEF LineDef;
	static FillDEF FillDef;
	char* tmp_font;

	OD_linedef(OD_SETLINE, 0L, 0L, 0L, (void *)GetLine(), 0);
	OD_filldef(OD_SETLINE, 0L, 0L, 0L, (void *)GetOutLine(), 0);
	OD_filldef(OD_SETFILL, 0L, 0L, 0L, (void *)GetFill(dUnits), 0);
	cb = rlp_strcpy(dt_info, 20, (char*)"today is ");
#ifdef USE_WIN_SECURE
	ctime_s((char*)dt_info+cb, 50-cb, &ti);
#else
	rlp_strcpy((char*)dt_info+cb, 50-cb, ctime(&ti));
#endif
	dt_info[cb+24] = 0;
	date_value((char*)(dt_info+13), (char*)"x z H:M:S Y", &dt);
	rlp_strcpy(date_info, 50, value_date(dt, defs.fmt_date));
	rlp_strcpy(datetime_info, 50, value_date(dt, defs.fmt_datetime));
	rlp_strcpy(time_info, 50, value_date(dt, defs.fmt_time));
	Dlg = new DlgRoot(DefsDlg, 0L);
	Dlg->TextFont(452, FONT_COURIER);	Dlg->TextFont(454, FONT_COURIER);
	Dlg->TextFont(456, FONT_COURIER);	Dlg->TextFont(458, FONT_COURIER);
	switch(dUnits) {
	case 1:		Dlg->SetCheck(421, 0L, true);	break;
	case 2:		Dlg->SetCheck(422, 0L, true);	break;
	default:	Dlg->SetCheck(420, 0L, true);	break;
		}
	Dlg->SetText(452, (unsigned char*)FontName(FONT_HELVETICA));
	Dlg->SetText(454, (unsigned char*)FontName(FONT_TIMES));
	Dlg->SetText(456, (unsigned char*)FontName(FONT_COURIER));
	Dlg->SetText(458, (unsigned char*)FontName(FONT_GREEK));
#ifdef _WINDOWS
	for(i = 360; i <= 361; i++) Dlg->TextSize(i, 12);
#else
	for(i = 360; i <= 361; i++) Dlg->TextSize(i, 10);
#endif
	hDlg = CreateDlgWnd((char*)SDLG_DEFS_HD1, 50, 50, 540, 336, Dlg, 0x4L);
	do{
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch(res) {
		case 0:
			if(bContinue) res = -1;
			break;
		case 362:				//update date/time display
			ti = time(0L);		cb = rlp_strcpy(dt_info, 20, (char*)"today is ");
#ifdef USE_WIN_SECURE
			ctime_s((char*)dt_info+cb, 50-cb, &ti);
#else
			rlp_strcpy((char*)dt_info+cb, 50-cb, ctime(&ti));
#endif
			dt_info[cb+24] = 0;														Dlg->SetText(350, dt_info);
			date_value((char*)(dt_info+13), (char*)"x z H:M:S Y", &dt);
			if(!(Dlg->GetText(352, date_info, 50))) rlp_strcpy(date_info, 50, defs.fmt_date);
			rlp_strcpy(date_info, 50, value_date(dt, (char*)date_info));					Dlg->SetText(353, date_info);
			if(!(Dlg->GetText(355, datetime_info, 50))) rlp_strcpy(datetime_info, 50, defs.fmt_datetime);
			rlp_strcpy(datetime_info, 50, value_date(dt, (char*)datetime_info));			Dlg->SetText(356, datetime_info);
			if(!(Dlg->GetText(358, time_info, 50))) rlp_strcpy(time_info, 50, defs.fmt_time);
			rlp_strcpy(time_info, 50, value_date(dt, (char*)time_info));			Dlg->SetText(359, time_info);
			bContinue = false;		res = -1;
			break;
		case 407:
			ConfigPrinter();
			bContinue = true;		res = -1;
			break;
		case 459:			//select new font sans serif
			if ((tmp_font = GetNewFont(defs.font_sans, FONT_HELVETICA))) {
				Dlg->SetText(452, (unsigned char*)FontName(FONT_HELVETICA));
				if (defs.font_sans) free(defs.font_sans);
				defs.font_sans = rlp_strdup(tmp_font);
				}
			bContinue = true;		res = -1;
			break;
		case 460:			//select new serif font
			if ((tmp_font = GetNewFont(defs.font_ser, FONT_TIMES))) {
				Dlg->SetText(454, (unsigned char*)FontName(FONT_TIMES));
				if (defs.font_ser) free(defs.font_ser);
				defs.font_ser = rlp_strdup(tmp_font);
				}
			bContinue = true;		res = -1;
			break;
		case 461:			//select new fixed space font
			if ((tmp_font = GetNewFont(defs.font_fix, FONT_COURIER))) {
				Dlg->SetText(456, (unsigned char*)tmp_font);
				if (defs.font_fix) free(defs.font_fix);
				defs.font_fix = rlp_strdup(tmp_font);
				}
			bContinue = true;		res = -1;
			break;
		case 462:			//select new greek base font
			if ((tmp_font = GetNewFont(defs.font_greek, FONT_GREEK))) {
				Dlg->SetText(458, (unsigned char*)tmp_font);
				if (defs.font_greek) free(defs.font_greek);
				defs.font_greek = rlp_strdup(tmp_font);
				}
			bContinue = true;		res = -1;
			break;
		case 361:			//call browser
			bContinue = true;		res = -1;
			break;
		case 4:		case 5:		case 6:		case 7:
			bContinue = false;		res = -1;
			break;
		case 8:
			bContinue = true;		res = -1;
			break;
		case 9:				//the 'Fonts' tab
			bContinue = false;		res = -1;
			break;
		case 1:
			if(Dlg->GetCheck(421)) dUnits = 1;
			else if(Dlg->GetCheck(422)) dUnits = 2;
			else dUnits = 0;
			}
		}while (res < 0);
	if(res == 1) {
		if(Dlg->GetText(402, (unsigned char*)TmpTxt, TMP_TXT_SIZE))	DecPoint[0] = TmpTxt[0];
		if(Dlg->GetText(404, (unsigned char*)TmpTxt, TMP_TXT_SIZE))	ColSep[0] = TmpTxt[0];
		OD_linedef(OD_GETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
		SetLine(tmpUnits, &LineDef, 0);
		OD_filldef(OD_GETLINE, 0L, 0L, 0L, (void *)&LineDef, 0);
		SetLine(tmpUnits, &LineDef, 2);
		OD_filldef(OD_GETFILL, 0L, 0L, 0L, (void *)&FillDef, 0);
		SetFill(tmpUnits, &FillDef);		Dlg->GetInt(301, &dlgtxtheight);
		if(Dlg->GetText(352, date_info, 50) && date_info[0] && strcmp((char*)date_info, defs.fmt_date)) {
			defs.fmt_date = (char*)realloc(defs.fmt_date, (cb = rlp_strlen(date_info) + 2));
			if (defs.fmt_date) rlp_strcpy(defs.fmt_date, cb, date_info);
			}
		if(Dlg->GetText(355, datetime_info, 50) && datetime_info[0] && strcmp((char*)datetime_info, defs.fmt_datetime)) {
			defs.fmt_datetime = (char*)realloc(defs.fmt_datetime, (cb = rlp_strlen(datetime_info) + 2));
			if (defs.fmt_datetime) rlp_strcpy(defs.fmt_datetime, cb, datetime_info);
			}
		if(Dlg->GetText(358, time_info, 50) && time_info[0] && strcmp((char*)time_info, defs.fmt_time)) {
			defs.fmt_time = (char*)realloc(defs.fmt_time, (cb = rlp_strlen(time_info) + 2));
			if (defs.fmt_time) rlp_strcpy(defs.fmt_time, cb, time_info);
			}
		if (Dlg->GetValue(305, &dwidth))defs.DlgSizeAdj.x = iround(dwidth);
		if (Dlg->GetValue(308, &dheight))defs.DlgSizeAdj.y = iround(dheight);
		defs.DlgSizeAdj.z = 0;
		bRet = true;
		}
	CloseDlgWnd(hDlg);
	delete Dlg;		free(DefsDlg);
	free(tab1);	free(tab2);	free(tab3);	free(tab4);	free(tab5);
	return bRet;
}
