//QT5_Spec.cpp, Copyright (c) 2001-2025 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This modules contains code specific for Qt5 graphics. It may be specific
//    to the Linux operating system
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "rlplot.h"
#include "menu.h"
#include "rlp_strings.h"
#include "QT6_Spec.h"

QApplication *QAppl;
extern GraphObj *CurrGO;		//Selected Graphic Objects
extern GraphObj *CopyGO;
static GraphObj *MouseObj = 0L;		//accepts mouse move events
extern def_vars defs;
extern tag_Units Units[];
extern UndoObj Undo;
extern Graph *CurrGraph;
extern char *WWWbrowser;
extern char *LoadFile;
extern char TmpTxt[];
extern void *prog_bar_ptr;
extern tagProgBarDlg *ProgressBarDlg;
extern notary *Notary;
QWidget *MainWidget =0L, *CurrWidget = 0L;
POINT CurrWidgetPos = {0,0};
char *gui_version = 0L;
static int mouse_buttons_down = 0;
static QIcon *rlp_icon = 0L;
static char UserFileName[1024];
static char *ShellCmd = 0L;
static QClipboard *clipboard;
static QPrinter *printer = 0L;
static POINT mouse_offs = {0,0};		//translate mouse position screen to widget
static int mouse_lx = 0, mouse_ly = 0;		//last mouse cursor position
static int desk_width = 0, desk_height = 0;	//desktop size
static char *char_map = 0L;
static QFont qf_sans(QString("Helvetica")), qf_ser(QString("Times")), qf_fix(QString("Courier"));
static QFont qf_greek(QString("Times")), qf_sys(QString("Helvetica"));

//forward declaration
void ExportBitmap(GraphObj *SourceGO, char *Filename, char *format = (char*)".png");

#define MK_QCOLOR(col) QColor(col&0xff,(col>>8)&0xff,(col>>16)&0xff,255-((col>>24)&0xff))

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Exute a file open dialog for the different situations
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a new file name to store data in
char *SaveDataAsName(char *oldname)
{
	int i, cb;
	char ext[1024];
	QFileDialog qf((QWidget*)0L, (Qt::WindowFlags)0);
	const QStringList filters({"RLPlot workbook *.rlw (*.rlw)", "comma separated file *.csv (*.csv)", "text file *.txt (*.txt)"});

	qf.setNameFilters(filters);
	qf.setDirectory(defs.currPath);			qf.setViewMode(QFileDialog::Detail);
	qf.selectFile(oldname);				qf.setFileMode(QFileDialog::AnyFile);
	qf.setAcceptMode(QFileDialog::AcceptSave);
	if(qf.exec() == QDialog::Accepted) {
		if(!qf.selectedFiles().isEmpty()) {
			rlp_strcpy(ext, 1023, qf.selectedNameFilter().toLatin1().data());
			cb = rlp_strcpy(UserFileName, 1024, (char*)qf.selectedFiles().first().toLatin1().data());
			if(cb < 5 || (UserFileName[cb-4] != '.' && UserFileName[cb-5] != '.')){
				for(i = 0; ext[i] && ext[i] != '*'; i++);
				rlp_strcpy(UserFileName+cb, 5, ext+i+1);
				}
			defs.FileHistory(UserFileName);
			return UserFileName;
			}
		}
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a new file name to store graph
char *SaveGraphAsName(char *oldname)
{
	int i, nb;
	QString fileName = QFileDialog::getSaveFileName(0L, "Save Graph", oldname?oldname:defs.currPath,
		"RLPlot files (*.rlp)");
	nb = (int)fileName.size();
	if(nb) {
		for(i = 0; i< nb; i++) {
			UserFileName[i] = (fileName.data()[i]).toLatin1();
			}
		UserFileName[i++] = '\0';
		if(0 != strcasecmp(".rlp", UserFileName+i-5)) {
			rlp_strcpy(UserFileName+i-1, 1024-i, (char*)".rlp"); 
			}
		defs.FileHistory(UserFileName);
		return UserFileName;
		}
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a file name to load data
char *OpenDataName(char *oldname)
{
	int i, nb;

	QString fileName = QFileDialog::getOpenFileName(0L, "Open File", oldname?oldname:defs.currPath,
		"RLPlot workbook *.rlw (*.rlw)\ncomma separated file *.csv (*.csv)\ntab separated file *.tsv, *.txt (*.tsv)\n"
		"RLPlot files *.rlp (*.rlp)\nall files *.* (*.*)");
	nb = (int)fileName.size();
	if(nb){
		for(i = 0; i< nb; i++) {
			UserFileName[i] = (fileName.data()[i]).toLatin1();
			}
		UserFileName[i++] = '\0';
		defs.FileHistory(UserFileName);
		return UserFileName;
		}
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a file name to export graph
void OpenExportName(GraphObj *g, char *oldname)
{
	int i, cb;
	QFileDialog qf((QWidget*)0L, (Qt::WindowFlags)0);
	const QStringList filters({"Scalable Vector Graphics (*.svg)", "Encapsulated Post Script (*.eps)",
		"Portable Network Graphics (*.png)"});
	char file_name[TMP_TXT_SIZE];

	if(!g) return;
	qf.setNameFilters(filters);
	qf.setDirectory(defs.currPath);			qf.setViewMode(QFileDialog::Detail);
	qf.selectFile(oldname);				qf.setFileMode(QFileDialog::AnyFile);
	qf.setAcceptMode(QFileDialog::AcceptSave);
	if(qf.exec() == QDialog::Accepted) {
		cb = rlp_strcpy(file_name, TMP_TXT_SIZE, (char*)qf.selectedFiles().first().toLatin1().data());
		if(cb < 5 || (file_name[cb-4] != '.' && file_name[cb-5] != '.')){
			char *ext = qf.selectedNameFilter().toLatin1().data();
			for(i = 0; ext[i] && ext[i] != '*'; i++);
			rlp_strcpy(file_name+cb, 5, ext+i+1);
			}
		i = rlp_strlen(file_name);
		g->Command(CMD_BUSY, 0L, 0L);
		if(0==strcasecmp(".svg", file_name+i-4)) {
			DoExportSvg(g, file_name, 0L);
			}
		else if(0==strcasecmp(".eps", file_name+i-4)) {
			DoExportEps(g, file_name, 0L);
			}
		else if(0==strcasecmp(".png", file_name+i-4)) {
			ExportBitmap(g, file_name, (char*)".png");
			}
		}
	g->Command(CMD_REDRAW, 0L, 0L);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Common alert boxes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char the_message[500];
void InfoBox(char *Msg)
{
	QMessageBox::information(QAppl->focusWidget(), "Info", Msg);
}

void ErrorBox(char *Msg)
{
	QMessageBox::critical(QAppl->focusWidget(), "ERROR", Msg);
}

bool YesNoBox(char *Msg)
{
	int cb;

	cb = rlp_strcpy(the_message, 450, Msg);
	rlp_strcpy(the_message+cb, 500-cb, (char*)"\n");
	cb = QMessageBox::information(QAppl->focusWidget(), "RLPlot", the_message,
		QMessageBox::Yes | QMessageBox::No);
	if(cb == QMessageBox::No) return false;
	return true;
}

int YesNoCancelBox(char *Msg)
{
	int res, cb;

	cb = rlp_strcpy(the_message, 450, Msg);
	rlp_strcpy(the_message+cb, 500-cb, (char*)"\n");
	res = QMessageBox::information(QAppl->focusWidget(), "RLPlot", the_message,
		QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
	switch(res) {
	case QMessageBox::Yes:		return 1;
	case QMessageBox::No:		return 0;
	default:			return 2;
		}
	return 0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a new font to be used in all graphs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
char *GetNewFont(char *, int family)
{
	QFont font;
	int i, cb;
	static char text[1024];
	bool ok = false;

	switch(family) {
	case FONT_HELVETICA:
		font = QFontDialog::getFont(&ok, qf_sans, MainWidget);
		if(ok) qf_sans = font;
		break;
	case FONT_TIMES:
		font = QFontDialog::getFont(&ok, qf_ser, MainWidget);
		if(ok) qf_ser = font;
		break;
	case FONT_COURIER:
		font = QFontDialog::getFont(&ok, qf_fix, MainWidget);
		if(ok) qf_fix = font;
		break;
	case FONT_GREEK:
		font = QFontDialog::getFont(&ok, qf_greek, MainWidget);
		if(ok) qf_greek = font;
		break;
	case FONT_SYS:
	default:
		font = QFontDialog::getFont(&ok, qf_sys, MainWidget);
		if(ok) qf_sys = font;
		break;
		}
	if(ok) {
		QString str = font.toString();
		QByteArray data = str.toLatin1();
		cb = data.size();
		if(cb >2) {
			for(i = 0; i < cb && (data.data())[i] != ','; i++) text[i] = (data.data())[i];
			text[i] = 0;
			return text;
			}
		}
	return 0L;
}

char *FontName(int fnt)
{
	QFont font;
	static char text[1024];
	int i, cb;
	
	switch (fnt) {
	case FONT_HELVETICA:	font = qf_sans;		break;
	case FONT_TIMES:	font = qf_ser;		break;
	case FONT_COURIER:	font = qf_fix;		break;
	case FONT_GREEK:	font = qf_greek;	break;
	case FONT_SYS:	default:
		font = qf_sys;	break;
		}
	QString str = font.toString();
	QByteArray data = str.toLatin1();
	cb = data.size();
	if(cb >2) {
		for(i = 0; i < cb && (data.data())[i] != ','; i++) text[i] = (data.data())[i];
		text[i] = 0;
		return text;
		}
	return(0L);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
int rlp_mbtowc(w_char *, unsigned char *, int)
{
//	printf("Stub: rlp_mbtowc()\n");
	return 0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Common code for all QT output classes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool com_QStringExt(QString txt, double *width, double *height, int cb, TextDEF *TxtSet, QPainter *qP)
{
	QRect rc;

	if(cb >0)txt.truncate(cb);
	rc = qP->boundingRect(0, 0, 10000, 1000, Qt::AlignLeft | Qt::AlignTop, txt);
	*width = rc.width();			*height = TxtSet->iSize +2;
	return true;
}

bool
com_GetTextExtent(unsigned char  *txt, double *width, double *height, int cb, TextDEF *TxtSet, QPainter *qP)
{
	QString ctxt((char*)txt);

	if(!txt || !txt[0]) return com_QStringExt(QString((char*)"a"), width, height, 1, TxtSet, qP);
	else com_QStringExt(ctxt, width, height, cb > 0 ? cb : rlp_strlen(txt), TxtSet, qP);
	return true;
}

bool
com_GetTextExtentW(w_char  *txt, double *width, double *height, int cb, TextDEF *TxtSet, QPainter *qP)
{
	int i;
	QString wtxt((char*)0L);

	if(!txt || !txt[0]) return com_QStringExt(QString("a"), width, height, 1, TxtSet, qP);
	for(i = 0; txt[i]; i++) wtxt.append(QChar(txt[i]));
	com_QStringExt(wtxt, width, height, cb, TxtSet, qP);
	return true;
}

bool com_QStringOut(int x, int y, QString txt, TextDEF *TxtSet, QPainter *qP, anyOutput *o)
{
	int ix, iy;
	double w, h;
	QBrush oldBrush;
	QPen oldPen, currPen;
	QPointF offset;

	if(!TxtSet->iSize && TxtSet->fSize > 0.0) TxtSet->iSize = o->un2ix(TxtSet->fSize);
	if(!TxtSet->iSize) return false;
	if(TxtSet->Align & TXA_VCENTER) y += iround(TxtSet->iSize * 0.4);
	else if(TxtSet->Align & TXA_VBOTTOM) y -= iround(TxtSet->iSize * 0.1);
	else y += iround(TxtSet->iSize);
	oldBrush = qP->brush();
	oldPen = currPen = qP->pen();				iy = y;
	com_QStringExt(txt, &w, &h, -1, TxtSet, qP);
	if(TxtSet->Align & TXA_HCENTER) ix = x - (w / 2.0);
	else if(TxtSet->Align & TXA_HRIGHT) ix = x - w;
	else ix = x;
	currPen.setColor(MK_QCOLOR(TxtSet->ColTxt));
	qP->setPen(currPen);
	if(fabs(TxtSet->RotBL) >.01 || fabs(TxtSet->RotCHAR) >.01) {
		offset.setX(x);		offset.setY(y);
		qP->translate(offset);
		qP->rotate(-TxtSet->RotBL);
		if(TxtSet->Mode == TXM_OPAQUE){
			currPen.setColor(MK_QCOLOR(TxtSet->ColBg));
			qP->setPen(currPen);
			qP->setBrush(MK_QCOLOR(TxtSet->ColBg));
			qP->drawRect(0, - iround(h*.8), w, h);
			currPen.setColor(MK_QCOLOR(TxtSet->ColTxt));
			qP->setPen(currPen);
			}
		if(TxtSet->Style & TXS_SUB) iy += ((TxtSet->iSize <<2)/10);
		else if(TxtSet->Style & TXS_SUPER) iy -= ((TxtSet->iSize <<2)/10);
		qP->drawText(ix-x, iy-y, txt);
		}
	else {
		if(TxtSet->Mode == TXM_OPAQUE){
			currPen.setColor(MK_QCOLOR(TxtSet->ColBg));
			qP->setPen(currPen);
			qP->setBrush(MK_QCOLOR(TxtSet->ColBg));
			qP->drawRect(ix, iy - iround(h*.8), w, h);
			currPen.setColor(MK_QCOLOR(TxtSet->ColTxt));
			qP->setPen(currPen);
			}
		if(TxtSet->Style & TXS_SUB) iy += o->un2iy(TxtSet->fSize*0.4);
		else if(TxtSet->Style & TXS_SUPER) iy -= o->un2iy(TxtSet->fSize*0.4);
		qP->drawText(ix, iy, txt);
		}
	oldPen.setCapStyle(Qt::RoundCap);
	oldPen.setJoinStyle(Qt::RoundJoin);
	qP->setPen(oldPen);
	qP->setBrush(oldBrush);
	qP->resetTransform();
	return true;
}

bool com_TextOut(int x, int y, unsigned char *ctxt, TextDEF *TxtSet, QPainter *qP, anyOutput *o)
{
	int i;
	QString txt((char*)ctxt);

	if(!ctxt || !ctxt[0] || !TxtSet || !qP || !o) return false;
	if(TxtSet->Font==FONT_GREEK) {
		txt.truncate(0);
		for(i = 0; ctxt[i]; i++) {
			if((ctxt[i] >= 'A' && ctxt[i] <= 'Z')) txt.append(QChar(ctxt[i] - 'A' + 0x391));
			else if((ctxt[i] >= 'a' && ctxt[i] <= 'z')) txt.append(QChar(ctxt[i] - 'a' + 0x3B1));
			else txt.append(QChar(ctxt[i]));
			}
		}
	return com_QStringOut(x, y, txt, TxtSet, qP, o);
}

bool com_TextOutW(int x, int y, w_char *wtxt, TextDEF *TxtSet, QPainter *qP, anyOutput *o)
{
	int i;
	QString txt((char*)0L);

	if(!wtxt || !wtxt[0] || !TxtSet || !qP || !o) return false;
	for(i = 0; wtxt[i]; i++) txt.append(QChar(wtxt[i]));
	return com_QStringOut(x, y, txt, TxtSet, qP, o);
}

bool com_SetTextSpec(TextDEF *set, TextDEF *TxtSet, anyOutput *o, QFont *qF, QPainter *qP)
{
	if(!set->iSize && (set->fSize > 0.0)) {
		if(o->OC_type == OC_PRINT) set->iSize = o->un2iy(set->fSize/12.0);
		else set->iSize = o->un2iy(set->fSize);
		}
	if(!set->iSize || !qF) return false;
	o->anyOutput::SetTextSpec(set);
	switch(TxtSet->Font){
	case FONT_HELVETICA:	*qF = qf_sans;		break;
	case FONT_TIMES:	*qF = qf_ser;		break;
	case FONT_COURIER:	*qF = qf_fix;		break;
	case FONT_GREEK:	*qF = qf_greek;		break;
	default:
	case FONT_SYS:		*qF = qf_sys;		break;
		}
	qF->setBold((TxtSet->Style & TXS_BOLD) ? true : false);
	qF->setItalic((TxtSet->Style & TXS_ITALIC) ? true : false);
	qF->setUnderline((TxtSet->Style &TXS_UNDERLINE) ? true : false);
	if((TxtSet->Style & TXS_SUPER) || (TxtSet->Style & TXS_SUB))
		qF->setPointSizeF((TxtSet->iSize > 2) ? (qreal)(TxtSet->iSize*0.71) : 1.0);
	else qF->setPointSizeF((TxtSet->iSize > 2) ? (qreal)TxtSet->iSize : 2.0);
	qP->setFont(*qF);
	if(TxtSet->fSize >0.0) TxtSet->iSize = o->un2iy(TxtSet->fSize);			//correct for printer
	return true;
}

bool
com_CircGrad(int cx, int cy, int r, POINT *pts, int cp, char *, anyOutput *o, QPainter *qPainter)
{
	int i, j;
	double v[3], vec[3], cf, cfx, cfy, pos;
	double positions[] = { 0.0, 0.2, 0.4, 0.6, 0.8, 1.0, 1.0};
	double tmp;
	RECT ibounds = { 0, 0, 0, 0 };
	qreal focX, focY;
	QPoint *a;
	QRect rect;
	QColor col;
	QGradientStops stops;

	ibounds.left = cx - r;				ibounds.right = cx + r;
	ibounds.top = cy - r;				ibounds.bottom = cy + r;
	v[0] = 0.0;		v[1] = 1.0;		v[2] = 0.0;
	for (i = 0; i < 3; i++) for (j = 0, vec[i] = 0.0; j < 3; j++)
		vec[i] += (o->light_vec[i][j] * v[j]);
	focX = (qreal)((double)cx + vec[0] * ((double)r));
	focY = (qreal)((double)cy - vec[2] * ((double)r));
	cf = ((double)focX - (double)cx) / (double)r;
	cf = cf * cf + (((double)focY - (double)cy) / (double)r) * (((double)focY - (double)cy) / (double)r);
	stops << QGradientStop(0.0, MK_QCOLOR(o->dFillCol2));
	for(i = 0; i < 6; i ++) {
		pos = positions[i] < 1.0 ? positions[i] + (1.0-positions[i]) * cf : 1.0;
		tmp = asin((1.0 -pos) *.95);
		col = MK_QCOLOR(IpolCol(o->dFillCol2, o->dFillCol, tmp < 1.0 ? tmp : 1.0));
		stops << QGradientStop(pos, col);
		}
	cfx = (double)(cx - iround(focX));		cfy = (double)(cy - iround(focY));
	cf = cfx > cfy ? cfx : cfy;							//keep circular
	cf *= 0.707;		focY += o->MenuHeight();
	rect.setX(ibounds.left - iround(cf));
	rect.setWidth(ibounds.right - ibounds.left + iround(cf*2.0));
	rect.setY(ibounds.top + o->MenuHeight() - iround(cf));
	rect.setHeight(ibounds.bottom - ibounds.top + iround(cf*2.0));
	QRadialGradient gradient(rect.x(), rect.y(), r<<1, focX, focY);
	gradient.setCenter(focX, focY);
	gradient.setStops(stops);
	QBrush brush(gradient);
	qPainter->setBrush(brush);
	if (pts) {							// outlined polygon
		if((a = (QPoint*)malloc(cp * sizeof(QPoint)))) {
			for(i = 0; i < cp; i++) {
				a[i].setX(pts[i].x);	a[i].setY(pts[i].y + o->MenuHeight());
				}
			qPainter->drawPolygon(a, cp);
			free(a);
			}
		}
	else {								// outlined circle
		qPainter->drawEllipse(ibounds.left, ibounds.top + o->MenuHeight(), 
			ibounds.right - ibounds.left, ibounds.bottom - ibounds.top);
		}
	return false;
}

bool
com_GradPG(fPOINT3D *pts, long npt, double, double, double pLo, double pHi, lfPOINT *grad, anyOutput *o, QPainter *qPainter)
{
	long i, lpts;
	double o_min, o_max, lc, hc;
	QPointF *points;
	FillDEF fill;
	QGradientStops stops;

	o_min = o->LoVal;					o_max = o->HiVal;
	lc = fmax((pLo - o_min) / (o_max - o_min), 0.0);
	hc = fmin((pHi - o_min) / (o_max - o_min), 1.0);
	points = (QPointF*)malloc((npt+2) * sizeof(QPointF));
	o->GetFill(&fill);
	for (i = 0, lpts = npt; i < npt; i++){
		points[i].setX(pts[i].fx);			points[i].setY(pts[i].fy + o->MenuHeight());
		}
	if (pts[i - 1].fx != pts[0].fx || pts[i - 1].fy != pts[0].fy) {		//close PG?
		points[i].setX(pts[0].fx);		points[i].setY(pts[0].fy + o->MenuHeight());
		lpts++;
		}
	if ((fill.type &0xf00) == FILL_GRADIENT){				//simple color to color gradient
		if (grad[0].fx != grad[1].fx || grad[0].fy != grad[1].fy){
			QLinearGradient gradient ((qreal)grad[0].fx, (qreal)grad[0].fy, (qreal)grad[1].fx, (qreal)grad[1].fy);
			stops << QGradientStop(0.0, MK_QCOLOR(IpolCol(o->dFillCol2, o->dFillCol, lc)));
			stops << QGradientStop(1.0, MK_QCOLOR(IpolCol(o->dFillCol2, o->dFillCol, hc)));
			gradient.setStops(stops);		QBrush brush(gradient);
			qPainter->setBrush(brush);		qPainter->drawPolygon(points, npt);
			}
		}
	else {									//rainbow colors
		if (grad[0].fx != grad[1].fx || grad[0].fy != grad[1].fy){
			QLinearGradient gradient ((qreal)grad[0].fx, (qreal)grad[0].fy, (qreal)grad[1].fx, (qreal)grad[1].fy);
			stops << QGradientStop(0.0, MK_QCOLOR(GradColor(2, o_min, o_max, pLo)));
			stops << QGradientStop(0.25, MK_QCOLOR(GradColor(2, o_min, o_max, pLo*0.75 + pHi*0.25)));
			stops << QGradientStop(0.5, MK_QCOLOR(GradColor(2, o_min, o_max, (pLo + pHi)*0.5)));
			stops << QGradientStop(0.75, MK_QCOLOR(GradColor(2, o_min, o_max, pLo*0.25 + pHi*0.75)));
			stops << QGradientStop(1.0, MK_QCOLOR(GradColor(2, o_min, o_max, pHi)));
			gradient.setStops(stops);		QBrush brush(gradient);
			qPainter->setBrush(brush);		qPainter->drawPolygon(points, npt);
			}
		}
	if ((o->dLineCol & 0xff000000) != 0xff000000) {
		for (i = 1; i < lpts; i++)qPainter->drawLine(points[i-1], points[i]);
		}
	free(points);
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Menu definitions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
struct men_def {
	char *name;
	int id;
	int action;
};

#define CM_SEPARATOR 10
#define CM_BREAK 11
#define CM_NOP 12

#pragma GCC diagnostic ignored "-Wwrite-strings"
static char *SCM_SEPARATOR = (char*)"separator";
static char *SCM_BREAK = (char*)"break";
//static char *SCM_CHARMAP = (char*)SCM_T_CHARMAP;
static men_def data_men_1[] = {{(char*)SMH_FILE, 0, 0},{(char*)SCM_NEWINST, CM_NEWINST, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_OPEN, CM_OPEN, 1}, {(char*)SCM_SAVE, CM_SAVE, 1},
	{(char*)SCM_SAVEAS, CM_SAVEAS, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {SCM_PRISETUP, CM_PRISETUP, 1},
	{(char*)SCM_PRINT, CM_PRINT, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_EXIT, CM_EXIT, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_FILE1, CM_FILE1, 1}, {(char*)SCM_FILE2, CM_FILE2, 1},
	{(char*)SCM_FILE3, CM_FILE3, 1}, {(char*)SCM_FILE4, CM_FILE4, 1}, {(char*)SCM_FILE5, CM_FILE5, 1},
	{(char*)SCM_FILE6, CM_FILE6, 1}, {(char*)SMH_EDIT, 0, 0}, {(char*)SCM_UNDO, CM_UNDO, 1}, 
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_ADDROWCOL, CM_ADDROWCOL, 1},
 	{(char*)SMH_INSERT, CM_NOP, 3}, {(char*)SCM_INSROW, CM_INSROW, 1}, {(char*)SCM_INSCOL, CM_INSCOL, 1},
	{(char*)SMH_DELETE, CM_NOP, 3}, {(char*)SCM_DELROW, CM_DELROW, 1}, {(char*)SCM_DELCOL, CM_DELCOL, 1},
	{SCM_BREAK, 0, CM_BREAK},{SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_COPY, CM_COPY, 1},
	{(char*)SCM_CUT, CM_CUT, 1}, {(char*)SCM_PASTE, CM_PASTE, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_FILLRANGE, CM_FILLRANGE, 1}, {(char*)SCM_ARITH, CM_ARITH, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_REDRAW, CM_REDRAW, 1}, {(char*)SMH_STATS, 0, 0}, {(char*)SCM_SMPLSTAT, CM_SMPLSTAT, 1},
	{(char*)SCM_REPCMEANS, CM_REPCMEANS, 1}, {(char*)SMH_ANOVA, CM_NOP, 3}, {(char*)SCM_REPANOV, CM_REPANOV, 1},
	{(char*)SCM_REPKRUSKAL, CM_REPKRUSKAL, 1}, {(char*)SCM_REPTWANOV, CM_REPTWANOV, 1},
	{(char*)SCM_REPFRIEDM, CM_REPFRIEDM, 1}, {(char*)SCM_REPTWANR, CM_REPTWANR, 1}, {(char*)SMH_REGRESS, CM_NOP, 3},
	{(char*)SCM_REPREGR, CM_REPREGR, 1},
//	{(char*)SCM_REPTWREGR, CM_REPTWREGR, 1},
	{(char*)SCM_ROBUSTLINE, CM_ROBUSTLINE, 1}, {(char*)SMH_CORREL, CM_NOP, 3}, {(char*)SCM_CORRELM, CM_CORRELM, 1},
	{(char*)SCM_CORRELT, CM_CORRELT, 1}, {SCM_BREAK, 0, CM_BREAK}, {(char*)SMH_BOOTSTRAP, CM_NOP, 3},
	{(char*)SCM_BT_MEAN, CM_BT_MEAN, 1}, {(char*)SCM_BT_CORREL, CM_BT_CORREL, 1},
	{(char*)SCM_BT_REGR, CM_BT_REGR, 1}, {SCM_BREAK, 0, CM_BREAK}, {(char*)SCM_REPTWOWAY, CM_REPTWOWAY, 1},
	{(char*)SMH_GRAPH, 0, 0}, {(char*)SCM_NEWGRAPH, CM_NEWGRAPH, 1}, {(char*)SCM_NEWPAGE, CM_NEWPAGE, 1},
	{(char*)SCM_DELGRAPH, CM_DELGRAPH, 1}, {(char*)SMH_ABOUT, 0, 0}, {(char*)SCM_ABOUT, CM_ABOUT, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_DEFAULTS, CM_DEFAULTS, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)0L, 0, 0}};

static men_def data_men_2[] = {{(char*)SMH_FILE, 0, 0}, {(char*)SCM_SAVEAS, CM_SAVEAS, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {SCM_PRISETUP, CM_PRISETUP, 1}, {(char*)SCM_PRINT, CM_PRINT, 1},
	{(char*)SCM_EXPORT, CM_EXPORT, 1}, {SCM_EXPORT, 0, CM_SEPARATOR}, {(char*)SCM_CLOSE, CM_EXIT, 1}, 
	{(char*)SMH_EDIT, 0, 0}, {(char*)SCM_UNDO, CM_UNDO, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_COPY, CM_COPYGRAPH, 1}, {(char*)SCM_PASTE, CM_PASTE, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_UPDATE, CM_UPDATE, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_DELOBJ, CM_DELOBJ, 1},
	{(char*)SMH_DISPLAY, 0, 0}, {(char*)SCM_REDRAW, CM_REDRAW, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SMH_ZOOM, CM_NOP, 3}, {(char*)SCM_ZOOMIN, CM_ZOOMIN, 1}, {(char*)SCM_ZOOMOUT, CM_ZOOMOUT, 1},
	{(char*)SCM_ZOOMFIT, CM_ZOOMFIT, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_ZOOM25, CM_ZOOM25, 1},
	{(char*)SCM_ZOOM50, CM_ZOOM50, 1}, {(char*)SCM_ZOOM100, CM_ZOOM100, 1}, {(char*)SCM_ZOOM200, CM_ZOOM200, 1},
	{(char*)SCM_ZOOM400, CM_ZOOM400, 1}, {SCM_BREAK, 0, CM_BREAK}, {(char*)SCM_LAYERS, CM_LAYERS, 1},
	{(char*)SMH_TOOLS, 0, 0}, {(char*)SCM_T_STANDARD, CM_T_STANDARD, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_T_DRAW, CM_T_DRAW, 1}, {(char*)SCM_T_POLYLINE, CM_T_POLYLINE, 1}, {(char*)SCM_T_POLYGON, CM_T_POLYGON, 1},
	{(char*)SCM_T_RECTANGLE, CM_T_RECTANGLE, 1},{(char*)SCM_T_ROUNDREC, CM_T_ROUNDREC, 1}, {(char*)SCM_T_ELLIPSE, CM_T_ELLIPSE, 1},
	{(char*)SCM_T_ARROW, CM_T_ARROW, 1}, {(char*)"&Text", CM_T_TEXT, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_T_CHARMAP, CM_T_CHARMAP, 1},
	{(char*)SMH_PLOTS, 0, 0}, {(char*)SCM_ADDPLOT, CM_ADDPLOT, 1}, {(char*)SCM_ADDAXIS, CM_ADDAXIS, 1},
	{(char*)SCM_LEGEND, CM_LEGEND, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_DEFAULTS, CM_DEFAULTS, 1},
	{(char*)SMH_ABOUT, 0, 0}, {(char*)SCM_ABOUT, CM_ABOUT, 1},
	{(char*)0L, 0, 0}};

static men_def data_men_3[] = {{(char*)SMH_FILE, 0, 0}, {(char*)SCM_SAVEAS, CM_SAVEAS, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {SCM_PRISETUP, CM_PRISETUP, 1}, {(char*)SCM_PRINT, CM_PRINT, 1},
	{(char*)SCM_EXPORT, CM_EXPORT, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_CLOSE, CM_EXIT, 1}, 
	{(char*)SMH_EDIT, 0, 0}, {(char*)SCM_UNDO, CM_UNDO, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_COPY, CM_COPYGRAPH, 1}, {(char*)SCM_PASTE, CM_PASTE, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_UPDATE, CM_UPDATE, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_DELOBJ, CM_DELOBJ, 1},
	{(char*)SMH_DISPLAY, 0, 0}, {(char*)SCM_REDRAW, CM_REDRAW, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SMH_ZOOM, CM_NOP, 3}, {(char*)SCM_ZOOMIN, CM_ZOOMIN, 1}, {(char*)SCM_ZOOMOUT, CM_ZOOMOUT, 1},
	{(char*)SCM_ZOOMFIT, CM_ZOOMFIT, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_ZOOM25, CM_ZOOM25, 1},
	{(char*)SCM_ZOOM50, CM_ZOOM50, 1}, {(char*)SCM_ZOOM100, CM_ZOOM100, 1}, {(char*)SCM_ZOOM200, CM_ZOOM200, 1},
	{(char*)SCM_ZOOM400, CM_ZOOM400, 1}, {SCM_BREAK, 0, CM_BREAK}, {(char*)SCM_LAYERS, CM_LAYERS, 1},
	{(char*)SMH_TOOLS, 0, 0}, {(char*)SCM_T_STANDARD, CM_T_STANDARD, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_T_DRAW, CM_T_DRAW, 1}, {(char*)SCM_T_POLYLINE, CM_T_POLYLINE, 1}, {(char*)SCM_T_POLYGON, CM_T_POLYGON, 1},
	{(char*)SCM_T_RECTANGLE, CM_T_RECTANGLE, 1}, {(char*)SCM_T_ROUNDREC, CM_T_ROUNDREC, 1}, {(char*)SCM_T_ELLIPSE, CM_T_ELLIPSE, 1},
	{(char*)SCM_T_ARROW, CM_T_ARROW, 1}, {(char*)SCM_T_TEXT, CM_T_TEXT, 1},
	{SCM_SEPARATOR, 0, CM_SEPARATOR}, {(char*)SCM_T_CHARMAP, CM_T_CHARMAP, 1},
	{(char*)SMH_PLOTS, 0, 0}, {(char*)SCM_NEWGRAPHP, CM_NEWGRAPH, 1}, {(char*)SCM_ADDPLOT, CM_ADDPLOT, 1},
	{(char*)SCM_ADDAXIS, CM_ADDAXIS, 1}, {(char*)SCM_LEGEND, CM_LEGEND, 1}, {SCM_SEPARATOR, 0, CM_SEPARATOR},
	{(char*)SCM_DEFAULTSP, CM_DEFAULTS, 1}, {(char*)SMH_ABOUT, 0, 0}, {(char*)SCM_ABOUT, CM_ABOUT, 1},
	{(char*)0L, 0, 0}};
#pragma GCC diagnostic warning "-Wwrite-strings"

static void add_menu(QWidget *parent, RLPmenu *menBar, men_def *data_men)
{
	int i, j, n, ns, m;
	QMenu *submenu = 0L;
	char *submenu_name = 0L, *currmenu_name = 0;
	QMenu *curr_menu[20];
	MenuAction **item_actions = 0L, **submenu_actions = 0L;
	
	for (i = m = n = ns = 0; data_men[i].name && data_men[i].name[0]; i++) {
		switch(data_men[i].action) {
		case 0:
			if(!currmenu_name) currmenu_name = strdup(data_men[i].name);
			if(item_actions) {
				for(j = 0; j< n; j++) {
					curr_menu[m-1]->addAction(item_actions[j]);
					}
				free(item_actions);	item_actions = 0L;	n = 0;
				}
			if(currmenu_name) {
				curr_menu[m] = menBar->addMenu(currmenu_name);
				free(currmenu_name);	currmenu_name = 0L;
				m++;
				}
			break;
		case 3:
			if(submenu_actions && submenu_name && submenu) {
				for(j = 0; j< ns; j++) {
					submenu->addAction(submenu_actions[j]);
					}
				free(submenu_name);	submenu_name = 0L;
				free(submenu_actions);	submenu_actions = 0L;
				ns = 0;
				}
			submenu_name = strdup(data_men[i].name);
			submenu_actions = (MenuAction**)calloc(30, sizeof(MenuAction*));
			if(item_actions) {
				for(j = 0; j< n; j++) {
					curr_menu[m-1]->addAction(item_actions[j]);
					}
				free(item_actions);	item_actions = 0L;	n = 0;
				}
			submenu = curr_menu[m-1]->addMenu(submenu_name);
			ns = 0;
			break;
		case 1:
			if(!item_actions) item_actions = (MenuAction**)calloc(30, sizeof(MenuAction*));
			if(submenu_actions && submenu_name) {
				submenu_actions[ns++] = new MenuAction(data_men[i].name, parent, data_men[i].id);
				}
			else {
				if(data_men[i].id == CM_T_CHARMAP) {
					if(char_map) item_actions[n++] = new MenuAction(data_men[i].name, parent, data_men[i].id);
					}
				else item_actions[n++] = new MenuAction(data_men[i].name, parent, data_men[i].id);
				}
			switch (data_men[i].id) {
			case CM_COPY:	case CM_COPYGRAPH:
				item_actions[n-1]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_C));
				break;
			case CM_PASTE:
				item_actions[n-1]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_V));
				break;
			case CM_UNDO:
				item_actions[n-1]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_Z));
				break;
			case CM_SAVE:
				item_actions[n-1]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_S));
				break;
			case CM_OPEN:
				item_actions[n-1]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_O));
				break;
			case CM_T_STANDARD: 	case CM_T_DRAW: 	case CM_T_POLYLINE:
			case CM_T_POLYGON:	case CM_T_RECTANGLE:	case CM_T_ELLIPSE:
			case CM_T_ROUNDREC:	case CM_T_ARROW:	case CM_T_TEXT:
				item_actions[n-1]->setCheckable(true);
				if (data_men[i].id == CM_T_STANDARD) item_actions[n-1]->setChecked(true);
				((DrawWidget*)parent)->Command(CMD_REG_MITEM, item_actions[n-1]);
				break;
			case CM_T_CHARMAP:
				break;
			case CM_FILE1:		case CM_FILE2:		case CM_FILE3:
			case CM_FILE4:		case CM_FILE5:		case CM_FILE6:
				item_actions[n-1]->setVisible(false);
				((DrawWidget*)parent)->Command(CMD_REG_MITEM, item_actions[n-1]);
				break;	
				}

			break;
		case CM_BREAK:
			if(submenu_actions && submenu_name && submenu) {
				for(j = 0; j< ns; j++) {
					submenu->addAction(submenu_actions[j]);
					}
				free(submenu_name);	submenu_name = 0L;
				free(submenu_actions);	submenu_actions = 0L;
				ns = 0;
				}
			break;
		case CM_SEPARATOR:
			if(item_actions) {
				for(j = 0; j< n; j++) {
					curr_menu[m-1]->addAction(item_actions[j]);
					}
				free(item_actions);	item_actions = 0L;	n = 0;
				}
			curr_menu[m-1]->addSeparator();
			break;
			}
		}
}
#undef CM_SEPARATOR
#undef CM_BREAK
#undef CM_NOP

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Icon definitions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

//this icon has been taken from trolltech's Qt: qmessagebox.cpp
const static char *information_xpm[]={
"32 32 5 1",		". c None",		"c c #000000",
"* c #999999",		"a c #ffffff",		"b c #0000ff",
"...........********.............",	"........***aaaaaaaa***..........",
"......**aaaaaaaaaaaaaa**........",	".....*aaaaaaaaaaaaaaaaaa*.......",
"....*aaaaaaaabbbbaaaaaaaac......",	"...*aaaaaaaabbbbbbaaaaaaaac.....",
"..*aaaaaaaaabbbbbbaaaaaaaaac....",	".*aaaaaaaaaaabbbbaaaaaaaaaaac...",
".*aaaaaaaaaaaaaaaaaaaaaaaaaac*..",	"*aaaaaaaaaaaaaaaaaaaaaaaaaaaac*.",
"*aaaaaaaaaabbbbbbbaaaaaaaaaaac*.",	"*aaaaaaaaaaaabbbbbaaaaaaaaaaac**",
"*aaaaaaaaaaaabbbbbaaaaaaaaaaac**",	"*aaaaaaaaaaaabbbbbaaaaaaaaaaac**",
"*aaaaaaaaaaaabbbbbaaaaaaaaaaac**",	"*aaaaaaaaaaaabbbbbaaaaaaaaaaac**",
".*aaaaaaaaaaabbbbbaaaaaaaaaac***",	".*aaaaaaaaaaabbbbbaaaaaaaaaac***",
"..*aaaaaaaaaabbbbbaaaaaaaaac***.",	"...caaaaaaabbbbbbbbbaaaaaac****.",
"....caaaaaaaaaaaaaaaaaaaac****..",	".....caaaaaaaaaaaaaaaaaac****...",
"......ccaaaaaaaaaaaaaacc****....",	".......*cccaaaaaaaaccc*****.....",
"........***cccaaaac*******......",	"..........****caaac*****........",
".............*caaac**...........",	"...............caac**...........",
"................cac**...........",	".................cc**...........",
"..................***...........",	"...................**..........."};

//thanks to Markus Bongard for the following icon
const static char *RLPlot_xpm[]={
/* width height ncolors chars_per_pixel */
"32 32 169 2",
/* colors */
"AA c #FFFFFFFFFFFF",	"BA c #FFFFF7F79494",	"CA c #FFFFF7F78484",
"DA c #FFFFF7F77373",	"EA c #FFFFF7F75252",	"FA c #FFFFF7F74242",
"GA c #FFFFF7F73939",	"HA c #FFFFEFEF8C8C",	"IA c #FFFFEFEF4A4A",
"JA c #FFFFEFEF2929",	"KA c #F7F7E7E77B7B",	"LA c #F7F7C6C6ADAD",
"MA c #F7F7B5B59C9C",	"NA c #F7F7ADAD9494",	"OA c #EFEFF7F7F7F7",
"PA c #EFEFF7F7EFEF",	"AB c #EFEFEFEFEFEF",	"BB c #EFEFEFEFDEDE",
"CB c #EFEFE7E7A5A5",	"DB c #EFEFDEDE7373",	"EB c #EFEFDEDE3939",
"FB c #EFEFDEDE2929",	"GB c #EFEFD6D64242",	"HB c #EFEFA5A58C8C",
"IB c #EFEF94947B7B",	"JB c #EFEF84847373",	"KB c #EFEF84846363",
"LB c #EFEF7B7B7373",	"MB c #E7E7E7E7CECE",	"NB c #E7E7E7E79494",
"OB c #E7E7DEDE6B6B",	"PB c #E7E7DEDE5252",	"AC c #E7E7CECE5252",
"BC c #E7E77B7B6363",	"CC c #E7E773735A5A",	"DC c #E7E76B6B5252",
"EC c #E7E75A5A4A4A",	"FC c #DEDEE7E7F7F7",	"GC c #DEDEE7E7EFEF",
"HC c #DEDEDEDEDEDE",	"IC c #DEDEDEDEBDBD",	"JC c #DEDECECE4A4A",
"KC c #DEDE4A4A3939",	"LC c #D6D6EFEFCECE",	"MC c #D6D6DEDEEFEF",
"NC c #D6D6DEDECECE",	"OC c #CECEDEDEDEDE",	"PC c #CECED6D6CECE",
"AD c #CECECECEB5B5",	"BD c #CECECECE5A5A",	"CD c #C6C6E7E7C6C6",
"DD c #C6C6DEDEEFEF",	"ED c #C6C6D6D67373",	"FD c #C6C6CECE4A4A",
"GD c #BDBDDEDEB5B5",	"HD c #BDBDCECEE7E7",	"ID c #BDBDCECECECE",
"JD c #BDBDC6C6D6D6",	"KD c #BDBD39394242",	"LD c #B5B5DEDEADAD",
"MD c #B5B5BDBDCECE",	"ND c #B5B5BDBD4242",	"OD c #B5B5B5B59C9C",
"PD c #ADADD6D6ADAD",	"AE c #ADADCECEDEDE",	"BE c #ADADBDBDB5B5",
"CE c #ADADB5B54242",	"DE c #ADAD4A4A5252",	"EE c #A5A5BDBDDEDE",
"FE c #A5A5BDBDCECE",	"GE c #A5A5ADADB5B5",	"HE c #A5A5ADAD4242",
"IE c #A5A563637B7B",	"JE c #9C9CD6D6A5A5",	"KE c #9C9CBDBDD6D6",
"LE c #9C9CBDBDBDBD",	"ME c #9C9CADADCECE",	"NE c #9C9CADADA5A5",
"OE c #9C9C52525252",	"PE c #9C9C4A4A5252",	"AF c #9494CECE9C9C",
"BF c #9494B5B5CECE",	"CF c #9494ADAD5A5A",	"DF c #9494ADAD3939",
"EF c #8C8CCECE9494",	"FF c #8C8CCECE8C8C",	"GF c #8C8CB5B5B5B5",
"HF c #8C8CADADD6D6",	"IF c #8C8CADADCECE",	"JF c #8C8CADADB5B5",
"KF c #8C8CADADA5A5",	"LF c #8C8C9C9CB5B5",	"MF c #8C8C9C9CADAD",
"NF c #8C8C8C8CA5A5",	"OF c #8C8C52526363",	"PF c #8C8C42425252",
"AG c #8484B5B5CECE",	"BG c #7B7BCECE9494",	"CG c #7B7BC6C68484",
"DG c #7B7BC6C67B7B",	"EG c #7B7BADADC6C6",	"FG c #7B7BADADADAD",
"GG c #7B7B9C9CC6C6",	"HG c #7B7B9C9CB5B5",	"IG c #7B7B9C9CADAD",
"JG c #7B7B84849C9C",	"KG c #7B7B42425252",	"LG c #737384848C8C",
"MG c #737373738484",	"NG c #737339395252",	"OG c #6B6BC6C68484",
"PG c #6B6BBDBD7B7B",	"AH c #6B6BA5A5CECE",	"BH c #6B6B9C9CBDBD",
"CH c #6B6B9C9CA5A5",	"DH c #6B6B94949C9C",	"EH c #6B6B8C8CA5A5",
"FH c #6B6B8C8C9C9C",	"GH c #6B6B6B6B7B7B",	"HH c #6B6B42425252",
"IH c #6363BDBD6B6B",	"JH c #6363B5B58C8C",	"KH c #6363A5A58C8C",
"LH c #63639C9CC6C6",	"MH c #5A5AB5B56363",	"NH c #5A5A9C9CBDBD",
"OH c #5A5A8C8C9C9C",	"PH c #5A5A7B7B8C8C",	"AI c #5A5A6B6B8484",
"BI c #5A5A63637B7B",	"CI c #5252BDBD7373",	"DI c #5252ADAD8484",
"EI c #5252A5A57B7B",	"FI c #4A4AB5B55A5A",	"GI c #4A4A8C8CBDBD",
"HI c #4A4A8C8CADAD",	"II c #4A4A7B7B9C9C",	"JI c #4A4A7B7B8484",
"KI c #4A4A6B6B7B7B",	"LI c #42429C9C7373",	"MI c #42428C8C7B7B",
"NI c #42425A5A7373",	"OI c #42424A4A6B6B",	"PI c #3939ADAD6B6B",
"AJ c #3939ADAD5A5A",	"BJ c #39399C9C6B6B",	"CJ c #39398484BDBD",
"DJ c #39397B7BADAD",	"EJ c #39397B7B8C8C",	"FJ c #393973737B7B",
"GJ c #313163637B7B",	"HJ c #31315A5A7B7B",	"IJ c #292984846B6B",
"JJ c #29296B6B7B7B",	"KJ c #21217B7BB5B5",	"LJ c #21217373A5A5",
"MJ c #212173739C9C",	"NJ c #21216B6B8C8C",	"OJ c #212152527373",
"PJ c #181884846363",	"AK c #18186B6B7B7B",	"BK c #18185A5A7373",
"CK c #10106B6B9C9C",	"DK c #10105A5A8484",	"EK c #08087B7B6363",
"FK c #08085A5A7B7B",	"GK c #00006B6B9C9C",	"HK c #000063637B7B",
"IK c #00005A5A8C8C",
/* pixels */
"CJAHAHAHAHHILHLHBHLHLJNHNHNHGINHGINHGINHGINHGINHGINHGINHGINHGIKJ",
"AHAAAAAAAGHDABBBGCBFBFHCPCOCEEHDJDEEJDEEHDJDEEHDJDEEJDEEHDEEJDNH",
"AHAAAGAAABABABKEOCHCHCPCFEFEABAEHDFCDDJDDDMCJDHDFCHDHDGCDDJDDDLH",
"EGAEDDABOAABOCEGHCHCPCHCBHJDJDEEJDEEJDEEJDAEJDEEAEJDEEJDEEJDEELH",
"AHAAAAAAMCJDGCABHCHCBFOCPCJDEEJDEEJDEEAEEEEEAEJDEEFEFEFEEEEEJDGI",
"AHAAAAPAAGGCABBBMCEGJDNCPCPCEEEEEEEEEEEEEEJDEEEEFEDIPIGFFEFEEENH",
"EGOAKEAAABPAABKEHCHCHCHCMEBFABJDAEMCOCEEMCMCFEOCDDLEKHIDIDEEDDLH",
"LHAEMCAAABABAEIFHCHCHCPCBHOCKEKEEEFEEEFEEEFEEEKEBFGEDIKFMEFEEEGI",
"AHAAAAABGCJDABGCBBOCHGHCPCMDEEFEKEEEFEKEFEKEEEMEBFJFDIGEIFBFFEGI",
"AHAAAAPAAHGCABBBMCJFMDOCPCJDMEKEKEMEKEKEKEKEMEIFGEFGEIIGJFGEMEGI",
"AHAAAEAAABABGCNBDBOCNCNCMDEEMCEEEEMCJDKEDDDDLFCIDIEIPICIDIDIJDNH",
"AHKEDDAAPAABOCKFPBICNCPCHGFEMEMEMEMEHFMEMEIFJFJELCCDCGBBCDLIIGHI",
"EGAAAAABABJDMBNCPBADGENCMDMDMEHFIFHFMEHFIFMECHJEPCLDBGGDGDLIIGDJ",
"AHAAAAAAAHEDKAOBEBBDCEODICMDIFIFMEIFIFIFIFIFCHEFLDPDOGGDPDMIFHHI",
"AHAAOCABMBEABABAFABABAGBNELEHDKEHFDDEEIFJDEEFGCGLDPDPGPDPDAJMFHI",
"AHDDEGABBBEABACAFABACBGBFHGEIFGGIFGGIFIFGGIFDHCGJEMHFIAFAFBJDHII",
"AHAAAAABMBEAHAEAEADACABDODBEHGIFGGIFHGGGIFHGCHJHAFAFEFAFFFFJDHMJ",
"AHAAAAAABEIADADAKACADAGBODNEGGHGIEIENFGGHGGGOHDGFFFFFFFFFFAJOHHI",
"AHAAABABPCGAKADADAKADAFDODMFHIHIAIPFIIIIHIHIEJCIFFDGFFDGDGJJFJCK",
"AHAAAHHCMBGAEAEAEAFAFAJCLGFKPEPEPEKDPEPEOIIKHKLIDGDGPGDGIHEKIKGK",
"EGFCDDAANCJAIAIAIAIAIANDBKOJJBLALALBLAMANGFKHKFIMHPGMHFILIEKFKGK",
"LHAAAABBNCFBJAJAJAEBGBDFBKOJBCNADCDCIBHBNGBKAKFIMHMHMHMHMHEKHKGK",
"AHAAAAABADEBJCEBGBACACHEBKNIBCJBJBHBJBJBHHOJJJPIIHPGIHIHIHPJDKGK",
"AHAAOCJDNCEBOBACOBOBOBHEEJNIKCLBKBBCKBCCKGHJJJMHDGIHPGDGIHEKEJCK",
"AHAAKEABMBEBDBDBOBOBOBHENIAIKCDCDCDCDCECKGKIGJCIDGFFLGDGDGPJNJMJ",
"CJAAOAABBBIADBHADBHAKAHEAIBIDCDCBCCCDCCCPFKIJIPGBGCFDGFFDGIJIIMJ",
"AHAAAAABMBIABADBBAHAHAHEAIGHKCJBBCKBKBBCKGPHJIPGFFGFFFFFFFIJJILJ",
"AHAAAAABMDFDFDBDFDNDBDHEFHJGDEDEDEDEOEDEOFPHMGBJBJAJBJBJBJMIOHDJ",
"EGAAAAJDHGNFFHMGFHJGMGEHJGNFNFMGMGMGMGMGLGJGCHEHFHPHMGOHPHFHIGDJ",
"LHAAOCIFIFIFLFLFMFMFJFLFIFJFHGIGMFIGNFMFMFLFIFLFHGIGFGIGIGHGJFCJ",
"EGDDKEKEMEBFBFBFBFMEBFBFFEKEMEMEMEGEBFMEBFBFBFKEMEBFMEBFBFKEKEGI",
"KJLHGINHGINHNHNHNHGINHNHGINHGINHNHGIGINHGINHNHGINHNHGIGINHNHGICJ"};


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bitmap class for display export etc.
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BitMapQT::BitMapQT(GraphObj *, RLPwidget *wi, int vr, int hr): anyOutput()
{
	int w, h;

	hres = (double)hr;		vres = (double)vr;
	image = 0L;			hgo = 0L;
	dlgwidget = 0L;			widget = wi;
	ShowObj = ShowAnimated = 0L;	minLW = 1;
	OC_type = OC_BITMAP;
	if(wi) {
		w = wi->width();		h = wi->height();
		}
	else {
		GetDesktopSize(&w, &h);
		}
	Box1.Xmin = Box1.Ymin = 0.0;
	Box1.Xmax = w;					Box1.Ymax = h;
	DeskRect.left = DeskRect.top = 0;
	GetDesktopSize(&w, &h);
	DeskRect.right = w;				DeskRect.bottom = h;
	mempic = new QPixmap(w, h);
	mempic->fill(0x00ffcbffL);
	qPainter.begin(mempic);
	qPen.setCapStyle(Qt::RoundCap);
	qPen.setJoinStyle(Qt::RoundJoin);
	qPainter.setPen(qPen);
	qFont = qPainter.font();
	Notary->ValPtr(this, true);
	
}

BitMapQT::BitMapQT(GraphObj *, QWidget *wi, int vr, int hr):anyOutput()
{
	int w, h;

	hres = (double)hr;		vres = (double)vr;
	image = 0L;			hgo = 0L;			
	dlgwidget = wi;			widget = 0L;
	ShowObj = ShowAnimated = 0L;	minLW = 1;
	OC_type = OC_BITMAP;
	if(wi) {
		w = wi->width();		h = wi->height();
		}
	else {
		GetDesktopSize(&w, &h);
		}
	Box1.Xmin = Box1.Ymin = 0.0;
	Box1.Xmax = w;					Box1.Ymax = h;
	DeskRect.left = DeskRect.top = 0;
	GetDesktopSize(&w, &h);
	DeskRect.right = w;				DeskRect.bottom = h;
	mempic = new QPixmap(w, h);
	mempic->fill(0x00ffffcbL);
	qPainter.begin(mempic);
	qPen.setCapStyle(Qt::RoundCap);
	qPen.setJoinStyle(Qt::RoundJoin);
	qPainter.setPen(qPen);
	qFont = qPainter.font();
	Notary->ValPtr(this, true);
}

BitMapQT::BitMapQT(int w, int h, double hr, double vr):anyOutput()
{
	hres = hr;		vres = vr;
	image = 0L;		hgo = 0L;			widget = 0L;
	w = abs(w);		h = abs(h);			ShowObj = ShowAnimated = 0L;
	Box1.Xmin = Box1.Ymin = 0.0;			minLW = 1;
	Box1.Xmax = w;					Box1.Ymax = h;
	DeskRect.right = w;				DeskRect.bottom = h;
	DeskRect.left = DeskRect.top = 0;		OC_type = OC_BITMAP;
	mempic = new QPixmap(w, h);
	mempic->fill(0x00cbffffL);
	qPainter.begin(mempic);
	qPen.setCapStyle(Qt::RoundCap);
	qPen.setJoinStyle(Qt::RoundJoin);
	qPainter.setPen(qPen);
	qFont = qPainter.font();
	Notary->ValPtr(this, true);
}

BitMapQT::~BitMapQT()
{
	Notary->ValPtr(this, false);
	Undo.KillDisp(this);
	if(ShowObj) delete((eph_obj*)ShowObj);
	ShowObj = 0L;
	if(qPainter.isActive()) qPainter.end();
	HideTextCursor();
	if(mempic) delete mempic;
	if(hgo) delete hgo;
	if(image) delete image;
	defs.Idle(CMD_FLUSH);
	mempic = 0L;	hgo = 0L;	image = 0L;
}

bool
BitMapQT::SetLine(LineDEF *lDef)
{
	int iw;
	double fiw;

	if(ShowObj) {
//		RestoreMark();
		}
	if(lDef->width != LineWidth || lDef->width != LineWidth ||
		lDef->pattern != dPattern || lDef->color != dLineCol) {
		LineWidth = lDef->width;
		fiw = un2fix(lDef->width);
		iw = iround(fiw);
		dPattern = lDef->pattern;
		RLP.finc = 256.0/un2fix(lDef->patlength*8.0);
		RLP.fp = 0.0;
		iLine = iw > minLW ? iw : minLW;
		if(fiw < minLW) fiw = minLW;
		dLineCol = lDef->color;
		qPen.setColor(MK_QCOLOR(dLineCol));
		qPen.setWidthF(fiw);
		qPen.setStyle(Qt::SolidLine);
		qPen.setCapStyle(Qt::RoundCap);
		qPen.setJoinStyle(Qt::RoundJoin);
		qPainter.setPen(qPen);
		}
	return true;
}

bool
BitMapQT::ClipRect(RECT *rec)
{
	int x, y, w, h;

	if (!rec){								//reset clipping
		hasClip = false;
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		qPainter.setClipping(false);
		}
	else if(rec->left == rec->right || rec->bottom == rec->top) {
		hasClip = false;
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		qPainter.setClipping(false);
		}
	else if(rec->left < rec->right && rec->bottom > rec->top){			//set up clipping rectangle
		memcpy(&ClipRC, rec, sizeof(RECT));
		hasClip = true;
		ClipRC.top += MenuHeight();	ClipRC.bottom += MenuHeight();
		x = ClipRC.left;		w = ClipRC.right - ClipRC.left;
		y = ClipRC.top;			h = ClipRC.bottom - ClipRC.top; 
		qPainter.setClipRect(x, y, w, h);
		qPainter.setClipping(true);
		}
	else return ClipRect(NULL);
	return true;
}

bool
BitMapQT::SetFill(FillDEF *fill)
{
	if(ShowObj) RestoreMark();
	if(!fill) return false;
	anyOutput::SetFill(fill);
	if((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		}
	else {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	qPainter.setBrush(MK_QCOLOR(fill->color));
	dFillCol = fill->color;			dFillCol2 = fill->color2;
	return true;
}

bool
BitMapQT::SetTextSpec(TextDEF *set)
{
	if(ShowObj) RestoreMark();
	return com_SetTextSpec(set, &TxtSet, this, &qFont, &qPainter);
}

bool
BitMapQT::Erase(DWORD color)
{
	if(!mempic) return false;
	mempic->fill(color);
	if(image) delete image;
	image = 0L;
	return true;
}

bool
BitMapQT::CopyBitmap(int x, int y, anyOutput* sr, int sx, int sy,
	int sw, int sh, bool invert)
{
	BitMapQT *src = (BitMapQT*)sr;

	if(!mempic) return false;
	if(invert) {
		QImage qi = src->mempic->copy(sx, sy, sw, sh).toImage();
		qi.invertPixels();
		qPainter.drawImage(x, y, qi);
		}
	else qPainter.drawPixmap(x, y, *src->mempic, sx, sy, sw, sh);
	return true;
}

bool
BitMapQT::oGetTextExtent(unsigned char *text, int cb, double *width, double *height)
{
	return com_GetTextExtent(text, width, height, cb, &TxtSet, &qPainter);
}

bool
BitMapQT::oGetTextExtentW(w_char *text, int cb, double *width, double *height)
{
	return com_GetTextExtentW(text, width, height, cb, &TxtSet, &qPainter);
}

bool
BitMapQT::oGetPix(int x, int y, DWORD *col)
{
	DWORD pix;

	if(!image && !(image = new QImage(mempic->toImage())))return false;
	if(x >= DeskRect.left && x < DeskRect.right && y >= DeskRect.top && y < DeskRect.bottom){
		pix = image->pixel(x,y);		*col = pix & 0x0000ff00L;
		*col |= (pix>>16)&0x000000ffL;		*col |= (pix<<16)&0x00ff0000L;
		return true;
		}
	return false;
}

bool
BitMapQT::oDrawIcon(int type, int x, int y)
{
	const char** xpm_data = 0L;

	switch (type) {
	case ICO_INFO:
		xpm_data = information_xpm;
		break;
	case ICO_ERROR:
//		xpm_data = critical_xpm;
		break;
	case ICO_RLPLOT:
		xpm_data = RLPlot_xpm;
		break;
	case ICO_QT:
//		xpm_data = qtlogo_xpm;
		break;
	default:
		return false;
		}
	if (xpm_data) {
		QPixmap pm(xpm_data);
		qPainter.drawPixmap(x, y + MenuHeight(), pm);
		return true;
		}
	return false;
}

bool
BitMapQT::oCircle(int x1, int y1, int x2, int y2, char*)
{
	qPainter.drawEllipse(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	if(hgo) return hgo->oCircle(x1, y1, x2, y2);
	return true;
}

bool 
BitMapQT::foCircle(double x1, double y1, double x2, double y2, char*)
{
	QRectF rect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	qPainter.drawEllipse(rect);
	if(hgo) return hgo->foCircle(x1, y1, x2, y2);
	return true;
}

bool
BitMapQT::oSphere(int cx, int cy, int r, POINT *pts, int cp, char *nam)
{
	if (dFillCol == dFillCol2) {
		if (pts) return oPolygon(pts, cp);
		else return oCircle(cx - r, cy - r, cx + r, cy + r);
		}
	return com_CircGrad(cx, cy, r, pts, cp, nam, this, &qPainter);
}

bool
BitMapQT::oPolyline(POINT * pts, int cp)
{
	int i;

	if(cp < 1) return false;
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		}
	else {
		for (i = 1; i < cp; i++)qPainter.drawLine(pts[i-1].x, pts[i-1].y + MenuHeight(), pts[i].x, pts[i].y + MenuHeight());
		}
	return true;
}

bool
BitMapQT::foPolyline(lfPOINT * fpts, int cp)
{
	int i;
	POINT *pts;

	if(cp < 1) return false;
	pts = (POINT*)calloc(cp + 2, sizeof(POINT));
	for (i = 0; i < cp; i++) {
		pts[i].x = fpts[i].fx;		pts[i].y = fpts[i].fy;
		}
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		}
	else {
		for (i = 1; i < cp; i++)qPainter.drawLine(pts[i-1].x, pts[i-1].y + MenuHeight(), pts[i].x, pts[i].y + MenuHeight());
		}
	free(pts);
	return true;
}

bool
BitMapQT::oRectangle(int x1, int y1, int x2, int y2, char *)
{
	qPainter.drawRect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	if(hgo) hgo->oRectangle(x1, y1, x2, y2, 0L);
	return true;
}

bool
BitMapQT::foRectangle(double x1, double y1, double x2, double y2, char *)
{
	QRectF rect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	qPainter.drawRect(rect);
	if(hgo) hgo->oRectangle(x1, y1, x2, y2, 0L);
	return true;
}

bool 
BitMapQT::oSolidRectangle(int x1, int y1, int x2, int y2)
{
	qPainter.fillRect(x1, y1 + MenuHeight(), x2-x1, y2-y1, MK_QCOLOR(dLineCol));
	return true;
}

bool 
BitMapQT::foSolidRectangle(double x1, double y1, double x2, double y2)
{
	QRectF rect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	qPainter.fillRect(rect, MK_QCOLOR(dLineCol));
	return true;
}

bool
BitMapQT::oSolidLine(POINT *p)
{
	qPainter.drawLine(p[0].x, p[0].y + MenuHeight(), p[1].x, p[1].y + MenuHeight());
	return true;
}

bool
BitMapQT::foSolidLine(lfPOINT *p)
{
	QPointF pt1, pt2;

	pt1.setX(p[0].fx);			pt1.setY(p[0].fy + MenuHeight());
	pt2.setX(p[1].fx);			pt2.setY(p[1].fy + MenuHeight());
	qPainter.drawLine(pt1, pt2);
	return true;
}

bool
BitMapQT::oTextOut(int x, int y, unsigned char *txt, int)
{
	if(!txt || !txt[0]) return false;
	return com_TextOut(x, y + MenuHeight(), txt, &TxtSet, &qPainter, this);
}

bool
BitMapQT::oTextOutW(int x, int y, w_char *txt, int)
{
	if(!txt || !txt[0]) return false;
	return com_TextOutW(x, y + MenuHeight(), txt, &TxtSet, &qPainter, this);
}

bool
BitMapQT::oPolygon(POINT *pts, int cp, char *)
{
	int i;
	QPoint *a;

	if((a = (QPoint*)malloc(cp * sizeof(QPoint)))) {
		for(i = 0; i < cp; i++) {
			a[i].setX(pts[i].x);	a[i].setY(pts[i].y + MenuHeight());
			}
		qPainter.drawPolygon(a, cp);
		free(a);
		}
	if(hgo) hgo->oPolygon(pts, cp);
	return true;
}

bool 
BitMapQT::foPolygon(lfPOINT *pts, int cp, char *)
{
	int i;
	QPointF *a;
	POINT *ipts;

	if((a = (QPointF*)malloc(cp * sizeof(QPointF)))) {
		for(i = 0; i < cp; i++) {
			a[i].setX(pts[i].fx);	a[i].setY(pts[i].fy + MenuHeight());
			}
		qPainter.drawPolygon(a, cp);
		free(a);
		}
	if (hgo && hgo->OC_type != OC_HATCH && (ipts = (POINT*)malloc(cp * sizeof(POINT)))) {
		for (i = 0; i < cp; i++) {
			ipts[i].x = iround(pts[i].fx);		ipts[i].y = iround(pts[i].fy);
			}
		hgo->oPolygon(ipts, cp, NULL);
		free(ipts);
		}
	return true;
}

bool 
BitMapQT::oSolidPolygon(POINT *pts, int cp)
{
	int i;
	QPoint *a;

	qPen.setStyle(Qt::NoPen);		qPainter.setPen(qPen);
	if((a = (QPoint*)malloc((cp+2) * sizeof(QPoint)))) {
		for(i = 0; i < cp; i++) {
			a[i].setX(pts[i].x);	a[i].setY(pts[i].y);
			}
		qPainter.drawPolygon(a, cp);
		free(a);
		}
	qPen.setStyle(Qt::SolidLine);		qPainter.setPen(qPen);
	return true;
}

bool
BitMapQT::GradPG(fPOINT3D *pts, long npt, double Lo, double Hi, double pLo, double pHi, lfPOINT *grad)
{
	return com_GradPG(pts, npt, Lo, Hi, pLo, pHi, grad, this, &qPainter);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The display output class
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
OutputQT::OutputQT(GraphObj *g):BitMapQT(g, (RLPwidget*)0L)
{
	int w, h;

	HScroll = VScroll = 0L;
	GetDesktopSize(&w, &h);
	CreateNewWindow(BaseObj = g);
	if(widget) {
		if(CurrWidgetPos.x >= ((w>>1)-100))CurrWidgetPos.x = CurrWidgetPos.y = 50;
		widget->show();
		widget->move(CurrWidgetPos.x+=50, CurrWidgetPos.y+=50);
		if(widget->x() || widget->y()) {
			CurrWidgetPos.x = widget->x();	CurrWidgetPos.y = widget->y();
			}
		}
	dlgwidget = 0L;
	Notary->ValPtr(this, true);
}

OutputQT::OutputQT(DlgWidget *wi):BitMapQT(0L, wi)
{
	//assume fixed size (dialog) widget
	if(dlgwidget == wi) {
		wi->move(CurrWidgetPos.x+50, CurrWidgetPos.y+50);
		}
	widget = 0L;
	dlgwidget = wi;
	HScroll = VScroll = 0L;			BaseObj = 0L;
	wi->OutputClass = this;
	wi->mempic = mempic;
	xAxis.flags = 0L;
	yAxis.flags = AXIS_INVERT;	//drawing origin upper left corner
	Notary->ValPtr(this, true);
}

OutputQT::~OutputQT()
{
	Notary->ValPtr(this, false);
	if(qPainter.isActive()) qPainter.end();
	if(widget) delete widget;
	widget = 0L;
	KillTextCursor();
	if(mempic) delete mempic;
	mempic = 0L;	
	if(hgo) delete hgo;
	if(image) delete image;
	image = 0L;	hgo = 0L;
	defs.Idle(CMD_FLUSH);
}

bool
OutputQT::ActualSize(RECT *rc)
{
	if(rc) {
		rc->left = rc->top = 0;
		if(widget) {
			rc->bottom = widget->height() - MenuHeight()-6;
			rc->right = widget->width();
			}
		else if(dlgwidget) {
			rc->bottom = dlgwidget->height() - MenuHeight()-6;
			rc->right = dlgwidget->width();
			}
		if(rc->bottom < 10 && rc->right < 10) {
			rc->right = 600;	rc->bottom = 400;
			}
		return (rc->right > 40 && rc->bottom > 40);
		}
	return false;
}

void
OutputQT::Caption(char *txt, bool bModified)
{
	QString cap(txt);

	if(bModified) cap.append(" [modified]");
	if(widget) widget->setWindowTitle(cap);
	else if(dlgwidget) dlgwidget->setWindowTitle(cap);
}

const static char* hand_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
".......11.......",	"...110100111....",
"..10011001001...",	"..1001100100101.",
"...1001001001101",	"...1001001001001",
".110100000001001",	"1001100000000001",
"100010000000001.",	".10000000000001.",
".10000000000001.",	"..100000000001..",
"...10000000001..",	"....100000001...",
".....10000001...",	".....10000001..."};

const static char* zoom_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"................",	"................",
".......111......",	".....11...11....",
"....1.......1...",	"...1....1....1..",
"...1....1....1..",	"..1.....0.....1.",
"..1..1100011..1.",	"..1.....0.....1.",
"...1....1....1..",	"...1....1....1..",
"....1.......1...",	".....11...11....",
".......111......",	"................"};


const static char* drawpen_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"11..............",	"1111............",	
".11111..........",	".1110011........",
"..1000001.......",	"..10001001......",
"...10001001.....",	"...100001001....",
"....100001001...",	".....100001001..",
"......100001001.",	".......100001011",
"........10000111",	".........100111.",
"..........1111..",	"...........11..."};

const static char* paste_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"..1.............",	"..1.............",
"11111...........",	"..1.............",
"..1...111111111.",	".....10000000101",
".....10100000101",	".....10111111001",
".....10000000001",	".....10000000001",
".....10000000001",	".....10000000001",
".....10000000001",	".....10000000001",
"......111111111.",	"................"};

const static char* drect_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"..1.............",	"..1.............",
"11111...........",	"..1.............",
"..1.............",	"................",
"...1111111111111",	"...1000000000001",
"...1000000000001",	"...1000000000001",
"...1000000000001",	"...1000000000001",
"...1000000000001",	"...1000000000001",
"...1111111111111",	"................"};

const static char* drrect_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"..1.............",	"..1.............",
"11111...........",	"..1.............",
"..1.............",	"................",
".....111111111..",	"....10000000001.",
"...1000000000001",	"...1000000000001",
"...1000000000001",	"...1000000000001",
"...1000000000001",	"....10000000001.",
".....111111111..",	"................"};

const static char* delly_xpm[] = {
"16 16 3 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"..1.............",	"..1.............",
"11111...........",	"..1.............",
"..1.............",	"................",
".......11111....",	".....110000011..",
"....10000000001.",	"...1000000000001",
"...1000000000001",	"...1000000000001",
"....10000000001.",	".....110000011..",
".......11111....",	"................"};

const static char* etracc_xpm[] = {
"16 16 4 1",		". c None",
"1 c #000000",		"0 c #ffffff",
"2 c #0000ff",
"11111...........",	"1...............",
"1..1111111111111",	"1..1.2.2.1222221",
"...12.2.21222221",	"...1.2.2.1222221",
"...1111111111111",	"...1.2.2.1.2.2.1",
"...12.2.212.2.21",	"...1.2.2.1.2.2.1",
"...1111111111111",	"...1.2.2.1.2.2.1",
"...12.2.212.2.21",	"...1.2.2.1.2.2.1",
"...1111111111111",	"................"};

void
OutputQT::MouseCursor(int cid, bool force)
{
	if(cid == cCursor && !force) return;
	if(cid == MC_LAST) cid = cCursor;
	if(widget)CurrWidget = widget;
	else if(dlgwidget) CurrWidget = dlgwidget;
	if(widget) switch(cid) {
	case MC_ARROW:	widget->setCursor(QCursor(Qt::ArrowCursor));	break;
	case MC_TXTFRM:
	case MC_CROSS:	widget->setCursor(QCursor(Qt::CrossCursor));	break;
	case MC_WAIT:	widget->setCursor(QCursor(Qt::WaitCursor));	break;
	case MC_TEXT:	widget->setCursor(QCursor(Qt::IBeamCursor));	break;
	case MC_NORTH:	widget->setCursor(QCursor(Qt::SizeVerCursor));	break;
	case MC_NE:	widget->setCursor(QCursor(Qt::SizeBDiagCursor));break;
	case MC_COLWIDTH:
	case MC_EAST:	widget->setCursor(QCursor(Qt::SizeHorCursor));	break;
	case MC_SE:	widget->setCursor(QCursor(Qt::SizeFDiagCursor));break;
	case MC_SALL:	widget->setCursor(QCursor(Qt::SizeAllCursor));	break;
	case MC_MOVE:
		widget->setCursor(QCursor(QPixmap(hand_xpm), 7, 7));
		break;
	case MC_ZOOM:
		widget->setCursor(QCursor(QPixmap(zoom_xpm), 7, 7));
		break;
	case MC_PASTE:
		widget->setCursor(QCursor(QPixmap(paste_xpm), 2, 2));
		break;
	case MC_DRAWPEN:
		widget->setCursor(QCursor(QPixmap(drawpen_xpm), 0, 0));
		break;
	case MC_DRAWREC:
		widget->setCursor(QCursor(QPixmap(drect_xpm), 0, 0));
		break;
	case MC_DRAWRREC:
		widget->setCursor(QCursor(QPixmap(drrect_xpm), 0, 0));
		break;
	case MC_DRAWELLY:
		widget->setCursor(QCursor(QPixmap(delly_xpm), 0, 0));
		break;
	case MC_ETRACC:
		widget->setCursor(QCursor(QPixmap(etracc_xpm), 0, 0));
		break;
	default:
//		printf("Unknown Mouse Cursor selected\n");
		return;
		}
	cCursor = cid;
}

bool
OutputQT::SetScroll(bool isVert, int iMin, int iMax, int iPSize, int iPos)
{
	if(widget) return widget->SetScroll(isVert, iMin, iMax, iPSize, iPos);
	return false;
}

bool
OutputQT::EndPage()
{
	if(ShowObj) {
		RestoreMark();
		}
	if(widget)widget->repaint();
	else if(dlgwidget)dlgwidget->repaint();
	return true;
}

bool
OutputQT::UpdateRect(RECT *rc, bool)
{
	ClipRect(NULL);
	if(widget)widget->update((int)rc->left, (int)(rc->top + MenuHeight()), (int)(rc->right-rc->left), (int)(rc->bottom-rc->top));
	else if(dlgwidget)dlgwidget->update((int)rc->left, (int)rc->top, (int)(rc->right-rc->left), (int)(rc->bottom-rc->top));
	return true;
}

void
OutputQT::ShowLine(POINT * pt, int cp, DWORD color, bool bShow)
{
	int i;
	RECT rec;
	POINT *pts;

	if(!(pts = (POINT*)malloc((cp+2) * sizeof(POINT))))return;
	for(i = 0; i < cp; i++) {
		pts[i].x = pt[i].x;	pts[i].y = pt[i].y + MenuHeight();
		}
	if(cp < 2){
		free(pts);	return;
		}
	rec.left = rec.right = pt[0].x;	rec.top = rec.bottom = pt[0].y;
	for(i = 1; i < cp; i++) {
		if(pt[i].x < rec.left) rec.left = pt[i].x;
		if(pt[i].x > rec.right) rec.right = pt[i].x;
		if(pt[i].y < rec.top) rec.top = pt[i].y;
		if(pt[i].y > rec.bottom) rec.bottom = pt[i].y;
		}
	IncrementMinMaxRect(&rec, 20);
	if(rec.bottom < 2 && rec.top < 2){
		free(pts);	return;
		}
	if(ShowObj && bShow) {
		((eph_obj*)ShowObj)->Restore();
		delete ((eph_obj*)ShowObj);
		ShowObj = 0L;
		}
 	if(!(ShowObj = new eph_line((eph_line*)ShowObj, pts, cp, color, this, bShow))){
		free(pts);	return;
		}
	((eph_obj*)ShowObj)->DoPlot(&qPainter);
	defs.UpdRect(this, rec.left, rec.top, rec.right, rec.bottom);
	defs.Idle(CMD_UPDATE);
	free(pts);
}

void
OutputQT::ShowEllipse(POINT p1, POINT p2, DWORD color)
{
	RECT rec;
	bool bShow = true;

	rec.left = rec.right = p1.x;			rec.top = rec.bottom = p1.y;
	UpdateMinMaxRect(&rec, p2.x, p2.y);		IncrementMinMaxRect(&rec, 2);
	if(rec.bottom < 2 && rec.top < 2) return;
	if(!(ShowObj = new eph_ellipse((eph_obj*)ShowObj, p1, p2, color, this, bShow)))return;
	UpdateRect(&rec, false);
}

void
OutputQT::ShowInvert(RECT *rec)
{
	POINT spts[5];

	spts[0].x = spts[4].x = spts[3].x = rec->left;
	spts[0].y = spts[4].y = spts[1].y = rec->top;
	spts[1].x = spts[2].x = rec->right;
	spts[2].y = spts[3].y = rec->bottom;
	ShowLine(spts, 5, 0x0);
}

bool
OutputQT::SetMenu(int type)
{
	RLPmenu *menu_bar;
	men_def *curr_men = 0L;

	if(widget && !dlgwidget) {
		menu_bar = new RLPmenu(((RLPwidget*)widget)->drawWidget);
		switch (type) {
		case MENU_SPREAD:
			curr_men = data_men_1;	break;
		case MENU_GRAPH:
			curr_men = data_men_2;	break;
		case MENU_PAGE:
			curr_men = data_men_3;	break;
			}
		if(curr_men) {
			add_menu(widget->drawWidget, menu_bar, curr_men);
			menu_bar->setVisible(true);
			if(type == MENU_SPREAD) {
				((RLPwidget*)widget)->CheckHistory();
				}
			return true;
			}
		}
	return false;
}

void
OutputQT::CheckMenu(int select, bool check)
{
	if(widget) {
		((RLPwidget*)widget)->CheckMenu(select+CM_T_STANDARD, check); 
		}
}

bool
OutputQT::Command(int cmd, void *data)
{
	unsigned char *dt = 0L;

	switch (cmd) {
	case CMD_CPY_DATA:				//coming here from the layers dialog
		if (((Plot*)data)->Id == GO_BOXPLOT) {
//			printf("OutputQT::Command(CMD_CPY_DATA) from Boxplot\n");
			if (((Plot*)data)->Command(CMD_CPY_DATA, &dt, 0L)){
//				printf("%s\n", dt);
				}
			}
		break;
	case CMD_TIMER:
		if (prog_bar_ptr) {
			((ProgressBar*)prog_bar_ptr)->Command(cmd, 0L, 0L);;
			}
//		printf("OutputQT::Command(CMD_TIMER)\n");
		break;
		}
	return true;
}

void
OutputQT::CreateNewWindow(GraphObj *g)
{
	int w=0, h=0;

	GetDesktopSize(&w, &h);
	if(w < 10 && h < 10) {
		w = 858;	h = 572;
		}
	widget = new RLPwidget(0, 0, this, g);
	if(widget) {
		widget->mempic = mempic;
		widget->setGeometry(0, 0, (int)(w*0.7), (int)(h*0.7));
		((RLPwidget*)widget)->mempic = mempic;
		HScroll = ((RLPwidget*)widget)->HScroll;
		VScroll = ((RLPwidget*)widget)->VScroll;
		}
}

int
OutputQT::MenuHeight()
{
	if(widget) return 25;
	return 0;
}

void 
OutputQT::RestoreMark()
{
	if(ShowObj) {
		((eph_obj*)ShowObj)->Restore();
		delete ((eph_obj*)ShowObj);
		ShowObj = 0L;
		}
}

//configure page size ... using the printer setup
bool ConfigPrinter()
{
	QRectF rc;
	
	QPageSetupDialog dlg(printer, NULL);
	if(dlg.exec()) {
		switch (defs.units()) {
		case 0:		default:
			rc = printer->paperRect(QPrinter::Millimeter);
			defs.pg_width = rc.right();		defs.pg_height = rc.bottom();
			break;
		case 1:
			rc = printer->paperRect(QPrinter::Millimeter);
			defs.pg_width = rc.right()/10.0;	defs.pg_height = rc.bottom()/10.0;
			break;
		case 2:
			rc = printer->paperRect(QPrinter::Inch);
			defs.pg_width = rc.right();		defs.pg_height = rc.bottom();
			break;
			} 
		}
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Output spreadsheet or graph to printer
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
PrintQT::PrintQT(GraphObj *g, char */*file*/): anyOutput()
{
	go = g;		hgo = 0L;	minLW = 1;
	hres = vres = 72;
	OC_type = OC_PRINT;		hasClip = false;
	Notary->ValPtr(this, true);
}

PrintQT::~PrintQT()
{
	Notary->ValPtr(this,false);
	if(hgo) delete(hgo);
	defs.Idle(CMD_FLUSH);
}

bool 
PrintQT::ActualSize(RECT *rc)
{
	if(printer && rc) {
		//debug: cannot read printer page rect ???
		QRectF qrc = printer->pageRect(QPrinter::DevicePixel);
		rc->top = rc->left = 0;
		rc->bottom = qrc.height();		rc->right = qrc.width();
		return true;
		}
	else {
		rc->left = rc->top = rc->right = rc->bottom = 0;
		}
	return false;
}

bool
PrintQT::SetLine(LineDEF *lDef)
{
	int iw;
	double fiw;

	if(lDef->width != LineWidth || lDef->width != LineWidth ||
		lDef->pattern != dPattern || lDef->color != dLineCol) {
		LineWidth = lDef->width;
		fiw = un2fix(lDef->width);
		iw = iround(fiw);
		dPattern = lDef->pattern;
		RLP.finc = 256.0/un2fix(lDef->patlength*8.0);
		RLP.fp = 0.0;
		iLine = iw > minLW ? iw : minLW;
		if(fiw < minLW) fiw = minLW;
		dLineCol = lDef->color;
		qPen.setColor(MK_QCOLOR(dLineCol));
		qPen.setWidthF(fiw);
		qPen.setStyle(Qt::SolidLine);
		qPen.setCapStyle(Qt::RoundCap);
		qPen.setJoinStyle(Qt::RoundJoin);
		qPainter.setPen(qPen);
		}
	return true;
}

bool
PrintQT::ClipRect(RECT *rec)
{
	int x, y, w, h;

	if (!rec){								//reset clipping
		hasClip = false;
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		qPainter.setClipping(false);
		}
	else if(rec->left == rec->right || rec->bottom == rec->top) {
		hasClip = false;
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		qPainter.setClipping(false);
		}
	else if(rec->left < rec->right && rec->bottom > rec->top){			//set up clipping rectangle
		memcpy(&ClipRC, rec, sizeof(RECT));
		hasClip = true;
		ClipRC.top += MenuHeight();	ClipRC.bottom += MenuHeight();
		x = ClipRC.left;		w = ClipRC.right - ClipRC.left;
		y = ClipRC.top;			h = ClipRC.bottom - ClipRC.top; 
		qPainter.setClipRect(x, y, w, h);
		qPainter.setClipping(true);
		}
	else return ClipRect(NULL);
	return true;
}

bool
PrintQT::SetFill(FillDEF *fill)
{
	if(!fill) return false;
	if (hgo && !(hgo->getDisp())) {
		delete hgo;				hgo = 0L;
		}
	if(!fill) return false;
	if((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		}
	else {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	qPainter.setBrush(MK_QCOLOR(fill->color));
	dFillCol = fill->color;			dFillCol2 = fill->color2;
	return true;
}

bool
PrintQT::SetTextSpec(TextDEF *set)
{
	return com_SetTextSpec(set, &TxtSet, this, &qFont, &qPainter);
}

bool
PrintQT::StartPage()
{
	QPrintDialog dialog(printer, 0L);
	if (dialog.exec()){
		qPainter.begin(printer);
		QRectF qrc = printer->pageRect(QPrinter::DevicePixel);
		DeskRect.left = DeskRect.top = 0;
		DeskRect.bottom = qrc.height();		DeskRect.right = qrc.width();
		hres = vres = (double)printer->resolution();
		minLW = hres/200.0;
		return bPrinting = true;
		}
	else return false;
}

bool
PrintQT::EndPage()
{
	qPainter.end();		bPrinting = false;
	return true;
}

bool 
PrintQT::Eject()
{
	printer->newPage();
	return true;
}

bool
PrintQT::oCircle(int x1, int y1, int x2, int y2, char*)
{
	qPainter.drawEllipse(x1, y1, x2-x1, y2-y1);
	if(hgo) return hgo->oCircle(x1, y1, x2, y2);
	return true;
}

bool 
PrintQT::foCircle(double x1, double y1, double x2, double y2, char*)
{
	QRectF rect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	qPainter.drawEllipse(rect);
	if(hgo) return hgo->foCircle(x1, y1, x2, y2);
	return true;
}

bool
PrintQT::oSphere(int cx, int cy, int r, POINT *pts, int cp, char *nam)
{
	if (dFillCol == dFillCol2) {
		if (pts) return oPolygon(pts, cp, nam);
		else return oCircle(cx - r, cy - r, cx + r, cy + r);
		}
	return com_CircGrad(cx, cy, r, pts, cp, nam, this, &qPainter);
}

bool
PrintQT::oPolyline(POINT * pts, int cp)
{
	int i;

	if(cp < 1) return false;
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		}
	else {
		for (i = 1; i < cp; i++)qPainter.drawLine(pts[i-1].x, pts[i-1].y, pts[i].x, pts[i].y);
		}
	return true;
}
	
bool 
PrintQT::oRectangle(int x1, int y1, int x2, int y2, char *)
{
	qPainter.drawRect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	if(hgo) hgo->oRectangle(x1, y1, x2, y2, 0L);
	return true;
}

bool
PrintQT::foRectangle(double x1, double y1, double x2, double y2, char *)
{
	QRectF rect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	qPainter.drawRect(rect);
	if(hgo) hgo->oRectangle(x1, y1, x2, y2, 0L);
	return true;
}


bool 
PrintQT::oSolidRectangle(int x1, int y1, int x2, int y2)
{
	qPainter.fillRect(x1, y1 + MenuHeight(), x2-x1, y2-y1, MK_QCOLOR(dLineCol));
	return true;
}

bool 
PrintQT::foSolidRectangle(double x1, double y1, double x2, double y2)
{
	QRectF rect(x1, y1 + MenuHeight(), x2-x1, y2-y1);
	qPainter.fillRect(rect, MK_QCOLOR(dLineCol));
	return true;
}

bool
PrintQT::oSolidLine(POINT *p)
{
	qPainter.drawLine(p[0].x, p[0].y, p[1].x, p[1].y);
	return true;
}
	
bool
PrintQT::oTextOut(int x, int y, unsigned char *txt, int)
{
	if(!txt || !txt[0]) return false;
	return com_TextOut(x, y, txt, &TxtSet, &qPainter, this);
}

bool
PrintQT::oTextOutW(int x, int y, w_char *txt, int)
{
	if(!txt || !txt[0]) return false;
	return com_TextOutW(x, y, txt, &TxtSet, &qPainter, this);
}

bool
PrintQT::oPolygon(POINT *pts, int cp, char *)
{
	int i;
	QPoint *a;

	if((a = (QPoint*)malloc(cp * sizeof(QPoint)))) {
		for(i = 0; i < cp; i++) {
			a[i].setX(pts[i].x);	a[i].setY(pts[i].y);
			}
		qPainter.drawPolygon(a, cp);
		free(a);
		}
	if(hgo) hgo->oPolygon(pts, cp);
	return true;
}

bool 
PrintQT::oSolidPolygon(POINT *pts, int cp)
{
	int i;
	QPoint *a;

	qPen.setStyle(Qt::NoPen);		qPainter.setPen(qPen);
	if((a = (QPoint*)malloc(cp * sizeof(QPoint)))) {
		for(i = 0; i < cp; i++) {
			a[i].setX(pts[i].x);	a[i].setY(pts[i].y);
			}
		qPainter.drawPolygon(a, cp);
		free(a);
		}
	qPen.setStyle(Qt::SolidLine);		qPainter.setPen(qPen);
	return true;
}

bool
PrintQT::GradPG(fPOINT3D *pts, long npt, double Lo, double Hi, double pLo, double pHi, lfPOINT *grad)
{
	return com_GradPG(pts, npt, Lo, Hi, pLo, pHi, grad, this, &qPainter);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Temporary visible objects: show action
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
eph_obj::eph_obj(eph_obj *nxt, anyOutput *o)
{
	next = nxt;	points = 0L;	lcolor = 0x0;
	bounds.left = bounds.top = bounds.right = bounds.bottom = 0;
	out = o;			source = NULL;
}

eph_obj::~eph_obj()
{
	int w, h;

	if(next) delete(next);
	w = bounds.right - bounds.left;		h = bounds.bottom - bounds.top;
	if(out && source && bounds.right > bounds.left && bounds.bottom > bounds.top) {
		out->CopyBitmap(bounds.left, bounds.top + out->MenuHeight(), source, 0, 0, w, h, false);
//		printf("delete eph_obj without previous restore\n");
		}
	bounds.left = bounds.top = bounds.right = bounds.bottom = 0;
	if(source)DelBitmapClass(source);
	source = NULL;		out = NULL;
	if(points) free(points);
	next = 0L;	points = 0L;
}

void
eph_obj::DoPlot(QPainter *qp)
{
	if(next) next->DoPlot(qp);
	DoPlot(qp);
}

void
eph_obj::Restore()
{
	int w, h;

	if(next)next->Restore();
//	printf("eph_obj::Restore()\n");
	w = bounds.right - bounds.left;		h = bounds.bottom - bounds.top;
	if(out && source) {
		if(out) out->CopyBitmap(bounds.left, bounds.top + out->MenuHeight(), source, 0, 0, w, h, false);
		DelBitmapClass(source);		source = NULL;
		if(points) free(points);
		next = 0L;	points = 0L;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
eph_line::eph_line(eph_obj *nxt, POINT * pts, int n, DWORD color, anyOutput *o, bool):eph_obj(nxt, o)
{
	int i;

	if(n >1 && (points = (POINT*)malloc(n * sizeof(POINT)))) {
		bounds.left = bounds.right = pts[0].x;	bounds.top = bounds.bottom = pts[0].y;
		for( i = 1; i < n; i++) {
			if(pts[i].x > bounds.right) bounds.right = pts[i].x;
			if(pts[i].x < bounds.left) bounds.left = pts[i].x;
			if(pts[i].y > bounds.bottom) bounds.bottom = pts[i].y;
			if(pts[i].y < bounds.top) bounds.top = pts[i].y;
			}
		bounds.top -= o->MenuHeight();
		IncrementMinMaxRect(&bounds, 5);
		if(bounds.left < 0) bounds.left = 0;
		if(bounds.top < 0) bounds.top = 0;
		source = GetRectBitmap(&bounds, out = o);
		memcpy(points, pts, n * sizeof(POINT));
		cp = n;			lcolor = color;
		}
	else {
		points = NULL;		cp = 0;
		}
}

eph_line::~eph_line()
{
	if(next) delete(next);
	if(points) free(points);
	next = 0L;			points = 0L;
}

void
eph_line::DoPlot(QPainter *qp)
{
	int i;
	QPen qpen;

	if(next)next->DoPlot(qp);
	if(!points || cp < 2) return;
	qpen.setColor(MK_QCOLOR(lcolor));	qpen.setWidth(1);
	qpen.setStyle(Qt::SolidLine);		qpen.setCapStyle(Qt::RoundCap);
	qpen.setJoinStyle(Qt::RoundJoin);	qp->setPen(qpen);
	for (i = 1; i < cp; i++)qp->drawLine(points[i-1].x, points[i-1].y, points[i].x, points[i].y);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
eph_ellipse::eph_ellipse(eph_obj *nxt, POINT pt1, POINT pt2, DWORD color, anyOutput *o, bool):eph_obj(nxt, o)
{
	p1.x = pt1.x;	p1.y = pt1.y;	p2.x = pt2.x;	p2.y = pt2.y;
	lcolor = color;
}

eph_ellipse::~eph_ellipse()
{
	if(next) delete(next);
	next = 0L;
}

void
eph_ellipse::DoPlot(QPainter *qp)
{
	QPen qpen;

	if(next)next->DoPlot(qp);
	qpen.setColor(MK_QCOLOR(lcolor));	qpen.setWidth(1);
	qpen.setStyle(Qt::SolidLine);		qpen.setCapStyle(Qt::RoundCap);
	qpen.setJoinStyle(Qt::RoundJoin);	qp->setPen(qpen);
	qp->drawArc(p1.x, p1.y, p2.x-p1.x, p2.y-p1.y, 0, 5760);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//eph_invert::eph_invert(eph_obj *nxt, QPixmap *src, int x, int y, int w, int h, anyOutput *o):eph_obj(nxt, o)
eph_invert::eph_invert(eph_obj *nxt, QPixmap *, int, int, int, int, anyOutput *o):eph_obj(nxt, o)
{
//	printf("eph_invert not implemented\n");
/*
	eph_obj *eo = nxt;

	source = src;			invert = true;
	rx = x;	ry = y;	rw = w;	rh = h;
	while(eo) {
		eo->invert = false;	eo = eo->next;
		}
*/
}

eph_invert::~eph_invert()
{
	if(next) delete(next);
	next = 0L;
}

void
//eph_invert::DoPlot(QPainter *qp)
eph_invert::DoPlot(QPainter *)
{
//	if(next)next->DoPlot(qp);
//	QImage qi = (source->copy(rx, ry, rw, rh)).toImage();
//	if(invert) qi.invertPixels();
//	qp->drawImage(rx, ry, qi);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// show animated copy mark support
RECT rCopyMark;
static anyOutput *oCopyMark = 0L, *copy_bm0 = 0L, *copy_bm1 = 0L;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//temporaly visible objects
eph_animated::eph_animated(eph_obj *nxt, int x, int y, int w, int h, anyOutput *o):eph_obj(nxt, o)
{
	rx = x;	ry = y;	rw = w;	rh = h;
	SetMinMaxRect(&bounds, x, y, x+w, y+h);
	IncrementMinMaxRect(&bounds, 2);
}

eph_animated::~eph_animated()
{
	if(next) delete(next);
	next = 0L;
}

void
eph_animated::DoPlot(QPainter *qp)
{
	int i, sx, sy, rlp = 0;
	QPen qpen;
	POINT pts[3];

	if(next)next->DoPlot(qp);
	if(!rw || !rh) return;

	qpen.setWidth(1);
	qpen.setStyle(Qt::SolidLine);			qpen.setCapStyle(Qt::RoundCap);
	qpen.setJoinStyle(Qt::RoundJoin);
	for(i = 0; i < 4; i++) {
		qpen.setColor((DWORD)0x00ffffffL);	qp->setPen(qpen);
		switch(i) {
		case 0:
			pts[0].x = rx;			pts[0].y = ry;
			pts[2].x = rx+rw;		pts[2].y = ry;
			sx = rw > 0 ? 1 : -1;		sy = 0;
			break;
		case 1:
			pts[0].x = rx+rw;		pts[0].y = ry;
			pts[2].x = rx+rw;		pts[2].y = ry+rh;
			sx = 0;				sy = rh > 0 ? 1 : -1;
			break;
		case 2:
			pts[0].x = rx+rw;		pts[0].y = ry+rh;
			pts[2].x = rx;			pts[2].y = ry+rh;
			sx = rw > 0 ? -1 : 1;		sy = 0;
			break;
		case 3:
			pts[0].x = rx;			pts[0].y = ry+rh;
			pts[2].x = rx;			pts[2].y = ry;
			sx = 0;				sy = rh > 0 ? -1: 1;
			break;
			}
		qp->drawLine(pts[0].x, pts[0].y, pts[2].x, pts[2].y);
		qpen.setColor((DWORD)0x0L);		qp->setPen(qpen);
		pts[1].x = pts[0].x;			pts[1].y = pts[0].y;
		for( ; pts[1].x != pts[2].x || pts[1].y != pts[2].y; ) {
			pts[1].x += sx;			pts[1].y += sy;
			rlp = (rlp+1) & 0x07;
			if(rlp == 0) {
				pts[0].x = pts[1].x;	pts[0].y = pts[1].y;
				}
			else if(rlp == 3) {
				qp->drawLine(pts[0].x, pts[0].y, pts[1].x, pts[1].y);
				}
			}
		if(rlp < 3 && (pts[0].x != pts[1].x || pts[0].y != pts[1].y)) 
			qp->drawLine(pts[0].x, pts[0].y, pts[1].x, pts[1].y);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Text cursor support
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static anyOutput *oTxtCur = 0L, *txtcur0 = 0L;
RECT rTxtCur = {0, 0, 0, 0}, rTxtUpd = {0, 0, 0, 0};
static bool bTxtCur = false,  bTxtCurIsVis = false,	bSuspend = false;
static TxtCurBlink *cTxtCur = 0L;
static POINT ptTxtCurLine[2];
static LineDEF liTxtCur = {0.1, 6.0, 0x000000ff, 0x0L};

static void DrawCursor()
{
	RECT clpRC;
	LineDEF oldLine;

	if(Notary->IsValidPtr(oTxtCur)) {
		oTxtCur->CopyClipRC(&clpRC);
		oTxtCur->GetLine(&oldLine);
		oTxtCur->SetLine(&liTxtCur);
		oTxtCur->oSolidLine(ptTxtCurLine);
		oTxtCur->SetLine(&oldLine);
		}
	else {
		bTxtCur = false;
		txtcur0 = oTxtCur = 0L;
		}
}

void HideTextCursor()
{
	if(txtcur0 && Notary->IsValidPtr(oTxtCur) && bTxtCur) {
		bTxtCur = false;
		oTxtCur->CopyBitmap(rTxtUpd.left, rTxtUpd.top, txtcur0, 0, 0, rTxtUpd.right - rTxtUpd.left, 
			rTxtUpd.bottom - rTxtUpd.top, false);
		if(((OutputQT*)oTxtCur)->widget) ((OutputQT*)oTxtCur)->widget->update(rTxtUpd.left, rTxtUpd.top, 
			rTxtUpd.right - rTxtUpd.left, rTxtUpd.bottom - rTxtUpd.top);
		else if(((OutputQT*)oTxtCur)->dlgwidget) ((OutputQT*)oTxtCur)->dlgwidget->update(rTxtUpd.left, rTxtUpd.top, 
			rTxtUpd.right - rTxtUpd.left, rTxtUpd.bottom - rTxtUpd.top);
		DelBitmapClass(txtcur0);
		}
	txtcur0 = oTxtCur = 0L;
}

void KillTextCursor()
{
	if(txtcur0 && Notary->IsValidPtr(oTxtCur) && bTxtCur) {
		DelBitmapClass(txtcur0);
		}
	bTxtCur = false;
	txtcur0 = oTxtCur = 0L;
}

void ShowTextCursor(anyOutput *out, RECT *disp, DWORD color)
{
	if(!out ||!Notary->IsValidPtr(out)) return;
	if((out->OC_type & 0xff) != OC_BITMAP) return;
	if(bTxtCur) HideTextCursor();
	else KillTextCursor();
	liTxtCur.color = color;
	liTxtCur.width = defs.GetSize(SIZE_HAIRLINE) * 2.0;
	oTxtCur = out;
	memcpy(&rTxtCur, disp, sizeof(RECT));
	rTxtCur.bottom += ((BitMapQT*)out)->MenuHeight();
	rTxtCur.top += ((BitMapQT*)out)->MenuHeight();
	rTxtUpd.top = rTxtCur.top -3;		rTxtUpd.left = rTxtCur.left -3;
	rTxtUpd.right = rTxtCur.right +3;	rTxtUpd.bottom = rTxtCur.bottom +3;
	rTxtUpd.top -= ((BitMapQT*)out)->MenuHeight(); 
	rTxtUpd.bottom -= ((BitMapQT*)out)->MenuHeight(); 
	txtcur0 = GetRectBitmap(&rTxtUpd, out);
	rTxtUpd.top += ((BitMapQT*)out)->MenuHeight(); 
	rTxtUpd.bottom += ((BitMapQT*)out)->MenuHeight(); 
	ptTxtCurLine[0].x = rTxtCur.left;	ptTxtCurLine[0].y = rTxtCur.top - oTxtCur->MenuHeight();
	ptTxtCurLine[1].x = rTxtCur.right;	ptTxtCurLine[1].y = rTxtCur.bottom - oTxtCur->MenuHeight();
	out->ClipRect(NULL);
	bTxtCur = true;				DrawCursor();
	if(((OutputQT*)oTxtCur)->widget) ((OutputQT*)oTxtCur)->widget->update(rTxtUpd.left, rTxtUpd.top, 
		rTxtUpd.right - rTxtUpd.left, rTxtUpd.bottom - rTxtUpd.top);
	else if(((OutputQT*)oTxtCur)->dlgwidget) ((OutputQT*)oTxtCur)->dlgwidget->update(rTxtUpd.left, rTxtUpd.top, 
		rTxtUpd.right - rTxtUpd.left, rTxtUpd.bottom - rTxtUpd.top);
	bSuspend = false;
}

void InitTextCursor(bool init)
{
	if(init && !cTxtCur) cTxtCur = new TxtCurBlink();
	else if(!init && cTxtCur) {
		delete cTxtCur;		cTxtCur = 0L;
		}
}

void RedrawTextCursor()
{
	if (bTxtCurIsVis && bTxtCur && oTxtCur) DrawCursor();
}

TxtCurBlink::TxtCurBlink():QObject(CurrWidget)
{
	isVis = false;
	count = 0;
}

void
TxtCurBlink::Blink()
{
	if(bSuspend) return;
	if(!Notary->IsValidPtr(oTxtCur) || (ptTxtCurLine[0].x == ptTxtCurLine[1].x &&
		ptTxtCurLine[0].y == ptTxtCurLine[1].y)) return;
	count++;
	if(count < 6) return;
	count -= 6;
	if(oTxtCur) {
		if(isVis) {
			oTxtCur->CopyBitmap(rTxtUpd.left, rTxtUpd.top, txtcur0, 0, 0, rTxtUpd.right - rTxtUpd.left, 
				rTxtUpd.bottom - rTxtUpd.top, false);
			isVis = bTxtCurIsVis = false;
			}
		else {
			DrawCursor();
			isVis = bTxtCurIsVis = true;
			}
		if(((OutputQT*)oTxtCur)->widget) ((OutputQT*)oTxtCur)->widget->update(rTxtUpd.left, rTxtUpd.top, 
			rTxtUpd.right - rTxtUpd.left, rTxtUpd.bottom - rTxtUpd.top);
		else if(((OutputQT*)oTxtCur)->dlgwidget) ((OutputQT*)oTxtCur)->dlgwidget->update(rTxtUpd.left, rTxtUpd.top, 
			rTxtUpd.right - rTxtUpd.left, rTxtUpd.bottom - rTxtUpd.top);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Copy / paste
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void SuspendAnimation(anyOutput *o, bool bSusp)
{
	if(!Notary->IsValidPtr(o) || (o->OC_type&0xff) != OC_BITMAP) return;
	bSuspend = bSusp;
}

void HideCopyMark()
{
	if(Notary->IsValidPtr(oCopyMark) && Notary->IsValidPtr(copy_bm0) && Notary->IsValidPtr(copy_bm1)) {
		DelBitmapClass(copy_bm0);		copy_bm0 = NULL;
		oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.top + oCopyMark->MenuHeight(), copy_bm1, 0, 0,
			rCopyMark.right - rCopyMark.left, 4, false);
		oCopyMark->CopyBitmap(rCopyMark.right-4, rCopyMark.top + oCopyMark->MenuHeight(), copy_bm1,
			rCopyMark.right-rCopyMark.left-4, 0, 4, rCopyMark.bottom - rCopyMark.top, false);
		oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.bottom + oCopyMark->MenuHeight()-4, copy_bm1, 0,
			rCopyMark.bottom - rCopyMark.top -4, rCopyMark.right - rCopyMark.left, 4, false);
		oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.top + oCopyMark->MenuHeight(), copy_bm1, 0, 0, 
			4, rCopyMark.bottom - rCopyMark.top, false);
		DelBitmapClass(copy_bm1);		copy_bm1 = NULL;
		oCopyMark->UpdateRect(&rCopyMark, true);
		oCopyMark = NULL;
		}
}

void HideCopyMark(bool)
{
	HideCopyMark();
}

void ShowCopyMark(anyOutput *out, RECT *mrk, int nRec)
{
	int i;
	eph_obj **animated;

	if(!out || (out->OC_type&0xff) != OC_BITMAP) return;
	HideTextCursor();
	HideCopyMark();			bSuspend = false;
	if(!out || !mrk || !nRec || !cTxtCur) return;
	animated = (eph_obj **)&((OutputQT*)out)->ShowAnimated;
	if(*animated) *animated = NULL;
	if(((OutputQT*)out)->ShowAnimated) delete (eph_obj*)(((OutputQT*)oCopyMark)->ShowAnimated);
	((OutputQT*)out)->ShowAnimated = (new eph_animated((eph_obj*) 0L, mrk[0].left, mrk[0].top, 
		mrk[0].right - mrk[0].left, mrk[0].bottom - mrk[0].top, NULL));
	oCopyMark = out;
	rCopyMark.left = mrk[0].left;	rCopyMark.right = mrk[0].right;
	rCopyMark.top = mrk[0].top;		rCopyMark.bottom = mrk[0].bottom;
	for(i = 1; i < nRec; i++) {
		UpdateMinMaxRect(&rCopyMark, mrk[i].left, mrk[i].top);
		UpdateMinMaxRect(&rCopyMark, mrk[i].right, mrk[i].bottom);
		}
	IncrementMinMaxRect(&rCopyMark, 2);
	out->UpdateRect(&rCopyMark, false);
	copy_bm0 = GetRectBitmap(&rCopyMark, out);		copy_bm1 = GetRectBitmap(&rCopyMark, out);
}

void InvalidateOutput(anyOutput *)
{
//	printf("Stub: InvalidateOutput()\n");
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Clipboard support
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
QString rlp_mime_text("text/plain");
QString rlp_mime_xml("application/x-rlp-xml");
QString rlp_mime_rlp("application/x-rlp");
QString rlp_mime_sylk("application/x-sylk");
QString rlp_mime_svg("image/svg+xml");
QString rlp_mime_uri("text/uri-list");
QString rlp_mime_utf("UTF8_STRING");

void
AnimCopyMrk()
{
	static LineDEF liCopy0 = {0.0, 0.0, 0x00cbcbcb, 0x0};
	static LineDEF liCopy1 = {0.0, 3.0, 0x0, 0x0f0f0f0f};
	static int cp_mark = 0;
	static POINT pts[5];

	if(!copy_bm0 || !copy_bm1 || !oCopyMark) return;
	pts[0].x = pts[4].x = 3;		pts[0].y = pts[4].y = 3;
	pts[1].x = rCopyMark.right - rCopyMark.left -4;
	pts[1].y = 3;				pts[2].x = rCopyMark.right - rCopyMark.left -4;
	pts[2].y = rCopyMark.bottom - rCopyMark.top -3;
	pts[3].x = 3;				pts[3].y = rCopyMark.bottom - rCopyMark.top -3;
	copy_bm0->SetLine(&liCopy0);		copy_bm0->oPolyline(pts, 5);
	copy_bm0->SetLine(&liCopy1);		copy_bm0->RLP.finc = 1.0;
	copy_bm0->RLP.fp = (cp_mark & 0xf);
	copy_bm0->oPolyline(pts, 5);	cp_mark++;
	oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.top + oCopyMark->MenuHeight(), copy_bm0, 0, 0,
		rCopyMark.right - rCopyMark.left, 4, false);
	oCopyMark->CopyBitmap(rCopyMark.right-4, rCopyMark.top + oCopyMark->MenuHeight(), copy_bm0,
		rCopyMark.right-rCopyMark.left-4, 0, 4, rCopyMark.bottom - rCopyMark.top, false);
	oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.bottom + oCopyMark->MenuHeight()-4, copy_bm0, 0,
		rCopyMark.bottom - rCopyMark.top -4, rCopyMark.right - rCopyMark.left, 4, false);
	oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.top + oCopyMark->MenuHeight(), copy_bm0, 0, 0, 
		4, rCopyMark.bottom - rCopyMark.top, false);
	oCopyMark->UpdateRect(&rCopyMark, true);
}

void
CopyText(char *txt, int, unsigned int cf)
{
	QMimeData *src = 0L;;

	src = new QMimeData();
	if(cf == CF_TEXT) {
		QByteArray data1(txt);
		src->setData(rlp_mime_text, data1);
		}
	if(cf == CF_SYLK) {
		QByteArray data1(txt);
		src->setData(rlp_mime_sylk, data1);
		}
	clipboard->setMimeData(src);
}

void ExportBitmap(GraphObj *g, char *Filename, char *format) 
{
	long w, h;
	double res, width, height;
	anyOutput *restore = 0L;
	BitMapQT *mempic = 0L;
	char *qs = (char*)rlp_strdup((unsigned char*)Filename);
	char title[80];
	DWORD ColGrect = 0x00ffffff, ColGrectLine = 0x00ffffff;

	res = 300.0;
	HideCopyMark(false);		HideTextCursor();
	if (g->Id == GO_GRAPH) {
		ColGrect = g->GetColor(COL_GRECT);
		ColGrectLine = g->GetColor(COL_GRECTLINE);
		}
	if(!strcasecmp(format, ".png")) {
		rlp_strcpy(title, 79, (char*)"Export Portable Network Graphics (*.png)"); 
		}
	else return;
	restore = g->getDisp();
	width = g->GetSize(SIZE_GRECT_RIGHT) - g->GetSize(SIZE_GRECT_LEFT);
	height = g->GetSize(SIZE_GRECT_BOTTOM) - g->GetSize(SIZE_GRECT_TOP);
	if (GetBitmapRes(&res, &width, &height, title)){
		w = (long)(width * res * Units[defs.units()].convert / 25.4);
		h = (long)(height * res * Units[defs.units()].convert / 25.4);
		mempic = new BitMapQT(w, h, res, res);
		mempic->minLW = res/200;	mempic->OC_type  = OC_UNKNOWN;
		g->SetColor(COL_GRECTLINE, ColGrect);
		g->DoPlot(mempic);
		QPixmap *pic = mempic->mempic;
		QImage qi = pic->toImage();
		qi.save(qs);
		g->SetColor(COL_GRECTLINE, ColGrectLine);
		g->DoPlot(restore);
		delete mempic;
		}
}

void
CopyData(GraphObj *SourceGO, unsigned int)
{
	char *text_plain = 0L;
	long buf_size = 0, w, h;
	double width, height;
	QMimeData *src;
	BitMapQT *mempic = 0L;
	anyOutput *restore = 0L;
	DWORD ColGrect = 0x00ffffff, ColGrectLine = 0x00ffffff;

	if(SourceGO->Id == GO_SPREADDATA) {
		src = new QMimeData();
		//copy data RLPlot OEM
		SourceGO->Command(CMD_COPY_XML, &text_plain, 0L);
		QByteArray data1(text_plain);
		src->setData(rlp_mime_xml, data1);
		if(text_plain) free(text_plain);
		text_plain = 0L;

		//copy tab separated file format
		SourceGO->Command(CMD_COPY_TSV, &text_plain, 0L);
		QByteArray data2(text_plain);
		src->setData(rlp_mime_text, data2);
		if(text_plain) free(text_plain);
		text_plain = 0L;

		//copy SYLK file format
		SourceGO->Command(CMD_COPY_SYLK, &text_plain, 0L);
		QByteArray data3(text_plain);
		src->setData(rlp_mime_sylk, data3);
		if(text_plain) free(text_plain);
		text_plain = 0L;

		clipboard->setMimeData(src);
		}
	else if(SourceGO->Id == GO_GRAPH || SourceGO->Id == GO_PAGE) {
		if (SourceGO->Id == GO_GRAPH) {
			ColGrect = SourceGO->GetColor(COL_GRECT);
			ColGrectLine = SourceGO->GetColor(COL_GRECTLINE);
			}
		restore = SourceGO->getDisp();
		width = SourceGO->GetSize(SIZE_GRECT_RIGHT) - SourceGO->GetSize(SIZE_GRECT_LEFT);
		height = SourceGO->GetSize(SIZE_GRECT_BOTTOM) - SourceGO->GetSize(SIZE_GRECT_TOP);
		w = (long)(width * 300.0 * Units[defs.units()].convert / 25.4);
		h = (long)(height * 300.0 * Units[defs.units()].convert / 25.4);
		mempic = new BitMapQT(w, h, 300.0, 300.0);
		mempic->minLW = 2;	mempic->OC_type  = OC_UNKNOWN;
		SourceGO->SetColor(COL_GRECTLINE, ColGrect);
		SourceGO->DoPlot(mempic);
		src = new QMimeData();
		QPixmap *pic = mempic->mempic;
		QImage qi = pic->toImage();
//		qi.setDotsPerMeterY(15240);		qi.setDotsPerMeterX(15240);
		src->setImageData(qi);
		SourceGO->SetColor(COL_GRECTLINE, ColGrectLine);
		if((text_plain = GraphToMem(SourceGO, &buf_size))){
			QByteArray data1(text_plain);
			src->setData(rlp_mime_rlp, data1);
			free(text_plain);	text_plain = 0L;
			}
		if((text_plain = DoCopySvg(SourceGO, &buf_size))){
			QByteArray data2(text_plain);
			src->setData(rlp_mime_svg, data2);
			free(text_plain);	text_plain = 0L;
			}
		clipboard->setMimeData(src);
		SourceGO->DoPlot(restore);
		delete mempic;
		}
	else if (SourceGO->Id == GO_BOXPLOT){
		text_plain = (char*)((BoxPlot*)SourceGO)->CopySylk();
		if(text_plain) {
			src = new QMimeData();
			QByteArray data1(text_plain);
			src->setData(rlp_mime_sylk, data1);
			free(text_plain);	text_plain = 0L;
			clipboard->setMimeData(src);
			}
		}
}

void 
CopyGraph(GraphObj *g)
{
	if(g->Id == GO_PAGE && CurrGraph) CopyData(CurrGraph, 0);
	else CopyData(g, 0);
}

//paste text to dialog item
unsigned char* 
PasteText()
{
	const QMimeData *src;
	long i, cb;
	static char *out_buff = NULL;
	char *sdata;
	long out_pos, out_size;
	unsigned char *ret;

	out_buff = NULL;	out_pos = out_size = 0;
	ret = NULL;
	src = clipboard->mimeData();
//	if(src->hasFormat(rlp_mime_text)) {
//		QByteArray data = src->data(rlp_mime_text);
	if(src->hasFormat(rlp_mime_utf)) {
		QByteArray data = src->data(rlp_mime_utf);
		sdata = data.data();
		cb = data.size();
		QString qst = QString::fromUtf8(sdata, cb);
		wchar_t *array;
		array = (wchar_t*)calloc(cb+2, sizeof(wchar_t));
		cb = qst.toWCharArray(array);
		for(i = 0; i < cb; i++ ){
			if(array[i] > 127) {
				sprintf(TmpTxt, "&#%x;", array[i]);
				add_to_buff(&out_buff, &out_pos, &out_size, TmpTxt, 0);
				}
			else {
				TmpTxt[0] = (char)array[i];	TmpTxt[1] = 0;
				add_to_buff(&out_buff, &out_pos, &out_size, TmpTxt, 1);
				}
			}
		if(out_pos >0) {
			ret = (unsigned char*) calloc(out_pos+2, sizeof(unsigned char));
			for (i = 0; out_buff[i]; i++) ret[i] = (unsigned char)out_buff[i];
			return ret;
			}
		return 0L;
		}
#ifdef _DEBUG
	printf("Illegal Clipboard format in PasteText()!\n");
#endif
	return 0L;
}

#ifdef _DEBUG
void
enumCBformats()
{
	int nb, i, level = 0;;
	char mime_name[1024];
	QMimeData *mimeData;
	QStringList stringList;
	QByteArray byteArray;

	mimeData = (QMimeData*)clipboard->mimeData();
	if(mimeData) {
		stringList = mimeData->formats();
		foreach (const QString &str, stringList) {
			if(!level)printf("Testing for Clipboard formats:\n");
			nb = (int)str.size();	level++;
			if(nb){
				for(i = 0; i< nb; i++) {
					mime_name[i] = (str.data()[i]).toLatin1();
					}
				mime_name[i++] = '\0';
				printf("   %s\n", mime_name);
				}
			}
		}
	if(!level)printf("Clipboard is empty\n");
}
#endif
void
TestClipboard(GraphObj *g)
{
const QMimeData *src;
long i, cb;
unsigned char *text = 0;
char *out_buff = NULL, *sdata;
long out_pos = 0, out_size = 0;

#ifdef _DEBUG
	enumCBformats();
#endif
	src = clipboard->mimeData();
	if(g->Id == GO_SPREADDATA) {
		if(src->hasFormat(rlp_mime_xml)) {
			QByteArray data = src->data(rlp_mime_xml);
			cb = data.size();
			text = (unsigned char*)calloc(cb+2, sizeof(unsigned char));
			for(i = 0; i < cb; i++) text[i] = (data.data())[i];
			g->Command(CMD_PASTE_XML, text, 0L);
			free(text);
			}
		else if(src->hasFormat(rlp_mime_rlp)) {
			QByteArray data = src->data(rlp_mime_rlp);
			cb = data.size();
			text = (unsigned char*)calloc(cb+2, sizeof(unsigned char));
			for(i = 0; i < cb; i++) text[i] = (data.data())[i];
			OpenGraph(g, 0L, text, true);
			free(text);
			}
		else if(src->hasFormat(rlp_mime_sylk)) {
			QByteArray data = src->data(rlp_mime_sylk);
			cb = data.size();
			text = (unsigned char*)calloc(cb+2, sizeof(unsigned char));
			for(i = 0; i < cb; i++) text[i] = (data.data())[i];
			g->Command(CMD_PASTE_SYLK, text, 0L);
			free(text);
			}
		else if(src->hasFormat(rlp_mime_text)) {
			QByteArray data = src->data(rlp_mime_text);
			if((cb = data.size())) {
				text = (unsigned char*)calloc(cb+2, sizeof(unsigned char));
				for(i = 0; i < cb; i++) text[i] = (data.data())[i];
				ProcMemData(g, text, true);
				free(text);
				}
			}
		}
	else if(g->Id == GO_PAGE || g->Id == GO_GRAPH) {
		if(src->hasFormat(rlp_mime_rlp)) {
			QByteArray data = src->data(rlp_mime_rlp);
			cb = data.size();
			text = (unsigned char*)calloc(cb+2, sizeof(unsigned char));
			for(i = 0; i < cb; i++) text[i] = (unsigned char)((data.data())[i]);
			OpenGraph(g, 0L, text, true);
			free(text);
			}
		}
	else if(g->Id == GO_LABEL) {
		if(src->hasFormat(rlp_mime_utf)) {
			QByteArray data = src->data(rlp_mime_utf);
			sdata = data.data();
			cb = data.size();
			QString qst = QString::fromUtf8(sdata, cb);
			wchar_t *array;
			array = (wchar_t*)calloc(cb+2, sizeof(wchar_t));
			cb = qst.toWCharArray(array);
			for(i = 0; i < cb; i++ ){
				if(array[i] > 127) {
					sprintf(TmpTxt, "&#%x;", array[i]);
					add_to_buff(&out_buff, &out_pos, &out_size, TmpTxt, 0);
					}
				else {
					TmpTxt[0] = (char)array[i];	TmpTxt[1] = 0;
					add_to_buff(&out_buff, &out_pos, &out_size, TmpTxt, 1);
					}
				}
			if(out_pos)g->Command(CMD_ADDTXT, (void *)out_buff, g->getDisp());
			if (out_buff) free(out_buff);
			out_buff = NULL; out_pos = out_size = 0;
			if (array) free(array);
			}
		else {
#ifdef _DEBUG
			printf("Illegal clipboard format\n");
#endif
			}
		}
}

void
EmptyClip()
{
	HideCopyMark();
	clipboard->clear(QClipboard::Clipboard);
	clipboard->clear(QClipboard::Selection);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get display (desktop) size
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void
GetDesktopSize(int *width, int *height)
{

	if(desk_width && desk_height) {
		*width = desk_width;	*height = desk_height;
		}
	else {
		*width = 1280;		*height = 1024;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Menu definitions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
MenuAction::MenuAction(const QString &text, QObject *par, unsigned int cmd)
	:QAction(text, par)
{
	val_cmd = cmd;		parent = par;
	QObject::connect(this, &QAction::triggered, this, &MenuAction::do_menu_cmd);
}

void
MenuAction::do_menu_cmd()
{
	if(parent)((DrawWidget*)parent)->Command(val_cmd, 0L);
}

RLPmenu::RLPmenu(QWidget *parent):QMenuBar(parent)
{
	par = parent;
}

void
RLPmenu::paintEvent(QPaintEvent *range)
{
	if(Notary->IsValidPtr(par)) QMenuBar::paintEvent(range);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Common widget support
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
RLPwidget::RLPwidget(QWidget *par, const char *, anyOutput *o , GraphObj *g)
	: QWidget(par)
{
	int w, h;

	GetDesktopSize(&w, &h);
	mempic = 0L;
	layout = new QGridLayout;	nMitems = 0;
	layout->setContentsMargins(0, 0, 16, 16);
	drawWidget = new DrawWidget(this);
	drawWidget->setGeometry(20, 20, 100, 100);
	layout->addWidget(drawWidget, 1, 1);
	setLayout(layout);
	parent = par;
	OutputClass = o;
	o->Erase(defs.Color(COL_BG));
	BaseObj = g;		parent = par;
	setMinimumSize(100, 80);
	setGeometry(0, 0, 800, 600);
	HScroll = new QScrollBar(Qt::Horizontal, this);
	HScroll->setRange(0, 1000);
	HScroll->setSingleStep(1);
	HScroll->setPageStep(16);
	connect(HScroll, SIGNAL(valueChanged(int)), SLOT(hScrollEvent(int)));
	VScroll = new QScrollBar(Qt::Vertical, this);
	VScroll->setRange(0, 1000);
	VScroll->setSingleStep(1);
	VScroll->setPageStep(16);
	connect(VScroll, SIGNAL(valueChanged(int)), SLOT(vScrollEvent(int)));
	setFocusPolicy(Qt::StrongFocus);
	if(!MainWidget) {
		MainWidget = this;
		setAcceptDrops(true);
		startTimer(100);
		}
}

RLPwidget::~RLPwidget()
{
	if(OutputClass)((OutputQT*)OutputClass)->widget = 0L;
	defs.Idle(CMD_FLUSH);
	OutputClass = 0L;	BaseObj = 0L;
}

void 
RLPwidget::Command(unsigned int cmd, void *data)
{
	PrintQT *out;
	DefsRW *drw;
	int cc;

	if(!Notary->IsValidPtr(BaseObj)) return;
	if(BaseObj)switch (cmd) {
	case CM_OPEN:
		CheckHistory();
		BaseObj->Command(CMD_OPEN, 0L, OutputClass);
		BaseObj->Command(CMD_REDRAW, 0L, OutputClass);
		break;
	case CM_SAVE:
		BaseObj->Command(CMD_SAVE, 0L, OutputClass);
		CheckHistory();
		break;
	case CM_SAVEAS:
		BaseObj->Command(CMD_SAVEAS, 0L, OutputClass);
		CheckHistory();
		break;
	case CM_FILE1:	case CM_FILE2:	case CM_FILE3:
	case CM_FILE4:	case CM_FILE5:	case CM_FILE6:
		openHistoryFile(cmd - CM_FILE1);
		break;
	case CM_UNDO:
		BaseObj->Command(CMD_UNDO, 0L, OutputClass);
		break;
	case CM_INSROW:
		BaseObj->Command(CMD_INSROW, 0L, OutputClass);
		break;
	case CM_INSCOL:
		BaseObj->Command(CMD_INSCOL, 0L, OutputClass);
		break;
	case CM_DELROW:
		BaseObj->Command(CMD_DELROW, 0L, OutputClass);
		break;
	case CM_DELCOL:
		BaseObj->Command(CMD_DELCOL, 0L, OutputClass);
		break;
	case CM_FILLRANGE:
		BaseObj->Command(CMD_FILLRANGE, 0L, OutputClass);
		break;
	case CM_ARITH:
		BaseObj->Command(CMD_ARITH, 0L, OutputClass);
		break;
	case CM_ADDROWCOL:
		BaseObj->Command(CMD_ADDROWCOL, 0L, OutputClass);
		break;
	case CM_NEWGRAPH:
		BaseObj->Command(CMD_NEWGRAPH, 0L, OutputClass);
		break;
	case CM_NEWPAGE:
		BaseObj->Command(CMD_NEWPAGE, 0L, OutputClass);
		break;
	case CM_DELGRAPH:
		BaseObj->Command(CMD_DELGRAPH, 0L, OutputClass);
		break;
	case CM_REDRAW:
		BaseObj->Command(CMD_REDRAW, 0L, OutputClass);
		break;
	case CM_UPDATE:
		BaseObj->Command(CMD_UPDATE, 0L, OutputClass);
		break;
	case CM_DEFAULTS:
		BaseObj->Command(CMD_CONFIG, 0L, OutputClass);
		break;
	case CM_REPANOV:
		rep_anova(BaseObj, BaseObj->data);
		break;
	case CM_REPBDANOV:
		rep_bdanova(BaseObj, BaseObj->data);
		break;
	case CM_REPKRUSKAL:
		rep_kruskal(BaseObj, BaseObj->data);
		break;
	case CM_REPTWANR:
		rep_twoway_anova(BaseObj, BaseObj->data);
		break;
	case CM_REPTWANOV:
		rep_twanova(BaseObj, BaseObj->data);
		break;
	case CM_REPFRIEDM:
		rep_fmanova(BaseObj, BaseObj->data);
		break;
	case CM_CORRELM:
		rep_correl(BaseObj, BaseObj->data, 0);
		break;
	case CM_CORRELT:
		rep_correl(BaseObj, BaseObj->data, 1);
		break;
	case CM_SMPLSTAT:
		rep_samplestats(BaseObj, BaseObj->data);
		break;
	case CM_REPCMEANS:
		rep_compmeans(BaseObj, BaseObj->data);
		break;
	case CM_REPTWOWAY:
		rep_twowaytable(BaseObj, BaseObj->data);
		break;
	case CM_REPREGR:
		rep_regression(BaseObj, BaseObj->data);
		break;
	case CM_REPTWREGR:
		rep_twregression(BaseObj, BaseObj->data);
		break;
	case CM_ROBUSTLINE:
		rep_robustline(BaseObj, BaseObj->data);
		break;
	case CM_ZOOMIN:
		BaseObj->Command(CMD_ZOOM, (void*)(&"+"), OutputClass);
		break;
	case CM_ZOOMOUT:
		BaseObj->Command(CMD_ZOOM, (void*)(&"-"), OutputClass);
		break;
	case CM_ZOOMFIT:
		BaseObj->Command(CMD_ZOOM, (void*)(&"fit"), OutputClass);
		break;
	case CM_ZOOM25:
		BaseObj->Command(CMD_ZOOM, (void*)(&"25"), OutputClass);
		break;
	case CM_ZOOM50:
		BaseObj->Command(CMD_ZOOM, (void*)(&"50"), OutputClass);
		break;
	case CM_ZOOM100:
		BaseObj->Command(CMD_ZOOM, (void*)(&"100"), OutputClass);
		break;
	case CM_ZOOM200:
		BaseObj->Command(CMD_ZOOM, (void*)(&"200"), OutputClass);
		break;
	case CM_ZOOM400:
		BaseObj->Command(CMD_ZOOM, (void*)(&"400"), OutputClass);
		break;
	case CM_ADDPLOT:
		BaseObj->Command(CMD_ADDPLOT, 0L, OutputClass);
		break;
	case CM_ADDAXIS:
		BaseObj->Command(CMD_ADDAXIS, 0L, OutputClass);
		break;
	case CM_LEGEND:
		BaseObj->Command(CMD_LEGEND, 0L, OutputClass);
		break;
	case CM_BT_MEAN:
		rep_bootstrap_mean(BaseObj, BaseObj->data);
		break;
	case CM_BT_CORREL:
		rep_bootstrap_correlation(BaseObj, BaseObj->data);
		break;
	case CM_BT_REGR:
		rep_bootstrap_regression(BaseObj, BaseObj->data);
		break;
	case CM_PRISETUP:
		ConfigPrinter();
		return;
	case CM_PRINT:
		if(!BaseObj) return;
		OutputClass->MouseCursor(MC_WAIT, false);
		out = new PrintQT(BaseObj, 0L);
		if(true) {
			if(BaseObj->Id == GO_SPREADDATA) {
				BaseObj->Command(CMD_PRINT, 0L, out);
				}
			else if(out->StartPage()){
				BaseObj->DoPlot(out);	out->EndPage();
				}
			BaseObj->DoPlot(OutputClass);
			}
		delete out;	OutputClass->MouseCursor(MC_ARROW, false);
		return;
	case CM_EXPORT:
		OutputClass->MouseCursor(MC_WAIT, false);
		OpenExportName(BaseObj, 0L);
		BaseObj->DoPlot(OutputClass);
		OutputClass->MouseCursor(MC_ARROW, true);
		return;
	case CM_COPY:		case CM_CUT:		case CM_COPYGRAPH:
//		OCrestore = this;
		CopyGO = BaseObj;
		if(CurrGO && BaseObj->Id != GO_SPREADDATA) {
			if(CurrGO->Id == GO_POLYLINE || CurrGO->Id == GO_POLYGON || CurrGO->Id == GO_RECTANGLE
				|| CurrGO->Id == GO_ROUNDREC || CurrGO->Id == GO_ELLIPSE || CurrGO->Id == GO_BEZIER) {
//				ShowCopyMark(this, &CurrGO->rDims, 1);
				CopyData(CurrGO, CF_TEXT);			return;
				}
			else if(CurrGO->Id == GO_TEXTFRAME) {
				if(CurrGO->Command(CMD_COPY, 0L, OutputClass))	return;
				}
			}	
		if(BaseObj->Id == GO_SPREADDATA && BaseObj->Command(cmd == CM_CUT ? CMD_CUT : CMD_QUERY_COPY, 0L, OutputClass)) {
			CopyData(BaseObj, CF_TEXT);
			}
		else if(cmd == CM_CUT) return;
		else if(BaseObj->Id == GO_PAGE) {
			if(CurrGraph) {
				ShowCopyMark(OutputClass, &CurrGraph->rDims, 1);
				CopyGraph(CurrGraph);
				}
			}
		else if(CurrGraph && CurrGraph->Id == GO_GRAPH){
			ShowCopyMark(OutputClass, &CurrGraph->rDims, 1);
			CopyGraph(CurrGraph);
			}
		else if(BaseObj->Id == GO_GRAPH) {
			ShowCopyMark(OutputClass, &BaseObj->rDims, 1);
			CopyGraph(BaseObj);
			}
		return;
	case CM_PASTE:
		Undo.SetDisp(OutputClass);
		BaseObj->Command(CMD_MOUSECURSOR, 0L, OutputClass);
		if(BaseObj->Id == GO_SPREADDATA) TestClipboard(BaseObj);
		else if(BaseObj->Id == GO_PAGE || BaseObj->Id == GO_GRAPH){
			if(CurrGO) {
				if(CurrGO->Id == GO_TEXTFRAME)CurrGO->Command(CMD_PASTE, 0L, 0L);
				else if(CurrGO->Id == GO_LABEL)CurrGO->Command(CMD_PASTE, 0L, 0L);
				else TestClipboard(BaseObj);
				}
			else TestClipboard(BaseObj);
			}
		BaseObj->Command(CMD_MOUSECURSOR, 0L, OutputClass);
		break;
	case CM_ABOUT:
		RLPlotInfo();	return;
	case CM_DELOBJ:
		if(CurrGO && CurrGO->parent && CurrGO->parent->
			Command(CMD_DELOBJ, (void*)CurrGO, OutputClass)) {
			CurrGO = 0L;
//			if(Erase(defs.Color(COL_BG))) BaseObj->DoPlot(OutputClass);
			}
		else if(!CurrGO) InfoBox((char*)"No Object Selected!");
		break;
	case CM_T_STANDARD:	case CM_T_DRAW:		case CM_T_POLYLINE:
	case CM_T_POLYGON:	case CM_T_RECTANGLE:	case CM_T_ROUNDREC:
	case CM_T_ELLIPSE:	case CM_T_ARROW:	case CM_T_TEXT:
		HideCopyMark(false);
		cc = Mitems[cmd - CM_T_STANDARD]->getCmd() - CM_T_STANDARD;
		if(BaseObj && OutputClass)BaseObj->Command(CMD_TOOLMODE,(void*)(&cc), OutputClass);
		return;
	case CM_T_CHARMAP:
		cc = system(char_map);
		break;
	case CMD_WRITE_GRAPHS:
		break;
	case CMD_REG_MITEM:
		Mitems[nMitems++] = (MenuAction*)data;
		break;
	case CM_EXIT:
		if(BaseObj->Command(CMD_CAN_CLOSE, 0L, 0L)){
			if(BaseObj->parent) close();
//			if(BaseObj->parent) BaseObj->parent->Command(CMD_DELOBJ, BaseObj, 0L);
			else if(BaseObj->Id == GO_SPREADDATA) QAppl->exit(0);
			}
		break;
	case CM_NEWINST:
		if(defs.IniFile) {
			if((drw = new DefsRW())){
				drw->FileIO(FILE_WRITE);		delete drw;
				}
			}
		if(ShellCmd &&ShellCmd[0]) {
			rlp_strcpy(TmpTxt, TMP_TXT_SIZE, ShellCmd);
			strcat(TmpTxt, " &");		cc = system(TmpTxt);
			}
		break;
	default:
//		printf("RLPwidget::MenuAction %d\n", cmd);
		break;
	}
}

bool
RLPwidget::SetScroll(bool isVert, int iMin, int iMax, int iPSize, int iPos)
{
	QScrollBar *sb;

	if(isVert) {
		if(!(sb = VScroll))return false;
		}
	else if(!(sb = HScroll)) return false;
	if(iPos < sb->minimum()) return false;
	sb->setRange(iMin, iMax);	sb->setPageStep(iPSize);
	if(BaseObj && BaseObj->Id == GO_GRAPH) sb->setSingleStep(8);
	else sb->setSingleStep(1);
	sb->setValue(iPos);
	NoWaitDlgLoop();		return true;
}

bool
RLPwidget::CheckMenu(int sel, bool check)
{
	int i;

	for(i = 0; i < nMitems; i++) {
		if(sel == Mitems[i]->getCmd()) {
			Mitems[i]->setChecked(check);
			}
		else switch(Mitems[i]->getCmd()) {
		case CM_T_STANDARD:	case CM_T_DRAW:		case CM_T_POLYLINE:
		case CM_T_POLYGON:	case CM_T_RECTANGLE:	case CM_T_ROUNDREC:
		case CM_T_ELLIPSE:	case CM_T_ARROW:	case CM_T_TEXT:
			Mitems[i]->setChecked(false);
			break;
			}
		}
	return true;
}
void
RLPwidget::CheckHistory()
{
	int i, j, k;
	unsigned char **history[] = {&defs.File1, &defs.File2, &defs.File3, &defs.File4, &defs.File5, &defs.File6};
	QString fil[6];

	for(i = 0; i < 6; i++) {
		if(*history[i]){
			k = rlp_strlen(*history[i]);
			for (j = 0; j < k && defs.currPath[j] == (*history[i])[j]; j++);
			if((*history[i])[j] == '\\' || (*history[i])[j] == '/') j++;
			fil[i] = QString((const char*)*history[i]+j);
			}
		}
	for(i = 0; i < nMitems; i++) {
		switch(Mitems[i]->getCmd()) {
		case CM_FILE1:	case CM_FILE2:	case CM_FILE3:
		case CM_FILE4:	case CM_FILE5:	case CM_FILE6:
			j = Mitems[i]->getCmd() - CM_FILE1;
			if(*history[j]) {
				Mitems[i]->setText(fil[j]);	Mitems[i]->setVisible(true);
				}
			else Mitems[i]->setVisible(false);
			break;
			}
		}
}

//public slots: menu items, events
void
RLPwidget::hScrollEvent(int pos)
{
	static long cpos;

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	cpos = pos;
	if(BaseObj && OutputClass){
		BaseObj->Command(CMD_SETHPOS, (void*)(&cpos), OutputClass);
		if(pos != HScroll->value()) HScroll->setValue(pos);
		}
}

void
RLPwidget::vScrollEvent(int pos)
{
	static long cpos;
	
	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	cpos = pos;
	if(BaseObj && OutputClass){
		BaseObj->Command(CMD_SETVPOS, (void*)(&cpos), OutputClass);
		if(pos != VScroll->value()) VScroll->setValue(pos);
		}
}

void 
RLPwidget::wheelEvent(QWheelEvent *evt)
{
	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	bSuspend = true;
	if(BaseObj && OutputClass){
		if(evt->angleDelta().y() > 0)BaseObj->Command(CMD_SCROLL_UP, 0L, OutputClass);
		if(evt->angleDelta().y() < 0)BaseObj->Command(CMD_SCROLL_DOWN, 0L, OutputClass);
		if(evt->angleDelta().x() > 0)BaseObj->Command(CMD_SCROLL_LEFT, 0L, OutputClass);
		if(evt->angleDelta().x() < 0)BaseObj->Command(CMD_SCROLL_RIGHT, 0L, OutputClass);
		BaseObj->Command(CMD_SETSCROLL, 0L, OutputClass);
		repaint();
		}
	evt->accept();
	bSuspend = false;
}

//protected: widget events
void
RLPwidget::paintEvent(QPaintEvent *range)
{
	QRect rc;
	QPainter qpainter(this);

	if(!mempic) return;
	rc = range->rect();
}

void
RLPwidget::resizeEvent(QResizeEvent *)
{
	CurrWidget = this;
	HScroll->resize(width() -16, 16);
	HScroll->move(0, height()-16);
	VScroll->resize(16, height()-32);
	VScroll->move(width()-16, 18);
	if(BaseObj) BaseObj->Command(CMD_SETSCROLL, 0L, OutputClass);
}

void
RLPwidget::closeEvent(QCloseEvent *e)
{
	GraphObj *Bobj;

	if(BaseObj){
		Bobj = BaseObj;					BaseObj = 0L;
		if(Bobj->Command(CMD_CAN_CLOSE, 0L, 0L)) {
 			if(Bobj->parent) Bobj->parent->Command(CMD_DELOBJ, Bobj, 0L);
			OutputClass = 0L;
			e->accept();
			}
		else e->ignore();
		}
	else e->accept();
	if(this == MainWidget) QAppl->exit(0);
}

void
RLPwidget::mouseDoubleClickEvent(QMouseEvent *e)
{
	QPointF pos = e->position();
	MouseEvent mev = {(unsigned short)1, (unsigned short)(e->button() == 
		Qt::LeftButton?MOUSE_LBDOUBLECLICK:-1),(int)iround(pos.x()),(int)iround(pos.y())};

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	if(OutputClass) mev.y -= OutputClass->MenuHeight();
	if (BaseObj)BaseObj->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
	e->accept();
}

void
RLPwidget::mousePressEvent(QMouseEvent *e)
{
	int i;
	QPoint pos, posa;
	MouseEvent mev = {(unsigned short)1, (unsigned short)(e->button() == Qt::LeftButton ? MOUSE_LBDOWN : -1), 0, 0};

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	posa = QCursor::pos();
	mouse_offs.x = posa.x() - pos.x();		mouse_offs.y = posa.y() - pos.y();
	mouse_lx = posa.x() - mouse_offs.x;		mouse_ly = posa.y() - mouse_offs.y;
	HideTextCursor();		CurrWidget = this;
	i = e->modifiers();		pos = e->pos();
	mev.x = pos.x();		mev.y = pos.y();
	if(OutputClass) mev.y -= OutputClass->MenuHeight();
	if(i & Qt::ShiftModifier) mev.StateFlags |= 0x08;
	if(i & Qt::ControlModifier) mev.StateFlags |= 0x10;
	if(mev.StateFlags |= MOUSE_LBDOWN) mouse_buttons_down |= 0x01;
	if (BaseObj){
		if(BaseObj->Id == GO_GRAPH || BaseObj->Id == GO_PAGE) {
			if(!(((Graph*)BaseObj)->ToolMode) && ((BitMapQT*)OutputClass)->ShowObj){
				((BitMapQT*)OutputClass)->RestoreMark();
				}
			}
		BaseObj->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
		}
	MouseObj = BaseObj;
	bSuspend = false;
	e->accept();
}

void
RLPwidget::mouseReleaseEvent(QMouseEvent *e)
{
	int i;
	QPoint posa;
	QPointF pos = e->position();
	MouseEvent mev = {0, 0, (int)iround(pos.x()),(int)iround(pos.y())};

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	posa = QCursor::pos();
	mouse_offs.x = posa.x() - mev.x;		mouse_offs.y = posa.y() - mev.y;
	mouse_lx = posa.x() - mouse_offs.x;		mouse_ly = posa.y() - mouse_offs.y;
	if(Notary->IsValidPtr(OutputClass) && Notary->IsValidPtr(BaseObj)) {
		if(OutputClass) mouse_offs.y += OutputClass->MenuHeight();
		MouseObj = BaseObj;
		if(OutputClass) mev.y -= OutputClass->MenuHeight();
		}
	if(e->button() == Qt::LeftButton) mev.Action = MOUSE_LBUP;
	else if(e->button() == Qt::RightButton) mev.Action = MOUSE_RBUP;
	else {
		e->accept();		return;
		}
	i = e->modifiers();
	if(i & Qt::ShiftModifier) mev.StateFlags |= 0x08;
	if(i & Qt::ControlModifier) mev.StateFlags |= 0x10;
	mouse_buttons_down = 0;
	if (BaseObj) BaseObj->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
	e->accept();
}

void
RLPwidget::mouseMoveEvent(QMouseEvent *e)
{
	int i;
	QPoint pos;
	static MouseEvent mev = {(unsigned short)mouse_buttons_down, (unsigned short)MOUSE_MOVE, 0, 0};

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	pos = e->pos();
	mev.x = pos.x();	mev.y = pos.y();
	if(OutputClass) mev.y -= OutputClass->MenuHeight();
	i = e->modifiers();
	if(i & Qt::ShiftModifier) mev.StateFlags |= 0x08;
	if(i & Qt::ControlModifier) mev.StateFlags |= 0x10;
	if (BaseObj) BaseObj->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
	e->accept();
}

static unsigned int ti_busy = false;		//timer busy: do not reenter

void
RLPwidget::keyPressEvent(QKeyEvent *e)
{
	int i, c, cc;
	QChar qc;
	w_char uc;
	bool is_shifted, is_control;

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	i = e->modifiers();
	is_shifted = ((i & Qt::ShiftModifier) != 0);
	is_control = ((i & Qt::ControlModifier) != 0);
	CurrWidget = this;

	if(x() || y()) {
		CurrWidgetPos.x = x();			CurrWidgetPos.y = y();
		}
	if(BaseObj) switch(c = e->key()) {
		case Qt::Key_PageUp:
			if(is_shifted) BaseObj->Command(CMD_SHPGUP, 0L, OutputClass);
			else BaseObj->Command(CMD_PAGEUP, 0L, OutputClass);
			break;
		case Qt::Key_PageDown:
			if(is_shifted) BaseObj->Command(CMD_SHPGDOWN, 0L, OutputClass);
			else BaseObj->Command(CMD_PAGEDOWN, 0L, OutputClass);
			break;
		case Qt::Key_Left:
			if(is_shifted) BaseObj->Command(CMD_SHIFTLEFT, 0L, OutputClass);
			else BaseObj->Command(CMD_CURRLEFT, 0L, OutputClass);
			break;
		case Qt::Key_Right:
			if(is_shifted) BaseObj->Command(CMD_SHIFTRIGHT, 0L, OutputClass);
			else BaseObj->Command(CMD_CURRIGHT, 0L, OutputClass);
			break;
		case Qt::Key_Up:
			if(is_shifted) BaseObj->Command(CMD_SHIFTUP, 0L, OutputClass);
			else BaseObj->Command(CMD_CURRUP, 0L, OutputClass);
			break;
		case Qt::Key_Down:
			if(is_shifted) BaseObj->Command(CMD_SHIFTDOWN, 0L, OutputClass);
			else BaseObj->Command(CMD_CURRDOWN, 0L, OutputClass);
			break;
		case Qt::Key_Delete:
			BaseObj->Command(CMD_DELETE, 0L, OutputClass);
			break;
		case Qt::Key_Tab:
			BaseObj->Command(CMD_TAB, 0L, OutputClass);
			break;
		case Qt::Key_Backtab:
			BaseObj->Command(CMD_SHTAB, 0L, OutputClass);
			break;
		case Qt::Key_Home:
			BaseObj->Command(CMD_POS_FIRST, 0L, OutputClass);
			break;
		case Qt::Key_End:
			BaseObj->Command(CMD_POS_LAST, 0L, OutputClass);
			break;
		case Qt::Key_Escape:
			cc = TM_STANDARD;
			BaseObj->Command(CMD_TOOLMODE, (void *)(&cc), OutputClass);
			break;
		case Qt::Key_Plus:	case Qt::Key_Minus:
			if(is_control && (BaseObj->Id == GO_GRAPH || BaseObj->Id == GO_PAGE)) {
				BaseObj->Command(CMD_ZOOM, c == Qt::Key_Plus ? (void*)(&"+") : (void*)(&"-"), OutputClass);
				}
			else {
				if (c == Qt::Key_Plus) cc = '+';
				else cc = '-';
				BaseObj->Command(CMD_ADDCHAR, (void *)(&cc), OutputClass);
				}
			break;
		default:
			QString kres = e->text();
			for(i = 0; i < kres.length(); i++) {
				qc = kres.at(i);	uc = qc.unicode();
				cc = (int)uc;
				if(cc == 3) break;
				else if(uc > 255) BaseObj->Command(CMD_ADDCHARW, (void *)(&uc), OutputClass);
				else BaseObj->Command(CMD_ADDCHAR, (void*)(&cc), OutputClass);
				}
			break;
		}
	ti_busy = false;	
	e->accept();
}

void
RLPwidget::timerEvent(QTimerEvent *event)
{
	QPoint posa;
	int x, y;
	static MouseEvent mev = {(unsigned short)0, (unsigned short) MOUSE_MOVE, 0, 0};

	if(ti_busy || bSuspend)return;
	//proces mouse events without any button down
	if(mouse_offs.x && mouse_offs.y && Notary->IsValidPtr(MouseObj) && !QAppl->mouseButtons()) {
		posa = QCursor::pos();
		x = posa.x() - mouse_offs.x;	y = posa.y() - mouse_offs.y;
		if((x != mouse_lx || y != mouse_ly) && x > 0 && y > 0 && x < width() && y < height()) {
			ti_busy = true;
			mev.x = x;		mev.y = y;
			MouseObj->Command(CMD_MOUSE_EVENT, &mev, 0L);
			mouse_lx = x;		mouse_ly = y;
			ti_busy = false;
			}
		}
	//any progress bar
	if (prog_bar_ptr) {
		ti_busy = true;
		((ProgressBar*)prog_bar_ptr)->Command(CMD_TIMER, 0L, 0L);;
		event->accept();
		ti_busy = false;
		return;
		}
	//any animation?
	if(oCopyMark) {
		ti_busy = true;		AnimCopyMrk();		ti_busy = false;
		}
	defs.Idle(CMD_UPDATE);
	if(cTxtCur) {
		ti_busy = true;		cTxtCur->Blink();	ti_busy = false;
		}
	event->accept();
}

void
RLPwidget::focusInEvent(QFocusEvent *)
{
	QRect rc;

	rc = geometry();
	mouse_offs.x = rc.x() + 1;	mouse_offs.y = rc.y()+OutputClass->MenuHeight() + 1;
	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	if(x() || y()) {
		CurrWidgetPos.x = x();			CurrWidgetPos.y = y();
		}
	CurrWidget = this;
	if(BaseObj) {
		MouseObj = BaseObj;
		if(BaseObj->Id == GO_GRAPH) CurrGraph = (Graph*)BaseObj;
		}
}

void
RLPwidget::dragEnterEvent(QDragEnterEvent *event)
{
	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	event->accept();
}

void 
RLPwidget::dropEvent(QDropEvent *event)
{
	int nb, i, j;
	const QMimeData *mimeData;
	unsigned char c, *text = 0;

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	mimeData = event->mimeData();
	if(mimeData->hasFormat(rlp_mime_uri)) {
		QByteArray data = mimeData->data(rlp_mime_uri);
		nb = data.size();
		text = (unsigned char*)calloc(nb+8, sizeof(unsigned char));
		for(i = 7, j = 0; i < nb; i++) {
			c = (data.data())[i];
			if(c >31) text[j++] = c;
			}
		text[j++] = 0;
		text = (unsigned char*)strreplace((char*)"%20", (char*)" ", (char*)text);
		if(BaseObj) {
			BaseObj->Command(CMD_DROPFILE, text, BaseObj->getDisp());
			defs.FileHistory((char*)text);
			BaseObj->Command(CMD_REDRAW, 0L, BaseObj->getDisp());
			}
		free(text);		event->accept();
		}
}

//private functions
void
RLPwidget::openHistoryFile(int idx)
{
	unsigned char *name = 0L;

	if(!Notary->IsValidPtr(BaseObj) || !Notary->IsValidPtr(OutputClass)) return;
	switch (idx) {
	case 0:		name = defs.File1;	break;
	case 1:		name = defs.File2;	break;
	case 2:		name = defs.File3;	break;
	case 3:		name = defs.File4;	break;
	case 4:		name = defs.File5;	break;
	case 5:		name = defs.File6;	break;
		}
	if(name && FileExist((char*)name)) {
		BaseObj->Command(CMD_DROPFILE, name, OutputClass);
		defs.FileHistory((char*)name);
		CheckHistory();
		BaseObj->Command(CMD_REDRAW, 0L, 0L);
		}
	else {
		ErrorBox((char*)SCMS_BAD_FILE);
		}
}

DrawWidget::DrawWidget(RLPwidget *par): QWidget(par)
{
	parent = par;		menu_bar = 0L;
	Notary->ValPtr(this, true);
}

DrawWidget::~DrawWidget()
{	Notary->ValPtr(this, false);
}

void 
DrawWidget::Command(unsigned int cmd, void* data)
{
	if(parent)((RLPwidget*)parent)->Command(cmd, data);
}

void
DrawWidget::paintEvent(QPaintEvent *range)
{
	QRect rc;
	RECT rlp_rec;
	QPainter qpainter(this);

	if((!parent->mempic)) return;
	rc = range->rect();
	rlp_rec.left = rc.left();
	rlp_rec.top = rc.top() >24 ? rc.top() : 24;
	rlp_rec.right = rc.right() < width() ? rc.right() : width();
	rlp_rec.bottom = rc.height() < height() ? rc.height() : height();
	qpainter.drawPixmap(rlp_rec.left, rlp_rec.top, *(parent->mempic), rlp_rec.left,
		rlp_rec.top, rlp_rec.right+1, rlp_rec.bottom+1);
}

void LoopDlgWnd() 	//keep message processing running
{
	QAppl->processEvents();
}

bool NoWaitDlgLoop()
{
	// keep the main event loop running
	// no waiting: processEvents will process all pending events.
	QAppl->processEvents();
	return false;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Find a suitable www browser and more initialization
void FindBrowser()
{	int cb;

	//find a suitable browser
	if(FileExist((char*)"/usr/bin/firefox")) WWWbrowser = strdup((char*)"firefox");
	else if(FileExist((char*)"/usr/bin/mozilla")) WWWbrowser = strdup((char*)"mozilla");
	else if(FileExist((char*)"/usr/bin/netscape")) WWWbrowser = strdup((char*)"netscape");
	else if(FileExist((char*)"/usr/bin/konqueror")) WWWbrowser = strdup((char*)"konqueror");
	//test if a charmap utility is installed
	if(FileExist((char*)"/usr/bin/gucharmap")) char_map = strdup((char*)"gucharmap &");
	else if(FileExist((char*)"/usr/bin/kcharselect")) char_map = strdup((char*)"kcharselect &");
	//use home as startup directory
	sprintf(TmpTxt, "%s", getenv("HOME"));
	defs.currPath = strdup(TmpTxt);	strcat(TmpTxt, "/.RLPlot");
	defs.IniFile = strdup(TmpTxt);
	//do more initialization: create application icon
	QPixmap pm(RLPlot_xpm);
	rlp_icon = new QIcon(pm);
	QAppl->setWindowIcon(*rlp_icon);
	cb = rlp_strcpy(TmpTxt, 100,(char*)" QT ");
	rlp_strcpy(TmpTxt + cb, 90, (char*)QT_VERSION_STR);
	gui_version = (char*)rlp_strdup((unsigned char*)TmpTxt);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The MAIN antry point
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include <QPageSize>
int main (int argc, char **argv)
{
	QAppl = new QApplication(argc, argv);
	DefsRW *drw;

	ShellCmd = (char*)calloc(rlp_strlen(argv[0])+2, sizeof(char));
	rlp_strcpy(ShellCmd, 1000, argv[0]);
	clipboard = QApplication::clipboard();
	printer = new QPrinter(QPrinter::HighResolution);
	QPageLayout pglay = printer->pageLayout();
	InitTextCursor(true);
	FindBrowser();
	Notary = new notary();
	if (argc > 1 && argv[1] && argv[1][0]) {
		sprintf(TmpTxt, "%s", argv[1]);
		if(TmpTxt[0]) {
			rmquot(TmpTxt);
 			if(TmpTxt[0] && FileExist(TmpTxt)) LoadFile= strdup(TmpTxt);
			}
		}
	ShowBanner(true);
	QAppl->exec();
	if(defs.IniFile) {
		if((drw = new DefsRW())){
			drw->FileIO(FILE_WRITE);		delete drw;
			}
		}
	if(printer) delete(printer);
	if(WWWbrowser) free(WWWbrowser);
	if(char_map) free(char_map);
	return 0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Dialog box support
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
DlgWidget::DlgWidget(QWidget *par, const char *, tag_DlgObj *d, DWORD flags): QWidget(0L, 
	flags & 0x00000004 ? Qt::WindowStaysOnTopHint: Qt::Window)
{
	parent = par;
	dlg = d;
	setFocusPolicy(Qt::StrongFocus);
}

DlgWidget::~DlgWidget()
{
	if(OutputClass){
		((OutputQT*)OutputClass)->widget=0L;
		delete ((OutputQT*)OutputClass);
		}
}

void
DlgWidget::paintEvent(QPaintEvent *range)
{
	QRect rc;
	QPainter qpainter(this);

	if(!Notary->IsValidPtr(OutputClass)) return;
	rc = range->rect();
	qpainter.drawPixmap(rc.left(), rc.top(), *mempic, rc.left(), rc.top(), rc.width()+1, rc.height()+1);
	if(OutputClass && ((OutputQT*)OutputClass)->ShowObj)
		((eph_obj*)(((OutputQT*)OutputClass)->ShowObj))->DoPlot(&qpainter);
	if(OutputClass && ((OutputQT*)OutputClass)->ShowAnimated)
		((eph_obj*)(((OutputQT*)OutputClass)->ShowAnimated))->DoPlot(&qpainter);
}

void
DlgWidget::mouseDoubleClickEvent(QMouseEvent *e)
{
	QPointF pos = e->position();
	MouseEvent mev = {(unsigned short)1, (unsigned short)(e->button() == 
		Qt::LeftButton?MOUSE_LBDOUBLECLICK:-1),(int)iround(pos.x()),(int)iround(pos.y())};
	if(!Notary->IsValidPtr(dlg) || !Notary->IsValidPtr(OutputClass)) return;
	if (dlg) dlg->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
}

void
DlgWidget::mousePressEvent(QMouseEvent *e)
{
	int i;
	QPointF pos = e->position();

	MouseEvent mev = {1, (unsigned short)(e->button() == Qt::LeftButton ? MOUSE_LBDOWN : -1), 
		(int)iround(pos.x()),(int)iround(pos.y())};
	if(!Notary->IsValidPtr(dlg) || !Notary->IsValidPtr(OutputClass)) return;
	KillTextCursor();		CurrWidget = this;
	i = e->modifiers();
	if(i & Qt::ShiftModifier) mev.StateFlags |= 0x08;
	if(i & Qt::ControlModifier) mev.StateFlags |= 0x10;
	if(mev.StateFlags |= MOUSE_LBDOWN) mouse_buttons_down |= 0x01;
	if (dlg)dlg->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
}

void
DlgWidget::mouseReleaseEvent(QMouseEvent *e)
{
	int i, button;
	QPointF pos = e->position();

	if(!Notary->IsValidPtr(dlg) || !Notary->IsValidPtr(OutputClass)) return;
	button =  e->button() == Qt::LeftButton? MOUSE_LBUP : e->button() == Qt::RightButton ? MOUSE_RBUP : -1;
	MouseEvent mev = {0, (unsigned short)button, (int)iround(pos.x()),(int)iround(pos.y())};
	i = e->modifiers();
	if(i & Qt::ShiftModifier) mev.StateFlags |= 0x08;
	if(i & Qt::ControlModifier) mev.StateFlags |= 0x10;
	mouse_buttons_down = 0;
	if (dlg) dlg->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
}

void
DlgWidget::mouseMoveEvent(QMouseEvent *e)
{
	int i;
	QPointF pos = e->position();
	MouseEvent mev = {(unsigned short)mouse_buttons_down, MOUSE_MOVE, (int)iround(pos.x()),(int)iround(pos.y())};

	if(!Notary->IsValidPtr(dlg) || !Notary->IsValidPtr(OutputClass)) return;
	i = e->modifiers();
	if(i & Qt::ShiftModifier) mev.StateFlags |= 0x08;
	if(i & Qt::ControlModifier) mev.StateFlags |= 0x10;
	if (dlg) dlg->Command(CMD_MOUSE_EVENT, (void *)&mev, OutputClass);
}

void
DlgWidget::keyPressEvent(QKeyEvent *e)
{
	int i, c, cc;
	QChar qc;
	w_char uc;
	bool is_shifted;

	if(!Notary->IsValidPtr(dlg) || !Notary->IsValidPtr(OutputClass)) return;
	i = e->modifiers();
	is_shifted = ((i & Qt::ShiftModifier) != 0);
	CurrWidget = this;
	if(x() || y()) {
		CurrWidgetPos.x = x();			CurrWidgetPos.y = y();
		}
	if(dlg) switch(c = e->key()) {
		case Qt::Key_Left:
			if(is_shifted) dlg->Command(CMD_SHIFTLEFT, 0L, OutputClass);
			else dlg->Command(CMD_CURRLEFT, 0L, OutputClass);
			break;
		case Qt::Key_Right:
			if(is_shifted) dlg->Command(CMD_SHIFTRIGHT, 0L, OutputClass);
			else dlg->Command(CMD_CURRIGHT, 0L, OutputClass);
			break;
		case Qt::Key_Up:
			if(is_shifted) dlg->Command(CMD_SHIFTUP, 0L, OutputClass);
			else dlg->Command(CMD_CURRUP, 0L, OutputClass);
			break;
		case Qt::Key_Down:
			if(is_shifted) dlg->Command(CMD_SHIFTDOWN, 0L, OutputClass);
			else dlg->Command(CMD_CURRDOWN, 0L, OutputClass);
			break;
		case Qt::Key_Delete:
			dlg->Command(CMD_DELETE, 0L, OutputClass);
			break;
		case Qt::Key_Tab:
			dlg->Command(CMD_TAB, 0L, OutputClass);
			break;
		case Qt::Key_Backtab:
			dlg->Command(CMD_SHTAB, 0L, OutputClass);
			break;
		case Qt::Key_Home:
			dlg->Command(CMD_POS_FIRST, 0L, OutputClass);
			break;
		case Qt::Key_End:
			dlg->Command(CMD_POS_LAST, 0L, OutputClass);
			break;
		default:
			QString kres = e->text();
			for(i = 0; i < kres.length(); i++) {
				qc = kres.at(i);	uc = qc.unicode();
				cc = (int)uc;
				if(cc == 3) dlg->Command(CMD_COPY, 0L, OutputClass);
				else if(cc == 22) dlg->Command(CMD_PASTE, 0L, OutputClass);
				else if(cc == 26) dlg->Command(CMD_UNDO, 0L, OutputClass);
				else if(uc > 255) dlg->Command(CMD_ADDCHARW, (void *)(&uc), OutputClass);
				else dlg->Command(CMD_ADDCHAR, (void *)(&cc), OutputClass);
				}
			break;
		}
	e->accept();
}

void
DlgWidget::focusInEvent(QFocusEvent *)
{
	mouse_offs.x = mouse_offs.y = 0;	//disable mouse tracking
	MouseObj = 0L;
	raise();
	CurrWidget = this;
	if(x() || y()) {
		CurrWidgetPos.x = x();		CurrWidgetPos.y = y();
		}
}

void
DlgWidget::focusOutEvent(QFocusEvent *)
{
	HideTextCursor();
	if(dlg) dlg->Command(CMD_ENDDIALOG, 0L, OutputClass);
}

void
DlgWidget::closeEvent(QCloseEvent *e)
{
	HideTextCursor();
	mouse_offs.x = mouse_offs.y = 0;	//disable mouse tracking
	MouseObj = 0L;
	e->ignore();
	if(dlg){
		dlg->Command(CMD_UNLOCK, 0L, OutputClass);
		dlg->Command(CMD_ENDDIALOG, 0L, OutputClass);
		}
}

void
DlgWidget::timerEvent(QTimerEvent *event)
{
	if(!Notary->IsValidPtr(dlg) || !Notary->IsValidPtr(OutputClass)) return;
	if(dlg) dlg->Command(CMD_ENDDIALOG, dlg, OutputClass);
	event->accept();
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void *CreateDlgWnd(char *title, int x, int y, int width, int height, tag_DlgObj *d, DWORD flags)
{
	DlgWidget *w;
	QWidget *pw;
	OutputQT *o;
	int dw, dh;

	if(flags & 0x1) {
		w = new DlgWidget(MainWidget, 0, d, flags);
		}
	else {
		w = new DlgWidget(pw = QAppl->activeWindow(), 0, d, flags);
		w->setWindowTitle(title);
		}
	if(flags & 0x2) w->setFixedSize(width, height);
	else w->setFixedSize(width-6, height-16);
	o = new OutputQT(w);
	o->Erase(0x00e0e0e0L);	if(flags & 0x08) w->startTimer(100);
	if(flags & 0x1) {
		desk_width = dw = w->screen()->geometry().right();
		desk_height = dh = w->screen()->geometry().height();
		w->move((dw>>1) - ((w->width())>>1), (dh>>1) - ((w->height())>>1));
		}
	else if(pw) {
		if(pw->x() || pw->y()) {
			x += pw->x();			y += pw->y();
			}
		else {
			x += CurrWidgetPos.x;	y += CurrWidgetPos.y;
			}
		w->move(x, y);
		}
	d->DoPlot(o);			w->show();
	((DlgRoot*)d)->hDialog = w;
	w->setWindowFlags(w->windowFlags()|Qt::WindowStaysOnTopHint);
	w->activateWindow();		w->raise();
	QAppl->processEvents();
	return w;
}

void CloseDlgWnd(void *hDlg)
{
	HideCopyMark();
	NoWaitDlgLoop();
	if(hDlg) {
		delete((DlgWidget*) hDlg);
		if(CurrWidgetPos.x > 50 && CurrWidgetPos.y > 50) {
			CurrWidgetPos.x -= 50;		CurrWidgetPos.y -= 50;
			}
		}
}

void ShowDlgWnd(void *hDlg)
{
	if(hDlg){
		((DlgWidget*)hDlg)->show();
		((DlgWidget*)hDlg)->activateWindow ();
		((DlgWidget*)hDlg)->raise();
		((DlgWidget*)hDlg)->setFocus();
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// OS independent interface to Qt specific classes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
anyOutput *NewDispClass(GraphObj *g)
{
	Notary->ValPtr(g, true);	//make sure g is a valid object
	return new OutputQT(g);
}

bool DelDispClass(anyOutput *w)
{
	HideTextCursor();
	if(Notary->IsValidPtr(w)) delete ((OutputQT*)w);
	return true;
}

anyOutput *NewBitmapClass(int w, int h, double hr, double vr)
{
	return new BitMapQT(w, h, hr, vr);
	return 0L;
}

bool DelBitmapClass(anyOutput *w)
{
	HideTextCursor();
	if (Notary->IsValidPtr(w)) delete((BitMapQT*)w);
	return true;
}


