//FileIO.cpp, Copyright (c) 2001-2025 R.Lackner
//read/write graphic objects
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#include "rlplot.h"
#include "rlp_strings.h"
#include "Version.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags

#ifdef _WINDOWS
#include <io.h>					//for read/write
#else
#define O_BINARY 0x0
#include <unistd.h>
#endif

extern GraphObj *CurrGO;			//Selected Graphic Objects
extern def_vars defs;
extern int dlgtxtheight;
extern char TmpTxt[];
extern int cPlots;
GraphObj *LastOpenGO;

notary *Notary = 0L;
static ReadCache *Cache = 0L;

unsigned long cObsW;				//count objects written

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// graphic input/output is driven by tables based on the descIO template
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
typedef struct {
	char *label;
	unsigned short type;
	void *ptr;
	long *count;
	}descIO;

//the type member of descIO describes the following data types pointed to by ptr
enum {
	typNONE, typNZINT, typINT, typLONG, typNZLONG, typLFLOAT, typNZLFLOAT,
	typDWORD, typFRECT, typNZLFPOINT, typLFPOINT, typPOINT3D, typIPOINT3D,
	typAXDEF, typPTRAXDEF, typLINEDEF, typFILLDEF, typGOBJ,	typOBJLST,
	typFPLST, typFPLST3D, typIPLST, typTEXT, typTXTDEF, typPTRTXTDEF, 
	typLAST = 0x100};

static char *ptr =0L;
static long cbOut, sizeOut = 0;
static int iFile = -1;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// output graph to file, elementary functions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static int OpenOutputFile(char *file)
{
	time_t ti;

	if(ptr) free(ptr);
	ptr = 0L;		cbOut = 0;		ti = time(0L);
	if(file && BackupFile(file)) {
#ifdef USE_WIN_SECURE
		if(_sopen_s(&iFile, file, O_RDWR | O_BINARY | O_CREAT | O_TRUNC, 
			0x40, S_IWRITE) || iFile < 0){
			ErrorBox((char*)"Open failed for output file");
			return -1;
			}
#else
		if(-1 ==(iFile = open(file, O_RDWR | O_BINARY | O_CREAT | O_TRUNC,
			S_IWRITE | S_IREAD))){
			ErrorBox((char*)"Open failed for output file");
			return -1;
			}
#endif
		ptr = (char *)malloc(sizeOut = 2000);
		if(!ptr) goto openerr;
		cbOut = rlp_strcpy(ptr, 20, (char*)";RLP 1.0\n;File \"");
		add_to_buff(&ptr, &cbOut, &sizeOut, file, 0);
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\" created by RLPlot version ", 0);
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)SZ_VERSION, 0);
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)" for ", 5);
#ifdef _WINDOWS
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"Windows", 7);
#else
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"Linux", 5);
#endif
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n;Date/Time: ", 13);
#ifdef USE_WIN_SECURE
		ctime_s(ptr+cbOut, 30, &ti);	cbOut += 25;
#else
		add_to_buff(&ptr, &cbOut, &sizeOut, ctime(&ti), 25);
#endif
		}
	return iFile;
openerr:
	ptr = 0L;	cbOut = sizeOut = 0;
#ifdef USE_WIN_SECURE
	if(iFile >=0) _close(iFile);
#else
	if(iFile >=0) close(iFile);
#endif
	return iFile = -1;
}

static void CloseOutputFile()
{
	int cb = 0;

	if(iFile >= 0){
#ifdef USE_WIN_SECURE
		if(cbOut) cb = _write(iFile, ptr, cbOut);
		_close(iFile);
#else
		if(cbOut) cb = write(iFile, ptr, cbOut);
		close(iFile);
#endif
		}
	cb = cb;
	if(ptr) free(ptr);
	cbOut = sizeOut = 0;
	ptr = 0L;		iFile = -1;
}

static void WriteTypObjLst(GraphObj **obs, long c1)
{
	long i, j, no, n, *idx;
	
	if(!obs || !(idx=(long*)malloc((c1+2)*sizeof(long)))) return;
	for(i = no = 0; i < c1; i++) {
		j = Notary->RegisterGO(obs[i]);
		if(j) idx[no++] = j;
		}
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"(", 1);
	add_long_to_buff(&ptr, &cbOut, &sizeOut, no, false);
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"){", 2);
	for(i = 0; i < no; i += 16) {
		if(i) add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"   ", 3);
		for(j = 0; (n=i+j) < no && j < 16; j++) {
			add_long_to_buff(&ptr, &cbOut, &sizeOut, idx[n], j != 0);
			}
		if(n >= no) add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"}", 1);
		else add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
		}
}

static void WriteTypIpLst(POINT *ppt, int count)
{
	long i, j, c, n;

	if(!ppt) return;
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"(", 1);
	add_int_to_buff(&ptr, &cbOut, &sizeOut, count, false);
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"){", 2);
	for(i = 0; i < count; i += 8) {
		for(j = c = 0; (n = i+j) <count && j < 8; j++) {
			add_int_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].x, (j != 0));
			add_int_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].y, true);
			}
		if(n >= count) add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"}", 1);
		else add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
		}
}

static void WriteTypFpLst(lfPOINT *ppt, long count, bool bPar)
{
	int i, j, n;

	if (bPar){
		if(!ppt) return;
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"(", 1);
		add_int_to_buff(&ptr, &cbOut, &sizeOut, count, false);
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"){", 2);
		}
	else {
		if(!ppt) count = 0;
		add_int_to_buff(&ptr, &cbOut, &sizeOut, count, true);
		if(!count) {
			add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
			return;
			}
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)" {", 2);
		}
	for(i = 0; i < count; i += 8) {
		for(j = 0; (n = i+j) <count && j < 8; j++) {
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].fx, (j != 0));
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].fy, true);
			}
		if(n >= count) add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"}", 1);
		else add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
		}
}

static void WriteTypFpLst3D(fPOINT3D *ppt, int count)
{
	long i, j, c, n;

	if(!ppt) return;
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"(", 1);
	add_int_to_buff(&ptr, &cbOut, &sizeOut, count, false);
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"){", 2);
	for(i = 0; i < count; i +=5) {
		for(j = c = 0; (n =i+j) <count && j < 5; j++) {
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].fx, (j != 0));
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].fy, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ppt[n].fz, true);
			}
		if(n >= count) add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"}", 1);
		else add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
		}
}

static char * esc_str = 0L;
static int esc_str_size = 0;
static void WriteEscString(char *txt)
{
	int i, j, l, lim = 60;

	if(!txt || !txt[0]) return;
	l = rlp_strlen(txt);
	if((l+10) > esc_str_size) if(!(esc_str = (char*)realloc(esc_str, esc_str_size = (l+100))))return;
	j = 0;	esc_str[j++] = '"';
	for(i = 0; txt[i]; i++) {
		switch(txt[i]) {
		case '\\':
			esc_str[j++] = '\\';	esc_str[j++] = '\\';
			break;
		case '\n':
			esc_str[j++] = '\\';	esc_str[j++] = 'n';
			break;
		default:	
			if(((unsigned char*)txt)[i] >= ' ') esc_str[j++] = txt[i];
			}
		if(j > (esc_str_size -10)) esc_str = (char*)realloc(esc_str, (esc_str_size += 100));
		if(j > lim && (l-i) > 3) {
			esc_str[j++] = '"';		esc_str[j++] = '\\';
			esc_str[j++] = '\n';	esc_str[j++] = ' ';
			esc_str[j++] = ' ';		esc_str[j++] = ' ';
			esc_str[j++] = '"';
			lim += 60;
			}
		}
	esc_str[j++] = '"';
	add_to_buff(&ptr, &cbOut, &sizeOut, esc_str, j);
}

bool ExecOutput(int id, char *Class, descIO *Desc)
{
	int i, last;
	fRECT *fr;
	AxisDEF *ax;
	LineDEF *ld;
	FillDEF *fd;
	TextDEF *tx;

	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n[", 2);
	add_int_to_buff(&ptr, &cbOut, &sizeOut, id, false);
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"=", 1);
	add_to_buff(&ptr, &cbOut, &sizeOut, Class, 0);
	add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"]\n", 2);
	cObsW++;
	for(i = 0; Desc[i].label; i++) {
		if(ptr[cbOut-1] != '\n') add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
		last = cbOut;
		add_to_buff(&ptr, &cbOut, &sizeOut, Desc[i].label, 0);
		add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"=", 1);
		switch(Desc[i].type & 0xff){
		case typNZINT:
			if(!(*(int*)Desc[i].ptr)) {
				cbOut = last;			break;
				}
			//if not zero value continue as if int
			add_int_to_buff(&ptr, &cbOut, &sizeOut, *(int*)Desc[i].ptr, true);
			break;
		case typINT:
			add_int_to_buff(&ptr, &cbOut, &sizeOut, *(int*)Desc[i].ptr, true);
			break;
		case typNZLONG:
			if(!(*(long*)Desc[i].ptr)) {
				cbOut = last;			break;
				}
			//if not zero value ccontinue as if long
			add_long_to_buff(&ptr, &cbOut, &sizeOut, *(long*)Desc[i].ptr, true);
			break;
		case typLONG:
			add_long_to_buff(&ptr, &cbOut, &sizeOut, *(long*)Desc[i].ptr, true);
			break;
		case typNZLFLOAT:
			if(*((double*)Desc[i].ptr) == 0.0) {
				cbOut = last;			break;
				}
			//if not zero or negative continue as if float
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, *(double*)Desc[i].ptr, true);
			break;
		case typLFLOAT:
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, *(double*)Desc[i].ptr, true);
			break;
		case typDWORD:
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, *(DWORD *)Desc[i].ptr, true);
			break;
		case typFRECT:
			fr = (fRECT*)Desc[i].ptr;
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, fr->Xmin, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, fr->Ymax, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, fr->Xmax, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, fr->Ymin, true);
			break;
		case typNZLFPOINT:
			if(((lfPOINT *)Desc[i].ptr)->fx == ((lfPOINT *)Desc[i].ptr)->fy &&
				((lfPOINT *)Desc[i].ptr)->fx == 0.0){
				cbOut = last;			break;
				}
			//if not zero continue as if fPOINT
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((lfPOINT *)Desc[i].ptr)->fx, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((lfPOINT *)Desc[i].ptr)->fy, true);
			break;
		case typLFPOINT:
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((lfPOINT *)Desc[i].ptr)->fx, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((lfPOINT *)Desc[i].ptr)->fy, true);
			break;
		case typPOINT3D:
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((fPOINT3D *)Desc[i].ptr)->fx, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((fPOINT3D *)Desc[i].ptr)->fy, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ((fPOINT3D *)Desc[i].ptr)->fz, true);
			break;
		case typIPOINT3D:
			add_long_to_buff(&ptr, &cbOut, &sizeOut, ((POINT3D *)Desc[i].ptr)->x, true);
			add_long_to_buff(&ptr, &cbOut, &sizeOut, ((POINT3D *)Desc[i].ptr)->y, true);
			add_long_to_buff(&ptr, &cbOut, &sizeOut, ((POINT3D *)Desc[i].ptr)->z, true);
			break;
		case typAXDEF:		case typPTRAXDEF:
			ax = (Desc[i].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[i].ptr : *(AxisDEF **)Desc[i].ptr;
			//we do not set ownership: reconstruct after read
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, ax->flags, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->min, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->max, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->loc[0].fx, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->loc[0].fy, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->loc[0].fz, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->loc[1].fx, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->loc[1].fy, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->loc[1].fz, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->Start, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->Step, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->Center.fx, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->Center.fy, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ax->Radius, true);
			WriteTypFpLst(ax->breaks, ax->nBreaks, false);
			break;
		case typLINEDEF:
			ld = (LineDEF *)Desc[i].ptr;
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ld->width, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, ld->patlength, true);
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, ld->color, true);
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, ld->pattern, true);
			break;
		case typFILLDEF:
			fd = (FillDEF *)Desc[i].ptr;
			//we set the 'hatch' member to zero: reconstruct after read
			add_int_to_buff(&ptr, &cbOut, &sizeOut, fd->type, true);
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, fd->color, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, fd->scale, true);
			add_to_buff(&ptr, &cbOut, &sizeOut, (char*)" 0x0", 4);
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, fd->color2, true);
			break;
		case typGOBJ:
			if(*(GraphObj **)(Desc[i].ptr)) add_int_to_buff(&ptr, &cbOut, &sizeOut, 
				Notary->RegisterGO(*(GraphObj **)(Desc[i].ptr)), true);
			else cbOut = last;
			break;
		case typOBJLST:
			if(!(*(GraphObj ***)(Desc[i].ptr)) || !(*Desc[i].count)){
				cbOut = last;				break;
				}
			WriteTypObjLst(*(GraphObj ***)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0);
			break;
		case typIPLST:
			if(!(*(POINT**)(Desc[i].ptr)) || !(*Desc[i].count)){
				cbOut = last;				break;
				}
			WriteTypIpLst(*(POINT**)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0L);
			break;
		case typFPLST:
			if(!(*(lfPOINT**)(Desc[i].ptr)) || !(*Desc[i].count)){
				cbOut = last;				break;
				}
			WriteTypFpLst(*(lfPOINT**)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0L, true);
			break;
		case typFPLST3D:
			if(!(*(fPOINT3D**)(Desc[i].ptr)) || !(*Desc[i].count)){
				cbOut = last;				break;
				}
			WriteTypFpLst3D(*(fPOINT3D**)Desc[i].ptr, Desc[i].count ? *Desc[i].count : 0);
			break;
		case typTEXT:
			if(!*(char**)(Desc[i].ptr)) cbOut = last;
			else WriteEscString(*((char**)Desc[i].ptr));
			break;
		case typTXTDEF:		case typPTRTXTDEF:
			tx = (Desc[i].type &0xff) == typTXTDEF ? (TextDEF *)Desc[i].ptr : *(TextDEF **)Desc[i].ptr;
			if(!tx) {
				cbOut = last;				break;
				}
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, tx->ColTxt, true);
			add_hex_to_buff(&ptr, &cbOut, &sizeOut, tx->ColBg, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, tx->fSize, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, tx->RotBL, true);
			add_dbl_to_buff(&ptr, &cbOut, &sizeOut, tx->RotCHAR, true);
			add_int_to_buff(&ptr, &cbOut, &sizeOut, tx->Align, true);
			add_int_to_buff(&ptr, &cbOut, &sizeOut, tx->Mode, true);
			add_int_to_buff(&ptr, &cbOut, &sizeOut, tx->Style, true);
			add_int_to_buff(&ptr, &cbOut, &sizeOut, tx->Font, true);
			if(tx->text && tx->text[0]) {
				add_to_buff(&ptr, &cbOut, &sizeOut, (char*)" \"", 2);
				add_to_buff(&ptr, &cbOut, &sizeOut, (char*)tx->text, 0);
				add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\"\n", 2);
				}
			break;
			}
		if(Desc[i].type & typLAST) break;
		}
	if(ptr[cbOut-1] != '\n') add_to_buff(&ptr, &cbOut, &sizeOut, (char*)"\n", 1);
	return true;
}

void ReadTypIpLst(POINT *ptr, long count, unsigned char *first)
{
	int i, j, k, f[20];

	if(!ptr || !first) return;
#ifdef USE_WIN_SECURE
	k = sscanf_s((char*)first, "%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d", 
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19]);
#else
	k = sscanf((char*)first, "%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d", 
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19]);
#endif
	for(i = 0,  j = 0; j < k && i < count; i++, j += 2) {
		ptr[i].x = f[j];	ptr[i].y = f[j+1];
		}
	while (i < count) {
		if(!Cache->GetInt(&ptr[i].x) || !Cache->GetInt(&ptr[i].y)) return;
		i++;
		}
}

void ReadTypFpLst(lfPOINT *ptr, long count, unsigned char *first)
{
	double f[20];
	int j, k;
	long i;

	if(!ptr || !first) return;
#ifdef USE_WIN_SECURE
	k = sscanf_s((char*)first, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf", 
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19]);
#else
	k = sscanf((char*)first, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf", 
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19]);
#endif
	for(i = 0,  j = 0; j < k && i < count; i++, j += 2) {
		ptr[i].fx = f[j];	ptr[i].fy = f[j+1];
		}
	while (i < count) {
		if(!Cache->GetFloat(&ptr[i].fx) || !Cache->GetFloat(&ptr[i].fy)) return;
		i++;
		}
}

void ReadTypFpLst3D(fPOINT3D *ptr, long count, unsigned char *first)
{
	double f[21];
	int j, k;
	long i;

	if(!ptr || !first) return;
#ifdef USE_WIN_SECURE
	k = sscanf_s((char*)first, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf",
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19], &f[20]);
#else
	k = sscanf((char*)first, "%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf",
		&f[0], &f[1], &f[2], &f[3], &f[4], &f[5], &f[6], &f[7], &f[8], &f[9],
		&f[10], &f[11], &f[12], &f[13], &f[14], &f[15], &f[16], &f[17], &f[18], &f[19], &f[20]);
#endif
	for(i = 0,  j = 0; j < k && i < count; i++, j += 3) {
		ptr[i].fx = f[j];	ptr[i].fy = f[j+1];		ptr[i].fz = f[j+2];
		}
	while (i < count) {
		if(!Cache->GetFloat(&ptr[i].fx) || !Cache->GetFloat(&ptr[i].fy) ||
			!Cache->GetFloat(&ptr[i].fz)) return;
		i++;
		}
}

void TranslateEscChar(char *txt)
{
	int i, j;

	if(txt && txt[0]) {
		for(i = j = 0; txt[i] && i < 900; i++) {
			if(txt[i] == '\\') {
				switch(txt[i+1]) {
				case 'n':
					txt[j++] = 0x0a;	i++;	break;
				case 'r':
					txt[j++] = 0x0d;	i++;	break;
				case '"':	case 0x27:			case '\\':
					txt[j++] = txt[++i];		break;
				default:
					txt[j++] = txt[i];	break;
					}
				}
			else txt[j++] = txt[i];
			}
		txt[j] = 0;
		}
}

void AddLines(char **txt)
{
	char tmp[1000], *ntxt;
	bool mlines;
	int i, j, cb = rlp_strlen(*txt);

	do {
		mlines = false;
		Cache->ReadLine(tmp, sizeof(tmp));
		for(i = rlp_strlen(tmp); i > 0 &&(tmp[i-1] < 33 || (tmp[i-1] == 
			'"' && tmp[i-2] != '\\') ||	(tmp[i-1] == '\\' && 
			(mlines = true))); tmp[--i] = 0);
		for(i = 0; tmp[i] && (tmp[i] < 33 || tmp[i] == '"'); i++);
		TranslateEscChar(tmp);
		if(tmp[0] && (j = rlp_strlen(tmp+i)) && (ntxt = (char*)realloc(*txt, cb + j + 1))) {
			rlp_strcpy(ntxt+cb, j+1, tmp+i);
			cb += j;	*(txt) = ntxt;
			}
		} while (mlines);
}

bool ExecInput(descIO *Desc)
{
	static unsigned char c, tmp1[1000], tmp2[200];
	int i, j, k, l;
	bool match, mlines;
	long il, jl;
	AxisDEF *ax;
	POINT *lp;
	lfPOINT *lfp;
	fPOINT3D *lfp3d;
	LineDEF *ld;
	FillDEF *fd;
	fRECT *fr;
	GraphObj **gobs;
	TextDEF *tx;
	POINT3D *lp3d;

	if(!Desc || !Desc[0].label) return false;
	for(j = k = 0; ; ) {
		do{
			c = Cache->Getc();
			switch (c) {
			case '[':					//next object
			case 0:						//probably eof
				return true;
			case '}':					//a lists hang over
				c = Cache->Getc();
				break;
				}
		} while(c <33);
		for(i = 1, tmp1[0] = c; i < int(sizeof(tmp1)-2) && '=' != (tmp1[i] = Cache->Getc()); i++){
			if(tmp1[i] < 32 && tmp1[i]) i = -1;			//some error conditions
			else if(!tmp1[i] && Cache->eof) return true;
			else if(tmp1[i] == '[') return true;
			}
		if(i < (int)sizeof(tmp1) && i >=0 ) tmp1[i] = 0;
		else return false;
		match = mlines = false;
		do {
			if(0 == strcmp((char*)tmp1, Desc[j].label)) {
				Cache->ReadLine((char*)tmp1, sizeof(tmp1));
				switch(Desc[j].type & 0xff){
				case typNZINT:				case typINT:
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%d", (int*)Desc[j].ptr);
#else
					sscanf((char*)tmp1, "%d", (int*)Desc[j].ptr);
#endif
					break;

				case typNZLONG:				case typLONG:
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%ld", (long*)Desc[j].ptr);
#else
					sscanf((char*)tmp1, "%ld", (long*)Desc[j].ptr);
#endif
					break;

				case typNZLFLOAT:		case typLFLOAT:
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%lf", (double*)Desc[j].ptr);
#else
					sscanf((char*)tmp1, "%lf", (double*)Desc[j].ptr);
#endif
					break;
				case typDWORD:
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%x", (DWORD*)Desc[j].ptr);
#else
					sscanf((char*)tmp1, "%x", (DWORD*)Desc[j].ptr);
#endif
					break;
				case typFRECT:
					fr = (fRECT*) Desc[j].ptr;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%lf%lf%lf%lf", &fr->Xmin, &fr->Ymax, &fr->Xmax, &fr->Ymin);
#else
					sscanf((char*)tmp1, "%lf%lf%lf%lf", &fr->Xmin, &fr->Ymax, &fr->Xmax, &fr->Ymin);
#endif
					break;
				case typNZLFPOINT:		case typLFPOINT:
					lfp = (lfPOINT*) Desc[j].ptr;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%lf%lf", &lfp->fx, &lfp->fy);
#else
					sscanf((char*)tmp1, "%lf%lf", &lfp->fx, &lfp->fy);
#endif
					break;
				case typPOINT3D:
					lfp3d = (fPOINT3D*) Desc[j].ptr;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%lf%lf%lf", &lfp3d->fx, &lfp3d->fy, &lfp3d->fz);
#else
					sscanf((char*)tmp1, "%lf%lf%lf", &lfp3d->fx, &lfp3d->fy, &lfp3d->fz);
#endif
					break;
				case typIPOINT3D:
					lp3d = (POINT3D*)Desc[j].ptr;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%ld%ld%ld", &lp3d->x, &lp3d->y, &lp3d->z);
#else
					sscanf((char*)tmp1, "%ld%ld%ld", &lp3d->x, &lp3d->y, &lp3d->z);
#endif
					break;
				case typPTRAXDEF:
				case typAXDEF:
					ax = (Desc[j].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[j].ptr : *(AxisDEF **)Desc[j].ptr;
					//pointer for typPTRAXDEF and memory allocated by the Axis module!
					if(!ax) break;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%x%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%d", &ax->flags, &ax->min, &ax->max,
						&ax->loc[0].fx,	&ax->loc[0].fy, &ax->loc[0].fz, &ax->loc[1].fx,
						&ax->loc[1].fy, &ax->loc[1].fz, &ax->Start, &ax->Step, &ax->Center.fx, 
						&ax->Center.fy, &ax->Radius, &ax->nBreaks);
#else
					sscanf((char*)tmp1, "%x%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%lf%d", &ax->flags, &ax->min, &ax->max,
						&ax->loc[0].fx,	&ax->loc[0].fy, &ax->loc[0].fz, &ax->loc[1].fx,
						&ax->loc[1].fy, &ax->loc[1].fz, &ax->Start, &ax->Step, &ax->Center.fx, 
						&ax->Center.fy, &ax->Radius, &ax->nBreaks);
#endif
					if(ax->nBreaks) {
						ax->breaks = (lfPOINT*)calloc(ax->nBreaks, sizeof(lfPOINT));
						for(i = 0; tmp1[i] && tmp1[i-1] != '{'; i++);
						if(tmp1[i]) {
							ReadTypFpLst(ax->breaks, ax->nBreaks, (unsigned char*)tmp1+i);
							SortAxisBreaks(ax);
							}
						}
					break;
				case typLINEDEF:
					ld = (LineDEF*) Desc[j].ptr;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1,"%lf%lf%x%x", &ld->width, &ld->patlength, &ld->color, &ld->pattern);
#else
					sscanf((char*)tmp1,"%lf%lf%x%x", &ld->width, &ld->patlength, &ld->color, &ld->pattern);
#endif
					break;
				case typFILLDEF:
					fd = (FillDEF*) Desc[j].ptr;
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%d%x%lf%x%x", &fd->type, &fd->color, &fd->scale, ((unsigned int*)(&fd->hatch)), &fd->color2);
#else
					sscanf((char*)tmp1, "%d%x%lf%x%x", &fd->type, &fd->color, &fd->scale, ((unsigned int*)(&fd->hatch)), &fd->color2);
#endif
					fd->hatch = 0L;
					break;
				case typGOBJ:
					*(GraphObj**)(Desc[j].ptr) = Notary->PopGO(atol((char*)tmp1));
					break;
				case typOBJLST:
					for(i = 0; i < 10 && tmp1[i] != '(' && tmp1[i]; i++);
					if(!tmp1[i]) break;
#ifdef USE_WIN_SECURE
					if(sscanf_s((char*)tmp1+i+1, "%ld", &il) && il) {
#else
					if(sscanf((char*)tmp1+i+1, "%ld", &il) && il) {
#endif
						*Desc[j].count = il;
						if(!*(GraphObj***)(Desc[j].ptr)){
							*(GraphObj***)(Desc[j].ptr) = (GraphObj**)calloc(il, sizeof(GraphObj*));
							}
						if((gobs = *(GraphObj***)(Desc[j].ptr))){
							i += 4;
							while(tmp1[i-1] != '{') i++; 
							while(tmp1[i-1] < 33) i++;
#ifdef USE_WIN_SECURE
							strcat_s((char*)tmp1, 1000, " ");
#else
							strcat((char*)tmp1, " ");
#endif
							for( ;il >0; il--) {
								for(l = 0; l < (int)sizeof(tmp2); ){
									if(tmp1[i]) c = tmp1[i++];
									else if(!(c = Cache->Getc())) break;
									if(c >='0' && c <='9') tmp2[l++] = c;
									else {
										tmp2[l] = 0;
										if(l)break;
										}
									}
#ifdef USE_WIN_SECURE
								sscanf_s((char*)tmp2, "%ld", &jl);
#else
								sscanf((char*)tmp2, "%ld", &jl);
#endif
								*gobs++ = Notary->PopGO(jl);
								if(c == '}') break;
								}
							}
						}	
					break;
				case typIPLST:
					for(i = 0; i < 10 && tmp1[i] != '(' && tmp1[i]; i++);
					if(!tmp1[i]) break;
#ifdef USE_WIN_SECURE
					if(sscanf_s((char*)tmp1+i+1, "%ld", &il) && il) {
#else
					if(sscanf((char*)tmp1+i+1, "%ld", &il) && il) {
#endif
						*Desc[j].count = il;
						if(!*(POINT**)(Desc[j].ptr)){
							*(POINT**)(Desc[j].ptr) = (POINT*)calloc(il, sizeof(POINT));
							}
						if(!(lp = *(POINT**)(Desc[j].ptr)))return false;
						while(tmp1[i-1] != '{') i++; 
						while(tmp1[i-1] < 33) i++;
						ReadTypIpLst(lp, il, (unsigned char*)tmp1+i);
						}
					break;
				case typFPLST:
					for(i = 0; i < 10 && tmp1[i] != '(' && tmp1[i]; i++);
					if(!tmp1[i]) break;
#ifdef USE_WIN_SECURE
					if(sscanf_s((char*)tmp1+i+1, "%ld", &il) && il) {
#else
					if(sscanf((char*)tmp1+i+1, "%ld", &il) && il) {
#endif
						*Desc[j].count = il;
						if(!*(lfPOINT**)(Desc[j].ptr)){
							*(lfPOINT**)(Desc[j].ptr) = (lfPOINT*)calloc(il, sizeof(lfPOINT));
							}
						if(!(lfp = *(lfPOINT**)(Desc[j].ptr)))return false;
						while(tmp1[i-1] != '{') i++; 
						while(tmp1[i-1] < 33) i++;
						ReadTypFpLst(lfp, il, (unsigned char*)tmp1+i);
						}
					break;
				case typFPLST3D:
					for(i = 0; i < 10 && tmp1[i] != '(' && tmp1[i]; i++);
					if(!tmp1[i]) break;
#ifdef USE_WIN_SECURE
					if(sscanf_s((char*)tmp1+i+1, "%ld", &il) && il) {
#else
					if(sscanf((char*)tmp1+i+1, "%ld", &il) && il) {
#endif
						*Desc[j].count = il;
						if(!*(fPOINT3D**)(Desc[j].ptr)){
							*(fPOINT3D**)(Desc[j].ptr) = (fPOINT3D*)calloc(il+2, sizeof(fPOINT3D));
							}
						if(!Desc[j].ptr)return false;
						while(tmp1[i-1] != '{') i++; 
						while(tmp1[i-1] < 33) i++;
						ReadTypFpLst3D(*(fPOINT3D**)(Desc[j].ptr), il, (unsigned char*)tmp1+i);
						}
					break;
				case typTEXT:
					for(i = rlp_strlen((char*)tmp1); i > 0 &&(tmp1[i-1] < 32 || (tmp1[i-1] == 
						'"' && tmp1[i-2] != '\\') ||	(tmp1[i-1] == '\\' && 
						(mlines = true))); tmp1[--i] = 0);
					for(i = 0; tmp1[i] && (tmp1[i] < 33 || tmp1[i] == '"'); i++);
					TranslateEscChar((char*)tmp1);
					if(tmp1[0]){
						if (rlp_strlen(tmp1) > 900)tmp1[900] = 0;
						*(char**)(Desc[j].ptr) = rlp_strdup((char*)(tmp1+i));
						if(mlines) AddLines((char**)(Desc[j].ptr));
						}
					break;
				case typPTRTXTDEF:
				case typTXTDEF:
					tx = (Desc[j].type & 0xff) == typTXTDEF ? (TextDEF *)Desc[j].ptr : *(TextDEF **)Desc[j].ptr;
					if(!tx) {
						if((Desc[j].type & 0xff) == typTXTDEF) break;	//prabably wrong usage of typTXTDEF instad of
																//    typPTRTXTDEF
						tx = *(TextDEF **)(Desc[j].ptr) = (TextDEF*)calloc(1, sizeof(TextDEF));
						if(!tx) return false;					//memory allocation error
						}
#ifdef USE_WIN_SECURE
					sscanf_s((char*)tmp1, "%x%x%lf%lf%lf%d%d%d%d", &tx->ColTxt, &tx->ColBg, 
						&tx->fSize, &tx->RotBL, &tx->RotCHAR,
						&tx->Align, &tx->Mode, &tx->Style, &tx->Font);
#else
					sscanf((char*)tmp1, "%x%x%lf%lf%lf%d%d%d%d", &tx->ColTxt, &tx->ColBg, 
						&tx->fSize, &tx->RotBL, &tx->RotCHAR,
						&tx->Align, &tx->Mode, &tx->Style, &tx->Font);
#endif
					tx->iSize = 0;
					for(i = rlp_strlen(tmp1); i >0 && tmp1[i] != '"'; i--);
					if(i > 0) {
						tmp1[i] = 0;
						for(l = 0; l <i && tmp1[l] != '"' && l < 900; l++);
						if(i && tmp1[l+1]) tx->text = rlp_strdup(tmp1+l+1);
						}
					break;
					}
				match = true;			j++;	k++;
				if(!Desc[j].label || (Desc[j-1].type & typLAST)) 
					j = k = 0;	//rewind: items in file not sorted
				}
			else {
				j++;
				k++;
				if(!Desc[j].label || (Desc[j-1].type & typLAST)) {				//Error:
					if(k > j){						//  item not defined in Desc
						match = true;				//  read parameters,
						Cache->ReadLine((char*)tmp1, sizeof(tmp1));	//   then continue
						}
					j= 0;
					}
				}
			}while(!match);
		}
}

bool SaveGraphAs(GraphObj *g)
{
	char *name = 0L;
	int i;
	bool bRet = true;

	if(!g) {
		ErrorBox((char*)"Output pending or\nno graph.");
		return false;
		}
	cObsW = 0;
	if(g->Id == GO_GRAPH || g->Id == GO_PAGE) {
		if(((Graph*)g)->filename) name = ((Graph*)g)->filename;
		}
	name = SaveGraphAsName(name);
	if (name && Notary) {
		iFile = OpenOutputFile(name);
		if(iFile >=0) {
			if(g && g->FileIO(FILE_WRITE)){
				if(g->Id == GO_GRAPH || g->Id == GO_PAGE) {
					g->Command(CMD_FILENAME, name, 0L);
					}
				for(i = rlp_strlen(name); i >=0 && name[i] != '/' && name[i] != '\\'; i--);
				if(name[i]) i++;
				g->Command(CMD_SETNAME, name+i, 0L);
				g->Command(CMD_UPDHISTORY, 0L, 0L);
				}
			else ErrorBox((char*)"Could not write\ndata to file.");
			}
		else ErrorBox((char*)"Open failed for\noutput file.");
		CloseOutputFile();
		}
	else bRet = false;
	Notary->FreeStack();
	return bRet;
}

char *GraphToMem(GraphObj *g, long *size)
{
	static char *ret;

	if(!g) {
		ErrorBox((char*)"Output pending or\nno graph.");
		return 0L;
		}
	cObsW = 0;				iFile = -1;
	cbOut = sizeOut = 0;	ptr = 0L;
	if (!Notary) Notary = new notary();
	if (Notary) {
		if(g && g->FileIO(FILE_WRITE)){
			//all done
			}
		if(ptr) ptr[cbOut] = 0;
		ret = ptr;						if(size) *size = cbOut;
		iFile = -1;
		cbOut = sizeOut = 0;	ptr = 0L;
		return ret;
		}
	return 0L;
}

void UpdGOfromMem(GraphObj *go, unsigned char *buff)
{
	int i=0;

	if(!go || !buff) return;
	iFile = -1;							cbOut = sizeOut = 0;	ptr = 0L;
	for(i = 0; buff[i] && buff[i] != ']'; i++);
	if(!buff[i])return;
	for(; buff[i] && buff[i] <33; i++);
	if(!buff[i] || i < 4) return;
	Cache = new MemCache(buff + i - 1);
	if(!Cache) return;
	if(!Notary) Notary = new notary();
	if ( Notary && go->Id > GO_UNKNOWN && go->Id < GO_DEFRW) {
		//notary not needed but saver if tree exists
		go->Command(CMD_FLUSH, 0L, 0L);
		go->FileIO(INIT_VARS);			go->FileIO(FILE_READ);
		Notary->FreeStack();
		}
	delete Cache;
	Cache = 0L;
}

bool OpenGraph(GraphObj *root, char *name, unsigned char *mem, bool bPaste)
{
	unsigned char c, tmp[1024];
	char *debug;
	long debg_pos = 0, debg_size = 200;
	long i, id, lid = 0;
	unsigned int hv;
	GraphObj *go;
	scaleINFO sc_info;

	LastOpenGO = 0L;
	if(Cache) {
		ErrorBox((char*)"Output pending:\nRead Error.");
		return false;
		}
	if(!Notary) Notary = new notary();
	if(mem) {
		if(!(Cache = new MemCache(mem))) return false;
		}
	else {
		Cache = new ReadCache();
		if (Cache){
			if (!Cache->Open(name)) {
				Notary->FreeStack();		 delete Cache;
				Cache = 0L;
				ErrorBox((char*)"Error open file");
				return false;
				}
			}
		else return false;
		}
	//DEBUG: skipping header
	do {
		c = Cache->Getc();
		} while(c && c != '[');
	if(!c) goto ReadErr;
	do {
		for(i = 0; i < (long)sizeof(tmp) && c != '=' && c; i++){
		tmp[i] = c = Cache->Getc();
			if(c == '[') i = -1;
			}
		if(!c) goto ReadErr;
		tmp[i] = tmp[i-1] = 0;			id=0;
#ifdef USE_WIN_SECURE
		sscanf_s((char*)tmp, "%ld", &id);
#else
		sscanf((char*)tmp, "%ld", &id);
#endif
		if(!id) goto ReadErr;
		//go to class name
		while((tmp[0] = Cache->Getc())<31 && tmp[0]);
		if(!tmp[0]) goto ReadErr;
		for (i = 1; i < (long)sizeof(tmp) && c != ']' && c; i++) {
			tmp[i] = c = Cache->Getc();
			}
		if(!c) goto ReadErr;
		tmp[i-1] = 0;
		go = 0L;
		hv = HashValue(tmp);
		switch(hv) {
		case 3895:	go = new Axis(FILE_READ);		break;
		case 886:	go = new Bar(FILE_READ);		break;
		case 81384:	go = new Symbol(FILE_READ);		break;
		case 62229:	go = new Bubble(FILE_READ);		break;
		case 948:	go = new Box(FILE_READ);		break;
		case 15411:	go = new Arrow(FILE_READ);		break;
		case 1052406:	go = new ErrorBar(FILE_READ);		break;
		case 324566:	go = new Whisker(FILE_READ);		break;
		case 1031437:	go = new DropLine(FILE_READ);		break;
		case 4839:	go = new Tick(FILE_READ);		break;
		case 16832:	go = new Label(FILE_READ);		break;
		case 1071373:	go = new GridLine(FILE_READ);		break;
		case 963085:	go = new DataLine(FILE_READ);		break;
		case 61662266:	go = new DataPolygon(FILE_READ);	break;
		case 435228:	go = new segment(FILE_READ);		break;
		case 1741325:	go = new polyline(FILE_READ);		break;
		case 435258:	go = new polygon(FILE_READ);		break;
		case 92534:	go = new Bezier(FILE_READ);		break;
		case 6888037:	go = new rectangle(FILE_READ);		break;
		case 1780087:	go = new roundrec(FILE_READ);		break;
		case 78813:	go = new Sphere(FILE_READ);		break;
		case 15463:	go = new Brick(FILE_READ);		break;
		case 69952:	go = new Line3D(FILE_READ);		break;
		case 386257:	go = new ellipse(FILE_READ); break;
		case 95680:	go = new mLabel(FILE_READ);		break;
		case 4819316:	go = new PlotScatt(FILE_READ);		break;
		case 117848:	go = new xyStat(FILE_READ);		break;
		case 15935312:	go = new BubblePlot(FILE_READ);		break;
		case 247376:	go = new BoxPlot(FILE_READ);		break;
		case 317384:	go = new StackBar(FILE_READ);		break;
		case 1205932:	go = new PieChart(FILE_READ);		break;
		case 16664:	go = new Graph(FILE_READ);		break;
		case 25108:	go = new GoGroup(FILE_READ);		break;
		case 300976:	go = new Scatt3D(FILE_READ);		break;
		case 297280:	go = new Plane3D(FILE_READ);		break;
		case 19227098:	go = new Regression(FILE_READ);		break;
		case 297997:	go = new RegLine(FILE_READ);		break;
		case 4318417:	go = new SDellipse(FILE_READ);		break;
		case 4843600:	go = new PolarPlot(FILE_READ);		break;
		case 977452:	go = new DensDisp(FILE_READ);		break;
		case 4465:	go = new Page(FILE_READ);		break;
		case 75120:	go = new Plot3D(FILE_READ);		break;
		case 17142080:	go = new GridLine3D(FILE_READ);		break;
		case 246688:	go = new Arrow3D(FILE_READ);		break;
		case 75562:	go = new Ribbon(FILE_READ);		break;
		case 16503104:	go = new DropLine3D(FILE_READ);		break;
		case 70259:	go = new Limits(FILE_READ);		break;
		case 17145824:	go = new GridRadial(FILE_READ);		break;
		case 1074714:	go = new Function(FILE_READ);		break;
		case 256075:	go = new FitFunc(FILE_READ);		break;
		case 273377:	go = new LegItem(FILE_READ);		break;
		case 1053744:	go = new FreqDist(FILE_READ);		break;
		case 68748:	go = new Legend(FILE_READ);		break;
		case 66800:	go = new Grid3D(FILE_READ);		break;
		case 967843:	go = new DefsRW(FILE_READ);		break;
		case 66848:	go = new Func3D(FILE_READ);		break;
		case 5001225:	go = new TextFrame(FILE_READ);		break;
		case 4743132:	go = new NormQuant(FILE_READ);		break;
		case 64333904:	go = new ContourPlot(FILE_READ);	break;
		case 1040336:	go = new ErrBar3D(FILE_READ);		break;
		case 20774705:	go = new SymTernary(FILE_READ);		break;
		case 71827249:	go = new LineTernary(FILE_READ);	break;
		case 19910878:	go = new TernaryXYZ(FILE_READ);		break;
		case 79643472:	go = new TernaryPlot(FILE_READ);	break;
		case 15679281:	go = new ArrTernary(FILE_READ);		break;
		case 435511770:	go = new myRegression(FILE_READ);	break;
		case 4714597:	go = new Phenology(FILE_READ);		break;
		case 6735952:	go = new myBarPlot(FILE_READ);		break;
		case 269493306:	go = new ErrorPolygon(FILE_READ);	break;
		case 4168:		go = new Grid(FILE_READ);			break;
		case 4867088:	go = new Polygon3D(FILE_READ);		break;
		default:
			if ((debug = (char*)malloc(debg_size))) {
				debg_pos = rlp_strcpy(debug, 20, (char*)" Object ");
				add_int_to_buff(&debug, &debg_pos, &debg_size, id, false);
				add_to_buff(&debug, &debg_pos, &debg_size, (char*)" in file\n (Class = \"", 0);
				add_to_buff(&debug, &debg_pos, &debg_size, (char*)tmp, 0);
				add_to_buff(&debug, &debg_pos, &debg_size, (char*)"\")\n hash #", 0);
				add_int_to_buff(&debug, &debg_pos, &debg_size, hv, false);
				add_to_buff(&debug, &debg_pos, &debg_size, (char*)"\n is unknown.", 0);
				InfoBox(debug);
				free(debug);
				}
			break;
			}
		if(go) {
			if(((int)id) < 0) DeleteGO(go);		//temporary objects have id < 0
			else if(!Notary->PushGO(lid = id, go)) DeleteGO(go);
			}
		if('[' != Cache->Lastc()) do {			//search next object
			c = Cache->Getc();
			} while(c && c != '[');
		tmp[0] = 0;
		}while (c);
	Cache->Close();
	if((LastOpenGO = go = Notary->PopGO(lid))) {
		go->Command(CMD_SET_DATAOBJ, 0L, 0L);
		if (name && name[0]) go->Command(CMD_FILENAME, name, 0L);
		Notary->FreeStack();
		if(bPaste && go->Id == GO_GRAPH) {
			sc_info.sx.fx = -((Graph*)go)->GRect.Xmin;				sc_info.sy.fx = -((Graph*)go)->GRect.Ymin;
			sc_info.sx.fy = sc_info.sy.fy = sc_info.sz.fy = 1.0;	sc_info.sz.fx = 0.0;
			go->Command(CMD_SCALE, &sc_info, 0L);
			sc_info.sx.fy = sc_info.sy.fy = sc_info.sz.fy = 1.0/go->GetSize(SIZE_SCALE);
			sc_info.sx.fx = sc_info.sy.fx = sc_info.sz.fx = 0.0;
			go->Command(CMD_SCALE, &sc_info, 0L);
			}
		if(bPaste && root->Command(CMD_PASTE_OBJ, (void *)go, 0L)) {
			// object accepted
			}
		else if(go->Id < GO_GRAPH
			&& !(root->Command(CMD_DROP_PLOT, (void *)go, 0L))){
			DeleteGO(go);	go = 0L;
			}
		else if(go->Id >= GO_GRAPH 
			&& !(root->Command(CMD_DROP_GRAPH, (void *)go, 0L))){
			DeleteGO(go);	go = 0L;
			}
		//go may not be valid any more at this stage
		}
	delete Cache;					Cache = 0L;
	return true;

ReadErr:
	Cache->Close();	
#ifdef USE_WIN_SECURE
	if(iFile >= 0) _close(iFile);
#else
	if(iFile >= 0) close(iFile);
#endif
	iFile = -1;			Notary->FreeStack();
	delete Cache;		Cache = 0L;
	if(!name || !defs.IniFile || strcmp(name, defs.IniFile))
		ErrorBox((char*)"An error occured during read.");
	return false;
}

bool InitVarsGO(descIO *Desc)
{
	int i;
	AxisDEF *ax;
	TextDEF *tx;

	for(i = 0; Desc[i].label; i++) {
		switch(Desc[i].type & 0xff) {
		case typNZINT:			case typINT:
			*(int*)Desc[i].ptr = 0;			
			break;
		case typNZLONG:			case typLONG:
			*(long*)Desc[i].ptr = 0L;
			break;
		case typNZLFLOAT:		case typLFLOAT:
			*(double*)Desc[i].ptr = 0.0;
			break;
		case typDWORD:
			*(DWORD*)Desc[i].ptr = 0x0L;
			break;
		case typFRECT:
			((fRECT*)Desc[i].ptr)->Xmin = ((fRECT*)Desc[i].ptr)->Xmax =
				((fRECT*)Desc[i].ptr)->Ymin = ((fRECT*)Desc[i].ptr)->Ymax = 0.0;
			break;
		case typNZLFPOINT:		case typLFPOINT:
			((lfPOINT*)Desc[i].ptr)->fx = ((lfPOINT*)Desc[i].ptr)->fy = 0.0;
			break;
		case typPOINT3D:
			((fPOINT3D*)Desc[i].ptr)->fx = ((fPOINT3D*)Desc[i].ptr)->fy =
				((fPOINT3D*)Desc[i].ptr)->fz = 0.0;
			break;
		case typIPOINT3D:
			((POINT3D*)Desc[i].ptr)->x = ((POINT3D*)Desc[i].ptr)->y =
				((POINT3D*)Desc[i].ptr)->z = 0;
			break;
		case typPTRAXDEF:		case typAXDEF:
			ax = (Desc[i].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[i].ptr : *(AxisDEF **)Desc[i].ptr;
			if(!ax) break;
			ax->owner = 0L;		ax->flags = 0L;		ax->breaks = 0L;	ax->nBreaks = 0;
			ax->min = ax->max = ax->Start = ax->Step = 0.0;
			ax->loc[0].fx = ax->loc[0].fy = ax->loc[0].fz = 0.0;
			ax->loc[1].fx = ax->loc[1].fy = ax->loc[1].fz = 0.0;
			ax->Center.fx = ax->Center.fy = ax->Radius = 0.0f;
			break;
		case typLINEDEF:
			((LineDEF*)Desc[i].ptr)->width = defs.GetSize(SIZE_HAIRLINE);
			((LineDEF*)Desc[i].ptr)->patlength = defs.GetSize(SIZE_PATLENGTH);
			((LineDEF*)Desc[i].ptr)->color = ((LineDEF*)Desc[i].ptr)->pattern = 0x0L;
			break;
		case typFILLDEF:
			((FillDEF*)Desc[i].ptr)->type = FILL_NONE;
			((FillDEF*)Desc[i].ptr)->color = 0x00ffffff;
			((FillDEF*)Desc[i].ptr)->scale = 1.0;
			((FillDEF*)Desc[i].ptr)->hatch = (LineDEF*)0L;
			((FillDEF*)Desc[i].ptr)->color2 = 0x00ffffff;
			break;
		case typGOBJ:
			*(GraphObj **)Desc[i].ptr = (GraphObj*)0L;
			break;
		case typOBJLST:
			*Desc[i].count = 0L;
			*(GraphObj ***)Desc[i].ptr = (GraphObj**)0L;
			break;
		case typIPLST:
			*Desc[i].count = 0L;
			*(POINT **)Desc[i].ptr = (POINT*)0L;
			break;
		case typFPLST:
			*Desc[i].count = 0L;
			*(lfPOINT **)Desc[i].ptr = (lfPOINT*)0L;
			break;
		case typFPLST3D:
			*Desc[i].count = 0L;
			*(fPOINT3D **)Desc[i].ptr = (fPOINT3D*)0L;
			break;
		case typTEXT:
			*(char **)Desc[i].ptr = (char*)0L;
			break;
		case typTXTDEF:
			tx = (TextDEF *)Desc[i].ptr;
			tx->ColTxt = 0x0L,			tx->ColBg = 0x00ffffffL;
			tx->fSize = defs.GetSize(SIZE_TEXT);
			tx->RotBL = tx->RotCHAR = 0.0;
			tx->Align = tx->Mode = tx->Style = tx->Font = 0L;
			tx->text = 0L;
			break;
		case typPTRTXTDEF:
			*(TextDEF**)Desc[i].ptr = (TextDEF*)0L;
			break;
			}
		if(Desc[i].type & typLAST) break;
		}
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Save object data to memory block
static unsigned char *SavVarBuf = 0L;
static long SavVarSize = 0,	SavVarPos = 0;
void SavVarInit(long len)
{
	if(SavVarBuf) free(SavVarBuf);
	SavVarBuf = (unsigned char *)malloc(SavVarSize = len+4096);
	SavVarPos = 0;
}

bool SavVarAdd(void *ptr, int size)
{
	int len, psize;

	if(SavVarBuf && size>0) {
		psize = (int)sizeof(unsigned char *);
		len = sizeof(unsigned char *) + sizeof(int) + size;
		while (SavVarSize <= SavVarPos+len*2) {
			SavVarBuf = (unsigned char *)realloc(SavVarBuf, SavVarSize + 4096);
			SavVarSize += 4096;
			}
		memcpy(SavVarBuf + SavVarPos, &ptr, psize);
		SavVarPos += psize;
		memcpy(SavVarBuf + SavVarPos, &size, sizeof(int));
		SavVarPos += sizeof(int);
		if(ptr) {
			memcpy(SavVarBuf+SavVarPos, ptr, size);			SavVarPos += size;
			}
		return true;
		}
	return false;
}

void *SavVarFetch()
{
	static void *tmp;

	SavVarAdd(0L, sizeof(unsigned char*));				
	tmp = SavVarBuf;	SavVarSize = SavVarPos = 0;		SavVarBuf = 0L;
	return tmp;
}

bool SaveVarGO(descIO *Desc)
{
	int i;
	AxisDEF *ax;
	TextDEF *tx;

	for(i = 0; Desc[i].label; i++) {
		switch(Desc[i].type & 0xff){
		case typNZINT:			case typINT:
			SavVarAdd(Desc[i].ptr, sizeof(int));
			break;
		case typNZLONG:			case typLONG:
			SavVarAdd(Desc[i].ptr, sizeof(long));
			break;
		case typNZLFLOAT:		case typLFLOAT:
			SavVarAdd(Desc[i].ptr, sizeof(double));
			break;
		case typDWORD:
			SavVarAdd(Desc[i].ptr, sizeof(DWORD));
			break;
		case typFRECT:
			SavVarAdd(Desc[i].ptr, sizeof(fRECT));
			break;
		case typNZLFPOINT:
		case typLFPOINT:
			SavVarAdd(Desc[i].ptr, sizeof(lfPOINT));
			break;
		case typPOINT3D:
			SavVarAdd(Desc[i].ptr, sizeof(fPOINT3D));
			break;
		case typAXDEF:			case typPTRAXDEF:
			ax = (Desc[i].type & 0xff) == typAXDEF ? (AxisDEF *)Desc[i].ptr : *(AxisDEF **)Desc[i].ptr;
			if(ax) {
				SavVarAdd(ax, sizeof(AxisDEF));
				if(ax->breaks && ax->nBreaks) {
					SavVarAdd(ax->breaks, ax->nBreaks * sizeof(lfPOINT));
					}
				}
			break;
		case typLINEDEF:
			SavVarAdd(Desc[i].ptr, sizeof(LineDEF));
			break;
		case typFILLDEF:
			SavVarAdd(Desc[i].ptr, sizeof(FillDEF));
			break;
		case typGOBJ:	case typOBJLST:		//not supported
			break;
		case typIPLST:						//probably in the future
			break;
		case typFPLST:
			SavVarAdd(Desc[i].count, sizeof(long));
			SavVarAdd(*((void **)Desc[i].ptr), sizeof(lfPOINT) * (*Desc[i].count));
			break;
		case typFPLST3D:
			SavVarAdd(Desc[i].count, sizeof(long));
			SavVarAdd(*((void **)Desc[i].ptr), sizeof(fPOINT3D) * (*Desc[i].count));
			break;
		case typTEXT:
			if(*(char**)(Desc[i].ptr)) SavVarAdd(Desc[i].ptr, rlp_strlen(*(char**)(Desc[i].ptr)));
			break;
		case typTXTDEF:				case typPTRTXTDEF:
			tx = (Desc[i].type &0xff) == typTXTDEF ? (TextDEF *)Desc[i].ptr : *(TextDEF **)Desc[i].ptr;
			if(tx) SavVarAdd(tx, sizeof(TextDEF) - sizeof(char*));
			break;
			}
		if(Desc[i].type & typLAST) break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Graphic object member funtions for IO
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifndef _WINDOWS
#pragma GCC diagnostic ignored "-Wmissing-field-initializers"
#endif
bool
Grid::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Line", typLINEDEF, &line, 0L},
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Spacing", typLAST | typLFLOAT, &spacing, 0L}};

	switch (rw) {
	case INIT_VARS:
		switch (defs.units()) {
		case 1:		spacing = 0.5;		break;
		case 2:		spacing = 0.25;		break;
		default:	spacing = 5.0;		break;
			}
		line.color = 0x00cbcbcb;		line.patlength = 0.0;
		line.pattern = 0;				line.width = defs.GetSize(SIZE_HAIRLINE);
		hidden = true;					name = 0L;
		break;
	case FILE_READ:
		ExecInput(Desc);		hidden = true;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Grid", Desc);
		}
	return false;
}
bool
Symbol::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Idx", typINT, &idx, 0L},
		{(char*)"Pos", typLFPOINT, &fPos, 0L},
		{(char*)"Size", typLFLOAT, &size, 0L},
		{(char*)"Line", typLINEDEF, &SymLine, 0L},
		{(char*)"FillCol", typDWORD, &SymFill.color, 0L},
		{(char*)"Text", typPTRTXTDEF, &SymTxt, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = DefSize(SIZE_SYMBOL);
		SymLine.color = parent ? parent->GetColor(COL_SYM_LINE) : defs.Color(COL_SYM_LINE);
		SymLine.width = parent ? parent->GetSize(SIZE_SYM_LINE) : DefSize(SIZE_SYM_LINE);
		SymFill.type = FILL_NONE;
		SymFill.color = parent ? parent->GetColor(COL_SYM_FILL) : defs.Color(COL_SYM_FILL);
		SymFill.scale = 1.0;		SymFill.hatch = (LineDEF *) 0L;
		last_pos.fx = last_pos.fy = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		SymFill.hatch = (LineDEF *) 0L;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Symbol", Desc);
		}
	return false;
}

bool
SymTernary::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Idx", typINT, &idx, 0L},
		{(char*)"Pos", typLFPOINT, &fPos, 0L},
		{(char*)"Pos3D", typPOINT3D, &fPos3D, 0L},
		{(char*)"Size", typLFLOAT, &size, 0L},
		{(char*)"Line", typLINEDEF, &SymLine, 0L},
		{(char*)"FillCol", typDWORD, &SymFill.color, 0L},
		{(char*)"Text", typPTRTXTDEF, &SymTxt, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = DefSize(SIZE_SYMBOL);
		SymLine.color = parent ? parent->GetColor(COL_SYM_LINE) : defs.Color(COL_SYM_LINE);
		SymLine.width = parent ? parent->GetSize(SIZE_SYM_LINE) : DefSize(SIZE_SYM_LINE);
		SymFill.type = FILL_NONE;
		SymFill.color = parent ? parent->GetColor(COL_SYM_FILL) : defs.Color(COL_SYM_FILL);
		SymFill.scale = 1.0f;
		SymFill.hatch = (LineDEF *) 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		SymFill.hatch = (LineDEF *) 0L;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"SymTernary", Desc);
		}
	return false;
}

bool
Bubble::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Pos", typLFPOINT, &fPos, 0L},
		{(char*)"Size", typLFLOAT, &fs, 0L},
		{(char*)"Line", typLINEDEF, &BubbleLine, 0L},
		{(char*)"FillLine", typLINEDEF, &BubbleFillLine, 0L},
		{(char*)"Fill", typFILLDEF, &BubbleFill, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		BubbleLine.width = DefSize(SIZE_BUBBLE_LINE);
		BubbleFillLine.width = DefSize(SIZE_BUBBLE_HATCH_LINE);
		BubbleLine.color = BubbleFillLine.color = defs.Color(COL_BUBBLE_LINE);
		BubbleFill.color = defs.Color(COL_BUBBLE_FILL);
		ssRef = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		BubbleFill.hatch = &BubbleFillLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Bubble", Desc);
		}
	return false;
}

bool
Bar::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Pos", typLFPOINT, &fPos, 0L},
		{(char*)"Size", typLFLOAT, &size, 0L},
		{(char*)"Org", typLFPOINT, &BarBase, 0L},
		{(char*)"Line", typLINEDEF, &BarLine, 0L},
		{(char*)"Fill", typFILLDEF, &BarFill, 0L},
		{(char*)"FillLine", typLINEDEF, &HatchLine, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		type = BAR_VERTB;
		memcpy(&BarFill, defs.GetFill(defs.units()), sizeof(FillDEF));
		if(BarFill.hatch) memcpy(&HatchLine, BarFill.hatch, sizeof(LineDEF));
		BarFill.hatch = &HatchLine;
		memcpy(&BarLine, defs.GetOutLine(), sizeof(LineDEF));
		size = DefSize(SIZE_BAR);
		mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		BarFill.hatch = &HatchLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Bar", Desc);
		}
	return false;
}

void
DataLine::FileValues(char *name, int type, double start, double step)
{
	FILE *file;
	int i = 0, c;
	double fx;
	lfPOINT *tfp;

#ifdef USE_WIN_SECURE
	if(fopen_s(&file, name, "r")) {
		sprintf_s(TmpTxt, TMP_TXT_SIZE, "DataLine: open failed for file \"%s\"", name);
#else
	if(!(file = fopen(name, "r"))) {
		sprintf(TmpTxt, "DataLine: open failed for file \"%s\"", name);
#endif
		ErrorBox(TmpTxt);
		return;
		}
	if(Values) free(Values);
	if(!(Values = (lfPOINT*)calloc( nPnt = 1000, sizeof(lfPOINT)))){
		fclose(file);
		return;
		}
	switch(type) {
	case 1:				//x and y values
		i = 0;
		do {
#ifdef USE_WIN_SECURE
			c = fscanf_s(file, "%lf%lf", &Values[i].fx, &Values[i].fy);
#else
			c = fscanf(file, "%lf%lf", &Values[i].fx, &Values[i].fy);
#endif
			i++;
			if(i >= nPnt &&(tfp = (lfPOINT*)realloc(Values, (nPnt+1000)*sizeof(lfPOINT)))){
				Values = tfp;			nPnt += 1000;
				}
			else if(i >= nPnt) break;
			}while(c == 2);
		i--;
		break;
	case 2:				//only y values
		i = 0;	fx = start;
		do {
#ifdef USE_WIN_SECURE
			c = fscanf_s(file, "%lf", &Values[i].fy);
#else
			c = fscanf(file, "%lf", &Values[i].fy);
#endif
			Values[i].fx = fx;
			i++;	fx += step;
			if(i >= nPnt &&(tfp = (lfPOINT*)realloc(Values, (nPnt+1000)*sizeof(lfPOINT)))){
				Values = tfp;			nPnt += 1000;
				}
			}while(c == 1);
		i--;
		break;
		}
	nPnt = i;
	fclose(file);
}

bool
DataLine::FileIO(int rw)
{
	char *file1 = 0L;
	char *file2 = 0L;
	double Start = 0.0f, Step = 0.0f;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssXref", typTEXT, &ssXref, 0L},
		{(char*)"ssYref", typTEXT, &ssYref, 0L},
		{(char*)"cond", typTEXT, &cond, 0L},
		{(char*)"BgCol", typDWORD, &BgColor, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"Data", typFPLST, &Values, &nPnt},
		{(char*)"file_xy", typTEXT, &file2, 0L},
		{(char*)"start_x", typNZLFLOAT, &Start, 0L},
		{(char*)"step_x", typNZLFLOAT, &Step, 0L},
		{(char*)"file_y", typTEXT, &file1, 0L},
		{(char*)"Desc", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		isPolygon = false;
		nPnt = cp = 0;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		BgColor = defs.Color(COL_BG);
		pts = 0L;		dirty = true;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = max.fx = max.fy = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(file2)FileValues(file2, 1, 0.0, 1.0);
		else if(file1)FileValues(file1, 2, Start, fabs(Step) > defs.min4log ? Step : 1.0);
		if(file1) free(file1);
		if(file2) free(file2);
		if(nPnt && Values)return true;
		break;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"DataLine", Desc);
		}
	return false;
}

bool
DataPolygon::FileIO(int rw)
{
	char *file1 = 0L;
	char *file2 = 0L;
	double Start = 0.0f, Step = 0.0f;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssXref", typTEXT, &ssXref, 0L},
		{(char*)"ssYref", typTEXT, &ssYref, 0L},
		{(char*)"BgCol", typDWORD, &BgColor, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"FillLine", typLINEDEF, &pgFillLine, 0L},
		{(char*)"Fill", typFILLDEF, &pgFill, 0L},
		{(char*)"Data", typFPLST, &Values, &nPnt},
		{(char*)"n_src", typNZLONG, &src_nPnt, 0L},
		{(char*)"file_xy", typTEXT, &file2, 0L},
		{(char*)"start_x", typNZLFLOAT, &Start, 0L},
		{(char*)"step_x", typNZLFLOAT, &Step, 0L},
		{(char*)"file_y", typLAST | typTEXT, &file1, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		isPolygon = true;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		LineDef.pattern = 0L;
		BgColor = defs.Color(COL_BG);
		memcpy(&pgFill, defs.GetFill(defs.units()), sizeof(FillDEF));
		if(pgFill.hatch) memcpy(&pgFillLine, pgFill.hatch, sizeof(LineDEF));
		pgFill.hatch = &pgFillLine;		dirty = true;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = max.fx = max.fy = 0.0;
		if (!src_nPnt) src_nPnt = nPnt;
		area = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(file2)FileValues(file2, 1, 0.0f, 1.0f);
		else if(file1)FileValues(file1, 2, Start, fabs(Step) > defs.min4log ? Step : 1.0);
		if(file1) free(file1);
		if(file2) free(file2);
		pgFill.hatch = &pgFillLine;
		if(nPnt && Values)return true;
		break;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"DataPolygon", Desc);
		}
	return false;
}

bool
ErrorPolygon::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Type", typNZLONG, &type, 0L },
		{ (char*)"ssXref", typTEXT, &ssXref, 0L },
		{ (char*)"ssYref", typTEXT, &ssYref, 0L },
		{ (char*)"ssErrRange", typTEXT, &ErrRange, 0L },
		{ (char*)"ssLoRange", typTEXT, &LoRange, 0L },
		{ (char*)"ssHiRange", typTEXT, &HiRange, 0L },
		{ (char*)"BgCol", typDWORD, &BgColor, 0L },
		{ (char*)"Line", typLINEDEF, &LineDef, 0L },
		{ (char*)"FillLine", typLINEDEF, &pgFillLine, 0L },
		{ (char*)"Fill", typFILLDEF, &pgFill, 0L },
		{ (char*)"Data", typFPLST | typLAST, &Values, &nPnt } };

	switch (rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);		isPolygon = true;		ErrRange = 0L;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		LineDef.width = defs.GetSize(SIZE_HAIRLINE) / 2.0;
		LineDef.pattern = 0L;		BgColor = defs.Color(COL_BG);
		memcpy(&pgFill, defs.GetFill(defs.units()), sizeof(FillDEF));
		if (pgFill.hatch) memcpy(&pgFillLine, pgFill.hatch, sizeof(LineDEF));
		pgFill.color = pgFill.color2 = 0xa4808080;
		LineDef.color = 0xff808080;
		pgFill.hatch = &pgFillLine;		dirty = true;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = max.fx = max.fy = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		pgFill.hatch = &pgFillLine;
		if (nPnt && Values)return true;
		break;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"ErrorPolygon", Desc);
	}
	return false;
}

bool
LineTernary::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"ssref1", typTEXT, &range1, 0L },
		{ (char*)"ssref2", typTEXT, &range2, 0L },
		{ (char*)"ssref3", typTEXT, &range3, 0L },
		{ (char*)"Line", typLINEDEF, &LineDef, 0L },
		{ (char*)"FillLine", typLINEDEF, &pgFillLine, 0L },
		{ (char*)"Fill", typFILLDEF, &pgFill, 0L },
		{ (char*)"values", typLAST | typFPLST3D, &values_xyz, &nValues } };

	switch (rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		memcpy(&Fill, defs.GetFill(defs.units()), sizeof(FillDEF));
		if (Fill.hatch) {
			memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
			}
		Fill.hatch = &FillLine;
		DispLine = NULL;		CurrDisp = NULL;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Fill.hatch = &FillLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"LineTernary", Desc);
		}
	return false;
}

bool
RegLine::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"type", typNZLONG, &type, 0L},
		{(char*)"nPoints", typINT, &nPoints, 0L},
		{(char*)"BgCol", typDWORD, &BgColor, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"Range", typFRECT, &lim, 0L},
		{(char*)"uClip", typFRECT, &uclip, 0L},
		{(char*)"mx", typNZLFLOAT, &mx, 0L},
		{(char*)"my", typNZLFLOAT, &my, 0L},
		{(char*)"li1", typLFPOINT, &l1, 0L},
		{(char*)"li2", typLFPOINT, &l2, 0L},
		{(char*)"li3", typLFPOINT, &l3, 0L},
		{(char*)"li4", typLFPOINT, &l4, 0L},
		{(char*)"li5", typLAST | typLFPOINT, &l5, 0L}};

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		cp = 0;
		memcpy(&LineDef, defs.GetLine(), sizeof(LineDEF));
		BgColor = defs.Color(COL_BG);
		pts = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"RegLine", Desc);
		}
	return false;
}

void
SDellipse::RegGO(void *n)
{
	if(n) {
		if(rl)rl->RegGO(n);
		((notary*)n)->AddRegGO(this);
	}
}

bool
SDellipse::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"type", typNZLONG, &type, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"Range", typFRECT, &lim, 0L},
		{(char*)"Regr", typGOBJ, &rl, 0L},
		{(char*)"Data", typLAST | typFPLST, &val, &nPoints}};

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&LineDef, defs.GetOutLine(), sizeof(LineDEF));
		pts = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(rl) rl->parent = this;
		return true;
	case FILE_WRITE:
		if(rl) rl->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"SDellipse", Desc);
		}
	return false;
}

bool
ErrorBar::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Pos", typLFPOINT, &fPos, 0L},
		{(char*)"Err", typLFLOAT, &ferr, 0L},
		{(char*)"Size", typLFLOAT, &SizeBar, 0L},
		{(char*)"Line", typLINEDEF, &ErrLine, 0L},
		{(char*)"Desc", typLAST | typTEXT, &name, 0L}};

	descIO DescSavVar[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Size", typLFLOAT, &SizeBar, 0L},
		{(char*)"Line", typLAST | typLINEDEF, &ErrLine, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(DescSavVar);
	case INIT_VARS:
		InitVarsGO(Desc);
		SizeBar = DefSize(SIZE_ERRBAR);
		ErrLine.width = DefSize(SIZE_ERRBAR_LINE);
		ErrLine.color = defs.Color(COL_SYM_LINE);
		mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"ErrorBar", Desc);
		}
	return false;
}

bool
ErrBar3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Pos", typPOINT3D, &fPos, 0L},
		{(char*)"Err", typLFLOAT, &ferr, 0L},
		{(char*)"Size", typLFLOAT, &SizeBar, 0L},
		{(char*)"Line", typLINEDEF, &ErrLine, 0L},
		{(char*)"Desc", typLAST | typTEXT, &name, 0L}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		SizeBar = DefSize(SIZE_ERRBAR);
		ErrLine.width = DefSize(SIZE_ERRBAR_LINE);
		ErrLine.color = defs.Color(COL_SYM_LINE);
		mo = 0L;	mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		for(i = 0; i < 3; i++) ls[i] = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"ErrBar3D", Desc);
		}
	return false;
}

bool
Arrow::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"p1", typLFPOINT, &pos1, 0L},
		{(char*)"p2", typLFPOINT, &pos2, 0L},
		{(char*)"CapW", typLFLOAT, &cw, 0L},
		{(char*)"CapL", typLFLOAT, &cl, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};


	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cw = DefSize(SIZE_ARROW_CAPWIDTH);
		cl = DefSize(SIZE_ARROW_CAPLENGTH);
		LineDef.color = parent ? parent->GetColor(COL_DATA_LINE) : defs.Color(COL_DATA_LINE);
		LineDef.width = parent ? parent->GetSize(SIZE_ARROW_LINE) : DefSize(SIZE_ARROW_LINE);
		type = ARROW_LINE;		dh1 = dh2 = 0L;		mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Arrow", Desc);
		}
	return false;
}

bool
ArrTernary::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef1", typIPOINT3D, &ssRef[0], 0L},
		{ (char*)"ssRef2", typIPOINT3D, &ssRef[1], 0L },
		{ (char*)"ssRef3", typIPOINT3D, &ssRef[2], 0L },
		{ (char*)"ssRef4", typIPOINT3D, &ssRef[3], 0L },
		{ (char*)"moveable", typNZINT, &moveable, 0L },
		{(char*)"p1", typPOINT3D, &pos1, 0L},
		{(char*)"p2", typPOINT3D, &pos2, 0L},
		{(char*)"CapW", typLFLOAT, &cw, 0L},
		{(char*)"CapL", typLFLOAT, &cl, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch (rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cw = DefSize(SIZE_ARROW_CAPWIDTH);
		cl = DefSize(SIZE_ARROW_CAPLENGTH);
		LineDef.color = parent ? parent->GetColor(COL_DATA_LINE) : defs.Color(COL_DATA_LINE);
		LineDef.width = parent ? parent->GetSize(SIZE_ARROW_LINE) : DefSize(SIZE_ARROW_LINE);
		type = ARROW_LINE;		dh1 = dh2 = 0L;		mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		ssRef[0].x = ssRef[0].y = ssRef[0].z = ssRef[1].x = ssRef[1].y = ssRef[1].z = -1;
		ssRef[2].x = ssRef[2].y = ssRef[2].z = ssRef[3].x = ssRef[3].y = ssRef[3].z = -1;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"ArrTernary", Desc);
	}
	return false;
}

bool
Box::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"High", typLFPOINT, &pos1, 0L},
		{(char*)"Low", typLFPOINT, &pos2, 0L},
		{(char*)"Size", typLFLOAT, &size, 0L},
		{(char*)"Line", typLINEDEF, &Outline, 0L},
		{(char*)"FillLine", typLINEDEF, &Hatchline, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&Outline, defs.GetOutLine(), sizeof(LineDEF));
		memcpy(&Fill, defs.GetFill(defs.units()), sizeof(FillDEF));
		if(Fill.hatch)memcpy(&Hatchline, Fill.hatch, sizeof(LineDEF));
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		Fill.hatch = &Hatchline;		size = DefSize(SIZE_BAR);
		ssRef = 0L;				mo = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Fill.hatch = &Hatchline;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Box", Desc);
		}
	return false;
}

bool
Whisker::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"High", typLFPOINT, &pos1, 0L},
		{(char*)"Low", typLFPOINT, &pos2, 0L},
		{(char*)"Size", typLFLOAT, &size, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"Desc", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = DefSize(SIZE_WHISKER);
		LineDef.width = DefSize(SIZE_WHISKER_LINE);
		LineDef.color = defs.Color(COL_WHISKER);
		mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Whisker", Desc);
		}
	return false;
}

bool
DropLine::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"Pos", typLFPOINT, &fPos, 0L},
		{(char*)"Line", typLAST | typLINEDEF, &LineDef, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		LineDef.color = defs.Color(COL_SYM_LINE);
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"DropLine", Desc);
		}
	return false;
}

bool
Sphere::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"Pos", typPOINT3D, &fPos, 0L},
		{(char*)"Size", typLFLOAT, &size, 0L},
		{(char*)"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};
	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		size = DefSize(SIZE_SYMBOL);
		Line.color = defs.Color(COL_SYM_LINE);
		Line.width = DefSize(SIZE_SYM_LINE);
		Fill.color = defs.Color(COL_SYM_FILL);
		scl = 0L;		nscl = 0;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Sphere", Desc);
		}
	return false;
}

bool
Plane3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"values", typLAST | typFPLST3D, &dt, &ndt}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		ipl = 0L;	pts = 0L;	PlaneVec = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Plane3D", Desc);
		}
	return false;
}

bool
Brick::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"Pos", typPOINT3D, &fPos, 0L},
		{(char*)"depth", typLFLOAT, &depth, 0L},
		{(char*)"width", typLFLOAT, &width, 0L},
		{(char*)"height", typLFLOAT, &height, 0L},
		{(char*)"flags", typDWORD, &flags, 0L},
		{(char*)"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_SYM_LINE);
		Line.width = DefSize(SIZE_SYM_LINE);
		Fill.color = defs.Color(COL_SYM_FILL);
		faces = (plane**)calloc(6, sizeof(plane*));
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		mo = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Brick", Desc);
		}
	return false;
}

bool
DropLine3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Pos", typPOINT3D, &fPos, 0L},
		{(char*)"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_SYM_LINE);
		Line.width = DefSize(SIZE_HAIRLINE);	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		ls[0] = ls[1] = ls[2] = ls[3] = ls[4] = ls[5] = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"DropLine3D", Desc);
		}
	return false;
}

bool
Arrow3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Org", typPOINT3D, &fPos1, 0L},
		{(char*)"Pos", typPOINT3D, &fPos2, 0L},
		{(char*)"CapW", typLFLOAT, &cw, 0L},
		{(char*)"CapL", typLFLOAT, &cl, 0L},
		{(char*)"ssRef", typLAST | typIPLST, &ssRef, &cssRef}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cw = DefSize(SIZE_ARROW_CAPWIDTH);
		cl = DefSize(SIZE_ARROW_CAPLENGTH);
		Line.color = defs.Color(COL_ARROW);
		Line.width = DefSize(SIZE_ARROW_LINE);
		ls[0] = ls[1] = ls[2] = 0L;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		cap = 0L;
		type = ARROW_LINE;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Arrow3D", Desc);
		}
	return false;
}

bool
Line3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"ssRefX", typTEXT, &x_range, 0L},
		{(char*)"ssRefY", typTEXT, &y_range, 0L},
		{(char*)"ssRefZ", typTEXT, &z_range, 0L},
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"values", typLAST | typFPLST3D, &values, &nPts}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_DATA_LINE);
		Line.width = DefSize(SIZE_DATA_LINE);
		ls = 0L;	pts = 0L;	npts = 0L;	mo=0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = min.fz = max.fx = max.fy = max.fz = 0.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(nPts > 1) ls = (line_segment **)calloc(nPts-1, sizeof(line_segment*));
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Line3D", Desc);
		}
	return false;
}

bool
Polygon3D::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Line", typLINEDEF, &Line, 0L },
		{ (char*)"Fill", typFILLDEF, &Fill, 0L },
		{ (char*)"ssRefX", typTEXT, &x_range, 0L },
		{ (char*)"ssRefY", typTEXT, &y_range, 0L },
		{ (char*)"ssRefZ", typTEXT, &z_range, 0L },
		{ (char*)"ssRef", typIPLST, &ssRef, &cssRef },
		{ (char*)"values", typLAST | typFPLST3D, &values, &nPts } };
	switch (rw){
	case INIT_VARS:
		InitVarsGO(Desc);
		Line.color = defs.Color(COL_DATA_LINE);
		Line.width = DefSize(SIZE_DATA_LINE);
		Fill.color = 0x00cbcbcb;
		ls = 0L;	pts = 0L;	npts = 0L;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		min.fx = min.fy = min.fz = max.fx = max.fy = max.fz = 0.0;
		plane = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if (values && nPts && (values[nPts - 1].fx != values[0].fx || values[nPts - 1].fy != values[0].fy ||
			values[nPts - 1].fz != values[0].fz)) {
			//close polygon if necessary
			values[nPts++] = values[0];
			}
		if (nPts > 1) ls = (line_segment **)calloc(nPts - 1, sizeof(line_segment*));
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Polygon3D", Desc);
		}
	return false;
}

bool
Label::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"ssRef", typIPLST, &ssRef, &cssRef},
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"Pos", typNZLFPOINT, &fPos, 0L},
		{(char*)"Dist", typNZLFPOINT, &fDist, 0L},
		{(char*)"Flags", typDWORD, &flags, 0L},
		{(char*)"Name", typTEXT, &name, 0L},
		{(char*)"TxtDef", typLAST | typTXTDEF, &TextDef, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);									TextDef.ColTxt = 0x0L;
		TextDef.ColBg = 0x00ffffffL;						TextDef.fSize = defs.GetSize(SIZE_TEXT);;
		TextDef.RotBL = TextDef.RotCHAR = 0.0;				TextDef.iSize = 0;
		TextDef.Align = TXA_VTOP | TXA_HLEFT;				TextDef.Mode = TXM_TRANSPARENT;
		TextDef.Style = TXS_NORMAL;							TextDef.Font = FONT_HELVETICA;
		TextDef.text = 0L;	bgcolor = 0x00ffffffL;			bgLine.width = 0.0;		bgLine.patlength = defs.GetSize(SIZE_TEXT);
		bgLine.color = bgcolor;	bgLine.pattern = 0L;		CursorPos = 0;	defDisp = 0L;	bBGvalid = bModified = false;
		curr_z = 0.0;										is3D = false;
		trackLine.color = TextDef.ColTxt;					trackLine.width = defs.GetSize(SIZE_HAIRLINE);
		trackLine.pattern = 0L;								trackLine.patlength = defs.GetSize(SIZE_TEXT);
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(parent && parent->Id != GO_MLABEL) {
			if(!TextDef.text || !TextDef.text[0]) return false;
			}
		return ExecOutput(Notary->RegisterGO(this), (char*)"Label", Desc);
		}
	return false;
}

void
mLabel::RegGO(void *n)
{
	int i;

	if(n) {
		if(Lines) for(i = 0; i < nLines; i++) if(Lines[i]) Lines[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
	}
}

bool
mLabel::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"Pos", typNZLFPOINT, &fPos, 0L},
		{(char*)"Dist", typNZLFPOINT, &fDist, 0L},
		{(char*)"lspc", typLFLOAT, &lspc, 0L},
		{(char*)"Flags", typDWORD, &flags, 0L},
		{(char*)"TxtDef", typTXTDEF, &TextDef, 0L},
		{(char*)"Name", typTEXT, &name, 0L },
		{(char*)"Lines", typLAST | typOBJLST, &Lines, &nLines}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		//The lines inherit settings from this object.
		//We need not save them in this context
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		TextDef.ColTxt = 0x0L;
		TextDef.ColBg = 0x00ffffffL;
		TextDef.fSize = DefSize(SIZE_TEXT);
		TextDef.RotBL = TextDef.RotCHAR = 0.0;
		TextDef.iSize = 0;
		TextDef.Align = TXA_VTOP | TXA_HLEFT;
		TextDef.Mode = TXM_TRANSPARENT;
		TextDef.Style = TXS_NORMAL;
		TextDef.Font = FONT_HELVETICA;
		TextDef.text = 0L;
		undo_flags = 0L;	lspc = 1.0;
		curr_z = 0.0;		is3D = false;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(Lines) for ( i = 0; i < nLines; i++)
			if(Lines[i]) Lines[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(Lines) for ( i = 0; i < nLines; i++)
			if(Lines[i]) Lines[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"mLabel", Desc);
		}
	return false;
}

bool
TextFrame::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"Pos1", typNZLFPOINT, &pos1, 0L},
		{(char*)"Pos2", typNZLFPOINT, &pos2, 0L},
		{(char*)"lspc", typLFLOAT, &lspc, 0L},
		{(char*)"Pad", typFRECT, &pad, 0L},
		{(char*)"TxtDef", typTXTDEF, &TextDef, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"FillLine", typLINEDEF, &FillLine, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"Name", typTEXT, &name, 0L },
		{(char*)"Text", typLAST | typTEXT, &text, 0L}};
	switch(rw) {
	case SAVE_VARS:
		return false;
	case INIT_VARS:
		InitVarsGO(Desc);
		TextDef.ColTxt = 0x0L;
		TextDef.ColBg = 0x00ffffffL;
		TextDef.fSize = DefSize(SIZE_TEXT);
		TextDef.RotBL = TextDef.RotCHAR = 0.0;
		TextDef.iSize = 0;
		TextDef.Align = TXA_VBOTTOM | TXA_HLEFT;
		TextDef.Mode = TXM_TRANSPARENT;
		TextDef.Style = TXS_NORMAL;
		TextDef.Font = FONT_HELVETICA;
		TextDef.text = 0L;
		lines = 0L;		nlines = 0;			drc = 0L;
		cur_pos.x = cur_pos.y = tm_c = 0;	tm_rec = 0L;
		bModified = bResize = has_m1 = has_m2 = false;
		if(Fill.hatch) memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;			c_char = m1_char = m2_char = '?';
		pad.Xmin = pad.Xmax = pad.Ymin = pad.Ymax = DefSize(SIZE_SYMBOL)/2.0;
		Cursor.left = Cursor.right = Cursor.top = Cursor.bottom = 0;
		pad.Xmax *= 2.0;
		return true;
	case FILE_READ:
		ExecInput(Desc);				Fill.hatch = &FillLine;
		return true;
	case FILE_WRITE:
		if(lines)lines2text();
		if(!text || !text[0]) return false;
		return ExecOutput(Notary->RegisterGO(this), (char*)"TextFrame", Desc);
		}
	return false;
}

bool
segment::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"cent", typLFPOINT, &fCent, 0L},
		{(char*)"ri", typNZLFLOAT, &radius1, 0L},
		{(char*)"ra", typLFLOAT, &radius2, 0L},
		{(char*)"start", typLFLOAT, &angle1, 0L},
		{(char*)"end", typLFLOAT, &angle2, 0L},
		{(char*)"shout", typNZLFLOAT, &shift, 0L},
		{ (char*)"Name", typTEXT, &name, 0L },
		{(char*)"Line", typLINEDEF, &segLine, 0L},
		{(char*)"FillLine", typLINEDEF, &segFillLine, 0L},
		{(char*)"Fill", typLAST | typFILLDEF, &segFill, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		segLine.width = DefSize(SIZE_SEGLINE);
		pts = 0L;	nPts = 0;	mo = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		segFill.hatch = &segFillLine;
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"segment", Desc);
		}
	return false;
}

bool
polyline::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"Data", typFPLST, &Values, &nPoints},
		{(char*)"Line", typLINEDEF, &pgLine, 0L},
		{(char*)"FillLine", typLINEDEF, &pgFillLine, 0L},
		{(char*)"Fill", typLAST | typFILLDEF, &pgFill, 0L}};
	
	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&pgLine, defs.plLineDEF(0L), sizeof(LineDEF));
		memcpy(&pgFill, defs.pgFillDEF(0L), sizeof(FillDEF));
		if(pgFill.hatch) memcpy(&pgFillLine, pgFill.hatch, sizeof(LineDEF));
		pgFill.hatch = &pgFillLine;
		pts = 0L;		nPts = 0;		name = 0L;
		pHandles = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		pgFill.hatch = &pgFillLine;
		if (!name) name = rlp_strdup(type == 1 ? (char *)"polygon" : (char*)"polyline");
		return true;
	case FILE_WRITE:
		if(type != 1) Desc[3].type |= typLAST;	//skip fill for polyline
		return ExecOutput(Notary->RegisterGO(this), 
			type == 1 ? (char*)"polygon" : (char*)"polyline", Desc);
		}
	return false;
}

bool
Bezier::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"Data", typFPLST, &Values, &nPoints},
		{(char*)"Line", typLINEDEF, &pgLine, 0L},
		{(char*)"FillLine", typLINEDEF, &pgFillLine, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L},
		{(char*)"Fill", typLAST | typFILLDEF, &pgFill, 0L}};
	
	switch(rw) {
	case INIT_VARS:
		//assume that all initialization is done by polyline::FileIO(int rw)
		return true;
	case FILE_READ:
		ExecInput(Desc);
		pgFill.hatch = &pgFillLine;
		if (name) free(name);
		name = rlp_strdup((char*)"curve");
		return true;
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"bezier", Desc);
		}
	return false;
}

bool
rectangle::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"p1", typLFPOINT, &fp1, 0L},
		{(char*)"p2", typLFPOINT, &fp2, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"FillLine", typLINEDEF, &FillLine, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"Rad", typNZLFLOAT, &rad, 0L},
		{(char*)"Name", typLAST | typTEXT, &name, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&Line, defs.pgLineDEF(0L), sizeof(LineDEF));
		memcpy(&Fill, defs.pgFillDEF(0L), sizeof(FillDEF));
		if(Fill.hatch) memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;
		pts = 0L;	nPts = 0L;
		rad = DefSize(SIZE_RRECT_RAD);
		drc = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Fill.hatch = &FillLine;
		if (!name) {
			switch (type) {
			case 0:			name = rlp_strdup((char*)"rectangle");		break;
			case 1:			name = rlp_strdup((char*)"ellipse");		break;
			case 2:			name = rlp_strdup((char*)"roundrec");		break;
				}
			}
		return true;
	case FILE_WRITE:
		if(type != 2) rad = 0.0;
		ExecOutput(Notary->RegisterGO(this), 
			type == 1? (char*)"ellipse" : type == 2? (char*)"roundrec" :
			(char*)"rectangle", Desc);
		return true;
		}
	return false;
}

void
LegItem::RegGO(void *n)
{
	if(n) {
		if(Sym) Sym->RegGO(n);
		if(Desc) Desc->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
LegItem::FileIO(int rw)
{
	descIO Des[] = {
		{(char*)"D_Line", typLINEDEF, &DataLine, 0L},
		{(char*)"O_Line", typLINEDEF, &OutLine, 0L},
		{(char*)"H_Line", typLINEDEF, &HatchLine, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"Sym", typGOBJ, &Sym, 0L},
		{(char*)"Text", typGOBJ, &Desc, 0L},
		{(char*)"flags", typLAST | typDWORD, &flags, 0L}};

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Des);
	case INIT_VARS:
		InitVarsGO(Des);
		Fill.hatch = &HatchLine;
		OutLine.width = defs.GetSize(SIZE_HAIRLINE);
		return true;
	case FILE_READ:
		ExecInput(Des);
		Fill.hatch = &HatchLine;
		if(Sym) Sym->parent=this;
		if(Desc) Desc->parent=this;
		return true;
	case FILE_WRITE:
		if(Sym) Sym->FileIO(rw);
		if(Desc) Desc->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"LegItem", Des);
		}
	return false;
}

void
Legend::RegGO(void *n)
{
	int i;

	if(n) {
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Legend::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"pos", typLFPOINT, &pos, 0L},
		{(char*)"rec1", typFRECT, &B_Rect, 0L},
		{(char*)"rec2", typFRECT, &D_Rect, 0L},
		{(char*)"rec3", typFRECT, &F_Rect, 0L},
		{(char*)"Items", typLAST | typOBJLST, &Items, &nItems}};
	int i;
	double d;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		B_Rect.Ymin = DefSize(SIZE_DRECT_TOP);			B_Rect.Xmin = DefSize(SIZE_DRECT_LEFT);
		B_Rect.Xmax = B_Rect.Xmin + 1.5*(d = DefSize(SIZE_BAR));
		B_Rect.Ymin += d*0.2;
		B_Rect.Ymax = B_Rect.Ymin + d/2.0;
		D_Rect.Ymin = 0.0;			D_Rect.Xmin = d*0.7;
		D_Rect.Xmax = d*1.3;		D_Rect.Ymax = d*0.4;
		F_Rect.Ymin = 0.0;			F_Rect.Xmin = d*0.2;
		F_Rect.Xmax = d*1.3;		F_Rect.Ymax = d*0.4;
		to = 0L;					hasLine = false;
		trc.left = trc.right = trc.top = trc.bottom = 0;
		if(!name) {
			name = (char*)malloc(20 * sizeof(char));
			rlp_strcpy(name, 20, (char*)"Legend");
			}
		return true;
	case FILE_READ:
		nItems = 0L;
		ExecInput(Desc);
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->parent=this;
		return true;
	case FILE_WRITE:
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Legend", Desc);
		}
	return false;
}

void
PlotScatt::RegGO(void *n)
{
	int i;

	if(n) {
		if(TheLine) TheLine->RegGO(n);
		if (ErrPg) ErrPg->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		if(Errors) for(i = 0; i < nErrs; i++) if(Errors[i]) Errors[i]->RegGO(n);
		if(Arrows) for(i = 0; i < nArrow; i++) if(Arrows[i]) Arrows[i]->RegGO(n);
		if(DropLines) for(i = 0; i < nDrops; i++) if(DropLines[i]) DropLines[i]->RegGO(n);
		if(Labels) for(i = 0; i < nLabel; i++) if(Labels[i]) Labels[i]->RegGO(n);
		if(Bars) for(i = 0; i < nBars; i++) if(Bars[i]) Bars[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
PlotScatt::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Name", typTEXT, &name, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"DefSym", typNZINT, &DefSym, 0L},
		{(char*)"baDist", typLFPOINT, &BarDist, 0L},
		{(char*)"xRange", typTEXT, &xRange, 0L},
		{(char*)"yRange", typTEXT, &yRange, 0L},
		{(char*)"eRange", typTEXT, &ErrRange, 0L},
		{(char*)"lRange", typTEXT, &LbRange, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"Bars", typOBJLST, &Bars, &nBars},
		{(char*)"Symbols", typOBJLST, &Symbols, &nPoints},
		{(char*)"PL", typGOBJ, &TheLine, 0L},
		{(char*)"ErrPg", typGOBJ, &ErrPg, 0L},
		{(char*)"ErrBars", typOBJLST, &Errors, &nErrs},
		{(char*)"Arrows", typOBJLST, &Arrows, &nArrow},
		{(char*)"dLines", typOBJLST, &DropLines, &nDrops},
		{(char*)"Labels", typOBJLST, &Labels, &nLabel},
		{(char*)"x_info", typTEXT, &x_info, 0L},
		{(char*)"y_info", typTEXT, &y_info, 0L},
		{(char*)"DataDesc", typLAST | typTEXT, &data_desc, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		x_info = y_info = z_info = 0L;	nPoints = 0;
		InitVarsGO(Desc);				DefSym = SYM_CIRCLE;
		DefSel = 0x01;					dirty = true;
		bar_dx = 0.0;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "xy-plot (%s)", name);
#else
			i = sprintf(TmpTxt, "xy-plot (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		else {
			name = rlp_strdup((char*)"xy-plot");
			}
		return true;
	case FILE_READ:
		nPoints = 0L;
		ExecInput(Desc);
		ForEach(FE_PARENT, 0L, 0L);
		return true;
	case FILE_WRITE:
		if(TheLine) TheLine->FileIO(rw);
		if (ErrPg) ErrPg->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Errors) for(i = 0; i < nErrs; i++) if(Errors[i]) Errors[i]->FileIO(rw);
		if(Arrows) for(i = 0; i < nArrow; i++) if(Arrows[i]) Arrows[i]->FileIO(rw);
		if(DropLines) for(i = 0; i < nDrops; i++) if(DropLines[i]) DropLines[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nLabel; i++) if(Labels[i]) Labels[i]->FileIO(rw);
		if(Bars) for(i = 0; i < nBars; i++) if(Bars[i]) Bars[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"PlotScatt", Desc);
		}
	return false;
}

bool
xyStat::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Name", typTEXT, &name, 0L },
		{ (char*)"Type", typNZLONG, &type, 0L },
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"DefSym", typNZINT, &DefSym, 0L},
		{(char*)"baDist", typLFPOINT, &BarDist, 0L},
		{(char*)"confi", typLFLOAT, &ci, 0L},
		{(char*)"xRange", typTEXT, &xRange, 0L},
		{(char*)"yRange", typTEXT, &yRange, 0L},
		{(char*)"prefix", typTEXT, &case_prefix, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"Bars", typOBJLST, &Bars, &nBars},
		{(char*)"Symbols", typOBJLST, &Symbols, &nPoints},
		{(char*)"PL", typGOBJ, &TheLine, 0L},
		{(char*)"ErrBars", typOBJLST, &Errors, &nErrs},
		{(char*)"Labels", typOBJLST, &Labels, &nLabel},
		{(char*)"x_info", typTEXT, &x_info, 0L},
		{(char*)"y_info", typLAST | typTEXT, &y_info, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		//most initialistion is done by PlotScatt::FileIO
		curr_data = 0L;
		case_prefix = 0L;
		ci = 95.0;
		if (!name) name = rlp_strdup((char*)"XY Stats");
		return true;
	case FILE_READ:
		nPoints = 0L;
		ExecInput(Desc);
		ForEach(FE_PARENT, 0L, 0L);
		return true;
	case FILE_WRITE:
		if(TheLine) TheLine->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Errors) for(i = 0; i < nErrs; i++) if(Errors[i]) Errors[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nLabel; i++) if(Labels[i]) Labels[i]->FileIO(rw);
		if(Bars) for(i = 0; i < nBars; i++) if(Bars[i]) Bars[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"xyStat", Desc);
		}
	return false;
}

void
FreqDist::RegGO(void *n)
{
	int i;

	if(n) {
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
FreqDist::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Type", typNZLONG, &type, 0L },
		{ (char*)"hide", typNZINT, &hidden, 0L },
		{ (char*)"ssRef", typTEXT, &ssRef, 0L },
		{ (char*)"x_axis", typNZINT, &use_xaxis, 0L },
		{ (char*)"y_axis", typNZINT, &use_yaxis, 0L },
		{ (char*)"cl_start", typNZLFLOAT, &start, 0L },
		{ (char*)"cl_size", typLFLOAT, &step, 0L },
		{ (char*)"BarLine", typLINEDEF, &BarLine, 0L },
		{ (char*)"BarFill", typFILLDEF, &BarFill, 0L },
		{ (char*)"FuncLine", typLINEDEF, &FuncLine, 0L },
		{ (char*)"BarFillLine", typLINEDEF, &HatchLine, 0L},
		{ (char*)"plots", typLAST | typOBJLST, &plots, &nPlots}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&BarFill, defs.GetFill(defs.units()), sizeof(FillDEF));
		BarFill.color = 0x00c0ffffL;
		if(BarFill.hatch) memcpy(&HatchLine, BarFill.hatch, sizeof(LineDEF));
		BarFill.hatch = &HatchLine;
		memcpy(&BarLine, defs.GetOutLine(), sizeof(LineDEF));
		memcpy(&FuncLine, defs.GetLine(), sizeof(LineDEF));
		FuncLine.color = 0x00ff0000L;
		curr_data=0L;		dirty = true;
		dmin = HUGE_VAL, dmax = -HUGE_VAL;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "freq. dist. (%s)", name);
#else
			i = sprintf(TmpTxt, "freq. dist. (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->parent=this;
		return true;
	case FILE_WRITE:
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"FreqDist", Desc);
		}
	return false;
}

void
Regression::RegGO(void *n)
{
	int i;

	if(n) {
		if(rLine) rLine->RegGO(n);
		if(sde) sde->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Regression::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"xRange", typTEXT, &xRange, 0L},
		{(char*)"yRange", typTEXT, &yRange, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"Line", typGOBJ, &rLine, 0L},
		{(char*)"Ellipse", typGOBJ, &sde, 0L},
		{(char*)"Symbols", typLAST | typOBJLST, &Symbols, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		dirty = true;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "regression (%s)", name);
#else
			i = sprintf(TmpTxt, "regression (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		nPoints = 0L;
		return ExecInput(Desc);
	case FILE_WRITE:
		if(rLine) rLine->FileIO(rw);
		if(sde) sde->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Regression", Desc);
		}
	return false;
}

bool
myRegression::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Type", typNZLONG, &type, 0L },
		{ (char*)"Name", typTEXT, &name, 0L },
		{ (char*)"Bounds", typFRECT, &Bounds, 0L },
		{ (char*)"xRange", typTEXT, &xRange, 0L },
		{ (char*)"yRange", typTEXT, &yRange, 0L },
		{ (char*)"x_axis", typNZINT, &use_xaxis, 0L },
		{ (char*)"y_axis", typNZINT, &use_yaxis, 0L },
		{ (char*)"PrevSym", typGOBJ, &PrevSym, 0L },
		{ (char*)"PrevErr", typGOBJ, &PrevErr, 0L },
		{ (char*)"Function", typGOBJ, &func, 0L },
		{ (char*)"CiLine1", typGOBJ, &ci_line1, 0L },
		{ (char*)"CiLine2", typGOBJ, &ci_line2, 0L },
		{ (char*)"Text", typGOBJ, &reg_res, 0L },
		{ (char*)"ConfInt", typLFLOAT, &ci, 0L },
		{ (char*)"Errors", typOBJLST, &Errors, &nErrs },
		{ (char*)"Line", typLINEDEF, &PrevLine, 0L },
		{ (char*)"CiLineDef", typLINEDEF, &PrevCiLine, 0L },
		{ (char*)"Symbols", typLAST | typOBJLST, &Symbols, &nPoints } };

	int i;

	switch (rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		memcpy(&PrevLine, defs.GetLine(), sizeof(LineDEF));
		memcpy(&PrevCiLine, defs.GetLine(), sizeof(LineDEF));
		PrevCiLine.color = 0x00000ff;		PrevCiLine.width *= 0.5;
		ci = 99.0;
		return true;
	case FILE_READ:
		nPoints = nErrs = 0L;
		return ExecInput(Desc);
	case FILE_WRITE:
		if (func) func->FileIO(rw);
		if (ci_line1) ci_line1->FileIO(rw);
		if (ci_line2) ci_line2->FileIO(rw);
		if (reg_res) reg_res->FileIO(rw);
		if (PrevSym) PrevSym->FileIO(rw);
		if (Symbols) for (i = 0; i < nPoints; i++) if (Symbols[i]) Symbols[i]->FileIO(rw);
		if (Errors) for (i = 0; i < nErrs; i++) if (Errors[i]) Errors[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"myRegression", Desc);
		}
	return false;
}

bool
myBarPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Name", typTEXT, &name, 0L },
		{ (char*)"hide", typNZINT, &hidden, 0L },
		{ (char*)"Bounds", typFRECT, &Bounds, 0L },
		{ (char*)"baDist", typLFPOINT, &BarDist, 0L },
		{ (char*)"x_axis", typNZINT, &use_xaxis, 0L },
		{ (char*)"y_axis", typNZINT, &use_yaxis, 0L },
		{ (char*)"Bars", typOBJLST, &Bars, &nBars },
		{ (char*)"Symbols", typOBJLST, &Symbols, &nPoints },
		{ (char*)"PL", typGOBJ, &TheLine, 0L },
		{ (char*)"ErrPg", typGOBJ, &ErrPg, 0L },
		{ (char*)"ErrBars", typOBJLST, &Errors, &nErrs },
		{ (char*)"x_info", typTEXT, &x_info, 0L },
		{ (char*)"y_info", typTEXT, &y_info, 0L },
		{ (char*)"PrevSym", typGOBJ, &PrevSym, 0L },
		{ (char*)"PrevBar", typGOBJ, &PrevBar, 0L },
		{ (char*)"PrevErr", typGOBJ, &PrevErr, 0L },
		{ (char*)"PrevLine", typLINEDEF, &PrevLine, 0L },
		{ (char*)"DataDesc", typLAST | typTEXT, &data_desc, 0L } };
	long i;

	switch (rw) {
	case INIT_VARS:
		//most initialisation already done by PlotScatt::FileIO()
		PrevSym = 0L;	PrevBar = 0L;		PrevErr = 0L;
		return true;
	case FILE_READ:
		ExecInput(Desc);			ForEach(FE_PARENT, 0L, 0L);
		if (ErrPg) ErrPg->parent = this;
		return true;
	case FILE_WRITE:
		if (TheLine) TheLine->FileIO(rw);
		if (ErrPg) ErrPg->FileIO(rw);
		if (Symbols) for (i = 0; i < nPoints; i++) if (Symbols[i]) Symbols[i]->FileIO(rw);
		if (Errors) for (i = 0; i < nErrs; i++) if (Errors[i]) Errors[i]->FileIO(rw);
		if (Bars) for (i = 0; i < nBars; i++) if (Bars[i]) Bars[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"myBarPlot", Desc);
		}
	return false;
}

void
BubblePlot::RegGO(void *n)
{
	int i;

	if(n) {
		if(Bubbles) for(i = 0; i < nPoints; i++) if(Bubbles[i]) Bubbles[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
BubblePlot::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"Line", typLINEDEF, &BubbleLine, 0L},
		{(char*)"FillLine", typLINEDEF, &BubbleFillLine, 0L},
		{(char*)"Fill", typFILLDEF, &BubbleFill, 0L},
		{(char*)"Bubbles", typLAST | typOBJLST, &Bubbles, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		BubbleFill.color = defs.Color(COL_BUBBLE_FILL);
		BubbleLine.color = defs.Color(COL_BUBBLE_LINE);
		BubbleLine.width = DefSize(SIZE_BUBBLE_LINE);
		BubbleFillLine.color = defs.Color(COL_BUBBLE_FILLLINE);
		BubbleFillLine.width = DefSize(SIZE_BUBBLE_HATCH_LINE);
		BubbleFill.hatch = &BubbleFillLine;
		dirty = true;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "Bubble Plot (%s)", name);
#else
			i = sprintf(TmpTxt, "Bubble Plot (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Bubbles) for(i = 0; i < nPoints; i++) if(Bubbles[i]) Bubbles[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"BubblePlot", Desc);
		}
	return false;
}

void
PolarPlot::RegGO(void *n)
{
	long i;

	if(n) {
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->RegGO(n);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
PolarPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"ang_offs", typLFLOAT, &offs, 0L},
		{(char*)"Plots", typOBJLST, &Plots, &nPlots},
		{(char*)"Axes", typOBJLST, &Axes, &nAxes},
		{(char*)"FillLine", typLINEDEF, &FillLine, 0L},
		{(char*)"Fill", typLAST | typFILLDEF, &Fill, 0L}};
	long i;

	switch(rw) {
	case INIT_VARS:
		CurrDisp = 0L;
		InitVarsGO(Desc);
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "polar plot root (%s)", name);
#else
			i = sprintf(TmpTxt, "polar plot root (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"PolarPlot", Desc);
		}
	return false;
}

void
TernaryPlot::RegGO(void *n)
{
	long i;

	if(n) {
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->RegGO(n);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
TernaryPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"Plots", typOBJLST, &Plots, &nPlots},
		{(char*)"Axes", typOBJLST, &Axes, &nAxes},
		{(char*)"FillLine", typLINEDEF, &FillLine, 0L},
		{(char*)"Fill", typLAST | typFILLDEF, &Fill, 0L}};
	long i;

	switch(rw) {
	case INIT_VARS:
		CurrDisp = 0L;
		InitVarsGO(Desc);
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "ternary plot root (%s)", name);
#else
			i = sprintf(TmpTxt, "ternary plot root (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"TernaryPlot", Desc);
		}
	return false;
}

void
TernaryXYZ::RegGO(void *n)
{
	long i;

	if(n) {
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->RegGO(n);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		if(Arrows)  for (i = 0; i < nArrow; i++) if (Arrows[i]) Arrows[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
TernaryXYZ::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"doPG", typNZINT, &doPG, 0L},
		{(char*)"Name", typTEXT, &name, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"Center", typLFPOINT, &cent, 0L},
		{(char*)"Rad_1", typLFLOAT, &rad_l, 0L},
		{(char*)"Rad_2", typLFLOAT, &rad_s, 0L},
		{(char*)"Plots", typOBJLST, &Plots, &nPlots},
		{(char*)"PL", typGOBJ, &TheLine, 0L},
		{(char*)"Axes", typOBJLST, &Axes, &nAxes},
		{(char*)"Symbols", typOBJLST, &Symbols, &nPoints},
		{(char*)"Arrows", typOBJLST, &Arrows, &nArrow },
		{(char*)"Outline", typLINEDEF, &Outline, 0L },
		{(char*)"FillLine", typLINEDEF, &FillLine, 0L},
		{(char*)"Fill", typLAST | typFILLDEF, &Fill, 0L}};
	long i;

	switch(rw) {
	case INIT_VARS:
		CurrDisp = 0L;
		InitVarsGO(Desc);
		rlp_strcpy(&TmpTxt, 40, (char*)"Ternary XYZ");
		if(name) free(name);
		name = rlp_strdup((char*)TmpTxt);
		memcpy(&Outline, defs.GetOutLine(), sizeof(LineDEF));
		memcpy(&FillLine, defs.GetLine(), sizeof(LineDEF));
		memcpy(&Fill, defs.GetFill(defs.units()), sizeof(FillDEF));
		if(Fill.hatch)memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;
		ang_disp = 0.57735;
		Plots = NULL;		nPlots = 0;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Fill.hatch = &FillLine;
		if(!name) {
			name = rlp_strdup((char*)"Ternary XYZ");
			}
		return true;
	case FILE_WRITE:
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		if (Arrows) for (i = 0; i < nArrow; i++) if (Arrows[i]) Arrows[i]->FileIO(rw);
		if (TheLine) TheLine->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"TernaryXYZ", Desc);
		}
	return false;
}


void
BoxPlot::RegGO(void *n)
{
	int i;

	if(n) {
		if(Boxes) for(i = 0; i < nBoxes; i++) if(Boxes[i]) Boxes[i]->RegGO(n);
		if(Whiskers) for(i = 0; i < nWhiskers; i++) if(Whiskers[i]) Whiskers[i]->RegGO(n);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		if(Labels) for(i = 0; i < nLabel; i++) if(Labels[i]) Labels[i]->RegGO(n);
		if(TheLine) TheLine->RegGO(n);
		if (ErrPg) ErrPg->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
BoxPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"Type", typNZLONG, &type, 0L },
		{ (char*)"hide", typNZINT, &hidden, 0L },
		{ (char*)"Bounds", typFRECT, &Bounds, 0L },
		{ (char*)"xRange", typTEXT, &xRange, 0L },
		{ (char*)"yRange", typTEXT, &yRange, 0L },
		{ (char*)"prefix", typTEXT, &case_prefix, 0L },
		{ (char*)"boDist", typLFPOINT, &BoxDist, 0L },
		{ (char*)"ci_box", typNZLFLOAT, &ci_box, 0L },
		{ (char*)"ci_err", typNZLFLOAT, &ci_err, 0L },
		{ (char*)"x_axis", typNZINT, &use_xaxis, 0L },
		{ (char*)"y_axis", typNZINT, &use_yaxis, 0L },
		{ (char*)"Boxes", typOBJLST, &Boxes, &nBoxes },
		{ (char*)"Whiskers", typOBJLST, &Whiskers, &nWhiskers },
		{ (char*)"Symbols", typOBJLST, &Symbols, &nPoints },
		{ (char*)"Labels", typOBJLST, &Labels, &nLabel },
		{ (char*)"Line", typGOBJ, &TheLine, 0L },
		{ (char*)"ErrPG", typLAST | typGOBJ, &ErrPg, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		dirty = true;		InitVarsGO(Desc);
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "boxes (%s)", name);
#else
			i = sprintf(TmpTxt, "boxes (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		curr_data = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Boxes) for(i = 0; i < nBoxes; i++) 
			if(Boxes[i]) Boxes[i]->FileIO(rw);
		if(Whiskers) for(i = 0; i < nWhiskers; i++) 
			if(Whiskers[i]) Whiskers[i]->FileIO(rw);
		if(Symbols) for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nLabel; i++) 
			if(Labels[i]) Labels[i]->FileIO(rw);
		if(TheLine) TheLine->FileIO(rw);
		if (ErrPg) ErrPg->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"BoxPlot", Desc);
		}
	return false;
}

void
DensDisp::RegGO(void *n)
{
	int i;

	if(n) {
		if(Boxes) for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
DensDisp::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"xRange", typTEXT, &xRange, 0L},
		{(char*)"yRange", typTEXT, &yRange, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"Line", typLINEDEF, &DefLine, 0L},
		{(char*)"FillLine", typLINEDEF, &DefFillLine, 0L},
		{(char*)"Fill", typFILLDEF, &DefFill, 0L},
		{(char*)"Boxes", typLAST | typOBJLST, &Boxes, &nPoints}};
	int i;

	switch(rw) {
	case INIT_VARS:
		return InitVarsGO(Desc);
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Boxes) for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"DensDisp", Desc);
		}
	return false;
}

void
StackBar::RegGO(void *n)
{
	long i;

	if(n) {
		if(Boxes) for(i = 0; i < numPlots; i++) if(Boxes[i]) Boxes[i]->RegGO(n);
		if(xyPlots) for(i = 0; i < numXY; i++) if(xyPlots[i]) xyPlots[i]->RegGO(n);
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->RegGO(n);
		if(Lines) for(i = 0; i < numPL; i++) if(Lines[i]) Lines[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
StackBar::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"cumData", typNZINT, &cum_data_mode, 0L},
		{(char*)"StartVal", typNZLFLOAT, &StartVal, 0L},
		{(char*)"Dspm", typNZLFPOINT, &dspm, 0L},
		{(char*)"ssXrange", typTEXT, &ssXrange, 0L},
		{(char*)"ssYrange", typTEXT, &ssYrange, 0L},
		{(char*)"BoxBars", typOBJLST, &Boxes, &numPlots},
		{(char*)"Plots", typOBJLST, &xyPlots, &numXY},
		{(char*)"Polygons", typOBJLST, &Polygons, &numPG},
		{(char*)"Lines", typLAST | typOBJLST, &Lines, &numPL}};
	long il;

	switch(rw) {
	case INIT_VARS:
		dirty = true;	CumData = 0L;
		InitVarsGO(Desc);
		if(name) {
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, "stack (%s)", name);
#else
			sprintf(TmpTxt, "stack (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		return true;
	case FILE_WRITE:
		if(Boxes) for(il = 0; il < numPlots; il++) if(Boxes[il]) Boxes[il]->FileIO(rw);
		if(xyPlots) for(il = 0; il < numXY; il++) if(xyPlots[il]) xyPlots[il]->FileIO(rw);
		if(Polygons) for(il = 0; il < numPG; il++) if(Polygons[il]) Polygons[il]->FileIO(rw);
		if(Lines) for(il = 0; il < numPL; il++) if(Lines[il]) Lines[il]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Stacked", Desc);
		}
	return false;
}
bool
Phenology::FileIO(int rw)
{
	descIO Desc[] = {
		{ (char*)"name", typTEXT, &name, 0L },
		{ (char*)"Bounds", typFRECT, &Bounds, 0L },
		{ (char*)"ssXrange", typTEXT, &ssXrange, 0L },
		{ (char*)"ssYrange", typTEXT, &ssYrange, 0 },
		{ (char*)"Plots", typLAST | typOBJLST, &Plots, &nPlots } };
	long i;

	switch (rw) {
	case INIT_VARS:
		dirty = true;		InitVarsGO(Desc);
		selRC_size = 0;		selRC_rec = 0L;		selRC_curr = -1;
		selRC_ref = -1;
		return true;
	case FILE_READ:
		ExecInput(Desc);
		return true;
	case FILE_WRITE:
		if (Plots) for (i = 0; i < nPlots; i++) if (Plots[i]) Plots[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Phenology", Desc);
		}
	return false;
}
void
PieChart::RegGO(void *n)
{
	int i;

	if(n) {
		if(Segments) for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
PieChart::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"ssRefA", typTEXT, &ssRefA, 0L},
		{(char*)"ssRefR", typTEXT, &ssRefR, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"CtDef", typLFPOINT, &CtDef, 0L},
		{(char*)"FacRad", typLFLOAT, &FacRad, 0L},
		{(char*)"Segs", typLAST | typOBJLST, &Segments, &nPts}};
	int i;

	switch(rw) {
	case INIT_VARS:
		Bounds.Xmax = Bounds.Ymax = 100.0f;
		Bounds.Xmin = Bounds.Ymin = -100.0f;
		InitVarsGO(Desc);
		CtDef.fx = 90.0;	CtDef.fy = 360.0;
		FacRad = 1.0;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "pie chart (%s)", name);
#else
			i = sprintf(TmpTxt, "pie chart (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Segments) for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"SegChart", Desc);
		}
	return false;
}

void
GoGroup::RegGO(void *n)
{
	long i;

	if(n) {
		if(Objects) for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
GoGroup::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Pos", typNZLFPOINT, &fPos, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Items", typLAST | typOBJLST, &Objects, &nObs}};
	long i;

	switch(rw) {
	case INIT_VARS:
		Bounds.Xmax = Bounds.Ymax = 100.0f;
		Bounds.Xmin = Bounds.Ymin = -100.0f;
		return InitVarsGO(Desc);
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		if(Objects) for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"group", Desc);
		}
	return false;
}

void
Scatt3D::RegGO(void *n)
{
	int i;

	if(n) {
		if(Line) Line->RegGO(n);
		if(rib) rib->RegGO(n);
		if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i]) Balls[i]->RegGO(n);
		if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i]) Columns[i]->RegGO(n);
		if(DropLines) for(i = 0; i < nDropLines; i++) if(DropLines[i]) DropLines[i]->RegGO(n);
		if(Arrows) for(i = 0; i < nArrows; i++) if(Arrows[i]) Arrows[i]->RegGO(n);
		if(Errors) for(i = 0; i < nErrors; i++) if(Errors[i]) Errors[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Scatt3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"ssRefX", typTEXT, &ssRefX, 0L},
		{(char*)"ssRefY", typTEXT, &ssRefY, 0L},
		{(char*)"ssRefZ", typTEXT, &ssRefZ, 0L},
		{(char*)"DataDesc", typTEXT, &data_desc, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"z_axis", typNZINT, &use_zaxis, 0L},
		{(char*)"Line", typGOBJ, &Line, 0L},
		{(char*)"Balls", typOBJLST, &Balls, &nBalls},
		{(char*)"Columns", typOBJLST, &Columns, &nColumns},
		{(char*)"DropLines", typOBJLST, &DropLines, &nDropLines},
		{(char*)"ParaV", typGOBJ, &rib, 0L},
		{(char*)"Arrows", typOBJLST, &Arrows, &nArrows},
		{(char*)"Errors", typLAST | typOBJLST, &Errors, &nErrors}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		c_flags = 0L;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		dirty = true;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "xyz-plot (%s)", name);
#else
			i = sprintf(TmpTxt, "xyz-plot (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		//now set parent in all children
		if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i]) Balls[i]->parent = this;
		if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i]) Columns[i]->parent = this;
		if(DropLines) for(i = 0; i < nDropLines; i++) if(DropLines[i]) DropLines[i]->parent = this;
		if(Arrows) for(i = 0; i < nArrows; i++) if(Arrows[i]) Arrows[i]->parent = this;
		if(Errors) for(i = 0; i < nErrors; i++) if(Errors[i]) Errors[i]->parent = this;
		if(Line) Line->parent = this;
		if(rib) rib->parent = this;
		return true;
	case FILE_WRITE:
		if(Line) Line->FileIO(rw);
		if(rib) rib->FileIO(rw);
		if (Balls) for (i = 0; i < nBalls; i++) {
			if (Balls[i]) Balls[i]->FileIO(rw);
			}
		if (Columns) for (i = 0; i < nColumns; i++) {
			if (Columns[i]) Columns[i]->FileIO(rw);
			}
		if (DropLines) for (i = 0; i < nDropLines; i++) {
			if (DropLines[i]) DropLines[i]->FileIO(rw);
			}
		if (Arrows) for (i = 0; i < nArrows; i++) {
			if (Arrows[i]) Arrows[i]->FileIO(rw);
			}
		if (Errors) for (i = 0; i < nErrors; i++) {
			if (Errors[i]) Errors[i]->FileIO(rw);
			}
		return ExecOutput(Notary->RegisterGO(this), (char*)"Scatt3D", Desc);
		}
	return false;
}

void
Ribbon::RegGO(void *n)
{
	int i;

	if(n) {
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Ribbon::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"z-pos", typNZLFLOAT, &z_value},
		{(char*)"z-width", typNZLFLOAT, &z_width},
		{(char*)"relwidth", typNZLFLOAT, &relwidth},
		{(char*)"ssRefX", typTEXT, &ssRefX, 0L},
		{(char*)"ssRefY", typTEXT, &ssRefY, 0L},
		{(char*)"ssRefZ", typTEXT, &ssRefZ, 0L},
		{(char*)"DataDesc", typTEXT, &data_desc, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"values", typFPLST3D, &values, &nVal},
		{(char*)"Planes", typLAST | typOBJLST, &planes, &nPlanes}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		relwidth = 0.6;				dirty = true;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "ribbon (%s)", name);
#else
			i = sprintf(TmpTxt, "ribbon (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		//now set parent in all children
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Ribbon", Desc);
		}
	return false;
}

void
Grid3D::RegGO(void *n)
{
	int i;

	if(n) {
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->RegGO(n);
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Grid3D::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Start", typPOINT3D, &start, 0L},
		{(char*)"Step", typPOINT3D, &step, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Bounds", typFRECT,&Bounds, 0L},
		{(char*)"xBounds", typLFPOINT, &xBounds, 0L},
		{(char*)"yBounds", typLFPOINT, &yBounds, 0L},
		{(char*)"zBounds", typLFPOINT, &zBounds, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"lines", typOBJLST, &lines, &nLines},
		{(char*)"planes", typLAST | typOBJLST, &planes, &nPlanes}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		step.fx = step.fz = 1.0;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "grid (%s)", name);
#else
			i = sprintf(TmpTxt, "grid (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		//now set parent in all children
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->parent = this;
		if (planes) for (i = 0; i < nPlanes; i++) if (planes[i]) {
			planes[i]->parent = this;
			planes[i]->Command(CMD_SET_DATAOBJ, 0L, 0L);
			}
		return true;
	case FILE_WRITE:
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->FileIO(rw);
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Grid3D", Desc);
		}
	return false;
}

bool
Limits::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Bounds", typLAST | typFRECT, &Bounds, 0L}};

	switch(rw) {
	case INIT_VARS:
		if(name) {
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, "limits (%s)", name);
#else
			sprintf(TmpTxt, "limits (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), (char*)"Limits", Desc);
		}
	return false;
}

void
Function::RegGO(void *n)
{
	if(n) {
		if(dl)dl->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Function::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"x1", typNZLFLOAT, &x1, 0L},
		{(char*)"x2", typNZLFLOAT, &x2, 0L},
		{(char*)"xstep", typNZLFLOAT, &xstep, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"f_xy", typTEXT, &cmdxy, 0L},
		{(char*)"param", typTEXT, &param, 0L},
		{(char*)"DataLine", typGOBJ, &dl, 0L},
		{(char*)"Desc", typLAST | typTEXT, &name, 0L}};
	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cmdxy = param = 0L;
		memcpy(&Line, defs.GetLine(), sizeof(LineDEF));
		if(name) {
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, "function (%s)", name);
#else
			sprintf(TmpTxt, "function (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(dl) dl->parent = this;
		return true;
	case FILE_WRITE:
		if(dl) dl->FileIO(rw);
		ExecOutput(Notary->RegisterGO(this), (char*)"Function", Desc);
		}
	return false;
}

void
FitFunc::RegGO(void *n)
{
	int i;

	if(n) {
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
FitFunc::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"ssXref", typTEXT, &ssXref, 0L},
		{(char*)"ssYref", typTEXT, &ssYref, 0L},
		{(char*)"x_desc", typTEXT, &x_info, 0L},
		{(char*)"y_desc", typTEXT, &y_info, 0L},
		{(char*)"x1", typNZLFLOAT, &x1, 0L},
		{(char*)"x2", typNZLFLOAT, &x2, 0L},
		{(char*)"xstep", typNZLFLOAT, &xstep, 0L},
		{(char*)"conv", typNZLFLOAT, &conv, 0L},
		{(char*)"chi2", typNZLFLOAT, &chi2, 0L},
		{(char*)"maxiter", typNZINT, &maxiter, 0L},
		{(char*)"method", typNZINT, &method, 0L },
		{(char*)"Line", typLINEDEF, &Line, 0L },
		{(char*)"f_xy", typTEXT, &cmdxy, 0L},
		{(char*)"p_xy", typTEXT, &parxy, 0L},
		{(char*)"nVals", typNZLONG, &nVals},
		{(char*)"Symbols", typLAST | typOBJLST, &Symbols, &nPoints}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		cmdxy = parxy = 0L;
		memcpy(&Line, defs.GetLine(), sizeof(LineDEF));
		conv = 1.0e-05;	maxiter = 100;
		method = 1;						//method: 1 = LevMarc, 2 = simplex;
		dl = 0L;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "fit function (%s)", name);
#else
			i = sprintf(TmpTxt, "fit function (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"FitFunc", Desc);
		}
	return false;
}

bool
NormQuant::FileIO(int rw)
{
	lfPOINT *dt;
	long cnt;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"nData", typINT, &nData, 0L},
		{(char*)"Data", typFPLST, &dt, &cnt},
		{(char*)"ssRef", typTEXT, &ssRef, 0L},
		{(char*)"Symbol", typGOBJ, &sy, 0L},
		{(char*)"x_info", typTEXT, &x_info, 0L},
		{(char*)"y_info", typLAST | typTEXT, &y_info, 0L}};
	int i, j, l;

	if(rw == FILE_WRITE) {
		if(nData < 4) return false;
		l = (nData >>1) +1;				cnt = l;
		if(!(dt = (lfPOINT *)calloc(l, sizeof(lfPOINT))))return false;
		for(i = j = 0; i < nData; i += 2, j++) {
			dt[j].fx = src_data[i];		dt[j].fy = src_data[i+1];
			}
		}
	else {
		dt = 0L;		cnt = 0;
		}
	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		x_vals = y_vals = src_data = 0L;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "normal quantiles (%s)", name);
#else
			i = sprintf(TmpTxt, "normal quantiles (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(dt && cnt > 1 && (src_data = (double*)malloc((nData+2)*sizeof(double)))) {
			for(i = j = 0, l = nData-1; i < nData; i += 2, j++) {
				src_data[i] = dt[j].fx;		if(i < l) src_data[i+1] = dt[j].fy;
				}
			free(dt);
			}
		if (!sy) {
			sy = new Symbol(this, data, 0.0, 0.0, SYM_CIRCLE);
			sy->SetSize(SIZE_SYMBOL, NiceValue(defs.GetSize(SIZE_SYMBOL)*0.75));
			}
		if (sy){
			sy->parent = this;		sy->Id = GO_SYMBOL;		sy->data = data;
			}
		return true;
	case FILE_WRITE:
		if(sy) sy->FileIO(rw);
		ExecOutput(Notary->RegisterGO(this), (char*)"NormQuant", Desc);
		free(dt);
		return true;
		}
	return false;
}

bool
GridLine::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Line", typLINEDEF, &LineDef, 0L},
		{(char*)"flags", typLAST | typDWORD, &flags, 0L}};

	switch(rw) {
	case SAVE_VARS:
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		ncpts = 0;		cpts = 0L;	gl1 = gl2 = 0L;	ls = 0L;
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;	mo = 0L;
		return true;
	case FILE_READ:
		return ExecInput(Desc);
	case FILE_WRITE:
		return ExecOutput(Notary->RegisterGO(this), 
			Id == GO_GRIDLINE ?(char*)"GridLine" : 
			Id == GO_GRIDRADIAL? (char*)"GridRadial" : (char*)"GridLine3D", Desc);
		}
	return false;
}

void
Tick::RegGO(void *n)
{
	int i;

	if(n) {
		if(Grid) Grid->RegGO(n);
		if(label) label->RegGO(n);
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Tick::FileIO(int rw)
{
	GraphObj *gl = Grid;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Val", typLFLOAT, &value, 0L},
		{(char*)"Flags", typDWORD, &flags, 0L},
		{(char*)"Rot", typNZLFLOAT, &angle, 0L},
		{(char*)"GridType", typINT, &gl_type, 0L},
		{(char*)"Grid", typGOBJ, &gl, 0L},
		{(char*)"Label", typGOBJ, &label, 0L},
		{(char*)"Polygons", typOBJLST, &Polygons, &numPG},
		{(char*)"Size", typLAST | typLFLOAT, &size, 0L}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		if (Grid && (flags & AXIS_GRIDLINE)) Grid->FileIO(rw);
		if(label) label->FileIO(rw);
//		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->FileIO(rw);
		return SaveVarGO(Desc);
	case INIT_VARS:
		InitVarsGO(Desc);
		mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
		fix = fiy = 0.0f;	ls = 0L;	Grid = 0L;	mo = 0L;
		n_seg = s_seg = 0;	seg = 0L;	bValidTick = false;
		size = DefSize(SIZE_AXIS_TICKS);
		return true;
	case FILE_READ:
		ExecInput(Desc);
		Grid = (GridLine*) gl;
		if(Grid)Grid->parent = this;
		if(label)label->parent = this;
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->parent = this;
		return true;
	case FILE_WRITE:
		if(Grid) Grid->FileIO(rw);
		else gl = 0L;
		if(label) label->FileIO(rw);
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i]) Polygons[i]->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Tick", Desc);
		}
	return false;
}

void
Axis::TickFile(char *name)
{
	ReadCache *ca;
	int i, j, k, nt;
	char line[500], item[20];
	Tick **ttck;

	if(!name) return;
	ca = new ReadCache();
	if(!ca) return;
	if(! ca->Open(name)) {
		delete ca;
#ifdef USE_WIN_SECURE
		sprintf_s(TmpTxt, TMP_TXT_SIZE, "Error open file \"%s\"\nfor axis ticks", name);
#else
		sprintf(TmpTxt, "Error open file \"%s\"\nfor axis ticks", name);
#endif
		ErrorBox(TmpTxt);
		return;
		}
	Command(CMD_FLUSH, 0L, 0L);
	if(!(Ticks = ((Tick**)calloc(nt = 100, sizeof(Tick*))))) return;
	for(i = 0; ; i++) {
		j = k = 0;
		ca->ReadLine(line, sizeof(line));
		if(!line[0]) break;
		while(line[j] && line[j] < 33) j++;
		do{ item[k] = line[j++]; }
			while(item[k] >32 && item[k++] != '=' && k < (int)sizeof(item) && j < (int)sizeof(line));
		item[k--] = 0;		if(!line[j-1])j--;
		while(k && !(isdigit(item[k])))item[k--]=0;
		while(line[j] && (line[j]<33 || line[j] == '"'))j++;
		k = rlp_strlen(line);
		while(k >=j && (line[k] < 33 || line[k] == '"')) line[k--] = 0;
		//realloc table if necessary
		if(NumTicks >= nt) {
			if((ttck= (Tick**)realloc(Ticks, (nt += 1000)*sizeof(Tick*))))Ticks= ttck;
			else NumTicks--;
			}
		//now add tick to table
		Ticks[NumTicks] = new Tick(this, data, atof(item), line[j] ? axis->flags : axis->flags | AXIS_MINORTICK);
		if (!(Ticks[NumTicks]))break;
		Ticks[NumTicks]->Command(CMD_SETTEXT, line+j, 0L);
		NumTicks++;
		}
	ca->Close();
	if(!NumTicks && Ticks) {
		free(Ticks);
		NumTicks = 0;
		}
	delete ca;
}

void
Axis::RegGO(void *n)
{
	long i;

	if(n) {
		for(i = 0; Ticks && i< NumTicks; i++) if(Ticks[i]) Ticks[i]->RegGO(n);
		if(axisLabel) axisLabel->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Axis::FileIO(int rw)
{
	char *tickfile = 0L;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"moveable", typNZINT, &moveable, 0L},
		{(char*)"sAxLine", typLFLOAT, &sizAxLine, 0L},
		{(char*)"sAxTick", typLFLOAT, &sizAxTick, 0L},
		{(char*)"BrkGap", typLFLOAT, &brkgap, 0L},
		{(char*)"BrkSymSize", typLFLOAT, &brksymsize, 0L},
		{(char*)"BrkSym", typINT, &brksym, 0L},
		{(char*)"sTickLabel", typLFLOAT, &sizAxTickLabel, 0L},
		{(char*)"tick_angle", typNZLFLOAT, &tick_angle, 0L},
		{(char*)"LbDist", typNZLFPOINT, &lbdist, 0L},
		{(char*)"TickLbDist", typNZLFPOINT, &tlbdist, 0L}, 
		{(char*)"tlbDef", typTXTDEF, &tlbdef, 0L},
		{(char*)"Color", typDWORD, &colAxis, 0L},
		{(char*)"AxisDef", typPTRAXDEF, &axis},
		{(char*)"TLBformat", typTEXT, &TLBformat},
		{(char*)"GridLine", typLINEDEF, &GridLine, 0L},
		{(char*)"GridType", typINT, &gl_type, 0L},
		{(char*)"Ticks", typOBJLST, &Ticks, &NumTicks},
		{(char*)"Label", typGOBJ, &axisLabel, 0L},
		{ (char*)"Name", typTEXT, &name, 0L },
		{ (char*)"TickFile", typTEXT, &tickfile, 0L },
		{(char*)"ssRefTV", typTEXT, &ssMATval, 0L},
		{(char*)"ssRefTL", typTEXT, &ssMATlbl, 0L},
		{(char*)"ssRefMT", typTEXT, &ssMITval, 0L},
		{(char*)"g_type", typINT, &grad_type, 0L},
		{(char*)"g_col_0", typDWORD, &gCol_0, 0L},
		{(char*)"g_col_1", typDWORD, &gCol_1, 0L},
		{(char*)"g_col_2", typDWORD, &gCol_2, 0L},
		{(char*)"Name", typTEXT, &name, 0L},
		{(char*)"gTrans", typLAST | typDWORD, &gTrans, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		sizAxLine = DefSize(SIZE_AXIS_LINE);
		sizAxTick = DefSize(SIZE_AXIS_TICKS);
		sizAxTickLabel = DefSize(SIZE_TICK_LABELS);
		colAxis = parent ? parent->GetColor(COL_AXIS) : defs.Color(COL_AXIS);
		memcpy(&GridLine, defs.GetGridline(), sizeof(LineDEF));
		brksymsize = DefSize(SIZE_TICK_LABELS);
		brkgap = DefSize(SIZE_AXIS_TICKS);
		brksym = 2;
		tlbdef.ColTxt = parent ? parent->GetColor(COL_AXIS) : defs.Color(COL_AXIS);
		tlbdef.ColBg = parent ? parent->GetColor(COL_BG) : defs.Color(COL_AXIS);
		tlbdef.RotBL = tlbdef.RotCHAR = 0.0f;
		tlbdef.fSize = DefSize(SIZE_TICK_LABELS);	tlbdef.Align = TXA_VCENTER | TXA_HCENTER;
		tlbdef.Style = TXS_NORMAL;					tlbdef.Mode = TXM_TRANSPARENT;
		tlbdef.Font = FONT_HELVETICA;				atv = 0L;
		tlbdef.text = 0L;		l_segs = 0L;		nl_segs = 0;
		drawOut = scaleOut = 0L;					bModified = false;
		mrc.left = mrc.right = mrc.top = mrc.bottom = mrc1.left = mrc1.right = mrc1.top = mrc1.bottom = 0;
		mo = mo1 = 0L;		grad_type = 1;		gTrans = 0x00000000L;
		gCol_0 = 0x00ffffffL;	gCol_1 = 0x00ff0000L;	gCol_2 = 0x000000ffL;
		tmpTLBformat = 0L;			Id = GO_AXIS;		hidden = 0;
		pIsoPG = 0L;				sIsoPG = 0;			nIsoPG = 0;
		return true;
	case FILE_READ:
		if(axisLabel)DeleteGO(axisLabel);
		if(tickfile) free(tickfile);
		if(ssMATval) free(ssMATval);
		if(ssMATlbl) free(ssMATlbl);
		if(ssMITval) free(ssMITval);
		tickfile = 0L;
		if(ExecInput(Desc) && tickfile && tickfile[0]){
			TickFile(tickfile);
			free(tickfile);					tickfile = 0L;
			}
		if(axis) axis->owner = this;
		if(axisLabel)axisLabel->parent = this;
		if(Ticks) for(i = 0; i < NumTicks; i++) if(Ticks[i]) Ticks[i]->parent = this;
		if(axis && !TLBformat)TLBformat = GetNumFormat(floor(log10(fabs(axis->max - axis->min))));
		return true;
	case FILE_WRITE:
		//do all ticks
		for(i = 0; Ticks && i< NumTicks; i++) if(Ticks[i]) Ticks[i]->FileIO(rw);
		if(axisLabel) axisLabel->FileIO(rw);
		if((type & 0x04)!=4) Desc[19].type |= typLAST;
		return ExecOutput(Notary->RegisterGO(this), (char*)"Axis", Desc);
		}
	return false;
}

void
ContourPlot::RegGO(void *n)
{
	int i;

	if(n) {
		if(Symbols) for(i = 0; i < nSym; i++) if(Symbols[i]) Symbols[i]->RegGO(n);
		if(Labels) for(i = 0; i < nLab; i++) if(Labels[i]) Labels[i]->RegGO(n);
		if(zAxis) zAxis->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
ContourPlot::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"xBounds", typLFPOINT, &xBounds, 0L},
		{(char*)"yBounds", typLFPOINT, &yBounds, 0L},
		{(char*)"zBounds", typLFPOINT, &zBounds, 0L},
		{(char*)"srz", typNZLFLOAT, &sr_zval, 0L},
		{(char*)"flags", typDWORD, &flags, 0L},
		{(char*)"zDef", typAXDEF, &z_axis, 0L},
		{(char*)"zAxis", typGOBJ, &zAxis, 0L},
		{(char*)"Values", typFPLST3D, &val, &nval},
		{(char*)"Symbols", typOBJLST, &Symbols, &nSym},
		{(char*)"Labels", typOBJLST, &Labels, &nLab},
		{(char*)"ssRefX", typTEXT, &ssRefX, 0L},
		{(char*)"ssRefY", typTEXT, &ssRefY, 0L},
		{(char*)"ssRefZ", typLAST | typTEXT, &ssRefZ, 0L}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		sr_zval = 0.0;		flags = 0L;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "Contour Plot (%s)", name);
#else
			i = sprintf(TmpTxt, "Contour Plot (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		ExecInput(Desc);
		if(Symbols) for(i = 0; i < nSym; i++) if(Symbols[i]) Symbols[i]->parent=this;
		if(Labels) for(i = 0; i < nLab; i++) if(Labels[i]) Labels[i]->parent=this;
		if(zAxis) zAxis->parent = this;
		return true;
	case FILE_WRITE:
		if(Symbols) for(i = 0; i < nSym; i++) if(Symbols[i]) Symbols[i]->FileIO(rw);
		if(Labels) for(i = 0; i < nLab; i++) if(Labels[i]) Labels[i]->FileIO(rw);
		if(zAxis) zAxis->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"ContourPlot", Desc);
		}
	return false;
}

void
Plot3D::RegGO(void *n)
{
	int i;

	if(n) {
		if(plots) for(i = 0; plots && i< nPlots; i++) if(plots[i]) plots[i]->RegGO(n);
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Plot3D::FileIO(int rw)
{
	fPOINT3D rot_vec, rot_ang;
	descIO Desc[] = {
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"xBounds", typLFPOINT, &xBounds, 0L},
		{(char*)"yBounds", typLFPOINT, &yBounds, 0L},
		{(char*)"zBounds", typLFPOINT, &zBounds, 0L},
		{(char*)"LightSrc", typLFPOINT, &light_source, 0L },
		{(char*)"Corner1", typPOINT3D, &cub1, 0L},
		{(char*)"Corner2", typPOINT3D, &cub2, 0L},
		{(char*)"Center", typPOINT3D, &rotC, 0L},
		{(char*)"rot_vec", typPOINT3D, &rot_vec, 0L},
		{(char*)"rot_ang", typPOINT3D, &rot_ang, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"z_axis", typNZINT, &use_zaxis, 0L},
		{(char*)"Axes", typOBJLST, &Axes, &nAxes},
		{(char*)"Plots", typLAST | typOBJLST, &plots, &nPlots}};
	int i;

	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		drag = 0L;		moveable = 1;
		//set up RotDef
		RotDef[0] = 0.919384;		RotDef[1] = 0.389104;		RotDef[2] = -0.057709;
		RotDef[3] = 0.327146;		RotDef[4] = 0.944974;		RotDef[5] = 1.0-RotDef[4];
		cub1.fx = DefSize(SIZE_GRECT_LEFT) + DefSize(SIZE_DRECT_LEFT);
		cub2.fx = DefSize(SIZE_GRECT_LEFT) + DefSize(SIZE_DRECT_RIGHT);
		cub1.fy = DefSize(SIZE_GRECT_TOP) + DefSize(SIZE_DRECT_BOTTOM);
		cub2.fy = DefSize(SIZE_GRECT_TOP) + DefSize(SIZE_DRECT_TOP);
		cub1.fy += DefSize(SIZE_DRECT_TOP);	cub2.fy += DefSize(SIZE_DRECT_TOP);
		cub1.fz = 0.0;
		cub2.fz = DefSize(SIZE_DRECT_BOTTOM) - DefSize(SIZE_DRECT_TOP);
		rotC.fx = (cub1.fx + cub2.fx)/2.0;		rotC.fy = (cub1.fy + cub2.fy)/2.0;
		rotC.fz = (cub1.fz + cub2.fz)/2.0;
		dispObs = 0L;		nmaxObs = 0;	crea_flags = 0L;		dirty = true;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		Sc_Plots = 0L;		nscp = 0;
		light_source.fx = 20.0;			light_source.fy = 16.0;
		memcpy(&track_line, defs.GetGridline(), sizeof(LineDEF));
		track_line.color = 0x00000ff;		//red track line for virtual trackball
		mo = 0L;		mrc.left = mrc.top = mrc.right = mrc.bottom = 0;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "3D-root (%s)", name);
#else
			i = sprintf(TmpTxt, "3D-root (%s)", name);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		rot_vec.fx = 0.919384;	rot_vec.fy = 0.389104;	rot_vec.fz = -0.057709;
		rot_ang.fx = 0.327146;	rot_ang.fy = 0.944974;	rot_ang.fz = 0.055026;
		ExecInput(Desc);
		RotDef[0] = rot_vec.fx;	RotDef[1] = rot_vec.fy;	RotDef[2] = rot_vec.fz;
		RotDef[3] = rot_ang.fx;	RotDef[4] = rot_ang.fy;	RotDef[5] = rot_ang.fz;
		return true;
	case FILE_WRITE:
		rot_vec.fx = RotDef[0];	rot_vec.fy = RotDef[1];	rot_vec.fz = RotDef[2];
		rot_ang.fx = RotDef[3];	rot_ang.fy = RotDef[4];	rot_ang.fz = RotDef[5];
		//do all plots
		for (i = 0; plots && i < nPlots; i++) {
			if (plots[i]) plots[i]->FileIO(rw);
			}
		//do all axes
		if (Axes) for (i = 0; i < nAxes; i++) {
			if (Axes[i]) Axes[i]->FileIO(rw);
			}
		return ExecOutput(Notary->RegisterGO(this), (char*)"Plot3D", Desc);
		}
	return false;
}

void
Func3D::RegGO(void *)
{
}

bool
Func3D::FileIO(int rw)
{
	fPOINT3D rot_vec, rot_ang;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"xBounds", typLFPOINT, &xBounds, 0L},
		{(char*)"yBounds", typLFPOINT, &yBounds, 0L},
		{(char*)"zBounds", typLFPOINT, &zBounds, 0L},
		{(char*)"Corner1", typPOINT3D, &cub1, 0L},
		{(char*)"Corner2", typPOINT3D, &cub2, 0L},
		{(char*)"Center", typPOINT3D, &rotC, 0L},
		{(char*)"rot_vec", typPOINT3D, &rot_vec, 0L},
		{(char*)"rot_ang", typPOINT3D, &rot_ang, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"z_axis", typNZINT, &use_zaxis, 0L},
		{(char*)"Axes", typOBJLST, &Axes, &nAxes},
		{(char*)"Grid", typGOBJ, &gob, 0L},
		{(char*)"x1", typNZLFLOAT, &x1, 0L},
		{(char*)"x2", typNZLFLOAT, &x2, 0L},
		{(char*)"xstep", typNZLFLOAT, &xstep, 0L},
		{(char*)"z1", typNZLFLOAT, &x1, 0L},
		{(char*)"z2", typNZLFLOAT, &x2, 0L},
		{(char*)"zstep", typNZLFLOAT, &xstep, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"f_xz", typTEXT, &cmdxy, 0L},
//		{(char*)"Plots", typOBJLST, &plots, &nPlots},
		{(char*)"param", typLAST | typTEXT, &param, 0L}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		x1 = -20.0;		x2 = 20.0;	xstep = 2.0;
		z1 = -20.0;		z2 = 20.0;	zstep = 2.0;
		gda = 0L;		gob = 0L;
		param = cmdxy = 0L;
		Line.width = DefSize(SIZE_HAIRLINE);		Line.patlength = DefSize(SIZE_PATLENGTH);
		Line.color = Line.pattern = 0x0L;			Fill.color = 0x00c0c0c0;
		Fill.color2 = 0x00ffffff;					Fill.hatch = 0L;
		Fill.type = FILL_LIGHT3D;					Fill.scale = 1.0;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "3D function (Plot %d)", cPlots);
#else
			i = sprintf(TmpTxt, "3D function (Plot %d)", cPlots);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		rot_vec.fx = 0.919384;	rot_vec.fy = 0.389104;	rot_vec.fz = -0.057709;
		rot_ang.fx = 0.327146;	rot_ang.fy = 0.944974;	rot_ang.fz = 0.055026;
		ExecInput(Desc);
		RotDef[0] = rot_vec.fx;	RotDef[1] = rot_vec.fy;	RotDef[2] = rot_vec.fz;
		RotDef[3] = rot_ang.fx;	RotDef[4] = rot_ang.fy;	RotDef[5] = rot_ang.fz;
		return true;
	case FILE_WRITE:
		rot_vec.fx = RotDef[0];	rot_vec.fy = RotDef[1];	rot_vec.fz = RotDef[2];
		rot_ang.fx = RotDef[3];	rot_ang.fy = RotDef[4];	rot_ang.fz = RotDef[5];
		//do all plots
		for (i = 0; plots && i < nPlots; i++) {
			if (plots[i]) plots[i]->FileIO(rw);
			}
		//do all axes
		if (Axes) for (i = 0; i < nAxes; i++) {
			if (Axes[i]) Axes[i]->FileIO(rw);
			}
		ExecOutput(Notary->RegisterGO(this), (char*)"Func3D", Desc);
		}
	return false;
}

void
FitFunc3D::RegGO(void *)
{
}

bool
FitFunc3D::FileIO(int rw)
{
	fPOINT3D rot_vec, rot_ang;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"hide", typNZINT, &hidden, 0L},
		{(char*)"ssXref", typTEXT, &ssXref, 0L},
		{(char*)"ssYref", typTEXT, &ssYref, 0L},
		{(char*)"ssZref", typTEXT, &ssZref, 0L},
		{(char*)"xBounds", typLFPOINT, &xBounds, 0L},
		{(char*)"yBounds", typLFPOINT, &yBounds, 0L},
		{(char*)"zBounds", typLFPOINT, &zBounds, 0L},
		{(char*)"Corner1", typPOINT3D, &cub1, 0L},
		{(char*)"Corner2", typPOINT3D, &cub2, 0L},
		{(char*)"Center", typPOINT3D, &rotC, 0L},
		{(char*)"rot_vec", typPOINT3D, &rot_vec, 0L},
		{(char*)"rot_ang", typPOINT3D, &rot_ang, 0L},
		{(char*)"x_axis", typNZINT, &use_xaxis, 0L},
		{(char*)"y_axis", typNZINT, &use_yaxis, 0L},
		{(char*)"z_axis", typNZINT, &use_zaxis, 0L},
		{(char*)"Axes", typOBJLST, &Axes, &nAxes},
		{(char*)"Plots", typOBJLST, &plots, &nPlots},
		{(char*)"x1", typNZLFLOAT, &x1, 0L},
		{(char*)"x2", typNZLFLOAT, &x2, 0L},
		{(char*)"xstep", typNZLFLOAT, &xstep, 0L},
		{(char*)"z1", typNZLFLOAT, &x1, 0L},
		{(char*)"z2", typNZLFLOAT, &x2, 0L},
		{(char*)"zstep", typNZLFLOAT, &xstep, 0L},
		{(char*)"maxiter", typNZINT, &maxiter, 0L},
		{(char*)"Line", typLINEDEF, &Line, 0L},
		{(char*)"Fill", typFILLDEF, &Fill, 0L},
		{(char*)"f_xz", typTEXT, &cmdxy, 0L},
		{(char*)"param", typLAST | typTEXT, &param, 0L}};
	int i;

	switch(rw) {
	case SAVE_VARS:
		return SaveVarGO(Desc);
	case INIT_VARS:
		x1 = -20.0;		x2 = 20.0;	xstep = 2.0;
		z1 = -20.0;		z2 = 20.0;	zstep = 2.0;
		gda = 0L;		gob = 0L;
		conv = 1.0e-15;	maxiter = 100;
		param = cmdxy = ssXref = ssYref = ssZref = 0L;
		Line.width = DefSize(SIZE_HAIRLINE);
		Line.patlength = DefSize(SIZE_PATLENGTH);
		Line.color = Line.pattern = 0x0L;
		Fill.color = 0x00c0c0c0;
		Fill.color2 = 0x00ffffff;
		Fill.hatch = 0L;
		Fill.type = FILL_LIGHT3D;
		if(name) {
#ifdef USE_WIN_SECURE
			i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "FitFunc3D (Plot %d)", cPlots);
#else
			i = sprintf(TmpTxt, "FitFunc3D (Plot %d)", cPlots);
#endif
			free(name);		name = rlp_strdup(TmpTxt);
			}
		return true;
	case FILE_READ:
		rot_vec.fx = 0.919384;	rot_vec.fy = 0.389104;	rot_vec.fz = -0.057709;
		rot_ang.fx = 0.327146;	rot_ang.fy = 0.944974;	rot_ang.fz = 0.055026;
		ExecInput(Desc);
		RotDef[0] = rot_vec.fx;	RotDef[1] = rot_vec.fy;	RotDef[2] = rot_vec.fz;
		RotDef[3] = rot_ang.fx;	RotDef[4] = rot_ang.fy;	RotDef[5] = rot_ang.fz;
		return true;
	case FILE_WRITE:
		rot_vec.fx = RotDef[0];	rot_vec.fy = RotDef[1];	rot_vec.fz = RotDef[2];
		rot_ang.fx = RotDef[3];	rot_ang.fy = RotDef[4];	rot_ang.fz = RotDef[5];
		//do all plots
		for(i = 0; plots && i< nPlots; i++) if(plots[i]) plots[i]->FileIO(rw);
		//do all axes
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		ExecOutput(Notary->RegisterGO(this), (char*)"FitFunc3D", Desc);
		}
	return false;
}

void
Graph::RegGO(void *n)
{
	int i;

	if(n) {
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->RegGO(n);
		if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]) Axes[i]->RegGO(n);
		if (theGrid) ((notary*)n)->AddRegGO(theGrid);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Graph::FileIO(int rw)
{
	int ixax, iyax, units;
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
		{(char*)"Units", typNZINT, &units, 0L},
		{(char*)"bClip", typNZINT, &bClip, 0L},
		{(char*)"tickstyle", typNZINT, &tickstyle, 0L},
		{(char*)"Scale", typNZLFLOAT, &scale, 0L},
		{(char*)"GRect", typFRECT, &GRect, 0L},
		{(char*)"DRect", typFRECT, &DRect, 0L},
		{(char*)"Bounds", typFRECT, &Bounds, 0L},
		{(char*)"ColFrame", typDWORD, &ColGR, 0L},
		{(char*)"ColFrameL", typDWORD, &ColGRL, 0L},
		{(char*)"ColRec", typDWORD, &ColDR, 0L},
		{(char*)"ColAxis", typDWORD, &ColAX, 0L},
		{(char*)"Xaxis", typAXDEF, &x_axis, 0L},
		{(char*)"Yaxis", typAXDEF, &y_axis, 0L},
		{(char*)"DefXAxis", typINT, &ixax, 0L},
		{(char*)"DefYAxis", typINT, &iyax, 0L},
		{(char*)"Axes", typOBJLST, &Axes, &NumAxes},
		{(char*)"Grid", typGOBJ, &theGrid, 0L},
		{(char*)"Plots", typLAST | typOBJLST, &Plots,&NumPlots}};
	int i;
	bool bConvert = false;

	ixax = iyax = -1;
	switch(rw) {
	case INIT_VARS:
		InitVarsGO(Desc);
		OwnDisp = bDialogOpen = false;			dirty = true;
		GRect.Ymin = defs.GetSize(SIZE_GRECT_TOP);			GRect.Ymax = defs.GetSize(SIZE_GRECT_BOTTOM);
		GRect.Xmin = defs.GetSize(SIZE_GRECT_LEFT);			GRect.Xmax = defs.GetSize(SIZE_GRECT_RIGHT);
		DRect.Ymin = defs.GetSize(SIZE_DRECT_TOP);			DRect.Ymax = defs.GetSize(SIZE_DRECT_BOTTOM);
		DRect.Xmin = defs.GetSize(SIZE_DRECT_LEFT);			DRect.Xmax = defs.GetSize(SIZE_DRECT_RIGHT);
		ColGR = defs.Color(COL_GRECT);						ColGRL = defs.Color(COL_GRECTLINE);
		ColDR = defs.Color(COL_DRECT);						ColBG = defs.Color(COL_BG);
		ColAX = defs.Color(COL_AXIS);
		x_axis.max = y_axis.max = 1.0;					x_axis.owner = y_axis.owner = (void *)this;
		rcDim.left = rcDim.right = rcDim.top = rcDim.bottom = 0;
		rcUpd.left = rcUpd.right = rcUpd.top = rcUpd.bottom = 0;
		CurrGO = 0L;		Disp = 0L;			Sc_Plots = 0L;
		AxisTempl = 0;		nscp = 0;			CurrDisp = 0L;
		ToolMode = TM_STANDARD;	bModified = false;		zoom_def = 0L;
		tl_pts = 0L;		tl_nPts = 0;		tickstyle = zoom_level = 0;
		frm_g = frm_d = 0L;	PasteObj = 0L;		filename = 0L;
		rc_mrk.left = -1; rc_mrk.right = -1; rc_mrk.top = -1; rc_mrk.bottom = -1;
		ClipRC.left = ClipRC.top = ClipRC.right = ClipRC.bottom = 0;
		return true;
	case FILE_READ:
		if((bConvert =ExecInput(Desc)) && ixax>=0 && iyax >=0 && Axes && 
			NumAxes >= ixax+1 && NumAxes >= iyax) {
			if(Axes[ixax]) Axes[ixax]->Command(CMD_SET_AXDEF, &x_axis, 0L);
			if(Axes[iyax]) Axes[iyax]->Command(CMD_SET_AXDEF, &y_axis, 0L);
			return true;
			}
		//should we set dUnits in def_vars if different??
		return bConvert;
	case FILE_WRITE:
		units = defs.units();
		bModified = false;		if(scale == 1.0) scale = 0.0;
		//find default axes
		if(Axes) for(i = 0; Axes && i < NumAxes; i++) {
			if(Axes[i] && Axes[i]->GetAxis() == &x_axis) ixax = i;
			else if(Axes[i] && Axes[i]->GetAxis() == &y_axis) iyax = i;
			}
		if(Id == GO_GRAPH)RegGO(Notary);
		//do all plots
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		//do all axes
		if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]) Axes[i]->FileIO(rw);
		//and the grid if defined
		if (theGrid)theGrid->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Graph", Desc);
		}
	return false;
}

void
Page::RegGO(void *n)
{
	int i;

	if(n) {
		if(Plots) for(i = 0; i< NumPlots; i++) 
			if(Plots[i] && Plots[i]->Id != GO_GRAPH) Plots[i]->RegGO(n);
		if (theGrid) ((notary*)n)->AddRegGO(theGrid);
		((notary*)n)->AddRegGO(this);
		}
}

bool
Page::FileIO(int rw)
{
	descIO Desc[] = {
		{(char*)"Type", typNZLONG, &type, 0L},
//		{(char*)"Units", typNZINT, &defs.dUnits, 0L},
		{(char*)"GRect", typFRECT, &GRect, 0L},
		{ (char*)"Grid", typGOBJ, &theGrid, 0L },
		{ (char*)"Plots", typLAST | typOBJLST, &Plots, &NumPlots } };
	int i;

	switch(rw) {
	case INIT_VARS:
		//assume that Graph::FileIO(INIT_VARS) has been executed
		GRect.Xmin = GRect.Ymin = 0.0;
		GetPaper(&GRect.Xmax, &GRect.Ymax);
		ColBG = 0x00e8e8e8L;
		LineDef.width = 0.0;	LineDef.patlength = 1.0;
		LineDef.color = LineDef.pattern = 0x0L;
		FillDef.type = FILL_NONE;
		FillDef.color = 0x00ffffffL;	//use white paper
		FillDef.scale = 1.0;
		FillDef.hatch = 0L;		filename =0L;
		return true;
	case FILE_READ:
		Graph::FileIO(rw);
		return true;
	case FILE_WRITE:
		//do all plots
		bModified = false;
		if(Id == GO_PAGE)RegGO(Notary);
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->FileIO(rw);
		//and the grid if defined
		if (theGrid)theGrid->FileIO(rw);
		return ExecOutput(Notary->RegisterGO(this), (char*)"Page", Desc);
		}
	return false;
}

bool
DefsRW::FileIO(int rw)
{
	unsigned char *fil1 = NULL, *fil2 = NULL, *fil3 = NULL, *fil4 = NULL, *fil5 = NULL,  *fil6 = NULL;
	char	*sColSep = 0L, *sDecPoint = 0L;
	static int units;
	fPOINT3D dlg_adj;
	descIO Desc[] = {
		{(char*)"dUnits", typINT, &units, 0L},
		{(char*)"dtHeight", typINT, &dlgtxtheight, 0L},
		{(char*)"ss_txt", typLFLOAT, &defs.ss_txt, 0L},
		{(char*)"fmt_date", typTEXT, &defs.fmt_date, 0L},
		{(char*)"fmt_datetime", typTEXT, &defs.fmt_datetime, 0L},
		{(char*)"fmt_time", typTEXT, &defs.fmt_time, 0L},
		{(char*)"font_sans", typTEXT, &defs.font_sans, 0L},
		{(char*)"font_ser", typTEXT, &defs.font_ser, 0L},
		{(char*)"font_fix", typTEXT, &defs.font_fix, 0L},
		{(char*)"font_greek", typTEXT, &defs.font_greek, 0L},
		{(char*)"curr_path", typTEXT, &defs.currPath, 0L},
		{(char*)"pg_width", typLFLOAT, &defs.pg_width, 0L},
		{(char*)"pg_height", typLFLOAT, &defs.pg_height, 0L},
		{(char*)"DlgSizeAdj", typPOINT3D, &dlg_adj, 0L},
		{(char*)"ColSep", typTEXT, &sColSep, 0L},
		{(char*)"DecPoint", typTEXT, &sDecPoint, 0L },
		{(char*)"File1", typTEXT, &fil1, 0L },
		{(char*)"File2", typTEXT, &fil2, 0L},
		{(char*)"File3", typTEXT, &fil3, 0L},
		{(char*)"File4", typTEXT, &fil4, 0L},
		{(char*)"File5", typTEXT, &fil5, 0L},
		{(char*)"File6", typLAST | typTEXT, &fil6, 0L}};

	unsigned char *fils[6] = {fil1, fil2, fil3, fil4, fil5, fil6 };
	char **files[6] = { (char**)&defs.File1, (char**)&defs.File2, (char**)&defs.File3, 
		(char**)&defs.File4, (char**)&defs.File5, (char**)&defs.File6 };
	int i, j;

		switch (rw) {
	case FILE_READ:
		dlg_adj.fx = defs.DlgSizeAdj.x;	dlg_adj.fy = defs.DlgSizeAdj.y;	dlg_adj.fz = defs.DlgSizeAdj.z;
		ExecInput(Desc);
		if (sColSep) {
			defs.ColSep[0] = sColSep[0];		free(sColSep);
			}
		if (sDecPoint) {
			defs.DecPoint[0] = sDecPoint[0];	free(sDecPoint);
			}
		fils[0] = fil1;		fils[1] = fil2;		fils[2] = fil3;
		fils[3] = fil4;		fils[4] = fil5;		fils[5] = fil6;
		for (i = j = 0; i < 6; i++) {
			if (fils[i] && fils[i][0] && FileExist((char*)fils[i])) *files[j++] = (char*)fils[i];
			}
		defs.DlgSizeAdj.x = iround(dlg_adj.fx);		defs.DlgSizeAdj.y = iround(dlg_adj.fy);
		defs.DlgSizeAdj.y = iround(dlg_adj.fy);
		while (j < 6) *files[j++] = NULL;
		//check for plausibility
		if(defs.ss_txt < 0.3 || defs.ss_txt > 3.0) defs.ss_txt = 0.9;
		if (defs.units() < 0 || defs.units() > 2) defs.set_units(0);
#ifdef _WINDOWS
		if(dlgtxtheight < 5 || dlgtxtheight > 60) dlgtxtheight = 16;
#else
		if(dlgtxtheight < 5 || dlgtxtheight > 60) dlgtxtheight = 10;
#endif
#ifdef _WINDOWS
		if (!defs.font_sans) defs.font_sans = rlp_strdup((char*)"Arial");
		if (!defs.font_ser) defs.font_ser = rlp_strdup((char*)"Times New Roman");
		if (!defs.font_fix) defs.font_fix = rlp_strdup((char*)"Courier New");
		if (!defs.font_greek) defs.font_greek = rlp_strdup((char*)"Times New Roman");
#else	//*nix system
		if (!defs.font_sans) defs.font_sans = rlp_strdup((char*)"Helvetica");
		if (!defs.font_ser) defs.font_ser = rlp_strdup((char*)"Times");
		if (!defs.font_fix) defs.font_fix = rlp_strdup((char*)"Courier");
		if (!defs.font_greek) defs.font_greek = rlp_strdup((char*)"Times");
#endif
		defs.NoAmp(defs.File1);			defs.NoAmp(defs.File2);
		defs.NoAmp(defs.File3);			defs.NoAmp(defs.File4);
		defs.NoAmp(defs.File5);			defs.NoAmp(defs.File6);
		if (defs.pg_width > 0.0 && defs.pg_height > 0.0) {
			if (defs.pg_width > defs.pg_height)FindPaper(defs.pg_height, defs.pg_width, 0.01);
			else FindPaper(defs.pg_width, defs.pg_height, 0.01);
			}
		defs.set_units(units);
		return true;
	case FILE_WRITE:
		units = defs.units();
		sColSep = defs.ColSep;			sDecPoint = defs.DecPoint;
		if(!Notary) Notary = new notary();
#ifdef USE_WIN_SECURE
		_unlink(defs.IniFile);
#else
		unlink(defs.IniFile);
#endif
		dlg_adj.fx = defs.DlgSizeAdj.x;	dlg_adj.fy = defs.DlgSizeAdj.y;	dlg_adj.fz = defs.DlgSizeAdj.z;
		fil1 = defs.File1;		fil2 = defs.File2;		fil3 = defs.File3;
		fil4 = defs.File4;		fil5 = defs.File5;		fil6 = defs.File6;
		iFile = OpenOutputFile(defs.IniFile);
		if(iFile >=0) {
			ExecOutput(-1, (char*)"Defaults", Desc);
			}
		CloseOutputFile();
		return true;
		}
	return false;
}
#ifndef _WINDOWS
#pragma GCC diagnostic warning "-Wmissing-field-initializers"
#endif

