//Export.cpp, Copyright (c) 2002-2024 R.Lackner
//export graph files
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include "Version.h"
#include "rlp_strings.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags
#ifdef USE_WIN_SECURE
	#include <share.h>			//I/O flags
#endif

#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif

extern char TmpTxt[];
extern def_vars defs;
extern GraphObj *CurrGO;
extern notary *Notary;

static char *str_ind = (char*)"                              ";

#define svg_scale 0.1

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// export to *.svg file (scalable vector graphic)
// this code is based on information from the following books
// H. Spona, 'Das Einsteigerseminar SVG-Webgrafiken mit XML',
//     vmi, ISBN 3-8266-7181-3
// M. Salathe, 'SVG Scalabe Vector Graphics ...fuer professionelle Einsteiger',
//     M&T, ISBN 3-8272-6188-0
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class ExportSVG:public anyOutput {
public:
	HatchOut *hgo;
	char *out_buff;
	long out_pos, out_size;

	ExportSVG(GraphObj *g, DWORD flags = 0);
	~ExportSVG();
	bool ClipRect(RECT *rec);
	bool SetLine(LineDEF *lDef);
	bool SetFill(FillDEF *fill);
	bool SetTextSpec(TextDEF *set);
	bool StartPage();
	bool EndPage();
	bool oCircle(int x1, int y1, int x2, int y2, char* nam = 0L);
	bool foCircle(double, double, double, double, char *nam = 0L);
	bool oPolyline(POINT * pts, int cp, char *nam = 0L);
	bool oBezier(POINT *pts, int cp, POINT **bpts = 0L, long *bcp = 0L, bool dodraw = true);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool foRectangle(double, double, double, double, char *nam = 0L);
	bool oSolidLine(POINT *p);
	bool foSolidLine(lfPOINT *);
	bool oTextOut(int x, int y, unsigned char *txt, int cb);
	bool oTextOutW(int x, int y, w_char *txt, int cb);
	bool oPolygon(POINT *pts, int cp, char *nam = 0L);
	bool foPolygon(lfPOINT *pts, int cp, char *nam = 0L);
	bool oSolidPolygon(POINT *pts, int cp);
	bool GradPG(fPOINT3D *pts, long npt, double Lo, double Hi, double pLo, double pHi, lfPOINT *grad);

private:
	int cb_out, cb_ind, clip_level, group_level, grad_id;
	double dLineWidth;
	char clip_url[50];
	bool bUseGroupLine, bOutputPending;
	GraphObj *go;
	char output[140], tHatchStyle[80];
	DWORD flags;

	bool com_TextOut(int x, int y, unsigned char *txt, int cb);
	void Indent(bool ind);
	void AddToOutput(char *txt, int len);
	char *ColName(DWORD col);
	char *Transparency(DWORD col, int type);
};

ExportSVG::ExportSVG(GraphObj *g, DWORD flg)
{
	hgo = 0L;		group_level = 0;	DeskRect.left = DeskRect.top = 0;
	DeskRect.right = DeskRect.bottom = 0x4fffffff;
	dFillCol = 0xffffffffL;		hres = vres = 1000.0f;	go = g;
	out_buff = (char*)malloc(out_size = 4000);
	out_pos = 0; grad_id = 0;
	flags = flg;		cb_ind = 3;			clip_level = 0;
	rlp_strcpy(tHatchStyle, 80, (char*)"style=\"stroke:black; stroke-width:1\"");
	bUseGroupLine = false;			OC_type = OC_EXPORT;
	Notary->ValPtr(this, true);
}

ExportSVG::~ExportSVG()
{
	Notary->ValPtr(this, false);
	if (hgo) delete hgo;
}

bool
ExportSVG::ClipRect(RECT *rec)
{
	if (!rec) {
		Indent(false);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		if (hasClip) {
			add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>  // end clipping\n", 0);
			group_level--;
			}
		hasClip = false;
		}
	else {
		memcpy(&ClipRC, rec, sizeof(RECT));
		clip_level++;
#ifdef USE_WIN_SECURE
		sprintf_s(clip_url, 49, "clip_%d", clip_level);
#else
		sprintf(clip_url, "clip_%d", clip_level);
#endif
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<defs>\n", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"   <clipPath id = \"", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, clip_url, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\">\n", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind); 
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"      <rect x = \"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, rec->left *svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" y = \"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, rec->top *svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" width = \"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, (rec->right - rec->left) *svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" height = \"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, (rec->bottom - rec->top) *svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" />\n", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"   </clipPath>\n", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</defs>\n", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g clip-path = \"url(#", 0);
		group_level++;
		add_to_buff(&out_buff, &out_pos, &out_size, clip_url, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)")\">   //start clipping\n", 0);
		Indent(true);		hasClip = true;
		}
	return false;
}

bool
ExportSVG::SetLine(LineDEF *lDef)
{
	anyOutput::SetLine(lDef);
	dLineWidth = floor(un2fix(lDef->width)*10.0)/10.0;
	if (2.0 > dLineWidth) dLineWidth = 2.0;
	return true;
}

bool
ExportSVG::SetFill(FillDEF *fill)
{
	double iL; 

	if(!fill) return false;
	anyOutput::SetFill(fill);
	if ((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		if(fill->hatch) {
			iL = ((double)iround(un2fix(fill->hatch->width)*10.0)/10.0);
#ifdef USE_WIN_SECURE
			sprintf_s(tHatchStyle, 80, "style=\"fill:none; stroke:%s; stroke-width:%.2g\"",
				ColName(fill->hatch->color), iL * svg_scale);
#else
			sprintf(tHatchStyle, "style=\"fill:none; stroke:%s; stroke-width:%.2g\"",
				ColName(fill->hatch->color), iL *svg_scale);
#endif
			}
		}
	else if (!(fill->type & FILL_KEEP_ALIVE)) {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	dFillCol = fill->color;
	dFillCol2 = fill->color2;
	return true;
}

bool
ExportSVG::SetTextSpec(TextDEF *set)
{
	if(set->fSize > 0.0) {
		if((set->Style & TXS_SUPER) || (set->Style & TXS_SUB))
			set->iSize = un2iy(set->fSize * 0.71);
		else set->iSize = un2iy(set->fSize);
		}
	if(!set->iSize) return false;
	return anyOutput::SetTextSpec(set);
}

bool 
ExportSVG::StartPage()
{
	int w, h;

	if(!go) return false;
	ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
	w = un2ix(go->GetSize(SIZE_GRECT_RIGHT) - go->GetSize(SIZE_GRECT_LEFT));
	h = un2iy(go->GetSize(SIZE_GRECT_BOTTOM) - go->GetSize(SIZE_GRECT_TOP));
	w++; h++;
	VPorg.fy = -un2fiy(go->GetSize(SIZE_GRECT_TOP));
	VPorg.fx = -un2fix(go->GetSize(SIZE_GRECT_LEFT));
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<?xml version=\"1.0\"?>\n", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<!-- Created with RLPlot Version ", 0L);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)SZ_VERSION, 0L);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"  -->\n", 0L);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n<svg width=\"", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, w *svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" height=\"", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, h * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" style=\"stroke-linecap:round; stroke-linejoin:round\"\n", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"   xmlns:svg=\"http://www.w3.org/2000/svg\"\n"
		"xmlns=\"http://www.w3.org/2000/svg\">\n", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g style=\"font-family:sans-serif\">\n", 0);			//check with svg_scale!!
	group_level++;
	return true;
}

bool
ExportSVG::EndPage()
{
	while (group_level >1) {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>", 0);
		group_level--;
		}
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n</svg>\n", 0);
	group_level--;
	return true;
}

bool
ExportSVG::oCircle(int x1, int y1, int x2, int y2, char* nam)
{
	if(x1 > x2) Swap(x1, x2);
	if(y1 > y2) Swap(y1, y2);
	double fx1 = x1, fy1 = y1, fx2 = x2, fy2 = y2;

	return foCircle(fx1, fy1, fx2, fy2, nam);
}

bool 
ExportSVG::foCircle(double x1, double y1, double x2, double y2, char *nam)
{
	if (hgo){
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g>  <!-- ", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (x2 - x1) == (y2 - y1) ? (char*)"circle" : (char*)"ellipse", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)" with pattern -->\n", 0);
		group_level++;
		Indent(true);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, (x2 - x1) == (y2 - y1) ? (char*)"<circle" : (char*)"<ellipse", 0);
	if (nam && nam[0]) {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)" name=\"", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, nam, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\"", 0);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)" cx=\"", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, ((x1 + x2) / 2.0) * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" cy=\"", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, ((y1 + y2) / 2.0) * svg_scale, false);
	if ((x2 - x1) == (y2 - y1)) {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" r=\"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, ((x2 - x1) / 2.0) * svg_scale, false);
		}
	else {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" rx=\"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, ((x2 - x1) / 2.0) * svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" ry=\"", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, ((y2 - y1) / 2.0) * svg_scale, false);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" style=\"fill:", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, ColName(dFillCol), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke:", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, ColName(dLineCol), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke-width:", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, dLineWidth * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)";", 1);
	add_to_buff(&out_buff, &out_pos, &out_size, Transparency(dFillCol, 2), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, Transparency(dLineCol, 1), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\"/>\n", 0);
	if (hgo) {
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g ", 3);
		group_level++;
		add_to_buff(&out_buff, &out_pos, &out_size, tHatchStyle, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)">  <!-- hatch -->\n", 0);
		Indent(true);		bUseGroupLine = true;
		hgo->foCircle(x1, y1, x2, y2);
		Indent(false);		bUseGroupLine = false;
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 0);
		group_level--;
		Indent(false);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 0);
		group_level--;
		}
	return true;
}

bool
ExportSVG::oPolyline(POINT *pts, int cp, char *)
{
	int i, cb;
	char tmptxt[120];

	if(cp < 2) return false;
	if (dPattern){
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g style=\"stroke:", 0);
		group_level++;
		add_to_buff(&out_buff, &out_pos, &out_size, ColName(dLineCol), 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke-width:", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, dLineWidth * svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke-linecap:round;", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, Transparency(dLineCol, 1), 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\"><!-- pattern line -->\n", 0);
		Indent(true);			bUseGroupLine = true;
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		Indent(false);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 0);
		group_level--;
		bUseGroupLine = false;
		}
	else {
		if(cp == 2) return oSolidLine(pts);
		bOutputPending = false;
		cb_out = rlp_strcpy(output, 20, (char*)"<polyline points=\""); 
		for(i = 0; i < cp; i++) {
#ifdef USE_WIN_SECURE
			cb = sprintf_s(tmptxt, 120, "%.2f %.2f ", (float)(pts[i].x * svg_scale), (float)(pts[i].y * svg_scale));
#else
			cb = sprintf(tmptxt, "%.2f %.2f ", (float)(pts[i].x * svg_scale), (float)(pts[i].y * svg_scale));
#endif
			AddToOutput(tmptxt, cb);
			}
		if(cb_out) output[cb_out-1] = '"';
		if(!bUseGroupLine) {
			cb = rlp_strcpy(tmptxt, 120, (char*)" style = \"fill:none; ");
			AddToOutput(tmptxt, cb);
			cb = rlp_strcpy(tmptxt, 120, (char*)"stroke:");	//bug fixed by vefremov
			cb += rlp_strcpy(tmptxt+cb, 120-cb, ColName(dLineCol));
			cb += rlp_strcpy(tmptxt+cb, 120-cb, (char*)"; ");
			AddToOutput(tmptxt, cb);
#ifdef USE_WIN_SECURE
			cb = sprintf_s(tmptxt, 120, "stroke-width:%.2g;", dLineWidth * svg_scale);
#else
			cb = sprintf(tmptxt, "stroke-width:%.2g;", dLineWidth *svg_scale);
#endif
			AddToOutput(tmptxt, cb);
			if((cb = rlp_strcpy(tmptxt, 40, Transparency(dLineCol, 1)))) AddToOutput(tmptxt, cb);
			AddToOutput((char*)"\"/>", 3); 
			}
		else AddToOutput((char*)"/>", 2);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
		if(bOutputPending)Indent(false);
		}
	return true;
}
bool
ExportSVG::oBezier(POINT *pts, int cp, POINT **bpts, long *bcp, bool dodraw)
{
	char txt[80];
	int cb;
	long i;

	if(bpts){
		anyOutput::oBezier(pts, cp, bpts, bcp, false);
		}
	if(!dodraw || cp < 4) return true;
#ifdef USE_WIN_SECURE
	cb_out = sprintf_s(output, 79, "<path d=\"M %.1lf %.1lf C", (pts[0].x * svg_scale), (pts[0].y *svg_scale));
#else
	cb_out = sprintf(output, "<path d=\"M %.1lf %.1lf C", (pts[0].x * svg_scale), (pts[0].y *svg_scale));
#endif
	for (i = 1; i < cp; i++) {
#ifdef USE_WIN_SECURE
		cb = sprintf_s(txt, 79, " %.1lf %.1lf", (pts[i].x * svg_scale), (pts[i].y * svg_scale));
#else
		cb = sprintf(txt, " %.1lf %.1lf", (pts[i].x * svg_scale), (pts[i].y * svg_scale));
#endif
		AddToOutput(txt, cb);
		}
#ifdef USE_WIN_SECURE
	cb = sprintf_s(txt, 79, "\" style=\"fill:none;stroke:%s; stroke-width:%.2lf\"/>", ColName(dLineCol), dLineWidth * svg_scale);
#else
	cb = sprintf(txt, "\" style=\"fill:none;stroke:%s; stroke-width:%.2lf\"/>", ColName(dLineCol), dLineWidth * svg_scale);
#endif
	AddToOutput(txt, cb);
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
	if(bOutputPending)Indent(false);
	return true;
}

bool
ExportSVG::oRectangle(int x1, int y1, int x2, int y2, char *nam)
{
	if(x1 > x2) Swap(x1, x2);
	if(y1 > y2) Swap(y1, y2);
	double fx1 = x1, fy1 = y1, fx2 = x2, fy2 = y2;
	return foRectangle(fx1, fy1, fx2, fy2, nam);
}

bool
ExportSVG::foRectangle(double x1, double y1, double x2, double y2, char *nam)
{
	if (hgo){
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g>  <!-- rectangle with pattern -->\n", 0);
		group_level++;
		Indent(true);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<rect", 5);
	if (nam && nam[0]) {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)" name=\"", 7);
		add_to_buff(&out_buff, &out_pos, &out_size, nam, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\"", 1);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)" x=\"", 4);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, x1 * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" y=\"", 5);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, y1 *svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" width=\"", 9);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, (x2 - x1 - 1)*svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" height=\"", 10);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, (y2 - y1 - 1)*svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" style=\"fill:", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, ColName(dFillCol), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke:", 0);
	add_to_buff(&out_buff, &out_pos, &out_size, ColName(dLineCol), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke-width:", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, dLineWidth * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)";", 1);
	add_to_buff(&out_buff, &out_pos, &out_size, Transparency(dFillCol, 2), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, Transparency(dLineCol, 1), 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\"/>\n", 0);
	if (hgo) {
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g ", 3);
		group_level++;
		add_to_buff(&out_buff, &out_pos, &out_size, tHatchStyle, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)">  <!-- hatch -->\n", 0);
		Indent(true);				bUseGroupLine = true;
		hgo->foRectangle(x1, y1, x2, y2, 0L);
		Indent(false);				bUseGroupLine = false;
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 5);
		group_level--;
		Indent(false);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 5);
		group_level--;
		}
	return true;
}

bool
ExportSVG::oSolidLine(POINT *p)
{
	lfPOINT fp[2];

	if (!p) return false;
	fp[0].fx = p[0].x;		fp[0].fy = p[0].y;
	fp[1].fx = p[1].x;		fp[1].fy = p[1].y;
	return foSolidLine(fp);
}

bool
ExportSVG::foSolidLine(lfPOINT *p)
{
	if (!p) return false;
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<line x1=\"", 0);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, p[0].fx * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" y1=\"", 6);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, p[0].fy *svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" x2=\"", 6);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, p[1].fx * svg_scale, false);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" y2=\"", 6);
	add_dbl_to_buff(&out_buff, &out_pos, &out_size, p[1].fy * svg_scale, false);
	if (!bUseGroupLine) {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\" style=\"stroke:", 0);
		add_to_buff(&out_buff, &out_pos, &out_size, ColName(dLineCol), 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"; stroke-width:", 0);
		add_dbl_to_buff(&out_buff, &out_pos, &out_size, dLineWidth * svg_scale, false);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)";", 1);
		add_to_buff(&out_buff, &out_pos, &out_size, Transparency(dLineCol, 1), 0);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\"/>\n", 4);
	return true;
}

bool
ExportSVG::com_TextOut(int x, int y, unsigned char *txt, int)
{
	int c, h, ix, iy, dy;
	char tmptxt[120];

	if(!txt || !txt[0]) return false;
	else h = TxtSet.iSize;
	if(TxtSet.Align & TXA_VCENTER) iy = y + h/3;
	else if(TxtSet.Align & TXA_VBOTTOM) iy = y;
	else iy = y + iround(h * 0.8);
	ix = x;		dy = 0;
	if(TxtSet.Style & TXS_SUB) {
		if((TxtSet.Align & TXA_VCENTER) == TXA_VCENTER) dy = un2iy(TxtSet.fSize*0.4);
		else if((TxtSet.Align & TXA_VBOTTOM) == TXA_VBOTTOM) dy = un2iy(TxtSet.fSize*0.2);
		else if((TxtSet.Align & TXA_VTOP) == TXA_VTOP) dy = un2iy(TxtSet.fSize*.6);
		}
	else if(TxtSet.Style & TXS_SUPER) {
		if((TxtSet.Align & TXA_VCENTER) == TXA_VCENTER) dy = -un2iy(TxtSet.fSize*0.4);
		else if((TxtSet.Align & TXA_VBOTTOM) == TXA_VBOTTOM) dy = -un2iy(TxtSet.fSize*0.6);
		else if((TxtSet.Align & TXA_VTOP) == TXA_VTOP) dy = -un2iy(TxtSet.fSize*0.2);
		}
#ifdef USE_WIN_SECURE
	cb_out = sprintf_s(output, 120, "<text x=\"%.2f\" y=\"%.2f\" dy=\"%.2f\" ", (float)(ix * svg_scale), (float)(iy * svg_scale), (float)(dy *svg_scale));
#else
	cb_out = sprintf(output, "<text x=\"%.2f\" y=\"%.2f\" dy=\"%.2f\" ", (float)(ix * svg_scale), (float)(iy * svg_scale), (float)(dy *svg_scale));
#endif
	if(fabs(TxtSet.RotBL) >.01 || fabs(TxtSet.RotCHAR) >.01) {
#ifdef USE_WIN_SECURE
		cb_out += sprintf_s(output+cb_out, 120-cb_out, "transform=\"rotate(%.2f,%.2f,%.2f)\" ", (float)(-TxtSet.RotBL), (float)(ix*svg_scale), (float)(iy*svg_scale));
#else
		cb_out += sprintf(output+cb_out,"transform=\"rotate(%.2f,%.2f,%.2f)\" ", (float)(-TxtSet.RotBL), (float)(ix*svg_scale), (float)(iy*svg_scale));
#endif
		}
	c = rlp_strcpy(tmptxt, 140, (char*)"style=\"font-family:");
	switch(TxtSet.Font) {
	case FONT_TIMES:	c += rlp_strcpy(tmptxt+c, 120-c, (char*)"Times;");		break;
	case FONT_COURIER:	c += rlp_strcpy(tmptxt+c, 120-c, (char*)"Courier;");	break;
	default:			c += rlp_strcpy(tmptxt+c, 120-c, (char*)"sans-serif;");	break;
		}
	if(TxtSet.Style & TXS_ITALIC) c += rlp_strcpy(tmptxt+c, 120-c, (char*)" font-style:italic;");
	if(TxtSet.Style & TXS_BOLD) c += rlp_strcpy(tmptxt+c, 120-c, (char*)" font-weight:bold;");
	if(TxtSet.Style & TXS_UNDERLINE) c += rlp_strcpy(tmptxt+c, 120-c, (char*)" text-decoration:underline;");
	AddToOutput(tmptxt, c);
#ifdef USE_WIN_SECURE
	c = sprintf_s(tmptxt, 120, " fill:%s; stroke-width:0; stroke:%s;%s ", ColName(TxtSet.ColTxt), ColName(TxtSet.ColTxt),Transparency(TxtSet.ColTxt, 0));
#else
	c = sprintf(tmptxt, " fill:%s; stroke-width:0; stroke:%s;%s ", ColName(TxtSet.ColTxt), ColName(TxtSet.ColTxt),Transparency(TxtSet.ColTxt, 0));
#endif
	AddToOutput(tmptxt, c);
#ifdef USE_WIN_SECURE
	c = sprintf_s(tmptxt, 120, "font-size:%.2f; text-anchor:%s \">", (float)(h * svg_scale), 
		(TxtSet.Align & TXA_HRIGHT) ? "end" : (TxtSet.Align & TXA_HCENTER) ? "middle":"start");
#else
	c = sprintf(tmptxt, "font-size:%.2f; text-anchor:%s \">", (float)(h *svg_scale), 
		(TxtSet.Align & TXA_HRIGHT) ? "end" : (TxtSet.Align & TXA_HCENTER) ? "middle":"start");
#endif
	AddToOutput(tmptxt, c);
	if((cb_ind+strlen((char*)txt)+cb_out) <110) cb_out += rlp_strcpy(output+cb_out, 120-cb_out, txt);
	else {
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
		cb_out=rlp_strcpy(output, 120, txt);
		}
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
	if((cb_ind + cb_out) <104) {
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</text>\n", 0);
		}
	else add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n</text>\n", 0);
	return true;
}

bool
ExportSVG::oTextOut(int x, int y, unsigned char *txt, int cb)
{
	unsigned char *nt;

	if(!txt || !txt[0]) return false;
	nt = str2xml(txt, TxtSet.Font == FONT_GREEK);
	return com_TextOut(x, y, nt, cb);
}

bool
ExportSVG::oTextOutW(int x, int y, w_char *txt, int)
{
	int i, j;

	for(i = j = 0; txt[i]; i++) {
		switch(txt[i]) {
			case '"':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, (char*)"&quot;");
				break;
			case '&':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, (char*)"&amp;");
				break;
			case '<':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, (char*)"&lt;");
				break;
			case '>':
				j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, (char*)"&gt;");
				break;
			case 176:				//degree character
				j += rlp_strcpy(TmpTxt + j, TMP_TXT_SIZE - j, (char*)"&#176;");
				break;
			default:
				if(txt[i] > 255) {
#ifdef USE_WIN_SECURE
					j += sprintf_s(TmpTxt+j, TMP_TXT_SIZE-j, "&#%d;", txt[i]);
#else
					j += sprintf(TmpTxt+j, "&#%d;", txt[i]);
#endif
					}
				else if(txt[i] > 127) {
#ifdef USE_WIN_SECURE
					j += sprintf_s(TmpTxt + j, TMP_TXT_SIZE - j, "&#%d;", txt[i]);
#else
					j += sprintf(TmpTxt + j, "&#%d;", txt[i]);
#endif
					}
				else TmpTxt[j++] = (char)txt[i];
				break;
			}
		}
	TmpTxt[j++] = 0;
	return com_TextOut(x, y, (unsigned char*)TmpTxt, j-1);
}

bool
ExportSVG::oPolygon(POINT *pts, int cp, char *nam)
{
	int i;
	lfPOINT *fpts;

	if (!pts || cp < 3) return false;
	fpts = (lfPOINT*)malloc((cp + 2) * sizeof(lfPOINT));
	for (i = 0; i < cp; i++) {
		fpts[i].fx = pts[i].x;		fpts[i].fy = pts[i].y;
		}
	foPolygon(fpts, cp, nam);
	free(fpts);
	return true;
}

bool
ExportSVG::foPolygon(lfPOINT *pts, int cp, char *)
{
	int i, cb;
	char tmptxt[40];

	if (cp <3) return false;
	if (hgo && hgo->OC_type != OC_HATCH){
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g>  <!-- polygon with pattern -->\n", 0);
		group_level++;
		Indent(true);
		}
	bOutputPending = false;
	cb_out = rlp_strcpy(output, 120, (char*)"<polygon points=\"");
	for (i = 0; i < cp; i++) {
#ifdef USE_WIN_SECURE
		cb = sprintf_s(tmptxt, 40, "%.2lf %.2lf ", pts[i].fx *svg_scale, pts[i].fy * svg_scale);
#else
		cb = sprintf(tmptxt, "%.2lf %.2lf ", pts[i].fx *svg_scale, pts[i].fy * svg_scale);
#endif
		AddToOutput(tmptxt, cb);
		}
	if (cb_out) output[cb_out - 1] = '"';
#ifdef USE_WIN_SECURE
	cb = sprintf_s(tmptxt, 40, " style=\"fill:%s; ", ColName(dFillCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf_s(tmptxt, 40, "stroke:%s; ", ColName(dLineCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf_s(tmptxt, 40, "stroke-width:%.2lf;", dLineWidth * svg_scale);
	AddToOutput(tmptxt, cb);
#else
	cb = sprintf(tmptxt, " style=\"fill:%s; ", ColName(dFillCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf(tmptxt, "stroke:%s; ", ColName(dLineCol));
	AddToOutput(tmptxt, cb);
	cb = sprintf(tmptxt, "stroke-width:%.2lf;", (float)(dLineWidth*svg_scale));
	AddToOutput(tmptxt, cb);
#endif
	if ((cb = rlp_strcpy(tmptxt, 40, Transparency(dFillCol, 2)))) AddToOutput(tmptxt, cb);
	if ((cb = rlp_strcpy(tmptxt, 40, Transparency(dLineCol, 1)))) AddToOutput(tmptxt, cb);
	AddToOutput((char*)"\"/>", 3);
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
	add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
	if (bOutputPending)Indent(false);
	if (hgo && hgo->OC_type != OC_HATCH){
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"<g ", 3);
		group_level++;
		add_to_buff(&out_buff, &out_pos, &out_size, tHatchStyle, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)">  <!-- hatch -->\n", 0);
		Indent(true);					bUseGroupLine = true;
		hgo->foPolygon(pts, cp);		Indent(false);
		bUseGroupLine = false;
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 5);
		group_level--;				Indent(false);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</g>\n", 5);
		group_level--;
		}
	return true;
}

bool 
ExportSVG::oSolidPolygon(POINT *pts, int cp)
{
	int i, cb;
	char tmptxt[80];

#ifdef USE_WIN_SECURE
	cb_out = sprintf_s(output, 120, "<polygon points=\"");
#else
	cb_out = sprintf(output, "<polygon points=\"");
#endif
	bOutputPending = false;
	for (i = 0; i < cp; i++) {
#ifdef USE_WIN_SECURE
		cb = sprintf_s(tmptxt, 80, "%.2f %.2f ", (float)(pts[i].x *svg_scale), (float)(pts[i].y * svg_scale));
#else
		cb = sprintf(tmptxt, "%.2f %.2f ", (float)(pts[i].x *svg_scale), (float)(pts[i].y * svg_scale));
#endif
		AddToOutput(tmptxt, cb);
		}
	if (cb_out) output[cb_out - 1] = '"';
#ifdef USE_WIN_SECURE
	cb = sprintf_s(tmptxt, 80, " style=\"fill:%s\" />\n", ColName(dFillCol));
#else
	cb = sprintf(tmptxt, " style=\"fill:%s\" />\n", ColName(dFillCol));
#endif
	AddToOutput(tmptxt, cb);
	add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
	add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
	return true;
}
bool 
ExportSVG::GradPG(fPOINT3D *pts, long npt, double, double, double pLo, double pHi, lfPOINT *grad)
{
	long i, lpts, cb;
	lfPOINT *points;
	double o_min, o_max, lc, hc;
	FillDEF fill;
	char tmptxt[120], grad_name[20];


	if (npt < 3) return false;
	points = (lfPOINT*)malloc((npt + 2) *sizeof(lfPOINT));
	GetFill(&fill);			bOutputPending = false;
	o_min = LoVal;					o_max = HiVal;
	lc = fmax((pLo - o_min) / (o_max - o_min), 0.0);
	hc = fmin((pHi - o_min) / (o_max - o_min), 1.0);
#ifdef USE_WIN_SECURE
	sprintf_s(grad_name, 20, "LinGrad%d", ++grad_id);
#else
	sprintf(grad_name, "LinGrad%d", ++grad_id);
#endif
	for (i = 0, lpts = npt; i < npt; i++){
		points[i].fx = pts[i].fx;					points[i].fy = pts[i].fy;
		}
	if (points[i - 1].fx != points[0].fx || points[i - 1].fy != points[0].fy) {		//close PG?
		points[i].fx = points[0].fx;					points[i].fy = points[0].fy;
		lpts++;
		}
	if (grad[0].fx != grad[1].fx || grad[0].fy != grad[1].fy){			//this polypon is not a line
		//define color gradient
		cb = rlp_strcpy(tmptxt, 120, (char*)"<linearGradient id=\"");
		cb += rlp_strcpy(tmptxt + cb, 120-cb, grad_name);
		cb += rlp_strcpy(tmptxt + cb, 120-cb, (char*)"\" gradientUnits=\"userSpaceOnUse\"");
#ifdef USE_WIN_SECURE
		cb += sprintf_s(tmptxt +cb, 120-cb, " x1=\"%.2f\" y1=\"%.2f\" x2=\"%.2f\" y2=\"%.2f\"",
			grad[0].fx * svg_scale, grad[0].fy * svg_scale, grad[1].fx * svg_scale, grad[1].fy * svg_scale);
#else
		cb += sprintf(tmptxt +cb, " x1=\"%.2f\" y1=\"%.2f\" x2=\"%.2f\" y2=\"%.2f\"",
			grad[0].fx * svg_scale, grad[0].fy * svg_scale, grad[1].fx * svg_scale, grad[1].fy * svg_scale);
#endif
		cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)">\n");
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
		if ((fill.type & 0xf00) == FILL_GRADIENT){			//simple color to color gradien
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"0%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName(IpolCol(dFillCol2, dFillCol, lc)));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind + 1);
			add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
			add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"100%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName(IpolCol(dFillCol2, dFillCol, hc)));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			}
		else {												//rainbow colors
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"0%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName((GradColor(2, o_min, o_max, pLo))));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind + 1);
			add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
			add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"25%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName((GradColor(2, o_min, o_max, pLo*0.75 +pHi*0.25))));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind + 1);
			add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
			add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"50%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName((GradColor(2, o_min, o_max, (pLo + pHi)*0.5))));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind + 1);
			add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
			add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"75%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName((GradColor(2, o_min, o_max, pLo*0.25 + pHi*0.75))));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind + 1);
			add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
			add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
			cb = rlp_strcpy(tmptxt, 120, (char*)"<stop offset=\"100%\" style=\"stop-color:");
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName((GradColor(2, o_min, o_max, pHi))));
			cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\"/>");
			}
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind+1);
		add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
		cb = rlp_strcpy(tmptxt, 20, (char*)"</linearGradient>\n");
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
		//draw polygon
		cb_out = rlp_strcpy(output, 120, (char*)"<polygon style=\"fill:url(#");
		cb_out += rlp_strcpy(output + cb_out, 120 - cb_out, grad_name);
		cb_out += rlp_strcpy(output + cb_out, 120 - cb_out, (char*)")\" points=\"");
		for (i = cb = 0; i < npt; i++) {
#ifdef USE_WIN_SECURE
			cb += sprintf_s(tmptxt+cb, 120-cb, "%.2f %.2f ", pts[i].fx * svg_scale, pts[i].fy * svg_scale);
#else
			cb = sprintf(tmptxt+cb, "%.2f %.2f ", pts[i].fx * svg_scale, pts[i].fy * svg_scale);
#endif
			}
		rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"\">\n");
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, tmptxt, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"</polygon>\n", 0);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {	//is outline transparent?
		bOutputPending = false;
		cb_out = rlp_strcpy(output, 20, (char*)"<polyline points=\"");
		for (i = 0; i < lpts; i++) {
#ifdef USE_WIN_SECURE
			cb = sprintf_s(tmptxt, 120, "%.2f %.2f ", pts[i].fx * svg_scale, pts[i].fy * svg_scale);
#else
			cb = sprintf(tmptxt, "%.2f %.2f ", pts[i].fx * svg_scale, pts[i].fy * svg_scale);
#endif
			AddToOutput(tmptxt, cb);
			}
		if (cb_out) output[cb_out - 1] = '"';
		cb = rlp_strcpy(tmptxt, 120, (char*)" style = \"fill:none; ");
		AddToOutput(tmptxt, cb);
		cb = rlp_strcpy(tmptxt, 120, (char*)"stroke:");	//bug fixed by vefremov
		cb += rlp_strcpy(tmptxt + cb, 120 - cb, ColName(dLineCol));
		cb += rlp_strcpy(tmptxt + cb, 120 - cb, (char*)"; ");
		AddToOutput(tmptxt, cb);
#ifdef USE_WIN_SECURE
		cb = sprintf_s(tmptxt, 120, "stroke-width:%.2g;", dLineWidth * svg_scale);
#else
		cb = sprintf(tmptxt, "stroke-width:%.2g;", dLineWidth *svg_scale);
#endif
		AddToOutput(tmptxt, cb);
		if ((cb = rlp_strcpy(tmptxt, 40, Transparency(dLineCol, 1)))) AddToOutput(tmptxt, cb);
		AddToOutput((char*)"\"/>", 3);
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
		if (bOutputPending)Indent(false);
		}
	free(points);
	return true;
}

void
ExportSVG::Indent(bool ind)
{
	if(ind) {
		if(cb_ind < 20) cb_ind += 3;
		}
	else {
		if(cb_ind > 5) cb_ind -= 3;
		}
}

void
ExportSVG::AddToOutput(char *txt, int len)
{
	if(!txt || !txt[0]) return;
	if(!len) len = (int)strlen(txt);
	if((len + cb_out + cb_ind) < 110){
		cb_out += rlp_strcpy(output+cb_out, 120, txt);
		}
	else {
		add_to_buff(&out_buff, &out_pos, &out_size, str_ind, cb_ind);
		add_to_buff(&out_buff, &out_pos, &out_size, output, 0);
		add_to_buff(&out_buff, &out_pos, &out_size, (char*)"\n", 1);
		if(!bOutputPending) Indent(true);
		bOutputPending = true;
		cb_out = rlp_strcpy(output, 120, txt);
		}
}

char *
ExportSVG::ColName(DWORD col)
{
	static char txt1[20], txt2[20];
	static int sw;
	char *ret;

	switch(col) {
	case 0x00000000:			return (char*)"black";
	case 0x000000ff:			return (char*)"red";
	case 0x0000ff00:			return (char*)"lime";
	case 0x0000ffff:			return (char*)"yellow";
	case 0x00ff0000:			return (char*)"blue";
	case 0x00ff00ff:			return (char*)"magenta";
	case 0x00ffff00:			return (char*)"cyan";
	case 0x00ffffff:			return (char*)"white";
		}
	sw = (sw+1) & 0x0f;
#ifdef USE_WIN_SECURE
	sprintf_s(ret = (sw & 0x01 ? txt1 : txt2), 20, "rgb(%d,%d,%d)", col & 0xff, (col>>8) & 0xff, (col>>16) &0xff);
#else
	sprintf(ret = (sw & 0x01 ? txt1 : txt2), "rgb(%d,%d,%d)", col & 0xff, (col>>8) & 0xff, (col>>16) &0xff);
#endif
	return ret;
}

char *
ExportSVG::Transparency(DWORD col, int type)
{
	static char txt1[30], txt2[30];
	static int sw;
	int cb;
	char *ret;
	double a;

	if((col & 0xfe000000) == 0) return (char*)"";
	a = ((double)(255-(col>>24)))/256.0;
	if(a > 0.99) return (char*)"";
	sw = (sw+1) & 0x0f;
	ret = sw & 0x01 ? txt1 : txt2;
	switch(type) {
	case 1:
		cb = rlp_strcpy(ret, 30, (char*)" stroke-opacity:");	break;
	case 2:	
		cb = rlp_strcpy(ret, 30, (char*)" fill-opacity:");	break;
	default:
		cb = rlp_strcpy(ret, 30, (char*)" opacity:");		break;
		}
#ifdef USE_WIN_SECURE
	sprintf_s(ret+cb, 30-cb, "%0.3lf;", a);
#else
	sprintf(ret+cb, "%0.3lf;", a);
#endif
	return ret;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Entry point to export graph to *.svg
void DoExportSvg(GraphObj *g, char *FileName, DWORD flags)
{
	ExportSVG *ex;
	anyOutput *o;
	FILE *oFile;
	
	if(!g) return;
	if(g->Id == GO_GRAPH || g->Id == GO_PAGE) {
		if((o = ((Graph*)g)->getDisp())) o->HideMark();
		}
	HideTextCursor();
	CurrGO = NULL;
	ex = new ExportSVG(g, flags);
	if(ex->StartPage()) {
		g->DoPlot(ex);
		ex->EndPage();
		if(FileName && ex->out_buff && ex->out_pos > 20) {
#ifdef USE_WIN_SECURE
				fopen_s(&oFile, FileName, "w");
#else
				oFile = fopen(FileName, "w");
#endif
				if(!oFile) {
					ErrorBox((char*)SCMS_NO_OUTPUTFILE);
					return;
					}
				fprintf(oFile, "%s", ex->out_buff);
				free(ex->out_buff);		fclose (oFile);
				}
		g->Command(CMD_REDRAW, 0L, 0L);
		}
	delete(ex);
}

char* DoCopySvg(GraphObj *g, long *buf_size)
{
	ExportSVG *ex;
	anyOutput *o;

	if (!g) return NULL;
	if (g->Id == GO_GRAPH || g->Id == GO_PAGE){
		if ((o = ((Graph*)g)->getDisp())) o->HideMark();
		}
	HideTextCursor();
	CurrGO = NULL;
	ex = new ExportSVG(g);
	if (ex->StartPage()) {
		g->DoPlot(ex);
		ex->EndPage();
		*buf_size = ex->out_pos;
		//the caller needs to free ex->out_buff !
		return ex->out_buff;
		}
	return NULL;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// export to *.eps file (encapsulated post script).
// This code is based on information from the following book
// G. Born, 'Referenzhandbuch Dateiformate', 
//     Addison-Wesley ISBN 3-89319-815-6
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class ExportEPS:public anyOutput {
public:
	HatchOut *hgo;

	ExportEPS(GraphObj *g, char *FileName, DWORD flags);
	~ExportEPS();
	bool ClipRect(RECT *rec);
	bool SetLine(LineDEF *lDef);
	bool SetFill(FillDEF *fill);
	bool SetTextSpec(TextDEF *set);
	bool StartPage();
	bool EndPage();
	bool oCircle(int x1, int y1, int x2, int y2, char* nam = 0L);
	bool oPolyline(POINT * pts, int cp, char *nam = 0L);
	bool oBezier(POINT *pts, int cp, POINT **bpts = 0L, long *bcp = 0L, bool dodraw = true);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oSolidLine(POINT *p);
	bool oTextOut(int x, int y, unsigned char *txt, int cb);
	bool oTextOutW(int x, int y, w_char *txt, int cb);
	bool oPolygon(POINT *pts, int cp, char *nam = 0L);
	bool oSolidPolygon(POINT *pts, int cp);

private:
	RECT BoundBox;
	fRECT SolLines[2000];
	int nSolLines;
	GraphObj *go;
	char *name, FontName[30];
	FILE *oFile;
	DWORD CurrCol;
	bool bFontChange;

	float ix2eps(int x);
	float iy2eps(int y);
	char *col2eps(DWORD color);
	void FlushSolLines();
	void AddSolLine(float x1, float y1, float x2, float y2);
};

ExportEPS::ExportEPS(GraphObj *g, char *FileName, DWORD)
{
	hgo =0L;		nSolLines = 0;			DeskRect.left = DeskRect.top = 0;
	DeskRect.right = DeskRect.bottom = 0x4fffffff;
	dFillCol = 0xffffffffL;					hres = vres = 720.0f;
	go = g;			bFontChange = false;	oFile = 0L;
	OC_type = OC_EXPORT;
	if(FileName)name = rlp_strdup(FileName);
	else name = 0L;
	Notary->ValPtr(this, true);
}

ExportEPS::~ExportEPS()
{
	Notary->ValPtr(this, false);
	if (hgo) delete hgo;
	if(name) free(name);
}

bool
ExportEPS::ClipRect(RECT *rec)
{
	if (!rec) {
		fprintf(oFile, "\ngrestore");
		hasClip = false;
		}
	else {
		memcpy(&ClipRC, rec, sizeof(RECT));
		fprintf(oFile, "\ngsave");
		fprintf(oFile, "\nnewpath %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f moveto 3 {lineto}"
			" repeat closepath clip", ix2eps(rec->left), iy2eps(rec->top), ix2eps(rec->right), iy2eps(rec->top),
			ix2eps(rec->right), iy2eps(rec->bottom), ix2eps(rec->left), iy2eps(rec->bottom));
		hasClip = true;
		}
	return false;
}

bool
ExportEPS::SetLine(LineDEF *lDef)
{
	if(LineWidth != lDef->width || dLineCol != lDef->color) {
		FlushSolLines();
		LineWidth = lDef->width;
		CurrCol = dLineCol = lDef->color;
		fprintf(oFile, "\nnewpath %.2f setlinewidth %s ",
			un2fix(LineWidth)/10.0f, col2eps(dLineCol));
		}
	dPattern = lDef->pattern;
	RLP.finc = (float)(256.0/un2fix(lDef->patlength*8.0));
	RLP.fp = 0.0;
	return true;
}

bool
ExportEPS::SetFill(FillDEF *fill)
{
	if(!fill) return false;
	anyOutput::SetFill(fill);
	if ((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		}
	else if (!(fill->type & FILL_KEEP_ALIVE)) {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	dFillCol = fill->color;						dFillCol2 = fill->color2;
	return true;
}

bool
ExportEPS::SetTextSpec(TextDEF *set)
{
	int cb;

	if(set->fSize > 0.0) {
		if((set->Style & TXS_SUPER) || (set->Style & TXS_SUB)) set->iSize = un2iy(set->fSize * 0.71);
		else set->iSize = un2iy(set->fSize);
		}
	if(!set->iSize) return false;
	anyOutput::SetTextSpec(set);
	switch(TxtSet.Font) {
	case FONT_TIMES:	cb = rlp_strcpy(FontName, 30, (char*)"(Times");		break;		//Serif
	case FONT_COURIER:	cb = rlp_strcpy(FontName, 30, (char*)"(Courier");		break;		//fixed spaced
	default:			cb = rlp_strcpy(FontName, 30, (char*)"(sans-serif");	break;		//Sans Serif	
		}
	if(TxtSet.Style & TXS_BOLD) cb += rlp_strcpy(FontName+cb, 30-cb, (char*)"-Bold");
	if(TxtSet.Style & TXS_ITALIC) cb += rlp_strcpy(FontName+cb, 30-cb, (char*)"-Italic");
	cb += rlp_strcpy(FontName+cb, 30-cb, (char*)")");		bFontChange = true;
	return true;
}

bool 
ExportEPS::StartPage()
{
	time_t ti;
	
	if(!go) return false;
	ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
	ti = time(0L);
	BoundBox.top = BoundBox.left = 0;
	BoundBox.right = un2ix(go->GetSize(SIZE_GRECT_RIGHT) - go->GetSize(SIZE_GRECT_LEFT))/10;
	BoundBox.bottom = un2iy(go->GetSize(SIZE_GRECT_BOTTOM) - go->GetSize(SIZE_GRECT_TOP))/10;
	BoundBox.right++;	BoundBox.bottom++;
	if(name) {
#ifdef USE_WIN_SECURE
		fopen_s(&oFile, name, "w");
#else
		oFile = fopen(name, "w");
#endif
		if(!oFile) {
			ErrorBox((char*)"Could not open\noutput file!");
			return false;
			}
		}
	else oFile = stdout;
	VPorg.fy = -un2fiy(go->GetSize(SIZE_GRECT_TOP));
	VPorg.fx = -un2fix(go->GetSize(SIZE_GRECT_LEFT));
	fprintf(oFile, "%%!PS-Adobe-1.0 EPSF-3.0\n"
		"%%%%BoundingBox: %ld %ld %ld %ld\n", BoundBox.left, BoundBox.top,
		BoundBox.right, BoundBox.bottom);
	fprintf(oFile, "%%%%Title: %s\n", name);
	fprintf(oFile,"%%%%Creator: RLPlot version %s\n", SZ_VERSION);
#ifdef USE_WIN_SECURE
	ctime_s(TmpTxt, 50, &ti);	TmpTxt[24] = 0;
	fprintf(oFile,"%%%%CreationDate: %s\n", TmpTxt);
#else
	fprintf(oFile,"%%%%CreationDate: %s\n", ctime(&ti));
#endif
	fprintf(oFile, "%%%%Pages: 1\n%%%%DocumentFonts: (atend)\n");
	fprintf(oFile, "%%%%EndComments\n"
		"%%%%BeginProlog\n"
		"%%%%EndProlog\n"
		"%%%%Page: 1 1");
	return true;
}

bool
ExportEPS::EndPage()
{
	fprintf(oFile, "\nshowpage\n%%%%Trailer\n");
	fprintf(oFile, "%%%%DocumentFonts: Helvetica\n");
	fprintf(oFile, "%%%%EOF\n");
	fclose (oFile);
	return true;
}

bool
ExportEPS::oCircle(int x1, int y1, int x2, int y2, char*)
{
	FlushSolLines();
	if((x2 - x1) == (y2 -y1)) {
		fprintf(oFile, "\nnewpath %.2f %.2f %.2f 0 360 arc ", ix2eps((x1+x2)/2), 
			iy2eps((y1+y2)/2), (float)(x2-x1)/20.0f);
		fprintf(oFile, "%s fill", col2eps(CurrCol = dFillCol));
		if(hgo){
			hgo->oCircle(x1, y1, x2, y2);
			FlushSolLines();
			}
		fprintf(oFile, "\nnewpath %.2f %.2f %.2f 0 360 arc ", ix2eps((x1+x2)/2), 
			iy2eps((y1+y2)/2), (float)(x2-x1)/20.0f);
		fprintf(oFile, "%s stroke", col2eps(CurrCol = dLineCol));
		}
	else if(x1 != x2 && y1 != y2){
		fprintf(oFile, "\ngsave %.2f %.2f translate", (ix2eps((x1+x2)>>1)),
			(iy2eps((y1+y2)>>1)));
		if(abs(x2-x1) > abs(y2-y1)) {
			fprintf(oFile, " 1 %lf scale", fabs(((double)(y2-y1))/((double)(x2-x1))));
			fprintf(oFile, " 0 0 %.1lf 0 360 arc ", fabs(((double)(x2-x1))/20.0));
			fprintf(oFile, "%s fill", col2eps(CurrCol = dFillCol));
			fprintf(oFile, "\n 0 0 %.1lf 0 360 arc ", fabs(((double)(x2-x1))/20.0));
			}
		else {
			fprintf(oFile, " %lf 1 scale", fabs(((double)(x2-x1))/((double)(y2-y1))));
			fprintf(oFile, " 0 0 %.1lf 0 360 arc ", fabs(((double)(y2-y1))/20.0));
			fprintf(oFile, "%s fill", col2eps(CurrCol = dFillCol));
			fprintf(oFile, "\n 0 0 %.1lf 0 360 arc ", fabs(((double)(y2-y1))/20.0));
			}
		fprintf(oFile, "%s stroke grestore", col2eps(CurrCol = dLineCol));
		if(hgo){
			hgo->oCircle(x1, y1, x2, y2);
			FlushSolLines();
			}
		}
	return true;
}

bool
ExportEPS::oPolyline(POINT * pts, int cp, char *)
{
	int i, j;

	if(cp <1) return false;
	if (dPattern){
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		return true;
		}
	else if(cp == 2) return oSolidLine(pts);
	FlushSolLines();
	if(CurrCol != dLineCol) {
		fprintf(oFile, "\n1 setlinecap %s ", col2eps(CurrCol = dLineCol));
		}
	else {
		fprintf(oFile, "\n1 setlinecap ");
		}
	for(i = cp-1, j = 0; i >= 0; i--, j++) {
		if(!(j & 0x07)) fprintf(oFile, "\n");
		fprintf(oFile, " %.2f %.2f", ix2eps(pts[i].x), iy2eps(pts[i].y));
		}
	fprintf(oFile, " moveto %d {lineto} repeat stroke ", cp-1);
	return true;
}

bool 
ExportEPS::oBezier(POINT *pts, int cp, POINT **bpts, long *bcp, bool dodraw)
{
	int i, j;

	if (cp <1) return false;
	if (dPattern){
		anyOutput::oBezier(pts, cp, bpts, bcp, dodraw);
		return true;
		}
	if (!dodraw || cp < 4) return true;
	FlushSolLines();
	fprintf(oFile, "\nnewpath 1 setlinecap %s", col2eps(CurrCol = dLineCol));
	fprintf(oFile, " %.2f %.2f moveto", ix2eps(pts[0].x), iy2eps(pts[0].y));
	for(i = 1; i < cp; i+= 3 ){
		for (j = 0; j < 3; j++){
			fprintf(oFile, " %.2f %.2f", ix2eps(pts[i+j].x), iy2eps(pts[i+j].y));
			}
		fprintf(oFile, " curveto\n");
		}
	fprintf(oFile, " stroke");
	return true;
}

bool
ExportEPS::oRectangle(int x1, int y1, int x2, int y2, char *)
{
	FlushSolLines();
	fprintf(oFile, "\nnewpath 1 setlinecap %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f moveto 3 {lineto}"
		" repeat closepath %s fill", ix2eps(x1), iy2eps(y1), ix2eps(x1), iy2eps(y2), ix2eps(x2), iy2eps(y2), 
		ix2eps(x2), iy2eps(y1), col2eps(CurrCol = dFillCol));
	if(hgo) hgo->oRectangle(x1, y1, x2, y2, 0L);
	fprintf(oFile, "\nnewpath 1 setlinecap %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f moveto 3 {lineto}"
		" repeat closepath %s stroke", ix2eps(x1), iy2eps(y1), ix2eps(x1), iy2eps(y2), ix2eps(x2), iy2eps(y2), 
		ix2eps(x2), iy2eps(y1), col2eps(CurrCol = dLineCol));
	return true;
}

bool
ExportEPS::oSolidLine(POINT *p)
{
	if(CurrCol != dLineCol) {
		FlushSolLines();
		fprintf(oFile, "\n%s ", col2eps(CurrCol = dLineCol));
		}
	AddSolLine(ix2eps(p[0].x), iy2eps(p[0].y), ix2eps(p[1].x), iy2eps(p[1].y));
	return true;
}

bool
ExportEPS::oTextOut(int x, int y, unsigned char *txt, int cb)
{
	int i, j, k, ix, iy;
	double w, h;
	float fx, fy, lw;
	unsigned char	*txt1;

	FlushSolLines();	if(!txt || !txt[0]) return true;
	oGetTextExtent(txt, cb, &w, &h);
	if(bFontChange)	{
		fprintf(oFile, "\n%s findfont %d scalefont setfont ", FontName, TxtSet.iSize/10);
		bFontChange = false;
		}
	if(TxtSet.Align & TXA_VCENTER) iy = y + iround(h/3.0);
	else if(TxtSet.Align & TXA_VBOTTOM) iy = y;
	else iy = y + iround(h*.8);
	if(TxtSet.Align & TXA_HRIGHT) ix = x-iround(w);
	else if(TxtSet.Align & TXA_HCENTER) ix = x-iround(w/2.0);
	else ix = x;
	lw = (float)((iy-y)/150.0);
	fprintf(oFile,"\n");
	//check for UNICODE !
	j = rlp_strlen(txt);
	txt1 = (unsigned char*)malloc((j<<1)+2);
	i = k = 0;
	do {
		if (txt[i] < 128) txt1[k++] = txt[i];
		i++;
		} while (i <= j);
	if(fabs(TxtSet.RotBL) >.01 || fabs(TxtSet.RotCHAR) >.01) {
		if(TxtSet.Style & TXS_SUB) iy += un2iy(TxtSet.fSize*0.6);
		else if(TxtSet.Style & TXS_SUPER) iy -= un2iy(TxtSet.fSize*.2);
		fprintf(oFile, "gsave %.1f %.1f translate %f rotate %.1f %.1f moveto\n",
			ix2eps(x), iy2eps(y), TxtSet.RotBL, (float)(ix-x)/10.0f, (float)(iy-y)/-10.0f);
		fprintf(oFile, "%s ", col2eps(CurrCol = TxtSet.ColTxt));
		fprintf(oFile, "(%s) show ", txt1);
		if(TxtSet.Style & TXS_UNDERLINE) {
			fprintf(oFile, "\ncurrentpoint %.1f exch pop moveto", (float)(iy-y)/-10.0f - lw*1.2);
			fprintf(oFile, " 0 %.1f lineto %s %.1f setlinewidth stroke ", (float)(iy-y)/-10.0f -lw*1.2,
				col2eps(TxtSet.ColTxt), lw);
			}
		fprintf(oFile, "grestore\n");
		}
	else {
		if(TxtSet.Style & TXS_SUB) iy += un2iy(TxtSet.fSize*0.6);
		else if(TxtSet.Style & TXS_SUPER) iy -= un2iy(TxtSet.fSize*.2);
		fx = ix2eps(ix);			fy = iy2eps(iy);
		fprintf(oFile, "%s ", col2eps(CurrCol = TxtSet.ColTxt));
		fprintf(oFile,"%.1f %.1f moveto (%s) show ", fx, fy, txt1);
		if(TxtSet.Style & TXS_UNDERLINE) {
			fprintf(oFile, "\ncurrentpoint %.1f exch pop moveto", fy - lw*1.2);
			fprintf(oFile, " %.1f %.1f lineto %s %.1f setlinewidth stroke\n", fx, fy - lw*1.2, 
				col2eps(TxtSet.ColTxt), lw);
			}
		}
	free(txt1);
	return true;
}

bool 
ExportEPS::oTextOutW(int x, int y, w_char *txt, int cb)
{
	int i, n;
	unsigned char *tmp_txt;

	if (!(n = rlp_strlenW(txt))) return false;
	tmp_txt = (unsigned char*)malloc((n + 2)*sizeof(unsigned char));
	for (i = 0; i < n; i++) {
		tmp_txt[i] = (unsigned char)txt[i];
		}
	tmp_txt[i] = 0;
	oTextOut(x, y, tmp_txt, cb ? cb : i);
	free(tmp_txt);
	return true;
}

bool
ExportEPS::oPolygon(POINT *pts, int cp, char *)
{
	int i, j;

	if(cp <1) return false;
	if(cp == 2) return oSolidLine(pts);
	FlushSolLines();
	for(i = cp-1, j = 0; i >= 0; i--, j++) {
		if(!(j & 0x07)) fprintf(oFile, "\n");
		fprintf(oFile, " %.1f %.1f", ix2eps(pts[i].x), iy2eps(pts[i].y));
		}
	fprintf(oFile, " moveto %d {lineto} repeat closepath %s fill ", cp-1, col2eps(CurrCol = dFillCol));
	if (hgo && hgo->OC_type != OC_HATCH) hgo->oPolygon(pts, cp);
	return oPolyline(pts, cp);
}

bool
ExportEPS::oSolidPolygon(POINT *pts, int cp)
{
	return oPolygon(pts, cp, 0L);
}

float
ExportEPS::ix2eps(int x)
{
	return (float)x/10.0f;
}

float
ExportEPS::iy2eps(int y)
{
	return (float)y/-10.0f + (float)BoundBox.bottom;
}

char *
ExportEPS::col2eps(DWORD color)
{
	static char txt[100];
	float r, g, b;

	r = (float)(color & 0xff)/255.0f;
	g = (float)((color>>8)&0xff)/255.0f;
	b = (float)((color>>16)&0xff)/255.0f;
#ifdef USE_WIN_SECURE
	sprintf_s(txt, 50, "%g %g %g setrgbcolor", r, g, b);
#else
	sprintf(txt, "%g %g %g setrgbcolor", r, g, b);
#endif
	return txt;
}

void
ExportEPS::FlushSolLines()
{
	int i, j;
	
	if(nSolLines <1) {
		nSolLines = 0;
		return;
		}
	if(nSolLines == 1) {
		fprintf(oFile, "\n1 setlinecap %.1f %.1f moveto %.1f %.1f lineto stroke ", 
			SolLines[0].Ymin, SolLines[0].Ymax, SolLines[0].Xmin, SolLines[0].Xmax);
		nSolLines = 0;
		return;
		}
	for(i = nSolLines-1, j = 0; i >=0; i--, j++) {
		if(!(j & 0x03)) fprintf(oFile, "\n");
		fprintf(oFile, " %.1f %.1f %.1f %.1f", SolLines[i].Xmin, SolLines[i].Xmax,
			SolLines[i].Ymin, SolLines[i].Ymax);
		}
	if(j > 8 && ((j & 0x3) >=2 || (j & 0x03) == 0)) fprintf(oFile, "\n");
	fprintf(oFile, " %d {moveto lineto} repeat stroke ", nSolLines);
	nSolLines = 0;
}

void
ExportEPS::AddSolLine(float x1, float y1, float x2, float y2)
{
	if(nSolLines >= 2000) FlushSolLines();
	SolLines[nSolLines].Ymin = x1;	SolLines[nSolLines].Ymax = y1;
	SolLines[nSolLines].Xmin = x2;	SolLines[nSolLines].Xmax = y2;
	nSolLines++;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Entry point to export graph to *.eps
void DoExportEps(GraphObj *g, char *FileName, DWORD flags)
{
	ExportEPS *ex;

	InfoBox((char*)SCMS_NOEPS);
	ex = new ExportEPS(g, FileName, flags);
	if(ex->StartPage()) {
		g->DoPlot(ex);
		ex->EndPage();
		}
//	delete(ex);			//this may crash on linux
}
