//use_gui.cpp, Copyright 2000-2025 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This modules contains code for builds using a graphical user interface.
// Builds running without GUI use no_gui.cpp instead.
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "rlp_strings.h"

extern char TmpTxt[];
extern def_vars defs;
extern GraphObj *CurrGO, *TrackGO;			//Selected Graphic Objects
extern Label *CurrLabel;
extern Graph *CurrGraph;
extern dragHandle *CurrHandle;
extern UndoObj Undo;
extern fmtText DrawFmtText;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Spread sheet buttons used for rows and columns
ssButton::ssButton(GraphObj *par, int x, int y, int w, int h):GraphObj(par, 0L)
{
	bLBdown = bSelected = bMarked = false;
	SetMinMaxRect(&rDims, x, y, x+w, y+h);
	Line.width = defs.GetSize(SIZE_HAIRLINE);	Line.patlength = defs.GetSize(SIZE_PATLENGTH);
	Line.color = 0x00000000L;			Line.pattern = 0x00000000L;
	Fill.type = FILL_NONE;				Fill.color = 0x00e8e8e8L;
	Fill.scale = 1.0;					Fill.hatch = NULL;
	TextDef.ColTxt = 0x00000000L;		TextDef.ColBg = 0x00ffffffL;
	TextDef.fSize = 4.0;				TextDef.RotBL = TextDef.RotCHAR = 0.0;
	TextDef.iSize = 0;					TextDef.Align = TXA_HLEFT | TXA_VTOP;
	TextDef.Mode = TXM_TRANSPARENT;		TextDef.Style = TXS_NORMAL;
	TextDef.Font = FONT_HELVETICA;		TextDef.text = 0L;
}

ssButton::~ssButton()
{
	if(TextDef.text) free(TextDef.text);
}

void
ssButton::DoPlot(anyOutput *o)
{
	POINT pts[3];

	Line.color = 0x00000000L;		Line.width = defs.GetSize(SIZE_HAIRLINE);
	if (bSelected) Fill.color = bMarked ? 0x00ffffc0L : 0x00ffffffL;
	else Fill.color = bMarked ? 0x00f0c0c0L : 0x00e0e0e0L;
	o->SetLine(&Line);				o->SetFill(&Fill);
	if(bLBdown)	o->oRectangle(rDims.left, rDims.top, rDims.right-1, rDims.bottom-1);
	else {
		o->oRectangle(rDims.left, rDims.top, rDims.right-2, rDims.bottom-2);
		Line.color = 0x00000000L;		o->SetLine(&Line);
		pts[0].x = rDims.left;					pts[0].y = pts[1].y = rDims.bottom-1;
		pts[1].x = pts[2].x = rDims.right-1;	pts[2].y = rDims.top-1;
		o->oPolyline(pts, 3);
		Line.color = 0x00ffffffL;		o->SetLine(&Line);
		pts[0].x = pts[1].x = rDims.left;		pts[0].y = rDims.bottom -3;
		pts[1].y = pts[2].y = rDims.top;		pts[2].x = rDims.right -2;
		o->oPolyline(pts, 3);
		}
	if(TextDef.text) {
		o->SetTextSpec(&TextDef);
#ifdef _WINDOWS
		o->oTextOut((rDims.left + rDims.right)/2-2, (rDims.top + rDims.bottom)/2-1, TextDef.text, 0);
#else
		o->oTextOut((rDims.left + rDims.right)/2-2, (rDims.top + rDims.bottom)/2+1, TextDef.text, 0);
#endif
		}

}
void
ssButton::DoMark(anyOutput *o, bool mark)
{
	bLBdown = mark;
	DoPlot(o);
	o->UpdateRect(&rDims, true);
}

bool
ssButton::Command(int cmd, void *tmpl, anyOutput *o)
{
	char *tmptxt;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_GETTEXT:
		if(TextDef.text && tmpl) {
			rlp_strcpy((char*)tmpl, 10, TextDef.text);
			return true;
			}
		return false;
	case CMD_SELECT:
		if(tmpl && *((int*)tmpl)) bSelected = true;
		else bSelected = false;
		if(o) {
			DoPlot(o);
			o->UpdateRect(&rDims, true);
			}
		return true;
	case CMD_SETSTYLE:
		if(tmpl && *((int*)tmpl)) bMarked = true;
		else bMarked = false;
		if(o) {
			DoPlot(o);		o->UpdateRect(&rDims, true);
			}
		return true;
	case CMD_SETTEXT:
		if(tmpl) {
			if(! TextDef.text) TextDef.text = (unsigned char*)malloc(10*sizeof(char));
			rlp_strcpy(TextDef.text, 10, (char*)tmpl);
			}
		return true;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		tmptxt = (char*)TextDef.text;
		memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = (unsigned char*)tmptxt;
		return true;
	case CMD_MOUSE_EVENT:
		if(!o || !(mev = (MouseEvent *) tmpl)) break;
		if(IsInRect(rDims, mev->x, mev->y)) {
			if(bLBdown) {
				if(!(mev->StateFlags & 0x01)) {
					o->HideMark();
					if(bLBdown) {
						bLBdown = false; DoPlot(o);
						}
					}
				}
			else if(mev->StateFlags & 0x01) {
				o->ShowMark(this, MRK_SSB_DRAW);
				}
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// drag handles are graphic objects which allow the user to interactively
//   change the size of an object
dragHandle::dragHandle(GraphObj *par, int which):GraphObj(par, 0L)
{
	type = which;
	Id = GO_DRAGHANDLE;
	LineDef.width = LineDef.patlength = 0.0;	LineDef.color = LineDef.pattern = 0L;
	FillDef.type = FILL_NONE;					FillDef.color = 0xcb00ff00;
	FillDef.scale = 1.0;						FillDef.hatch = 0L;
	minRC = maxRC = 0L;							x_pos = y_pos = 0.0;
}

dragHandle::~dragHandle()
{
	if(minRC) free(minRC);
	if(maxRC) free(maxRC);
	if(CurrHandle == this) CurrHandle = 0L;
}

double
dragHandle::GetSize(int select)
{
	switch (select) {
	case SIZE_XPOS:		return x_pos;
	case SIZE_YPOS:		return y_pos;
	}
	return 0.0;
}

void
dragHandle::DoPlot(anyOutput *o)
{
	double fx, fy, dx, dy;
	int ix, iy, idx;
	fPOINT3D fp3d, ifp3d;
	bool is3D = false;

	if(!o || !parent) return;
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	SetMinMaxRect(&drec, o->co2ix(parent->GetSize(SIZE_XPOS)+dx), 
		o->co2iy(parent->GetSize(SIZE_YPOS)+dy), o->co2ix(parent->GetSize(SIZE_XPOS+1)+dx),
		o->co2iy(parent->GetSize(SIZE_YPOS+1)+dy));
	switch(type) {
	case DH_19:		case DH_12:
		fx = parent->GetSize(SIZE_XPOS);	fy = parent->GetSize(SIZE_YPOS);
		break;
	case DH_99:		case DH_22:
		fx = parent->GetSize(SIZE_XPOS+1);	fy = parent->GetSize(SIZE_YPOS+1);
		break;
	case DH_29:
		fx = (parent->GetSize(SIZE_XPOS) + parent->GetSize(SIZE_XPOS+1))/2.0;
		fy = parent->GetSize(SIZE_YPOS);
		break;
	case DH_39:
		fx = parent->GetSize(SIZE_XPOS+1);	fy = parent->GetSize(SIZE_YPOS);
		break;
	case DH_49:
		fx = parent->GetSize(SIZE_XPOS);
		fy = (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
		break;
	case DH_59:
		fx = (parent->GetSize(SIZE_XPOS) + parent->GetSize(SIZE_XPOS+1))/2.0;
		fy = (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
		break;
	case DH_69:
		fx = parent->GetSize(SIZE_XPOS+1);
		fy = (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
		break;
	case DH_79:
		fx = parent->GetSize(SIZE_XPOS);
		fy = parent->GetSize(SIZE_YPOS+1);
		break;
	case DH_89:
		fx = (parent->GetSize(SIZE_XPOS) + parent->GetSize(SIZE_XPOS+1))/2.0;
		fy = parent->GetSize(SIZE_YPOS+1);
		break;
	case DH_18:	case DH_28:	case DH_38:	case DH_48:
	case DH_58:	case DH_68:	case DH_78:	case DH_88:
		fp3d.fx = parent->GetSize(SIZE_XPOS + type - DH_18);
		fp3d.fy = parent->GetSize(SIZE_YPOS + type - DH_18);
		fp3d.fz = parent->GetSize(SIZE_ZPOS + type - DH_18);
		is3D = true;
		break;
	default:
		if(type >= DH_DATA) {
			fx = parent->GetSize(SIZE_XPOS + type - DH_DATA);
			fy = parent->GetSize(SIZE_YPOS + type - DH_DATA);
			FillDef.color = (this == CurrHandle) ? 0x0L : 0x00ffffffL;
			}
		else return;
		}
	if(is3D) {
		o->cvec2ivec(&fp3d, &ifp3d);
		x_pos = ifp3d.fx;				y_pos = ifp3d.fy;
		ix = iround(ifp3d.fx);			iy = iround(ifp3d.fy);
		}
	else {
		x_pos = o->co2fix(fx + dx);		y_pos = o->co2fiy(fy + dy);
		ix = iround(x_pos);				iy = iround(y_pos);
		}
	SetMinMaxRect(&rDims, ix-4, iy-4, ix+4, iy+4);
	memcpy(&upd, &rDims, sizeof(RECT));
	switch(type) {
	case DH_12:	case DH_22:	case DH_19:	case DH_29:	case DH_39:
	case DH_49:	case DH_69:	case DH_79:	case DH_89:	case DH_99:
		o->SetLine(&LineDef);		o->SetFill(&FillDef);
		o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		defs.UpdRect(o, &rDims);
		break;
	case DH_59:
		o->SetLine(&LineDef);		o->SetFill(&FillDef);
		IncrementMinMaxRect(&rDims, 2);
		o->oCircle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		IncrementMinMaxRect(&rDims, 1);
		defs.UpdRect(o, &rDims);
		break;
	default:
		if(type >= DH_DATA) {
			idx = (type - DH_DATA);
			o->SetLine(&LineDef);		o->SetFill(&FillDef);
			if(parent->Id == GO_BEZIER && (idx %3)) {
				IncrementMinMaxRect(&rDims, -1);
				o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
				}
			else if(parent->Id == GO_POLYLINE || parent->Id == GO_BEZIER) {
				o->oCircle(rDims.left, rDims.top, rDims.right, rDims.bottom);
				}
			else {
				o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
				}
			defs.UpdRect(o, &rDims);
			}
		else {
			IncrementMinMaxRect(&rDims, 1);		o->SetFill(&FillDef);
			o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
			IncrementMinMaxRect(&rDims, 1);
			defs.UpdRect(o, &rDims);
			}
		break;
		}
}

bool
dragHandle::Command(int cmd, void *tmpl, anyOutput *o)
{
	lfPOINT pos;
	int idx;

	if(!parent) return false;
	switch (cmd) {
	case CMD_MOUSECURSOR:
		if(o) switch(type) {
		case DH_19:	case DH_99:
			o->MouseCursor(MC_SE, false);							break;
		case DH_29:	case DH_89:
			o->MouseCursor(MC_NORTH, false);						break;
		case DH_39:	case DH_79:
			o->MouseCursor(MC_NE, false);							break;
		case DH_49:	case DH_69:
			o->MouseCursor(MC_EAST, false);							break;
		case DH_19+4:
			o->MouseCursor(MC_MOVE, false);
			break;
		default:
			if(type >= DH_DATA) o->MouseCursor(MC_SALL, false);
			else if(parent->parent->parent->Id != GO_AXIS){
				o->HideMark();
				o->MouseCursor(MC_MOVE, false);
				}
			break;
			}
		return true;
	case CMD_MINRC:
		if(!(minRC)) minRC = (RECT*)calloc(1, sizeof(RECT));
		if(tmpl && minRC) SetMinMaxRect(minRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		return true;
	case CMD_MAXRC:
		if(!(maxRC)) maxRC = (RECT*)calloc(1, sizeof(RECT));
		if(tmpl && maxRC) SetMinMaxRect(maxRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		return true;
	case CMD_MOVE:
		idx = type >= DH_DATA ? type - DH_DATA : 0;
		pos.fx = NiceValue(((lfPOINT*)tmpl)[0].fx);
		pos.fy = NiceValue(((lfPOINT*)tmpl)[0].fy);
		if(pos.fx == 0.0 && pos.fy == 0.0) return false;
		parent->Command(CMD_SAVEPOS, &idx, o);
		switch(type) {
		case DH_12:
			parent->SetSize(SIZE_XPOS, pos.fx + parent->GetSize(SIZE_XPOS));
			parent->SetSize(SIZE_YPOS, pos.fy + parent->GetSize(SIZE_YPOS));
			break;
		case DH_22:
			parent->SetSize(SIZE_XPOS+1, pos.fx + parent->GetSize(SIZE_XPOS+1));
			parent->SetSize(SIZE_YPOS+1, pos.fy + parent->GetSize(SIZE_YPOS+1));
			break;
		case DH_19: 
			parent->parent->SetSize(SIZE_XPOS, pos.fx + parent->GetSize(SIZE_XPOS));
			parent->parent->SetSize(SIZE_YPOS, pos.fy + parent->GetSize(SIZE_YPOS));
			break;
		case DH_29: parent->parent->SetSize(SIZE_YPOS, pos.fy + parent->GetSize(SIZE_YPOS));
			break;
		case DH_39: 
			parent->parent->SetSize(SIZE_YPOS, pos.fy + parent->GetSize(SIZE_YPOS));
			parent->parent->SetSize(SIZE_XPOS + 1, pos.fx + parent->GetSize(SIZE_XPOS + 1));
			break;
		case DH_69: parent->parent->SetSize(SIZE_XPOS+1, pos.fx + parent->GetSize(SIZE_XPOS+1));
			break;
		case DH_99: 
			parent->parent->SetSize(SIZE_XPOS+1, pos.fx + parent->GetSize(SIZE_XPOS+1));
			parent->parent->SetSize(SIZE_YPOS + 1, pos.fy + parent->GetSize(SIZE_YPOS + 1));
			break;
		case DH_89: parent->parent->SetSize(SIZE_YPOS+1, pos.fy + parent->GetSize(SIZE_YPOS+1));
			break;
		case DH_79: 
			parent->parent->SetSize(SIZE_YPOS+1, pos.fy + parent->GetSize(SIZE_YPOS+1));
			parent->parent->SetSize(SIZE_XPOS, pos.fx + parent->GetSize(SIZE_XPOS));
			break;
		case DH_49:	parent->parent->SetSize(SIZE_XPOS, pos.fx + parent->GetSize(SIZE_XPOS));
			break;
		case DH_18:	case DH_28:	case DH_38:	case DH_48:
		case DH_58:	case DH_68:	case DH_78:	case DH_88:
		case DH_59:
			return parent->parent->Command(cmd, tmpl, o);
		default:
			if (type >= DH_DATA) {
				parent->SetSize(SIZE_XPOS + idx, pos.fx + parent->GetSize(SIZE_XPOS + idx));
				parent->SetSize(SIZE_YPOS + idx, pos.fy + parent->GetSize(SIZE_YPOS + idx));
				CurrGO = parent;
				}
			break;
			}
		return parent->Command(CMD_REDRAW, 0L, o);
		break;
		}
	return false;
}

void *
dragHandle::ObjThere(int x, int y)
{
	if(IsInRect(rDims, x, y)) return (CurrHandle = this);
	else return 0L;
}


void
dragHandle::Track(POINT *p, anyOutput *o, bool)
{
	POINT p1, p2, pts[5], bez[256];
	double dx, dy;
	int i, first=0, last = 0, idx;
	long nbez;
	DWORD color;

	if(!parent || !o) return;
	o->MrkMode = MRK_NONE;
	Command(CMD_MOUSECURSOR, 0L, o);
	if(upd.right < upd.left) Swap(upd.right, upd.left);
	if(upd.bottom < upd.top) Swap(upd.bottom, upd.top);
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	IncrementMinMaxRect(&upd, 2);
	InvalidateOutput(o);
	o->UpdateRect(&upd, false);
	if(type >= DH_19 && type <= DH_99) memcpy(&upd, &drec, sizeof(RECT));
	color = parent->GetColor(COL_DATA_LINE);
	switch(type) {
	case DH_12:
	case DH_22:
		pts[0].x = o->co2ix(parent->GetSize(SIZE_XPOS)+dx);
		pts[0].y = o->co2iy(parent->GetSize(SIZE_YPOS)+dy);
		pts[1].x = o->co2ix(parent->GetSize(SIZE_XPOS+1)+dx);
		pts[1].y = o->co2iy(parent->GetSize(SIZE_YPOS+1)+dy);
		if(type == DH_12) {
			pts[0].x += p->x;			pts[0].y += p->y;
			}
		else if(type == DH_22) {
			pts[1].x += p->x;			pts[1].y += p->y;
			}
		UpdateMinMaxRect(&upd, pts[0].x, pts[0].y);
		UpdateMinMaxRect(&upd, pts[1].x, pts[1].y);
		last = 2;
		break;
	case DH_19:
		if(minRC && IsInRect((*minRC), (upd.left+p->x), ((upd.top+upd.bottom)>>1)))
			upd.left = minRC->left;
		else if(maxRC && !IsInRect((*maxRC), (upd.left+p->x), (upd.top+upd.bottom)>>1))
			upd.left = upd.left+p->x >= maxRC->right ? maxRC->right : maxRC->left;
		else upd.left += p->x;
		if (minRC && IsInRect((*minRC), ((upd.left + upd.right) >> 1), (upd.top + p->y)))
			upd.top = minRC->top;
		else if (maxRC && !IsInRect((*maxRC), ((upd.left + upd.right) >> 1), (upd.top + p->y)))
			upd.top = upd.top + p->y >= maxRC->bottom ? maxRC->bottom : maxRC->top;
		else upd.top += p->y;
		break;
	case DH_29:
		if(minRC && IsInRect((*minRC), ((upd.left + upd.right)>>1), (upd.top+p->y)))
			upd.top = minRC->top;
		else if(maxRC && !IsInRect((*maxRC), ((upd.left + upd.right)>>1), (upd.top+p->y)))
			upd.top = upd.top +p->y >= maxRC->bottom? maxRC->bottom : maxRC->top;
		else upd.top += p->y;
		break;
	case DH_39:
		if(minRC && IsInRect((*minRC), ((upd.left + upd.right)>>1), (upd.top+p->y)))
			upd.top = minRC->top;
		else if(maxRC && !IsInRect((*maxRC), (upd.left + upd.right)>>1, upd.top+p->y))
			upd.top = upd.top+p->y >= maxRC->bottom ? maxRC->bottom : maxRC->top;
		else upd.top += p->y;
		if (minRC && IsInRect((*minRC), (upd.right + p->x), ((upd.top + upd.bottom) >> 1)))
			upd.right = minRC->right;
		else if (maxRC && !IsInRect((*maxRC), (upd.right + p->x), ((upd.top + upd.bottom) >> 1)))
			upd.right = upd.right + p->x <= maxRC->left ? maxRC->left : maxRC->right;
		else upd.right += p->x;
		break;
	case DH_69:
		if(minRC && IsInRect((*minRC), (upd.right+p->x), ((upd.top+upd.bottom)>>1)))
			upd.right = minRC->right;
		else if(maxRC && !IsInRect((*maxRC), (upd.right+p->x), ((upd.top+upd.bottom)>>1)))
			upd.right = upd.right+p->x <= maxRC->left ? maxRC->left : maxRC->right;
		else upd.right += p->x;
		break;
	case DH_99:
		if (minRC && IsInRect((*minRC), (upd.right + p->x), ((upd.top + upd.bottom) >> 1)))
			upd.right = minRC->right;
		else if (maxRC && !IsInRect((*maxRC), (upd.right + p->x), ((upd.top + upd.bottom) >> 1)))
			upd.right = upd.right + p->x <= maxRC->left ? maxRC->left : maxRC->right;
		else upd.right += p->x;
		if (minRC && IsInRect((*minRC), ((upd.left + upd.right) >> 1), (upd.bottom + p->y)))
			upd.bottom = minRC->bottom;
		else if (maxRC && !IsInRect((*maxRC), ((upd.left + upd.right) >> 1), (upd.bottom + p->y)))
			upd.bottom = upd.bottom + p->y <= maxRC->top ? maxRC->top : maxRC->bottom;
		else upd.bottom += p->y;
		break;
	case DH_89:
		if (minRC && IsInRect((*minRC), ((upd.left + upd.right) >> 1), (upd.bottom + p->y)))
			upd.bottom = minRC->bottom;
		else if (maxRC && !IsInRect((*maxRC), ((upd.left + upd.right) >> 1), (upd.bottom + p->y)))
			upd.bottom = upd.bottom + p->y <= maxRC->top ? maxRC->top : maxRC->bottom;
		else upd.bottom += p->y;
		break;
	case DH_79:
		if(minRC && IsInRect((*minRC), ((upd.left + upd.right)>>1), (upd.bottom+p->y)))
			upd.bottom = minRC->bottom;
		else if(maxRC && !IsInRect((*maxRC), ((upd.left + upd.right)>>1), (upd.bottom+p->y)))
			upd.bottom = upd.bottom+p->y <= maxRC->top? maxRC->top : maxRC->bottom;
		else upd.bottom += p->y;
		if (minRC && IsInRect((*minRC), (upd.left + p->x), ((upd.top + upd.bottom) >> 1)))
			upd.left = minRC->left;
		else if (maxRC && !IsInRect((*maxRC), (upd.left + p->x), ((upd.top + upd.bottom) >> 1)))
			upd.left = upd.left + p->x >= maxRC->right ? maxRC->right : maxRC->left;
		else upd.left += p->x;
		break;
	case DH_49:
		if(minRC && IsInRect((*minRC), (upd.left+p->x), ((upd.top+upd.bottom)>>1)))
			upd.left = minRC->left;
		else if(maxRC && !IsInRect((*maxRC), (upd.left+p->x), ((upd.top+upd.bottom)>>1)))
			upd.left = upd.left+p->x >= maxRC->right ? maxRC->right : maxRC->left;
		else upd.left += p->x;
		break;
	case DH_18:	case DH_28:	case DH_38:	case DH_48:
	case DH_58:	case DH_68:	case DH_78:	case DH_88:
		CurrGO = this;
		parent->parent->Track(p, o, false);		return;
	case DH_59:
		parent->parent->Track(p, o, false);		return;
	default:
		if(type >= DH_DATA) {
			idx = type - DH_DATA;
			pts[1].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx)+dx);
			pts[1].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx)+dy);
			pts[1].x += p->x;				pts[1].y += p->y;
			if(type > DH_DATA) {
				pts[0].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx -1)+dx);
				pts[0].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx -1)+dy);
				}
			else {
				pts[0].x = pts[1].x;		pts[0].y = pts[1].y;
				}
			pts[3].x = pts[2].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx +1)+dx);
			pts[3].y = pts[2].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx +1)+dy);
			UpdateMinMaxRect(&upd, pts[0].x, pts[0].y);
			UpdateMinMaxRect(&upd, pts[1].x, pts[1].y);
			UpdateMinMaxRect(&upd, pts[2].x, pts[2].y);
			if(parent ->Id == GO_BEZIER) {
				switch(idx % 3) {
				case 0:
					o->ShowLine(pts, 3, 0x00b4b4b4L);
					pts[0].x = pts[1].x;		pts[0].y = pts[1].y;
					for(nbez = 0, i = 1; i < 4; i++) {
						pts[i].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx +i)+dx);
						pts[i].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx +i)+dy);
						}
					DrawBezier(&nbez, bez, pts[0], pts[1], pts[2], pts[3], 0);
					o->ShowLine(bez, nbez, color);
					if(idx < 3) return;
					pts[3].x = pts[0].x;		pts[3].y = pts[0].y;
					for(i = nbez = 0, idx -= 3; i < 3; i++) {
						pts[i].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx +i)+dx);
						pts[i].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx +i)+dy);
						}
					DrawBezier(&nbez, bez, pts[0], pts[1], pts[2], pts[3], 0);
					o->ShowLine(bez, nbez, color);
					return;
				case 1:		
					pts[3].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx +2)+dx);
					pts[3].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx +2)+dy);
					last = 2;				break;
				case 2:	
					pts[2].x = pts[1].x;	pts[2].y = pts[1].y;
					pts[1].x = pts[0].x;	pts[1].y = pts[0].y;
					pts[0].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx -2)+dx);
					pts[0].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx -2)+dy);
					first = 2;	last = 4;	break;
					}
				if(idx %3) {
					nbez = 0;
					DrawBezier(&nbez, bez, pts[0], pts[1], pts[2], pts[3], 0);
					o->ShowLine(bez, nbez, color);
					}
				color = 0x00b8b8b8L;
				}
			else last = 3;
			if(color == 0x0L || color == 0x00ffffffL) color = 0x00b8b8b8L;
			}
		else return;
		}
	if(type >= DH_19 && type <= DH_99) {
		pts[0].x = pts[4].x = pts[3].x = p1.x = upd.left;	
		pts[0].y = pts[1].y = pts[4].y = p1.y = upd.top;
		pts[1].x = pts[2].x = p2.x = upd.right;	
		pts[2].y = pts[3].y = p2.y = upd.bottom;
		last = 5;
		if(parent->parent->Id == GO_ELLIPSE) o->ShowEllipse(p1, p2, color);
		}
	o->ShowLine(pts+first, last-first, color);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the dragRect object uses nine dragHandles to create a user interface
//   for modification of rectangular shapes
dragRect::dragRect(GraphObj *par, int which):GraphObj(par, 0L)
{
	int i;

	type = which;
	Id = GO_DRAGRECT;
	if ((handles = (dragHandle**)calloc(9, sizeof(dragHandle*)))){
		if (par->parent->Id == GO_AXIS) {
//			handles[0] = new dragHandle(this, DH_19);
//			handles[1] = new dragHandle(this, DH_99);
			handles[2] = new dragHandle(this, DH_19 + 4);
			}
		else for (i = 0; i < 9; i++){
			if (i == 4 && type == 0) handles[i] = new dragHandle(this, DH_19 + i);
			else if (i != 4) handles[i] = new dragHandle(this, DH_19 + i);
			}
		}
}

dragRect::~dragRect()
{
	int i;

	if (handles) {
		for (i = 0; i < 9; i++) {
			if (handles[i]) DeleteGO(handles[i]);
			}
		}
}

void
dragRect::DoPlot(anyOutput *o)
{
	int i;
	RECT clpRC;
	bool hasClp;

	if((hasClp = o->GetClipRec(&clpRC))) o->ClipRect(NULL);
	if (handles) {
		for (i = 0; i < 9; i++){
			if (handles[i]) handles[i]->DoPlot(o);
			}
		}
	if (hasClp) o->ClipRect(&clpRC);
}

bool
dragRect::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;

	if(!parent) return false;
	switch (cmd) {
	case CMD_MINRC:
	case CMD_MAXRC:
		if(handles) for(i = 0; i < 9; i++) {
			if(handles[i]) handles[i]->Command(cmd, tmpl, o);
			}
		break;
	case CMD_SAVEPOS:
	case CMD_REDRAW:
		return parent->Command(cmd, tmpl, o);
		}
	return false;
}

void *
dragRect::ObjThere(int x, int y)
{
	int i;
	void *go;

	if(handles)	for(i = 0; i < 9; i++) 
		if(handles[i] && (go = (handles[i])->ObjThere(x, y))) return go;
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// implement some kind of virtual trackball for 3D plots
Drag3D::Drag3D(GraphObj *par):GraphObj(par, 0L)
{
	int i;

	Id = GO_DRAG3D;
	if((handles = (dragHandle**)calloc(8, sizeof(dragHandle*))))
		for(i = 0; i < 8; i++){
			handles[i] = new dragHandle(this, DH_18 + i);
			}
}

Drag3D::~Drag3D()
{
	int i;

	if(handles) for(i = 0; i < 8; i++) if(handles[i]) DeleteGO(handles[i]);
}

void
Drag3D::DoPlot(anyOutput *o)
{
	int i;
	static LineDEF line;

	line.color = 0x80000000;			line.patlength = 0.0;
	line.width = defs.GetSize(SIZE_HAIRLINE);
	line.pattern = 0;
	o->SetLine(&line);
	if(handles) for(i = 0; i < 8; i++) if(handles[i]) handles[i]->DoPlot(o);
}

void *
Drag3D::ObjThere(int x, int y)
{
	int i;
	void *go;

	if(handles)	for(i = 0; i < 8; i++) 
		if(handles[i] && (go = (handles[i])->ObjThere(x, y))) return go;
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// frame rectangle
FrmRect::FrmRect(GraphObj *par, fRECT *lim, fRECT *c, fRECT *chld):GraphObj(par, 0L)
{
	parent = par;
	limRC = lim;		cRC = c;		chldRC = chld;
	if (cRC) {
		CurrRect.Xmax = cRC->Xmax;	CurrRect.Xmin = cRC->Xmin;
		CurrRect.Ymax = cRC->Ymax;	CurrRect.Ymin = cRC->Ymin;
		}
	else {
		CurrRect.Xmax = CurrRect.Xmin = CurrRect.Ymax = CurrRect.Ymin = 0.0;
		}
	drag = 0L;		mo = 0L;
	Id = GO_FRAMERECT;
	moveable = true;
	Fill.type = FILL_NONE;
	Line.color = FillLine.color = Fill.color = 0x00ffffffL;
	Line.width = FillLine.width = 0.0;
	Line.patlength = FillLine.patlength = Fill.scale = 1.0;
	Line.pattern = FillLine.pattern = 0x0L;
	Fill.hatch = &FillLine;
	minRC = maxRC = 0L;
}

FrmRect::~FrmRect()
{
	if(drag) DeleteGO(drag);
	drag = 0L;
	if(minRC) free(minRC);
	minRC = 0L;	
	if(maxRC) free(maxRC);
	maxRC = 0L;
	if(mo) DelBitmapClass(mo);
	mo = 0L;
}

double
FrmRect::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:			return CurrRect.Xmin;
	case SIZE_XPOS+1:		return CurrRect.Xmax;
	case SIZE_YPOS:			return CurrRect.Ymin;
	case SIZE_YPOS+1:		return CurrRect.Ymax;
	}
	return 0.0;
}

bool
FrmRect::SetSize(int select, double value)
{
	double tmp, o_left, o_top;

	o_left = cRC->Xmin;		o_top = cRC->Ymin;
	switch (select & 0xfff) {
	case SIZE_XPOS:
		if(limRC) value -= limRC->Xmin;
		if(swapX) cRC->Xmax = value;
		else cRC->Xmin = value;
		break;
	case SIZE_XPOS+1:
		if(limRC) value -= limRC->Xmin;
		if(swapX) cRC->Xmin = value;
		else cRC->Xmax = value;
		break;
	case SIZE_YPOS:
		if(limRC) value -= limRC->Ymin;
		if(swapY) cRC->Ymin = value;
		else cRC->Ymax = value;
		break;
	case SIZE_YPOS+1:
		if(limRC) value -= limRC->Ymin;
		if(swapY) cRC->Ymax = value;
		else cRC->Ymin = value;
		break;
	default: return false;
		}
	if((swapX && cRC->Xmin < cRC->Xmax) || (!swapX && cRC->Xmin > cRC->Xmax)) {
		tmp = cRC->Xmin;	cRC->Xmin = cRC->Xmax;	cRC->Xmax = tmp;
		}
	if((swapY && cRC->Ymin > cRC->Ymax) || (!swapY && cRC->Ymin < cRC->Ymax)) {
		tmp = cRC->Ymin;	cRC->Ymin = cRC->Ymax;	cRC->Ymax = tmp;
		}
	if(chldRC) {		//check if new rectangle is not inside child rectangle
		if(cRC->Xmin > o_left+ chldRC->Xmin) cRC->Xmin = o_left + chldRC->Xmin;
		if(cRC->Xmax < o_left+ chldRC->Xmax) cRC->Xmax = o_left + chldRC->Xmax;
		if(cRC->Ymin > o_top+ chldRC->Ymin) cRC->Ymin = o_top + chldRC->Ymin;
		if(cRC->Ymax < o_top+ chldRC->Ymax) cRC->Ymax = o_top + chldRC->Ymax;
		}
	if(chldRC && (o_left != cRC->Xmin || o_top != cRC->Ymin)) {
		chldRC->Xmin -= (tmp = cRC->Xmin - o_left);		chldRC->Xmax -= tmp;
		chldRC->Ymin -= (tmp = cRC->Ymin - o_top);		chldRC->Ymax -= tmp;
		}
	return true;
}

bool
FrmRect::SetColor(int select, DWORD col)
{
	switch(select & 0xfff){
	case COL_DRECT:
		Line.color = col;
		Fill.color = col;		return true;
	case COL_BG:
		Fill.color = col;		return true;
	case COL_GRECT:
		Fill.color = col;		return true;
	case COL_GRECTLINE:
		Line.color = col;		return true;
		}
	return false;
}

void 
FrmRect::DoMark(anyOutput *o, bool mark)
{
	if(!parent || !o) return;
	if (!drag && (drag = new dragRect(this, (!limRC && parent->moveable) ? 0 : 1))){
		if(minRC) drag->Command(CMD_MINRC, minRC, o);
		if(maxRC) drag->Command(CMD_MAXRC, maxRC, o);
		}
	if(mark && drag){
		SetMinMaxRect(&rDims, o->co2ix(cRC->Xmin), o->co2iy(cRC->Ymax), o->co2ix(cRC->Xmax), o->co2iy(cRC->Ymin));
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		if (parent->parent->Id == GO_PAGE) {
			mrc.left = o->co2ix(parent->parent->GetSize(SIZE_GRECT_LEFT));
			mrc.top = o->co2ix(parent->parent->GetSize(SIZE_GRECT_TOP));
			mrc.right = o->co2ix(parent->parent->GetSize(SIZE_GRECT_RIGHT));
			mrc.bottom = o->co2ix(parent->parent->GetSize(SIZE_GRECT_BOTTOM));
			}
		else {
			memcpy(&mrc, &rDims, sizeof(RECT));
			IncrementMinMaxRect(&mrc, 8);
			}
		mo = GetRectBitmap(&mrc, o, false);
		drag->DoPlot(o);
		o->UpdateRect(&mrc, true);
		}
	else RestoreRectBitmap(&mo, &mrc, o, false);
}

void
FrmRect::DoPlot(anyOutput *o)
{
	int x1, y1, x2, y2;
	double tmp;

	if(!(cRC) || !o || !parent) return;
	CurrRect.Xmin = cRC->Xmin;				CurrRect.Xmax = cRC->Xmax;
	CurrRect.Ymin = cRC->Ymax;				CurrRect.Ymax = cRC->Ymin;
	if (parent->Id != GO_AXIS) {
		if (limRC) {
			CurrRect.Xmin += limRC->Xmin;		CurrRect.Xmax += limRC->Xmin;
			CurrRect.Ymin += limRC->Ymin;		CurrRect.Ymax += limRC->Ymin;
			}
		if ((swapX = (CurrRect.Xmin > CurrRect.Xmax))) {
			tmp = CurrRect.Xmin;	CurrRect.Xmin = CurrRect.Xmax;	CurrRect.Xmax = tmp;
			}
		if ((swapY = (CurrRect.Ymin > CurrRect.Ymax))) {
			tmp = CurrRect.Ymin;	CurrRect.Ymin = CurrRect.Ymax;	CurrRect.Ymax = tmp;
			}
		}
	o->SetLine(&Line);						o->SetFill(&Fill);
	x1 = o->co2ix(CurrRect.Xmin);		y1 = o->co2iy(CurrRect.Ymin);
	x2 = o->co2ix(CurrRect.Xmax);		y2 = o->co2iy(CurrRect.Ymax);
	o->oRectangle(x1, y1, x2, y2);
	SetMinMaxRect(&rDims, x1, y1, x2, y2);
}

bool
FrmRect::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	AxisDEF *ax;

	if(!parent) return false;
	if (parent->Id == GO_AXIS) ax = ((Axis*)parent)->GetAxis();
	else ax = 0L;
	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !(CurrGO) && (o)){
				o->ShowMark(this, MRK_GODRAW);
				if(!CurrGraph && parent && parent->Id == GO_GRAPH) CurrGraph = (Graph*)parent;
				return true;
				}
			}
		return false;
	case CMD_SELECT:
		memcpy(&mrc, &rDims, sizeof(RECT));
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MINRC:
		if(!(minRC)) minRC = (RECT*)calloc(1, sizeof(RECT));
		if(minRC && tmpl) SetMinMaxRect(minRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		if(drag) drag->Command(cmd, tmpl, o);
		return true;
	case CMD_MAXRC:
		if(!(maxRC)) maxRC = (RECT*)calloc(1, sizeof(RECT));
		if(maxRC && tmpl) SetMinMaxRect(maxRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		if(drag) drag->Command(cmd, tmpl, o);
		return true;
	case CMD_MOVE:
		if (parent->Id == GO_GRAPH || parent->Id == GO_PAGE) {
			return parent->Command(cmd, tmpl, o);
			}
		cRC->Xmin += NiceValue(((lfPOINT*)tmpl)[0].fx);
		cRC->Ymin += NiceValue(((lfPOINT*)tmpl)[0].fy);
		cRC->Xmax += NiceValue(((lfPOINT*)tmpl)[0].fx);
		cRC->Ymax += NiceValue(((lfPOINT*)tmpl)[0].fy);
		if (ax) {
			ax->loc[0].fx += NiceValue(((lfPOINT*)tmpl)[0].fx);
			ax->loc[1].fx += NiceValue(((lfPOINT*)tmpl)[0].fx);
			ax->loc[0].fy += NiceValue(((lfPOINT*)tmpl)[0].fy);
			ax->loc[1].fy += NiceValue(((lfPOINT*)tmpl)[0].fy);
			}
		if (mo) {
			DelBitmapClass(mo);			mo = 0L;
			mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
			}
		return parent->Command(CMD_REDRAW, 0L, o);
	case CMD_REDRAW:
		return parent->Command(CMD_REDRAW, 0L, o);
	case CMD_SAVEPOS:
		return parent->Command(cmd, tmpl, o);
	case CMD_SETCHILD:
		chldRC = (fRECT*)tmpl;
		return true;
		}
	return false;
}

void *
FrmRect::ObjThere(int x, int y)
{
	if(drag) return drag->ObjThere(x, y);
	return 0L;
}

void
FrmRect::Track(POINT *p, anyOutput *o, bool)
{
	POINT tpts[5];
	RECT old_rc;
	GridView *view;
//	AxisDEF *ax;

//	if (parent && parent->Id == GO_AXIS) ax = ((Axis*)parent)->GetAxis();
//	else ax = 0L;

	if(o){
		view = 0L;
		memcpy(&old_rc, &rDims, sizeof(rDims));
		InvalidateOutput(o);
		tpts[0].x = tpts[1].x = tpts[4].x = o->co2ix(cRC->Xmin)+p->x;		
		tpts[0].y = tpts[3].y = tpts[4].y = o->co2iy(cRC->Ymin)+p->y;
		tpts[1].y = tpts[2].y = o->co2iy(cRC->Ymax)+p->y;
		tpts[2].x = tpts[3].x = o->co2ix(cRC->Xmax)+p->x;
		UpdateMinMaxRect(&rDims, tpts[0].x, tpts[0].y);
		UpdateMinMaxRect(&rDims, tpts[2].x, tpts[2].y);	
		if (parent && parent->Id == GO_GRAPH && (parent->parent->Id == GO_PAGE || parent->parent->Id == GO_GRAPH) 
			&& (view = new GridView(o))) {
			view->setVPorg(o->co2ix(0) + p->x, o->co2iy(0) + p->y, o->getVPorgScale());
			if (!mo) {
				mrc.left = o->co2ix(parent->parent->GetSize(SIZE_GRECT_LEFT));
				mrc.top = o->co2ix(parent->parent->GetSize(SIZE_GRECT_TOP));
				mrc.right = o->co2ix(parent->parent->GetSize(SIZE_GRECT_RIGHT));
				mrc.bottom = o->co2ix(parent->parent->GetSize(SIZE_GRECT_BOTTOM));
				mo = GetRectBitmap(&mrc, o);
				}
			else {
				o->CopyBitmap(mrc.left, mrc.top + o->MenuHeight(), mo, 0, 0, mrc.right - mrc.left,
					mrc.bottom - mrc.top, false);
				}
			parent->DoPlot(view);
			defs.UpdRect(o, &mrc);
			delete view;
			}
		else if (TrackGO && TrackGO->Id == GO_DRAGHANDLE && parent->Id == GO_AXIS) {
			switch (TrackGO->type) {
			case DH_19+4:
				((Axis*)parent)->Track(p, o, false);
				break;
				}
			}
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void * 
Arrow::ObjThere(int x, int y)
{
	if(!IsInRect(rDims, x, y)) return 0L;
	if(IsCloseToLine(&pts[0], &pts[1], x, y, 5.0) ||
		IsCloseToLine(&pts[2], &pts[3], x, y, 5.0) ||
		IsCloseToLine(&pts[3], &pts[4], x, y, 5.0)){
		if(dh1 && dh1->ObjThere(x, y)) return dh1;
		if(dh2 && dh2->ObjThere(x, y)) return dh2;
		return this;
		}
	return 0L;
}

void
Arrow::Track(POINT *p, anyOutput *o, bool)
{
	POINT *tpts;
	RECT old_rc;
	int i;

	if(o && (tpts = (POINT*)malloc(6*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		InvalidateOutput(o);
		for(i = 0; i < 5; i++) {
			tpts[i].x = pts[i].x+p->x;
			tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		switch(type & 0xff) {
		case ARROW_LINE:
			o->ShowLine(tpts+2, 3, LineDef.color);
			o->ShowLine(tpts, 2, LineDef.color);
			break;
		case ARROW_NOCAP:
			o->ShowLine(tpts, 2, LineDef.color);
			break;
		case ARROW_TRIANGLE:
			tpts[5].x = tpts[2].x;		tpts[5].y = tpts[2].y;
			o->ShowLine(tpts+2, 4, LineDef.color);
			tpts[1].x = (tpts[2].x + tpts[4].x)>>1;
			tpts[1].y = (tpts[2].y + tpts[4].y)>>1;
			o->ShowLine(tpts, 2, LineDef.color);
			break;
			}
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

void
Label::ShowCursor(anyOutput *o)
{
	if(o) {
		HideTextCursor();
		DrawFmtText.DrawText(o);
		CalcRect(o);
		if((o->OC_type & 0xff) == OC_BITMAP) ShowTextCursor(o, &Cursor, TextDef.ColTxt);
		}
}

bool
Label::AddChar(int ci, anyOutput *o)
{
	unsigned char c, *txt1 = 0L;
	int i, j, k, txt1_size;
	GraphObj *golist[2];
	DWORD uflags;

	if(!o || (ci != 13 && ci < 32)) return false;
	if (CheckMark() && !bBusy) {
		Undo.String(this, (char**)(&TextDef.text), 0L);
		Undo.ValInt(this, &m1, UNDO_CONTINUE);
		Undo.ValInt(this, &m2, UNDO_CONTINUE);
		rlp_strcpy(TextDef.text+m1, (int)strlen((char*)(TextDef.text+m1)+1), (char*)(TextDef.text+m2));
		CursorPos = m1;		uflags = UNDO_CONTINUE;
		}
	else uflags = 0L;
	m1 = m2 = -1;
	if(ci == 13 && parent){		//CR
		if(parent->Id == GO_MLABEL){
			parent->Command(CMD_SETFOCUS, this, o);
			return parent->Command(CMD_ADDCHAR, &ci, o);
			}
		if((golist[1] = new mLabel(parent, data, fPos.fx, fPos.fy, &TextDef, 
			(char*)TextDef.text, CursorPos, flags))){
			golist[1]->moveable = moveable;
			golist[1]->SetSize(SIZE_LB_XDIST, fDist.fx);
			golist[1]->SetSize(SIZE_LB_YDIST, fDist.fy);
			if (moveable) {
				golist[1]->name = rlp_strdup((char*)"ml text");
				}
			}
		golist[0] = this;
		if(!parent->Command(CMD_MUTATE, golist, o)) DeleteGO(golist[1]);
		return false;
		}
	if(TextDef.text && TextDef.text[0]) i = rlp_strlen((char*)TextDef.text);
	else i = 0;
	if(CursorPos > i)CursorPos = i;
	if(ci > 254) {
		Undo.String(this, (char**)(&TextDef.text), uflags);
		Undo.ValInt(this, &CursorPos, uflags = UNDO_CONTINUE);
		txt1_size = i + 20;
		txt1 = (unsigned char*)malloc(txt1_size*sizeof(unsigned char));
		if(txt1) {
			j = CursorPos;
			if(CursorPos) k = rlp_strcpy(txt1, CursorPos+1, TextDef.text);
			else k = 0;
#ifdef USE_WIN_SECURE
			k += sprintf_s((char*)(txt1+k), i+12-k, "&#%x;", ci);
#else
			k += sprintf((char*)(txt1+k), "&#%x;",ci);
#endif
			CursorPos = k;
			if (k > txt1_size - 5) {
				txt1_size += 20;
				txt1 = (unsigned char*)realloc(txt1, txt1_size*sizeof(unsigned char));
				}
			k += rlp_strcpy(txt1+k, i+20-k, TextDef.text+j);
			if(TextDef.text) free(TextDef.text);
			TextDef.text = txt1;			RedrawEdit(o);
			}
		return true;
		}
	else if( ci > 31) c = (char)ci;
	else return false;
	if(!TextDef.text || CursorPos < 10 || c == ' ') {
		if (!bBusy)Undo.String(this, (char**)(&TextDef.text), uflags);
		if (!bBusy)Undo.ValInt(this, &CursorPos, uflags = UNDO_CONTINUE);
		}
	i = (TextDef.text && TextDef.text[0]) ? rlp_strlen(TextDef.text) : 0;
	if (i && i < (rlp_strlen(TextDef.text)-2)) {
		TextDef.text[i + 1] = 0;
		for (j = i; j > CursorPos; j--) TextDef.text[j] = TextDef.text[j - 1];
		TextDef.text[j] = c;
		CursorPos++;
		if (!bBusy) RedrawEdit(o);
		return true;
		}
	txt1 = (unsigned char*)malloc((i + 10)* sizeof(char));
	if(txt1) {
		if(CursorPos) k = rlp_strcpy(txt1, CursorPos+1, TextDef.text);
		else k = 0;
		txt1[k++] = c;
		if (k < rlp_strlen(TextDef.text)) {
			k += rlp_strcpy(txt1 + k, 80, TextDef.text + CursorPos);
			}
 		if(TextDef.text) free(TextDef.text);
		TextDef.text = txt1;	txt1[k] = 0;
		CursorPos++;
		if (!bBusy) RedrawEdit(o);
		}
	return true;
}

bool
Label::AddText(unsigned char *src, anyOutput *o)
{
	int i, cc;

	if(!src || !src[0]) return false;
	bBusy = true;
	if (*src < 32) i = 1;
	else i = 0;
	Undo.String(this, (char**)(&TextDef.text), 0L);
	Undo.ValInt(this, &CursorPos, UNDO_CONTINUE);
	//do not allow paste of multiline text to label
	for (; src[i] > 31; i++) {
		cc = src[i];		AddChar(cc, o);
		}
	bBusy = false;
	RedrawEdit(o);
	return true;
}

void
Label::CalcCursorPos(int x, int y, anyOutput *o)
{
//	int i, j, ix, x1, y1;
//	double x2, h;
	int j, ix, x1, y1;

	if(!o || !TextDef.text) return;
	TextDef.iSize = o->un2iy(TextDef.fSize);
	if(parent && parent->Id != GO_MLABEL) o->SetTextSpec(&TextDef);
	x1 = ((pts[3].x + pts[4].x)>>1);	y1 = ((pts[3].y + pts[4].y)>>1);
	j = x1 - x;		ix = j*j;
	j = y1 - y;		ix += (j*j);		ix = isqr(ix);
	DrawFmtText.SetText(0L, TextDef.text, 0L, 0L, false);
	CursorPos = DrawFmtText.CalcCursorPos(ix, o);
/*	for (i = 1, x1 = 0; TextDef.text[i - 1]; i++) {
		DrawFmtText.oGetTextExtent(o, &x2, &h, i);

		if(x1 <= ix && x2 >= ix) {
			if((ix-x1) < (x2-ix))CursorPos = i-1;
			else CursorPos = i;
			if(CursorPos && (TextDef.text[CursorPos-1]& 0xc0)== 0xc0 &&
				(TextDef.text[CursorPos]& 0xc0) == 0x80) CursorPos--;		//UTF
			return;
			}
		else if(ix < x2){
			CursorPos = i-1;
			if(CursorPos && (TextDef.text[CursorPos-1]& 0xc0)== 0xc0 &&
				(TextDef.text[CursorPos]& 0xc0) == 0x80) CursorPos--;		//UTF
			return;
			}
		if((TextDef.text[i] & 0xc0) == 0xc0) {								//UTF
			while((TextDef.text[i+1] & 0xc0) == 0x80) i++;
			}
		x1 = iround(x2);
		}
	if(pts[3].x < pts[2].x && x < pts[3].x) CursorPos = 0;
	else CursorPos = rlp_strlen(TextDef.text);
*/
}

void 
TextFrame::ShowCursor(anyOutput *o)
{
	double cx, cy, w, h;

	if(!o || (o->OC_type & 0xff) != OC_BITMAP) return;
	TextDef.iSize = o->un2iy(TextDef.fSize);
#ifdef _WINDOWS
	linc = o->un2iy(TextDef.fSize*lspc);
#else
	linc = o->un2iy(TextDef.fSize*lspc*1.2);
#endif
	o->SetTextSpec(&TextDef);
	cy = rDims.top + linc * cur_pos.y;			cx = rDims.left;
	cx += ipad.left;							cy += ipad.top;
	if(cur_pos.y < nlines) {
		if(lines[cur_pos.y] && lines[cur_pos.y][0] && cur_pos.x) {
			fmt_txt.SetText(0L, lines[cur_pos.y], &cx, &cy, false);
			if(!fmt_txt.oGetTextExtent(o, &w, &h, cur_pos.x))return;
			}
		else {
			if(!(o->oGetTextExtent((unsigned char*)"A", 1, &w, &h)))return;	
			w = 0;
			}
		Cursor.left = iround(cx+w);		Cursor.right = iround(cx+w);
		Cursor.top = iround(cy);		Cursor.bottom = iround(cy)+linc;
		ShowTextCursor(o, &Cursor, TextDef.ColTxt);
		}
}

void
TextFrame::CalcCursorPos(int x, int y, anyOutput *o)
{
	int i, iy, ix;
	double x1, x2, h;

	if(!o || !lines || !nlines) return;
	TextDef.iSize = o->un2iy(TextDef.fSize);
	o->SetTextSpec(&TextDef);
	iy = (y-rDims.top-ipad.top)/linc;		x1 = x2 = 0;
	if(iy >= nlines) iy = nlines-1;
	if (iy < 0) iy = 0;
	cur_pos.y = iy;
	x2 = rDims.right - rDims.left -ipad.left - ipad.right;
	x = x - rDims.left -ipad.left;
	fmt_txt.SetText(0L, lines[iy], &x1, &x2, false);
	x1 = 0.0;
	for(i = ix = 0; lines[iy][i]; i++) {
		if(i) fmt_txt.oGetTextExtent(o, &x1, &h, i);
		x1 -= (TextDef.iSize >> 2);
		if(i && x1 <= x2 && x1 >= x) {
			if(i >1) fmt_txt.oGetTextExtent(o, &x1, &h, i-1);
			else x1 = 0;
			fmt_txt.oGetTextExtent(o, &x2, &h, i);
			ix = (fabs(x1-x)<fabs(x2-x)) ? i = (i-1) : i;
			break;
			}
		}
	if(ix) cur_pos.x = ix;
	else cur_pos.x = i;
}

void
Axis::DoMark(anyOutput *o, bool mark)
{
	LineDEF mrkLine;
	double minw = DefSize(SIZE_DATA_LINE);
	POINT *arc;
	long narc = 0;
	RECT clp_rc;

	if(axis->flags & AXIS_ANGULAR) {
		if(mark) {
			if(mo) DelBitmapClass(mo);
			mo = 0L;
			o->GetSize(&mrc);						mo = GetRectBitmap(&mrc, o, false);
			memcpy(&mrkLine, &axline, sizeof(LineDEF));
			mrkLine.width = axline.width < minw ? minw * 3.0 : axline.width *3.0;
			o->SetLine(&mrkLine);
			arc = MakeArc(pts[0].x, pts[0].y, pts[1].x, 0x0f, &narc);
			if(arc) {
				o->oPolyline(arc, (int)narc);
				mrkLine.color ^= 0x00ffffffL;		mrkLine.width = axline.width;
				o->SetLine(&mrkLine);				o->oPolyline(arc, (int)narc);
				free(arc);
				}
			o->UpdateRect(&mrc, true);
			}
		else RestoreRectBitmap(&mo, &mrc, o, false);
		return;
		}
	//disable clipping by graph during marking
	if (o->GetClipRec(&clp_rc)) o->ClipRect(0L);
	else clp_rc.left = clp_rc.right = clp_rc.top = clp_rc.bottom = 0;
	if (mark){
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 3 + o->un2ix(axline.width));
		if (!moveable) {
			mo = GetRectBitmap(&mrc, o, false);
			if (type & 0x04) InvertLine(gradient_box, 5, &axline, &rDims, o, mark);
			else InvertLine(pts, 2, &axline, &rDims, o, mark);
			}
		else {
			if (frm_a) delete frm_a;
			frm_a = new FrmRect(this, 0L, &frm_rc, 0L);
			frm_a->DoMark(o, true);
			o->UpdateRect(&mrc, true);
			}
		}
	else {
		if(mo) RestoreRectBitmap(&mo, &mrc, o, false);
		mo = 0L;
		if (frm_a)	{
			frm_a->DoMark(o, false);
			delete frm_a;
			}
		frm_a = 0L;
		}
	if (clp_rc.left < clp_rc.right && clp_rc.top < clp_rc.bottom) o->ClipRect(&clp_rc);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Graphs are graphic objects containing plots, axes, and drawn objects
bool
Graph::ExecTool(MouseEvent *mev)
{
	static POINT pl = {0, 0}, pc = {0, 0};
	static DWORD color = 0L;
	scaleINFO scale;
	POINT line[5], pt1;
	GraphObj **tmpPlots, *new_go;
	TextDEF *td;
	lfPOINT *lfp, lf;
	long i = 0, j;
	double x, y;
	anyOutput *ScaleOut;

	if(!mev || !CurrDisp) return false;
	td = 0L;	if(mev->Action == MOUSE_LBUP) SuspendAnimation(CurrDisp, false);
	if(ToolMode & TM_MOVE) switch (mev->Action) {
		case MOUSE_LBDOWN:
			if (TrackGO && TrackGO->Id == GO_DRAGHANDLE) {
				pt1.x = iround(TrackGO->GetSize(SIZE_XPOS));
				pt1.y = iround(TrackGO->GetSize(SIZE_YPOS));
				}
			else {
				pt1.x = mev->x;					pt1.y = mev->y;
				}
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pl.x = pc.x = pt1.x;				pl.y = pc.y = pt1.y;
			if (TrackGO && TrackGO->Id != GO_AXIS && TrackGO->Command(CMD_MOUSECURSOR, 0L, CurrDisp)) return true;
			CurrDisp->HideMark();			CurrDisp->MouseCursor(MC_MOVE, false);
			return true;
		case MOUSE_MOVE:
			CurrDisp->MrkMode = MRK_NONE;
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			if (TrackGO && TrackGO->Id != GO_AXIS) {
				if(TrackGO->Id == GO_DRAGHANDLE && TrackGO->type >= DH_18 &&
					TrackGO->type <= DH_88) {
					lf.fx = CurrDisp->fix2un((double)(pt1.x - pl.x));
					lf.fy = CurrDisp->fiy2un((double)(pt1.y - pl.y));
					TrackGO->Track((POINT*)&lf, CurrDisp, false);
					}
				else {
					pc.x = pt1.x - pl.x;				pc.y = pt1.y - pl.y;
					TrackGO->Track(&pc, CurrDisp, false);
					}
				}
			return true;
		case MOUSE_LBUP:
			CurrDisp->MrkMode = MRK_NONE;
			if (TrackGO && TrackGO->Id != GO_AXIS) {
				pt1.x = mev->x;					pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				ToolMode &= ~TM_MOVE;
				pc.x = pt1.x - pl.x;				pc.y = pt1.y - pl.y;
				if(!pc.x && !pc.y){
					Command(CMD_TOOLMODE, (void*)(& ToolMode), CurrDisp);
					return false;
					}
				TrackGO->Track(&pc, CurrDisp, false);
				lf.fx = CurrDisp->fix2un((double)(pt1.x - pl.x));
				lf.fy = CurrDisp->fiy2un((double)(pt1.y - pl.y));
				CurrDisp->MrkMode = MRK_NONE;
				TrackGO->Command(CMD_MOVE, &lf, CurrDisp);
				bModified = true;
				}
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_TOOLMODE, (void*)(& ToolMode), CurrDisp);
			return true;
		default:
			return true;
		}
	if(ToolMode == TM_MARK || ToolMode == TM_ZOOMIN) {
		switch (mev->Action) {
		case MOUSE_LBDOWN:			//we should not receive that !
			rc_mrk.left = mev->x;	rc_mrk.top = mev->y;
			return false;
		case MOUSE_LBUP:
			CurrDisp->MrkMode = MRK_NONE;
			i = rc_mrk.left - mev->x;			j = i*i;
			i = rc_mrk.top - mev->y;			j += (i*i);
			if(j < 20) {						//glitch
				rc_mrk.left = rc_mrk.right = rc_mrk.top = rc_mrk.bottom = 0;
				ToolMode = TM_STANDARD;		CurrDisp->MouseCursor(MC_ARROW, false);
				return false;
				}
			if(ToolMode == TM_ZOOMIN) {
				rc_mrk.right = mev->x;	rc_mrk.bottom = mev->y;
				ToolMode = TM_STANDARD;
				return Command(CMD_ZOOM, (void*) &"+", 0L);
				}
			//fall through
		default:			//come here on MOUSE_MOVE
			if (mev->Action != MOUSE_MOVE) ToolMode = TM_STANDARD;
			if((mev->StateFlags &1) && rc_mrk.left >= 0 && rc_mrk.top >= 0) {
				InvalidateOutput(CurrDisp);
				line[0].x = line[3].x = line[4].x = rc_mrk.left;
				line[0].y = line[1].y = line[4].y = rc_mrk.top;
				line[1].x = line[2].x = mev->x;
				line[2].y = line[3].y = mev->y;
				CurrDisp->ShowLine(line, 5, 0x00b4b4b4L);
				rcUpd.left = rcUpd.right = rc_mrk.left;
				rcUpd.top = rcUpd.bottom = rc_mrk.top;
				UpdateMinMaxRect(&rcUpd, rc_mrk.right = mev->x, rc_mrk.bottom = mev->y);
				IncrementMinMaxRect(&rcUpd,2);
				}
			return true;
			}
		}
	if(ToolMode == TM_PASTE) {
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pl.x = pc.x = pt1.x;		pl.y = pc.y = pt1.y;
			rcDim.left = rcDim.right = rcUpd.left = rcUpd.right = pt1.x;
			rcDim.top = rcDim.bottom = rcUpd.top = rcUpd.bottom = pt1.y;
			return true;
		case MOUSE_MOVE:
			if(mev->StateFlags &1) {
				InvalidateOutput(CurrDisp);
				pt1.x = mev->x;					pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				line[0].x = line[4].x = pl.x;	line[0].y = line[4].y = pl.y;
				line[1].x = pc.x = pt1.x;		line[1].y = pl.y;
				line[2].x = pt1.x;				line[2].y = pc.y = pt1.y;
				line[3].x = pl.x;				line[3].y = pt1.y;
				CurrDisp->ShowLine(line, 5, 0x00b8b8b8L);
				memcpy(&rcUpd, &rcDim, sizeof(RECT));
				UpdateMinMaxRect(&rcUpd, pt1.x, pt1.y);
				IncrementMinMaxRect(&rcUpd, 2);
				return true;
				}
			break;
		case MOUSE_LBUP:
			CurrDisp->MrkMode = MRK_NONE;
			if (!PasteObj) return false;
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pc.x = pt1.x;					pc.y = pt1.y;
			if (pl.x > rDims.right || pl.y > rDims.bottom) {
				//attempt to place object outside the drawing area
				//we probably should issue a warning
				}
			if((lfp = (lfPOINT*)malloc(2 * sizeof(lfPOINT)))){
				lfp[0].fx = CurrDisp->fix2un(pl.x - CurrDisp->getVPorgX());
				lfp[0].fy = CurrDisp->fiy2un(pl.y - CurrDisp->getVPorgY());
				lfp[1].fx = CurrDisp->fix2un(pc.x - CurrDisp->getVPorgX());
				lfp[1].fy = CurrDisp->fiy2un(pc.y - CurrDisp->getVPorgY());
				if(lfp[0].fx > lfp[1].fx) {
					x = lfp[0].fx;		lfp[0].fx = lfp[1].fx;		lfp[1].fx = x;
					}
				if(lfp[0].fy > lfp[1].fy) {
					y = lfp[0].fy;		lfp[0].fy = lfp[1].fy;		lfp[1].fy = y;
					}
				}
			else {
				DeleteGO(PasteObj);			PasteObj = 0L;
				ToolMode = TM_STANDARD;		CurrDisp->MouseCursor(MC_ARROW, false);
				return true;
				}
			scale.sx.fx = lfp[0].fx;	scale.sy.fx = lfp[0].fy;	scale.sz.fx = 0.0;
			scale.sx.fy = scale.sy.fy = scale.sz.fy = 1.0;
			if(PasteObj->Id == GO_GRAPH) {
				if(abs(pc.x -pl.x) > 20 && abs(pc.y -pl.y) >20) {
					x = ((Graph*)PasteObj)->GRect.Xmax - ((Graph*)PasteObj)->GRect.Xmin;
					y = ((Graph*)PasteObj)->GRect.Ymax - ((Graph*)PasteObj)->GRect.Ymin;
					scale.sx.fy = (lfp[1].fx - lfp[0].fx)/x;	scale.sy.fy = (lfp[1].fy - lfp[0].fy)/y;
					scale.sx.fy = (scale.sx.fy + scale.sy.fy) /2.0;		//preserve aspect ratio
					scale.sy.fy = scale.sx.fy;
					}
				((Graph*)PasteObj)->GRect.Xmax -= ((Graph*)PasteObj)->GRect.Xmin;
				((Graph*)PasteObj)->GRect.Ymax -= ((Graph*)PasteObj)->GRect.Ymin;
				((Graph*)PasteObj)->GRect.Ymin = ((Graph*)PasteObj)->GRect.Xmin = 0.0;
				PasteObj->Command(CMD_SCALE, &scale, 0L);
				PasteObj->moveable = 1;
				Command(CMD_DROP_GRAPH, (void*)PasteObj, 0L);
				}
			else {
				ScaleOut = NewBitmapClass(10, 10, CurrDisp->hres, CurrDisp->vres);
				if (ScaleOut) {
					PasteObj->parent = this;	PasteObj->DoPlot(ScaleOut);
					switch(PasteObj->Id){
					case GO_POLYLINE:		case GO_POLYGON:
						IncrementMinMaxRect(&PasteObj->rDims, -(3*CurrDisp->un2ix(((polyline*)PasteObj)->pgLine.width)+3));
						break;
					case GO_BEZIER:
						IncrementMinMaxRect(&PasteObj->rDims, 3);
						break;
						}
					scale.sx.fx = CurrDisp->fix2un(-PasteObj->rDims.left);
					scale.sy.fx = CurrDisp->fiy2un(-PasteObj->rDims.top);
					PasteObj->Command(CMD_SCALE, &scale, 0L);
					scale.sx.fx = lfp[0].fx;	scale.sy.fx = lfp[0].fy;	scale.sz.fx = 0.0;
					if(abs(pc.x -pl.x) > 8 && abs(pc.y -pl.y) > 8) {
						x = CurrDisp->fix2un(PasteObj->rDims.right - PasteObj->rDims.left);
						y = CurrDisp->fiy2un(PasteObj->rDims.bottom - PasteObj->rDims.top);
						scale.sx.fy = (lfp[1].fx - lfp[0].fx)/x;	
						scale.sy.fy = (lfp[1].fy - lfp[0].fy)/y;
						}
					PasteObj->Command(CMD_SCALE, &scale, 0L);
					DelBitmapClass(ScaleOut);
					}
				Command(CMD_DROP_GRAPH, (void*)PasteObj, 0L);
				}
			PasteObj = 0L;
			ToolMode = TM_STANDARD;			CurrDisp->MouseCursor(MC_ARROW, false);
			break;
			}
		return true;
		}
	if(NumPlots && !Plots[NumPlots-1]) {
		Undo.StoreListGO(this, &Plots, &NumPlots, UNDO_CONTINUE);
		for(i = j = 0; i < NumPlots; i++) if(Plots[i]) Plots[j++] = Plots[i];
		NumPlots = j;
		}
	else if (!Plots) {
		Plots = (GraphObj**)calloc(2, sizeof(GraphObj*));
		if (!Plots)return false;
		}
	else if(Plots[NumPlots]){
		tmpPlots = (GraphObj**)memdup(Plots, (NumPlots + 2) * sizeof(GraphObj*), 0L);
		if (tmpPlots){
			Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
			free(Plots);	Plots = tmpPlots;
			Plots[NumPlots] = Plots[NumPlots+1] = 0L;
			}
		else return false;
		}
	switch(ToolMode & 0x0f) {
	case TM_DRAW:
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_REDRAW, 0L, CurrDisp);
			color = defs.Color(COL_POLYLINE);
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pl.x = pc.x = pt1.x;		pl.y = pc.y = pt1.y;
			if(tl_pts) free(tl_pts);
			tl_nPts = 0;
			tl_pts = (POINT*)malloc(4096 * sizeof(POINT));
			if(tl_pts) AddToPolygon(&tl_nPts, tl_pts, &pc);
			return true;
		case MOUSE_LBUP:
			pl.x = mev->x;				pl.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pl, 0L);
			else Command(CMD_SNAP, &pl, 0L);
			if (tl_pts) AddToPolygon(&tl_nPts, tl_pts, &pl);
			// create line object
			if(tl_pts && tl_nPts >1) {			//DEBUG: check for plausibility
				lfp = (lfPOINT*)malloc(tl_nPts * sizeof(lfPOINT));
				if(lfp){
					for(i = 0; i < tl_nPts; i++) {
						lfp[i].fx = CurrDisp->fix2un(tl_pts[i].x - CurrDisp->getVPorgX());
						lfp[i].fy = CurrDisp->fiy2un(tl_pts[i].y - CurrDisp->getVPorgY());
						}
					if(Plots[NumPlots]) i = NumPlots+1;
					else i = NumPlots;
					Undo.SetGO(this, &Plots[i], new Bezier(this, data, lfp, (int)tl_nPts, 0,
						(CurrDisp->hres / CurrDisp->getVPorgScale() + CurrDisp->vres / CurrDisp->getVPorgScale()) / 2.0), 0L);
					if(Plots[i]){
						CurrDisp->HideMark();
						NumPlots = i+1;
						Plots[i]->moveable = 1;
						Command(CMD_REDRAW, 0L, 0L);
						bModified = true;
						}
					free(lfp);
					}
				if(tl_pts) free(tl_pts);
				tl_pts = 0L;	tl_nPts = 0;
				return true;
				}
			if(tl_pts) free(tl_pts);
			tl_pts = 0L;	tl_nPts = 0;
			return false;
		case MOUSE_MOVE:
			if((mev->StateFlags &1) && tl_pts && tl_nPts < 4095) {
				if ((abs(pc.x - mev->x) + abs(pc.y - mev->y)) < 4) return true;
				memcpy(&pl, &pc, sizeof(POINT));
				line[1].x = pc.x = mev->x;		line[1].y = pc.y = mev->y;
				line[0].x = pl.x;				line[0].y = pl.y;
				AddToPolygon(&tl_nPts, tl_pts, &pc);
#ifdef _WINDOWS
				CurrDisp->ShowLine(line, 2, color);
//				CurrDisp->ShowLine(tl_pts, tl_nPts, color);
#else
				CurrDisp->ShowLine(tl_pts, tl_nPts, color);
#endif
				return true;
				}
			break;
			}
		break;
	case TM_POLYLINE:	case TM_POLYGON:
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			if (!tl_pts) {
				CurrGO = 0L;
				CurrDisp->MrkMode = MRK_NONE;
				Command(CMD_REDRAW, 0L, CurrDisp);
				color = defs.Color((ToolMode & 0x0f) == TM_POLYLINE ? COL_POLYLINE : COL_POLYGON);
				pt1.x = mev->x;					pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				pl.x = pc.x = pt1.x;		pl.y = pc.y = pt1.y;
				tl_nPts = 0;
				tl_pts = (POINT*)malloc(4096 * sizeof(POINT));
				if (tl_pts) AddToPolygon(&tl_nPts, tl_pts, &pc);
				rcDim.left = rcDim.right = rcUpd.left = rcUpd.right = pt1.x;
				rcDim.top = rcDim.bottom = rcUpd.top = rcUpd.bottom = pt1.y;
				}
			return true;
		case MOUSE_MOVE:
			if (tl_pts && tl_nPts) {
				InvalidateOutput(CurrDisp);
				if (tl_nPts > 1) CurrDisp->ShowLine(tl_pts, tl_nPts, color, true);
				pt1.x = mev->x;					pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				line[1].x = pt1.x;				line[1].y = pt1.y;
				line[0].x = pc.x;				line[0].y = pc.y;
				CurrDisp->ShowLine(line, 2, color, tl_nPts < 2 ? true : false);
				UpdateMinMaxRect(&rcUpd, pt1.x, pt1.y);
				memcpy(&rcUpd, &rcDim, sizeof(RECT));
				UpdateMinMaxRect(&rcUpd, pt1.x, pt1.y);
				IncrementMinMaxRect(&rcUpd, 2);
				return true;
				}
			break;
		case MOUSE_LBUP:
			if(tl_pts && tl_nPts) {
				memcpy(&pl, &pc, sizeof(POINT));
				pt1.x = mev->x;				pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				pc.x = pt1.x;				pc.y = pt1.y;
				UpdateMinMaxRect(&rcDim, pt1.x, pt1.y);
				AddToPolygon(&tl_nPts, tl_pts, &pc);
				}
			return true;
		default:			// create line or polygon object upon double click
			if(tl_pts && tl_nPts >0) {			//DEBUG: check for plausibility
				pc.x = mev->x;				pc.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pc, 0L);
				else Command(CMD_SNAP, &pc, 0L);
				AddToPolygon(&tl_nPts, tl_pts, &pc);
				lfp = (lfPOINT*)malloc(tl_nPts * sizeof(lfPOINT));
				if(lfp){
					for(i = 0; i < tl_nPts; i++) {
						lfp[i].fx = CurrDisp->fix2un(tl_pts[i].x - CurrDisp->getVPorgX());
						lfp[i].fy = CurrDisp->fiy2un(tl_pts[i].y - CurrDisp->getVPorgY());
						}
					if(Plots[NumPlots]) i = NumPlots+1;
					else i = NumPlots;
					Undo.SetGO(this, &Plots[i], ToolMode == TM_POLYLINE ?
						new polyline(this, data, lfp, (int)tl_nPts) :
						(GraphObj*) new polygon(this, data, lfp, (int)tl_nPts), 0L);
					if(Plots[i]){
						NumPlots = i+1;
						Plots[i]->moveable = 1;
						Command(CMD_REDRAW, 0L, 0L);
						bModified = true;
						}
					free(lfp);
					}
				free(tl_pts);			tl_pts = 0L;		tl_nPts = 0;
				return true;
				}
			}
//		if(mev->x == pc.x && mev->y == pc.y) return true;	//rebounce
		break;
	case TM_RECTANGLE:	case TM_ELLIPSE:	case TM_ROUNDREC:	case TM_ARROW:
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_REDRAW, 0L, CurrDisp);
			color = defs.Color((ToolMode & 0x0f) != TM_ARROW ? COL_POLYGON : COL_DATA_LINE);
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pl.x = pc.x = pt1.x;		pl.y = pc.y = pt1.y;
			rcDim.left = rcDim.right = rcUpd.left = rcUpd.right = mev->x;
			rcDim.top = rcDim.bottom = rcUpd.top = rcUpd.bottom = mev->y;
			return true;
		case MOUSE_MOVE:
			if(mev->StateFlags &1) {
				pt1.x = mev->x;					pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				InvalidateOutput(CurrDisp);
				line[0].x = line[4].x = pl.x;	line[0].y = line[4].y = pl.y;
				if((ToolMode & 0x0f)==TM_ARROW) {
					line[1].x = pc.x = pt1.x;		line[1].y = pc.y = pt1.y;
					CurrDisp->ShowLine(line, 2, color);
					}
				else {
					line[1].x = pc.x = pt1.x;		line[1].y = pl.y;
					line[2].x = pt1.x;				line[2].y = pc.y = pt1.y;
					line[3].x = pl.x;				line[3].y = pt1.y;
					CurrDisp->ShowLine(line, 5, color);
					if((ToolMode & 0x0f) == TM_ELLIPSE) 
						CurrDisp->ShowEllipse(pc, pl, color);
					}
				memcpy(&rcUpd, &rcDim, sizeof(RECT));
				UpdateMinMaxRect(&rcUpd, pt1.x, pt1.y);
				IncrementMinMaxRect(&rcUpd, 2);
				return true;
				}
			break;
		case MOUSE_LBUP:
			CurrDisp->MrkMode = MRK_NONE;
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pc.x = pt1.x;			pc.y = pt1.y;
			if(((ToolMode & 0x0f)==TM_ARROW || (abs(pc.x-pl.x) >5 && abs(pc.y-pl.y) >5)) && 
				(lfp = (lfPOINT*)malloc(2 * sizeof(lfPOINT)))){
				lfp[0].fx = CurrDisp->fix2un(pl.x - CurrDisp->getVPorgX());
				lfp[0].fy = CurrDisp->fiy2un(pl.y - CurrDisp->getVPorgY());
				lfp[1].fx = CurrDisp->fix2un(pc.x - CurrDisp->getVPorgX());
				lfp[1].fy = CurrDisp->fiy2un(pc.y - CurrDisp->getVPorgY());
				if(Plots[NumPlots]) i = NumPlots+1;
				else i = NumPlots;
				if(ToolMode == TM_RECTANGLE) new_go = new rectangle(this, data,
					&lfp[0], &lfp[1]);
				else if(ToolMode == TM_ELLIPSE) new_go = new ellipse(this, data,
					&lfp[0], &lfp[1]);
				else if(ToolMode == TM_ROUNDREC) new_go = new roundrec(this, data,
					&lfp[0], &lfp[1]);
				else if (ToolMode == TM_ARROW && (lfp[0].fx != lfp[1].fx || lfp[0].fy != lfp[1].fy)){
					new_go = new Arrow(this, data, &lfp[0], &lfp[1], ARROW_UNITS | ARROW_LINE);
					new_go->name = rlp_strdup((char*)"arrow");
					}
				else new_go = 0L;
				if(new_go) Undo.SetGO(this, &Plots[i], new_go, 0L);
				if(Plots[i]){
					NumPlots = i+1;				CurrDisp->HideMark();
					Plots[i]->moveable = 1;
					Command(CMD_REDRAW, 0L, 0L);
					bModified = true;
					}
				free(lfp);
				}
			else if(rcUpd.left != rcUpd.right && rcUpd.top != rcUpd.bottom) {
				CurrDisp->UpdateRect(&rcUpd, true);
				}
			}
		break;
	case TM_TEXT:
		if(Plots[NumPlots]) i = NumPlots+1;
		else i = NumPlots;
		switch(mev->Action) {
		case MOUSE_LBDOWN:
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_REDRAW, 0L, CurrDisp);
			color = 0x00cbcbcb;
			pt1.x = mev->x;					pt1.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			pl.x = pc.x = pt1.x;		pl.y = pc.y = pt1.y;
			rcDim.left = rcDim.right = rcUpd.left = rcUpd.right = pt1.x;
			rcDim.top = rcDim.bottom = rcUpd.top = rcUpd.bottom = pt1.y;
			return true;
		case MOUSE_MOVE:
			if(mev->StateFlags &1) {
				CurrDisp->MouseCursor(MC_TXTFRM, false);
				InvalidateOutput(CurrDisp);
				pt1.x = mev->x;					pt1.y = mev->y;
				if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pt1, 0L);
				else Command(CMD_SNAP, &pt1, 0L);
				line[0].x = line[4].x = pl.x;	line[0].y = line[4].y = pl.y;
				if((ToolMode & 0x0f)==TM_ARROW) {
					line[1].x = pc.x = pt1.x;		line[1].y = pc.y = pt1.y;
					CurrDisp->ShowLine(line, 2, color);
					}
				else {
					line[1].x = pc.x = pt1.x;		line[1].y = pl.y;
					line[2].x = pt1.x;				line[2].y = pc.y = pt1.y;
					line[3].x = pl.x;				line[3].y = pt1.y;
					CurrDisp->ShowLine(line, 5, color);
					}
				memcpy(&rcUpd, &rcDim, sizeof(RECT));
				UpdateMinMaxRect(&rcUpd, pt1.x, pt1.y);
				IncrementMinMaxRect(&rcUpd, 2);
				return true;
				}
			break;
		case MOUSE_LBUP:
			CurrDisp->MrkMode = MRK_NONE;
			pc.x = mev->x;			pc.y = mev->y;
			if (Id != GO_PAGE && CurrGraph) CurrGraph->Command(CMD_SNAP, &pc, 0L);
			else Command(CMD_SNAP, &pt1, 0L);
			if (((abs(pc.x - pl.x) >20 && abs(pc.y - pl.y) >20)) &&
				(lfp = (lfPOINT*)malloc(2 * sizeof(lfPOINT)))){
				lfp[0].fx = CurrDisp->fix2un(pl.x - CurrDisp->getVPorgX());
				lfp[0].fy = CurrDisp->fiy2un(pl.y - CurrDisp->getVPorgY());
				lfp[1].fx = CurrDisp->fix2un(pc.x - CurrDisp->getVPorgX());
				lfp[1].fy = CurrDisp->fiy2un(pc.y - CurrDisp->getVPorgY());
				if(Plots[NumPlots]) i = NumPlots+1;
				else i = NumPlots;
				new_go = new TextFrame(this, data, &lfp[0], &lfp[1], 0L);
				new_go->name = rlp_strdup((char*)"textframe");
				if(new_go) Undo.SetGO(this, &Plots[i], new_go, 0L);
				if(Plots[i]){
					NumPlots = i+1;
					Plots[i]->DoPlot(CurrDisp);							//init
					CurrDisp->ShowMark(CurrGO = Plots[i], MRK_GODRAW);				//edit
					Plots[i]->moveable = 1;
					bModified = true;
					}
				free(lfp);
				CurrDisp->CheckMenu(ToolMode & 0x0f, false);
				CurrDisp->CheckMenu(ToolMode = TM_STANDARD, true);
				CurrDisp->MouseCursor(MC_ARROW, false);
				}
			else {
				CurrDisp->MouseCursor(MC_TEXT, false);
				if(!(td = (TextDEF *)calloc(1, sizeof(TextDEF))))return false;
				x = CurrDisp->fix2un(pc.x - CurrDisp->getVPorgX());
				y = CurrDisp->fiy2un(pc.y - CurrDisp->getVPorgY());
				td->ColTxt = defs.Color(COL_TEXT);		td->ColBg = defs.Color(COL_BG);
				td->RotBL = td->RotCHAR = 0.0f;			td->fSize = DefSize(SIZE_TEXT);
				td->Align = TXA_VTOP | TXA_HLEFT;		td->Style = TXS_NORMAL;
				td->Mode = TXM_TRANSPARENT;				td->Font = FONT_HELVETICA;
				td->text = 0L;
				CurrGO = 0L;
				CurrDisp->MrkMode = MRK_NONE;
				Command(CMD_REDRAW, 0L, CurrDisp);
				y -= td->fSize/2.0;
				new_go = new Label(this, data, x, y, td, 0L, 0L);
				new_go->name = rlp_strdup((char*)"text");
				Undo.SetGO(this, &Plots[i], new_go, 0L);
				if(Plots[i]){
					NumPlots = i+1;
					Plots[i]->DoPlot(CurrDisp);							//init
					CurrDisp->ShowMark(CurrGO = Plots[i], MRK_GODRAW);	//edit
					Plots[i]->moveable = 1;
					}
				free(td);
				}
			return true;
			}
		break;
		}
	return false;
}

bool
Graph::MoveObj(int cmd, GraphObj *g)
{
	long i, j;
	GraphObj *g1 = 0;

	if(!g || NumPlots <2 || g->parent != this) return false;
	switch(cmd) {
	case CMD_MOVE_TOP:
		for(i = j = 0; i <NumPlots; i++){
			if(g == Plots[i]) g1 = Plots[i];
			else Plots[j++] = Plots[i];
			}
		if(g1) {
			Plots[j++] = g1;
			return true;
			}
		break;
	case CMD_MOVE_UP:
		for(i = 0; i<NumPlots-1; i++){
			if(g == Plots[i]){
				g1 = Plots[i];	Plots[i] = Plots[i+1];	Plots[i+1] = g1;
				return true;
				}
			}
		break;
	case CMD_MOVE_DOWN:
		for(i = 1; i<NumPlots; i++){
			if(g == Plots[i]){
				g1 = Plots[i];	Plots[i] = Plots[i-1];	Plots[i-1] = g1;
				return true;
				}
			}
		break;
	case CMD_MOVE_BOTTOM:
		if(Plots[0] == g) return false;
		for(i =  j = NumPlots-1; i >= 0; i--) {
			if(g == Plots[i]) g1 = Plots[i];
			else Plots[j--] = Plots[i];
			}
		if(g1) {
			Plots[j--] = g1;
			return true;
			}
		break;
		}
	return false;
}

bool
Graph::DoZoom(char *z)
{
	RECT cw;
	double fac, f1, f2, tmp, x, y;
	ZoomDEF *tz;
	Graph *cg;

	if(!z) return false;
	HideCopyMark(false);
	if(0==strcmp("fit", z)) {
		if(CurrGraph) cg = CurrGraph;
		else cg = this;
		rc_mrk.left = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_LEFT))-4;
		rc_mrk.right = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_RIGHT))+4
			+ iround(CurrDisp->MenuHeight());
		rc_mrk.top = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_TOP))-4 
			- iround(CurrDisp->MenuHeight());
		rc_mrk.bottom = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_BOTTOM))+4;
		if(!(CurrDisp->ActualSize(&cw)))return false;
		f1 = (double)(cw.bottom - cw.top)/(double)(rc_mrk.bottom - rc_mrk.top);
		f2 = ((double)(cw.right - cw.left)/(double)(rc_mrk.right - rc_mrk.left));
		fac = f1 < f2 ? f1 : f2;
		if ((CurrDisp->getVPorgScale() * fac) > 100.0) fac = 100.0 / CurrDisp->getVPorgScale();
		if ((CurrDisp->getVPorgScale() * fac) < 0.05) fac = 0.05 / CurrDisp->getVPorgScale();
		if(fac == 1.0) return false;
		tz = (ZoomDEF*)memdup(zoom_def, sizeof(ZoomDEF)*(zoom_level + 1), 0);
		if(tz){
			if(zoom_def) free(zoom_def);
			zoom_def = tz;
			zoom_def[zoom_level].org.fx = CurrDisp->getVPorgX();
			zoom_def[zoom_level].org.fy = CurrDisp->getVPorgY();
			zoom_def[zoom_level].scale = CurrDisp->getVPorgScale();
			zoom_level++;
			}
		tmp = CurrDisp->getVPorgScale() * fac;
		if (tmp < 0.05) tmp = 0.05;
		if (tmp > 100.0) tmp = 100.0;
		x = -rc_mrk.left * fac;
		y = -rc_mrk.top * fac;
		CurrDisp->setVPorg(x, y, tmp);
		HideTextCursor();		CurrDisp->HideMark();
		Command(CMD_SETSCROLL, 0L, CurrDisp);
		return true;
		}
	else if(0==strcmp("+", z)) {
		if(rc_mrk.left >= 0 && rc_mrk.right >= 0 && rc_mrk.top >= 0 && rc_mrk.bottom >= 0) {
			if(rc_mrk.left > rc_mrk.right) Swap(rc_mrk.left, rc_mrk.right);
			if(rc_mrk.top > rc_mrk.bottom) Swap(rc_mrk.top, rc_mrk.bottom);
			if(5 > (rc_mrk.right - rc_mrk.left) || 5 > (rc_mrk.bottom - rc_mrk.top)) {
				rc_mrk.left = -1; rc_mrk.right = -1; rc_mrk.top = -1; rc_mrk.bottom = -1;
				ToolMode = TM_STANDARD;		Command(CMD_TOOLMODE, &ToolMode, CurrDisp);
				return false;
				}
			if(!(CurrDisp->ActualSize(&cw)))return false;
			fac = (double)(cw.bottom - cw.top)/(double)(rc_mrk.bottom - rc_mrk.top);
			fac += (((double)(cw.right - cw.left)/(double)(rc_mrk.right - rc_mrk.left)));
			fac /= 2.0;
			if ((CurrDisp->getVPorgScale() * fac) > 100.0) fac = 100.0 / CurrDisp->getVPorgScale();
			if ((CurrDisp->getVPorgScale() * fac) < 0.05) fac = 0.05 / CurrDisp->getVPorgScale();
			if(fac == 1.0) return false;
			tz = (ZoomDEF*)memdup(zoom_def, sizeof(ZoomDEF)*(zoom_level + 1), 0);
			if(tz){
				if(zoom_def) free(zoom_def);
				zoom_def = tz;
				zoom_def[zoom_level].org.fx = CurrDisp->getVPorgX();
				zoom_def[zoom_level].org.fy = CurrDisp->getVPorgY();
				zoom_def[zoom_level].scale = CurrDisp->getVPorgScale();
				zoom_level++;
				}
			tmp = CurrDisp->getVPorgScale() * fac;
			if (tmp < 0.05) tmp = 0.05;
			if (tmp > 100.0) tmp = 100.0;
			x = CurrDisp->getVPorgX() * fac - rc_mrk.left * fac;
			y = CurrDisp->getVPorgY() * fac - rc_mrk.top * fac;
			CurrDisp->setVPorg(x, y, tmp);
			HideTextCursor();
			Command(CMD_SETSCROLL, 0L, CurrDisp);
			CurrDisp->MouseCursor(MC_ARROW, false);
			rc_mrk.left = -1; rc_mrk.top = -1; rc_mrk.right = -1; rc_mrk.bottom = -1;
			ToolMode = TM_STANDARD;		Command(CMD_TOOLMODE, &ToolMode, CurrDisp);
			CurrDisp->HideMark();
			return true;
			}
		else {
			ToolMode = TM_ZOOMIN;			CurrDisp->MouseCursor(MC_ZOOM, true);
			}
		}
	else if(0==strcmp("-", z)) {
		HideTextCursor();
		if(zoom_def && zoom_level > 0) {
			zoom_level--;
			if (CurrDisp->getVPorgScale() == zoom_def[zoom_level].scale &&
				CurrDisp->getVPorgX() == zoom_def[zoom_level].org.fx &&
				CurrDisp->getVPorgY() == zoom_def[zoom_level].org.fy) {
				DoZoom(z);
				}
			else {
				CurrDisp->setVPorg(zoom_def[zoom_level].org.fx, zoom_def[zoom_level].org.fy, zoom_def[zoom_level].scale);
				}
			}
		else {
			CurrDisp->setVPorg(0.0, (double)CurrDisp->MenuHeight(), Id == GO_PAGE ? 0.5 : 1.0);
			}
		Command(CMD_SETSCROLL, 0L, CurrDisp);
		return true;
		}
	else if(0==strcmp((char*)"25", z)){
		CurrDisp->setVPorgScale(0.25);		return DoZoom((char*)"org");
		}
	else if(0==strcmp((char*)"50", z)){
		CurrDisp->setVPorgScale(0.50);		return DoZoom((char*)"org");
		}
	else if(0==strcmp((char*)"100", z)){
		CurrDisp->setVPorgScale(1.0);		return DoZoom((char*)"org");
		}
	else if(0==strcmp((char*)"200", z)){
		CurrDisp->setVPorgScale(2.0);		return DoZoom((char*)"org");
		}
	else if(0==strcmp((char*)"400", z)){
		CurrDisp->setVPorgScale(4.0);		return DoZoom((char*)"org");
		}
	else if(0==strcmp((char*)"org", z)){
//		CurrDisp->setVPorg(0.0, (double)CurrDisp->MenuHeight(), Id == GO_PAGE ? 0.5 : 1.0);
		CurrDisp->setVPorg(0.0, (double)CurrDisp->MenuHeight(), CurrDisp->getVPorgScale());
		HideTextCursor();
		Command(CMD_SETSCROLL, 0L, CurrDisp);
		if (zoom_def && zoom_level > 0) free(zoom_def);
		zoom_def = 0L;			zoom_level = 0;
		}
	else if(0==strcmp((char*)"reset", z)){
		if(zoom_def && zoom_level > 0) free(zoom_def);
		zoom_def = 0L;			zoom_level = 0;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a tree structure of all interesting objects
//   This object is used e.g. for layer control
ObjTree::ObjTree(GraphObj *par, DataObj *d, GraphObj *root):GraphObj(par, d)
{
	Id = GO_OBJTREE;					base = root;		list=0L;
	TextDef.ColTxt = 0x00000000L;		TextDef.ColBg = 0x00ffffffL;
	TextDef.fSize = 4.0;				TextDef.RotBL = TextDef.RotCHAR = 0.0;
	TextDef.iSize = 0;					TextDef.Align = TXA_HLEFT | TXA_VTOP;
	TextDef.Mode = TXM_TRANSPARENT;		TextDef.Style = TXS_NORMAL;
	TextDef.Font = FONT_HELVETICA;		TextDef.text = 0L;
	Command(CMD_LAYERS, 0L, 0L);
}

ObjTree::~ObjTree()
{
	if(list) free(list);
	list = 0L;
}

void
ObjTree::DoPlot(anyOutput *o)
{
	int i, n, ix, iy, lin_space;
	GraphObj *curr_obj;

	if(!o || !list) return;
	lin_space = o->un2ix(TextDef.fSize)+1;
	o->Erase(0x00ffffffL);
	ix = 10;	iy = 0;
	for(i = 0; i < count; i++, iy += lin_space) {
		if(list[i]) {
			curr_obj = list[i];			n = 0;
			if(i) while(curr_obj && curr_obj != list[0]) {
				if(curr_obj != list[0]) n += rlp_strcpy(TmpTxt+n, TMP_TXT_SIZE -n, (char*)" -");
				if(curr_obj) curr_obj = curr_obj->parent; 
				}
			if(n) TmpTxt[n++] = ' ';
			n += rlp_strcpy(TmpTxt+n, TMP_TXT_SIZE -n, get_name(i));
			if(list[i]->Id >= GO_PLOT && list[i]->Id < GO_GRAPH) {
				TextDef.ColTxt = ((Plot*)list[i])->hidden ? 0x00000080L : 0x00008000L;
				}
			else if (list[i]->Id == GO_AXIS) {
				TextDef.ColTxt = ((Axis*)list[i])->hidden ? 0x00000080L : 0x00008000L;
				}
			else {
				switch (list[i]->Id){
				case GO_RECTANGLE:		case GO_ELLIPSE:		case GO_ROUNDREC:
				case GO_BEZIER:			case GO_POLYLINE:		case GO_POLYGON:
				case GO_ARROW:			case GO_LABEL:			case GO_TEXTFRAME:
					TextDef.ColTxt = ((GraphObj*)list[i])->hidden ? 0x00000080L : 0x00008000L;
					break;
				default:
					TextDef.ColTxt = 0x00000000L;
					break;
					}
				}
			o->SetTextSpec(&TextDef);
			o->oTextOut(ix, iy, (unsigned char*)TmpTxt, 0);
			}
		}
}

bool
ObjTree::Command(int cmd, void *tmpl, anyOutput *)
{
	switch(cmd){
	case CMD_LAYERS:
		if(list) free(list);
		count = 0;		maxcount = 100;
		if(base) {
			list = (GraphObj**)calloc(maxcount, sizeof(GraphObj*));
			if(list) {
				list[count++] = base;
				}
			base->Command(CMD_OBJTREE, this, 0L);
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_OBJTREE;
		return true;
	case CMD_UPDATE:
		if (tmpl) {
			if(count >= maxcount) {
				maxcount += 100;
				list = (GraphObj**)realloc(list, maxcount * sizeof(GraphObj*));
				}
			if (list) {
				list[count++] = (GraphObj*)tmpl;
				list[count] = 0L;
				}
			}
		return true;
	case CMD_TEXTDEF:
		if(tmpl) memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = 0L;
		return true;
		}
	return false;
}

anyOutput *
ObjTree::CreateBitmap(int *bw, int *bh, anyOutput *tmpl)
{
	anyOutput *bmp = 0L;
	int h;
	
	h = (tmpl->un2iy(TextDef.fSize) + 1) * (count+1);
	if(h > *bh) *bh = h;
	bmp = NewBitmapClass(*bw, *bh, tmpl->hres, tmpl->vres);
	if(bmp) DoPlot(bmp);
	return bmp;
}

char *
ObjTree::get_name(int li)
{
	static char *text = (char*)"text";

	if(li < count && list[li] && list[li]->name) return list[li]->name;
	else if (list[li]->Id == GO_LABEL) return text;
	return (char*)"(unknown)";
}

int
ObjTree::get_vis(int li)
{
	if (li < count && list[li] && (list[li]->Id >= GO_PLOT && list[li]->Id < GO_GRAPH))
		return ((Plot*)list[li])->hidden ? 0 : 1;
	else if (li < count && list[li]) 	return list[li]->hidden ? 0 : 1;
	return 2;
}

bool
ObjTree::set_vis(int li, bool vis)
{
	if(li < count && list[li] && list[li]->Id >= GO_PLOT && list[li]->Id < GO_GRAPH) {
		((Plot*)list[li])->hidden = vis ? 0 : 1;
		list[li]->Command(CMD_MRK_DIRTY, 0L, 0L);
		list[li]->Command(CMD_REDRAW, 0L, 0L);
		}
	else if (li < count && list[li] && list[li]->Id == GO_AXIS) {
		((Axis*)list[li])->hidden = vis ? 0 : 1;
		list[li]->Command(CMD_REDRAW, 0L, 0L);
		}
	else if (li < count && list[li]){
		list[li]->hidden = vis ? 0 : 1;
		if (list[li]->parent) (list[li]->parent)->Command(CMD_REDRAW, 0L, 0L);
		}
	return false;
}

bool 
ObjTree::set_movable(int li, bool bMov)
{
	if (li < count && list[li] && list[li]->Id == GO_AXIS) {
		((Axis*)list[li])->moveable = bMov ? 1 : 0;
		}
	return false;
}

GraphObj*
ObjTree::get_obj(int li)
{
	if(li < count && list[li]) return list[li];
	return 0L;
}

