//RLPlot.h, Copyright (c) 2000-2025 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version F2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#ifndef _RLPLOT_H
#define _RLPLOT_H
#define NUM_UNITS  3
#define TMP_TXT_SIZE 4096
#define MAXDRIVE	8
#define MAXDIR		256
#define MAXFILE		512
#define MAXEXT		16
#define MAXPATH		1024

#include <stdio.h>
#include <string.h>
#include "Version.h"

#define Swap(a,b) {a^=b;b^=a;a^=b;}
#define IsInRect(rc, x, y) (x > rc.left && x < rc.right && y > rc.top && y < rc.bottom)
inline long iround(double a) {return a < 0.0 ?(long)(a-0.499) : (long)(a+0.499);}

#define _PI	3.1415926535897932384626433832795028841971693993751
#define _SQRT2	1.4142135623730950488016887242096980785696718753769

#ifdef _WIN64
#ifndef _WINDOWS 
#define _WINDOWS
#endif
#ifndef DWORD
#define DWORD COLORREF			//unsigned int32
#endif
#endif

#ifdef _WIN32
#ifndef _WINDOWS 
#define _WINDOWS
#endif
#ifndef DWORD
#define DWORD COLORREF			//unsigned int32
#endif
#endif

#ifdef _WINDOWS					//platform is windows
#include <windows.h>
#if _MSC_VER >= 1400
#define USE_WIN_SECURE
#endif
#define w_char WCHAR
#define _TXH	4.0				//graph text default size in mm

#else							//platform is *nix
#include <sys/types.h>
#define DWORD u_int32_t
#define CF_TEXT	1				//defined in WinUser.h
#define CF_SYLK 4
#define w_char wchar_t
#define _TXH	3.0				//graph text default size in mm

typedef struct tagPOINT { // pt 
    long x; 
    long y; 
} POINT; 

typedef struct _RECT {    // rc 
    long left; 
    long top; 
    long right; 
    long bottom; 
} RECT; 

#endif	//_WINDOWS

inline void UpdateMinMaxRect(RECT *rc, int x, int y){
	rc->left = (x < rc->left) ? x : rc->left;		rc->right = (x > rc->right) ? x : rc->right;
	rc->top = (y < rc->top) ? y : rc->top;			rc->bottom = (y > rc->bottom) ? y : rc->bottom;
}

enum {SIZE_MINE, SIZE_SYMBOL, SIZE_SYM_LINE, SIZE_DATA_LINE, SIZE_TEXT,
	SIZE_GRECT_TOP, SIZE_GRECT_BOTTOM, SIZE_GRECT_LEFT, SIZE_GRECT_RIGHT,
	SIZE_DRECT_LEFT, SIZE_DRECT_RIGHT, SIZE_DRECT_TOP, SIZE_DRECT_BOTTOM,
	SIZE_DATA_LINE_PAT, SIZE_XPOS, SIZE_XPOS_LAST = SIZE_XPOS+200, 
	SIZE_YPOS, SIZE_YPOS_LAST = SIZE_YPOS+200, SIZE_ZPOS, 
	SIZE_ZPOS_LAST = SIZE_ZPOS+200, SIZE_BOUNDS_XMIN,
	SIZE_BOUNDS_XMAX, SIZE_BOUNDS_YMIN, SIZE_BOUNDS_YMAX, SIZE_BOUNDS_ZMIN,
	SIZE_BOUNDS_ZMAX, SIZE_BOUNDS_LEFT, SIZE_BAR_BASE,
	SIZE_BOUNDS_RIGHT, SIZE_BOUNDS_TOP, SIZE_BOUNDS_BOTTOM, SIZE_XAXISY,
	SIZE_YAXISX, SIZE_AXIS_LINE, SIZE_PATLENGTH, SIZE_BAR_DEPTH,
	SIZE_AXIS_TICKS, SIZE_TICK_LABELS, SIZE_ERRBAR, SIZE_ERRBAR_LINE, SIZE_ERRBAR_VALUE,
	SIZE_BAR_LINE, SIZE_BAR, SIZE_XBASE, SIZE_YBASE, SIZE_ZBASE, SIZE_BUBBLE_LINE,
	SIZE_BUBBLE_HATCH_LINE, SIZE_BARMINX, SIZE_BARMINY, SIZE_ARROW_LINE,
	SIZE_ARROW_CAPWIDTH, SIZE_ARROW_CAPLENGTH, SIZE_HAIRLINE, SIZE_WHISKER,
	SIZE_WHISKER_LINE, SIZE_BOX_LINE, SIZE_BOXMINX, SIZE_BOXMINY, SIZE_BOX,
	SIZE_RADIUS1, SIZE_RADIUS2, SIZE_SEGLINE, SIZE_LB_XPOS, SIZE_LB_YPOS,
	SIZE_LB_XDIST, SIZE_LB_YDIST, SIZE_TLB_XDIST, SIZE_TLB_YDIST, SIZE_ANGLE1,
	SIZE_ANGLE2, SIZE_XCENTER, SIZE_YCENTER, SIZE_ZCENTER, SIZE_CELLWIDTH, SIZE_CELLTEXT,
	SIZE_A, SIZE_B, SIZE_MX, SIZE_MY, SIZE_MIN_Z, SIZE_MAX_Z, SIZE_MIN_X, SIZE_MAX_X,
	SIZE_MIN_Y, SIZE_MAX_Y, SIZE_TICK_ANGLE, SIZE_RRECT_RAD, SIZE_XCENT, SIZE_YCENT,
	SIZE_ZCENT, SIZE_DRADIUS, SIZE_CURSORPOS, SIZE_CURSOR_XMIN, SIZE_CURSOR_YMIN,
	SIZE_CURSOR_XMAX, SIZE_CURSOR_YMAX, SIZE_XSTEP, SIZE_LSPC, SIZE_SCALE, SIZE_ANG_DISP,
	SIZE_MVINC, SIZE_BAR_DX};
enum {COL_SYM_LINE, COL_SYM_FILL, COL_DATA_LINE, COL_TEXT, COL_BG,
	COL_AXIS, COL_BAR_LINE, COL_BAR_FILL, COL_ERROR_LINE, COL_BUBBLE_LINE, 
	COL_BUBBLE_FILLLINE, COL_BUBBLE_FILL, COL_ARROW, COL_WHISKER, COL_BOX_LINE,
	COL_DRECT, COL_GRECT, COL_GRECTLINE, COL_POLYLINE, COL_POLYGON};
enum {MRK_NONE, MRK_INVERT, MRK_GODRAW, MRK_SSB_DRAW};
enum {
	CMD_NONE, CMD_ADDCHAR, CMD_ADDCHARW, CMD_SETFOCUS, CMD_KILLFOCUS, CMD_DELETE,
	CMD_BACKSP, CMD_CURRLEFT, CMD_CURRIGHT, CMD_CURRUP, CMD_CURRDOWN,
	CMD_SHIFTLEFT, CMD_SHIFTRIGHT, CMD_SHIFTUP, CMD_SHIFTDOWN,
	CMD_NEWGRAPH, CMD_DELGRAPH, CMD_DELOBJ, CMD_DROP_PLOT, CMD_DROP_GRAPH, CMD_OPEN,
	CMD_SAVEAS, CMD_ADDROWCOL, CMD_MOUSE_EVENT, CMD_REDRAW, CMD_DOPLOT,
	CMD_LBUP, CMD_ENDDIALOG, CMD_RADIOBUTT, CMD_ADDCHILD, CMD_BAR_TYPE,
	CMD_BAR_FILL, CMD_SET_AXDEF, CMD_SET_DATAOBJ, CMD_SETTEXT, CMD_GETTEXT,
	CMD_SETTEXTDEF, CMD_GETTEXTDEF, CMD_ADDPLOT, CMD_SYM_TYPE, CMD_SYMTEXT,
	CMD_SYMTEXTDEF, CMD_RANGETEXT, CMD_SYM_RANGETEXT, CMD_FOCTXT, CMD_CONTINUE,
	CMD_ERR_TYPE, CMD_ARROW_ORG, CMD_ARROW_TYPE, CMD_FLUSH, CMD_BOX_FILL,
	CMD_TABDLG, CMD_NOTABDLG, CMD_TAB, CMD_SHTAB, CMD_BOX_TYPE, CMD_BUBBLE_TYPE,
	CMD_BUBBLE_ATTRIB, CMD_BUBBLE_FILL, CMD_BUBBLE_LINE, CMD_DL_LINE,
	CMD_DL_TYPE, CMD_SEG_FILL, CMD_SEG_LINE, CMD_SELECT, CMD_MOVE, CMD_CUT,
	CMD_SETSCROLL, CMD_SETHPOS, CMD_SETVPOS, CMD_PG_FILL, CMD_BOUNDS,
	CMD_SHIFT_OUT, CMD_CAN_CLOSE, CMD_RESET_LINE, CMD_SET_TICKSTYLE,
	CMD_GET_GRIDLINE, CMD_SET_GRIDLINE, CMD_SET_GRIDTYPE, CMD_TLB_TXTDEF,
	CMD_DROP_LABEL, CMD_DROP_OBJECT, CMD_PAGEUP, CMD_PAGEDOWN, CMD_AUTOSCALE,
	CMD_MRK_DIRTY, CMD_SETNAME, CMD_TOOLMODE, CMD_DROPFILE, CMD_AXIS, CMD_INIT,
	CMD_GET_CELLDIMS, CMD_SET_CELLDIMS, CMD_TEXTSIZE, CMD_PASTE_TSV, CMD_PASTE_XML, CMD_PASTE_SYLK,
	CMD_PASTE_SSV, CMD_PASTE_CSV, CMD_COPY_TSV, CMD_COPY_XML, CMD_COPY_SYLK, CMD_QUERY_COPY,
	CMD_MOUSECURSOR, CMD_NEWPAGE, CMD_MINRC, CMD_MAXRC, CMD_SETCHILD, CMD_SYM_FILL,
	CMD_LINEUP, CMD_LINEDOWN, CMD_CONFIG, CMD_FINDTEXT, CMD_MOVE_TOP, CMD_MOVE_UP,
	CMD_MOVE_DOWN, CMD_MOVE_BOTTOM, CMD_UPDATE, CMD_CURRPOS, CMD_POS_FIRST, CMD_POS_LAST,
	CMD_ADDAXIS, CMD_REG_AXISPLOT, CMD_USEAXIS, CMD_SET_GO3D, CMD_UNDO, CMD_DELOBJ_CONT,
	CMD_RMU, CMD_REPL_GO, CMD_UNDO_MOVE, CMD_SAVEPOS, CMD_WHISKER_STYLE, CMD_TICK_TYPE,
	CMD_ZOOM, CMD_CLIP, CMD_STARTLINE, CMD_ADDTOLINE, CMD_REQ_POINT, CMD_DRAW_LATER,
	CMD_SEG_MOVEABLE, CMD_ARROW_ORG3D, CMD_MUTATE, CMD_PRINT, CMD_UPDHISTORY, CMD_ALLTEXT,
	CMD_SET_LINE, CMD_SAVE_SYMBOLS, CMD_SAVE_TICKS, CMD_SAVE_BARS, CMD_SAVE_BARS_CONT,
	CMD_SAVE_ERRS, CMD_SAVE_ARROWS, CMD_SAVE_DROPLINES, CMD_SAVE_LABELS, CMD_UNLOCK, CMD_SYMTEXT_UNDO,
	CMD_FILLRANGE, CMD_BUSY, CMD_ERROR, CMD_CLEAR_ERROR, CMD_SETPARAM, CMD_SETFUNC,
	CMD_LEGEND, CMD_FILENAME, CMD_LAYERS, CMD_OBJTREE, CMD_TEXTDEF,
	CMD_HASSTACK, CMD_WRITE_GRAPHS, CMD_SETFONT, CMD_SETSTYLE, CMD_COPY, CMD_PASTE,
	CMD_INSROW, CMD_INSCOL, CMD_DELROW, CMD_DELCOL, CMD_ADDTXT, CMD_ETRACC, CMD_SHPGUP,
	CMD_SHPGDOWN, CMD_ERRDESC, CMD_SAVE, CMD_GETMARK, CMD_PASTE_OBJ, CMD_COL_MOUSE,
	CMD_MARKOBJ, CMD_SCALE, CMD_GETFILENAME, CMD_TEXTTHERE, CMD_HIDEMARK,
	CMD_RECALC, CMD_DRAWPG, CMD_UPDPG, CMD_MINMAX, CMD_STEP, CMD_SIGNAL_POL,
	CMD_SCROLL_LEFT, CMD_SCROLL_RIGHT, CMD_SCROLL_UP, CMD_SCROLL_DOWN, CMD_USE_FORMAT, CMD_SET_COND,
	CMD_ARITH, CMD_TIMER, CMD_SCHED_PG, CMD_MARKALL, CMD_CLEAR, CMD_PREVSYM, CMD_CPY_DATA,
	CMD_ADDTXT_PLUS, CMD_MOUSE_REPEAT, CMD_SUSPEND, CMD_REG_MITEM, CMD_SNAP, CMD_DOSNAP, CMD_SAVE_PG};
enum {SYM_CIRCLE, SYM_CIRCLEF, SYM_RECT, SYM_RECTF, SYM_TRIAU, SYM_TRIAUF,
	SYM_TRIAD, SYM_TRIADF, SYM_DIAMOND, SYM_DIAMONDF, SYM_PLUS, SYM_CROSS,
	SYM_STAR, SYM_HLINE, SYM_VLINE, SYM_CIRCLEC, SYM_RECTC, SYM_TRIAUC, SYM_TRIAUL, SYM_TRIAUR,
	SYM_TRIADC, SYM_TRIADL, SYM_TRIADR, SYM_DIAMONDC, SYM_4STAR, SYM_4STARF, SYM_5GON, SYM_5GONF, 
	SYM_5GONC, SYM_5STAR, SYM_5STARF, SYM_6STAR, SYM_6STARF, SYM_1QUAD, SYM_2QUAD, SYM_3QUAD,
	SYM_TEXT = 0x004f, SYM_POS_PARENT = 0x1000};
//types of graphic objects: stored in Id and used for proper destruction of objects
//  and retrieving information.
enum {
	GO_UNKNOWN, GO_AXIS, GO_TICK, GO_GRIDLINE, GO_SYMBOL, GO_SYMTERN, GO_BUBBLE, GO_BAR,
	GO_ERRBAR, GO_ARROW, GO_BOX, GO_LABEL, GO_MLABEL, GO_WHISKER, GO_DROPLINE,
	GO_DATALINE, GO_DATAPOLYGON, GO_ERRORPOLYGON, GO_REGLINE, GO_SDELLIPSE, GO_SEGMENT,
	GO_POLYLINE, GO_POLYGON, GO_RECTANGLE, GO_ELLIPSE, GO_ROUNDREC,
	GO_DRAGHANDLE, GO_DRAGRECT, GO_DRAG3D, GO_FRAMERECT, GO_SPHERE,	GO_PLANE, 
	GO_BRICK, GO_LINESEG, GO_LINE3D, GO_POLYGON3D, GO_GRIDLINE3D, GO_GRIDRADIAL,
	GO_SPHSCANL, GO_DROPL3D, GO_ARROW3D, GO_PLANE3D, GO_LEGITEM, GO_LEGEND,
	GO_OBJTREE, GO_BEZIER, GO_TEXTFRAME, GO_ERRBAR3D, GO_ARRTERN, GO_GRID,
	GO_PLOT = 0x100, GO_PLOTSCATT, GO_REGRESSION, GO_BARCHART, GO_BUBBLEPLOT,
	GO_BOXPLOT, GO_DENSDISP, GO_STACKBAR, GO_STACKPG, GO_WATERFALL, GO_POLARPLOT,
	GO_PIECHART, GO_RINGCHART, GO_GROUP, GO_STARCHART, GO_SCATT3D, GO_PLOT3D,
	GO_RIBBON, GO_LIMITS, GO_FUNCTION, GO_FITFUNC, GO_FREQDIST, GO_GRID3D, GO_FUNC3D,
	GO_XYSTAT, GO_FITFUNC3D, GO_NORMQUANT, GO_CONTOUR, GO_TERNARY, GO_TERNARYXYZ, 
	GO_MYREGR, GO_PHENOLOG, GO_TERNLINE, GO_MYBARS,
	GO_GRAPH = 0x200, GO_PAGE, GO_SPREADDATA = 0x300, GO_DEFRW
#ifndef _WINDDOWS 
	,GO_SCROLLBAR = 0x400
#endif
};

enum { OC_UNKNOWN, OC_BITMAP, OC_GRIDVIEW, OC_HATCH, OC_PRINT, OC_EXPORT};
enum {
	FILL_NONE, FILL_HLINES, FILL_VLINES, FILL_HVCROSS, FILL_DLINEU, FILL_DLINED,
	FILL_DCROSS, FILL_STIPPLE1, FILL_STIPPLE2, FILL_STIPPLE3, FILL_STIPPLE4, 
	FILL_STIPPLE5, FILL_ZIGZAG, FILL_COMBS, FILL_BRICKH, FILL_BRICKV, FILL_BRICKDU, 
	FILL_BRICKDD, FILL_TEXTURE1, FILL_TEXTURE2, FILL_WAVES1, FILL_SCALES, FILL_SHINGLES, 
	FILL_WAVES2, FILL_HERRING, FILL_CIRCLES, FILL_GRASS, FILL_FOAM, FILL_RECS, 
	FILL_HASH, FILL_WATER, FILL_CHESS, NUM_FILLS, FILL_LIGHT3D = 0x100, FILL_GRADIENT = 0x200,
	FILL_RAINBOW = 0x400, FILL_KEEP_ALIVE = 0x800};
enum {ERRBAR_VSYM, ERRBAR_VUP, ERRBAR_VDOWN, ERRBAR_HSYM, ERRBAR_HLEFT,
	ERRBAR_HRIGHT};
enum {BAR_NONE, BAR_VERTB, BAR_VERTT, BAR_VERTU, BAR_HORL, BAR_HORR, BAR_HORU, 
	BAR_RELWIDTH = 0x100, BAR_CENTERED = 0x200, BAR_WIDTHDATA = 0x400, BAR_ALIGN_LEFT = 0x1000, BAR_ALIGN_RIGHT = 0x2000};
enum {TM_STANDARD, TM_DRAW, TM_POLYLINE, TM_POLYGON, TM_RECTANGLE, TM_ELLIPSE,
	TM_ROUNDREC, TM_ARROW, TM_TEXT, TM_MARK, TM_ZOOMIN, TM_MOVE = 0x100, 
	TM_PASTE=0x200};
enum {MC_LAST, MC_ARROW, MC_CROSS, MC_TEXT, MC_WAIT, MC_MOVE, MC_NORTH,
	MC_NE, MC_EAST, MC_SE, MC_SALL, MC_ZOOM, MC_PASTE, MC_DRAWPEN, MC_DRAWREC, 
	MC_DRAWRREC, MC_DRAWELLY, MC_TXTFRM, MC_COLWIDTH, MC_ETRACC};
enum {FILE_ERROR, FILE_READ, FILE_WRITE, INIT_VARS, SAVE_VARS};
enum {ARROW_NOCAP, ARROW_LINE, ARROW_TRIANGLE, ARROW_UNITS = 0x100, ARROW_SYMSPACE = 0x200};
enum {MENU_NONE, MENU_SPREAD, MENU_GRAPH, MENU_PAGE};
enum {GT_UNKNOWN, GT_STANDARD, GT_CIRCCHART, GT_POLARPLOT, GT_3D, GT_TERNARY, GT_TERNARYXYZ};
enum {ICO_NONE, ICO_INFO, ICO_ERROR, ICO_RLPLOT, ICO_QT};
enum {FF_UNKNOWN, FF_CSV, FF_TSV, FF_XML, FF_SYLK, FF_RLP, FF_SVG, FF_EPS,
	FF_RLW, FF_SSV};
enum {LB_X_DATA = 0x01, LB_X_PARENT = 0x02, LB_Y_DATA = 0x10, LB_Y_PARENT = 0x20};
enum {BUBBLE_CIRCLE = 0x000, BUBBLE_SQUARE = 0x001, BUBBLE_UPTRIA = 0x002,
	BUBBLE_DOWNTRIA = 0x003, BUBBLE_UNITS = 0x000, BUBBLE_XAXIS = 0x010,
	BUBBLE_YAXIS = 0x020, BUBBLE_DIAMET = 0x000, BUBBLE_CIRCUM = 0x100,
	BUBBLE_AREA = 0x200};
enum {DH_UNKNOWN, DH_12, DH_22, DH_19, DH_29, DH_39, DH_49, DH_59, DH_69, DH_79,
	DH_89, DH_99, DH_18, DH_28, DH_38, DH_48, DH_58, DH_68, DH_78, DH_88, DH_DATA};
enum {FE_NONE = 0x1000, FE_PARENT, FE_PLOT, FE_FLUSH, FE_DELOBJ, FE_REPLGO, FE_MUTATE};

//drop line styles
#define DL_LEFT          0x001
#define DL_RIGHT         0x002
#define DL_YAXIS         0x004
#define DL_TOP           0x010
#define DL_BOTTOM        0x020
#define DL_XAXIS         0x040
#define DL_CIRCULAR      0x100

typedef struct {
	int num;
	char* display;
	double convert;			//multiply to get mm
	}tag_Units;

typedef struct {
	long x, y, z;
	}POINT3D;

typedef struct {
	double Xmin;
	double Ymax;
	double Xmax;
	double Ymin;
	}fRECT;

typedef struct {
	double fx;
	double fy;
	double fz;
	}fPOINT3D;

typedef struct {
	double fx;
	double fy;
	}lfPOINT;

typedef struct {
	lfPOINT sx, sy, sz;
	}scaleINFO;

typedef struct {
	double finc, fp;
	}RunLinePat;				//used for line patterns

typedef struct {
	double width, patlength;
	DWORD color, pattern;
	}LineDEF;

typedef struct {
	int type;					//pattern
	DWORD color;
	double scale;
	LineDEF *hatch;
	DWORD color2;
	}FillDEF;

typedef struct {
	lfPOINT org;				//zoom origin
	double scale;				//zoom factor
	}ZoomDEF;

typedef struct rlp_datetime {
	int aday, year, doy, month, dom, dow, hours, minutes;
	double seconds;
}rlp_datetime;

// Axis definitions are stored in the following structure
// not to be confused with the Axis class grapic object
// bits stored in flags havethe following meaning
#define AXIS_NOTICKS      0x0			// no ticks at all
#define AXIS_POSTICKS     0x1			// positive direction of ticks
#define AXIS_NEGTICKS     0x2			// negative direction 
#define AXIS_SYMTICKS     0x3			// ticks are symmetrical
#define AXIS_GRIDLINE     0x4			// ticks control a gridline
#define AXIS_MINORTICK    0x8			// its a small tick only
#define AXIS_USER         0x00			// axis placement by user
#define AXIS_LEFT         0x10			// left of plot
#define AXIS_RIGHT        0x20			// right  -"-
#define AXIS_TOP          0x30			// top    -"-
#define AXIS_BOTTOM       0x40			// bottom -"-
#define AXIS_AUTOSCALE    0x80			// rescale upon editing
#define AXIS_INVERT       0x100			// axis top->bottom, right to left
#define AXIS_AUTOTICK     0x200			// select tick s automatically
#define AXIS_DEFRECT      0x400			// use axis to define drawing rectangle
#define AXIS_LINEAR       0x0000		// transforms ...
#define AXIS_LOG          0x1000
#define AXIS_RECI         0x2000
#define AXIS_SQR          0x3000
#define AXIS_DATETIME     0x4000		// its a date- or time axis
#define AXIS_X_DATA       0x10000		// loc.x is data coordinates
#define AXIS_Y_DATA       0x20000       // loc.y is data coordinates
#define AXIS_Z_DATA       0x40000       // loc.z is data coordinates
#define AXIS_ANGULAR      0x100000      // angular (circular) axis
#define AXIS_RADIAL       0x200000		// radial axis
#define AXIS_3D           0x400000		// three dimensional axis
#define AXIS_TRIA         0x800000		// triangular plot axis

typedef struct {
	void *owner;				//owners are usually graph, output or axis classes
	DWORD flags;
	double min, max;
	fPOINT3D loc[2];			//placement of axis coordinates
	double Start, Step;			//used for linear axis
	lfPOINT Center;				//of polar plot
	double Radius;				//   -"-
	int nBreaks;				//axis break ...
	lfPOINT *breaks;
	}AxisDEF;

//Attributes for text properties
//Text fonts
enum {FONT_HELVETICA, FONT_TIMES, FONT_COURIER, FONT_GREEK};
//Text styles
#define TXA_VTOP        0
#define TXA_VCENTER     4
#define TXA_VBOTTOM     8
#define TXA_HLEFT       0
#define TXA_HCENTER     1
#define TXA_HRIGHT      2
#define TXM_OPAQUE      0
#define TXM_TRANSPARENT 1
#define TXS_NORMAL      0
#define TXS_ITALIC      1
#define TXS_BOLD        2
#define TXS_UNDERLINE   4
#define TXS_SUPER       8
#define TXS_SUB			16
typedef struct {
	DWORD ColTxt, ColBg;				//colors ..
	double fSize;						//Text size in units
	double RotBL, RotCHAR;				//Rotation in degrees
	int iSize;							//Text size is given in iSize as pix
	int Align, Mode, Style;				//Text Alignement 0   1   2
										//                4   5   6
										//                8   9  10
										//Mode 0 = opaque, 1 = transparent
	int Font;
	unsigned char *text;
	}TextDEF;

// Store mouse events in the following structure
// Action defines the type of event:
enum {MOUSE_NONE, MOUSE_LBDOWN, MOUSE_LBUP, MOUSE_LBDOUBLECLICK, MOUSE_MBDOWN, MOUSE_MBUP, 
	MOUSE_MBDOUBLECLICK, MOUSE_RBDOWN, MOUSE_RBUP, MOUSE_RBDOUBLECLICK,
	MOUSE_MOVE};
typedef struct {
	unsigned short StateFlags;	//  1 Mouse left button down
								//  2       middle button down
								//  4       right button down
								//  8       shift pressed
								// 16       control pressed
	unsigned short Action;
	int x, y;
	} MouseEvent;

//use this structure if type of data is not known
typedef struct {
	int type;
	double value;
	char *text;
	double *a_data;
	long a_count;
	} anyResult;

//the AccRange class allows to access data objects with a 'a1:b1' style
class AccRange{
public:
	AccRange(char *asc);
	~AccRange();
	long CountItems();
	bool GetFirst(long *x, long *y);
	bool GetNext(long *x, long *y);
	bool NextRow(long *y);
	bool NextCol(long *x);
	bool IsInRange(long x, long y);
	bool BoundRec(RECT *rec);
	char *RangeDesc(void *d, int style);	//d points to a DataObj class
	int DataTypes(void *d, int *numerical, int *strings, int *datetime);

private:
	char *txt;
	long x1, y1, x2, y2, cx, cy, curridx;

	bool Reset();
	bool Parse(long start);
};

class Triangle {
public:
	Triangle *next;
	fPOINT3D pt[4];
	int order[3];					//sort order
	DWORD flags;					// 0x01 has ref to super rectangle

	Triangle();
	Triangle(fPOINT3D *p1, fPOINT3D *p2, fPOINT3D *p3);
	void SetRect();
	bool TestVertex(double x, double y);
	bool Sort();
	void LinePoint(int i1, int i2, double z, lfPOINT *res);
	bool IsoLine(double z, void *dest);

private:
	double cx, cy, r2;				//circumcircle
	bool bSorted;					//vertices are sorted

	bool Circumcircle();
};

class Triangulate {
public:
	Triangle *trl;

	Triangulate(Triangle *t_list);
	bool AddEdge(fPOINT3D *p1, fPOINT3D *p2);
	bool AddVertex(fPOINT3D *v);

private:
	struct edge {
		edge *next;
		fPOINT3D p1, p2;
	};
	edge *edges;
};

class anyOutput{
public:
	double minLW;						//minimum line width in pix
	int MrkMode;						//existing mark on screen
	int OC_type;						//specify display, printer clipboard ...
	void *MrkRect;						//pointer to e.g. the marked rectangle
	void *MrkBitmap;					//use anyOutput class for marking
	fRECT Box1;							//User drawing rectangle
	lfPOINT Box1z;						// add 3D to Box1
	RECT DeskRect;						//this is maximum size Rectangle
	double ddx, ddy, ddz;				//convert to device coordinates
	double hres, vres;					//resolution in dpi
	TextDEF TxtSet;						//store current text settings here
	RunLinePat RLP;						//continuous setings of pattern line
	AxisDEF xAxis, yAxis, zAxis;		//axis and transform definitions
	void *x_owner, *y_owner, *z_owner;	//actual (original owner of AxisDEFs
	int cCursor;						//mouse cursor identifier
	double rotM[3][3];					//rotation matrix for 3D
	fPOINT3D rotC;						//rotation center
	bool hasHistMenu;					//File History List
	int HistMenuSize;					//     -"-
	lfPOINT light_source;					//angles for shading
	double light_vec[3][3];					//     -"-
	void *ShowObj;						//eph_obj
	DWORD dFillCol, dFillCol2;
	double LineWidth;					//line width in units
	DWORD dLineCol;						//current color of line;
	LineDEF oLine;
	double HiVal, LoVal;				//needed for color gradient

	anyOutput();
	virtual ~anyOutput(){return;};
	void SetRect(fRECT rec, int units, AxisDEF *x_ax, AxisDEF *y_ax);
	void UseAxis(AxisDEF *ax, int type, double dx, double dy);
	void SetSpace(fPOINT3D*,fPOINT3D*,int,double*,fPOINT3D*,AxisDEF*,AxisDEF*,AxisDEF*);
	void LightSource(lfPOINT *ls);
	DWORD VecColor(double *, DWORD, DWORD);
	bool GetSize(RECT *rc);
	double fix2fx(double);
	double fiy2fy(double);
	virtual bool ActualSize(RECT *);
	double fx2fix(double);
	long fx2ix(double);
	double fy2fiy(double);
	long fy2iy(double);
	bool fp2fip(lfPOINT *, lfPOINT *);
	bool fvec2ivec(fPOINT3D *, fPOINT3D *);
	bool cvec2ivec(fPOINT3D *, fPOINT3D *);
	bool uvec2ivec(fPOINT3D *, fPOINT3D *);
	double un2fix(double);
	long un2ix(double);
	long co2ix(double);
	double co2fix(double);
	double un2fiy(double);
	long un2iy(double);
	long co2iy(double);
	double co2fiy(double);
	double un2fiz(double);
	double fz2fiz(double);
	double fix2un(double);
	double fiy2un(double);
	virtual bool SetLine(LineDEF *);
	bool GetLine(LineDEF *);
	bool GetFill(FillDEF *);
	virtual bool ClipRect(RECT *);
	virtual int ClipStatus(RECT *);
	virtual bool GetClipRec(RECT *);
	virtual void Focus();
	virtual void Caption(char *, bool);
	virtual void MouseCursor(int , bool);
	virtual bool SetScroll(bool, int, int, int, int);
	virtual bool SetFill(FillDEF *);
	virtual bool SetTextSpec(TextDEF *);
	virtual bool Erase(DWORD);
	virtual bool StartPage();
	virtual bool EndPage();
	virtual bool Eject();		//printers only
	virtual bool UpdateRect(RECT *, bool);
	virtual bool CopyBitmap(int, int, anyOutput*, int, int, int, int, bool);
	virtual bool CopyObject(void *);
	virtual void ShowBitmap(int, int, anyOutput*);
	bool ShowMark(void *rc, int);
	virtual bool HideMark();
	int CalcCursorPos(unsigned char *, POINT, POINT *);
	bool TextCursor(unsigned char *, POINT, POINT *, int *, int);
	bool PatLine(POINT, POINT);
	bool PatLine(lfPOINT, lfPOINT);
	virtual void ShowLine(POINT *, int, DWORD, bool bShow = true);
	virtual void ShowEllipse(POINT, POINT, DWORD, bool bShow = true); 
	virtual bool SetMenu(int);
	virtual void CheckMenu(int, bool);
	virtual void FileHistory();
	virtual bool oGetPix(int, int, DWORD *);
	virtual bool oDrawIcon(int, int, int);
	virtual bool oGetTextExtent(unsigned char *, int, double *, double *);
	virtual bool oGetTextExtentW(w_char *, int, double *, double *);
	virtual bool oCircle(int, int, int, int, char *nam = 0L);
	virtual bool foCircle(double, double, double, double, char *nam = 0L);
	virtual bool oCircArc(int, int, int, int);
	virtual bool oSphere(int, int, int, POINT *, int, char *nam = 0L);
	virtual bool oPolyline(POINT *, int);
	virtual bool foPolyline(lfPOINT *, int);
	virtual bool oBezier(POINT *, int, POINT **bpts = 0L, long *bcp = 0L, bool dodraw = true);
	virtual bool oRectangle(int, int, int, int, char *nam = 0L);
	virtual bool oSolidRectangle(int, int, int, int);
	virtual bool foSolidRectangle(double, double, double, double);
	virtual bool oFillRect(RECT *, DWORD);
	virtual bool foRectangle(double, double, double, double, char *nam = 0L);
	virtual bool oSolidLine(POINT *);
	virtual bool foSolidLine(lfPOINT *);
	virtual bool oTextOut(int, int, unsigned char *, int);
	virtual bool oTextOutW(int, int, w_char *, int);
	virtual bool oPolygon(POINT *, int, char *nam = 0L);
	virtual bool foPolygon(lfPOINT *, int, char *nam = 0L);
	virtual bool oSolidPolygon(POINT *, int);
	virtual bool Command(int, void *);
	virtual void *GetGO(){return 0L;};
	virtual int MenuHeight(){ return 0;};		//ofset due to pull down menus
	bool setVPorg(double x, double y, double scale);
	bool setVPorgScale(double scale);
	double getVPorgX(){ return VPorg.fx; };
	double getVPorgY(){ return VPorg.fy; };
	bool incVPorg(double dx, double dy);
	double getVPorgScale(){ return VPscale; };
	double getdispx() { return disp_x; };
	double getdispy() { return disp_y; };
	void setdispx(double disp) { disp_x = disp;};
	void setdispy(double disp) { disp_y = disp;};
	bool isDlg(){ return isDialog; };
	void FlagIsDialog(bool state) { isDialog = state; };
	void CopyClipRC(RECT *rc_dest) { memcpy(rc_dest, &ClipRC, sizeof(RECT)); };
	virtual void RestoreMark() { return; };
	virtual bool GradPG(fPOINT3D *, long, double Lo, double Hi, double pLo, double pHi, lfPOINT *);

protected:
	lfPOINT VPorg;						//zoom origin
	double VPscale;						//zoom factor
	int iLine;							//current width of line in pixels
	DWORD dBgCol;						//color of background
	DWORD dPattern;						//current line bit-pattern
	double disp_x, disp_y;				//displacement on page
	bool isDialog;
	RECT ClipRC;						//the current clipping rectangle
	bool hasClip;

private:
	FillDEF FillDef;
	LineDEF FillLine;
};

class GridView:public anyOutput {
public:
	GridView(anyOutput *parent);
	~GridView();
	bool SetLine(LineDEF *lDef);
	bool SetTextSpec(TextDEF *set);
	bool SetFill(FillDEF *fill);
	bool oGetTextExtent(unsigned char *text, int cb, double *width, double *height);
	bool oGetTextExtentW(w_char *text, int cb, double *width, double *height);
	bool oCircle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oPolyline(POINT * pts, int cp);
	bool foPolyline(lfPOINT * pts, int cp);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oSolidLine(POINT *p);
	bool oTextOut(int x, int y, unsigned char *txt, int cb);
	bool oTextOutW(int x, int y, w_char *txt, int cb);
	bool oPolygon(POINT *pts, int cp, char *nam = 0L);
private:
	anyOutput *out;

	bool ShowTxtRec(int x, int y, int width, int height);
};

enum {ET_UNKNOWN, ET_VALUE, ET_TEXT, ET_FORMULA, ET_ERROR, ET_BOOL, 
	ET_DATE, ET_TIME, ET_DATETIME, ET_BUSY=0x100, ET_CIRCULAR=0x200, ET_EMPTY=0x400, 
	ET_NODRAW=0x800, ET_NODRAW_EMPTY=0xc00, ET_DIALOG=0x1000, ET_NOEDIT = 0x2000};

class EditText {
public:
	unsigned char *text, *ftext;
	unsigned int type;
	void *parent;				//points to a data object: defined below
	POINT loc, rb, crb;
	double Value;

	EditText(void *par, unsigned char *msg, int r, int c);
	~EditText();
	bool AddChar(int c, anyOutput *Out, void *data_obj);
	void Update(int select, anyOutput *Out, POINT *MousePos);
	bool Redraw(anyOutput *Out, bool display);
	void Mark(anyOutput *Out, int mark);
	bool Command(int cmd, anyOutput *Out, void *data_obj);
	bool GetValue(double *v);
	bool GetText(char *t, int size, bool bTranslate);
	bool GetResult(anyResult *r, bool use_last = false);
	bool SetValue(double v);
	bool SetText(unsigned char *t);
	void SetRec(RECT *rc);
	int Cursor() {return CursorPos;};
	bool isValue();
	bool isFormula();
	void formula2int(unsigned char *text);
	bool isInRect(POINT *p) {return (p->x>loc.x && p->x<crb.x && p->y>loc.y && p->y<rb.y);};
	bool hasMark() {return (m1 != m2 && m1 >= 0 && m2 >= 0);};
	void FindType();

protected:
	int Align;
	long row, col;
	anyOutput *disp;
	double *a_data;
	long a_count;

private:
	LineDEF *bgLine;
	FillDEF *bgFill;
	int length, CursorPos, m1, m2, mx1, mx2;
	DWORD TextCol;

	void set_etracc();
};

class clear_sym {
public:
	clear_sym();
	void set_rect(POINT *pos);
	void do_plot(anyOutput *o);
	bool select(int x, int y);

private:
	RECT rec, mrc;
	POINT pts[6];
	anyOutput *mo, *currdisp;
};

class fmtText {

typedef struct _fmt_txt_info {
	int tag, uc, uc_len;
	unsigned char *txt;
}fmt_txt_info;

typedef struct _fmt_uc_info {
	int tag, cb;
	w_char *uc_txt;
}fmt_uc_info;

public:
	fmtText();
	fmtText(anyOutput *o, int x, int y, unsigned char *txt);
	~fmtText();
	bool StyleAt(int idx,  TextDEF *txt_def, int *style, int *font);
	int rightTag(unsigned char *txt, int cb);
	int leftTag(unsigned char *txt, int cb);
	void cur_right(int *pos);
	void cur_left(int *pos);
	bool oGetTextExtent(anyOutput *o, double *width, double *height, int cb);
	void SetText(anyOutput *o, unsigned char *txt, double *px, double *py, bool bDraw);
	void DrawText(anyOutput *o);
	void EditMode(bool bEdit);
	int CalcCursorPos(int x, anyOutput *o);


private:
	bool SetTextDef(TextDEF *td, int idx);
	bool Parse();
	bool DrawFormattedW(anyOutput *o);
	void DrawFormatted(anyOutput *o);

	unsigned char *src;
	POINT pos;
	int n_split, n_split_W, uc_state;
	DWORD flags;
	fmt_txt_info *split_text;
	fmt_uc_info *split_text_W;
};

class TextValue {

typedef struct _TxtValItem {
	unsigned int hv1, hv2;
	char *text;
	double val;
}TextValItem;

public:
	TextValue();
	TextValue(TextValItem **tvi, int ntvi);
	~TextValue();
	double GetValue(char* txt);
	bool HasValue(char* txt, double *value);
	bool GetItem(int idx, char **text, double *value);
	void Reset();
	TextValue *Copy();
	int Count() {return nitems;};

private:
	TextValItem **items;
	int nitems;
	double next, inc;
};

class RangeInfo {
public:
	int col_width, row_height, first_width;
	RangeInfo(int sel, int cw, int rh, int fw, RangeInfo *next);
	RangeInfo(int sel, char *range, RangeInfo *next);
	~RangeInfo();
	int SetWidth(int w);
	int SetDefWidth(int w);
	int SetHeight(int h);
	int SetFirstWidth(int w);
	int GetWidth(int col);
	int GetHeight(int row);
	int GetFirstWidth();
	int Type(){return r_type;};
	RangeInfo *Next(){return ri_next;};

private:
	int r_type;		// 0 default
					// 1 column info
					// 2 row info
	RangeInfo *ri_next;
	AccRange *ar;
};

class DataObj{
public:
	long cRows, cCols, c_disp, r_disp;
	RangeInfo *ri;
	EditText ***etRows;

	DataObj();
	virtual ~DataObj();
	virtual bool Init(long nRows, long nCols);
	virtual bool mpos2dpos(POINT *, POINT *, bool){return false;};
	virtual bool SetValue(long row, long col, double val);
	virtual bool SetText(long row, long col, char *txt);
	virtual bool GetValue(long row, long col, double *v);
	virtual bool GetText(long row, long col, char *txt, int len, bool bTranslate = true);
	char **GetTextPtr(long row, long col);
	virtual bool GetResult(anyResult *r, long row, long col, bool use_last = false);
	virtual bool GetSize(long *width, long *height);
	virtual bool isEmpty(long row, long col);
	virtual bool Select(POINT *){return false;};
	virtual bool WriteData(char *) {return false;};
	virtual bool AddCols(long){return false;};
	virtual bool AddRows(long){return false;};
	virtual bool ChangeSize(long, long, bool){return false;};
	virtual bool Command(int, void *, anyOutput *){return false;};
	virtual bool ReadData(char *, unsigned char *, int){return false;};
	virtual void FlushData();
	bool ValueRec(RECT *rc, bool any);
	virtual bool hasEtracc(){ return false; };
};

class StrData {
public:
	StrData(DataObj *par, RECT *rc = 0L);
	~StrData();
	bool GetSize(int *w, int *h);
	void RestoreData(DataObj *dest);

private:
	long pw, ph;
	RECT drc;
	DataObj *src;
	char ***str_data;
};

class HatchOut:public anyOutput {
public:
	HatchOut(anyOutput *Parent);
	bool SetFill(FillDEF *fill);
	bool StartPage();
	bool oCircle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oSphere(int cx, int cy, int r, POINT *pts, int cp, char *nam = 0L);
	bool oRectangle(int x1, int y1, int x2, int y2, char *nam = 0L);
	bool oPolygon(POINT *pts, int cp, char *nam = 0L);
	anyOutput* getDisp(){ return out; };

private:
	anyOutput *out;
	RECT pClipRC;
	double xbase, ybase;
	LineDEF ParLineDef, MyLineDef;
	FillDEF ParFillDef, MyFillDef;
	bool ParInit;
	int ho, ht;
	DWORD ParLinPat;
	RECT UseRect;
	struct {
		int cx, cy, r;
		}circ_grad;

	bool PrepareParent(bool Restore);
	bool DoHatch();
	bool MkPolyLine(POINT *p, anyOutput *o, bool bFill = false);
	bool HatchLine(POINT p1, POINT p2, bool bFill = false);
	bool HatchArc(long x, long y, long r, int qad, bool start);
	bool IsInside(POINT p);
	void Lines000();
	void Lines090();
	void Lines045();
	void Lines315();
	void Stipple(int type);
	void Zigzag();
	void Combs();
	void BricksH();
	void BricksV();
	void Bricks045();
	void Bricks315();
	void Texture1();
	void Texture2();
	void Arcs(int type);
	void Waves2(int type);
	void Herringbone();
	void Circles();
	void Grass();
	void Foam();
	void Recs();
	void Hash();
	void CircGrad();
	void Chess();
};

class GraphObj {				//the base class form  all objects
public:
	unsigned long Id;			//accepts an identifier during read from file
								//  it is set to an object identifier after
								//  construction
	GraphObj *parent;
	DataObj *data;
	int moveable, hidden;
	long type;
	RECT rDims;
	char *name;

	GraphObj(GraphObj *par, DataObj *d);
	virtual ~GraphObj();
	virtual double GetSize(int select);
	virtual bool SetSize(int, double){return false;};
	virtual DWORD GetColor(int select);
	virtual bool SetColor(int, DWORD) {return false;};
	virtual void DoPlot(anyOutput *){return;};
	virtual void DoMark(anyOutput *, bool){ return; };
	virtual bool Command(int, void *, anyOutput *){return false;};
	virtual bool PropertyDlg(){return false;};
	virtual void RegGO(void *n);
	virtual bool FileIO(int) {return false;};
	virtual void CheckBounds3D(double, double, double) { return; };
	virtual double Area() { return 0.0; };
	virtual void * ObjThere(int x, int y);
	virtual void Track(POINT *p, anyOutput *o, bool accept);
	virtual double DefSize(int select);
	virtual bool hasTransp(){return false;};
	virtual bool MemList(unsigned char **, int){ return false; };
	virtual anyOutput* getDisp(){ return 0L; };
};

class ssButton:public GraphObj {
public:

	ssButton(GraphObj *par, int x, int y, int w, int h);
	~ssButton();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *target, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);

private:
	bool bLBdown, bSelected, bMarked;
	TextDEF TextDef;
	LineDEF Line;
	FillDEF Fill;
};

class dragHandle:public GraphObj {
public:

	dragHandle(GraphObj *par, int type);
	~dragHandle();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

private:
	RECT upd, drec, *minRC, *maxRC;
	double x_pos, y_pos;				//the actual position of handle in px
	LineDEF LineDef;
	FillDEF FillDef;
};

class dragRect:public GraphObj {
public:

	dragRect(GraphObj *par, int type);
	~dragRect();
	double GetSize(int select){return parent->GetSize(select);};
	DWORD GetColor(int select){return parent->GetColor(select);};
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void*tmpl, anyOutput *o);
	void * ObjThere(int x, int y);

private:
	dragHandle **handles;
};

class Drag3D:public GraphObj {
public:
	Drag3D(GraphObj *par);
	~Drag3D();
	double GetSize(int select){return parent->GetSize(select);};
	void DoPlot(anyOutput *o);
	void * ObjThere(int x, int y);

private:
	dragHandle **handles;
};

class FrmRect:public GraphObj {
public:

	FrmRect(GraphObj *par, fRECT *limRC, fRECT *cRC, fRECT *chld);
	~FrmRect();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoMark(anyOutput *o, bool mark);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg(){return parent ? parent->Command(CMD_CONFIG, 0L, 0L) : false;};
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

private:
	RECT *minRC, *maxRC;
	anyOutput *mo;
	RECT mrc;
	dragRect *drag;
	bool swapX, swapY;
	fRECT *limRC, *cRC, *chldRC, CurrRect;
	LineDEF Line, FillLine;
	FillDEF Fill;
};

class Grid:public GraphObj {
public:
	Grid(GraphObj *par);
	Grid(int src);
	~Grid();
	double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int);
	bool SetColor(int, DWORD);
	void DoPlot(anyOutput *);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);

	void Snap(POINT *pt, anyOutput *);		//snap to grid
	void Snap(lfPOINT *pt, anyOutput *);

private:
	double spacing;
	LineDEF line;
};

class Symbol:public GraphObj{
public:
	int idx;
	anyOutput *mo;
	lfPOINT fPos;
	POINT *ssRef;
	long cssRef;
	double size;
	LineDEF SymLine;
	FillDEF SymFill;
	TextDEF *SymTxt;
	lfPOINT last_pos;

	Symbol(GraphObj *, DataObj *, double x, double y, int which, 
		int xc = -1, int xr = -1, int yc = -1, int yr = -1);
	Symbol(int src);
	~Symbol();
	virtual double GetSize(int select);
	virtual bool SetSize(int select, double value);
	DWORD GetColor(int select);
	bool SetColor(int select, DWORD col);
	virtual void DoPlot(anyOutput *target);
	virtual void DoMark(anyOutput *o, bool mark);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	virtual bool FileIO(int rw);

	void DoPlotXY(anyOutput *target, double fix, double fiy);

private:
	RECT mrc;
};

class SymTernary:public Symbol {
public:
	fPOINT3D fPos3D;

	SymTernary(GraphObj *, DataObj *, double x, double y, double z, int which, 
		int xc = -1, int xr = -1, int yc = -1, int yr = -1, int zc = -1, int zr = -1);
	SymTernary(int src);
	~SymTernary();
	double GetSize(int select);
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);
};

class Bubble:public GraphObj{
public:
	Bubble(GraphObj *, DataObj *, double x, double y, double s, 
		int which, FillDEF *fill, LineDEF *outline, int xc = -1, 
		int xr = -1, int yc = -1, int yr = -1, int sc = -1, int sr = -1);
	Bubble(int src);
	~Bubble();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	bool DoAutoscale(anyOutput *o);

	lfPOINT fPos;
	double fs;
	LineDEF BubbleLine, BubbleFillLine;
	FillDEF BubbleFill;
	POINT pts[5];
	lfPOINT fPts[5];
	POINT *ssRef;
	long cssRef;
};

class Bar:public GraphObj {
public:
	LineDEF BarLine, HatchLine;
	FillDEF BarFill;

	Bar(GraphObj *, DataObj *, double x, double y, int which,
		int xc = -1, int xr = -1, int yc = -1, int yr = -1, char *desc = 0L);
	Bar(int src);
	~Bar();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *target);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	anyOutput *mo;
	RECT mrc;
	double size;
	lfPOINT fPos, BarBase;
	POINT *ssRef;
	long cssRef;
};

class DataLine:public GraphObj{
public:
	bool isPolygon, dirty;
	lfPOINT *Values;
	lfPOINT min, max;
	LineDEF LineDef;
	FillDEF pgFill;
	LineDEF pgFillLine;
	long nPnt, cp;
	DWORD BgColor;
	POINT *pts;
	char *ssXref, *ssYref, *cond;
	anyOutput *mo;
	RECT mrc;

	DataLine(GraphObj *, DataObj *, char *xrange=0L, char *yrange=0L, char *name=0L, char *cond = 0L);
	DataLine(GraphObj *, DataObj *, lfPOINT *val, long nval, char *name);
	DataLine(int src);
	virtual ~DataLine();
	bool SetColor(int select, DWORD col);
	bool SetSize(int select, double value);
	virtual void DoPlot(anyOutput *target);
	virtual void DoMark(anyOutput *o, bool mark);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	virtual bool PropertyDlg();
	virtual bool FileIO(int rw);

	void FileValues(char *name, int type, double start, double step);
	void SetValues();
	void LineData(lfPOINT *val, long nval);
	void DrawCurve(anyOutput *target);
	void DrawSpline(anyOutput *target);
	bool MemList(unsigned char **ptr, int type);

};

class DataPolygon:public DataLine{
public:
	DataPolygon(GraphObj *, DataObj *, char *xrange=0L, char *yrange=0L, char *name=0L);
	DataPolygon(GraphObj *, DataObj *, lfPOINT *val, long nval, char *name = 0L);
	DataPolygon(int src);
	~DataPolygon();
	virtual void DoPlot(anyOutput *target);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	double Area();

private:
	long src_nPnt;
	double area;
};

class ErrorPolygon :public DataPolygon{
public:
	ErrorPolygon(GraphObj *, DataObj *, int nlow, lfPOINT *lowvals, int nhigh, lfPOINT *hivals);
	ErrorPolygon(GraphObj *, DataObj *, char *xrange, char *yrange, char *erange, char * name);
	ErrorPolygon(GraphObj *, DataObj *, char *xrange, char *lo_range, char *hi_range);
	ErrorPolygon(int src);
	~ErrorPolygon();
	void DoPlot(anyOutput *target);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	bool SetValues(long nlow, lfPOINT *lowvals, long nhigh, lfPOINT *hivals);
	bool LoadValues();
	bool Update();

private:
	char *ErrRange, *LoRange, *HiRange;
	fRECT pgBounds;
};

class LineTernary :public DataLine {
public:
	int doPG;
	LineDEF LineDef, FillLine;

	LineTernary(GraphObj *, DataObj *, char *rng1, char *rng2, char* rng3, char *name = 0L);
	LineTernary(int src);
	~LineTernary();
	virtual void DoPlot(anyOutput *target);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);
	anyOutput* getDisp(){ return CurrDisp; };


private:
	char *range1, *range2, *range3;
	FillDEF Fill;
	DataLine *DispLine;
	fPOINT3D* values_xyz;
	long nValues;
	anyOutput *CurrDisp;
	lfPOINT zoomOrg;
	double zoomScale;

	bool MkDispLine(anyOutput *target);
	bool DoUpdate();
};

class RegLine:public GraphObj{
public:
	RegLine(GraphObj *, DataObj *, lfPOINT *values, long n, int type);
	RegLine(int src);
	~RegLine();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

	void Recalc(lfPOINT *values, long n);
	LineDEF *GetLine(){return &LineDef;};

private:
	long nPoints, cp;
	double mx, my;
	LineDEF LineDef;
	fRECT lim, uclip;
	lfPOINT l1, l2, l3, l4, l5;
	DWORD BgColor;
	POINT *pts;
};

class SDellipse:public GraphObj{
public:
	SDellipse(GraphObj *, DataObj *, lfPOINT *values, long n, int sel);
	SDellipse(int src);
	~SDellipse();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

	void Recalc(lfPOINT *values, long n);

private:
	long nPoints, cp;
	double sd1, sd2, mx, my;
	POINT *pts;
	LineDEF LineDef;
	fRECT lim;
	lfPOINT *val;
	RegLine *rl;
};

class ErrorBar:public GraphObj{
public:
	ErrorBar(GraphObj *, DataObj *, double x, double y, double err, int type,
		int xc=-1, int xr=-1, int yc=-1, int yr=-1, int ec=-1, int er=-1);
	ErrorBar(int src);
	~ErrorBar();
	bool SetColor(int select, DWORD col);
	bool SetSize(int select, double value);
	DWORD GetColor(int select);
	double GetSize(int select);
	void DoPlot(anyOutput *target);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	anyOutput *mo;
	RECT mrc;
	lfPOINT fPos;
	double ferr, SizeBar;
	POINT ebpts[6];
	LineDEF ErrLine;
	POINT *ssRef;
	long cssRef;
};

class Arrow:public GraphObj {
public:
	anyOutput *mo;
	RECT mrc;
	dragHandle *dh1, *dh2;
	double cw, cl;
	LineDEF LineDef;
	bool bModified;

	Arrow(GraphObj *, DataObj *, lfPOINT *fp1, lfPOINT *fp2, int which = 0,
		int xc1=-1, int xr1=-1, int yc1=-1, int yr1=-1, int xc2=-1, int xr2=-1,
		int yc2=-1, int yr2=-1);
	Arrow(int src);
	~Arrow();
	double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int){return LineDef.color;};
	bool SetColor(int select, DWORD col);
	virtual void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	virtual bool FileIO(int rw);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

	void Redraw(anyOutput *o);
	void DoPlotXY(anyOutput *o, double fix1, double fiy1, double fix2, double fiy2);

private:
	POINT pts[5];
	POINT *ssRef;
	long cssRef;
	lfPOINT pos1, pos2;
};

class ArrTernary:public Arrow {
public:
	ArrTernary(GraphObj *, DataObj *, fPOINT3D *fp1, fPOINT3D *fp2, int which = 0, int xc1 = -1, int xr1 = -1,
		int yc1 = -1, int yr1 = -1, int zc1 = -1, int zr1 = -1, int xc2 = -1, int xr2 = -1, int yc2 = -1,
		int yr2 = -1, int zc2 = -1, int zr2 =-1);
	ArrTernary(int src);
	~ArrTernary();
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);

private:
	fPOINT3D pos1, pos2;
	POINT3D ssRef[4];
};

class Box:public GraphObj {
public:
	Box(GraphObj *, DataObj *, lfPOINT fp1, lfPOINT fp2, int which, 
		int xc1=-1, int xr1=-1, int yc1=-1, int yr1=-1, int xc2=-1, int xr2=-1,
		int yc2=-1, int yr2=-1);
	Box(int src);
	~Box();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	anyOutput *mo;
	RECT mrc;
	double size;
	lfPOINT pos1, pos2;
	POINT pts[5];
	LineDEF Outline, Hatchline;
	FillDEF Fill;
	POINT *ssRef;
	long cssRef;
};

class Whisker:public GraphObj {
public:
	Whisker(GraphObj *, DataObj *, lfPOINT fp1, lfPOINT fp2, int which,
		int xc1=-1, int xr1=-1, int yc1=-1, int yr1=-1, int xc2=-1, int xr2=-1,
		int yc2=-1, int yr2=-1);
	Whisker(int src);
	~Whisker();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	anyOutput *mo;
	RECT mrc;
	double size;
	POINT pts[6];
	lfPOINT pos1, pos2;
	LineDEF LineDef;
	POINT *ssRef;
	long cssRef;
};

class DropLine:public GraphObj{
public:
	DropLine(GraphObj *, DataObj *, double x, double y, int which, int xc = -1, 
		int xr = -1, int yc = -1, int yr = -1);
	DropLine(int src);
	~DropLine();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	lfPOINT fPos;
	POINT pts[4];
	LineDEF LineDef;
	POINT *ssRef;
	long cssRef;
	bool bModified;
};

class line_segment:public GraphObj{
public:
	line_segment(GraphObj *, DataObj *, LineDEF *ld, fPOINT3D *p1, fPOINT3D *p2);
	~line_segment();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void * ObjThere(int x, int y);

private:
	LineDEF Line;
	fPOINT3D **ldata;
	fPOINT3D fmin, fmax;
	int *nldata, nli, ndf_go;
	double prop;
	bool bDrawDone;
	GraphObj *co, **df_go;
	lfPOINT xBounds, yBounds, zBounds;

	void DoClip(GraphObj *co);
};

class sph_scanline:public GraphObj{
public:
	double rad;
	bool vert;

	sph_scanline(fPOINT3D *center, double radius, bool bVert);
	bool Command(int cmd, void *tmpl, anyOutput *o);

	void DoClip(GraphObj *co);
	bool GetPoint(POINT *p, int sel);

private:
	fPOINT3D p1, p2, cent;
	bool bValid1, bValid2;
};

class ErrBar3D:public GraphObj{
public:
	ErrBar3D(GraphObj *, DataObj *, double x, double y, double z, double err, int type,
		int xc=-1, int xr=-1, int yc=-1, int yr=-1, int zc=-1, int zr=-1, int ec=-1, int er=-1);
	ErrBar3D(int src);
	~ErrBar3D();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *target);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	anyOutput *mo;
	RECT mrc;
	fPOINT3D fPos;
	double ferr, SizeBar;
	line_segment *ls[3];
	LineDEF ErrLine;
	POINT *ssRef, mpts[3][2];
	long cssRef;
};

class Sphere:public GraphObj{
public:
	Sphere(GraphObj *, DataObj *, int sel, double x, double y, double z, double r,
		int xc = -1, int xr = -1, int yc = -1, int yr = -1, int zc = -1, int zr = -1,
		int rc = -1, int rr = -1);
	Sphere(int src);
	~Sphere();
	double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int select);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	int ix, iy, rx, ry, rz, nscl;
	LineDEF Line;
	FillDEF Fill;
	fPOINT3D fPos, fip;
	double size;
	POINT *ssRef;
	long cssRef;
	GraphObj *co;
	bool bModified, bDrawDone;
	sph_scanline **scl;
	anyOutput *mo;

	void DoClip(GraphObj *co);
	void DrawPG(anyOutput *o, int start);
};

class plane:public GraphObj{
public:
	plane(GraphObj *, DataObj *, fPOINT3D *pts, int nPts, fPOINT3D *oData, int nOdt, LineDEF *line, 
		FillDEF *fill);
	~plane();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void * ObjThere(int x, int y);

	bool GetPolygon(fPOINT3D **pla, int *npt, int sel);
	double *GetVec() {return PlaneVec;};
	bool DoGradient(anyOutput *o, fPOINT3D *pts, int npt);

private:
	LineDEF Line;
	FillDEF Fill;
	double *PlaneVec;
	fPOINT3D **ldata, ReqPoint;
	int *nldata, nli, n_ipts, n_lines, n_linept;
	POINT *ipts;
	lfPOINT xBounds, yBounds, zBounds;
	bool bDrawDone, bReqPoint, bSigPol;
	GraphObj *co;
	fPOINT3D *src_pts;
	line_segment **lines;
	double totalArea;
	fPOINT3D *srcDt;					//original data of parent
	int srcN;							//and values in srcDt

	void DoClip(GraphObj *co);
	bool IsValidPG(fPOINT3D *pg, int npg);
};

class Plane3D:public GraphObj {
public:
	Plane3D(GraphObj *, DataObj *, fPOINT3D *pt, long npt);
	Plane3D(int src);
	~Plane3D();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	double PlaneValue(lfPOINT v);

private:
	plane *ipl;
	double *PlaneVec;
	fPOINT3D *dt, *pts;
	long ndt;
	LineDEF Line;
	FillDEF Fill;

	bool CreatePlaneVec();
};

class Brick:public GraphObj{
public:
	Brick(GraphObj *, DataObj *, double x, double y, double z, 
		double d, double w, double h, DWORD flags, int xc = -1,
		int xr = -1, int yc = -1, int yr = -1, int zc = -1, int zr = -1,
		int dc = -1, int dr = -1, int wc = -1, int wr = -1, int hc = -1,
		int hr = -1);
	Brick(int src);
	~Brick();
	bool SetSize(int select, double value);
	double GetSize(int select);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	LineDEF Line;
	FillDEF Fill;
	fPOINT3D fPos;
	double depth, width, height;
	DWORD flags;
	POINT *ssRef;
	long cssRef;
	plane **faces;
	anyOutput *mo;
	RECT mrc;
	bool bModified;
};

class DropLine3D:public GraphObj{
public:
	DropLine3D(GraphObj *, DataObj *, fPOINT3D *p1, int xc = -1,
		int xr = -1, int yc = -1, int yr = -1, int zc = -1, int zr = -1);
	DropLine3D(int src);
	~DropLine3D();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	line_segment *ls[6];
	POINT mpts[6][2];
	LineDEF Line;
	fPOINT3D fPos;
	POINT *ssRef;
	long cssRef;
	bool bModified;
	anyOutput *mo;
	RECT mrc;
};

class Arrow3D:public GraphObj{
public:
	Arrow3D(GraphObj *, DataObj *, fPOINT3D *p1, fPOINT3D *p2, int xc1 = -1,
		int xr1 = -1, int yc1 = -1, int yr1 = -1, int zc1 = -1, int zr1 = -1, 
		int xc2 = -1, int xr2 = -1, int yc2 = -1, int yr2 = -1, int zc2 = -1, 
		int zr2 = -1);
	Arrow3D(int src);
	~Arrow3D();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	line_segment *ls[3];
	plane *cap;
	double cw, cl;
	POINT mpts[3][2];
	LineDEF Line;
	fPOINT3D fPos1, fPos2;
	POINT *ssRef;
	long cssRef;
	bool bModified;
	anyOutput *mo;
	RECT mrc;
};

class Line3D:public GraphObj{
public:
	LineDEF Line;
	FillDEF Fill;

	Line3D(GraphObj *, DataObj *);
	Line3D(GraphObj *, DataObj *, char *rx, char *ry, char *rz);
	Line3D(GraphObj *, DataObj *, fPOINT3D *pt, int n_pt, int xc1 = -1,
		int xr1 = -1, int yc1 = -1, int yr1 = -1, int zc1 = -1, int zr1 = -1, 
		int xc2 = -1, int xr2 = -1, int yc2 = -1, int yr2 = -1, int zc2 = -1, 
		int zr2 = -1);
	Line3D(int src);
	~Line3D();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	line_segment **ls;
	fPOINT3D *values, min, max;
	POINT *pts;
	long nPts, npts;
	char *x_range, *y_range, *z_range;
	POINT *ssRef;
	long cssRef;
	bool bModified;
	anyOutput *mo;
	RECT mrc;

	void DoUpdate();
};

class Polygon3D:public Line3D{
public:
	Polygon3D(GraphObj *, DataObj *, char *rx, char *ry, char *rz);
	Polygon3D(GraphObj *, DataObj *, fPOINT3D *pt, int n_pt, int xc1 = -1,
		int xr1 = -1, int yc1 = -1, int yr1 = -1, int zc1 = -1, int zr1 = -1,
		int xc2 = -1, int xr2 = -1, int yc2 = -1, int yr2 = -1, int zc2 = -1,
		int zr2 = -1);
	Polygon3D(int src);
	~Polygon3D();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);

private:
	line_segment **ls;
	Plane3D *plane;
	fPOINT3D *values, min, max;
	POINT *pts;
	long nPts, npts;
	char *x_range, *y_range, *z_range;
	POINT *ssRef;
	long cssRef;
	bool bModified;
	anyOutput *mo;
	RECT mrc;

	void DoUpdate();

};

class Label:public GraphObj{
public:
	Label(GraphObj *, DataObj *, double x, double y, TextDEF *td, DWORD flg, char *txt,
		int xc, int xr, int yc, int yr, int tc, int tr);
	Label(GraphObj *, DataObj *, double x, double y, TextDEF *td, DWORD flg, char *txt);
	Label(int src);
	~Label();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

	bool CalcRect(anyOutput *o);
	void RedrawEdit(anyOutput *o);
	void ShowCursor(anyOutput *o);
	bool AddChar(int ci, anyOutput *o);
	bool AddText(unsigned char *src, anyOutput *o);
	void CalcCursorPos(int x, int y, anyOutput *o);
	void SetModified();
	void DoPlotText(anyOutput *o);
	bool CheckMark();
	TextDEF *GetTextDef(){return &TextDef;};
	int Append(char* txt);
	anyOutput *getDisp(){ return (parent ? parent->getDisp() : 0L); };

private:
	lfPOINT fPos, fDist;
	double si, csi, curr_z;
	DWORD flags, bgcolor;
	TextDEF TextDef;
	LineDEF bgLine, trackLine;
	double fix, fiy;
	int m1, m2, CursorPos;
	bool is3D;
	RECT Cursor, rm1, rm2;
	POINT pts[5];
	POINT *ssRef;
	long cssRef;
	anyOutput *defDisp;
	bool bModified, bBGvalid, bBusy;
	RECT mrc;
	anyOutput *mo;
};

class mLabel:public GraphObj{
public:
	mLabel(GraphObj *, DataObj *, double, double, TextDEF *, char *, int, DWORD);
	mLabel(GraphObj *, DataObj *, double, double, TextDEF *, char *);
	mLabel(int src);
	~mLabel();
	double GetSize(int select);
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void RegGO(void *n);
	bool FileIO(int rw);
	void Track(POINT *p, anyOutput *o, bool accept);

private:
	lfPOINT fPos, fDist, cPos, cPos1, dist;
	double si, csi, curr_z, lspc;
	DWORD flags, undo_flags;
	TextDEF TextDef;
	long nLines;
	int cli;
	bool is3D;
	Label **Lines;
};

class TextFrame:public GraphObj{
	enum {TF_MAXLINE=120};
public:
	TextFrame(GraphObj *, DataObj *, lfPOINT *p1, lfPOINT *p2, char *txt);
	TextFrame(int src);
	~TextFrame();
	double GetSize(int select);
	bool SetSize(int select, double value);
	void DoMark(anyOutput *o, bool mark);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);
	anyOutput* getDisp(){ return CurrDisp; };

private:
	fRECT pad;
	RECT ipad, Cursor, *tm_rec;
	fmtText fmt_txt;
	anyOutput *CurrDisp;
	int nlines, linc, tm_c;
	long csize, cpos;
	bool bModified, bResize, has_m1, has_m2, bFocus;
	unsigned char c_char, m1_char, m2_char;
	double lspc;
	lfPOINT pos1, pos2;
	POINT cur_pos, m1_pos, m2_pos, m1_cpos, m2_cpos;
	TextDEF TextDef;
	dragRect *drc;
	unsigned char *text, **lines;
	LineDEF Line, FillLine;
	FillDEF Fill;
	DWORD UndoFlags;

	void text2lines(anyOutput *o);
	void lines2text();
	void ShowCursor(anyOutput *o);
	void AddChar(anyOutput *o, int c);
	void DelChar(anyOutput *o);
	void CalcCursorPos(int x, int y, anyOutput *o);
	void ReplMark(anyOutput *o, char *ntext);
	void procTokens();
	bool DoPaste(anyOutput *o);
	void TextMark(anyOutput *o, int mode);
	bool CopyText(anyOutput *o, bool b_cut);
};

class segment:public GraphObj{
public:
	segment(GraphObj *, DataObj *, lfPOINT *c, double r1, double r2, double a1, double a2, char *name = 0L);
	segment(int src);
	~segment();
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

private:
	lfPOINT fCent;
	long nPts;
	double radius1, radius2, angle1, angle2, shift;
	LineDEF segLine, segFillLine;
	FillDEF segFill;
	POINT *pts;
	bool bModified;
	anyOutput *mo;
	RECT mrc;
};

class polyline:public GraphObj {
public:
	dragHandle **pHandles;
	lfPOINT *Values;
	long nPoints, nPts;
	POINT *pts;
	LineDEF pgLine, pgFillLine;
	FillDEF pgFill;
	bool bModified;

	polyline(GraphObj *, DataObj *, lfPOINT *fpts, int cpts);
	polyline(int src);
	virtual ~polyline();
	double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int select);
	virtual void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	virtual bool PropertyDlg();
	virtual bool FileIO(int rw);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

private:
	void ShowPoints(anyOutput *o);
};

class Bezier:public polyline {
public:
	Bezier(GraphObj *, DataObj *, lfPOINT *fpts, int cpts, int mode, double res);
	Bezier(int src);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);

private:
	void AddPoints(int n, lfPOINT *p);

	void FitCurve(lfPOINT *d, int npt, double error);
	void IpolBez(lfPOINT *d, lfPOINT tHat1, lfPOINT tHat2);
	void FitCubic(lfPOINT *d, int first, int last, lfPOINT tHat1, lfPOINT tHat2, double error);
	void RemovePoint(lfPOINT *d, int sel);
	void GenerateBezier(lfPOINT *d, int first, int last, double * uPrime, lfPOINT tHat1, lfPOINT tHat2, lfPOINT *bezCurve);
	double *Reparameterize(lfPOINT *d, int first, int last, double *u, lfPOINT *bezCurve);
	lfPOINT fBezier(int degree, lfPOINT *V, double t);
	double *ChordLengthParameterize(lfPOINT *d, int first, int last);
	double ComputeMaxError(lfPOINT *d, int first, int last, lfPOINT *bezCurve, double *u, int *splitPoint);
};

class polygon:public polyline {
public:
	polygon(GraphObj *, DataObj *, lfPOINT *fpts, int cpts);
	polygon(int src):polyline(src){};
	bool PropertyDlg();
};

class rectangle:public GraphObj {
public:
	lfPOINT fp1, fp2;
	LineDEF Line, FillLine;
	FillDEF Fill;
	double rad;
	bool bModified;

	rectangle(GraphObj *, DataObj *, lfPOINT *p1, lfPOINT *p2);
	rectangle(int src);
	virtual ~rectangle();
	double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int){return Line.color;};
	void DoMark(anyOutput *o, bool mark);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);
	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

private:
	POINT *pts;
	long nPts;
	dragRect *drc;
	void PlotRoundRect(anyOutput *o);
};

class ellipse:public rectangle {
public:
	ellipse(GraphObj *, DataObj *, lfPOINT *p1, lfPOINT *p2);
	ellipse(int src);
};

class roundrec:public rectangle {
public:
	roundrec(GraphObj *, DataObj *, lfPOINT *p1, lfPOINT *p2);
	roundrec(int src);
};

class LegItem:public GraphObj{
public:
	DWORD flags;

	LegItem(GraphObj *, DataObj *, LineDEF *ld, LineDEF *lf, FillDEF *fill, char *desc);
	LegItem(GraphObj *, DataObj *, LineDEF *ld, Symbol *sy);
	LegItem(GraphObj *, DataObj *, LineDEF *ld, int err, char *desc);
	LegItem(int src);
	~LegItem();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void RegGO(void *n);
	bool FileIO(int rw);
	void Track(POINT *p, anyOutput *o, bool accept);

	bool HasFill(LineDEF *ld, FillDEF *fd, char *desc);
	bool HasSym(LineDEF *ld, GraphObj *sy);
	bool HasErr(LineDEF *ld, int err);

private:
	LineDEF DataLine, OutLine, HatchLine;
	FillDEF Fill;
	Symbol *Sym;
	Label *Desc;
	RECT hcr;

	void DefDesc(char *txt);
};

class Legend:public GraphObj{
public:
	Legend(GraphObj *, DataObj *);
	Legend(int src);
	~Legend();
	double GetSize(int select);
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void RegGO(void *n);
	bool FileIO(int rw);
	void Track(POINT *p, anyOutput *o, bool accept);

	bool HasFill(LineDEF *ld, FillDEF *fd, char *desc);
	bool HasSym(LineDEF *ld, GraphObj *sy, char *desc);
	bool HasErr(LineDEF *ld, int err, char *desc);

private:
	lfPOINT pos, lb_pos;
	RECT trc;
	anyOutput *to;
	fRECT B_Rect, C_Rect, D_Rect, E_Rect, F_Rect;
	long nItems;
	bool hasLine;
	LegItem **Items;
};

class Plot:public GraphObj{
public:
	fRECT Bounds;					//contains minima and maxima for x and y
	bool dirty;						//rescale before redraw;
	int use_xaxis, use_yaxis, use_zaxis;		//this plot uses its own axes
	lfPOINT xBounds, yBounds, zBounds;	//like Bounds but in 3D space
	int x_dtype, y_dtype, z_dtype;	//data types
	char *x_info, *y_info, *z_info;	//descriptor used e.g. for axis label or legend
	char *data_desc;				//descriptor for data, used for legend etc
	TextValue *x_tv, *y_tv;			//TextValue object for ordinal axes

	Plot(GraphObj *, DataObj *);
	virtual ~Plot(){return;};
	void DefName(char* first, AccRange *rX, AccRange *rY);
	virtual double GetSize(int select);
	virtual bool SetSize(int, double){return false;};
	virtual DWORD GetColor(int select);
	virtual bool SetColor(int, DWORD){return false;};
	virtual void DoPlot(anyOutput *){return;};
	virtual bool Command(int, void *, anyOutput *){return false;};
	virtual bool PropertyDlg(){return false;};
	virtual void RegGO(void *){return;};
	virtual bool FileIO(int) {return false;};

	virtual void CheckBounds(double x, double y);
	void CheckBounds3D(double x, double y, double z);
	bool UseAxis(int idx);
	void ApplyAxes(anyOutput *o);
	bool SavVarObs(GraphObj **gol, long ngo, DWORD flags);
	DataObj *CreaCumData(char *xr, char *yr, int mode, double base);
};

class PlotScatt:public Plot{
public:
	char *xRange, *yRange;
	long nPoints, nBars, nErrs, nLabel, nDrops, nArrow;
	int DefSym;
	lfPOINT BarDist;
	Bar **Bars;
	Symbol **Symbols;
	Arrow **Arrows;
	DataLine *TheLine;
	ErrorBar **Errors;
	Label **Labels;
	ErrorPolygon *ErrPg;
	double bar_dx;

	PlotScatt(GraphObj *, DataObj *, DWORD presel);
	PlotScatt(GraphObj *, DataObj *);
	PlotScatt(GraphObj *, DataObj *, int cBars, Bar **bars, ErrorBar **errs);
	PlotScatt(GraphObj *, DataObj *, int nPts, Symbol **sym, DataLine *lin, int nB=0, Bar **bar = NULL);
	PlotScatt(int src);
	virtual ~PlotScatt();
	virtual double GetSize(int select);
	virtual bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *target);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	virtual bool PropertyDlg();
	void RegGO(void *n);
	virtual bool FileIO(int rw);
	virtual void *ObjThere(int x, int y);

	bool ForEach(int cmd, void *tmp, anyOutput *o);

private:
	DWORD DefSel;
	char *ErrRange, *LbRange;
	DropLine **DropLines;
	lfPOINT lbDist;

	bool CreateBarChart();
};

class xyStat:public PlotScatt{
public:
	xyStat(GraphObj *, DataObj *);
	xyStat(int src);
	~xyStat();
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	virtual bool FileIO(int rw);

	void CreateData();
	unsigned char *CopySylk();

private:
	double ci;
	DataObj *curr_data;
	char *case_prefix;

};

class BarChart:public PlotScatt{
public:
	BarChart(GraphObj *, DataObj *);
};

class FreqDist:public Plot {
public:
	FreqDist(GraphObj *, DataObj *);
	FreqDist(GraphObj *, DataObj *, char* range, bool bOnce);
	FreqDist(GraphObj *, DataObj *, double *vals, int nvals, int nclasses);
	FreqDist(int src);
	~FreqDist();
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	double dmin, dmax, start, step;
	long nPlots;
	char *ssRef;
	LineDEF BarLine, HatchLine, FuncLine;
	FillDEF BarFill;
	DataObj *curr_data;
	GraphObj **plots;

	void ProcData(int sel);
};

class Regression:public Plot{
public:
	Regression(GraphObj *, DataObj *);
	Regression(int src);
	~Regression();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *target);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	long nPoints;
	char *xRange, *yRange;
	Symbol **Symbols;
	RegLine *rLine;
	SDellipse *sde;

	void Recalc();
};

class myRegression:public Plot {
public:
	myRegression(GraphObj *, DataObj *);
	myRegression(int src);
	~myRegression();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *target);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	long nPoints, nErrs;
	double ci;
	char *xRange, *yRange;
	Symbol **Symbols, *PrevSym;
	ErrorBar *PrevErr;
	ErrorBar **Errors;
	GraphObj *func;						//points to Function object
	GraphObj *ci_line1, *ci_line2;		//  and the confidence intervall
	Label *reg_res;
	LineDEF PrevLine, PrevCiLine;

	bool Recalc();

};

class myBarPlot :public PlotScatt {
public:
	myBarPlot(GraphObj *, DataObj *);
	myBarPlot(int src);
	~myBarPlot();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	Symbol *PrevSym;
	ErrorBar *PrevErr;
	Bar *PrevBar;
	LineDEF PrevLine;

	bool Recalc();

};

class BubblePlot:public Plot{
public:
	BubblePlot(GraphObj *, DataObj *);
	BubblePlot(int src);
	~BubblePlot();
	void DoPlot(anyOutput *target);
	DWORD GetColor(int select);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	long nPoints;
	LineDEF BubbleLine, BubbleFillLine;
	FillDEF BubbleFill;
	Bubble **Bubbles;
};

class PolarPlot:public Plot{
public:
	PolarPlot(GraphObj *, DataObj *);
	PolarPlot(int src);
	~PolarPlot();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void *ObjThere(int x, int y);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);
	anyOutput* getDisp(){ return CurrDisp; };

	bool AddPlot();
	bool Config();

private:
	long nPlots, nAxes;
	double offs;
	anyOutput *CurrDisp;
	fRECT CurrRect;
	LineDEF FillLine;
	FillDEF Fill;
	Plot **Plots;
	GraphObj **Axes;			//Axis not yet defined
};

class TernaryPlot:public Plot{
public:
	TernaryPlot(GraphObj *, DataObj *);
	TernaryPlot(int src);
	~TernaryPlot();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);
	anyOutput* getDisp(){ return CurrDisp; };

	bool AddPlot();
	bool Config();

private:
	long nPlots, nAxes;
	anyOutput *CurrDisp;
	fRECT CurrRect;
	LineDEF FillLine;
	FillDEF Fill;
	Plot **Plots;
	GraphObj **Axes;	
};

class TernaryXYZ:public PlotScatt{
public:
	TernaryXYZ(GraphObj *, DataObj *);
	TernaryXYZ(GraphObj *, DataObj *, Symbol **syms, long nsyms);
	TernaryXYZ(int src);
	~TernaryXYZ();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);
	void *ObjThere(int x, int y);
	anyOutput* getDisp(){ return CurrDisp; };

	bool AddPlot();
	bool Config();
	bool TransformVal(anyOutput *o, fPOINT3D *lfp, double *ix, double *iy); 

private:
	long nPlots, nAxes, doPG;
	double ang_disp;
	lfPOINT cent;
	double rad_l, rad_s;
	anyOutput *CurrDisp;
	fRECT CurrRect;
	LineDEF FillLine, Outline;
	FillDEF Fill;
	Plot **Plots;
	GraphObj **Axes;
};

class BoxPlot:public Plot {
public:
	BoxPlot(GraphObj *, DataObj *);
	BoxPlot(int src);
	BoxPlot(GraphObj *, DataObj *, int mode, int c1, int c2, int c3, char *box_name);
	~BoxPlot();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

	bool ForEach(int cmd, void *tmp, anyOutput *o);
	void CreateData();
	void CopyData1();
	char *CopySylk();

private:
	char *xRange, *yRange, *case_prefix;
	long nPoints, nBoxes, nWhiskers, nLabel;
	double ci_box, ci_err;
	DataObj *curr_data;
	lfPOINT BoxDist;
	Box **Boxes;
	Whisker **Whiskers;
	Symbol **Symbols;
	Label **Labels;
	DataLine *TheLine;
	ErrorPolygon *ErrPg;
};

class DensDisp:public Plot {
public:
	DensDisp(GraphObj *, DataObj *);
	DensDisp(int src);
	~DensDisp();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	LineDEF DefLine, DefFillLine;
	FillDEF DefFill;
	long nPoints;
	char *xRange, *yRange;
	Box **Boxes;

	void DoUpdate();

};

class StackBar:public Plot{
public:
	long numPlots, numXY, numPG, numPL;
	BoxPlot **Boxes;
	PlotScatt **xyPlots;
	DataPolygon **Polygons;
	DataLine **Lines;
	lfPOINT dspm;

	StackBar(GraphObj *, DataObj *);
	StackBar(int src);
	virtual ~StackBar();
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	char *ssXrange, *ssYrange;
	double StartVal;
	int cum_data_mode;
	DataObj *CumData;
};

class Phenology:public Plot {
public:
	Phenology(GraphObj *, DataObj *);
	Phenology(int src);
	~Phenology();
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

	int canAddPlot() {	return selRC_curr;};

private:
	char *ssXrange, *ssYrange;
	Plot **Plots;
	long nPlots;
	long selRC_size;
	RECT *selRC_rec;
	int selRC_curr, selRC_ref;
};

class StackPG:public StackBar{
public:
	StackPG(GraphObj *par, DataObj *d);
};

class GroupBars:public StackBar{
public:
	GroupBars(GraphObj *par, DataObj *d, int type):StackBar(par, d){mode = type;};
	bool PropertyDlg();

private:
	int mode;
};

class Waterfall:public StackBar{
public:
	Waterfall(GraphObj *, DataObj *);
	bool PropertyDlg();
};

class MultiLines:public StackBar{
public:
	MultiLines(GraphObj *, DataObj *);
	bool PropertyDlg();
};

class PieChart:public Plot {
public:
	PieChart(GraphObj *, DataObj *);
	PieChart(int src);
	virtual ~PieChart();
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);
	void DoUpdate();

private:
	long nPts;
	char *ssRefA, *ssRefR;
	lfPOINT CtDef;
	double FacRad;
	segment **Segments;
};

class RingChart:public PieChart {
public:
	RingChart(GraphObj *, DataObj *);
};

class GoGroup:public Plot {
public:
	GraphObj **Objects;
	long nObs;
	lfPOINT fPos;

	GoGroup(GraphObj *, DataObj *);
	GoGroup(int src);
	virtual ~GoGroup();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void RegGO(void *n);
	bool FileIO(int rw);
};

class StarChart:public GoGroup {
public:
	StarChart(GraphObj *, DataObj *);
	bool PropertyDlg();

private:
};

class Ribbon:public Plot {
public:
	Ribbon(GraphObj *, DataObj *, double z, double width, char *xr, char *yr);
	Ribbon(GraphObj *, DataObj *, int which, char *xr, char *yr, char *zr);
	Ribbon(GraphObj *, DataObj *, GraphObj **go, int ngo);
	Ribbon(int src);
	~Ribbon();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void RegGO(void *n);
	bool FileIO(int rw);
	bool HasPlanes() {return (nPlanes != 0);};

private:
	long nPlanes, nVal;
	double z_value, z_width, relwidth;
	char *ssRefX, *ssRefY, *ssRefZ;
	FillDEF Fill;
	LineDEF Line;
	fPOINT3D *values;
	Plane3D **planes;

	void CreateObs();
	void UpdateObs(bool bNewData);
};

class Grid3D:public Plot {
public:
	Grid3D(GraphObj *, DataObj *, int sel, double x1=0.0, double xstep=1.0, double z1=0.0, double zstep=1.0);
	Grid3D(int src);
	~Grid3D();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

	void CreateObs(bool set_undo);

private:
	long nLines, nPlanes;
	fPOINT3D start, step;
	LineDEF Line;
	FillDEF Fill;
	Line3D **lines;
	Plane3D **planes;

	bool Configure();
};

class Scatt3D:public Plot{
public:
	Scatt3D(GraphObj *, DataObj *, DWORD flags);
	Scatt3D(GraphObj *, DataObj *, Brick **cols, long nob);
	Scatt3D(GraphObj *, DataObj *, Sphere **ba, long nob);
	Scatt3D(int src);
	~Scatt3D();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	long nBalls, nColumns, nDropLines, nArrows, nErrors;
	DWORD c_flags;
	char *ssRefX, *ssRefY, *ssRefZ;
	Line3D *Line;
	Sphere **Balls;
	Brick **Columns;
	DropLine3D **DropLines;
	Arrow3D **Arrows;
	ErrBar3D **Errors;
	Ribbon *rib;
};

class Limits:public Plot {
public:
	Limits(int src);
	~Limits();
	double GetSize(int select);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool FileIO(int rw);
};

class Function:public Plot{
public:
	char *cmdxy;

	Function(GraphObj *, DataObj *, char *desc);
	Function(int src);
	~Function();
	bool SetSize(int select, double value);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

	void CheckBounds(double x, double y);
	bool Update(anyOutput *o, DWORD flags);
	LineDEF *GetLine() {return &Line;};

private:
	double x1,x2, xstep;
	LineDEF Line;
	char *param;
	DataLine *dl;
};

class FitFunc:public Plot{
public:
	FitFunc(GraphObj *, DataObj *);
	FitFunc(int src);
	~FitFunc();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	double x1, x2, xstep, conv, chi2;
	char *ssXref, *ssYref;
	long nPoints, nVals;
	int maxiter, method;
	LineDEF Line;
	Symbol **Symbols;
	char *cmdxy, *parxy;
	Function *dl;
};

class NormQuant:public Plot{
public:
	NormQuant(GraphObj *, DataObj *, char* range);
	NormQuant(GraphObj *, DataObj *, double *data, int ndata, char *Ref = 0L);
	NormQuant(int src);
	~NormQuant();
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	char *ssRef;
	int nData, nValidData;
	double *x_vals, *y_vals, *src_data;
	Symbol *sy;

	bool ProcessData();
};

	
class GridLine:public GraphObj{
public:
	DWORD flags;
	long ncpts;
	POINT pts[6], *cpts;
	LineDEF LineDef;
	line_segment **ls;
	bool bModified;
	anyOutput *mo;
	RECT mrc;

	GridLine(GraphObj *, DataObj *, int type, DWORD df);
	GridLine(int src);
	virtual ~GridLine();
	virtual void DoPlot(anyOutput *o);
	virtual void DoMark(anyOutput *o, bool mark);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	bool FileIO(int rw);

private:
	POINT3D *gl1, *gl2;
};

class GridLine3D:public GridLine {
public:
	GridLine3D(GraphObj *, DataObj *, int type, DWORD df);
	GridLine3D(int src);
	~GridLine3D();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);

private:
	fPOINT3D *gl1, *gl2, *gl3;
};

class GridRadial:public GridLine {
public:
	GridRadial(GraphObj *, DataObj *, int type, DWORD df);
	GridRadial(int src);
	~GridRadial();
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
};

class Tick:public GraphObj{
public:
	Tick(GraphObj *, DataObj *, double val, DWORD Flags);
	Tick(int src);
	~Tick();
	double GetSize(int select);
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);
	void Track(POINT *p, anyOutput *o, bool accept);

	void DoPlot(double six, double csx, anyOutput *o);
	bool ProcSeg();
	anyOutput *getDisp(){ return (parent ? parent->getDisp() : 0L); };

private:
	double value, size, fix, fiy, fiz, lsi, lcsi, angle, lbx, lby;
	int n_seg, s_seg;
	double *seg;
	bool bModified, bValidTick;
	long numPG;
	anyOutput *mo;
	RECT mrc;
	int gl_type;
	GridLine *Grid;
	Label *label;
	DataPolygon **Polygons;
	DWORD flags;
	POINT pts[2];
	line_segment *ls;

	bool CmpPoints(double x1, double y1, double x2, double y2);
	bool StoreSeg(lfPOINT *line);
};

class Axis:public GraphObj{
public:
	AxisDEF *axis;
	LineDEF axline;
	TextValue *atv;
	long NumTicks;
	Tick **Ticks;
	char *TLBformat, *tmpTLBformat;

	Axis(GraphObj *, DataObj *, AxisDEF *ax, DWORD flags);
	Axis(int src);
	~Axis();
	double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int select);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);
	void *ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);

	AxisDEF *GetAxis() {return axis;};
	bool GetValuePos(double val, double *fix, double *fiy, double *fiz, anyOutput *o);
	void TickFile(char *name);
	void BreakSymbol(fPOINT3D *p1, double dsi, double dcsi, bool connect, anyOutput *o);
	void DrawBreaks(anyOutput *o);
	void CreateTicks();
	DWORD GradColor(double value);
	anyOutput *getDisp(){ return (parent ? parent->getDisp() : 0L); };

private:
	double sizAxLine, sizAxTick, sizAxTickLabel;
	double si, csi;
	double brksymsize, brkgap, tick_angle;
	int brksym, nl_segs, gl_type, tick_type, grad_type;
	bool bModified;
	DWORD colAxis, gCol_0, gCol_1, gCol_2, gTrans;
	LineDEF GridLine;
	GraphObj *axisLabel;
	POINT pts[2], gradient_box[5];
	fPOINT3D pts3D[2];
	fPOINT3D flim[2];
	lfPOINT lbdist, tlbdist;
	TextDEF tlbdef;
	anyOutput *drawOut, *scaleOut;
	line_segment **l_segs;
	char *ssMATval, *ssMATlbl, *ssMITval; 
	anyOutput *mo, *mo1;
	RECT mrc, mrc1;
	fRECT frm_rc;
	GraphObj **pIsoPG;				//pointers to isopleth polygons
	long sIsoPG, nIsoPG;			//size of isopleth buffer and size of buffer
	FrmRect *frm_a;					//for tracking

	void SetTick(long idx, double val, DWORD flags, char *txt);
	void mkTimeAxis();
	void ManuTicks(double sa, double st, int n, DWORD flags);
	void UpdateTicks();
	bool ssTicks();
	void GradientBar(anyOutput *o);
};

class ContourPlot:public Plot {
public:
	ContourPlot(GraphObj *, DataObj *);
	ContourPlot(int src);
	~ContourPlot();
	bool SetSize(int select, double value);
	bool SetColor(int select, DWORD col);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	char *ssRefX, *ssRefY, *ssRefZ;
	Symbol **Symbols;
	Label **Labels;
	long nval, nSym, nLab;
	DWORD flags;
	double sr_zval;
	fPOINT3D *val;
	AxisDEF z_axis;
	Axis *zAxis;

	bool LoadData(char *xref, char *yref, char *zref);
	bool DoTriangulate();
	bool DoAxis(anyOutput *o);
	void DoSymbols(Triangle *trl);
};

class Plot3D:public Plot{
	typedef struct {
		double Zmin, Zmax;
		GraphObj *go;
		}obj_desc;
public:
	long nPlots, nAxes;
	Axis **Axes;
	GraphObj **plots, **Sc_Plots;
	double *RotDef;
	fPOINT3D cub1, cub2, rotC;
	int nscp;

	Plot3D(GraphObj *, DataObj *, DWORD flags);
	Plot3D(int src);
	virtual ~Plot3D();
	double GetSize(int select);
	bool SetColor(int select, DWORD col);
	virtual void DoPlot(anyOutput *o);
	void DoMark(anyOutput *o, bool mark);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	virtual bool PropertyDlg();
	virtual void RegGO(void *n);
	virtual bool FileIO(int rw);
	anyOutput* getDisp(){ return parent ? parent->getDisp() : 0L; };

	void * ObjThere(int x, int y);
	void Track(POINT *p, anyOutput *o, bool accept);
	void CreateAxes();
	void DoAutoscale();
	void CalcRotation(double dx, double dy, anyOutput *o, bool accept);
	bool AcceptObj(GraphObj *go);
	void SortObj();
	bool Rotate(double dx, double dy, double dz, anyOutput *o, bool accept, int direction);
	bool AddAxis();
	bool Configure();

private:
	long nObs, nmaxObs;
	DWORD crea_flags;
	Drag3D *drag;
	fPOINT3D cu1, cu2, rc;
	lfPOINT light_source;
	obj_desc *dispObs;
	LineDEF track_line;
	anyOutput *CurrDisp, *mo;
	RECT mrc;
	int LastDirect;

	bool AddPlot(int family);
};

class Chart25D:public Plot3D {
public:
	Chart25D(GraphObj *, DataObj *, DWORD flags);
	~Chart25D();
	bool PropertyDlg();

private:
	fPOINT3D dspm;
};

class Ribbon25D:public Plot3D {
public:
	Ribbon25D(GraphObj *, DataObj *, DWORD flags);
	~Ribbon25D();
	bool PropertyDlg();

private:
	fPOINT3D dspm;
};

class BubblePlot3D:public Plot3D {
public:
	BubblePlot3D(GraphObj *, DataObj *);
	~BubblePlot3D();
	bool PropertyDlg();
};

class Func3D:public Plot3D {
public:
	DataObj *gda;
	Grid3D  *gob;

	Func3D(GraphObj *, DataObj *);
	Func3D(GraphObj *, DataObj *, double x1, double x2, double xstep, double z1, double z2, double zstep, char *cmd, char *pm);
	Func3D(int src);
	~Func3D();
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	bool Update();

	double x1, x2, xstep, z1, z2, zstep;
	LineDEF Line;
	FillDEF Fill;
	char *param, *cmdxy;
};

class FitFunc3D:public Plot3D {
public:
	FitFunc3D(GraphObj *, DataObj *);
	FitFunc3D(int src);
	~FitFunc3D();
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	bool FileIO(int rw);

private:
	bool Update();

	double x1, x2, xstep, z1, z2, zstep, conv, chi2;
	char *ssXref, *ssYref, *ssZref;
	long nVals;
	int maxiter;
	LineDEF Line;
	FillDEF Fill;
	char *param, *cmdxy;
	DataObj *gda;
	Grid3D  *gob;
};

class Graph:public GraphObj{
public:
	int ToolMode, nscp, tickstyle, bClip;
	fRECT GRect, DRect, Bounds;
	bool OwnDisp, bModified;
	fRECT CurrRect;
	GraphObj **Plots, *PasteObj;
	DWORD ColBG, ColAX;
	GraphObj **Sc_Plots;
	Axis **Axes;
	long NumAxes, NumPlots;
	char *filename;
	RECT ClipRC;

	Graph(GraphObj *, DataObj *, anyOutput *o, int style);
	Graph(int src);
	~Graph();
	virtual double GetSize(int select);
	bool SetSize(int select, double value);
	DWORD GetColor(int select);
	bool SetColor(int select, DWORD col);
	virtual void DoPlot(anyOutput *o);
	virtual bool Command(int cmd, void *tmpl, anyOutput *o);
	bool PropertyDlg();
	void RegGO(void *n);
	virtual bool FileIO(int rw);
	void *ObjThere(int x, int y);
	virtual double DefSize(int select);
	void CheckBounds(double x, double y);
	void CreateAxes(int templ, int numAxes = 0);
	anyOutput* getDisp(){ return CurrDisp; };

protected:
	anyOutput *Disp, *CurrDisp;
	Grid *theGrid;

private:
	int AxisTempl, zoom_level;
	double scale;
	RECT rcDim, rcUpd, rc_mrk;
	DWORD ColDR, ColGR, ColGRL;
	AxisDEF x_axis, y_axis;
	FrmRect *frm_g, *frm_d;
	bool dirty, bDialogOpen;
	POINT *tl_pts;
	long tl_nPts;
	ZoomDEF *zoom_def;

	bool AddPlot(int family);
	void DoAutoscale();
	bool ExecTool(MouseEvent *mev);
	bool Configure();
	bool AddAxis();
	bool MoveObj(int cmd, GraphObj *g); 
	bool DoZoom(char *z);
	bool DoScale(scaleINFO *sc, anyOutput *o);
	bool DoMouseEvent(int cmd, void *tmpl, anyOutput *o);
};

class Page:public Graph{
public:
	Page(GraphObj *, DataObj *);
	Page(int src);
	~Page();
	double GetSize(int select);
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	void RegGO(void *n);
	bool FileIO(int rw);
	double DefSize(int select);

private:
	LineDEF LineDef;
	FillDEF FillDef;

	bool Configure();
};

class ObjTree:public GraphObj {
public:
	ObjTree(GraphObj *, DataObj *, GraphObj *root);
	~ObjTree();
	void DoPlot(anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);

	anyOutput *CreateBitmap(int *w, int *h, anyOutput *tmpl);
	int count_lines(){ return count;};
	char *get_name(int line);
	int get_vis(int line);
	bool set_vis(int line, bool vis);
	bool set_movable(int line, bool bMov);
	GraphObj *get_obj(int line);

private:
	GraphObj **list, *base;
	int count, maxcount;
	TextDEF TextDef;
};

class notary{
public:
	notary();
	virtual ~notary();
	int RegisterGO(GraphObj *go);
	void AddRegGO(GraphObj *go);
	bool PushGO(unsigned int id, GraphObj *go);
	GraphObj *PopGO(unsigned int id);
	void FreeStack();
	bool ValPtr(void* ptr, bool add_remove);
	bool IsValidPtr(void* ptr);

private:
	unsigned int NextPopGO, NextPushGO, NextRegGO;
	GraphObj ***gObs;
	GraphObj ***goStack;
	void **valPtr_a;
	long valPtr_size, valPtr_pos;
};

class def_vars{
public:
	int dUnits;
	char DecPoint[2], ColSep[2];
	char *currPath, *IniFile;
	unsigned char *File1, *File2, *File3, *File4, *File5, *File6;
	char *fmt_date, *fmt_time, *fmt_datetime;
	double min4log, ss_txt;
	double pg_width, pg_height;				//current paper size and page default
	RECT clipRC;
	POINT3D DlgSizeAdj;

	def_vars();
	~def_vars();
	void NoAmp(unsigned char *txt);
	void SetDisp(anyOutput *o);
	double GetSize(int select);
	DWORD Color(int select);
	LineDEF *GetLine();
	void SetLine(int u, LineDEF *l, int which);
	FillDEF *GetFill(int units);
	void SetFill(int u, FillDEF *fd);
	LineDEF *GetOutLine();
	bool PropertyDlg();
	LineDEF *plLineDEF(LineDEF *ld);
	LineDEF *pgLineDEF(LineDEF *ol);
	FillDEF *pgFillDEF(FillDEF *fd);
	double rrectRad(double rad);
	void FileHistory(char *path);
	void Idle(int cmd);
	void UpdRect(anyOutput *o, int x1, int y1, int x2, int y2);
	void UpdRect(anyOutput *o, RECT *rc);
	void UpdAdd(anyOutput * o, int x, int y);
	void FreeOut(anyOutput *o);
	LineDEF *GetGridline();

private:
	LineDEF Line_0, Line_1, Line_2, *pl, *pgl;
	FillDEF Fill_0, Fill_1, Fill_2, *pg;
	LineDEF FillLine_0, FillLine_1, FillLine_2, *pg_fl;
	LineDEF OutLine_0, OutLine_1, OutLine_2;
	double *rrect_rad;
	anyOutput *cdisp, *out_upd;
	RECT rec_upd;
	bool can_upd, upd_suspend;
	DWORD axis_color;
};

class DefsRW:public GraphObj{
public:
	DefsRW():GraphObj(0, 0){Id = 0; return;};
	DefsRW(int rw):GraphObj(0,0){FileIO(rw);Id=GO_DEFRW;return;};
	~DefsRW() {return;};
	bool FileIO(int rw);
};

class tagProgBarDlg:public GraphObj{
public:
	int ti_count, state;

	tagProgBarDlg():GraphObj(0, 0){Id = 0; ti_count = state = 0; return;};
	~tagProgBarDlg() {return;};
	virtual void Timer() {return;};
	virtual void Start(long, long *, char *){return;};

};

class ReadCache{
public:
	unsigned char last, *Cache, Line[4096];
	int iFile;
	long idx, max;
	long curr_pos, max_pos;	//for progress bar
	bool eof;

	ReadCache();
	virtual ~ReadCache();
	virtual bool Open(char *name);
	virtual void Close();
	virtual unsigned char Getc();
	virtual unsigned char *GetField();
	void ReadLine(char *dest, int size);
	bool GetInt(long *in);
	bool GetFloat(double *fn);
	unsigned char Lastc();
	bool IsEOF();
};

class MemCache:public ReadCache{
public:
	MemCache(unsigned char *ptr);
	virtual ~MemCache();
	bool Open(char *){return false;};
	void Close(){return;};
	unsigned char Getc();
	unsigned char *GetField();
};

class DrawLaterObj {
public:
	DrawLaterObj();
	~DrawLaterObj();
	bool AddObj(GraphObj * obj);
	bool DoPlot(anyOutput *o);
	void Clear();

private:
	GraphObj **buff;
	int buff_size, buff_pos;
};

#define UNDO_CONTINUE 0x01
#define UNDO_STORESET 0x1000
class UndoObj {
	enum {UNDO_UNDEFINED, UNDO_DEL_GO, UNDO_GOLIST, UNDO_DROPMEM,
		UNDO_VALDWORD, UNDO_VALINT, UNDO_VALLONG, UNDO_OBJCONF, UNDO_OBJCONF_1,
		UNDO_LFP, UNDO_POINT, UNDO_VOIDPTR, UNDO_MOVE, UNDO_RECT, UNDO_STRING, 
		UNDO_ETSTRING, UNDO_ROTDEF, UNDO_SETGO, UNDO_LINEDEF, UNDO_FILLDEF,
		UNDO_AXISDEF, UNDO_LFP3D, UNDO_FLOAT, UNDO_MEM, UNDO_MUTATE, UNDO_DROPGOLIST, 
		UNDO_TEXTDEF, UNDO_SAVVAR, UNDO_DATA, UNDO_ET, UNDO_TEXTBUF, UNDO_SEPARATOR};
	typedef struct _UndoInfo {
		int cmd;
		DWORD flags;
		GraphObj *owner;
		void *data;
		void **loc;
		ZoomDEF zd;
		}UndoInfo;

	typedef struct _UndoList {
		void *array;
		void **loc_arr;
		long count, size;
		long *loc_count;
		}UndoList;

	typedef struct _UndoBuff {
		int count;
		UndoInfo **buff;
		anyOutput *disp;
		}UndoBuff;

	typedef struct _EtBuff {
		char *txt;
		DataObj *DaO;
		int *cur, *m1, *m2, vcur, vm1, vm2, row, col;
		}EtBuff;

	typedef struct _TextBuff {
		long *psize, size, *ppos, pos;
		unsigned char **pbuff, *buff;
		}TextBuff;

public:
	UndoObj();
	~UndoObj();
	void Flush();
	bool isEmpty(anyOutput *o);
	void SetDisp(anyOutput *o);
	void KillDisp(anyOutput *o);
	void InvalidGO(GraphObj *go);
	void Pop(anyOutput *o);
	void Restore(bool redraw, anyOutput *o);
	void ListGOmoved(GraphObj **oldlist, GraphObj **newlist, long size);
	void DeleteGO(GraphObj **go, DWORD flags, anyOutput *o);
	void MutateGO(GraphObj **old, GraphObj *repl, DWORD flags, anyOutput *o);
	void StoreListGO(GraphObj *parent, GraphObj ***go, long *count, DWORD flags);
	void DropListGO(GraphObj *parent, GraphObj ***go, long *count, DWORD flags);
	void DropMemory(GraphObj *parent, void **mem, DWORD flags);
	void SavVarBlock(GraphObj *parent, void **mem, DWORD flags);
	void ValDword(GraphObj *parent, DWORD *val, DWORD flags);
	void Point(GraphObj *parent, POINT *pt, anyOutput * o, DWORD flags);
	void VoidPtr(GraphObj *parent, void **pptr, void *ptr, anyOutput * o, DWORD flags);
	void ValInt(GraphObj *parent, int *val, DWORD flags);
	void ValLong(GraphObj *parent, long *val, DWORD flags);
	void ObjConf(GraphObj *go, DWORD flags);
	int SaveLFP(GraphObj *go, lfPOINT *lfp, DWORD flags);
	void MoveObj(GraphObj *go, lfPOINT *lfp, DWORD flags);
	void ValRect(GraphObj *go, fRECT *rec, DWORD flags);
	int String(GraphObj *go, char **s, DWORD flags);
	int EtString(EditText *go, char **s, DWORD flags);
	void RotDef(GraphObj *go, double **d, DWORD flags);
	void SetGO(GraphObj *parent, GraphObj **where, GraphObj *go, DWORD flags);
	void Line(GraphObj *go, LineDEF *ld, DWORD flags);
	void Fill(GraphObj *go, FillDEF *fd, DWORD flags);
	void AxisDef(GraphObj *go, AxisDEF *ad, DWORD flags);
	void TextDef(GraphObj *go, TextDEF *td, DWORD flags);
	void ValLFP3D(GraphObj *go, fPOINT3D *lfp, DWORD flags);
	void ValFloat(GraphObj *parent, double *val, DWORD flags);
	void DataMem(GraphObj *go, void **mem, int size, long *count, DWORD flags);
	void DataObject(GraphObj *go, anyOutput *o, DataObj *d, RECT *rc, DWORD flags);
	void TextBuffer(GraphObj *parent, long *psize, long *ppos, unsigned char **pbuff, DWORD flags, anyOutput *o); 
	void Separator();
	anyOutput * Cdisp(){ return cdisp; };
	bool isBusy(){ return busy; };
	int level(){ return *pcb; };

private:
	UndoInfo **buff, **buff0;
	int stub1, ndisp;
	UndoBuff **buffers;
	anyOutput *cdisp, *ldisp;
	bool busy;
	int *pcb;

	int NewItem(int cmd, DWORD flags, GraphObj *owner, void *data, void **loc);
	void FreeInfo(UndoInfo** inf);
	void RestoreConf(UndoInfo *inf);
	void RestoreData(UndoInfo *inf);
};

//prototypes: spreadwi.cpp
int ProcMemData(GraphObj *g, unsigned char *ptr, bool dispatch);
void SpreadMain(bool show);
void sylk_cell_ref(char** ptr, long *cbd, long *size, char *first, char* trail, int row, int col);

//prototypes: WinSpec.cpp, QT_Spec.cpp, or Gtk_spec.cpp
char *SaveDataAsName(char *oldname);
char *SaveGraphAsName(char *oldname);
char *OpenDataName(char *oldname);
void InfoBox(char *Msg);
void ErrorBox(char *Msg);
bool YesNoBox(char *Msg);
int YesNoCancelBox(char *Msg);
void Qt_Box();
int rlp_mbtowc(w_char *wc, unsigned char *txt, int n);
void HideTextCursor();
void KillTextCursor();
void ShowTextCursor(anyOutput *out, RECT *disp, DWORD color);
void RedrawTextCursor();
void HideCopyMark(bool bUpd);
void ShowCopyMark(anyOutput *out, RECT *mrk, int nRec);
void InvalidateOutput(anyOutput *o);
void SuspendAnimation(anyOutput *o, bool bSusp);
void InitTextCursor(bool init);
void TestClipboard(GraphObj *g);
void EmptyClip();
void CopyText(char *txt, int len, unsigned int cf);
void CopyData(GraphObj *g, unsigned int cf);
unsigned char* PasteText();
void GetDesktopSize(int *width, int *height);
void FindBrowser();
void LoopDlgWnd();
bool NoWaitDlgLoop();
void CloseDlgWnd(void *hDlg);
void ShowDlgWnd(void *hDlg);
void ResizeDlgWnd(void *hDlg, int w, int h);
anyOutput *NewDispClass(GraphObj *g);
bool DelDispClass(anyOutput *w);
anyOutput *NewBitmapClass(int w, int h, double hr, double vr);
bool DelBitmapClass(anyOutput *w);
bool com_GradPG(fPOINT3D *pts, long npt, double Lo, double Hi, double pLo, double pHi, lfPOINT *grad, anyOutput *o);

//prototypes: FileIO.cpp
bool SaveGraphAs(GraphObj *g);
char *GraphToMem(GraphObj *g, long *size);
void UpdGOfromMem(GraphObj *go, unsigned char *buff);
bool OpenGraph(GraphObj *root, char *name, unsigned char *mem, bool bPaste);
void SavVarInit(long len);
void *SavVarFetch();

//prototypes:TheDialog.cpp
DWORD GetNewColor(DWORD oldcol);
bool ShowLayers(GraphObj *root);
void ShowBanner(bool show);
void RLPlotInfo();
bool DoSpShSize(DataObj *dt, GraphObj *parent);
bool FillSsRange(DataObj *d, char **range, GraphObj *msg_go);
bool ArithSsRange(DataObj *d, char **range, GraphObj *msg_go);
bool GetBitmapRes(double *res, double *width, double *height, char *header);
void OD_scheme(int, void *, RECT *, anyOutput *, void *, int);
FillDEF *GetSchemeFill(int *i);
void OD_linedef(int, void *, RECT *, anyOutput *o, void *, int);
void OD_filldef(int, void *, RECT *, anyOutput *o, void *, int);
void OD_paperdef(int, void *, RECT *, anyOutput *o, void *, int);
void FindPaper(double w, double h, double tol);
bool GetPaper(double *w, double *h);
void OD_axisplot(int, void *, RECT *, anyOutput *o, void *, int);
void ScheduleProgBar();
void KillProgBar();

//prototypes: Utils.cpp
anyOutput *GetRectBitmap(RECT *rc, anyOutput *src, bool invert = false);
void RestoreRectBitmap(anyOutput **pmo, RECT *mrc, anyOutput *o, bool invert = false);
void MemoryError(anyOutput *o, bool bReset);
void NiceAxis(AxisDEF *axis, int nTick);
void NiceStep(AxisDEF *axis, int nTick);
double base4log(AxisDEF *axis, int direc);
double TransformValue(AxisDEF *axis, double val, bool transform);
void SortAxisBreaks(AxisDEF *axis);
double GetAxisFac(AxisDEF *axis, double delta, int direc);
char *str_ltrim(char *str);
char *str_rtrim(char *str);
char *str_trim(char *str);
void rmquot(char *str);
int strpos(char *needle, char *haystack);
char *strreplace(char *needle, char *replace, char *haystack);
char *substr(char *text, int pos1, int pos2);
//int rlp_strcpy(char*dest, int size, char*src);
int rlp_strcpy(void *dest, int size, void *src);
int rlp_strlen(char* txt);
int rlp_strlen(unsigned char* txt);
int rlp_strlenW(w_char *txt);
unsigned char *rlp_strdup(unsigned char *src);
char *rlp_strdup(char *src);
int hasUTF(unsigned char *txt, int *cpc);
void ReshapeFormula(char **text);
void TranslateResult(anyResult *res);
void CleanTags(char *txt, int *i1, int *i2, int *i3);
void ChangeChar(char *text, char c1, char c2);
char *Int2Nat(char *Text);
char *Nat2Int(char *Text);
void WriteNatFloatToBuff(char *buff, double val, char *fmt = 0L);
bool Txt2Flt(char *txt, double *val);
void RmTrail(char *txt);
double NiceValue(double fv);
char *NiceTime(double val);
char *Int2ColLabel(int nr, bool uc);
char *mkCellRef(int row, int col);
char *mkRangeRef(int r1, int c1, int r2, int c2);
unsigned char *str2xml(unsigned char *str, bool bGreek);
char **split(char *str, char sep, int *nl);
unsigned char *fit_num_rect(anyOutput *o, int max_width, unsigned char *num_str);
void add_to_buff(char** dest, long *pos, long *csize, char *txt, int len);
void add_int_to_buff(char** dest, long *pos, long *csize, int value, bool lsp);
void add_long_to_buff(char** dest, long *pos, long *csize, long value, bool lsp);
void add_dbl_to_buff(char** dest, long *pos, long *csize, double value, bool lsp, char *format = 0L);
void add_hex_to_buff(char** dest, long *pos, long *csize, DWORD value, bool lsp);
void SetMinMaxRect(RECT *rc, int x1, int y1, int x2, int y2);
void IncrementMinMaxRect(RECT *rc, int i);
bool IsCloseToLine(POINT *p1, POINT *p2, long x, long y, double tolerance);
bool IsCloseToPL(POINT p, POINT *pts, long cp, double tolerance);
bool IsCloseToPL(lfPOINT p, lfPOINT *pts, long cp, double tolerance);
bool IsInPolygon(POINT *p, POINT *pts, long cp, double tolerance);
bool IsInPolygon(lfPOINT *p, lfPOINT *pts, long cp, double tolerance);
bool OverlapRect(RECT *rc1, RECT *rc2);
void AddToPolygon(long *cp, POINT *pts, POINT *np);
void AddToPolygon(long *cp, POINT *pts, lfPOINT *np);
void DrawBezier(long *cp, POINT *pts, POINT p0, POINT p1, POINT p2, POINT p3, int depth = 0);
void ClipBezier(long *cp, POINT *pts, POINT p0, POINT p1, POINT p2, POINT p3, POINT *clp1, POINT *clp2);
int mkCurve(lfPOINT *src, int n1, lfPOINT **dst, bool bClosed);
POINT *MakeArc(int ix, int iy, int r, int qad, long *npts);
void InvertLine(POINT*, int, LineDEF*, RECT*, anyOutput*, bool);
unsigned int ColDiff(DWORD col1, DWORD col2);
DWORD IpolCol(DWORD color1, DWORD color2, double fact);
DWORD GradColor(int grad_type, double Lo, double Hi, double val);
double ran2(long *idum);
unsigned long isqr(unsigned long n);
bool MatMul(double a[3][3], double b[3][3], double c[3][3]);
char *GetNumFormat(double Magn);
void DeleteGO(GraphObj *go);
bool DeleteGOL(GraphObj ***gol, long n, GraphObj *go, anyOutput *o);
bool BackupFile(char *FileName);
bool IsRlpFile(char *FileName);
bool IsXmlFile(char *FileName);
bool FileExist(char *FileName);
bool IsPlot3D(GraphObj *g);
void *memdup(void *ptr, int cb_old, int cb_new);
double sininv(double val);
double trig2deg(double si, double csi);
bool ReplaceGO(GraphObj **oldobj, GraphObj **newobj);
unsigned int HashValue(unsigned char *str);
unsigned int Hash2(unsigned char * str);
bool cmpLineDEF(LineDEF *l1, LineDEF *l2);
bool cmpFillDEF(FillDEF *f1, FillDEF *f2);
bool cmpAxisDEF(AxisDEF *a1, AxisDEF *a2);
bool cmpTextDEF(TextDEF *t1, TextDEF *t2);
DWORD CheckNewFloat(double *loc, double old_v, double new_v, GraphObj *par, DWORD flags);
DWORD CheckNewInt(int *loc, int old_v, int new_v, GraphObj *par, DWORD flags);
DWORD CheckNewLong(long *loc, long old_v, long new_v, GraphObj *par, DWORD flags);
DWORD CheckNewDword(DWORD *loc, DWORD old_v, DWORD new_v, GraphObj *par, DWORD flags);
DWORD CheckNewLFPoint(lfPOINT *loc, lfPOINT *old_v, lfPOINT *new_v, GraphObj *par, DWORD flags);
DWORD CheckNewLFPoint3D(fPOINT3D *loc, fPOINT3D *old_v, fPOINT3D *new_v, GraphObj *par, DWORD flags);
DWORD CheckNewString(unsigned char **loc, unsigned char *s_old, unsigned char *s_new, GraphObj *par, DWORD flags);
void clip_line_sphere(GraphObj *par, fPOINT3D **li, double r, double cx, double cy, double cz);
void clip_line_plane(GraphObj *par, fPOINT3D **li, fPOINT3D *pg, int np, double *vec);
void clip_sphline_sphere(GraphObj *par, fPOINT3D *lim1, fPOINT3D *lim2, fPOINT3D *cent, 
	double r1, double r2, double cx, double cy, double cz);
void clip_sphline_plane(GraphObj *par, fPOINT3D *lim1, fPOINT3D *lim2, fPOINT3D *cent, 
	double r1, fPOINT3D *pg, int np, double *vec);
void clip_plane_plane(GraphObj *par, fPOINT3D *pg1, int n1, double *v1, fPOINT3D *pg2, 
	int n2, double *v2, POINT *m, int nm);
bool fnsplit(void *FileName, void* drive, void* dir, void *name, void *ext);
bool fnmerge(void *FileName, void* drive, void* dir, void *name, void *ext);


//prototypes Export.cpp
void DoExportSvg(GraphObj *g, char *FileName, DWORD flags);
char *DoCopySvg(GraphObj *g, long *buf_size);
void DoExportEps(GraphObj *g, char *FileName, DWORD flags);

//prototypes mfcalc.cpp
void LockData(bool lockExec, bool lockWrite);
char  *yywarn(char *txt, bool bNew);
bool do_xyfunc(DataObj *, double, double, double, char *, lfPOINT **, long *, char *);
bool do_func3D(DataObj *d, double x1, double x2, double xstep, double z1, double z2, double zstep, 
	char *expr, char *param);
void ClearRangeInfo();
anyResult *do_formula(DataObj *, char *);
bool MoveFormula(DataObj *d, char *of, char *nf, int nfsize, int dx, int dy, int r0, int c0);
long do_simplex(DataObj *d, char *rx, char *ry, char *rz, char **par, char *expr,
	double conv, long maxiter, long *nv, double *chi_2, bool bProgress);
int do_fitfunc(DataObj *d, char *rx, char *ry, char *rz, char **par, char *expr, double conv, int maxiter, long *nv,
	double *chi_2, double *x_data, double *y_data, double *z_data, long n_data,int *fit_nparam, char ***pnames, double **pvals);

//prototypes rlp_math.cp
long ExecSimplex(DataObj *d, double **s_data, int nd, double *s_par, unsigned char **s_pnam, int np,
	char *formula, double vg, int maxfit, double *chi_2, bool bProgress);
bool mrqmin(double *, double *, double *, int, double **, int, int *, int, double **, double **, double *,
	void (*funcs)(double, double, double **, double *, double *, int), double *);
bool Check_MRQerror();
bool matinv(double **a, int n, double *d);
bool hdiag(double **h, int n, double **u);
void SortArray(long n, double *vals);
void SortArray2(long n, double *a1, double *a2);
void SortFpArray(long n, lfPOINT *vals);
double *randarr(double *v0, int n, long *seed);
double *resample(double *v0, int n, long *seed);
void spline(lfPOINT *v, int n, double *y2);
double gammln(double x);
double factrl(int n);
double gammp(double a, double x);
double gammq(double a, double x);
double betai(double a, double b, double x);
double bincof(double n, double k);
double binomdistf(double k, double n, double p);
double betaf(double z, double w);
double errf(double x);
double errfc(double x);
double norm_dist(double x, double m, double s);
double norm_freq(double x, double m, double s);
double exp_dist(double x, double l, double s);
double exp_inv(double p, double l, double s);
double exp_freq(double x, double l, double s);
double lognorm_dist(double x, double m, double s);
double lognorm_freq(double x, double m, double s);
double chi_dist(double x, double df, double);
double chi_freq(double x, double df);
double t_dist(double t, double df, double);
double t_freq(double t, double df);
double pois_dist(double x, double m, double);
double f_dist(double f, double df1, double df2);
double f_freq(double f, double df1, double df2);
double weib_dist(double x, double shape, double scale);
double weib_freq(double x, double shape, double scale);
double geom_freq(double x, double p);
double geom_dist(double x, double p);
double hyper_freq(double k, double n0, double m, double n1);
double hyper_dist(double k, double n0, double m, double n1);
double cauch_dist(double x, double loc, double scale);
double cauch_freq(double x, double loc, double scale);
double logis_dist(double x, double loc, double scale);
double logis_freq(double x, double loc, double scale);
double ks_dist(int n, double d);
void swilk1(int n, double *v0, double (*func)(double, double, double), double p1, double p2, 
	bool bsorted, double *w, double *p);
void KolSmir(int n, double *v0, double (*func)(double, double, double),
	double p1, double p2, bool bsorted, double *d, double *p);
double distinv(double (*sf)(double, double, double), double df1, double df2, double p, double x0);
void d_quartile(int n, double *v, double *q1, double *q2, double *q3);
double d_variance(int n, double *v, double *mean = 0L, double *ss = 0L);
double d_amean(int n, double *v);
double d_kurt(int n, double *v);
double d_skew(int n, double *v);
double d_gmean(int n, double *v);
double d_hmean(int n, double *v);
double d_classes(DataObj *d, double start, double step, double *v, int nv, char *range);
double d_pearson(double *x, double *y, int n, char *dest, DataObj *data, double *ra);
double d_rank(int n, double *v, double v1);
void crank(int n, double *w0, double *s);
double d_spearman(double *x, double *y, int n, char *dest, DataObj *data, double *ra);
double d_kendall(double *x, double *y, int n, char *dest, DataObj *data, double *ra);
double d_regression(double *x, double *y, int n, char *dest, DataObj *data, double ra[]);
double d_covar(double *x, double *y, int n, char *dest, DataObj *data);
double d_utest(double *x, double *y, int n1, int n2, char *dest, DataObj *data, double *results);
double d_ttest(double *x, double *y, int n1, int n2, char *dest, DataObj *data, double *results);
double d_ttest2(double *x, double *y, int n, char *dest, DataObj *data, double *results);
double d_ftest(double *x, double *y, int n1, int n2, char *dest, DataObj *data, double *results);
bool do_anova1(int n, int *nv, double **vals, double **res_tab, double *gm, double **means, double **ss);
bool bartlett(int n, int *nc, double *ss, double *chi2);
bool levene(int type, int n, int *nv, double *means, double **vals, double *F, double *P);
double wprob(double w, double rr, double cc);
double ptukey(double q, double rr, double cc, double df, int lower_tail, int log_p);
double qtukey(double p, double rr, double cc, double df, int lower_tail, int log_p);
int year2aday(int y);
void add_date(rlp_datetime *base, rlp_datetime *inc);
char *date2text(rlp_datetime *dt, char *fmt);
double date2value(rlp_datetime *dt);
void parse_datevalue(rlp_datetime *dt, double dv);
bool date_value(char *desc, char *fmt, double *value);
char *value_date(double dv, char *fmt);
double now_today();
void split_date(double dv, int *y, int *mo, int *dom, int *dow, int *doy, int *h, int *m, double *s);
bool line_intersect(double x1, double y1, double x2, double y2, double x3, double y3,
	double x4, double y4, double *x, double *y, bool bBrack);
bool bToBaseRect(double x, double y, double z);
Triangle* Triangulate1(char *xr, char *yr, char *zr, DataObj *data);
void PlaneEquation(fPOINT3D *, int, double *);

//prototypes reports.cpp
void rep_anova(GraphObj *, DataObj *);
void rep_bdanova(GraphObj *, DataObj *);
void rep_twanova(GraphObj *, DataObj *);
void rep_fmanova(GraphObj *, DataObj *);
void rep_twoway_anova(GraphObj *, DataObj *);
void rep_kruskal(GraphObj *, DataObj *);
void rep_samplestats(GraphObj *, DataObj *);
void rep_pca(GraphObj *, DataObj *);
void rep_regression(GraphObj *, DataObj *);
void rep_twregression(GraphObj *, DataObj *);
void rep_robustline(GraphObj *, DataObj *);
void rep_twowaytable(GraphObj *, DataObj *);
void rep_compmeans(GraphObj *, DataObj *);
void rep_correl(GraphObj *, DataObj *, int style);
void rep_bootstrap_mean(GraphObj *, DataObj *);
void rep_bootstrap_correlation(GraphObj *, DataObj *);
void rep_bootstrap_regression(GraphObj *, DataObj *);
bool rep_BT_Function(GraphObj *, DataObj *, unsigned char *, unsigned char *, unsigned char *,
	unsigned char *,unsigned char *,long,int, double);

#endif //_RLPLOT_H
