//rlplot.cpp, Copyright 2000-2025 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include "rlp_strings.h"
#include "menu.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

extern tag_Units Units[];
extern char TmpTxt[];
extern def_vars defs;
extern clear_sym ClearSym;
extern notary *Notary;

GraphObj *CurrGO = 0L, *TrackGO = 0L, *CopyGO=0L;			//Selected Graphic Objects
Label *CurrLabel = 0L;
Graph *CurrGraph = 0L;
Axis **CurrAxes = 0L;
Plot *CurrPlot = 0L;
static Grid *CurrGrid = 0L;
long NumCurrAxes = 0;
dragHandle *CurrHandle = 0L;
UndoObj Undo;
DrawLaterObj DrawLater;
fmtText DrawFmtText;

int cGraphs = 0;
int cPlots = 0;
int cPages = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// grapic objects
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
GraphObj::GraphObj(GraphObj *par, DataObj *d)
{
	parent = par;	data = d;	Id = GO_UNKNOWN;
	type = moveable = 0;		name = 0L;
	rDims.left = rDims.right = rDims.top = rDims.bottom = 0;
	hidden = 0;
}

GraphObj::~GraphObj()
{
	if(name)free(name);
	name = 0L;
	if(CurrGO == this)	CurrGO = 0L;
	if(TrackGO == this)	TrackGO = 0L;
	if (CopyGO == this) CopyGO = 0L;
}

double
GraphObj::GetSize(int select)
{
	if(parent) return parent->GetSize(select);
	else return defs.GetSize(select);
}

DWORD 
GraphObj::GetColor(int select){
	return defs.Color(select);
}

void
GraphObj::RegGO(void *n)
{
	((notary*)n)->AddRegGO(this);
}

void *
GraphObj::ObjThere(int x, int y)
{
	if(IsInRect(rDims, x, y)) return this;
	else return 0L;
}

void
GraphObj::Track(POINT *, anyOutput *, bool)
{
}

double
GraphObj::DefSize(int select)
{
	if(parent) return parent->DefSize(select);
	else return defs.GetSize(select);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Show a background grid on a Graph
Grid::Grid(GraphObj *par):GraphObj(par, 0L)
{
	FileIO(INIT_VARS);
	Id = GO_GRID;
}

Grid::Grid(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_GRID;
}

Grid::~Grid()
{
	CurrGrid = 0L;
}

double
Grid::GetSize(int select)
{
	switch (select){
	default:
	case SIZE_MINE:
		return spacing;
		}
	return 0.0;
}

bool
Grid::SetSize(int select, double value)
{
	switch (select) {
	case SIZE_MINE:
		spacing = value;
		return true;
		}
	return false;
}

DWORD
Grid::GetColor(int)
{
	return line.color;
}

bool
Grid::SetColor(int, DWORD col)
{
	line.color = col;
	return true;
}

void
Grid::DoPlot(anyOutput *o)
{
	double x, x1, x2, y, y1, y2;
	lfPOINT pt[2];
	double size = 2.0;

	if (hidden || !o || !parent) return;
	x1 = parent->GetSize(SIZE_GRECT_LEFT);		x2 = parent->GetSize(SIZE_GRECT_RIGHT);
	y1 = parent->GetSize(SIZE_GRECT_TOP);		y2 = parent->GetSize(SIZE_GRECT_BOTTOM);
	o->SetLine(&line);
	size = o->un2fix(spacing) / 10.0;
	if (size > 5.0) size = 5.0;
	for (x = x1 + spacing; x < x2; x += spacing) {
		for (y = y1 + spacing; y < y2; y += spacing) {
			pt[0].fx = o->co2fix(x) - size;			pt[1].fx = o->co2fix(x) + size;
			pt[0].fy = pt[1].fy = o->co2fiy(y);
			if(pt[1].fx >0.0) o->foSolidLine(pt);
			pt[0].fy = o->co2fiy(y) - size;			pt[1].fy = o->co2fiy(y) + size;
			pt[0].fx = pt[1].fx = o->co2fix(x);
			if (pt[1].fy >0.0) o->foSolidLine(pt);
			}
		}
}
bool
Grid::Command(int cmd, void *tmpl, anyOutput *)
{
	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_GRID;
		if (tmpl) data = (DataObj *)tmpl;
		return true;
		}
	return false;
}

void
Grid::Snap(POINT *pt, anyOutput *o)		//snap to grid: integer  coordinates
{
	double x, x1, x2, y, y1, y2;
	int ix, iy, dx, dy;
	long d, dist;
	POINT new_pt;
	bool b_match = false;

	if (hidden || !pt || !o || !(type &0x01)) return;
	CurrGrid = this;
	x1 = parent->GetSize(SIZE_GRECT_LEFT);		x2 = parent->GetSize(SIZE_GRECT_RIGHT);
	y1 = parent->GetSize(SIZE_GRECT_TOP);		y2 = parent->GetSize(SIZE_GRECT_BOTTOM);
	dist = o->un2ix(spacing) + 2;			dist = dist * dist;
	for (x = x1 + spacing; x < x2; x += spacing) {
		for (y = y1 + spacing; y < y2; y += spacing) {
			ix = o->co2ix(x);			iy = o->co2iy(y);
			dx = ix - pt->x;			dy = iy - pt->y;
			d = dx * dx + dy * dy;
			if (d < dist) {
				dist = d;				new_pt.x = ix;	
				new_pt.y = iy;			b_match = true;
				}
			}
		}
	if(b_match) {
		pt->x = new_pt.x;				pt->y = new_pt.y;
		}
}

void
Grid::Snap(lfPOINT *pt, anyOutput *o)		//snap to grid: floating  coordinates
{
	double x, x1, x2, y, y1, y2;
	double d, dist, ix, iy, dx, dy;
	lfPOINT new_pt;
	bool b_match = false;

	if (hidden || !pt || !o || !(type & 0x01)) return;
	CurrGrid = this;
	x1 = parent->GetSize(SIZE_GRECT_LEFT);		x2 = parent->GetSize(SIZE_GRECT_RIGHT);
	y1 = parent->GetSize(SIZE_GRECT_TOP);		y2 = parent->GetSize(SIZE_GRECT_BOTTOM);
	dist = o->un2fix(spacing) + 2.0;			dist = dist * dist;
	for (x = x1 + spacing; x < x2; x += spacing) {
		for (y = y1 + spacing; y < y2; y += spacing) {
			ix = o->co2fix(x);			iy = o->co2fiy(y);
			dx = ix - pt->fx;			dy = iy - pt->fy;
			d = dx * dx + dy * dy;
			if (d < dist) {
				dist = d;				new_pt.fx = ix;
				new_pt.fy = iy;			b_match = true;
				}
			}
		}
	if (b_match) {
		pt->fx = new_pt.fx;				pt->fy = new_pt.fy;
	}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Symbols are graphic objects
Symbol::Symbol(GraphObj *par, DataObj *d, double x, double y, int which,
		int xc, int xr, int yc, int yr):GraphObj(par, d)
{
	//Symbols with no parent are part of a dialog
	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;		type = which;
	Id = GO_SYMBOL;		mo = 0L;
	if(xc >= 0 && xr >= 0 && yc >= 0 && yr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 2);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			cssRef = 2;
			}
		}
}

Symbol::Symbol(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		SymFill.hatch = (LineDEF *) NULL;
		}
	mo = 0L;
}

Symbol::~Symbol()
{
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
}

double
Symbol::GetSize(int select)
{
	switch(select) {
	case SIZE_MINE:			case SIZE_SYMBOL:		return size;
	case SIZE_SYM_LINE:		return SymLine.width;
	case SIZE_XPOS:			return fPos.fx;
	case SIZE_YPOS:			return fPos.fy;
	case SIZE_ZPOS:			return 0.0;
	default:
		return DefSize(select);
		}
}

bool
Symbol::SetSize(int select, double value)
{
	switch(select & 0xfff){
	case SIZE_MINE:			case SIZE_SYMBOL:
		size = value;		return true;
	case SIZE_SYM_LINE:		SymLine.width = value;		return true;
	case SIZE_XPOS:			fPos.fx = value;			return true;
	case SIZE_YPOS:			fPos.fy = value;			return true;
	case SIZE_ZPOS:			return true;
		}
	return false;
}

DWORD
Symbol::GetColor(int select)
{
	switch(select) {
	case COL_SYM_LINE:		return SymLine.color;
	case COL_SYM_FILL:		return SymFill.color;
	default:
		return parent ? parent->GetColor(select) : defs.Color(select);
		}
}

bool
Symbol::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_SYM_LINE:
		SymLine.color = col;
		if(SymTxt) SymTxt->ColTxt = col;
		return true;
	case COL_SYM_FILL:
		SymFill.color = col;
		return true;
	default:
		return false;
		}
}

void
Symbol::DoPlot(anyOutput *target)
{
	int ix, iy, i;
	lfPOINT fip;
	RECT currClipRC;

	if(type & SYM_POS_PARENT) {
		if(!parent) return;
		fip.fx = parent->GetSize(SIZE_XCENTER);
		fip.fy = parent->GetSize(SIZE_YCENTER);
		}
	else if(!parent) {		//Dialog?
		fip.fx = fPos.fx;	fip.fy = fPos.fy;
		}
	else if(!target->fp2fip(&fPos, &fip)) return;
	ix = iround(fip.fx);		iy = iround(fip.fy);
	rDims.left = rDims.right = ix;		rDims.top = rDims.bottom = iy;
	i = target->un2iy(size / 2.0);
	IncrementMinMaxRect(&rDims, i);
	i = target->ClipStatus(&rDims);
	switch (i) {
	default:
	case 0:		case 7:
		DoPlotXY(target, fip.fx, fip.fy);		break;
	case 1:		case 2:		case 3:
		if (target->GetClipRec(&currClipRC)) {		//temporarily disable clipping to show symbols on the axis
			target->ClipRect(0L);
			DoPlotXY(target, fip.fx, fip.fy);
			target->ClipRect(&currClipRC);
			}
		else DrawLater.AddObj(this);
		break;
	case 8:
		break;								//a hidden symbol
		}
	last_pos.fx = fip.fx;		last_pos.fy = fip.fy;
}

void
Symbol::DoPlotXY(anyOutput *target, double fix, double fiy)
{
	int atype, minc, ix, iy;
	double crx, cry, sc, rx, ry, tmp;
	long ncpts;
	POINT pts[14], *cpts = NULL;
	lfPOINT fPts[14];
	FillDEF cf;

	ix = iround(fix);			iy = iround(fiy);
	atype = (type  & 0xfff);
	memcpy(&cf, &SymFill, sizeof(FillDEF));
	if(atype == SYM_CIRCLEF || atype == SYM_RECTF || atype == SYM_TRIAUF ||
		atype == SYM_TRIADF || atype == SYM_DIAMONDF || atype == SYM_4STARF ||
		atype == SYM_5GONF || atype == SYM_5STARF || atype == SYM_6STARF) cf.color = SymLine.color;
	target->SetLine(&SymLine);
	minc = target->un2iy(size/6.0)+ 1;
	switch(atype){
	default:
	case SYM_CIRCLE:		//circle
	case SYM_CIRCLEF:		//filled circle
	case SYM_CIRCLEC:		//circle with center point
	case SYM_1QUAD:		case SYM_2QUAD:		case SYM_3QUAD:
		rx = target->un2fix(size/1.8);		ry = target->un2fiy(size/1.8);
		if(rx < 1.0) rx = 1.0;
		if(ry < 1.0) ry = 1;
		target->SetFill(&cf);
		target->foCircle(fix-rx, fiy-ry, fix+rx, iy+ry, name);
		if(atype == SYM_CIRCLEC) {
			crx = target->un2fix(size/5.0);	cry = target->un2fiy(size/5.0);
			cf.color = SymLine.color;		target->SetFill(&cf);
			target->foCircle(fix-crx, fiy-cry, fix+crx, fiy+cry, name);
			}
		else if(atype == SYM_1QUAD || atype == SYM_2QUAD || atype == SYM_3QUAD) {
			ncpts = 0L;			cf.color = SymLine.color;		target->SetFill(&cf);
			if(atype == SYM_1QUAD) {
				if(!(cpts = MakeArc(ix, iy, iround(rx), 0x04, &ncpts)) || !ncpts) return;
				cpts[0].x = ix + iround(rx);		cpts[0].y = iy;
				}
			else if(atype == SYM_2QUAD) {
				if(!(cpts = MakeArc(ix, iy, iround(rx), 0x06, &ncpts)) || !ncpts) return;
				cpts[0].x = ix;				cpts[0].y = iy+iround(rx);
				}
			else if(atype == SYM_3QUAD) {
				if(!(cpts = MakeArc(ix, iy, iround(rx), 0x07, &ncpts)) || !ncpts) return;
				cpts[0].x = ix-iround(rx);			cpts[0].y = iy;
				}
			cpts[ncpts-1].x = ix;			cpts[ncpts-1].y = iy-iround(rx);
			cpts[ncpts].x = ix;				cpts[ncpts].y = iy;				ncpts++;
			cpts[ncpts].x = cpts[0].x;		cpts[ncpts].y = cpts[0].y;		ncpts++;
			target->oPolygon(cpts, ncpts);	free(cpts);
			}
		rx -= 0.5;		ry-= 0.5;			//smaller marking rectangle
		break;
	case SYM_RECT:			//rectange (square)
	case SYM_RECTF:			//filled rectangle
	case SYM_RECTC:			//square with center point
		rx = target->un2fix(size/2.25676);	ry = target->un2fiy(size/2.25676);
		if(rx < 1.0) rx = 1.0;
		if(ry < 1.0) ry = 1.0;
		target->SetFill(&cf);
		if (atype == SYM_RECTF) {
			target->foSolidRectangle(fix - rx, fiy - ry, fix + rx, fiy + ry);
			}
		else {
			target->foRectangle(fix - rx, fiy - ry, ix + rx, iy + ry, name);
			}
		if(atype == SYM_RECTC) {
			crx = target->un2fix(size/6.0);	cry = target->un2fiy(size/6.0);
			cf.color = SymLine.color;		target->SetFill(&cf);
			target->foCircle(fix-crx, fiy-cry, fix+crx, fiy+cry, name);
			}
		break;
	case SYM_TRIAU:			//triangles up and down, open or closed
	case SYM_TRIAUF:	case SYM_TRIAD:		case SYM_TRIADF:		case SYM_TRIADC:
	case SYM_TRIAUC:	case SYM_TRIAUL:	case SYM_TRIAUR:		case SYM_TRIADL:
	case SYM_TRIADR:
		rx = target->un2fix(size/1.48503);	ry = target->un2fiy(size/1.48503);
		if(rx < 1.0) rx = 1.0;
		if(ry < 5.0) ry = 1.0;
		target->SetFill(&cf);
		fPts[0].fx = fPts[3].fx = fix - rx;		fPts[1].fx = fix;		fPts[2].fx = fix + rx;
		//patch by anonymous
		if(atype == SYM_TRIAU || atype == SYM_TRIAUF || atype == SYM_TRIAUL 
			|| atype == SYM_TRIAUR || atype == SYM_TRIAUC) {
			fPts[0].fy = fPts[2].fy = fPts[3].fy = fiy + target->un2fiy(size*0.38878);
			fPts[1].fy = fiy - target->un2fiy(size*0.77756);
			}
		else {
			fPts[0].fy = fPts[2].fy = fPts[3].fy = fiy - target->un2fiy(size*0.38878);
			fPts[1].fy = fiy + target->un2fiy(size*0.77756);
			}
		target->foPolygon(fPts, 4);
		if(atype == SYM_TRIAUC || atype == SYM_TRIADC) {
			crx = target->un2fix(size/6.0);	cry = target->un2fiy(size/6.0);
			cf.color = SymLine.color;		target->SetFill(&cf);
			target->foCircle(fix-crx, fiy-cry, fix+crx, fiy+cry, name);
			}
		else if(atype == SYM_TRIAUL || atype == SYM_TRIADL) {
			cf.color = SymLine.color;		target->SetFill(&cf);
			fPts[2].fx = fPts[1].fx;		target->foPolygon(fPts, 4);
			}
		else if(atype == SYM_TRIAUR || atype == SYM_TRIADR) {
			cf.color = SymLine.color;		target->SetFill(&cf);
			fPts[0].fx = fPts[3].fx = fPts[1].fx;	target->foPolygon(fPts, 4);
			}
		rx -= 0.5; ry -= 0.5;
		break;
	case SYM_DIAMOND:	case SYM_DIAMONDF:		case SYM_DIAMONDC:
		rx = target->un2fix(size/1.59588);	ry = target->un2fiy(size/1.59588);
		if(rx < 1.0) rx = 1.0;
		if(ry < 1.0) ry = 1.0;
		target->SetFill(&cf);
		pts[0].x = pts[2].x = pts[4].x = ix;		
		pts[0].y = pts[4].y = iy - iround(ry);
		pts[1].x = ix + iround(rx);					pts[1].y = pts[3].y = iy;
		pts[2].y = iy + iround(ry);					pts[3].x = ix - iround(rx);
		target->oPolygon(pts, 5);
		if(atype == SYM_DIAMONDC) {
			crx = target->un2ix(size/6.0);	cry = target->un2iy(size/6.0);
			cf.color = SymLine.color;		target->SetFill(&cf);
			target->foCircle(fix-crx, fiy-cry, fix+crx, fiy+cry, name);
			}
		rx--;									ry--;
		break;
	case SYM_4STAR:		case SYM_4STARF:
		rx = target->un2fix(size/1.4);	ry = target->un2fiy(size/1.4);
		crx = target->un2ix(size/6.0);	cry = target->un2iy(size/6.0);
		pts[0].x = pts[8].x = ix - iround(rx);		pts[0].y = pts[4].y = pts[8].y = iy;
		pts[1].x = pts[7].x = ix-iround(crx);		pts[1].y = pts[3].y = iy - iround(cry);
		pts[2].x = pts[6].x = ix;					pts[2].y = iy - iround(ry);
		pts[3].x = pts[5].x = ix+iround(crx);		pts[4].x = ix + iround(rx);
		pts[5].y = pts[7].y = iy+iround(cry);		pts[6].y = iy + iround(ry);
		target->SetFill(&cf);						target->oPolygon(pts, 9);
		break;
	case SYM_5GON:		case SYM_5GONF:		case SYM_5GONC:
		sc = 1.4;
		rx = target->un2fix(size/sc);	ry = target->un2fiy(size/sc);
		crx = target->un2ix(size/sc * 0.951057);	
		cry = target->un2iy(size/sc * 0.309017);
		pts[0].x = ix-iround(crx);			pts[0].y = pts[2].y = iy-iround(cry);	pts[1].x = ix;
		pts[1].y = iy-iround(ry);			pts[2].x = ix+iround(crx);
		crx = target->un2ix(size/sc * 0.587785);	
		cry = target->un2iy(size/sc * 0.809017);
		pts[3].x = ix + iround(crx);		pts[4].x = ix - iround(crx);
		pts[3].y = pts[4].y = iy+iround(cry);
		pts[5].x = pts[0].x;	pts[5].y = pts[0].y;
		target->SetFill(&cf);			target->oPolygon(pts, 6);
		if(atype == SYM_5GONC) {
			crx = target->un2ix(size/6.0);	cry = target->un2iy(size/6.0);
			cf.color = SymLine.color;		target->SetFill(&cf);
			target->foCircle(fix-crx, fiy-cry, fix+crx+1, fiy+cry+1, name);
			}
		break;
	case SYM_5STAR:		case SYM_5STARF:
		sc = 1.4;
		rx = target->un2fix(size/sc);	ry = target->un2fiy(size/sc);
		crx = target->un2fix(size/sc * 0.951057);	
		cry = target->un2fiy(size/sc * 0.309017);
		pts[0].x = ix-iround(crx);		pts[0].y = pts[1].y = pts[3].y = pts[4].y = iy-iround(cry);
		pts[2].x = pts[7].x = ix;		pts[2].y = iy-iround(ry);		pts[4].x = ix+iround(crx);
		crx = target->un2fix(size/sc * 0.23);
		pts[1].x = ix - iround(crx);	pts[3].x = ix + iround(crx);
		crx =  target->un2fix(size/sc * 0.36);
		cry = target->un2fiy(size/sc * 0.11);
		pts[5].x = ix + iround(crx);		pts[5].y = pts[9].y = iy +	iround(cry);	pts[9].x = ix - iround(crx);
		pts[7].y = iy + target->un2iy(size/sc * 0.38);
		crx = target->un2fix(size/sc * 0.587785);	
		cry = target->un2fiy(size/sc * 0.809017);
		pts[6].x = ix + iround(crx);	pts[8].x = ix - iround(crx);
		pts[6].y = pts[8].y = iy+iround(cry);
		pts[10].x = pts[0].x;	pts[10].y = pts[0].y;
		target->SetFill(&cf);			target->oPolygon(pts, 11);
		break;
	case SYM_6STAR:		case SYM_6STARF:
		sc = 1.4 / 0.86;				rx = target->un2fix(size/sc);
		sc = 1.4 / 0.5;					ry = target->un2fiy(size/sc);
		pts[0].x = pts[10].x = pts[12].x = ix - iround(rx);
		pts[4].x = pts[6].x = ix + iround(rx);	pts[2].x = pts[8].x = ix;
		pts[0].y = pts[1].y = pts[3].y = pts[4].y = pts[12].y = iy - iround(ry);
		pts[6].y = pts[7].y = pts[9].y = pts[10].y = iy + iround(ry);	
		sc = 1.4 / 0.29;				rx = target->un2fix(size/sc);
		pts[1].x = pts[9].x = ix - iround(rx);	pts[3].x = pts[7].x = ix + iround(rx);
		sc = 1.4 / 0.52;				rx = target->un2fix(size/sc);
		pts[5].x = ix + iround(rx);		pts[11].x = ix - iround(rx);		pts[5].y = pts[11].y = iy;
		sc = 1.4;
		rx = target->un2fix(size/sc);	ry = target->un2fiy(size/sc);
		pts[2].y = iy - iround(ry);		pts[8].y = iy + iround(ry);
		target->SetFill(&cf);			target->oPolygon(pts, 13);
		break;
	case SYM_STAR:			//star is a combination of + and x symbols
	case SYM_PLUS:			//draw a + sign
	case SYM_HLINE:		case SYM_VLINE:
		rx = target->un2fix(size/2.0f);		ry = target->un2fiy(size/2.0f);
		if(rx < 1.0) rx = 1.0;
		if(ry < 1.0) ry = 1.0;
		fPts[0].fx = fPts[1].fx = fix;
		fPts[0].fy = fiy - ry;					fPts[1].fy = iy + ry;
		if(type != SYM_HLINE) target->foSolidLine(fPts);
		fPts[0].fx = ix - rx;					fPts[1].fx = ix + rx;
		fPts[0].fy = fPts[1].fy = fiy;
		if (atype != SYM_VLINE) target->foSolidLine(fPts);
		if(atype == SYM_VLINE){ rx = 2.0; break;}
		if(atype == SYM_HLINE){ ry = 2.0; break;}
		if(atype == SYM_PLUS) break;		//continue with x symbol for star
		rx = target->un2fix(size / 2.5);	ry = target->un2fiy(size / 2.5);
		fPts[0].fx = fix - rx;				fPts[1].fx = fix + rx;
		fPts[0].fy = fiy - ry;				fPts[1].fy = fiy + ry;
		target->foSolidLine(fPts);
		//swap values
		tmp = fPts[0].fy;	fPts[0].fy = fPts[1].fy;	fPts[1].fy = tmp;
		//and draw second line
		target->foSolidLine(fPts);
		break;
	case SYM_CROSS:			//draw a x symbol
		rx = target->un2fix(size/2.6);		ry = target->un2fiy(size/2.6);
		if(rx < 1.0) rx = 1.0;
		if(ry < 1.0) ry = 1.0;
		fPts[0].fx = fix - rx;				fPts[1].fx = fix + rx;
		fPts[0].fy = fiy - ry;				fPts[1].fy = fiy + ry;
		target->foSolidLine(fPts);
		//swap values
		tmp = fPts[0].fy;	fPts[0].fy = fPts[1].fy;	fPts[1].fy = tmp;
		//and draw second line
		target->foSolidLine(fPts);
		break;
	case SYM_TEXT:
		if(!SymTxt) Command(CMD_SETTEXT, (void *)"text", target);
		if(!SymTxt || !SymTxt->text || !SymTxt->text[0])return;
		SymTxt->iSize = target->un2iy(SymTxt->fSize = size *1.5);
		target->SetTextSpec(SymTxt);
		DrawFmtText.SetText(target, SymTxt->text, &fix, &fiy, true);
		if (target->oGetTextExtent(SymTxt->text, 0, &rx, &ry)){
			rx /= 2.0;		ry /= 2.0;
			}
		else rx = ry = 10;
		}
	rDims.left = ix-iround(rx)-minc;				rDims.right = ix+iround(rx)+minc+1;
	rDims.top = iy-iround(ry)-minc;					rDims.bottom = iy+iround(ry)+minc+1;
}

void
Symbol::DoMark(anyOutput *o, bool mark)
{
	double  old_line_w;
	POINT pl[2];

	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		switch (type  & 0xfff) {
		case SYM_STAR:	case SYM_PLUS:	case SYM_HLINE:		case SYM_VLINE:		case SYM_CROSS:
			mo = GetRectBitmap(&rDims, o);
			old_line_w = SymLine.width;
			SymLine.width *= 3;								DoPlotXY(o, last_pos.fx, last_pos.fy);
			SymLine.width = old_line_w;						SymLine.color ^= 0x00ffffffL;
			DoPlotXY(o, last_pos.fx, last_pos.fy);			SymLine.color ^= 0x00ffffffL;
			break;
		case SYM_TEXT:
			o->SetLine(&SymLine);
			mo = GetRectBitmap(&rDims, o);
			pl[0].x = rDims.left+2;					pl[1].x = rDims.right-2;
			pl[0].y = pl[1].y = rDims.top+2;		o->oSolidLine(pl);
			pl[0].y = pl[1].y = rDims.bottom-2;		o->oSolidLine(pl);
			pl[0].y = rDims.top +2;		pl[0].x = pl[1].x = rDims.left+2;	o->oSolidLine(pl);
			pl[0].x = pl[1].x = rDims.right-2;		o->oSolidLine(pl);
			break;
		default:
			mo = GetRectBitmap(&rDims, o);
			SymFill.color ^= 0x00ffffffL;
			DoPlotXY(o, last_pos.fx, last_pos.fy);
			SymFill.color ^= 0x00ffffffL;
			break;
			}
		defs.Idle(CMD_DOPLOT);
		}
	else {
		if(mo) {
			RestoreRectBitmap(&mo, &rDims, o);
			DelBitmapClass(mo);			mo = 0L;
			return;
			}
		else if(parent) parent->DoPlot(o);
		else DoPlot(o);
		}
	o->UpdateRect(&rDims, true);
}

bool 
Symbol::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	unsigned char *tmptxt;
	AccRange *ac;
	long i, r, c;
	Symbol *prevsym;

	switch (cmd) {
	case CMD_SCALE:
		if(!tmpl) return false;
		size *= ((scaleINFO*)tmpl)->sy.fy;
		SymLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		if(SymTxt) {
			SymTxt->fSize *= ((scaleINFO*)tmpl)->sy.fy;
			SymTxt->iSize = 0;
			}
		return true;
	case CMD_FLUSH:
		if(SymTxt) {
			if(SymTxt->text) free(SymTxt->text);
			free(SymTxt);
			}
		if(ssRef) free(ssRef);
		ssRef = 0L;						if(name)free(name);
		name = 0L;
		return true;
	case CMD_REDRAW:
		//if we come here its most likely the result of Undo
		if(parent && parent->Id==GO_REGRESSION)
			return parent->Command(CMD_MRK_DIRTY, 0L, o);
		return false;
	case CMD_GETTEXT:
		if(SymTxt && SymTxt->text && tmpl) {
			rlp_strcpy((char*)tmpl, 50, SymTxt->text);
			return true;
			}
		return false;
	case CMD_SYMTEXT_UNDO:		case CMD_SYMTEXT:		case CMD_SETTEXT:
		if (cmd == CMD_SYMTEXT_UNDO) {
			if (SymTxt && SymTxt->text){
				c = Undo.String(this, (char**)(&SymTxt->text), UNDO_CONTINUE);
				i = (int)strlen((char*)tmpl);		i = i > c ? i + 2 : c + 2;
				if (tmpl) {
					SymTxt->text = (unsigned char*)realloc(SymTxt->text, i);
					if (SymTxt->text) rlp_strcpy(SymTxt->text, i, (char*)tmpl);
					}
				else if (SymTxt->text) SymTxt->text[0] = 0;
				return true;
				}
			}
		if(!SymTxt && (SymTxt = (TextDEF *) calloc(1, sizeof(TextDEF)))) {
			SymTxt->ColTxt = SymLine.color;			SymTxt->fSize = size*1.5;
			SymTxt->ColBg = parent ? parent->GetColor(COL_BG) : 0x00ffffffL;
			SymTxt->Align = TXA_VCENTER | TXA_HCENTER;
			SymTxt->Style = TXS_NORMAL;				SymTxt->Mode = TXM_TRANSPARENT;
			SymTxt->Font = FONT_HELVETICA;			SymTxt->text = 0L;
			}
		if(!SymTxt) return false;
		if(tmpl) {
			i = (int) strlen((char*)tmpl) + 2;
			SymTxt->text = (unsigned char*)realloc(SymTxt->text, i);
			if (SymTxt->text) rlp_strcpy(SymTxt->text, i, (char*)tmpl);
			}
		else if(SymTxt->text) SymTxt->text[0] = 0;
		return true;
	case CMD_SYM_TYPE:
		if(tmpl)type = *((int*)tmpl);
		return true;
	case CMD_GETTEXTDEF:
		if(!SymTxt || !tmpl) return false;
		memcpy(tmpl, SymTxt, sizeof(TextDEF));
		return true;
	case CMD_SYMTEXTDEF:		case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		if(SymTxt) tmptxt = SymTxt->text;
		else tmptxt = 0L;
		if(!SymTxt && !(SymTxt = (TextDEF *) calloc(1, sizeof(TextDEF)))) return false;
		memcpy(SymTxt, tmpl, sizeof(TextDEF));
		SymTxt->text = tmptxt;
		return true;
	case CMD_SYM_RANGETEXT:		case CMD_RANGETEXT:
		if(!data || !tmpl) return false;
		if(!(tmptxt = (unsigned char*)malloc(500)))return false;
		if((ac = new AccRange((char*)tmpl)) && ac->GetFirst(&c, &r)) {
			for(i = 0, tmptxt[0] = 0; i <= idx; i++) ac->GetNext(&c, &r);
			data->GetText(r, c, (char*)tmptxt, 500);
			delete(ac);
			}
		Command(CMD_SETTEXT, tmptxt, 0L);
		free(tmptxt);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_SYMBOL;
		if(tmpl) data = (DataObj *)tmpl;
		return true;
	case CMD_SELECT:
		memcpy(&mrc, &rDims, sizeof(RECT));
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				memcpy(&mrc, &rDims, sizeof(RECT));
				o->ShowMark(this, MRK_GODRAW);
				return true;
				}
			break;
			}
		break;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			return true;
			}
		return false;
	case  CMD_PREVSYM:
		if (tmpl && ((GraphObj*)tmpl)->Id == GO_SYMBOL) {
			prevsym = (Symbol*)tmpl;
			SymLine.color = prevsym->SymLine.color;			SymLine.width = prevsym->SymLine.width;
			SymFill.color = prevsym->SymFill.color;			size = prevsym->GetSize(SIZE_SYMBOL);
			type = prevsym->type;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Symbols are graphic objects
SymTernary::SymTernary(GraphObj *par, DataObj *d, double x, double y, double z, int which,
		int xc, int xr, int yc, int yr, int zc, int zr):Symbol(par, d, x, y, which, xc, xr, yc, yr)
{
	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;		type = which;
	fPos3D.fx = x;		fPos3D.fy = y;		fPos3D.fz = z;
	Id = GO_SYMTERN;	mo = 0L;
	if(xc >= 0 && xr >= 0 && yc >= 0 && yr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 3);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			cssRef = 3;
			}
		}
}

SymTernary::SymTernary(int src):Symbol(0)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		SymFill.hatch = (LineDEF *) NULL;
		}
	mo = 0L;
}

SymTernary::~SymTernary()
{
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
}

double
SymTernary::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:
		return fPos3D.fx;
	case SIZE_YPOS:
		return fPos3D.fy;
	case SIZE_ZPOS:
		return fPos3D.fz;
	default:
		return Symbol::GetSize(select);
		}
	return 0.0;
}

bool
SymTernary::SetSize(int select, double value)
{
	switch(select & 0xfff){
	case SIZE_XPOS:
		fPos.fx = fPos3D.fx = value;
		return true;
	case SIZE_YPOS:
		fPos.fy = fPos3D.fy = value;
		return true;
	case SIZE_ZPOS:
		fPos3D.fz = value;
		return true;
	default:
		return Symbol::SetSize(select, value);
	}
	return false;
}

void
SymTernary::DoPlot(anyOutput *o)
{
	double fix, fiy;

	if(parent && parent->Id == GO_TERNARYXYZ) {
		if(!((TernaryXYZ *)parent)->TransformVal(o, &fPos3D, &fix, &fiy)) Symbol::DoPlot(o);
		last_pos.fx = fix;		last_pos.fy = fiy;
		DoPlotXY(o, fix, fiy);
		}
	else Symbol::DoPlot(o);
}

void
SymTernary::DoMark(anyOutput *o, bool mark)
{
	double  old_line_w;
	POINT pl[2];

	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		switch (type  & 0xfff) {
		case SYM_STAR:	case SYM_PLUS:	case SYM_HLINE:		case SYM_VLINE:		case SYM_CROSS:
			mo = GetRectBitmap(&rDims, o);
			old_line_w = SymLine.width;
			SymLine.width *= 3;
			DoPlotXY(o, last_pos.fx, last_pos.fy);
			SymLine.width = old_line_w;								SymLine.color ^= 0x00ffffffL;
			DoPlotXY(o, last_pos.fx, last_pos.fy);
			SymLine.color ^= 0x00ffffffL;
			break;
		case SYM_TEXT:
			o->SetLine(&SymLine);
			mo = GetRectBitmap(&rDims, o);
			pl[0].x = rDims.left+2;					pl[1].x = rDims.right-2;
			pl[0].y = pl[1].y = rDims.top+2;		o->oSolidLine(pl);
			pl[0].y = pl[1].y = rDims.bottom-2;		o->oSolidLine(pl);
			pl[0].y = rDims.top +2;		pl[0].x = pl[1].x = rDims.left+2;	o->oSolidLine(pl);
			pl[0].x = pl[1].x = rDims.right-2;		o->oSolidLine(pl);
			break;
		default:
			SymFill.color ^= 0x00ffffffL;
			DoPlotXY(o, last_pos.fx, last_pos.fy); 
			SymFill.color ^= 0x00ffffffL;
			break;
			}
		}
	else {
		if(mo) {
			RestoreRectBitmap(&mo, &rDims, o);
			DelBitmapClass(mo);			mo = 0L;
			return;
			}
		else if(parent) parent->DoPlot(o);
		else DoPlot(o);
		}
	o->UpdateRect(&rDims, true);
}

bool
SymTernary::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_SYMTERN;
		if(tmpl)data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			fPos3D.fx = fPos.fx;		fPos3D.fy = fPos.fy;
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos3D.fz);
			return true;
			}
		return false;
	default:
		return Symbol::Command(cmd, tmpl, o);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bubbles are graphic objects
Bubble::Bubble(GraphObj *par, DataObj *d, double x, double y, double s, int which, 
	FillDEF *fill, LineDEF *outline, int xc, int xr, int yc, int yr, int sc,
	int sr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;	fPos.fy = y;	fs = s;
	type = which;
	if(fill) {
		memcpy(&BubbleFill,fill, sizeof(FillDEF));
		if(BubbleFill.hatch) memcpy(&BubbleFillLine, BubbleFill.hatch, sizeof(LineDEF));
		}
	BubbleFill.hatch = &BubbleFillLine;
	if(outline)memcpy(&BubbleLine, outline, sizeof(LineDEF));
	Id = GO_BUBBLE;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || sc >= 0 || sr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 3);
		if(ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = sc;	ssRef[2].y = sr;
			cssRef = 3;
			}
		}
}

Bubble::Bubble(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Bubble::~Bubble()
{
	Command(CMD_FLUSH, 0L, 0L);
}

void
Bubble::DoPlot(anyOutput *o)
{
	int i;
	double x1, y1, x2, y2, ix, iy, tmp;
	double fix = 0.0, fiy = 0.0;
	RECT currClipRC, rcBounds;

	o->SetLine(&BubbleLine);		o->SetFill(&BubbleFill);
	switch(type & 0x0f0) {
	case BUBBLE_UNITS:
		fix = o->un2fix(fs);		fiy = o->un2fiy(fs);
		break;
	case BUBBLE_XAXIS:
		fix = (o->fx2fix(fPos.fx+fs) - o->fx2fix(fPos.fx-fs))/2.0;
		fiy = fix * (o->un2fiy(10.0f)/o->un2fix(10.0f));	//x and y resolution different ?
		break;
	case BUBBLE_YAXIS:
		fix = (o->fy2fiy(fPos.fy-fs) - o->fy2fiy(fPos.fy+fs))/2.0;
		fiy = fix * (o->un2fiy(10.0f)/o->un2fix(10.0f));	//x and y resolution different ?
		break;
		}
	fix = fix < 0.0 ? -fix : fix;							//sign must be positive
	fiy = fiy < 0.0 ? -fiy : fiy;
	rDims.left = rDims.right = iround(o->fx2fix(fPos.fx));
	rDims.top = rDims.bottom = iround(o->fy2fiy(fPos.fy));
	switch (type & 0x00f) {
	case BUBBLE_CIRCLE:
		ix = fix / 2.0;			iy = fiy / 2.0;
		tmp = o->fx2fix(fPos.fx);		x1 = tmp - ix;		x2 = tmp + ix;
		tmp = o->fy2fiy(fPos.fy);		y1 = tmp - iy;		y2 = tmp + iy;
		rcBounds.left = iround(x1);				rcBounds.right = iround(x2);
		rcBounds.top = iround(y1);				rcBounds.bottom = iround(y2);
		i = o->ClipStatus(&rcBounds);
		if (o->OC_type == OC_GRIDVIEW) o->oCircArc(iround(x1), iround(y1), iround(x2), iround(y2));
		else switch (i) {
		default:
		case 0:		case 7:
			o->foCircle(x1, y1, x2, y2, name);
			break;
		case 1:		case 2:		case 3:
			if (o->GetClipRec(&currClipRC)) {		//temporarily disable clipping to show symbols on the axis
				o->ClipRect(0L);
				o->foCircle(x1, y1, x2, y2, name);
				o->ClipRect(&currClipRC);
			}
			else DrawLater.AddObj(this);
			break;
		case 8:
			break;								//a hidden symbol
		}
		UpdateMinMaxRect(&rDims, iround(x1), iround(y1));
		UpdateMinMaxRect(&rDims, iround(x2), iround(y2));
		break;
	case BUBBLE_SQUARE:
		if ((type & 0xf00) == BUBBLE_CIRCUM) {
			ix = fix*.392699081;		iy = fiy*.392699081;
		}
		else if ((type & 0xf00) == BUBBLE_AREA) {
			ix = fix*.443113462;		iy = fiy*.443113462;
		}
		else {
			ix = fix*.353553391;		iy = fiy*.353553391;
		}
		tmp = o->fx2fix(fPos.fx);		x1 = tmp - ix;		x2 = tmp + ix;
		tmp = o->fy2fiy(fPos.fy);		y1 = tmp - iy;		y2 = tmp + iy;
		rcBounds.left = iround(x1);				rcBounds.right = iround(x2);
		rcBounds.top = iround(y1);				rcBounds.bottom = iround(y2);
		i = o->ClipStatus(&rcBounds);
		switch (i) {
		default:
		case 0:		case 7:
			o->foRectangle(x1, y1, x2, y2, name);
			break;
		case 1:		case 2:		case 3:
			if (o->GetClipRec(&currClipRC)) {		//temporarily disable clipping to show symbols on the axis
				o->ClipRect(0L);
				o->foRectangle(x1, y1, x2, y2, name);
				o->ClipRect(&currClipRC);
			}
			else DrawLater.AddObj(this);
			break;
		case 8:
			break;								//a hidden symbol
		}
		UpdateMinMaxRect(&rDims, iround(x1), iround(y1));
		UpdateMinMaxRect(&rDims, iround(x2), iround(y2));
		break;
	case BUBBLE_UPTRIA:						case BUBBLE_DOWNTRIA:
		if ((type & 0xf00) == BUBBLE_CIRCUM) {
			fix *= .523598775;		fiy *= .523598775;
			}
		else if ((type & 0xf00) == BUBBLE_AREA) {
			fix *= .673386843;		fiy *= .673386843;
			}
		else {
			fix *= .433012702;		fiy *= .433012702;
			}
		ix = fix;		iy = fiy*.57735;
		tmp = o->fx2fix(fPos.fx);
		fPts[0].fx = fPts[3].fx = tmp - ix;				fPts[1].fx = tmp + ix;				fPts[2].fx = tmp;
		pts[0].x = pts[3].x = iround(fPts[0].fx);		pts[1].x = iround(fPts[1].fx);		pts[2].x = iround(fPts[2].fx);
		tmp = o->fy2fiy(fPos.fy);
		if ((type & 0x00f) == BUBBLE_UPTRIA) {
			pts[0].y = pts[1].y = pts[3].y = iround(tmp + iy);
			fPts[0].fy = fPts[1].fy = fPts[3].fy = tmp + iy;
			pts[2].y = iround(tmp - fiy*1.1547);		fPts[2].fy = tmp - fiy*1.1547;
			rcBounds.top = iround(fPts[2].fy);			rcBounds.bottom = iround(fPts[0].fy);
			}
		else {
			pts[0].y = pts[1].y = pts[3].y = iround(tmp - iy);
			fPts[0].fy = fPts[1].fy = fPts[3].fy = tmp - iy;
			pts[2].y = iround(tmp + fiy*1.1547);		fPts[2].fy = tmp + fiy*1.1547;
			rcBounds.top = iround(fPts[0].fy);			rcBounds.bottom = iround(fPts[2].fy);
			}
		rcBounds.left = iround(fPts[0].fx);			rcBounds.right = iround(fPts[1].fx);
		i = o->ClipStatus(&rcBounds);
		switch (i) {
		default:
		case 0:		case 7:
			o->foPolygon(fPts, 4);
			break;
		case 1:		case 2:		case 3:
			if (o->GetClipRec(&currClipRC)) {		//temporarily disable clipping to show symbols on the axis
				o->ClipRect(0L);
				o->foPolygon(fPts, 4);
				o->ClipRect(&currClipRC);
				}
			else DrawLater.AddObj(this);
			break;
		case 8:
			break;								//a hidden symbol
			}
		UpdateMinMaxRect(&rDims, pts[0].x, pts[0].y);
		UpdateMinMaxRect(&rDims, pts[1].x, pts[2].y);
		IncrementMinMaxRect(&rDims, 2);
		break;
		}
}

void
Bubble::DoMark(anyOutput *o, bool mark)
{
	if(mark) {
		BubbleFillLine.color ^= 0x00ffffffL;
		BubbleFill.color ^= 0x00ffffffL;
		DoPlot(o);
		BubbleFill.color ^= 0x00ffffffL;
		BubbleFillLine.color ^= 0x00ffffffL;
		}
	else {
		if(parent) parent->DoPlot(o);
		else DoPlot(o);
		}
	o->UpdateRect(&rDims, true);
}

bool 
Bubble::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	bool bSelected = false;
	unsigned long n, s;
	POINT p;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);
		ssRef = 0L;						if(name)free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		if(!tmpl) return false;
		if((type & 0x0f0)== BUBBLE_UNITS) fs *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleFillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleFillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleFill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		((Legend*)tmpl)->HasFill(&BubbleLine, &BubbleFill, 0L);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				p.x = mev->x, p.y = mev->y;
				switch(type & 0x00f) {
				case BUBBLE_CIRCLE:
					n = s = p.x - ((rDims.right+rDims.left)>>1);
					s *= n;
					n = p.y - ((rDims.bottom+rDims.top)>>1);
					n = isqr(s += n*n) -2;
					bSelected = ((unsigned)((rDims.right-rDims.left)>>1) > n);
					break;
				case BUBBLE_SQUARE:
					bSelected = true;
					break;
				case BUBBLE_UPTRIA:
				case BUBBLE_DOWNTRIA:
					if(!(bSelected = IsInPolygon(&p, pts, 4, 3.0)))
						bSelected = IsCloseToPL(p, pts, 4, 3.0);
					break;
					}
				if(bSelected) o->ShowMark(this, MRK_GODRAW);
				return bSelected;
				}
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_BUBBLE;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fs);
			return true;
			}
		return false;
	case CMD_BUBBLE_ATTRIB:
		if(tmpl) {
			type &= ~0xff0;
			type |= (*((int*)tmpl) & 0xff0);
			return true;
			}
		return false;
	case CMD_BUBBLE_TYPE:
		if(tmpl) {
			type &= ~0x00f;
			type |= (*((int*)tmpl) & 0x00f);
			return true;
			}
		return false;
	case CMD_BUBBLE_FILL:
		if(tmpl) {
			BubbleFill.type = ((FillDEF*)tmpl)->type;
			BubbleFill.color = ((FillDEF*)tmpl)->color;
			BubbleFill.scale = ((FillDEF*)tmpl)->scale;
			if(((FillDEF*)tmpl)->hatch)
				memcpy(&BubbleFillLine, ((FillDEF*)tmpl)->hatch, sizeof(LineDEF));
			}
		return true;
	case CMD_BUBBLE_LINE:
		if(tmpl) memcpy(&BubbleLine, tmpl, sizeof(LineDEF));
		return true;
	case CMD_AUTOSCALE:
		return DoAutoscale(o);
		break;
		}
	return false;
}

bool
Bubble::DoAutoscale(anyOutput *o) 
{
	double dx = 0.0, dy = 0.0;

	switch (type & 0x0f0) {
	case BUBBLE_XAXIS:			case BUBBLE_YAXIS:
		if (rDims.right > rDims.left){
			((Plot*)parent)->CheckBounds(o->fix2fx(rDims.left), o->fiy2fy(rDims.top));
			((Plot*)parent)->CheckBounds(o->fix2fx(rDims.right), o->fiy2fy(rDims.bottom));
			return true;
			}
		else {
			dx = dy = fs / 2.0;		break;
			}
		dx = dy = fs/2.0;		break;
	case BUBBLE_UNITS:
		dx = fPos.fx/20;		dy = fPos.fy/20;		break;
		}
	((Plot*)parent)->CheckBounds(fPos.fx+dx, fPos.fy-dy);
	((Plot*)parent)->CheckBounds(fPos.fx-dx, fPos.fy+dy);
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bars are graphic objects
Bar::Bar(GraphObj *par, DataObj *d, double x, double y, int which,int xc, int xr,
		int yc, int yr, char *desc):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;			fPos.fy = y;			type = which;
	if(type & BAR_RELWIDTH) size = 60.0;
	Id = GO_BAR;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 2);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			cssRef = 2;
			}
		}
	mo = 0L;
	if(desc && desc[0]) name = rlp_strdup(desc);
}

Bar::Bar(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	mo = 0L;
}

Bar::~Bar()
{
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
}

double
Bar::GetSize(int select)
{
	switch(select){
	case SIZE_XPOS:				return fPos.fx;
	case SIZE_YPOS:				return fPos.fy;
	case SIZE_BAR:				return size;
		}
	return 0.0;
}

bool
Bar::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_BAR: 
		size = value;				return true;
	case SIZE_BAR_LINE:
		BarLine.width = value;		return true;
	case SIZE_XBASE:
		BarBase.fx = value;			return true;
	case SIZE_YBASE:
		BarBase.fy = value;			return true;
	case SIZE_XPOS:
		fPos.fx = value;			return true;
	case SIZE_YPOS:
		fPos.fy = value;			return true;
		}
	return false;
}

bool
Bar::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_BAR_LINE:
		BarLine.color = col;		return true;
	case COL_BAR_FILL:
		BarFill.color = col;		return true;
		}
	return false;
}

void
Bar::DoPlot(anyOutput *target)
{
	int i, w;
	double fBase, rsize;
	POINT pts[2];
	RECT currClipRC;

	target->SetLine(&BarLine);			target->SetFill(&BarFill);
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	switch(type & 0xff) {
	case BAR_VERTU:		case BAR_VERTT:		case BAR_VERTB:
		if (!parent) {				//is it part of a dialog?
			fBase = BarBase.fx;
			pts[0].y = iround(fPos.fy);
			pts[1].y = iround(fBase);
			}
		else switch(type & 0xff) {	//its part of user data!
		case BAR_VERTB:
			fBase = parent->GetSize(SIZE_BOUNDS_BOTTOM);
			break;
		case BAR_VERTT:
			fBase = parent->GetSize(SIZE_BOUNDS_TOP);
			break;
		case BAR_VERTU:
			fBase = BarBase.fy;
			break;
			}
		if (type & BAR_RELWIDTH) {
			if (!parent) {				//dialog ?
				pts[0].x = iround(fPos.fx - size / 2.0);
				pts[1].x = iround(fPos.fx + size / 2.0);
				target->oRectangle(pts[0].x, pts[0].y, pts[1].x, pts[1].y, 0L);
				return;
				}
			else {						//user plot!
				rsize = size * parent->GetSize(SIZE_BARMINX) / 100.0;
				if ((type & 0x3000) == BAR_ALIGN_LEFT) {
					pts[0].x = iround(target->fx2fix(fPos.fx));
					pts[1].x = iround(target->fx2fix(fPos.fx + rsize));
					parent->SetSize(SIZE_BAR_DX, (double)(pts[1].x - pts[0].x) / 2.0);
					}
				else if ((type & 0x3000) == BAR_ALIGN_RIGHT) {
					pts[0].x = iround(target->fx2fix(fPos.fx - rsize));
					pts[1].x = iround(target->fx2fix(fPos.fx));
					parent->SetSize(SIZE_BAR_DX, (double)(pts[1].x - pts[0].x) / -2.0);
					}
				else {
					pts[0].x = iround(target->fx2fix(fPos.fx - rsize / 2.0));
					pts[1].x = iround(target->fx2fix(fPos.fx + rsize / 2.0));
					parent->SetSize(SIZE_BAR_DX, 0.0);
					}
				}
			}
		else {
			if (!parent) {				//dialog ?
				pts[0].x = iround(fPos.fx - size / 2.0);
				pts[1].x = iround(fPos.fx + size / 2.0);
				target->oRectangle(pts[0].x, pts[0].y, pts[1].x, pts[1].y, 0L);
				return;
				}
			else {						//user plot!
				w = target->un2ix(size);
				if ((type & 0x3000) == BAR_ALIGN_LEFT) {
					pts[0].x = iround(target->fx2fix(fPos.fx));
					pts[1].x = iround(target->fx2fix(fPos.fx) + w);
					parent->SetSize(SIZE_BAR_DX, (double)(pts[1].x - pts[0].x) / 2.0);
					}
				else if ((type & 0x3000) == BAR_ALIGN_RIGHT) {
					pts[0].x = iround(target->fx2fix(fPos.fx) - w);
					pts[1].x = iround(target->fx2fix(fPos.fx));
					parent->SetSize(SIZE_BAR_DX, (double)(pts[1].x - pts[0].x) / -2.0);
					}
				else {
					pts[0].x = iround(target->fx2fix(fPos.fx)) - (w >> 1);
					pts[1].x = pts[0].x + w;
					parent->SetSize(SIZE_BAR_DX, 0.0);
					}
				}
			}
		if(type & BAR_CENTERED) {
			pts[0].y = iround(target->fy2fiy(fBase - (fPos.fy - fBase)));
			pts[1].y = iround(target->fy2fiy(fBase + (fPos.fy - fBase)));
			}
		else {
			pts[0].y = iround(target->fy2fiy(fBase));
			pts[1].y = iround(target->fy2fiy(fPos.fy));
			}
		break;
	case BAR_HORU:		case BAR_HORR:		case BAR_HORL:
		if(parent) switch(type & 0xff) {
		case BAR_HORL:
			fBase = parent->GetSize(SIZE_BOUNDS_LEFT);
			break;
		case BAR_HORR:
			fBase = parent->GetSize(SIZE_BOUNDS_RIGHT);
			break;
		case BAR_HORU:
			fBase = BarBase.fx;
			break;
			}
		else fBase = BarBase.fx;
		if(type & BAR_RELWIDTH) {
			if(parent) rsize = size * parent->GetSize(SIZE_BARMINY)/100.0;
			else rsize = size / 10.0;
			pts[0].y = iround(target->fy2fiy(fPos.fy - rsize/2.0));
			pts[1].y = iround(target->fy2fiy(fPos.fy + rsize/2.0));
			}
		else {
			w = target->un2iy(size);
			pts[0].y = target->fy2iy(fPos.fy) - w/2;
			pts[1].y = pts[0].y+w;
			}
		if(type & BAR_CENTERED) {
			pts[0].x = target->fx2ix(fBase - (fPos.fx - fBase));
			pts[1].x = target->fx2ix(fBase + (fPos.fx - fBase));
			}
		else {
			pts[0].x = target->fx2ix(fBase);
			pts[1].x = target->fx2ix(fPos.fx);
			}
		break;
	default:
		return;
		}
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	i = target->ClipStatus(&rDims);
	switch (i) {
	default:
	case 0:								break;		//no clipping
	case 1:											//vertically clipped
		if (target->GetClipRec(&currClipRC)) {		//temporarily disable clipping to show bars on the y-axis
			target->ClipRect(0L);
			target->oRectangle(pts[0].x, pts[0].y, pts[1].x, pts[1].y, name);
			target->ClipRect(&currClipRC);
			return;
			}
		else DrawLater.AddObj(this);
		break;
	case 8:
		return;										//a hidden bar
		}
	if (i == 7 && target->GetClipRec(&currClipRC)) {	//make sure clipping works
		target->ClipRect(&currClipRC);
		}
	if (pts[0].x == pts[1].x || pts[0].y == pts[1].y) {
		target->oSolidLine(pts);
		}
	else target->oRectangle(pts[0].x, pts[0].y, pts[1].x, pts[1].y, name);
}

void
Bar::DoMark(anyOutput *o, bool mark)
{
	POINT mpts[5];

	if (mark){
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		mpts[0].x = mpts[4].x = mpts[3].x = rDims.left;
		mpts[0].y = mpts[4].y = mpts[1].y = rDims.bottom;
		mpts[1].x = mpts[2].x = rDims.right;	mpts[2].y = mpts[3].y = rDims.top;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 5 * o->un2ix(BarLine.width) + 2);
		mo = GetRectBitmap(&mrc, o);
		InvertLine(mpts, 5, &BarLine, &mrc, o, mark);
		}
	else {
		RestoreRectBitmap(&mo, &mrc, o);
		if (mo) DelBitmapClass(mo);
		mo = 0L;
		}
}

bool 
Bar::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	FillDEF *TmpFill;
	lfPOINT bl = { 0.0, 0.0 };
	Bar* prevsym;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);
		ssRef = 0L;						if(name)free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		if(!tmpl) return false;
		if(!(type & BAR_RELWIDTH)) size *= ((scaleINFO*)tmpl)->sy.fy;
		BarLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		BarLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		HatchLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		HatchLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		BarFill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(name)((Legend*)tmpl)->HasFill(&BarLine, &BarFill, name);
		else if(parent && parent->Id == GO_PLOTSCATT)
			((Legend*)tmpl)->HasFill(&BarLine, &BarFill, ((PlotScatt*)parent)->data_desc);
		else ((Legend*)tmpl)->HasFill(&BarLine, &BarFill, 0L);
		break;
	case CMD_SELECT:
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, o->un2ix(BarLine.width) + 3);
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				IncrementMinMaxRect(&mrc, o->un2ix(BarLine.width) + 3);
				o->ShowMark(CurrGO = this, MRK_GODRAW);
				return true;
				}
			break;
			}
		return false;
	case CMD_BAR_FILL:
		TmpFill = (FillDEF *)tmpl;
		if(TmpFill) {
			BarFill.type = TmpFill->type;
			BarFill.color = TmpFill->color;
			BarFill.scale = TmpFill->scale;
			if(TmpFill->hatch) memcpy(&HatchLine, TmpFill->hatch, sizeof(LineDEF));
			}
		return true;
	case CMD_BAR_TYPE:
		if(tmpl) type = *((int*)tmpl);
		return true;
	case CMD_PREVSYM:
		if (tmpl && ((GraphObj*)tmpl)->Id == GO_BAR) {
			prevsym = (Bar*)tmpl;
			memcpy(&BarFill, &prevsym->BarFill, sizeof(FillDEF));		memcpy(&BarLine, &prevsym->BarLine, sizeof(LineDEF));
			memcpy(&HatchLine, &prevsym->HatchLine, sizeof(LineDEF));	BarFill.hatch = &HatchLine;
			size = prevsym->GetSize(SIZE_BAR);							type = prevsym->type;
			}
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_BAR;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			switch(type & 0xff) {
			case BAR_VERTU:					case BAR_VERTT:			case BAR_VERTB:
				bl.fx = fPos.fx;
				switch (type & 0xff) {
				case BAR_VERTU:
					bl.fy = BarBase.fy;
					break;
				case BAR_VERTT:				case BAR_VERTB:
					bl.fy = 0.0f;		//cannot resolve
					break;
					}
				if(type & BAR_CENTERED) bl.fy -= fPos.fy;
				break;
			case BAR_HORU:					case BAR_HORR:			case BAR_HORL:
				bl.fy = fPos.fy;
				switch(type & 0xff) {
				case BAR_HORU:				case BAR_HORR:			case BAR_HORL:
					if ((type & 0xff) == BAR_HORU) bl.fx = BarBase.fx;
					else bl.fx = 0.0f;		//cannot resolve
					}
				if(type & BAR_CENTERED) bl.fx -= fPos.fx;
				break;
				}
			((Plot*)parent)->CheckBounds(bl.fx, bl.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Data line is a graphic object
DataLine::DataLine(GraphObj *par, DataObj *d, char *xrange, char *yrange, char *nam, char *condition):GraphObj(par, d)
{
	size_t cb;

	FileIO(INIT_VARS);
	Id = GO_DATALINE;
	if(xrange && xrange[0]) {
		cb = rlp_strlen(xrange) +2;		ssXref = (char*)malloc(cb);		rlp_strcpy(ssXref, (int)cb, xrange);
		}
	if(yrange && yrange[0]) {
		cb = rlp_strlen(yrange) +2;		ssYref = (char*)malloc(cb);		rlp_strcpy(ssYref, (int)cb, yrange);
		}
	if (nam && nam[0]) name = rlp_strdup(nam);
	else name = 0L;
	if (condition && condition[0]) cond = rlp_strdup(condition);
	SetValues();
}
	
DataLine::DataLine(GraphObj *par, DataObj *d, lfPOINT *val, long nval, char *na):GraphObj(par, d)
{  
	FileIO(INIT_VARS);
	Values = (lfPOINT *)calloc(nval + 2, sizeof(lfPOINT));
	memcpy(Values, val, nval * sizeof(lfPOINT));
	nPnt = nval;
	if (na && na[0]) name = rlp_strdup(na);
	else name = 0L;
	Id = GO_DATALINE;
}

DataLine::DataLine(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_DATALINE;
}

DataLine::~DataLine()
{
	if(Values)free(Values);
	Values = 0L;				if(pts) free(pts);
	pts = 0L;					if(ssXref) free(ssXref);
	ssXref = 0L;				if(ssYref) free(ssYref);
	ssYref = 0L;				if(cond) free(cond);
	cond = 0L;					if(mo) DelBitmapClass(mo);
	mo = 0L;					if(name) free(name);
	name = 0L;
}

bool
DataLine::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_DATA_LINE:
		LineDef.color = col;		return true;
	case COL_POLYGON:
		pgFill.color = col;
		LineDef.color = ((col & 0x00fefefeL)>>1);
		return true;
		}
	return false;
}
bool
DataLine::SetSize(int select, double value)
{
	switch (select & 0xfff) {
	case SIZE_DATA_LINE:
		LineDef.width = value;		return true;
		}
	return false;
}

void
DataLine::DoPlot(anyOutput *target)
{
	int i;
	lfPOINT fip;
	POINT pn, *tmppts;

	if(!Values || nPnt < 1) return;
	if (target->OC_type == OC_GRIDVIEW) return;
	if(mo) DelBitmapClass(mo);
	mo = 0L;						if(pts) free(pts);
	pts = 0L;
	if((type & 0xff) == 9 || (type & 0xff) == 10)	//splines
		pts = (POINT *)malloc(sizeof(POINT)*1000);
	else if((type & 0xff) == 11 || (type & 0xff) == 12)			// curve
		pts = (POINT *) malloc(sizeof(POINT)* (nPnt+2)*192);
	else if(type & 0xff) pts = (POINT *)malloc(sizeof(POINT)*(nPnt+4)*2);
	else pts = (POINT *)malloc(sizeof(POINT)*(nPnt+4));
	if(!pts) return;
	if(max.fx > min.fx && max.fy > min.fy) dirty = false;
	else if(dirty) Command(CMD_AUTOSCALE, 0L, target);
	cp = 0;
	switch(type & 0x0f) {
	case 0:		default:
		for (i = 0; i < nPnt; i++){
			if (parent && parent->Id == GO_TERNLINE) {
				pn.x = iround(Values[i].fx);		pn.y = iround(Values[i].fy);
				}
			else {
				target->fp2fip(Values + i, &fip);
				pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
				}
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 5:									case 1:
		target->fp2fip(Values, &fip);
		if ((type & 0x0f) == 5) {
			pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
			target->fp2fip(Values + 1, &fip);
			pn.y += (pn.y - iround(fip.fy)) >> 1;
			AddToPolygon(&cp, pts, &pn);
			}
		pn.x = iround(fip.fx);		pn.y = iround(+fip.fy);
		for (i = 0; i < nPnt; i++){
			target->fp2fip(Values+i, &fip);
			pn.x = iround(fip.fx);			AddToPolygon(&cp, pts, &pn);
			pn.y = iround(fip.fy);			AddToPolygon(&cp, pts, &pn);
			}
		if((type &0xf) == 5) {
			target->fp2fip(Values+i-2, &fip);
			pn.x += (pn.x - iround(fip.fx))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 6:								case 2:
		target->fp2fip(Values, &fip);
		if ((type & 0x0f) == 6) {
			pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
			target->fp2fip(Values + 1, &fip);
			pn.x += (pn.x - iround(fip.fx)) >> 1;
			AddToPolygon(&cp, pts, &pn);
			}
		pn.x = iround(fip.fx);			pn.y = iround(fip.fy);
		for (i = 0; i < nPnt; i++){
			target->fp2fip(Values+i, &fip);
			pn.y = iround(fip.fy);			AddToPolygon(&cp, pts, &pn);
			pn.x = iround(fip.fx);			AddToPolygon(&cp, pts, &pn);
			}
		if((type &0xf) == 6) {
			target->fp2fip(Values+i-2, &fip);
			pn.y += (pn.y - iround(fip.fy))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 7:							case 3:
		target->fp2fip(Values, &fip);
		if ((type & 0x0f) == 7) {
			pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
			target->fp2fip(Values + 1, &fip);
			pn.x += (pn.x - iround(fip.fx)) >> 1;
			AddToPolygon(&cp, pts, &pn);
			}
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		for (i = 0; i < nPnt; i++){
			target->fp2fip(Values+i, &fip);
			pn.x = (pn.x + iround(fip.fx))>>1;		AddToPolygon(&cp, pts, &pn);
			pn.y = iround(fip.fy);				AddToPolygon(&cp, pts, &pn);
			pn.x = iround(fip.fx);
			}
		AddToPolygon(&cp, pts, &pn);
		if((type &0xf) == 7) {
			target->fp2fip(Values+i-2, &fip);
			pn.x += (pn.x - iround(fip.fx))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 8:										case 4:
		target->fp2fip(Values, &fip);
		if ((type & 0x0f) == 8) {
			pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
			target->fp2fip(Values + 1, &fip);
			pn.y += (pn.y - iround(fip.fy)) >> 1;
			AddToPolygon(&cp, pts, &pn);
			}
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		for (i = 0; i < nPnt; i++){
			target->fp2fip(Values+i, &fip);
			pn.y = (pn.y + iround(fip.fy))>>1;		AddToPolygon(&cp, pts, &pn);
			pn.x = iround(fip.fx);				AddToPolygon(&cp, pts, &pn);
			pn.y = iround(fip.fy);
			}
		AddToPolygon(&cp, pts, &pn);
		if((type &0xf) == 8) {
			target->fp2fip(Values+i-2, &fip);
			pn.y += (pn.y - iround(fip.fy))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 9:		case 10:
		DrawSpline(target);
		break;
	case 11:	case 12:
		DrawCurve(target);
		break;
		}
	if(cp < 2) return;
	if (isPolygon && cp != 2) {			//for mark polygon only !!
		AddToPolygon(&cp, pts, pts);
		}
	else{
		target->SetLine(&LineDef);		target->oPolyline(pts, cp);
		}
	tmppts = (POINT*)realloc(pts, cp *sizeof(POINT));
	if (tmppts) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < cp; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 2*target->un2ix(LineDef.width);		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i);
}

void
DataLine::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark){
			if(mo) DelBitmapClass(mo);
			mo = 0L;
			memcpy(&mrc, &rDims, sizeof(RECT));
			IncrementMinMaxRect(&mrc, 6 + o->un2ix(LineDef.width));
			mo = GetRectBitmap(&mrc, o);
			InvertLine(pts, cp, &LineDef, &mrc, o, mark);
			}
		else if (mo) {
			RestoreRectBitmap(&mo, &mrc, o);
			DelBitmapClass(mo);			mo = 0L;
			}
		}
}

bool
DataLine::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	bool bFound = false;
	POINT p1;
	int i;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(rDims, (p1.x= mev->x), (p1.y= mev->y)) || CurrGO || !o || nPnt <1)
				return false; 
			if(isPolygon && IsInPolygon(&p1, pts, cp, 3.0)) bFound = true;
			if (bFound || IsCloseToPL(p1, pts, cp, 3.0)) {
				return o->ShowMark(this, MRK_GODRAW);
				}
			}
		break;
	case CMD_SCALE:
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;		LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		break;
	case CMD_SET_DATAOBJ:
		Id = isPolygon ? GO_DATAPOLYGON : GO_DATALINE;
		data = (DataObj*)tmpl;
		return true;
	case CMD_MRK_DIRTY:
		dirty= true;
		return false;
	case CMD_REDRAW:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, 0L);
		return false;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND) {
			if(Id == GO_DATALINE) ((Legend*)tmpl)->HasFill(&LineDef, 0L, name);
			}
		break;
	case CMD_SET_LINE:
		if(tmpl) memcpy(&LineDef, tmpl, sizeof(LineDEF));
		return true;
	case CMD_UPDATE:
		Undo.DataMem(this, (void**)&Values, nPnt * sizeof(lfPOINT), &nPnt, UNDO_CONTINUE);
		Undo.ValLong(this, &nPnt, UNDO_CONTINUE);
		if(Id != GO_ERRORPOLYGON) SetValues();
		return true;
	case CMD_AUTOSCALE:
		if(nPnt < 1 || !Values) return false;
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			if(dirty) {
				min.fx = max.fx = Values[0].fx;	min.fy = max.fy = Values[0].fy;
				for (i = 1; i < nPnt; i++){
					min.fx = Values[i].fx < min.fx ? Values[i].fx : min.fx;
					max.fx = Values[i].fx > max.fx ? Values[i].fx : max.fx;
					min.fy = Values[i].fy < min.fy ? Values[i].fy : min.fy;
					max.fy = Values[i].fy > max.fy ? Values[i].fy : max.fy;
					}
				}
			((Plot*)parent)->CheckBounds(min.fx, min.fy);
			((Plot*)parent)->CheckBounds(max.fx, max.fy);
			dirty = false;
			return true;
			}
		return false;
	case CMD_SET_COND:
		if(((char*)tmpl)[0]) {
			if(cond) free(cond);
#ifdef USE_WIN_SECURE
			cond = _strdup((char*)tmpl);
#else
			cond = strdup((char*)tmpl);
#endif
			if (cond && cond[0]) return true;
			}
		return false;
	case CMD_COPY:
		if (parent) return parent->Command(CMD_COPY, this, 0L);
		return false;
	case CMD_COPY_TSV:
		return MemList((unsigned char**)tmpl, FF_TSV);
		}
	return false;
}

void
DataLine::SetValues()
{
	AccRange *rX, *rY=0L;
	anyResult *cRes, xRes, yRes;
	bool bValid = false;
	long i, j, k, l;
	double x = 0.0, y = 0.0;
	lfPOINT *tmpValues = Values;
	TextValue *c_xtv, *c_ytv;

	if(!ssXref || !ssYref || !data) return;
	if (parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
		c_xtv = ((Plot*)parent)->x_tv;			c_ytv = ((Plot*)parent)->y_tv;
		}
	else return;
	dirty = true;					nPnt = 0;
	rX = new AccRange(ssXref);		rY = new AccRange(ssYref);
	if (!rX || !rY){
		if (rX) delete(rX);
		if (rY) delete(rY);
		return;
		}
	if (!name) name = rlp_strdup(rY->RangeDesc(data, 1));
	cp = 0;
	if ((nPnt = rX->CountItems()) != (rY->CountItems())) return;
	if (!(Values = (lfPOINT *)calloc(nPnt + 2, sizeof(lfPOINT)))) return;
	if (rX->GetFirst(&i, &j) && rY->GetFirst(&k, &l) &&	rX->GetNext(&i, &j) && rY->GetNext(&k, &l)) do {
		if (data->GetResult(&xRes, j, i, false) && data->GetResult(&yRes, l, k, false)
			&& xRes.type != ET_UNKNOWN && yRes.type != ET_UNKNOWN) {
			switch (xRes.type) {
			case ET_VALUE:	case ET_DATE:	case ET_DATETIME:	case ET_TIME:	case ET_BOOL:
				x = xRes.value;		bValid = true;		 break;
			case ET_TEXT:
				if (c_xtv && !c_xtv->HasValue(xRes.text, &x)) bValid = false;
				if (!c_xtv) bValid = false;				//probably range header
				break;
			default:				bValid = false;		break;
				}
			switch (yRes.type) {
			case ET_VALUE:	case ET_DATE:	case ET_DATETIME:	case ET_TIME:	case ET_BOOL:
				y = yRes.value;		bValid = true;		 break;
			case ET_TEXT:
				if (c_ytv && !c_ytv->HasValue(yRes.text, &y)) bValid = false;
				if (!c_ytv) bValid = false;				//probably range header
				break;
			default:				bValid = false;		break;
				}
			if (bValid && cond && cond[0]) {
#ifdef USE_WIN_SECURE
				sprintf_s((char*)TmpTxt, 100, "x=%g;y=%g;%s", x, y, cond);
#else
				sprintf((char*)TmpTxt, "x=%g;y=%g;%s", x, y, cond);
#endif
				cRes = do_formula(data, TmpTxt);
				if (cRes->type == ET_BOOL && cRes->value != 0.0) {
					Values[cp].fx = x;				Values[cp++].fy = y;
					}
				}
			else if (bValid){
				Values[cp].fx = x;				Values[cp++].fy = y;
				}
			}
		} while (rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
	nPnt = cp;			dirty = true;
	Command(CMD_AUTOSCALE, 0L, 0L);
	if (tmpValues && Values != tmpValues) Undo.InvalidGO(this);
	if (rX) delete(rX);
	if (rY) delete(rY);
}

void
DataLine::LineData(lfPOINT *val, long nval)
{
	lfPOINT *ov = Values;

	if(!val || nval <2) return;
	dirty = true;
	if(nval > nPnt && nPnt > 1 && Values){
		if(!(Values = (lfPOINT *)realloc(Values, ((nval*2+2) * sizeof(lfPOINT))))) return; 
		if(ov != Values) Undo.InvalidGO(this);
		}
	else if(!Undo.isBusy()) Undo.DataMem(this, (void**)&Values, nPnt * sizeof(lfPOINT), &nPnt, UNDO_CONTINUE);
	memcpy(Values, val, nval * sizeof(lfPOINT));
	if(pts) free(pts);
	pts = 0L;					dirty = true;			
	free(val);					nPnt = nval;
}

void
DataLine::DrawCurve(anyOutput *target)
{
	lfPOINT *sdata, *bdata;
	POINT *tmppts;
	int i, j, n;

	if(!(sdata = (lfPOINT *)malloc(nPnt * sizeof(lfPOINT))))return;
	sdata[0].fx = Values[0].fx;				sdata[0].fy = Values[0].fy;
	for(i = j = 1; i < nPnt; i++) {
		if(Values[i].fx != sdata[j-1].fx || Values[i].fy != sdata[j-1].fy) {
			sdata[j].fx = Values[i].fx;		sdata[j++].fy = Values[i].fy;
			}
		}
	n = mkCurve(sdata, j, &bdata, (type&0x0f) != 11);
	if(!(tmppts = (POINT*)malloc((n*64+2)*sizeof(POINT))))return;
	for(i = 0; i < n; i++){
		tmppts[i].x = target->fx2ix(bdata[i].fx);	tmppts[i].y = target->fy2iy(bdata[i].fy);
		}
	for(i = cp = 0; i< (n-2); i += 3) {
		if(parent->Id == GO_TICK) ClipBezier(&cp, pts, tmppts[i], tmppts[i+1], tmppts[i+2], tmppts[i+3], 0L, 0L);
		else DrawBezier(&cp, pts, tmppts[i], tmppts[i+1], tmppts[i+2], tmppts[i+3], 0);
		}
	if(bdata) free(bdata);
	free(sdata);
}

void
DataLine::DrawSpline(anyOutput *target)
{
	int i, j, k, klo, khi, ptsize = 1000;
	double *y2, min, max, x, y, h, b, a;
	POINT pn;
	lfPOINT *scvals;
	
	if(!(y2 = (double*)malloc(sizeof(double)*(nPnt)))) return;
	if(!(scvals = (lfPOINT*)malloc(sizeof(lfPOINT)*(nPnt)))){
		free(y2);
		return;
		}
	if((type & 0x0f) == 9 || (type & 0x0f) == 10) {
		if((type & 0x0f) == 9) for(i = 0; i < nPnt; i++) {
			scvals[i].fx = target->fx2fix(Values[i].fx);
			scvals[i].fy = target->fy2fiy(Values[i].fy);
			}
		else for(i = 0; i < nPnt; i++) {
			scvals[i].fy = target->fx2fix(Values[i].fx);
			scvals[i].fx = target->fy2fiy(Values[i].fy);
			}
		SortFpArray(nPnt, scvals);
		min = scvals[0].fx;			max = scvals[nPnt-1].fx;
		for(i = j = 0; i < (nPnt-1); i++, j++) {
			y = scvals[i].fy;			scvals[j].fx = scvals[i].fx;
			for(k = 1; scvals[i+1].fx == scvals[i].fx; k++) {
				y += scvals[i+1].fy;		i++;
				}
			scvals[j].fy = y/((double)k);
			}
		if(scvals[i].fx > scvals[i-1].fx) {
			scvals[j].fx = scvals[i].fx;	scvals[j].fy = scvals[i].fy;
			j++;
			}
		spline(scvals, j, y2);
		h = scvals[1].fx - scvals[0].fx;	// klo and khi bracket the input value of x
		for(x = min, klo = 0, i = khi = 1; x < max && i < j; x += 1.0) {
			while(x > scvals[i].fx) {
				klo++;		khi++;	i++;
				h = scvals[khi].fx - scvals[klo].fx;
				}
			a = (scvals[khi].fx - x) / h;		b = (x - scvals[klo].fx) / h;
			y = a * scvals[klo].fy + b * scvals[khi].fy + ((a*a*a - a) * y2[klo] + (b*b*b - b) * y2[khi]) * (h*h)/6.0;
			if((type & 0x0f) == 9) {
				pn.x = iround(x);		pn.y = iround(y);
				}
			else {
				pn.x = iround(y);		pn.y = iround(x);
				}
			if(cp >= ptsize) {
				ptsize += 1000;
				pts = (POINT*)realloc(pts, sizeof(POINT)*ptsize); 
				}
			AddToPolygon(&cp, pts, &pn);
			}
		}
	free(y2);	free(scvals);
}

bool
DataLine::MemList(unsigned char **ptr, int type)
{
	long i, nc, cbd = 0, size, start, end;

	start = 0;		end = nPnt;
	if (!nPnt) return false;
	switch (type){
	case FF_TSV:
		if (!(*ptr = (unsigned char *)malloc(size = 10000)))return false;
		for (i = start; i < end; i++){
#ifdef USE_WIN_SECURE
			nc = sprintf_s(TmpTxt, TMP_TXT_SIZE, "%g\t%g\n", Values[i].fx, Values[i].fy);
#else
			nc = sprintf(TmpTxt, "%g\t%g\n", Values[i].fx, Values[i].fy);
#endif
			add_to_buff((char**)ptr, &cbd, &size, TmpTxt, nc);
			}
		if (cbd > 5) return true;
		free(*ptr);	*ptr = 0L;
		return false;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// DataPolygon is a graphic object based on DataLine
DataPolygon::DataPolygon(GraphObj *par, DataObj *d, char *xrange, char *yrange, char *nam):
	DataLine(par, d, xrange, yrange, nam)
{
	lfPOINT *fp = Values;
	char *rx = ssXref;
	char *ry = ssYref;
	long np = nPnt;

	FileIO(INIT_VARS);
	Values = fp;			//FileIO will just set Values to 0L !
	ssXref = rx;			ssYref = ry;
	src_nPnt = nPnt = np;
	Id = GO_DATAPOLYGON;
}

DataPolygon::DataPolygon(GraphObj *par, DataObj *d, lfPOINT *val, long nval, char *na):
	DataLine(par, d, val, nval, 0L)
{
	FileIO(INIT_VARS);
	Values = (lfPOINT *)calloc(nval + 2, sizeof(lfPOINT));
	memcpy(Values, val, nval * sizeof(lfPOINT));
	nPnt = nval;
	if (!name && na && na[0]) name = rlp_strdup(na);
	src_nPnt = nPnt;
	Id = GO_DATAPOLYGON;
}

DataPolygon::DataPolygon(int src):DataLine(0L, 0)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_DATAPOLYGON;
}

DataPolygon::~DataPolygon()
{
	if(Values)free(Values);
	Values =0L;					if(pts) free (pts);
	pts = 0L;					if(ssXref) free(ssXref);
	ssXref = 0L;				if(ssYref) free(ssYref);
	ssYref = 0L;				if(mo) DelBitmapClass(mo);
	mo = 0L;					if(name) free(name);
	name = 0;
	if(parent)parent->Command(CMD_MRK_DIRTY, 0L, 0L);
}

void
DataPolygon::DoPlot(anyOutput *o)
{
	LineDEF currLine;

	if(!Values || !o || nPnt < 2) return;
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	if (type & 0x100) {								//expand polygon to baseline
		Values = (lfPOINT*)realloc(Values, (nPnt + 3) * sizeof(lfPOINT));
		Values[nPnt].fy = Values[nPnt + 1].fy = o->yAxis.min;
		Values[nPnt].fx = Values[nPnt-1].fx;
		Values[nPnt + 1].fx = Values[0].fx;
		nPnt += 2;
		}
	else if((type & 0x0f) != 12 && (type & 0x0f)) {		//close polygon if necessary
		if(Values[nPnt].fx != Values[0].fx || Values[nPnt].fy != Values[0].fy) {
			Values = (lfPOINT*)realloc(Values, (nPnt+2) * sizeof(lfPOINT));
			Values[nPnt+1].fx = Values[0].fx;	Values[nPnt+1].fy = Values[0].fy;
			nPnt++;
			}
		}
	DataLine::DoPlot(o);			//no drawing but fill pts only
	memcpy(&currLine, &LineDef, sizeof(LineDEF));
	if(currLine.width < 1.0e-10) currLine.color = pgFill.color;
	if (cp > 2){
		o->SetLine(&currLine);		o->SetFill(&pgFill);
		o->oPolygon(pts, cp);
		}
	if (type & 0x100) {								//revoke expanded polygon to baseline
		nPnt -= 2;
		}
}

void
DataPolygon::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark){
			if(mo) DelBitmapClass(mo);
			mo = 0L;
			memcpy(&mrc, &rDims, sizeof(RECT));
			IncrementMinMaxRect(&mrc, 6 + o->un2ix(LineDef.width));
			mo = GetRectBitmap(&mrc, o);
			InvertLine(pts, cp, &LineDef, &mrc, o, mark);
			}
		else if(mo){
			RestoreRectBitmap(&mo, &mrc, o);
			DelBitmapClass(mo);
			mo = 0L;
			}
		}
}

bool
DataPolygon::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_PG_FILL:
		if(tmpl) {
			memcpy((void*)&pgFill, tmpl, sizeof(FillDEF));
			if(pgFill.hatch) memcpy((void*)&pgFillLine, (void*)pgFill.hatch, sizeof(LineDEF));
			pgFill.hatch = (LineDEF*)&pgFillLine;
			}
		return true;
	case CMD_SCALE:
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;			LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		pgFillLine.width *= ((scaleINFO*)tmpl)->sy.fy;		pgFillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		pgFill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		break;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND) {
			if(Id == GO_DATAPOLYGON || Id == GO_ERRORPOLYGON) ((Legend*)tmpl)->HasFill(&LineDef, &pgFill, name);
			}
		break;
	default:
		return DataLine::Command(cmd, tmpl, o);
		}
	return false;
}

//integrate polygon: trapezoidal integration
double
DataPolygon::Area()
{
	long i;

	if (area > 0.0) return area;
	area = 0.0;
	for (i = 0; i < (nPnt-1); i++) {
		if ((Values[i + 1].fx) != Values[i].fx) {
			area += (Values[i + 1].fx - Values[i].fx) * (Values[i + 1].fy + Values[i].fy);
			}
		}
	area += (Values[i].fx - Values[0].fx) * (Values[i].fy + Values[0].fy);
	area = fabs(area / 2.0);		if (area == 0.0) area = 1.0e-16;
	return (area);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// ErrorPolygon draws the error or confidence range as a polygon
ErrorPolygon::ErrorPolygon(GraphObj *par, DataObj *d, int nlow, lfPOINT *lowvals, int nhigh, lfPOINT *hivals) :
	DataPolygon(par, d, 0L, 0L, (char *)"Error Polygon")
{
	FileIO(INIT_VARS);		Values = 0L;
	if (lowvals && hivals && nlow && nhigh) SetValues(nlow, lowvals, nhigh, hivals);
	Id = GO_ERRORPOLYGON;
}

ErrorPolygon::ErrorPolygon(GraphObj *par, DataObj *d, char *xrange, char *yrange, char *erange, char *name):
	DataPolygon(par, d, 0L, 0L, name)
{
	FileIO(INIT_VARS);
	ssXref = rlp_strdup(xrange);		ssYref = rlp_strdup(yrange);
	ErrRange = rlp_strdup(erange);		Id = GO_ERRORPOLYGON;
}

ErrorPolygon::ErrorPolygon(GraphObj *par, DataObj *d, char *xrange, char *lo_range, char *hi_range) :
	DataPolygon(par, d, 0L, 0L, (char *)"Error Polygon")
{
	FileIO(INIT_VARS);		Values = 0L;
	if (xrange && xrange[0] && lo_range && lo_range[0] && hi_range && hi_range[0]) {
		ssXref = rlp_strdup(xrange);			LoRange = rlp_strdup(lo_range);
		HiRange = rlp_strdup(hi_range);
		}
	Id = GO_ERRORPOLYGON;
}

ErrorPolygon::ErrorPolygon(int src) :DataPolygon(0L, 0L, 0L, 0L, (char *)"Error Polygon")
{
	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_ERRORPOLYGON;
}

ErrorPolygon::~ErrorPolygon()
{
	if (Values) free(Values);
	Values = 0L;					if (name)	free(name);
	name = 0L;						if (ErrRange) free(ErrRange);
	ErrRange = 0L;					if (ssXref) free(ssXref);
	ssXref = 0L;					if (LoRange) free(LoRange);
	LoRange = 0L;					if (HiRange) free(HiRange);
	HiRange = 0L;
}

void
ErrorPolygon::DoPlot(anyOutput *target)
{
	if (!Values && ssXref && ssYref && ErrRange) Update();
	DataPolygon::DoPlot(target);
}

bool 
ErrorPolygon::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_SET_DATAOBJ:
		data = (DataObj*)tmpl;
		Id = GO_ERRORPOLYGON;
		return true;
	case CMD_UPDATE:
		Update();
		}
	return DataPolygon::Command(cmd, tmpl, o);
}

bool
ErrorPolygon::SetValues(long nlow, lfPOINT *lowvals, long nhigh, lfPOINT *hivals)
{
	long i;
	double *tmpx, *tmpy;

	if (!lowvals || !hivals || !nlow || !nhigh) return false;
	if (Values) free(Values);
	Values = (lfPOINT*)malloc((nlow + nhigh + 2)*sizeof(lfPOINT));
	tmpx = (double*)malloc((nlow + nhigh + 2)*sizeof(double));
	tmpy = (double*)malloc((nlow + nhigh + 2)*sizeof(double));
	for (i = 0; i < nlow; i++) {
		tmpx[i] = lowvals[i].fx;		tmpy[i] = lowvals[i].fy;
		}
	SortArray2(nlow, tmpx, tmpy);
	for (nPnt = i = 0; i < nlow; i++) {
		Values[nPnt].fx = tmpx[i];		Values[nPnt++].fy = tmpy[i];
		}
	for (i = 0; i < nhigh; i++) {
		tmpx[i] = hivals[i].fx;			tmpy[i] = hivals[i].fy;
		}
	SortArray2(nhigh, tmpx, tmpy);
	for (i = nhigh - 1; i >=0; i--) {
		Values[nPnt].fx = tmpx[i];		Values[nPnt++].fy = tmpy[i];
		}
	free(tmpx);			free(tmpy);
	return true;
}

//get error values from the spreadsheet
bool
ErrorPolygon::LoadValues()
{
	long nv, i, j, k, l, m, n, n1 = 0, n2 = 0;
	AccRange *rX = 0L, *rY1 = 0L, *rY2 = 0L;
	double x, y1, y2;
	double *tmpx1, *tmpx2, *tmpy1, *tmpy2;

	if (!ssXref || !LoRange || !HiRange)return false;
	if (Values) free(Values);
	Values = 0L;					nPnt = 0;
	rX = new AccRange(ssXref);		rY1 = new AccRange(LoRange);
	rY2 = new AccRange(HiRange);	nv = rX->CountItems();
	tmpx1 = (double*)malloc((nv + 2)*sizeof(double));
	tmpx2 = (double*)malloc((nv + 2)*sizeof(double));
	tmpy1 = (double*)malloc((nv + 2)*sizeof(double));
	tmpy2 = (double*)malloc((nv + 2)*sizeof(double));
	rX->GetFirst(&i, &j);			rY1->GetFirst(&k, &l);			rY2->GetFirst(&m, &n);
	rX->GetNext(&i, &j);			rY1->GetNext(&k, &l);			rY2->GetNext(&m, &n);
	do {
		if (data->GetValue(j, i, &x)){
			if (data->GetValue(l, k, &y1)) {
				tmpx1[n1] = x;	tmpy1[n1++] = y1;
				}
			if (data->GetValue(n, m, &y2)) {
				tmpx2[n2] = x;	tmpy2[n2++] = y2;
				}
			}
		} while (rX->GetNext(&i, &j) && rY1->GetNext(&k, &l) && rY2->GetNext(&m, &n));
	if (n1 > 2 && n2 > 2){
		Values = (lfPOINT*)malloc((n1 + n2 + 2)*sizeof(lfPOINT));
		SortArray2(n1, tmpx1, tmpy1);				SortArray2(n2, tmpx2, tmpy2);
		for (nPnt = i = 0; i < n1; i++) {
			Values[nPnt].fx = tmpx1[i];		Values[nPnt++].fy = tmpy1[i];
			}
		for (i = n2 - 1; i >= 0; i--) {
			Values[nPnt].fx = tmpx2[i];		Values[nPnt++].fy = tmpy2[i];
			}
		}
	if (rX) delete rX;
	rX = 0L;		if (rY1) delete rY1;
	rY1 = 0L;		if (rY2) delete rY2;
	rY2 = 0L;		if (tmpx1) free(tmpx1);
	tmpx1 = 0L;		if (tmpx2) free(tmpx2);
	tmpx2 = 0L;		if (tmpy1) free(tmpy1);
	tmpy1 = 0L;		if (tmpy2) free(tmpy2);
	tmpy2 = 0L;
	return (nPnt > 3);
}

bool
ErrorPolygon::Update()
{
	int n;
	long c1, r1, c2, r2, c3, r3, nval = 0;
	double x, y, e;
	lfPOINT *low_vals, *high_vals;
	AccRange *rX, *rY, *rE;
	anyResult xRes, yRes, eRes;
	bool RetVal = false, bValid_x;
	TextValue *c_xtv = 0L;

	if (!ssXref || !ssYref || !ErrRange) return false;
	pgBounds.Xmin = pgBounds.Ymin = HUGE_VAL;		pgBounds.Xmax = pgBounds.Ymax = -HUGE_VAL;
	if (!(rX = new AccRange(ssXref)) || !(rY = new AccRange(ssYref)) || !(rE = new AccRange(ErrRange))
		|| !(n = rX->CountItems()) || !(low_vals = (lfPOINT*)malloc((n+2)*sizeof(lfPOINT))) 
		|| !(high_vals = (lfPOINT*)malloc((n+2)*sizeof(lfPOINT)))) return false;
	rX->GetFirst(&c1, &r1);		rY->GetFirst(&c2, &r2);		rE->GetFirst(&c3, &r3);
	rX->GetNext(&c1, &r1);		rY->GetNext(&c2, &r2);		rE->GetNext(&c3, &r3);
	do {
		if (data->GetResult(&xRes, r1, c1, false) && data->GetResult(&yRes, r2, c2, false)
			&& data->GetResult(&eRes, r3, c3, false)){
			bValid_x = false;
			if (xRes.type == ET_TEXT) {
				if (c_xtv && c_xtv->HasValue(xRes.text, &xRes.value)) bValid_x = true;
				}
			else bValid_x = true;
			if (bValid_x && yRes.type == ET_VALUE && eRes.type == ET_VALUE) {
				x = xRes.value;				y = yRes.value;			e = eRes.value;
				low_vals[nval].fx = high_vals[nval].fx = x;
				high_vals[nval].fy = y + e;						low_vals[nval].fy = y - e;
				if (pgBounds.Xmin > x) pgBounds.Xmin = x;
				if (pgBounds.Xmax < x) pgBounds.Xmax = x;
				if (pgBounds.Ymin > low_vals[nval].fy) pgBounds.Ymin = low_vals[nval].fy;
				if (pgBounds.Ymax < high_vals[nval].fy) pgBounds.Ymax = high_vals[nval].fy;
				nval++;
				}
			}
		} while (rX->GetNext(&c1, &r1) && rY->GetNext(&c2, &r2) && rE->GetNext(&c3, &r3));
	if (nval > 2){
		SetValues(nval, low_vals, nval, high_vals);
		RetVal = true;
		}
	if (parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
		((Plot*)parent)->CheckBounds(pgBounds.Xmin, pgBounds.Ymin);
		((Plot*)parent)->CheckBounds(pgBounds.Xmax, pgBounds.Ymax);
		}
	free(low_vals);					free(high_vals);
	delete rX;		delete rY;		delete rE;
	return RetVal;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// LineTernary is a graphic object based on DataLine in a ternary plot
LineTernary::LineTernary(GraphObj *par, DataObj *d, char *rng1, char *, char* rng3, char *name) :
DataLine(par, d, 0L, 0L, name)
{
	FileIO(INIT_VARS);
	range1 = rlp_strdup(rng1);		range2 = rlp_strdup(rng3);
	Id = GO_TERNLINE;
	DoUpdate();
}

LineTernary::LineTernary(int src) : DataLine(0L, 0)
{
	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_TERNLINE;
}

LineTernary::~LineTernary()
{
	if (range1) free(range1);
	range1 = NULL;
	if (range2) free(range2);
	range2 = NULL;
	if (range3) free(range3);
	range3 = NULL;
	if (DispLine) delete DispLine;
	DispLine = NULL;
	if (values_xyz) free(values_xyz);
	values_xyz = NULL;
}

void
LineTernary::DoPlot(anyOutput *o)
{
	if (!o) return;						//Check first if the viewpoint or output has changed
	if (o != CurrDisp || o->getVPorgX() != zoomOrg.fx || o->getVPorgY() != zoomOrg.fy || o->getVPorgScale() != zoomScale) {
		if (DispLine) {
			Undo.InvalidGO(DispLine);
			delete(DispLine);			DispLine = NULL;
			}
		CurrDisp = o;
		}
	if (!DispLine) MkDispLine(o);
	if (DispLine){
		DispLine->Command(CMD_SET_LINE, (void*)&LineDef, o);
		DispLine->DoPlot(o);
		}
}
bool 
LineTernary::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd){
	case CMD_SET_DATAOBJ:
		Id = GO_TERNLINE;
		if (DispLine) DispLine->parent = this;
		return true;
	case CMD_UPDATE:
		DoUpdate();
		if (DispLine) {
			Undo.InvalidGO(DispLine);
			delete DispLine;
			DispLine = 0L;
			}
		return true;
	case CMD_SET_LINE:
		memcpy(&LineDef, tmpl, sizeof(LineDEF));
		if (DispLine) DispLine->Command(cmd, tmpl, o);
		return true;
	case CMD_PG_FILL:
		memcpy(&Fill, tmpl, sizeof(FillDEF));
		if (Fill.hatch)memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;
		if (DispLine) DispLine->Command(cmd, tmpl, o);
		return true;
	case CMD_MOUSE_EVENT:
		if (DispLine) {
			if (DispLine->Id == GO_DATAPOLYGON) return ((DataPolygon*)DispLine)->Command(cmd, tmpl, o);
			else if (DispLine->Id == GO_DATALINE) return ((DataLine*)DispLine)->Command(cmd, tmpl, o);
			}
		return false;
	case CMD_LEGEND:
		if (DispLine && doPG) return ((DataPolygon*)DispLine)->Command(cmd, tmpl, o);
		return false;
		}
	return false;
}

bool
LineTernary::MkDispLine(anyOutput *o)
{
	long nPts;
	lfPOINT *val;
	int i;
	double fix, fiy;

	if(!parent || parent->Id != GO_TERNARYXYZ) return false;
	if (!values_xyz || !nValues) DoUpdate();
	if (nValues < 2) return false;
	if (DispLine) {
		Undo.InvalidGO(DispLine);
		delete (DispLine);
		DispLine = NULL;
		}
	zoomOrg.fx = o->getVPorgX();		zoomOrg.fy = o->getVPorgY();		zoomScale = o->getVPorgScale();
	val = (lfPOINT*)malloc((nValues + 2)*sizeof(lfPOINT));
	for (i = 0, nPts = 0; i < nValues; i++){
		if (((TernaryXYZ *)parent)->TransformVal(o, &values_xyz[i], &fix, &fiy)) {
			val[nPts].fx = iround(fix);		val[nPts].fy = iround(fiy);		nPts++;
			}
		}
	//The DataLine or DataPolygon object take ownership of the val data array
	if (doPG){
		DispLine = new DataPolygon(this, data, val, nPts, NULL);
		DispLine->Command(CMD_PG_FILL, &Fill, o);
		DispLine->Command(CMD_SET_LINE, &LineDef, o);
		}
	else {
		DispLine = new DataLine(this, data, val, nPts, NULL);
		DispLine->Command(CMD_SET_LINE, &LineDef, o);
		}
	if (val) free(val);
	return (nPts > 1);
}

bool
LineTernary::DoUpdate()
{
	int n1 = 0, ic = 0;
	long i, j, k, l, m, n;
	double x, y, z;
	AccRange *rX = 0L, *rY = 0L, *rZ = 0L;

	if (!range1 || !range2 || !range3) return false;
	rX = new AccRange(range1);			rY = new AccRange(range2);
	rZ = new AccRange(range3);
	if (values_xyz) free(values_xyz);
	values_xyz = NULL;
	if (rX) n1 = rX->CountItems();
	if (n1 && rY && rZ && (values_xyz = (fPOINT3D*)malloc(n1 * sizeof(fPOINT3D)))){
		rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
		rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
		rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
		do {
			if (data->GetValue(j, i, &x) && data->GetValue(l, k, &y) &&
				data->GetValue(n, m, &z)){
				values_xyz[ic].fx = x;	values_xyz[ic].fy = y;	values_xyz[ic].fz = z;
				ic++;
				}
			} while (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
		nValues = ic;
		}
	if (rX) delete(rX);
	if (rY) delete(rY);
	if (rZ) delete(rZ);
	return (nValues > 1);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a regression line
// Ref.: "Biometry" third edition 1995 (ed. R.R. Sokal and F.J. Rohlf),
// W.H. Freeman and Company, New York; ISBN 0-7167-2411-1; pp. 451ff
RegLine::RegLine(GraphObj *par, DataObj *d, lfPOINT *values, long n, int sel):
	GraphObj(par, d)
{
	FileIO(INIT_VARS);
	type = sel;
	Id = GO_REGLINE;
	uclip.Xmin = uclip.Ymin = lim.Xmin = lim.Ymin = -1.0;
	uclip.Xmax = uclip.Ymax = lim.Xmax = lim.Ymax = 1.0;
	Recalc(values, n);
}

RegLine::RegLine(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) FileIO(FILE_READ);
}

RegLine::~RegLine()
{
	if(pts) free(pts);
	pts = 0L;
	if(parent)parent->Command(CMD_MRK_DIRTY, 0L, 0L);
}

double
RegLine::GetSize(int select)
{
	double a, b;

	switch(select) {
	case SIZE_MX:		return mx;
	case SIZE_MY:		return my;
	case SIZE_A:
	case SIZE_B:
		switch(type & 0x07) {
		case 1:		a = l2.fx;	b = l2.fy;	break;
		case 2:		a = l3.fx;	b = l3.fy;	break;
		case 3:		a = l4.fx;	b = l4.fy;	break;
		case 4:		a = l5.fx;	b = l5.fy;	break;
		default:	a = l1.fx;	b = l1.fy;	break;
			}
		if(select == SIZE_A) return a;
		else return b;
		}
	return 0.0;
}

void
RegLine::DoPlot(anyOutput *o)
{
	int i;
	POINT pn;
	double x, x1, y, d, a, b;
	fRECT cliprc;
	bool dValid;

	switch (type & 0x70) {
	case 0x20:	memcpy(&cliprc, &uclip, sizeof(fRECT));		break;
	case 0x10:	default:
		if (parent) {
			cliprc.Xmin = parent->GetSize(SIZE_BOUNDS_LEFT);	cliprc.Xmax = parent->GetSize(SIZE_BOUNDS_RIGHT);
			cliprc.Ymin = parent->GetSize(SIZE_BOUNDS_BOTTOM);	cliprc.Ymax = parent->GetSize(SIZE_BOUNDS_TOP);
			break;
			}
		else {
			memcpy(&cliprc, &lim, sizeof(fRECT));
			}
		break;
		}
	if(cliprc.Xmax < cliprc.Xmin) {
		x = cliprc.Xmax;	cliprc.Xmax = cliprc.Xmin;	cliprc.Xmin = x;
		}
	if(cliprc.Ymax < cliprc.Ymin) {
		y = cliprc.Ymax;	cliprc.Ymax = cliprc.Ymin;	cliprc.Ymin = y;
		}
	if(cliprc.Xmin == cliprc.Xmax || cliprc.Ymin == cliprc.Ymax) return;
	if((!pts) && (!(pts = (POINT *)malloc(sizeof(POINT)*202))))return;
	switch(type & 0x07) {
	case 1:		a = l2.fx;	b = l2.fy;	break;
	case 2:		a = l3.fx;	b = l3.fy;	break;
	case 3:		a = l4.fx;	b = l4.fy;	break;
	case 4:		a = l5.fx;	b = l5.fy;	break;
	default:	a = l1.fx;	b = l1.fy;	break;
		}
	x = cliprc.Xmin;	d = (cliprc.Xmax - cliprc.Xmin)/200.0;
	for (cp = i = 0; i <= 200; i++){
		dValid = true;
		switch(type & 0x700) {
		case 0x100:						//logarithmic x
			dValid = x > defs.min4log;
			if(dValid) x1 = log10(x);
			break;
		case 0x200:						//reciprocal x
			dValid = fabs(x) > defs.min4log;
			if (dValid) x1 = 1.0 / x;
			break;
		case 0x300:						//square root x
			dValid = fabs(x) > defs.min4log;
			if (dValid) x1 = sqrt(x);
			break;
		default:	x1 = x;	break;		//linear x
			}
		if(dValid) {
			y = a + b*x1;
			switch(type & 0x7000) {
			case 0x1000:				//logarithmic y
				y = pow(10.0, y);
				break;
			case 0x2000:				//reciprocal y
				dValid = fabs(y) > 0.0001;
				if (dValid) y = 1.0 / y;
				break;
			case 0x3000:				//square root y
				dValid = fabs(y) > 0.0001;
				if (dValid) y = y*y;
				break;
				}
			if(y >= cliprc.Ymin && y <= cliprc.Ymax) {
				pn.x = o->fx2ix(x);	pn.y = o->fy2iy(y);
				AddToPolygon(&cp, pts, &pn);
				}
			}
		x += d;
		}
	if(cp < 2) return;
	o->SetLine(&LineDef);			o->oPolyline(pts, cp);
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < cp; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 2*o->un2ix(LineDef.width);		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i);
}

void
RegLine::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark)InvertLine(pts, cp, &LineDef, &rDims, o, mark);
		else if(parent) parent->Command(CMD_REDRAW, 0L, o);
		}
}

bool
RegLine::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(rDims, (p1.x= mev->x), (p1.y= mev->y)) || CurrGO || !o || nPoints <2)
				return false; 
			if(IsCloseToPL(p1,pts,cp, 3.0)) return o->ShowMark(CurrGO= this, MRK_GODRAW);
			}
		break;
	case CMD_SCALE:
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;		LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_REGLINE;
		return true;
	case CMD_BOUNDS:
		if(tmpl) {
			memcpy(&lim, tmpl, sizeof(fRECT));
			memcpy(&uclip, tmpl, sizeof(fRECT));
			}
		return true;
	case CMD_AUTOSCALE:
		if(nPoints < 2) return false;
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(lim.Xmin, lim.Ymin);
			((Plot*)parent)->CheckBounds(lim.Xmax, lim.Ymax);
			return true;
			}
		return false;
		}
	return false;
}

void
RegLine::Recalc(lfPOINT *values, long n)
{
	double sx, sy, dx, dy, sxy, sxx, syy;
	double a, b, k;
	long ic;

	sx = sy = 0.0;
	if((nPoints = n)<2) return;
	for(ic = 0; ic < n; ic++) {
		sx += values[ic].fx;		sy += values[ic].fy;
		}
	mx = sx /((double)nPoints);	my = sy/((double)nPoints);
	sxy = sxx = syy = 0.0;
	for(ic = 0; ic < n; ic++) {
		dx = mx - values[ic].fx;	dy = my - values[ic].fy;
		sxx += (dx*dx);	syy += (dy*dy);	sxy += (dx*dy);
		}
	l1.fy = sxy / sxx;			l1.fx = my - (sxy / sxx) * mx;
	b = sxy / syy;				a = mx - (sxy / syy) * my;
	l2.fy = 1.0/b;				l2.fx = -a / b;
	l3.fy = (l1.fy+l2.fy)/2.0;	l3.fx = (l1.fx+l2.fx)/2.0;
	l4.fy = sy/sx;				l4.fx = 0.0;
	if(l5.fx == 0.0 && l5.fx == 0.0){
		l5.fy = l1.fy;				l5.fx = l1.fx;
		}
	//calculate distance point from line algorithm
	//Ref: K. Thompson, 1990: Vertical Distance from a Point to a Line. In:
	//   Graphic Gems (Andrew S. Glassner, ed.), Academic Press,
	//   pp. 47-48; ISBN 0-12-286165-5
	k = (sqrt(1.0/(1.0+l1.fy*l1.fy))+sqrt(1.0/(1.0+l2.fy*l2.fy)))/2.0;
	b = sqrt(1.0/(k*k) -1.0);
	l3.fy = l3.fy > 0.0 ? b : -b;
	l3.fx = my - mx * l3.fy;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a statnard deviation (SD-) ellipse
SDellipse::SDellipse(GraphObj *par, DataObj *d, lfPOINT *values, long n, int sel):
	GraphObj(par, d)
{
	FileIO(INIT_VARS);
	type = sel;					Id = GO_SDELLIPSE;
	val = (lfPOINT*)malloc(n * sizeof(lfPOINT));
	if(val){
		memcpy(val, values, (nPoints = n)*sizeof(lfPOINT));
		rl = new RegLine(this, data, values, n, type);
		}
}

SDellipse::SDellipse(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) FileIO(FILE_READ);
}

SDellipse::~SDellipse()
{
	if(val) free(val);
	if(pts) free(pts);
	if(!(type & 0x10000) && parent && rl && 
		parent->Command(CMD_DROP_OBJECT, rl, 0L)) return;
	if(rl) DeleteGO(rl);
}

void
SDellipse::DoPlot(anyOutput *o)
{
	int i;
	double a1, b1, a2, b2, fv, k1, k2, ss1, ss2, np, x, dx, si, csi, fac, fac2;
	lfPOINT fp, fip;
	POINT p1, *tmppts;

	if(!rl) return;
	if(pts) free(pts);
	if(!(pts = (POINT *)malloc(sizeof(POINT)*420)))return;
	//get line data from regression line object
	mx = rl->GetSize(SIZE_MX);		my = rl->GetSize(SIZE_MY);
	a1 = rl->GetSize(SIZE_A);		b1 = rl->GetSize(SIZE_B);
	b2 = -1.0/b1;	a2 = my - b2 * mx;
	//calculate sine and cosine for back rotation
	fv = sqrt(1.0+b1*b1);			si = b1/fv;			csi = 1.0/fv;
	//calculate distance from line for each point and squared sum of distances
	//Ref: K. Thompson, 1990: Vertical Distance from a Point to a Line. In:
	//   Graphic Gems (Andrew S. Glassner, ed.), Academic Press,
	//   pp. 47-48; ISBN 0-12-286165-5
	k1 = sqrt(1.0/(1.0+b1*b1));			k2 = sqrt(1.0/(1.0+b2*b2));
	// y = a + b*x;
	ss1 = ss2 = 0.0;
	for(i = 0; i < nPoints; i++) {
		fv = (a1 + b1 * val[i].fx - val[i].fy) * k1;	ss1 += (fv*fv);
		fv = (a2 + b2 * val[i].fx - val[i].fy) * k2;	ss2 += (fv*fv);
		}
	np = ((double)(nPoints-1));
	//SD perpendicular and in direction of regression line
	sd1 = sqrt(ss1 /= np);		sd2 = sqrt(ss2 /= np);
	switch(type & 0x60000) {
		case 0x20000:		fac = 2.0;		break;
		case 0x40000:		fac = 3.0;		break;
		default:			fac = 1.0;		break;
		}
	fac2 = fac*fac;			dx = sd2/100.0*fac;
	for(i = 0, cp = 0, x = -(sd2*fac); i < 2; i++) {
		do {
			fv = (x*x)/(ss2*fac2);
			fv = fv < 0.99999 ? sqrt((1.0-fv)*ss1*fac2) : 0.0;
			fv = i ? fv : -fv;
			fp.fx = mx + x * csi - fv * si;
			fp.fy = my + x * si + fv * csi;
			switch(type & 0x700) {
			case 0x100:					//logarithmic x
				fp.fx = pow(10.0, fp.fx);
				break;
			case 0x200:					//reciprocal x
				if(fabs(fp.fx) > defs.min4log) fp.fx = 1.0/fp.fx;
				else fp.fx = 0.0;
				break;
			case 0x300:					//square root x
				if(fabs(fp.fx) > defs.min4log) fp.fx = fp.fx*fp.fx;
				else fp.fx = 0.0;
				break;
				}
			switch(type & 0x7000) {
			case 0x1000:				//logarithmic y
				fp.fy = pow(10.0, fp.fy);
				break;
			case 0x2000:				//reciprocal y
				if(fabs(fp.fy) > defs.min4log) fp.fy = 1.0/fp.fy;
				else fp.fy = 0.0;
				break;
			case 0x3000:				//square root y
				if(fabs(fp.fy) > defs.min4log) fp.fy = fp.fy*fp.fy;
				else fp.fy = 0.0;
				break;
				}
			o->fp2fip(&fp, &fip);	p1.x = iround(fip.fx);		p1.y = iround(fip.fy);
			AddToPolygon(&cp, pts, &p1);
			}while((x += dx) < (sd2*fac) && x > (-sd2*fac));
		x = sd2*fac;
		dx *= -1.0;
		}
	o->SetLine(&LineDef);
	if(cp > 2) {
		AddToPolygon(&cp, pts, pts);		//close polygon
		tmppts = (POINT*)realloc(pts, cp *sizeof(POINT));
		if (tmppts) pts = tmppts;
		SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
		for(i = 2; i < cp; i++) 
			UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
		i = 3*o->un2ix(LineDef.width);		//increase size of rectangle for marks
		IncrementMinMaxRect(&rDims, i);
		o->oPolyline(pts, cp);
		}
	else {
		free(pts);
		cp = 0;
		}
	if(!(type & 0x10000))rl->DoPlot(o);
}

void
SDellipse::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark)InvertLine(pts, cp, &LineDef, &rDims, o, mark);
		else if(parent) parent->Command(CMD_REDRAW, 0L, o);
		}
}

bool
SDellipse::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!(type & 0x10000) && rl && rl->Command(cmd, tmpl, o)) return true;
			if(!IsInRect(rDims, (p1.x= mev->x), (p1.y= mev->y)) || CurrGO || !o || nPoints <2)
				return false; 
			if(IsCloseToPL(p1,pts,cp, 3.0)) return o->ShowMark(CurrGO= this, MRK_GODRAW);
			}
		break;
	case CMD_SCALE:
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;		LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		if(rl) rl->Command(cmd, tmpl, o);
		break;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_RMU:
		if(!(type & 0x10000) && parent && rl && parent->Command(CMD_DROP_OBJECT, rl, o)){
			rl = 0L;
			return true;
			}
		return false;
	case CMD_INIT:
		if(rl) return rl->PropertyDlg();
		break;
	case CMD_DROP_OBJECT:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_REGLINE && !rl) {
			rl = (RegLine *)tmpl;
			rl->parent = this;
			return true;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_SDELLIPSE;
		if(rl) rl->Command(cmd, tmpl, o);
		return true;
	case CMD_BOUNDS:
		if(tmpl) {
			if(rl) rl->Command(cmd, tmpl, o);
			memcpy(&lim, tmpl, sizeof(fRECT));
			}
		return true;
	case CMD_DELOBJ:
		if(tmpl && tmpl == (void*)rl) {
			Undo.ValLong(parent, &type, 0L);
			type |= 0x10000;
			return true;
			}
		break;
	case CMD_AUTOSCALE:
		if(nPoints < 2) return false;
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(lim.Xmin, lim.Ymin);
			((Plot*)parent)->CheckBounds(lim.Xmax, lim.Ymax);
			return true;
			}
		break;
		}
	return false;
}

void
SDellipse::Recalc(lfPOINT *values, long n)
{
	if(val) free(val);
	val = (lfPOINT*)malloc(n * sizeof(lfPOINT));
	if (val) memcpy(val, values, (nPoints = n)*sizeof(lfPOINT));
	if(rl) rl->Recalc(values, n);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Error bars are simple graphic objects
ErrorBar::ErrorBar(GraphObj *par, DataObj *d, double x, double y, double err, int which,
	int xc, int xr, int yc, int yr, int ec, int er):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;
	ferr = err;			type = which;
	Id = GO_ERRBAR;		data = d;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || ec >= 0 || er >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 3);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = ec;	ssRef[2].y = er;
			cssRef = 3;
			}
		}
	Command(CMD_AUTOSCALE, 0L, 0L);
}

ErrorBar::ErrorBar(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	type = ERRBAR_VSYM;
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_ERRBAR;
}

ErrorBar::~ErrorBar()
{
	if(mo) DelBitmapClass(mo);
	mo = 0L;					if(ssRef) free(ssRef);
	ssRef = 0L;					if(name) free(name);
	name = 0L;
}

bool
ErrorBar::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ERRBAR: 
		SizeBar = NiceValue(value);		return true;
	case SIZE_ERRBAR_LINE:
		ErrLine.width = value;			return true;
	case SIZE_XPOS:
		fPos.fx = value;				return true;
	case SIZE_YPOS:
		fPos.fy = value;				return true;
	case SIZE_ERRBAR_VALUE:
		ferr = value;					return true;
		}
	return false;
}

bool
ErrorBar::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ERROR_LINE:
		ErrLine.color = col;
		return true;
		}
	return false;
}

DWORD
ErrorBar::GetColor(int select)
{
	switch (select & 0xfff) {
	case COL_ERROR_LINE:
		return ErrLine.color;
		}
	return 0L;
}

double
ErrorBar::GetSize(int select)
{
	switch (select & 0xfff) {
	case SIZE_ERRBAR:
		return SizeBar;
	case SIZE_ERRBAR_LINE:
		return ErrLine.width;
	case SIZE_ERRBAR_VALUE:
		return ferr;
		}
	return 0.0;
}

void
ErrorBar::DoPlot(anyOutput *target)
{
	int ie;

	switch (type & 0x0ff) {
	case ERRBAR_VSYM:		case ERRBAR_VUP:		case ERRBAR_VDOWN:
		ie = target->un2ix(SizeBar/2.0);			break;
	default:
		ie = target->un2iy(SizeBar/2.0);			break;
		}
	target->SetLine(&ErrLine);
	switch(type & 0x0ff) {
	case ERRBAR_VSYM:
		if (!parent) {		//Dialog?
			ebpts[0].x = ebpts[1].x = iround(fPos.fx);
			ebpts[0].y = iround(fPos.fy - ferr);
			ebpts[4].y = ebpts[5].y = ebpts[1].y = iround(fPos.fy + ferr);
			}
		else {				//user data!
			if ((type & 0x300) == 0x300){
				ebpts[0].x = ebpts[1].x = iround(target->fx2fix(fPos.fx) + parent->GetSize(SIZE_BAR_DX));
				}
			else {
				ebpts[0].x = ebpts[1].x = target->fx2ix(fPos.fx);
				}
			ebpts[0].y = target->fy2iy(fPos.fy - ferr);
			ebpts[4].y = ebpts[5].y = ebpts[1].y = target->fy2iy(fPos.fy + ferr);
			}
		if(ebpts[1].y != ebpts[0].y) target->oSolidLine(ebpts);
		ebpts[4].x = ebpts[2].x = ebpts[0].x - ie;		ebpts[5].x = ebpts[3].x = ebpts[1].x + ie+1;
		ebpts[2].y = ebpts[3].y = ebpts[0].y;
		if(ebpts[3].x > ebpts[2].x) {
			target->oSolidLine(ebpts+2);						target->oSolidLine(ebpts+4);
			}
		rDims.left =  ebpts[2].x;		rDims.right = ebpts[3].x;
		rDims.top = ebpts[0].y;			rDims.bottom = ebpts[1].y;
		break;
	case ERRBAR_VUP:		case ERRBAR_VDOWN:
		if (!parent) {		//Dialog?
			ebpts[0].x = ebpts[1].x = iround(fPos.fx);			ebpts[0].y = iround(fPos.fy);
			ebpts[2].y = ebpts[3].y = ebpts[1].y =
				iround((type & 0x0ff) == ERRBAR_VUP ? fPos.fy - ferr : fPos.fy + ferr);
			}
		else {				//user data!
			if ((type & 0x300) == 0x300){
				ebpts[0].x = ebpts[1].x = iround(target->fx2fix(fPos.fx) + parent->GetSize(SIZE_BAR_DX));
				}
			else {
				ebpts[0].x = ebpts[1].x = target->fx2ix(fPos.fx);
				}
			ebpts[0].y = target->fy2iy(fPos.fy);				ebpts[2].y = ebpts[3].y = ebpts[1].y =
				target->fy2iy((type & 0x0ff) == ERRBAR_VUP ? fPos.fy + ferr : fPos.fy - ferr);
			}
		if(ebpts[1].y != ebpts[0].y) target->oSolidLine(ebpts);
		ebpts[2].x = ebpts[0].x - ie;		ebpts[3].x = ebpts[1].x + ie+1;
		if(ebpts[3].x > ebpts[2].x) target->oSolidLine(ebpts+2);
		rDims.left =  ebpts[2].x;		rDims.right = ebpts[3].x;
		rDims.top = ebpts[0].y;			rDims.bottom = ebpts[1].y;
		break;
	case ERRBAR_HSYM:
		if (!parent) {		//Dialog?
			ebpts[2].x = ebpts[3].x = ebpts[0].x = iround(fPos.fx - ferr);
			ebpts[4].x = ebpts[5].x = ebpts[1].x = iround(fPos.fx + ferr);
			ebpts[0].y = ebpts[1].y = iround(fPos.fy);
			}
		else {				//user data!
			ebpts[2].x = ebpts[3].x = ebpts[0].x = target->fx2ix(fPos.fx - ferr);
			ebpts[4].x = ebpts[5].x = ebpts[1].x = target->fx2ix(fPos.fx + ferr);
			ebpts[0].y = ebpts[1].y = target->fy2iy(fPos.fy);
			}
		if(ebpts[1].x != ebpts[0].x) target->oSolidLine(ebpts);
		ebpts[2].y = ebpts[4].y = ebpts[0].y - ie;
		ebpts[3].y = ebpts[5].y = ebpts[1].y + ie+1;
		if(ebpts[3].y >ebpts[2].y) {
			target->oSolidLine(ebpts+2);			target->oSolidLine(ebpts+4);
			}
		rDims.left =  ebpts[0].x;		rDims.right = ebpts[1].x;
		rDims.top = ebpts[2].y;			rDims.bottom = ebpts[3].y;
		break;
	case ERRBAR_HLEFT:		case ERRBAR_HRIGHT:
		if (!parent) {		//Dialog?
			ebpts[0].x = iround(fPos.fx);
			ebpts[0].y = ebpts[1].y = iround(fPos.fy);
			ebpts[2].x = ebpts[3].x = ebpts[1].x =
				iround((type & 0x0ff) == ERRBAR_HRIGHT ? fPos.fx + ferr : fPos.fx - ferr);
			}
		else {				//user data!
			ebpts[0].x = target->fx2ix(fPos.fx);
			ebpts[0].y = ebpts[1].y = target->fy2iy(fPos.fy);
			ebpts[2].x = ebpts[3].x = ebpts[1].x =
				target->fx2ix((type & 0x0ff) == ERRBAR_HRIGHT ? fPos.fx + ferr : fPos.fx - ferr);
			}
		if(ebpts[1].x != ebpts[0].x) target->oSolidLine(ebpts);
		ebpts[2].y = ebpts[0].y - ie;			ebpts[3].y = ebpts[1].y + ie+1;
		if(ebpts[3].y > ebpts[2].y) target->oSolidLine(ebpts+2);
		rDims.left =  ebpts[0].x;		rDims.right = ebpts[1].x;
		rDims.top = ebpts[2].y;			rDims.bottom = ebpts[3].y;
		break;
		}
	if(rDims.left > rDims.right) Swap(rDims.left, rDims.right);
	if(rDims.top > rDims.bottom) Swap(rDims.top, rDims.bottom);
	IncrementMinMaxRect(&rDims, 2);
}

void
ErrorBar::DoMark(anyOutput *o, bool mark)
{
	int i;
	LineDEF OldLine;

	if(mark){
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		memcpy(&OldLine, &ErrLine, sizeof(LineDEF));
		i = 3*o->un2ix(ErrLine.width);		//increase size of rectangle for marks
		IncrementMinMaxRect(&mrc, i);		mo = GetRectBitmap(&mrc, o);
		ErrLine.width *= 5.0;				DoPlot(o);
		ErrLine.width = OldLine.width;		ErrLine.color = OldLine.color ^ 0x00ffffffL;
		DoPlot(o);							o->UpdateRect(&mrc, true);
		memcpy(&ErrLine, &OldLine, sizeof(LineDEF));
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
ErrorBar::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	ErrorBar *prevsym;
	bool bFound;
	int cb;

	switch (cmd) {
	case CMD_SCALE:
		ErrLine.width *= ((scaleINFO*)tmpl)->sy.fy;		ErrLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		SizeBar *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_SELECT:
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		bFound = false;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(rDims, mev->x, mev->y) || CurrGO) return false; 
			switch (type &0x0ff) {
			case ERRBAR_HSYM:		case ERRBAR_HLEFT:		case ERRBAR_HRIGHT:
				if(mev->y >= (ebpts[0].y-2) && mev->y <= (ebpts[1].y+2)) bFound = true;
				else if(mev->x >= (ebpts[2].x-2) && mev->x <= (ebpts[2].x+2)) bFound = true;
				else if((type & 0x0ff) == ERRBAR_HSYM && mev->x >= (ebpts[4].x-2) &&
					mev->x <= (ebpts[4].x + 2)) bFound = true;
				break;
			case ERRBAR_VSYM:		case ERRBAR_VUP:		case ERRBAR_VDOWN:
				if(mev->x >= (ebpts[0].x-2) && mev->x <= (ebpts[1].x+2)) bFound = true;
				else if(mev->y >= (ebpts[2].y-2) && mev->y <= (ebpts[2].y+2)) bFound = true;
				else if((type & 0x0ff) == ERRBAR_VSYM && mev->y >= (ebpts[4].y-2) &&
					mev->y <= (ebpts[4].y + 2)) bFound = true;
				break;
				}
			if(bFound) o->ShowMark(this, MRK_GODRAW);
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_ERRBAR;
		data = (DataObj *) tmpl;
		return true;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &ferr);
			return true;
			}
		return false;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		switch(type & 0x0ff) {
			case ERRBAR_VSYM:		case ERRBAR_VUP:		case ERRBAR_VDOWN:
				((Legend*)tmpl)->HasErr(&ErrLine, 1, name);
				break;
			case ERRBAR_HSYM:		case ERRBAR_HLEFT:		case ERRBAR_HRIGHT:
				((Legend*)tmpl)->HasErr(&ErrLine, 2, name);
				break;
			}
		break;
	case CMD_ERRDESC:
		if(tmpl && *((char *)tmpl)){
			cb = rlp_strlen((char*)tmpl)+2;
			name = (char*)realloc(name, cb);
			if(name) rlp_strcpy(name, cb, (char*)tmpl);
			return true;
			}
		return false;
	case CMD_ERR_TYPE:
		if(tmpl) type = *((int*)tmpl);
		return true;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			switch(type & 0xff) {
			default:
			case ERRBAR_VSYM:	
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy+ferr);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy-ferr);		break;
			case ERRBAR_VUP:	
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy+ferr);		break;
			case ERRBAR_VDOWN:
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy-ferr);		break;
			case ERRBAR_HSYM:
				((Plot*)parent)->CheckBounds(fPos.fx+ferr, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx-ferr, fPos.fy);		break;
			case ERRBAR_HLEFT:
				((Plot*)parent)->CheckBounds(fPos.fx-ferr, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);				break;
			case ERRBAR_HRIGHT:
				((Plot*)parent)->CheckBounds(fPos.fx+ferr, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);				break;
				}
			return true;
			}
		break;
	case  CMD_PREVSYM:
		if (tmpl && ((GraphObj*)tmpl)->Id == GO_ERRBAR) {
			prevsym = (ErrorBar*)tmpl;
			SizeBar = prevsym->GetSize(SIZE_ERRBAR);				ErrLine.width = prevsym->GetSize(SIZE_ERRBAR_LINE);
			ErrLine.color = prevsym->GetColor(COL_ERROR_LINE);		type = prevsym->type;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Error bars in 3D space
ErrBar3D::ErrBar3D(GraphObj *par, DataObj *d, double x, double y, double z, double err, int which,
	int xc, int xr, int yc, int yr, int zc, int zr, int ec, int er):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;		fPos.fz = z;
	ferr = err;		type = which;
	Id = GO_ERRBAR3D;		data = d;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || zc >= 0 || zr >= 0 || ec >= 0 || er >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 4);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			ssRef[3].x = ec;	ssRef[3].y = er;
			cssRef = 4;
			}
		}
	Command(CMD_AUTOSCALE, 0L, 0L);
}

ErrBar3D::ErrBar3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	type = 0;
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

ErrBar3D::~ErrBar3D()
{
	int i;

	if(mo) DelBitmapClass(mo);
	mo = 0L;						if(ssRef) free(ssRef);
	ssRef = 0L;						if(name) free(name);
	name = 0L;
	for (i = 0; i < 3; i++){
		if (ls[i]) delete(ls[i]);
		}
}

bool
ErrBar3D::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ERRBAR: 
		SizeBar = value;
		return true;
	case SIZE_ERRBAR_LINE:
		ErrLine.width = value;
		return true;
		}
	return false;
}

bool
ErrBar3D::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ERROR_LINE:
		ErrLine.color = col;
		return true;
		}
	return false;
}

void
ErrBar3D::DoPlot(anyOutput *o)
{
	fPOINT3D fip1, fip2, pos1, pos2, tmp, itmp;
	fPOINT3D p1, p2;
	int i;

	if(mo) DelBitmapClass(mo);
	mo = 0L;
	pos1.fx = pos2.fx = fPos.fx;	pos1.fy = pos2.fy = fPos.fy;	pos1.fz = pos2.fz = fPos.fz;
	tmp.fx = tmp.fy = tmp.fz = 0.0;
	switch(type & 0x0f) {
	case 0:
		pos1.fy = fPos.fy - ferr; 	pos2.fy = fPos.fy + ferr;
		tmp.fx = SizeBar/2.0;
		break;
	case 1:
		pos2.fy = fPos.fy + ferr;	tmp.fx = SizeBar/2.0;
		break;
	case 2:
		pos1.fy = fPos.fy - ferr;	tmp.fx = SizeBar/2.0;
		break;
	case 3:
		pos1.fx = fPos.fx - ferr;	pos2.fx = fPos.fx + ferr;
		tmp.fy = SizeBar/2.0;
		break;
	case 4:
		pos1.fx = fPos.fx + ferr;	tmp.fy = SizeBar/2.0;
		break;
	case 5:
		pos2.fx = fPos.fx - ferr;	tmp.fy = SizeBar/2.0;
		break;
	case 6:
		pos1.fz = fPos.fz - ferr;	pos2.fz = fPos.fz + ferr;
		tmp.fy = SizeBar/2.0;
		break;
	case 7:
		pos2.fz = fPos.fz + ferr;	tmp.fy = SizeBar/2.0;
		break;
	case 8:
		pos2.fz = fPos.fz - ferr;	tmp.fy = SizeBar/2.0;
		break;
	default:
		return;
		}
	if(!parent || !o || !o->fvec2ivec(&pos1, &fip1) ||!o->fvec2ivec(&pos2, &fip2)) return;
	for(i = 0; i < 3; i++){
		if(ls[i]) delete(ls[i]);
		ls[i] = 0L;
		}
//	p1.x = iround(fip1.fx);	p1.y = iround(fip1.fy);	p1.z = iround(fip1.fz);
//	p2.x = iround(fip2.fx);	p2.y = iround(fip2.fy);	p2.z = iround(fip2.fz);
	p1.fx = fip1.fx;		p1.fy = fip1.fy;	p1.fz = fip1.fz;
	p2.fx = fip2.fx;		p2.fy = fip2.fy;	p2.fz = fip2.fz;
	if (p1.fx == p2.fx && p1.fy == p2.fy && p1.fz == p2.fz) return;	//zero length errror
	mpts[0][0].x = iround(p1.fx);	mpts[0][0].y = iround(p1.fy);
	mpts[0][1].x = iround(p2.fx);	mpts[0][1].y = iround(p2.fy);
	rDims.left = rDims.right = iround(p1.fx);	rDims.top = rDims.bottom = iround(p1.fy);
	UpdateMinMaxRect(&rDims, iround(p2.fx), iround(p2.fy));
	ls[0] = new line_segment(this, data, &ErrLine, &p1, &p2);
	if (ls[0]) ls[0]->DoPlot(o);
	o->uvec2ivec(&tmp, &itmp);
	if(pos1.fx != fPos.fx || pos1.fy != fPos.fy || pos1.fz != fPos.fz) {
		p1.fx = fip1.fx-itmp.fx;		p1.fy = fip1.fy-itmp.fy;		p1.fz = fip1.fz-itmp.fy;
		p2.fx = fip1.fx+itmp.fx;		p2.fy = fip1.fy+itmp.fy;		p2.fz = fip1.fz+itmp.fy;
		if(p1.fx == p2.fx && p1.fy == p2.fy && p1.fz == p2.fz) return;	//zero length cap
		mpts[1][0].x = iround(p1.fx);	mpts[1][0].y = iround(p1.fy);
		mpts[1][1].x = iround(p2.fx);	mpts[1][1].y = iround(p2.fy);
		UpdateMinMaxRect(&rDims, iround(p1.fx), iround(p1.fy));
		UpdateMinMaxRect(&rDims, iround(p2.fx), iround(p2.fy));
		ls[1] = new line_segment(this, data, &ErrLine, &p1, &p2);
		if (ls[1]) ls[1]->DoPlot(o);
		}
	if(pos2.fx != fPos.fx || pos2.fy != fPos.fy || pos2.fz != fPos.fz) {
		p1.fx = fip2.fx-itmp.fx;		p1.fy = fip2.fy-itmp.fy;		p1.fz = fip2.fz-itmp.fy;
		p2.fx = fip2.fx+itmp.fx;		p2.fy = fip2.fy+itmp.fy;		p2.fz = fip2.fz+itmp.fy;
		mpts[2][0].x = iround(p1.fx);	mpts[2][0].y = iround(p1.fy);
		mpts[2][1].x = iround(p2.fx);	mpts[2][1].y = iround(p2.fy);
		UpdateMinMaxRect(&rDims, iround(p1.fx), iround(p1.fy));
		UpdateMinMaxRect(&rDims, iround(p2.fx), iround(p2.fy));
		ls[2] = new line_segment(this, data, &ErrLine, &p1, &p2);
		if (ls[2]) ls[2]->DoPlot(o);
		}
	IncrementMinMaxRect(&rDims, 5);
}

void
ErrBar3D::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(ErrLine.width));
		mo = GetRectBitmap(&mrc, o);
		for(i = 2; i >= 0 ; i--) if(ls[i]) InvertLine(mpts[i], 2, &ErrLine, 0L, o, mark);
		o->UpdateRect(&rDims, true);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
ErrBar3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i, cb;

	switch (cmd) {
	case CMD_SCALE:
		ErrLine.width *= ((scaleINFO*)tmpl)->sy.fy;		ErrLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		SizeBar *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				for(i = 0; i < 3; i++) {
					if(ls[i] && ls[i]->ObjThere(mev->x, mev->y)){
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
					}
				}
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_ERRBAR3D;
		data = (DataObj *) tmpl;
		return true;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			data->GetValue(ssRef[3].y, ssRef[3].x, &ferr);
			return true;
			}
		return false;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		((Legend*)tmpl)->HasErr(&ErrLine, 1, name);
		break;
	case CMD_ERRDESC:
		if(tmpl && *((char *)tmpl)){
			cb = (int)strlen((char*)tmpl)+2;
			name = (char*)realloc(name, cb);
			if (name) rlp_strcpy(name, cb, (char*)tmpl);
			return true;
			}
		return false;
	case CMD_ERR_TYPE:
		if(tmpl) type = *((int*)tmpl);
		return true;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			switch (type & 0x0f) {
			case 0:
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy+ferr, fPos.fz);
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy-ferr, fPos.fz);
				break;
			case 1:
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy+ferr, fPos.fz);
				break;
			case 2:
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy-ferr, fPos.fz);
				break;
			case 3:
				((Plot*)parent)->CheckBounds3D(fPos.fx+ferr, fPos.fy, fPos.fz);
				((Plot*)parent)->CheckBounds3D(fPos.fx-ferr, fPos.fy, fPos.fz);
				break;
			case 4:
				((Plot*)parent)->CheckBounds3D(fPos.fx+ferr, fPos.fy, fPos.fz);
				break;
			case 5:
				((Plot*)parent)->CheckBounds3D(fPos.fx-ferr, fPos.fy, fPos.fz);
				break;
			case 6:
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz+ferr);
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz-ferr);
				break;
			case 7:
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz+ferr);
				break;
			case 8:
				((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz-ferr);
				break;
				}
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// arrows to data points or with absolute coordinates
Arrow::Arrow(GraphObj * par, DataObj *d, lfPOINT *fp1, lfPOINT *fp2, int which,
	int xc1, int xr1, int yc1, int yr1, int xc2, int xr2, int yc2, int yr2):
	GraphObj(par, d)
{
	double dx, dy;

	FileIO(INIT_VARS);
	if (fp1){
		memcpy(&pos1, fp1, sizeof(lfPOINT));
		}
	else {
		pos1.fx = pos1.fy = 0;
		}
	if (fp2) {
		memcpy(&pos2, fp2, sizeof(lfPOINT));
		}
	else {
		pos2.fx = pos2.fy = 0.0;
		}
	type = which;
	if(parent && (type & ARROW_UNITS)) {
		dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
		pos1.fx -= dx;	pos1.fy -= dy;			pos2.fx -= dx;	pos2.fy -= dy;
		}
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || xc2 >= 0 || xr2 >= 0 || 
		yc2 >= 0 || yr2 >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 4);
		if (ssRef) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = xc2;	ssRef[2].y = xr2;
			ssRef[3].x = yc2;	ssRef[3].y = yr2;
			cssRef = 4;
			}
		}
	Id = GO_ARROW;		bModified = false;
}

Arrow::Arrow(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_ARROW;		bModified = false;
}

Arrow::~Arrow()
{
	Command(CMD_FLUSH, 0L, 0L);
	if (name) free(name);
	name = 0L;
	if(bModified) Undo.InvalidGO(this);
}

double
Arrow::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:				return pos1.fx;
	case SIZE_XPOS+1:			return pos2.fx;
	case SIZE_YPOS:				return pos1.fy;
	case SIZE_YPOS+1:			return pos2.fy;
	case SIZE_ARROW_LINE:		return LineDef.width;
	case SIZE_ARROW_CAPWIDTH:	return cw;
	case SIZE_ARROW_CAPLENGTH:	return cl;
	case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
	case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
Arrow::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ARROW_LINE:		LineDef.width = value;		return true;
	case SIZE_ARROW_CAPWIDTH:	cw = value;					return true;
	case SIZE_ARROW_CAPLENGTH:	cl = value;					return true;
	case SIZE_XPOS:				pos1.fx = value;			return true;
	case SIZE_XPOS+1:			pos2.fx = value;			return true;
	case SIZE_YPOS:				pos1.fy = value;			return true;
	case SIZE_YPOS+1:			pos2.fy = value;			return true;
		}
	return false;
}

bool
Arrow::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ARROW:
		LineDef.color = col;
		return true;
		}
	return false;
}

void
Arrow::DoPlot(anyOutput *o)
{
	double dx, dy, fix1, fiy1, fix2, fiy2, len, size;
	GraphObj *go, *sy;

	if (!o || o->OC_type == OC_GRIDVIEW) return;
	if (mo) DelBitmapClass(mo);
	mo = 0L;
	if (!parent){		// ist this arrow part of a dialog?
		//position is given in pixel coordinates
		fix1 = pos1.fx;			fix2 = pos2.fx;
		fiy1 = pos1.fy;			fiy2 = pos2.fy;
		}
	else if (type & ARROW_UNITS) {
		dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
		fix1 = o->co2fix(pos1.fx + dx);	fix2 = o->co2fix(pos2.fx + dx);
		fiy1 = o->co2fiy(pos1.fy + dy);	fiy2 = o->co2fiy(pos2.fy + dy);
		}
	else {
		fix1 = o->fx2fix(pos1.fx);		fix2 = o->fx2fix(pos2.fx);
		fiy1 = o->fy2fiy(pos1.fy);		fiy2 = o->fy2fiy(pos2.fy);
		if (type & ARROW_SYMSPACE) {
			go = parent;
			while (go && go->Id != GO_GRAPH){
				go = go->parent;
				}
			if (go){
				dx = fix2 - fix1;	dy = fiy2 - fiy1;
				len = sqrt(dx * dx + dy * dy);
				if (len < 1.0e-12) return;
				dx /= len;			dy /= len;
				if ((sy = (GraphObj*)go->ObjThere((int)fix2, (int)fiy2)) && sy->Id == GO_SYMBOL){
					size = o->un2fix(((Symbol*)sy)->GetSize(SIZE_SYMBOL) / 2.0 + LineDef.width);
					fix2 -= size * dx;				fiy2 -= size * dy;
					}
				if ((sy = (GraphObj*)go->ObjThere((int)fix1, (int)fiy1)) && sy->Id == GO_SYMBOL){
					size = o->un2fix(((Symbol*)sy)->GetSize(SIZE_SYMBOL) / 2.0 + LineDef.width);
					fix1 += size * dx;				fiy1 += size * dy;
					}
				}
			}
		}
	DoPlotXY(o, fix1, fiy1, fix2, fiy2);
}

void
Arrow::DoPlotXY(anyOutput *o, double fix1, double fiy1, double fix2, double fiy2)
{
	double si, csi, tmp;

	if (fix1 == fix2 && fiy1 == fiy2) return;	//zero length
	//draw arrow line
	pts[0].x = iround(fix1);		pts[1].x = iround(fix2);
	pts[0].y = iround(fiy1);		pts[1].y = iround(fiy2);
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	//calculate sine and cosine for cap
	si = fiy1-fiy2;
	tmp = fix2 - fix1;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fix2-fix1;
	tmp = fiy2 - fiy1;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	//draw cap
	pts[2].x = pts[1].x - o->un2ix(csi*cl + si*cw/2.0);
	pts[2].y = pts[1].y + o->un2iy(si*cl - csi*cw/2.0);
	pts[3].x = pts[1].x;		pts[3].y = pts[1].y;
	pts[4].x = pts[1].x - o->un2ix(csi*cl - si*cw/2.0);
	pts[4].y = pts[1].y + o->un2iy(si*cl + csi*cw/2.0);
	switch(type & 0xff) {
	case ARROW_NOCAP:
		pts[2].x = pts[3].x = pts[4].x = pts[1].x;
		pts[2].y = pts[3].y = pts[4].y = pts[1].y;
		break;
		}
	UpdateMinMaxRect(&rDims, pts[2].x, pts[2].y);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	IncrementMinMaxRect(&rDims, 3*o->un2ix(LineDef.width)+3);
	Redraw(o);
}

void
Arrow::DoMark(anyOutput *o, bool mark)
{
	LineDEF OldLine;

	if(type & ARROW_UNITS) {
		if(!dh1) dh1 = new dragHandle(this, DH_12);
		if(!dh2) dh2 = new dragHandle(this, DH_22);
		}
	else {
		if (dh1) DeleteGO(dh1);
		if (dh2) DeleteGO(dh2);
		dh1 = dh2 = 0L;
		}
	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));				IncrementMinMaxRect(&mrc, 2);
		if(dh1 && dh2) {
			memcpy(&OldLine, &LineDef, sizeof(LineDEF));
			mo = GetRectBitmap(&mrc, o);				Redraw(o);
			dh1->DoPlot(o);		dh2->DoPlot(o);
			}
		else {
			memcpy(&OldLine, &LineDef, sizeof(LineDEF));
			mo = GetRectBitmap(&mrc, o);				LineDef.color = 0x00000000L;
			LineDef.width = OldLine.width*3.0 +2;		Redraw(o);
			LineDef.width = OldLine.width;				LineDef.color = OldLine.color ^ 0x00ffffffL;
			Redraw(o);									o->UpdateRect(&mrc, true);
			memcpy(&LineDef, &OldLine, sizeof(LineDEF));
			}
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);
		mo = 0L;
		}
}

bool
Arrow::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_SAVEPOS:
		bModified = true;
		Undo.SaveLFP(this, &pos1, 0L);
		Undo.SaveLFP(this, &pos2, UNDO_CONTINUE);
		return true;
	case CMD_SCALE:
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;		LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		cw *= ((scaleINFO*)tmpl)->sy.fy;				cl *= ((scaleINFO*)tmpl)->sy.fy;
		if(type & ARROW_UNITS) {
			pos1.fx = ((scaleINFO*)tmpl)->sx.fx + pos1.fx * ((scaleINFO*)tmpl)->sx.fy;
			pos1.fy = ((scaleINFO*)tmpl)->sy.fx + pos1.fy * ((scaleINFO*)tmpl)->sy.fy;
			pos2.fx = ((scaleINFO*)tmpl)->sx.fx + pos2.fx * ((scaleINFO*)tmpl)->sx.fy;
			pos2.fy = ((scaleINFO*)tmpl)->sy.fx + pos2.fy * ((scaleINFO*)tmpl)->sy.fy;
			}
		return true;
	case CMD_FLUSH:
		if (dh1) DeleteGO(dh1);	
		dh1 = 0L;						if (dh2) DeleteGO(dh2);
		dh2 = 0L;						if(mo) DelBitmapClass(mo);
		mo = 0L;						if(ssRef) free(ssRef);
		ssRef = 0L;						if(name) free(name);
		name = 0L;
		return true;
	case CMD_SELECT:
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
		if(!CurrGO && ObjThere(mev->x, mev->y)){
			o->ShowMark(this, MRK_GODRAW);
			return true;
			}
		}
		break;
	case CMD_ARROW_ORG:
		memcpy(&pos1, tmpl, sizeof(lfPOINT));
		if(ssRef && cssRef >3) 
			ssRef[0].x = ssRef[0].y = ssRef[1].x = ssRef[1].y = -1;
		return true;
	case CMD_ARROW_TYPE:
		if(tmpl) {
			type = (*((long*)tmpl)) & 0x2ff;
			return true;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_ARROW;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &pos1.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &pos1.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &pos2.fx);
			data->GetValue(ssRef[3].y, ssRef[3].x, &pos2.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy);
			((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy);
			return true;
			}
		break;
	case CMD_MRK_DIRTY:				//from Undo ?
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOVE:					case CMD_UNDO_MOVE:
		if (cmd == CMD_MOVE) bModified = true;
		if(type & ARROW_UNITS) {
			if(cmd == CMD_MOVE) Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			pos1.fx += ((lfPOINT*)tmpl)[0].fx;	pos1.fy += ((lfPOINT*)tmpl)[0].fy;
			pos2.fx += ((lfPOINT*)tmpl)[0].fx;	pos2.fy += ((lfPOINT*)tmpl)[0].fy;
			if(o){
				o->StartPage();		parent->DoPlot(o);		o->EndPage();
				}
			return true;
			}
		break;
		}
	return false;
}

void
Arrow::Redraw(anyOutput *o)
{
	FillDEF FillCap;

	o->SetLine(&LineDef);
	o->oSolidLine(pts);
	switch(type & 0xff) {
	case ARROW_NOCAP:
		break;
	case ARROW_LINE:
		o->oSolidLine(pts+2);
		o->oSolidLine(pts+3);
		break;
	case ARROW_TRIANGLE:
		FillCap.type = FILL_NONE;
		FillCap.color = LineDef.color;
		FillCap.scale = 1.0f;
		FillCap.hatch = 0L;
		o->SetFill(&FillCap);
		o->oPolygon(pts+2, 3);
		break;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// arrows for Ternary Plot
ArrTernary::ArrTernary(GraphObj *par, DataObj *d, fPOINT3D *fp1, fPOINT3D *fp2, int which, int xc1, int xr1,
	int yc1, int yr1, int zc1, int zr1, int xc2, int xr2, int yc2,
	int yr2, int zc2, int zr2):Arrow(par, d, 0L, 0L, which)
{
	FileIO(INIT_VARS);
	if (fp1) memcpy(&pos1, fp1, sizeof(fPOINT3D));
	else { pos1.fx = pos1.fy = pos1.fz = 0.0; }
	if (fp2) memcpy(&pos2, fp2, sizeof(fPOINT3D));
	else { pos2.fx = pos2.fy = pos2.fz = 0.0; }
	type = which;
	ssRef[0].x = xc1;	ssRef[0].y = yc1;	ssRef[0].z = zc1;
	ssRef[1].x = xr1;	ssRef[1].y = yr1;	ssRef[1].z = zr1;
	ssRef[2].x = xc2;	ssRef[2].y = yc2;	ssRef[2].z = zc2;
	ssRef[3].x = xr2;	ssRef[3].y = yr2;	ssRef[3].z = zr2;
	Id = GO_ARRTERN;		bModified = false;
}

ArrTernary::ArrTernary(int src) :Arrow(0L, 0L, 0L, 0L, ARROW_LINE)
{
	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_ARRTERN;		bModified = false;
}

ArrTernary::~ArrTernary()
{
	Command(CMD_FLUSH, 0L, 0L);
	if (name) free(name);
	name = 0L;
	if (bModified) Undo.InvalidGO(this);
}
void 
ArrTernary::DoPlot(anyOutput *o)
{
	double fix1 = 0.0, fiy1 = 0.0, fix2 = 0.0, fiy2 = 0.0;
	double dx, dy, len, size;
	GraphObj *go, *sy;

	if (!o || o->OC_type == OC_GRIDVIEW) return;
	if (parent && parent->Id == GO_TERNARYXYZ) {
		((TernaryXYZ *)parent)->TransformVal(o, &pos1, &fix1, &fiy1);
		((TernaryXYZ *)parent)->TransformVal(o, &pos2, &fix2, &fiy2); 
		if (type & ARROW_SYMSPACE) {
			go = parent;
			while (go && go->Id != GO_GRAPH) go = go->parent;
			if (go){
				dx = fix2 - fix1;	dy = fiy2 - fiy1;
				len = sqrt(dx * dx + dy * dy);
				if (len < 1.0e-12) return;
				dx /= len;			dy /= len;
				if ((sy = (GraphObj*)go->ObjThere((int)fix2, (int)fiy2)) && (sy->Id == GO_SYMTERN || sy->Id == GO_SYMBOL)){
					size = o->un2fix(((Symbol*)sy)->GetSize(SIZE_SYMBOL) / 2.0 + LineDef.width);
					fix2 -= size * dx;				fiy2 -= size * dy;
				}
				if ((sy = (GraphObj*)go->ObjThere((int)fix1, (int)fiy1)) && (sy->Id == GO_SYMTERN || sy->Id == GO_SYMBOL)){
					size = o->un2fix(((Symbol*)sy)->GetSize(SIZE_SYMBOL) / 2.0 + LineDef.width);
					fix1 += size * dx;				fiy1 += size * dy;
					}
				}
			}
		DoPlotXY(o, fix1, fiy1, fix2, fiy2);
	}
}

bool
ArrTernary::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_SAVEPOS:
		bModified = true;
		Undo.ValLFP3D(this, &pos1, 0L);
		Undo.ValLFP3D(this, &pos2, UNDO_CONTINUE);
		Undo.ValLFP3D(this, &pos2, UNDO_CONTINUE);
		return true;
	case CMD_SCALE:
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;		LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		cw *= ((scaleINFO*)tmpl)->sy.fy;				cl *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_UPDATE:
		if (!data) return false;
		if (ssRef[0].x >= 0 && ssRef[0].y >= 0 && ssRef[0].z >= 0 && ssRef[1].x >= 0 && ssRef[1].y >= 0 && ssRef[1].z >= 0) {
			data->GetValue(ssRef[1].x, ssRef[0].x, &pos1.fx);			data->GetValue(ssRef[1].y, ssRef[0].y, &pos1.fy);
			data->GetValue(ssRef[1].z, ssRef[0].z, &pos1.fz);
			}
		if (ssRef[2].x >= 0 && ssRef[2].y >= 0 && ssRef[2].z >= 0 && ssRef[3].x >= 0 && ssRef[3].y >= 0 && ssRef[3].z >= 0) {
			data->GetValue(ssRef[3].x, ssRef[2].x, &pos2.fx);			data->GetValue(ssRef[3].y, ssRef[2].y, &pos2.fy);
			data->GetValue(ssRef[3].z, ssRef[2].z, &pos2.fz);
			}
		return true;
	case CMD_FLUSH:
		if (dh1) DeleteGO(dh1);
		dh1 = 0L;						if (dh2) DeleteGO(dh2);
		dh2 = 0L;						if (mo) DelBitmapClass(mo);
		mo = 0L;						if (name) free(name);
		name = 0L;
		return true;
	case CMD_ARROW_TYPE:
		if (tmpl) {
			type &= ~0xff;		type |= (*((int*)tmpl));
			return true;
		}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_ARRTERN;
		data = (DataObj *)tmpl;
		return true;
	case CMD_MRK_DIRTY:			//from Undo ?
	case CMD_REDRAW:			case CMD_MOVE:
		if (cmd == CMD_MOVE) bModified = true;
		if (parent) return parent->Command(cmd, tmpl, o);
		return true;
	case CMD_MOUSE_EVENT:
		return Arrow::Command(cmd, tmpl, o);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// universal boxes
Box::Box(GraphObj * par, DataObj *d, lfPOINT fp1, lfPOINT fp2, int which,
	int xc1, int xr1, int yc1, int yr1, int xc2, int xr2,
		int yc2, int yr2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	memcpy(&pos1, &fp1, sizeof(lfPOINT));
	memcpy(&pos2, &fp2, sizeof(lfPOINT));
	type = which;
	Id = GO_BOX;
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || xc2 >= 0 || xr2 >= 0 || 
		yc2 >= 0 || yr2 >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 4);
		if (ssRef) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = xc2;	ssRef[2].y = xr2;
			ssRef[3].x = yc2;	ssRef[3].y = yr2;
			cssRef = 4;
			}
		}
}

Box::Box(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Box::~Box()
{
	Command(CMD_FLUSH, 0L, 0L);
}

double
Box::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:		return (pos1.fx + pos2.fx)/2.0;
	case SIZE_YPOS:		return (pos1.fy + pos2.fy)/2.0;
	case SIZE_XPOS + 1:	return pos1.fx;
	case SIZE_YPOS + 1:	return pos1.fy;
	case SIZE_XPOS + 2:	return pos2.fx;
	case SIZE_YPOS + 2:	return pos2.fy;
	}
	return 1.0;
}

bool
Box::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_BOX_LINE:		Outline.width = value;		return true;
	case SIZE_BOX:			size = value;				return true;
	case SIZE_XPOS:			pos1.fx = value;			return true;
	case SIZE_XPOS+1:		pos2.fx = value;			return true;
	case SIZE_YPOS:			pos1.fy = value;			return true;
	case SIZE_YPOS+1:		pos2.fy = value;			return true;
		}
	return false;
}

bool
Box::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_BOX_LINE:
		Outline.color = col;
		return true;
		}
	return false;
}

void
Box::DoPlot(anyOutput *o)
{
	int i;
	double si, csi, tmp, fix1, fiy1, fix2, fiy2, fsize, dx, dy;

	if(!parent || !o || size <= 0.001) return;
	o->SetLine(&Outline);		o->SetFill(&Fill);
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	//calculate coordinates
	fix1 = o->fx2fix(pos1.fx);	fix2 = o->fx2fix(pos2.fx);
	fiy1 = o->fy2fiy(pos1.fy);	fiy2 = o->fy2fiy(pos2.fy);
	//calculate sine and cosine
	si = fiy1-fiy2;
	tmp = fix2 - fix1;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fix2-fix1;
	tmp = fiy2 - fiy1;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	if(type & BAR_WIDTHDATA) {			//use e.g. for density distribution
		dx = si * size;					dy = csi * size;
		pts[0].x = o->fx2ix(pos1.fx + dx);	pts[1].x = o->fx2ix(pos2.fx + dx);
		pts[2].x = o->fx2ix(pos2.fx - dx);	pts[3].x = o->fx2ix(pos1.fx - dx);
		pts[0].y = o->fy2iy(pos1.fy + dy);	pts[1].y = o->fy2iy(pos2.fy + dy);
		pts[2].y = o->fy2iy(pos2.fy - dy);	pts[3].y = o->fy2iy(pos1.fy - dy);
		}
	else if(type & BAR_RELWIDTH) {
		if(!parent || (pos1.fy == pos2.fy && pos1.fx == pos2.fx)) return;
		fsize = parent->GetSize(pos1.fy == pos2.fy ? SIZE_BOXMINY : SIZE_BOXMINX);
		fsize = fsize * size /200.0;
		dx = si * fsize;					dy = csi * fsize;
		pts[0].x = o->fx2ix(pos1.fx + dx);	pts[1].x = o->fx2ix(pos2.fx + dx);
		pts[2].x = o->fx2ix(pos2.fx - dx);	pts[3].x = o->fx2ix(pos1.fx - dx);
		pts[0].y = o->fy2iy(pos1.fy + dy);	pts[1].y = o->fy2iy(pos2.fy + dy);
		pts[2].y = o->fy2iy(pos2.fy - dy);	pts[3].y = o->fy2iy(pos1.fy - dy);
		}
	else {
		dx = o->un2fix(si*size/2.0);		dy = o->un2fiy(csi*size/2.0);
		pts[0].x = iround(fix1 + dx);	pts[1].x = iround(fix2 + dx);
		pts[2].x = iround(fix2 - dx);	pts[3].x = iround(fix1 - dx);
		pts[0].y = iround(fiy1 + dy);	pts[1].y = iround(fiy2 + dy);
		pts[2].y = iround(fiy2 - dy);	pts[3].y = iround(fiy1 - dy);
		}
	pts[4].x = pts[0].x;		pts[4].y = pts[0].y;	//close polygon
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[2].x, pts[2].y);
	UpdateMinMaxRect(&rDims, pts[1].x, pts[1].y);
	UpdateMinMaxRect(&rDims, pts[3].x, pts[3].y);
	i = o->ClipStatus(&rDims);
	switch (i) {
	default:
	case 0:		break;							//not clipped
	case 1:	
		DrawLater.AddObj(this);		break;		//horizontal clipped
	case 8:
		break;									//a hidden box
		}
	if (pts[0].x == pts[1].x){
		o->oRectangle(pts[3].x, pts[3].y, pts[1].x, pts[1].y, name);
		}
	else {
		o->oPolygon(pts, 5);
		}
	IncrementMinMaxRect(&rDims, o->un2ix(Outline.width*6)+3);
}

void
Box::DoMark(anyOutput *o, bool mark)
{
	if(mark){
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, o->un2ix(Outline.width*6)+3);
		mo = GetRectBitmap(&mrc, o);
		InvertLine(pts, 5, &Outline, &rDims, o, mark);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);
		mo = 0L;
		}
}

bool
Box::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p;

	switch (cmd) {
	case CMD_SCALE:
		Outline.width *= ((scaleINFO*)tmpl)->sy.fy;
		Hatchline.width *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		if(!(type & BAR_RELWIDTH) && parent->Id!= GO_DENSDISP)size *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_FLUSH:
		if(ssRef) free(ssRef);
		ssRef = 0L;						if(name)free(name);
		name = 0L;						if(mo) DelBitmapClass(mo);
		mo = 0L;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		((Legend*)tmpl)->HasFill(&Outline, &Fill, name);
		break;
	case CMD_MRK_DIRTY:					case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SELECT:
		o->ShowMark(CurrGO = this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, (p.x = mev->x), (p.y = mev->y)) && !CurrGO) {
				if(IsInPolygon(&p, pts, 5, 3.0)) {
					o->ShowMark(CurrGO = this, MRK_GODRAW);
					return true;
					}
				}
			break;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_BOX;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &pos1.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &pos1.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &pos2.fx);
			data->GetValue(ssRef[3].y, ssRef[3].x, &pos2.fy);
			return true;
			}
		return false;
	case CMD_BOX_TYPE:
		if(tmpl)type = *((int*)tmpl);
		return true;
	case CMD_BOX_FILL:
		if(tmpl) {
			memcpy(&Fill, tmpl, sizeof(FillDEF));
			if(Fill.hatch) memcpy(&Hatchline, Fill.hatch, sizeof(LineDEF));
			Fill.hatch = &Hatchline;
			return true;
			}
		break;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			if(type & BAR_WIDTHDATA) {
				if(pos1.fy != pos2.fy) {
					((Plot*)parent)->CheckBounds(pos1.fx+size, pos1.fy);
					((Plot*)parent)->CheckBounds(pos2.fx-size, pos2.fy);
					}
				else {
					((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy+size);
					((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy-size);
					}
				}
			else {
				((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy);
				((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy);
				if(parent && (type & BAR_RELWIDTH)) {
					if(pos1.fy == pos2.fy) {
						((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy + parent->GetSize(SIZE_BOXMINY));
						((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy - parent->GetSize(SIZE_BOXMINY));
						}
					else if(pos1.fx == pos2.fx) {
						((Plot*)parent)->CheckBounds(pos2.fx + parent->GetSize(SIZE_BOXMINX), pos2.fy);
						((Plot*)parent)->CheckBounds(pos2.fx - parent->GetSize(SIZE_BOXMINX), pos2.fy);
						}
					}
				}
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// whisker 
Whisker::Whisker(GraphObj *par, DataObj *d, lfPOINT fp1, lfPOINT fp2, int which,
	int xc1, int xr1, int yc1, int yr1, int xc2, int xr2,
	int yc2, int yr2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	memcpy(&pos1, &fp1, sizeof(lfPOINT));
	memcpy(&pos2, &fp2, sizeof(lfPOINT));
	type = which;
	Id = GO_WHISKER;
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || xc2 >= 0 || xr2 >= 0 || 
		yc2 >= 0 || yr2 >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 4);
		if (ssRef) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = xc2;	ssRef[2].y = xr2;
			ssRef[3].x = yc2;	ssRef[3].y = yr2;
			cssRef = 4;
			}
		}
	Command(CMD_AUTOSCALE, 0L, 0L);
}

Whisker::Whisker(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Whisker::~Whisker()
{
	if(mo) DelBitmapClass(mo);
	mo = 0L;					if(ssRef) free(ssRef);
	ssRef = 0L;					if(name) free(name);
	name = 0L;
}

double
Whisker::GetSize(int select)
{
	switch (select & 0xfff) {
	case SIZE_XPOS:			return pos1.fx;
	case SIZE_XPOS+1:		return pos2.fx;
	case SIZE_YPOS:			return pos1.fy;
	case SIZE_YPOS+1:		return pos2.fy;
		}
	return 0.0;
}

bool
Whisker::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_WHISKER: 
		size = value;
		return true;
	case SIZE_WHISKER_LINE:
		LineDef.width = value;
		return true;
		}
	return false;
}

bool
Whisker::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_WHISKER:
		LineDef.color = col;
		return true;
		}
	return false;
}

void
Whisker::DoPlot(anyOutput *o)
{
	double si, csi, tmp, fix1, fiy1, fix2, fiy2, dx, dy;
	int i;

	fix1 = o->fx2fix(pos1.fx);	fix2 = o->fx2fix(pos2.fx);
	fiy1 = o->fy2fiy(pos1.fy);	fiy2 = o->fy2fiy(pos2.fy);
	if(fix1 == fix2 && fiy1 == fiy2) return;	//zero length
	pts[2].x = iround(fix1);		pts[3].x = iround(fix2);
	pts[2].y = iround(fiy1);		pts[3].y = iround(fiy2);
	//calculate sine and cosine
	si = fiy1-fiy2;
	tmp = fix2 - fix1;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fix2-fix1;
	tmp = fiy2 - fiy1;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	dx = o->un2fix(si*size/2.0);
	dy = o->un2fiy(csi*size/2.0);
	//calc cap
	pts[0].x = iround(fix1 - dx);	pts[4].x = iround(fix2 - dx);
	pts[0].y = iround(fiy1 - dy);	pts[4].y = iround(fiy2 - dy);
	pts[1].x = iround(fix1 + dx);	pts[5].x = iround(fix2 + dx);
	pts[1].y = iround(fiy1 + dy);	pts[5].y = iround(fiy2 + dy);
	//draw whisker
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	UpdateMinMaxRect(&rDims, pts[5].x, pts[5].y);
	IncrementMinMaxRect(&rDims, 3+(o->un2ix(LineDef.width)<<1));
	o->SetLine(&LineDef);
	switch(type & 0x0f) {
	case 2:				case 3:			case 0:
		if ((type & 0x0f) == 2){
			pts[4].x = pts[3].x;	pts[4].y = pts[3].y;
			pts[1].x = pts[2].x;	pts[1].y = pts[2].y;
			}
		if ((type & 0x0f) == 3) {
			pts[5].x = pts[3].x;	pts[5].y = pts[3].y;
			pts[0].x = pts[2].x;	pts[0].y = pts[2].y;
			}
		for(i = 0; i < 5; i+=2) o->oSolidLine(pts+i);
		break;
	case 1:
		pts[4].x = pts[5].x = pts[3].x;		pts[4].y = pts[5].y = pts[3].y;
		pts[1].x = pts[0].x = pts[2].x;		pts[1].y = pts[0].y = pts[2].y;
		o->oSolidLine(pts+2);
		break;
		}
}

void
Whisker::DoMark(anyOutput *o, bool mark)
{
	int i;
	LineDEF OldLine;

	if(mark){
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		memcpy(&OldLine, &LineDef, sizeof(LineDEF));
		i = 3*o->un2ix(LineDef.width);		//increase size of rectangle for marks
		IncrementMinMaxRect(&mrc, i);		mo = GetRectBitmap(&mrc, o);
		LineDef.width *= 5.0;				DoPlot(o);
		LineDef.width = OldLine.width;		LineDef.color = OldLine.color ^ 0x00ffffffL;
		DoPlot(o);							o->UpdateRect(&mrc, true);
		memcpy(&LineDef, &OldLine, sizeof(LineDEF));
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);
		mo = 0L;
		}
}

bool
Whisker::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int cb;

	switch (cmd) {
	case CMD_SCALE:
		size *= ((scaleINFO*)tmpl)->sy.fy;
		LineDef.width *= ((scaleINFO*)tmpl)->sy.fy;
		LineDef.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_SELECT:
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				if(IsCloseToLine(&pts[2], &pts[3], mev->x, mev->y, 3.0) ||
					IsCloseToLine(&pts[0], &pts[1], mev->x, mev->y, 3.0) ||
					IsCloseToLine(&pts[4], &pts[5], mev->x, mev->y, 3.0)) {
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
				}
			break;
			}
		return false;
	case CMD_ERRDESC:
		if(tmpl && *((char*)tmpl)) {
			cb = (int)strlen((char*)tmpl)+2;
			name = (char*)realloc(name, cb);
			if(name) rlp_strcpy(name, cb, (char*)tmpl);
			return true;
			}
		return false;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		switch(type & 0x0f) {
		case 1:		((Legend*)tmpl)->HasErr(&LineDef, 5, name);		break;
		case 2:		((Legend*)tmpl)->HasErr(&LineDef, 4, name);		break;
		case 3:		((Legend*)tmpl)->HasErr(&LineDef, 3, name);		break;
		default:
			if((rDims.right - rDims.left) < (rDims.bottom - rDims.top))
				((Legend*)tmpl)->HasErr(&LineDef, 1, name);
			else ((Legend*)tmpl)->HasErr(&LineDef, 2, name);
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_WHISKER;		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &pos1.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &pos1.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &pos2.fx);
			data->GetValue(ssRef[3].y, ssRef[3].x, &pos2.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy);
			((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy);
			return true;
			}
		break;
	case CMD_WHISKER_STYLE:
		if(tmpl) type = *((int*)tmpl);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// drop line 
DropLine::DropLine(GraphObj *par, DataObj *d, double x, double y, int which, int xc, 
	int xr, int yc, int yr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;
	fPos.fy = y;
	type = which;
	Id = GO_DROPLINE;
	if(xc >= 0 && xr >= 0 && yc >= 0 && yr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 2);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			cssRef = 2;
			}
		}
	bModified = false;
}

DropLine::DropLine(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

DropLine::~DropLine()
{
	if(bModified) Undo.InvalidGO(this);
	if(ssRef) free(ssRef);
	ssRef = 0L;
}

void
DropLine::DoPlot(anyOutput *o)
{
	long tmp;

	if (o->OC_type == OC_GRIDVIEW) return;
	o->RLP.fp = 0.0f;		//reset line pattern start
	if(parent) {
		pts[0].x = pts[1].x = pts[2].x = pts[3].x = o->fx2ix(fPos.fx);
		pts[0].y = pts[1].y = pts[2].y = pts[3].y = o->fy2iy(fPos.fy);
		if(type & DL_LEFT) {
			tmp = o->fx2ix(parent->GetSize(SIZE_BOUNDS_LEFT));
			tmp += iround(o->getVPorgX());
			if(tmp < pts[0].x) pts[0].x = tmp;
			if(tmp > pts[1].x) pts[1].x = tmp;
			}
		if(type & DL_RIGHT) {
			tmp = o->fx2ix(parent->GetSize(SIZE_BOUNDS_RIGHT));
			tmp += iround(o->getVPorgX());
			if(tmp < pts[0].x) pts[0].x = tmp;
			if(tmp > pts[1].x) pts[1].x = tmp;
			}
		if(type & DL_YAXIS) {
			tmp = iround(o->un2fix(parent->GetSize(SIZE_GRECT_LEFT)));
			tmp += iround(o->co2fix(o->xAxis.loc[0].fx));
			if (tmp < pts[0].x) pts[0].x = tmp;
			if(tmp > pts[1].x) pts[1].x = tmp;
			}
		if(type & DL_TOP) {
			tmp = o->fy2iy(parent->GetSize(SIZE_BOUNDS_TOP));
			tmp += iround(o->getVPorgY());
			if (tmp < pts[2].y) pts[2].y = tmp;
			if(tmp > pts[3].y) pts[3].y = tmp;
			}
		if(type & DL_BOTTOM) {
			tmp = o->fy2iy(parent->GetSize(SIZE_BOUNDS_BOTTOM));
			tmp += iround(o->getVPorgY());
			if (tmp < pts[2].y) pts[2].y = tmp;
			if(tmp > pts[3].y) pts[3].y = tmp;
			}
		if(type & DL_XAXIS) {
			tmp = iround(o->un2fiy(parent->GetSize(SIZE_GRECT_TOP)));
			tmp += iround(o->co2fiy(o->xAxis.loc[0].fy));
			if (tmp < pts[2].y) pts[2].y = tmp;
			if(tmp > pts[3].y) pts[3].y = tmp;
			}
		SetMinMaxRect(&rDims, pts[0].x, pts[2].y, pts[1].x, pts[3].y);
		IncrementMinMaxRect(&rDims, 3);
		o->SetLine(&LineDef);		o->oPolyline(pts, 2);		o->oPolyline(pts+2, 2);
		}
}

void
DropLine::DoMark(anyOutput *o, bool mark)
{

	InvertLine(pts, 2, &LineDef, 0L, o, mark);
	InvertLine(pts+2, 2, &LineDef, &rDims, o, mark);
}

bool
DropLine::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_SELECT:
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				if(IsCloseToLine(&pts[0], &pts[1], mev->x, mev->y, 3.0) ||
					IsCloseToLine(&pts[2], &pts[3], mev->x, mev->y, 3.0)) {
					o->ShowMark(this, MRK_GODRAW);
					return true;
					}
				}
			break;
			}
		return false;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_DL_LINE:
		if(tmpl) memcpy(&LineDef, tmpl, sizeof(LineDEF));
		return true;
	case CMD_DL_TYPE:
		if(tmpl)type = *((int*)tmpl);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_DROPLINE;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define a spherical scanline used for clipping spheres
sph_scanline::sph_scanline(fPOINT3D *center, double radius, bool bVert):GraphObj(0, 0)
{
	Id = GO_SPHSCANL;
	rad = radius >= 0 ? radius : -radius;
	memcpy(&p1, center, sizeof(fPOINT3D));	memcpy(&p2, center, sizeof(fPOINT3D));
	memcpy(&cent, center, sizeof(fPOINT3D));
	vert = bVert;
	if(vert) {
		p1.fy -= rad;		p2.fy += rad;
		}
	else {
		p1.fx -= rad;		p2.fx += rad;
		}
	if(p1.fx < 1.0) p1.fx = 1.0;
	if(p1.fy < 1.0) p1.fy = 1.0;
	if(p2.fx < p1.fx) p2.fx = p1.fx;
	if(p2.fy < p1.fy) p2.fy = p1.fy;
	bValid1 = bValid2 = true;
}

bool
sph_scanline::Command(int cmd, void *tmpl, anyOutput *)
{
	switch(cmd) {
	case CMD_ADDTOLINE:
		if(bValid1 && tmpl){
			memcpy(&p2, tmpl, sizeof(fPOINT3D));
			bValid2 = true;
			return true;
			}
		break;
	case CMD_STARTLINE:
		if(!bValid1 && tmpl){
			memcpy(&p1, tmpl, sizeof(fPOINT3D));
			bValid1 = true;
			}
		break;
		return true;
		}
	return false;
}

void
sph_scanline::DoClip(GraphObj *co)
{
	fPOINT3D *pla;
	int np, i;

	if(!bValid1 || !bValid2) return;
	switch(co->Id){
	case GO_SPHERE:
		bValid1 = bValid2 = false;
		clip_sphline_sphere(this, &p1, &p2, &cent, rad, co->GetSize(SIZE_RADIUS1), 
			co->GetSize(SIZE_XPOS), co->GetSize(SIZE_YPOS), co->GetSize(SIZE_ZPOS));
		break;
	case GO_PLANE:
		for(i=0; ((plane*)co)->GetPolygon(&pla, &np, i); i++) {
			bValid1 = bValid2 = false;
			clip_sphline_plane(this, &p1, &p2, &cent, rad, pla, np, ((plane*)co)->GetVec());
			}
		break;
		}
}
	
bool
sph_scanline::GetPoint(POINT *p, int sel)
{
	if(!bValid1 || !bValid2) {
		p->x = p->y = 0;
		return false;
		}
	switch(sel) {
	case 1:
		p->x = (long)p1.fx;		p->y = (long)p1.fy;
		return true;
	case 2:
		p->x = (long)p2.fx;		p->y = (long)p2.fy;
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Sphere: a symbol in three dimensional space
Sphere::Sphere(GraphObj *par, DataObj *d, int sel, double x, double y, double z, 
	double r, int xc, int xr, int yc, int yr, int zc, int zr, int rc, int rr)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_SPHERE;
	fPos.fx = x;	fPos.fy = y;	fPos.fz = z;	size = r;
	if(xc >=0 || xr >=0 || yc >=0 || yr >=0 || zc >=0 || zr >=0 || rc >=0 || rr >=0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 4);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			ssRef[3].x = rc;	ssRef[3].y = rr;
			cssRef = 4;
			}
		}
	type = sel;		mo = 0L;
	bModified = false;
}

Sphere::Sphere(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_SPHERE;		bModified = false;
}

Sphere::~Sphere()
{
	if(bModified) Undo.InvalidGO(this);
	Command(CMD_FLUSH, 0L, 0L);
}

double
Sphere::GetSize(int select)
{
	switch(select) {
	case SIZE_MIN_X:		return fip.fx - (double)rx;
	case SIZE_MAX_X:		return fip.fx + (double)rx;
	case SIZE_MIN_Y:		return fip.fy - (double)ry;
	case SIZE_MAX_Y:		return fip.fy + (double)ry;
	case SIZE_MIN_Z:		return fip.fz - (double)rz;
	case SIZE_MAX_Z:		return fip.fz + (double)rz;
	case SIZE_XPOS:			return fip.fx;
	case SIZE_YPOS:			return fip.fy;
	case SIZE_ZPOS:			return fip.fz;
	case SIZE_RADIUS1:	case SIZE_RADIUS2:	return (double)rx;
	case SIZE_XCENT:		return fPos.fx;
	case SIZE_YCENT:		return fPos.fy;
	case SIZE_ZCENT:		return fPos.fz;
	case SIZE_DRADIUS:		return size;
	case SIZE_SYMBOL:
		if(!type) return size;
		else return DefSize(SIZE_SYMBOL);
	case SIZE_SYM_LINE:		return Line.width;
		}
	return 0.0;
}

bool
Sphere::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_SYMBOL:		size = value;			break;
	case SIZE_SYM_LINE:		Line.width = value;		break;
		}
	return true;
}

DWORD
Sphere::GetColor(int select)
{
	switch(select) {
	case COL_SYM_LINE:		return Line.color;
	case COL_SYM_FILL:		return Fill.color;
	default: return defs.Color(select);
		}
}

bool
Sphere::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_SYM_LINE:		Line.color = col;		break;
	case COL_SYM_FILL:		Fill.color = col;		break;
		}
	return true;
}

void
Sphere::DoPlot(anyOutput *o)
{
	int i;

	if(size <= 0.001 || !o) return;
	if(!o->fvec2ivec(&fPos, &fip)) return;
	bDrawDone = false;
	if(scl){
		for (i = 0; i < nscl; i++) {
			if (scl[i]) delete(scl[i]);
			}
		free(scl);
		scl = 0L;	nscl = 0;
		}
	ix = iround(fip.fx);			iy = iround(fip.fy);
	rz = iround(o->fz2fiz(size));
	switch (type) {
	case 1:
		rx = ry = iround(size * 0.5 * o->ddx);		break;
	case 2:
		rx = ry = iround(size * 0.5 * o->ddy);		break;
	case 3:
		rx = ry = iround(size * 0.5 * o->ddz);		break;
	default:			case 5:
		rx = o->un2ix(size / 2.0);	ry = o->un2iy(size / 2.0);
		break;
		}
	rDims.left = ix - rx;			rDims.right = ix + rx;
	rDims.top = iy - ry;			rDims.bottom = iy + ry;
	if (o->getVPorgScale() > 1.5 && (
		(rDims.right < defs.clipRC.left) || (rDims.left > defs.clipRC.right) ||
		(rDims.bottom < defs.clipRC.top) || (rDims.top > defs.clipRC.bottom))){
		bDrawDone = true;		return;
		}
	if(parent && parent->Command(CMD_SET_GO3D, this, o)) return;
	Command(CMD_REDRAW, 0L, o);
}

void
Sphere::DoMark(anyOutput *o, bool mark)
{
	if (mark) {
		if (mo) DelBitmapClass(mo);
		mo = GetRectBitmap(&rDims, o);
		o->CopyBitmap(rDims.left, rDims.top + o->MenuHeight(), o, 0, 0, rDims.right-rDims.left,
			rDims.bottom - rDims.top, true);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &rDims, o);
		DelBitmapClass(mo);			mo = 0L;
		return;
		}
	o->UpdateRect(&rDims, true);
}

bool
Sphere::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);
		ssRef = 0L;
		if(name) free(name);
		name = 0L;
		if(scl){
			for (i = 0; i < nscl; i++) {
				if (scl[i]) delete(scl[i]);
				}
			free(scl);
			scl = 0L;	nscl = 0;
			}
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		if(!type || type == 5)size *= ((scaleINFO*)tmpl)->sy.fy;;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		((Legend*)tmpl)->HasFill(&Line, &Fill, 0L);
		break;
	case CMD_SYM_FILL:
		if(tmpl) memcpy(&Fill, tmpl, sizeof(FillDEF));
		return true;
	case CMD_MRK_DIRTY:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_SPHERE;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		if(bDrawDone) return false;
		bDrawDone = true;
		if(scl) DrawPG(o, 0);
		else {
			o->SetLine(&Line);				o->SetFill(&Fill);
			if(Fill.type & FILL_LIGHT3D) o->oSphere(ix, iy, rx, 0L, 0, 0L);
			else o->oCircle(ix-rx, iy-ry, ix+rx+1, iy+ry+1, name);
			}
		rx--;ry--;			//smaller marking rectangle
		rDims.left = ix-rx-1;				rDims.right = ix+rx+1;
		rDims.top = iy-ry-1;				rDims.bottom = iy+ry+1;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				o->ShowMark(this, MRK_GODRAW);
				CurrGO = this;
				return true;
				}
			break;
			}
		break;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			if(cssRef >3) data->GetValue(ssRef[3].y, ssRef[3].x, &size);
			return true;
			}
		return false;
	case CMD_CLIP:
		co = (GraphObj*)tmpl;
		if(co){
			switch(co->Id) {
			case GO_PLANE:
			case GO_SPHERE:
				DoClip(co);
				break;
				}
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			return true;
			}
		break;
	case CMD_RECALC:
		if (size > 0.001 && o){
			if (!o->fvec2ivec(&fPos, &fip)) return false;
			rz = iround(o->fz2fiz(size));
			}
		}
	return false;
}

void
Sphere::DoClip(GraphObj *co)
{
	RECT cliprc;
	double d, d1;
	fPOINT3D cscl;
	int x, y, q, di, de, lim, cx, cy;
	int i;

	if(co && co->Id == GO_SPHERE) {
//		if(co->GetSize(SIZE_ZPOS) > fip.fz) return;
		d = co->GetSize(SIZE_XPOS) - fip.fx;	d1 = d * d;
		d = co->GetSize(SIZE_YPOS) - fip.fx;	d1 += (d * d);
		d = co->GetSize(SIZE_ZPOS) - fip.fz;	d1 = sqrt(d1 + d * d);
		if (d1 > (co->GetSize(SIZE_RADIUS1) + rx)) return;
		}
	else {
		cliprc.left = iround(co->GetSize(SIZE_MIN_X));
		cliprc.right = iround(co->GetSize(SIZE_MAX_X));
		cliprc.top = iround(co->GetSize(SIZE_MIN_Y));
		cliprc.bottom = iround(co->GetSize(SIZE_MAX_Y));
		if(!OverlapRect(&rDims, &cliprc))return;
		}
	//use a list of horizontal scanlines created by a circular Bresenham's algorithm
	//Ref: C. Montani, R. Scopigno (1990) "Speres-To-Voxel Conversion", in:
	//   Graphic Gems (A.S. Glassner ed.) Academic Press, Inc.; 
	//   ISBN 0-12-288165-5 
	if(!scl && (scl = (sph_scanline**)calloc(rx*7+2, sizeof(sph_scanline*)))) {
		cscl.fz = fip.fz;			nscl = 0;
		for(q = 0; q < 2; q++) {
			x = lim = 0;	y = rx;		di = 2*(1-rx);
			while( y >= lim) {
				if(di < 0) {
					de = 2*di + 2*y -1;
					if(de > 0) {
						x++;	y--;	di += (2*x -2*y +2);
						}
					else {
						x++;	di += (2*x +1);
						}
					}
				else {
					de = 2*di -2*x -1;
					if(de > 0) {
						y--;	di += (-2*y +1);
						}
					else {
						x++;	y--;	di += (2*x -2*y +2);
						}
					}
				switch(q) {
				case 0:
					cy = rx - y;			cx = x;
					break;
				case 1:
					cy = rx + x;			cx = y; 
					break;
					}
				cscl.fy = iround(fip.fy);		cscl.fx = iround(fip.fx);
				cscl.fy = cscl.fy - rx + cy;
				if(cx > 1) scl[nscl++] = new sph_scanline(&cscl, cx, false);
				}
			}
		}
	if(!scl) return;
	//do clip for every scanline
	for(i = 0; i < nscl; i++) if(scl[i]) scl[i]->DoClip(co);
}

void
Sphere::DrawPG(anyOutput *o, int start)
{
	POINT *pts, np;
	long cp = 0;
	int i = start, step = 1;

	if(!o || !scl ||!nscl) return;
	if((pts = (POINT*)calloc(nscl*2, sizeof(POINT)))) {
		do {
			if(scl[i]) {
				if(step > 0) scl[i]->GetPoint(&np, 1);
				else scl[i]->GetPoint(&np, 2);
				if(np.x && np.y){
					AddToPolygon(&cp, pts, &np);
					}
				else if(cp){
					if(step > 0) DrawPG(o, i);			//split sphere
					step = -1;
					}
				}
			if(i == nscl && step > 0) step = -1;
			else i += step;
			}while(i > start);
		}
	o->SetLine(&Line);				o->SetFill(&Fill);
	if(cp) o->oSphere(ix, iy, rx, pts, cp, name);
	free(pts);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// plane: utility object to draw a flat plane in 3D space
plane::plane(GraphObj *par, DataObj *d, fPOINT3D *pts, int nPts, fPOINT3D *oData, int nOdt, LineDEF *line,
	  FillDEF *fill):GraphObj(par, d)
{
	int i;
	long area;
	double vlength, v1[3], v2[3], vp[3], area1;
	bool v_valid = false;
	fPOINT3D p1, p2;
	
	nli = n_ipts = n_lines = 0;
	nldata = 0L;  ldata = 0L;	co = 0L;	lines = 0L;		PlaneVec = 0L;
	srcN = nOdt;	srcDt = oData;
	Id = GO_PLANE;	totalArea = 0;	bSigPol = false;
	memcpy(&Line, line, sizeof(LineDEF));
	memcpy(&Fill, fill, sizeof(FillDEF));
	src_pts = (fPOINT3D *)memdup(pts, nPts * sizeof(fPOINT3D), 0);
	rDims.left = rDims.right = rDims.top = rDims.bottom = 0;
	if(nPts > 2 && (ldata =(fPOINT3D**)calloc(1, sizeof(fPOINT3D*))) &&
		(ldata[0] = (fPOINT3D *)calloc(nPts+1, sizeof(fPOINT3D))) &&
		(nldata = (int*)calloc(1, sizeof(int)))	&&
		(ipts = (POINT*)calloc(nPts, sizeof(POINT)))){
		for(i = 0; i < nPts; i++) {
			ldata[0][i].fx = pts[i].fx;			ipts[i].x = iround(pts[i].fx);
			ldata[0][i].fy = pts[i].fy;			ipts[i].y = iround(pts[i].fy);
			ldata[0][i].fz = pts[i].fz;
			}
		nldata[0] = nPts;		nli = 1;	n_ipts = nPts;
		xBounds.fx = xBounds.fy = pts[0].fx;	yBounds.fx = yBounds.fy = pts[0].fy;
		zBounds.fx = zBounds.fy = pts[0].fz;
		rDims.left = rDims.right = ipts[0].x;	rDims.top = rDims.bottom = ipts[0].y;
		for (i = 1; i < nPts; i++){
			UpdateMinMaxRect(&rDims, ipts[i].x, ipts[i].y);
			if (pts[i].fx < xBounds.fx) xBounds.fx = pts[i].fx;
			if (pts[i].fx > xBounds.fy) xBounds.fy = pts[i].fx;
			if (pts[i].fy < yBounds.fx) yBounds.fx = pts[i].fy;
			if (pts[i].fy > yBounds.fy) yBounds.fy = pts[i].fy;
			if (pts[i].fz < zBounds.fx) zBounds.fx = pts[i].fz;
			if (pts[i].fz > zBounds.fy) zBounds.fy = pts[i].fz;
			}
		//test if plane vertical
		area1 = (xBounds.fx - xBounds.fy) * (yBounds.fx - yBounds.fy);
		for(area = 0, i = 1; i < nPts; i++) {
			area += (ipts[i].x - ipts[i-1].x) * ((ipts[i].y + ipts[i-1].y)>>1);
			}
		totalArea= area = abs(area);
		area1 = area ? fabs(area1/area) : 101.0;
		if(area < 100 && area1 > 100.0) {			//its small or vertical !
			lines = (line_segment**)calloc(nPts+2, sizeof(line_segment*));
			if(lines){
				for(i = 1; i < nPts; i++) {
					p1.fx = (ldata[0][i - 1]).fx;		p1.fy = (ldata[0][i - 1]).fy;
					p1.fz = (ldata[0][i - 1]).fz;
					p2.fx = (ldata[0][i]).fx;			p2.fy = (ldata[0][i]).fy;
					p2.fz = (ldata[0][i]).fz;
					lines[i - 1] = new line_segment(par, d, line, &p1, &p2);
					}
				n_lines = nPts-1;
				}
			}
		else {					//for a visible plane get vector perpendicular to plane
			for (i = 1; i < (nPts-1); i++) {
				v1[0] = pts[i].fx - pts[i-1].fx;	v1[1] = pts[i].fy - pts[i-1].fy;
				v1[2] = pts[i].fz - pts[i-1].fz;	v2[0] = pts[i+1].fx - pts[i].fx;
				v2[1] = pts[i+1].fy - pts[i].fy;	v2[2] = pts[i+1].fz - pts[i].fz;
				vp[0] = v1[1]*v2[2] - v1[2]*v2[1];	vp[1] = v1[2]*v2[0] - v1[0]*v2[2];
				vp[2] = v1[0]*v2[1] - v1[1]*v2[0];
				vlength = sqrt(vp[0]*vp[0]+vp[1]*vp[1]+vp[2]*vp[2]);
				v_valid = (vlength > 100.0);
				if (v_valid) break;
				}
			if(v_valid && (PlaneVec = (double*)malloc(4 * sizeof(double)))) {
				PlaneVec[0] = vp[0]/vlength;		PlaneVec[1] = vp[1]/vlength;
				PlaneVec[2] = -vp[2]/vlength;
				PlaneVec[3] = PlaneVec[0] * pts[i].fx + PlaneVec[1] * pts[i].fy - PlaneVec[2] * pts[i].fz;
				}
			}
		}
}

plane::~plane() 
{
	int i;

	if(ldata) {
		for(i = 0; i < nli; i++) if(ldata[i]) free(ldata[i]);
		free(ldata);	ldata = 0L;		nli = 0;
		}
	if(lines) {
		for (i = 0; i < n_lines; i++) {
			if (lines[i]) delete(lines[i]);
			}
		free(lines);	lines = 0L;		n_lines = 0;
		}
	if(nldata) free(nldata);
	if (src_pts) free(src_pts);
	src_pts = 0L;
	nldata = 0L;					if(ipts) free(ipts);
	ipts = 0L;						if(PlaneVec) free(PlaneVec);
	PlaneVec = 0L;
}

double
plane::GetSize(int select)
{
	switch(select) {
	case SIZE_MIN_X:		return xBounds.fx;
	case SIZE_MAX_X:		return xBounds.fy;
	case SIZE_MIN_Y:		return yBounds.fx;
	case SIZE_MAX_Y:		return yBounds.fy;
	case SIZE_MIN_Z:		return zBounds.fx;
	case SIZE_MAX_Z:		return zBounds.fy;
		}
	return 0.0;
}

void
plane::DoPlot(anyOutput *o)
{
	int i;

	bDrawDone = bReqPoint = false;
	if(Fill.type & FILL_LIGHT3D){
		Fill.color = o->VecColor(PlaneVec, Fill.color2, Fill.color);
		Fill.type &= ~FILL_LIGHT3D;
		}
	if (o->getVPorgScale() > 1.5 && (
		//ignore objects outside the display ara
		(rDims.right < defs.clipRC.left) || (rDims.left > defs.clipRC.right) ||
		(rDims.bottom < defs.clipRC.top) || (rDims.top > defs.clipRC.bottom))){
		bDrawDone = true;		return;
		}
	if(lines) {
		if(Line.width == 0.0) return;
		//draw line segments for vertical plane
		for(i = 0; i < n_lines; i++) if(lines[i]) lines[i]->DoPlot(o);
		bDrawDone = true;		return;
		}
	if(parent && parent->parent && parent->parent->Command(CMD_SET_GO3D, this, o)) return;
}

void
plane::DoMark(anyOutput *o, bool mark)
{
	FillDEF tmpfill;
	LineDEF tmpline;

	memcpy(&tmpfill, &Fill, sizeof(FillDEF));
	memcpy(&tmpline, &Line, sizeof(LineDEF));
	if(mark){
		tmpfill.color ^= 0x00ffffffL;		tmpline.color ^= 0x00ffffffL;
		}
	o->SetLine(&tmpline);					o->SetFill(&tmpfill);
	o->oPolygon(ipts, n_ipts);
}

bool 
plane::Command(int cmd, void *tmpl, anyOutput *o)
{
	lfPOINT *pt;
	fPOINT3D *ap;
	int i, j;
	
	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_REDRAW:
		if(bDrawDone) return false;
		bDrawDone = true;
		if(o && nldata){
			if(Line.width == 0.0) Line.color = Fill.color;
			o->SetLine(&Line);			o->SetFill(&Fill);
			for(i = 0; i < nli; i++){
				if(nldata[i] > 2 && (pt = (lfPOINT*)malloc(nldata[i]*sizeof(lfPOINT)))){
					if ((Fill.type == FILL_GRADIENT || Fill.type == FILL_RAINBOW )&& PlaneVec) {
						DoGradient(o, ldata[i], nldata[i]);
						}
					else {
						for (j = 0; j < nldata[i]; j++) {
							pt[j].fx = ldata[i][j].fx;		pt[j].fy = ldata[i][j].fy;
							}
						o->foPolygon(pt, nldata[i]);
						}
					free(pt);
					}
				}
			}
		return true;
	case CMD_STARTLINE:
		ap = (fPOINT3D*)tmpl;
		if(ap) {
			if(ldata && nldata && nli) {
				if(bReqPoint) {
					Command(CMD_ADDTOLINE, &ReqPoint, o);
					bReqPoint = false;
					}
				i = nli-1;			j = nldata[i]-1;
				if(ldata[i][j].fx == ap->fx && ldata[i][j].fy == ap->fy && ldata[i][j].fz == ap->fz){
					return true;
					}
				if(IsValidPG(ldata[i], nldata[i])) {
					//close previous polygon first
					if(ldata[i][0].fx != ldata[i][j].fx || ldata[i][0].fy != ldata[i][j].fy ||
						ldata[i][0].fz != ldata[i][j].fz){
						j++;
						ldata[i][j].fx = ldata[i][0].fx;	ldata[i][j].fy = ldata[i][0].fy;
						ldata[i][j].fz = ldata[i][0].fz;	nldata[i]++;
						}
					ldata = (fPOINT3D**)realloc(ldata, sizeof(fPOINT3D*) * (nli+1));
					ldata[nli] = (fPOINT3D*)malloc(sizeof(fPOINT3D)*2);
					nldata = (int*)realloc(nldata, sizeof(int) * (nli+1));
					}
				else {					//drop incomplete or invalid polygon
					nli--;
					}
				}
			else {
				ldata = (fPOINT3D**)calloc(1, sizeof(fPOINT3D*));
				ldata[nli = 0] = (fPOINT3D*)malloc(sizeof(fPOINT3D));
				nldata = (int*)calloc(1, sizeof(int));
				bReqPoint = false;
				}
			if(bSigPol) {
				n_linept = 1;	bSigPol = false;
				}
//			else n_linept = 0;
			if(ldata && nldata) {
				ldata[nli][0].fx = ap->fx;	ldata[nli][0].fy = ap->fy;
				ldata[nli][0].fz = ap->fz;	nldata[nli++] = 1;
				return true;
				}
			}
		break;
	case CMD_REQ_POINT:
		ap = (fPOINT3D*)tmpl;
		if(ap) {
			ReqPoint.fx = ap->fx;		ReqPoint.fy = ap->fy;		ReqPoint.fz = ap->fz;
			bReqPoint = true;
			}
		return true;
	case CMD_SIGNAL_POL:			//signal: next point is on outline
		bSigPol = true;
		return true;
	case CMD_ADDTOLINE:				case CMD_CLIP:
		if (cmd == CMD_ADDTOLINE) {
			if ((ap = (fPOINT3D*)tmpl) && ldata && nldata && nli) {
				i = nli - 1;	j = nldata[i] - 1;
				if (ldata[i][j].fx == ap->fx && ldata[i][j].fy == ap->fy && ldata[i][j].fz == ap->fz){
					return j>0 ? true : false;		//probably nothing to add
					}
				ldata[i] = (fPOINT3D*)realloc(ldata[i], ((j = nldata[i]) + 2) * sizeof(fPOINT3D));
				ldata[i][j].fx = ap->fx;			ldata[i][j].fy = ap->fy;
				ldata[i][j].fz = ap->fz;			nldata[i]++;
				if (bSigPol) n_linept++;
				bSigPol = false;
				return true;
				}
			}
		bSigPol = false;
		co = (GraphObj*)tmpl;
		if(co){
			switch(co->Id) {
			case GO_PLANE:
				if(nli){
					DoClip(co);
					if(nli && ldata) {
						i = nli-1;			j = nldata[i]-1;
						//is last part valid ?
						if(j < 2) {
							free(ldata[i]);		ldata[i] = 0L;
							nldata[i] = 0;
							nli--;
							}
						//close last polygon
						else if(ldata[i][0].fx != ldata[i][j].fx ||
							ldata[i][0].fy != ldata[i][j].fy ||
							ldata[i][0].fz != ldata[i][j].fz){
							if(bSigPol) {
								nldata[i]--;
								}
							else j++;
							ldata[i][j].fx = ldata[i][0].fx;
							ldata[i][j].fy = ldata[i][0].fy;
							ldata[i][j].fz = ldata[i][0].fz;
							nldata[i]++;
							}
						}
					}
				else xBounds.fx=xBounds.fy=yBounds.fx=yBounds.fy=zBounds.fx=zBounds.fy=0.0;
				break;
				}
			}
		break;
	case CMD_RECALC:
		//coordinates in src_pts are in pix
		for (i = 0; i < n_ipts; i++) {
			if (src_pts[i].fx < xBounds.fx) xBounds.fx = src_pts[i].fx;
			if (src_pts[i].fx > xBounds.fy) xBounds.fy = src_pts[i].fx;
			if (src_pts[i].fy < yBounds.fx) yBounds.fx = src_pts[i].fy;
			if (src_pts[i].fy > yBounds.fy) yBounds.fy = src_pts[i].fy;
			if (src_pts[i].fz < zBounds.fx) zBounds.fx = src_pts[i].fz;
			if (src_pts[i].fz > zBounds.fy) zBounds.fy = src_pts[i].fz;
			}
		break;
		}
	return false;
}

void * 
plane::ObjThere(int x, int y)
{
	POINT p1;

	if(bDrawDone && IsInRect(rDims, (p1.x = x), (p1.y = y)) &&
		(IsInPolygon(&p1, ipts, n_ipts, 3.0) || IsCloseToPL(p1, ipts, n_ipts, 3.0))) return this;
	return 0L;
}

bool
plane::GetPolygon(fPOINT3D **pla, int *npt, int n)
{
	if(n < nli && ldata && ldata[n]) {
		*pla = ldata[n];	*npt = nldata[n];
		return true;
		}
	return false;
}

void
plane::DoClip(GraphObj *co)
{
	RECT cliprc;
	int o_nli, *o_nldata = 0L;
	fPOINT3D **o_ldata = 0L;
	fPOINT3D *tpg;
	int i, j, tnpt;
	bool is_valid = false;
	double *co_vec, d;

	//if two planes have the same parent it means they are part of one object
	// do not clip!
	if (co->parent->parent == parent->parent) return;
	//test if two planes are part of the same superplane
#define TOL3D 1.0e-12
	if(co->Id == GO_PLANE && (co_vec = ((plane*)co)->GetVec()) && PlaneVec && fabs(co_vec[0] - PlaneVec[0]) < TOL3D
		&& fabs(co_vec[1]-PlaneVec[1]) < TOL3D && fabs(co_vec[2]-PlaneVec[2]) < TOL3D && ldata && nldata) {
		d = (ldata[0][0].fx * co_vec[0] + ldata[0][0].fy * co_vec[1] - co_vec[3])/co_vec[2] - ldata[0][0].fz;
		if(fabs(d) < 2) return;
		}
#undef TOL3D
	cliprc.left = iround(co->GetSize(SIZE_MIN_X));
	cliprc.right = iround(co->GetSize(SIZE_MAX_X));
	cliprc.top = iround(co->GetSize(SIZE_MIN_Y));
	cliprc.bottom = iround(co->GetSize(SIZE_MAX_Y));
	if(OverlapRect(&rDims, &cliprc) && co != this) {
		o_nli = nli;		nli = 0;
		o_nldata = nldata;	nldata = 0L;
		o_ldata = ldata;	ldata = 0L;
		switch(co->Id) {
		case GO_PLANE:
			//clip all parts of this plane with all from another plane
			for(i = 0; ((plane*)co)->GetPolygon(&tpg, &tnpt, i); i++){
				for(j = 0; j < o_nli; j++) {
					n_linept = 0;
					if(o_nldata[j] >2) clip_plane_plane(this, o_ldata[j], o_nldata[j], PlaneVec, 
						tpg, tnpt, ((plane*)co)->GetVec(), ipts, n_ipts );
					}
				if(bReqPoint){
					if(bSigPol) nldata[nli-1]--;
					Command(CMD_ADDTOLINE, &ReqPoint, 0L);
					bReqPoint = false;
					}
				if(nli) is_valid = true;
				if(o_ldata) {
					for(j = 0; j < o_nli; j++) if(o_ldata[j]) free(o_ldata[j]);
					free(o_ldata);
					}
				if(o_nldata) free(o_nldata);
				o_nli = nli;		nli = 0;
				o_nldata = nldata;	nldata = 0L;
				o_ldata = ldata;	ldata = 0L;
				if(!o_nli) {					//plane is completly hidden
					return;
					}
				}
			if(is_valid || i==0){
				nli = o_nli;		o_nli = 0;
				nldata = o_nldata;	o_nldata = 0L;
				ldata = o_ldata;	o_ldata = 0L;
				}
			if(nli > 1) for(i = 1; i < nli; i++) {
				if(nldata[i] > 3 && ldata[i][nldata[i]-1].fx == ldata[i-1][0].fx
					&& ldata[i][nldata[i]-1].fy == ldata[i-1][0].fy) {
					nldata[i]--;	//bad vis: ignore last point
					}
				}
			break;
		default:
			nli = o_nli;		o_nli = 0;
			nldata = o_nldata;	o_nldata = 0L;
			ldata = o_ldata;	o_ldata = 0L;
			break;
			}
		//check shape and recalc some values
		if(is_valid && nli) {
			xBounds.fx = xBounds.fy = ldata[0][0].fx;	yBounds.fx = yBounds.fy = ldata[0][0].fy;
			zBounds.fx = zBounds.fy = ldata[0][0].fz;
			rDims.left = rDims.right = iround(ldata[0][0].fx);
			rDims.top = rDims.bottom = iround(ldata[0][0].fy);
			for(i = 0; i < nli; i++) if(nldata[i] > 2){
				if(ldata[i][0].fx != ldata[i][nldata[i]-1].fx || ldata[i][0].fy != ldata[i][nldata[i]-1].fy
					|| ldata[i][0].fz != ldata[i][nldata[i]-1].fz) {
					ldata[i][nldata[i]].fx = ldata[i][0].fx;	ldata[i][nldata[i]].fy = ldata[i][0].fy;
					ldata[i][nldata[i]].fz = ldata[i][0].fz;	nldata[i]++;
					}
				for(j = 0; j < (nldata[i]-1); j++) {
					UpdateMinMaxRect(&rDims, iround(ldata[i][j].fx), iround(ldata[i][j].fy));
					if(ldata[i][j].fx < xBounds.fx) xBounds.fx = ldata[i][j].fx;
					if(ldata[i][j].fx > xBounds.fy) xBounds.fy = ldata[i][j].fx;
					if(ldata[i][j].fy < yBounds.fx) yBounds.fx = ldata[i][j].fy;
					if(ldata[i][j].fy > yBounds.fy) yBounds.fy = ldata[i][j].fy;
					if(ldata[i][j].fz < zBounds.fx) zBounds.fx = ldata[i][j].fz;
					if(ldata[i][j].fz > zBounds.fy) zBounds.fy = ldata[i][j].fz;
					}
				}
			}
		}
	if(o_ldata) {
		for(i = 0; i < o_nli; i++) if(o_ldata[i]) free(o_ldata[i]);
		free(o_ldata);
		}
	if(o_nldata) free(o_nldata);
}

bool
plane::IsValidPG(fPOINT3D *pg, int npg)
{
	int i;
	double area, ratio;

	//if all points on an edge: not valid
	if(n_linept >= npg) return false;
	//a polygon must have more than 3 Points
	if(npg < 3) return false;
	//check for a reasonable size
	for(area = 0, i = 1; i < npg; i++) {
		area += (pg[i].fx - pg[i-1].fx) * ((pg[i].fy + pg[i-1].fy)/2.0);
		}
	area += (pg[0].fx - pg[i-1].fx) * ((pg[0].fy + pg[i-1].fy)/2.0);
	area = fabs(area);					if(area < 25) return false;
	ratio = totalArea/area;
	if(ratio > 100.0) return false;
	return true;
}

bool
plane::DoGradient(anyOutput *o, fPOINT3D *pts, int npt)
{
	double x, y, z;
	double *px, *py, *pz;
	double x_mean, y_mean, z_mean;
	double Ymin, Ymax;
	long i, n, i1, i2, im, direc;
	static lfPOINT grad[2];
	fPOINT3D fg1, fg2;
	fPOINT3D gp1, gp2;

#ifdef _WINDOWS
	double d;
#endif

	if (!parent) return false;
	if (!parent->Id) parent->Command(CMD_SET_DATAOBJ, 0L, o);
	if (parent->Id != GO_PLANE3D) return false;
	
	if (!srcDt || !srcN) return false;
	px = (double*)calloc(npt + 1, sizeof(double));
	py = (double*)calloc(npt + 1, sizeof(double));
	pz = (double*)calloc(npt + 1, sizeof(double));
	Ymax = fmax(srcDt[0].fy, srcDt[1].fy);		Ymin = fmin(srcDt[0].fy, srcDt[1].fy);
	if (pts[npt - 1].fx == pts[0].fx && pts[npt - 1].fy == pts[0].fy) n = npt - 1;
	else n = npt;			//last pt equal to 1st ?
	for (i = 0, x_mean = y_mean = z_mean = 0.0; i < n; i++){
		x = srcDt[i].fx;		y = srcDt[i].fy;		z = srcDt[i].fz;
		if (i > 1) {
			if (y > Ymax) Ymax = y;
			if (y < Ymin) Ymin = y;
			}
		px[i] = x;		py[i] = y;		pz[i] = z;
		x_mean += x;	y_mean += y;	z_mean += z;
		}
	x_mean /= i;	y_mean /= i;	z_mean /= i;
#ifdef _WINDOWS
	//increase bounding rectangle
	if (o->OC_type != OC_EXPORT) {
		d = (Ymax - Ymin) / 10.0;		Ymax += d;		Ymin -= d;
		}
#endif
	if(true) {					//general case or triangle
		i1 = i2 = im = -1;
		gp1.fx = gp2.fx = px[0];		gp1.fy = gp2.fy = py[0];
		gp1.fz= gp2.fz = pz[0];
		if (n == 3)	i1 = i2 = 0;
		for (i = 1; i < n; i++) {
			if (py[i] < gp1.fy) {
				gp1.fx = px[i];		gp1.fy = py[i];		gp1.fz = pz[i];		i1 = i;
				}
			if (py[i] > gp2.fy) {
				gp2.fx = px[i];		gp2.fy = py[i];		gp2.fz = pz[i];		i2 = i;
				}
			if (py[i] >= gp1.fy && py[i] <= gp2.fy) im = i;
			}
		direc = 0;
		if (n == 3 && i1 >= 0 && i2 >= 0 && im >= 0) {
			if (true){
				gp1.fx = (px[i1] + px[im]) / 2.0;			gp1.fy = (py[i1] + py[im]) / 2.0;
				gp1.fz = (pz[i1] + pz[im]) / 2.0;			direc = 1;
				}
			else {
				gp2.fx = (px[i2] + px[im]) / 2.0;			gp2.fy = (py[i2] + py[im]) / 2.0;
				gp2.fz = (pz[i2] + pz[im]) / 2.0;			direc = 2;
				}
			}
		direc = direc;							//debug: disable linux gcc warning
		o->fvec2ivec(&gp1, &fg1);				o->fvec2ivec(&gp2, &fg2);
		grad[0].fx = fg1.fx;					grad[0].fy = fg1.fy;
		grad[1].fx = fg2.fx;					grad[1].fy = fg2.fy;
#ifdef _WINDOWS
		if (direc == 1 && o->OC_type != OC_EXPORT) {
			d = (grad[1].fx - grad[0].fx) / 7.0;	grad[0].fx -= d;
			d = (grad[1].fy - grad[0].fy) / 7.0;	grad[0].fy -= d;
			}
#endif
		o->GradPG(pts, n, 0.0, 0.0, Ymin, Ymax, grad);
		free(px);			free(py);		free(pz);
		return true;
		}
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a simple plane in three dimensional space
Plane3D::Plane3D(GraphObj *par, DataObj *da, fPOINT3D *pt, long npt)
	:GraphObj(par, da) 
{
	FileIO(INIT_VARS);
	Id = GO_PLANE3D;
	dt = (fPOINT3D*) memdup(pt, sizeof(fPOINT3D)*npt, 0L);
	ndt = npt;
	CreatePlaneVec();
}

Plane3D::Plane3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_PLANE3D;
	CreatePlaneVec();
}

Plane3D::~Plane3D()
{
	if(dt) free(dt);
	if(pts) free(pts);
	dt = 0L;		ndt = 0L;
	if(ipl) delete (ipl);
	if (PlaneVec) free(PlaneVec);
	ipl = 0L;		PlaneVec = 0L;
}

bool
Plane3D::SetSize(int select, double value)
{
	int i;

	if((select & 0xfff) >= SIZE_XPOS && (select & 0xfff) <=  SIZE_XPOS_LAST){
		if((i = select-SIZE_XPOS) >=0 && i <= 200 && i < (int)ndt){
			dt[i].fx = value;			return true;
			}
		}
	else if((select & 0xfff) >= SIZE_YPOS && (select & 0xfff) <= SIZE_YPOS_LAST){
		if((i = select-SIZE_YPOS) >=0 && i <= 200 && i < (int)ndt){
			dt[i].fy = value;			return true;
			}
		}
	else if((select & 0xfff) >= SIZE_ZPOS && (select & 0xfff) <= SIZE_ZPOS_LAST){
		if((i = select-SIZE_ZPOS) >=0 && i <= 200 && i < (int)ndt){
			dt[i].fz = value;			return true;
			}
		}
	else switch(select) {
	case SIZE_SYM_LINE:
		Line.width = value;
		}
	return false;
}

bool
Plane3D::SetColor(int select, DWORD col)
{
	switch(select) {
	case COL_POLYLINE:		Line.color = col;		return true;
	case COL_POLYGON:		Fill.color = col;		return true;
		}
	return false;
}

void
Plane3D::DoPlot(anyOutput *o)
{
	int i;

	if(ipl) delete ipl;
	ipl = 0L;								if(pts) free(pts);
	pts = 0L;
	if(!(o->ActualSize(&rDims)))return;
	rDims.left = rDims.right;	rDims.top = rDims.bottom;
	rDims.right = rDims.bottom = 0;
	pts = (fPOINT3D*)malloc(sizeof(fPOINT3D)*ndt);
	if (pts){
		for(i = 0; i < ndt; i++) {
			if(!o->fvec2ivec(&dt[i], &pts[i])){
				free(pts);	pts = 0L; return;
				}
			UpdateMinMaxRect(&rDims, iround(pts[i].fx), iround(pts[i].fy));
			}
		ipl = new plane(this, data, pts, i, dt, i, &Line, &Fill);
		if (ipl)ipl->DoPlot(o);
		}
	IncrementMinMaxRect(&rDims, o->un2ix(Line.width)+1);
}

void
Plane3D::DoMark(anyOutput *o, bool mark)
{
	if(mark){
		if(pts && ipl)ipl->DoMark(o, mark);
		o->UpdateRect(&rDims, true);
		}
	else if(parent) parent->Command(CMD_REDRAW, 0L, o);
}

bool
Plane3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_PLANE3D;
		if(tmpl) data = (DataObj *)tmpl;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH) {
			((Legend*)tmpl)->HasFill(&Line, &Fill, ((Plot*)parent)->data_desc);
			}
		else ((Legend*)tmpl)->HasFill(&Line, &Fill, 0L);
		break;
	case CMD_MRK_DIRTY:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SYM_FILL:
		if(tmpl) memcpy(&Fill, tmpl, sizeof(FillDEF));
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				if(ipl && ipl->ObjThere(mev->x, mev->y)){
					o->ShowMark(CurrGO=this, MRK_GODRAW);
					return true;
					}
				}
			break;
			}
		break;
	case CMD_PG_FILL:
		if(tmpl) {
			memcpy((void*)&Fill, tmpl, sizeof(FillDEF));
			Fill.hatch = 0L;
			}
		break;
	case CMD_SET_LINE:
		if (tmpl) {
			memcpy((void*)&Line, tmpl, sizeof(LineDEF));
			}
		break;
	case CMD_AUTOSCALE:
		if((parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH && dt) || parent->Id == GO_POLYGON3D) {
			for (i = 0; i < ndt; i++) {
				if (parent->Id == GO_POLYGON3D)((Plot*)parent->parent)->CheckBounds3D(dt[i].fx, dt[i].fy, dt[i].fz);
				else ((Plot*)parent)->CheckBounds3D(dt[i].fx, dt[i].fy, dt[i].fz);
				}
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

double
Plane3D::PlaneValue(lfPOINT v)
{
	if (!PlaneVec) CreatePlaneVec();
	if (PlaneVec){
		return  (v.fx * PlaneVec[0] + v.fy * PlaneVec[1] - PlaneVec[3]) / PlaneVec[2];
		}
	else if (dt) {
		return dt[0].fz;
		}
	return 0.0;
}

//for a visible plane get vector perpendicular to plane
bool 
Plane3D::CreatePlaneVec()
{
	int i, n;
	double vlength, v1[3], v2[3], vp[3];
	bool v_valid = false;

	if (!dt || !ndt) return false;
	if (dt[ndt - 1].fx == dt[0].fx && dt[ndt - 1].fy == dt[0].fy) n = ndt-1;
	else n = ndt;
	for (i = 1; i < (n - 1); i++) {
		v1[0] = dt[i].fx - dt[i - 1].fx;			v1[1] = dt[i].fz - dt[i - 1].fz;
		v1[2] = dt[i].fy - dt[i - 1].fy;			v2[0] = dt[i + 1].fx - dt[i].fx;
		v2[1] = dt[i + 1].fz - dt[i].fz;			v2[2] = dt[i + 1].fy - dt[i].fy;
		vp[0] = v1[1] * v2[2] - v1[2] * v2[1];		vp[1] = v1[2] * v2[0] - v1[0] * v2[2];
		vp[2] = v1[0] * v2[1] - v1[1] * v2[0];
		vlength = sqrt(vp[0] * vp[0] + vp[1] * vp[1] + vp[2] * vp[2]);
		v_valid = (vlength > 1.0e-10);
		if (v_valid) break;
		}
	if (v_valid && (PlaneVec = (double*)malloc(4 * sizeof(double)))) {
		PlaneVec[0] = vp[0] / vlength;		PlaneVec[1] = vp[1] / vlength;
		PlaneVec[2] = -vp[2] / vlength;
		PlaneVec[3] = PlaneVec[0] * dt[i].fx + PlaneVec[1] * dt[i].fz - PlaneVec[2] * dt[i].fy;
	}
	return v_valid;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Brick: a bar in three dimensional space
Brick::Brick(GraphObj *par, DataObj *da, double x, double y, double z, 
		double d, double w, double h, DWORD flg, int xc, int xr, int yc,
		int yr, int zc, int zr, int dc, int dr, int wc, int wr, int hc,
		int hr):GraphObj(par, da)
{
	FileIO(INIT_VARS);
	Id = GO_BRICK;
	fPos.fx = x;	fPos.fy = y;	fPos.fz = z;
	depth = d;		width = w;		height = h;
	flags = flg;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || zc >= 0 || zr >= 0 ||
		dc >= 0 || dr >= 0 || wc >= 0 || wr >= 0 || hc >= 0 || hr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 6);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			ssRef[3].x = dc;	ssRef[3].y = dr;
			ssRef[4].x = wc;	ssRef[4].y = wr;
			ssRef[5].x = hc;	ssRef[5].y = hr;
			cssRef = 6;
			}
		}
	bModified = false;
}

Brick::Brick(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}
	
Brick::~Brick()
{
	int i;

	if(faces) {
		for (i = 0; i < 6; i++) {
			if (faces[i]) delete(faces[i]);
			}
		free(faces);
		}
	faces = 0L;
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified) Undo.InvalidGO(this);
}

bool
Brick::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_BAR_LINE:		Line.width = value;		return true;
	case SIZE_BAR_BASE:		fPos.fy = value;		return true;
	case SIZE_BAR:			width = value;			return true;
	case SIZE_BAR_DEPTH:	depth = value;			return true;
		}
	return false;
}

double
Brick::GetSize(int)
{
	return 0.0;
}


bool
Brick::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_BAR_LINE:		Line.color = col;		return true;
	case COL_BAR_FILL:		Fill.color = col;		return true;
		}
	return false;
}

void
Brick::DoPlot(anyOutput *o)
{
	fPOINT3D cpt[8], fip1, fip2, tmp, itmp, *pg;
	lfPOINT ipg[10];
	plane *npl;
	double dtmp;
	int i;

	if(mo) DelBitmapClass(mo);
	mo = 0L;
	if(!faces || !o || !o->fvec2ivec(&fPos, &fip1)) return;
	if(!(pg = (fPOINT3D *)malloc(5*sizeof(fPOINT3D)))) return;
	for(i = 0; i < 6; i++) {
		if(faces[i]) delete(faces[i]);
		faces[i] = 0L;
		}
	if(flags & 0x800L) {		//height is data
		tmp.fx = fPos.fx;			tmp.fy = height;
		tmp.fz = fPos.fz;			o->fvec2ivec(&tmp, &fip2);
		}
	else {						//height is units
		tmp.fx = tmp.fz = 0.0;		tmp.fy = height;
		o->uvec2ivec(&tmp, &fip2);
		fip2.fx += fip1.fx;			fip2.fy += fip1.fy;
		fip2.fz += fip1.fz;
		}
	//calc output-device coordinates of cubic brick: 8 corners
	tmp.fx = -(width/2.0);			tmp.fz = (depth/2.0);	tmp.fy = 0.0;
	o->uvec2ivec(&tmp, &itmp);
	cpt[0].fx= fip1.fx+itmp.fx;	cpt[0].fy= fip1.fy+itmp.fy;	cpt[0].fz= fip1.fz+itmp.fz;
	cpt[2].fx= fip1.fx-itmp.fx;	cpt[2].fy= fip1.fy-itmp.fy;	cpt[2].fz= fip1.fz-itmp.fz;
	cpt[4].fx= fip2.fx+itmp.fx;	cpt[4].fy= fip2.fy+itmp.fy;	cpt[4].fz= fip2.fz+itmp.fz;
	cpt[6].fx= fip2.fx-itmp.fx;	cpt[6].fy= fip2.fy-itmp.fy;	cpt[6].fz= fip2.fz-itmp.fz;
	tmp.fx = (width/2.0);			tmp.fz = (depth/2.0);	tmp.fy = 0.0;
	o->uvec2ivec(&tmp, &itmp);
	cpt[1].fx= fip1.fx+itmp.fx;	cpt[1].fy= fip1.fy+itmp.fy;	cpt[1].fz= fip1.fz+itmp.fz;
	cpt[3].fx= fip1.fx-itmp.fx;	cpt[3].fy= fip1.fy-itmp.fy;	cpt[3].fz= fip1.fz-itmp.fz;
	cpt[5].fx= fip2.fx+itmp.fx;	cpt[5].fy= fip2.fy+itmp.fy;	cpt[5].fz= fip2.fz+itmp.fz;
	cpt[7].fx= fip2.fx-itmp.fx;	cpt[7].fy= fip2.fy-itmp.fy;	cpt[7].fz= fip2.fz-itmp.fz;
	//set up 6 faces or a grid view
	if (o->OC_type == OC_GRIDVIEW) {
		ipg[0].fx = cpt[0].fx;			ipg[0].fy = cpt[0].fy;
		ipg[1].fx = cpt[1].fx;			ipg[1].fy = cpt[1].fy;
		ipg[2].fx = cpt[2].fx;			ipg[2].fy = cpt[2].fy;
		ipg[3].fx = cpt[3].fx;			ipg[3].fy = cpt[3].fy;
		ipg[4].fx = cpt[0].fx;			ipg[4].fy = cpt[0].fy;
		ipg[5].fx = cpt[4].fx;			ipg[5].fy = cpt[4].fy;
		o->foPolyline(ipg, 6);
		ipg[0].fx = cpt[4].fx;			ipg[0].fy = cpt[4].fy;
		ipg[1].fx = cpt[5].fx;			ipg[1].fy = cpt[5].fy;
		ipg[2].fx = cpt[1].fx;			ipg[2].fy = cpt[1].fy;
		o->foPolyline(ipg, 3);
		ipg[0].fx = cpt[5].fx;			ipg[0].fy = cpt[5].fy;
		ipg[1].fx = cpt[6].fx;			ipg[1].fy = cpt[6].fy;
		ipg[2].fx = cpt[2].fx;			ipg[2].fy = cpt[2].fy;
		o->foPolyline(ipg, 3);
		ipg[0].fx = cpt[6].fx;			ipg[0].fy = cpt[6].fy;
		ipg[1].fx = cpt[7].fx;			ipg[1].fy = cpt[7].fy;
		ipg[2].fx = cpt[3].fx;			ipg[2].fy = cpt[3].fy;
		o->foPolyline(ipg, 3);
		ipg[0].fx = cpt[7].fx;			ipg[0].fy = cpt[7].fy;
		ipg[1].fx = cpt[4].fx;			ipg[1].fy = cpt[4].fy;
		o->foPolyline(ipg, 2);
		return;
		}
	pg[0].fx = pg[4].fx = cpt[0].fx;	pg[1].fx = cpt[1].fx;	
	pg[2].fx = cpt[2].fx;				pg[3].fx = cpt[3].fx;
	pg[0].fy = pg[4].fy = cpt[0].fy;	pg[1].fy = cpt[1].fy;	
	pg[2].fy = cpt[2].fy;				pg[3].fy = cpt[3].fy;
	pg[0].fz = pg[4].fz = cpt[0].fz;	pg[1].fz = cpt[1].fz;	
	pg[2].fz = cpt[2].fz;				pg[3].fz = cpt[3].fz;
	faces[0] = new plane(this, data, pg, 5, 0L, 0L, &Line, &Fill);
	pg[2].fx = cpt[5].fx;				pg[3].fx = cpt[4].fx;
	pg[2].fy = cpt[5].fy;				pg[3].fy = cpt[4].fy;
	pg[2].fz = cpt[5].fz;				pg[3].fz = cpt[4].fz;
	npl = new plane(this, data, pg, 5, 0L, 0L, &Line, &Fill);
	if(npl->GetSize(SIZE_MAX_Z) > faces[0]->GetSize(SIZE_MAX_Z)) faces[1] = npl;
	else {
		faces[1] = faces[0];		faces[0] = npl;		
		}
	pg[0].fx = pg[4].fx = cpt[2].fx;	pg[1].fx = cpt[6].fx;	
	pg[2].fx = cpt[5].fx;				pg[3].fx = cpt[1].fx;
	pg[0].fy = pg[4].fy = cpt[2].fy;	pg[1].fy = cpt[6].fy;	
	pg[2].fy = cpt[5].fy;				pg[3].fy = cpt[1].fy;
	pg[0].fz = pg[4].fz = cpt[2].fz;	pg[1].fz = cpt[6].fz;	
	pg[2].fz = cpt[5].fz;				pg[3].fz = cpt[1].fz;
	npl = new plane(this, data, pg, 5, 0L, 0L, &Line, &Fill);
	if((dtmp = npl->GetSize(SIZE_MAX_Z)) > faces[1]->GetSize(SIZE_MAX_Z)) faces[2] = npl;
	else {
		faces[2] = faces[1];
		if(dtmp > faces[0]->GetSize(SIZE_MAX_Z)) faces[1] = npl;
		else {
			faces[1] = faces[0];	faces[0] = npl;
			}
		}
	pg[2].fx = cpt[7].fx;				pg[3].fx = cpt[3].fx;
	pg[2].fy = cpt[7].fy;				pg[3].fy = cpt[3].fy;
	pg[2].fz = cpt[7].fz;				pg[3].fz = cpt[3].fz;
	npl = new plane(this, data, pg, 5, 0L, 0L, &Line, &Fill);
	dtmp = npl->GetSize(SIZE_MAX_Z);
	for (i = 3; i; i--) {
		if(dtmp >faces[i-1]->GetSize(SIZE_MAX_Z)) {
			faces[i] = npl;			break;
			}
		else faces[i] = faces[i-1];
		}
	if(!i) faces[0] = npl;
	pg[0].fx = pg[4].fx = cpt[4].fx;	pg[1].fx = cpt[7].fx;	
	pg[2].fx = cpt[3].fx;				pg[3].fx = cpt[0].fx;
	pg[0].fy = pg[4].fy = cpt[4].fy;	pg[1].fy = cpt[7].fy;	
	pg[2].fy = cpt[3].fy;				pg[3].fy = cpt[0].fy;
	pg[0].fz = pg[4].fz = cpt[4].fz;	pg[1].fz = cpt[7].fz;	
	pg[2].fz = cpt[3].fz;				pg[3].fz = cpt[0].fz;
	npl = new plane(this, data, pg, 5, 0L, 0L, &Line, &Fill);
	dtmp = npl->GetSize(SIZE_MAX_Z);
	for (i = 4; i; i--) {
		if(dtmp >faces[i-1]->GetSize(SIZE_MAX_Z)) {
			faces[i] = npl;			break;
			}
		else faces[i] = faces[i-1];
		}
	if(!i) faces[0] = npl;
	pg[2].fx = cpt[6].fx;				pg[3].fx = cpt[5].fx;
	pg[2].fy = cpt[6].fy;				pg[3].fy = cpt[5].fy;
	pg[2].fz = cpt[6].fz;				pg[3].fz = cpt[5].fz;
	npl = new plane(this, data, pg, 5, 0L, 0L, &Line, &Fill);
	dtmp = npl->GetSize(SIZE_MAX_Z);
	for (i = 5; i; i--) {
		if(dtmp >faces[i-1]->GetSize(SIZE_MAX_Z)) {
			faces[i] = npl;			break;
			}
		else faces[i] = faces[i-1];
		}
	if(!i) faces[0] = npl;
	rDims.left = rDims.right = (int)pg[0].fx;
	rDims.top = rDims.bottom = (int)pg[0].fy;
	for (i= 3; i < 6; i++) if(faces[i]) {
		faces[i]->DoPlot(o);
		UpdateMinMaxRect(&rDims, faces[i]->rDims.left, faces[i]->rDims.top);
		UpdateMinMaxRect(&rDims, faces[i]->rDims.right, faces[i]->rDims.bottom);
		}
	free(pg);
}

void
Brick::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark){
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		if(faces) for(i = 3; i < 6; i++)
			if(faces[i]) faces[i]->DoMark(o, mark);
		o->UpdateRect(&rDims, true);
		}
	else if (mo)	{
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
Brick::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);
		ssRef = 0L;
		if(name) free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;			depth *= ((scaleINFO*)tmpl)->sz.fy;
		width *= ((scaleINFO*)tmpl)->sx.fy;		if(!(flags & 0x800L)) height *=  ((scaleINFO*)tmpl)->sx.fy;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH) {
			((Legend*)tmpl)->HasFill(&Line, &Fill, ((Plot*)parent)->data_desc);
			}
		else ((Legend*)tmpl)->HasFill(&Line, &Fill, 0L);
		break;
	case CMD_BAR_FILL:
		if(tmpl) {
			memcpy(&Fill, tmpl, sizeof(FillDEF));
			Fill.hatch = 0L;
			return true;
			}
		break;
	case CMD_MRK_DIRTY:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_BRICK;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		//Should we ever come here ?
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				if(faces && faces[3] && faces[4] && faces[5] &&
					(faces[3]->ObjThere(mev->x, mev->y) || 
					faces[4]->ObjThere(mev->x, mev->y) ||
					faces[5]->ObjThere(mev->x, mev->y))){
						o->ShowMark(CurrGO=this, MRK_GODRAW);
						return true;
						}
				}
			break;
			}
		break;
	case CMD_UPDATE:
		if(ssRef && cssRef > 5 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			data->GetValue(ssRef[3].y, ssRef[3].x, &depth);
			data->GetValue(ssRef[4].y, ssRef[4].x, &width);
			data->GetValue(ssRef[5].y, ssRef[5].x, &height);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			if(flags & 0x800L) {		//height is data
				((Plot*)parent)->CheckBounds3D(fPos.fx, height, fPos.fz);
				}
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_RECALC:
		if (faces) for (i = 0; i < 6; i++) {
			if (faces[i]) faces[i]->Command(cmd, tmpl, o);
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// line_segment: utility object to draw a piece of a polyline in 3D space
line_segment::line_segment(GraphObj *par, DataObj *d, LineDEF *ld, fPOINT3D *p1, fPOINT3D *p2)
	:GraphObj(par, d)
{
	double tmp, tmp1, tmp2;

	nli = 0;    nldata = 0L;	  ldata = 0L;	prop = 1.0;		df_go = 0L;
	fmin.fx = fmax.fx = fmin.fy = fmax.fy = fmin.fz = fmax.fz = 0.0;
	ndf_go = 0;
	if(ld) memcpy(&Line, ld, sizeof(LineDEF));
	if(p1 && p2 &&(ldata =(fPOINT3D**)calloc(2, sizeof(fPOINT3D*))) &&
		(ldata[0] = (fPOINT3D*)malloc(4 * sizeof(fPOINT3D))) &&
		(nldata = (int*)calloc(2, sizeof(int)))){
			if(Line.pattern) {
				tmp = p2->fx - p1->fx;				tmp1 = tmp * tmp;
				tmp = p2->fy - p1->fy;				tmp1 += (tmp * tmp);
				tmp2 = tmp1;						tmp = p2->fz - p1->fz;
				tmp1 += (tmp * tmp);
				if(tmp1 > 1.0) prop = sqrt(tmp2)/sqrt(tmp1);
				}
			memcpy(&ldata[0][0], p1, sizeof(fPOINT3D));
			memcpy(&ldata[0][1], p2, sizeof(fPOINT3D));
			nldata[0] = 2;		nli = 1;
			rDims.left = rDims.right = iround(p1->fx);
			rDims.top = rDims.bottom = iround(p1->fy);
			UpdateMinMaxRect(&rDims, iround(p2->fx), iround(p2->fy));
			fmin.fx = (double)rDims.left;	fmin.fy = (double)rDims.top;
			fmax.fx = (double)rDims.right;	fmax.fy = (double)rDims.bottom;
			if(p2->fz > p1->fz) {
				fmin.fz = (double)p1->fz;	fmax.fz = (double)p2->fz;
				}
			else {
				fmin.fz = (double)p2->fz;	fmax.fz = (double)p1->fz;
				}
		}
	Id = GO_LINESEG;
}

line_segment::~line_segment()
{
	int i;

	if(ldata) {
		for(i = 0; i < nli; i++) if(ldata[i]) free(ldata[i]);
		free(ldata);
		}
	if(nldata) free(nldata);
	nldata = 0L;
}

double
line_segment::GetSize(int select)
{
	switch (select) {
	case SIZE_MIN_X:		return xBounds.fx;
	case SIZE_MAX_X:		return xBounds.fy;
	case SIZE_MIN_Y:		return yBounds.fx;
	case SIZE_MAX_Y:		return yBounds.fy;
	case SIZE_MIN_Z:		return zBounds.fx;
	case SIZE_MAX_Z:		return zBounds.fy;
		}
	return 0.0;
}

void
line_segment::DoPlot(anyOutput *o)
{
	bDrawDone = false;		co = 0L;
	if(df_go) free(df_go);
	df_go = 0L;				ndf_go = 0;
	if (o->getVPorgScale() > 1.5 && (
		(rDims.right < defs.clipRC.left) || (rDims.left > defs.clipRC.right) ||
		(rDims.bottom < defs.clipRC.top) || (rDims.top > defs.clipRC.bottom))){
		bDrawDone = true;		return;
		}
	if(parent && parent->Command(CMD_SET_GO3D, this, o)) return;
	Command(CMD_REDRAW, 0L, o);
}

bool
line_segment::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j;
	POINT pts[2];
	LineDEF cLine;
	fPOINT3D *ap;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_DRAW_LATER:
		if(!co) return false;
		if(df_go){
			for(i = 0; i < ndf_go; i++) if(df_go[i] == co) return true;
			df_go = (GraphObj**)realloc(df_go, (ndf_go + 1) * sizeof(GraphObj*));
			if (df_go) {
				df_go[ndf_go++] = co;
				}
			}
		else {
			if((df_go = (GraphObj**)malloc(sizeof(GraphObj*)))){
				df_go[0] = co;	ndf_go = 1;
				}
			}
		return true;
	case CMD_REDRAW:
		if(bDrawDone) return false;
		bDrawDone = true;
		if(!nli) return false;
		if(df_go) {
			for(i = 0; i < ndf_go; i++) if(df_go[i]) df_go[i]->Command(cmd, tmpl, o);
			free(df_go);	df_go = 0L;			ndf_go = 0L;
			}
		if(o && ldata && nldata){
			memcpy(&cLine, &Line, sizeof(LineDEF));
			cLine.patlength *= prop;
			o->SetLine(&cLine);
			for(i = 0; i < nli; i++) for(j = 0; j < (nldata[i]-1); j++) {
				pts[0].x = iround(ldata[i][j].fx);		pts[0].y = iround(ldata[i][j].fy);
				pts[1].x = iround(ldata[i][j+1].fx);	pts[1].y = iround(ldata[i][j+1].fy);
				if(pts[0].x != pts[1].x || pts[0].y != pts[1].y) o->oPolyline(pts, 2);
				}
			}
		return true;
	case CMD_CLIP:
		co = (GraphObj*)tmpl;
		if(co){
			switch(co->Id) {
			case GO_PLANE:			case GO_SPHERE:
				DoClip(co);
				break;
				}
			}
		return false;
	case CMD_STARTLINE:
		if(tmpl) {
			if(ldata && nldata) {
				ldata = (fPOINT3D**)realloc(ldata, sizeof(fPOINT3D*) * (nli+2));
				ldata[nli] = (fPOINT3D*)malloc(4 * sizeof(fPOINT3D));
				nldata = (int*)realloc(nldata, sizeof(int)*(nli+2));
				}
			else {
				ldata = (fPOINT3D**)calloc(2, sizeof(fPOINT3D*));
				ldata[nli = 0] = (fPOINT3D*)malloc(4 * sizeof(fPOINT3D));
				nldata = (int*)calloc(2, sizeof(int));
				}
			if(ldata && nldata) {
				memcpy(&ldata[nli][0], tmpl, sizeof(fPOINT3D));
				memcpy(&ldata[nli][1], tmpl, sizeof(fPOINT3D));
				nldata[nli++] = 1;
				return true;
				}
			}
		break;
	case CMD_ADDTOLINE:
		if((ap = (fPOINT3D*)tmpl) && ldata && nldata && nli && nldata[i =(nli-1)]) {
			j = nldata[i];
			ldata[i] = (fPOINT3D*)realloc(ldata[i], (nldata[i]+2) * sizeof(fPOINT3D));
			if(j && ldata[i][j-1].fx == ap->fx && ldata[i][j-1].fy == ap->fy &&
				ldata[i][j-1].fz == ap->fz) return true;
			else {
				j = (nldata[i]++);
				}
			memcpy(&ldata[i][j-1], tmpl, sizeof(fPOINT3D));
			return true;
			}
		break;
	case CMD_RECALC:
		//ldata has pix resolution
		if (ldata[0][0].fx < ldata[0][1].fx) {
			xBounds.fx = ldata[0][0].fx;			xBounds.fy = ldata[0][1].fx;
			}
		else {
			xBounds.fx = ldata[0][1].fx;			xBounds.fy = ldata[0][0].fx;
			}
		if (ldata[0][0].fy < ldata[0][1].fy) {
			yBounds.fx = ldata[0][0].fy;			yBounds.fy = ldata[0][1].fy;
			}
		else {
			yBounds.fy = ldata[0][1].fy;			yBounds.fy = ldata[0][0].fy;
			}
		if (ldata[0][0].fz < ldata[0][1].fz) {
			zBounds.fx = ldata[0][0].fz;			zBounds.fy = ldata[0][1].fz;
			}
		else {
			zBounds.fy = ldata[0][1].fz;			zBounds.fy = ldata[0][0].fz;
			}
		break;
		}
	return false;
}

void *
line_segment::ObjThere(int x, int y)
{
	int i, j;
	POINT pts[2];

	if(ldata && nldata){
		for(i = 0; i < nli; i++) for(j = 0; j < nldata[i]; j +=2) {
			pts[0].x = iround(ldata[i][j].fx);		pts[0].y = iround(ldata[i][j].fy);
			pts[1].x = iround(ldata[i][j+1].fx);	pts[1].y = iround(ldata[i][j+1].fy);
			if(IsCloseToLine(&pts[0], &pts[1], x, y, 3.0)) return this;
			}
		}
	return 0L;
}

void
line_segment::DoClip(GraphObj *co)
{
	RECT cliprc;
	int o_nli, *o_nldata = 0L;
	fPOINT3D **o_ldata = 0L, *pts = 0L;
	fPOINT3D *pla;
	int i, j, k, np;
	double r, cx, cy, cz;
	bool is_valid = false;

	cliprc.left = iround(co->GetSize(SIZE_MIN_X));
	cliprc.right = iround(co->GetSize(SIZE_MAX_X));
	cliprc.top = iround(co->GetSize(SIZE_MIN_Y));
	cliprc.bottom = iround(co->GetSize(SIZE_MAX_Y));
	if(OverlapRect(&rDims, &cliprc)) {
		if(!(pts = (fPOINT3D*)calloc(2, sizeof(fPOINT3D))))return;
		o_nli = nli;		nli = 0;
		o_nldata = nldata;	nldata = 0L;
		o_ldata = ldata;	ldata = 0L;
		switch(co->Id) {
		case GO_SPHERE:
			cx = co->GetSize(SIZE_XPOS);			cy = co->GetSize(SIZE_YPOS);
			cz = co->GetSize(SIZE_ZPOS);			r = co->GetSize(SIZE_RADIUS1);
			for(i = 0; i < o_nli; i++) for(j = 0; j < (o_nldata[i]-1); j ++) {
				pts[0].fx = o_ldata[i][j].fx;			pts[0].fy = o_ldata[i][j].fy;
				pts[0].fz = o_ldata[i][j].fz;			pts[1].fx = o_ldata[i][j+1].fx;
				pts[1].fy = o_ldata[i][j+1].fy;			pts[1].fz = o_ldata[i][j+1].fz;
				clip_line_sphere(this, &pts, r, cx, cy, cz);
				}
			break;
		case GO_PLANE:
			for(i = 0; ((plane*)co)->GetPolygon(&pla, &np, i); i++){
				for(j = 0; j < o_nli; j++) {
					if(o_nldata[j] >1) for(k = 0; k < (o_nldata[j]-1); k++){
						pts[0].fx = o_ldata[j][k].fx;			pts[0].fy = o_ldata[j][k].fy;
						pts[0].fz = o_ldata[j][k].fz;			pts[1].fx = o_ldata[j][k+1].fx;
						pts[1].fy = o_ldata[j][k+1].fy;			pts[1].fz = o_ldata[j][k+1].fz;
						if (pts[0].fx != pts[1].fx || pts[0].fy != pts[1].fy || pts[0].fz != pts[1].fz)
							clip_line_plane(this, &pts, pla, np, ((plane*)co)->GetVec());
						}
					}
				if(nli) is_valid = true;
				if(o_ldata) {
					for(j = 0; j < o_nli; j++) if(o_ldata[j]) free(o_ldata[j]);
					free(o_ldata);
					}
				if(o_nldata) free(o_nldata);
				o_nli = nli;		nli = 0;
				o_nldata = nldata;	nldata = 0L;
				o_ldata = ldata;	ldata = 0L;
				if(!o_nli) return;				//line is completly hidden
				}
			if(is_valid || i==0){
				nli = o_nli;		o_nli = 0;
				nldata = o_nldata;	o_nldata = 0L;
				ldata = o_ldata;	o_ldata = 0L;
				}
			break;
		default:
			nli = o_nli;		o_nli = 0;
			nldata = o_nldata;	o_nldata = 0L;
			ldata = o_ldata;	o_ldata = 0L;
			break;
			}
		if(pts) free(pts);
		}
	if(o_ldata) {
		for(i = 0; i < o_nli; i++) if(o_ldata[i]) free(o_ldata[i]);
		free(o_ldata);
		}
	if(o_nldata) free(o_nldata);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define a drop line in 3D space
DropLine3D::DropLine3D(GraphObj *par, DataObj *d, fPOINT3D *p1, int xc,
		int xr, int yc, int yr, int zc, int zr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_DROPL3D;
	memcpy(&fPos, p1, sizeof(fPOINT3D));
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || zc >= 0 || zr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 3);
		if (ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			cssRef = 3;
			}
		}
	bModified = false;
	type = 0x01;
}

DropLine3D::DropLine3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

DropLine3D::~DropLine3D()
{
	if(bModified) Undo.InvalidGO(this);
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
}

void
DropLine3D::DoPlot(anyOutput *o)
{
	fPOINT3D fip, fp, fp1;
	fPOINT3D p1, p2;
	int i;

	if(!parent || !o || !o->fvec2ivec(&fPos, &fip)) return;
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	for(i = 0; i < 6; i++){
		if(ls[i]) delete(ls[i]);
		ls[i] = 0L;
		}
	p1.fx = fip.fx;			p1.fy = fip.fy;			p1.fz = fip.fz;
	rDims.left = rDims.right = iround(p1.fx);		rDims.top = rDims.bottom = iround(p1.fy);
	for(i = 0; i < 6; i++) {
		fp.fx = fPos.fx;	fp.fy = fPos.fy;	fp.fz = fPos.fz;
		if(type & (1 << i)){
			switch (i) {
			case 0:	fp.fy = parent->GetSize(SIZE_BOUNDS_YMIN);	break;
			case 1:	fp.fy = parent->GetSize(SIZE_BOUNDS_YMAX);	break;
			case 2:	fp.fz = parent->GetSize(SIZE_BOUNDS_ZMIN);	break;
			case 3:	fp.fz = parent->GetSize(SIZE_BOUNDS_ZMAX);	break;
			case 4:	fp.fx = parent->GetSize(SIZE_BOUNDS_XMIN);	break;
			case 5:	fp.fx = parent->GetSize(SIZE_BOUNDS_XMAX);	break;
				}
			o->fvec2ivec(&fp, &fp1);		p2.fx = fp1.fx;
			p2.fy = fp1.fy;					p2.fz = fp1.fz;
			UpdateMinMaxRect(&rDims, iround(p2.fx), iround(p2.fy));
			ls[i] = new line_segment(this, data, &Line, &p1, &p2);
			if (ls[i]) ls[i]->DoPlot(o);
			mpts[i][0].x = iround(p1.fx);	mpts[i][0].y = iround(p1.fy);
			mpts[i][1].x = iround(p2.fx);	mpts[i][1].y = iround(p2.fy);
			}
		}
	IncrementMinMaxRect(&rDims, 5);
}

void
DropLine3D::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		for(i = 0; i < 6; i++) {
			if(type & (1 << i)){
				InvertLine(mpts[i], 2, &Line, 0L, o, mark);
				}
			}
		o->UpdateRect(&mrc, true);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
DropLine3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		for(i = 0; i < 6; i++){
			if(ls[i]) delete(ls[i]);
			ls[i] = 0L;
			}
		if(ssRef) free(ssRef);
		ssRef = 0L;
		if(name) free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_DL_TYPE:
		if(tmpl && *((int*)tmpl)) type = *((int*)tmpl);
		return true;
	case CMD_DL_LINE:
		if(tmpl) memcpy(&Line, tmpl, sizeof(LineDEF));
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_DROPL3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				for(i = 0; i < 6; i++) {
					if(ls[i] && ls[i]->ObjThere(mev->x, mev->y)){
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
					}
				}
			break;
			}
		break;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define an arrow in 3D space
Arrow3D::Arrow3D(GraphObj *par, DataObj *d, fPOINT3D *p1, fPOINT3D *p2, int xc1,
		int xr1, int yc1, int yr1, int zc1, int zr1, int xc2, int xr2, int yc2, 
		int yr2, int zc2, int zr2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_ARROW3D;
	memcpy(&fPos1, p1, sizeof(fPOINT3D));	memcpy(&fPos2, p2, sizeof(fPOINT3D));
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || zc1 >= 0 || zr1 >= 0 || 
		xc2 >= 0 || xr2 >= 0 || yc2 >= 0 || yr2 >= 0 || zc2 >= 0 || zr2 >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 6);
		if (ssRef) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = zc1;	ssRef[2].y = zr1;
			ssRef[3].x = xc2;	ssRef[3].y = xr2;
			ssRef[4].x = yc2;	ssRef[4].y = yr2;
			ssRef[5].x = zc2;	ssRef[5].y = zr2;
			cssRef = 6;
			}
		}
	bModified = false;
}

Arrow3D::Arrow3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

Arrow3D::~Arrow3D()
{
	if(bModified) Undo.InvalidGO(this);
	Command(CMD_FLUSH, 0L, 0L);
}

bool
Arrow3D::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ARROW_LINE:
		Line.width = value;
		return true;
	case SIZE_ARROW_CAPWIDTH:
		cw = value;
		return true;
	case SIZE_ARROW_CAPLENGTH:
		cl = value;
		return true;
		}
	return false;
}

bool
Arrow3D::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ARROW:
		Line.color = col;
		return true;
		}
	return false;
}

void
Arrow3D::DoPlot(anyOutput *o)
{
	double si, csi, tmp, cwr, clr, d, d1, d2;
	fPOINT3D fip1, fip2, tria[3];
	fPOINT3D p1, p2;
	fPOINT3D cp1, cp2;
	FillDEF fill;
	int i;

	if(!parent || !o || !o->fvec2ivec(&fPos1, &fip1) ||!o->fvec2ivec(&fPos2, &fip2)) return;
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	for(i = 0; i < 3; i++){
		if(ls[i]) delete(ls[i]);
		ls[i] = 0L;
		}
	p1.fx = fip1.fx;		p1.fy = fip1.fy;		p1.fz = fip1.fz;
	p2.fx = fip2.fx;		p2.fy = fip2.fy;		p2.fz = fip2.fz;
	if(p1.fx == p2.fx && p1.fy == p2.fy && p1.fz == p2.fz) return;	//zero length arrow
	rDims.left = rDims.right = iround(p1.fx);		rDims.top = rDims.bottom = iround(p1.fy);
	UpdateMinMaxRect(&rDims, iround(p2.fx), iround(p2.fy));
	IncrementMinMaxRect(&rDims, 5);
	ls[0] = new line_segment(this, data, &Line, &p1, &p2);
	if (ls[0]) ls[0]->DoPlot(o);
	mpts[0][0].x = iround(p1.fx);	mpts[0][0].y = iround(p1.fy);
	mpts[0][1].x = iround(p2.fx);	mpts[0][1].y = iround(p2.fy);
	if(p1.fx == p2.fx && p1.fy == p2.fy) return;			//zero length in 2D
	if((type & 0xff) == ARROW_NOCAP) return;				//no cap;
	//calculate sine and cosine for cap
	si = fip1.fy-fip2.fy;				tmp = fip2.fx - fip1.fx;
	si = si/sqrt(si*si + tmp*tmp);		csi = fip2.fx-fip1.fx;
	tmp = fip2.fy - fip1.fy;			csi = csi/sqrt(csi*csi + tmp*tmp);
	//cap corners
	d = fip2.fx - fip1.fx;				d1 = d * d;
	d = fip2.fy - fip1.fy;				d1 += (d * d);
	d = fip2.fz - fip1.fz;
	d2 = d1 + (d * d);	d1 = sqrt(d1);	d2 = sqrt(d2);	d = d1/d2;
	cwr = cw;	clr = cl*d;
	cp1.fx = p2.fx - o->un2fix(csi*clr + si*cwr/2.0);
	cp1.fy = p2.fy + o->un2fiy(si*clr - csi*cwr/2.0);
	cp2.fx = p2.fx - o->un2fix(csi*clr - si*cwr/2.0);
	cp2.fy = p2.fy + o->un2fiy(si*clr + csi*cwr/2.0);
	cp1.fz = cp2.fz = p2.fz;
	mpts[1][0].x = iround(p2.fx);		mpts[1][0].y = iround(p2.fy);
	mpts[1][1].x = iround(cp1.fx);		mpts[1][1].y = iround(cp1.fy);
	mpts[2][0].x = iround(p2.fx);		mpts[2][0].y = iround(p2.fy);
	mpts[2][1].x = iround(cp2.fx);		mpts[2][1].y = iround(cp2.fy);
	if((type & 0xff) == ARROW_LINE) {
		ls[1] = new line_segment(this, data, &Line, &p2, &cp1);
		if (ls[1]) ls[1]->DoPlot(o);
		ls[2] = new line_segment(this, data, &Line, &p2, &cp2);
		if (ls[2]) ls[2]->DoPlot(o);
		}
	else if((type & 0xff) == ARROW_TRIANGLE) {
		fill.type = FILL_NONE;	fill.color = Line.color;
		fill.scale = 1.0;		fill.hatch = 0L;
		tria[0].fz = tria[1].fz = tria[2].fz = fip2.fz;
		tria[0].fx = cp1.fx;		tria[0].fy = cp1.fy;
		tria[1].fx = fip2.fx;	tria[1].fy = fip2.fy;
		tria[2].fx = cp2.fx;		tria[2].fy = cp2.fy;
		cap = new plane(this, data, tria, 3, 0L, 0L, &Line, &fill);
		if (cap) cap->DoPlot(o);
	}
}

void
Arrow3D::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		for(i = 2; i >= 0 ; i--) if(ls[i]) InvertLine(mpts[i], 2, &Line, 0L, o, mark);
		o->UpdateRect(&rDims, true);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
Arrow3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		for(i = 0; i < 3; i++){
			if(ls[i]) delete(ls[i]);
			ls[i] = 0L;
			}
		if(cap) delete(cap);
		cap = 0L;					if(mo) DelBitmapClass(mo);
		mo = 0L;					if(ssRef) free(ssRef);
		ssRef = 0L;					if(name) free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		 cw *= ((scaleINFO*)tmpl)->sy.fy;					cl *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_ARROW3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_MRK_DIRTY:			//from Undo ?
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !CurrGO) {
				for(i = 0; i < 3; i++) {
					if(ls[i] && ls[i]->ObjThere(mev->x, mev->y)){
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
					}
				}
			break;
			}
		break;
	case CMD_ARROW_ORG3D:
		if(tmpl) memcpy(&fPos1, tmpl, sizeof(fPOINT3D));
		return true;
	case CMD_ARROW_TYPE:
		if(tmpl) type = *((int*)tmpl);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >5 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos2.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos2.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos2.fz);
			data->GetValue(ssRef[3].y, ssRef[3].x, &fPos1.fx);
			data->GetValue(ssRef[4].y, ssRef[4].x, &fPos1.fy);
			data->GetValue(ssRef[5].y, ssRef[5].x, &fPos1.fz);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos1.fx, fPos1.fy, fPos1.fz);
			((Plot*)parent)->CheckBounds3D(fPos2.fx, fPos2.fy, fPos2.fz);
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a data line in 3D space
Line3D::Line3D(GraphObj *par, DataObj *d) :GraphObj(par, d)
{
}

Line3D::Line3D(GraphObj *par, DataObj *d, char *rx, char *ry, char *rz)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(rx && rx[0]) x_range = rlp_strdup(rx);
	if(ry && ry[0]) y_range = rlp_strdup(ry);
	if(rz && rz[0]) z_range = rlp_strdup(rz);
	DoUpdate();
	Id = GO_LINE3D;
	bModified = false;
}

Line3D::Line3D(GraphObj *par, DataObj *d, fPOINT3D *pt, int n_pt, int xc1, int xr1, int yc1, int yr1,
		int zc1, int zr1, int xc2, int xr2, int yc2, int yr2, int zc2, int zr2)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(pt && n_pt) {
		values = (fPOINT3D*)memdup(pt, n_pt * sizeof(fPOINT3D), 0L);
		nPts = n_pt;
		ls = (line_segment **)calloc(nPts-1, sizeof(line_segment*));
		}
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || zc1 >= 0 || zr1 >= 0 || 
		xc2 >= 0 || xr2 >= 0 || yc2 >= 0 || yr2 >= 0 || zc2 >= 0 || zr2 >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 6);
		if (ssRef) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = zc1;	ssRef[2].y = zr1;
			ssRef[3].x = xc2;	ssRef[3].y = xr2;
			ssRef[4].x = yc2;	ssRef[4].y = yr2;
			ssRef[5].x = zc2;	ssRef[5].y = zr2;
			cssRef = 6;
			}
		}
	Id = GO_LINE3D;
	bModified = false;
}

Line3D::Line3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;		Id = GO_LINE3D;
}

Line3D::~Line3D()
{
	int i;

	if(bModified) Undo.InvalidGO(this);
	if(ls){
		for (i = 0; i < (nPts - 1); i++) {
			if (ls[i]) delete(ls[i]);
			}
		free(ls);
		}
	if(pts && npts) free(pts);
	pts = 0L;		npts = 0;	cssRef = 0;
	if(x_range) free(x_range);
	x_range = 0L;					if(y_range) free(y_range);
	y_range = 0L;					if(z_range) free(z_range);
	z_range = 0L;					if(values) free(values);
	values = 0L;					if(ssRef) free(ssRef);
	ssRef = 0L;						if(mo) DelBitmapClass(mo);
	mo = 0L;
}

void
Line3D::DoPlot(anyOutput *o)
{
	int i, j;
	fPOINT3D fip;
	fPOINT3D p1, p2;
	POINT np;


	if(mo) DelBitmapClass(mo);
	mo = 0L;
	if(ls) {
		if(pts && npts) free(pts);
		npts = 0;	if(!(pts = (POINT*)calloc(nPts+1, sizeof(POINT))))return;
		for(i = 0; i< nPts; i++) {
			if(!o->fvec2ivec(&values[i], &fip)) return;
			p2.fx = fip.fx;			p2.fy = fip.fy;			p2.fz = fip.fz;
			np.x = iround(p2.fx);	np.y = iround(p2.fy);
			AddToPolygon(&npts, pts, &np);
			if(i) {
				UpdateMinMaxRect(&rDims, np.x, np.y);
				j = i-1;
				if(ls[j]) delete(ls[j]);
				ls[j] = new line_segment(this, data, &Line, &p1, &p2);
				if (ls[j]) ls[j]->DoPlot(o);
				}
			else {
				rDims.left = rDims.right = iround(p2.fx);
				rDims.top = rDims.bottom = iround(p2.fy);
				}
			p1.fx = p2.fx;	p1.fy = p2.fy;	p1.fz = p2.fz;
			}
		if((rDims.right - rDims.left) < 3 || (rDims.bottom - rDims.top) < 3)
			IncrementMinMaxRect(&rDims, 3);
		}
}

void
Line3D::DoMark(anyOutput *o, bool mark)
{
	if(mark) {
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		InvertLine(pts, npts, &Line, &rDims, o, true);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
Line3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		if(name) free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_SET_LINE:
		if(tmpl) {
			memcpy(&Line, tmpl, sizeof(LineDEF));
			return true;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_LINE3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued  by Undo (no output given)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		//Should we ever come here ?
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(rDims, (p1.x= mev->x), (p1.y=mev->y))|| CurrGO || !o || nPts <2 ||
				!IsCloseToPL(p1, pts, npts, 3.0))return false;
			o->ShowMark(CurrGO=this, MRK_GODRAW);
			return true;
			}
		return false;
	case CMD_UPDATE:					case CMD_AUTOSCALE:
		if (cmd == CMD_UPDATE) {
			if (parent && parent->Id != GO_GRID3D) {
				Undo.DataMem(this, (void**)&values, nPts * sizeof(fPOINT3D), &nPts, UNDO_CONTINUE);
				}
			if (ssRef && cssRef >5 && data && nPts == 2) {
				data->GetValue(ssRef[0].y, ssRef[0].x, &values[0].fx);
				data->GetValue(ssRef[1].y, ssRef[1].x, &values[0].fy);
				data->GetValue(ssRef[2].y, ssRef[2].x, &values[0].fz);
				data->GetValue(ssRef[3].y, ssRef[3].x, &values[1].fx);
				data->GetValue(ssRef[4].y, ssRef[4].x, &values[1].fy);
				data->GetValue(ssRef[5].y, ssRef[5].x, &values[1].fz);
				return true;
				}
			else DoUpdate();
			}
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH){
			if(min.fx == max.fx || min.fy == max.fy){	//z's may be equal !
				min.fx = min.fy = min.fz = HUGE_VAL;
				max.fx = max.fy = max.fz = -HUGE_VAL;
				for(i = 0; i < nPts; i++) {
					if(values[i].fx < min.fx) min.fx = values[i].fx;
					if(values[i].fy < min.fy) min.fy = values[i].fy;
					if(values[i].fz < min.fz) min.fz = values[i].fz;
					if(values[i].fx > max.fx) max.fx = values[i].fx;
					if(values[i].fy > max.fy) max.fy = values[i].fy;
					if(values[i].fz > max.fz) max.fz = values[i].fz;
					}
				}
			((Plot*)parent)->CheckBounds3D(min.fx, min.fy, min.fz);
			((Plot*)parent)->CheckBounds3D(max.fx, max.fy, max.fz);
			return true;
			}
		return false;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

void
Line3D::DoUpdate()
{
	int n1 = 0, ic = 0;
	long i, j, k, l, m, n;
	double x, y, z;
	AccRange *rX=0L, *rY=0L, *rZ=0L;

	if(!x_range || !y_range || !z_range) return;
	if(values) free(values);
	values = 0L;
	if(ls) free(ls);
	ls = 0L;
	rX = new AccRange(x_range);
	rY = new AccRange(y_range);
	rZ = new AccRange(z_range);
	min.fx = min.fy = min.fz = HUGE_VAL;	max.fx = max.fy = max.fz = -HUGE_VAL;
	if(rX) n1 = rX->CountItems();
	values = (fPOINT3D*)malloc(n1 * sizeof(fPOINT3D));
	if(n1 && rY && rZ && values){
		rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
		rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
		rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
		do {
			if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && 
				data->GetValue(n, m, &z)){
				values[ic].fx = x;	values[ic].fy = y;	values[ic].fz = z;
				if(x < min.fx) min.fx = x;
				if(x > max.fx) max.fx = x;
				if(y < min.fy) min.fy = y;
				if(y > max.fy) max.fy = y;
				if(z < min.fz) min.fz = z;
				if(z > max.fz) max.fz = z;
				ic++;
				}
			}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
		nPts = ic;
		if(ic > 1) ls = (line_segment **)calloc(ic-1, sizeof(line_segment*));
		}
	if(rX) delete(rX);
	if(rY) delete(rY);
	if(rZ) delete(rZ);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a polygon in 3D space
Polygon3D::Polygon3D(GraphObj *par, DataObj *d, char *rx, char *ry, char *rz)
	:Line3D(par, d, rx, ry, rz)
{
	FileIO(INIT_VARS);
	if (rx && rx[0]) x_range = rlp_strdup(rx);
	if (ry && ry[0]) y_range = rlp_strdup(ry);
	if (rz && rz[0]) z_range = rlp_strdup(rz);
	DoUpdate();
	Id = GO_POLYGON3D;
	bModified = false;
}

Polygon3D::Polygon3D(GraphObj *par, DataObj *d, fPOINT3D *pt, int n_pt, int xc1, int xr1, int yc1, int yr1,
	int zc1, int zr1, int xc2, int xr2, int yc2, int yr2, int zc2, int zr2)
	:Line3D(par, d, pt, n_pt, xc1, xr1, yc1, yr1, zc1, zr1, xc2, xr2, yc2, yr2, zc2, zr2)
{
	FileIO(INIT_VARS);
	if (pt && n_pt) {
		int i;
		values = (fPOINT3D*)calloc(n_pt + 2, sizeof(fPOINT3D));
		for (i = 0; i < n_pt; i++) {
			values[i] = pt[i];
			}
		if (values[i - 1].fx != pt[0].fx || values[i - 1].fy != pt[0].fy || values[i - 1].fz != pt[0].fz) {
			//close polygon if necessary
			values[i] = pt[0];
			n_pt++;
			}
		nPts = n_pt;
		ls = (line_segment **)calloc(nPts, sizeof(line_segment*));
		}
	if (xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || zc1 >= 0 || zr1 >= 0 ||
		xc2 >= 0 || xr2 >= 0 || yc2 >= 0 || yr2 >= 0 || zc2 >= 0 || zr2 >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 6);
		if (ssRef) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = zc1;	ssRef[2].y = zr1;
			ssRef[3].x = xc2;	ssRef[3].y = xr2;
			ssRef[4].x = yc2;	ssRef[4].y = yr2;
			ssRef[5].x = zc2;	ssRef[5].y = zr2;
			cssRef = 6;
			}
		}
	Id = GO_POLYGON3D;
	bModified = false;
}

Polygon3D::Polygon3D(int src):Line3D(NULL, NULL)
{
	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_POLYGON3D;
	bModified = false;
}

Polygon3D::~Polygon3D()
{
	int i;

	if (bModified) Undo.InvalidGO(this);
	if (ls){
		for (i = 0; i < (nPts - 1); i++) {
			if (ls[i]) delete(ls[i]);
		}
		free(ls);
	}
	if (pts && npts) free(pts);
	pts = 0L;		npts = 0;	cssRef = 0;
	if (x_range) free(x_range);
	x_range = 0L;					if (y_range) free(y_range);
	y_range = 0L;					if (z_range) free(z_range);
	z_range = 0L;					if (values) free(values);
	values = 0L;					if (ssRef) free(ssRef);
	ssRef = 0L;						if (mo) DelBitmapClass(mo);
	mo = 0L;
}

double 
Polygon3D::GetSize(int select)
{
	return parent->GetSize(select);
}

void
Polygon3D::DoPlot(anyOutput *o)
{
	if (!values || !nPts) return;
	if (!plane){
		plane = new Plane3D(this, data, values, nPts);
		}
	if (plane) {
		plane->Command(CMD_PG_FILL, &Fill, o);
		plane->Command(CMD_SET_LINE, &Line, o);
		plane->DoPlot(o);
		}
}

void 
Polygon3D::DoMark(anyOutput *, bool )
{

}

bool 
Polygon3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_FLUSH:
		if (name) free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_SET_LINE:
		if (tmpl) {
			memcpy(&Line, tmpl, sizeof(LineDEF));
			return true;
			}
		return false;
	case CMD_PG_FILL:
		if (tmpl) {
			memcpy(&Fill, tmpl, sizeof(FillDEF));
			return true;
			}
		return false;
	case CMD_SAVE_PG:
		Undo.Fill(this, &Fill, 0L);
		Undo.Line(this, &Line, UNDO_CONTINUE);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_POLYGON3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued  by Undo (no output given)
		if (!parent) return false;
		if (!o) return parent->Command(cmd, tmpl, o);
		//Should we ever come here ?
		return true;
	case CMD_MOUSE_EVENT:
		return false;
	case CMD_UPDATE:					case CMD_AUTOSCALE:
		if (!values | !nPts) return false;
		if (!plane){
			plane = new Plane3D(this, data, values, nPts);
			}
		if (cmd == CMD_UPDATE)	DoUpdate();
		else if (cmd == CMD_AUTOSCALE && plane) {
			plane->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_SET_GO3D:
		if (parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return Line3D::Command(cmd, tmpl, o);
}

void 
Polygon3D::DoUpdate()
{
	int n1 = 0, ic = 0;
	long i, j, k, l, m, n;
	double x, y, z;
	AccRange *rX = 0L, *rY = 0L, *rZ = 0L;

	if (!x_range || !y_range || !z_range) return;
	if (values) free(values);
	values = 0L;
	if (ls) free(ls);
	ls = 0L;
	rX = new AccRange(x_range);
	rY = new AccRange(y_range);
	rZ = new AccRange(z_range);
	min.fx = min.fy = min.fz = HUGE_VAL;	max.fx = max.fy = max.fz = -HUGE_VAL;
	if (rX) n1 = rX->CountItems();
	values = (fPOINT3D*)calloc(n1+2, sizeof(fPOINT3D));
	if (n1 && rY && rZ && values){
		rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
		rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
		rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
		do {
			if (data->GetValue(j, i, &x) && data->GetValue(l, k, &y) &&
				data->GetValue(n, m, &z)){
				values[ic].fx = x;	values[ic].fy = y;	values[ic].fz = z;
				if (x < min.fx) min.fx = x;
				if (x > max.fx) max.fx = x;
				if (y < min.fy) min.fy = y;
				if (y > max.fy) max.fy = y;
				if (z < min.fz) min.fz = z;
				if (z > max.fz) max.fz = z;
				ic++;
				}
			} while (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
		if (values[ic - 1].fx != values[0].fx || values[ic - 1].fy != values[0].fy || values[ic - 1].fz != values[0].fz) {
			//close polygon if necessary
			values[ic++] = values[0];
			}
		nPts = ic;
		if (ic > 1) ls = (line_segment **)calloc(ic - 1, sizeof(line_segment*));
		}
	if (rX) delete(rX);
	if (rY) delete(rY);
	if (rZ) delete(rZ);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the text label class 
Label::Label(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, DWORD flg, char *txt,
	 int xc, int xr, int yc, int yr, int tc, int tr):GraphObj(par, d)
{
	int cb;

	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;		flags = flg;
	bBusy = false;
	if(parent){
		fDist.fx = parent->GetSize(SIZE_LB_XDIST);
		fDist.fy = parent->GetSize(SIZE_LB_YDIST);
		}
	Id = GO_LABEL;
	if(td){
		memcpy(&TextDef, td, sizeof(TextDEF));
		if (txt && txt[0]) {
			cb = (int)strlen((char*)(txt)) + 8;			if (cb < 20) cb = 20;
			TextDef.text = (unsigned char*)malloc(cb *sizeof(unsigned char));
			rlp_strcpy(TextDef.text, cb, txt);
			}
		else if(td->text && td->text[0]) {
			cb = (int)strlen((char*)(td->text))+8;			if(cb < 20) cb = 20;
			TextDef.text = (unsigned char*)malloc(cb *sizeof(unsigned char));
			rlp_strcpy(TextDef.text, cb, td->text);
			}

		}
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || tc >= 0 || tr >= 0) {
		ssRef = (POINT*)malloc(sizeof(POINT) * 3);
		if(ssRef) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = tc;	ssRef[2].y = tr;
			cssRef = 3;
			}
		}
}

Label::Label(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, DWORD flg, char *txt)
	:GraphObj(par, d)
{
	int cb;

	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;		flags = flg;
	if (parent){
		fDist.fx = parent->GetSize(SIZE_LB_XDIST);
		fDist.fy = parent->GetSize(SIZE_LB_YDIST);
		}
	Id = GO_LABEL;	bBusy = false;
	if (td){
		memcpy(&TextDef, td, sizeof(TextDEF));
		if (txt && txt[0]) {
			cb = (int)strlen((char*)(txt)) + 8;			if (cb < 20) cb = 20;
			TextDef.text = (unsigned char*)malloc(cb *sizeof(unsigned char));
			rlp_strcpy(TextDef.text, cb, txt);
			}
		else if (td->text && td->text[0]) {
			cb = (int)strlen((char*)(td->text)) + 8;			if (cb < 20) cb = 20;
			TextDef.text = (unsigned char*)malloc(cb *sizeof(unsigned char));
			rlp_strcpy(TextDef.text, cb, td->text);
			}
		}
}

Label::Label(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Label::~Label()
{
	HideTextCursor();
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified)Undo.InvalidGO(this);
	if (name) free(name);
	name = 0L;
}

double
Label::GetSize(int select)
{
	switch(select){
	case SIZE_CURSORPOS:
		return (double)CursorPos;
	case SIZE_CURSOR_XMIN:
		return (double) (Cursor.right > Cursor.left ? Cursor.left : Cursor.right);
	case SIZE_CURSOR_XMAX:
		return (double) (Cursor.right > Cursor.left ? Cursor.right : Cursor.left);
	case SIZE_CURSOR_YMIN:
		return (double) (Cursor.bottom > Cursor.top ? Cursor.top : Cursor.bottom);
	case SIZE_CURSOR_YMAX:
		return (double) (Cursor.bottom > Cursor.top ? Cursor.bottom : Cursor.top);
	case SIZE_MIN_Z:	case SIZE_MAX_Z:	case SIZE_ZPOS:
		return curr_z;
	case SIZE_XPOS:		return fPos.fx;
	case SIZE_YPOS:		return fPos.fy;
		}
	return 0.0;
}

bool
Label::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_LB_XDIST:			fDist.fx = value;		return true;
	case SIZE_LB_YDIST:			fDist.fy = value;		return true;
	case SIZE_XPOS:				fPos.fx = value;		return true;
	case SIZE_YPOS:				fPos.fy = value;		return true;
	case SIZE_ZPOS:
		curr_z = value;			return is3D = true;
	case SIZE_TEXT:				TextDef.fSize = value;		return true;
		}
	return false;
}

bool
Label::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_TEXT:
		if(select & UNDO_STORESET) {
			Undo.ValDword(this, &TextDef.ColTxt, UNDO_CONTINUE);
			bModified = true;
			}
		TextDef.ColTxt = col;			bBGvalid = false;
		return true;
	case COL_BG:
		bgcolor = col;	bBGvalid = true;
		return true;
		}
	return false;
}

void
Label::DoPlot(anyOutput *o)
{
	if (hidden) return;
	bBusy = false;
	if (this != CurrGO && m1 >= 0 && m2 >= 0) {
		m1 = m2 = -1;
		}
	if(is3D && parent && parent->Command(CMD_SET_GO3D, this, o)) return;
	DoPlotText(o);
}

void
Label::DoMark(anyOutput *o, bool mark)
{
	DWORD bgpix[16];
	int i, d, d1, ix, iy, dx, dy, n = 0;

	if(mark) {
		//find color with maximum contrast to text
		if(!bBGvalid) {
			mrc.left = rDims.left;		mrc.right = rDims.right;
			mrc.top = rDims.top;		mrc.bottom = rDims.bottom;
			dx = mrc.right - mrc.left;	dy = mrc.bottom - mrc.top;
			if(dx <= 0 || dy <= 0) return;
			mo = GetRectBitmap(&mrc, o);
			if(mo) {
				for(i = 0,  ix = 1; ix < dx; ix += dx<<2) {
					for(iy = 1; iy < dy && i < 16; dy += dy<<2) {
						if(!(mo->oGetPix(pts[ix].x, pts[iy].y, &bgpix[i]))) bgpix[i] = 0x00ffffff;
						i++;
						}
					}
				DelBitmapClass(mo);		n = i;
				mo = 0L;
				}
			bgcolor = bgpix[0];
			d = ColDiff(bgcolor, TextDef.ColTxt);
			for(i = 1; i < n; i++) {
				if(d < (d1 = ColDiff(bgpix[i], TextDef.ColTxt))) {
					d = d1;		bgcolor = bgpix[i];
					}
				}
			if(!d) bgcolor = TextDef.ColTxt ^ 0x00ffffffL;
			bBGvalid = true;
			}
		//in dialogs parent has no parent
		InvalidateOutput(o);
#ifdef _WINDOWS
		if(parent && parent->parent) o->ShowLine(pts, 5, TextDef.ColTxt);
#endif
		ShowCursor(o);	CurrGO = this;		CurrLabel = this;
		}
	else if(m1 >= 0 && m2 >= 0 && m1!= m2) {
		m1 = m2 = -1;
		parent->Command(CMD_REDRAW, 0L, o);
		}
	else {
		HideTextCursor();				m1 = m2 = -1;
		bgLine.color = bgcolor;			o->SetLine(&bgLine);
		//in dialogs parent has no parent
		if(parent && parent->parent) o->oPolyline(pts, 5);
		IncrementMinMaxRect(&rDims, 3);
		o->UpdateRect(&rDims, true);	IncrementMinMaxRect(&rDims, -3);
		if(CurrLabel == this) CurrLabel = 0L;
		if(parent && parent->Id != GO_MLABEL && (!TextDef.text || !TextDef.text[0]))
			parent->Command(CMD_DELOBJ, (void*)this, o);
		}
}

bool
Label::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	scaleINFO *scale;
	int i, cb;

	if(cmd != CMD_SET_DATAOBJ && !parent) return false;
	switch (cmd) {
	case CMD_SCALE:
		scale = (scaleINFO*)tmpl;
		if((flags & 0x03)!= LB_X_DATA) fPos.fx *= scale->sx.fy;
		if((flags & 0x30)!= LB_Y_DATA) fPos.fy *= scale->sy.fy;
		fDist.fx *= scale->sx.fy;				fDist.fy *= scale->sy.fy;
		TextDef.fSize *= scale->sx.fy;			TextDef.iSize = 0;
		return true;
	case CMD_CONFIG:
		return PropertyDlg();
	case CMD_HIDEMARK:
		if(m1 >=0 && m2 >=0 && m1 != m2) {
			m1 = m2 = -1;						return true;
			}
		return false;
	case CMD_FLUSH:
		if(CurrLabel == this) CurrLabel = 0L;
		if(TextDef.text) free(TextDef.text);
		TextDef.text = 0L;
		if(ssRef) free(ssRef);
		ssRef = 0L;
		return true;
	case CMD_POS_FIRST:		case CMD_POS_LAST:
		Undo.ValInt(this, &CursorPos, 0L);
		bModified = true;
		if(o && TextDef.text) {
			CursorPos = (cmd == CMD_POS_LAST) ? rlp_strlen((char*)TextDef.text) : 0;
			ShowCursor(o);
			return true;
			}
		return false;
	case CMD_SHIFTLEFT:		case CMD_CURRLEFT:
		bBusy = false;
		if(o && CursorPos >0 && TextDef.text) {
			Undo.ValInt(this, &CursorPos, 0L);
			DrawFmtText.SetText(0L, TextDef.text, 0L, 0L, true);
			bModified = true;
			if(cmd == CMD_SHIFTLEFT) {
				if(CursorPos && CursorPos == m1) {
					DrawFmtText.cur_left(&CursorPos);
					m1 = CursorPos;		ShowCursor(o);
					}
				else if(CursorPos && CursorPos == m2) {
					DrawFmtText.cur_left(&CursorPos);
					m2 = CursorPos;		ShowCursor(o);
					if(parent) parent->Command(CMD_REDRAW, 0L, o); 
					}
				else if(CursorPos){
					m2 = CursorPos;
					DrawFmtText.cur_left(&CursorPos);
					m1 = CursorPos;		ShowCursor(o);
					}
				}
			else {
				if(m1 >= 0 && m2 >= 0 && parent) {
					m1 = m2 = -1;						parent->Command(CMD_REDRAW, 0L, o);
					}
				DrawFmtText.cur_left(&CursorPos);		ShowCursor(o);
				}
			if(m1 >=0 && m2 >=0 && m1 != m2) DoPlot(o);
			CurrGO = CurrLabel = this;
			return true;
			}
		return false;
	case CMD_SHIFTRIGHT:	case CMD_CURRIGHT:
		bBusy = false;
		if(o && TextDef.text && CursorPos < rlp_strlen(TextDef.text)) {
			Undo.ValInt(this, &CursorPos, 0L);
			DrawFmtText.SetText(0L, TextDef.text, 0L, 0L, true);
			bModified = true;
			if(cmd == CMD_SHIFTRIGHT) {
				if(CursorPos == m1 && TextDef.text[m1]) {
					DrawFmtText.cur_right(&CursorPos);
					m2 = CursorPos;		ShowCursor(o);
					memcpy(&rm2, &Cursor, sizeof(RECT));
					if(parent) parent->Command(CMD_REDRAW, 0L, o); 
					}
				else if(CursorPos == m2 && TextDef.text[m2]) {
					DrawFmtText.cur_right(&CursorPos);
					m2 = CursorPos;		ShowCursor(o);
					memcpy(&rm2, &Cursor, sizeof(RECT));
					}
				else if(TextDef.text[CursorPos]){
					if(m1 < 0) {
						m1 = CursorPos;		ShowCursor(o);
						}
					DrawFmtText.cur_right(&CursorPos);
					m2 = CursorPos;		ShowCursor(o);
					}
				else return false;
				}
			else {
				if(m1 >= 0 && m2 >= 0 && parent) {
					m1 = m2 = -1;						parent->Command(CMD_REDRAW, 0L, o);
					}
				DrawFmtText.cur_right(&CursorPos);		ShowCursor(o);
				}
			if(m1 >=0 && m2 >=0 && m1 != m2) DoPlot(o);
			CurrGO = CurrLabel = this;
			return true;
			}
		return false;
	case CMD_ADDCHAR:		case CMD_ADDCHARW:			case CMD_BACKSP:
		bBusy = false;
		if (cmd == CMD_ADDCHAR || cmd == CMD_ADDCHARW){
			SetModified();				//value 8 == backspace
			if (tmpl && 8 != *((int*)tmpl)) return AddChar(*((int*)tmpl), o);
			}
		SetModified();
		if(CursorPos <=0 && o) {
			if(parent && parent->Id == GO_MLABEL) {
				parent->Command(CMD_SETFOCUS, this, o);
				return parent->Command(CMD_BACKSP, tmpl, o);
				}
			RedrawEdit(o);
			return true;
			}
		DrawFmtText.SetText(0L, TextDef.text, 0L, 0L, true);
		DrawFmtText.cur_left(&CursorPos);					//continue as if delete
		//fall through
	case CMD_DELETE:
		SetModified();
		if(TextDef.text && TextDef.text[CursorPos]) {
			Undo.String(this, (char**)(&TextDef.text), 0L);
			if(cmd == CMD_DELETE) Undo.ValInt(this, &CursorPos, UNDO_CONTINUE);
			if(CheckMark() && m2 < (cb = (int) strlen((char*)TextDef.text))) {
				Undo.ValInt(this, &m1, UNDO_CONTINUE);
				Undo.ValInt(this, &m2, UNDO_CONTINUE);
				rlp_strcpy(TextDef.text + m1, cb, TextDef.text + m2);
				CursorPos = m1;			m1 = m2 = -1;
				}
			else {
				DrawFmtText.SetText(0L, TextDef.text, 0L, 0L, true);
				cb = CursorPos;		DrawFmtText.cur_right(&cb);
				cb -= CursorPos;	if(cb < 1) cb = 1;
				rlp_strcpy(TextDef.text + CursorPos, TMP_TXT_SIZE, TextDef.text + CursorPos + cb);
				}
			if(o) {
				RedrawEdit(o);		ShowCursor(o);
				}
			}
		else if(TextDef.text && parent->Id == GO_MLABEL) {
			parent->Command(CMD_SETFOCUS, this, o);
			return parent->Command(CMD_DELETE, tmpl, o);
			}
		else o->HideMark();
		break;
	case CMD_GETTEXT:
		if(TextDef.text && TextDef.text[0] && tmpl) {
			rlp_strcpy((char*)tmpl, TMP_TXT_SIZE, TextDef.text);
			return true;
			}
		return false;
	case CMD_SETTEXT:
		if(TextDef.text) Undo.String(this, (char**)(&TextDef.text), UNDO_CONTINUE);
		if(TextDef.text) free(TextDef.text);
		TextDef.text = 0L;
		if(tmpl && *((char*)tmpl)) {
			TextDef.text = rlp_strdup((unsigned char*)tmpl);
			}
		return true;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		memcpy(&TextDef, tmpl, sizeof(TextDEF)-sizeof(char*));
		if(((TextDEF*)tmpl)->text) Command(CMD_SETTEXT, ((TextDEF*)tmpl)->text, o);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_LABEL;
		data = (DataObj*)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			if(data->GetText(ssRef[2].y, ssRef[2].x, TmpTxt, TMP_TXT_SIZE)) {
				Undo.String(this, (char**)(&TextDef.text), UNDO_CONTINUE);
				TextDef.text = (unsigned char*)realloc(TextDef.text, cb = (int)strlen(TmpTxt)+2);
				if(TmpTxt[0]) rlp_strcpy(TextDef.text, cb, TmpTxt);
				else TextDef.text[0] = 0;
				}
			return true;
			}
		return false;
	case CMD_SELECT:
		if(!o) return false;
		if(tmpl) CalcCursorPos(((POINT*)tmpl)->x, ((POINT*)tmpl)->y, o);
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_MOVE:
			if((mev->StateFlags & 0x01) && ObjThere(mev->x, mev->y)) {
				i = CursorPos;				CalcCursorPos(mev->x, mev->y, o);
				if(CurrLabel && CurrLabel != this) {
					CurrLabel->Command(CMD_HIDEMARK, tmpl, o);
					}
				CurrGO = CurrLabel = this;
				if(i == CursorPos) return true;
				if(CursorPos > m1 && CursorPos < m2) {
					if(m1 < 0) m1 = CursorPos;
					else if(m2 != CursorPos)m2 = CursorPos;
					parent->Command(CMD_REDRAW, 0L, o);
					}
				else {
					if(m1 < 0) m1 = CursorPos;
					else if(m2 != CursorPos)m2 = CursorPos;
					if(m1 >=0 && m2 >=0 && m1 != m2) DoPlot(o);
					}
				return true;
				}
			break;
		case MOUSE_LBUP:
			if (bBusy) return false;
			if(ObjThere(mev->x, mev->y)) {
				bBusy = false;
				HideTextCursor();
				if(parent && parent->Id == GO_MLABEL) parent->Command(CMD_SETFOCUS, this, o);
				CalcCursorPos(mev->x, mev->y, o);		
				if(o->MrkRect && (void*)o->MrkRect == (void*)this) o->MrkMode = MRK_NONE;
				if(m1 < 0) m1 = CursorPos;
				ShowCursor(o);
				o->ShowMark(this, MRK_GODRAW);
				}
			else if(m1 >= 0 && m2 >= 0) {
				m1 = m2 = -1;		DoPlot(o);
				}
			break;
			}
		break;
	case CMD_TEXTTHERE:
		if(ObjThere(((MouseEvent *)tmpl)->x, ((MouseEvent *)tmpl)->y)) {
			CalcCursorPos(((MouseEvent *)tmpl)->x, ((MouseEvent *)tmpl)->y, o);
			CalcRect(o);							ShowCursor(o);
			m1 = m2 = CursorPos;					CurrGO = this;								
			return true;
			}
		m1 = m2 = -1;
		return false;
	case CMD_AUTOSCALE:
		if(parent->Id >= GO_PLOT && parent->Id < GO_GRAPH
			&& (flags & LB_X_DATA) && (flags & LB_Y_DATA)) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
	case CMD_REDRAW:
		if(is3D && o) {
			DoPlotText(o);
			is3D = false;										//enable edit
			}
		else if(CurrGO == this) {
			if(parent && parent->parent) RedrawEdit(defDisp);	//not a dialog
			else ShowCursor(defDisp);							//dialog !
			}
		else return parent->Command(cmd, tmpl, o);
		return true;
	case CMD_MOVE:				case CMD_UNDO_MOVE:
		if (cmd == CMD_MOVE) {
			if (parent && (parent->Id == GO_MLABEL || parent->Id == GO_LEGITEM))
				return parent->Command(cmd, tmpl, o);
			Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			if(mo) DelBitmapClass(mo);
			mo = 0L;
			}
		if(!(flags & 0x03)) fPos.fx += ((lfPOINT*)tmpl)[0].fx;
		if(!(flags & 0x30)) fPos.fy += ((lfPOINT*)tmpl)[0].fy;
		if(o){
			o->StartPage();		parent->DoPlot(o);		o->EndPage();
			}
		CurrGO = this;
		return bModified = true;
	case CMD_PASTE:
		TestClipboard(this);		CurrLabel = this;
		break;
	case CMD_ADDTXT:
		return AddText((unsigned char*)tmpl, o);
		break;
		}
	return false;
}

void *
Label::ObjThere(int x, int y)
{
	POINT p1;

	if (IsInRect(rDims, (p1.x = x), (p1.y = y)) && IsInPolygon(&p1, pts, 5, 3.0)){
		DrawFmtText.SetText(0L, TextDef.text, &fix, &fiy, false);
		return this;
		}
	return 0L;
}

void
Label::Track(POINT *p, anyOutput *o, bool)
{
	POINT *tpts;
	int i;

	if(!parent || !TextDef.text || !TextDef.text[0] || !o) return;
	m1 = m2 = -1;
	if(parent->Id == GO_MLABEL || parent->Id == GO_LEGITEM){
		parent->Track(p, o, false);
		return;
		}
	if(o && (tpts = (POINT*)malloc(5*sizeof(POINT)))){
		trackLine.color = TextDef.ColTxt;
		o->SetLine(&trackLine);
		for (i = 0; i < 5; i++) {
			tpts[i].x = pts[i].x+p->x;	tpts[i].y = pts[i].y+p->y;
			}
		if (moveable) {
			if (mo) {
				RestoreRectBitmap(&mo, &mrc, o);
				DelBitmapClass(mo);				mo = 0L;
				}
			mrc.left = mrc.right = tpts[0].x;		mrc.top = mrc.bottom = tpts[0].y;
			for (i = 1; i < 5; i++) {
				UpdateMinMaxRect(&mrc, tpts[i].x, tpts[i].y);
				}
			IncrementMinMaxRect(&mrc, 5);		mo = GetRectBitmap(&mrc, o);
			o->dLineCol = trackLine.color;		o->LineWidth = defs.GetSize(SIZE_HAIRLINE);
			o->oPolyline(tpts, 5);				o->UpdateRect(&mrc, false);
			}
		else o->oPolyline(tpts, 5);
		free(tpts);
		}
}

bool
Label::CalcRect(anyOutput *o)
{
	double rx1;
	double rx, ry;
	fRECT rc, rcc;

	if(parent && parent->Id != GO_MLABEL) o->SetTextSpec(&TextDef);
	DrawFmtText.SetText(o, TextDef.text, &fix, &fiy, false);
	if(TextDef.text && TextDef.text[0]) {
		if(!(DrawFmtText.oGetTextExtent(o, &rx, &ry, 0))) return false;
		rx++;
		}
	else {
		if(!(o->oGetTextExtent((unsigned char*)"A", 1, &rx, &ry))) return false;
		rx = 1;
		}
	rx += 4;	rc.Xmin = -2.0f;	rc.Ymin = 0.0f;		rc.Xmax = rx;		rc.Ymax = ry;
	si = sin(TextDef.RotBL *0.01745329252);	csi = cos(TextDef.RotBL *0.01745329252);
	if(TextDef.Align & TXA_HCENTER) {
		rc.Xmin -= rx/2.0-1.0;		rc.Xmax -= rx/2.0-1.0;
		}
	else if(TextDef.Align & TXA_HRIGHT) {
		rc.Xmin -= rx-2.0;			rc.Xmax -= rx-2.0;
		}
	if(TextDef.Align & TXA_VCENTER) {
		rc.Ymin -= ry/2.0;			rc.Ymax -= ry/2.0;
		}
	else if(TextDef.Align & TXA_VBOTTOM) {
		rc.Ymin -= ry;				rc.Ymax -= ry;
		}
	if(DrawFmtText.oGetTextExtent(o, &rx1, &ry, CursorPos)){
		rx = CursorPos ? (int)rx1 : 0;
		}
	else rx = 0;
	rcc.Xmax = rc.Xmin + (double)rx+2.0;	rcc.Ymin = rc.Ymin+2.0;
	rcc.Xmin = rc.Xmin;						rcc.Ymax = rc.Ymax-2.0;
	rc.Xmin -= 2.0;		rc.Ymin -= 2.0;
	rc.Xmax += 2.0;		rc.Ymax += 2.0;
	pts[0].x = iround(rc.Xmin*csi + rc.Ymin*si)+iround(fix);
	pts[0].y = iround(rc.Ymin*csi - rc.Xmin*si)+iround(fiy);
	pts[1].x = iround(rc.Xmax*csi + rc.Ymin*si)+iround(fix);
	pts[1].y = iround(rc.Ymin*csi - rc.Xmax*si)+iround(fiy);
	pts[2].x = iround(rc.Xmax*csi + rc.Ymax*si)+iround(fix);
	pts[2].y = iround(rc.Ymax*csi - rc.Xmax*si)+iround(fiy);
	pts[3].x = iround(rc.Xmin*csi + rc.Ymax*si)+iround(fix);
	pts[3].y = iround(rc.Ymax*csi - rc.Xmin*si)+iround(fiy);
	pts[4].x = pts[0].x;	pts[4].y = pts[0].y;
	rc.Xmin += 2.0;		rc.Ymin += 2.0;
	rc.Xmax -= 2.0;		rc.Ymax -= 2.0;
	Cursor.left = iround(rcc.Xmax*csi + rcc.Ymin*si)+iround(fix);
	Cursor.top = iround(rcc.Ymin*csi - rcc.Xmax*si)+iround(fiy);
	Cursor.right = iround(rcc.Xmax*csi + rcc.Ymax*si)+iround(fix);
	Cursor.bottom = iround(rcc.Ymax*csi - rcc.Xmax*si)+iround(fiy);
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	UpdateMinMaxRect(&rDims, pts[2].x, pts[2].y);
	UpdateMinMaxRect(&rDims, pts[3].x, pts[3].y);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	return true;
}

void 
Label::RedrawEdit(anyOutput *o)
{
	FillDEF bgFill = {FILL_NONE, bgcolor, 1.0, 0L, bgcolor};

	if(!o || !parent) return;
	bgLine.color = bgcolor;		o->SetLine(&bgLine);	o->SetFill(&bgFill);
	o->oPolygon(pts, 5);		IncrementMinMaxRect(&rDims, 3);		
	o->UpdateRect(&rDims, true);
	CalcRect(o);			bgLine.color ^= 0x00ffffffL;
	o->SetLine(&bgLine);		o->oPolygon(pts, 5);
	if(parent->Id == GO_MLABEL) {
		if(parent->parent && parent->parent->Id == GO_LEGITEM && parent->parent->parent)
			parent->parent->parent->DoPlot(o);
		else parent->DoPlot(o);
		}
	else if(parent->Id == GO_LEGITEM && parent->parent) parent->parent->DoPlot(o);
	else DoPlot(o);
	o->UpdateRect(&rDims, true);
	DoMark(o, true);			ShowCursor(o);
}

void
Label::SetModified()
{
	AxisDEF *adef;

	bModified = true;
	if(parent && parent->Id==GO_TICK && parent->parent && parent->parent->Id==GO_AXIS){
	adef = ((Axis*)(parent->parent))->GetAxis();
	adef->flags &= ~AXIS_AUTOSCALE;
	}
}

void
Label::DoPlotText(anyOutput *o)
{
	static LineDEF yLine = {0.0, 1.0, 0x0000ffff, 0x0};
	static FillDEF yFill = {0, 0x0000ffff, 1.0, 0, 0x0000ffff};
	POINT mpts[5];
	int i;

	if(!parent || !o) return;
	if(m1 >= 0 && m2 >= 0 && m1 != m2 && CurrGO == this) {
		i = CursorPos;							CursorPos = m1;
		CalcRect(o);							memcpy(&rm1, &Cursor, sizeof(RECT));
		CursorPos = m2;							CalcRect(o);
		memcpy(&rm2, &Cursor, sizeof(RECT));	CursorPos = i;
		if(CurrGO == this) ShowCursor(o);
		else CalcRect(o);
		if(m2 > m1) {
			mpts[0].x = mpts[4].x = rm1.left;	mpts[1].x = rm1.right;
			mpts[0].y = mpts[4].y = rm1.top;	mpts[1].y = rm1.bottom;
			mpts[2].x = rm2.right;				mpts[2].y = rm2.bottom;
			mpts[3].x = rm2.left;				mpts[3].y = rm2.top;
			}
		else {
			mpts[0].x = mpts[4].x = rm2.left;	mpts[1].x = rm2.right;
			mpts[0].y = mpts[4].y = rm2.top;	mpts[1].y = rm2.bottom;
			mpts[2].x = rm1.right;				mpts[2].y = rm1.bottom;
			mpts[3].x = rm1.left;				mpts[3].y = rm1.top;
			}
		o->SetLine(&yLine);						o->SetFill(&yFill);
		o->oPolygon(mpts, 5, 0L);
		}
	else {
		m1 = m2 = -1;
		if(CurrGO == this) ShowCursor(o);
		}
	defDisp = o;
	if(parent && parent->Id == GO_MLABEL) parent->Command(CMD_SETFOCUS, this, o);
	switch(flags & 0x03) {
	case LB_X_DATA:	fix = o->fx2fix(fPos.fx);		break;
	case LB_X_PARENT: 
		fix = parent->GetSize(SIZE_LB_XPOS);
		break;
	default:
		fix = o->co2fix(fPos.fx + parent->GetSize(SIZE_GRECT_LEFT));
		break;
		}
	switch(flags & 0x30) {
	case LB_Y_DATA:	fiy = o->fy2fiy(fPos.fy);		break;
	case LB_Y_PARENT: 
		fiy = parent->GetSize(SIZE_LB_YPOS);
		break;
	default:
		fiy = o->co2fiy(fPos.fy +parent->GetSize(SIZE_GRECT_TOP));
		break;
		}
	fix += o->un2fix(fDist.fx);		fiy += o->un2fiy(fDist.fy);
	TextDef.iSize = 0;				o->SetTextSpec(&TextDef);
	if(TextDef.text && TextDef.text[0]){
		DrawFmtText.SetText(o, TextDef.text, &fix, &fiy, true);
		}
	if(!(CalcRect(o))) return;
	if(m1 >= 0 && m2 >= 0 && m1 != m2 && CurrGO == this && fabs(TextDef.RotBL) < 0.01) {
		o->CopyBitmap(mpts[0].x, mpts[0].y, o, mpts[0].x, mpts[0].y,
			mpts[2].x - mpts[0].x, mpts[2].y - mpts[0].y, true);
		}
	if(m1 >= 0 && m2 >= 0) o->UpdateRect(&rDims, true);
}

bool
Label::CheckMark()
{
	int m;

	if(m1 < 0 || m2 < 0 || m1 == m2) return false;
	if(m1 < m2) return true;
	//come here on right to left mark: swap m1 and m2
	m = m1;		m1 = m2;	m2 = m;
	return	true;
}

//Append string to the crrent text.
//This function does not need Undo!
int
Label::Append(char* txt)
{
	int l1 = 0, l2 = 0, cb;

	if (TextDef.text && TextDef.text[0]) l1 = rlp_strlen(TextDef.text);
	if (txt && txt[0]) l2 = rlp_strlen(txt);
	cb = l1 + l2;
	if (cb < 20) cb = 20;
	if (TextDef.text) {
		TextDef.text = (unsigned char*) realloc(TextDef.text, cb+10);
		}
	else {
		TextDef.text = (unsigned char*) calloc(cb+10, sizeof(unsigned char));
		}
	rlp_strcpy(TextDef.text + l1, cb + 2, txt);
	return l1 + l2;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a multiline label consists of several Label objects
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
mLabel::mLabel(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, char *txt, 
	int cp, DWORD flg):GraphObj(par, d)
{
	int i;

	memcpy(&TextDef, td, sizeof(TextDEF));
	TextDef.text = 0L;		fPos.fx = x;		fPos.fy = y;
	Lines = 0L;				flags = flg;		lspc = 1.0;
	fDist.fx = 0.0;			fDist.fy = 0.0;
	CurrGO = CurrLabel = 0L;
	if(txt && (Lines = (Label**)calloc(2, sizeof(Label*)))) {
		for(i = 0; i < cp && txt[i]; i ++) TmpTxt[i] = txt[i];
		TmpTxt[i] = 0;
		Lines[0] = new 	Label(this, d, x, y, &TextDef, LB_X_PARENT | LB_Y_PARENT, 0L);
		if (Lines[0]) Lines[0]->Command(CMD_SETTEXT, TmpTxt, 0L);
		Lines[1] = new 	Label(this, d, x, y, &TextDef, LB_X_PARENT | LB_Y_PARENT, 0L);
		if (Lines[1]){
			Lines[1]->Command(CMD_SETTEXT, txt+cp, 0L);
			CurrGO = CurrLabel = Lines[1];
			}
		nLines = 2;		cli = 1;
		}
	Id = GO_MLABEL;
}

mLabel::mLabel(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, char *txt)
	:GraphObj(par, d)
{
	int i, nll;
	char **llist;

	memcpy(&TextDef, td, sizeof(TextDEF));
	TextDef.text = 0L;		fPos.fx = x;		fPos.fy = y;
	Lines = 0L;				flags = 0L;			Id = GO_MLABEL;
	fDist.fx = 0.0;			fDist.fy = 0.0;		lspc = 1.0;
	CurrGO = CurrLabel = 0L;
	if(txt){
		if((llist=split(txt,'\n',&nll)) && nll && (Lines=(Label**)calloc(nll, sizeof(Label*)))){
			for(i = 0; i < nll; i++) {
				if(llist[i]){
					Lines[i] = new 	Label(this, d, x, y, &TextDef, LB_X_PARENT | LB_Y_PARENT, llist[i]);
					free(llist[i]);
					}
				}
			free(llist);	nLines = nll;		cli = 0;
			}
		}
}

mLabel::mLabel(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

mLabel::~mLabel()
{
	int i;
	
	Undo.InvalidGO(this);
	if(Lines){
		for(i = 0; i < nLines; i++) if(Lines[i]) DeleteGO(Lines[i]);
		free(Lines);
		}
	if (name) free(name);
	name = 0L;
}

double
mLabel::GetSize(int select)
{
	switch(select){
	case SIZE_LB_XPOS:	return cPos1.fx;
	case SIZE_LB_YPOS:	return cPos1.fy;
	case SIZE_GRECT_TOP:	
		if (parent) return parent->GetSize(select);
		break;
	case SIZE_MIN_Z:	case SIZE_MAX_Z:	case SIZE_ZPOS:
		return curr_z;
	case SIZE_LSPC:
		return lspc;
		}
	return 0.0;
}

bool
mLabel::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff) {
	case SIZE_LB_XDIST:
		undo_flags = CheckNewFloat(&fDist.fx, fDist.fx, value, this, undo_flags);
		return true;
	case SIZE_LB_YDIST:
		undo_flags = CheckNewFloat(&fDist.fy, fDist.fy, value, this, undo_flags);
		return true;
	case SIZE_XPOS:
		undo_flags = CheckNewFloat(&fPos.fx, fPos.fx, value, this, undo_flags);
		return true;
	case SIZE_YPOS:
		undo_flags = CheckNewFloat(&fPos.fy, fPos.fy, value, this, undo_flags);
		return true;
	case SIZE_ZPOS:
		curr_z = value;
		if(Lines) for(i = 0; i < nLines; i++) if(Lines[i]){
			Lines[i]->SetSize(select, value);
			}
		return is3D = true;
	case SIZE_LSPC:
		undo_flags = CheckNewFloat(&lspc, lspc, value, this, undo_flags);
		return true;
		}
	return false;
}

void
mLabel::DoPlot(anyOutput *o)
{
	int i;
	double dh, dx, dy;

	if(!o || !Lines) return;
	undo_flags = 0L;
	if(parent){		//if this object is part of a dialog we dont have a parent
		dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
		}
	else dx = dy = 0.0;
	cPos.fx = cPos.fy = 0.0;
	TextDef.iSize = o->un2iy(TextDef.fSize);
	switch(flags & 0x03) {
	case LB_X_DATA:		cPos.fx = o->fx2fix(fPos.fx);				break;
	case LB_X_PARENT:	if(parent) cPos.fx = parent->GetSize(SIZE_LB_XPOS);	break;
	default:
		//if no parent its a dialog
		cPos.fx = parent ? o->co2fix(fPos.fx + dx) : fPos.fx;
		break;
		}
	switch(flags & 0x30) {
	case LB_Y_DATA:		cPos.fy = o->fy2fiy(fPos.fy);				break;
	case LB_Y_PARENT:	if(parent) cPos.fy = parent->GetSize(SIZE_LB_YPOS);	break;
	default:	
		//if no parent its a dialog
		cPos.fy = parent ? o->co2fiy(fPos.fy + dy) : fPos.fy;
		break;
		}
	si = sin(TextDef.RotBL *0.01745329252f);	csi = cos(TextDef.RotBL *0.01745329252f);
	if(TextDef.Align & TXA_VBOTTOM) dh = (double)(nLines-1);
	else if(TextDef.Align & TXA_VCENTER) dh = ((double)(nLines-1))/2.0;
	else dh = 0.0;
	dh *= TextDef.fSize;
	cPos.fx -= o->un2fix(dh*si);		cPos.fy -= o->un2fiy(dh*csi);
	memcpy(&cPos1, &cPos, sizeof(lfPOINT));
	if(lspc < 0.5 || lspc > 5) lspc = 1.0;
#ifdef _WINDOWS
	dist.fx = floor(o->un2fix(TextDef.fSize * si * lspc));
	dist.fy = floor(o->un2fiy(TextDef.fSize * csi * lspc));
#else
	dist.fx = floor(o->un2fix(TextDef.fSize * si * 1.2 * lspc));
	dist.fy = floor(o->un2fiy(TextDef.fSize * csi * 1.2 * lspc));
#endif
	o->SetTextSpec(&TextDef);
	rDims.left = rDims.top = rDims.right = rDims.bottom = 0;
	for(i = 0; i < nLines; i++)	if(Lines[i]){
		Lines[i]->Command(CMD_SETTEXTDEF, &TextDef, o);
		Lines[i]->SetSize(SIZE_LB_XDIST, fDist.fx);	Lines[i]->SetSize(SIZE_LB_YDIST, fDist.fy);
		Lines[i]->SetSize(SIZE_XPOS, fPos.fx);		Lines[i]->SetSize(SIZE_YPOS, fPos.fy);
		Lines[i]->moveable = moveable;				Lines[i]->DoPlot(o);
		if (i) {
			UpdateMinMaxRect(&rDims, Lines[i]->rDims.left, Lines[i]->rDims.top);
			UpdateMinMaxRect(&rDims, Lines[i]->rDims.right, Lines[i]->rDims.bottom);
			}
		else {
			rDims.left = Lines[i]->rDims.left;		rDims.bottom = Lines[i]->rDims.bottom;
			rDims.right = Lines[i]->rDims.right;	rDims.top = Lines[i]->rDims.top;
			}
		}
	if(CurrLabel && o && Command(CMD_SETFOCUS, CurrLabel, o))
		Lines[cli]->ShowCursor(o);
}

bool
mLabel::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j, k;
	fRECT t_cur;
	MouseEvent mev;
	scaleINFO *scale;

	switch (cmd) {
	case CMD_MOUSE_EVENT:		case CMD_TEXTTHERE:
		if(Lines && tmpl && IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) 
			for(i = 0; i<nLines; i++) if(Lines[i] && Lines[i]->Command(cmd, tmpl, o)) return true;
		break;
	case CMD_ADDCHAR:
		if(!tmpl || 13 != *((int*)tmpl)) return false;
		if(!Lines[cli] || !Lines[cli]->Command(CMD_GETTEXT, &TmpTxt, o)) return false;
		k = iround(Lines[cli]->GetSize(SIZE_CURSORPOS));
		if(parent)Undo.ObjConf(this, 0L);
		if(tmpl && Lines && (Lines = (Label**)realloc(Lines, (nLines+1) * sizeof(Label*)))) {
			for(i = nLines-1, j = nLines; i >= cli; i--, j-- ) {
				Lines[j] = Lines[i];
				}
			i++, j++;
			if(Lines[j]) DeleteGO(Lines[j]);
			nLines++;
			Lines[j] = new 	Label(this, data, fPos.fx, fPos.fy, &TextDef, LB_X_PARENT | LB_Y_PARENT, 0L);
			if (Lines[j]){
				Lines[j]->Command(CMD_SETTEXT, TmpTxt+k, o);
				Lines[j]->Command(CMD_POS_FIRST, 0L, o);
				CurrGO = CurrLabel = Lines[j];
				TmpTxt[k] = 0;
				}
			Lines[i] = new 	Label(this, data, fPos.fx, fPos.fy, &TextDef, LB_X_PARENT | LB_Y_PARENT, 0L);
			if (Lines[i]) Lines[i]->Command(CMD_SETTEXT, TmpTxt, 0L);
			Command(CMD_SETFOCUS, CurrGO, o);
			if(parent) return parent->Command(CMD_REDRAW, 0L, o);
			else if(o) DoPlot(o);
			}
		break;
	case CMD_SETFOCUS:
		for(i = 0; i< nLines; i++) if(Lines[i] == (Label*)tmpl) {
			cli = i;
			cPos1.fx = cPos.fx + dist.fx*i;
			cPos1.fy = cPos.fy + dist.fy*i;
			return true;
			}
		break;
	case CMD_GETTEXT:
		if(tmpl && Lines && nLines && Lines[0]) return Lines[0]->Command(CMD_GETTEXT, tmpl, o);
		break;
	case CMD_ALLTEXT:							//used e.g from dialog text boxes
		if(tmpl && Lines && nLines){
			for(i = j = 0; i < nLines; i++) {
				if(Lines[i] && Lines[i]->Command(CMD_GETTEXT, TmpTxt+500, o) && TmpTxt[500]){
					j += rlp_strcpy((char*)tmpl+j, 500-j, TmpTxt+500);
					((char*)tmpl)[j++] = '\n';
					}
				((char*)tmpl)[j] = 0;
				}
			if(j >2) return true;
			}
		return false;
	case CMD_BACKSP:	case CMD_DELETE:
		if (cmd == CMD_DELETE) cli++;
		if(cli > 0 && cli < nLines && Lines && Lines[cli] && Lines[cli-1]) {
			Lines[cli-1]->Command(CMD_POS_LAST, 0L, o);
			TmpTxt[0] = 0;
			Lines[cli-1]->Command(CMD_GETTEXT, TmpTxt, o);
			Lines[cli]->Command(CMD_GETTEXT, TmpTxt+strlen(TmpTxt), o);
			Lines[cli-1]->Command(CMD_SETTEXT, TmpTxt, o);
			DeleteGO(Lines[cli]);	Lines[cli] = 0L;
			for(i = cli+1; i < nLines; i++){
				Lines[i-1] = Lines[i], Lines[i]= 0L;
				}
			nLines--;
			CurrGO = CurrLabel = Lines[cli-1];
			if(parent) parent->Command(CMD_REDRAW, 0L, o);
			if(CurrLabel) CurrLabel->RedrawEdit(o);
			return true;
			}
		return false;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		if(parent)Undo.TextDef(this, &TextDef, undo_flags);
		undo_flags |= UNDO_CONTINUE;
		memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = 0L;
		return true;
	case CMD_SET_DATAOBJ:
		if(Lines) {
			for(i = 0; i< nLines; i++) if(Lines[i]) Lines[i]->Command(cmd, tmpl, o);
			}
		Id = GO_MLABEL;
		data = (DataObj*)tmpl;
		return true;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH
			&& (flags & LB_X_DATA) && (flags & LB_Y_DATA)) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
	case CMD_CURRUP:		case CMD_CURRDOWN:
		if(!o) return false;
		o->SetTextSpec(&TextDef);
		Command(CMD_SETFOCUS, CurrGO, o);
		if(cli >= 0 && cli < nLines && Lines && Lines[cli]) {
			t_cur.Xmin = Lines[cli]->GetSize(SIZE_CURSOR_XMIN);
			t_cur.Xmax = Lines[cli]->GetSize(SIZE_CURSOR_XMAX);
			t_cur.Ymin = Lines[cli]->GetSize(SIZE_CURSOR_YMIN);
			t_cur.Ymax = Lines[cli]->GetSize(SIZE_CURSOR_YMAX);
			mev.StateFlags = 0;		mev.Action = MOUSE_LBUP;
			mev.x = iround((t_cur.Xmax+t_cur.Xmin)/2.0);
			mev.y = iround((t_cur.Ymax+t_cur.Ymin)/2.0);
			i = o->un2ix(TextDef.fSize*si);		j = o->un2iy(TextDef.fSize*csi);
			if(cmd == CMD_CURRUP && cli > 0 && Lines[cli-1]) {
				Lines[cli-1]->CalcCursorPos(mev.x -= i, mev.y -= j, o);
				o->ShowMark(CurrGO = CurrLabel = Lines[cli-=1], MRK_GODRAW);
				return Command(CMD_SETFOCUS, Lines[cli], o);
				}
			if(cmd == CMD_CURRDOWN && cli < (nLines-1) && Lines[cli+1]) {
				Lines[cli+1]->CalcCursorPos(mev.x += i, mev.y += j, o);
				o->ShowMark(CurrGO = CurrLabel = Lines[cli+=1], MRK_GODRAW);
				return Command(CMD_SETFOCUS, Lines[cli], o);
				}
			}
		else return false;
		break;
	case CMD_SCALE:
		scale = (scaleINFO*)tmpl;
		if((flags & 0x03)!= LB_X_DATA) fPos.fx *= scale->sx.fy;
		if((flags & 0x30)!= LB_Y_DATA) fPos.fy *= scale->sy.fy;
		fDist.fx *= scale->sx.fy;				fDist.fy *= scale->sy.fy;
		TextDef.fSize *= scale->sx.fy;			TextDef.iSize = 0;
		return true;
	case CMD_SELECT:
		if(o && tmpl) for(i = 0; i < nLines; i++){
			o->SetTextSpec(&TextDef);
			if(Lines[i] && ((POINT*)tmpl)->y > Lines[i]->rDims.top && ((POINT*)tmpl)->y < 
				Lines[i]->rDims.bottom) return Lines[i]->Command(cmd, tmpl, o);
			}
		break;
	case CMD_DELOBJ:
		if(parent && Lines) for(i = 0; i< nLines; i++) 
			if(Lines[i] && Lines[i] == (Label*)tmpl) return parent->Command(cmd, this, o);
		break;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		else if(o) DoPlot(o);
		break;
	case CMD_MOVE:						case CMD_UNDO_MOVE:
		if (cmd == CMD_MOVE){
			if (parent && parent->Id == GO_LEGITEM) return parent->Command(cmd, tmpl, o);
			Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			}
		if(!(flags & 0x03)) fPos.fx += ((lfPOINT*)tmpl)[0].fx;
		if(!(flags & 0x30)) fPos.fy += ((lfPOINT*)tmpl)[0].fy;
		if(o && parent){
			o->StartPage();		parent->DoPlot(o);		o->EndPage();
			}
		return true;
		}
	return false;
}

void
mLabel::Track(POINT *p, anyOutput *o, bool)
{
	int i;

	if(!parent) return;
	if(parent->Id == GO_LEGITEM){
		parent->Track(p, o, false);
		return;
		}
	for(i = 0; i < nLines; i++)	if(Lines[i]){
		if(!i) memcpy(&rDims, &Lines[i]->rDims, sizeof(RECT));
		else {
			UpdateMinMaxRect(&rDims, Lines[i]->rDims.left, Lines[i]->rDims.top);
			UpdateMinMaxRect(&rDims, Lines[i]->rDims.right, Lines[i]->rDims.bottom);
			}
		}
	defs.UpdRect(o, &rDims);
	Id = 0L;			//disable reentrance from Label objects
	for(i = 0; i < nLines; i++)	if(Lines[i]) Lines[i]->Track(p, o, false);
	Id = GO_MLABEL;		//enable entrance from Label objects
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a rectangular range to accept any text
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
TextFrame::TextFrame(GraphObj *parent, DataObj *data, lfPOINT *p1, lfPOINT *p2, char *txt)
	:GraphObj(parent, data)
{
	FileIO(INIT_VARS);					lspc = 1.0;
	pos1.fx = p1->fx;	pos1.fy = p1->fy;	pos2.fx = p2->fx;	pos2.fy = p2->fy;
	if(txt && txt[0]) {
		text = rlp_strdup((unsigned char*)txt);
		}
	else {
		lines = (unsigned char**)malloc(sizeof(char*));
		if (lines){
			lines[0] = (unsigned char*)calloc(TF_MAXLINE, sizeof(unsigned char));
			if (lines[0]) lines[0][0] = 0;
			nlines = 1;
			}
		}
	bFocus = false;		 CurrDisp = 0L;			 Id = GO_TEXTFRAME;
	UndoFlags = 0;
}

TextFrame::TextFrame(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	moveable = 1;				bModified = false;
	CurrDisp = 0L;				UndoFlags = 0;

}

TextFrame::~TextFrame()
{
	int i;

	if(text)free(text);
	text = 0L;					if(drc) delete(drc);
	drc = 0L;					if(tm_rec) free(tm_rec);
	tm_rec = 0L;
	if(lines) {
		for (i = 0; i < nlines; i++)	{
			if (lines[i]) free(lines[i]);
			}
		free(lines);	nlines = 0;		lines = 0L;
		}
	HideTextCursor();
	if (name) free(name);
	name = 0L;
}
double
TextFrame::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:		return pos1.fx;
	case SIZE_XPOS+1:	return pos2.fx;
	case SIZE_YPOS:		return pos1.fy;
	case SIZE_YPOS+1:	return pos2.fy;
	case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
	case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
TextFrame::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_XPOS:				pos1.fx = value;			return bResize = true;
	case SIZE_XPOS+1:			pos2.fx = value;			return bResize = true;
	case SIZE_YPOS:				pos1.fy = value;			return bResize = true;
	case SIZE_YPOS+1:			pos2.fy = value;			return bResize = true;
		}
	return false;
}

void 
TextFrame::DoMark(anyOutput *o, bool mark)
{
	RECT upd;

	if(has_m1 && has_m2) TextMark(o, 3);
	has_m1 = has_m2 = false;
	if(parent && !drc) drc = new dragRect(this, 0);		//in dialog no parent
	memcpy(&upd, &rDims, sizeof(RECT));
	if(mark){
		if(drc) drc->DoPlot(o);
		ShowCursor(o);
		}
	else if(parent)	parent->DoPlot(o);
	IncrementMinMaxRect(&upd, 6);
	o->UpdateRect(&upd, true);
}

void
TextFrame::DoPlot(anyOutput *o)
{
	int i, j;
	double tmp, dx, dy, x1, y1, x2, y2;
	POINT p1;
	RECT rc_clp;
	bool has_clp = true;

	if(!o) return;
	CurrDisp = o;
	if (!o->GetClipRec(&rc_clp)) has_clp = false;
	if(parent){		//if this object is part of a dialog we dont have a parent
		dx = parent->GetSize(SIZE_GRECT_LEFT);			dy = parent->GetSize(SIZE_GRECT_TOP);
		x1 = o->co2ix(pos1.fx+dx);						y1 = o->co2iy(pos1.fy+dy);
		x2 = o->co2ix(pos2.fx+dx);						y2 = o->co2iy(pos2.fy+dy);
		ipad.left = o->un2ix(pad.Xmin);					ipad.right = o->un2ix(pad.Xmax);
		ipad.top = o->un2iy(pad.Ymin);					ipad.bottom = o->un2iy(pad.Ymax);
		}
	else {
		dx = dy = 0.0;
		x1 = pos1.fx;		y1 = pos1.fy;		x2 = pos2.fx;		y2 = pos2.fy;
		ipad.left = ipad.right = ipad.top = ipad.bottom = 4;
		fmt_txt.EditMode(true);
		}
	if(pos1.fx > pos2.fx) {
		tmp = pos2.fx;	pos2.fx = pos1.fx;	pos1.fx = tmp;
		}
	if(pos1.fy > pos2.fy) {
		tmp = pos2.fy;	pos2.fy = pos1.fy;	pos1.fy = tmp;
		}
	o->SetLine(&Line);						o->SetFill(&Fill);
	o->oRectangle(iround(x1), iround(y1), iround(x2), iround(y2), name);
	SetMinMaxRect(&rDims, iround(x1), iround(y1), iround(x2), iround(y2));
	o->ClipRect(&rDims);
	x1 += ipad.left;						y1 += ipad.top;
	TextDef.iSize = o->un2iy(TextDef.fSize);
#ifdef _WINDOWS
	linc = o->un2iy(TextDef.fSize*lspc);
#else
	linc = o->un2iy(TextDef.fSize*lspc*1.2);
#endif
	o->SetTextSpec(&TextDef);				y1 += linc;
	if(text && text[0] && !(lines)) text2lines(o);
	else if(bResize && lines) {
		o->ClipRect(has_clp ? &rc_clp : 0L);
		c_char = lines[cur_pos.y][cur_pos.x];			lines[cur_pos.y][cur_pos.x] = 0x01;
		if(!c_char) lines[cur_pos.y][cur_pos.x+1] = 0x00; 
		lines2text();				text2lines(o);		bResize = false;
		if(parent)o->ShowMark(this, MRK_GODRAW);
		return;
		}
	if(has_m1 && has_m2) TextMark(o, 1);
	if(lines) for(i = 0; i < nlines; i++) {
		if(lines[i] && lines[i][0]){
			j = rlp_strlen(lines[i]);
			if(lines[i][j-1] == '\n') {
				lines[i][j-1] = 0;
				fmt_txt.SetText(o, lines[i], &x1, &y1, true);
				lines[i][j-1] = '\n';
				}
			else fmt_txt.SetText(o, lines[i], &x1, &y1, true);
			}
		y1 += linc;
		}
	if(has_m1 && has_m2) TextMark(o, 2);
	bModified = bResize = false;
	defs.UpdRect(o, &rDims);
	if (!parent) defs.Idle(CMD_UPDATE);
	o->ClipRect(has_clp ? &rc_clp : 0L);
	if (!parent && bFocus && lines && lines[0]){							//dialog?
		p1.x = rDims.right;		p1.y = rDims.top;
		ClearSym.set_rect(&p1);		ClearSym.do_plot(o);
		}
}

bool
TextFrame::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;
	double fi;

	switch (cmd) {
	case CMD_SELECT:
		if(!o || !tmpl) return false;
		CalcCursorPos(((POINT*)tmpl)->x, ((POINT*)tmpl)->y, o);
		if(parent)o->ShowMark(this, MRK_GODRAW);
		CurrGO = this;			bFocus = true;					DoPlot(o);
		ShowCursor(o);
		return true;
	case CMD_UNDO:				//from UndoObj::Restore()
		text2lines(o ? o : CurrDisp);
		DoPlot(CurrDisp);
		return true;
	case CMD_CLEAR:
		if (!o || !lines) return false;
		if (text)free(text);
		text = 0L;		lines[0][0] = 0;		nlines = 1;
		HideTextCursor();
		CalcCursorPos(rDims.left+1, rDims.top+1, o);
		DoPlot(o);						o->UpdateRect(&rDims, true);
		CurrGO = this;					ShowCursor(o);
		return true;
	case CMD_COPY:
		return CopyText(o, false);
	case CMD_SAVEPOS:
		bModified = true;
		Undo.SaveLFP(this, &pos1, 0L);
		Undo.SaveLFP(this, &pos2, UNDO_CONTINUE);
		return true;
	case CMD_SCALE:
		if(tmpl) {
			pos1.fx = ((scaleINFO*)tmpl)->sx.fx + pos1.fx * ((scaleINFO*)tmpl)->sx.fy;
			pos1.fy = ((scaleINFO*)tmpl)->sy.fx + pos1.fy * ((scaleINFO*)tmpl)->sy.fy;
			pos2.fx = ((scaleINFO*)tmpl)->sx.fx + pos2.fx * ((scaleINFO*)tmpl)->sx.fy;
			pos2.fy = ((scaleINFO*)tmpl)->sy.fx + pos2.fy * ((scaleINFO*)tmpl)->sy.fy;
			TextDef.fSize *= ((scaleINFO*)tmpl)->sy.fy;		TextDef.iSize = 0;
			pad.Xmax *= ((scaleINFO*)tmpl)->sx.fy;			pad.Xmin *= ((scaleINFO*)tmpl)->sx.fy;
			pad.Ymax *= ((scaleINFO*)tmpl)->sy.fy;			pad.Ymin *= ((scaleINFO*)tmpl)->sy.fy;
			Line.width *= ((scaleINFO*)tmpl)->sy.fy;		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;
			FillLine.width *= ((scaleINFO*)tmpl)->sy.fy;	FillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
			Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
			}
		return true;
	case CMD_GETTEXT:	case CMD_ALLTEXT:
		if(lines && lines[0] && lines[0][0] && nlines) {
			lines2text();	i = rlp_strcpy((char*)tmpl, TMP_TXT_SIZE-2, (char*)text);
			while(i && ((char*)tmpl)[i-1] == '\n') i--; 
			((char*)tmpl)[i++] = '\n';						((char*)tmpl)[i] = 0;
			return true;
			}
		return false;
	case CMD_ADDCHARW:	case CMD_ADDCHAR:
		bFocus = true;
		if(tmpl && o) AddChar(o, *((int *)tmpl));
		return true;
	case CMD_DELETE:
		bFocus = true;
		if(o) DelChar(o);
		return true;
	case CMD_SHIFTLEFT:					case CMD_CURRLEFT:
		bFocus = true;
		if (cmd == CMD_SHIFTLEFT){
			if (!has_m1) {
				m1_pos.x = cur_pos.x;	m1_pos.y = cur_pos.y;
				}
			has_m1 = has_m2 = true;
			}
		if(!(lines[cur_pos.y]))return false;
		if(cmd == CMD_CURRLEFT && has_m1 && has_m2) TextMark(o, 3);
		if(cur_pos.x > 0) {
			fi = 0.0;					fmt_txt.SetText(0L, lines[cur_pos.y], &fi, &fi, true);
			fi = cur_pos.x;				i = iround(fi);	 fmt_txt.cur_left(&i);
			fi = i;						cur_pos.x = iround(fi);
			}
		else if(cur_pos.y) {
			cur_pos.y--;			cur_pos.x = (int)strlen((char*)lines[cur_pos.y]);
			}
		if(cmd == CMD_SHIFTLEFT){
			m2_pos.x = cur_pos.x;		m2_pos.y = cur_pos.y;
			DoPlot(o);					o->UpdateRect(&rDims, true);
			}
		ShowCursor(o);
		return false;
	case CMD_SHIFTRIGHT:				case CMD_CURRIGHT:
		bFocus = true;
		if (cmd == CMD_SHIFTRIGHT){
			if (!has_m1) {
				m1_pos.x = cur_pos.x;		m1_pos.y = cur_pos.y;
				}
			has_m1 = has_m2 = true;
			}
		else if(!(lines[cur_pos.y]))return false;
		if(cmd == CMD_CURRIGHT && has_m1 && has_m2) TextMark(o, 3);
		if(cur_pos.x >= (int)strlen((char*)lines[cur_pos.y])) {
			if(cur_pos.y < (nlines-1)) {
				cur_pos.y++;		cur_pos.x = 0;
				}
			else if(cur_pos.y == (nlines-1) && lines[cur_pos.y][0]) {
				if(!(lines = (unsigned char**)realloc(lines, (nlines+1)*sizeof(char*))))return false;
				if(!(lines[cur_pos.y+1] = (unsigned char*)malloc(TF_MAXLINE))) return false;
				cur_pos.y++;		cur_pos.x = 0;		nlines++;
				lines[cur_pos.y][0] = 0;
				}
			}
		else {
			fi = 0.0;					fmt_txt.SetText(0L,lines[cur_pos.y], &fi, &fi, true);
			fi = cur_pos.x;				i = iround(fi);			 fmt_txt.cur_right(&i);
			fi = i;						 cur_pos.x = iround(fi);
			}
		if(cmd == CMD_SHIFTRIGHT){
			m2_pos.x = cur_pos.x;		m2_pos.y = cur_pos.y;
			DoPlot(o);					o->UpdateRect(&rDims, true);
			}
		ShowCursor(o);
		return false;
	case CMD_SHIFTUP:						case CMD_CURRUP:
		bFocus = true;
		if (cmd == CMD_SHIFTUP){
			if (!has_m1) {
				m1_pos.x = cur_pos.x;		m1_pos.y = cur_pos.y;
				}
			has_m1 = has_m2 = true;
			}
		else if(cmd == CMD_CURRUP && has_m1 && has_m2) TextMark(o, 3);
		if(cur_pos.y && o) {
			i = ((Cursor.bottom + Cursor.top)>>1)-linc;
			CalcCursorPos(Cursor.left, i, o);
			if(cmd == CMD_SHIFTUP){
				m2_pos.x = cur_pos.x;	m2_pos.y = cur_pos.y;
				DoPlot(o);				o->UpdateRect(&rDims, true);
				}
			ShowCursor(o);
			return true;
			}
		return false;
	case CMD_SHIFTDOWN:					case CMD_CURRDOWN:
		bFocus = true;
		if (cmd == CMD_SHIFTDOWN){
			if (!has_m1) {
				m1_pos.x = cur_pos.x;	m1_pos.y = cur_pos.y;
				}
			has_m1 = has_m2 = true;
			}
		else if(cmd == CMD_CURRDOWN && has_m1 && has_m2) TextMark(o, 3);
		if(cur_pos.y < (nlines-1) && o && lines[cur_pos.y][0]) {
			i = ((Cursor.bottom + Cursor.top)>>1)+linc;
			if(i >= (rDims.bottom-ipad.bottom)) return false;
			CalcCursorPos(Cursor.left, i, o);
			if(cmd == CMD_SHIFTDOWN){
				m2_pos.x = cur_pos.x;	m2_pos.y = cur_pos.y;
				DoPlot(o);				o->UpdateRect(&rDims, true);
				}
			ShowCursor(o);
			return true;
			}
		return false;
	case CMD_POS_FIRST:
		bFocus = true;
		if(has_m1 && has_m2) TextMark(o, 3);
		cur_pos.x = 0;									ShowCursor(o);
		return true;
	case CMD_POS_LAST:
		bFocus = true;
		if(has_m1 && has_m2) TextMark(o, 3);
		cur_pos.x = (int)strlen((char*)lines[cur_pos.y]);
		ShowCursor(o);
		return true;
	case CMD_TEXTTHERE:
		if(IsInRect(rDims, ((MouseEvent *)tmpl)->x, ((MouseEvent *)tmpl)->y)) return true;
		return false;
	case CMD_KILLFOCUS:
		bFocus = false;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if (IsInRect(rDims, mev->x, mev->y)) {
				bFocus = true;
				if (!parent && ClearSym.select(mev->x, mev->y)) {
					Command(CMD_CLEAR, 0L, o);
					return true;
					}
				if ((!(CurrGO) || !has_m2) && o){
					CalcCursorPos(mev->x, mev->y, o);
					if (has_m1 && has_m2) {
						if (o) {
							DoPlot(o);						o->UpdateRect(&rDims, true);
							CurrGO = this;					ShowCursor(o);
							}
						return true;
						}
					else {
						if(parent) o->ShowMark(this, MRK_GODRAW);
						CurrGO = this;						ShowCursor(o);
						return true;
						}
					}
				else if (CurrGO == this && o){
					ShowCursor(o);
					return IsInRect(rDims, mev->x, mev->y);
					}
				}
			else if (bFocus){
				bFocus = false;
				DoPlot(o);							o->UpdateRect(&rDims, true);
				}
			break;
		case MOUSE_LBDOWN:				case MOUSE_MOVE:
			if (mev->Action == MOUSE_LBDOWN) {
				if (has_m1 && has_m2) {
					has_m1 = has_m2 = false;
					if (Notary->IsValidPtr(o)){
						DoPlot(o);		o->UpdateRect(&rDims, true);
						}
					}
				has_m1 = has_m2 = false;
				}
			if(!(mev->StateFlags & 0x1)) return false;
			if(!IsInRect(rDims, mev->x, mev->y))return false;
			if(!CurrGO) CurrGO = this;
			CalcCursorPos(mev->x, mev->y, o);
			if(!has_m1) {
				m1_pos.x = m2_pos.x = cur_pos.x;		m1_pos.y = m2_pos.y = cur_pos.y;
				return has_m1 = true;
				}
			else if(cur_pos.x != m2_pos.x || cur_pos.y != m2_pos.y) {
				m2_pos.x = cur_pos.x;					m2_pos.y = cur_pos.y;
				has_m2 = true;
				if(o)DoPlot(o);
				o->UpdateRect(&rDims, true);
				return true;
				}
			return true;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_TEXTFRAME;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_PASTE:
#ifdef _WINDOWS
		return DoPaste(o);
#else
		TestClipboard(this);
		return true;
#endif
	case CMD_ADDTXT:
#ifndef _WINDOWS
		if (!tmpl) {		//Linux: Clipboard data ready!
			if(CurrDisp) DoPaste(CurrDisp);
			}
#endif
		return true;
	case CMD_MOVE:		case CMD_UNDO_MOVE:		case CMD_REDRAW:
		if (Notary->IsValidPtr(o))o->MrkMode = MRK_NONE;
		if (cmd == CMD_MOVE){
			bModified = true;
			Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			}
		if (cmd == CMD_UNDO_MOVE || cmd == CMD_MOVE){
			pos1.fx += ((lfPOINT*)tmpl)[0].fx;	pos1.fy += ((lfPOINT*)tmpl)[0].fy;
			pos2.fx += ((lfPOINT*)tmpl)[0].fx;	pos2.fy += ((lfPOINT*)tmpl)[0].fy;
			CurrGO = this;
			}
		if (parent){
			if(o && cmd == CMD_REDRAW) DoPlot(o);		//trickle down ?
			if(!o && cmd == CMD_REDRAW) {
				//coming from Undo
				bModified = true;
				if(lines) {
					for(i = 0; i < nlines; i++) if(lines[i]) free(lines[i]);
					free(lines);			lines = 0L;
					}
				HideTextCursor();			cur_pos.x = cur_pos.y = 0;
				parent->Command(CMD_REDRAW, tmpl, o);
				}
			else parent->Command(CMD_REDRAW, tmpl, o);
			}
		return true;
	case CMD_SETSCROLL:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = 0L;
		return true;
	case CMD_SETTEXT:
		if(lines) {
			for(i = 0; i < nlines; i++) if(lines[i]) free(lines[i]);
			free(lines);			lines = 0L;
			}
		if(text) free(text);
		text = 0L;
		if(tmpl && *((unsigned char*)tmpl)) text = rlp_strdup((unsigned char*)tmpl);
		return true;
	case CMD_CONFIG:
		return PropertyDlg();
		}
	return false;
}

void
TextFrame::Track(POINT *p, anyOutput *o, bool)
{
	POINT tpts[5];
	RECT old_rc;
	double dx, dy;

	if(Notary->IsValidPtr(o) && parent){							//dialog frames are not moveable
		o->MrkMode = MRK_NONE;
		dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
		memcpy(&old_rc, &rDims, sizeof(rDims));
		defs.UpdAdd(o, rDims.left, rDims.top);		defs.UpdAdd(o, rDims.right, rDims.bottom);
		tpts[0].x = tpts[1].x = tpts[4].x = o->co2ix(pos1.fx+dx)+p->x;		
		tpts[0].y = tpts[3].y = tpts[4].y = o->co2iy(pos1.fy+dy)+p->y;
		tpts[1].y = tpts[2].y = o->co2iy(pos2.fy+dy)+p->y;
		tpts[2].x = tpts[3].x = o->co2ix(pos2.fx+dx)+p->x;
		UpdateMinMaxRect(&rDims, tpts[0].x, tpts[0].y);
		UpdateMinMaxRect(&rDims, tpts[2].x, tpts[2].y);	
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		o->ShowLine(tpts, 5, 0x00cbcbcb);		//assume grey to be visible
		defs.Idle(CMD_UPDATE);
		}
}

void *
TextFrame::ObjThere(int x, int y)
{
	if(drc) return drc->ObjThere(x, y);
	return 0L;
}

void
TextFrame::text2lines(anyOutput *o)
{
	int i, j, cl, maxlines, maxw;
	double w, h;
	unsigned char tmp_line[TF_MAXLINE];
	bool hasMark = false;

	if(lines) {
		for(i = 0; i < nlines; i++) if(lines[i]) free(lines[i]);
		free(lines);			lines = 0L;
		}
	has_m1 = has_m2 = false;
	nlines = 0;
	if(!text || !text[0]) return;
	maxlines = (rDims.bottom -rDims.top)/linc +1;
	maxw = rDims.right - rDims.left - ipad.left - ipad.right;
	lines = (unsigned char**)calloc(maxlines, sizeof(char*));
	w = h = 0.0;
	for(cl = cpos = 0; cl < maxlines && text[cpos]; cl++) {
		if(!(lines[cl] = (unsigned char*)malloc(TF_MAXLINE))) return;
		for(i = 0; text[cpos] && i < TF_MAXLINE; i++) {
			tmp_line[i] = text[cpos++];				tmp_line[i+1] = 0;
			fmt_txt.SetText(0L, tmp_line, &w, &h, false);
			if(!(fmt_txt.oGetTextExtent(o, &w, &h, i))) break;
			if(tmp_line[i] == '\n'){
				break;													//new line character found
				}
			if(tmp_line[i] < ' ') switch(tmp_line[i]) {
				case 0x01:
					if(c_char == 0 && text[cpos]) c_char = text[cpos++];	//cursor at end of line
					hasMark = true;				break;
				case 0x02:	case 0x03:
					hasMark = true;				break;
				}
			else if(w >= maxw){
				for(j = i; j > (i>>1); j--) {
					if(tmp_line[j] == ' ' || tmp_line[j] == '-' || tmp_line[j] < ' ') break;
					}
				if(j == (i>>1)) {
					cpos--;							tmp_line[i] = 0;
					}
				else {
					for(tmp_line[j+1] = 0; j < i; j++, cpos--);
					}
				break;
				}
			}
		if(i || tmp_line[i] == '\n') rlp_strcpy((char*)lines[cl], TF_MAXLINE, tmp_line);
		else lines[cl][0] = 0;
		}
	nlines = cl;
	if(hasMark)procTokens();
}

void
TextFrame::lines2text()
{
	int i;

	if(text) free(text);
	cpos = 0;
	text = (unsigned char*)malloc(csize = 1000);
	for(i = 0; i < nlines; i++) {
		if(lines[i] && lines[i][0]) add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i], 0);
		}
}

void
TextFrame::AddChar(anyOutput *o, int chr)
{
	int i, j, c;
	double fj;
	double h, w;
	char txt1[60];

	if(cur_pos.y >= nlines) return;
	if(!lines || !lines[cur_pos.y]) return;
	c = (chr & 0xfff);
	if(c == '\r') c = '\n';
	if(has_m1 && has_m2){
		TmpTxt[0] = c;		TmpTxt[1] = 0;
		if(c == 8) ReplMark(o, (char*)"");
		else if(c >= 32 || c == '\n') ReplMark(o, TmpTxt);
		return;
		}
	else if(!text) {
		lines2text();
		Undo.TextBuffer(this, &csize, &cpos, &text, 0L, o);
		}
	has_m1 = has_m2 = false;
	if (c == '\n') {
		lines2text();		Undo.TextBuffer(this, &csize, &cpos, &text, 0L, o);
		while (nlines > 0 && (!lines[nlines - 1] || *lines[nlines - 1] < (unsigned char)' ')) nlines--;
		lines = (unsigned char **)realloc(lines, (nlines + 1) * sizeof(unsigned char*));
		for (i = nlines-1; i >= cur_pos.y; i--) lines[i + 1] = lines[i];
		lines[i+1] = (unsigned char*)calloc(TF_MAXLINE, sizeof(unsigned char));
		j = rlp_strcpy(lines[i + 1], cur_pos.x +1, lines[i + 2]);
		lines[i + 1][j++] = (unsigned char)'\n';		lines[i + 1][j] = 0;
		rlp_strcpy(lines[i + 2], TF_MAXLINE, lines[i + 2] + cur_pos.x);
		nlines++;		cur_pos.x = 0;		cur_pos.y++;
		DoPlot(o);					o->UpdateRect(&rDims, false);
		ShowCursor(o);				UndoFlags = 0;
		return;
		}
	i = j = rlp_strlen(lines[cur_pos.y]) + 1;
	if (c >= 32) {
		lines2text();		Undo.TextBuffer(this, &csize, &cpos, &text, UndoFlags, o);
		UndoFlags |= UNDO_CONTINUE;
		if(c > 254) {
#ifdef USE_WIN_SECURE
			w = sprintf_s(txt1, 10, "&#%x;", c);
#else
			w = sprintf(txt1, "&#%x;", c);
#endif
			for(j = j+iround(w); j>0; j--) {
				lines[cur_pos.y][j] = lines[cur_pos.y][j-iround(w)];
				if((j-w) == cur_pos.x){
					for(i = 0; i < w; i++) lines[cur_pos.y][j-iround(w)+i] = txt1[i];
					j = 0;					cur_pos.x += iround(w);
					}
				}
			}
		else while(j) {
			lines[cur_pos.y][j] = lines[cur_pos.y][j-1];	j--;
			if(j == cur_pos.x){
				lines[cur_pos.y][j] = c;					j = 0;
				cur_pos.x++;
				}
			}
		fj = j;
		fmt_txt.SetText(0L, lines[cur_pos.y], &fj, &fj, true);
		fmt_txt.oGetTextExtent(o, &w, &h, i);
		}
	else if(c == 8 && (cur_pos.x || cur_pos.y)) {			//Backspace
		if(!cur_pos.x) Command(CMD_CURRLEFT, 0L, o);
		Command(CMD_CURRLEFT, 0L, o);						DelChar(o);
		return;
		}
	else return;
	DoPlot(o);		o->UpdateRect(&rDims, true);			ShowCursor(o);
}

void
TextFrame::DelChar(anyOutput *o)
{
	int i, cb, x;

	if(has_m1 && has_m2){
		ReplMark(o, (char*)"");	return;
		}
	if(lines[cur_pos.y][cur_pos.x]) {
		lines2text();
		Undo.TextBuffer(this, &csize, &cpos, &text, 0L, o);
		cb = x = cur_pos.x;			fmt_txt.cur_right(&x);
		cb = x - cb;	if(cb < 1) cb = 1;
		for(i = cur_pos.x; lines[cur_pos.y][i]; i++) {
			if(!(lines[cur_pos.y][i] = lines[cur_pos.y][i+cb])) break;
			}
		c_char = lines[cur_pos.y][cur_pos.x];			lines[cur_pos.y][cur_pos.x] = 0x01;
		if(!c_char) lines[cur_pos.y][cur_pos.x+1] = 0x00; 
		lines2text();			text2lines(o);
		DoPlot(o);				o->UpdateRect(&rDims, true);
		ShowCursor(o);			return;
		}
	else if(cur_pos.y < (nlines-1)){
		cur_pos.y++;			cur_pos.x = 0;
		DelChar(o);				return;
		}
}

void
TextFrame::ReplMark(anyOutput *o, char *ntext)
{
	int i, j;

	if(!has_m1 || !has_m2 || !o || !ntext) return;
	lines2text();
	Undo.TextBuffer(this, &csize, &cpos, &text, 0L, o);
	for(i = cpos = 0; i < nlines && i < m1_cpos.y; i++) {
		if(lines[i] && lines[i][0]) add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i], 0);
		}
	if(m1_cpos.x)add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i], m1_cpos.x);
	add_to_buff((char**)&text, &cpos, &csize, ntext, 0);			j = cpos;
	if(m1_cpos.y == m2_cpos.y)add_to_buff((char**)&text, &cpos, &csize, (char*)(lines[i]+m2_cpos.x), 0);
	for( ; i < nlines && i < m2_cpos.y; i++);
	if(i == m2_cpos.y && m2_cpos.y > m1_cpos.y)add_to_buff((char**)&text, &cpos, &csize, (char*)(lines[i]+m2_cpos.x), 0);
	for(i++; i < nlines; i++) {
		if(lines[i] && lines[i][0]) add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i], 0);
		}
	if(tm_rec)free(tm_rec);
	tm_rec = 0L;								has_m1 = has_m2 = false;
	if(text[j]) {
		c_char = text[j];						text[j] = 0x01;
		}
	else if(i < (nlines-1) && lines[i+1] && lines[i+1][0]) {
		c_char = lines[i+1][0];					lines[i+1][0] = 0x01;
		}
	else if(j == cpos){
		c_char = 0;		text[cpos++] = 0x01;		text[cpos] = 0;
		}
	else cur_pos.x = cur_pos.y = 0;
	text2lines(o);		DoPlot(o);	o->UpdateRect(&rDims, true);
	ShowCursor(o);
}

void
TextFrame::procTokens()
{
	int i, j;

	for(i = 0; i < nlines; i++) {
		if(lines[i] && lines[i][0]){
			for(j = 0; lines[i][j]; j++) switch(lines[i][j]) {
				case 0x01:
					cur_pos.y = i;				cur_pos.x = j;
					lines[i][j] = c_char;		c_char = '?';
					break;
				case 0x02:
					m1_pos.y = i;				m1_pos.x = j;
					if(m1_char == 0x01) {
						lines[i][j] = c_char;	c_char = '?';
						cur_pos.y = i;			cur_pos.x = j;
						}
					else lines[i][j] = m1_char;
					has_m1 = true;				m1_char = '?';
					break;
				case 0x03:
					m2_pos.y = i;				m2_pos.x = j;
					if(m2_char == 0x01) {
						lines[i][j] = c_char;	c_char = '?';
						cur_pos.y = i;			cur_pos.x = j;
						}
					else lines[i][j] = m2_char;
					has_m2 = true;				m2_char = '?';
					break;
					}
			}
		}
}

bool
TextFrame::DoPaste(anyOutput *o)
{
	int i, j, k;
	char *ntxt;
	unsigned char *ptxt;

	if((ptxt = PasteText()) && ptxt[0]) {
		lines2text();
		Undo.TextBuffer(this, &csize, &cpos, &text, 0L, o);
		if(!(ntxt = (char*)malloc(strlen((char*)ptxt)+1))) return false;
		for(i = j = cpos = 0; ptxt[i]; i++) {
			if(ptxt[i] >= ' ' || ptxt[i] == '\n') ntxt[j++] = ptxt[i];
			else if(ptxt[i] == 9)ntxt[j++] = ' ';		//convert tab->space 
			}
		ntxt[j] = 0;
		if(!ntxt[0]) {
			free(ntxt);				return false;
			}
		if(has_m1 && has_m2) {
			ReplMark(o, ntxt);		free(ntxt);
			return true;
			}
		m1_char = ntxt[0];			ntxt[0] = 0x02;
		ntxt[j] = 0;				if(text) free(text);
		if(!(text = (unsigned char*)malloc(csize = 1000)))return false;
		for(i = k = 0, text[0] = 0; i < nlines; i++) {
			if(lines[i] && lines[i][0]){
				if(i == cur_pos.y) {
					if(cur_pos.x) add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i], cur_pos.x);
					add_to_buff((char**)&text, &cpos, &csize, ntxt, 0);	k = cpos;
					add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i]+cur_pos.x, 0);
					free(ntxt);		ntxt = 0L;
					}
				else add_to_buff((char**)&text, &cpos, &csize, (char*)lines[i], 0);
				}
			}
		if(ntxt) {
			add_to_buff((char**)&text, &cpos, &csize, ntxt, 0);			k = cpos;
			}
		m2_char = 0x01;
		if(text[k]) {
			c_char = text[k];						text[k] = 0x03;
			}
		else if(i < (nlines-1) && lines[i+1] && lines[i+1][0]) {
			c_char = lines[i+1][0];					lines[i+1][0] = 0x03;
			}
		else if(k == cpos){
			c_char = 0;		text[cpos++] = 0x03;	text[cpos] = 0;
			}
		text2lines(o);						DoPlot(o);
		o->UpdateRect(&rDims, true);
		ShowCursor(o);						if(ntxt) free(ntxt);
		}
	return false;
}
void
TextFrame::TextMark(anyOutput *o, int mode)
{
	int i, j;
	double w, h;
	LineDEF ld;
	FillDEF fd;

	if(m1_pos.y > m2_pos.y) {
		m1_cpos.y = m2_pos.y;				m1_cpos.x = m2_pos.x;
		m2_cpos.y = m1_pos.y;				m2_cpos.x = m1_pos.x;
		}
	else if(m1_pos.y == m2_pos.y  && m1_pos.x > m2_pos.x) {
		m1_cpos.x = m2_pos.x;				m2_cpos.x = m1_pos.x;
		m1_cpos.y = m2_cpos.y = m1_pos.y;
		}
	else {
		m1_cpos.y = m1_pos.y;				m1_cpos.x = m1_pos.x;
		m2_cpos.y = m2_pos.y;				m2_cpos.x = m2_pos.x;
		}
	if(!has_m1 || !has_m2 || !o) return;
	if(m1_pos.y == m2_pos.y && m1_pos.x == m2_pos.x) return;
	if(mode == 1){							//create background mark
		if(tm_rec)free(tm_rec);
		tm_rec = 0L;
		if(((tm_c = m2_cpos.y - m1_cpos.y +1)<1)) return;
		if(!(tm_rec = (RECT*)malloc(tm_c * sizeof(RECT))))return;
		w = 0.0;
		for(i = 0, j = m1_cpos.y; j <= m2_cpos.y; i++, j++) {
			h = TextDef.iSize;
			if(j == m1_cpos.y) {
				fmt_txt.SetText(0L, lines[j], 0L, 0L, true);
				if(m1_cpos.x) fmt_txt.oGetTextExtent(o, &w, &h, m1_cpos.x);
				else w = 0;
				tm_rec[0].left = iround(w) + rDims.left + ipad.left + 1;			
				fmt_txt.SetText(0L, (lines[j]+m1_cpos.x), 0L, 0L, false);
				fmt_txt.oGetTextExtent(o, &w, &h, 0);
				tm_rec[0].right = tm_rec[0].left + iround(w);
				}
			if(j == m2_cpos.y) {
				fmt_txt.SetText(0L, lines[j], 0L, 0L, false);
				if(m2_cpos.x) fmt_txt.oGetTextExtent(o, &w, &h, m2_cpos.x);
				else w = 0;
				tm_rec[i].right = iround(w) + rDims.left + ipad.left - 1;			
				if(m2_cpos.y > m1_cpos.y) {
					tm_rec[i].left = rDims.left + ipad.left;
					}
				}
			else if(j < m2_cpos.y && j > m1_cpos.y) {
				tm_rec[i].left = rDims.left + ipad.left;
				tm_rec[i].top = j * linc + rDims.top + ipad.top;
				fmt_txt.SetText(0L, lines[j], 0L, 0L, false);
				fmt_txt.oGetTextExtent(o, &w, &h, 0);
				tm_rec[i].right = iround(w) + rDims.left + ipad.left;			
				}
			tm_rec[i].top = j * linc + rDims.top + ipad.top;
			tm_rec[i].bottom = tm_rec[i].top + linc;
			}
		ld.color = 0x0000ffff;				ld.patlength = 1.0;
		ld.pattern = 0x0L;					ld.width = 0;
		fd.color = fd.color2 = ld.color;	fd.hatch = 0L;
		fd.scale = 1.0;						fd.type = 0;
		o->SetLine(&ld);					o->SetFill(&fd);
		for(i = 0; i < tm_c; i++) {
			o->oRectangle(tm_rec[i].left, tm_rec[i].top, tm_rec[i].right, tm_rec[i].bottom, 0L);
			}
		}
	if(mode == 2){							//invert rectangles
		if(tm_rec) for(i = 0; i < tm_c; i++) {
			o->CopyBitmap(tm_rec[i].left, tm_rec[i].top, o, tm_rec[i].left, tm_rec[i].top,
				tm_rec[i].right - tm_rec[i].left, tm_rec[i].bottom - tm_rec[i].top, true);
			}
		}
	if(mode == 3){							//clear mark
		if(tm_rec)free(tm_rec);
		tm_rec = 0L;
		tm_c = 0;						has_m1 = has_m2 = false;
		DoPlot(o);						o->UpdateRect(&rDims, true);
		}
}

bool
TextFrame::CopyText(anyOutput *o, bool b_cut)
{
	long i, csize, pos = 0;
	char *ntxt;

	if(!lines || !lines[0][0] || !o) return false;
	if(!has_m1 || !has_m2) {
		m1_pos.x = m1_pos.y = 0;		m2_pos.y = nlines-1;
		if(lines[nlines-1]) m2_pos.x = (int)strlen((char*)lines[nlines-1]);
		else m2_pos.x = 0;
		has_m1 = has_m2 = true;
		DoPlot(o);						o->UpdateRect(&rDims, true);
		return CopyText(o, false);
		}
	ShowCopyMark(o, &rDims, 1);
	if(!(ntxt = (char*)malloc(csize = 1000))) return false;
	if(m1_cpos.y == m2_cpos.y) {
		add_to_buff(&ntxt, &pos, &csize, (char*)(lines[m1_cpos.y]) + m1_cpos.x, m2_cpos.x - m1_cpos.x);
		::CopyText(ntxt, pos, CF_TEXT);			free(ntxt);
		}
	else if(m1_cpos.y < m2_cpos.y){
		for(i = m1_cpos.y; i < m2_cpos.y; i++) {
			add_to_buff(&ntxt, &pos, &csize, (char*)(lines[i]), 0);
			}
		add_to_buff(&ntxt, &pos, &csize, (char*)(lines[i]), m2_pos.x);
		::CopyText(ntxt, pos, CF_TEXT);			free(ntxt);
		}
	else {
		free(ntxt);						return false;
		}
	if(b_cut) ReplMark(o, (char*)"");
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The segment object is either a pie slice or a ring segment
//
segment::segment(GraphObj *par, DataObj *d, lfPOINT *c, double r1, double r2,
				 double a1, double a2, char *nam):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	segFill.hatch = &segFillLine;
	segFill.color = 0x00c0c0c0L;
	fCent.fx = c->fx;		fCent.fy = c->fy;
	radius1 = r1;			radius2 = r2;
	angle1 = a1;			angle2 = a2;
	Id = GO_SEGMENT;		bModified = false;
	if (nam) name = rlp_strdup(nam);
}

segment::segment(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

segment::~segment()
{
	if(pts && nPts) free(pts);
	pts = 0L;
	if(bModified) Undo.InvalidGO(this);
	if(mo) DelBitmapClass(mo);
	mo = 0L;
}
	
bool 
segment::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_XPOS:		fCent.fx = value;		break;
	case SIZE_YPOS:		fCent.fy = value;		break;
	case SIZE_RADIUS1:	radius1 = value;		break;
	case SIZE_RADIUS2:	radius2 = value;		break;
	case SIZE_ANGLE1:	angle1 = value;			break;
	case SIZE_ANGLE2:	angle2 = value;			break;
	default:			return false;
		}
	return true;
}

void
segment::DoPlot(anyOutput *o)
{
	double dsize, dpt, frx1, frx2, dtmp, da, dda, sia, csia;
	int i, n, npt = 12;
	POINT np, of, cp,  *tmppts;

	if(!o || angle1 == angle2) return;
	if(mo) DelBitmapClass(mo);
	mo = 0L;
	dsize = angle1 > angle2 ? angle1 -angle2 : (angle1+360.0)-angle2;
	dpt = dsize*0.01745329252;		//degrees to rad
	frx1 = (double)o->un2fix(radius1);
	frx2 = (double)o->un2fix(radius2);
	dtmp = frx1*dpt;
	npt += (int)(dtmp < dsize ? dtmp : dsize);
	dtmp = frx2*dpt;
	npt += (int)(dtmp < dsize ? dtmp : dsize);
	if(!(pts = (POINT*)malloc(npt*sizeof(POINT))))return;
	nPts = 0;
	n = (dtmp < dsize) ? (int)dtmp : (int)dsize;
	while (n<2) n++;
	da = angle1*0.01745329252;
	dda = (dsize*0.01745329252)/(double)n;
	sia = sin(0.5*((angle2 < angle1 ? angle1 : angle1 +360) + angle2) * 0.01745329252);
	csia = cos(0.5*((angle2 < angle1 ? angle1 : angle1 +360) + angle2) * 0.01745329252);
	of.x = o->un2ix(shift * csia);
	of.y = - (o->un2iy(shift * sia));
	cp.x = o->co2ix(fCent.fx + (parent ? parent->GetSize(SIZE_GRECT_LEFT): 0.0));
	cp.y = o->co2iy(fCent.fy + (parent ? parent->GetSize(SIZE_GRECT_TOP): 0.0));
	for(i = 0; i < n; i++) {
		sia = sin(da);					csia = cos(da);
		np.x = of.x + cp.x;				np.y = of.y + cp.y;
		np.x += o->un2ix(csia*radius2);	np.y -= o->un2iy(sia*radius2);
		AddToPolygon(&nPts, pts, &np);
		da -= dda;
		}
	sia = sin(angle2 *0.01745329252);
	csia = cos(angle2 *0.01745329252);
	np.x = of.x + cp.x;					np.y = of.y + cp.y;
	np.x += o->un2ix(csia*radius2);		np.y -= o->un2iy(sia*radius2);
	AddToPolygon(&nPts, pts, &np);
	dtmp = frx1*dpt;
	n = dtmp < dsize ? (int)dtmp : (int)dsize;
	da = angle2*0.01745329252;
	if(n>1)dda = (dsize*0.01745329252)/(double)n;
	else dda = 0.0;
	for(i = 0; i < n; i++) {
		sia = sin(da);					csia = cos(da);
		np.x = of.x + cp.x;				np.y = of.y + cp.y;
		np.x += o->un2ix(csia*radius1);	np.y -= o->un2iy(sia*radius1);
		AddToPolygon(&nPts, pts, &np);
		da += dda;
		}
	sia = sin(angle1 *0.01745329252);	csia = cos(angle1 *0.01745329252);
	np.x = of.x + cp.x;					np.y = of.y + cp.y;
	np.x += o->un2ix(csia*radius1);		np.y -= o->un2iy(sia*radius1);
	AddToPolygon(&nPts, pts, &np);
	if(nPts <3) return;
	AddToPolygon(&nPts, pts, &pts[0]);	//close polygon
	o->SetLine(&segLine);				o->SetFill(&segFill);
	o->oPolygon(pts, nPts);
	tmppts = (POINT*)realloc(pts, nPts *sizeof(POINT));
	if(tmppts) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < nPts; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 3*o->un2ix(segLine.width);		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i+6);
}

void
segment::DoMark(anyOutput *o, bool mark)
{
	POINT *tpts;
	long i;

	if (mark && o && (tpts = (POINT*)malloc(nPts*sizeof(POINT)))){
		if(mo) DelBitmapClass(mo);
		mo = 0L;
		memcpy(&mrc, &rDims, sizeof(RECT));
		mo = GetRectBitmap(&mrc, o);
		for (i = 0; i < nPts; i++) {
			tpts[i].x = pts[i].x;			tpts[i].y = pts[i].y;
			}
		InvertLine(tpts, nPts, &segLine, &rDims, o, mark);
		free(tpts);
		}
	else if (mo) {
		RestoreRectBitmap(&mo, &mrc, o);
		DelBitmapClass(mo);			mo = 0L;
		}
}

bool
segment::Command(int cmd, void *tmpl, anyOutput *o)
{
	FillDEF *TmpFill;
	LegItem *leg;

	switch (cmd) {
	case CMD_SCALE:
		fCent.fx *= ((scaleINFO*)tmpl)->sx.fy;			fCent.fy *= ((scaleINFO*)tmpl)->sx.fy;
		radius1 *= ((scaleINFO*)tmpl)->sx.fy;			radius2 *= ((scaleINFO*)tmpl)->sx.fy;
		segLine.width *= ((scaleINFO*)tmpl)->sx.fy;		segLine.patlength *= ((scaleINFO*)tmpl)->sx.fy;
		segFillLine.width *= ((scaleINFO*)tmpl)->sx.fy;	segFillLine.patlength *= ((scaleINFO*)tmpl)->sx.fy;
		segFill.scale *= ((scaleINFO*)tmpl)->sx.fy;		shift *= ((scaleINFO*)tmpl)->sx.fy;
		return true;
	case CMD_LEGEND:
		if(tmpl) {
			leg = new LegItem(this, data, 0L, &segLine, &segFill, name);
			if(!((Legend*)tmpl)->Command(CMD_DROP_OBJECT, leg, o)) DeleteGO(leg);
			}
		return true;
	case CMD_MOUSE_EVENT:
		switch (((MouseEvent*)tmpl)->Action) {
		case MOUSE_LBUP:
			if(ObjThere(((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) 
				return o->ShowMark(CurrGO=this, MRK_GODRAW);
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_SEGMENT;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_REDRAW:
		return parent ? parent->Command(cmd, tmpl, o) : false;
	case CMD_SEG_FILL:
		TmpFill = (FillDEF *)tmpl;
		if(TmpFill) {
			segFill.type = TmpFill->type;			segFill.color = TmpFill->color;
			segFill.scale = TmpFill->scale;
			if(TmpFill->hatch) memcpy(&segFillLine, TmpFill->hatch, sizeof(LineDEF));
			}
		return true;
	case CMD_SEG_LINE:
		if(tmpl) memcpy(&segLine, tmpl, sizeof(LineDEF));
		return true;
	case CMD_SEG_MOVEABLE:
		if(tmpl) moveable = *((int*)tmpl);
		return true;
	case CMD_SHIFT_OUT:
		if(tmpl) shift = *((double*)tmpl);
		return true;
	case CMD_MOVE:								case CMD_UNDO_MOVE:
		if (Notary->IsValidPtr(o))o->MrkMode = MRK_NONE;
		if (cmd == CMD_UNDO_MOVE) {
			bModified = true;					Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			}
		fCent.fx += ((lfPOINT*)tmpl)[0].fx;		fCent.fy += ((lfPOINT*)tmpl)[0].fy;
		if(parent)parent->Command(CMD_REDRAW, 0L, o);
		return true;
		}
	return false;
}

void *
segment::ObjThere(int x, int y)
{
	bool bFound = false;
	POINT p1;

	if(IsInRect(rDims, (p1.x = x), (p1.y = y))) {
		bFound = IsInPolygon(&p1, pts, nPts, 3.0);
		if(bFound || IsCloseToPL(p1, pts, nPts, 3.0)) return this;
		}
	return 0L;
}

void
segment::Track(POINT *p, anyOutput *o, bool)
{
	POINT *tpts;
	RECT old_rc;
	long i;

	if(o && (tpts = (POINT*)malloc(nPts*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		InvalidateOutput(o);
		for(i = 0; i < nPts; i++) {
			tpts[i].x = pts[i].x+p->x;			tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		o->ShowLine(tpts, nPts, segLine.color);
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the polyline object
//
polyline::polyline(GraphObj *par, DataObj *d, lfPOINT *fpts, int cpts):
	GraphObj(par, d)
{
	double dx = 0.0, dy = 0.0;
	int i;

	FileIO(INIT_VARS);
	if(parent){
		dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
		}
	if(cpts && (Values = (lfPOINT*)malloc((nPoints = cpts)* sizeof(lfPOINT)))){
		memcpy(Values, fpts, cpts*sizeof(lfPOINT));
		for(i = 0; i < cpts; i++) {
			Values[i].fx -= dx;		Values[i].fy -= dy;
			}
		}
	Id = GO_POLYLINE;
	name = rlp_strdup((char*)"polyline");
	bModified = false;
}

polyline::polyline(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

polyline::~polyline()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(this == CurrGO) CurrGO = 0L;
	if(bModified) Undo.InvalidGO(this);
}

double
polyline::GetSize(int select)
{
	int i;

	if(select >= SIZE_XPOS && select <=  SIZE_XPOS_LAST){
		if((i = select-SIZE_XPOS) >=0 && i <= 200)
			return i < nPoints ? Values[i].fx : Values[nPoints-2].fx;
		}
	if(select >= SIZE_YPOS && select <= SIZE_YPOS_LAST){
		if((i = select-SIZE_YPOS) >=0 && i <= 200)
			return i < nPoints ? Values[i].fy : Values[nPoints-2].fy;
		}
	return parent ? parent->GetSize(select) : 0.0;
}

DWORD
polyline::GetColor(int)
{
	return pgLine.color;
}

bool
polyline::SetSize(int select, double value)
{
	int i;

	if((select & 0xfff) >= SIZE_XPOS && (select & 0xfff) <=  SIZE_XPOS_LAST){
		if((i = select-SIZE_XPOS) >=0 && i <= 200 && i < (int)nPoints){
			Values[i].fx = value;
			return true;
			}
		}
	if((select & 0xfff) >= SIZE_YPOS && (select & 0xfff) <= SIZE_YPOS_LAST){
		if((i = select-SIZE_YPOS) >=0 && i <= 200 && i < (int)nPoints){
			Values[i].fy = value;
			return true;
			}
		}
	if (select == SIZE_DATA_LINE) pgLine.width = value;
	return false;
}

void
polyline::DoPlot(anyOutput *o)
{
	POINT np, *tmppts;
	double dx, dy;
	int i;

	if(hidden || !Values || !nPoints || !o || !parent) return;
	if(pts) free(pts);
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	if(!(pts = (POINT*)malloc((nPoints+2)*sizeof(POINT))))return;
	for(i = nPts = 0; i < nPoints; i++){
		np.x = o->co2ix(Values[i].fx + dx);		np.y = o->co2iy(Values[i].fy + dy);
		AddToPolygon(&nPts, pts, &np);
		}
	if (np.x != pts[nPts - 1].x || np.y != pts[nPts - 1].y) {
		pts[nPts].x = np.x;			pts[nPts].y = np.y;
		nPts++;
		}
	if(type == 1) AddToPolygon(&nPts, pts, &pts[0]);	//close polygon
	tmppts = (POINT*)realloc(pts, nPts *sizeof(POINT));
	if (tmppts) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < nPts; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 3*o->un2ix(pgLine.width)+3;		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i);
	if(this == CurrGO) o->ShowMark(this, MRK_GODRAW);
	else switch(type) {
	case 0:				//line
		o->SetLine(&pgLine);							o->oPolyline(pts, nPts);
		break;
	case 1:				//polygon
		o->SetLine(&pgLine);	o->SetFill(&pgFill);	o->oPolygon(pts, nPts);
		break;
		}
}

void
polyline::DoMark(anyOutput *o, bool mark)
{
	RECT upd;

	memcpy(&upd, &rDims, sizeof(RECT));
	IncrementMinMaxRect(&upd, 6 + o->un2ix(pgLine.width)*4);
	if(mark) {
		o->SetLine(&pgLine);
		if(nPoints < 200){
			if(pts && nPts) switch(type) {
			case 0:				//line
				o->oPolyline(pts, nPts);
				break;
			case 1:				//polygon
				o->SetFill(&pgFill);		o->oPolygon(pts, nPts);
				break;
				}
			ShowPoints(o);
			}
		else InvertLine(pts, nPts, &pgLine, &upd, o, true);
		defs.UpdRect(o, &upd);
 		}
	else {
		if(parent)	parent->DoPlot(o);
		}
}

bool
polyline::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;
	int i;

	switch (cmd) {
	case CMD_MRK_DIRTY:					case CMD_FLUSH:
		if (cmd == CMD_MRK_DIRTY){		//issued by Undo
			CurrGO = this;				bModified = true;
			}
		if(pHandles) {
			for (i = 0; i < nPoints; i++) {
				if (pHandles[i]) delete(pHandles[i]);
				}
			free(pHandles);		pHandles = 0L;
			}
		if(cmd == CMD_FLUSH && Values && nPoints){
			free(Values);
			Values = 0L;	nPoints = 0;
			}
		if(pts && nPts) free(pts);
		pts = 0L;		nPts = 0;
		return true;
	case CMD_CONFIG:
		return PropertyDlg();
	case CMD_SCALE:
		if(Values) for(i = 0; i < nPoints; i++){
			Values[i].fx = ((scaleINFO*)tmpl)->sx.fx + Values[i].fx * ((scaleINFO*)tmpl)->sx.fy;
			Values[i].fy = ((scaleINFO*)tmpl)->sy.fx + Values[i].fy * ((scaleINFO*)tmpl)->sy.fy;
			}
		pgLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;		pgLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		pgFillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;	pgFillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		pgFill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!ObjThere(p1.x= mev->x, p1.y=mev->y)|| CurrGO || !o || nPoints <2)return false;
			return o->ShowMark(CurrGO=this, MRK_GODRAW);
			}
		return false;
	case CMD_DELOBJ:
		if(pHandles && tmpl && tmpl == (void*)CurrHandle) {
			for(i = 0; i < nPoints; i++) if(pHandles[i] == CurrHandle) {
				Undo.DataMem(this, (void**)&Values, nPoints * sizeof(lfPOINT), &nPoints, 0L);
				for( ; i < nPoints-1; i++) {
					Values[i].fx = Values[i+1].fx;	Values[i].fy = Values[i+1].fy;
					}
				nPoints--;
				if(pHandles[nPoints])delete(pHandles[nPoints]);
				pHandles[nPoints] = 0L;				CurrHandle = 0L;
				CurrGO = this;						bModified = true;
				return true;
				}
			}
		return false;
	case CMD_SAVEPOS:
		if(tmpl && Values) {
			bModified = true;
			i = *(int*)tmpl;
			if(i >= 0 && i < nPoints) Undo.SaveLFP(this, Values + i, 0L);
			}
		return true;
	case CMD_SET_DATAOBJ:
		Id = type == 1 ? GO_POLYGON : GO_POLYLINE;
		if(tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_MOVE:				case CMD_UNDO_MOVE:
		if (Notary->IsValidPtr(o)) o->MrkMode = MRK_NONE;
		if (cmd == CMD_MOVE) {
			bModified = true;
			Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			}
		for(i = 0; i < nPoints; i++) {
			Values[i].fx += ((lfPOINT*)tmpl)[0].fx;
			Values[i].fy += ((lfPOINT*)tmpl)[0].fy;
			}
		if (Notary->IsValidPtr(o)) {
//			o->StartPage();		parent->DoPlot(o);		o->EndPage();
			parent->Command(CMD_REDRAW, 0L, 0L);
			CurrGO = this;
			}
		return true;
		}
	return false;
}

void * 
polyline::ObjThere(int x, int y)
{
	bool bFound = false;
	POINT p1;
	int i;
	void *ret;

	if(IsInRect(rDims, (p1.x = x), (p1.y = y))) {
		if(CurrGO == this && pHandles) for(i = nPoints-1; i >= 0; i--) 
			if((pHandles[i]) && (ret = pHandles[i]->ObjThere(x, y))) return ret;
		if(type == 1) bFound = IsInPolygon(&p1, pts, nPts, 3.0);
		if(bFound || IsCloseToPL(p1,pts,nPts, 3.0)) return this;
		}
	return 0L;
}

void
polyline::Track(POINT *p, anyOutput *o, bool)
{
	POINT *tpts;
	RECT old_rc;
	int i;

	if (Notary->IsValidPtr(o) && (tpts = (POINT*)malloc(nPts*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		InvalidateOutput(o);
		for(i = 0; i < nPts; i++) {
			tpts[i].x = pts[i].x+p->x;
			tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		o->ShowLine(tpts, nPts, pgLine.color);
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

void
polyline::ShowPoints(anyOutput *o)
{
	int i;
	double dx, dy;
	POINT hpts[3];
	LineDEF gl = {0.0, 1.0, 0x00c0c0c0, 0};

	if(nPoints >= 200 || !o) return;
	if(!pHandles && (pHandles = (dragHandle**)calloc(nPoints+4, sizeof(dragHandle*)))){
		for(i = 0; i < nPoints; i++) pHandles[i] = new dragHandle(this, DH_DATA+i);
		}
	if(!pHandles) return;
	if(Id == GO_BEZIER && parent && nPoints > 3) {
		dx = parent->GetSize(SIZE_GRECT_LEFT);			dy = parent->GetSize(SIZE_GRECT_TOP);
		o->SetLine(&gl);
		hpts[0].x = o->co2ix(Values[0].fx+dx);			hpts[0].y = o->co2iy(Values[0].fy+dy);
		hpts[1].x = o->co2ix(Values[1].fx+dx);			hpts[1].y = o->co2iy(Values[1].fy+dy);
		o->oPolyline(hpts, 2);
		for(i = 3; i < (nPoints-2); i += 3) {
			hpts[0].x = o->co2ix(Values[i-1].fx+dx);	hpts[0].y = o->co2iy(Values[i-1].fy+dy);
			hpts[1].x = o->co2ix(Values[i].fx+dx);		hpts[1].y = o->co2iy(Values[i].fy+dy);
			hpts[2].x = o->co2ix(Values[i+1].fx+dx);	hpts[2].y = o->co2iy(Values[i+1].fy+dy);
			o->oPolyline(hpts, 3);
			}
		hpts[0].x = o->co2ix(Values[i-1].fx+dx);		hpts[0].y = o->co2iy(Values[i-1].fy+dy);
		hpts[1].x = o->co2ix(Values[i].fx+dx);			hpts[1].y = o->co2iy(Values[i].fy+dy);
		o->oPolyline(hpts, 2);
		}
	for(i = 0; i < nPoints; i++) if(pHandles[i]) pHandles[i]->DoPlot(o);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Beziers are based on the polyline object
Bezier::Bezier(GraphObj *par, DataObj *d, lfPOINT *fpts, int cpts, int mode, double res):
	polyline(par, d, 0L, 0)
{
	double dx, dy, merr;
	int i;

	type = mode;
	Id = GO_BEZIER;

	if(!parent) return;
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	if(type == 0 && (Values = (lfPOINT*)malloc(4 * cpts * sizeof(lfPOINT)))) {
		merr = 0.01 * res / Units[defs.dUnits].convert;
		FitCurve(fpts, cpts, merr);
		Values[nPoints].fx = fpts[cpts-1].fx;	Values[nPoints].fy = fpts[cpts-1].fy;
		for(i = 0; i < nPoints; i++) {
			Values[i].fx -= dx;		Values[i].fy -= dy;
			}
		nPoints ++;
		}
	if (name) free(name);
	name = rlp_strdup((char*)"curve");
	return;
}

Bezier::Bezier(int src):polyline(0L, 0L, 0L, 0)
{
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

void
Bezier::DoPlot(anyOutput *o)
{
	POINT *tmppts;
	double dx, dy;
	int i;

	if(!Values || !nPoints || !o || !parent) return;
	if(pts) free(pts);
	pts = 0L;
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	tmppts = (POINT*)malloc((nPoints + 2)*sizeof(POINT));
	if (!tmppts)return;
	for(i = 0; i < nPoints; i++){
		tmppts[i].x = o->co2ix(Values[i].fx + dx);	tmppts[i].y = o->co2iy(Values[i].fy + dy);
		}
	rDims.left = rDims.right = tmppts[0].x;		rDims.top = rDims.bottom = tmppts[0].y;
	for(i = 1; i < nPoints; i++) {
		if(tmppts[i].x < rDims.left) rDims.left = tmppts[i].x;
		else if(tmppts[i].x > rDims.right) rDims.right = tmppts[i].x;
		if(tmppts[i].y < rDims.top) rDims.top = tmppts[i].y;
		else if(tmppts[i].y > rDims.bottom) rDims.bottom = tmppts[i].y;
		}
	//DrawBezier returns not more than 2^MAXDEPTH points
	if(this == CurrGO) {
		o->oBezier(tmppts, nPoints, &pts, &nPts, true);
		o->ShowMark(this, MRK_GODRAW);
		}
	else {
		o->SetLine(&pgLine);
		o->oBezier(tmppts, nPoints, &pts, &nPts, true);
		}
	IncrementMinMaxRect(&rDims, 3);
}

bool
Bezier::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, i1, i2;

	switch (cmd) {
	case CMD_MRK_DIRTY:		case CMD_FLUSH:
		if (cmd == CMD_MRK_DIRTY) {				//issued by Undo
			CurrGO = this;						bModified = true;
			}
		if(pHandles) {
			for (i = 0; i < nPoints; i++) {
				if (pHandles[i]) delete(pHandles[i]);
				}
			free(pHandles);		pHandles = 0L;
			}
		if(cmd == CMD_FLUSH && Values && nPoints){
			free(Values);
			Values = 0L;	nPoints = 0;
			}
		if(pts && nPts) free(pts);
		pts = 0L;		nPts = 0;
		return true;
	case CMD_DELOBJ:
		if(pHandles && tmpl && tmpl == (void*)CurrHandle) {
			i = CurrHandle->type - DH_DATA;
			if (i >= 0 && i < nPoints) {
				i = CurrHandle->type - DH_DATA;
				Undo.DataMem(this, (void**)&Values, nPoints * sizeof(lfPOINT), &nPoints, 0L);
				if (i < 2) {
					i1 = 0;					i2 = 3;
					}
				else if (i > (nPoints-3)) {
					i1 = nPoints -3;		i2 = nPoints;
					}
				else {
					i -= (((i-2) % 3)-1);	i1 = i -1;		i2 = i +2;
					RemovePoint(Values, i1+1);
					Values[i2].fx = Values[i1].fx;	Values[i2].fy = Values[i1].fy;
					}
				for (i = 0; i < nPoints; i++){
					if (pHandles[i]) delete(pHandles[i]);
					}
				free(pHandles);		pHandles = 0L;		CurrHandle = 0L;
				i = i2 - i1;
				for( ; i2 < nPoints; i1++, i2++) {
					Values[i1].fx = Values[i2].fx;	Values[i1].fy = Values[i2].fy;
					}
				nPoints -= i;	CurrGO = this;		bModified = true;
				return true;
				}
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_BEZIER;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
		}
	return polyline::Command(cmd, tmpl, o);
}

void
Bezier::AddPoints(int n, lfPOINT *p)
{
	int i;

	for(i = 0; i< n; i++) {
		Values[nPoints].fx = p[i].fx;	Values[nPoints].fy = p[i].fy;
		nPoints ++;
		}
}

//Fitting of a Bezier curve to digitized points is based on:
//   P.J. Schneider (1990) An Algorithm for Automatically Fitting Digitized
//   Curves. In: Graphics Gems (Ed. A. Glassner, Academic Press Inc., 
//   ISBN 0-12-286165-5) pp. 612ff
void
Bezier::FitCurve(lfPOINT *d, int npt, double error)
{
	double len;
	lfPOINT tHat1, tHat2;

	tHat1.fx = d[1].fx - d[0].fx;		tHat1.fy = d[1].fy - d[0].fy;
	if(0.0 != (len = sqrt((tHat1.fx * tHat1.fx) + (tHat1.fy * tHat1.fy)))) {
		tHat1.fx /= len;					tHat1.fy /= len;
		}
	tHat2.fx = d[npt-2].fx - d[npt-1].fx;		tHat2.fy = d[npt-2].fy - d[npt-1].fy;
	if(0.0 != (len = sqrt((tHat2.fx * tHat2.fx) + (tHat2.fy * tHat2.fy)))) {
		tHat2.fx /= len;					tHat2.fy /= len;
		}
	FitCubic(d, 0, npt -1, tHat1, tHat2, error);
}

void
Bezier::RemovePoint(lfPOINT *d, int sel)
{
	lfPOINT tHat1, tHat2;

	tHat1.fx = d[sel-2].fx - d[sel-3].fx;	tHat1.fy = d[sel-2].fy - d[sel-3].fy;
	tHat2.fx = d[sel+2].fx - d[sel+3].fx;	tHat2.fy = d[sel+2].fy - d[sel+3].fy;
	d[sel] = d[sel+3];
	IpolBez(d+sel-3, tHat1, tHat2);
}

//heuristic interpolation: other methods failed
void
Bezier::IpolBez(lfPOINT *d, lfPOINT tHat1, lfPOINT tHat2)
{
	double b0, b1, b2;						//temp variables

	b1 = d[3].fx - d[0].fx;					b2 = d[3].fy - d[0].fy;
	b0 = sqrt(b1 * b1 + b2 * b2)/3.0;
	b1 = b0/sqrt(tHat1.fx * tHat1.fx + tHat1.fy * tHat1.fy);
	b2 = b0/sqrt(tHat2.fx * tHat2.fx + tHat2.fy * tHat2.fy);
	d[1].fx = d[0].fx + tHat1.fx * b1;		d[1].fy = d[0].fy + tHat1.fy * b1;
	d[2].fx = d[3].fx + tHat2.fx * b2;		d[2].fy = d[3].fy + tHat2.fy * b2;
}

//Fit a Bezier curve to a (sub)set of digitized points
void
Bezier::FitCubic(lfPOINT *d, int first, int last, lfPOINT tHat1, lfPOINT tHat2, double error)
{
	lfPOINT bezCurve[4];
	double *u, *uPrime, maxError, iterationError, len;
	int i, splitPoint, npt;
	lfPOINT tHatCenter;
	int maxIterations = 8;

	iterationError = error * error;
	npt = last - first +1;
	if(npt == 2) {
		bezCurve[0] = d[first];					bezCurve[3] = d[last];
		IpolBez(bezCurve, tHat1, tHat2);
		AddPoints(3, bezCurve);
		return;
		}
	u = ChordLengthParameterize(d, first, last);
	GenerateBezier(d, first, last, u, tHat1, tHat2, bezCurve);
	maxError = ComputeMaxError(d, first, last, bezCurve, u, &splitPoint);
	if(maxError < error) {
		if(maxError < 0.0) {					//Failure fitting curve
			IpolBez(bezCurve, tHat1, tHat2);
			}
		AddPoints(3, bezCurve);					return;
		}
	if(maxError < iterationError) {
		for (i = 0; i < maxIterations; i++) {
			uPrime = Reparameterize(d, first, last, u, bezCurve);
			GenerateBezier(d, first, last, uPrime, tHat1, tHat2, bezCurve);
			maxError = ComputeMaxError(d, first, last, bezCurve, uPrime, &splitPoint);
			if (maxError < error) {
				if(maxError < 0.0) IpolBez(bezCurve, tHat1, tHat2);
				AddPoints(3, bezCurve);			return;
				}
			free(u);							u = uPrime;
			}
		}
	//Fitting failed: split at max error point and recurse
	tHatCenter.fx = d[splitPoint-1].fx - d[splitPoint+1].fx;
	tHatCenter.fy = d[splitPoint-1].fy - d[splitPoint+1].fy;
	if(0.0 != (len = sqrt((tHatCenter.fx * tHatCenter.fx) + (tHatCenter.fy * tHatCenter.fy)))) {
		tHatCenter.fx /= len;					tHatCenter.fy /= len;
		}
	FitCubic(d, first, splitPoint, tHat1, tHatCenter, error * _SQRT2);
	tHatCenter.fx = -tHatCenter.fx;				tHatCenter.fy = -tHatCenter.fy;
	FitCubic(d, splitPoint, last, tHatCenter, tHat2, error * _SQRT2);
}

//Use least-squares method to find Bezier control points for region.
void
Bezier::GenerateBezier(lfPOINT *d, int first, int last, double *uPrime, 
	lfPOINT tHat1, lfPOINT tHat2, lfPOINT *bezCurve)
{
	int i, npt;
	lfPOINT *A0, *A1, tmp, v1, v2;
	double C[2][2], X[2];
	double det_C0_C1, det_C0_X, det_X_C1, alpha_l, alpha_r;
	double b0, b1, b2, b3;		//temp variables

	npt = last - first +1;
	A0 = (lfPOINT*) malloc(npt * sizeof(lfPOINT));
	A1 = (lfPOINT*) malloc(npt * sizeof(lfPOINT));
	for (i = 0; i < npt; i++) {
		v1 = tHat1;								v2 = tHat2;
		b2 = 1.0 - uPrime[i];
		b1 = 3.0 * uPrime[i] * b2 * b2;			b2 = 3.0 * uPrime[i] * uPrime[i] * b2;
		if(0.0 != (b0 = sqrt((v1.fx * v1.fx) + (v1.fy * v1.fy)))) {
			v1.fx *= b1/b0;						v1.fy *= b1/b0;
			}
		if(0.0 != (b0 = sqrt((v2.fx * v2.fx) + (v2.fy * v2.fy)))) {
			v2.fx *= b2/b0;						v2.fy *= b2/b0;
			}
		A0[i] = v1;								A1[i] = v2;
		}
	C[0][0] = C[0][1] = C[1][0] = C[1][1] = X[0] = X[1] = 0.0;
	for (i = 0; i < npt; i++) {
		C[0][0] += ((A0[i].fx * A0[i].fx) + (A0[i].fy * A0[i].fy));
		C[0][1] += ((A0[i].fx * A1[i].fx) + (A0[i].fy * A1[i].fy));
		C[1][0] = C[0][1];
		C[1][1] += ((A1[i].fx * A1[i].fx) + (A1[i].fy * A1[i].fy));
		b2 = 1.0 - uPrime[i];					b0 = b2 * b2 * b2;
		b1 = 3.0 * uPrime[i] * b2 * b2;			b2 = 3.0 * uPrime[i] * uPrime[i] * b2;
		b3 = uPrime[i] * uPrime[i] * uPrime[i];
		tmp.fx = d[last].fx * b2 + d[last].fx * b3;
		tmp.fy = d[last].fy * b2 + d[last].fy * b3;
		tmp.fx += d[first].fx * b1;				tmp.fy += d[first].fy * b1;
		tmp.fx += d[first].fx * b0;				tmp.fy += d[first].fy * b0;
		tmp.fx = d[first+i].fx - tmp.fx;		tmp.fy = d[first+i].fy - tmp.fy;
		X[0] += (A0[i].fx * tmp.fx + A0[i].fy * tmp.fy);
		X[1] += (A1[i].fx * tmp.fx + A1[i].fy * tmp.fy);
		}
	det_C0_C1 = C[0][0] * C[1][1] - C[1][0] * C[0][1];
	det_C0_X = C[0][0] * X[1] - C[0][1] * X[0];
	det_X_C1 = X[0] * C[1][1] - X[1] * C[0][1];
	if(det_C0_C1 == 0.0) det_C0_C1 = (C[0][0] * C[1][1]) * 10e-12;
	alpha_l = det_X_C1 / det_C0_C1;				alpha_r = det_C0_X / det_C0_C1;
	bezCurve[0] = d[first];						bezCurve[3] = d[last];
	if(alpha_l < 0.0 || alpha_r < 0.0) {
		//use Wu/Barsky heuristic
		b1 = d[last].fx - d[first].fx;			b2 = d[last].fy - d[first].fy;
		b0 = sqrt(b1 * b1 + b2 * b2)/3.0;
		b1 = b0/sqrt(tHat1.fx * tHat1.fx + tHat1.fy * tHat1.fy);
		b2 = b0/sqrt(tHat2.fx * tHat2.fx + tHat2.fy * tHat2.fy);
		}
	else {
		b1 = alpha_l/sqrt(tHat1.fx * tHat1.fx + tHat1.fy * tHat1.fy);
		b2 = alpha_r/sqrt(tHat2.fx * tHat2.fx + tHat2.fy * tHat2.fy);
		}
	bezCurve[1].fx = bezCurve[0].fx + tHat1.fx * b1;
	bezCurve[1].fy = bezCurve[0].fy + tHat1.fy * b1;
	bezCurve[2].fx = bezCurve[3].fx + tHat2.fx * b2;
	bezCurve[2].fy = bezCurve[3].fy + tHat2.fy * b2;
	free(A0);									free(A1);
}

//Given set of points and their parameterization, try to find
//  a better parameterization.
double *
Bezier::Reparameterize(lfPOINT *d, int first, int last, double *u, lfPOINT *bezCurve)
{
	int i, j, k, npt = last-first+1;
	double *uPrime, num, den, *pl, *ph, *pq;
	lfPOINT Q1[3], Q2[2], Q_u, Q1_u, Q2_u;

	uPrime = (double*)malloc(npt * sizeof(double));
	//Use Newton-Raphson iteration to find better root
	for (i = first, j = 0; i <= last; i++, j++) {
		for(pl=(double*)bezCurve, ph=pl+2, pq=(double*)Q1, k=0; k <= 4; pl++, ph++, pq++, k++) {
			*pq = (*ph - *pl ) * 3.0;
			}
		for(pl=(double*)Q1, ph=pl+2, pq=(double*)Q2, k=0; k <= 2; pl++, ph++, pq++, k++) {
			*pq = (*ph - *pl ) * 2.0;
			}
		Q_u = fBezier(3, bezCurve, u[j]);
		Q1_u = fBezier(2, Q1, u[j]);	
		Q2_u = fBezier(1, Q2, u[j]);
		num = (Q_u.fx - d[i].fx) * (Q1_u.fx) + (Q_u.fy - d[i].fy) * (Q1_u.fy);
		den = (Q1_u.fx) * (Q1_u.fx) + (Q1_u.fy) * (Q1_u.fy) + 
			(Q_u.fx - d[i].fx) * (Q2_u.fx) + (Q_u.fy - d[i].fy) * (Q2_u.fy);
		uPrime[j] = u[j] - (num/den);
		}
	return uPrime;
}

//evaluate a Bezier curve at a particular parameter value
lfPOINT
Bezier::fBezier(int degree, lfPOINT *V, double t)
{
	int i, j;
	lfPOINT Q;
	lfPOINT *Vtemp;

	Vtemp = (lfPOINT *)malloc((degree+1) * sizeof(lfPOINT));
	for (i = 0; i <= degree; i++) {
		Vtemp[i] = V[i];
		}
	for (i = 1; i <= degree; i++) {
		for (j = 0; j <= degree-i; j++) {
			Vtemp[j].fx = (1.0 -t) * Vtemp[j].fx + t * Vtemp[j+1].fx;
			Vtemp[j].fy = (1.0 -t) * Vtemp[j].fy + t * Vtemp[j+1].fy;
			}
		}
	Q = Vtemp[0];
	free(Vtemp);
	return Q;
}

double *
Bezier::ChordLengthParameterize(lfPOINT *d, int first, int last)
{ 
	int i;
	double tmp1, tmp2, *u;

	u = (double*)malloc((last-first+1) * sizeof(double));
	u[0] = 0.0;
	for(i = first+1; i <= last; i++) {
		tmp1 = d[i].fx - d[i - 1].fx;		tmp2 = d[i].fy - d[i - 1].fy;
		u[i-first] = u[i-first-1] + sqrt((tmp1*tmp1) + (tmp2*tmp2));
		}
	for(i = first +1; i <= last; i++) {
		u[i-first] = u[i-first] / u[last-first];
		}
	return u;
}

double
Bezier::ComputeMaxError(lfPOINT *d, int first, int last, lfPOINT *bezCurve, double *u, int *splitPoint)
{
	int i;
	double maxDist, dist;
	lfPOINT P, v;

	*splitPoint = (last - first + 1)>>1;
	maxDist = -1.0;
	for(i = first +1; i < last; i++) {
		P = fBezier(3, bezCurve, u[i-first]);
		v.fx = P.fx - d[i].fx;			v.fy = P.fy - d[i].fy;
		dist = v.fx * v.fx + v.fy * v.fy;
		if(dist >= maxDist) {
			maxDist = dist;		*splitPoint = i;
			}
		}
	return maxDist;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// polygons are based on the polyline object
polygon::polygon(GraphObj *par, DataObj *d, lfPOINT *fpts, int cpts):
	polyline(par, d, fpts, cpts)
{
	Id = GO_POLYGON;
	memcpy(&pgLine, defs.pgLineDEF(0L), sizeof(LineDEF));
	type = 1;
	if (name) free(name);
	name = rlp_strdup((char *)"polygon");
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// rectangle with absolute coordinates
rectangle::rectangle(GraphObj *par, DataObj *d, lfPOINT *p1, lfPOINT *p2):
	GraphObj(par, d)
{
	double dx = 0.0, dy = 0.0;

	FileIO(INIT_VARS);
	if(parent){
		dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
		}
	memcpy(&fp1, p1, sizeof(lfPOINT));	memcpy(&fp2, p2, sizeof(lfPOINT));
	fp1.fx -= dx;	fp1.fy -= dy;		fp2.fx -= dx;	fp2.fy -= dy;
	type = 0;
	Id = GO_RECTANGLE;
	name = rlp_strdup((char*)"rectangle");
	bModified = false;
}

rectangle::rectangle(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	type = 0;
	bModified = false;
}

rectangle::~rectangle()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified) Undo.InvalidGO(this);
	if(drc) delete(drc);
	drc = 0L;
	if (name) free(name);
	name = 0L;
}

double
rectangle::GetSize(int select)
{
	if(parent && parent->Id== GO_GROUP){
	switch(select) {
		case SIZE_XPOS:		return fp1.fx + parent->GetSize(SIZE_XPOS);
		case SIZE_XPOS+1:	return fp2.fx + parent->GetSize(SIZE_XPOS);
		case SIZE_YPOS:		return fp1.fy + parent->GetSize(SIZE_YPOS);
		case SIZE_YPOS+1:	return fp2.fy + parent->GetSize(SIZE_YPOS);
		case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
		case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
			return parent->GetSize(select);
			}
		return 0.0;
		}
	switch(select) {
	case SIZE_XPOS:		return fp1.fx;
	case SIZE_XPOS+1:	return fp2.fx;
	case SIZE_YPOS:		return fp1.fy;
	case SIZE_YPOS+1:	return fp2.fy;
	case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
	case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
rectangle::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_XPOS:				fp1.fx = value;			return true;
	case SIZE_XPOS+1:			fp2.fx = value;			return true;
	case SIZE_YPOS:				fp1.fy = value;			return true;
	case SIZE_YPOS+1:			fp2.fy = value;			return true;
		}
	return false;
}

void 
rectangle::DoMark(anyOutput *o, bool mark)
{
	RECT upd;

	if(!drc) drc = new dragRect(this, 0);
	memcpy(&upd, &rDims, sizeof(RECT));
	if(mark){
		if(drc) drc->DoPlot(o);
		}
	else if(parent)	parent->DoPlot(o);
	IncrementMinMaxRect(&upd, 6);
	o->UpdateRect(&upd, true);
}

void
rectangle::DoPlot(anyOutput *o)
{
	int x1, y1, x2, y2;
	double tmp, dx, dy;

	if(!parent || !o || hidden) return;
	dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
	if(fp1.fx > fp2.fx) {
		tmp = fp2.fx;	fp2.fx = fp1.fx;	fp1.fx = tmp;
		}
	if(fp1.fy > fp2.fy) {
		tmp = fp2.fy;	fp2.fy = fp1.fy;	fp1.fy = tmp;
		}
	if(type == 2) PlotRoundRect(o);
	else {
		x1 = o->co2ix(fp1.fx+dx);			y1 = o->co2iy(fp1.fy+dy);
		x2 = o->co2ix(fp2.fx+dx);			y2 = o->co2iy(fp2.fy+dy);
		o->SetLine(&Line);				o->SetFill(&Fill);
		if(type == 1) o->oCircle(x1, y1, x2, y2, name);
		else o->oRectangle(x1, y1, x2, y2, name);
		SetMinMaxRect(&rDims, x1, y1, x2, y2);
		}
	o->MrkMode = MRK_NONE;
	if(CurrGO == this) o->ShowMark(this, MRK_GODRAW);
}

bool
rectangle::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_FLUSH:
		if(pts) free(pts);
		pts = 0L;				if(name) free(name);
		name = 0L;
		return true;
	case CMD_SCALE:
		fp1.fx = ((scaleINFO*)tmpl)->sx.fx + fp1.fx * ((scaleINFO*)tmpl)->sx.fy;
		fp2.fx = ((scaleINFO*)tmpl)->sx.fx + fp2.fx * ((scaleINFO*)tmpl)->sx.fy;
		fp1.fy = ((scaleINFO*)tmpl)->sy.fx + fp1.fy * ((scaleINFO*)tmpl)->sy.fy;
		fp2.fy = ((scaleINFO*)tmpl)->sy.fx + fp2.fy * ((scaleINFO*)tmpl)->sy.fy;
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		FillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;	FillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;			rad *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_SAVEPOS:
		bModified = true;
		Undo.SaveLFP(this, &fp1, 0L);
		Undo.SaveLFP(this, &fp2, UNDO_CONTINUE);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(rDims, mev->x, mev->y) && !(CurrGO) && (o)){
				return o->ShowMark(this, MRK_GODRAW);
				}
			}
		return false;
	case CMD_SET_DATAOBJ:
		switch (type) {
		case 1:		Id = GO_ELLIPSE;		break;
		case 2:		Id = GO_ROUNDREC;		break;
		default:	Id = GO_RECTANGLE;		break;
			}
		if (tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_MOVE:			case CMD_UNDO_MOVE:			case CMD_REDRAW:
		if (cmd == CMD_MOVE) {
			bModified = true;
			Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			}
		if (cmd == CMD_MOVE || cmd == CMD_UNDO_MOVE) {
			fp1.fx += ((lfPOINT*)tmpl)[0].fx;	fp1.fy += ((lfPOINT*)tmpl)[0].fy;
			fp2.fx += ((lfPOINT*)tmpl)[0].fx;	fp2.fy += ((lfPOINT*)tmpl)[0].fy;
			}
		if (parent && cmd != CMD_UNDO_MOVE){
			parent->Command(CMD_REDRAW, tmpl, o);
			}
		CurrGO = this;
		return true;
	case CMD_SETSCROLL:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_CONFIG:
		return PropertyDlg();
		}
	return false;
}

void
rectangle::Track(POINT *p, anyOutput *o, bool)
{
	POINT tpts[5];
	RECT old_rc;
	double dx, dy;

	if (Notary->IsValidPtr(o) && parent){
		dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
		memcpy(&old_rc, &rDims, sizeof(rDims));
		o->UpdateRect(&rDims, true);
		tpts[0].x = tpts[1].x = tpts[4].x = o->co2ix(fp1.fx+dx)+p->x;		
		tpts[0].y = tpts[3].y = tpts[4].y = o->co2iy(fp1.fy+dy)+p->y;
		tpts[1].y = tpts[2].y = o->co2iy(fp2.fy+dy)+p->y;
		tpts[2].x = tpts[3].x = o->co2ix(fp2.fx+dx)+p->x;
		UpdateMinMaxRect(&rDims, tpts[0].x, tpts[0].y);
		UpdateMinMaxRect(&rDims, tpts[2].x, tpts[2].y);	
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		o->ShowLine(tpts, 5, Line.color);
		if(type == 1) o->ShowEllipse(tpts[0], tpts[2], Line.color);
		}
}

void *
rectangle::ObjThere(int x, int y)
{
	if(drc) return drc->ObjThere(x, y);
	return 0L;
}

//use circular Bresenham's algorithm to draw rounded rectangles
//Ref: C. Montani, R. Scopigno (1990) "Speres-To-Voxel Conversion", in:
//   Graphic Gems (A.S. Glassner ed.) Academic Press, Inc.; 
//   ISBN 0-12-288165-5 
void
rectangle::PlotRoundRect(anyOutput *o)
{
	int i, m, x, y, di, de, lim, ir, x1, x2, y1, y2;
	double dx, dy;
	POINT np;

	dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
	ir = o->un2ix(rad);
	x1 = o->co2ix(fp1.fx+dx);			y1 = o->co2iy(fp1.fy+dy);
	x2 = o->co2ix(fp2.fx+dx);			y2 = o->co2iy(fp2.fy+dy);
	if (x1 > x2) Swap(x1, x2);
	if(y1 > y2) Swap(y1, y2);
	if(pts) free(pts);
	nPts = 0;
	m = ir*4+10;
	pts = (POINT*)malloc(m*sizeof(POINT));
	if(!pts)return;
	for(i = 0; i < 4; i++) {
		x = lim = 0;	y = ir;	di = 2*(1-ir);
		while (y >= lim){
			if(di < 0) {
				de = 2*di + 2*y -1;
				if(de > 0) {
					x++;	y--;	di += (2*x -2*y +2);
					}
				else {
					x++;	di += (2*x +1);
					}
				}
			else {
				de = 2*di -2*x -1;
				if(de > 0) {
					y--;	di += (-2*y +1);
					}
				else {
					x++;	y--;	di += (2*x -2*y +2);
					}
				}
			switch(i) {
			case 0:
				np.x = x2-ir+x;		np.y = y2-ir+y;
				break;
			case 1:
				np.x = x2-ir+y;		np.y = y1+ir-x;
				break;
			case 2: 
				np.x = x1+ir-x;		np.y = y1+ir-y;
				break;
			case 3:
				np.x = x1+ir-y;		np.y = y2-ir+x;
				break;
				}
			AddToPolygon(&nPts, pts, &np);
			}
		}
	AddToPolygon(&nPts, pts, pts);	//close polygon
	o->SetLine(&Line);				o->SetFill(&Fill);
	o->oPolygon(pts, nPts, name);
	SetMinMaxRect(&rDims, x1, y1, x2, y2);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// ellipse with absolute coordinates
ellipse::ellipse(GraphObj *par, DataObj *d, lfPOINT *p1, lfPOINT *p2)
	:rectangle(par, d, p1, p2)
{
	type = 1;
	Id = GO_ELLIPSE;
	if (name) free(name);
	name = rlp_strdup((char*)"ellipse");
}

ellipse::ellipse(int src)
	:rectangle(src)
{
	type = 1;
	Id = GO_ELLIPSE;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// rounded rectangle
roundrec::roundrec(GraphObj *par, DataObj *d, lfPOINT *p1, lfPOINT *p2)
	:rectangle(par, d, p1, p2)
{
	type = 2;
	Id = GO_ROUNDREC;
	if (name) free(name);
	name = rlp_strdup((char*)"round rec.");
}

roundrec::roundrec(int src)
	:rectangle(src)
{
	type = 2;
	Id = GO_ROUNDREC;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Add a legend to the graph
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
LegItem::LegItem(GraphObj *par, DataObj *d, LineDEF *ld, LineDEF *lf, FillDEF *fd, char *desc)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(!ld && !fd && lf) ld = lf;
	if(ld) {
		memcpy(&DataLine, ld, sizeof(LineDEF));
		flags |= 0x01;
		}
	if (lf) {
		memcpy(&OutLine, lf, sizeof(LineDEF));
		OutLine.width /= 2.0;
		}
	if(fd) {
		if(fd->hatch) memcpy(&HatchLine, fd->hatch, sizeof(LineDEF));
		memcpy(&Fill, fd, sizeof(FillDEF));
		Fill.hatch = &HatchLine;
		flags |= 0x02;
		}
	DefDesc(desc);		Id = GO_LEGITEM;		moveable = 1;
}

LegItem::LegItem(GraphObj *par, DataObj *d, LineDEF *ld, Symbol *sy)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(ld) {
		memcpy(&DataLine, ld, sizeof(LineDEF));		flags |= 0x01;
		}
	if(sy) {
		Sym = sy;		Sym->parent = this;			flags |= 0x04;
		}
	DefDesc(sy ? sy->name : 0L);		Id = GO_LEGITEM;		moveable = 1;
}

LegItem::LegItem(GraphObj *par, DataObj *d, LineDEF *ld, int err, char *desc)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);		flags |= (0x80 | (err & 0x7f));
	if(ld) {
		memcpy(&OutLine, ld, sizeof(LineDEF));
		}
	DefDesc(desc);		Id = GO_LEGITEM;		moveable = 1;
}

LegItem::LegItem(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	moveable = 1;
}

LegItem::~LegItem()
{
	if(Sym) DeleteGO(Sym);
	Sym = 0L;
	if(Desc) DeleteGO(Desc);
	Desc = 0L;
}

double
LegItem::GetSize(int select)
{
	switch(select) {
	case SIZE_XCENTER:
		return (parent->GetSize((flags & 0x01) ? SIZE_XPOS+2 : SIZE_XPOS) + 
			parent->GetSize((flags & 0x01) ? SIZE_XPOS+3 : SIZE_XPOS+1))/2.0;
	case SIZE_YCENTER:
		return (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
	case SIZE_LB_XPOS:		case SIZE_LB_YPOS:
	case SIZE_GRECT_TOP:	case SIZE_GRECT_LEFT:
	default:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

void
LegItem::DoPlot(anyOutput *o)
{
	POINT pts[3];
	int ie, cy;

	if(!parent || !o) return;
	hcr.top = iround(parent->GetSize(SIZE_YPOS));
	hcr.bottom = iround(parent->GetSize(SIZE_YPOS+1));
	if(flags & 0x80) {
		hcr.left = iround(parent->GetSize((flags & 0x40) ? SIZE_XPOS+2 : SIZE_XPOS));
		hcr.right = iround(parent->GetSize((flags & 0x40) ? SIZE_XPOS+3 :SIZE_XPOS+1));
		ie = o->un2ix(DefSize(SIZE_ERRBAR)/2.0);
		o->SetLine(&OutLine);						cy = (hcr.top + hcr.bottom)>>1;
		if((flags & 0x3f) == 0x01) {
			pts[0].x = pts[1].x = (hcr.right + hcr.left)>>1;
			pts[0].y = hcr.top;		pts[1].y = hcr.bottom;
			o->oPolyline(pts, 2);		
			pts[0].x -= (ie-1);		pts[1].x += ie;		pts[0].y = pts[1].y = hcr.top;
			o->oPolyline(pts, 2);					pts[0].y = pts[1].y = hcr.bottom;
			o->oPolyline(pts, 2);
			}
		else if((flags & 0x3f) == 0x02) {
			pts[0].x = pts[1].x = ((hcr.right + hcr.left)>>1);
			pts[0].y = pts[1].y = cy;			cy = ((hcr.bottom - hcr.top)>>1);
			pts[0].x -= cy;						pts[2].x = (pts[1].x += cy);
			o->oPolyline(pts, 2);				pts[1].x = pts[0].x;
			pts[0].y -= ie;						pts[1].y += ie;
			o->oPolyline(pts, 2);
			pts[0].x = pts[1].x = pts[2].x;		o->oPolyline(pts, 2);
			}
		else if((flags & 0x3f) == 0x03) {
			pts[0].x = pts[1].x = (hcr.right + hcr.left)>>1;
			pts[0].y = hcr.top;		pts[1].y = hcr.bottom;
			o->oPolyline(pts, 2);			pts[0].x -= (ie-1);
			pts[0].y = pts[1].y = hcr.top;		o->oPolyline(pts, 2);
			pts[0].y = pts[1].y = hcr.bottom;	pts[0].x += (ie + ie -1);
			o->oPolyline(pts, 2);
			}
		else if((flags & 0x3f) == 0x04) {
			pts[0].x = pts[1].x = (hcr.right + hcr.left)>>1;
			pts[0].y = hcr.top;		pts[1].y = hcr.bottom;
			o->oPolyline(pts, 2);				pts[0].x += ie;
			pts[0].y = pts[1].y = hcr.top;		o->oPolyline(pts, 2);
			pts[0].y = pts[1].y = hcr.bottom;	pts[0].x -= (ie + ie -1);
			o->oPolyline(pts, 2);
			}
		else if((flags & 0x3f) == 0x05) {
			pts[0].x = pts[1].x = (hcr.right + hcr.left)>>1;
			pts[0].y = hcr.top;		pts[1].y = hcr.bottom;
			o->oPolyline(pts, 2);
			}
		}
	else {
		hcr.left = iround(parent->GetSize((flags & 0x01) ? SIZE_XPOS+2 : SIZE_XPOS));
		hcr.right = iround(parent->GetSize((flags & 0x01) ? SIZE_XPOS+3 :SIZE_XPOS+1));
		if(flags & 0x02){
			o->SetLine(&OutLine);	o->SetFill(&Fill);
			o->oRectangle(hcr.left, hcr.top, hcr.right, hcr.bottom, name);
			}
		if(flags & 0x01){
			pts[0].x = hcr.left;	pts[1].x = hcr.right;
			pts[0].y = pts[1].y = iround(GetSize(SIZE_YCENTER))+1;
			o->SetLine(&DataLine);	o->oPolyline(pts, 2);
			}
		if(flags & 0x04){
			if(Sym) Sym->DoPlot(o);
			}
		}
	SetMinMaxRect(&rDims, hcr.left, hcr.top, hcr.right, hcr.bottom);
	if(Desc) {
		Desc->moveable = 1;			Desc->DoPlot(o);
		if(Desc->rDims.bottom > rDims.bottom){
			parent->SetSize(SIZE_YPOS+1, (double)Desc->rDims.bottom);
			}
		UpdateMinMaxRect(&rDims, Desc->rDims.left, Desc->rDims.top);
		UpdateMinMaxRect(&rDims, Desc->rDims.right, Desc->rDims.bottom);
		}
}

void
LegItem::DoMark(anyOutput *o, bool mark)
{
	RECT cr;
	LineDEF ld = {0.0, 1.0, 0x00000000L, 0x00000000L};
	POINT pts[5];

	if(!parent || !o) return;
	cr.left = hcr.left-5;			cr.right = hcr.right+3;
	cr.top = hcr.top-3;				cr.bottom = hcr.bottom+1;
	ld.color = mark ? 0x00000000L : 0x00ffffffL;	o->SetLine(&ld);
	pts[0].x = pts[3].x = pts[4].x = cr.left;	pts[0].y = pts[1].y = pts[4].y = cr.top;
	pts[1].x = pts[2].x = cr.right;				pts[2].y = pts[3].y = cr.bottom;
	o->oPolyline(pts, 5);						IncrementMinMaxRect(&cr, 3);
	o->UpdateRect(&cr, true);
}

bool
LegItem::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;

	switch(cmd){
	case CMD_TEXTTHERE:
		if(Desc && Desc->Command(cmd, tmpl, o)) return true;
		return false;
	case CMD_MOUSE_EVENT:
		if(tmpl && IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y) && o) {
			if(Desc && Desc->Command(cmd, tmpl, o)) return true;
			if(!CurrGO) o->ShowMark(CurrGO=this, MRK_GODRAW);
			}
		break;
	case CMD_SCALE:
		DataLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;		DataLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		OutLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;			OutLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		HatchLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;		HatchLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;	
		if(Sym) Sym->Command(cmd, tmpl, o);
		if(Desc) Desc->Command(cmd, tmpl, o);
		break;
	case CMD_REDRAW:	case CMD_MOVE:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MUTATE:
		if(!parent || !(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Desc == tmpPlots[0]) {
			Undo.MutateGO((GraphObj**)&Desc, tmpPlots[1], 0L, o);
			return true;
			}
		break;
	case CMD_SET_DATAOBJ:
		if(Desc) Desc->Command(cmd, tmpl, o);
		Id = GO_LEGITEM;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
		}
	return false;
}

void
LegItem::Track(POINT *p, anyOutput *o, bool)
{
	if(parent) parent->Track(p, o, false);
}

bool
LegItem::HasFill(LineDEF *ld, FillDEF *fd, char *)
{
	if (OutLine.color != ld->color) return false;
	if(fd && cmpFillDEF(fd, &Fill)) return false;
	if(fd && fd->hatch && cmpLineDEF(fd->hatch, &HatchLine)) return false;
	return true;
}

bool
LegItem::HasSym(LineDEF *ld, GraphObj *sy)
{
	if(sy && !Sym) return false;
	if(sy->Id == GO_SYMBOL && (sy->type & 0xfff) != (Sym->type & 0xfff)) return false;
	if(sy && Sym) {
		if(Sym->GetSize(SIZE_SYMBOL) != sy->GetSize(SIZE_SYMBOL)) return false;
		if(Sym->GetSize(SIZE_SYM_LINE) != sy->GetSize(SIZE_SYM_LINE)) return false;
		if(Sym->GetColor(COL_SYM_LINE) != sy->GetColor(COL_SYM_LINE)) return false;
		if(Sym->GetColor(COL_SYM_FILL) != sy->GetColor(COL_SYM_FILL)) return false;
		}
	if(ld && cmpLineDEF(ld, &DataLine)) return false;
	return true;
}

bool
LegItem::HasErr(LineDEF *ld, int err)
{
	if((((DWORD)err & 0x1f) | 0x80) != (flags & 0x9f)) return false;
	if(ld && cmpLineDEF(ld, &OutLine)) return false;
	return true;
}

void
LegItem::DefDesc(char *txt)
{
	TextDEF td;
	int cb;

	cb = txt && *txt ? (int)strlen(txt) : 20;
	if(cb < 20) cb = 20;
	td.ColTxt = 0x00000000L;		td.ColBg = 0x00ffffffL;
	td.fSize = DefSize(SIZE_TICK_LABELS);
	td.RotBL = td.RotCHAR = 0.0;
	td.iSize = 0;					td.Align = TXA_VTOP | TXA_HLEFT;
	td.Mode = TXM_TRANSPARENT;		td.Style = TXS_NORMAL;
	td.Font = FONT_HELVETICA;		td.text = (unsigned char*)malloc(cb+2);
	rlp_strcpy(td.text, cb+2, txt ? txt : (char*)"text");
	Desc = new Label(this, data, 0, 0, &td, LB_X_PARENT | LB_Y_PARENT, 0L);
}

Legend::Legend(GraphObj *par, DataObj *d):GraphObj(par, d)
{
	FileIO(INIT_VARS);		Id = GO_LEGEND;		moveable = true;
}

Legend::Legend(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	moveable = true;
}

Legend::~Legend()
{
	int i;

	if(Items) {
		for (i = 0; i < nItems; i++){
			if (Items[i]) DeleteGO(Items[i]);
			}
		free(Items);	Items = 0L;
		}
	if(to) DelBitmapClass(to);
	to = 0L;
}
double
Legend::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:			return C_Rect.Xmin;
	case SIZE_XPOS+1:		return C_Rect.Xmax;
	case SIZE_XPOS+2:		return E_Rect.Xmin;
	case SIZE_XPOS+3:		return E_Rect.Xmax;
	case SIZE_YPOS:			return C_Rect.Ymin;
	case SIZE_YPOS+1:		return C_Rect.Ymax;
	case SIZE_LB_XPOS:		return lb_pos.fx;
	case SIZE_LB_YPOS:		return lb_pos.fy;
	case SIZE_GRECT_TOP:	case SIZE_GRECT_LEFT:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
Legend::SetSize(int select, double value)
{
	double tmp;

	switch (select & 0xfff){
	case SIZE_XPOS:		pos.fx = value;		return true;
	case SIZE_YPOS:		pos.fy = value;		return true;
	case SIZE_YPOS+1:
		tmp = value - C_Rect.Ymax;
		C_Rect.Ymin += tmp;		C_Rect.Ymax += tmp;		lb_pos.fy +=tmp;
		}
	return false;
}

void
Legend::DoPlot(anyOutput *o)
{
	int i;
	double y_inc, dx, dy;

	if(!o || !Items) return;
	if(to) DelBitmapClass(to);
	to = 0L;
	dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
	C_Rect.Xmin = o->co2fix(pos.fx + B_Rect.Xmin + D_Rect.Xmin + dx);
	C_Rect.Ymin = o->co2fiy(pos.fy + B_Rect.Ymin + D_Rect.Ymin + dy);
	C_Rect.Xmax = C_Rect.Xmin + o->un2fix(D_Rect.Xmax - D_Rect.Xmin);
	C_Rect.Ymax = C_Rect.Ymin + o->un2fix(D_Rect.Ymax - D_Rect.Ymin);
	E_Rect.Ymin = C_Rect.Ymin;	E_Rect.Ymax = C_Rect.Ymax;
	E_Rect.Xmin = o->co2fix(pos.fx + B_Rect.Xmin + F_Rect.Xmin + dx);
	E_Rect.Xmax = E_Rect.Xmin + o->un2fix(F_Rect.Xmax - F_Rect.Xmin);
	y_inc = floor(0.5+o->un2fiy(B_Rect.Ymax - B_Rect.Ymin));
	rDims.top = iround(C_Rect.Ymin); rDims.bottom = iround(C_Rect.Ymax);
	rDims.left = rDims.right = iround(C_Rect.Xmin);	
	lb_pos.fx = o->co2fix(pos.fx + B_Rect.Xmax + dx);	lb_pos.fy = C_Rect.Ymin;
	//draw all items
	for(i = 0; i < nItems; i++) {
		if(Items[i]){
			if((Items[i]->flags & 0x11) == 0x01) hasLine = true;
			Items[i]->DoPlot(o);
			UpdateMinMaxRect(&rDims, Items[i]->rDims.left, Items[i]->rDims.top);
			UpdateMinMaxRect(&rDims, Items[i]->rDims.right, Items[i]->rDims.bottom);
			C_Rect.Ymin += y_inc;		C_Rect.Ymax += y_inc;
			lb_pos.fy += y_inc;
			}
		}
	IncrementMinMaxRect(&rDims, 6);
}

void
Legend::DoMark(anyOutput *o, bool mark)
{
	RECT cr;
	LineDEF ld = {0.0, 1.0, 0x00c0c0c0L, 0x00000000L};
	POINT pts[5];

	if(!parent || !o) return;
	cr.left = rDims.left;			cr.right = rDims.right;
	cr.top = rDims.top;				cr.bottom = rDims.bottom;
	ld.color = mark ? 0x00c0c0c0L : 0x00ffffffL;	o->SetLine(&ld);
	pts[0].x = pts[3].x = pts[4].x = cr.left;	pts[0].y = pts[1].y = pts[4].y = cr.top;
	pts[1].x = pts[2].x = cr.right;				pts[2].y = pts[3].y = cr.bottom;
	o->oPolyline(pts, 5);						IncrementMinMaxRect(&cr, 3);
	o->UpdateRect(&cr, true);
}

bool
Legend::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;

	switch(cmd){
	case CMD_TEXTTHERE:
		if(Items) for(i = 0; i< nItems; i++) 
			if(Items[i] && Items[i]->Command(cmd, tmpl, o)) return true;
		return false;
	case CMD_MOUSE_EVENT:
		if(o && tmpl && IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) {
			if(Items) for(i = 0; i< nItems; i++) 
				if(Items[i] && Items[i]->Command(cmd, tmpl, o)) return true;
			if(!CurrGO) o->ShowMark(CurrGO = this, MRK_GODRAW);
			}
		break;
	case CMD_SET_DATAOBJ:
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->Command(cmd, tmpl, o);
		Id = GO_LEGEND;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_SCALE:
		pos.fx *= ((scaleINFO*)tmpl)->sx.fy;			pos.fy *= ((scaleINFO*)tmpl)->sy.fy;
		lb_pos.fx *= ((scaleINFO*)tmpl)->sx.fy;			lb_pos.fy *= ((scaleINFO*)tmpl)->sy.fy;
		B_Rect.Xmax *= ((scaleINFO*)tmpl)->sx.fy;		B_Rect.Xmin *= ((scaleINFO*)tmpl)->sx.fy;
		B_Rect.Ymax *= ((scaleINFO*)tmpl)->sy.fy;		B_Rect.Ymin *= ((scaleINFO*)tmpl)->sy.fy;
		C_Rect.Xmax *= ((scaleINFO*)tmpl)->sx.fy;		C_Rect.Xmin *= ((scaleINFO*)tmpl)->sx.fy;
		C_Rect.Ymax *= ((scaleINFO*)tmpl)->sy.fy;		C_Rect.Ymin *= ((scaleINFO*)tmpl)->sy.fy;
		D_Rect.Xmax *= ((scaleINFO*)tmpl)->sx.fy;		D_Rect.Xmin *= ((scaleINFO*)tmpl)->sx.fy;
		D_Rect.Ymax *= ((scaleINFO*)tmpl)->sy.fy;		D_Rect.Ymin *= ((scaleINFO*)tmpl)->sy.fy;
		E_Rect.Xmax *= ((scaleINFO*)tmpl)->sx.fy;		E_Rect.Xmin *= ((scaleINFO*)tmpl)->sx.fy;
		E_Rect.Ymax *= ((scaleINFO*)tmpl)->sy.fy;		E_Rect.Ymin *= ((scaleINFO*)tmpl)->sy.fy;
		F_Rect.Xmax *= ((scaleINFO*)tmpl)->sx.fy;		F_Rect.Xmin *= ((scaleINFO*)tmpl)->sx.fy;
		F_Rect.Ymax *= ((scaleINFO*)tmpl)->sy.fy;		F_Rect.Ymin *= ((scaleINFO*)tmpl)->sy.fy;
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		o->HideMark();
		if(Items && parent) for(i = 0; i < nItems; i++) {
			if(Items[i] && tmpl == (void *)Items[i]) {
				Undo.DeleteGO((GraphObj**)(&Items[i]), 0L, o);
				parent->Command(CMD_REDRAW, NULL, o);
				return true;
				}
			}
		break;
	case CMD_MOVE:		case CMD_UNDO_MOVE:			case CMD_REDRAW:
		if (cmd == CMD_MOVE) Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
		if (cmd == CMD_MOVE || cmd == CMD_UNDO_MOVE) {
			pos.fx += ((lfPOINT*)tmpl)[0].fx;	pos.fy += ((lfPOINT*)tmpl)[0].fy;
			CurrGO = this;
			}
		if (parent && cmd != CMD_UNDO_MOVE){
			parent->Command(CMD_REDRAW, tmpl, o);
			}
		return true;
	case CMD_DROP_OBJECT:
		if(!tmpl) return false;
		if(!(Items = (LegItem**)realloc(Items, (2+nItems)*sizeof(LegItem*))))return false;
		Items[nItems++] = (LegItem*)tmpl;
		Items[nItems-1]->parent = this;
		return true;
		}
	return false;
}

void
Legend::Track(POINT *p, anyOutput *o, bool)
{
	POINT pts[5];
	LineDEF tld = {0.0, 1.0, 0x00c0c0c0, 0x0L};

	if(!p || !o) return;
	if(to) {
		o->CopyBitmap(trc.left, trc.top, to, 0, 0, trc.right - trc.left, 
			trc.bottom - trc.top, false);
		DelBitmapClass(to);		to = 0L;
		o->UpdateRect(&trc, true);
		}
	trc.left = pts[0].x = pts[1].x = pts[4].x = rDims.left + p->x;		
	trc.top = pts[0].y = pts[3].y = pts[4].y = rDims.top + p->y;
	trc.bottom = pts[1].y = pts[2].y = rDims.bottom + p->y;
	trc.right = pts[2].x = pts[3].x = rDims.right + p->x;
	IncrementMinMaxRect(&trc, 3);	to = GetRectBitmap(&trc, o);
	o->SetLine(&tld);				o->oPolyline(pts, 5);
	o->UpdateRect(&trc, true);
} 

bool
Legend::HasFill(LineDEF *ld, FillDEF *fd, char *desc)
{
	int i;
	LegItem *li;

	if(Items) for(i = 0; i < nItems; i++) {
		if(Items[i] && Items[i]->HasFill(ld, fd, desc)) return true;
		}
	li = new LegItem(this, data, 0L, ld, fd, desc);
	if(li){
		if(!(Command(CMD_DROP_OBJECT, li, 0L))) DeleteGO(li);
		}
	return false;
}

bool
Legend::HasSym(LineDEF *ld, GraphObj *sy, char *desc)
{
	int i, sym;
	Symbol *ns;
	LegItem *li;

	if(!parent || !sy) return true;
	if(ld) hasLine = true;
	if(Items) for(i = 0; i < nItems; i++) {
		if(Items[i] && Items[i]->HasSym(ld, sy)) return true;
		}
	sym = (sy->Id == GO_SYMBOL || sy->Id == GO_SYMTERN) ? (sy->type & 0xff) : 0;
	if(!(ns = new Symbol(this, data, 0.0, 0.0, sym | SYM_POS_PARENT))) return true;
	ns->SetSize(SIZE_SYMBOL, sy->GetSize(SIZE_SYMBOL));
	ns->SetSize(SIZE_SYM_LINE, sy->GetSize(SIZE_SYM_LINE));
	ns->SetColor(COL_SYM_LINE, sy->GetColor(COL_SYM_LINE));
	ns->SetColor(COL_SYM_FILL, sy->GetColor(COL_SYM_FILL));
	if(desc && desc[0]) {
		ns->name = rlp_strdup(desc); 
		}
	else if(sy->name && sy->name[0]) {
		ns->name = rlp_strdup(sy->name); 
		}
	else if(sy->parent && sy->parent->Id < GO_GRAPH && sy->parent->Id >= GO_PLOT) {
		if(((Plot*)(sy->parent))->data_desc) ns->name = rlp_strdup(((Plot*)(sy->parent))->data_desc);
		}
	else if(sy->parent && sy->parent->name && sy->parent->name[0]) {
		ns->name = rlp_strdup(sy->parent->name); 
		}
	li = new LegItem(this, data, ld, ns);
	if(li){
		if(!(Command(CMD_DROP_OBJECT, li, 0L))) DeleteGO(li);
		}
	return false;
}

bool
Legend::HasErr(LineDEF *ld, int err, char *desc)
{
	int i;
	LegItem *li;

	if(Items) for(i = 0; i < nItems; i++) {
		if(Items[i] && Items[i]->HasErr(ld, err)) return true;
		}
	li = new LegItem(this, data, ld, err | (hasLine ? 0x40 : 0), desc);
	if(li){
		if(!(Command(CMD_DROP_OBJECT, li, 0L))) DeleteGO(li);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Graphs are graphic objects containing plots, axes, and drawn objects
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Graph::Graph(GraphObj *par, DataObj *d, anyOutput *o, int style):GraphObj(par, d)
{
	Graph::FileIO(INIT_VARS);
	Disp = o;		Id = GO_GRAPH;		cGraphs++;	bModified = true;
	if(style & 0x10) y_axis.flags |= AXIS_INVERT;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
	Notary->ValPtr(this, true);
}

Graph::Graph(int src):GraphObj(0L, 0L)
{
	int i;

	Graph::FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		x_axis.owner = y_axis.owner = (void *)this;
		//do all axes
		for(i = 0; Axes && i< NumAxes; i++) if(Axes[i]) Axes[i]->parent = this;
		//do all plots
		for(i = 0; Plots && i< NumPlots; i++) if(Plots[i]) Plots[i]->parent = this;
		if(x_axis.max > x_axis.min && y_axis.max > y_axis.min &&
			Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) dirty = false;
		if (theGrid) theGrid->parent = this;
		}
	cGraphs++;		bModified = false;
	rc_mrk.left = -1; rc_mrk.left = -1; rc_mrk.left = -1; rc_mrk.left = -1;
	Notary->ValPtr(this, true);
}

Graph::~Graph()
{
	long i;

	Notary->ValPtr(this, false);		parent = 0L;
	Undo.InvalidGO(this);			DoZoom((char*)"reset");
	if(CurrGraph == this) CurrGraph = 0L;
	if(Plots) {
		for (i = 0; i < NumPlots; i++) {
			if (Plots[i]) DeleteGO(Plots[i]);
			}
		free(Plots);		Plots = 0L;		NumPlots = 0;
		}
	if(Axes) {
		for (i = 0; i < NumAxes; i++){
			if (Axes[i]) DeleteGO(Axes[i]);
			}
		free(Axes);			Axes = 0L;		NumAxes = 0;
		}
	if (theGrid) DeleteGO(theGrid);
	theGrid = 0L;
	if (OwnDisp && Disp) Disp->Command(CM_EXIT, 0L);
	OwnDisp = false;	Disp = 0L;
	if(frm_g) DeleteGO(frm_g);
	if(frm_d) DeleteGO(frm_d);
	if(x_axis.breaks && x_axis.owner == this) free(x_axis.breaks);
	if(y_axis.breaks && y_axis.owner == this) free(y_axis.breaks);
	if(tl_pts) free(tl_pts);
	if(nscp > 0 && nscp <= NumPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;				Sc_Plots = 0L;
	if(name) free(name);
	name = 0L;				if(filename) free(filename);
	filename= 0L;
}

double
Graph::GetSize(int select)
{
	return Graph::DefSize(select);
}

bool
Graph::SetSize(int select, double val)
{
	switch(select & 0xfff) {
	case SIZE_GRECT_TOP:	GRect.Ymin = val;	return true;
	case SIZE_GRECT_BOTTOM:	GRect.Ymax = val;	return true;
	case SIZE_GRECT_LEFT:	GRect.Xmin = val;	return true;
	case SIZE_GRECT_RIGHT:	GRect.Xmax = val;	return true;
	case SIZE_DRECT_TOP:	DRect.Ymin = val;	return true;
	case SIZE_DRECT_BOTTOM:	DRect.Ymax = val;	return true;
	case SIZE_DRECT_LEFT:	DRect.Xmin = val;	return true;
	case SIZE_DRECT_RIGHT:	DRect.Xmax = val;	return true;
	case SIZE_BOUNDS_XMIN:	Bounds.Xmin = val;	return (dirty = true);
	case SIZE_BOUNDS_XMAX:	Bounds.Xmax = val;	return (dirty = true);
	case SIZE_BOUNDS_YMIN:	Bounds.Ymin = val;	return (dirty = true);
	case SIZE_BOUNDS_YMAX:	Bounds.Ymax = val;	return (dirty = true);
	default: return false;
		}
}

DWORD
Graph::GetColor(int select)
{
	switch(select & 0xfff) {
	case COL_AXIS:		return ColAX;
	case COL_BG:		return ColDR;
	case COL_DRECT:		return ColDR;
	case COL_GRECT:		return ColGR;
	case COL_GRECTLINE: return ColGRL;
		}
	if(parent) return parent->GetColor(select);
	else return defs.Color(select);
}

bool
Graph::SetColor(int select, DWORD col)
{
	switch(select) {
	case COL_DRECT:
		ColDR = col;		return true;
	case COL_GRECT:
		ColGR = col;		return true;
	case COL_GRECTLINE:
		ColGRL = col;		return true;
		}
	return false;
}

void
Graph::DoPlot(anyOutput *target)
{
	int i, cb;
	AxisDEF *ax;

	if(nscp > 0 && nscp <= NumPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;		Sc_Plots = 0L;
	rc_mrk.left = rc_mrk.right = rc_mrk.top = rc_mrk.bottom = -1;
	CurrAxes = Axes;		NumCurrAxes = NumAxes;
	if(data) do_formula(data, 0L);			//init mfcalc
	//every graph needs axes!
	if(type == GT_STANDARD && !NumAxes) CreateAxes(AxisTempl);
	//verify ownership of axes
	if(Axes) for (i = 0; i < NumAxes; i++){
		if(Axes[i] && (ax = Axes[i]->GetAxis()))
			if(ax == &x_axis || ax == &y_axis) ax->owner = this;
		}
	if(!name){
#ifdef USE_WIN_SECURE
		cb = sprintf_s(TmpTxt, TMP_TXT_SIZE, "Graph %d", cGraphs) + 2;
#else
		cb = sprintf(TmpTxt, "Graph %d", cGraphs) + 2;
#endif
		name = (char*)realloc(name, cb);		rlp_strcpy(name, cb, TmpTxt);
		}
	if(!target && !Disp) {
		Disp = NewDispClass(this);
		Disp->SetMenu(MENU_GRAPH);
		if(name) Disp->Caption(name, false);
		Disp->CheckMenu(ToolMode, true);
		OwnDisp = true;						defs.SetDisp(Disp);
		bModified = false;			//first graph is not modified!
		Undo.SetDisp(Disp);
		}
	//the first output class is the display class
	if(!Disp && (!(Disp = target))) return;
	Disp->ActualSize(&defs.clipRC);
	if (parent && parent->Id == GO_SPREADDATA) Disp->Erase(Disp->OC_type == OC_BITMAP ? defs.Color(COL_BG) : ColGR);
	CurrDisp = target ? target : Disp;
	CurrDisp->MrkMode = MRK_NONE;
	CurrRect.Xmin=GRect.Xmin + DRect.Xmin;	CurrRect.Xmax=GRect.Xmin + DRect.Xmax;
	CurrRect.Ymin=GRect.Ymin + DRect.Ymax;	CurrRect.Ymax=GRect.Ymin + DRect.Ymin;
	if (dirty) DoAutoscale();
	CurrDisp->SetRect(CurrRect,defs.dUnits, &x_axis, &y_axis);
	CurrDisp->setdispx(CurrDisp->un2fix(GRect.Xmin));
	CurrDisp->setdispy(CurrDisp->un2fiy(GRect.Ymin));
	if(!frm_g && !(frm_g = new FrmRect(this, 0L, &GRect, 0L))) return;
	frm_g->SetColor(COL_GRECT, ColGR);	frm_g->SetColor(COL_GRECTLINE, ColGRL);
	frm_g->DoPlot(CurrDisp);
	if(type == GT_STANDARD) {
		if(!frm_d && !(frm_d = new FrmRect(this, &GRect, &DRect, 0L))) return;
		SetMinMaxRect(&rDims, CurrDisp->co2ix(CurrRect.Xmin), CurrDisp->co2iy(CurrRect.Ymax), 
			CurrDisp->co2ix(CurrRect.Xmax), CurrDisp->co2iy(CurrRect.Ymin));
		frm_g->Command(CMD_MINRC, &rDims, CurrDisp);
		SetMinMaxRect(&rDims, CurrDisp->co2ix(GRect.Xmin), CurrDisp->co2iy(GRect.Ymax), 
			CurrDisp->co2ix(GRect.Xmax), CurrDisp->co2iy(GRect.Ymin));
		frm_d->Command(CMD_MAXRC, &rDims, CurrDisp);
		frm_d->SetColor(COL_DRECT, ColDR);		frm_d->DoPlot(CurrDisp);
		frm_g->Command(CMD_SETCHILD, &DRect, CurrDisp);
		}
	else {
		SetMinMaxRect(&rDims, CurrDisp->co2ix(GRect.Xmin), CurrDisp->co2iy(GRect.Ymax),
			CurrDisp->co2ix(GRect.Xmax), CurrDisp->co2iy(GRect.Ymin));
		}
	defs.Idle(CMD_SUSPEND);
	//if there is a grid, draw it first
	if (theGrid) theGrid->DoPlot(CurrDisp);
	//do all axes
	if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]){
		Axes[i]->SetColor(COL_BG, ColGR);
		Axes[i]->DoPlot(CurrDisp);
		}
	//do all plots
	if (Id == GO_GRAPH && bClip) {
		if (type == GT_STANDARD) {
			ClipRC.left = CurrDisp->co2ix(GRect.Xmin + DRect.Xmin);
			ClipRC.top = CurrDisp->co2iy(GRect.Ymin + DRect.Ymin);
			ClipRC.right = CurrDisp->co2ix(GRect.Xmin + DRect.Xmax);
			ClipRC.bottom = CurrDisp->co2iy(GRect.Ymin + DRect.Ymax);
			CurrDisp->ClipRect(&ClipRC);
			if (Plots) for (i = 0; i < NumPlots; i++) if (Plots[i]) {
				if (Plots[i]->Id == GO_GRAPH) ((Graph*)Plots[i])->bClip = 0;
				if (Plots[i]->Id == GO_PLOTSCATT || Plots[i]->Id == GO_STACKBAR) {
					if (((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(CurrDisp);
					}
				else if (Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH) {
					if (Plots[i]->Id == GO_BOXPLOT) CurrPlot = (Plot*)Plots[i];
					else CurrPlot = NULL;
					CurrDisp->ClipRect(&ClipRC);
					if (((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(CurrDisp);
					}
				else {
					if (bClip && (Plots[i]->Id < GO_PLOT || Plots[i]->Id >= GO_GRAPH)) {		//do not clip draw objects
						if(!Plots[i]->hidden) DrawLater.AddObj(Plots[i]);
						}
					else Plots[i]->DoPlot(CurrDisp);
					}
				}
//			CurrDisp->ClipRect(NULL);
			DrawLater.DoPlot(CurrDisp);
			}
		else for (i = 0; i < NumPlots; i++) if (Plots[i]){
			if (Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH) {
				if (((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(CurrDisp);
				}
			else {
				if (!Plots[i]->hidden) Plots[i]->DoPlot(CurrDisp);
				if (Plots[i]->Id >= GO_GRAPH) CurrDisp->ClipRect(0L);
				}
			}
		}
	else if (Plots) {
		if (Id == GO_GRAPH) CurrDisp->ClipRect(NULL);
		for (i = 0; i < NumPlots; i++) if (Plots[i]) {
			if (Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH ) {
				if (((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(CurrDisp);
				}
			else Plots[i]->DoPlot(CurrDisp);
			}
		}
	if(bModified && data) data->Command(CMD_MRK_DIRTY, 0L, 0L);
	if(parent && parent->Id == GO_GRAPH && parent->type == GT_STANDARD) {
		parent->Command(CMD_AXIS, 0L, 0L);
		}
	if(Disp)switch(ToolMode) {
		case TM_STANDARD:
			Disp->MouseCursor(MC_ARROW, true);
			break;
		case TM_DRAW:		case TM_POLYLINE:		case TM_POLYGON:
			Disp->MouseCursor(MC_DRAWPEN, true);
			break;
		case TM_RECTANGLE:
			Disp->MouseCursor(MC_DRAWREC, true);
			break;
		case TM_ELLIPSE:
			Disp->MouseCursor(MC_DRAWELLY, true);
			break;
		case TM_ROUNDREC:
			Disp->MouseCursor(MC_DRAWRREC, true);
			break;
		case TM_ARROW:
			Disp->MouseCursor(MC_CROSS, true);
			break;
		case TM_TEXT:
			Disp->MouseCursor(MC_TEXT, true);
			break;
		case TM_MARK:
			Disp->MouseCursor(MC_CROSS, true);
			break;
		case TM_ZOOMIN:
			Disp->MouseCursor(MC_ZOOM, true);
			break;
		case TM_MOVE:
			Disp->MouseCursor(MC_MOVE, true);
			break;
		case TM_PASTE:
			Disp->MouseCursor(MC_PASTE, true);
			break;
		default:
			break;
		}
	if(CurrDisp) defs.UpdAdd(CurrDisp, 0, 0);
	defs.Idle(CMD_DOPLOT);
}

bool
Graph::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	GraphObj **tmpPlots;
	RECT rc;
	long i, j, k;
	double tmp;
	DWORD delflg = 0L;
	lfPOINT mv_inc;
	lfPOINT p1, p2;

	if(!o) o = Disp ? Disp : CurrDisp;
	switch (cmd){
	case CMD_CAN_CLOSE:
		HideTextCursor();
		if (bModified) {
			if (Undo.isEmpty(CurrDisp)) return true;
			bModified = false;
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SCMG_NOTSAVED, name);
#else
			sprintf(TmpTxt, SCMG_NOTSAVED, name);
#endif
			i = YesNoCancelBox(TmpTxt);
			if (i == 2) return false;
			else if (i == 1) if (!SaveGraphAs(this)) return false;
			}
		return true;
	case CMD_SCROLL_UP:
		if (CurrDisp) CurrDisp->incVPorg(0.0, 16.0);
		DoPlot(CurrDisp);
		return true;
	case CMD_SCROLL_DOWN:
		if (CurrDisp)CurrDisp->incVPorg(0.0, -16.0);
		DoPlot(CurrDisp);
		return true;
	case CMD_SCROLL_LEFT:
		if (CurrDisp)CurrDisp->incVPorg(-16.0, 0.0);
		DoPlot(CurrDisp);
		return true;
	case CMD_SCROLL_RIGHT:
		if (CurrDisp)CurrDisp->incVPorg(16.0, 0.0);
		DoPlot(CurrDisp);
		return true;
		if (CurrDisp)CurrDisp->incVPorg(0.0, 80.0);
		Command(CMD_SETSCROLL, 0L, o);
		return true;
	case CMD_PAGEDOWN:
		if (CurrDisp)CurrDisp->incVPorg(0.0, -80.0);
		Command(CMD_SETSCROLL, 0L, o);
		return true;
	case CMD_TAB:
		if (CurrGO && CurrGO->parent && (CurrGO->parent->parent->Id == GO_GRAPH
			|| CurrGO->parent->parent->Id == GO_STACKBAR || CurrGO->parent->parent->Id == GO_PHENOLOG
			|| CurrGO->parent->parent->Id == GO_FREQDIST)) {
			if (CurrGO->parent->Command(cmd, CurrGO, o)) return true;
			}
		if (CurrDisp)CurrDisp->incVPorg(-80.0, 0.0);
		Command(CMD_SETSCROLL, 0L, o);
		return true;
	case CMD_SHTAB:
		if (CurrGO && CurrGO->parent && (CurrGO->parent->parent->Id == GO_GRAPH
			|| CurrGO->parent->parent->Id == GO_STACKBAR || CurrGO->parent->parent->Id == GO_PHENOLOG
			|| CurrGO->parent->parent->Id == GO_FREQDIST)) {
			if (CurrGO->parent->Command(cmd, CurrGO, o)) return true;
			}
		if (CurrDisp)CurrDisp->incVPorg(80.0, 0.0);
		Command(CMD_SETSCROLL, 0L, o);
		return true;
	case CMD_SAVEAS:
		return SaveGraphAs(this);
	case CMD_SCALE:
		return DoScale((scaleINFO*)tmpl, o);
	case CMD_LAYERS:
		Undo.SetDisp(o ? o : CurrDisp);
		if (ShowLayers(this)) return true;
		return false;
	case CMD_HASSTACK:
		return(NumPlots >= 1 && Plots  && Plots[0]);
	case CMD_SAVEPOS:
		Undo.ValRect(this, &GRect, 0L);
		Undo.ValRect(this, &DRect, UNDO_CONTINUE);
		return true;
	case CMD_LEGEND:
		Undo.SetDisp(o ? o : CurrDisp);
		if (Id == GO_PAGE) {
			if (CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			if (!CurrGraph && NumPlots == 1 && Plots[0] && Plots[0]->Id == GO_GRAPH){
				CurrGraph = (Graph*)Plots[0];
				return CurrGraph->Command(cmd, tmpl, o);
				}
			InfoBox((char*)SCMG_NOGRAPHLEGEND);
			return false;
			}
		if (Id == GO_GRAPH && !tmpl && Plots) {
			for (i = 0; i < NumPlots; i++){
				if (Plots[i] && Plots[i]->Id == GO_LEGEND) {
					tmpl = (void*)Plots[i];
					Undo.ObjConf(Plots[i], 0L);
					break;
					}
				}
			if (!tmpl) {
				if (!(tmpl = (void*) new Legend(this, data)))return false;
				if (Disp) {
					Undo.SetDisp(Disp);
					tmpPlots = (GraphObj**)memdup(Plots, sizeof(GraphObj*) * (NumPlots + 2), 0);
					if (!tmpPlots) return false;
					Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
					Undo.SetGO(this, &tmpPlots[NumPlots++], (Plot *)tmpl, 0L);
					free(Plots);		Plots = tmpPlots;	bModified = true;
					}
				else if ((Plots = (GraphObj**)realloc(Plots, sizeof(GraphObj*) * (NumPlots + 2)))){
					Plots[NumPlots++] = (GraphObj*)tmpl;	Plots[NumPlots] = 0L;
					}
				else return false;
				}
			if (type == GT_CIRCCHART)((Legend*)tmpl)->SetSize(SIZE_XPOS, DRect.Xmin*3.0);
			for (i = 0; i < NumPlots; i++) if (Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			if (Disp) Command(CMD_REDRAW, 0L, CurrDisp);
			}
		else if (Id == GO_GRAPH) {
			for (i = 0; i < NumPlots; i++) if (Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_REPL_GO:
		if (!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if (Axes) for (i = 0; i < NumAxes; i++) if (Axes[i] && Axes[i] == tmpPlots[0]){
			tmpPlots[1]->parent = this;
			tmpPlots[1]->Command(CMD_SET_DATAOBJ, data, o);
			Axes[i] = (Axis *)tmpPlots[1];
			tmpPlots[0]->parent = 0L;		//disable messaging
			//check for default axes
			if (((Axis*)tmpPlots[0])->GetAxis() == &x_axis) {
				if (x_axis.breaks) free(x_axis.breaks);
				memcpy(&x_axis, Axes[i]->axis, sizeof(AxisDEF));
				if (x_axis.owner == Axes[i]) free(Axes[i]->axis);
				Axes[i]->axis = &x_axis;			x_axis.owner = this;
				}
			else if (((Axis*)tmpPlots[0])->GetAxis() == &y_axis) {
				if (y_axis.breaks) free(y_axis.breaks);
				memcpy(&y_axis, Axes[i]->axis, sizeof(AxisDEF));
				if (y_axis.owner == Axes[i]) free(Axes[i]->axis);
				Axes[i]->axis = &y_axis;			y_axis.owner = this;
				}
			DeleteGO(tmpPlots[0]);
			return bModified = dirty = true;
			}
		if (Plots) for (i = 0; i < NumPlots; i++) if (Plots[i] && Plots[i] == tmpPlots[0]) {
			return bModified = dirty = ReplaceGO((GraphObj**)&Plots[i], tmpPlots);
			}
		return false;
	case CMD_MUTATE:
		if (!parent || !(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if (Plots) for (i = 0; i < NumPlots; i++) if (Plots[i] && Plots[i] == tmpPlots[0]) {
			Undo.MutateGO((GraphObj**)&Plots[i], tmpPlots[1], 0L, o);
			if (ToolMode) Command(CMD_TOOLMODE, 0L, o);
			return bModified = true;
			}
		break;
	case CMD_REDRAW:
		if (Disp)CurrDisp = Disp;
		bDialogOpen = false;
		if (CurrDisp) CurrDisp->HideMark();
		if (parent && (parent->Id == GO_PAGE || parent->Id == GO_GRAPH)) return parent->Command(cmd, tmpl, o);
		if (CurrDisp && CurrDisp->Erase(CurrDisp->OC_type == OC_BITMAP ? ColBG : ColGR)) {
			CurrDisp->StartPage();					CurrDisp->ClipRect(0L);
			CurrDisp->MrkMode = MRK_NONE;			DoPlot(CurrDisp);
			CurrDisp->EndPage();
			if (CurrGO && CurrGO == CurrLabel && CurrLabel->Id == GO_LABEL)
				CurrDisp->ShowMark(CurrLabel, MRK_GODRAW);
			defs.UpdRect(CurrDisp, &rDims);
			}
		bDialogOpen = false;
		if (CurrGO == this) CurrGO = 0L;
		defs.Idle(CMD_UPDATE);
		RedrawTextCursor();
		return true;
	case CMD_ZOOM:
		return DoZoom((char*)tmpl);
	case CMD_MOUSECURSOR:
		if (o)o->MouseCursor(PasteObj ? MC_PASTE : MC_ARROW, false);
		return true;
	case CMD_AXIS:			//one of the plots has changed scaling: reset
		if (CurrAxes == Axes) {
			NumCurrAxes = NumAxes;
			if (o)o->SetRect(CurrRect, defs.dUnits, &x_axis, &y_axis);
			}
		return true;
	case CMD_REG_AXISPLOT:	//notification: plot can handle its own axes
		if (nscp > 0 && nscp <= NumPlots && Sc_Plots)  {
			for (i = 0; i < nscp; i++){
				if (Sc_Plots[i] == (GraphObj*)tmpl) return true;
				}
			tmpPlots = (GraphObj**)realloc(Sc_Plots, (nscp + 1)*sizeof(GraphObj*));
			if (tmpPlots){
				tmpPlots[nscp++] = (GraphObj *)tmpl;
				Sc_Plots = tmpPlots;
				}
			else {		//memory allocation error
				nscp = 0;
				Sc_Plots = 0L;
				}
			}
		else {
			Sc_Plots = (GraphObj **)calloc(1, sizeof(GraphObj*));
			if (Sc_Plots){
				Sc_Plots[0] = (GraphObj *)tmpl;
				nscp = 1;
				}
			else nscp = 0;
			}
		return true;
	case CMD_BUSY:
		if (Disp) Disp->MouseCursor(MC_WAIT, true);
		break;
	case CMD_UNDO:
		Command(CMD_TOOLMODE, 0L, o);
		if (CurrDisp) CurrDisp->MouseCursor(MC_WAIT, true);
		Undo.Restore(true, CurrDisp);
		if (CurrDisp) CurrDisp->MouseCursor(MC_ARROW, true);
		return true;
	case CMD_ADDAXIS:
		Undo.SetDisp(o ? o : CurrDisp);		Command(CMD_TOOLMODE, 0L, o);
		if (Id == GO_PAGE) {
			if (CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			if (!CurrGraph && NumPlots == 1 && Plots[0] && Plots[0]->Id == GO_GRAPH){
				CurrGraph = (Graph*)Plots[0];
				return CurrGraph->Command(cmd, tmpl, o);
				}
			InfoBox((char*)SCMG_NOGRAPHAXIS);
			return false;
			}
		else {
			if (type == GT_3D && Plots){
				for (i = 0; i < NumPlots; i++)
					if (Plots[i] && (Plots[i]->Id == GO_PLOT3D || Plots[i]->Id == GO_FUNC3D
						|| Plots[i]->Id == GO_FITFUNC3D))
						return Plots[i]->Command(cmd, tmpl, o);
				}
			else {
				if (AddAxis()) {
					Command(CMD_REDRAW, tmpl, o);		return true;
					}
				else {
					ToolMode = TM_STANDARD;				Command(CMD_TOOLMODE, (void*)(&ToolMode), o);
					return false;
					}
				}
			}
		return false;
	case CMD_CONFIG:
		Command(CMD_TOOLMODE, 0L, o);		if (CurrGO == this) CurrGO = 0L;
		if (bDialogOpen) return false;
		bDialogOpen = true;
		Configure();
		ToolMode = TM_STANDARD;				Command(CMD_TOOLMODE, (void*)(&ToolMode), o);
		bDialogOpen = false;
		return false;
	case CMD_FILENAME:
		if (tmpl) {
			i = (int)strlen((char*)tmpl) + 2;
			filename = (char*)realloc(filename, i);
			rlp_strcpy(filename, i, (char*)tmpl);
			}
		break;
	case CMD_SETNAME:
		if (OwnDisp && CurrDisp && tmpl && *((char*)tmpl)){
			CurrDisp->Caption((char*)tmpl, false);
			i = rlp_strlen((char*)tmpl);
			j = name && name[0] ? rlp_strlen(name) : i;
			name = (char*)realloc(name, i > j ? i + 2 : j + 2);
			rlp_strcpy(name, i + 2, (char*)tmpl);
			}
		else return false;
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_GRAPH;
		if (tmpl) data = (DataObj *)tmpl;
		//do axes
		if (Axes) for (i = 0; i < NumAxes; i++) if (Axes[i]) Axes[i]->Command(cmd, data, o);
		//do all plots
		if (Plots) for (i = 0; i < NumPlots; i++) if (Plots[i]) Plots[i]->Command(cmd, data, o);
		//do grid
		if (theGrid) theGrid->Command(cmd, data, o);	//set the Id property
		return true;
	case CMD_UPDHISTORY:
		if (data) data->Command(CMD_UPDHISTORY, 0L, 0L);
		return true;
	case CMD_UPDATE:
		Command(CMD_TOOLMODE, 0L, o);
		Undo.SetDisp(CurrDisp);
		if (parent && parent->Id != GO_PAGE && parent->Id != GO_GRAPH){
			Undo.ValInt(this, &ToolMode, 0L);		//stub, all plots have UNDO_CONTINUE
			}
		if (CurrDisp) CurrDisp->MouseCursor(MC_WAIT, true);
		for (i = 0; i < NumAxes; i++) if (Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		if (CurrDisp) CurrDisp->MouseCursor(MC_WAIT, false);
		for (i = 0; Plots && i < NumPlots; i++)
			if (Plots[i]) Plots[i]->Command(cmd, (tmpl != 0L ? tmpl : (void*)data), o);
		dirty = bModified = true;		CurrDisp->StartPage();
		if (CurrDisp) CurrDisp->MouseCursor(MC_WAIT, false);
		DoPlot(CurrDisp);		CurrDisp->EndPage();		CurrGO = 0L;
		return true;
	case CMD_DELOBJ:		case CMD_DELOBJ_CONT:
		if (cmd == CMD_DELOBJ_CONT) delflg = UNDO_CONTINUE;
		Command(CMD_TOOLMODE, 0L, o);
		bModified = true;
		if (((GraphObj*)tmpl)->parent != this) return ((GraphObj*)tmpl)->parent->Command(cmd, tmpl, o);
		if (!tmpl) return false;
		for (i = 0; i < NumAxes; i++) if (Axes[i] && (void*)Axes[i] == tmpl){
			if (Axes[i]->Command(CMD_CAN_CLOSE, 0L, o)) {
				Undo.DeleteGO((GraphObj**)(&Axes[i]), delflg, o);
				return Command(CMD_REDRAW, 0L, o);
				}
			else {
				InfoBox((char*)SCMG_NODELSCALEAX);
				return false;
				}
			}
		for (i = 0; Plots && i < NumPlots; i++) if (Plots[i] && (void*)Plots[i] == tmpl) {
			Undo.DeleteGO(&Plots[i], delflg, o);			HideTextCursor();
			Undo.StoreListGO(this, &Plots, &NumPlots, UNDO_CONTINUE);
			for (i = j = 0; i < NumPlots; i++) if (Plots[i]) Plots[j++] = Plots[i];
			NumPlots = j;			//compress list of objects
			return Command(CMD_REDRAW, NULL, o);
			}
		if (tmpl == (void*)frm_g && parent && (parent->Id == GO_PAGE || parent->Id == GO_GRAPH))
			return parent->Command(CMD_DELOBJ_CONT, (void*)this, o);
		return false;
	case CMD_TOOLMODE:
		if (o && tmpl) {
			Undo.SetDisp(o);
			o->CheckMenu(ToolMode, false);
			o->CheckMenu(ToolMode = tmpl ? (*((int*)tmpl)) & 0x0f : TM_STANDARD, true);
			}
		if (tl_pts && tl_nPts) free(tl_pts);
		tl_pts = 0L;	tl_nPts = 0;
		if (CurrDisp) CurrDisp->MrkMode = MRK_NONE;
		if (o) switch (ToolMode & 0x0f) {
		case TM_TEXT:
			o->MouseCursor(MC_TEXT, false);			break;
		case TM_DRAW:		case TM_POLYLINE:		case TM_POLYGON:
			o->MouseCursor(MC_DRAWPEN, false);		break;
		case TM_RECTANGLE:
			o->MouseCursor(MC_DRAWREC, false);		break;
		case TM_ELLIPSE:
			o->MouseCursor(MC_DRAWELLY, false);		break;
		case TM_ROUNDREC:
			o->MouseCursor(MC_DRAWRREC, false);		break;
		case TM_ARROW:
			o->MouseCursor(MC_CROSS, false);		break;
		default:
			o->MouseCursor(MC_ARROW, true);			break;
			}
		if ((ToolMode & 0x0f) != TM_TEXT) KillTextCursor();
		return Command(CMD_REDRAW, 0L, CurrDisp);
	case CMD_ADDPLOT:
		Undo.SetDisp(o ? o : CurrDisp);
		if (Id == GO_PAGE) {
			if (CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			if (!CurrGraph && NumPlots == 1 && Plots[0] && Plots[0]->Id == GO_GRAPH){
				CurrGraph = (Graph*)Plots[0];
				return CurrGraph->Command(cmd, tmpl, o);
				}
			InfoBox((char*)SCMG_NOGRAFORPLOT);
			return false;
			}
		else if (Id == GO_GRAPH) {
			if (type == GT_3D && Plots){
				for (i = 0; i < NumPlots; i++) {
					if (Plots[i] && (Plots[i]->Id == GO_PLOT3D || Plots[i]->Id == GO_FUNC3D
						|| Plots[i]->Id == GO_FITFUNC3D)) {
						if (Plots[i]->Command(cmd, tmpl, CurrDisp)) return Command(CMD_REDRAW, 0L, o);
						else return false;
						}
					}
				return false;
				}
			else {
				j = NumPlots;
				if (Plots && Plots[0]->Id == GO_PHENOLOG) {
					if (((Phenology*)Plots[0])->canAddPlot() < 0){
						ErrorBox((char*)SCMG_NOGRAPHPHEN);
						return false;
						}
					if (AddPlot(0x0)){
						i = ((Phenology*)Plots[0])->canAddPlot();
						k = i * 2 + 1;				//axes are intermittent x and y axes
						if (k >= NumAxes) k--;		//the last plot uses the common x-axis
						if (NumPlots > j) ((Plot*)Plots[j])->use_yaxis = k;
						Command(CMD_REDRAW, 0L, o);					Command(CMD_TOOLMODE, 0L, o);
						return true;
						}
					return false;
					}
				if (AddPlot(0x0)) return true;
				Command(CMD_TOOLMODE, 0L, o);
				ToolMode = TM_STANDARD;				Disp->MouseCursor(MC_ARROW, true);
				return false;
				}
			}
		return false;
	case CMD_MRK_DIRTY:
		return (dirty = true);
	case CMD_ADDTXT:	case CMD_ADDCHAR:	case CMD_ADDCHARW:
		defs.SetDisp(o);
		if (CurrLabel) return CurrLabel->Command(cmd, tmpl, o);
		if (CurrGO)return CurrGO->Command(cmd, tmpl, o);
		if (CurrGraph && type == GT_3D && Plots && NumPlots) return Plots[0]->Command(cmd, tmpl, o);
		return false;
	case CMD_CURRLEFT:	case CMD_CURRIGHT:
	case CMD_BACKSP:	case CMD_POS_FIRST:	case CMD_POS_LAST:
		defs.SetDisp(o);
		if (tmpl && *((int*)tmpl) == 27) {			//Escape
			HideCopyMark(false);
			if (CurrGO && CurrGO->Id == GO_TEXTFRAME) {
				CurrGO->DoMark(o, false);				o->MrkMode = MRK_NONE;
				CurrGO = 0L;
				}
			else o->HideMark();
			CurrLabel = 0L;								HideTextCursor();
			}
		if (CurrLabel) return CurrLabel->Command(cmd, tmpl, o);
		else if (CurrGO && type != GT_3D && (CurrGO->Id == GO_AXIS || CurrGO->moveable) && (cmd == CMD_CURRLEFT || cmd == CMD_CURRIGHT)) {
			mv_inc.fy = 0.0;
			mv_inc.fx = (cmd == CMD_CURRLEFT ? -defs.GetSize(SIZE_MVINC) : defs.GetSize(SIZE_MVINC));
			CurrGO->Command(CMD_MOVE, (void*)&mv_inc, o);
			}
		else if (CurrGO && CurrGO->Id == GO_TEXTFRAME) return CurrGO->Command(cmd, tmpl, o);
		else if (CurrGO && CurrGO->Id != GO_LABEL && tmpl && *((int*)tmpl) == 13) {
			if (CurrGO->PropertyDlg()) {
				ToolMode = TM_STANDARD;
				return Command(CMD_REDRAW, 0L, 0L);
				}
			else {
				ToolMode = TM_STANDARD;
				o->MouseCursor(MC_ARROW, true);
				return false;
				}
			}
		else if (type == GT_3D && Plots) {
			for (i = 0; i < NumPlots; i++) {
				if (Plots[i] && (Plots[i]->Id == GO_PLOT3D || Plots[i]->Id == GO_FUNC3D
					|| Plots[i]->Id == GO_FITFUNC3D))
					return Plots[i]->Command(cmd, tmpl, CurrDisp);
				}
			}
		return false;
	case CMD_CURRUP:	case CMD_CURRDOWN:
		if (CurrLabel && CurrLabel == CurrGO){
			if (CurrLabel->parent && CurrLabel->parent->Id == GO_MLABEL)
				return CurrLabel->parent->Command(cmd, tmpl, o);
			return true;
			}
		else if (CurrGO && CurrGO->Id == GO_TEXTFRAME) return CurrGO->Command(cmd, tmpl, o);
		else if (type == GT_3D && Plots) {
			for (i = 0; i < NumPlots; i++) {
				if (Plots[i] && (Plots[i]->Id == GO_PLOT3D || Plots[i]->Id == GO_FUNC3D
					|| Plots[i]->Id == GO_FITFUNC3D))
					return Plots[i]->Command(cmd, tmpl, CurrDisp);
				}
			}
		else if (CurrGO && (CurrGO->Id == GO_AXIS || CurrGO->moveable)){
			mv_inc.fx = 0.0;	mv_inc.fy = (cmd == CMD_CURRUP ? -defs.GetSize(SIZE_MVINC) : defs.GetSize(SIZE_MVINC));
			CurrGO->Command(CMD_MOVE, (void*)&mv_inc, o);
			}
		return false;
	case CMD_SHIFTLEFT:	case CMD_SHIFTRIGHT:	case CMD_SHIFTUP:	case CMD_SHIFTDOWN:
		if (Id == GO_PAGE || (CurrGraph && CurrGraph->parent == this)) {
			if (CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			else if (CurrGO && CurrGO->Id == GO_TEXTFRAME) return CurrGO->Command(cmd, tmpl, o);
			else if (CurrGO && CurrGO->Id == GO_LABEL) return CurrGO->Command(cmd, tmpl, o);
			}
		else {
			if (type == GT_3D && Plots) {
				for (i = 0; i < NumPlots; i++) {
					if (Plots[i] && (Plots[i]->Id == GO_PLOT3D || Plots[i]->Id == GO_FUNC3D
						|| Plots[i]->Id == GO_FITFUNC3D))
						return Plots[i]->Command(cmd, tmpl, CurrDisp);
					}
				}
			else if (CurrGO && CurrGO->Id == GO_TEXTFRAME) return CurrGO->Command(cmd, tmpl, o);
			else if (CurrGO && CurrGO->Id == GO_LABEL) return CurrGO->Command(cmd, tmpl, o);
			else if (CurrGO && (CurrGO->Id == GO_AXIS || CurrGO->moveable)) {
				mv_inc.fx = mv_inc.fy = 0.0;
				switch (cmd) {
				case CMD_SHIFTLEFT:				case CMD_SHIFTRIGHT:
					mv_inc.fx = (cmd == CMD_SHIFTLEFT ? -defs.GetSize(SIZE_MVINC) / 4.0 : defs.GetSize(SIZE_MVINC)) / 4.0;
					break;
				case CMD_SHIFTUP:				case CMD_SHIFTDOWN:
					mv_inc.fy = (cmd == CMD_SHIFTUP ? -defs.GetSize(SIZE_MVINC) / 4.0 : defs.GetSize(SIZE_MVINC)) / 4.0;
					break;
				default:						return false;
				}
				CurrGO->Command(CMD_MOVE, (void*)&mv_inc, o);
				}
			}
		return false;
	case CMD_MOVE_TOP:	case CMD_MOVE_UP:
	case CMD_MOVE_DOWN:	case CMD_MOVE_BOTTOM:
		Undo.StoreListGO(this, &Plots, &NumPlots, 0L);
		if (MoveObj(cmd, (GraphObj *)tmpl)){
			bModified = true;
			CurrDisp->StartPage();			DoPlot(CurrDisp);
			CurrDisp->EndPage();			return true;
			}
		return false;
	case CMD_DELETE:
		if (!CurrGO) return false;
		bModified = true;
		if (CurrGO->Id == GO_TEXTFRAME) return CurrGO->Command(cmd, tmpl, o);
		if (CurrGO == CurrLabel) return CurrLabel->Command(cmd, tmpl, o);
		if (CurrGO->Id == GO_FRAMERECT) if (!(CurrGO = CurrGO->parent))return false;
		if (CurrGO->parent == this) return Command(CMD_DELOBJ, (void*)CurrGO, o);
		if (CurrGO->parent)return CurrGO->parent->Command(CMD_DELOBJ, (void*)CurrGO, o);
		return false;
	case CMD_DROP_GRAPH:
		if (Disp) {
			tmpPlots = (GraphObj**)memdup(Plots, (NumPlots + 2)*sizeof(GraphObj*), 0);
			if (!(tmpPlots)) return false;
			HideCopyMark(false);	Undo.SetDisp(Disp);
			Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
			free(Plots);			Plots = tmpPlots;	Plots[NumPlots] = Plots[NumPlots + 1] = 0L;
			Undo.SetGO(this, &Plots[NumPlots], (GraphObj*)tmpl, 0L);
			}
		else if ((Plots = (GraphObj**)realloc(Plots, sizeof(GraphObj*) * (NumPlots + 2)))){
			Plots[NumPlots] = (GraphObj*)tmpl;	Plots[NumPlots + 1] = 0L;
			}
		else return false;
		if (Plots[NumPlots]){
			Plots[NumPlots]->parent = this;
			Plots[NumPlots]->Command(CMD_SET_DATAOBJ, data, 0L);
			if (Plots[NumPlots]->Id == GO_GRAPH) CurrGraph = (Graph*)Plots[NumPlots];
			Plots[NumPlots]->moveable = 1;		//all page items should be freely
			}									//   moveable by user
		NumPlots++;
		if (CurrDisp) {
			CurrDisp->StartPage();		DoPlot(CurrDisp);		CurrDisp->EndPage();
			}
		return true;
	case CMD_SAVE_TICKS:		case CMD_SET_GRIDLINE:
		//do all axes
		for (i = 0; Axes && i < NumAxes; i++){
			Axes[i] && Axes[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_DROP_PLOT:
		if (!tmpl) return false;
		HideCopyMark(false);
		((GraphObj*)tmpl)->parent = this;
		if (((GraphObj*)tmpl)->Id < GO_PLOT) {
			if (!NumPlots && Id != GO_PAGE) return false;
			Plots = (GraphObj**)realloc(Plots, (NumPlots + 2)*sizeof(GraphObj*));
			Plots[NumPlots] = (GraphObj*)tmpl;
			NumPlots++;
			if (CurrDisp) {
				CurrDisp->StartPage();
				DoPlot(CurrDisp);
				CurrDisp->EndPage();
				}
			return true;
			}
		if (Id == GO_GRAPH) CurrGraph = this;
		bModified = true;
		if (!NumPlots) {
			Plots = (GraphObj**)calloc(2, sizeof(GraphObj*));
			if (Plots) {
				Plots[0] = (Plot *)tmpl;	Plots[0]->parent = this;
				switch (Plots[0]->Id) {
				case GO_PIECHART:		case GO_RINGCHART:	case GO_STARCHART:
					type = GT_CIRCCHART;
					break;
				case GO_POLARPLOT:
					type = GT_POLARPLOT;
					break;
				case GO_TERNARY:
					type = GT_TERNARY;
					break;
				case GO_TERNARYXYZ:
					type = GT_TERNARYXYZ;
					break;
				case GO_SCATT3D:		case GO_PLOT3D:		case GO_FUNC3D:
				case GO_FITFUNC3D:
					type = GT_3D;
					break;
				default:
					type = GT_STANDARD;
					if (Plots[0]->Id != GO_CONTOUR) bClip = 1;
					break;
					}
				Bounds.Xmin = x_axis.min = ((Plot*)Plots[0])->Bounds.Xmin;
				Bounds.Xmax = x_axis.max = ((Plot*)Plots[0])->Bounds.Xmax;
				Bounds.Ymin = y_axis.min = ((Plot*)Plots[0])->Bounds.Ymin;
				Bounds.Ymax = y_axis.max = ((Plot*)Plots[0])->Bounds.Ymax;
				if (Bounds.Ymax == Bounds.Ymin) {
					if (Bounds.Ymax != 0.0f) {
						Bounds.Ymax = y_axis.max = Bounds.Ymax + (Bounds.Ymax) / 10.0f;
						Bounds.Ymin = y_axis.min = Bounds.Ymin - (Bounds.Ymax) / 10.0f;
					}
					else {
						Bounds.Ymax = y_axis.max = 1.0f;
						Bounds.Ymin = y_axis.min = -1.0f;
					}
				}
				if (Bounds.Xmax == Bounds.Xmin) {
					if (Bounds.Xmax != 0.0f) {
						Bounds.Xmax = x_axis.max = Bounds.Xmax + (Bounds.Xmax) / 10.0f;
						Bounds.Xmin = x_axis.min = Bounds.Xmin - (Bounds.Xmax) / 10.0f;
						}
					else {
						Bounds.Xmax = 1.0f;
						Bounds.Xmin = -1.0f;
						}
					}
				NiceAxis(&x_axis, 4);				NiceAxis(&y_axis, 4);
				NumPlots = 1;
				dirty = false;
				return true;
			}
			return false;
		}
		else {
			if (Disp) {
				Undo.SetDisp(Disp);
				tmpPlots = (GraphObj**)memdup(Plots, sizeof(GraphObj*) * (NumPlots + 2), 0);
				Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
				Undo.SetGO(this, &tmpPlots[NumPlots++], (Plot *)tmpl, 0L);
				free(Plots);			Plots = tmpPlots;
				}
			else if ((Plots = (GraphObj**)realloc(Plots, sizeof(GraphObj*) * (NumPlots + 2)))){
				Plots[NumPlots++] = (Plot*)tmpl;	Plots[NumPlots] = 0L;
				}
			else return false;
			if (Plots && NumPlots && Plots[0]->Id == GO_PHENOLOG) {
				dirty = false;
				//Redraw graph to show all plots
				if (CurrDisp) {
					CurrDisp->StartPage(); 	DoPlot(CurrDisp); 	CurrDisp->EndPage();
					}
				return true;
				}
			if (type == GT_STANDARD && ((x_axis.flags & AXIS_AUTOSCALE) ||
				(y_axis.flags & AXIS_AUTOSCALE))) {
				if (x_axis.flags & AXIS_AUTOSCALE) {
					Bounds.Xmin = x_axis.min = ((Plot *)tmpl)->Bounds.Xmin < Bounds.Xmin ?
						((Plot *)tmpl)->Bounds.Xmin : Bounds.Xmin;
					Bounds.Xmax = x_axis.max = ((Plot *)tmpl)->Bounds.Xmax > Bounds.Xmax ?
						((Plot *)tmpl)->Bounds.Xmax : Bounds.Xmax;
					NiceAxis(&x_axis, 4);
					if (Axes)for (i = 0; i < NumAxes; i++)
						if (Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &x_axis, o);
					}
				if (y_axis.flags & AXIS_AUTOSCALE) {
					Bounds.Ymin = y_axis.min = ((Plot *)tmpl)->Bounds.Ymin < Bounds.Ymin ?
						((Plot *)tmpl)->Bounds.Ymin : Bounds.Ymin;
					Bounds.Ymax = y_axis.max = ((Plot *)tmpl)->Bounds.Ymax > Bounds.Ymax ?
						((Plot *)tmpl)->Bounds.Ymax : Bounds.Ymax;
					NiceAxis(&y_axis, 4);
					if (Axes)for (i = 0; i < NumAxes; i++)
						if (Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &y_axis, o);
					}
				}
			dirty = false;
			//Redraw graph to show all plots
			if (CurrDisp) {
				CurrDisp->StartPage(); 	DoPlot(CurrDisp); 	CurrDisp->EndPage();
				}
			return true;
			}
		break;
	case CMD_PASTE_OBJ:
		if (!tmpl) return false;
		Undo.SetDisp(o ? o : CurrDisp);
		PasteObj = (GraphObj*)tmpl;
		if (PasteObj->Id == GO_GRAPH || PasteObj->Id == GO_POLYLINE || PasteObj->Id == GO_POLYGON
			|| PasteObj->Id == GO_RECTANGLE || PasteObj->Id == GO_ROUNDREC || PasteObj->Id == GO_ELLIPSE
			|| PasteObj->Id == GO_BEZIER) {
			ToolMode = TM_PASTE;			o->MouseCursor(MC_PASTE, false);
			return true;
			}
		PasteObj = 0L;
		return false;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *)tmpl;
		if (CurrGO && CurrGO->moveable && mev->Action == MOUSE_LBDOWN &&
			ToolMode == TM_STANDARD && (mev->StateFlags & 24) == 0  &&
			(TrackGO = (GraphObj*)CurrGO->ObjThere(mev->x, mev->y))){
			ToolMode |= TM_MOVE;
			}
		else if(mev->Action == MOUSE_LBDOWN){
			if(CurrGO && (CurrGO->Id == GO_TEXTFRAME || CurrGO->Id == GO_LABEL) && CurrGO->Command(cmd, tmpl, o)) return true;
			CurrGO = 0L;		rc_mrk.left = mev->x;		rc_mrk.top = mev->y;
			if((ToolMode == TM_TEXT || ToolMode == TM_STANDARD) && Command(CMD_TEXTTHERE, tmpl, o)) {
				o->CheckMenu(TM_TEXT, false);		o->CheckMenu(TM_STANDARD, true);
				ToolMode = TM_STANDARD;				return true;
				}
			}
		if (Id == GO_GRAPH && IsInRect(rDims, mev->x, mev->y)){
			CurrGraph = this;
			}
		if(ToolMode != TM_STANDARD && ExecTool(mev)) return true;
		switch (mev->Action) {
		case MOUSE_RBUP:	case MOUSE_LBDOUBLECLICK:
		case MOUSE_LBUP:	case MOUSE_MOVE:
			defs.SetDisp(CurrDisp);
			return DoMouseEvent(cmd, tmpl, o? o: CurrDisp);
			}
		return false;
	case CMD_TEXTTHERE:
		//do all axes
		for(i = 0; Axes && i< NumAxes; i++)
			if(Axes[i] && Axes[i]->Command(cmd, tmpl,o)) return true;
		//do all plots
		if(Plots)for(i = NumPlots-1; i>=0; i--)
			if(Plots[i] && Plots[i]->Command(cmd, tmpl,o)) return true;
		break;
	case CMD_SETSCROLL:
		if(Notary->IsValidPtr(o)) {
			o->MrkMode = MRK_NONE;
			if(!(o->ActualSize(&rc)))return false;
			i = o->un2iy(GRect.Ymax);
			o->SetScroll(true, -i - (i >> 2), i + (i >> 2), (rc.bottom - rc.top) >> 2, -iround(o->getVPorgY()));
			i = o->un2ix(GRect.Xmax);
			o->SetScroll(false, -i - (i >> 2), i + (i >> 2), (rc.right - rc.left) >> 2, -iround(o->getVPorgX()));
			if (o && o->Erase(o->OC_type == OC_BITMAP ? ColBG : ColGR)) Command(CMD_REDRAW, 0L, o);
			return true;
			}
		return false;
	case CMD_SETHPOS:
		if (Notary->IsValidPtr(o) && tmpl && ((long*)tmpl)){
			tmp = -(double)(*((long*)tmpl));
			if(o->setVPorg(tmp, o->getVPorgY(), o->getVPorgScale())) Command(CMD_SETSCROLL, tmpl, o);
			return true;
			}
		return false;
	case CMD_SETVPOS:
		if (Notary->IsValidPtr(o) && tmpl && ((long*)tmpl)){
			tmp = -(double)(*((long*)tmpl));
			if(o->setVPorg(o->getVPorgX(), tmp, o->getVPorgScale())) Command(CMD_SETSCROLL, tmpl, o);
			return true;
			}
		return false;
	case CMD_OBJTREE:
		for (i = 0; i < NumAxes; i++) if (Axes[i]) {
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Axes[i], 0L);
			}
		for(i = 0; Plots && i < NumPlots; i++) if(Plots[i]) {
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Plots[i], 0L);
			Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_COPY:
		if (Disp) ShowCopyMark(Disp, &((GraphObj*)tmpl)->rDims, 1);
		if (Disp && tmpl) return Disp->CopyObject(tmpl);
		return false;
	case CMD_SNAP:
		if (theGrid) {
			if (theGrid->type & 0x01) theGrid->Snap((POINT*)tmpl, Disp);
			return true;
			}
		return true;
	case CMD_DOSNAP:
		if (theGrid) {
			if (tmpl) theGrid->type |= 0x01;	//enable snap to grid
			else theGrid->type &= (~0x01);
			}
		return true;
	case CMD_MOVE:
		if (CurrGrid && CurrGrid->type & 0x01) {
			p1.fx = p2.fx = Disp->co2fix(GRect.Xmin);
			p1.fy = p2.fy = Disp->co2fix(GRect.Ymin);
			p2.fx += Disp->un2fix(((lfPOINT*)tmpl)[0].fx);
			p2.fy += Disp->un2fiy(((lfPOINT*)tmpl)[0].fy);
			CurrGrid->Snap(&p2, Disp);
			tmp = Disp->fix2un(p2.fx- p1.fx);
			GRect.Xmax += tmp;		GRect.Xmin += tmp;
			tmp = Disp->fiy2un(p2.fy - p1.fy);
			GRect.Ymax += tmp;		GRect.Ymin += tmp;
			}
		else {
			GRect.Xmax += ((lfPOINT*)tmpl)[0].fx;
			GRect.Xmin += ((lfPOINT*)tmpl)[0].fx;
			GRect.Ymax += ((lfPOINT*)tmpl)[0].fy;
			GRect.Ymin += ((lfPOINT*)tmpl)[0].fy;
			}
		return true;
		}
	return false;
}

void *
Graph::ObjThere(int x, int y)
{
	long i;
	void *ptr;

	//do all axes
	for (i = 0; Axes && i < NumAxes; i++){
		if (Axes[i] && (ptr = Axes[i]->ObjThere(x, y))) return ptr;;
		}
	for (i = 0; i < NumPlots; i++){
		if (Plots[i] && (ptr = Plots[i]->ObjThere(x, y))) return ptr;;
		}
	return 0L;
}

double 
Graph::DefSize(int select)
{
	switch(select) {
	case SIZE_LB_XDIST:
	case SIZE_LB_YDIST:			return 0.0f;
	case SIZE_GRECT_TOP:		return GRect.Ymin;
	case SIZE_GRECT_BOTTOM:		return GRect.Ymax;
	case SIZE_GRECT_LEFT:		return GRect.Xmin;
	case SIZE_GRECT_RIGHT:		return GRect.Xmax;
	case SIZE_DRECT_TOP:		return DRect.Ymin;
	case SIZE_DRECT_BOTTOM:		return DRect.Ymax;
	case SIZE_DRECT_LEFT:		return DRect.Xmin;
	case SIZE_DRECT_RIGHT:		return DRect.Xmax;
	case SIZE_BOUNDS_XMIN:		return Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		return Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		return Bounds.Ymin;
	case SIZE_BOUNDS_YMAX:		return Bounds.Ymax;
	case SIZE_SCALE:			return scale > 0.0 ? scale : 1.0;
	case SIZE_BOUNDS_LEFT:		return x_axis.flags & AXIS_INVERT ? x_axis.max : x_axis.min;
	case SIZE_BOUNDS_RIGHT:		return x_axis.flags & AXIS_INVERT ? x_axis.min : x_axis.max;
	case SIZE_BOUNDS_TOP:		return y_axis.flags & AXIS_INVERT ? y_axis.min : y_axis.max;
	case SIZE_BOUNDS_BOTTOM:	return y_axis.flags & AXIS_INVERT ? y_axis.max : y_axis.min;
	case SIZE_YAXISX:
		if(y_axis.flags & AXIS_X_DATA) return CurrDisp->fx2fix(y_axis.loc[0].fx);
		else return CurrDisp->co2fix(y_axis.loc[0].fx);
	case SIZE_XAXISY:
		if(x_axis.flags & AXIS_Y_DATA) return CurrDisp->fy2fiy(x_axis.loc[0].fy);
		else return CurrDisp->co2fiy(x_axis.loc[0].fy);
		}
	if (scale > 0.0) return scale * (CurrDisp ? defs.GetSize(select) : defs.GetSize(select));
	else return defs.GetSize(select);
}

void 
Graph::CheckBounds(double x, double y)
{
	if (x < Bounds.Xmin) Bounds.Xmin = x;
	if (x > Bounds.Xmax) Bounds.Xmax = x;
	if (y < Bounds.Ymin) Bounds.Ymin = y;
	if (y > Bounds.Ymax) Bounds.Ymax = y;
}

void
Graph::DoAutoscale()
{
	int i;
	fRECT oB;

	memcpy(&oB, &Bounds, sizeof(fRECT));
	if(type == GT_STANDARD && ((x_axis.flags & AXIS_AUTOSCALE) || (y_axis.flags & AXIS_AUTOSCALE))) {
		for(i = 0; i < NumPlots; i++) {
			if(Plots[i] && Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH){
				bModified = true;
				if(dirty) {
					if(x_axis.flags & AXIS_AUTOSCALE) {
						Bounds.Xmin = HUGE_VAL;			Bounds.Xmax = -HUGE_VAL;
						}
					if(y_axis.flags & AXIS_AUTOSCALE) {
						Bounds.Ymin = HUGE_VAL;			Bounds.Ymax = -HUGE_VAL;
						}
					dirty = false;
					}
				if (!((Plot*)Plots[i])->hidden) Plots[i]->Command(CMD_AUTOSCALE, 0L, CurrDisp);
				}
			}
		if(Bounds.Xmax <= Bounds.Xmin) {
			Bounds.Xmax = oB.Xmax > oB.Xmin ? oB.Xmax : oB.Xmax + 1.0;
			Bounds.Xmin = oB.Xmin < oB.Xmax ? oB.Xmin : oB.Xmin - 1.0;
			}
		if(Bounds.Ymax <= Bounds.Ymin) {
			Bounds.Ymax = oB.Ymax > oB.Ymin ? oB.Ymax : oB.Ymax + 1.0;
			Bounds.Ymin = oB.Ymin < oB.Ymax ? oB.Ymin : oB.Ymin - 1.0;
			}
		if(x_axis.flags & AXIS_AUTOSCALE){								//rescale x-axis
			x_axis.min = Bounds.Xmin;	x_axis.max = Bounds.Xmax;
			NiceAxis(&x_axis, 4);
			if(x_axis.min <= 0.0 && ((x_axis.flags & 0xf000) == AXIS_LOG ||
				(x_axis.flags & 0xf000) == AXIS_RECI)) {
				x_axis.min = base4log(&x_axis, 0);
				}
			if(Axes)for(i = 0; i < NumAxes; i++)
				if(Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &x_axis, 0L);
			}
		if(y_axis.flags & AXIS_AUTOSCALE){								//rescale y-axis
			y_axis.min = Bounds.Ymin;	y_axis.max = Bounds.Ymax;
			NiceAxis(&y_axis, 4);
			if(y_axis.min <= 0.0 && ((y_axis.flags & 0xf000) == AXIS_LOG ||
				(y_axis.flags & 0xf000) == AXIS_RECI)) {
				y_axis.min = base4log(&y_axis, 1);
				}
			if(Axes)for(i = 0; i < NumAxes; i++)
				if(Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &y_axis, 0L);
			}
		}
	dirty = false;
}

void
Graph::CreateAxes(int templ, int numAxes)
{
	AxisDEF tmp_axis;
	TextDEF label_def, tlbdef;
	char label_text[500];
	Label *label;
	double ts, lb_ydist, lb_xdist, tlb_dist;
	DWORD ptick, ntick, utick;
	char xa_desc[50], ya_desc[50];

	if(Axes && NumAxes) return;
	label_def.ColTxt = defs.Color(COL_AXIS);				label_def.ColBg = 0x00ffffffL;
	label_def.fSize = DefSize(SIZE_TICK_LABELS)*1.2;		label_def.RotBL = label_def.RotCHAR = 0.0;
	label_def.iSize = 0;									label_def.Align = TXA_VTOP | TXA_HCENTER;
	label_def.Mode = TXM_TRANSPARENT;						label_def.Style = TXS_NORMAL;
	label_def.Font = FONT_HELVETICA;						label_def.text = (unsigned char*)label_text;
	tlbdef.ColTxt = defs.Color(COL_AXIS);					tlbdef.ColBg = 0x00ffffffL;
	tlbdef.RotBL = tlbdef.RotCHAR = 0.0;					tlbdef.iSize = 0;
	tlbdef.fSize = DefSize(SIZE_TICK_LABELS);				tlbdef.Align = TXA_VCENTER | TXA_HCENTER;
	tlbdef.Style = TXS_NORMAL;								tlbdef.Mode = TXM_TRANSPARENT;
	tlbdef.Font = FONT_HELVETICA;							tlbdef.text = 0L;
	ts = DefSize(SIZE_AXIS_TICKS);
	rlp_strcpy(xa_desc, 50, (char*)"x-axis");				rlp_strcpy(ya_desc, 50, (char*)"y-axis");
	if(Plots && NumPlots && Plots[0] && Plots[0]->Id >= GO_PLOT && Plots[0]->Id < GO_GRAPH) {
		if(((Plot*)Plots[0])->x_info) rlp_strcpy(xa_desc, 50,((Plot*)Plots[0])->x_info); 
		if(((Plot*)Plots[0])->y_info) rlp_strcpy(ya_desc, 50,((Plot*)Plots[0])->y_info); 
		}
	switch (tickstyle & 0x07){
	case 1:						//ticks inside
		ntick = AXIS_POSTICKS;		ptick = AXIS_POSTICKS;	utick = AXIS_NEGTICKS;
		ts *= 0.5;
		break;
	case 2:						//centered, symetrical
		ptick = ntick = utick = AXIS_SYMTICKS;
		ts *= 0.75;
		break;
	default:					//ticks outside
		ptick = AXIS_NEGTICKS;		ntick = AXIS_NEGTICKS; utick = AXIS_POSTICKS;
		break;
		}
	tlb_dist = NiceValue(ts * 2.0);
	lb_ydist = NiceValue((ts+DefSize(SIZE_AXIS_TICKS))*2.0);
	lb_xdist = NiceValue((ts+DefSize(SIZE_AXIS_TICKS))*3.0);
	switch(templ) {
	case 0:					//standard layout with four axes
		Axes = (Axis**)calloc(5, sizeof(Axis *));
		x_axis.loc[0].fx = DRect.Xmin;
		x_axis.loc[1].fx = DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy =  DRect.Ymax;
		y_axis.loc[0].fy = DRect.Ymin;
		y_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = DRect.Xmin;
		Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick |
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0));
		if (Axes[0]){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			rlp_strcpy(label_text, 500, xa_desc);
			label = new Label(Axes[0], data, (DRect.Xmin + DRect.Xmax) / 2.0,
				DRect.Ymax + DefSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT, 0L);
			if (label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		Axes[1] = new Axis(this, data, &y_axis, y_axis.flags | AXIS_LEFT | ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0));
		if (Axes[1]){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			rlp_strcpy(label_text, 500, ya_desc);
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, DRect.Xmin - DefSize(SIZE_AXIS_TICKS)*6.0,
				(DRect.Ymax + DRect.Ymin) / 2.0, &label_def, LB_X_PARENT, 0L);
			if (label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		memcpy(&tmp_axis, &x_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fy = tmp_axis.loc[1].fy = DRect.Ymax;
		Axes[2] = new Axis(this, data, &tmp_axis, AXIS_TOP | AXIS_NOTICKS | AXIS_AUTOTICK);
		if (Axes[2]){
			Axes[2]->SetSize(SIZE_LB_YDIST, -lb_ydist); 
			Axes[2]->SetSize(SIZE_TLB_YDIST, -tlb_dist); 
			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			Axes[2]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		memcpy(&tmp_axis, &y_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fx = tmp_axis.loc[1].fx = DRect.Xmax;
		(Axes[3] = new Axis(this, data, &tmp_axis, AXIS_RIGHT | AXIS_NOTICKS | AXIS_AUTOTICK));
		if (Axes[3]){
			Axes[3]->SetSize(SIZE_LB_XDIST, lb_xdist); 
			Axes[3]->SetSize(SIZE_TLB_XDIST, tlb_dist); 
			tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
			Axes[3]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		NumAxes = 4;
		break;
	case 1:
		Axes = (Axis**)calloc(7, sizeof(Axis *));
		if(x_axis.Start >= 0.0) x_axis.min = x_axis.Start = -x_axis.Step;
		if(y_axis.Start >= 0.0) y_axis.min = y_axis.Start = -y_axis.Step;
		x_axis.loc[0].fx = DRect.Xmin;
		x_axis.loc[1].fx = DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = 0.0;
		y_axis.loc[0].fy = DRect.Ymin;
		y_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = 0.0;
		(Axes[0] = new Axis(this, data, &x_axis, ptick | AXIS_Y_DATA |
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)));
		if (Axes[0]){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			rlp_strcpy(label_text, 500, xa_desc);
			label_def.Align = TXA_VTOP | TXA_HCENTER;
			label = new Label(Axes[0], data, (DRect.Xmin + DRect.Xmax) / 2.0,
				DRect.Ymax + DefSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT, 0L);
			if (label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		(Axes[1] = new Axis(this, data, &y_axis, ntick | AXIS_AUTOTICK | AXIS_X_DATA |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)));
		if (Axes[1]){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist);
			rlp_strcpy(label_text, 500, ya_desc);
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, DRect.Xmin - DefSize(SIZE_AXIS_TICKS)*6.0,
				(DRect.Ymax + DRect.Ymin) / 2.0, &label_def, LB_X_PARENT, 0L);

			if (label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		memcpy(&tmp_axis, &x_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fy = tmp_axis.loc[1].fy = DRect.Ymax;
		Axes[2] = new Axis(this, data, &tmp_axis, AXIS_TOP | AXIS_NOTICKS | AXIS_AUTOTICK);
		if (Axes[2]){
			Axes[2]->SetSize(SIZE_LB_YDIST, -lb_ydist); 
			Axes[2]->SetSize(SIZE_TLB_YDIST, -tlb_dist);
			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			Axes[2]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		tmp_axis.loc[0].fy = tmp_axis.loc[1].fy = DRect.Ymin;
		Axes[3] = new Axis(this, data, &tmp_axis, AXIS_BOTTOM | AXIS_NOTICKS | AXIS_AUTOTICK);
		if (Axes[3]){
			Axes[3]->SetSize(SIZE_LB_YDIST, lb_xdist); 
			Axes[3]->SetSize(SIZE_TLB_YDIST, tlb_dist); 
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[3]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		memcpy(&tmp_axis, &y_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fx = tmp_axis.loc[1].fx = DRect.Xmin;
		Axes[4] = new Axis(this, data, &tmp_axis, AXIS_LEFT | AXIS_NOTICKS | AXIS_AUTOTICK);
		if (Axes[4]){
			Axes[4]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[4]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[4]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		tmp_axis.loc[0].fx = tmp_axis.loc[1].fx = DRect.Xmax;
		Axes[5] = new Axis(this, data, &tmp_axis, AXIS_RIGHT | AXIS_NOTICKS | AXIS_AUTOTICK);
		if (Axes[5]){
			Axes[5]->SetSize(SIZE_LB_XDIST, lb_xdist); 
			Axes[5]->SetSize(SIZE_TLB_XDIST, tlb_dist); 
			tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
			Axes[5]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		NumAxes = 6;
		break;
	case 2:
		Axes = (Axis**)calloc(3, sizeof(Axis *));
		x_axis.loc[0].fx = DRect.Xmin;
		x_axis.loc[1].fx = DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fy = DRect.Ymin;
		y_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = DRect.Xmin;
		if ((Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick |
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			rlp_strcpy(label_text, 500, xa_desc);
			label = new Label(Axes[0], data, (DRect.Xmin + DRect.Xmax) / 2.0f,
				DRect.Ymax + DefSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT, 0L);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, AXIS_LEFT | ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			rlp_strcpy(label_text, 500, ya_desc);
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, DRect.Xmin - DefSize(SIZE_AXIS_TICKS)*6.0,
				(DRect.Ymax + DRect.Ymin) / 2.0, &label_def, LB_X_PARENT, 0L);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		NumAxes = 2;
		break;
	case 3:
		label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
		Axes = (Axis**)calloc(3, sizeof(Axis *));
		x_axis.loc[0].fx = DRect.Xmin;
		x_axis.loc[1].fx = DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = DRect.Ymin;
		y_axis.loc[0].fy = DRect.Ymin;
		y_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = DRect.Xmin;
		if((Axes[0] = new Axis(this, data, &x_axis, AXIS_TOP | utick |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, -lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, -tlb_dist);
			rlp_strcpy(label_text, 500, xa_desc);
			label = new Label(Axes[0], data, (DRect.Xmin + DRect.Xmax) / 2.0,
				DRect.Ymax + DefSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT, 0L);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, AXIS_LEFT | ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | AXIS_INVERT | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			rlp_strcpy(label_text, 500, ya_desc);
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, DRect.Xmin - DefSize(SIZE_AXIS_TICKS)*6.0,
				(DRect.Ymax + DRect.Ymin) / 2.0, &label_def, LB_X_PARENT, 0L);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		NumAxes = 2;
		break;
	case 4:
		Axes = (Axis**)calloc(3, sizeof(Axis *));
		if(x_axis.Start >= 0.0f) x_axis.min = -x_axis.Step;
		x_axis.loc[0].fx = DRect.Xmin;
		x_axis.loc[1].fx = DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fy = DRect.Ymin;
		y_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = 0.0;
		if((Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			rlp_strcpy(label_text, 500, xa_desc);
			label = new Label(Axes[0], data, (DRect.Xmin + DRect.Xmax) / 2.0,
				DRect.Ymax + DefSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT, 0L);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, ntick | AXIS_AUTOTICK | AXIS_X_DATA |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			rlp_strcpy(label_text, 500, ya_desc);
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, DRect.Xmin - DefSize(SIZE_AXIS_TICKS)*6.0,
				(DRect.Ymax + DRect.Ymin) / 2.0, &label_def, LB_X_PARENT, 0L);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		NumAxes = 2;
		break;
	case 20:	case 21:						//special axis layout for phenology plot
		Axes = (Axis**)calloc(numAxes > 3 ? numAxes : 3, sizeof(Axis *));
		x_axis.loc[0].fx = DRect.Xmin;
		x_axis.loc[1].fx = DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fy = DRect.Ymin;
		y_axis.loc[1].fy = DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = DRect.Xmin / 2.0;
		if ((Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick | (templ == 21 ? AXIS_DATETIME :0) |
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			rlp_strcpy(label_text, 500, xa_desc);
			label = new Label(Axes[0], data, (DRect.Xmin + DRect.Xmax) / 2.0,
				DRect.Ymax + DefSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT, 0L);
			if (label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if (label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			Axes[0]->name = rlp_strdup((char*)"Common x axis");
			}
		if ((Axes[1] = new Axis(this, data, &y_axis, ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist);
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			Axes[1]->name = rlp_strdup((char*)"Master y axis");
			}
		label = 0L;
		NumAxes = 2;
		break;
		}
	if(Plots && Plots[0] && Plots[0]->Id >= GO_PLOT && Plots[0]->Id < GO_GRAPH && NumAxes > 1) {
		if (((Plot*)Plots[0])->x_tv && Axes[0]){
			Axes[0]->atv = ((Plot*)Plots[0])->x_tv->Copy();
			Axes[0]->axis->flags &= ~(AXIS_AUTOSCALE | AXIS_AUTOTICK);
			Axes[0]->axis->min = 0.0;
			}
		else if(((Plot*)Plots[0])->x_dtype == ET_DATETIME)x_axis.flags |= AXIS_DATETIME;
		if (((Plot*)Plots[0])->y_tv && Axes[1]){
			Axes[1]->atv = ((Plot*)Plots[0])->y_tv->Copy();
			Axes[1]->axis->flags &= ~(AXIS_AUTOSCALE | AXIS_AUTOTICK);
			Axes[1]->axis->min = 0.0;
			}
		else if(((Plot*)Plots[0])->y_dtype == ET_DATETIME)y_axis.flags |= AXIS_DATETIME;
		}
}

bool
Graph::DoScale(scaleINFO* sc, anyOutput *o)
{
	int i;
	scaleINFO sc0;

	if(sc->sy.fy <= 0.0) return false;
	GRect.Xmax = sc->sx.fx + GRect.Xmax* sc->sx.fy;
	GRect.Xmin = sc->sx.fx + GRect.Xmin * sc->sx.fy;
	GRect.Ymax = sc->sy.fx + GRect.Ymax * sc->sy.fy;
	GRect.Ymin = sc->sy.fx + GRect.Ymin * sc->sy.fy;
	DRect.Xmax *= sc->sx.fy;	DRect.Xmin *= sc->sx.fy;
	DRect.Ymax *= sc->sy.fy;	DRect.Ymin *= sc->sy.fy;
	if(sc->sx.fy == 1.0 && sc->sx.fy  == sc->sy.fy && sc->sx.fy == sc->sz.fy) return true;
	memcpy(&sc0, sc, sizeof(scaleINFO));
	sc0.sx.fx = sc0.sy.fx = sc0.sz.fx = 0.0;	sc0.sx.fy = sc0.sz.fy = sc->sy.fy;
	if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]) Axes[i]->Command(CMD_SCALE, &sc0, o);
	if(Plots) for(i = 0; i < NumPlots; i++) if(Plots[i]){
		if(Plots[i]->Id == GO_GRAPH || Plots[i]->Id == GO_PAGE) Plots[i]->Command(CMD_SCALE, sc, o);
		else Plots[i]->Command(CMD_SCALE, &sc0, o);
		}
	scale = scale > 0.0 ? scale * sc->sy.fy : sc->sy.fy;
	return true;
}

bool
Graph::DoMouseEvent(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	long i;

	mev = (MouseEvent *)tmpl;		defs.SetDisp(o);
	switch (mev->Action) {
	case MOUSE_LBUP:		case MOUSE_MOVE:
		if (mev->Action == MOUSE_LBUP){
			if (bDialogOpen) {
				bDialogOpen = false;					return false;
			}
			Undo.SetDisp(o);		SuspendAnimation(o, false);
			if (Id == GO_GRAPH && parent && parent->Id == GO_SPREADDATA){
				CurrGO = 0L;			CurrGraph = 0L;
			}
			if (ToolMode != TM_STANDARD) return false;
		}
		if (mev->Action == MOUSE_MOVE && !(mev->StateFlags & 0x01)) return false;
		//do all axes
		for (i = 0; Axes && i< NumAxes; i++)
			if (Axes[i] && Axes[i]->Command(cmd, tmpl, o)) return true;
		//do all plots
		if (Plots && NumPlots > 0) for (i = NumPlots - 1; i >= 0; i--){
			if (Plots[i] && Plots[i]->Command(cmd, tmpl, o)){
				if (Plots[i]->Id != GO_GRAPH && Id == GO_GRAPH) CurrGraph = this;
				return true;
			}
		}
		if ((frm_d && frm_d->Command(cmd, tmpl, o)) || (frm_g && frm_g->Command(cmd, tmpl, o))) {
			if (Id == GO_GRAPH) CurrGraph = this;
			return true;
		}
		if (mev->Action == MOUSE_MOVE && ToolMode == TM_STANDARD &&
			rc_mrk.left >= 0 && rc_mrk.top >= 0) ToolMode = TM_MARK;
		if (!CurrGO) CurrGraph = 0L;
		return false;
	case MOUSE_RBUP:	case MOUSE_LBDOUBLECLICK:
		// the default behaviour for right button click is the same as for
		//   double click: execute properties dialog, just continue.
		if (!o) o = CurrDisp;
		TrackGO = 0L;			Undo.SetDisp(o);
		if (Plots && NumPlots == 1 && IsPlot3D(Plots[0])){
			CurrGO = 0L;
			mev->Action = MOUSE_LBUP;				//fake select
			Plots[0]->Command(cmd, mev, o);
			mev->Action = MOUSE_RBUP;
			Plots[0]->Command(cmd, mev, o);
		}
		if (!CurrGO && o->MrkRect && o->MrkMode == MRK_GODRAW) {
			CurrGO = (GraphObj*)((anyOutput*)o)->MrkRect;
		}
		if (CurrGO && !CurrGO->Id) CurrGO->Command(CMD_SET_DATAOBJ, 0L, o);
		if (!CurrGO) {
			mev->Action = MOUSE_LBUP;				//fake select
			Command(cmd, tmpl, o);
		}
		else if (CurrGO && CurrGO->Id < GO_PLOT && CurrGO->Id > GO_UNKNOWN) {
			if (CurrDisp) CurrDisp->HideMark();
			if (CurrGO->PropertyDlg()) {
				Command(CMD_REDRAW, 0L, o);					Command(CMD_TOOLMODE, 0L, o);
				return (bModified = true);
			}
			else {
				ToolMode = TM_STANDARD;
				Command(CMD_TOOLMODE, 0L, o);
				return true;
			}
		}
		else if (CurrGO->Id > GO_UNKNOWN && CurrGO->Id < GO_SPREADDATA && CurrGO->Command(CMD_CONFIG, 0L, o)){
			ToolMode = TM_STANDARD;				Command(CMD_TOOLMODE, 0L, o);
		}
		else o->HideMark();
		CurrLabel = 0L;			bDialogOpen = false;
		if (CurrGO == this) CurrGO = 0L;
		if (CurrGO) {
			defs.Idle(CMD_FLUSH);
			CurrGO->Command(CMD_SELECT, 0L, o);
		}
		else if (Id == GO_PAGE) {
			Command(CMD_CONFIG, 0L, o);
			Command(CMD_REDRAW, 0L, o);
		}
	}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Pages are graphic objects containing graphs and drawn objects
Page::Page(GraphObj *par, DataObj *d):Graph(par, d, 0L, 0)
{
	FileIO(INIT_VARS);
	cGraphs--;		cPages++;		Id = GO_PAGE;	bModified = true;
	Notary->ValPtr(this, true);
}

Page::Page(int src):Graph(src)
{
	int i;

	//most of the object is read by Graph::FileIO()
	ColBG = 0x00e8e8e8L;
	LineDef.width = 0.0;	LineDef.patlength = 1.0;
	LineDef.color = LineDef.pattern = 0x0L;
	FillDef.type = FILL_NONE;
	FillDef.color = 0x00ffffffL;	//use white paper
	FillDef.scale = 1.0;
	FillDef.hatch = 0L;
	cGraphs--;		cPages++;	bModified = false;
	if(Plots) for(i = 0; i < NumPlots; i++) if(Plots[i]) Plots[i]->moveable = 1;
	Notary->ValPtr(this, true);
}

Page::~Page()
{
	long i;

	Notary->ValPtr(this, false);	Undo.InvalidGO(this);
//	DrawFmtText.SetText(0L, 0L, 0L, 0L, false);
	if (Plots) {
		for (i = 0; i < NumPlots; i++) {
			if (Plots[i]) DeleteGO(Plots[i]);
			}
		free(Plots);		Plots = 0L;		NumPlots = 0;
		}
	if (theGrid) DeleteGO(theGrid);
	theGrid = 0L;
	if (OwnDisp && Disp) DelDispClass(Disp);
	OwnDisp = false;	Disp = 0L;
	if (nscp > 0 && nscp <= NumPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;				Sc_Plots = 0L;
	if (name) free(name);
	name = 0L;				if (filename) free(filename);
	filename = 0L;
}

double
Page::GetSize(int select)
{
	switch (select) {
	case SIZE_GRECT_TOP:		return GRect.Ymin;
	case SIZE_GRECT_BOTTOM:		return GRect.Ymax;
	case SIZE_GRECT_LEFT:		return GRect.Xmin;
	case SIZE_GRECT_RIGHT:		return GRect.Xmax;
	default:
		return Graph::GetSize(select);
		}
	return 0.0;
}

void
Page::DoPlot(anyOutput *o)
{
	int i;
	POINT pts[3];

	if(!o && !Disp) {
		Disp = NewDispClass(this);
		Disp->SetMenu(MENU_PAGE);
#ifdef USE_WIN_SECURE
		i = sprintf_s(TmpTxt, TMP_TXT_SIZE, "Page %d", cPages);
#else
		i = sprintf(TmpTxt, "Page %d", cPages);
#endif
		if(!name && (name = (char*)malloc(i += 2)))rlp_strcpy(name, i, TmpTxt);
		Disp->Caption(TmpTxt, false);
		Disp->setVPorgScale(0.5);
		Disp->CheckMenu(ToolMode, true);
		OwnDisp = true;
		}
	//the first output class is the display class
	if(!Disp && (!(Disp = o))) return;
	CurrDisp = o ? o : Disp;
	CurrDisp->Erase(CurrDisp->OC_type == OC_BITMAP ? ColBG : FillDef.color)		;
	if(OwnDisp && CurrDisp == Disp)LineDef.color = 0x0L;
	else LineDef.color = FillDef.color;
	CurrDisp->SetLine(&LineDef);
	CurrDisp->SetFill(&FillDef);
	CurrDisp->oRectangle(rDims.left = CurrDisp->co2ix(GRect.Xmin), 
		rDims.top = CurrDisp->co2iy(GRect.Ymin), rDims.right = CurrDisp->co2ix(GRect.Xmax),
		rDims.bottom = CurrDisp->co2iy(GRect.Ymax));
	pts[0].x = rDims.left+7;				pts[0].y = pts[1].y = rDims.bottom;
	pts[1].x = pts[2].x = rDims.right;		pts[2].y = rDims.top +3;
	CurrDisp->oPolyline(pts, 3);
	//if there is a grid, draw it first
	if (theGrid) theGrid->DoPlot(CurrDisp);
	//do all plots
	if (Plots) for (i = 0; i < NumPlots; i++) if (Plots[i]) {
		Plots[i]->DoPlot(CurrDisp);
		if(Plots[i]->Id == GO_GRAPH) CurrDisp->ClipRect(0L);
		}
	if(PasteObj) {
		ToolMode = TM_PASTE;	CurrDisp->MouseCursor(MC_PASTE, false);
		}
}

bool
Page::Command(int cmd, void *tmpl, anyOutput *o)
{
	Graph *g;

	switch(cmd) {
	case CMD_MOUSE_EVENT:
		return Graph::Command(cmd, tmpl, o);
	case CMD_REDRAW:
		if(Disp) {
			Disp->StartPage();		DoPlot(Disp);		Disp->EndPage();
			}
		return true;
	case CMD_CONFIG:
		if(Configure()) return true;
		ToolMode = TM_STANDARD;				if(Disp) Disp->MouseCursor(MC_ARROW, true);
		return false;
	case CMD_NEWGRAPH:
		if((g = new Graph(this, data, Disp, 0)) && g->PropertyDlg() && 
			Command(CMD_DROP_GRAPH, g, o))return true;
		else if(g) DeleteGO(g);
		ToolMode = TM_STANDARD;
		Command(CMD_TOOLMODE, (void*)(&ToolMode), o );
		return false;
	case CMD_SET_DATAOBJ:
		Graph::Command(cmd, tmpl, o);
		Id = GO_PAGE;
		if (tmpl) data = (DataObj*)tmpl;
		return true;
	case CMD_SCALE:
		return true;
	default:
		return Graph::Command(cmd, tmpl, o);
	}
	return false;
}

double 
Page::DefSize(int select)
{
	return defs.GetSize(select);
}

