//WinSpec.cpp, Copyright (c) 2000-2025 R.Lackner
//the entire code of this module is highly specific to Windows!
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include <stdio.h>
#include <math.h>
#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags
#include <io.h>					//for read/write
#include "rlplot.h"
#include "WinSpec.h"
#include "rlplot.rc"
#include "TheDialog.h"
#include "menu.h"
#include "rlp_strings.h"

#define RLP_FROZEN_FILE "\\RLP_FROZEN.tmp"

extern int dlgtxtheight;
HINSTANCE hInstance;
HWND MainWnd = 0L;
HACCEL accel;
extern tag_Units Units[];
extern GraphObj *CurrGO;			//Selected Graphic Objects
extern GraphObj *CopyGO;
extern Graph *CurrGraph;
extern Plot *CurrPlot;
extern GraphObj *CurrSS;
extern char *WWWbrowser;
extern char *LoadFile;
extern def_vars defs;
extern char TmpTxt[];
extern UndoObj Undo;
extern void *prog_bar_ptr;
extern tagProgBarDlg *ProgressBarDlg;
extern notary *Notary;

const char name[] = "RLPLOT1";
char *gui_version = 0L;

static unsigned int cf_rlpobj = RegisterClipboardFormat("rlp_obj");
static unsigned int cf_rlpxml = RegisterClipboardFormat("rlp_xml");
//static unsigned int cf_svg = RegisterClipboardFormat(CFSTR_MIME_SVG_XML);
static unsigned int cf_svg = RegisterClipboardFormat("image/svg+xml");
static unsigned int cf_png = RegisterClipboardFormat("image/png");
//static unsigned int cf_eps = RegisterClipboardFormat(CFSTR_MIME_POSTSCRIPT);
static char *ShellCmd = 0L;
static anyOutput *oCopyMark = 0L, *OCrestore = 0L;
static PRINTDLG PriDlg = { 0 };
PrintWin *Printer = 0L;

//forward declaration 
void DoExportPng(GraphObj *g, char *FileName, DWORD flags);

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// I/O File name dialogs
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a new file name to store data in
char *SaveDataAsName(char *oldname)
{
	static char szFile[500], szFileTitle[256];
	static char szFilter[] = "RLPlot workbook (*.rlw)\0*.rlw\0data files (*.csv)\0*.csv\0tab separated (*.tsv)\0"
		"*.tsv\0XML (*.xml)\0*.xml\0";
	static OPENFILENAME ofn;
	int i, j, cb;
	char *ext;

	szFile[0] = '\0';
	if(oldname)rlp_strcpy((void*)szFile, 500, (void*)oldname);
	memset(&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = GetFocus();
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = szFile;
	ofn.nMaxFile = sizeof(szFile);
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = defs.currPath;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
	ofn.lpstrTitle = "Save Data As";

	if(GetSaveFileName(&ofn)){
		if(!(cb = (int)strlen(szFile)) || !szFile[0])return 0L;
		if(cb < 4 || szFile[cb-4] != '.'){
			for(i = j = 0; (j>>1) < (int)(ofn.nFilterIndex-1); i++) {
				if(szFilter[i] == '\0') j++;
				}
			ext = szFilter+i;
			for(i = 0; ext[i] && ext[i] != '*'; i++);
			rlp_strcpy((void*)(szFile+cb), 5, (void*)(ext+i+1));
			}
		defs.FileHistory(szFile);
		return szFile;
		}
	else return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a new file name to store graph
char *SaveGraphAsName(char *oldname)
{
	static char szFile[500], szFileTitle[256];
	static char szFilter[] = "RLPlot Graph (*.rlp)\0*.rlp\0";
	OPENFILENAME ofn;
	int i, j, cb;
	char *ext;

	szFile[0] = '\0';
	if(oldname)rlp_strcpy((void*)szFile, 500, (void*)oldname);
	memset(&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = GetFocus();
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = szFile;
	ofn.nMaxFile = sizeof(szFile);
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = defs.currPath;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
	ofn.lpstrTitle = "Save Graph As";

	if(GetSaveFileName(&ofn)){
		if(!(cb = (int)strlen(szFile)) || !szFile[0])return 0L;
		if(cb < 4 || szFile[cb-4] != '.'){
			for(i = j = 0; (j>>1) < (int)(ofn.nFilterIndex-1); i++) {
				if(szFilter[i] == '\0') j++;
				}
			ext = szFilter+i;
			for(i = 0; ext[i] && ext[i] != '*'; i++);
			rlp_strcpy((void*)(szFile+cb), 5, (void*)(ext+i+1));
			}
		defs.FileHistory(szFile);
		return szFile;
		}
	else return NULL;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a file name to load data
char *OpenDataName(char *oldname)
{
	static char szFile[500], szFileTitle[256];
	static char szFilter[] = "RLPlot workbook (*rlw)\0*.rlw\0data files (*.csv)\0*.csv\0"
		"tab separated file (*.tsv)\0*.tsv\0"
		"RLPlot Graph (*.rlp)\0*.rlp\0all files (*.*)\0*.*\0";
	OPENFILENAME ofn;

	szFile[0] = '\0';
	if(oldname)rlp_strcpy((void*)szFile, 500, (void*)oldname);
	memset(&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = GetFocus();
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = szFile;
	ofn.nMaxFile = sizeof(szFile);
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = defs.currPath;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
	ofn.lpstrTitle = "Open Data File";

	if(GetOpenFileName(&ofn)){
		defs.FileHistory(szFile);
		return szFile;
		}
	else return NULL;
}


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get a file name to export graph
static DWORD last_filter = 1;
void OpenExportName(GraphObj *g, char *oldname)
{
	static char szFile[1024], szFileTitle[512];
	static char szFilter[] = "Scalable Vector Graphics (*.svg)\0*.svg\0"
		"Encapsulated Post Script (*.eps)\0*.eps\0"
		"Portable Network Graphics (*.png)\0*.png\0";
	OPENFILENAME ofn;
	int i;

	szFile[0] = '\0';
	if(!g) return;
	if ((g->Id == GO_GRAPH || g->Id == GO_PAGE) && ((Graph*)g)->getDisp()) (((Graph*)g)->getDisp())->HideMark();
	HideTextCursor();		CurrGO = NULL;
	if(oldname)rlp_strcpy((void*)szFile, 500, (void*)oldname);
	memset(&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = GetFocus();
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = last_filter;
	ofn.lpstrFile = szFile;
	ofn.nMaxFile = sizeof(szFile);
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = defs.currPath;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
	ofn.lpstrTitle = "Export Graph";

	if(g && GetSaveFileName(&ofn)){
		last_filter = ofn.nFilterIndex;
		i = (int)strlen(szFile);
		g->Command(CMD_BUSY, 0L, 0L);
		if(0 == _stricmp(".svg", szFile+i-4)) {
			DoExportSvg(g, szFile, 0L);
			}
		else if(0 == _stricmp(".eps", szFile+i-4)) {
			DoExportEps(g, szFile, 0L);
			}
		else if(0 == _stricmp(".png", szFile + i - 4)) {
			DoExportPng(g, szFile, 0L);
			}
		else {
			switch (ofn.nFilterIndex) {
			case 1:
				rlp_strcpy((void*)(szFile+i), 1024-i, (void*)".svg");
				DoExportSvg(g, szFile, 0L);
				break;
			case 2:
				rlp_strcpy((void*)(szFile + i), 1024 - i, (void*)".eps");
				DoExportEps(g, szFile, 0L);
				break;
			case 3:
				rlp_strcpy((void*)(szFile + i), 1024 - i, (void*)".png");
				DoExportPng(g, szFile, 0L);
				break;
				}
			}
		g->Command(CMD_MOUSECURSOR, 0L, 0L);
		}
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Common alert boxes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void InfoBox(char *Msg)
{
	MessageBox(0, Msg, "Info", MB_OK | MB_ICONINFORMATION);
}

void ErrorBox(char *Msg)
{
	MessageBox(0, Msg, "ERROR", MB_OK | MB_ICONSTOP);
}

bool YesNoBox(char *Msg)
{
	if(IDYES == MessageBox(0, Msg, "RLPlot", MB_YESNO | MB_ICONQUESTION)) return true;
	return false;
}

int YesNoCancelBox(char *Msg)
{
	int res;

	res = MessageBox(0, Msg, "RLPlot", MB_YESNOCANCEL | MB_ICONQUESTION);
	switch(res) {
	case IDYES: return 1;
	case IDNO:	return 0;
	default:	return 2;
		}
	return 0;
}

int AbortRetryIgnoreBox(char *Msg)
{
	int res;

	res = MessageBox(0, Msg, "RLPlot", MB_ABORTRETRYIGNORE | MB_ICONSTOP);
	switch (res) {
	case IDYES: return 1;
	case IDNO:	return 0;
	default:	return 2;
	}
	return 0;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Convert a multibyte character (UTF8) to wide char
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
int rlp_mbtowc(w_char *wc, unsigned char *txt, int n)
{
	int cp;
	w_char *uc;

	if(!(uc=(WCHAR *)calloc(4, sizeof(WCHAR)))) return false;
	cp = MultiByteToWideChar(CP_UTF8, 0, (LPCSTR)txt, n, (LPWSTR)uc, 4);
	*wc = *uc;
	return cp;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Temporary visible objects: show action
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class eph_obj {
public:
	eph_obj(eph_obj *nxt, RECT *bounds, anyOutput *o);
	virtual ~eph_obj();
	virtual void DoPlot(anyOutput *o, bool bShow) {if(next) next->DoPlot(o, bShow);};
	virtual void Animate(anyOutput *o) {if(next) next->Animate(o);};
	virtual void Restore(anyOutput *o, bool bShow);

	int cp, rx, ry, rw, rh, anim_base;
	eph_obj *next;
	POINT *points, p1, p2;
	DWORD lcolor;
	RECT rBounds;
	anyOutput *source;
	bool invert;
};

class eph_line:public eph_obj {
public:
	eph_line(eph_obj *nxt, POINT * pts, int n, DWORD color, RECT *bounds, anyOutput *o);
	~eph_line();
	void DoPlot(anyOutput *o, bool bShow);
};

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
eph_obj::eph_obj(eph_obj *nxt, RECT *bounds, anyOutput *o)
{
	next = nxt;	points = 0L;	lcolor = 0x0;	oCopyMark = source = o;
	if (bounds) memcpy(&rBounds, bounds, sizeof(RECT));
	else rBounds.left = rBounds.top = rBounds.right = rBounds.bottom = 0;
}

eph_obj::~eph_obj()
{
	if(next) delete(next);				next = 0L;
	if(points) free(points);			points = 0L;
}

void
eph_obj::Restore(anyOutput *o, bool bShow)
{
	defs.UpdRect(o, &rBounds);
	if(next)next->Restore(o, false);
	if(bShow) defs.Idle(CMD_UPDATE);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
eph_line::eph_line(eph_obj *nxt, POINT * pts, int n, DWORD color, RECT *bounds, anyOutput *o):eph_obj(nxt, bounds, o)
{
	int i;

	if(points = (POINT*)malloc((n+2) * sizeof(POINT))) {
		memcpy(points, pts, n * sizeof(POINT));
		cp = n;			lcolor = color;
		if(rBounds.left >= rBounds.right || rBounds.top >= rBounds.bottom) {
			rBounds.left = rBounds.right = points[0].x;
			rBounds.top = rBounds.bottom = points[0].y;
			for(i = 1; i < n; i++) UpdateMinMaxRect(&rBounds, points[i].x, points[i].y);
			}
		DoPlot(o, true);
		}
}

eph_line::~eph_line()
{
	if(next) delete(next);				next = 0L;
	if(points) free(points);			points = 0L;
}

void
eph_line::DoPlot(anyOutput *o, bool)
{
	int i;
	HDC dc;
	Gdiplus::Point *pts;

	if (next)next->DoPlot(o, false);
	if (!points || cp < 2) return;
	pts = (Gdiplus::Point *)malloc((cp+2) * sizeof(Gdiplus::Point)+2);
	for (i = 0; i < cp; i++) {
		pts[i].X = points[i].x;			pts[i].Y = points[i].y;
		}
	Gdiplus::Pen pen(GDI_COLOR(lcolor), 1.0);
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	dc = GetDC(((OutputWin*)o)->hWnd);
	Graphics *g = new Graphics(dc);
	g->DrawLines(&pen, pts, cp);
	delete g;
	free(pts);			ReleaseDC(((OutputWin*)o)->hWnd, dc);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Display blinking text cursor
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static anyOutput *oTxtCur = 0L;
RECT rTxtCur = { 0, 0, 0, 0 };
RECT rCopyMark;
static bool bTxtCur = false, bTxtCurIsVis = false, bSuspend = false;
static DWORD cTxtCur = 0x0L;
static HWND hwndTxtCur = 0L;
static int iTxtCurCount = 0;
static POINT ptTxtCurLine[2];
static BitMapWin *bmCopyMark = 0L, *bmCopyBG =0L;
static int TxtCurWidth = 2;

static void DrawCursor(HWND hwnd, DWORD)
{
	HDC dc;
	if (!Notary->IsValidPtr(oTxtCur)) return;
	if (!bTxtCur || !oTxtCur) return;
	static Gdiplus::Point points[2];
	Gdiplus::Pen pen(GDI_COLOR(cTxtCur), (Gdiplus::REAL)TxtCurWidth);
	dc = GetDC(hwnd);
	Graphics *g = new Graphics(dc);
	points[0].X = ptTxtCurLine[0].x;		points[0].Y = ptTxtCurLine[0].y;
	points[1].X = ptTxtCurLine[1].x;		points[1].Y = ptTxtCurLine[1].y;
	g->DrawLine(&pen, points[0], points[1]);
	delete g;
	ReleaseDC(hwnd, dc);
}

void HideTextCursor()
{
	HDC dc;
	RECT updRec;

	memcpy(&updRec, &rTxtCur, sizeof(RECT));
	IncrementMinMaxRect(&updRec, TxtCurWidth+5);
	if (bTxtCur && bTxtCurIsVis && Notary->IsValidPtr(oTxtCur)) {
		bTxtCur = false;
		if (dc = GetDC(((OutputWin*)oTxtCur)->hWnd)) {
			BitBlt(dc, updRec.left, updRec.top, updRec.right - updRec.left, updRec.bottom - updRec.top, 
				((OutputWin*)oTxtCur)->memDC, updRec.left, updRec.top, SRCCOPY);
			ReleaseDC(((OutputWin*)oTxtCur)->hWnd, dc);
			}
		}
	oTxtCur = 0L;		bTxtCur = false;
}

void KillTextCursor()
{
	rTxtCur.left = rTxtCur.top = rTxtCur.right = rTxtCur.bottom = 0;
	oTxtCur = 0L;		bTxtCur = bTxtCurIsVis = false;
}

void ShowTextCursor(anyOutput *out, RECT *disp, DWORD color)
{
	int k1, k2, h;

	if (!Notary->IsValidPtr(out))return;
	if (!out || (out->OC_type & 0xff) != OC_BITMAP) return;
	defs.Idle(CMD_UPDATE);
	k1 = disp->bottom - disp->top;	k2 = disp->right - disp->left;
	h = isqr(k1*k1 + k2*k2);	TxtCurWidth = h / 10;
	if (TxtCurWidth < 2) TxtCurWidth = 2;
	if (TxtCurWidth > 5) TxtCurWidth = 5;
	cTxtCur = color;
	if(bTxtCur) HideTextCursor();
	else KillTextCursor();
	oTxtCur = out;				bSuspend = false;
	iTxtCurCount = 8;
	memcpy(&rTxtCur, disp, sizeof(RECT));
	ptTxtCurLine[0].x = rTxtCur.left;	ptTxtCurLine[0].y = rTxtCur.top;
	ptTxtCurLine[1].x = rTxtCur.right;	ptTxtCurLine[1].y = rTxtCur.bottom;
	rTxtCur.bottom++;		rTxtCur.right++;
	out->ClipRect(NULL);
	DrawCursor(((OutputWin*)out)->hWnd, color);
	bTxtCurIsVis = bTxtCur = true;
}

void RedrawTextCursor()
{
	if (!Notary->IsValidPtr(oTxtCur)) return;
	if (bTxtCurIsVis && bTxtCur && oTxtCur && iTxtCurCount > 5) {
		DrawCursor(((OutputWin*)oTxtCur)->hWnd, cTxtCur);
		}
}

void HideCopyMark(bool)
{
	if (!Notary->IsValidPtr(oCopyMark)) return;
	if (bmCopyMark && oCopyMark) {
		if (bmCopyBG){
			oCopyMark->CopyBitmap(rCopyMark.left, rCopyMark.top, bmCopyBG, 0, 0,
				rCopyMark.right - rCopyMark.left, rCopyMark.bottom - rCopyMark.top, false);
			oCopyMark->UpdateRect(&rCopyMark, true);
			}
		}
	if (bmCopyBG) delete bmCopyBG;	bmCopyBG = 0L;
	bmCopyMark = 0L;	oCopyMark = 0L;
}

void ShowCopyMark(anyOutput *out, RECT *mrk, int nRec)
{
	int i;

	if (!Notary->IsValidPtr(out)) return;
	if (!out || (out->OC_type & 0xff) != OC_BITMAP) return;
	HideCopyMark(true);			bSuspend = false;
	if(!out || !mrk || !nRec) return;
	oCopyMark = out;
	rCopyMark.left = mrk[0].left;	rCopyMark.right = mrk[0].right;
	rCopyMark.top = mrk[0].top;		rCopyMark.bottom = mrk[0].bottom;
	for(i = 1; i < nRec; i++) {
		UpdateMinMaxRect(&rCopyMark, mrk[i].left, mrk[i].top);
		UpdateMinMaxRect(&rCopyMark, mrk[i].right, mrk[i].bottom);
		}
	bmCopyMark = new BitMapWin(rCopyMark.right - rCopyMark.left,
		rCopyMark.bottom - rCopyMark.top, out->hres, out->vres);
	bmCopyBG = new BitMapWin(rCopyMark.right - rCopyMark.left, 
		rCopyMark.bottom - rCopyMark.top, out->hres, out->vres);
	bmCopyBG->CopyBitmap(0, 0, out, rCopyMark.left, rCopyMark.top,
		rCopyMark.right - rCopyMark.left, rCopyMark.bottom - rCopyMark.top, false);
}

void InvalidateOutput(anyOutput *o)
{
	eph_obj *mark = 0L;

	if (!Notary->IsValidPtr(o)) return;
	if(!o || (o->OC_type & 0xff) != OC_BITMAP) return;
	if(oCopyMark && (mark = (eph_obj*)((BitMapWin*)oCopyMark)->ShowObj)){
		mark->Restore(oCopyMark, true);
		delete (mark);
		((BitMapWin*)oCopyMark)->ShowObj = 0L;
		} 
	if(o == oCopyMark) {
		oCopyMark = 0L;
		if(bmCopyMark) delete bmCopyMark;		bmCopyMark = 0L;
		if (bmCopyBG)	delete bmCopyBG;		bmCopyBG = 0L;
		}
	if(o == oTxtCur) {
		oTxtCur = 0L;	bTxtCur = false;
		}
}

void SuspendAnimation(anyOutput *o, bool bSusp)
{
	if(!o || (o->OC_type & 0xff) != OC_BITMAP) return;
	if(!bSusp) bSuspend = false;
	else {
		if(o == oCopyMark) bSuspend = bSusp;
		if(o == oTxtCur) bSuspend = bSusp;
		}
}

static LineDEF liCopyMark1 = {0.0f, 1.0f, 0x00ffffffL, 0L};
static LineDEF liCopyMark2 = {0.0f, 6.0f, 0x0L, 0xf0f0f0f0L};

void InitTextCursor(bool)
{
	if(!hwndTxtCur) {
		DestroyWindow(hwndTxtCur);
		hwndTxtCur = 0L;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process paste command: check for clipboard contents
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
int ClipFormats()
{
#if defined _DEBUG
	UINT cf = 0;
	char format[40];

	cf = EnumClipboardFormats(cf);
	if (cf) OutputDebugString("\nAvailable Clipboard Formats:\n");
	rlp_strcpy((void*)format, 40, (void*)"     ");
	while (cf){
		switch (cf){
		case CF_TEXT:			OutputDebugString("   CF_TEXT");				break;
		case CF_BITMAP:			OutputDebugString("   CF_BITMAP");				break;
		case CF_METAFILEPICT:	OutputDebugString("   CF_METAFILEPICT");		break;
		case CF_SYLK:			OutputDebugString("   CF_SYLK");				break;
		case CF_DIF:			OutputDebugString("   CF_DIF");					break;
		case CF_TIFF:			OutputDebugString("   CF_TIFF");				break;
		case CF_OEMTEXT:		OutputDebugString("   CF_OEMTEXT");				break;
		case CF_DIB:			OutputDebugString("   CF_DIB");					break;
		case CF_PALETTE:		OutputDebugString("   CF_PALETTE");				break;
		case CF_PENDATA:		OutputDebugString("   CF_PENDATA");				break;
		case CF_RIFF:			OutputDebugString("   CF_RIFF");				break;
		case CF_WAVE:			OutputDebugString("   CF_WAVE");				break;
		case CF_UNICODETEXT:	OutputDebugString("   CF_UNICODETEXT");			break;
		case CF_ENHMETAFILE:	OutputDebugString("   CF_ENHMETAFILE");			break;
		default:
			GetClipboardFormatName(cf, format + 3, 79);
			OutputDebugString(format);
			break;
			}
		OutputDebugString("\n");
		cf = EnumClipboardFormats(cf);
		}
#endif
	return 0;
}

void TestClipboard(GraphObj *g)
{
	HANDLE hmem = 0;
	unsigned char *ptr;
	w_char *wc_ptr, wc_single[2];
	long out_pos = NULL, out_size = NULL;
	char text[10],  *out_buff = NULL;
	int i;
	size_t cb;

	if(!g) return;
	OpenClipboard(MainWnd);
	ClipFormats();
	if(g->Id == GO_SPREADDATA) {
		if((hmem = GetClipboardData(cf_rlpxml)) &&
			(ptr = (unsigned char*) GlobalLock(hmem))) g->Command(CMD_PASTE_XML, ptr, 0L);
		else if ((hmem = GetClipboardData(CF_SYLK)) &&
			(ptr = (unsigned char*)GlobalLock(hmem))) g->Command(CMD_PASTE_SYLK, ptr, 0L);
		else if((hmem = GetClipboardData(CF_TEXT)) &&
			(ptr = (unsigned char*) GlobalLock(hmem))) ProcMemData(g, ptr, true);
		else if((hmem = GetClipboardData(cf_rlpobj)) &&
			(ptr = (unsigned char*) GlobalLock(hmem))) OpenGraph(g, 0L, ptr, true);
		}
	else if(g->Id == GO_PAGE || g->Id == GO_GRAPH) {
		if((hmem = GetClipboardData(cf_rlpobj)) &&
			(ptr = (unsigned char*) GlobalLock(hmem))) OpenGraph(g, 0L, ptr, true);
			}
	else if(g->Id == GO_LABEL) {
		if ((hmem = GetClipboardData(CF_UNICODETEXT)) &&
			(wc_ptr = (w_char*)GlobalLock(hmem))) {
			for (i = 0; wc_ptr[i]; i++) {
				wc_single[0] = wc_ptr[i];			wc_single[1] = 0;
				if (wcstombs_s(&cb, text, 8, wc_single, 8)){
					sprintf_s(TmpTxt, 20, "&#%x;", (unsigned int)wc_ptr[i]);
					add_to_buff(&out_buff, &out_pos, &out_size, TmpTxt, 0);
					}
				else {
					add_to_buff(&out_buff, &out_pos, &out_size, text, 0);
					}
				}
			g->Command(CMD_ADDTXT, (void *)out_buff, g->getDisp());
			}
		else if ((hmem = GetClipboardData(CF_TEXT)) && (ptr = (unsigned char*)GlobalLock(hmem))){
			g->Command(CMD_ADDTXT, ptr, g->getDisp());
			}
		if (out_buff) free(out_buff);
		}
	else TestClipboard(g->parent);
	if(hmem) GlobalUnlock(hmem);
	CloseClipboard();
}

void EmptyClip()
{
	HideCopyMark(false);
	OpenClipboard(MainWnd);
	EmptyClipboard();
	CloseClipboard();
}

void CopyText(char *txt, int len, unsigned int cf)
{
	HGLOBAL hmem;
	unsigned char* buf;

	if(!txt || !txt[0]) return;
	if(!len) len = (int)strlen(txt);
	OpenClipboard(MainWnd);
	EmptyClipboard();
	if(hmem = GlobalAlloc(GMEM_MOVEABLE, len+2)) {
		if(buf = (unsigned char *)GlobalLock(hmem)) {
			memcpy(buf, txt, len);	buf[len] = 0;
			GlobalUnlock(hmem);
			SetClipboardData(cf, hmem);
			}
		}
	CloseClipboard();
}

unsigned char* PasteText()
{
	HANDLE hmem = 0;
	unsigned char *ptr, *ret=0L;
	
	OpenClipboard(MainWnd);
	if((hmem = GetClipboardData(CF_TEXT)) && (ptr = (unsigned char*) GlobalLock(hmem))){
		ret = (unsigned char*) _strdup((char*)ptr);
		}
	if(hmem) GlobalUnlock(hmem);
	CloseClipboard();
	return ret;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get display (desktop) size
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void GetDesktopSize(int *width, int *height)
{
	RECT rc;

	GetClientRect(GetDesktopWindow(), &rc);
	*width = rc.right - rc.left;
	*height = rc.bottom - rc.top;
	if(*width < 800 || *height < 600){
		*width = 800;		*height = 600;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Common code for any Windows output class
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool com_oTextOutW(int x, int y, w_char *txt, int cb, HFONT *hFont, HDC *dc, 
	TextDEF *td, anyOutput *o)
{
	XFORM xf;
	int ix, iy, dtflags;
	double w, h;
	double rx, ry;
	double si, csi;
	RECT dtrc;
	fRECT rc;
	TextDEF ttd;


	if (!*hFont || !txt || !txt[0] || !Notary->IsValidPtr(o)) return false;
	if(cb < 1) cb = (int)wcslen(txt);
	//test for transparency
	if(((td->ColTxt & 0xff000000L) || (td->ColBg & 0xff000000L)) && (o->OC_type & 0xff) == OC_BITMAP) {
		o->oGetTextExtentW(txt, cb, &rx, &ry);
		rx += 4;	rc.Xmin = -2.0;		rc.Ymin = 0.0;		rc.Xmax = rx;		rc.Ymax = ry;
		si = sin(td->RotBL *0.01745329252);	csi = cos(td->RotBL *0.01745329252);
		if(td->Align & TXA_HCENTER) {
			rc.Xmin -= rx/2.0-1.0;		rc.Xmax -= rx/2.0-1.0;
			}
		else if(td->Align & TXA_HRIGHT) {
			rc.Xmin -= rx-2.0;			rc.Xmax -= rx-2.0;
			}
		if(td->Align & TXA_VCENTER) {
			rc.Ymin -= ry/2.0;			rc.Ymax -= ry/2.0;
			}
		else if(td->Align & TXA_VBOTTOM) {
			rc.Ymin -= ry;				rc.Ymax -= ry;
			}
		SetMinMaxRect(&((BitMapWin*)o)->tr_rec, iround(rc.Xmin*csi + rc.Ymin*si)+x, iround(rc.Ymin*csi - rc.Xmin*si)+y,
			iround(rc.Xmax*csi + rc.Ymin*si)+x, iround(rc.Ymin*csi - rc.Xmax*si)+y);
		UpdateMinMaxRect(&((BitMapWin*)o)->tr_rec, iround(rc.Xmax*csi + rc.Ymax*si)+x, iround(rc.Ymax*csi - rc.Xmax*si)+y);
		UpdateMinMaxRect(&((BitMapWin*)o)->tr_rec, iround(rc.Xmin*csi + rc.Ymax*si)+x, iround(rc.Ymax*csi - rc.Xmin*si)+y);
		IncrementMinMaxRect(&((BitMapWin*)o)->tr_rec, (td->iSize>>1) +6);
		((BitMapWin*)o)->tr_out = GetRectBitmap(&((BitMapWin*)o)->tr_rec, o);
		((BitMapWin*)o)->tr_out->hres = ((BitMapWin*)o)->hres;
		((BitMapWin*)o)->tr_out->vres = ((BitMapWin*)o)->vres;
		memcpy(&ttd, td, sizeof(TextDEF));
		ttd.ColTxt = td->ColTxt & 0x00ffffffL;	ttd.ColBg = td->ColBg & 0x00ffffffL;
		((BitMapWin*)o)->tr_out->SetTextSpec(&ttd);
		((BitMapWin*)o)->tr_out->oTextOutW(x-((BitMapWin*)o)->tr_rec.left, y-((BitMapWin*)o)->tr_rec.top,
			txt, cb);
		return true;
		}
	SelectObject(*dc, *hFont);					SetTextColor(*dc, (td->ColTxt)&0x00ffffffL);
	SetBkColor(*dc, (td->ColBg)&0x00ffffffL);	SetBkMode(*dc, td->Mode ? TRANSPARENT : OPAQUE);
	ix = iy = 0;								SetTextAlign(*dc, TA_LEFT | TA_TOP);
	if(td->Style & TXS_SUB) {
		if((td->Align & TXA_VCENTER) == TXA_VCENTER) iy += o->un2iy(td->fSize*0.4);
		else if((td->Align & TXA_VBOTTOM) == TXA_VBOTTOM) iy += o->un2iy(td->fSize*0.2);
		else if((td->Align & TXA_VTOP) == TXA_VTOP) iy += o->un2iy(td->fSize*.6);
		}
	else if(td->Style & TXS_SUPER) {
		if((td->Align & TXA_VCENTER) == TXA_VCENTER) iy -= o->un2iy(td->fSize*0.4);
		else if((td->Align & TXA_VBOTTOM) == TXA_VBOTTOM) iy -= o->un2iy(td->fSize*0.6);
		else if((td->Align & TXA_VTOP) == TXA_VTOP) iy += o->un2iy(td->fSize*.0);
		}
	else {
		if((td->Align & TXA_VBOTTOM) == TXA_VBOTTOM) iy -= td->iSize;
		else if((td->Align & TXA_VCENTER) == TXA_VCENTER) iy -= (td->iSize>>1);
		}
	dtflags = DT_NOCLIP | DT_NOPREFIX;
	o->oGetTextExtentW(txt, cb, &w, &h);
	if((td->Align & TXA_HCENTER) == TXA_HCENTER) {
		dtrc.left = x+ix- iround(w/2.0);	dtflags |= DT_CENTER;
		}
	else if((td->Align & TXA_HRIGHT) == TXA_HRIGHT) {
		dtrc.left = x+ix-iround(w);			dtflags |= DT_RIGHT;
		}
	else {
		dtrc.left = x+ix;			dtflags |= DT_LEFT;
		}
	dtrc.top = iy + y;				dtrc.right = dtrc.left+iround(w);		dtrc.bottom = dtrc.top+iround(h);
	if(fabs(td->RotBL) >.01 || fabs(td->RotCHAR) >.01) {
		SetGraphicsMode(*dc, GM_ADVANCED);
		xf.eM11 = xf.eM22 = (float)cos(td->RotBL *0.01745329252);
		xf.eM12 = (float)-sin(td->RotBL *0.01745329252);
		xf.eM21 = -xf.eM12;
		xf.eDx = (float)x;		xf.eDy = (float)y;			SetWorldTransform(*dc, &xf);
		dtrc.left -= x;		dtrc.right -= x;
		dtrc.top = iy;		dtrc.bottom = dtrc.top + iround(h);
		DrawTextW(*dc, txt, cb, &dtrc, dtflags);
		ModifyWorldTransform(*dc, &xf, MWT_IDENTITY);
		SetGraphicsMode(*dc, GM_COMPATIBLE);
		return true;
		}
	else {
		DrawTextW(*dc, txt, cb, &dtrc, DT_CENTER | DT_NOCLIP | DT_NOPREFIX);
		return true;
		}
	return false;
}

bool com_oTextOut(int x, int y, unsigned char *atxt, int cb, HFONT *hFont, HDC *dc, TextDEF *td, anyOutput *o)
{
	unsigned char *utxt = (unsigned char*)atxt;
	w_char *uc;
	int i, n;
	bool bRet;

	if(!*hFont || !atxt || !atxt[0]) return false;
	if(cb < 1) cb = (int)strlen((char*)atxt);
	if(!(uc=(WCHAR *)calloc(cb+20, sizeof(WCHAR)))) return false;
	if(td->Font==FONT_GREEK) {
		for(i = 0; utxt[i]; i++) {
			if((utxt[i] >= 'A' && utxt[i] <= 'Z')) uc[i] = utxt[i] - 'A' + 0x391;
			else if((utxt[i] >= 'a' && utxt[i] <= 'z')) uc[i] = utxt[i] - 'a' + 0x3B1;
			else uc[i] = utxt[i];
			}
		}
	else if(n = hasUTF(atxt, NULL)) {		//process UTF8
		n = MultiByteToWideChar(CP_UTF8, 0, (LPCSTR)atxt, cb, (LPWSTR)uc, 1000);
		bRet = com_oTextOutW(x, y, uc, n, hFont, dc, td, o);
		free(uc);		return bRet;
		}
	else for(i = 0; utxt[i]; i++) uc[i] = utxt[i];
	bRet = com_oTextOutW(x, y, uc, cb, hFont, dc, td, o);
	free(uc);		return bRet;
}

bool com_SetTextSpec(TextDEF *set, anyOutput *o, HFONT *hFont,	TextDEF *TxtSet, 
	HDC *dc)
{	
	bool IsModified, RetVal;
	LOGFONT FontRec;
	HFONT newFont, oldFont;

	if(!set->iSize && set->fSize > 0.001) set->iSize = o->un2iy(set->fSize);
	if(!set->iSize) return false;
	if(!*hFont || TxtSet->iSize != set->iSize || TxtSet->Style != set->Style ||
		TxtSet->RotBL != set->RotBL || TxtSet->RotCHAR != set->RotCHAR ||
		TxtSet->Font != set->Font || TxtSet->fSize != set->fSize) IsModified = true;
	else IsModified = false;
	RetVal = o->anyOutput::SetTextSpec(set);
	if (IsModified && RetVal) {
		// create font
		if((TxtSet->Style & TXS_SUPER) || (TxtSet->Style & TXS_SUB))
			FontRec.lfHeight = o->un2iy(set->fSize*0.71);
		else FontRec.lfHeight = TxtSet->iSize;
		if(FontRec.lfHeight <2) FontRec.lfHeight = 2;
		FontRec.lfWidth = 0;
		FontRec.lfEscapement = 0;		//text angle
		FontRec.lfOrientation = 0;		//base line angle
		FontRec.lfWeight = (TxtSet->Style & TXS_BOLD) ? FW_BOLD : FW_NORMAL;
		FontRec.lfItalic = (TxtSet->Style & TXS_ITALIC) ? TRUE : FALSE;
		FontRec.lfUnderline = (TxtSet->Style & TXS_UNDERLINE) ? TRUE : FALSE;
		FontRec.lfStrikeOut = 0;
		FontRec.lfOutPrecision = OUT_DEFAULT_PRECIS;
		FontRec.lfClipPrecision = CLIP_DEFAULT_PRECIS;
		FontRec.lfQuality = PROOF_QUALITY;
		switch(TxtSet->Font){
		case FONT_HELVETICA:
		default:
			FontRec.lfCharSet = ANSI_CHARSET;
			FontRec.lfPitchAndFamily = VARIABLE_PITCH | FF_SWISS;
			rlp_strcpy((void*)FontRec.lfFaceName, 32, (void*)"Arial");
			break;
		case FONT_GREEK:		case FONT_TIMES:
			FontRec.lfCharSet = ANSI_CHARSET;
			FontRec.lfPitchAndFamily = VARIABLE_PITCH | FF_ROMAN;
			rlp_strcpy((void*)FontRec.lfFaceName, 32, (void*)"Times New Roman");
			break;
		case FONT_COURIER:
			FontRec.lfCharSet = ANSI_CHARSET;
			FontRec.lfPitchAndFamily = FIXED_PITCH | FF_MODERN;
			rlp_strcpy((void*)FontRec.lfFaceName, 32, (void*)"Courier New");
			break;
			}
		newFont = CreateFontIndirect(&FontRec);
		oldFont = (HFONT)SelectObject(*dc, newFont);
		*hFont = newFont;
		if(!(*hFont)) return false;
		}
	return RetVal;
}

bool com_oGetTextExtentW(w_char *text, int cb, double *width, double *height, HDC dc, TextDEF *TxtSet)
{
	SIZE TextExtent;
	double si, csi, d;

	if (!text || !TxtSet || !text[0]) {
		*width = 0.0;		*height = 0.0;
		return false;
		}
	if (!GetTextExtentPoint32W(dc, text, cb ? cb : (int)wcslen(text), &TextExtent)) {
		*width = 0.0;		*height = 0.0;
		return false;
		}
	if(fabs(TxtSet->RotBL) >0.01) {
		si = fabs(sin(TxtSet->RotBL * 0.01745329252));	csi = fabs(cos(TxtSet->RotBL * 0.01745329252));
		d = si > csi ? 1.0/si : 1.0/csi;
		d = (TextExtent.cx * ((7.0 + d)/8.0));
		TextExtent.cx = iround(d);
		}
	*width = TextExtent.cx;					*height = TextExtent.cy;
	return true;
}

bool com_oGetTextExtent(unsigned char *text, int cb, double *width, double *height, HDC dc, TextDEF *TxtSet)
{
	int i;
	unsigned char *utext;
	w_char *uc;
	bool bRet;

	if (!text || !TxtSet || !text[0]) {
		*width = 0.0;		*height = 0.0;
		return false;
		}
	if(!cb) cb = (int)strlen((char*)text);
	if(!(uc = (w_char *) malloc((cb+2)*sizeof(w_char))))return false;
	for(i = 0, utext = (unsigned char*)text ; i <= cb; i++) uc[i] = utext[i];
	uc[i] = 0;
	bRet = com_oGetTextExtentW(uc, cb, width, height, dc, TxtSet);
	free(uc);	
	return bRet;
}

bool
com_CircGrad(int cx, int cy, int r, POINT *pts, int cp, char *nam, anyOutput *o, HDC dc)
{
	int i, j;
	double v[3], vec[3], cf, cfx, cfy, tmp;
	RECT ibounds = { 0, 0, 0, 0 };
	Gdiplus::PointF point1;
	Gdiplus::Color preset[7];
	Gdiplus::REAL positions[] = { 0.0f, 0.2f, 0.4f, 0.6f, 0.8f, 1.0f, 1.0f };
	Gdiplus::Point *points;
	Gdiplus::Rect rect;
	ibounds.left = cx - r;				ibounds.right = cx + r;
	ibounds.top = cy - r;				ibounds.bottom = cy + r;
	v[0] = 0.0;		v[1] = 1.0;		v[2] = 0.0;
	for (i = 0; i < 3; i++) for (j = 0, vec[i] = 0.0; j < 3; j++)
		vec[i] += (o->light_vec[i][j] * v[j]);
	point1.X = Gdiplus::REAL((double)cx + vec[0] * ((double)r));
	point1.Y = Gdiplus::REAL((double)cy - vec[2] * ((double)r));
	cf = ((double)point1.X - (double)cx) / (double)r;
	tmp = (((double)point1.Y - (double)cy) / (double)r);
	cf = cf * cf + tmp * tmp;
	for (i = 0; i < 6; i++){		//no color gradien on lower hemisphere
		positions[i] = positions[i] < (Gdiplus::REAL)1.0 ? positions[i] + positions[i] * ((Gdiplus::REAL)cf) : (Gdiplus::REAL)1.0;
		}
	for (i = 0; i < 6; i++) {
		tmp = asin(positions[i]*0.95);
		preset[i] = GDI_COLOR(IpolCol(o->dFillCol2, o->dFillCol, tmp < 1.0 ? tmp : 1.0));
		}
	preset[i] = GDI_COLOR(o->dFillCol);
	Graphics *g = new Graphics(dc);
	g->SetPageUnit(UnitPixel);
	GraphicsPath *path = new GraphicsPath();
	cfx = (double)(cx - iround(point1.X));		cfy = (double)(cy - iround(point1.Y));
	cf = cfx > cfy ? cfx : cfy;							//keep circular
	cf *= 0.707;
	rect.X = ibounds.left - iround(cf);			rect.Width = ibounds.right - ibounds.left + iround(cf*2.0);
	rect.Y = ibounds.top - iround(cf);			rect.Height = ibounds.bottom - ibounds.top + iround(cf*2.0);
	path->AddEllipse(rect);
	PathGradientBrush brush(path);
	brush.SetInterpolationColors(preset, positions, 7);
	brush.SetCenterPoint(point1);
	if (pts) {							// outline polygon
		points = (Gdiplus::Point *)malloc((cp + 2) * sizeof(Gdiplus::Point));
		Gdiplus::Pen pen(GDI_COLOR(o->dLineCol), (Gdiplus::REAL)o->un2fiy(o->LineWidth*0.7));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinRound);
		for (i = 0; i < cp; i++) {
			points[i].X = pts[i].x;			points[i].Y = pts[i].y;
			}
		if (points[i - 1].X != points[0].X || points[i - 1].Y != points[0].Y){
			points[i].X = pts[0].x;				points[i++].Y = pts[0].y;	//close polygon
			}
		g->FillPolygon(&brush, points, cp);
		if ((o->oLine.color & 0xff000000) != 0xff000000) {
			g->DrawLines(&pen, points, i);
			}
		free(points);
	}
	else {
		g->FillEllipse(&brush, cx - r, cy - r, r << 1, r << 1);
		if ((o->oLine.color & 0xff000000) != 0xff000000) {
			Gdiplus::Pen pen(GDI_COLOR(o->oLine.color), (Gdiplus::REAL)o->un2fiy(o->oLine.width*0.7));
			pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
			pen.SetLineJoin(LineJoin::LineJoinRound);
			g->DrawEllipse(&pen, cx - r, cy - r, r << 1, r << 1);
		}
	}
	delete path;	delete g;
	return true;
}

bool
com_GradPG(fPOINT3D *pts, long npt, double, double, double pLo, double pHi, lfPOINT *grad, anyOutput *o, HDC dc)
{
	long i, lpts;
	Gdiplus::PointF point1, point2, *points;
	double o_min, o_max, lc, hc;
	int rb_i;					//indices to raibow and pos
	FillDEF fill;

	point1.X = (Gdiplus::REAL)grad[0].fx;			point1.Y = (Gdiplus::REAL)grad[0].fy;
	point2.X = (Gdiplus::REAL)grad[1].fx;			point2.Y = (Gdiplus::REAL)grad[1].fy;
	o->GetFill(&fill);
	Gdiplus::Color preset[10];						Gdiplus::REAL positions[10];
	Gdiplus::PointF cp;

	if (npt < 3) return false;
	o_min = o->LoVal;					o_max = o->HiVal*1.07;
	lc = fmax((pLo - o_min) / (o_max - o_min), 0.0);
	hc = fmin((pHi - o_min) / (o_max - o_min), 1.0);
	Graphics *g = new Graphics(dc);					g->SetPageUnit(UnitPixel);
	points = (Gdiplus::PointF *)malloc((npt + 2) *sizeof(Gdiplus::PointF));
	for (i = 0, lpts = npt; i < npt; i++){
		points[i].X = (Gdiplus::REAL)pts[i].fx ;			points[i].Y = (Gdiplus::REAL)pts[i].fy;
		}
	if (points[i - 1].X != points[0].X || points[i - 1].Y != points[0].Y) {		//close PG?
		points[i].X = points[0].X;					points[i].Y = points[0].Y;
		lpts++;
		}
	if (fill.type == FILL_GRADIENT && (grad[0].fx != grad[1].fx || grad[0].fy != grad[1].fy)){
		//simple color to color gradient
		rb_i = 0;
		Gdiplus::LinearGradientBrush brush(point1, point2, Gdiplus::Color(255, 0, 0), Gdiplus::Color(0, 0, 255));
		preset[rb_i] = GDI_COLOR(IpolCol(o->dFillCol2, o->dFillCol, lc));
		positions[rb_i++] = 0.0;
		preset[rb_i] = GDI_COLOR(IpolCol(o->dFillCol2, o->dFillCol, hc));
		positions[rb_i++] = 1.0;
		brush.SetInterpolationColors(preset, positions, rb_i);
		g->FillPolygon(&brush, points, npt);
		}
	else if (grad[0].fx != grad[1].fx || grad[0].fy != grad[1].fy){
		//rainbow colors
		rb_i = 0;
		Gdiplus::LinearGradientBrush brush(point1, point2, Gdiplus::Color(255, 0, 0), Gdiplus::Color(0, 0, 255));
		preset[rb_i] = GDI_COLOR(GradColor(2, o_min, o_max, pLo));
		positions[rb_i++] = 0.0;
		preset[rb_i] = GDI_COLOR(GradColor(2, o_min, o_max, pLo*0.75 + pHi*0.25));
		positions[rb_i++] = 0.25;
		preset[rb_i] = GDI_COLOR(GradColor(2, o_min, o_max, (pLo + pHi)*0.5));
		positions[rb_i++] = 0.5;
		preset[rb_i] = GDI_COLOR(GradColor(2, o_min, o_max, pLo*0.25 + pHi*0.75));
		positions[rb_i++] = 0.75;
		preset[rb_i] = GDI_COLOR(GradColor(2, o_min, o_max, pHi));
		positions[rb_i++] = 1.0;
		brush.SetInterpolationColors(preset, positions, rb_i);
		g->FillPolygon(&brush, points, npt);
		}
	if ((o->dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(o->dLineCol), (Gdiplus::REAL)o->un2fiy(o->LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(npt < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
		g->DrawLines(&pen, points, lpts);
		}
	free(points);
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Output to windows bitmap
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BitMapWin::BitMapWin(GraphObj *g, HWND hw):anyOutput()
{
	HDC dc;
	HWND hwndDesk;

	memDC = 0L;		hgo = 0L;		go = g;
	dc = GetDC(hwndDesk = GetDesktopWindow());
	hres = (double)GetDeviceCaps(dc, LOGPIXELSX);
	vres = (double)GetDeviceCaps(dc, LOGPIXELSY);
	if(hw) GetClientRect(hw, &DeskRect);
	else GetClientRect(hwndDesk, &DeskRect);
	Box1.Xmin = DeskRect.left;		Box1.Xmax = DeskRect.right;
	Box1.Ymin = DeskRect.top;		Box1.Ymax = DeskRect.bottom;
	scr = CreateCompatibleBitmap(dc, DeskRect.right, DeskRect.bottom);
	memDC = CreateCompatibleDC(NULL);
	SelectObject(memDC, scr);		ReleaseDC(hwndDesk, dc);
	hPen = CreatePen(PS_SOLID, 1, dLineCol = 0x00ffffffL);
	hBrush = CreateSolidBrush(dFillCol =dBgCol = 0x00ffffffL);
	dPattern = 0L;				minLW = 1.0;		ShowObj = 0L;
	tr_rec.left = tr_rec.top = tr_rec.right = tr_rec.bottom = 0;
	if(memDC) {
		oldPen = (HPEN)SelectObject(memDC, hPen);
		oldBrush = (HBRUSH)SelectObject(memDC, hBrush);
		}
	else {
		oldBrush = 0L;		oldPen = 0L;
		}
	hFont = 0L;	OC_type = OC_BITMAP;	hasClip = false;
	Notary->ValPtr(this, true);
}

BitMapWin::BitMapWin(int w, int h, double hr, double vr)
{
	HDC dc;
	HWND hwndDesk;

	memDC = 0L;		hgo = 0L;		go = 0L;
	hres = hr;		vres = vr;		minLW = 1.0;
	DeskRect.right = w;				DeskRect.bottom = h;
	DeskRect.left = DeskRect.top = 0;
	VPorg.fx = VPorg.fy = 0.0;
	dc = GetDC(hwndDesk = GetDesktopWindow());
	scr = CreateCompatibleBitmap(dc, DeskRect.right, DeskRect.bottom);
	memDC = CreateCompatibleDC(NULL);
	ReleaseDC(hwndDesk, dc);
	SelectObject(memDC, scr);
	hPen = CreatePen(PS_SOLID, 1, dLineCol = 0x00ffffffL);
	hBrush = CreateSolidBrush(dFillCol =dBgCol = 0x00ffffffL);
	dPattern = 0L;	ShowObj = 0L;
	if(memDC) {
		SelectObject(memDC, hPen);
		SelectObject(memDC, hBrush);
		}
	hFont = 0L;				OC_type = OC_BITMAP;
	Notary->ValPtr(this, true);
}

BitMapWin::BitMapWin(GraphObj *g):anyOutput()
{
	HDC dc;
	HWND hwndDesk;

	memDC = 0L;		hgo = 0L;		go = g;
	dc = GetDC(hwndDesk = GetDesktopWindow());
	hres = vres = 300.0;	minLW = 1.0;
	DeskRect.right = un2ix(go->GetSize(SIZE_GRECT_RIGHT) - go->GetSize(SIZE_GRECT_LEFT));
	DeskRect.bottom = un2iy(go->GetSize(SIZE_GRECT_BOTTOM) - go->GetSize(SIZE_GRECT_TOP));
	DeskRect.top = DeskRect.left = 0;
	VPorg.fy = -co2fiy(go->GetSize(SIZE_GRECT_TOP));
	VPorg.fx = -co2fix(go->GetSize(SIZE_GRECT_LEFT));
	scr = CreateCompatibleBitmap(dc, DeskRect.right, DeskRect.bottom);
	memDC = CreateCompatibleDC(NULL);
	SelectObject(memDC, scr);			ReleaseDC(hwndDesk, dc);
	hPen = CreatePen(PS_SOLID, 1, dLineCol = 0x00ffffffL);
	hBrush = CreateSolidBrush(dFillCol =dBgCol = 0x00ffffffL);
	dPattern = 0L;		ShowObj = 0L;
	if(memDC) {
		SelectObject(memDC, hPen);		SelectObject(memDC, hBrush);
		}
	hFont = 0L;		OC_type = OC_BITMAP;	hasClip = false;
	Notary->ValPtr(this, true);
}

BitMapWin::~BitMapWin()
{
	Notary->ValPtr(this, false);
	if (hgo) delete hgo;				defs.FreeOut(this);
	SelectObject(memDC, oldPen);		SelectObject(memDC, oldBrush);
	if(memDC) DeleteDC(memDC);		//if(hPen) DeleteObject(hPen);
	hgo = 0L;	hFont = 0L;	scr = 0L;	
	memDC = 0L;	hPen = 0L;	hBrush = 0L;
	if (ShowObj) {
		delete((eph_obj*)ShowObj);		ShowObj = 0L;
		}
	Undo.KillDisp(this);
	if (hwndTxtCur) hwndTxtCur = 0L;
}

bool
BitMapWin::ClipRect(RECT *rec)
{
	HRGN rgn;

	if (!rec && hasClip){						//reset clipping
		SelectClipRgn(memDC, NULL);
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		hasClip = false;
		}
	else if(rec){										//set up clipping rectangle
		memcpy(&ClipRC, rec, sizeof(RECT));
		rgn = CreateRectRgn(rec->left, rec->top, rec->right, rec->bottom);
		SelectClipRgn(memDC,rgn);
		DeleteObject(rgn);
		hasClip = true;
		}
	return false;
}


bool
BitMapWin::SetLine(LineDEF *lDef)
{
	return anyOutput::SetLine(lDef);
}

bool
BitMapWin::SetFill(FillDEF *fill)
{
	HBRUSH newBrush;

	if (!fill) {
		if (hgo) delete hgo;		hgo = 0L;
		return false;
		}
	if (hgo && !hgo->getDisp()) {
		delete hgo;					hgo = 0L;
		}
	anyOutput::SetFill(fill);
	if ((fill->type & 0xff) != FILL_NONE) {
		if(!hgo) hgo = new HatchOut(this);
		if(hgo) hgo->SetFill(fill);
		}
	else if (!(fill->type & FILL_KEEP_ALIVE)) {
		if(hgo) delete hgo;
		hgo = 0L;
		}
	if(dFillCol != fill->color) {
		dFillCol = fill->color;
		newBrush = CreateSolidBrush(dFillCol);
		SelectObject(memDC, newBrush);
		if(hBrush) DeleteObject(hBrush);
		}
	dFillCol = fill->color;
	dFillCol2 = fill->color2;
	return true;
}

bool
BitMapWin::SetTextSpec(TextDEF *set)
{
	return com_SetTextSpec(set, this, &hFont, &TxtSet, &memDC);
}

bool
BitMapWin::Erase(DWORD Color)
{
	Gdiplus::Point pts[5];

	if (!memDC) return false;
	pts[0].X = pts[3].X = pts[4].X = 0;
	pts[0].Y = pts[1].Y = pts[4].Y = 0;
	pts[1].X = pts[2].X = DeskRect.right;
	pts[2].Y = pts[3].Y = DeskRect.bottom;
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(Color));
	g->FillPolygon(&brush, pts, 5);
	delete g;
	return true;
}

bool
BitMapWin::CopyBitmap(int x, int y, anyOutput* sr, int sx, int sy,
	int sw, int sh, bool invert)
{
	BitMapWin *src = (BitMapWin*)sr;

	return(0 != BitBlt(memDC, x, y, sw, sh, src->memDC, sx, sy, 
		invert ? DSTINVERT : SRCCOPY));
}

bool
BitMapWin::oGetTextExtent(unsigned char *text, int cb, double *width, double *height)
{
	return com_oGetTextExtent(text, cb, width, height, memDC, &TxtSet);
}

bool
BitMapWin::oGetTextExtentW(w_char *text, int cb, double *width, double *height)
{
	return com_oGetTextExtentW(text, cb, width, height, memDC, &TxtSet);
}

bool
BitMapWin::oGetPix(int x, int y, DWORD *col)
{
	DWORD pix;
	
	if(x >= DeskRect.left && x < DeskRect.right &&
		y >= DeskRect.top && y < DeskRect.bottom) {
		pix = GetPixel(memDC, x, y);
		*col = pix;
		return true;
		}
	else return false;
}

bool
BitMapWin::oDrawIcon(int type, int x, int y)
{
	HICON icon = 0L;
	
	switch(type) {
	case ICO_INFO:
		icon = LoadIcon(0L, IDI_ASTERISK);
		break;
	case ICO_ERROR:
		icon = LoadIcon(0L, IDI_HAND);
		break;
	case ICO_RLPLOT:
		icon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_EVAL));
		break;
		}
	if(icon){
		DrawIcon(memDC, x, y, icon);
		return true;
		}
	return false;
}

bool
BitMapWin::oCircle(int x1, int y1, int x2, int y2, char *nam)
{
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillEllipse(&brush, x1, y1, x2 - x1, y2 - y1);
	if (hgo && hgo->OC_type != OC_HATCH) {
		hgo->oCircle(x1, y1, x2, y2, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinRound);
		g->DrawEllipse(&pen, x1, y1, x2 - x1, y2 - y1);
		}
	delete g;
	return true;
}

bool
BitMapWin::foCircle(double x1, double y1, double x2, double y2, char *)
{
	Gdiplus::REAL x, y, w, h;
	x = (Gdiplus::REAL) x1;					y = (Gdiplus::REAL) y1;
	w = (Gdiplus::REAL) (x2 - x1);			h = (Gdiplus::REAL) (y2 - y1);
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillEllipse(&brush, x, y, w, h);
	if (hgo && hgo->OC_type != OC_HATCH) {
		hgo->foCircle(x1, y1, x2, y2, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinRound);
		g->DrawEllipse(&pen, x, y, w, h);
		}
	delete g;
	return true;
}

bool
BitMapWin::oSphere(int cx, int cy, int r, POINT *pts, int cp, char *nam)
{
	if (dFillCol == dFillCol2) {
		if (pts) return oPolygon(pts, cp, nam);
		else return oCircle(cx - r, cy - r, cx + r, cy + r);
		}
	return com_CircGrad(cx, cy, r, pts, cp, nam, this, memDC);
}

bool
BitMapWin::oPolyline(POINT * pts, int cp)
{
	int i;
	Gdiplus::Point *points;

	if(!pts || cp < 1) return false;
	if ((dLineCol & 0xff000000) == 0xff000000) return false;	//do not draw transparent lines
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i - 1], pts[i]);
		return true;
		}
	points = (Gdiplus::Point *)malloc((cp+2) * sizeof(Gdiplus::Point));
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(cp < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
	Graphics *g = new Graphics(memDC);
	for (i = 0; i < cp; i++) {
		points[i].X = pts[i].x;			points[i].Y = pts[i].y;
		}
	g->DrawLines(&pen, points, cp);
	free(points);	delete g;
	return true;
}

bool
BitMapWin::foPolyline(lfPOINT * pts, int cp)
{
	int i;
	Gdiplus::PointF *points;

	if (!pts || cp < 1) return false;
	if ((dLineCol & 0xff000000) == 0xff000000) return false;	//do not draw transparent lines
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i - 1], pts[i]);
		return true;
		}
	points = (Gdiplus::PointF *)malloc((cp + 2) * sizeof(Gdiplus::PointF));
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(cp < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
	Graphics *g = new Graphics(memDC);
	for (i = 0; i < cp; i++) {
		points[i].X = (Gdiplus::REAL)pts[i].fx;			points[i].Y = (Gdiplus::REAL)pts[i].fy;
		}
	g->DrawLines(&pen, points, cp);
	free(points);	delete g;
	return true;
}

bool
BitMapWin::oRectangle(int x1, int y1, int x2, int y2, char *)
{
	Gdiplus::Point pts[5];
	Gdiplus::REAL lw = (float)un2fiy(LineWidth);
	pts[0].X = pts[3].X = pts[4].X = x1;	pts[0].Y = pts[1].Y = pts[4].Y = y1;
	pts[1].X = pts[2].X = x2;		pts[2].Y = pts[3].Y = y2;
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, 5);
	if (hgo && hgo->OC_type != OC_HATCH) {
		if (hasClip) ClipRect(&ClipRC);
		hgo->oRectangle(x1, y1, x2, y2, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), lw);
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinMiter);
		g->DrawPolygon(&pen, pts, 5);
		}
	delete g;
	return true;
}

bool
BitMapWin::foRectangle(double x1, double y1, double x2, double y2, char *)
{
	Gdiplus::PointF pts[5];
	pts[0].X = pts[3].X = pts[4].X = (Gdiplus::REAL)x1;	
	pts[0].Y = pts[1].Y = pts[4].Y = (Gdiplus::REAL)y1;
	pts[1].X = pts[2].X = (Gdiplus::REAL)x2;
	pts[2].Y = pts[3].Y = (Gdiplus::REAL)y2;
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, 5);
	if (hgo && hgo->OC_type != OC_HATCH) {
		if (hasClip) ClipRect(&ClipRC);
		hgo->oRectangle(iround(x1), iround(y1), iround(x2), iround(y2), NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinMiter);
		g->DrawPolygon(&pen, pts, 5);
		}
	delete g;
	return true;
}

bool
BitMapWin::oSolidRectangle(int x1, int y1, int x2, int y2)
{
	Gdiplus::PointF pts[5];
	pts[0].X = pts[3].X = pts[4].X = (Gdiplus::REAL)x1;
	pts[0].Y = pts[1].Y = pts[4].Y = (Gdiplus::REAL)y1;
	pts[1].X = pts[2].X = (Gdiplus::REAL)x2;
	pts[2].Y = pts[3].Y = (Gdiplus::REAL)y2;
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	if (hasClip) ClipRect(&ClipRC);
	g->FillPolygon(&brush, pts, 5);
	delete g;
	return true;
}

bool
BitMapWin::foSolidRectangle(double x1, double y1, double x2, double y2)
{
	Gdiplus::PointF pts[5];
	pts[0].X = pts[3].X = pts[4].X = (Gdiplus::REAL)x1;
	pts[0].Y = pts[1].Y = pts[4].Y = (Gdiplus::REAL)y1;
	pts[1].X = pts[2].X = (Gdiplus::REAL)x2;
	pts[2].Y = pts[3].Y = (Gdiplus::REAL)y2;
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	if (hasClip) ClipRect(&ClipRC);
	g->FillPolygon(&brush, pts, 5);
	delete g;
	return true;
}

bool
BitMapWin::oSolidLine(POINT *p)
{
	static Gdiplus::Point points[2];

	if ((dLineCol & 0xff000000L) == 0xff000000L) return false;	//do not draw transparent lines
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	Graphics *g = new Graphics(memDC);
	points[0].X = p[0].x;		points[0].Y = p[0].y;
	points[1].X = p[1].x;		points[1].Y = p[1].y;
	g->DrawLine(&pen, points[0], points[1]);
	delete g;
	return true;
}

bool 
BitMapWin::foSolidLine(lfPOINT *pts)
{
	static Gdiplus::PointF points[2];

	if ((dLineCol & 0xff000000L) == 0xff000000L) return false;	//do not draw transparent lines
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	Graphics *g = new Graphics(memDC);
	points[0].X = (Gdiplus::REAL)pts[0].fx;		points[0].Y = (Gdiplus::REAL)pts[0].fy;
	points[1].X = (Gdiplus::REAL)pts[1].fx;		points[1].Y = (Gdiplus::REAL)pts[1].fy;
	g->DrawLine(&pen, points[0], points[1]);
	delete g;
	return true;
}

bool
BitMapWin::oTextOut(int x, int y, unsigned char *txt, int cb)
{
	return com_oTextOut(x, y, txt, cb, &hFont, &memDC, &TxtSet, this);
}

bool
BitMapWin::oTextOutW(int x, int y, w_char *txt, int cb)
{
	return com_oTextOutW(x, y, txt, cb, &hFont, &memDC, &TxtSet, this);
}

bool
BitMapWin::oPolygon(POINT *pt, int cp, char *nam)
{
	int i;
	Gdiplus::Point *pts;
	if (!(pts = (Gdiplus::Point*) malloc((cp +2) * sizeof(Gdiplus::Point)))) return false;
	for (i = 0; i < cp; i++) {
		pts[i].X = pt[i].x;		pts[i].Y = pt[i].y;
		}
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, cp);
	if (hgo && hgo->OC_type != OC_HATCH) {
		if (hasClip) ClipRect(&ClipRC);
		hgo->oPolygon(pt, cp, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		if (pts[i - 1].X != pts[0].X || pts[i - 1].Y != pts[0].Y) {		//close polygon ?
			pts[i].X = pts[0].X;		pts[i].Y = pts[0].Y;		cp++;
			}
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineJoin(cp < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		g->DrawLines(&pen, pts, cp);
		}
	free(pts);		delete g;
	return true;
}

bool
BitMapWin::foPolygon(lfPOINT *pt, int cp, char *nam)
{
	int i;
	Gdiplus::PointF *pts;
	POINT *ipts;
	if (!(pts = (Gdiplus::PointF*) malloc((cp+2) * sizeof(Gdiplus::PointF)))) return false;
	for (i = 0; i < cp; i++) {
		pts[i].X = (Gdiplus::REAL)pt[i].fx;		pts[i].Y = (Gdiplus::REAL)pt[i].fy;
		}
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, cp);
	if (hgo && hgo->OC_type != OC_HATCH && (ipts = (POINT*)malloc(cp * sizeof(POINT)))) {
		for (i = 0; i < cp; i++) {
			ipts[i].x = iround(pt[i].fx);		ipts[i].y = iround(pt[i].fy);
			}
		if (hasClip) ClipRect(&ClipRC);
		hgo->oPolygon(ipts, cp, NULL);
		free(ipts);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		if (pts[i - 1].X != pts[0].X || pts[i - 1].Y != pts[0].Y) {		//close polygon ?
			pts[i].X = pts[0].X;		pts[i].Y = pts[0].Y;		cp++;
			}
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(cp < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
		g->DrawLines(&pen, pts, cp);
		}
	free(pts);		delete g;
	return true;
}

bool 
BitMapWin::oSolidPolygon(POINT *pt, int cp)
{
	int i;
	Gdiplus::Point *pts;
	if (!(pts = (Gdiplus::Point*) malloc((cp + 2) * sizeof(Gdiplus::Point)))) return false;
	for (i = 0; i < cp; i++) {
		pts[i].X = pt[i].x;		pts[i].Y = pt[i].y;
		}
	Graphics *g = new Graphics(memDC);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, cp);
	free(pts);		delete g;
	return true;
}

bool
BitMapWin::GradPG(fPOINT3D *pts, long npt, double Lo, double Hi, double pLo, double pHi, lfPOINT *grad)
{
	return com_GradPG(pts, npt, Lo, Hi, pLo, pHi, grad, this, memDC);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Output to windows window
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
OutputWin::OutputWin(GraphObj *g, HWND hw):BitMapWin(g, hw)
{
	minLW = 1.0;					bHasFocus = false;
	if(g) {
		if(!hw) CreateNewWindow(g);
		else hWnd = hw;
		if (!MainWnd) {
			MainWnd = hWnd;			SetTimer(hWnd, 1, 150, 0L);
			}
		}
	else {							//its a dialog window
		hWnd = hw;
		yAxis.flags = AXIS_INVERT;	//drawing origin upper left corner
		}
	Notary->ValPtr(this, true);
}

OutputWin::~OutputWin()
{
	defs.Idle(CMD_UPDATE);
	Undo.KillDisp(this);	defs.FreeOut(this);
	SendMessage(hWnd, WM_CLOSE, 0, 0L);
	if (hwndTxtCur) hwndTxtCur = 0L;
	Notary->ValPtr(this, false);
	//Note: HGDI objects are deleted by the BitMapWin destructor
}

bool
OutputWin::ActualSize(RECT *rc)
{
	if(GetClientRect(hWnd, rc)&& (rc->right-rc->left) > 40 && (rc->bottom - rc->top) > 40) return true;
	return false;
}

void
OutputWin::Caption(char *txt, bool bModified)
{
	char txt1[200];
	int cb;

	cb = rlp_strcpy((void*)txt1, 180, (void*)txt);
	if(bModified)rlp_strcpy((void*)(txt1+cb), 20, (void*)" [modified]");
	SetWindowText(hWnd, txt1);
}

const static unsigned char hand_bits[] =	{	//hand cursor bitmap
	0x01, 0x80, 0x1b, 0xf0, 0x3f, 0xf8, 0x3f, 0xfa,
	0x1f, 0xff, 0x1f, 0xff, 0x6f, 0xff, 0xff, 0xff,
	0xff, 0xfe, 0x7f, 0xfe, 0x7f, 0xfe, 0x3f, 0xfc,
	0x1f, 0xfc, 0x0f, 0xf8, 0x07, 0xf8, 0x07, 0xf8};

const static unsigned char hand_mask[] =	{	//hand cursor mask
	0xff, 0xff, 0xfe, 0x7f, 0xe6, 0x4f, 0xe6, 0x4f,
	0xf2, 0x4d, 0xf2, 0x49, 0x78, 0x09, 0x98, 0x01,
	0x88, 0x03, 0xc0, 0x03, 0xc0, 0x03, 0xe0, 0x07,
	0xf0, 0x07, 0xf8, 0x0f, 0xfc, 0x0f, 0xfc, 0x0f};

const static unsigned char zoom_bits[] =	{	//zoom cursor bitmap
	0x00, 0x00, 0x00, 0x00, 0x01, 0xa0, 0x06, 0x30,
	0x08, 0x08, 0x10, 0x84, 0x10, 0x84, 0x20, 0x02,
	0x26, 0x32, 0x20, 0x02, 0x10, 0x84, 0x10, 0x84,
	0x08, 0x08, 0x06, 0x30, 0x01, 0xa0, 0x00, 0x00};

const static unsigned char zoom_mask[] =	{	//zoom cursor mask
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

const static unsigned char paste_bits[] =	{	//paste cursor bitmap
	0x20, 0x00, 0x20, 0x00, 0xf8, 0x00, 0x20, 0x00,
	0x23, 0xfe, 0x07, 0xff, 0x07, 0xff, 0x07, 0xff,
	0x07, 0xff, 0x07, 0xff, 0x07, 0xff, 0x07, 0xff,
	0x07, 0xff, 0x07, 0xff, 0x03, 0xfe, 0x00, 0x00};

const static unsigned char paste_mask[] =	{	//paste cursor mask
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xfc, 0x05, 0xfd, 0x05, 0xfd, 0xf9,
	0xfc, 0x01, 0xfc, 0x01, 0xfc, 0x01, 0xfc, 0x01,
	0xfc, 0x01, 0xfc, 0x01, 0xff, 0xff, 0xff, 0xff};

const static unsigned char drawpen_bits[] =	{	//draw cursor bitmap
	0xc0, 0x00, 0xf0, 0x00, 0x7c, 0x00, 0x7f, 0x00,
	0x3f, 0x80, 0x3f, 0xc0, 0x1f, 0xe0, 0x1f, 0xf0,
	0x0f, 0xf8, 0x07, 0xfc, 0x03, 0xfe, 0x01, 0xff,
	0x00, 0xff, 0x00, 0x7e, 0x00, 0x3c, 0x00, 0x18};

const static unsigned char drawpen_mask[] =	{	//draw cursor mask
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf3, 0xff,
	0xe0, 0xff, 0xe2, 0x7f, 0xf1, 0x3f, 0xf0, 0x9f,
	0xf8, 0x4f, 0xfc, 0x27, 0xfe, 0x13, 0xff, 0x0b,
	0xff, 0x87, 0xff, 0xcf, 0xff, 0xff, 0xff, 0xff};

const static unsigned char drect_bits[] =	{	//draw rectangle bitmap
	0x20, 0x00, 0x20, 0x00, 0xf8, 0x00, 0x20, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x1f, 0xff, 0x1f, 0xff,
	0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xff,
	0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xff, 0x00, 0x00};

const static unsigned char drect_mask[] =	{	//draw rectangle mask
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x01,
	0xf0, 0x01, 0xf0, 0x01, 0xf0, 0x01, 0xf0, 0x01,
	0xf0, 0x01, 0xf0, 0x01, 0xff, 0xff, 0xff, 0xff};

const static unsigned char drrect_bits[] =	{	//draw rounded rectangle bitmap
	0x20, 0x00, 0x20, 0x00, 0xf8, 0x00, 0x20, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x07, 0xfc, 0x0f, 0xfe,
	0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xff,
	0x1f, 0xff, 0x0f, 0xfe, 0x07, 0xfc, 0x00, 0x00};

const static unsigned char drrect_mask[] =	{	//draw rounded rectangle mask
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x03,
	0xf0, 0x01, 0xf0, 0x01, 0xf0, 0x01, 0xf0, 0x01,
	0xf0, 0x01, 0xf8, 0x03, 0xff, 0xff, 0xff, 0xff};

const static unsigned char delly_bits[] =	{	//draw ellipse bitmap
	0x20, 0x00, 0x20, 0x00, 0xf8, 0x00, 0x20, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x01, 0xf0, 0x07, 0xfc,
	0x0f, 0xfe, 0x1f, 0xff, 0x1f, 0xff, 0x1f, 0xff,
	0x0f, 0xfe, 0x07, 0xfc, 0x01, 0xf0, 0x00, 0x00};

const static unsigned char delly_mask[] =	{	//draw ellipse mask
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x0f,
	0xf8, 0x03, 0xf0, 0x01, 0xf0, 0x01, 0xf0, 0x01,
	0xf8, 0x03, 0xfe, 0x0f, 0xff, 0xff, 0xff, 0xff};

const static unsigned char etracc_bits[] = {	//draw etracc bitmap
	0xf8, 0x00,	0x80, 0x00,	0x80, 0x00,	0x9f, 0xff,
	0x90, 0x01,	0x10, 0x01, 0x10, 0x01,	0x1f, 0xff,
	0x10, 0x01, 0x10, 0x01, 0x10, 0x01,	0x1f, 0xff,
	0x10, 0x01,	0x10, 0x01, 0x10, 0x01,	0x1f, 0xff };

const static unsigned char etracc_mask[] = {	//draw etracc mask
	0xff, 0xff,	0xff, 0xff,	0xff, 0xff,	0xff, 0xff,
	0xff, 0x83,	0xff, 0x83,	0xff, 0x83,	0xff, 0xff,
	0xff, 0x83,	0xff, 0x83,	0xff, 0x83,	0xff, 0xff,
	0xf4, 0x3f,	0xf4, 0x3f,	0xf4, 0x3f,	0xff, 0xff };

//display 16x16 cursor data: developers utility
/*
void disp_bm(unsigned char *tb)
{
	char txt[512];
	unsigned char currbyte;
	int i, j, pos;

	for(i = pos = 0; i < 32; i++) {
		currbyte = tb[i];
		for (j = 0; j < 8; j++) {
			if(currbyte & 0x80) pos += sprintf(txt+pos, "1");
			else pos += sprintf(txt+pos, "0");
			currbyte <<= 1;
			}
		if(i & 1) pos += sprintf(txt+pos, "\n");
		}
	InfoBox((char*)txt);
}
*/

void
OutputWin::MouseCursor(int cid, bool force)
{
	HCURSOR hc, hoc = 0L;

	if(cid == cCursor && !force) return;
	if(cid == MC_LAST) cid = cCursor;
	switch(cid) {
	case MC_ARROW:
		hoc = SetCursor(LoadCursor(NULL, IDC_ARROW));	break;
	case MC_TXTFRM:
	case MC_CROSS:	hoc = SetCursor(LoadCursor(NULL, IDC_CROSS));	break;
	case MC_WAIT:
		hoc = SetCursor(LoadCursor(NULL, IDC_WAIT));	break;
	case MC_TEXT:	hoc = SetCursor(LoadCursor(NULL, IDC_IBEAM));	break;
	case MC_NORTH:	hoc = SetCursor(LoadCursor(NULL, IDC_SIZENS));	break;
	case MC_NE:		hoc = SetCursor(LoadCursor(NULL, IDC_SIZENESW));break;
	case MC_COLWIDTH:
	case MC_EAST:	hoc = SetCursor(LoadCursor(NULL, IDC_SIZEWE));	break;
	case MC_SE:		hoc = SetCursor(LoadCursor(NULL, IDC_SIZENWSE));break;
	case MC_SALL:	hoc = SetCursor(LoadCursor(NULL, IDC_SIZEALL));	break;	
	case MC_MOVE:
		hc = CreateCursor(hInstance, 7, 7, 16, 16, hand_mask, hand_bits);
		hoc = SetCursor(hc);
		break;
	case MC_ZOOM:
		hc = CreateCursor(hInstance, 7, 7, 16, 16, zoom_mask, zoom_bits);
		hoc = SetCursor(hc);
		break;
	case MC_PASTE:
		hc = CreateCursor(hInstance, 2, 2, 16, 16, paste_mask, paste_bits);
		hoc = SetCursor(hc);
		break;
	case MC_DRAWPEN:
		hc = CreateCursor(hInstance, 0, 0, 16, 16, drawpen_mask, drawpen_bits);
		hoc = SetCursor(hc);
		break;
	case MC_DRAWREC:
		hc = CreateCursor(hInstance, 2, 2, 16, 16, drect_mask, drect_bits);
		hoc = SetCursor(hc);
		break;
	case MC_DRAWRREC:
		hc = CreateCursor(hInstance, 2, 2, 16, 16, drrect_mask, drrect_bits);
		hoc = SetCursor(hc);
		break;
	case MC_DRAWELLY:
		hc = CreateCursor(hInstance, 2, 2, 16, 16, delly_mask, delly_bits);
		hoc = SetCursor(hc);
		break;
	case MC_ETRACC:
		hc = CreateCursor(hInstance, 2, 2, 16, 16, etracc_mask, etracc_bits);
		hoc = SetCursor(hc);
	default:	return;
		}
	if(hoc) DestroyCursor(hoc);
	cCursor = cid;
}

bool
OutputWin::SetScroll(bool isVert, int iMin, int iMax, int iPSize, int iPos)
{
	SCROLLINFO si;

	if (iPos < iMin) return false;
	si.cbSize = sizeof(SCROLLINFO);			si.fMask = SIF_ALL;
	si.nMin = iMin;							si.nMax = iMax;
	si.nPage = iPSize < iMax ? iPSize : iMax;
	si.nPos = iPos;							si.nTrackPos = 0;
	SetScrollInfo(hWnd, isVert ? SB_VERT : SB_HORZ, &si, TRUE);
	return true;
}

bool
OutputWin::Erase(DWORD Color)
{
	bool bRet;
	RECT ClientRect;

	if(bRet = BitMapWin::Erase(Color)) {
		GetClientRect(hWnd, &ClientRect);
		InvalidateRect(hWnd, &ClientRect, FALSE);
		}
	return bRet;
}

bool
OutputWin::StartPage()
{
	MrkMode = MRK_NONE;			MrkRect = 0L;
	bSuspend = true;
	if(hgo) hgo->StartPage();
	ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
	return true;
}

bool
OutputWin::EndPage()
{
	RECT ClientRect;

	bSuspend = false;
	if (hgo) hgo->EndPage();
	GetClientRect(hWnd, &ClientRect);
//	return UpdateRect(&ClientRect, true);
	return true;
}

static void FrozenHWND(DataObj *d, OutputWin *o)
{
	char tmp_txt[1024], cmd[1024];

	Notary->ValPtr(o, false);				//disable further input to frozen window
	sprintf_s(tmp_txt, 500, "%s%s", defs.currPath, RLP_FROZEN_FILE);
	sprintf_s(cmd, 1000, "%s %s", ShellCmd, tmp_txt);
	if (FileExist(tmp_txt)){
#ifdef USE_WIN_SECURE
		_unlink(tmp_txt);
#else
		unlink(tmp_txt);
#endif
		}
	if (d->WriteData(tmp_txt)) {
		if (ShellCmd && ShellCmd[0])WinExec(cmd, SW_SHOW);
		PostQuitMessage(234);
		}
}
	
bool
OutputWin::UpdateRect(RECT *rc, bool bStore)
{
	HDC dc;
	bool RetVal = false;

//	ClipRect(NULL);
	if (bStore){
		defs.UpdRect(this, rc);
		RetVal = true;
		}
	else if(dc = GetDC(hWnd)) {
		if (BitBlt(dc, rc->left, rc->top, rc->right - rc->left,
			rc->bottom - rc->top, memDC, rc->left, rc->top, SRCCOPY)) {
			RetVal = true;
			}
		ReleaseDC(hWnd, dc);
		}
	else {			//Window is frozen!
		FrozenHWND(go->data, this);
		}
	return RetVal;
}

bool
OutputWin::UpdateRect(HDC dc, RECT rc)		//come here on WM_PAINT event
{
	if(BitBlt(dc, rc.left, rc.top, rc.right - rc.left,
			rc.bottom - rc.top, memDC, rc.left, rc.top, SRCCOPY))return true;
	return false;
}

void
OutputWin::ShowBitmap(int x, int y, anyOutput* src)
{
	int w, h;
	HDC dc;
	BitMapWin *sr;

	if(!src) return;
	sr = (BitMapWin *) src;
	w = sr->DeskRect.right - sr->DeskRect.left;
	h = sr->DeskRect.bottom - sr->DeskRect.top;
	if(dc = GetDC(hWnd)) {
		BitBlt(dc, x, y, w,	h, sr->memDC, 0, 0, SRCCOPY);
		ReleaseDC(hWnd, dc);
		}
}

bool
OutputWin::CopyObject(void *obj)
{
	GraphObj *g;

	g = ((GraphObj *)obj);
	if (g && g->Id == GO_DATALINE) {
		CopyGO = g;
		OpenClipboard(hWnd);
		EmptyClipboard();
		SetClipboardData(CF_TEXT, NULL);
//		SetClipboardData(CF_SYLK, NULL);
//		SetClipboardData(cf_rlpxml, NULL);
		CloseClipboard();
		return true;
		}
	return false;
}

bool
OutputWin::HideMark()
{
	anyOutput::HideMark();
	if(ShowObj) {
		((eph_obj*)ShowObj)->Restore(this, true);
		delete((eph_obj*)ShowObj);			ShowObj = 0L;
		}
	return true;
}

void
OutputWin::ShowLine(POINT * pts, int cp, DWORD color, bool)
{
	ShowObj = new eph_line((eph_obj*)ShowObj, pts, cp, color, 0L, this);
}

void
OutputWin::ShowEllipse(POINT p1, POINT p2, DWORD color)
{
	HDC dc;
	HPEN hP, oP;

	if((hP = CreatePen(PS_SOLID, 0, color)) && (dc = GetDC(hWnd))) {
		oP = (HPEN)SelectObject(dc, hP);
		Arc(dc, p1.x, p1.y, p2.x, p2.y, 0, 0, 0, 0);
		SelectObject(dc, oP);
		ReleaseDC(hWnd, dc);
		}
}

bool
OutputWin::SetMenu(int type)
{
	HMENU hMenu = 0L;

	switch(type) {
	case MENU_NONE:
		break;
	case MENU_SPREAD:
		hMenu = LoadMenu(hInstance, MAKEINTRESOURCE(MENU_2));
		break;
	case MENU_GRAPH:
		hMenu = LoadMenu(hInstance, MAKEINTRESOURCE(MENU_1));
		break;
	case MENU_PAGE:
		hMenu = LoadMenu(hInstance, MAKEINTRESOURCE(MENU_3));
		}
	::SetMenu(hWnd, hMenu);
	return true;
}

void
OutputWin::CheckMenu(int mid, bool check)
{
	HMENU hMenu = GetMenu(hWnd);

	if(mid < CM_T_STANDARD) switch(mid){					//tool mode identifier
	case TM_STANDARD:	mid = CM_T_STANDARD;	break;
	case TM_DRAW:		mid = CM_T_DRAW;		break;
	case TM_POLYLINE:	mid = CM_T_POLYLINE;	break;
	case TM_POLYGON:	mid = CM_T_POLYGON;		break;
	case TM_RECTANGLE:	mid = CM_T_RECTANGLE;	break;
	case TM_ROUNDREC:	mid = CM_T_ROUNDREC;	break;
	case TM_ELLIPSE:	mid = CM_T_ELLIPSE;		break;
	case TM_ARROW:		mid = CM_T_ARROW;		break;
	case TM_TEXT:		mid = CM_T_TEXT;		break;
	default:	return;
		}
	if(hMenu) CheckMenuItem(hMenu, mid, check ? MF_CHECKED : MF_UNCHECKED);
}

void
OutputWin::FileHistory() 
{
	HMENU hSubMenu;
	unsigned char **history[] = {&defs.File1, &defs.File2, &defs.File3, &defs.File4, &defs.File5, &defs.File6};
	int i, j, k;

	if(!hasHistMenu || !defs.File1) return;
	if (!defs.currPath)return;
	if(!(hSubMenu = GetSubMenu(GetMenu(hWnd), 0))) return;
	if(!HistMenuSize) AppendMenu(hSubMenu, MF_SEPARATOR, 0L, 0L);
    for(i = 0; i < 6 && *history[i]; i++) {
		k = rlp_strlen((char*) *history[i]);
		for (j = 0; j < k && defs.currPath[j] == (*history[i])[j]; j++);
		if((*history[i])[j] == '\\' || (*history[i])[j] == '/') j++;
		if(i < HistMenuSize) {
			ModifyMenu(hSubMenu, CM_FILE1+i, MF_BYCOMMAND | MF_STRING, CM_FILE1+i, (char*)*history[i]+j);
			}
		else {
			AppendMenu(hSubMenu, MF_STRING, CM_FILE1+i, (char*)*history[i]+j);
			}
		}
	HistMenuSize = i;
}

bool
OutputWin::Command(int cmd, void *data)
{
	switch (cmd) {
	case CMD_CPY_DATA:				//coming here from the layers dialog
		if (((Plot*)data)->Id == GO_BOXPLOT) {
			CopyGO = (GraphObj*)data;
			OpenClipboard(hWnd);
			SetClipboardData(CF_SYLK, NULL);
			CloseClipboard();
			}
		else ((Plot*)data)->Command(cmd, 0L, this);
		break;
	case CM_EXIT:
		DestroyWindow(hWnd);
		break;
		}
	return false;
}

void
OutputWin::CreateNewWindow(void *g)
{
	RECT ClientRect;

	if (MainWnd) {
		hWnd = CreateWindow(name, "RLPlot",	WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL,
		CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInstance, NULL);
		}
	else {
		hWnd = CreateWindowEx(WS_EX_ACCEPTFILES, name, "RLPlot", WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL,
			CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInstance, NULL);
		}
	SetWindowLongPtr(hWnd, 0,(LONG_PTR) g);		// g is the parent graphic obj
#ifdef _WIN64
	SetWindowLongPtr(hWnd, GWLP_USERDATA, (LONG_PTR) this);
#else
	SetWindowLongPtr(hWnd, GWL_USERDATA, (LONG_PTR) this);
#endif
	if (BitMapWin::Erase(0x00cbcbcb)) {
		GetClientRect(hWnd, &ClientRect);
		InvalidateRect(hWnd, &ClientRect, FALSE);
		}
	UpdateWindow(hWnd);
	ShowWindow(hWnd, SW_SHOW);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Printer setup
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool ConfigPrinter()
{
	double pw, ph;
	DEVMODE *modes;
	short orientation;

	PriDlg.Flags = PD_PRINTSETUP;
	if (PrintDlg(&PriDlg) == TRUE && (modes = (DEVMODE *)GlobalLock(PriDlg.hDevMode))) 		{
		pw = ((double)(modes->dmPaperWidth)) / 254.0;		//? to device caps!
		ph = ((double)(modes->dmPaperLength)) / 254.0;		//?
		orientation = modes->dmOrientation;
		GlobalUnlock(PriDlg.hDevMode);
		switch (defs.dUnits){
		case 1:		pw *= 2.54;		ph *= 2.54;		break;
		case 2:		break;
		default:	pw *= 25.4;		ph *= 25.4;		break;
			}
		if (orientation == 2) FindPaper(ph, pw, 0.01);
		else FindPaper(pw, ph, 0.01);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Printer class 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
PrintWin::PrintWin()
{
	int i, j;
	double pw, ph;
	DEVMODE *modes;
	short orientation;

	PrintDriver = PrintDevice = PrintPort = 0L;
	hPen = 0L;		hBrush = 0L;			hFont = 0L;		hDC = 0L;
	hgo = 0L;		i = j = 0;		minLW = 1.0;
	if (PriDlg.hDevMode && (modes = (DEVMODE *)GlobalLock(PriDlg.hDevMode))) {
		pw = ((double)(modes->dmPaperWidth)) /254.0;		//? to device caps!
		ph = ((double)(modes->dmPaperLength)) / 254.0;		//?
		orientation = modes->dmOrientation;
		GlobalUnlock(PriDlg.hDevMode);
		}
	else {
		pw = 8.266;				//paper size in inch
		ph = 11.69;
		orientation = 1;		//portrait
		}
	switch (defs.dUnits){
		case 1:		pw *= 2.54;		ph *= 2.54;		break;
		case 2:		break;
		default:	pw *= 25.4;		ph *= 25.4;		break;
		}
	OC_type = OC_PRINT;			hasClip = false;
	if(orientation == 2) FindPaper(ph, pw, 0.01);
	else FindPaper(pw, ph, 0.01);
	Notary->ValPtr(this, true);
}

PrintWin::~PrintWin()
{
	defs.FreeOut(this);
	if(PrintDriver) free(PrintDriver);		if(PrintDevice) free(PrintDevice);
	if(PrintPort) free(PrintPort);
	defs.Idle(CMD_FLUSH);
	Notary->ValPtr(this, false);
}

bool
PrintWin::ClipRect(RECT *rec)
{
	HRGN rgn;

	if (!rec && hasClip){
		SelectClipRgn(hDC, NULL);
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		hasClip = false;
		}
	else if(rec){
		memcpy(&ClipRC, rec, sizeof(RECT));
		rgn = CreateRectRgn(rec->left, rec->top, rec->right, rec->bottom);
		SelectClipRgn(hDC, rgn);		DeleteObject(rgn);
		hasClip = true;
		}
	return false;
}

bool
PrintWin::SetLine(LineDEF *lDef)
{
	return anyOutput::SetLine(lDef);
}

bool
PrintWin::SetFill(FillDEF *fill)
{
	HBRUSH newBrush;

	if (!fill) {
		if (hgo) delete hgo;		hgo = 0L;
		return false;
		}
	if (hgo && !hgo->getDisp()) {
		delete hgo;					hgo = 0L;
		}
	anyOutput::SetFill(fill);
	if ((fill->type & 0xff) != FILL_NONE) {
		if (!hgo) hgo = new HatchOut(this);
		if (hgo) hgo->SetFill(fill);
		}
	else if (!(fill->type & FILL_KEEP_ALIVE)) {
		if (hgo) delete hgo;
		hgo = 0L;
		}
	if (dFillCol != fill->color) {
		dFillCol = fill->color;
		newBrush = CreateSolidBrush(dFillCol);
		SelectObject(hDC, newBrush);
		hBrush = newBrush;
		}
	dFillCol = fill->color;			dFillCol2 = fill->color2;
	return true;
}

bool
PrintWin::SetTextSpec(TextDEF *set)
{
	return com_SetTextSpec(set, this, &hFont, &TxtSet, &hDC);
}

bool
PrintWin::oGetTextExtent(unsigned char *text, int cb, double *width, double *height)
{
	return com_oGetTextExtent(text, cb, width, height, hDC, &TxtSet);
}

bool
PrintWin::oGetTextExtentW(w_char *text, int cb, double *width, double *height)
{
	return com_oGetTextExtentW(text, cb, width, height, hDC, &TxtSet);
}

bool
PrintWin::StartPage()
{
	DOCINFO DocInfo;
	bool bRet = false;
	double res;

	ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
	PriDlg.Flags = PD_RETURNDC;
	if (PrintDlg(&PriDlg) == TRUE){
		hDC = PriDlg.hDC;

		DEVMODE *modes;
		if (PriDlg.hDevMode && (modes = (DEVMODE *)GlobalLock(PriDlg.hDevMode))) {
			res = modes->dmYResolution;
			GlobalUnlock(PriDlg.hDevMode);
			}
		}
	else hDC = NULL;
	if (hDC) {
		hPen = CreatePen(PS_SOLID, 1, dLineCol = 0x00ffffffL);
		hBrush = CreateSolidBrush(dFillCol =dBgCol = 0x00ffffffL);
		dPattern = 0L;
		SelectObject(hDC, hPen);
		SelectObject(hDC, hBrush);
		memset(&DocInfo, 0, sizeof(DOCINFO));
		DocInfo.lpszDocName = "RLPlot graph";
		DocInfo.cbSize = sizeof(DOCINFO);
		DeskRect.left = DeskRect.top = 0;
		DeskRect.right = GetDeviceCaps(hDC, HORZRES);
		DeskRect.bottom = GetDeviceCaps(hDC, VERTRES);
		hres = (double)GetDeviceCaps(hDC, LOGPIXELSX);
		vres = (double)GetDeviceCaps(hDC, LOGPIXELSY);
		if(StartDoc(hDC, &DocInfo) >= 0) {
			if(::StartPage(hDC)>0) bRet = true;
			}
		}
	return bRet;
}

bool
PrintWin::EndPage()
{

	if(hDC) {
		::EndPage(hDC);		EndDoc(hDC);		DeleteDC(hDC);
		}
	hDC = 0L;
	return true;
}

bool
PrintWin::Eject()
{
	if(hDC) {
		ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
		::EndPage(hDC);		::StartPage(hDC);	return true;
		}
	return false;
}

bool
PrintWin::CopyBitmap(int x, int y, anyOutput* sr, int sx, int sy,
	int sw, int sh, bool invert)
{
	BitMapWin *src = (BitMapWin*)sr;

	return(0 != BitBlt(hDC, x, y, sw, sh, src->memDC, sx, sy, 
		invert ? DSTINVERT : SRCCOPY));
}

bool
PrintWin::oCircle(int x1, int y1, int x2, int y2, char* nam)
{
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillEllipse(&brush, x1, y1, x2 - x1, y2 - y1);
	if (hgo && hgo->OC_type != OC_HATCH) {
		hgo->oCircle(x1, y1, x2, y2, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinRound);
		g->DrawEllipse(&pen, x1, y1, x2 - x1, y2 - y1);
		}
	delete g;
	return true;
}

bool
PrintWin::foCircle(double x1, double y1, double x2, double y2, char *)
{
	Gdiplus::REAL x, y, w, h;
	x = (Gdiplus::REAL) x1;					y = (Gdiplus::REAL) y1;
	w = (Gdiplus::REAL) (x2 - x1);			h = (Gdiplus::REAL) (y2 - y1);
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillEllipse(&brush, x, y, w, h);
	if (hgo && hgo->OC_type != OC_HATCH) {
		hgo->foCircle(x1, y1, x2, y2, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinRound);
		g->DrawEllipse(&pen, x, y, w, h);
		}
	delete g;
	return true;
}

bool 
PrintWin::oSphere(int cx, int cy, int r, POINT *pts, int cp, char *nam)
{
	if (dFillCol == dFillCol2) {
		if (pts) return oPolygon(pts, cp, nam);
		else return oCircle(cx - r, cy - r, cx + r, cy + r);
		}
	return com_CircGrad(cx, cy, r, pts, cp, nam, this, hDC);
}

bool
PrintWin::oPolyline(POINT * pts, int cp)
{
	long i;
	static Gdiplus::PointF *points;
	Gdiplus::REAL lw;

	if(cp < 2) return false;
	if ((dLineCol & 0xfe000000L) == 0xfe000000L) return false;
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i-1], pts[i]);
		return true;
		}
	points = (Gdiplus::PointF*)malloc(cp * sizeof(Gdiplus::PointF));
	lw = (Gdiplus::REAL)un2fiy(LineWidth);
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), lw);
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	points[0].X = (Gdiplus::REAL)pts[0].x;			points[0].Y = (Gdiplus::REAL)pts[0].y;
	for (i = 1; i < cp; i++) {
		points[i].X = (Gdiplus::REAL)pts[i].x;		points[i].Y = (Gdiplus::REAL)pts[i].y;
		}
	g->DrawLines(&pen, points, cp);
	free(points);	delete g;
	return true;
}

bool
PrintWin::foPolyline(lfPOINT * pts, int cp)
{
	long i;
	static Gdiplus::PointF *points;
	Gdiplus::REAL lw;

	if (cp < 2) return false;
	if ((dLineCol & 0xfe000000L) == 0xfe000000L) return false;
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	if (dPattern) {
		for (i = 1; i < cp; i++) PatLine(pts[i - 1], pts[i]);
		return true;
		}
	points = (Gdiplus::PointF*)malloc(cp * sizeof(Gdiplus::PointF));
	lw = (Gdiplus::REAL)un2fiy(LineWidth);
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), lw);
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	points[0].X = (Gdiplus::REAL)pts[0].fx;			points[0].Y = (Gdiplus::REAL)pts[0].fy;
	for (i = 1; i < cp; i++) {
		points[i].X = (Gdiplus::REAL)pts[i].fx;		points[i].Y = (Gdiplus::REAL)pts[i].fy;
		}
	g->DrawLines(&pen, points, cp);
	free(points);	delete g;
	return true;
}

bool
PrintWin::oRectangle(int x1, int y1, int x2, int y2, char *nam)
{
	Gdiplus::Point pts[5];
	Gdiplus::REAL lw = (float)un2fiy(LineWidth);
	pts[0].X = pts[3].X = pts[4].X = x1;	pts[0].Y = pts[1].Y = pts[4].Y = y1;
	pts[1].X = pts[2].X = x2;		pts[2].Y = pts[3].Y = y2;
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, 5);
	if (hgo && hgo->OC_type != OC_HATCH) {
		if (hasClip) ClipRect(&ClipRC);
		hgo->oRectangle(x1, y1, x2, y2, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), lw);
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinMiter);
		g->DrawPolygon(&pen, pts, 5);
		}
	delete g;
	return true;
}

bool
PrintWin::foRectangle(double x1, double y1, double x2, double y2, char *)
{
	Gdiplus::PointF pts[5];
	pts[0].X = pts[3].X = pts[4].X = (Gdiplus::REAL)x1;
	pts[0].Y = pts[1].Y = pts[4].Y = (Gdiplus::REAL)y1;
	pts[1].X = pts[2].X = (Gdiplus::REAL)x2;
	pts[2].Y = pts[3].Y = (Gdiplus::REAL)y2;
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, 5);
	if (hgo && hgo->OC_type != OC_HATCH) {
		if (hasClip) ClipRect(&ClipRC);
		hgo->oRectangle(iround(x1), iround(y1), iround(x2), iround(y2), NULL);
	}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(LineJoin::LineJoinMiter);
		g->DrawPolygon(&pen, pts, 5);
	}
	delete g;
	return true;
}

bool
PrintWin::oSolidRectangle(int x1, int y1, int x2, int y2)
{
	Gdiplus::PointF pts[5];
	pts[0].X = pts[3].X = pts[4].X = (Gdiplus::REAL)x1;
	pts[0].Y = pts[1].Y = pts[4].Y = (Gdiplus::REAL)y1;
	pts[1].X = pts[2].X = (Gdiplus::REAL)x2;
	pts[2].Y = pts[3].Y = (Gdiplus::REAL)y2;
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	if (hasClip) ClipRect(&ClipRC);
	g->FillPolygon(&brush, pts, 5);
	delete g;
	return true;
}

bool
PrintWin::foSolidRectangle(double x1, double y1, double x2, double y2)
{
	Gdiplus::PointF pts[5];
	pts[0].X = pts[3].X = pts[4].X = (Gdiplus::REAL)x1;
	pts[0].Y = pts[1].Y = pts[4].Y = (Gdiplus::REAL)y1;
	pts[1].X = pts[2].X = (Gdiplus::REAL)x2;
	pts[2].Y = pts[3].Y = (Gdiplus::REAL)y2;
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	if (hasClip) ClipRect(&ClipRC);
	g->FillPolygon(&brush, pts, 5);
	delete g;
	return true;
}

bool
PrintWin::oSolidLine(POINT *p)
{
	static Gdiplus::PointF points[2];
	Gdiplus::REAL lw;

	if ((dLineCol & 0xff000000L) == 0xff000000L) return false;	//do not draw transparent lines
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	lw = (Gdiplus::REAL)un2fiy(LineWidth);
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), lw);
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	points[0].X = (Gdiplus::REAL)p[0].x;			points[0].Y = (Gdiplus::REAL)p[0].y;
	points[1].X = (Gdiplus::REAL)p[1].x;			points[1].Y = (Gdiplus::REAL)p[1].y;
	g->DrawLine(&pen, points[0], points[1]);
	delete g;
	return true;
}

bool 
PrintWin::foSolidLine(lfPOINT *p)
{
	static Gdiplus::PointF points[2];
	Gdiplus::REAL lw;

	if ((dLineCol & 0xff000000L) == 0xff000000L) return false;	//do not draw transparent lines
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	lw = (Gdiplus::REAL)un2fiy(LineWidth);
	Gdiplus::Pen pen(GDI_COLOR(dLineCol), lw);
	pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
	pen.SetLineJoin(LineJoin::LineJoinRound);
	points[0].X = (Gdiplus::REAL)p[0].fx;		points[0].Y = (Gdiplus::REAL)p[0].fy;
	points[1].X = (Gdiplus::REAL)p[1].fx;		points[1].Y = (Gdiplus::REAL)p[1].fy;
	g->DrawLine(&pen, points[0], points[1]);
	delete g;
	return true;
}

bool
PrintWin::oTextOut(int x, int y, unsigned char *txt, int cb)
{
	return com_oTextOut(x, y, txt, cb, &hFont, &hDC, &TxtSet, this);
}

bool
PrintWin::oTextOutW(int x, int y, w_char *txt, int cb)
{
	return com_oTextOutW(x, y, txt, cb, &hFont, &hDC, &TxtSet, this);
}

bool
PrintWin::oPolygon(POINT *pt, int cp, char *nam)
{
	int i;
	Gdiplus::Point *pts;

	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	if (!(pts = (Gdiplus::Point*) malloc((cp+2) * sizeof(Gdiplus::Point)))) return false;
	for (i = 0; i < cp; i++) {
		pts[i].X = pt[i].x;		pts[i].Y = pt[i].y;
		}
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, cp);
	if (hgo && hgo->OC_type != OC_HATCH) {
		if (hasClip) ClipRect(&ClipRC);
		hgo->oPolygon(pt, cp, NULL);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		if (pts[i - 1].X != pts[0].X || pts[i - 1].Y != pts[0].Y) {		//close polygon ?
			pts[i].X = pts[0].X;		pts[i].Y = pts[0].Y;		cp++;
			}
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(cp < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
		g->DrawLines(&pen, pts, cp);
		}
	free(pts);		delete g;
	return true;
}

bool
PrintWin::foPolygon(lfPOINT *pt, int cp, char *nam)
{
	int i;
	Gdiplus::PointF *pts;
	POINT *ipts;
	if (!(pts = (Gdiplus::PointF*) malloc((cp+2) * sizeof(Gdiplus::PointF)))) return false;
	for (i = 0; i < cp; i++) {
		pts[i].X = (Gdiplus::REAL)pt[i].fx;		pts[i].Y = (Gdiplus::REAL)pt[i].fy;
		}
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);

	Gdiplus::SolidBrush *brush;
	brush = new Gdiplus::SolidBrush(GDI_COLOR(dFillCol));
	g->FillPolygon(brush, pts, cp);
	delete brush;

	if (hgo && hgo->OC_type != OC_HATCH && (ipts = (POINT*)malloc(cp * sizeof(POINT)))) {
		for (i = 0; i < cp; i++) {
			ipts[i].x = iround(pt[i].fx);		ipts[i].y = iround(pt[i].fy);
			}
		if (hasClip) ClipRect(&ClipRC);
		hgo->oPolygon(ipts, cp, NULL);
		free(ipts);
		}
	if ((dLineCol & 0xff000000) != 0xff000000) {
		if (pts[i - 1].X != pts[0].X || pts[i - 1].Y != pts[0].Y) {		//close polygon ?
			pts[i].X = pts[0].X;		pts[i].Y = pts[0].Y;		cp++;
			}
		Gdiplus::Pen pen(GDI_COLOR(dLineCol), (Gdiplus::REAL)un2fiy(LineWidth));
		pen.SetLineCap(LineCap::LineCapRound, LineCap::LineCapRound, DashCap::DashCapRound);
		pen.SetLineJoin(cp < 5 ? LineJoin::LineJoinMiter : LineJoin::LineJoinRound);
		g->DrawLines(&pen, pts, cp);
		}
	free(pts);		delete g;
	return true;
}

bool
PrintWin::oSolidPolygon(POINT *pt, int cp)
{
	int i;
	Gdiplus::Point *pts;
	if (!(pts = (Gdiplus::Point*) malloc((cp + 2) * sizeof(Gdiplus::Point)))) return false;
	for (i = 0; i < cp; i++) {
		pts[i].X = pt[i].x;		pts[i].Y = pt[i].y;
		}
	Graphics *g = new Graphics(hDC);
	g->SetPageUnit(UnitPixel);
	Gdiplus::SolidBrush brush(GDI_COLOR(dFillCol));
	g->FillPolygon(&brush, pts, cp);
	free(pts);		delete g;
	return true;
}

bool
PrintWin::GradPG(fPOINT3D *pts, long npt, double Lo, double Hi, double pLo, double pHi, lfPOINT *grad)
{
	return com_GradPG(pts, npt, Lo, Hi, pLo, pHi, grad, this, hDC);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Find a suitable www browser
void FindBrowser()
{
	char text[600], *tmp1;
	long size = 599;
	HKEY hdl;
	size_t cb;
	int i;

	//find the default browser
	if(ERROR_SUCCESS == RegQueryValue(HKEY_CLASSES_ROOT, "http\\shell\\open\\command", 
		text, &size) && size > 7) {
		if(text[0] == '"') {
			for(i = 1; i< size; i++) {
				if(text[i] == '"') {
					text[i] = '\0';
					break;
					}
				}
			WWWbrowser = _strdup(text+1);
			}
		else {
			for(i = size-1; i >5; i--) {
				if(0 == _stricmp(text+i-3, ".exe")) break;
				else text[i] = 0;
				}
			WWWbrowser = _strdup(text);
			}
		}
	//find user default data directory
	if(ERROR_SUCCESS == RegOpenKeyEx(HKEY_CURRENT_USER, "Environment", NULL,
		KEY_READ, &hdl)) {
		text[0] = 0;	size=599;
		RegQueryValueEx(hdl, "HOMEDRIVE", 0, 0, (unsigned char*)text, (unsigned long*)&size);
		size= 599;
		RegQueryValueEx(hdl, "HOMEPATH", 0, 0, (unsigned char*)(text+strlen(text)), 
			(unsigned long*)&size);
		if(text[0]) defs.currPath = _strdup(text);
		else {
			tmp1 = (char*)calloc(600, sizeof(char));
#ifdef USE_WIN_SECURE
			_dupenv_s(&tmp1, &cb,"HOMEDRIVE");
#else
			rlp_strcpy(tmp1, 600, getenv("HOMEDRIVE"));
#endif
			i = rlp_strcpy(text, 500, tmp1);
#ifdef USE_WIN_SECURE
			_dupenv_s((char**)&tmp1, &cb, "HOMEPATH");
#else
			rlp_strcpy(tmp1, 600, getenv("HOMEPATH"));
#endif
			rlp_strcpy(text + i, 500-i, tmp1);
			defs.currPath = _strdup(text);
			free(tmp1);
			}
		}
	//find user application data directory
	if(ERROR_SUCCESS == RegOpenKeyEx(HKEY_CURRENT_USER, 
		"Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders", NULL,
		KEY_READ, &hdl)) {
		text[0] = 0;	size=599;
		RegQueryValueEx(hdl, "AppData", 0, 0, (unsigned char*)text, (unsigned long*)&size);
#ifdef USE_WIN_SECURE
		strcat_s(text, 600, "\\RLPlot");
#else
		strcat(text, "\\RLPlot");
#endif
		defs.IniFile = _strdup(text);
		}
	//find country specific information
	//  its not a perfect place to do it, but a good one
	GetProfileString("intl", "sDecimal", ".", text, 2);
	if(text[0]) defs.DecPoint[0] = text[0];
	GetProfileString("intl", "sList", ".", text, 2);
	if(text[0]) defs.ColSep[0] = text[0];
	if(GetProfileInt("intl", "iMeasure", 0)) defs.dUnits = 2;
}

void CopyData(GraphObj *g, unsigned int cf)
{
	HGLOBAL hmem;
	long cb;
	unsigned char *dt = 0L;
	unsigned char *buf;

	if (!g)return;
	if (g->Id == GO_SPREADDATA){
		switch (cf) {
		case CF_TEXT:
			if (!g->Command(CMD_COPY_TSV, &dt, 0L))return;
			break;
		case CF_SYLK:
			if (!g->Command(CMD_COPY_SYLK, &dt, 0L))return;
			break;
		default:
			if (cf == cf_rlpxml && g->Command(CMD_COPY_XML, &dt, 0L)) break;
			else return;
			}
		}
	else if (g->Id == GO_DATALINE) {
		switch (cf) {
		case CF_TEXT:
			if (!g->Command(CMD_COPY_TSV, &dt, 0L)) return;
			break;
			}
		}
	else if (g->Id == GO_BOXPLOT){
		dt = (unsigned char*)((BoxPlot*)g)->CopySylk();
		}
	else if (g->Id == GO_XYSTAT){
		dt = ((xyStat*)g)->CopySylk();
		}
	else return;
	cb = (long)strlen((char*)dt);
	if (!cb) return;
	if (hmem = GlobalAlloc(GMEM_MOVEABLE, cb + 2)) {
		if (buf = (unsigned char *)GlobalLock(hmem)) {
			memcpy(buf, dt, cb + 1);
			GlobalUnlock(hmem);
			SetClipboardData(cf, hmem);
			free(dt);
			}
		}
}

void CopyGraph(GraphObj *g, unsigned int cf, anyOutput *o)
{
	HGLOBAL hmem;
	char *dt, *buf;
	long cb;

	if (!(dt = GraphToMem(g, &cb)))return;
	if (o) ShowCopyMark(o, &g->rDims, 1);
	if (hmem = GlobalAlloc(GMEM_MOVEABLE, cb + 1)) {
		if (buf = (char *)GlobalLock(hmem)) {
			memcpy(buf, dt, cb);
			buf[cb] = 0;
			GlobalUnlock(hmem);
			SetClipboardData(cf, hmem);
			}
		}
	free(dt);
}

void ScrollEvent(bool bVert, HWND hwnd, UINT type, GraphObj *g, OutputWin *w, bool isWheel)
{
	SCROLLINFO si;
	int LineStep, cmd;
	long pos;

	if (hwnd && g && w) {
		cmd = bVert ? CMD_SETVPOS : CMD_SETHPOS;
		si.fMask = SIF_ALL;
		si.cbSize = sizeof(SCROLLINFO);
		if (!(GetScrollInfo(hwnd, bVert ? SB_VERT : SB_HORZ, &si)))return;
		LineStep = (g->Id == GO_GRAPH || g->Id == GO_PAGE) ? 16 : isWheel? 5 : 1;
		switch (type){
		case SB_LINEUP:
			pos = si.nPos - LineStep;
			break;
		case SB_LINEDOWN:
			pos = si.nPos + LineStep;
			break;
		case SB_PAGEUP:
			if (g->Id == GO_SPREADDATA && bVert) {
				g->Command(CMD_PAGEUP, 0L, w);
				return;
				}
			pos = (si.nPos - (int)si.nPage) >= si.nMin ? (si.nPos - si.nPage) : si.nMin;
			break;
		case SB_PAGEDOWN:
			if (g->Id == GO_SPREADDATA && bVert) {
				g->Command(CMD_PAGEDOWN, 0L, w);
				return;
				}
			pos = (si.nPos + (int)si.nPage * 2) < si.nMax ? (si.nPos + si.nPage) : (si.nMax - si.nPage + 1);
			break;
		case SB_THUMBTRACK:		case SB_THUMBPOSITION:
			pos = si.nTrackPos;
			break;
		default:
			return;
			}
		g->Command(cmd, (void*)(&pos), w);
		}
}

long OpenFileFromHistory(OutputWin *w, GraphObj *g, int id)
{
	unsigned char *name = 0L;

	switch (id) {
	case 0:			name = defs.File1;			break;
	case 1:			name = defs.File2;			break;
	case 2:			name = defs.File3;			break;
	case 3:			name = defs.File4;			break;
	case 4:			name = defs.File5;			break;
	case 5:			name = defs.File6;			break;
	default:		return 0;
		}
	if (name && name[0] && FileExist((char*)name)) {
		g->Command(CMD_DROPFILE, name, w);
		defs.FileHistory((char*)name);
		w->FileHistory();
		}
	else {
		ErrorBox((char*)SCMS_FILE_MISSING);
		}
	return 0;
}

void DoExportPng(GraphObj *g, char *FileName, DWORD flags)
{
	/*
	List of image encoders
	image/bmp : {557cf400-1a04-11d3-9a73-0000f81ef32e}
	image/jpeg : {557cf401-1a04-11d3-9a73-0000f81ef32e}
	image/gif : {557cf402-1a04-11d3-9a73-0000f81ef32e}
	image/tiff : {557cf405-1a04-11d3-9a73-0000f81ef32e}
	image/png : {557cf406-1a04-11d3-9a73-0000f81ef32e}
	*/
	Gdiplus::Bitmap *bitmap;
	CLSID clsid;
	double res, width, height;
	long i, w, h, cb;
	anyOutput *bmo;
	WCHAR *wcFilename;
	DWORD ColGrect = 0x00ffffff, ColGrectLine = 0x00ffffff;

	if (!g || !FileName) return;
	res = 300.0;
	HideCopyMark(false);		HideTextCursor();
	width = g->GetSize(SIZE_GRECT_RIGHT) - g->GetSize(SIZE_GRECT_LEFT);
	height = g->GetSize(SIZE_GRECT_BOTTOM) - g->GetSize(SIZE_GRECT_TOP);
	if (GetBitmapRes(&res, &width, &height, (char*)"Export Portable Network Graphics (*.png)")){
		if (g->Id == GO_GRAPH) {
			ColGrect = ((Graph*)g)->GetColor(COL_GRECT);
			ColGrectLine = ((Graph*)g)->GetColor(COL_GRECTLINE);
			((Graph*)g)->SetColor(COL_GRECTLINE, ColGrect);
			}
		cb = rlp_strlen((unsigned char*)FileName);
		wcFilename = (WCHAR*)calloc(cb + 2, sizeof(WCHAR));
		for (i = 0; i < cb; i++) {
			MultiByteToWideChar(CP_ACP, 0, (LPCSTR)&FileName[i], 1, (LPWSTR)&wcFilename[i], sizeof(WCHAR));
			}
		w = (long)(width * res * Units[defs.dUnits].convert / 25.4);
		h = (long)(height * res * Units[defs.dUnits].convert / 25.4);
		bmo = NewBitmapClass(w, h, res, res);
		g->DoPlot(bmo);
		if (g->Id == GO_GRAPH) {
			((Graph*)g)->SetColor(COL_GRECTLINE, ColGrectLine);
			g->DoPlot(0L);
			}
		bitmap = new  Bitmap(((BitMapWin*)bmo)->scr, NULL);
		CLSIDFromString(L"{557cf406-1a04-11d3-9a73-0000f81ef32e}", &clsid);
		bitmap->Save(wcFilename, &clsid);
		delete(bitmap);
		DelBitmapClass(bmo);
		free(wcFilename);
		}
}

static int RenderClipboard(OutputWin *w, GraphObj *g, WPARAM wParam)
{
	Gdiplus::Bitmap *bitmap;
	HBITMAP hBitmap = NULL;
	DIBSECTION ds;
	HBITMAP hDDB;
	BitMapWin *CopyBMP = NULL;
	char *cb_txt;		//text for clipboard
	unsigned char *buf;	//clipboard string memory
	long buf_size = 0;
	HGLOBAL hmem;		//clipboard memory
	GraphObj *target = 0L;
	DWORD ColGrect = 0x00ffffff, ColGrectLine = 0x00ffffff;

	if (!CopyGO) return 0;
	HideCopyMark(false);		HideTextCursor();
	if (CopyGO->Id >= GO_PLOT && CopyGO->Id < GO_GRAPH) {
		target = CopyGO->parent;
		if (target && target->Id == GO_GRAPH) {
			ColGrect = ((Graph*)target)->GetColor(COL_GRECT);
			ColGrectLine = ((Graph*)target)->GetColor(COL_GRECTLINE);
			}
		else if (CurrGraph) target = CurrGraph;
		}
	else target = CopyGO;
	if ((wParam & 0xff) == CF_BITMAP){
#ifdef _DEBUG
		OutputDebugString("Rendering BMP\n");
#endif
		if (CopyGO->Id == GO_GRAPH) {
			ColGrect = ((Graph*)target)->GetColor(COL_GRECT);
			ColGrectLine = ((Graph*)target)->GetColor(COL_GRECTLINE);
			}
		CopyBMP = new BitMapWin(target);
		CopyBMP->OC_type = OC_UNKNOWN;
		CopyBMP->StartPage();
		if (target->Id = GO_GRAPH) ((Graph*)target)->SetColor(COL_GRECTLINE, ColGrect);
		target->DoPlot(CopyBMP);
		CopyBMP->EndPage();
		bitmap = new Bitmap(CopyBMP->scr, nullptr);
		bitmap->GetHBITMAP(NULL, &hBitmap);
		delete CopyBMP;						CopyBMP = NULL;
		ds.dsBmih.biCompression = BI_RGB;
		GetObject(hBitmap, sizeof(ds), &ds);
		hDDB = CreateDIBitmap(w->memDC, &ds.dsBmih, CBM_INIT, ds.dsBm.bmBits, (BITMAPINFO*)&ds.dsBmih, DIB_RGB_COLORS);
		SetClipboardData(CF_BITMAP, hDDB);
		DeleteObject(hBitmap);
		if (target->Id == GO_GRAPH) {
			((Graph*)target)->SetColor(COL_GRECTLINE, ColGrectLine);
			target->DoPlot(0L);
			}
		else if (CopyGO->Id == GO_PAGE) CopyGO->DoPlot(0L);
		}
	else if (((unsigned int)(wParam & 0xffff)) == cf_svg){
#ifdef _DEBUG
		OutputDebugString("Rendering SVG\n");
#endif
		cb_txt = DoCopySvg(target, &buf_size);
		if (cb_txt && buf_size && (hmem = GlobalAlloc(GMEM_MOVEABLE, buf_size + 2))){
			if (buf = (unsigned char *)GlobalLock(hmem)) {
				memcpy(buf, cb_txt, buf_size);	buf[buf_size] = 0;
				GlobalUnlock(hmem);			SetClipboardData(cf_svg, hmem);
				}
			}
		if (cb_txt) free(cb_txt);
		if (OCrestore) {
			if (CurrGraph) CurrGraph->DoPlot(OCrestore);
			OCrestore->HideMark();
			}
		}
	return 0;
}

long FAR PASCAL WndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static BitMapWin *CopyBMP = NULL;
	static bool CtrlDown = false, bAltKey = false;
	static w_char uc_char;
	int cc;
	static POINT line[5], mpos;
	static int cp_mark = 0;
	PAINTSTRUCT ps;
	POINTS pts;			//mouse coordinates
	OutputWin *w;
	GraphObj *g;
	static MouseEvent mev;
	HDC dc;
	long buf_size = 0;
	RECT rec;
	HDROP hDrop;
	RECT updRec;

	g = (GraphObj*) GetWindowLongPtr(hwnd, 0);
#ifdef _WIN64
	w = (OutputWin*) GetWindowLongPtr(hwnd, GWLP_USERDATA);
#else
	w = (OutputWin*)GetWindowLongPtr(hwnd, GWL_USERDATA);
#endif
	if (!Notary->IsValidPtr(w) || !Notary->IsValidPtr(g)) {
		switch (message){
		case WM_CLOSE:
		case WM_DESTROY:
		case WM_KILLFOCUS:
			break;
		default:
			return (long)DefWindowProc(hwnd, message, wParam, lParam);
			}
		}
	if (g && w) switch (message) {
	case WM_DROPFILES:
		hDrop = (HDROP)wParam;
		if (DragQueryFile(hDrop, 0, (LPSTR)TmpTxt, TMP_TXT_SIZE)) {
			if (TmpTxt && TmpTxt[0] && FileExist(TmpTxt)) {
				defs.FileHistory(TmpTxt);
				g->Command(CMD_DROPFILE, TmpTxt, w);
				w->FileHistory();
				}
			}
		return 0;
	case WM_SETFOCUS:
		if (g->Id == GO_GRAPH) CurrGraph = (Graph*)g;
		else CurrGraph = 0L;
		break;
	case WM_TIMER:
		if (bSuspend) return 0;
		defs.Idle(CMD_UPDATE);
		if (ProgressBarDlg) {
			ProgressBarDlg->Timer();
			return 0;
			}
		if (w->bHasFocus && !CurrGO && g->Id == GO_SPREADDATA) {
			if (GetAsyncKeyState(VK_LBUTTON) && GetCursorPos(&mpos)){	//Debug: Why not CurrGO ???
				if (ScreenToClient(hwnd, &mpos)) {
					mev.x = mpos.x;		mev.y = mpos.y;			mev.StateFlags = 1;
					if (GetAsyncKeyState(VK_SHIFT))mev.StateFlags |= 8;
					if (GetAsyncKeyState(VK_CONTROL))mev.StateFlags |= 16;
					g->Command(CMD_MOUSE_REPEAT, (void *)&mev, w);
					}
				return 0;
				}
			}
		if (bmCopyMark && oCopyMark && ((oCopyMark->OC_type & 0xff) == OC_BITMAP)) {
			bmCopyMark->CopyBitmap(0, 0, oCopyMark, rCopyMark.left, rCopyMark.top,
				rCopyMark.right - rCopyMark.left, rCopyMark.bottom - rCopyMark.top, false);
			bmCopyMark->SetLine(&liCopyMark1);
			line[0].x = line[1].x = line[4].x = 0;
			line[0].y = line[3].y = line[4].y = 0;
			line[1].y = line[2].y = rCopyMark.bottom - rCopyMark.top - 1;
			line[2].x = line[3].x = rCopyMark.right - rCopyMark.left - 1;
			bmCopyMark->oPolyline(line, 5);
			bmCopyMark->SetLine(&liCopyMark2);
			bmCopyMark->RLP.finc = 1.0;		bmCopyMark->RLP.fp = (cp_mark & 0x7);
			bmCopyMark->oPolyline(line, 5);
			oCopyMark->ShowBitmap(rCopyMark.left, rCopyMark.top, bmCopyMark);
			cp_mark++;
			if (bTxtCurIsVis && oTxtCur && ptTxtCurLine[0].y != ptTxtCurLine[1].y &&
				oTxtCur == oCopyMark && OverlapRect(&rCopyMark, &rTxtCur)){
				DrawCursor(((OutputWin*)oTxtCur)->hWnd, cTxtCur);
				}
			}
		if (!oTxtCur || (ptTxtCurLine[0].x == ptTxtCurLine[1].x &&
			ptTxtCurLine[0].y == ptTxtCurLine[1].y)) return 0;
		iTxtCurCount++;
		if (iTxtCurCount == -2) DrawCursor(((OutputWin*)oTxtCur)->hWnd, cTxtCur);
		if (iTxtCurCount < 4) break;
		iTxtCurCount = 0;
		if (bTxtCur && oTxtCur) {
			if (!bTxtCurIsVis) {
				DrawCursor(((OutputWin*)oTxtCur)->hWnd, cTxtCur);
				bTxtCurIsVis = true;
				}
			else {
				memcpy(&updRec, &rTxtCur, sizeof(RECT));
				IncrementMinMaxRect(&updRec, TxtCurWidth);
				if (dc = GetDC(((OutputWin*)oTxtCur)->hWnd)) {
					BitBlt(dc, updRec.left, updRec.top, updRec.right - updRec.left, updRec.bottom - updRec.top,
						((OutputWin*)oTxtCur)->memDC, updRec.left, updRec.top, SRCCOPY);
					ReleaseDC(((OutputWin*)oTxtCur)->hWnd, dc);
				}
				bTxtCurIsVis = false;
				}
			}
		defs.Idle(CMD_UPDATE);
		return 0;
	case WM_MOUSEMOVE:		case WM_RBUTTONUP:		case WM_LBUTTONUP:
	case WM_LBUTTONDOWN:	case WM_LBUTTONDBLCLK:
		if (message == WM_LBUTTONDOWN || message == WM_LBUTTONDBLCLK){
			HideTextCursor();
			}
		pts = MAKEPOINTS(lParam);		mev.x = pts.x;		mev.y = pts.y;
		mev.StateFlags = 0;
		if (wParam & MK_LBUTTON) mev.StateFlags |= 1;
		if (wParam & MK_MBUTTON) mev.StateFlags |= 2;
		if (wParam & MK_RBUTTON) mev.StateFlags |= 4;
		if (wParam & MK_SHIFT) mev.StateFlags |= 8;
		if (wParam & MK_CONTROL) mev.StateFlags |= 16;
		switch (message) {
		case WM_LBUTTONUP:
			mev.Action = MOUSE_LBUP;				break;
		case WM_RBUTTONUP:
			mev.Action = MOUSE_RBUP;				break;
		case WM_LBUTTONDBLCLK:
			mev.Action = MOUSE_LBDOUBLECLICK;		break;
		case WM_LBUTTONDOWN:
			mev.Action = MOUSE_LBDOWN;				break;
		case WM_MOUSEMOVE:
			mev.Action = MOUSE_MOVE;				break;
		default:
			return 0;
			}
		if(Notary->IsValidPtr(g)) g->Command(CMD_MOUSE_EVENT, (void *)&mev, w);
		return 0;
	case WM_MOUSEWHEEL:
		ScrollEvent(true, hwnd, (int)wParam > 0 ? SB_LINEUP : SB_LINEDOWN, g, w, true);
		return 0;
	case WM_KEYDOWN:
		w->bHasFocus = true;
		cc = (wParam & 0xff);
		if (g && w && (GetKeyState(VK_LCONTROL) || GetKeyState(VK_RCONTROL))){
			if (cc == 0xbb || cc == 0x6b) g->Command(CMD_ZOOM, (void*)&"+", w);
			else if (cc == 0xbd || cc == 0x6d) g->Command(CMD_ZOOM, (void*)&"-", w);
			else break;
			return 0;
			}
		break;
	case WM_CHAR:
		cc = (wParam & 0xff);
		if (bAltKey && uc_char) {
			if (uc_char >= 125) {
				g->Command(CMD_ADDCHARW, (void *)(&uc_char), w);
				uc_char = 0;	bAltKey = false;
				return 0;
				}
			else cc = (int)uc_char;
			uc_char = 0;	bAltKey = false;
			}
		g->Command(CMD_ADDCHAR, (void *)(&cc), w);
		return 0;
	case WM_SYSKEYDOWN:
		if ((wParam & 0xff) == VK_MENU) {
			bAltKey = true;		uc_char = 0;
			}
		break;
	case WM_SYSKEYUP:
		if ((wParam & 0xff) == VK_MENU) bAltKey = false;
		if (bAltKey) {
			cc = (wParam & 0xff);
			if (cc >= 0x60 && cc <= 0x69) {
				uc_char *= 10;	uc_char += (cc - 0x60);
				return 0;
				}
			else {
				uc_char *= 10;
				uc_char += (cc - '0');
				return 0;
				}
			}
		break;
	case WM_VSCROLL:		case WM_HSCROLL:
		ScrollEvent(message == WM_VSCROLL, hwnd, (unsigned int)wParam & 0x7fff, g, w, false);
		return 0;
		}
	switch (message) {
	case WM_CREATE:
		break;
	case WM_SIZE:
		if (g && w) g->Command(CMD_SETSCROLL, 0L, w);
		break;
	case WM_INITMENUPOPUP:		case WM_NCMOUSEMOVE:
		SetCursor(LoadCursor(NULL, IDC_ARROW));
		break;
	case WM_SETCURSOR:
		if (w) w->MouseCursor(MC_LAST, true);
		break;
	case WM_SETFOCUS:
		if (g && w) {
			if (!g->Command(message == WM_SETFOCUS ? CMD_SETFOCUS : CMD_KILLFOCUS, NULL, w))
				SetCursor(LoadCursor(NULL, IDC_ARROW));
			w->bHasFocus = true;
			}
		break;
	case WM_KILLFOCUS:
		if (g && w) {
			w->bHasFocus = false;
			}
		break;
	case WM_DESTROYCLIPBOARD:
		if (g && w) g->Command(CMD_HIDEMARK, 0L, w);
		HideCopyMark(false);
		return 0;
	case WM_RENDERALLFORMATS:
		// we do not support leaving data on the clipboard after exit
		OpenClipboard(hwnd);
		EmptyClipboard();
		CloseClipboard();
		return 0;
	case WM_RENDERFORMAT:
		if (g && w) switch (wParam & 0xff){
		case CF_BITMAP:
			return RenderClipboard(w, g, wParam);
		case CF_SYLK:		case CF_TEXT:
			if (!g) return 0;
			if (g->Id == GO_SPREADDATA && !CopyGO) CopyData(g, (unsigned int)(wParam & 0x7ff));
			else if (g->Id == GO_GRAPH && CopyGO) {
				if (CopyGO->Id == GO_DATALINE) CopyData(CopyGO, (unsigned int)(wParam & 0x7ff));
				}
			if(CopyGO){
				if (CopyGO->Id == GO_BOXPLOT) CopyData(CopyGO, (unsigned int)(wParam & 0x7ff));
				}
			break;
		default:
			if (((unsigned int)(wParam & 0xffff)) == cf_svg){
				return RenderClipboard(w, g, wParam);
				}
			else if ((unsigned int)(wParam & 0xffff) == cf_rlpxml) CopyData(g, (unsigned int)(wParam & 0xffff));
			break;
		}
		OCrestore = 0;
		return 0;
	case WM_COMMAND:
		wParam &= 0x7fff;
		if (g && w) switch (wParam) {
		case CM_EXIT:
			if (g->Command(CMD_CAN_CLOSE, 0L, 0L)) {
				if (g->parent && g->parent->Id == GO_SPREADDATA) {
					g->parent->Command(CMD_DELOBJ, g, 0L);
					}
				if (w)InvalidateOutput(w);
#ifdef _WIN64
				SetWindowLongPtr(hwnd, 0, 0L);				SetWindowLongPtr(hwnd, GWLP_USERDATA, 0L);
#else
				SetWindowLongPtr(hwnd, 0, 0L);				SetWindowLongPtr(hwnd, GWL_USERDATA, 0L);
#endif
				w->go = 0L;
				DestroyWindow(hwnd);
				}
			return 0;
		case CM_NEWINST:
			if (ShellCmd && ShellCmd[0])WinExec(ShellCmd, SW_SHOW);
			return 0;
		case CM_T_CHARMAP:
			WinExec("charmap", SW_SHOW);
			return 0;
		case CM_PRISETUP:
			ConfigPrinter();
			return true;
		case CM_PASTE:
			Undo.SetDisp(w);
			w->MouseCursor(MC_WAIT, true);
			if (g->Id == GO_SPREADDATA) TestClipboard(g);
			else if (g->Id == GO_PAGE || g->Id == GO_GRAPH){
				if (CurrGO) {
					if (CurrGO->Id == GO_TEXTFRAME)CurrGO->Command(CMD_PASTE, 0L, w);
					else if (CurrGO->Id == GO_LABEL)CurrGO->Command(CMD_PASTE, 0L, w);
					else TestClipboard(g);
					}
				else TestClipboard(g);
				}
			g->Command(CMD_MOUSECURSOR, 0L, w);
			return 0;
		case CM_COPY:			case CM_CUT:		case CM_COPYGRAPH:
			OCrestore = w;
			EmptyClip();
			if (CurrGO && g->Id != GO_SPREADDATA) {
				if (CurrGO->Id == GO_POLYLINE || CurrGO->Id == GO_POLYGON || CurrGO->Id == GO_RECTANGLE
					|| CurrGO->Id == GO_ROUNDREC || CurrGO->Id == GO_ELLIPSE || CurrGO->Id == GO_BEZIER) {
					OpenClipboard(hwnd);
					CopyGraph(CurrGO, cf_rlpobj, w);		CopyGO = CurrGO;
					CloseClipboard();						return 0;
					}
				else if (CurrGO->Id == GO_TEXTFRAME) {
					if (CurrGO->Command(CMD_COPY, 0L, w)) return 0;
					}
				else if (CurrGO->Id == GO_DATALINE) {
					if (CurrGO->Command(CMD_COPY, 0L, w)) return 0;
					}
				}
			OpenClipboard(hwnd);
			if (g->Id == GO_SPREADDATA && g->Command((unsigned int)(wParam & 0x7ff) == CM_CUT ? CMD_CUT : CMD_QUERY_COPY, 0L, w) && !CurrPlot) {
				SetClipboardData(CF_TEXT, NULL);
				SetClipboardData(CF_SYLK, NULL);
				SetClipboardData(cf_rlpxml, NULL);
				}
			else if (CurrPlot && CurrPlot->Id == GO_BOXPLOT) {
				CopyGO = CurrPlot;
				SetClipboardData(CF_SYLK, NULL);
				SetClipboardData(CF_BITMAP, NULL);
				SetClipboardData(cf_svg, NULL);
				CopyGraph(g, cf_rlpobj, w);
				}
			else if (g->Id == GO_PAGE) {
				SetClipboardData(CF_BITMAP, NULL);
				if (CurrGraph) {
					CopyGraph(CurrGraph, cf_rlpobj, w);		CopyGO = CurrGraph;
					}
				}
			else if ((unsigned int)(wParam & 0x7ff) == CM_CUT)return 0;
			else if (CurrGraph && CurrGraph->Id == GO_GRAPH){
				SetClipboardData(CF_BITMAP, NULL);
				SetClipboardData(cf_svg, NULL);
				CopyGraph(CurrGraph, cf_rlpobj, w);			CopyGO = CurrGraph;
				w->bHasFocus = true;
				}
			else if (g->Id == GO_GRAPH){
				SetClipboardData(CF_BITMAP, NULL);
				SetClipboardData(cf_svg, NULL);
				CopyGraph(g, cf_rlpobj, w);					CopyGO = g;
				w->bHasFocus = true;
				}
			CloseClipboard();
			return 0;
		case CM_UPDATE:
			g->Command(CMD_UPDATE, 0L, w);
			return 0;
		case CM_OPEN:
			g->Command(CMD_OPEN, 0L, w);
			return 0;
		case CM_FILE1:	case CM_FILE2:	case CM_FILE3:
		case CM_FILE4:	case CM_FILE5:	case CM_FILE6:
			return OpenFileFromHistory(w, g, (int)(wParam & 0x7ff) - CM_FILE1);
		case CM_FILLRANGE:
			g->Command(CMD_FILLRANGE, (void *)NULL, w);
			return 0;
		case CM_ARITH:
			g->Command(CMD_ARITH, (void *)NULL, w);
			return 0;
		case CM_ABOUT:
			RLPlotInfo();
			return 0;
		case CM_ZOOM25:
			g->Command(CMD_ZOOM, (void*)&"25\0", w);
			return 0;
		case CM_ZOOM50:
			g->Command(CMD_ZOOM, (void*)&"50\0", w);
			return 0;
		case CM_ZOOM100:
			g->Command(CMD_ZOOM, (void*)&"100\0", w);
			return 0;
		case CM_ZOOM200:
			g->Command(CMD_ZOOM, (void*)&"200\0", w);
			return 0;
		case CM_ZOOM400:
			g->Command(CMD_ZOOM, (void*)&"400\0", w);
			return 0;
		case CM_ZOOMIN:
			g->Command(CMD_ZOOM, (void*)&"+\0", w);
			return 0;
		case CM_ZOOMOUT:
			g->Command(CMD_ZOOM, (void*)&"-\0", w);
			return 0;
		case CM_ZOOMFIT:
			g->Command(CMD_ZOOM, (void*)&"fit\0", w);
			return 0;
		case CM_ADDPLOT:
			g->Command(CMD_ADDPLOT, 0L, w);
			return 0;
		case CM_LEGEND:
			g->Command(CMD_LEGEND, 0L, w);
			return 0;
		case CM_LAYERS:
			g->Command(CMD_LAYERS, 0L, w);
			return 0;
		case CM_NEWGRAPH:
			g->Command(CMD_NEWGRAPH, 0L, w);
			return 0;
		case CM_NEWPAGE:
			g->Command(CMD_NEWPAGE, 0L, w);
			return 0;
		case CM_DELGRAPH:
			g->Command(CMD_DELGRAPH, 0L, w);
			return 0;
		case CM_SAVE:
			g->Command(CMD_SAVE, 0L, w);
			return 0;
		case CM_SAVEAS:
			g->Command(CMD_SAVEAS, 0L, w);
			return 0;
		case CM_REDRAW:
			HideCopyMark(false);		w->HideMark();		HideTextCursor();
			if (w->Erase(defs.Color(COL_BG))) g->Command(CMD_REDRAW, 0L, w);
			defs.UpdRect(w, &g->rDims);
			return 0;
		case CM_DELOBJ:
			if (CurrGO && CurrGO->parent && CurrGO->parent->
				Command(CMD_DELOBJ, (void*)CurrGO, w)) {
				CurrGO = 0L;
				if (w->Erase(defs.Color(COL_BG))) g->DoPlot(w);
				}
			else if (!CurrGO) InfoBox((char*)"No object selected!");
			return 0;
		case CM_EXPORT:
			w->MouseCursor(MC_WAIT, false);			OpenExportName(g, 0L);
			g->DoPlot(w);							w->MouseCursor(MC_ARROW, true);
			return 0;
		case CM_PRINT:
			if ((g->Id == GO_GRAPH || g->Id == GO_PAGE) && ((Graph*)g)->getDisp()) (((Graph*)g)->getDisp())->HideMark();
			HideTextCursor();		CurrGO = NULL;
			w->MouseCursor(MC_WAIT, false);
			Printer = new PrintWin();
			if (g->Id == GO_SPREADDATA){
				g->Command(CMD_PRINT, 0L, Printer);
				}
			else if (Printer && Printer->StartPage()) {
				SetCursor(LoadCursor(0L, IDC_WAIT));
				rec.left = Printer->un2ix(g->GetSize(SIZE_GRECT_LEFT));
				rec.right = Printer->un2ix(g->GetSize(SIZE_GRECT_RIGHT));
				rec.top = Printer->un2iy(g->GetSize(SIZE_GRECT_TOP));
				rec.bottom = Printer->un2iy(g->GetSize(SIZE_GRECT_BOTTOM));
				if (g->hasTransp()) {
					if ((CopyBMP = new BitMapWin(rec.right - rec.left, rec.bottom - rec.top,
						Printer->hres, Printer->vres)) && CopyBMP->StartPage()) {
						CopyBMP->setVPorg(-Printer->co2fix(g->GetSize(SIZE_GRECT_LEFT)), -Printer->co2fiy(g->GetSize(SIZE_GRECT_TOP)), 1.0);
						g->DoPlot(CopyBMP);			CopyBMP->EndPage();
						Printer->CopyBitmap(rec.left, rec.top, CopyBMP, 0, 0, CopyBMP->DeskRect.right,
							CopyBMP->DeskRect.bottom, false);
						delete CopyBMP;						CopyBMP = NULL;
						Printer->EndPage();
						}
					}
				else {
					g->DoPlot(Printer);
					Printer->EndPage();
					}
				g->DoPlot(w);
				SetCursor(LoadCursor(0L, IDC_ARROW));
				}
			delete Printer;			Printer = NULL;
			w->MouseCursor(MC_ARROW, false);
			return 0;
		case CM_ADDROWCOL:
			g->Command(CMD_ADDROWCOL, (void *)NULL, w);
			return 0;
		case CM_DEFAULTS:
			g->Command(CMD_CONFIG, 0L, w);
			return 0;
		case CM_ADDAXIS:
			g->Command(CMD_ADDAXIS, 0L, w);
			return 0;
		case CM_T_STANDARD:
			cc = TM_STANDARD;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_DRAW:
			cc = TM_DRAW;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_POLYLINE:
			cc = TM_POLYLINE;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_POLYGON:
			cc = TM_POLYGON;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_RECTANGLE:
			cc = TM_RECTANGLE;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_ROUNDREC:
			cc = TM_ROUNDREC;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_ELLIPSE:
			cc = TM_ELLIPSE;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_ARROW:
			cc = TM_ARROW;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_T_TEXT:
			cc = TM_TEXT;
			g->Command(CMD_TOOLMODE, (void*)(&cc), w);
			return 0;
		case CM_DELKEY:		case CM_LEFTARRKEY:		case CM_RIGHTARRKEY:
		case CM_UPARRKEY:   case CM_DOWNARRKEY:		case CM_POS_FIRST:
		case CM_POS_LAST:	case CM_SHLEFT:			case CM_SHRIGHT:
		case CM_SHUP:		case CM_SHDOWN:			case CM_TAB:
		case CM_SHTAB:		case CM_SHPGUP:			case CM_SHPGDOWN:
		case CM_PGUP:		case CM_PGDOWN:
			if (bSuspend) return 0;
			cc = CMD_NONE;	bSuspend = true;
			switch ((unsigned int)(wParam & 0x7ff)) {
			case CM_DELKEY:			cc = CMD_DELETE;		break;
			case CM_LEFTARRKEY:		cc = CMD_CURRLEFT;		break;
			case CM_RIGHTARRKEY:	cc = CMD_CURRIGHT;		break;
			case CM_UPARRKEY:		cc = CMD_CURRUP;		break;
			case CM_DOWNARRKEY:		cc = CMD_CURRDOWN;		break;
			case CM_POS_FIRST:		cc = CMD_POS_FIRST;		break;
			case CM_POS_LAST:		cc = CMD_POS_LAST;		break;
			case CM_SHLEFT:			cc = CMD_SHIFTLEFT;		break;
			case CM_SHRIGHT:		cc = CMD_SHIFTRIGHT;	break;
			case CM_SHUP:			cc = CMD_SHIFTUP;		break;
			case CM_SHDOWN:			cc = CMD_SHIFTDOWN;		break;
			case CM_TAB:			cc = CMD_TAB;			break;
			case CM_SHTAB:			cc = CMD_SHTAB;			break;
			case CM_SHPGUP:			cc = CMD_SHPGUP;		break;
			case CM_SHPGDOWN:		cc = CMD_SHPGDOWN;		break;
			case CM_PGUP:			cc = CMD_PAGEUP;		break;
			case CM_PGDOWN:			cc = CMD_PAGEDOWN;		break;
			}
			g->Command(cc, (void *)NULL, w);
			bSuspend = false;
			return 0;
		case CM_UNDO:
			g->Command(CMD_UNDO, 0L, w);
			return 0;
		case CM_INSROW:
			g->Command(CMD_INSROW, 0L, w);
			return 0;
		case CM_INSCOL:
			g->Command(CMD_INSCOL, 0L, w);
			return 0;
		case CM_DELROW:
			g->Command(CMD_DELROW, 0L, w);
			return 0;
		case CM_DELCOL:
			g->Command(CMD_DELCOL, 0L, w);
			return 0;
		case CM_SMPLSTAT:
			if (g->data) rep_samplestats(g, g->data);
			return 0;
		case CM_REPCMEANS:
			if (g->data) rep_compmeans(g, g->data);
			return 0;
		case CM_REPANOV:
			if (g->data) rep_anova(g, g->data);
			return 0;
		case CM_REPBDANOV:
			if (g->data) rep_bdanova(g, g->data);
			return 0;
		case CM_REPTWANR:
			if (g->data) rep_twoway_anova(g, g->data);
			return 0;
		case CM_REPKRUSKAL:
			if (g->data) rep_kruskal(g, g->data);
			return 0;
		case CM_REPTWANOV:
			if (g->data) rep_twanova(g, g->data);
			return 0;
		case CM_REPFRIEDM:
			if (g->data) rep_fmanova(g, g->data);
			return 0;
		case CM_REPREGR:
			if (g->data) rep_regression(g, g->data);
			return 0;
		case CM_REPTWREGR:
			rep_twregression(g, g->data);
			return 0;
		case CM_REPPCA:
			rep_pca(g, g->data);
			return 0;
		case CM_ROBUSTLINE:
			if (g->data) rep_robustline(g, g->data);
			return 0;
		case CM_CORRELM:
			if (g->data) rep_correl(g, g->data, 0);
			return 0;
		case CM_CORRELT:
			if (g->data) rep_correl(g, g->data, 1);
			return 0;
		case CM_REPTWOWAY:
			if (g->data) rep_twowaytable(g, g->data);
			return 0;
		case CM_BT_MEAN:
			rep_bootstrap_mean(g, g->data);
			return 0;
		case CM_BT_CORREL:
			rep_bootstrap_correlation(g, g->data);
			return 0;
		case CM_BT_REGR:
			rep_bootstrap_regression(g, g->data);
			return 0;
		default:
#ifdef _DEBUG
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, "Command 0x%x (%d)\nreceived", (int)(wParam & 0x7ff),
				(int)(wParam & 0x7ff) & 0xffff);
#else
			sprintf(TmpTxt, "Command 0x%x (%d)\nreceived", (int)(wParam & 0x7ff),
				(int)(wParam & 0x7ff));
#endif
			MessageBox(hwnd, TmpTxt, "Info", MB_OK | MB_ICONINFORMATION);
#endif
			break;
			}
		return 0;
	case WM_CLOSE:
		if (Notary->IsValidPtr(g) && g->Command(CMD_CAN_CLOSE, 0L, 0L)) {
			if (Notary->IsValidPtr(w))InvalidateOutput(w);
#ifdef _WIN64
			SetWindowLongPtr(hwnd, 0, 0L);		SetWindowLongPtr(hwnd, GWLP_USERDATA, 0L);
#else
			SetWindowLongPtr(hwnd, 0, 0L);		SetWindowLongPtr(hwnd, GWL_USERDATA, 0L);
#endif
			w->go = 0L;
			if (g && Notary->IsValidPtr(g) && g->parent) g->parent->Command(CMD_DELOBJ, g, w);
			DestroyWindow(hwnd);
			}
		else if (Notary->IsValidPtr(g)) return 0L;
		else if (!g) DestroyWindow(hwnd);
		break;
	case WM_DESTROY:
		if (hwnd == MainWnd){
			KillTimer(hwnd, 1);			PostQuitMessage(0);
			}
		break;
	case WM_PAINT:
		if (w) {
			dc = BeginPaint(hwnd, &ps);
			if (dc) {
				w->UpdateRect(dc, ps.rcPaint);
				}
			else {
				//RLPlot is probably frozen
				FrozenHWND(g->data, w);
				}
			EndPaint(hwnd, &ps);
			}
		break;
		}
	return (long)DefWindowProc(hwnd, message, wParam, lParam);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Windos entry point
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrevInstance,
	 LPSTR lpCmdLine, int nCmdShow )
{
	WNDCLASS wndclass;
	MSG	msg;
	DefsRW *drw;
	HWND hwnd = NULL;
	GdiplusStartupInput gdiplusStartupInput;
	ULONG_PTR gdiplusToken;
	DWORD hasMsg;
	bool bShutDown = false;
	int i = 50;
	char *tmp_txt;

	ShellCmd = rlp_strdup(GetCommandLine());;
	FindBrowser();
	tmp_txt = rlp_strdup(ShellCmd);
	for (i = rlp_strlen(tmp_txt); i; i--) {
		if (tmp_txt[i] = '\\') {
			tmp_txt[i] = '\0';
			}
		}
	//OS dependent initialization
	dlgtxtheight = 16;
	//init printer dialog
	ZeroMemory(&PriDlg, sizeof(PriDlg));
	PriDlg.lStructSize = sizeof(PriDlg);
	PriDlg.hwndOwner = hwnd;
	PriDlg.hDevMode = NULL;     // Don't forget to free or store hDevMode.
	PriDlg.hDevNames = NULL;     // Don't forget to free or store hDevNames.
	PriDlg.Flags = PD_RETURNDEFAULT;
	PriDlg.nCopies = 1;
	PriDlg.nFromPage = 0xFFFF;
	PriDlg.nToPage = 0xFFFF;
	PriDlg.nMinPage = 1;
	PriDlg.nMaxPage = 0xFFFF;

	//check command line
	if (lpCmdLine && lpCmdLine[0]) {
#ifdef USE_WIN_SECURE
		sprintf_s(TmpTxt, TMP_TXT_SIZE, "%s", lpCmdLine);
#else
		sprintf(TmpTxt, "%s", lpCmdLine);
#endif
		if(TmpTxt[0]) {
			rmquot(TmpTxt);
 			if(TmpTxt && FileExist(TmpTxt)) LoadFile= _strdup(TmpTxt);
			}
		}
#ifdef _WIN64
	gui_version = rlp_strdup((char*)" 64 bit Windows");
#else
	gui_version = rlp_strdup((char*)" 32 bit Windows");
#endif
	//set main window class
	hInstance = hInst;
	wndclass.style = CS_BYTEALIGNWINDOW | CS_DBLCLKS;
	wndclass.lpfnWndProc = (WNDPROC)WndProc;
	wndclass.cbClsExtra = 0;
	wndclass.cbWndExtra = sizeof(GraphObj*);
	wndclass.hInstance = hInstance;
	wndclass.hIcon = LoadIcon(hInst, MAKEINTRESOURCE(IDI_EVAL));
	wndclass.hCursor = 0L;
	wndclass.hbrBackground = NULL;
	wndclass.lpszMenuName = 0L;
	wndclass.lpszClassName = name;

	Notary = new notary();
	RegisterClass(&wndclass);
	InitTextCursor(true);
	accel = LoadAccelerators(hInstance, MAKEINTRESOURCE(ACCELERATORS_1));
	GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);
	ShowBanner(true);
	do {
		hasMsg = WAIT_OBJECT_0;
		if (bShutDown) {
			GetMessageA(&msg, NULL, 0, 0);
			TranslateMessage(&msg);
			DispatchMessage(&msg);
			hasMsg++;
			}
		else {
			GetMessage(&msg, NULL, 0, 0);
			TranslateMessage(&msg);
			TranslateAccelerator(msg.hwnd, accel, &msg);
			if (msg.message == WM_QUIT) bShutDown = true;
			DispatchMessage(&msg);
			}
		} while (hasMsg == WAIT_OBJECT_0);
	GdiplusShutdown(gdiplusToken);
	if(defs.IniFile) {
		if(drw = new DefsRW()){
			drw->FileIO(FILE_WRITE);		delete drw;
			}
		}
	if(WWWbrowser) free(WWWbrowser);
	if(LoadFile) free(LoadFile);
	if (ShellCmd) free(ShellCmd);
	SpreadMain(false);
	if(Printer) delete Printer;
	InitTextCursor(false);
	UnregisterClass(name, hInstance);
	return (int)msg.wParam;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Dialog window: Windows specific
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
const char dlgname[] = "RLDLGWIN";

LRESULT FAR PASCAL DlgWndProc(HWND hwnd, UINT message, UINT wParam, LPARAM lParam)
{
	OutputWin *w;
	tag_DlgObj *d;
	PAINTSTRUCT ps;
	HDC dc;
	MouseEvent mev;
	POINTS pts;			//mouse coordinates
	int i;
	w_char cc;

	d = (tag_DlgObj*) GetWindowLongPtr(hwnd, 0);
#ifdef _WIN64
	w = (OutputWin *)(GetWindowLongPtr(hwnd, GWLP_USERDATA));
#else
	w = (OutputWin *)(GetWindowLongPtr(hwnd, GWL_USERDATA));
#endif
	if (!Notary->IsValidPtr(w) || !Notary->IsValidPtr(d)) {
		switch (message){
		case WM_CLOSE:
		case WM_DESTROY:
			break;
		default:
			return (long)DefWindowProc(hwnd, message, wParam, lParam);
			}
		}
	switch (message) {
	case WM_CREATE:
		break;
	case WM_KILLFOCUS:
		HideTextCursor();
		if(d) d->Command(CMD_ENDDIALOG, NULL, w);
		return 0;
	case WM_TIMER:
		if(prog_bar_ptr) ((ProgressBar*)prog_bar_ptr)->Command(CMD_TIMER, d, w);
		else if(d) d->Command(CMD_ENDDIALOG, d, w);
		return 0;
	case WM_DESTROY:	case WM_CLOSE:
		if (w && w->isDlg()) {
			w->FlagIsDialog(false);
			InfoBox((char*)SCMS_DLG_CLOSE);
			}
		if (d) {
			d->Command(CMD_UNLOCK, 0L, w);
			d->Command(CMD_ENDDIALOG, 0L, w);
			SetWindowLongPtr(hwnd, 0, NULL);
			}
		if(w) {
			w->hWnd = 0L;
#ifdef _WIN64
			SetWindowLongPtr(hwnd, GWLP_USERDATA, NULL);
#else
			SetWindowLongPtr(hwnd, GWL_USERDATA, NULL);
#endif
			delete w;
			}
		break;
	case WM_CHAR:
		if(0x09 == (cc = wParam & 0xff)) break;		//ignore Tab
		if(d && w) d->Command(CMD_ADDCHAR, (void *)(& cc), w);
		break;
	case WM_LBUTTONDOWN:	case WM_LBUTTONDBLCLK:
		HideTextCursor();
	case WM_RBUTTONUP:		case WM_LBUTTONUP:		case WM_MOUSEMOVE:
		pts = MAKEPOINTS(lParam);	mev.x = pts.x;		mev.y = pts.y;
		mev.StateFlags = mev.Action = 0;
		if(wParam & MK_LBUTTON) mev.StateFlags |= 1;
		if(wParam & MK_MBUTTON) mev.StateFlags |= 2;
		if(wParam & MK_RBUTTON) mev.StateFlags |= 4;
		if(wParam & MK_SHIFT) mev.StateFlags |= 8;
		if(wParam & MK_CONTROL) mev.StateFlags |= 16;
		if(message == WM_MOUSEMOVE) mev.Action = MOUSE_MOVE;
		else if(message == WM_LBUTTONDOWN) mev.Action = MOUSE_LBDOWN;
		else if(message == WM_LBUTTONUP) mev.Action = MOUSE_LBUP;
		else if(message == WM_RBUTTONUP) mev.Action = MOUSE_RBUP;
		else if(message == WM_LBUTTONDBLCLK) mev.Action = MOUSE_LBDOUBLECLICK;
		if(d && w) d->Command(CMD_MOUSE_EVENT, (void *)&mev, w);
		return 0;
	case WM_MOUSEWHEEL:
		d->Command((int)wParam > 0 ? CMD_SCROLL_UP : CMD_SCROLL_DOWN, 0L, w);
		return 0;
	case WM_COMMAND:
		wParam &= 0x7fff;
		i = 0;
		switch(wParam) {
		case CM_DELKEY:			i = CMD_DELETE;		break;
		case CM_LEFTARRKEY:		i = CMD_CURRLEFT;	break;
		case CM_RIGHTARRKEY:	i = CMD_CURRIGHT;	break;
		case CM_UPARRKEY:		i = CMD_CURRUP;		break;
		case CM_DOWNARRKEY:		i = CMD_CURRDOWN;	break;
		case CM_TAB:			i = CMD_TAB;		break;
		case CM_SHTAB:			i = CMD_SHTAB;		break;
		case CM_POS_FIRST:		i = CMD_POS_FIRST;	break;
		case CM_POS_LAST:		i = CMD_POS_LAST;	break;
		case CM_SHLEFT:			i = CMD_SHIFTLEFT;	break;
		case CM_SHRIGHT:		i = CMD_SHIFTRIGHT;	break;
		case CM_UNDO:			i = CMD_UNDO;		break;
		case CM_COPY:			i = CMD_COPY;		break;
		case CM_PASTE:			i = CMD_PASTE;		break;
		case CM_CUT:			i = CMD_CUT;		break;
			}
		if (i && d && w) d->Command(i, 0L, w);
		return 0;
	case WM_PAINT:
		dc = BeginPaint(hwnd, &ps);
		if(w && dc){
      		w->HideMark();		w->UpdateRect(dc, ps.rcPaint);
			}
		EndPaint(hwnd, &ps);
		break;
	}
	return DefWindowProc(hwnd, message, wParam, lParam);
}

void *CreateDlgWnd(char *title, int x, int y, int width, int height, tag_DlgObj *d, DWORD flags)
{
	WNDCLASS wndclass;
	OutputWin *w;
	HWND hDlg, hFoc;
	RECT rec, BoxRec, DeskRect;
	DWORD ws;

	wndclass.style = CS_BYTEALIGNWINDOW | CS_DBLCLKS;
	wndclass.lpfnWndProc = (WNDPROC)DlgWndProc;
	wndclass.cbClsExtra = 0;
	wndclass.cbWndExtra = sizeof(tag_DlgObj *);
	wndclass.hInstance = hInstance;
	wndclass.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_EVAL));
	wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground = NULL;
	wndclass.lpszMenuName = NULL;
	wndclass.lpszClassName = dlgname;
	RegisterClass(&wndclass);
	hFoc = GetFocus();
	if(hFoc) {
		GetWindowRect(hFoc, &rec);
		x += rec.left;		y += rec.top;
		}
	width += (int)defs.DlgSizeAdj.x;			height += (int)defs.DlgSizeAdj.y;
	ws = WS_POPUP | WS_VISIBLE | WS_CLIPSIBLINGS;
	if(!(flags & 0x2)) ws |= WS_CAPTION;
	if (flags & 0x01) ws |= WS_EX_TOPMOST;
	hDlg = CreateWindow(dlgname, title,	ws,
		x, y, width, height, GetFocus(), NULL, hInstance, NULL);
	w = new OutputWin(0L, hDlg);
	if(hDlg && w && w->Erase(0x00e0e0e0L)) {
		((DlgRoot*)d)->hDialog = hDlg;
#ifdef _WIN64
		SetWindowLongPtr(hDlg, GWLP_USERDATA, (LONG_PTR)w);
#else
		SetWindowLongPtr(hDlg, GWL_USERDATA, (LONG_PTR)w);
#endif
		SetWindowLongPtr(hDlg, 0, (LONG_PTR)d);
		if(flags & 0x01) {					//center on screen
			GetWindowRect(hDlg, &BoxRec);
			GetClientRect(GetDesktopWindow(), &DeskRect);
			SetWindowPos(hDlg, HWND_TOPMOST, (DeskRect.right -DeskRect.left)/2 -
				(BoxRec.right - BoxRec.left)/2, (DeskRect.bottom -DeskRect.top)/2 -
				(BoxRec.bottom- BoxRec.top)/2, BoxRec.right - BoxRec.left,
				BoxRec.bottom - BoxRec.top, 0);
			}
		else w->FlagIsDialog(true);
		if (flags & 0x08) {
			SetTimer(hDlg, 1, 100, 0L);
			}
		UpdateWindow(hDlg);			d->DoPlot(w);
		ShowWindow(hDlg, SW_SHOW);
		}
	else {
		if(w) delete (w);			return 0L;
		}
	return hDlg;
}

void LoopDlgWnd() 	//keep message processing running
{
	MSG	msg;
	
	GetMessage(&msg, NULL, 0, 0);
	TranslateMessage(&msg);
	TranslateAccelerator(msg.hwnd, accel, &msg);
	DispatchMessage(&msg);
}

bool NoWaitDlgLoop()
{
	MSG	msg;
	
	if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)){
		if(msg.message == WM_TIMER && prog_bar_ptr) {
			((ProgressBar *)prog_bar_ptr)->Command(CMD_TIMER, NULL, NULL);
			return true;
			}
		TranslateMessage(&msg);
		TranslateAccelerator(msg.hwnd, accel, &msg);
		DispatchMessage(&msg);
		return true;
		}
	return false;
}

void CloseDlgWnd(void *hDlg)
{
	OutputWin *w;

#ifdef _WIN64
	w = (OutputWin*) GetWindowLongPtr((HWND)hDlg, GWLP_USERDATA);
#else
	w = (OutputWin*)GetWindowLongPtr((HWND)hDlg, GWL_USERDATA);
#endif
	if (w) w->FlagIsDialog(false);
	HideTextCursor();
	HideCopyMark(false);
	if(hDlg) SendMessage((HWND)hDlg, WM_CLOSE, 0, 0);
}

void ShowDlgWnd(void *hDlg)
{
	ShowWindow((HWND)hDlg, SW_SHOW);
	SetFocus((HWND)hDlg);
}

void ResizeDlgWnd(void *hDlg, int w, int h)
{
	HWND hwnd = (HWND)hDlg;
	RECT rc;

	GetWindowRect(hwnd, &rc);
	SetWindowPos(hwnd, HWND_TOPMOST, rc.left, rc.top, w, h, 0);
	ShowDlgWnd(hDlg);
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// OS independent interface to Windows specific classes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
anyOutput *NewDispClass(GraphObj *g)
{
	return new OutputWin(g, 0L);
}

bool DelDispClass(anyOutput *w)
{
	int i;

	if (!Notary->IsValidPtr(w)) return false;
	InvalidateOutput(w);			//kill text cursor and animated rectangle
	for (i = 0; i < 20 && NoWaitDlgLoop(); i++);
	delete (OutputWin*)w;
	return true;
}

anyOutput *NewBitmapClass(int w, int h, double hr, double vr)
{
	return new BitMapWin(w, h, hr, vr);
}

bool DelBitmapClass(anyOutput *w)
{
	int i;

	if (!Notary->IsValidPtr(w)) return false;
//	InvalidateOutput(w);			//kill text cursor and animated rectangle
	for (i = 0; i < 20 && NoWaitDlgLoop(); i++);
	delete (BitMapWin*)w;
	return true;
}
