//UtilObj.cpp, (c) 2000-2025 by R. Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include "rlp_strings.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags
#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif

def_vars defs;

static LineDEF ETbgnn = { 0.0, 1.0, 0x00f0f0f0L, 0L };
static LineDEF ETbgna = { 0.0, 1.0, 0x00ffffffL, 0L };
static LineDEF ETbgmn = { 0.0, 1.0, 0x00cbcbcbL, 0L };
static LineDEF ETbgma = { 0.0, 1.0, 0x00ffffd0L, 0L };
static LineDEF yLine = { 0.0, 1.0, 0x0000ffffL, 0L };
static LineDEF BlackLine = { 0.0, 1.0, 0x00000000L, 0L };
static LineDEF GrayLine = { 0.0, 1.0, 0x00c0c0c0L, 0L };
static LineDEF WhiteLine = { 0.0, 1.0, 0x00ffffffL, 0L };

static FillDEF ETfbnn = {FILL_NONE, 0x00f4f4f4L, 1.0, NULL, 0x00ffffffL};
static FillDEF ETfbna = {FILL_NONE, 0x00ffffffL, 1.0, NULL, 0x00ffffffL};
static FillDEF ETfbmn = {FILL_NONE, 0x00f8e0e0L, 1.0, NULL, 0x00ffffffL};
static FillDEF ETfbma = {FILL_NONE, 0x00ffffb0L, 1.0, NULL, 0x00ffffffL};
static FillDEF yFill = {FILL_NONE, 0x0000ffffL, 1.0, NULL, 0x0000ffffL};
static FillDEF BlackFill = {FILL_NONE, 0x0L, 1.0, NULL, 0x0L};
static FillDEF GreyFill = { FILL_NONE, 0x00cbcbcbL, 1.0, NULL, 0x0L };

extern char TmpTxt[500];
extern unsigned long cObsW;				//count objects written
extern GraphObj *CurrGO, *TrackGO;		//Selected Graphic Objects
extern dragHandle *CurrHandle;
extern UndoObj Undo;

extern void *prog_bar_ptr;
extern long prog_bar_max, *prog_bar_current;
extern DWORD prog_bar_color;
extern tagProgBarDlg *ProgressBarDlg;
extern int prog_bar_mode;
extern notary *Notary;

clear_sym ClearSym;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process fields with user input text
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
EditText *CurrText = 0L, *scroll_et = 0;
int scroll_dist = 0;

EditText::EditText(void *par, unsigned char *msg, int r, int c)
{
	loc.x = loc.y = crb.x = rb.x = crb.y = rb.y = 0;	Value = 0.0;
	row = r;	col = c;	disp = 0L;		text = 0L;
	CursorPos = length = Align = 0;		type = ET_UNKNOWN;		TextCol=0x00000000L;
	bgLine = &ETbgnn;					bgFill = &ETfbnn;		parent = par;
	if(msg && msg[0]) {
		SetText(msg);		FindType();
		}
	else {
		Align = TXA_VCENTER | TXA_HRIGHT;
		type = ET_UNKNOWN;
		}
	m1 = m2 = -1;						//cursor positions track marks
	mx1 = mx2 = -1;
	ftext = 0L;							//store formula text result here
	a_data = 0L;		a_count = 0;	//cell contains array data
}

EditText::~EditText()
{
	HideCopyMark(false);
	if(CurrText == this) CurrText = 0L;
	if(text) free(text);
	text = 0L;
	if(ftext) free(ftext);
	ftext = 0L;
	if (a_data  && a_count) {			//free array data
		free(a_data);	a_data = 0L;	a_count = 0;
		}
}

bool
EditText::AddChar(int ci, anyOutput *Out, void *)
{
	unsigned char byte1, byte2, c, *tmp;
	POINT MyPos;
	int i;

	if (type & ET_NOEDIT) return false;
	if (ci < 254 && ci > 31) c = (unsigned char)ci;
	else if(ci == 27) {						//Esc
		m1 = m2 = -1;		Redraw(Out, true);		return true;
		}
	else return false;
	Undo.EtString(this, (char**)&text, 0L);
	if(parent) {
		((DataObj*)parent)->Command(CMD_MRK_DIRTY, 0L, 0L);
		((DataObj*)parent)->Command(CMD_SAVEPOS, 0L, 0L);
		}
	bgLine = &ETbgna; bgFill = &ETfbna; TextCol = 0x00000000L;
	if(text)length = rlp_strlen((char*)text);
	else length = 0;
	if(text) tmp = (unsigned char *)realloc(text, length+2);
	else tmp = (unsigned char *)calloc(2, sizeof(unsigned char));
	if(!tmp) return false;
	text = (unsigned char*)tmp;
	byte1 = byte2 = 0;
	//replace mark by character if mark exists
	if(hasMark()) {			//delete marked part of text
			if(m1 > m2) Swap(m1, m2);
			if(m2 >= (short int)rlp_strlen((char*)text)) text[m1] = 0;
			else rlp_strcpy((unsigned char*)(text+m1), TMP_TXT_SIZE, (unsigned char*)(text+m2));
			CursorPos = m1;
			m1 = m2 = -1;
			}
	byte1 = text[CursorPos];
	i = CursorPos;
	text[i++] = c;
	while(byte1) {
		byte2 = byte1;			byte1 = text[i];			text[i++] = byte2;
		}
	text[i] = byte1;			CursorPos++;				type = ET_UNKNOWN;
	Redraw(Out, true);
	MyPos.y = ((loc.y +crb.y)>>1);
	MyPos.x = Align & TXA_HRIGHT ? crb.x - 2 : loc.x + 2;
	if(Out)Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos, 
		scroll_et == this ? scroll_dist : scroll_dist=0);
	set_etracc();
	return true;
}

void
EditText::Update(int select, anyOutput *Out, POINT *MousePos)
{
	POINT MyPos;

	if (Out) disp = Out;
	if (!disp) return;
	if (!Out) Out = disp;
	if (select != 1 && select != 5){
		m1 = m2 = -1;		//no mark;
		}
	switch(select) {
		case 0:							//just redraw with current settings
			Redraw(Out ? Out : disp, true);
			break;
		case 5:							//dialog control
		case 1:							//active spread sheet cell with cursor
			if (select == 5) {
				if(!text) Align = TXA_VCENTER | TXA_HLEFT;
				}
			if (CurrText && CurrText != this && (CurrText->type & ET_DIALOG)) {
				CurrText->Update(2, Out, MousePos);		CurrText = 0L;
				}
			if((type & 0xff) == ET_FORMULA) Align = TXA_VCENTER | TXA_HLEFT;
			if(!text && !(text = (unsigned char *) calloc(10, sizeof(unsigned char))))return;
			if(CursorPos > rlp_strlen((char*)text)) CursorPos = rlp_strlen((char*)text);
			if(MousePos && (type & 0xff) == ET_TEXT && (text && text[0] == '\'' && (bgLine == &ETbgnn || bgLine == &ETbgmn))) {
				MousePos->x += 4;
				}
			if (a_data && a_count){
				free(a_data);		a_data = 0L;		a_count = 0;
				}
			bgLine = &ETbgna; bgFill = &ETfbna; TextCol = 0x00000000L;
			if(Out) {
				if ((type & ET_DIALOG) && select == 1 && (Align & TXA_HRIGHT)!= TXA_HRIGHT){
					MyPos.x = crb.x;		MyPos.y = loc.y;
					if(text[0]) ClearSym.set_rect(&MyPos);
					if (MousePos && ClearSym.select(MousePos->x, MousePos->y)) {
						Command(CMD_CLEAR, 0L, Out);
						}
					Redraw(Out, true);
					if(parent) ((DataObj*)parent)->Command(CMD_ETRACC, this, 0L);
					}
				else Redraw(Out, true);
				MyPos.y = ((loc.y +crb.y)>>1);
				MyPos.x = Align & TXA_HRIGHT ? crb.x - 4 : loc.x + 4;
				if(MousePos && MousePos->x && MousePos->y) {
					Out->TextCursor((unsigned char*)text, MyPos, MousePos,&CursorPos, 
					scroll_et == this ? scroll_dist : scroll_dist=0);
					}
				else if(select == 1) Out->TextCursor(text, MyPos, NULL, &CursorPos, 
					scroll_et == this ? scroll_dist : (scroll_dist=0)+2);
				if(!(type & ET_DIALOG)) set_etracc();
				}
			break;
		case 2:							//inactive spreadsheet cell
			if(CurrText && CurrText == this) {
				HideTextCursor();				FindType();
				CurrText = 0L;
				}
			if(crb.x > rb.x) {
				crb.x = rb.x;	crb.y = rb.y;
				}
			bgLine = &ETbgnn; bgFill = &ETfbnn; TextCol = 0x00000000L;
			if(Out) Redraw(Out, true);
			break;
		case 10:						//value filled in by external app.
			if(text && text[0]) {
				type = (type & ~0xfff) | ET_VALUE;
				Align = TXA_VCENTER | TXA_HRIGHT;
#ifdef USE_WIN_SECURE
				sscanf_s((char*)text, "%lf", &Value);
#else
				sscanf((char*)text, "%lf", &Value);
#endif
				}
			break;
		case 20:						//update value only
			FindType();
			break;
		}
}

bool
EditText::Redraw(anyOutput *Out, bool display)
{
	RECT rc;
	POINT MyPos;
	unsigned char *txt, tmptxt[500];
	int i, o_crbx;
	double w, h;
	bool b_clip = false;
	anyOutput *opc;
	anyResult cres;
	POINT grid[3];

	if((!parent || !Out) && disp) Out = disp;
	ETbgnn.width = ETbgna.width = ETbgmn.width = ETbgma.width = defs.GetSize(SIZE_HAIRLINE);
	yLine.width = BlackLine.width = GrayLine.width = WhiteLine.width = defs.GetSize(SIZE_HAIRLINE);
	if ((type & ET_NODRAW_EMPTY) == ET_NODRAW_EMPTY && CurrText != this){
		type &= ~ET_NODRAW;
		return true;
		}
	if(loc.x <1 || rb.x < 1 || loc.y <1 || rb.y <1) return false;
	cres.a_data = 0L;		cres.a_count = 0;
	o_crbx = crb.x;			crb.x = rb.x;				crb.y = rb.y;
	if (Out) {
		if (m1 >m2) Swap(m1, m2);
		if(((type & 0xff) == ET_UNKNOWN) && text && text[0] && (bgLine == &ETbgnn || bgLine == &ETbgmn)) FindType();
		Out->TxtSet.Align = Align;		Out->TxtSet.ColTxt = TextCol;
		Out->TxtSet.ColBg = bgLine->color;
		if(text && text[0]) {
			Out->oGetTextExtent(text, rlp_strlen((char*)text), &w, &h);
			if(CurrText == this && parent && col >= 0 && parent) {
				for(i = col+1; (crb.x - loc.x) < (w+(h /2.0)); i++) {
					crb.x += ((DataObj*)parent)->ri->GetWidth(i);
					}
				if(o_crbx > loc.x && o_crbx > crb.x && o_crbx < 4000) crb.x = o_crbx;
				}
			else if((crb.x - loc.x) < (w+(h/2.01))) b_clip = true;
			}
		if (type & ET_NOEDIT) bgFill = &GreyFill;
		Out->SetFill(bgFill);		Out->SetLine(bgLine);
		rc.left = loc.x;			rc.right = crb.x;
		rc.top = loc.y+1;			rc.bottom = crb.y-2;
		Out->oRectangle(loc.x, loc.y, crb.x-1, crb.y-1);
		if (bgLine == &ETbgna && row >= 0 && col >= 0) {
			Out->SetLine((LineDEF*)&GrayLine);
			Out->oRectangle(loc.x, loc.y, crb.x - 2, crb.y - 2);
			}
		if(!text || !text[0]){
			if((type & 0xff) == ET_VALUE){
#ifdef USE_WIN_SECURE
				sprintf_s((char*)tmptxt, 500, "%g", Value);
#else
				sprintf((char*)tmptxt, "%g", Value);
#endif
				}
			else if((type & 0xff) == ET_BOOL) {
				rlp_strcpy(tmptxt, 500, Value != 0.0 ? (char*)"true" : (char*)"false");
				}
			else tmptxt[0] = 0;
			if(tmptxt[0] && (text = (unsigned char*)realloc(text, rlp_strlen((char*)tmptxt)+2))) rlp_strcpy(text, 500, tmptxt);
			CursorPos = 0;
			}
		if(ftext) free(ftext);
		ftext = 0L;
		if(text && text[0]){
			if(bgLine == &ETbgnn || bgLine == &ETbgmn) {
				Out->TxtSet.Align = TXA_HLEFT | TXA_VCENTER;
				GetResult(&cres, false);				TranslateResult(&cres);
				Value = cres.value;
				switch (cres.type) {
				case ET_VALUE:
					Out->TxtSet.Align = TXA_HRIGHT | TXA_VCENTER;
					b_clip = false;				rlp_strcpy(tmptxt, 500, cres.text);
					fit_num_rect(Out, rb.x - loc.x, tmptxt);
					Int2Nat((char*)tmptxt);			break;
				case ET_BOOL:	case ET_DATE:	case ET_TIME:	case ET_DATETIME:
				case ET_TEXT:
					Out->TxtSet.Align = cres.type == ET_TEXT ? 
						TXA_HLEFT | TXA_VCENTER : TXA_HRIGHT | TXA_VCENTER;
					i = rlp_strlen(cres.text)+2;
					ftext = (unsigned char*)realloc(ftext, i > 20 ? i : 20);
					if(ftext)rlp_strcpy(ftext, i, cres.text);
					if(cres.text && i < (int)sizeof(tmptxt)) 
						rlp_strcpy(tmptxt, 500, cres.text[0] != '\'' ? cres.text : cres.text +1);
					else if(cres.text)rlp_strcpy(tmptxt, 500, (char*)"#SIZE");
					else tmptxt[0] = 0;			
					Out->oGetTextExtent(tmptxt, rlp_strlen((char*)tmptxt), &w, &h);
					b_clip = (crb.x - loc.x) < (w+(h/2.0)) ? true : false;
					break;
				case ET_ERROR:
					rlp_strcpy(tmptxt, 500, (char*)"#ERROR");	break;
				default: 
					rlp_strcpy(tmptxt, 500, (char*)"#VALUE");	break;
					}
				txt = tmptxt;
				}
			else txt = text;
			if(b_clip && parent && col >= 0 && row >=0) {
				Out->oGetTextExtent(txt, rlp_strlen((char*)txt), &w, &h);
				for(i = col+1; (crb.x - loc.x) < (w+(h/2.0)); i++) {
					if(((DataObj*)parent)->isEmpty(row, i)) {
						crb.x += ((DataObj*)parent)->ri->GetWidth(i);
						((DataObj*)parent)->etRows[row][i]->type |= ET_NODRAW;
						}
					else break;
					}
				if((crb.x - loc.x) >= (w+(h/2.0))) b_clip = false;
				else rc.right = crb.x;
				}
			MyPos.y = (loc.y+rb.y)>>1;
			if(Out->TxtSet.Align & TXA_HRIGHT) {	//right justified text
				MyPos.x = crb.x-4;
				}
			else {									//left justified text
				MyPos.x = loc.x+4;
				}
			if(b_clip && (opc = NewBitmapClass(iround(w)+22, rb.y-loc.y, Out->hres, Out->vres))){
				if(scroll_et != this || parent) {
					scroll_et = this;	scroll_dist = 0;
					}
				opc->Erase(bgFill->color);
				opc->SetTextSpec(&Out->TxtSet);		opc->TxtSet.Align = TXA_HLEFT | TXA_VCENTER;
				opc->oTextOut(4,(rb.y-loc.y)>>1, txt, rlp_strlen((char*)txt));
				if(!parent && CursorPos) {
					Out->oGetTextExtent(txt, CursorPos, &w, &h);
					while((scroll_dist + w)>(rc.right-rc.left-10)) scroll_dist -=10;
					while((scroll_dist + w)<12) scroll_dist +=10;
					if(scroll_dist >0) scroll_dist=0;
					}
				else scroll_dist=0;
				Out->CopyBitmap(rc.left+1, rc.top+1, opc, 1-scroll_dist, 1, 
					rc.right-rc.left-4, rc.bottom-rc.top-2, false);
				DelBitmapClass(opc);
				}
			else {
				if(display && hasMark() && mx1 > loc.x && mx2 < crb.x) {
					Out->SetFill(&yFill);		Out->SetLine(&yLine);
					Out->oRectangle(mx1, rc.top, mx2, rc.bottom);
					Out->SetFill(bgFill);		Out->SetLine(bgLine);
					}
				scroll_dist = 0;
				Out->oTextOut(MyPos.x, MyPos.y, txt, 0);
				if(display && hasMark() && mx1 > loc.x && mx2 < crb.x) {
					rc.left = mx1;		rc.right = mx2;
					Out->CopyBitmap(mx1, rc.top, Out, mx1, rc.top, mx2-mx1, rc.bottom-rc.top, true);
					Out->MrkMode = MRK_NONE;
					}
				}
			}
		Out->SetLine((LineDEF*)&GrayLine);
		grid[0].x = loc.x;					grid[0].y = grid[1].y = crb.y-1;
		grid[1].x = grid[2].x = crb.x-1;	grid[2].y = loc.y-1;
		Out->oPolyline(grid, 3);
		if (type & ET_DIALOG) {
			if(text && text[0] && (rc.right-rc.left)>100)ClearSym.do_plot(Out);
			}
		if(display) {
			Out->UpdateRect(&rc, false);
			}
		return true;
	}
	return false;
}

void
EditText::Mark(anyOutput *Out, int mark)
{
	LineDEF *ol = bgLine;
	FillDEF *of = bgFill;
	DWORD ocol = TextCol;

	m1 = m2 = -1;
	if(!parent) return;
	switch (mark){
	case 0:				//normal not active
		bgLine = &ETbgnn; bgFill = &ETfbnn; TextCol = 0x00000000L;
		break;
	case 1:				//normal active
		bgLine = &ETbgna; bgFill = &ETfbna; TextCol = 0x00000000L;
		break;
	case 2:				//mark not active
		bgLine = &ETbgmn; bgFill = &ETfbmn; TextCol = 0x00c00000L;
		break;
	case 3:				//mark active
		bgLine = &ETbgma; bgFill = &ETfbma; TextCol = 0x00ff0000L;
		break;
		}
	if(!mark || mark == 2) {
		loc.y--;	rb.y++;
		}
	Redraw(Out, true);
	if(!mark || mark == 2) {
		loc.y++;	rb.y--;
		}
	bgLine = ol;	bgFill = of;	TextCol = ocol;
}

bool
EditText::Command(int cmd, anyOutput *Out, void *data_obj)
{
	int i, j, k, iw;
	double w, h;
	POINT MyPos;
	MouseEvent *mev;
	static RECT rMark;
	bool bRet;
	unsigned char *tag1, *tag2;
	unsigned char *pt;

	MyPos.y = ((loc.y+crb.y)>>1);
	MyPos.x = Align & TXA_HRIGHT ? crb.x - 4 : loc.x + 4;
	if(!(text)) return false;
	if((type & ET_DIALOG) || !Out) Out = disp;		//Dialog ?
	Undo.SetDisp(disp);
	switch(cmd) {
		case CMD_MRK_DIRTY:
			type = ET_UNKNOWN;
			if(CurrText == this) {
				Command(CMD_REDRAW, Out, data_obj);
				if(parent)((DataObj*)parent)->Command(CMD_MRK_DIRTY, Out, 0L);
				}
			else if(parent) {
				((DataObj*)parent)->Command(CMD_REDRAW, Out, 0L);
				((DataObj*)parent)->Command(CMD_MRK_DIRTY, Out, 0L);
				}
			else return Command(CMD_REDRAW, Out, data_obj);
			return true;
		case CMD_SETFONT:
			if (!text || !text[0]) return false;
			if(Out) Undo.SetDisp(Out);
			type = (type & ~0xfff) | ET_TEXT;
			if(hasMark()) {
				Undo.EtString(this, (char**)&text, 0L);
				switch (*((int*)data_obj)) {
				case FONT_HELVETICA:
					tag1 = (unsigned char*)"<face=helvetica>";		tag2 = (unsigned char*)"</face>";		break;
				case FONT_TIMES:
					tag1 = (unsigned char*)"<face=times>";			tag2 = (unsigned char*)"</face>";		break;
				case FONT_COURIER:
					tag1 = (unsigned char*)"<face=courier>";			tag2 = (unsigned char*)"</face>";		break;
				case FONT_GREEK:
					tag1 = (unsigned char*)"<face=greek>";			tag2 = (unsigned char*)"</face>";		break;
				default:
					return false;
					}
				if(m1 < m2) {
					j = m1;	k = m2;
					}
				else if(m1 > m2) {
					j = m2; k = m1;
					}
				else return false;			//empty mark !
				for(i = 0; i < j; i++) TmpTxt[i] = text[i];
				for(j = 0, iw = i; tag1[j]; j++) TmpTxt[i++] = tag1[j];
				for( ; iw < k; iw++) TmpTxt[i++] = text[iw];
				for (j = 0; tag2[j]; j++) {
					TmpTxt[i++] = tag2[j];
					}
				for (; text[iw]; iw++){
					TmpTxt[i++] = (char)text[iw];
					}
				TmpTxt[i++] = (char)'\0';
				m1 += iround((iw = rlp_strlen((char*)tag1)));	m2 += iw;	CursorPos += iw;
				CleanTags(TmpTxt, &m1, &m2, &CursorPos);
				text = (unsigned char*)realloc(text, rlp_strlen(TmpTxt) + 2);
				if(text) rlp_strcpy(text, TMP_TXT_SIZE, TmpTxt);
				Command(CMD_REDRAW, Out, 0L);
				return true;
				}
			return false;
		case CMD_MARKALL:
			m1 = 0;		m2 = rlp_strlen(text);
			return true;
		case CMD_SETSTYLE:
			if (!text || !text[0]) return false;
			if(Out) Undo.SetDisp(Out);
			type = (type & ~0xfff) | ET_TEXT;
			if(hasMark()) {
				Undo.EtString(this, (char**)&text, 0L);
				switch (*((int*)data_obj)) {
				case TXS_BOLD:
					tag1 = (unsigned char*)"<b>";		tag2 = (unsigned char*)"</b>";		break;
				case ~TXS_BOLD:
					tag1 = (unsigned char*)"</b>";		tag2 = (unsigned char*)"<b>";		break;
				case TXS_ITALIC:
					tag1 = (unsigned char*)"<i>";		tag2 = (unsigned char*)"</i>";		break;
				case ~TXS_ITALIC:
					tag1 = (unsigned char*)"</i>";		tag2 = (unsigned char*)"<i>";		break;
				case TXS_UNDERLINE:
					tag1 = (unsigned char*)"<u>";		tag2 = (unsigned char*)"</u>";		break;
				case ~TXS_UNDERLINE:
					tag1 = (unsigned char*)"</u>";		tag2 = (unsigned char*)"<u>";		break;
				case TXS_SUPER:
					tag1 = (unsigned char*)"<sup>";		tag2 = (unsigned char*)"</sup>";		break;
				case ~TXS_SUPER:
					tag1 = (unsigned char*)"</sup>";		tag2 = (unsigned char*)"<sup>";		break;
				case TXS_SUB:
					tag1 = (unsigned char*)"<sub>";		tag2 = (unsigned char*)"</sub>";		break;
				case ~TXS_SUB:
					tag1 = (unsigned char*)"</sub>";		tag2 = (unsigned char*)"<sub>";		break;
				default:
					return false;
					}
				if(m1 < m2) {
					j = m1;	k = m2;
					}
				else if(m1 > m2) {
					j = m2; k = m1;
					}
				else return false;			//empty mark !
				for(i = 0; i < j; i++) TmpTxt[i] = text[i];
				for(j = 0, w = i; tag1[j]; j++) TmpTxt[i++] = tag1[j];
				for( ; w < k; w++) TmpTxt[i++] = text[iround(w)];
				for(j = 0; tag2[j]; j++) TmpTxt[i++] = tag2[j];
				for( ; (TmpTxt[i++] = (char)text[iround(w)]); w += 1.0);
				m1 += iround(w = (double)strlen((char*)tag1));	m2 += iround(w);	CursorPos += iround(w);
				CleanTags(TmpTxt, &m1, &m2, &CursorPos);
				text = (unsigned char*)realloc(text, rlp_strlen(TmpTxt) + 8);
				if(text) rlp_strcpy(text, TMP_TXT_SIZE, TmpTxt);
				Command(CMD_REDRAW, Out, 0L);
				return true;
				}
			return false;
		case CMD_ADDTXT:	case CMD_ADDTXT_PLUS:
			HideTextCursor();
			if (cmd == CMD_ADDTXT_PLUS && CursorPos) AddChar(';', 0L, 0L);
			k = type & 0xff;
			if(k == ET_VALUE || k == ET_FORMULA || k == ET_ERROR || k == ET_BOOL || k == ET_DATE ||
				k == ET_TIME || k == ET_DATETIME || k == ET_EMPTY) k = ET_UNKNOWN;
			type &= ~0xff;		type |= k;
			if((tag1 = (unsigned char*)data_obj) && *tag1 && text && 
				(k == ET_TEXT || k == ET_UNKNOWN || k == ET_FORMULA)){
				if(hasMark()) Command(CMD_DELETE, 0L, 0L);
				else Undo.EtString(this, (char**)&text, 0L);
				if(m1 > -1 && m2 > -1) Command(CMD_DELETE, 0L, 0L);
				for(k = 0; tag1[k] && tag1[k] == text[k]; k++);
				if(tag1[k]!=';') k = 0;
				for(i = 0; i < CursorPos && text[i]; i++) TmpTxt[i] = text[i];
				j = i + rlp_strcpy(TmpTxt+i, TMP_TXT_SIZE-i, tag1+k);
				if(text[i]) j += rlp_strcpy(TmpTxt+j, TMP_TXT_SIZE-j, text+i);
				text = (unsigned char*)realloc(text, j + 2);
				if(text) rlp_strcpy(text, j+2, TmpTxt);
				CursorPos += rlp_strlen((char*)(tag1+k));
				Out->Focus();					Update(1, Out, 0L);
				}
			set_etracc();
			return true;
		case CMD_BACKSP:		case CMD_DELETE:
			HideTextCursor();
			if (cmd == CMD_BACKSP){
				if (!text) return false;
				if (CursorPos <= 0){
					Out->TextCursor(text, MyPos, (POINT *)NULL, &CursorPos,
						scroll_et == this ? scroll_dist : scroll_dist = 0);
					return false;
				}
				Undo.EtString(this, (char**)&text, 0L);
				CursorPos--;						//continue as if delete
				}
			if(!text) return false;
			if (cmd == CMD_DELETE) Undo.EtString(this, (char**)&text, 0L);
			if(parent) {
				((DataObj*)parent)->Command(CMD_MRK_DIRTY, 0L, 0L);
				((DataObj*)parent)->Command(CMD_SAVEPOS, 0L, 0L);
				}
			bRet = false;
			if(!text || !text[0]) {
				type = ET_UNKNOWN;	CursorPos = 0;
				}
			if(hasMark()) {			//delete marked part of text
				if (!text || !text[0]) return false;
				if(m1 > m2) Swap(m1, m2);
				if(m2 >= (short int)rlp_strlen((char*)text)) text[m1] = 0;
				else rlp_strcpy((unsigned char*)(text+m1), rlp_strlen((char*)text)+m1, (char*)(text+m2));
				CursorPos = m1;						m1 = m2 = -1;
				if(!text[0]) {
					type = ET_UNKNOWN;	CursorPos = 0;
					}
				if(Out) Redraw(Out, (bRet = true));
				}
			else if(text[CursorPos]) {
				rlp_strcpy((char*)(text + CursorPos), rlp_strlen((char*)(text + CursorPos)), (char*)(text + CursorPos + 1));
				if(!text || !text[0]) {
					type = ET_UNKNOWN;	CursorPos = 0;
					}
				if(Out)Redraw(Out, (bRet = true));
				}
			set_etracc();
			if(Out)Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos,
				scroll_et == this ? scroll_dist : scroll_dist=0);
			return bRet;
		case CMD_COPY:
			if(text && text[0]) {
				rMark.left = loc.x+2;		rMark.right = rb.x-2;
				rMark.top = loc.y+1;		rMark.bottom = rb.y-2;
				if(m1 != m2 && m1 >=0 && m2 >=0) {
					if (m1 >m2) Swap(m1, m2);
					rMark.left = mx1;		rMark.right = mx2;
					if(Out) Out->UpdateRect(&rMark, false);
					CopyText((char*)(text+m1), m2-m1, CF_TEXT);
					if(Out) {
						Out->MrkMode = MRK_NONE;
						ShowCopyMark(Out, &rMark, 1);
						Out->UpdateRect(&rMark, true);
						}
					return false;
					}
				CopyText((char*)text, rlp_strlen((char*)text), CF_TEXT);
				if(Out)Out->UpdateRect(&rMark, true);
				return false;
				}
			return false;
		case CMD_CLEAR:
			if (text && text[0]) {
				Undo.EtString(this, (char**)&text, 0L);
				text[0] = 0;		if (Out) Redraw(Out, true);	
				if(parent)((DataObj*)parent)->Command(CMD_ETRACC, this, 0L);
				}
			return true;
		case CMD_PASTE:
			pt = PasteText();
			if(pt) {
				Undo.EtString(this, (char**)&text, 0L);
				for(i = 0; pt[i] > 0x20 && i < 81; i++) this->AddChar(pt[i], 0L, 0L);
				if(Out) Redraw(Out, true);
				free(pt);
				set_etracc();
				if(i) return true;
				}
			return false;
		case CMD_SHIFTRIGHT:		case CMD_SHIFTLEFT:		case CMD_REDRAW:
			HideTextCursor();
			if (cmd == CMD_SHIFTRIGHT) {
				if (CursorPos == m1 && text[m1]) m1++;
				else if (CursorPos == m2 && text[m2]) m2++;
				else if (text[CursorPos]){
					m1 = CursorPos;	m2 = CursorPos + 1;
					}
				set_etracc();
				if (text[CursorPos]) CursorPos++;
				else return false;
				}
			if (cmd == CMD_SHIFTLEFT || cmd == CMD_SHIFTRIGHT) {
				if (cmd == CMD_SHIFTLEFT) {
					set_etracc();
					if (!(CursorPos)) return false;
					if (CursorPos == m1 && m1 > 0) m1--;
					else if (CursorPos == m2 && m2 > 0) m2--;
					else if (CursorPos > 0){
						m1 = CursorPos;	m2 = CursorPos - 1;
						}
					if (CursorPos > 0) CursorPos--;
					}
				}
			if(m1 >=0 && m2 >= 0 && m1 != m2 && Out) {
				if(m1 > m2) Swap(m1, m2);
				w = h = 0;
				if(Align & TXA_HRIGHT) {	//right justified text
					Out->oGetTextExtent(text, 0, &w, &h);
					mx1 = crb.x-4 - iround(w);
					}
				else {						//left justified text
					mx1 = loc.x +4;
					}
				Out->oGetTextExtent(text, m1, &w, &h);
				mx1 += (m1 ? iround(w) : 0);
				Out->oGetTextExtent(text+m1, m2-m1, &w, &h);
				mx2 = mx1 + iround(w);
				}
			HideTextCursor();		 ClearSym.set_rect(0L);			Redraw(Out, true);
			Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos,
				scroll_et == this ? scroll_dist : scroll_dist=0);
			return true;
		case CMD_CURRLEFT:
			m1 = m2 = -1;					set_etracc();
			HideTextCursor();
			if(CursorPos >0) {
				CursorPos--;
				if(Redraw(Out, true) && Out->TextCursor(text, MyPos, (POINT *) NULL,
					&CursorPos, scroll_et == this ? scroll_dist : scroll_dist=0)) return true;
				else return false;
				}
			else if (data_obj) {
				MyPos.x = loc.x-4;			MyPos.y = (rb.y+loc.y)/2;
				if(((DataObj*)data_obj)->Select(&MyPos))return true;
				MyPos.x = loc.x+4;
				((DataObj*)data_obj)->Select(&MyPos);
				}
			return false;
		case CMD_CURRIGHT:
			m1 = m2 = -1;					set_etracc();
			HideTextCursor();
			if(text[CursorPos]){
				CursorPos++;
				if(Redraw(Out, true) && Out->TextCursor(text, MyPos, (POINT *) NULL,
					&CursorPos, scroll_et == this ? scroll_dist : scroll_dist=0)) return true;
				else return false;
				}
			else if (data_obj) {
				MyPos.x = rb.x+4;		MyPos.y = (rb.y+loc.y)/2;	crb.x = rb.x;
				if(((DataObj*)data_obj)->Select(&MyPos)) return true;
				MyPos.x = rb.x-4;
				((DataObj*)data_obj)->Select(&MyPos);
				}
			return false;
		case CMD_UPDATE:
			m1 = m2 = -1;
			Redraw(Out, true);
			return true;
		case CMD_POS_FIRST:		case CMD_POS_LAST:
			HideTextCursor();
			CursorPos = (cmd == CMD_POS_LAST && text && text[0]) ? rlp_strlen((char*)text) : 0;
			m1 = m2 = -1;		Redraw(Out, true);
			Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos,
				scroll_et == this ? scroll_dist : scroll_dist=0);
			set_etracc();
			return true;
		case CMD_CURRDOWN:		case CMD_CURRUP:
			if (data_obj) {
			//the following calculation of the cursor position is crude
            //it is based on a aspect of 2:1 for digits
				if(Align & TXA_HRIGHT)		//right justified text
					MyPos.x = rb.x-4-((rb.y-loc.y-4)*((long)rlp_strlen((char*)text)-CursorPos))/2;
				else MyPos.x = loc.x+4+((rb.y-loc.y-4)*CursorPos)/2;
				MyPos.y = (cmd == CMD_CURRUP) ? loc.y-2 : rb.y+2;
				if(MyPos.x < loc.x) MyPos.x = loc.x +4;
				if(MyPos.x > rb.x) MyPos.x = rb.x -4;
				if(((DataObj*)data_obj)->Select(&MyPos))return true;
				MyPos.y = rb.y;
				((DataObj*)data_obj)->Select(&MyPos);
				}
			return false;
		case CMD_MOUSE_EVENT:					//track left mouse button
			mev = (MouseEvent*) data_obj;
			if(!text || !text[0]) return false;
			if(mev->x <loc.x || mev->x >crb.x || mev->y <loc.y || mev->y >rb.y)return false;
			if(mev->Action == MOUSE_LBDOWN) {
				m1 = m2 = -1;//					set_etracc();
				return true;
				}
			if(mev->Action == MOUSE_LBDOUBLECLICK) {
				rMark.top = loc.y+1;			rMark.bottom = rb.y-2;
				if(!Out->oGetTextExtent(text, rlp_strlen((char*)text), &w, &h)) return false;
				m1 = 0;							m2 = rlp_strlen((char*)text);
				if(Align & TXA_HRIGHT) {		//right justfied text
					rMark.right = crb.x -4;		rMark.left = crb.x - iround(w) - 4;
					}
				else {							//left justified text
					rMark.left = loc.x +4;		rMark.right = rMark.left + iround(w);
					}
				mx1 = rMark.left;				mx2 = rMark.right;
				Redraw(Out, true);				set_etracc();
				return true;
				}
			MyPos.x = Align & TXA_HRIGHT ? mev->x + 4 : mev->x - 4;
			MyPos.y = mev->y;
			Out->TxtSet.Align = Align;
			j = Out->CalcCursorPos(text, Align & TXA_HRIGHT ? crb :loc, &MyPos);
			if(j == m1 || j == m2) return true;
			if(Align & TXA_HRIGHT) {			//right justfied text
				if((i = rlp_strlen((char*)text)-j)){
					if(!Out->oGetTextExtent(text+j, i, &w, &h)) return false;
					w = crb.x - w - 4;
					}
				else w = crb.x-1;
				}
			else {								//left justified text
				if(!j) w = 0;
				else if(!Out->oGetTextExtent(text, j, &w, &h))return false;
				w += (loc.x+4);
				}
			if(m1 == m2 && m1 == -1) {
				mx1 = (short)(rMark.left = iround(w));
				m1 = j;
				}
			else if(j != m2){
				m2 = j;
				if(m2 >= 0)Out->UpdateRect(&rMark, false);
				mx2 = (short)(rMark.right = iround(w));
				rMark.top = loc.y+1;				rMark.bottom = rb.y-2;
				if(rMark.right < crb.x && rMark.right > loc.x &&
					rMark.left > loc.x && rMark.left < crb.x)
					Redraw(Out, true);
				}
			if(hasMark() && parent)
				//remove range-mark of data 
				((DataObj*)parent)->Command(CMD_UNLOCK, 0L, 0L);
			return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// return the value (i.e. the floating point equivalent) of text
bool
EditText::GetValue(double *v)
{
	anyResult * res;

	if(((type & 0xff) == ET_UNKNOWN) && (text)) FindType();
	if(!text || !text[0]) {
		if((type & 0xff) == ET_VALUE) {
			*v = Value;		return true;
			}
		return false;
		}
	if(CurrText == this && !(type & ET_BUSY)) FindType();
	if((type & 0xff) == ET_VALUE || (type & 0xff) == ET_BOOL || (type & 0xff) == ET_DATE 
		|| (type & 0xff) == ET_TIME || (type & 0xff) == ET_DATETIME){
		a_data = 0L;		a_count = 0;
		*v = Value;			return true;
		}
	if((type & 0xff) == ET_FORMULA && text && text[0]){
		if(!(type & ET_BUSY)){
			type |= ET_BUSY;
			res = do_formula((DataObj*)parent, (char*)(text + 1));
			if(res) {
				if(res->type == ET_VALUE || res->type == ET_DATE || res->type == ET_TIME 
					|| res->type == ET_DATETIME || res->type == ET_BOOL){
					res->a_data = 0L;		res->a_count = 0;
					a_data = 0L;			a_count = 0;
					*v = Value = res->value;	type &= ~ET_BUSY;	return true;
					}
				else{
					res->a_data = 0L;		res->a_count = 0;
					a_data = 0L;			a_count = 0;
					}
				}
			*v = Value = 0.0;	type &= ~ET_BUSY;	return false;
			}
		else type |= ET_CIRCULAR;
		*v = Value;
		return true;
		}
	return false;
}

bool
EditText::GetText(char *tx, int size, bool bTranslate)
{
	unsigned char *t = 0L;
	static unsigned char tmp_txt[40];
	anyResult res;

	if((type & 0xff) == ET_TEXT && text && text[0]) {
		if(text[0] =='\'' && text[1]) t = text + 1;
		else t = text;
		}
	else if(bTranslate) {
		GetResult(&res, false);						TranslateResult(&res);
		switch (res.type) {
		case ET_VALUE:
			rlp_strcpy(tmp_txt, 40, res.text);		t = tmp_txt;
			Int2Nat((char*)tmp_txt);							break;
		case ET_BOOL:		case ET_DATE:			case ET_TIME:	
		case ET_DATETIME:	case ET_TEXT:
			t = (unsigned char*)res.text;						break;
		default:
			t = 0L;									break;
			}
		}
	else if(text && text[0]) t = (text[0] =='\'' && text[1]) ? text+1 : text;
	if(t) {
		rlp_strcpy(tx, size, t);
		return true;
		}
	else if((type & 0xff) == ET_VALUE) {
		text = (unsigned char*)realloc(text, 30);
#ifdef USE_WIN_SECURE
		if(text) sprintf_s((char*)text, 30, "%g", Value);
#else
		if(text) sprintf((char*)text, "%g", Value);
#endif
		if(text && text[0]) return(GetText(tx, size, false));
		}
	return false;
}

bool
EditText::GetResult(anyResult *r, bool use_last)
{
	anyResult * res;

	if(!text || !text[0]) {
		r->text = 0L;
		if((type & 0xff) == ET_VALUE) {
			r->value = Value;		r->type = ET_VALUE;
			}
		else {
			r->value = 0.0;			r->type = ET_UNKNOWN;
			}
		return true;
		}
    if((type & 0xff) == ET_UNKNOWN) FindType();
	if((type & 0xff) == ET_VALUE || (type & 0xff) == ET_BOOL || (type & 0xff) == ET_DATE 
		|| (type & 0xff) == ET_TIME || (type & 0xff) == ET_DATETIME){
		r->text = 0L;	r->value = Value;		r->type = (type & 0xff);
		return true;
		}
	if((type & 0xff) == ET_TEXT) {
		r->text = (char*)text;	r->value = 0.0;				r->type = ET_TEXT;
		return true;
		}
	if((type & 0xff) == ET_FORMULA){
		if(use_last) {
			if(ftext) {
				r->text = (char*)ftext;	r->value = 0.0;		r->type = ET_TEXT;
				}
			else {
				r->text = 0L;		r->value = Value;			r->type = ET_VALUE;
				if (r->a_data && r->a_count){
					free(r->a_data);		r->a_data = 0L;		r->a_count = 0;
					}
				}
			return true;
			}
		if(!(type & ET_BUSY)){
			type |= ET_BUSY;
			if(defs.DecPoint[0] != '.') formula2int(text);
			if (a_data && a_count) {
				r->a_count = a_count;			r->a_data = a_data;
				r->text = (char*)"#ARRAY";		r->type = ET_TEXT;
				r->value = 0.0;					type &= ~ET_BUSY;
				return true;
				}
			res = do_formula((DataObj*)parent, (char*)(text + 1));
			if(res) {
				if(res->type == ET_VALUE) Value = res->value;
				else if(res->type == ET_ERROR) {
					res->text = (char*)"#ERROR";	res->type = ET_TEXT;
					}
				else Value = 0.0;
				type &= ~ET_BUSY;
				memcpy(r, res, sizeof(anyResult));
				if (res->a_count && res->a_data) {
					a_data = (double*)calloc(res->a_count + 2, sizeof(double));
					memcpy(a_data, res->a_data, sizeof(double)*res->a_count);
					a_count = res->a_count;
					r->a_count = a_count;		r->a_data = a_data;
					}
				else if(r->a_data && r->a_count){
					free(r->a_data);		r->a_data = 0L;		r->a_count = 0;
					}
				return true;
				}
			type &= ~ET_BUSY;
			return false;
			}
		else {
			type |= ET_CIRCULAR;
			r->text = (char*)"#CIRC.";	r->value = 0.0;			r->type = ET_TEXT;
			r->a_data = 0L;		r->a_count = 0;
			return true;
			}
		return false;
		}
	return false;
}

bool
EditText::SetValue(double v)
{
	if(text) text[0] = 0;
	Value = v;	type = ET_VALUE;
	return true;
}

bool
EditText::SetText(unsigned char *t)
{
	int cb = 0;

	Value = 0.0;	type = ET_UNKNOWN;
	bgLine = &ETbgnn; bgFill = &ETfbnn; TextCol = 0x00000000L;
	if(t && t[0] && (text = (unsigned char*)realloc(text, cb = (rlp_strlen((char*)t)+2)))) rlp_strcpy(text, cb, t);
	else if (text) text[0] = 0;
	if (text && text[0]){
		if (!strcmp((char*)text, (char*)"TRUE")) rlp_strcpy(text, cb, (char*)"true");
		else if (!strcmp((char*)text, (char*)"FALSE")) rlp_strcpy(text, cb, (char*)"false");
		}
	return false;
}

void
EditText::SetRec(RECT *rc)
{
	if (type & ET_DIALOG) return;
	loc.x = rc->left;				loc.y = rc->top;
	crb.x = rb.x = rc->right;		crb.y = rb.y = rc->bottom;
}

	
bool
EditText::isValue()
{
	if((type & 0xff)==ET_UNKNOWN) FindType();
	return (type == ET_VALUE || type == ET_FORMULA || type == ET_BOOL
		|| type == ET_DATE || type == ET_TIME || type == ET_DATETIME);
}

bool
EditText::isFormula()
{
	if((type & 0xff)==ET_UNKNOWN) FindType();
	return (type == ET_FORMULA);
}

void
EditText::formula2int(unsigned char *text)
{
	int i;
	bool is_txt = 0;

	for(i = 1; text[i]; i++) {
		if(text[i] == defs.DecPoint[0] && (isdigit(text[i-1]) || isdigit(text[i+1]))) {
			if(!is_txt) text[i] = '.';
			}
		else if(text[i] == '"') is_txt ^= 0x01;
		}
}

//test if text str contains character c
static int str_has_c(unsigned char *str, unsigned char c)
{
	int i;

	if (c == defs.DecPoint[0]) return 50;			//the value of 50 is longer than length of str
	for (i = 0; str[i]; i++) {
		if (str[i] == c) return i + 1;
		}
	return 0;
}

void
EditText::FindType()
{
	static unsigned char *templ1 = (unsigned char*)"1234567890-+.";
	static unsigned char *templ2 = (unsigned char*)"1234567890.\0";
	int i, c1, c2, c3, c4, c5;

	if(!text || !text[0]) {
		Align = TXA_VCENTER | TXA_HRIGHT;
		if ((type & 0xff) == ET_VALUE) type = ET_VALUE;
		else type = ET_UNKNOWN | ET_EMPTY;
		return;
		}
	while (text[0] <= ' ' && text[0]) {
		for (i = 1; text[i-1]; i++) {
			text[i - 1] = text[i];
			}
		}
	if (!text[0]) {
		FindType();					return;
		}
	if(text[0] == '=') {
		Align = TXA_VCENTER | TXA_HRIGHT;
		type = ET_FORMULA;
		}
	else if (isdigit(text[0]) && !text[1]){
		Txt2Flt((char*)text, &Value);
		Align = TXA_VCENTER | TXA_HRIGHT;
		type = ET_VALUE;
		}
	else if (str_has_c(templ1, text[0]) && str_has_c(templ2, text[1])) {
		for (i = c1 = c2 = c3 = c4 = c5 = 0; text[i]; i++) {
			switch (text[i]) {
			case '.':		c1++;		break;
			case '-':		c2++;		break;
			case '/':		c3++;		break;
			case ':':		c4++;		break;
			case 'e':		break;
			case 'E':		break;
			default:
				if (text[i] == defs.DecPoint[0])c1++;
				else if (isalpha(text[i])) c5++;
				break;
				}
			}
			if(c5 > 0){
				Align = TXA_VCENTER | TXA_HLEFT;
				type = ET_TEXT;
				}
			else if (c1 < 2 && c2 < 2 && !c3  && !c4 && Txt2Flt((char*)text, &Value)) {
				Align = TXA_VCENTER | TXA_HRIGHT;
				type = ET_VALUE;
				}
			else if((c1 == 2 || c2 == 2 || c3 == 2) && date_value((char*)text, 0L, &Value)) {
				Align = TXA_VCENTER | TXA_HRIGHT;
				type = c4 == 2 ? ET_DATETIME : ET_DATE;
				}
			else if((c4 == 1 || c4 == 2) && date_value((char*)text, (char*)"H:M:S", &Value)) {
				Align = TXA_VCENTER | TXA_HRIGHT;
				type = ET_TIME;
				}
			else {
				Align = TXA_VCENTER | TXA_HLEFT;
				type = ET_TEXT;
				}
			}
	else {
		if(0 == strcmp((char*)text, "true")) {
			Value = 1.0;
			Align = TXA_VCENTER | TXA_HRIGHT;
			type = ET_BOOL;
			}
		else if(0 == strcmp((char*)text, "false")) {
			Value = 0.0;
			Align = TXA_VCENTER | TXA_HRIGHT;
			type = ET_BOOL;
			}
		else if(0 == strcmp((char*)text, "inf")) {
			Value = HUGE_VAL;
			Align = TXA_VCENTER | TXA_HRIGHT;
			type = ET_VALUE;
			}
		else if(0 == strcmp((char*)text, "-inf")) {
			Value = -HUGE_VAL;
			Align = TXA_VCENTER | TXA_HRIGHT;
			type = ET_VALUE;
			}
		else {
			Align = TXA_VCENTER | TXA_HLEFT;
			type = ET_TEXT;
			}
		}
}

void
EditText::set_etracc()
{
	int i;
	bool accept_range;
	anyResult *res;

	if(parent) {
		if(hasMark()) i = m1 < m2 ? m1 : m2;
		else i = CursorPos;
		accept_range = (i && text && text[0] == '=' && text[i-1]!=')'
			&& text[i-1] > 31 && !(isdigit(text[i-1]) || isalpha(text[i-1]))
			&& !(isdigit(text[i]) || isalpha(text[i])));
		if(accept_range) {
			LockData(true, true);
			res = do_formula((DataObj*)parent, (char*)(text+1));
			if(res->type != ET_ERROR) accept_range = false;
			if(!accept_range) {
				if(text[i-1] == '(' || text[i-1] == ',' || text[i-1] == ';')
					accept_range = true;
				}
			((DataObj*)parent)->Command(CMD_CLEAR_ERROR, 0L, 0L);
			LockData(false, false);
			}
		((DataObj*)parent)->Command(CMD_ETRACC, accept_range ? this : 0L, 0L);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// display a clear-all icon in top right corner
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
clear_sym::clear_sym()
{
	rec.left = rec.right = rec.top = rec.bottom = 0;
	mrc.left = mrc.right = mrc.top = mrc.bottom = 0;
	mo = currdisp = 0L;
}

void
clear_sym::set_rect(POINT *pos)
{
	if (mo) {
		if(currdisp && Notary->IsValidPtr(currdisp)) RestoreRectBitmap(&mo, &mrc, currdisp);
		DelBitmapClass(mo);			mo = 0L;
		}
	if (pos && pos->x > 0 && pos->y >0) {
		rec.left = pos->x - 16;			rec.right = pos->x;
		rec.top = pos->y;				rec.bottom = pos->y + 16;
		memcpy(&mrc, &rec, sizeof(RECT));
		}
	else {
		rec.left = rec.right = rec.top = rec.bottom = 0;
		}
}

void
clear_sym::do_plot(anyOutput *o)
{
	POINT mark[2];

	if (!o || rec.left == rec.right || rec.top == rec.bottom) return;

	if (mo){
		if (currdisp && Notary->IsValidPtr(currdisp)) RestoreRectBitmap(&mo, &mrc, currdisp);
		DelBitmapClass(mo);			mo = 0L;
		}
	mo = GetRectBitmap(&mrc, currdisp = o);

	pts[0].x = pts[5].x = rec.right - 3;		pts[0].y = pts[1].y = rec.top + 4;
	pts[4].x = pts[3].x = pts[1].x = rec.right - 11;		pts[2].x = rec.right - 15;
	pts[2].y = rec.top + 8;								pts[4].y = pts[5].y = pts[3].y = rec.top + 12;
	o->SetLine((LineDEF*)&BlackLine);					o->SetFill(&BlackFill);
	o->oPolygon(pts, 6);
	WhiteLine.width *= 5.0;				o->SetLine((LineDEF*)&WhiteLine);
	mark[0].x = rec.right - 9;			mark[1].x = rec.right - 5;
	mark[0].y = rec.top + 6;			mark[1].y = rec.top + 10;
	o->oSolidLine(mark);				Swap(mark[0].x, mark[1].x);
	o->oSolidLine(mark);
	WhiteLine.width = defs.GetSize(SIZE_HAIRLINE);
	o->UpdateRect(&mrc, true);
}

bool
clear_sym::select(int x, int y)
{
	POINT pt;

	if (rec.left != rec.right && rec.top != rec.bottom){
		pt.x = x;		pt.y = y;
		if (IsInRect(rec, x, y))  return IsInPolygon(&pt, pts, 6, 1.0);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// output formated text - style and font changes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
typedef struct _tag_info {
	char *tag;
	int and_style, or_style;
	int font, op;
}tag_info;

static tag_info tags[] = {
	{(char*)"<i>", ~TXS_NORMAL, TXS_ITALIC, -1, 0},
	{(char*)"</i>", ~TXS_ITALIC, TXS_NORMAL, -1, 0},
	{(char*)"<b>", ~TXS_NORMAL, TXS_BOLD, -1, 0},
	{(char*)"</b>", ~TXS_BOLD, TXS_NORMAL, -1, 0},
	{(char*)"<u>", ~TXS_NORMAL, TXS_UNDERLINE, -1, 0},
	{(char*)"</u>", ~TXS_UNDERLINE, TXS_NORMAL, -1, 0},
	{(char*)"<ssup>", ~TXS_NORMAL, TXS_SUPER, -1, 0},
	{(char*)"</ssup>", ~TXS_SUPER, TXS_NORMAL, -1, 0},
	{(char*)"<ssub>", ~TXS_NORMAL, TXS_SUB, -1, 0},
	{(char*)"</ssub>", ~TXS_SUB, TXS_NORMAL, -1, 0},
	{(char*)"<face=helvetica>", 0, 0, FONT_HELVETICA, 0},
	{(char*)"<face=times>", 0, 0, FONT_TIMES, 0},
	{(char*)"<face=courier>", 0, 0, FONT_COURIER, 0},
	{(char*)"<face=greek>", 0, 0, FONT_GREEK, 0},
	{(char*)"</face>", 0, 0, -2, 0},
	{(char*)"<white>", 0, 0, -1, 100},	{(char*)"<black>", 0, 0, -1, 101},
	{(char*)"<red>", 0, 0, -1, 102},	{(char*)"<blue>", 0, 0, -1, 103},
	{(char*)"<green>", 0, 0, -1, 104},	{(char*)"<yellow>", 0, 0, -1, 105},
	{(char*)"<cyan>", 0, 0, -1, 106},	{(char*)"<purple>", 0, 0, -1, 107},
	{(char*)"<smaller>", 0, 0, -1, 200},	{(char*)"<larger>", 0, 0, -1, 201},
	{0L, 0, 0, 0, 0}};

static tag_info sstags[] = {
	{(char*)"<sup>", ~TXS_NORMAL, TXS_SUPER, -1, 0},
	{(char*)"</sup>", ~TXS_SUPER, TXS_NORMAL, -1, 0},
	{ (char*)"<sub>", ~TXS_NORMAL, TXS_SUB, -1, 0 },
	{ (char*)"</sub>", ~TXS_SUB, TXS_NORMAL, -1, 0 },
	{ 0L, 0, 0, 0, 0 } };

static int font_buff[256];
static unsigned font_idx=0;

static int char2uc(unsigned char*src, w_char* dest, bool isGreek, int *Style)
{
	int i, j, k, cb;
	unsigned int xuc, uc_flag = 0;
	unsigned char tmp_tag[45];

	if (!src || !*src) return 0;
	for (i = j = cb = 0;; i++){
		if (src[i] == '&') uc_flag |= 0x01;
		if (src[i] == '#') uc_flag |= 0x02;
		if (src[i] == ';') uc_flag |= 0x04;
		if (isGreek && src[i] >= 'A' && src[i] <= 'Z') dest[i] = (src[i] - 'A' + 0x391);
		else if (isGreek && src[i] >= 'a' && src[i] <= 'z') dest[i] = (src[i] - 'a' + 0x3B1);
		else if (src[i] == '<') {
			for (j = 0; j < 10  && src[i+j] && src[i+j] != '>'; j++) {	//copy possible tag
				tmp_tag[j] = src[i + j];
				}
			tmp_tag[j++] = '>';							tmp_tag[j] = 0;
			for (k = 0; sstags[k].tag; k++){
				if (!strcmp((char*)sstags[k].tag, (char*)tmp_tag)) {
					i += j-1;
					*Style &= sstags[k].and_style;		*Style |= sstags[k].or_style;
					break;
					}
				}
			}
		else {
			if (*Style & TXS_SUPER) {
				switch (src[i]) {
				case '+':		dest[cb++] = 0x207a;		break;
				case '-':		dest[cb++] = 0x207b;		break;
				case '=':		dest[cb++] = 0x207c;		break;
				case '(':		dest[cb++] = 0x207d;		break;
				case ')':		dest[cb++] = 0x207e;		break;
				case 'n':		dest[cb++] = 0x207f;		break;
				case '0':		dest[cb++] = 0x2070;		break;
				case '1':		dest[cb++] = 0x00b9;		break;
				case '2':		dest[cb++] = 0x00b2;		break;
				case '3':		dest[cb++] = 0x00b3;		break;
				default:
					if (src[i] > '3' && src[i] <= '9'){
						dest[cb++] = 8256+src[i];
						}
					else dest[cb++] = src[i];
					}
				}
			else if (*Style & TXS_SUB) {
				switch (src[i]) {
				case '+':		dest[cb++] = 0x208a;		break;
				case '-':		dest[cb++] = 0x208b;		break;
				case '=':		dest[cb++] = 0x208c;		break;
				case '(':		dest[cb++] = 0x208d;		break;
				case ')':		dest[cb++] = 0x208e;		break;
				case 'a':		dest[cb++] = 0x2090;		break;
				case 'e':		dest[cb++] = 0x2091;		break;
				case 'o':		dest[cb++] = 0x2092;		break;
				case 'x':		dest[cb++] = 0x2093;		break;
				default:
					if (src[i] >= '0' && src[i] <= '9'){
						dest[cb++] = 8272 + src[i];
						}
					else dest[cb++] = src[i];
					}
				}
			else dest[cb++] = src[i];
			}
		if (!src[i]) {
			if ((uc_flag & 0x07) == 0x07) {
				for (i = 0; dest[i]; i++) {
					if (dest[i] == '&' && dest[i + 1] == '#') {
						for (k = i + 2; dest[k] && dest[k] != ';'; k++){
							tmp_tag[k - i - 2] = (unsigned char)dest[k];
							}
						tmp_tag[k - i - 2] = 0;		xuc = 0;
#ifdef USE_WIN_SECURE
						sscanf_s((char*)(tmp_tag), "%x", &xuc);
#else
						sscanf((char*)(tmp_tag), "%x", &xuc);
#endif
						if (xuc > 0) {
							dest[j++] = xuc;				i += (k - i);
							}
						else dest[j++] = dest[i];
						}
					else dest[j++] = dest[i];
					}
				dest[j] = 0;
				return j - 1;
				}
			return cb - 1;
			}
		}
	return 0;
}

fmtText::fmtText()
{
	src=0L;		split_text=0L;		split_text_W = 0;
	n_split = n_split_W = uc_state = 0;
	pos.x = pos.y = 0;				flags =0x0;
}

fmtText::fmtText(anyOutput *o, int x, int y, unsigned char *txt)
{
	if(txt && txt[0]) src = rlp_strdup(txt);
	else src = 0L;
	split_text = 0L;	split_text_W = 0L;
	n_split = n_split_W = uc_state = 0;		flags = 0x0;
	pos.x = x;	pos.y = y;		if(src)Parse();
	if(o) DrawText(o);
}

fmtText::~fmtText()
{
	SetText(0L, 0L, 0L, 0L, false);
}

bool
fmtText::StyleAt(int idx, TextDEF *txt_def, int *style, int *font)
{
	TextDEF td;
	int i, j, n;

	if(!src || !split_text || (idx > rlp_strlen((char*)src))) return false;
	memcpy(&td, txt_def, sizeof(TextDEF));
	for(i = j = 0; i < n_split; i++) {
		if((n=split_text[i].tag) >= 0 && SetTextDef(&td, n)) j += rlp_strlen(tags[n].tag);
		if(j > idx) break;
		if(split_text[i].txt && split_text[i].txt[0]) j += rlp_strlen((char*)split_text[i].txt);
		if(j >= idx) break;
		}
	if(style) *style = td.Style;
	if(font) *font = td.Font;
	return true;
}

int
fmtText::rightTag(unsigned char *txt, int cb)
{
	int i, j;

	for(i = 0; tags[i].tag; i++) {
		for(j=1; tags[i].tag[j] && txt[cb+j]; j++) if(tags[i].tag[j] != txt[cb+j]) break;
		if(!tags[i].tag[j]) return i;
		}
	return -1;
}

int
fmtText::leftTag(unsigned char *txt, int cb)
{
	int i, j, k;

	for(i = 0; tags[i].tag; i++) {
		for(j = 0, k=rlp_strlen(tags[i].tag)-1; tags[i].tag[j] && k <=cb; j++, k--) {
			if(tags[i].tag[j] != txt[cb-k]) break;
			}
		if(!tags[i].tag[j]) return i;
		}
	return -1;
}

void
fmtText::cur_right(int *pos)
{
	int n;

	if(!src || !pos || *pos >= rlp_strlen(src) || !src[*pos]) return;
	if(src[*pos] == '<' && (n=rightTag(src, *pos)) >= 0) {
		*pos += rlp_strlen(tags[n].tag);
		cur_right(pos);
		}
	else if((src[*pos] & 0xc0) == 0xc0) {			//UTF8
		(*pos)++;
		while((src[*pos] & 0xc0) == 0x80) (*pos)++;
		}
	else (*pos)++;
}

void 
fmtText::cur_left(int *pos)
{
	int n/*, tl*/;

	if(!src || !pos || !(*pos)) return;
	(*pos)--;
	if((src[*pos] & 0xc0) == 0x80){					//UTF8
		while((src[*pos] & 0xc0) == 0x80 && *pos) (*pos)--;
		}
	if(*pos >= (n=rlp_strlen((char*)src))){
		*pos = n-1;		return;
		}
	while (src[*pos] == '>' && (n=leftTag(src, *pos)) >= 0) {
		*pos -= rlp_strlen(tags[n].tag);
		}
}

bool
fmtText::oGetTextExtent(anyOutput *o, double *width, double *height, int cb)
{
	TextDEF td1, td2;
	int i, n, l, l1;
	double w1, h1, w, h;
	w_char *uc_txt, last_cc;

	if (!src || !o) return false;
	if(!width || !height) return false;
	if (!src[0]){
		o->oGetTextExtent((unsigned char*)"A", cb, &w1, &h1);
		*width = 0.0;		*height = h1;
		return false;
		}
	if(!cb) cb = rlp_strlen((char*)src);
	if (!split_text) {
		uc_txt = (w_char*)calloc(rlp_strlen(src) + 1, sizeof(w_char));
		l = char2uc(src, uc_txt, false, &o->TxtSet.Style);
		last_cc = uc_txt[cb];						uc_txt[cb] = 0;
		o->oGetTextExtentW(uc_txt, cb, width, height);
		uc_txt[cb] = last_cc;			free(uc_txt);		uc_txt = 0L;
		return true;
		}
	memcpy(&td1, &o->TxtSet, sizeof(TextDEF));		memcpy(&td2, &o->TxtSet, sizeof(TextDEF));
	w = h = 0.0;
	for(i = l = l1 = 0; i < n_split; i++){
		if((n=split_text[i].tag) >= 0 && SetTextDef(&td2, n)) {
			o->SetTextSpec(&td2);			l += rlp_strlen(tags[n].tag);
			}
		if(tags[n].font == -1 && tags[n].op > 0 && tags[n].op < 100) {
			w += o->un2fix(td2.fSize/2.0);
			}
		if(split_text[i].txt && split_text[i].txt[0]){
			l1 = l;							l += rlp_strlen((char*)split_text[i].txt);
			if (l1 >= cb) break;
			uc_txt = (w_char*)calloc(2 + rlp_strlen((unsigned char*)split_text[i].txt), sizeof(w_char));
			n = char2uc(split_text[i].txt, uc_txt, false, &o->TxtSet.Style);
			if ((l1 + n) >= cb) n = cb >= l ? cb - l : n;
			last_cc = uc_txt[n];			uc_txt[n] = 0;
			o->oGetTextExtentW(uc_txt, l >= cb ? cb - l1 : 0, &w1, &h1);
			free(uc_txt);		uc_txt = 0L;				w += w1;
			if (h < h1) h = h1;
			if (l >= cb) break;
			}
		}
	*width = w;			*height = h;		o->SetTextSpec(&td1);
	return true;
}

void
fmtText::SetText(anyOutput *o, unsigned char *txt, double *px, double *py, bool bDraw)
{
	int i;

	if(px) pos.x = iround(*px);	
	if(py) pos.y = iround(*py);
	if(src && txt && !strcmp((char*)src, (char*)txt)) {
		if(bDraw && o) DrawText(o);
		return;
		}
	if(src) free(src);
	src = 0L;
	if(split_text) {
		for (i = 0; i < n_split; i++) if (split_text[i].txt){
			free(split_text[i].txt);		split_text[i].txt = 0L;
			}
		free(split_text);		split_text = 0L;	n_split = 0;
		}
	if(split_text_W) {
		for(i = 0; i < n_split_W; i++) {
			if (split_text_W[i].uc_txt) {
				free((split_text_W[i].uc_txt)); split_text_W[i].uc_txt = 0L;
				}
			}
		free(split_text_W);		split_text_W = 0L;	n_split_W = 0;
		}
	if(txt && txt[0]) src = rlp_strdup(txt);
	if(src)Parse();
	if(bDraw && o) DrawText(o);
}

void
fmtText::DrawText(anyOutput *o)
{
	w_char *uc_txt;

	if(!o || !src || !src[0] || !Notary->IsValidPtr(o)) return;
	if(split_text)DrawFormatted(o);
	else {
		uc_txt = (w_char*)calloc(rlp_strlen(src) + 10, sizeof(w_char));
		char2uc(src, uc_txt, false, &o->TxtSet.Style);
		o->oTextOutW(pos.x, pos.y, uc_txt, 0);
		free(uc_txt);	uc_txt = 0L;
		}
}

bool
fmtText::SetTextDef(TextDEF *td, int idx)
{
	if(tags[idx].and_style != tags[idx].or_style) {
		td->Style &= tags[idx].and_style;			td->Style |= tags[idx].or_style;
		}
	else if(tags[idx].font >= 0) {
		font_buff[font_idx & 0xff] = td->Font;	font_idx++;
		td->Font = tags[idx].font;
		}
	else if(tags[idx].font == -2) {
		font_idx--;			td->Font=font_buff[font_idx & 0xff];
		}
	else return false;
	return true;
}

bool
fmtText::Parse()
{
	int i, li, j, n;
	unsigned char *tmp;

	if((flags & 0x01) || !src || !(tmp = rlp_strdup(src))) return false;
	for(i = li = uc_state = 0; src[i]; i++) {
		if(i-li == 1 && split_text) {
			if(src[li] == '<' && (n=rightTag(src, li))>=0) i--;
			}
		if(src[i] == '<' && (n=rightTag(src, i))>=0) {
			if(tags[n].font == FONT_GREEK) uc_state |= 0x02;
			if(tags[n].op) uc_state |= 0x04;
			if(split_text) {				//more tags in text
				if(!(split_text = (fmt_txt_info *)realloc(split_text, (n_split+1)*sizeof(fmt_txt_info)))){
					free(tmp);		tmp = 0L;				return false;
					}
				for(j = li; j < i; j++) tmp[j-li] = src[j];
				tmp[j-li]=0;
				split_text[n_split-1].txt = rlp_strdup(tmp);
				i += rlp_strlen(tags[n].tag);	split_text[n_split].tag = n;
				split_text[n_split++].txt = 0L;
				}
			else {							//first tag of text
				if(!(split_text = (fmt_txt_info *)calloc(2, sizeof(fmt_txt_info)))){
					free(tmp);		tmp = 0L;			return false;
					}
				for(j = 0; j < i; j++) tmp[j]= src[j];
				tmp[j]=0;
				split_text[0].tag = -1;		
				split_text[0].txt = rlp_strdup(tmp);
				i += rlp_strlen(tags[n].tag);	split_text[1].tag = n;
				n_split = 2;
				}
			li = i;							i--;
			}
		}
	if(split_text && n_split && li < i && src[li]) 
		split_text[n_split-1].txt = rlp_strdup(src+li);
	free(tmp);		tmp = 0L;
	return true;
}

bool 
fmtText::DrawFormattedW(anyOutput *o)
{
	int i, j, n, cb, x, y, x1, y1;
	double w, h;
	double si, csi, fx, fy;
	TextDEF td1, td2;
	bool bGreek = false;
	
	if(!o || !(split_text_W = (fmt_uc_info *)calloc(n_split, sizeof(fmt_uc_info))))return false;
	memcpy(&td1, &o->TxtSet, sizeof(TextDEF));	memcpy(&td2, &o->TxtSet, sizeof(TextDEF));
	bGreek = (td1.Font == FONT_GREEK);
	for(i = n_split_W = 0; i < n_split; i++){
		if (true) {
			if(!i) return false;
			bGreek = true;			j = n_split_W-1;
			cb = split_text[i].txt ? rlp_strlen((char*)split_text[i].txt) : 0;
			split_text_W[j].uc_txt = (w_char*)realloc(split_text_W[j].uc_txt, (2 + cb + split_text_W[j].cb) * sizeof(w_char));
			if (split_text_W[j].uc_txt) {
				if (cb) split_text_W[j].cb += char2uc(split_text[i].txt, split_text_W->uc_txt + split_text_W[j].cb, bGreek, &o->TxtSet.Style);
				}
			}
		else if(bGreek && split_text[i].tag >= 0 && tags[split_text[i].tag].font == -2){
			if(!i) return false;
			bGreek = false;			j = n_split_W-1;
			cb = split_text[i].txt ? rlp_strlen((char*)split_text[i].txt) : 0;
			split_text_W[j].uc_txt = (w_char*)realloc(split_text_W[j].uc_txt, (2 + cb + split_text_W[j].cb) * sizeof(w_char));
			if (split_text_W[j].uc_txt) {
				if (cb) split_text_W[j].cb += (char2uc(split_text[i].txt, split_text_W->uc_txt + split_text_W[j].cb, bGreek, &o->TxtSet.Style));
				}
			}
		else {
			if((n=split_text[i].tag) >= 0) SetTextDef(&td2, n);
			bGreek = (td2.Font == FONT_GREEK);
			split_text_W[n_split_W].tag = split_text[i].tag;
			split_text_W[n_split_W].cb = split_text[i].txt ? rlp_strlen((char*)split_text[i].txt) : 0;
			if (split_text_W[n_split_W].cb && (split_text_W[n_split_W].uc_txt =
				(w_char*)calloc(2 + split_text_W[n_split_W].cb, sizeof(w_char)))){
				char2uc(split_text[i].txt, split_text_W[n_split_W].uc_txt, bGreek, &o->TxtSet.Style);
				}
			else split_text_W[n_split_W].uc_txt = 0L;
			n_split_W++;
			}
		}
	memcpy(&td2, &td1, sizeof(TextDEF));
	si = sin(td1.RotBL *0.01745329252);	csi = cos(td1.RotBL *0.01745329252);
	fx = pos.x;		fy = pos.y;	
	oGetTextExtent(o, &w, &h, 0);
	if(td2.Align & TXA_HRIGHT) {
		fx -= w*csi;		fy += w*si;
		}
	else if(td2.Align & TXA_HCENTER){
		fx -= (w / 2.0)*csi;	fy += (w / 2.0)*si;
		}
	x = iround(fx);			y = iround(fy);
	td2.Align &= ~(TXA_HRIGHT | TXA_HCENTER);			o->SetTextSpec(&td2);
	for(i = 0; i < n_split_W; i++) {
		if((n=split_text_W[i].tag) >= 0 && SetTextDef(&td2, n)) o->SetTextSpec(&td2);
		else if(n >= 0 && tags[n].op) {
			x1 = x + iround(o->un2fix(td2.fSize*0.25)*csi);
			y1 = y - iround(o->un2fiy(td2.fSize*0.25)*si);
			if((td2.Align & TXA_VTOP) == TXA_VTOP){
				y1 += iround(o->un2fiy(td2.fSize*0.5)*csi);
				x1 += iround(o->un2fix(td2.fSize*0.5)*si);
				}
			if((td2.Align & TXA_VCENTER) == TXA_VCENTER){
				y1 -= iround(o->un2fiy(td2.fSize*0.5)*csi);
				x1 -= iround(o->un2fix(td2.fSize*0.5)*si);
				}
			if((td2.Align & TXA_VBOTTOM) == TXA_VBOTTOM){
				y1 -= iround(o->un2fiy(td2.fSize)*csi);
				x1 -= iround(o->un2fix(td2.fSize)*si);
				}
			switch (tags[n].op) {
			case 100:	td2.ColTxt = 0x00ffffffL;	break;
			case 101:	td2.ColTxt = 0x00000000L;	break;
			case 102:	td2.ColTxt = 0x000000ffL;	break;
			case 103:	td2.ColTxt = 0x00ff0000L;	break;
			case 104:	td2.ColTxt = 0x0000ff00L;	break;
			case 105:	td2.ColTxt = 0x0000ffffL;	break;
			case 106:	td2.ColTxt = 0x00ffff00L;	break;
			case 107:	td2.ColTxt = 0x00ff00ffL;	break;
			case 200:	td2.fSize *= 0.81;	td2.iSize = 0;	break;
			case 201:	td2.fSize /= 0.81;	td2.iSize = 0;	break;
				}
			o->SetTextSpec(&td2);
			}
		if(split_text_W[i].uc_txt && split_text_W[i].uc_txt[0]){
			o->oTextOutW(x, y, split_text_W[i].uc_txt, 0);
			o->oGetTextExtentW(split_text_W[i].uc_txt, 0, &w, &h);
			x = iround(fx += (w*csi));		y = iround(fy -= (w*si));
			}
		}
	return true;
}

void 
fmtText::DrawFormatted(anyOutput *o)
{
	int i, n, x, y;
	double w, h;
	TextDEF td1, td2;
	double si, csi, fx, fy;
	w_char *uc_txt = NULL;

	if(!o || !split_text) return;
	if(uc_state && DrawFormattedW(o)) return;
	memcpy(&td1, &o->TxtSet, sizeof(TextDEF));	memcpy(&td2, &o->TxtSet, sizeof(TextDEF));
	si = sin(td1.RotBL *0.01745329252);	csi = cos(td1.RotBL *0.01745329252);
	fx = pos.x;		fy = pos.y;	
	oGetTextExtent(o, &w, &h, 0);
	if(td2.Align & TXA_HRIGHT) {
		fx -= w*csi;		fy += w*si;
		}
	else if(td2.Align & TXA_HCENTER){
		fx -= (w / 2.0)*csi;	fy += (w / 2.0)*si;
		}
	x = iround(fx);			y = iround(fy);
	td2.Align &= ~(TXA_HRIGHT | TXA_HCENTER);			o->SetTextSpec(&td2);
	for(i = 0; i < n_split; i++) if(split_text[i].txt) {
		if((n=split_text[i].tag) >= 0 && SetTextDef(&td2, n)) o->SetTextSpec(&td2);
		if(split_text[i].txt && split_text[i].txt[0]){
			if ((td2.Style & TXS_SUPER) == TXS_SUPER){
#ifdef _WINDOWS
				x = iround(fx - (o->un2fix(td1.fSize)*si*0.7));
#else
				x = iround(fx - (o->un2fix(td1.fSize)*si));
#endif
				uc_txt = (w_char*)malloc(sizeof(w_char) * (rlp_strlen(split_text[i].txt) + 1));
				char2uc(split_text[i].txt, uc_txt, false, &o->TxtSet.Style);
				o->oTextOutW(x, y, uc_txt, 0);
				}
			else if ((td2.Style & TXS_SUB) == TXS_SUB){
#ifdef _WINDOWS
				x = iround(fx - (o->un2fix(td1.fSize)*si*0.7));
#else
				x = iround(fx - (o->un2fix(td1.fSize)*si));
#endif
				uc_txt = (w_char*)malloc(sizeof(w_char) * (rlp_strlen(split_text[i].txt) + 1));
				char2uc(split_text[i].txt, uc_txt, false, &o->TxtSet.Style);
				o->oTextOutW(x, y, uc_txt, 0);
				}
			else {
				uc_txt = (w_char*)malloc(sizeof(w_char) * (rlp_strlen(split_text[i].txt) + 1));
				char2uc(split_text[i].txt, uc_txt, false, &o->TxtSet.Style);
				o->oTextOutW(x, y, uc_txt, 0);
				}
			if (uc_txt) {
				o->oGetTextExtentW(uc_txt, 0, &w, &h);
				free(uc_txt);		uc_txt = NULL;
				}
			else o->oGetTextExtent(split_text[i].txt, 0, &w, &h);
			x = iround(fx += (w*csi));		y = iround(fy -= (w*si));
			}
		}
}

void
fmtText::EditMode(bool bEdit)
{
	if(bEdit) flags |= 0x01;
	else flags &= ~(0x01);
}

int
fmtText::CalcCursorPos(int x, anyOutput *o)
{
	w_char *uc_txt;
	int i, j, n, l, l1, pos0, pos1;
	double w, h, xf = 0.0, dx, tmp;
	TextDEF td1, td2;

	if (!o) return 0;
	pos1 = pos0 = 0;
	if (!split_text) {
		uc_txt = (w_char*)malloc(sizeof(w_char) * (rlp_strlen(src) + 1));
		l = char2uc(src, uc_txt, false, &o->TxtSet.Style);
		xf = dx = x;
		for (i = 1; uc_txt[i - 1]; i++) {
			oGetTextExtent(o, &w, &h, i);
			tmp = fabs(xf - w);
			if (tmp < dx) {
				dx = tmp;			pos1 = i;
				}
			else break;
			}
		free(uc_txt);			uc_txt = 0L;
		return pos1;
		}
	memcpy(&td1, &o->TxtSet, sizeof(TextDEF));		memcpy(&td2, &o->TxtSet, sizeof(TextDEF));
	uc_txt = (w_char*)malloc(sizeof(w_char*) * (rlp_strlen(src) + 1));
	for (i = l = l1 = 0; i < n_split; i++){
		if ((n = split_text[i].tag) >= 0 && SetTextDef(&td2, n)) {
			o->SetTextSpec(&td2);			l += rlp_strlen(tags[n].tag);
			pos0 += rlp_strlen(tags[n].tag);
			}
		if (split_text[i].txt && split_text[i].txt[0] && (l = char2uc(split_text[i].txt, uc_txt, false, &o->TxtSet.Style))){
			xf = dx = x;
			pos1 = 0;
			for (j = 1; uc_txt[j - 1]; j++) {
				o->oGetTextExtentW(uc_txt, j, &w, &h);
				tmp = fabs(xf - w);
				if (tmp < dx) {
					dx = tmp;			pos1 = j;
					}
				else break;
				}
			if (pos1 < l) {
				free(uc_txt);
				return (pos0 + pos1);
				}
			if (i < n_split && split_text[i+1].tag >= 0) {
	//			pos0 += (rlp_strlen(tags[split_text[i+1].tag].tag));
				}
			pos0 += l;
			o->oGetTextExtentW(uc_txt, 0, &w, &h);
			x -= (int)w;
			if (pos1 < l) break;
			}
		}
	free(uc_txt);		uc_txt = 0L;
	return pos0;
}


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// assign a value to a string
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
TextValue::TextValue()
{
	items = 0L;		nitems = 0;		next = inc = 1.0;
}

TextValue::TextValue(TextValItem **tvi, int ntvi)
{
	int i;

	items = 0L;		nitems = 0;		next = inc = 1.0;
	if(ntvi && (items = (TextValItem **)malloc(ntvi*sizeof(TextValItem*)))){
		for(i = 0, nitems = ntvi; i < ntvi; i++) {
			items[i] = (TextValItem*)memdup(tvi[i], sizeof(TextValItem), 0);
			if(items[i]){
				if(tvi[i]->text) items[i]->text = rlp_strdup(tvi[i]->text);
				}
			}
		nitems = ntvi;	if(items[nitems-1]) next = items[nitems-1]->val + inc;
		}
}

TextValue::~TextValue()
{
	Reset();
}

double
TextValue::GetValue(char *txt)
{
	unsigned int hv1, hv2;
	int i;

	if(!txt || !txt[0]) return 0.0;
	hv1 = HashValue((unsigned char*)txt);
	hv2 = Hash2((unsigned char*)txt);
	if(items && nitems) for(i = 0; i < nitems; i++) {
		if(items[i] && items[i]->hv1 == hv1 && items[i]->hv2 == hv2) return items[i]->val;
		}
	else if(!items &&(items = (TextValItem **)malloc(sizeof(TextValItem*)))) {
		if((items[0] = (TextValItem*)calloc(1, sizeof(TextValItem)))) {
			items[0]->hv1 = hv1;		items[0]->hv2 = hv2;
			items[0]->text = rlp_strdup(txt);
			items[0]->val = next;		next += inc;
			nitems = 1;					return (next-inc);
			}
		return 0.0;
		}
	else return 0.0;
	items = (TextValItem **)realloc(items, (nitems + 1)*sizeof(TextValItem*));
	if(items){
		items[nitems] = (TextValItem*)calloc(1, sizeof(TextValItem));
		if(items[nitems]) {
			items[nitems]->hv1 = hv1;		items[nitems]->hv2 = hv2;
			items[nitems]->text = rlp_strdup(txt);
			items[nitems]->val = next;		next += inc;	nitems++;
			return (next-inc);
			}
		}
	return 0.0;
}

bool
TextValue::HasValue(char* txt, double *value)
{
	unsigned int hv1, hv2;
	int i;

	if (!txt || !txt[0] || !value) return false;
	hv1 = HashValue((unsigned char*)txt);
	hv2 = Hash2((unsigned char*)txt);
	if (items && nitems) for (i = 0; i < nitems; i++) {
		if (items[i] && items[i]->hv1 == hv1 && items[i]->hv2 == hv2){
			*value = items[i]->val;
			return true;
			}
		}

	return false;
}

bool
TextValue::GetItem(int idx, char **text, double *value)
{
	if(items && idx >=0 && idx < nitems) {
		if(text) *text = items[idx]->text;
		if(value) *value = items[idx]->val;
		return true;
		}
	return false;
}

void
TextValue::Reset()
{
	int i;

	if(items) for(i = 0; i < nitems; i++){
		if(items[i]->text) free(items[i]->text);
		free(items[i]);
		}
	if(items && nitems) free(items);
	next = inc = 1.0;	items = 0L;		nitems = 0;
}

TextValue *
TextValue::Copy()
{
	return new TextValue(items, nitems);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// manage formats and style of a spreadsheet range
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
RangeInfo::RangeInfo(int sel, int cw, int rh, int fw, RangeInfo *next)
{
	r_type = sel;		col_width = cw;		row_height = rh;	first_width = fw;
	ri_next = next;		ar = 0L;
}

RangeInfo::RangeInfo(int sel, char *range, RangeInfo *next)
{
	r_type = sel;		ri_next = next;
	if(range && range[0]) {
		ar = new AccRange(range);
		}
	else ar = 0L;
}

RangeInfo::~RangeInfo()
{
	if(ar) delete ar;
}

int
RangeInfo::SetWidth(int w)
{
	if(r_type == 0 || r_type == 1) return col_width = w;
	else if (ri_next) return ri_next->SetWidth(w);
	return w;
}

int
RangeInfo::SetDefWidth(int w)
{
	if(r_type == 0) return col_width = w;
	else if (ri_next) return ri_next->SetDefWidth(w);
	return w;
}

int
RangeInfo::SetHeight(int h)
{
	if(r_type == 0) return row_height = h;
	else if (ri_next) return ri_next->SetHeight(h);
	return h;
}

int
RangeInfo::SetFirstWidth(int w)
{
	if(r_type == 0) return first_width = w;
	else if (ri_next) return ri_next->SetFirstWidth(w);
	return w;
}

int
RangeInfo::GetWidth(int col)
{
	long r, c;

	if(r_type == 0) return col_width;
	else if (ar && r_type == 1){
		ar->GetFirst(&c, &r);
		while(ar->NextCol(&c)) {
			if(c == col) return col_width;
			}
		}
	if(ri_next) return ri_next->GetWidth(col);
	return 0;
}

int
RangeInfo::GetHeight(int row)
{
	if(r_type == 0) return row_height;
	else if (ri_next) return ri_next->GetHeight(row);
	return row_height;
}

int
RangeInfo::GetFirstWidth()
{
	if(r_type == 0) return first_width;
	else if (ri_next) return ri_next->GetFirstWidth();
	return first_width;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the basic data object
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
DataObj::DataObj()
{
	cRows = cCols = 0;
	etRows = 0L;
	ri = new RangeInfo(0, 76, 19, 32, 0L);
}

DataObj::~DataObj()
{
	FlushData();
	delete ri;
}

bool
DataObj::Init(long nR, long nC)
{
	long i, j;

	if(etRows)FlushData();
	if(!(etRows = (EditText ***)calloc (cRows = nR, sizeof(EditText **)))) return false;
	for(i = 0, cCols = nC; i < cRows; i++) {
		if(!(etRows[i] = (EditText **)calloc(cCols, sizeof(EditText *)))) {
			FlushData();	return false;
			}
		if(etRows[i]) for(j = 0; j < cCols; j++) {
			etRows[i][j] = new EditText(this, 0L, i, j);
			}
		}
	return true;
}

bool
DataObj::SetValue(long row, long col, double val)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->SetValue(val);
	return false;
}

bool
DataObj::SetText(long row, long col, char *txt)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->SetText((unsigned char*)txt);
	return false;
}

bool
DataObj::GetValue(long row, long col, double *v)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->GetValue(v);
	return false;
}

bool
DataObj::GetText(long row, long col, char *txt, int len, bool bTranslate)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(txt && etRows[row][col]) return etRows[row][col]->GetText(txt, len, bTranslate);
	return false;
}

char ** 
DataObj::GetTextPtr(long row, long col)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return 0L;
	if(etRows[row][col]) return (char**)(&etRows[row][col]->text);
	return 0L;
}

bool
DataObj::GetResult(anyResult *r, long row, long col, bool use_last)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->GetResult(r, use_last);
	return false;
}

bool
DataObj::GetSize(long *width, long *height)
{
	if(width)*width = cCols;
	if(height)*height = cRows;
	return true;
}

bool
DataObj::isEmpty(long row, long col)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) {
		if((etRows[row][col]->type & 0xff) == ET_UNKNOWN)etRows[row][col]->Update(20, 0L, 0L);
		if((etRows[row][col]->type & ET_EMPTY) == ET_EMPTY) return true;
		}
	return false;
}

void
DataObj::FlushData()
{
	long i, j;

	if(etRows){
		for(i = 0; i < cRows; i++) if(etRows[i]) {
			for (j = 0; j< cCols; j++) if(etRows[i][j]) delete etRows[i][j];
			free(etRows[i]);
			}
		free(etRows);
		}
	etRows = 0L;
}

bool
DataObj::ValueRec(RECT *rc, bool any)
{
	long r, r1 = cRows-1, r2 = -1, c, c1 = cCols-1, c2 = -1;
	double val;
	int bDoneRow, bDoneCol;
	char tmptxt[80];

	if (etRows && rc) {
		bDoneRow = bDoneCol  = 0;		// find upper left data corner
		for (r = 0; r < cRows && bDoneRow < 5 && bDoneCol < 5; r++) if (etRows[r]) {
			if (any)for (c = 0; c < cCols && bDoneRow <5 && bDoneCol < 5; c++) {
				if (GetText(r, c, tmptxt, 79, true)) {
					if (c < c1 ) {
						c1 = c;		bDoneCol++;
						}
					if (r < r1 ) {
						r1 = r;		bDoneRow++;
						}
					}
				}
			else for (c = 0; c < cCols && bDoneRow < 5 && bDoneCol< 5; c++) {
				if (etRows[r][c] && etRows[r][c]->GetValue(&val)) {
					if (c < c1) {
						c1 = c;		bDoneCol++;
						}
					if (r <r1) {
						r1 = r;		bDoneRow++;
						}
					}
				}
			}
		bDoneRow = bDoneCol = 0;		//find lower right data corner
		for (r = cRows - 1; r >= 0 && bDoneRow < 5 && bDoneCol < 5; r--) if (etRows[r]) {
			if (any) for (c = cCols - 1; c >= 0 && bDoneRow < 5 && bDoneCol < 5; c--) {
				if (GetText(r, c, tmptxt, 79, true)) {
					if (c > c2 ) {
						c2 = c;		bDoneCol++;
						}
					if (r > r2) {
						r2 = r;		bDoneRow++;
						}
					}
				}
			else for (c = cCols - 1; c >= 0 && bDoneRow < 5 && bDoneCol < 5; c--) {
				if (etRows[r][c] && etRows[r][c]->GetValue(&val)) {
					if (c > c2) {
						c2 = c;		bDoneCol++;
						}
					if (r > r2) {
						r2 = r;		bDoneRow++;
						}
					}
				}
			}
		}
	rc->left = c1;	rc->right = c2;
	rc->top = r1;	rc->bottom = r2;
	if (c2 > c1 && r2 > r1) return true;
	rc->left = rc->right = rc->top = rc->bottom = 0;
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Store Data Object as strings: less memory required than with DataObj
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
StrData::StrData(DataObj *par, RECT *rc)
{
	int r1, c1, r2, c2, w, h;
	char **tx;

	pw = ph = 0;		str_data = 0L;
	drc.left = drc.right = drc.top = drc.bottom = 0;
	if(!(src = par)) return;
	src->GetSize(&pw, & ph);
	if(rc) {
		if(0>(h = (rc->bottom - rc->top)) || 0>(w = (rc->right - rc->left))) return;
		if(!(str_data = (char***)calloc(h+1, sizeof(char**))))return;
		drc.left = rc->left;				drc.right = rc->right;
		drc.top = rc->top;					drc.bottom = rc->bottom;
		for (r1 = 0, r2 = drc.top; r1 <= h; r1++, r2++) {
			if(!(str_data[r1] = (char**)malloc((w+1) * sizeof(char*)))) break;
			for(c1 = 0, c2= drc.left; c1 <= w; c1++, c2++) {
				tx = src->GetTextPtr(r2, c2);
				if(tx && *tx && *tx[0]) {
					str_data[r1][c1] = rlp_strdup(*tx);
					}
				else str_data[r1][c1] = 0L;
				}
			}
		}
	else {
		if(!(str_data = (char***)calloc(ph, sizeof(char**))))return;
		for (r1 = 0; r1 < ph; r1++) {
			if(!(str_data[r1] = (char**)malloc(pw * sizeof(char*)))) break;
			for(c1 = 0; c1 < pw; c1++) {
				tx = src->GetTextPtr(r1, c1);
				if(tx && *tx && *tx[0]) {
					str_data[r1][c1] = rlp_strdup(*tx);
					}
				else str_data[r1][c1] = 0L;
				}
			}
		drc.right = pw-1;		drc.bottom = ph-1;
		}
}

StrData::~StrData()
{
	int r, c, w, h;

	w = drc.right-drc.left;			h = drc.bottom-drc.top;
	if(str_data) for (r = 0; r <= h; r++) {
		if(str_data[r]) {
			for(c = 0; c <= w; c++) if(str_data[r][c]) free(str_data[r][c]);
			free(str_data[r]);
			}
		}
	if(str_data) free(str_data);
}

bool
StrData::GetSize(int *uw, int *uh)
{
	if(uw) *uw = pw;
	if(uh) *uh = ph;
	return true;
}

void
StrData::RestoreData(DataObj *dest)
{
	int r1, c1, r2, c2;

	if(!dest || !str_data) return;
	for (r1 = 0, r2 = drc.top; r2 <= drc.bottom; r1++, r2++) {
		if(str_data[r1]) {
			for(c1 = 0, c2 = drc.left; c2 <= drc.right; c1++, c2++) 
				dest->SetText(r2, c2, str_data[r1][c1]);
			}
		}
	return;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The notary class handles different types of supervision and indexing
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
notary::notary()
{
	gObs = 0L;
	goStack = 0L;
	NextPopGO = NextPushGO = NextRegGO = 0L;
	valPtr_a = 0L;	valPtr_size = 0; valPtr_pos = 0;
}

notary::~notary()
{
	FreeStack();
}

int
notary::RegisterGO(GraphObj *go)
{
	int i, j;

	if(!go) return 0L;
	if(!gObs) {
		gObs = (GraphObj ***)calloc(0x2000L, sizeof(GraphObj **));
		gObs[0] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
		if(gObs && gObs[0]) {
			gObs[0][0] = go;
			return 1L;
			}
		return 0L;
		}
	i = (int)(NextRegGO >> 13);
	j = (int)(NextRegGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;	j = 0;
		}
	if(gObs[i] && gObs[i][j] && gObs[i][j] == go) {
		NextRegGO = ((i << 13) | j);
		return i*0x2000+j+1;
		}
	if(gObs && gObs[0]) {
		for(i = 0; i < 0x2000; i++) {
			for(j = 0; j < 0x2000L; j++) {
				if(gObs[i][j] == go) {
					NextRegGO = ((i << 13) | j);
					return i*0x2000+j+1;
					}
				if(!gObs[i][j]) {
					gObs[i][j] = go;
					NextRegGO = ((i << 13) | j);
					return i*0x2000+j+1;
					}
				}
			if(i < 0x1fffL && !gObs[i+1])
				gObs[i+1] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
			if(i < 0x1fff && !gObs[i+1]) return 0;
			}
		}
	return 0;
}

void
notary::AddRegGO(GraphObj *go)
{
	int i, j;

	if(!go) return;
	if(!gObs) {
		gObs = (GraphObj ***)calloc(0x2000L, sizeof(GraphObj **));
		gObs[0] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
		if(gObs && gObs[0]) {
			gObs[0][0] = go;
			return;
			}
		return;
		}
	i = (int)(NextRegGO >> 13);
	j = (int)(NextRegGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;
		j = 0;
		}
	if(!gObs[i]) gObs[i] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
	if(gObs[i] && !gObs[i][j]) {
		gObs[i][j] = go;
		NextRegGO = ((i << 13) | j);
		}
	else RegisterGO(go);
}

bool
notary::PushGO(unsigned int id, GraphObj *go)
{
	int i, j;

	NextPopGO = 0L;
	if(!go) return true;
	go->Id = id;
	if(!goStack) {
		if(!(goStack = (GraphObj ***)calloc(8192, sizeof(GraphObj **))))return false;
		goStack[0] = (GraphObj **)calloc(8192, sizeof(GraphObj *));
		if(goStack && goStack[0]) {
			goStack[0][0] = go;
			return true;
			}
		return false;
		}
	i = (int)(NextPushGO >> 13);
	j = (int)(NextPushGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;
		j = 0;
		}
	if(!goStack || !goStack[0]) return false;
	if(goStack[i] && !goStack[i][j]) {
		goStack[i][j] = go;
		NextPushGO = ((i << 13) | j);
		return true;
		}
	for(i = 0; i < 0x2000; i++) {
		for(j = 0; j < 0x2000; j++) {
			if(!goStack[i][j]) {
				goStack[i][j] = go;
				NextPushGO = ((i << 13) | j);
				return true;
				}
			}
		if(i < 0x1fff && !goStack[i+1] && !(goStack[i+1] =
			(GraphObj **)calloc(0x2000, sizeof(GraphObj *)))) return false;
		}
	return false;
}

GraphObj *
notary::PopGO(unsigned int id)
{
	int i, j;
	GraphObj *go;

	NextPushGO = 0L;
	if(!id || !goStack || !goStack[0]) return 0L;
	i = (int)(NextPopGO >> 13);
	j = (int)(NextPopGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;
		j = 0;
		}
	if(goStack[i] && goStack[i][j] && goStack[i][j]->Id == id) {
		go = goStack[i][j];
		goStack[i][j] = 0L;
		go->Id = 0L;
		NextPopGO = ((i << 13) | j);
		return go;
		}
	for(i = 0; i < 0x2000; i++) {
		for(j = 0; j < 0x2000; j++) {
			if(goStack[i][j] && goStack[i][j]->Id == id) {
				go = goStack[i][j];
				goStack[i][j] = 0L;
				go->Id = 0L;
				NextPopGO = ((i << 13) | j);
				return go;
				}
			}
		if(i < 0x1fff && !goStack[i+1]) return 0L;
		}
	return 0L;
}

void
notary::FreeStack()
{
	int i, j, k;

	if(gObs) {
		for(i = 0; gObs[i] && i <8192; i++) free(gObs[i]);
		free(gObs);
		gObs = 0L;
		}
	if(goStack) {
		for(i = k = 0; goStack[i] && i <8192; i++){
			for(j = 0; j < 8192; j++){
				if(goStack[i][j]) {
					goStack[i][j]->Id = 0L;
					DeleteGO(goStack[i][j]);
					k++;
					}
				}
			free(goStack[i]);	goStack[i] = 0L;
			}
		free(goStack);			goStack = 0L;
		if(k){
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, SCMS_DELBYNOTARY, k);
#else
			sprintf(TmpTxt,SCMS_DELBYNOTARY, k);
#endif
			ErrorBox(TmpTxt);
			}
		}
}

bool 
notary::ValPtr(void* ptr, bool add_remove)
{
	long i, j;

	if (!valPtr_a || !valPtr_size) {
		valPtr_a = (void**)calloc(valPtr_size = 1000, sizeof(void*));
		valPtr_pos = 0;
		}
	if (!ptr || !valPtr_a) return false;
	if (add_remove){							//new valid pointer
		if (valPtr_pos > (valPtr_size - 2)){
			valPtr_a = (void**)realloc(valPtr_a, valPtr_size += 1000);
			}
		for (i = 0; i < valPtr_pos; i++){
			if (valPtr_a[i] == ptr) return false;
			}
		valPtr_a[valPtr_pos++] = ptr;
		}
	else {										//invalidate pointer
		for (i = j = 0; i < valPtr_pos; i++) {
			if (valPtr_a[i] != ptr) {
				valPtr_a[j++] = valPtr_a[i];
				}
			}
		if (j < valPtr_pos) {
			valPtr_a[i] = 0;			valPtr_pos = j;
			return true;
			}
		}
	return false;
}

bool
notary::IsValidPtr(void* ptr)
{
	long i;

	if (!ptr || !valPtr_a) return false;
	for (i = valPtr_pos - 1; i >= 0; i--){
		if (valPtr_a[i] == ptr) return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate continuous index to a range given by an ASCII string
// string examples include  "a1:a12"  or  "a1:a4;a12:a24"
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
AccRange::AccRange(char *asc)
{
	long i, j, l;

	if(asc && *asc && (l=rlp_strlen(asc)) >1){
		txt = (char *)malloc(l+2);
		for(i = j = 0; i< rlp_strlen(asc); i++)
			if(asc[i] > 32) txt[j++] = asc[i];
		txt[j] = 0;
		}
	else txt = 0L;
	x1 = y1 = x2 = y2 = 0;
}

AccRange::~AccRange()
{
	if (txt) free(txt);
	txt = 0L;
}

long
AccRange::CountItems()
{
	long RetVal = 0, l;

	if(txt && txt[0] && Reset()){
		l = rlp_strlen(txt);
		do {
			RetVal += ((x2-x1+1)*(y2-y1+1));
			} while((curridx < l) && Parse(curridx));
		}
	return RetVal;
}

bool
AccRange::GetFirst(long *x, long *y)
{
	if(txt && Reset()) {
		if(x && y) {*x = x1; *y = y1;}
		return true;
		}
	return false;
}

bool
AccRange::GetNext(long *x, long *y)
{
	if(txt && x && y) {
		if(cx <= x2) {*x = cx; *y = cy; cx++; return true;}
		else {
			cx = x1; cy++;
			if(cy <= y2) return GetNext(x, y);
			else if(txt[curridx]){
				if(Parse(curridx)) return GetNext(x, y);
				return false;
				}
			}
		}
	return false;
}

bool
AccRange::NextRow(long *y)
{
	if(cy <= y2) {
		*y = cy;	cy++;	cx = x1;	return true;
		}
	else if(txt[curridx] && Parse(curridx)) return NextRow(y);
	return false;
}

bool
AccRange::NextCol(long *x)
{
	if(cx <= x2) {
		*x = cx;	cx++;	return true;
		}
	else if(txt[curridx] && Parse(curridx)) return NextCol(x);
	return false;
}

bool
AccRange::IsInRange(long x, long y)
{
	if(txt && Reset())	do {
		if(x >= x1 && x <= x2 && y >= y1 && y <= y2) return true;
		} while((curridx < rlp_strlen(txt)) && Parse(curridx));
	return false;
}

bool
AccRange::BoundRec(RECT *rec)
{
	if(txt && Reset()){
		SetMinMaxRect(rec, x1, y1, x2, y2);
		while((curridx < rlp_strlen(txt)) && Parse(curridx)) {
			UpdateMinMaxRect(rec, x1, y1);	UpdateMinMaxRect(rec, x2, y2);
			}
		Reset();
		return true;
		}
	return false;
}

bool
AccRange::Reset()
{
	curridx = 0;
	return Parse(curridx);
}

bool
AccRange::Parse(long start)
{
	long i, l, step, *v;

	i = start;
	if(!txt) return false;
	while(txt[i] == ';' || txt[i] == ',') i++;
	if(!txt[i]) return false;
	step = x1 = y1 = x2 = y2 = 0;
	v = &x1;
	for (l=rlp_strlen(txt)+1 ; i < l; i++) {
		if(txt[i] == '$') i++;
		switch(step) {
		case 0:
		case 2:
			if((txt[i] >= 'a') && (txt[i] <= 'z')){
				*v *= 26;
				*v += (txt[i]-'a'+1);
				}
			else if((txt[i] >= 'A') && (txt[i] <= 'Z')){
				*v *= 26;
				*v += (txt[i]-'A'+1);
				}
			else if((txt[i] >= '0') && (txt[i] <= '9')){
				v = step == 0 ? &y1 : &y2;
				*v = txt[i]-'0';
				step++;
				}
			else return false;
			break;
		case 1:
		case 3:
			if((txt[i] >= '0') && (txt[i] <= '9')){
				*v *= 10;
				*v += (txt[i]-'0');
				}
			else if((txt[i] >= 'a' && txt[i] <= 'z') ||
				(txt[i] >= 'A' && txt[i] <= 'Z')){
				if(step == 1) v =  &x2;
				else return false;
				*v = txt[i] >='a' && txt[i] <= 'z' ? 
					txt[i]-'a' : txt[i]-'A';
				step++;
				}
			else if(step == 1 && (txt[i] == ':')) {
				v = &x2;
				step++;
				}
			else if((txt[i] == ';') || (txt[i] == ',') || (txt[i] == 0)) {
				if(step == 1) {		//one single cell selected
					x2 = x1;	y2 = y1;
					}
				if(x2<x1) Swap(x1,x2);
				if(y2<y1) Swap(y1,y2);
				if(y1 >=0) y1--;
				if(y2 >=0) y2--;
				if(x1 >=0) x1--;
				if(x2 >=0) x2--;
				curridx = i;
				cx = x1; cy = y1;
				return true;
				}
			break;
			}
		}
	return false;
}

//get a description for the current range from the data object
char *
AccRange::RangeDesc(void *d, int style)
{
	anyResult res, res1;
	int cb;
	static char desc[100];

	if(!d || !txt || !Reset())return 0L;
	if(!((DataObj*)d)->GetResult(&res, y1, x1, false))return 0L;
	if(style == 3) {
		if(x1 == x2 && y1 > 0 && res.type == ET_TEXT) {
			if (((DataObj*)d)->GetResult(&res1, 0, x1, false) && res1.type == ET_TEXT){
				rlp_strcpy(desc, 98, res1.text);
				return desc;
				}
			}
		if(y1 == y2 && x1 > 0 && res.type == ET_TEXT) {
			if (((DataObj*)d)->GetResult(&res1, y1, 0, false) && res1.type == ET_TEXT){
				rlp_strcpy(desc, 98, res1.text);
				return desc;
				}
			}
		}
	switch(res.type) {
	case ET_TEXT:
		if (res.text && res.text[0]) {
			rlp_strcpy(desc, 98, res.text);
			return desc;
			}
		else return 0L;
	case ET_VALUE:
	case ET_DATE:	case ET_DATETIME:	case ET_TIME:
		if (res.type == ET_VALUE && style != 4)  break;
		TranslateResult(&res);
		if (res.text && res.text[0]){
			rlp_strcpy(desc, 98, res.text);
			return desc;
			}
		else return 0L;
		}
	if(x1 == x2) {
		if(style == 1 || style == 3) cb = rlp_strcpy(desc, 40, (char*)"Col. ");
		else if(style == 2) cb = rlp_strcpy(desc, 40, (char*)"Column ");
		else return 0L;
		rlp_strcpy(desc+cb, 40-cb, Int2ColLabel(x1, true));
		return desc;
		}
	else if(y1 == y2) {
#ifdef USE_WIN_SECURE
		sprintf_s(desc, 40, "Row %ld", y1+1);
#else
		sprintf(desc, "Row %ld", y1+1);
#endif
		return desc;
		}
	else return txt;
	return 0L;
}

int 
AccRange::DataTypes(void *d, int *numerical, int *strings, int *datetime)
{
	anyResult res;
	int n, c_num, c_txt, c_dattim;
	long r, c;

	if(!d || !Reset()) return 0;
	for(n = c_num = c_txt = c_dattim = 0, GetFirst(&c, &r); GetNext(&c, &r); n++) {
		if(((DataObj*)d)->GetResult(&res, r, c, false)) {
			switch(res.type) {
			case ET_VALUE:		c_num++;		break;
			case ET_TEXT:		c_txt++;		break;
			case ET_DATE:		case ET_TIME:	case ET_DATETIME:		
				c_dattim++;		break;
				}
			}
		}
	if(numerical) *numerical = c_num;
	if(strings) *strings = c_txt;
	if(datetime) *datetime = c_dattim;
	return n;
}

//---------------------------------------------------------------------------
// Use the Delauney triangulation to create a 3D mesh of dispersed data
//---------------------------------------------------------------------------
Triangle::Triangle()
{
	bSorted = false;	flags = 0L;		r2 = 0.0;
}

Triangle::Triangle(fPOINT3D *p1, fPOINT3D *p2, fPOINT3D *p3)
{
	bSorted = false;	flags = 0L;		r2 = 0.0;
	if (p1 && p2 && p3) {
		memcpy(&pt[0], p1, sizeof(fPOINT3D));
		memcpy(&pt[1], p2, sizeof(fPOINT3D));
		memcpy(&pt[2], p3, sizeof(fPOINT3D));
		}
	next = 0L;
}

//calculate circumcircle of triangle
bool
Triangle:: Circumcircle()
{
	double dx, dy, len, csi, si;
	lfPOINT line_point1[3], line_point2[3], centers[3];
	double radii[3], sum_radii, sum_center_x, sum_center_y;
	int n_radii;
	bool bValidLine[3];
	int i;

	//close polygon
	pt[3].fx = pt[0].fx;	pt[3].fy = pt[0].fy;	pt[3].fz = pt[0].fz;
	//get three line equations, each defined by two points
	for (i = 0; i < 3; i++) {
		if (pt[i + 1].fx != pt[i].fx && pt[i + 1].fy != pt[i].fy){
			dx = (pt[i + 1].fx - pt[i].fx);
			dy = (pt[i + 1].fy - pt[i].fy);
			len = sqrt(dx * dx + dy * dy);
			csi = dx/len;		si = dy/len;
			line_point1[i].fx = (pt[i].fx + pt[i + 1].fx) / 2.0;
			line_point1[i].fy = (pt[i].fy + pt[i + 1].fy) / 2.0;
			line_point2[i].fx = line_point1[i].fx - si;
			line_point2[i].fy = line_point1[i].fy + csi;
			}
		else if (pt[i + 1].fx == pt[i].fx) {
			line_point1[i].fx = pt[i].fx;
			line_point2[i].fx = line_point1[i].fx + 1.0;
			line_point1[i].fy = line_point2[i].fy = (pt[i].fy + pt[i + 1].fy) / 2.0;
			}
		else if (pt[i + 1].fy == pt[i].fy) {
			line_point1[i].fy = pt[i].fy;
			line_point2[i].fy = line_point1[i].fy + 1.0;
			line_point1[i].fx = line_point2[i].fx = (pt[i].fx + pt[i + 1].fx)/2.0;
			}
		else return false;
		}
	//now we calculate three intersection points
	bValidLine[0] = line_intersect(line_point1[0].fx, line_point1[0].fy, line_point2[0].fx, line_point2[0].fy,
		line_point1[1].fx, line_point1[1].fy, line_point2[1].fx, line_point2[1].fy, &centers[0].fx, &centers[0].fy, false);
	bValidLine[1] = line_intersect(line_point1[0].fx, line_point1[0].fy, line_point2[0].fx, line_point2[0].fy,
		line_point1[2].fx, line_point1[2].fy, line_point2[2].fx, line_point2[2].fy, &centers[1].fx, &centers[1].fy, false);
	bValidLine[2] = line_intersect(line_point1[1].fx, line_point1[1].fy, line_point2[1].fx, line_point2[1].fy,
		line_point1[2].fx, line_point1[2].fy, line_point2[2].fx, line_point2[2].fy, &centers[2].fx, &centers[2].fy, false);
	//get radii: distance of each point to the center
	sum_center_x = sum_center_y = sum_radii = 0.0;
	for (i = n_radii = 0; i < 3; i++){
		if (bValidLine[i]) {
			dx = pt[i].fx - centers[i].fx;		dy = pt[i].fy - centers[i].fy;
			radii[i] = dx * dx + dy * dy;		sum_radii += radii[i];	n_radii++;
			sum_center_x += centers[i].fx;		sum_center_y += centers[i].fy;
			}
		}
	if (!n_radii) return false;
	r2 = sum_radii / ((double)n_radii);
	cx = sum_center_x / ((double)n_radii);		cy = sum_center_y / ((double)n_radii);
	return true;
}

void
Triangle::SetRect()
{
	//and calculate the bounding circle
	Circumcircle();
}

bool
Triangle::TestVertex(double x, double y)
{
	double dx, dy;

	if (r2 < 1.0e-10) Circumcircle();
	dx = x-cx;		dx = dx * dx;		dy = y-cy;		dy = dy * dy;
	return (dx+dy)<r2;
}

bool
Triangle::Sort()
{
	int i, j;

	for (i = 0; i < 3; i++) order[i] = i;
	for (i = 0; i < 2; i++) {
		for (j = 0; j < 2; j++) {
			if (pt[order[j]].fz > pt[order[j + 1]].fz) Swap(order[j + 1], order[j]);
			}
		}
	return (bSorted = true);
}
	
void
Triangle::LinePoint(int i1, int i2, double z, lfPOINT *res)
{
	double dx, dy, dz, l, f;
	fPOINT3D v;

	dx = pt[i2].fx - pt[i1].fx;		dy = pt[i2].fy - pt[i1].fy;
	dz = pt[i2].fz - pt[i1].fz;
	l = sqrt(dx * dx + dy * dy + dz * dz);
	v.fx = dx / l;	v.fy = dy / l;	v.fz = dz / l;
	f = (z - pt[i1].fz) * v.fz;
	res->fx = pt[i1].fx + v.fx * f;
	res->fy = pt[i1].fy + v.fy * f;
}

bool
Triangle::IsoLine(double z, void *dest)
{
	lfPOINT li[2];

	if(!bSorted) Sort();
	if(z < pt[order[0]].fz) return false;		//below lowest point
	if (z == pt[order[0]].fz) {
		if(pt[order[1]].fz > z) return true;	//crossing single lowest point
		li[0].fx = pt[order[0]].fx;		li[0].fy = pt[order[0]].fy;
		li[1].fx = pt[order[1]].fx;		li[1].fy = pt[order[1]].fy;
		return ((GraphObj*)dest)->Command(CMD_ADDTOLINE, &li, 0L);
		}
	if(z < pt[order[1]].fz) {
		LinePoint(order[0], order[1], z, &li[0]);
		LinePoint(order[0], order[2], z, &li[1]);
		return ((GraphObj*)dest)->Command(CMD_ADDTOLINE, &li, 0L);
		}
	if(z == pt[order[1]].fz && pt[order[1]].fz < pt[order[2]].fz) {
		li[0].fx = pt[order[1]].fx;		li[0].fy = pt[order[1]].fy;
		LinePoint(order[0], order[2], z, &li[1]);
		return ((GraphObj*)dest)->Command(CMD_ADDTOLINE, &li, 0L);
		}
	if(z < pt[order[2]].fz) {
		LinePoint(order[1], order[2], z, &li[0]);
		LinePoint(order[0], order[2], z, &li[1]);
		return ((GraphObj*)dest)->Command(CMD_ADDTOLINE, &li, 0L);
		}
	if(z == pt[order[1]].fz && z == pt[order[2]].fz) {
		li[0].fx = pt[order[1]].fx;		li[0].fy = pt[order[1]].fy;
		li[1].fx = pt[order[2]].fx;		li[1].fy = pt[order[2]].fy;
		return ((GraphObj*)dest)->Command(CMD_ADDTOLINE, &li, 0L);
		}
	return false;
}

Triangulate::Triangulate(Triangle *t_list)
{
	trl = t_list;		edges = 0L;
}

bool
Triangulate::AddEdge(fPOINT3D *p1, fPOINT3D *p2)
{
	edge *ce, *ne;

	//if edge exists delete both the new and the existing edge
	for(ce = edges, ne = 0L; (ce); ) {
		if((ce->p1.fx == p1->fx && ce->p1.fy == p1->fy && ce->p1.fz == p1->fz
			&& ce->p2.fx == p2->fx && ce->p2.fy == p2->fy && ce->p2.fz == p2->fz)
			|| (ce->p2.fx == p1->fx && ce->p2.fy == p1->fy && ce->p2.fz == p1->fz
			&& ce->p1.fx == p2->fx && ce->p1.fy == p2->fy && ce->p1.fz == p2->fz)) {
			if(ne) ne->next = ce->next;
			else edges = ce->next;
			delete ce;					return true;
			}
		ne = ce;	ce = ce->next;
		}
	//come here for new edge
	ne = new edge();
	if(ne) {
		ne->p1.fx = p1->fx;		ne->p1.fy = p1->fy;		ne->p1.fz = p1->fz;
		ne->p2.fx = p2->fx;		ne->p2.fy = p2->fy;		ne->p2.fz = p2->fz;
		ne->next = edges;		edges = ne;
		}
	return false;
}

bool
Triangulate::AddVertex(fPOINT3D *v)
{
	Triangle *trc, *trn, *tr1;
	edge *ce, *ae;
	DWORD flags = 0L;

	for(trc = trl, trn = 0L, edges = 0L; (trc);) {
		tr1 = trc->next;
		//delete triangles whose circumcircle enclose the new vertex
		if(trc->TestVertex(v->fx, v->fy)) {
			AddEdge(&trc->pt[0], &trc->pt[1]);		AddEdge(&trc->pt[1], &trc->pt[2]);
			AddEdge(&trc->pt[0], &trc->pt[2]);
			if(trn) trn->next = trc->next;
			else trl = trc->next;
			if(trl == trc) trl = 0L;
			flags |= trc->flags;
			delete trc;
			}
		else trn = trc;
		trc = tr1;
		}
	//create new triangles from those edges which where found only once
	for(ce = edges; (ce); ) {
		trn = new Triangle();
		if(trn) {
			if (!(bToBaseRect(ce->p1.fx, ce->p1.fy, ce->p1.fz) || bToBaseRect(ce->p2.fx, ce->p2.fy, ce->p2.fz)
				|| bToBaseRect(v->fx, v->fy, v->fz))) flags &= ~0x01;
			else flags |= 0x01;				//flag reference to super triangles
			trn->flags = flags;
			trn->pt[0].fx = ce->p1.fx;	trn->pt[0].fy = ce->p1.fy;	trn->pt[0].fz = ce->p1.fz;
			trn->pt[1].fx = ce->p2.fx;	trn->pt[1].fy = ce->p2.fy;	trn->pt[1].fz = ce->p2.fz;
			trn->pt[2].fx = v->fx;		trn->pt[2].fy = v->fy;		trn->pt[2].fz = v->fz;
			trn->SetRect();				trn->next = trl;			trl = trn;
			ae = ce->next;				delete(ce);					ce = ae;
			}
		}
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Default data vault
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def_vars::def_vars()
{
	dUnits = 0;
	rlp_strcpy(DecPoint, 2, (char*)".\0");		rlp_strcpy(ColSep, 2, (char*)";\0");
	Line_0.width = .4;		Line_1.width = .04,		Line_2.width = 0.016;
	Line_0.patlength = 6.0;	Line_1.patlength = 0.6;	Line_2.patlength = 0.24;
	Line_0.color = Line_1.color = Line_2.color = 0x00000000L;	//black
	Line_0.pattern = Line_1.pattern = Line_2.pattern = 0L;		//solid line
	FillLine_0.width = FillLine_1.width = FillLine_2.width = 0.05;
	FillLine_0.patlength = 6.0;	FillLine_1.patlength = 0.6;	FillLine_2.patlength = 0.24;
	Line_0.color = Line_1.color = Line_2.color = 0x00000000L;	//black
	Line_0.pattern = Line_1.pattern = Line_2.pattern = 0L;		//solid line
	Fill_0.type = Fill_1.type = Fill_2.type = FILL_NONE;
	Fill_0.color = Fill_1.color = Fill_2.color = 0x00ffffffL;	//white background
	Fill_0.scale = Fill_1.scale = Fill_2.scale = 1.0;			//size = 100%
	Fill_0.hatch = &FillLine_0;	Fill_1.hatch = &FillLine_1;	Fill_2.hatch = &FillLine_2;
	Fill_0.color2 = Fill_1.color2 = Fill_2.color2 = 0x00ffffffL;	//white background
	OutLine_0.width = .2;	OutLine_1.width = .02;	OutLine_2.width = 0.008;
	OutLine_0.patlength = 6.0;	OutLine_1.patlength = 0.6;	OutLine_2.patlength = 0.24;
	OutLine_0.color = OutLine_1.color = OutLine_2.color = 0x00000000L;
	OutLine_0.pattern = OutLine_1.pattern = OutLine_2.pattern = 0L;
	pl = pgl = 0L;	pg = 0L;	pg_fl = 0L;	rrect_rad = 0L;
	cdisp = 0L;		min4log = 0.000001;		axis_color = 0x0L;
	ss_txt = 0.9;	out_upd = 0L;			can_upd = false;
	upd_suspend = false;
#ifdef _WINDOWS								//Windows MSVC
//	iMenuHeight = 0;
//	_WIN32_WINNT
//	WINVER
#if _WIN32_WINNT > 0x0600
	DlgSizeAdj.x = DlgSizeAdj.y = DlgSizeAdj.z = 0;
#else
	DlgSizeAdj.x = DlgSizeAdj.y = DlgSizeAdj.z = -10;
#endif
#else										//Linux 
//	iMenuHeight = 25;
	DlgSizeAdj.x = -20;
	DlgSizeAdj.y = DlgSizeAdj.z = -35;
#endif
	currPath = IniFile = 0L;
	File1 = File2 = File3 = File4 = File5 = File6 = 0L;
	fmt_date = rlp_strdup((char*)"Z.V.Y");
	fmt_time = rlp_strdup((char*)"H:M:S");
	fmt_datetime = rlp_strdup((char*)"Z.V.Y H:M:S");
}

def_vars::~def_vars()
{
	if(currPath) free(currPath);
	currPath = 0L;
	if (IniFile) free(IniFile);
	if(pl) free(pl);
	if(pgl) free(pgl);
	if(pg_fl) free(pg_fl);
	if(pg) free(pg);
	if(rrect_rad) free(rrect_rad);
	if(File1) free(File1);
	if(File2) free(File2);
	if(File3) free(File3);
	if(File4) free(File4);
	if(File5) free(File5);
	if(File6) free(File6);
	if(fmt_date) free(fmt_date);
	if(fmt_time) free(fmt_time);
	if(fmt_datetime) free(fmt_datetime);
}

void
def_vars::NoAmp(unsigned char *txt)
{
	unsigned char *new_txt;
	int i, j;

	if (!txt || !txt[0]) return;
	new_txt = rlp_strdup(txt);
	for(i = j = 0; txt[i]; i++) {
		if (new_txt[i] && new_txt[i] != '"' && new_txt[i] != '\'') txt[j++] = new_txt[i];
		}
	txt[j++] = 0;				free(new_txt);
}

void
def_vars::SetDisp(anyOutput *o)
{
	if(o && o != cdisp) {
		Undo.SetDisp(o);
		cdisp = o;
		}
}

double
def_vars::GetSize(int select)
{
	double RetVal = 0.0;

	switch (select){
	case SIZE_SYMBOL:				RetVal = 3.0;		break;
	case SIZE_SYM_LINE:				RetVal = 0.2;		break;
	case SIZE_DATA_LINE:
		switch(dUnits) {
		case 1:		return Line_1.width;
		case 2:		return Line_2.width;
		default:	return Line_0.width;
		}
	case SIZE_TEXT:					RetVal = _TXH;		break;
	case SIZE_GRECT_TOP:			RetVal = 0.0;		break;
	case SIZE_GRECT_BOTTOM:			RetVal = 110.0;		break;
	case SIZE_GRECT_LEFT:			RetVal = 0.0;		break;
	case SIZE_GRECT_RIGHT:			RetVal = 160.0;		break;
	case SIZE_DRECT_TOP:			RetVal = 10.0;		break;
	case SIZE_DRECT_BOTTOM:			RetVal = 90.0;		break;
	case SIZE_DRECT_LEFT:			RetVal = 25.0;		break;
	case SIZE_DRECT_RIGHT:			RetVal = 140.0;		break;
	case SIZE_PATLENGTH:
	case SIZE_DATA_LINE_PAT:
		switch(dUnits) {
		case 1:		return Line_1.patlength;
		case 2:		return Line_2.patlength;
		default:	return Line_0.patlength;
		}
	case SIZE_AXIS_TICKS:			RetVal = 2.0;		break;
	case SIZE_TICK_LABELS:			RetVal = _TXH;		break;
	case SIZE_WHISKER:
	case SIZE_ERRBAR:				RetVal = 3.0;		break;
	case SIZE_WHISKER_LINE:
	case SIZE_AXIS_LINE:
	case SIZE_BAR_LINE:			
	case SIZE_ERRBAR_LINE:
		switch(dUnits) {
		case 1:		return OutLine_1.width;
		case 2:		return OutLine_2.width;
		default:	return OutLine_0.width;
		}
	case SIZE_BOX:
	case SIZE_BAR:					RetVal = 10.0;				break;
	case SIZE_BUBBLE_LINE:			RetVal = 0.4;				break;
	case SIZE_BUBBLE_HATCH_LINE:	RetVal = 0.1;				break;
	case SIZE_ARROW_LINE:			RetVal = 0.4;				break;
	case SIZE_ARROW_CAPWIDTH:		RetVal = 3.0;				break;
	case SIZE_ARROW_CAPLENGTH:		RetVal = 4.0;				break;
	case SIZE_HAIRLINE:				RetVal = 0.1;				break;
	case SIZE_SEGLINE:				RetVal = 0.4;				break;
	case SIZE_CELLWIDTH:			RetVal = 20.0;				break;
	case SIZE_CELLTEXT:
#ifdef _WINDOWS
		RetVal = 4.5*ss_txt;		break;
#else
		RetVal = 4.5*ss_txt*0.7;	break;
#endif
	case SIZE_RRECT_RAD:			
		return rrect_rad ? *rrect_rad : GetSize(SIZE_SYMBOL);
	case SIZE_SCALE:				return 1.0;
	case SIZE_MVINC:				RetVal = 2.4;				break;
	default:	return 0.0;
		}
	switch(dUnits) {
	case 1:	RetVal /= 10.0;	break;
	case 2:	RetVal = NiceValue(RetVal/25.4);	break;
		}
	return RetVal;
}

DWORD
def_vars::Color(int select)
{
	switch (select){
	case COL_ERROR_LINE:
	case COL_WHISKER:
	case COL_SYM_LINE:			return 0x00000000L;
	case COL_SYM_FILL:			return 0x00ffffffL;
	case COL_ARROW:
	case COL_DATA_LINE:			return Line_0.color;
	case COL_TEXT:				return 0x00000000L;
	case COL_BG:				return 0x00ebebebL;
	case COL_AXIS:				return axis_color;
	case COL_BAR_LINE:			return OutLine_0.color;
	case COL_BAR_FILL:			return 0x00ffffffL;
	case COL_BUBBLE_LINE:
	case COL_BUBBLE_FILLLINE:	return 0x00ff0000L;
	case COL_BUBBLE_FILL:		return 0x00ffc0c0L;
	case COL_DRECT:				return 0x00ffffffL;
	case COL_GRECT:				return 0x00ffffffL;
	case COL_GRECTLINE:			return 0x00e0e0e0L;
	case COL_POLYLINE:			return pl ? pl->color : OutLine_0.color;
	case COL_POLYGON:			return pgl ? pgl->color : OutLine_0.color;
	default:					return 0x00C0C0C0L;	//Error
	}
}

LineDEF *
def_vars::GetLine()
{
	switch (dUnits) {
	case 1:		return &Line_1;
	case 2:		return &Line_2;
	default:	return &Line_0;
		}
}

void
def_vars::SetLine(int u, LineDEF *l, int which)
{
	double lw, pl;
	LineDEF *l1, *l2, *l3;

	switch (which) {
	case 0:	l1 = &Line_0;		l2 = &Line_1;		l3 = &Line_2;		break;
	case 1:	l1 = &FillLine_0;	l2 = &FillLine_1;	l3 = &FillLine_2;	break;
	case 2:	l1 = &OutLine_0;	l2 = &OutLine_1;	l3 = &OutLine_2;	break;
	default: return;
		}
	l1->color = l2->color = l3->color = l->color;
	l1->pattern = l2->pattern = l3->pattern = l->pattern;
	switch (u) {
	case 1:
		lw = l->width*10.0;	pl = l->patlength*10.0;
		l1->width = lw;			l1->patlength = pl;
		l2->width = l->width;	l2->patlength = l->patlength;
		l3->patlength = NiceValue(pl/25.4);
		l3->patlength = NiceValue(pl/25.4);
		break;
	case 2:
		lw = NiceValue(l->width*25.4);	pl = NiceValue(l->patlength*25.4);
		l1->width = lw;			l1->patlength = pl;
		l2->width = lw/10.0;	l2->patlength = pl/10.0;
		l3->width = l->width;	l3->patlength = l->patlength;
		break;
	default:
		lw = l->width;		pl = l->patlength;
		l1->width = l->width;	l1->patlength = l->patlength;
		l2->width = lw/10.0;		l2->patlength = pl/10.0;
		l3->patlength = NiceValue(pl/25.4);
		l3->patlength = NiceValue(pl/25.4);
		break;
		}
}

FillDEF *
def_vars::GetFill(int units)
{
	switch (units) {
	case 1:		return &Fill_1;
	case 2:		return &Fill_2;
	default:	return &Fill_0;
		}
}

void
def_vars::SetFill(int u, FillDEF *fd)
{
	memcpy(&Fill_0, fd, sizeof(FillDEF));
	memcpy(&Fill_1, fd, sizeof(FillDEF));
	memcpy(&Fill_2, fd, sizeof(FillDEF));
	if(fd->hatch) SetLine(u, fd->hatch, 1);
	Fill_0.hatch = &FillLine_0;
	Fill_1.hatch = &FillLine_1;
	Fill_2.hatch = &FillLine_2;
}

LineDEF *
def_vars::GetOutLine()
{
	switch (dUnits) {
	case 1:		return &OutLine_1;
	case 2:		return &OutLine_2;
	default:	return &OutLine_0;
		}
}

LineDEF *
def_vars::plLineDEF(LineDEF *ld)
{
	if(ld) {
		if(pl) free(pl);
		pl = (LineDEF *)malloc(sizeof(LineDEF));
		if(pl) memcpy(pl, ld, sizeof(LineDEF));
		}
	if(pl) return pl;
	else return GetOutLine();
}

LineDEF *
def_vars::pgLineDEF(LineDEF *ol)
{
	if(ol) {
		if(pgl) free(pgl);
		pgl = (LineDEF *)malloc(sizeof(LineDEF));
		if(pgl) memcpy(pgl, ol, sizeof(LineDEF));
		}
	if(pgl) return pgl;
	else return GetOutLine();
}

FillDEF *
def_vars::pgFillDEF(FillDEF *fd)
{
	if(fd) {
		if(pg) free(pg);
		pg = (FillDEF *)malloc(sizeof(FillDEF));
		if(pg){
			memcpy(pg, fd, sizeof(FillDEF));
			if(pg->hatch) {
				if(pg_fl) free(pg_fl);
				pg_fl = (LineDEF *)malloc(sizeof(LineDEF));
				if(pg_fl) memcpy(pg_fl, pg->hatch, sizeof(LineDEF));
				pg->hatch = pg_fl;
				}
			}
		}
	if(pg) return pg;
	else return GetFill(dUnits);
}

double
def_vars::rrectRad(double rad)
{
	if(!rrect_rad)rrect_rad=(double*)malloc(sizeof(double));
	if(rrect_rad) return (*rrect_rad = rad);
	return 0.0;
}

void
def_vars::FileHistory(char *path)
{
	unsigned char *tmp_path = 0L, *tmp;
	unsigned char **history[] = {&File1, &File2, &File3, &File4, &File5, &File6};
	int i, j;

	if(path && (tmp_path=(unsigned char*)malloc(( i= rlp_strlen(path))+10))){
		rlp_strcpy(tmp_path, i+1, path);
		NoAmp((unsigned char*)tmp_path);
		for(j = i ; i > 0 && tmp_path[i] != '/' && tmp_path[i] != '\\'; i--);
		tmp_path[i] = 0;
		if(currPath) free(currPath);
		if(tmp_path[0]) currPath = rlp_strdup((char*)tmp_path);
		else currPath = 0L;
		rlp_strcpy(tmp_path, j+1, path);
		if(File1 && strcmp((char*)File1, (char*)tmp_path)) {
			for(i = 0; i < 6 && tmp_path; i++) {
				if(i && *history[i] && !strcmp((char*)File1, (char*)*history[i])){
					free(*history[i]);
					*history[i] = tmp_path;
					tmp_path = 0L;
					break;
					}
				if(*history[i]) {
					if(strcmp((char*)tmp_path, (char*)*history[i])){
						tmp = *history[i];
						*history[i] = tmp_path;
						tmp_path = tmp;
						}
					else { 
						free(tmp_path);
						tmp_path = 0L;
						}
					}
				else{
					tmp = *history[i];
					*history[i] = tmp_path;
					tmp_path = tmp;
					}
				}
			}
		if(!(*history[0])) {
			*history[0] = tmp_path;
			tmp_path = 0L;
			}
		}
	if(tmp_path) free(tmp_path);
}

void
def_vars::Idle(int cmd)
{
	if(cmd == CMD_UPDATE && can_upd && Notary->IsValidPtr(out_upd) && !upd_suspend) {
		can_upd = false;					//no reentry
		IncrementMinMaxRect(&rec_upd, 6);
		if(rec_upd.left < 0) rec_upd.left = 0;
		if(rec_upd.top < 0) rec_upd.top = 0;
		if(out_upd->OC_type == OC_BITMAP) out_upd->UpdateRect(&rec_upd, false);
		out_upd = 0L;	can_upd = false;
		RedrawTextCursor();
		}
	else if (cmd == CMD_FLUSH) {
		rec_upd.left = rec_upd.right = rec_upd.top = rec_upd.bottom = 0;
		out_upd = 0L;	can_upd = false;
		}
	else if (cmd == CMD_SUSPEND) {
		upd_suspend = true;			return;
		}
	else if (cmd == CMD_DOPLOT) {
		upd_suspend = false;		can_upd = true;
		Idle(CMD_UPDATE);
		return;
		}
}

void
def_vars::UpdRect(anyOutput *o, int x1, int y1, int x2, int y2)
{
	if (!o)	o = out_upd;
	if (!out_upd && o) out_upd = o;
	if (out_upd && out_upd->OC_type != OC_BITMAP) return;
	if(out_upd && o != out_upd) Idle(CMD_UPDATE);
	out_upd = o;
	if(can_upd){
		UpdAdd(o, x1, y1);		UpdAdd(o, x2, y2);
		}
	else SetMinMaxRect(&rec_upd, x1, y1, x2, y2);
	can_upd = true;
}

void
def_vars::UpdRect(anyOutput *o, RECT *rc)
{
	if (!o)	o = out_upd;
	if (!out_upd && o) out_upd = o;
	if (out_upd && out_upd->OC_type != OC_BITMAP) return;
	if (out_upd && o != out_upd) Idle(CMD_UPDATE);
	out_upd = o;
	if (can_upd){
		UpdateMinMaxRect(&rec_upd, rc->left, rc->top);
		UpdateMinMaxRect(&rec_upd, rc->right, rc->bottom);
		}
	else memcpy(&rec_upd, rc, sizeof(RECT));
	can_upd = true;
}

void
def_vars::UpdAdd(anyOutput * o, int x, int y)
{
	if (!o)	o = out_upd;
	if (!out_upd && o) out_upd = o;
	if (!out_upd && o) out_upd = o;
	if (out_upd && o && out_upd != o) Idle(CMD_UPDATE);
	if ((x || y) && o == out_upd && can_upd)
		UpdateMinMaxRect(&rec_upd, x, y);
}

void
def_vars::FreeOut(anyOutput *o)
{
	if (out_upd == o){
		rec_upd.left = rec_upd.right = rec_upd.top = rec_upd.bottom = 0;
		out_upd = 0L;	can_upd = false;
		}
}

LineDEF *
def_vars::GetGridline()
{
	static LineDEF gl;

	gl.color = 0x00d0d0d0L;						gl.width = GetSize(SIZE_HAIRLINE);
	gl.patlength = GetSize(SIZE_SYMBOL)*2.0;	gl.pattern = 0L;
	return &gl;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Chache file input for read operations
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#define CharCacheSize 2048
ReadCache::ReadCache()
{
	Cache = 0L;		idx = max = 0;		eof = true;
}

ReadCache::~ReadCache()
{
	if(Cache) free(Cache);
	Cache = 0L;
}

bool
ReadCache::Open(char *name)
{
	struct stat st;
	char Header[200];
	int i;

	idx = max = 0;
	eof = true;
	if(!name) iFile = -1;
#ifdef USE_WIN_SECURE
	else if(_sopen_s(&iFile, name, O_BINARY, 0x40, S_IWRITE) || iFile < 0) return false;
#else
	else if(-1 ==(iFile = open(name, O_BINARY))) return false;
#endif
	curr_pos = max_pos = 0; 
	if(ProgressBarDlg) {
		if(stat(name, &st) == 0) max_pos = st.st_size;
#ifdef _WINDOWS
		for(i = rlp_strlen(name)-1; i>0 && name[i] !='\\'; i--);
#else
		for(i = rlp_strlen(name)-1; i>0 && name[i] !='/'; i--);
#endif
#ifdef USE_WIN_SECURE
		sprintf_s(Header, 199, (char*)SCMS_INF_FREAD, name + i + 1);
#else
		sprintf(Header, (char*)SCMS_INF_FREAD, name + i + 1);
#endif
		if((i = rlp_strlen(Header)>35)) rlp_strcpy(Header+30, 10, (char*)" ...");
		if(max_pos > 10) ProgressBarDlg->Start(max_pos, &curr_pos, Header);
		}
	Cache = (unsigned char *)malloc((unsigned)(CharCacheSize + 1));
	if(Cache) return true;
	return false;
}

void
ReadCache::Close()
{
#ifdef USE_WIN_SECURE
	if(iFile >= 0) _close(iFile);
#else
	if(iFile >= 0) close(iFile);
#endif
	if(Cache) free(Cache);
	Cache = 0L;
}

unsigned char
ReadCache::Getc()
{
	if(Cache){
		if(idx < max) {
			last = Cache[idx++];	curr_pos++;
			if(last == 0x0a) NoWaitDlgLoop();
			return (last);
			}
		else {
			do {
#ifdef USE_WIN_SECURE
				max = _read(iFile, Cache, CharCacheSize);
#else
				max = read(iFile, Cache, CharCacheSize);
#endif
				if(max <=0) {
					eof = true;
					return 0;
					}
				else eof = false;
				}while(max == 0);
			idx = 1;	curr_pos++;
			return(last = Cache[0]);
			}
		}
	return 0;
}

unsigned char *
ReadCache::GetField()
{
	int i;
	static unsigned char *ret;

	if(Cache && max) {
		while(idx < max && Cache[idx] < 43) idx++;
		if(idx == max){
			if(max == CharCacheSize) {
#ifdef USE_WIN_SECURE
				max = _read(iFile, Cache, CharCacheSize);
#else
				max = read(iFile, Cache, CharCacheSize);
#endif
				idx = 0;
				return GetField();
				}
			else return 0L;
			}
		i = idx;
		while(i < max && Cache[i] > 32 && Cache[i] <= 'z') i++;
		if(i == max) {
			for(i = 0; (Line[i] = Getc()) >32 && Line[i] <= 'z' && i < 4096; i++);
			Line[i] = 0;
			return Line;
			}
		else {
			ret = Cache+idx;
			idx = i;
			return ret;
			}
		}
	return 0L;
}

void
ReadCache::ReadLine(char *dest, int size)
{
	int i=0;
	unsigned char c;

	dest[0] = 0;
	do {
		c =  Getc();
		if(c == 0x09) c = 0x32;			// tab to space
		if(c > 31) dest[i++] = (char)c;
		}while(c && c != 0x0a && i < size);
	dest[i] = 0;
	curr_pos += i;
	NoWaitDlgLoop();
}

bool
ReadCache::GetInt(long *in)
{
	unsigned char *field;

	field = GetField();
	if(field && field[0]) {
		*in = atol((char*)field);
		if(*in == 0 && field[0] == '}') return false;
		return true;
		}
	return false;
}

bool
ReadCache::GetFloat(double *fn)
{
	unsigned char *field;

	field = GetField();
	if(field && field[0]) {
		*fn = atof((char*)field);
		if(*fn == 0.0 && field[0] == '}') return false;
		return true;
		}
	return false;
}

unsigned char
ReadCache::Lastc()
{
	return last;
}

bool
ReadCache::IsEOF()
{
	if (!Cache) return true;
	return eof;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process memory block as if file input
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
MemCache::MemCache(unsigned char *ptr):ReadCache()
{
	if(ptr) {
		max = (long)strlen((char*)ptr);			//rlp_strlen  is limited to 10.000 !!
		if (ptr && ptr[0]) {
			Cache = (unsigned char*)calloc(max + 2, sizeof(unsigned char));
			memcpy(Cache, ptr, max);			eof = false;
			}
		else eof = true;
		}
	if(ProgressBarDlg) {
		max_pos = max;
		if(max_pos > 10) ProgressBarDlg->Start(max_pos, &idx, (char*)"Processing Data ...");
		}
}

MemCache::~MemCache()
{
	if(Cache) free(Cache);
	Cache = 0L;
}

unsigned char
MemCache::Getc()
{
	if(Cache){
		if(idx < max){
			last = Cache[idx++];
			if(last == 0x0a) NoWaitDlgLoop();
			return last;
			}
		else {
			eof = true;
			return 0;
			}
		}
	else eof = true;
	return 0;
}

unsigned char *
MemCache::GetField()
{
	long i;
	static unsigned char *ret;

	if(Cache && max) {
		while(idx < max && Cache[idx] < 43) idx++;
		i = idx;
		while(i < max && Cache[i] > 32 && Cache[i] <= 'z') i++;
		ret = Cache+idx;
		idx = i;
		return ret;
		}
	else if (!Cache) eof = true;
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Postpone drawing after clipping
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
DrawLaterObj::DrawLaterObj()
{
	buff = (GraphObj **)malloc((buff_size = 20) * sizeof(GraphObj*));
	buff_pos = 0;
}

DrawLaterObj::~DrawLaterObj()
{
	if (buff) free(buff);
	buff = NULL;
	buff_size = buff_pos = 0;
}

bool
DrawLaterObj::AddObj(GraphObj * obj)
{
	if (buff_pos > (buff_size - 2)) {
		if (!(buff = (GraphObj**)realloc(buff, (buff_size += 20) * sizeof(GraphObj*)))) return false;
		}
	buff[buff_pos++] = obj;
	return true;
}

bool
DrawLaterObj::DoPlot(anyOutput *o)
{
	int i;
	RECT last_clip;
	bool bRet = false;

	o->CopyClipRC(&last_clip);			// suspend clipping
//	o->ClipRect(NULL);
	if (o && buff && buff_pos) for (i = 0, bRet = true; i < buff_pos; i++) {
		if(buff[i]) buff[i]->DoPlot(o);
		}
	buff_pos = 0;						// only draw once
	o->ClipRect(&last_clip);			// restore clipping
	return bRet;
}

void
DrawLaterObj::Clear()
{
	buff_pos = 0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process Undo 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#define UNDO_RING_SIZE 0x100
#define UNDO_IDX_MASK 0xff
UndoObj::UndoObj()
{
	buff = buff0 = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
	stub1 = ndisp = 0;					busy = false;
	pcb = &stub1;	cdisp = ldisp = 0L;	buffers = 0L;
}

UndoObj::~UndoObj()
{
	Flush();
	if(buff0) free(buff0);
}

//free all resources associated with undo operations
void
UndoObj::Flush()
{
	int i, j;

	if(!buffers) return;
	for(i = 0; i < ndisp; i++) if(buffers[i]) {
		if(buffers[i]->buff) {
			for(j = 0; j < UNDO_RING_SIZE; j++) {
				if(buffers[i]->buff[j]) FreeInfo(&buffers[i]->buff[j]);
				}
			free(buffers[i]->buff);
			}
		free(buffers[i]);
		}
	free(buffers);
	stub1 = ndisp = 0;
	pcb = &stub1;	cdisp = 0L;	buffers = 0L;
}

bool
UndoObj::isEmpty(anyOutput *o)
{
	int i;

	if(!buffers) return true;
	for (i = 0; i < ndisp; i++) if(buffers[i]){
		if(o && buffers[i]->disp == o ) {
			if(buffers[i]->count > 0) return false;
			else return true;
			}
		else if(!o && buffers[i]->count > 0) return false;
		}
	return true;
}

//select buffers associated with the current window/widget,
//create new buffers if called for the first time
void
UndoObj::SetDisp(anyOutput *o)
{
	int i;
	void *ptmp;

	if (o == cdisp) return;
	if(o && o != cdisp) {
		ldisp = cdisp;
		if(buffers) {
			for(i = 0; i < ndisp; i++) {
				if(buffers[i] && buffers[i]->disp == o){
					cdisp = o;
					buff = buffers[i]->buff;	pcb = &buffers[i]->count;
					return;
					}
				else if(!buffers[i] && (buffers[i] = (UndoBuff*)calloc(1, sizeof(UndoBuff)))) {
					buffers[i]->buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
					buffers[i]->disp = cdisp = o;
					buff = buffers[i]->buff;	pcb = &buffers[i]->count;
					return;
					}
				}
			ptmp = memdup(buffers, sizeof(UndoBuff*) *(ndisp + 1), 0);
			if (ptmp){
				free(buffers);		buffers = (UndoBuff**)ptmp;
				buffers[ndisp] = (UndoBuff*)calloc(1, sizeof(UndoBuff));
				if (buffers[ndisp]) {
					buffers[ndisp]->buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
					buffers[ndisp]->disp = cdisp = o;
					buff = buffers[ndisp]->buff;	pcb = &buffers[ndisp]->count;
					ndisp++;
					}
				}
			}
		else if((buffers = (UndoBuff**)calloc(1, sizeof(UndoBuff*)))){
			buffers[0] = (UndoBuff*)calloc(1, sizeof(UndoBuff));
			if (buffers[0]) {
				buffers[0]->buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
				buffers[0]->disp = cdisp = o;			ndisp = 1;
				buff = buffers[0]->buff;				pcb = &buffers[0]->count;
				}
			}
		}
}

//free memory for a closed output
void 
UndoObj::KillDisp(anyOutput *o)
{
	int i, j, c_buf;

	if (ldisp == o) ldisp = NULL;
	if(o && buffers) {
		for(i = 0, c_buf = -1; i < ndisp; i++) {
			if(buffers[i] && buffers[i]->disp == o)	{
				c_buf = i;	break;
				}
			}
		if(c_buf < 0) return;
		if(buffers[c_buf]->buff) {
			for(j = 0; j < UNDO_RING_SIZE; j++) {
				if(buffers[c_buf]->buff[j]) FreeInfo(&buffers[c_buf]->buff[j]);
				}
			free(buffers[c_buf]->buff);
			}
		free(buffers[c_buf]);	buffers[c_buf] = 0L;
		}
	if(ldisp && o == cdisp) SetDisp(ldisp);
	else cdisp = 0L;
}

//remove references to an invalid (probabbly deleted) object
void
UndoObj::InvalidGO(GraphObj *go)
{
	int i, i1, i2;

	if(*pcb >10) {
		if(*pcb < UNDO_RING_SIZE){
			i1 = 0;		i2 = *pcb;
			}
		else {
			i1 = ((*pcb) & UNDO_IDX_MASK);
			i2 = (((*pcb)-UNDO_RING_SIZE) & UNDO_IDX_MASK);
			if(i1 > i2) Swap(i1, i2);
			}
		}
	else {
		i1 = 0;		i2 = *pcb;
		}
	for(i = i1; i < i2; i++) {
		if(buff[i] && buff[i]->owner == go) FreeInfo(&buff[i]);
		if(buff[i]) switch(buff[i]->cmd) {
//		case UNDO_OBJCONF:
		case UNDO_OBJCONF_1:
			if(buff[i]->loc == (void**)go) FreeInfo(&buff[i]);
			break;
			}
		}
}

void
UndoObj::Pop(anyOutput *o)
{
	int idx;

	if(o) {
		SetDisp(o);
		if(*pcb < 1) return;
		idx = ((*pcb-1) & UNDO_IDX_MASK);
		if(buff[idx]) FreeInfo(&buff[idx]);
		(*pcb)--;
		}
}

void
UndoObj::Restore(bool redraw, anyOutput*o)
{
	int i, j, idx;
	DWORD flags;
	UndoList *ul;
	GraphObj **gol;
	unsigned char *savbuf, *target;

	if(o) SetDisp(o);
	else if(cdisp) o = cdisp;
	CurrGO = 0L;
	if(*pcb < 1){
		InfoBox((char*)SCMS_UNDO_EMPTY);
		return;
		}
	do {
		idx = ((*pcb-1) & UNDO_IDX_MASK);
		if(!buff[idx] && *pcb > 0) (*pcb)--;
		} while(!buff[idx] && *pcb > 0);
	if(!buff[idx]) return;
	if (buff[idx]->zd.org.fx != cdisp->getVPorgX() ||
		buff[idx]->zd.org.fy != cdisp->getVPorgY() || buff[idx]->zd.scale != cdisp->getVPorgScale()){
		HideCopyMark(false);
		cdisp->setVPorg(buff[idx]->zd.org.fx, buff[idx]->zd.org.fy, buff[idx]->zd.scale);
		if (cdisp->getVPorgScale() > 100.0) cdisp->setVPorgScale(100.0);
		if (cdisp->getVPorgScale() < 0.05) cdisp->setVPorgScale(0.05);
		if(buff[idx]->owner && buff[idx]->owner->Id < GO_PLOT) CurrGO = buff[idx]->owner;
		if(!((buff[idx]->owner)->Command(CMD_SETSCROLL, 0L, cdisp))) (buff[idx]->owner)->Command(CMD_REDRAW, 0L, cdisp);
		return;
		}
	(*pcb)--;
	busy = true;
	if (buff[idx]->cmd == UNDO_SEPARATOR && idx > 1) {
		free(buff[idx]);	buff[idx] = 0L;		idx--;
		}
	if(buff[idx]) {
		flags = buff[idx]->flags;
		switch(buff[idx]->cmd) {
		case UNDO_DATA:
			RestoreData(buff[idx]);
			break;
		case UNDO_TEXTBUF:
			if(buff[idx]->loc && buff[idx]->data) {
				target = *((TextBuff*)buff[idx]->data)->pbuff;
				target = ((TextBuff*)buff[idx]->data)->buff;
				if(*((TextBuff*)buff[idx]->data)->pbuff) free(*((TextBuff*)buff[idx]->data)->pbuff);
				*(((TextBuff*)buff[idx]->data)->pbuff) = ((TextBuff*)buff[idx]->data)->buff;
				*(((TextBuff*)buff[idx]->data)->psize) = ((TextBuff*)buff[idx]->data)->size;
				*(((TextBuff*)buff[idx]->data)->ppos) = ((TextBuff*)buff[idx]->data)->pos;
				}
			break;
		case UNDO_MUTATE:
		case UNDO_DEL_GO:
			((GraphObj*)(buff[idx]->data))->parent = buff[idx]->owner;
			if(buff[idx]->cmd == UNDO_MUTATE && *(buff[idx]->loc))
				::DeleteGO((GraphObj*)*(buff[idx]->loc));
			else CurrGO = (GraphObj*) buff[idx]->data;
			*(buff[idx]->loc) = buff[idx]->data;
			(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
			break;
		case UNDO_DROPGOLIST:
			if((ul = (UndoList *)(buff[idx]->data)) && (ul->array)){
				gol = (GraphObj**)(*(ul->loc_arr));
				if(gol) for (i = 0; i < *(ul->loc_count); i++) if(gol[i]) ::DeleteGO(gol[i]);
				*(ul->loc_count) = ul->count;				if(gol) free(gol);
				*(ul->loc_arr) = ul->array;					free(ul);
				}
			break;
		case UNDO_GOLIST:
			if((ul = (UndoList *)(buff[idx]->data)) && (ul->array)){
				memcpy(*(ul->loc_arr), ul->array, ul->count * sizeof(GraphObj*));
				*(ul->loc_count) = ul->count;				free(ul->array);
				free(ul);
				}
			break;
		case UNDO_DROPMEM:
			*(buff[idx]->loc) = buff[idx]->data;				break;
		case UNDO_SAVVAR:
			if(!(savbuf = (unsigned char *)buff[idx]->data))break;
			for(i = 0; ; ) {
				memcpy(&target, savbuf+i, sizeof(unsigned char*));	i += sizeof(unsigned char*);
				if (target){
					j = *((int*)(savbuf + i));								i += sizeof(int);
					memcpy(target, savbuf + i, j);							i += j;
					}
				else break;
				}
			if(buff[idx]->owner)(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
			free(savbuf);
			break;
		case UNDO_VALDWORD:
			*((DWORD*)(buff[idx]->loc)) = *((DWORD*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_POINT:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(POINT));
			free(buff[idx]->data);								break;
		case UNDO_VOIDPTR:
			*buff[idx]->loc = buff[idx]->data;					break;
		case UNDO_VALINT:
			*((int*)(buff[idx]->loc)) = *((int*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_VALLONG:
			*((long*)(buff[idx]->loc)) = *((long*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_OBJCONF_1:			//single object restore
			UpdGOfromMem((GraphObj *)buff[idx]->loc, (unsigned char *)buff[idx]->data);
			free(buff[idx]->data);								break;
		case UNDO_OBJCONF:				//tree of objects to restore
			RestoreConf(buff[idx]);
			if(buff[idx] && buff[idx]->data) free(buff[idx]->data);
			break;
		case UNDO_LFP:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(lfPOINT));
			free(buff[idx]->data);								break;
		case UNDO_MOVE:
			(buff[idx]->owner)->Command(CMD_UNDO_MOVE, buff[idx]->data, 0L);
			free(buff[idx]->data);								break;
		case UNDO_RECT:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(fRECT));
			free(buff[idx]->data);								break;
		case UNDO_STRING:
			if(*(buff[idx]->loc) && buff[idx]->data)
				rlp_strcpy ((char*)(*buff[idx]->loc), 100, (char*)(buff[idx]->data));
			else if(*(buff[idx]->loc))((char*)*(buff[idx]->loc))[0] = 0;
			if(buff[idx]->data) free(buff[idx]->data);
			CurrGO = buff[idx]->owner;							break;
		case UNDO_ETSTRING:
			if (*(buff[idx]->loc) && buff[idx]->data)
				rlp_strcpy((char*)(*buff[idx]->loc), 100, (char*)(buff[idx]->data));
			else if (*(buff[idx]->loc))((char*)*(buff[idx]->loc))[0] = 0;
			if (buff[idx]->data) free(buff[idx]->data);
			if (CurrText) CurrText->Update(2, 0L, 0L);
			((EditText *)buff[idx]->owner)->Update(1, 0L, 0L);
			CurrGO = 0L;		 break;
		case UNDO_ROTDEF:
			memcpy(*(buff[idx]->loc), buff[idx]->data, 6 * sizeof(double));
			free(buff[idx]->data);								break;
		case UNDO_SETGO:
			::DeleteGO(*((GraphObj**)(buff[idx]->loc)));
			*((GraphObj**)(buff[idx]->loc)) = 0L;				break;
		case UNDO_LINEDEF:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(LineDEF));
			free(buff[idx]->data);								break;
		case UNDO_FILLDEF:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(FillDEF) - (sizeof(LineDEF*) + sizeof(DWORD)));
			((FillDEF*)(buff[idx]->loc))->color2 = ((FillDEF*)(buff[idx]->data))->color2;
			free(buff[idx]->data);								break;
		case UNDO_AXISDEF:
			if(((AxisDEF*)(buff[idx]->loc))->breaks) free(((AxisDEF*)(buff[idx]->loc))->breaks);
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(AxisDEF));
			free(buff[idx]->data);								break;
		case UNDO_TEXTDEF:
			if(((TextDEF*)(buff[idx]->loc))->text) free(((TextDEF*)(buff[idx]->loc))->text);
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(TextDEF));
			free(buff[idx]->data);								break;
		case UNDO_LFP3D:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(fPOINT3D));
			free(buff[idx]->data);								break;
		case UNDO_FLOAT:
			*((double*)(buff[idx]->loc)) = *((double*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_SEPARATOR:
			flags &= ~UNDO_CONTINUE;	//break UNDO_CONTINUE chain
			break;
		case UNDO_MEM:
			if((ul = (UndoList *)(buff[idx]->data)) && (ul->array)){
				memcpy(*(ul->loc_arr), ul->array, ul->size);
				*(ul->loc_count) = ul->count;
				free(ul->array);
				if(buff[idx]->owner->Id < GO_PLOT) CurrGO = (GraphObj*) buff[idx]->owner;
				if(buff[idx]->owner)(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
				}
			break;
			}
		if(flags & UNDO_CONTINUE){
			free(buff[idx]);	buff[idx] = 0L;
			Restore(redraw, cdisp);
			}
		else if (buff[idx]->cmd != UNDO_SEPARATOR){
			if (o) {
				o->HideMark();
				o->MrkMode = MRK_NONE;
				}
			if (redraw && buff[idx] && buff[idx]->owner){
				if (buff[idx]->cmd == UNDO_ETSTRING) {
					//buff[idx] is not a graphic object but a text cell or input text dialog
					((EditText*)(buff[idx]->owner))->Command(CMD_REDRAW, 0L, 0L);
					}
				else if (buff[idx]->owner->Id == GO_TEXTFRAME) {
					((TextFrame*)(buff[idx]->owner))->Command(CMD_UNDO, 0L, 0L);
					}
				else {
					(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
					CurrGO = 0L;
					(buff[idx]->owner)->Command(CMD_REDRAW, 0L, 0L);
					}
				}
			CurrGO = 0L;		CurrText = 0L;		HideTextCursor();
			if(buff[idx]) free(buff[idx]);
			buff[idx] = 0L;
			}
		}
	else {
		InfoBox((char*)SCMS_UNDO_EMPTY);
		}
	busy = false;
}

void
UndoObj::ListGOmoved(GraphObj **oldlist, GraphObj **newlist, long size)
{
	long i;
	int c;

	if(!oldlist || !newlist || oldlist == newlist) return;
	for(i = 0; i < size; i++) if(oldlist[i] == newlist[i]) {
		for(c = 0; c < UNDO_RING_SIZE; c++) {
			if(buff[c]) switch(buff[c]->cmd) {
			case UNDO_DEL_GO:
			case UNDO_SETGO:
			case UNDO_OBJCONF_1:
			case UNDO_OBJCONF:
				if(buff[c]->loc == (void**) &oldlist[i]){
					buff[c]->loc = (void**) &newlist[i];
					}
				break;
				}
			}
		}
}

void
UndoObj::DeleteGO(GraphObj **go, DWORD flags, anyOutput *o)
{
	if(!go || !(*go)) return;
	HideCopyMark(true);
	if(o){
		SetDisp(o);					 o->HideMark();
		}
	if((*go)->Id == GO_POLYLINE || (*go)->Id == GO_POLYGON || (*go)->Id == GO_BEZIER){
		if(CurrHandle && CurrHandle->parent==*go) {
			if((*go)->Command(CMD_DELOBJ, CurrHandle, 0l)) return;
			}
		}
	NewItem(UNDO_DEL_GO, flags, (*(go))->parent, *(go), (void**)go);
	(*(go))->parent = 0L;
	*(go) = 0L;	 CurrGO = 0L;
}

void
UndoObj::MutateGO(GraphObj **old, GraphObj *repl, DWORD flags, anyOutput *o)
{
	if(!old || !(*old)) return;
	if(o){
		SetDisp(o);					 o->HideMark();
		}
	if(!(*old))return;	//HideMark might delete object: should never happen
	if(CurrGO == *old) CurrGO = 0L;
	NewItem(UNDO_MUTATE, flags, (*(old))->parent, *(old), (void**)old);
	repl->parent = (*(old))->parent;
	(*(old))->parent = 0L;
	*(old) = repl;
	repl->parent->Command(CMD_REDRAW, 0L, o);
}

void
UndoObj::StoreListGO(GraphObj *parent, GraphObj ***go, long *count, DWORD flags)
{
	UndoList *ul;

	ul = (UndoList *)malloc(sizeof(UndoList));
	if(ul) {
		ul->array = memdup(*go, *count * sizeof(GraphObj*), 0);
		if (ul->array){
			ul->loc_arr = (void **)go;			ul->count = *count;
			ul->loc_count = count;
			if(0 > NewItem(UNDO_GOLIST, flags, parent, ul, 0L)) {
				free(ul->array);			free(ul);
				}
			}
		else free(ul);
		}
}

void
UndoObj::DropListGO(GraphObj *parent, GraphObj ***go, long *count, DWORD flags)
{
	UndoList *ul;

	ul = (UndoList *)malloc(sizeof(UndoList));
	if(ul) {
		ul->array = *go;
		if (ul->array) {
			ul->loc_arr = (void **)go;		*go = 0L;
			ul->count = *count;			*count = 0;
			ul->loc_count = count;
			if(0 > NewItem(UNDO_DROPGOLIST, flags, parent, ul, 0L)) {
				free(ul->array);			free(ul);
				}
			}
		else free(ul);
		}
}

void
UndoObj::DropMemory(GraphObj *parent, void **mem, DWORD flags)
{
	NewItem(UNDO_DROPMEM, flags, parent, *(mem), mem);
	*mem = 0L;
}

void
UndoObj::SavVarBlock(GraphObj *parent, void **mem, DWORD flags)
{
	NewItem(UNDO_SAVVAR, flags, parent, *(mem), mem);
	*mem = 0L;
}

void
UndoObj::ValDword(GraphObj *parent, DWORD *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(DWORD), 0))) return;
	if(0 > NewItem(UNDO_VALDWORD, flags, parent, ptr, (void**)val)) free(ptr);
}

void 
UndoObj::Point(GraphObj *parent, POINT *pt, anyOutput * o, DWORD flags)
{
	void *ptr;

	if(o) SetDisp(o);
	if(!(ptr = memdup(pt, sizeof(POINT), 0))) return;
	if(0 > NewItem(UNDO_POINT, flags, parent, ptr, (void**)pt)) free(ptr);
}

void 
UndoObj::VoidPtr(GraphObj *parent, void **pptr, void *ptr, anyOutput *o, DWORD flags)
{
	if(o) SetDisp(o);
	NewItem(UNDO_VOIDPTR, flags, parent, ptr, pptr);
}

void
UndoObj::ValInt(GraphObj *parent, int *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(int), 0))) return;
	if(0 > NewItem(UNDO_VALINT, flags, parent, ptr, (void**)val)) free(ptr);
}

void
UndoObj::ValLong(GraphObj *parent, long *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(long), 0))) return;
	if(0 > NewItem(UNDO_VALINT, flags, parent, ptr, (void**)val)) free(ptr);
}

void
UndoObj::ObjConf(GraphObj *go, DWORD flags)
{
	long sz;
	int idx;
	
	InvalidGO(go);
	if(0<=(idx = NewItem(UNDO_OBJCONF, flags, go->parent, GraphToMem(go, &sz), (void**)go))){
		if(cObsW == 1) buff[idx]->cmd = UNDO_OBJCONF_1;
		(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
		}
}

int
UndoObj::SaveLFP(GraphObj *go, lfPOINT *lfp, DWORD flags)
{
	int idx;
	void *ptr;

	if(!(ptr = memdup(lfp, sizeof(lfPOINT), 0))) return -1;
	if(0 > (idx = NewItem(UNDO_LFP, flags, go, ptr, (void**)lfp))) free(ptr);
	return idx;
}

void
UndoObj::MoveObj(GraphObj *go, lfPOINT *lfp, DWORD flags)
{
	int idx;
	lfPOINT dsp;

	if(!lfp) return;
	dsp.fx = -1.0 * lfp->fx;		dsp.fy = -1.0 * lfp->fy;
	idx = (*pcb & UNDO_IDX_MASK);
	if (idx > 0) {
		if (buff[idx - 1]->cmd == UNDO_MOVE && buff[idx - 1]->owner == go) {
			((lfPOINT*)(buff[idx - 1]->data))->fx += dsp.fx;
			((lfPOINT*)(buff[idx - 1]->data))->fy += dsp.fy;
			return;
			}
		}
	if((idx = SaveLFP(go, &dsp, flags)) <0) return;
	buff[idx]->cmd = UNDO_MOVE;
}

void
UndoObj::ValRect(GraphObj *go, fRECT *rec, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(rec, sizeof(fRECT), 0))) return;
	if(0 > NewItem(UNDO_RECT, flags, go, ptr, (void**)rec)) free(ptr);
}

int
UndoObj::String(GraphObj *go, char **s, DWORD flags)
{
	char *ptr;
	int iret;

	if(s && *s &&  *(*s)){
		iret = rlp_strlen(*(s));
		ptr = rlp_strdup(*(s));
		}
	else {
		ptr = 0L;				iret = 0;
		}
	if(0 > NewItem(UNDO_STRING, flags, go, ptr, (void**)s)) if(ptr) free(ptr);
	return iret;
}

int
UndoObj::EtString(EditText *go, char **s, DWORD flags)
{
	char *ptr;
	int iret;
	int idx;

	if (s && *s &&  *(*s)){
		iret = rlp_strlen(*(s));
		ptr = rlp_strdup(*(s));
		}
	else {
		ptr = 0L;				iret = 0;
		}
	idx = (*pcb & UNDO_IDX_MASK);
	if (idx > 0 && buff[idx - 1]) {
		if (buff[idx - 1]->cmd == UNDO_ETSTRING && buff[idx - 1]->owner == (GraphObj*)go && buff[idx - 1]->loc == (void**)s 
			&& rlp_strlen(*s) > rlp_strlen((char*)buff[idx - 1]->data)) {
			if (ptr) free(ptr);
			return 0;
			}
		}
	if (ptr && 0 > NewItem(UNDO_ETSTRING, flags, (GraphObj*)go, ptr, (void**)s)) if (ptr) free(ptr);
	return iret;
}

void
UndoObj::RotDef(GraphObj *go, double **d, DWORD flags)
{
	void *ptr;
	int idx;
	double *orot;

	if (flags & UNDO_CONTINUE) {
		idx = (*pcb & UNDO_IDX_MASK);
		if (idx > 1 && buff[idx - 1]) {
			if (buff[idx - 1]->cmd == UNDO_ROTDEF && buff[idx - 2]->cmd == UNDO_ROTDEF && buff[idx - 1]->owner == (GraphObj*)go
				&& buff[idx - 2]->owner == (GraphObj*)go) {
				orot = (double*)buff[idx - 1]->data;
				memcpy(orot, *d, 6 * sizeof(double));
				return;
				}
			}
		}
	if(!(ptr = memdup(*d, 6 * sizeof(double), 0))) return;
	NewItem(UNDO_ROTDEF, 0, go, ptr, (void**)d);
}

void
UndoObj::SetGO(GraphObj *parent, GraphObj **pos, GraphObj *go, DWORD flags)
{
	*pos = go;
	NewItem(UNDO_SETGO, flags, parent, 0L, (void**)pos);
}

void
UndoObj::Line(GraphObj *go, LineDEF *ld, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(ld, sizeof(LineDEF), 0))) return;
	if(0 > NewItem(UNDO_LINEDEF, flags, go, ptr, (void**)ld)) free(ptr);
}

void
UndoObj::Fill(GraphObj *go, FillDEF *fd, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(fd, sizeof(FillDEF), 0))) return;
	if(0 > NewItem(UNDO_FILLDEF, flags, go, ptr, (void**)fd)) free(ptr);
}

void
UndoObj::AxisDef(GraphObj *go, AxisDEF *ad, DWORD flags)
{
	AxisDEF *ptr;

	if(!(ptr = (AxisDEF*) memdup(ad, sizeof(AxisDEF), 0))) return;
	if(ptr->nBreaks && ptr->breaks) ptr->breaks = 
		(lfPOINT*)memdup(ad->breaks, ad->nBreaks * sizeof(lfPOINT), 0);
	if(0 > NewItem(UNDO_AXISDEF, flags, go, ptr, (void**)ad)) free(ptr);
}

void
UndoObj::TextDef(GraphObj *go, TextDEF *td, DWORD flags)
{
	TextDEF *ptr;

	ptr = (TextDEF*)memdup(td, sizeof(TextDEF), 0);
	if(!ptr) return;
	if(td->text && td->text[0]) ptr->text = rlp_strdup(td->text);
	if(0 > NewItem(UNDO_TEXTDEF, flags, go, ptr, (void**)td)){
		if(ptr->text) free(ptr->text);
		free(ptr);
		}
}

void
UndoObj::ValLFP3D(GraphObj *go, fPOINT3D *lfp, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(lfp, sizeof(fPOINT3D), 0))) return;
	if(0 > NewItem(UNDO_LFP3D, flags, go, ptr, (void**)lfp)) free(ptr);
}

void
UndoObj::ValFloat(GraphObj *parent, double *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(double), 0))) return;
	if(0 > NewItem(UNDO_FLOAT, flags, parent, ptr, (void**)val)) free(ptr);
}

void
UndoObj::DataMem(GraphObj *go, void **mem, int size, long *count, DWORD flags)
{
	UndoList *ul;

	ul = (UndoList *)malloc(sizeof(UndoList));
	if(ul) {
		ul->array = memdup(*mem, size, 0);
		if (ul->array){
			ul->loc_arr = (void **)mem;
			ul->size = size;
			ul->count = *count;
			ul->loc_count = count;
			if(0 > NewItem(UNDO_MEM, flags, go, ul, 0L)) {
				free(ul->array);			free(ul);
				}
			}
		else free(ul);
		}
}

void
UndoObj::DataObject(GraphObj *go, anyOutput *o, DataObj *d, RECT *rc, DWORD flags)
{
	StrData *save;

	if(!go || !d) return;
	if(o) SetDisp(o);
	if(!(save = new StrData(d, rc))) return;
	if(0 > NewItem(UNDO_DATA, flags, go, save, (void**)d)) if(save) delete(save);
}

void
UndoObj::TextBuffer(GraphObj *parent, long *psize, long *ppos, unsigned char **pbuff, DWORD flags, anyOutput *o)
{
	TextBuff *ptr;
	int idx;

	if(o) SetDisp(o);
	if(!parent || !psize || !ppos || !pbuff) return;
	if (flags & UNDO_CONTINUE) {
		idx = (*pcb & UNDO_IDX_MASK);
		if (idx > 1 && buff[idx - 1]) {
			if (buff[idx - 1]->cmd == UNDO_TEXTBUF && buff[idx - 1]->owner == (GraphObj*)parent) {
				ptr = (TextBuff *)buff[idx - 1]->data;
				ptr->size = *psize;			ptr->psize = psize;
				ptr->pos = *ppos;			ptr->ppos = ppos;
				ptr->pbuff = pbuff;			free(ptr->buff);
				ptr->buff = rlp_strdup(*pbuff);
				return;
				}
			}
		}

	ptr = (TextBuff*)calloc(1, sizeof(TextBuff));
	if(ptr) {
		ptr->size = *psize;			ptr->psize = psize;
		ptr->pos = *ppos;			ptr->ppos = ppos;
		ptr->pbuff = pbuff;			ptr->buff = rlp_strdup(*pbuff);
		if(0 > NewItem(UNDO_TEXTBUF, flags, parent, ptr, (void**)pbuff)) {
			if(ptr->buff) free(ptr->buff);
			free(ptr);
			}
		}
}

void 
UndoObj::Separator()
{
	if (*pcb) {
		NewItem(UNDO_SEPARATOR, 0x0, NULL, NULL, (void**)NULL);
		}
}

int
UndoObj::NewItem(int cmd, DWORD flags, GraphObj *owner, void *data, void **loc)
{
	UndoInfo *tmp;
	int idx;

	if(!buff || !cdisp) return -1;
	if(!(tmp = (UndoInfo *)malloc(sizeof(UndoInfo))))return -1;
	tmp->cmd = cmd;			tmp->flags = flags;
	tmp->owner = owner;		tmp->data = data;
	tmp->loc = loc;
	tmp->zd.org.fx = cdisp->getVPorgX();	tmp->zd.org.fy = cdisp->getVPorgY();
	tmp->zd.scale = cdisp->getVPorgScale();
	idx = (*pcb & UNDO_IDX_MASK);
	if (idx && buff[idx - 1] && buff[idx - 1]->cmd == UNDO_SAVVAR) tmp->flags |= UNDO_CONTINUE;
	if (!idx) tmp->flags &= ~UNDO_CONTINUE;
	if(buff[idx]) FreeInfo(&buff[idx]);
	buff[idx] = tmp;
	(*pcb)++;
	return idx;
}

void
UndoObj::FreeInfo(UndoInfo** inf)
{
	int i;
	UndoList *ul;
	GraphObj *go, **gol;

	if(!inf || !(*inf)) return;
	switch((*inf)->cmd) {
	case UNDO_SETGO:
		break;
	case UNDO_DATA:
		delete ((StrData*)((*inf)->data));
		break;
	case UNDO_TEXTBUF:
		if(((TextBuff*)((*inf)->data))->buff) free(((TextBuff*)((*inf)->data))->buff);
		free((*inf)->data);
		break;
	case UNDO_MUTATE:
	case UNDO_DEL_GO:
		go = (GraphObj*)((*inf)->data);
		(*inf)->data = 0L;		::DeleteGO(go);
		break;
	case UNDO_DROPGOLIST:
		if((ul = (UndoList *)((*inf)->data)) && (ul->array)) {
			gol = (GraphObj**)(ul->array);
			for (i = 0; i < ul->count; i++) if(gol[i]) ::DeleteGO(gol[i]);
			free(ul->array);				free(ul);
			}
		break;
	case UNDO_GOLIST:	case UNDO_MEM:
		if((ul = (UndoList *)((*inf)->data)) && (ul->array)) {
			free(ul->array);				free(ul);
			}
		break;
	case UNDO_AXISDEF:
		if(((AxisDEF*)(*inf)->data)->breaks) free(((AxisDEF*)(*inf)->data)->breaks);
		free((*inf)->data);
		break;
	case UNDO_TEXTDEF:
		if(((TextDEF*)(*inf)->data)->text) free(((TextDEF*)(*inf)->data)->text);
		free((*inf)->data);
		break;
	case UNDO_DROPMEM:		case UNDO_VALDWORD:		case UNDO_VALINT:
	case UNDO_OBJCONF:		case UNDO_OBJCONF_1:	case UNDO_LFP:
	case UNDO_MOVE:			case UNDO_RECT:			case UNDO_STRING:
	case UNDO_ROTDEF:		case UNDO_LINEDEF:		case UNDO_FILLDEF:
	case UNDO_LFP3D:		case UNDO_FLOAT:		case UNDO_SAVVAR:
	case UNDO_POINT:		case UNDO_VALLONG:
		free((*inf)->data);
		break;
		}
	free(*inf);
	*inf = 0L;
}

class UndoUtil:public GraphObj {
public:
	GraphObj *res;

	UndoUtil(GraphObj *p, GraphObj *old):GraphObj(0L, 0L){root = p; optr = old; res = 0L;};
	bool Command(int cmd, void *tmpl, anyOutput *o);

private:
	GraphObj *root, *optr;
};

bool
UndoUtil::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj *xch[2];

	switch(cmd){
	case CMD_DROP_GRAPH:
		//we come here if conversion of undo-information is
		//   successfully converted into a tree of graphic objects.
		//   Now ask the parent object to replace the modified
		//   object with a previous version (i.e. undo modifications).
		xch[0] = optr;
		xch[1] = res = (GraphObj*)tmpl;
		if(root) return root->Command(CMD_REPL_GO, xch, o);
		break;
		}
	return false;
}

void
UndoObj::RestoreConf(UndoInfo *inf)
{
	UndoUtil *proc;
	int i;

	//Create a message object which will accept the translated graphic
	//   object or tree of objects, and which will forward it to the parent
	//   of the tree finalizing undo.
	if(!inf->data) return;
	if(!(proc = new UndoUtil(inf->owner, (GraphObj*)inf->loc))) return; 
	OpenGraph(proc, 0L, (unsigned char *)inf->data, false);
	if(proc->res) for(i = 0; i < UNDO_RING_SIZE; i++) {
		if(buff[i] && buff[i]->owner == (GraphObj*)inf->loc) FreeInfo(&buff[i]);
		if(buff[i] && buff[i]->cmd == UNDO_OBJCONF){
			if(buff[i]->loc == inf->loc) buff[i]->loc = (void**)proc->res;
			}
		}
	delete proc;
}

void
UndoObj::RestoreData(UndoInfo *inf)
{
	DataObj *od;
	StrData *nd;
	int w, h;

	nd = (StrData*)inf->data;	od = (DataObj*)inf->loc;
	if(!nd || !od){
		if(nd) delete(nd);
		return;
		}
	nd->GetSize(&w, &h);		od->ChangeSize(w, h, false);
	nd->RestoreData(od);		delete(nd);
}

#undef UNDO_RING_SIZE
#undef UNDO_IDX_MASK
