//PlotObs.cpp, Copyright (c) 2001-2025 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This modules contains code for the differnt Plot objects. Plots are
// graphic objects containing more objects, which represent the data.
// Several Plots may be contained in a Graph: Plots are the different layers
// of a Graph.
// Most part of this module has been moved here from rlplot.cpp of
// earlier versions. 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "rlp_strings.h"

extern char TmpTxt[];
extern def_vars defs;
extern int cPlots;
extern GraphObj *CurrGO, *TrackGO;			//Selected Graphic Objects
extern Axis **CurrAxes;						//axes of current graph
extern long NumCurrAxes;
extern Plot *CurrPlot;
extern UndoObj Undo;

int AxisTempl3D = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Plot::Plot(GraphObj *par, DataObj *d):GraphObj(par, d)
{
	long pos, nsize = 30;

	Id = GO_PLOT;
	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	name = (char*)malloc((nsize)*sizeof(char));
	if(name){
		pos = rlp_strcpy(name, nsize, (char*)"Plot");
		add_int_to_buff(&name, &pos, &nsize, ++cPlots, true);
		}
	use_xaxis = use_yaxis = use_zaxis = 0;	hidden = 0;
	x_info = y_info = z_info = data_desc = 0L;
	x_tv = y_tv = 0L;	x_dtype = y_dtype = z_dtype = 0;
}

void
Plot::DefName(char* first, AccRange *rX, AccRange *rY)
{
	int i;

	if (!rY) return;
	if(!data_desc || !data_desc[0]) data_desc = rlp_strdup(rY->RangeDesc(data, 1));
	i = rlp_strcpy(TmpTxt, 40, first);			i += rlp_strcpy(TmpTxt + i, 40 - i, data_desc);
	if (rX) {
		x_info = rlp_strdup(rX->RangeDesc(data, 1));
		y_info = rlp_strdup(rY->RangeDesc(data, 1));
		i += rlp_strcpy(TmpTxt + i, 40 - i, (void*)" vs. ");		i += rlp_strcpy(TmpTxt + i, 40 - i, rX->RangeDesc(data, 1));
		}
	if (name) free(name);
	name = rlp_strdup(TmpTxt);
}

double
Plot::GetSize(int select)
{
	switch(select){
	case SIZE_MINE:				return 0.0;
	//The Bounds values must be returned by every plot:
	//   they are necessary for scaling !
	case SIZE_BOUNDS_XMIN:		
		return parent ? parent->GetSize(SIZE_BOUNDS_XMIN) : Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		
		return parent ? parent->GetSize(SIZE_BOUNDS_XMAX) : Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		
		return parent ? parent->GetSize(SIZE_BOUNDS_YMIN) : Bounds.Ymin;	
	case SIZE_BOUNDS_YMAX:		
		return parent ? parent->GetSize(SIZE_BOUNDS_YMAX) : Bounds.Ymax;
	case SIZE_BARMINX:
	case SIZE_BARMINY:
		return 1.0f;
	default:
		return DefSize(select);
		}
}

DWORD
Plot::GetColor(int select)
{
	if(parent) return parent->GetColor(select);
	else return defs.Color(select);
}

void
Plot::CheckBounds(double x, double y)
{
	if(x < Bounds.Xmin) Bounds.Xmin = x;
	if(x > Bounds.Xmax) Bounds.Xmax = x;
	if(y < Bounds.Ymin) Bounds.Ymin = y;
	if(y > Bounds.Ymax) Bounds.Ymax = y;
	if (parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
		((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
		((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
		}
	else if (parent && parent->Id == GO_GRAPH) {
		((Graph*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
		((Graph*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
		}
}

bool
Plot::UseAxis(int idx)
{
	if(CurrAxes && CurrAxes[idx]) {
		switch(CurrAxes[idx]->type & 0xf) {
		case 1:									// x-axis
			Undo.ValInt(parent, &use_xaxis, 0L);
			use_xaxis = idx;			return true;
		case 2:									// y-axis
			Undo.ValInt(parent, &use_yaxis, 0L);
			use_yaxis = idx;			return true;
		case 3:									// z-axis
			Undo.ValInt(parent, &use_zaxis, 0L);
			use_zaxis = idx;			return true;
			}
		}
	return false;
}

void
Plot::ApplyAxes(anyOutput *o)
{
	double dx, dy;

	dx = o->co2fix(parent->GetSize(SIZE_GRECT_LEFT));
	dy = o->co2fiy(parent->GetSize(SIZE_GRECT_TOP));
	if (!o || !CurrAxes || !parent) return;
	if(use_xaxis && use_xaxis < NumCurrAxes && CurrAxes[use_xaxis]) {
		o->UseAxis(CurrAxes[use_xaxis]->axis, CurrAxes[use_xaxis]->type & 0xf, dx, dy);
		}
	else use_xaxis = 0;
	if(use_yaxis && use_yaxis < NumCurrAxes && CurrAxes[use_yaxis]) {
		o->UseAxis(CurrAxes[use_yaxis]->axis, CurrAxes[use_yaxis]->type & 0xf, dx, dy);
		}
	else use_yaxis = 0;
	if(use_zaxis && use_zaxis < NumCurrAxes && CurrAxes[use_zaxis]) {
		o->UseAxis(CurrAxes[use_zaxis]->axis, CurrAxes[use_zaxis]->type & 0xf, dx, dy);
		}
	else use_zaxis = 0;
	return;
}

void
Plot::CheckBounds3D(double x, double y, double z)
{
	if(x < xBounds.fx) xBounds.fx = x;
	if(x > xBounds.fy) xBounds.fy = x;
	if(y < yBounds.fx) yBounds.fx = y;
	if(y > yBounds.fy) yBounds.fy = y;
	if(z < zBounds.fx) zBounds.fx = z;
	if(z > zBounds.fy) zBounds.fy = z;
	CheckBounds(x, y);
}

bool
Plot::SavVarObs(GraphObj **gol, long ngo, DWORD flags)
{
	int i;
	void *ptr;

	if(!gol || !ngo) return false;
	SavVarInit(20 * ngo);
	for (i = 0; i < ngo; i++) {
		if (gol[i]) gol[i]->FileIO(SAVE_VARS);
		}
	ptr = SavVarFetch();
	Undo.SavVarBlock(this, &ptr, flags);
	return true;
}

DataObj *
Plot::CreaCumData(char *xr, char *yr, int mode, double base)
{
	char **yranges;
	int i, j, nc, nr, n, c_num, c_txt, c_datetime;
	long ir, ic;
	double value, old_val;
	DataObj *CumData = 0L;
	anyResult ares;
	AccRange *ax = 0L, **ayy = 0L;
	TextValue *tv = 0L;
	bool *validRows;

	if(!xr || !yr || !mode || !data) return 0L;
	if(!(CumData = new DataObj()))return 0L;
	//count valid data lines
	if(!(ax = new AccRange(xr))) {
		delete CumData;		CumData = 0L;	return 0L;
		}
	ax->DataTypes(data, &c_num, &c_txt, &c_datetime);
	nr = ax->CountItems();
	if(!(yranges = split(yr, '&', &nc))){
		delete CumData;		delete ax;		return 0L;
		}
	if(x_tv) x_tv->Reset();
	if(y_tv) y_tv->Reset();
	j = mode == 1 || mode == 2 ? nr : nr * 2;
	if(CumData->Init(j , nc+2) && (validRows = (bool*)calloc(j, sizeof(bool)))){
		if(!c_num && (c_txt + c_datetime) > 0 ) {
			if(x_tv) tv = x_tv;
			else if(y_tv) tv = y_tv;
			else tv = x_tv = new TextValue();
			}
		//setup all ranges
		if(!(ayy = (AccRange**)calloc(nc, sizeof(AccRange*))))return 0L;
		for(i = 0; i < nc; i++) {
			if(yranges[i] && *yranges[i] && (ayy[i] = new AccRange(yranges[i]))) {
				if(!ayy[i]->GetFirst(&ic, &ir)) return 0L;
				}
			}
		// set x values as first column
		for(i = n = 0, ax->GetFirst(&ic, &ir); ax->GetNext(&ic, &ir); i++, n++) {
			if(data->GetResult(&ares, ir, ic, false)) {
				if(tv) {
					switch(ares.type) {
					case ET_TEXT:
						value = tv->GetValue(ares.text);			break;
					default:
						TranslateResult(&ares);
						value = tv->GetValue(ares.text);			break;
						}
					CumData->SetValue(n, 0, value);		CumData->SetValue(n, 1, base);
					}
				else if(ares.type == ET_VALUE && ares.value > -HUGE_VAL && ares.value < HUGE_VAL) {
					CumData->SetValue(n, 0, value = ares.value);	CumData->SetValue(n, 1, base);
					}
				else {
					CumData->SetValue(n, 0, value = 0.0);	CumData->SetValue(n, 1, base);
					}
				if(mode == 3 || mode == 4){				//complete polygon data
					CumData->SetValue((nr<<1)-i-1, 0, value);
					}
				for(j = 0; j < nc; j++) {
					if(CumData->GetValue(n, j+1, &value)) CumData->SetValue(n, j+2, value);
					if(ayy[j]->GetNext(&ic, &ir) && data->GetResult(&ares, ir, ic, false)){
						if(ares.type == ET_VALUE && ares.value > -HUGE_VAL && ares.value < HUGE_VAL){
							value = ares.value;		validRows[i] = true;
							}
						else value = 0.0;
						old_val = 0.0;
						CumData->GetValue(n, j+2, &old_val);
						switch (mode) {
						case 1:	case 3:	value += old_val;			break;
						case 2:	case 4: value = old_val -value;		break;
							}
						CumData->SetValue(n, j+2, value);
						}
					if(mode == 3 || mode == 4)			//complete polygon data
						if(CumData->GetValue(n, j+1, &value)){
							if(validRows[n]) validRows[(nr<<1)-i-1] = true;
							CumData->SetValue((nr<<1)-i-1, j+2, value);
							}
					}
				}
			else {
				for(j = 0; j < nc; j++) ayy[j]->GetNext(&ic, &ir);
				}
			}
		for(i = 0; i < nc; i++) delete ayy[i];
		free(ayy);
		for(i = 0; i < CumData->cRows; i++) {
			if(!validRows[i]) {
				CumData->cRows--;
				for(j = 0; j < CumData->cCols; j++) {
					if(CumData->etRows[i][j]) delete CumData->etRows[i][j];
					}
				free(CumData->etRows[i]);
				for(j = i; j < CumData->cRows; j++) {
					CumData->etRows[j] = CumData->etRows[j+1];
					validRows[j] = validRows[j+1];
					}
				if(!validRows[i] && i < CumData->cRows) i--;
				}
			}
		free(validRows);
		}
	for (i = 0; i < nc; i++) {
		if (yranges[i]) free(yranges[i]);
		}
	if(ax) delete ax;
	free(yranges);
	return CumData;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// PlotScatt handles most XY-Plots: its a Plot-Class
PlotScatt::PlotScatt(GraphObj *par, DataObj *d, DWORD presel):Plot(par, d)
{
	FileIO(INIT_VARS);
	DefSel = presel;
	Id = GO_PLOTSCATT;
	if (!d && par) {
		if(parent && parent->Command(CMD_DELOBJ, this, NULL)) return;
		ErrorBox((char*)SCMS_NODATA4PLOT);
		return;
		}
	lbDist.fx = lbDist.fy = 0.0;
}

PlotScatt::PlotScatt(GraphObj *par, DataObj *d) :Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_PLOTSCATT;
	lbDist.fx = lbDist.fy = 0.0;
}

PlotScatt::PlotScatt(GraphObj *par, DataObj *d, int cBars, Bar **bars, ErrorBar **errs):Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	if(cBars && bars) {
		if((Bars = (Bar**)calloc(cBars, sizeof(Bar*)))) {
			nPoints = nBars = cBars;
			for(i = 0; i < cBars; i++) {
				if((Bars[i] = bars[i])) Bars[i]->parent = this;
				bars[i] = 0L;
				}
			}
		}
	if(cBars && errs) {
		if((Errors = (ErrorBar**)calloc(cBars, sizeof(Bar*)))) {
			nPoints = nErrs = cBars;
			for(i = 0; i < cBars; i++) {
				if((Errors[i] = errs[i])) Errors[i]->parent = this;
				errs[i] = 0L;
				}
			}
		}
	lbDist.fx = lbDist.fy = 0.0;
	Id = GO_PLOTSCATT;
}

PlotScatt::PlotScatt(GraphObj *par, DataObj *d, int nPts, Symbol **sym, DataLine *lin, int nB , Bar **bar) :
	Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	nPoints = nPts;			nBars = nB;
	Symbols = sym;		TheLine = lin;			Bars = bar;
	if (Symbols) for (i = 0; i < nPts; i++) {
		if (Symbols[i]) Symbols[i]->parent = this;
		}
	if (TheLine) TheLine->parent = this;
	if (Bars) for (i = 0; i < nB; i++) {
		if (Bars[i]) Bars[i]->parent = this;
		}
	lbDist.fx = lbDist.fy = 0.0;
	Id = GO_PLOTSCATT;
}

PlotScatt::PlotScatt(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	lbDist.fx = lbDist.fy = 0.0;
	Id = GO_PLOTSCATT;
}

PlotScatt::~PlotScatt()
{
	ForEach(FE_FLUSH, 0L, 0L);
	if(name) free(name);
	name=0L;						if(data_desc) free(data_desc);
	data_desc = 0L;					if(x_tv) delete(x_tv);
	x_tv = 0L;						if(y_tv) delete(y_tv);
	y_tv = 0L;						if (ErrPg) delete(ErrPg);
	ErrPg = 0L;
	Undo.InvalidGO(this);
}

double
PlotScatt::GetSize(int select)
{
	int i;
	double ft1, ft2, d;

	switch(select){
	case SIZE_BARMINX:
		if(BarDist.fx >= 0.0001) return BarDist.fx;
		if((!Bars) || (nBars < 2)) return BarDist.fx = 1.0;
		ft1 = -HUGE_VAL;	ft2 = HUGE_VAL;		BarDist.fx= HUGE_VAL;
		for(i = 0; i < nBars; i++) {
			if(Bars[i]) {
				ft2 = Bars[i]->GetSize(SIZE_XPOS);
				d = fabs(ft2-ft1);
				if(d != 0.0 && d < BarDist.fx) BarDist.fx = d;
				}
			ft1 = ft2;
			}
		return BarDist.fx = BarDist.fx > 0.0001 && BarDist.fx != HUGE_VAL  ? BarDist.fx : 1.0;
	case SIZE_BARMINY:
		if(BarDist.fy >= 0.0001) return BarDist.fy;
		if((!Bars) || (nBars < 2)) return BarDist.fy = 1.0;
		ft1 = -HUGE_VAL;	ft2 = HUGE_VAL;		BarDist.fy= HUGE_VAL;
		for(i = 0; i < nBars; i++) {
			if(Bars[i]) {
				ft2 = Bars[i]->GetSize(SIZE_YPOS);
				d = fabs(ft2-ft1);
				if(d != 0.0 && d < BarDist.fy) BarDist.fy = d;
				}
			ft1 = ft2;
			}
		return BarDist.fy = BarDist.fy > 0.0001 && BarDist.fy != HUGE_VAL  ? BarDist.fy : 1.0;
	case SIZE_LB_XDIST:
		return lbDist.fx;
	case SIZE_LB_YDIST:
		return lbDist.fy;
	case SIZE_BAR_DX:
		return bar_dx;
	default:
		return Plot::GetSize(select);
		}
}

bool
PlotScatt::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_BARMINX:
		BarDist.fx = value;
		return true;
	case SIZE_BARMINY:
		BarDist.fy = value;
		return true;
	case SIZE_SYMBOL:		case SIZE_SYM_LINE:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	case SIZE_WHISKER:		case SIZE_WHISKER_LINE:
	case SIZE_ERRBAR:		case SIZE_ERRBAR_LINE:
		if(Errors)	for(i = 0; i < nErrs; i++) 
			if(Errors[i]) Errors[i]->SetSize(select, value);
		return true;
	case SIZE_BAR_LINE:		case SIZE_BAR:		case SIZE_XBASE:		case SIZE_YBASE:
		if(Bars) for(i = 0; i < nBars; i++) 
			if(Bars[i]) Bars[i]->SetSize(select, value);
		return true;
	case SIZE_LB_XDIST:		case SIZE_LB_YDIST:
		if(Labels) for(i = 0; i < nLabel; i++)
			if(Labels[i]) Labels[i]->SetSize(select, value);
		if((select & 0xfff) == SIZE_LB_XDIST) lbDist.fx = value;
		else if((select & 0xfff) == SIZE_LB_YDIST) lbDist.fy = value;
		return true;
	case SIZE_ARROW_LINE:	case SIZE_ARROW_CAPWIDTH:	case SIZE_ARROW_CAPLENGTH:
		if(Arrows) for(i = 0; i < nArrow; i++)
			if(Arrows[i]) Arrows[i]->SetSize(select, value);
		return true;
	case SIZE_BAR_DX:
		bar_dx = value;			return true;
		}
	return false;
}

bool
PlotScatt::SetColor(int select, DWORD col)
{
	int i;
	long no = 0;
	GraphObj **go = 0L;

	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:		go = (GraphObj**)Symbols;		no = nPoints;	break;
	case COL_WHISKER:
	case COL_ERROR_LINE:	go = (GraphObj**)Errors;		no = nErrs;		break;
	case COL_BAR_LINE:
	case COL_BAR_FILL:		go = (GraphObj**)Bars;			no = nBars;		break;
	case COL_ARROW:			go = (GraphObj**)Arrows;		no = nArrow;	break;
	default:				return false;
		}
	if(go) for(i = 0; i < no; i++)
		if(go[i]) go[i]->SetColor(select, col);
	return true;
}

void
PlotScatt::DoPlot(anyOutput *o)
{
	if(!parent || hidden) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) {
		ApplyAxes(o);
		ForEach(FE_PLOT, 0L, o);
		parent->Command(CMD_AXIS, 0L, o);
		}
	else {
		ForEach(FE_PLOT, 0L, o);
		}
	dirty = false;
}

bool
PlotScatt::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	GraphObj *go;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		if(!CurrGO && ((MouseEvent*)tmpl)->Action == MOUSE_LBUP)
			return ForEach(cmd, tmpl, o);
		return false;
	case CMD_LEGEND:
		if(((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(Bars) for (i = 0; i < nBars; i++)
			if(Bars[i]) Bars[i]->Command(cmd, tmpl, o);
		if(Symbols) {
			if(TheLine && TheLine->Id == GO_DATALINE) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(&TheLine->LineDef, Symbols[i], 0L);
				}
			else {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(0L, Symbols[i], 0L);
				}
			if(TheLine && TheLine->Id == GO_DATAPOLYGON) TheLine->Command(cmd, tmpl, o);
			}
		else if(TheLine) TheLine->Command(cmd, tmpl, o);
		if(Errors) for (i = 0; i < nErrs; i++)
			if(Errors[i]) Errors[i]->Command(cmd, tmpl, o);
		break;
	case CMD_MRK_DIRTY:
		dirty = true;
		if (parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		return UseAxis(*((int*)tmpl));
	case CMD_FLUSH:
		return ForEach(FE_FLUSH, 0L, 0L);
	case CMD_TEXTTHERE:
		if(Labels) for(i = 0; i < nLabel; i++)	if(Labels[i] &&  Labels[i]->Command(cmd, tmpl, o))	return true;
		return false;
	case CMD_AUTOSCALE:
		if(hidden) return false;
		if(dirty){
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		else{
			if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH && 
				Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) {
				((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
				((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
				return true;
				}
			}
		dirty = false;
		//fall through
	case CMD_UPDATE:
		if(cmd == CMD_UPDATE){
			if (Symbols) SavVarObs((GraphObj **)Symbols, nPoints, UNDO_CONTINUE);
			if (Bars) SavVarObs((GraphObj **)Bars, nBars, UNDO_CONTINUE);
			if (Errors) SavVarObs((GraphObj **)Errors, nErrs, UNDO_CONTINUE);
			if (Arrows) SavVarObs((GraphObj **)Arrows, nArrow, UNDO_CONTINUE);
			if (DropLines) SavVarObs((GraphObj **)DropLines, nDrops, UNDO_CONTINUE);
			if (Labels) SavVarObs((GraphObj **)Labels, nLabel, UNDO_CONTINUE);
			dirty = true;
			}
		//fall through
	case CMD_SET_DATAOBJ:
		if(cmd == CMD_SET_DATAOBJ) {
			Id = GO_PLOTSCATT;
			if(data && data == (DataObj *) tmpl) return true;
			data = (DataObj *)tmpl;	
			}
		ForEach(cmd, tmpl, o);
		if(cmd == CMD_AUTOSCALE) {
			if(x_tv) {
				Bounds.Xmin = 0.5;		Bounds.Xmax = ((double)x_tv->Count())+0.5;
				}
			if(y_tv) {
				Bounds.Ymin = 0.5;		Bounds.Xmax = ((double)y_tv->Count())+0.5;
				}
			}
		if(cmd == CMD_AUTOSCALE && parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH
			&& Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) {
			if (ErrPg) ErrPg->Update();
			((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
			((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
			}
		return true;
	case CMD_SCALE:
		return ForEach(cmd, tmpl, o);
	case CMD_MUTATE:		case CMD_REPL_GO:
		dirty = true;
		return ForEach(cmd == CMD_REPL_GO ? FE_REPLGO : FE_MUTATE, tmpl, o);
	case CMD_SYMTEXT:		case CMD_SYMTEXT_UNDO:	case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		dirty = true;
		return true;
	case CMD_SETTEXTDEF:
		if(Labels) for(i = 0; i < nLabel; i++)
			if(Labels[i]) Labels[i]->Command(cmd, tmpl, o);
		dirty = true;
		return true;
	case CMD_DL_LINE:		case CMD_DL_TYPE:
		if(DropLines) for(i = 0; i < nDrops; i++)
			if(DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
		dirty = true;
		return true;
	case CMD_ERR_TYPE:		case CMD_WHISKER_STYLE:		case CMD_ERRDESC:
		if(Errors) for(i = 0; i < nErrs; i++) {
			if(Errors[i]) Errors[i]->Command(cmd, tmpl, o);
			}
		dirty = true;
		return true;
	case CMD_BAR_TYPE:		case CMD_BAR_FILL:
		if(Bars) for(i = 0; i < nBars; i++) {
			if(Bars[i]) Bars[i]->Command(cmd, tmpl, o);
			}
		dirty = true;
		return true;
	case CMD_ARROW_TYPE:	case CMD_ARROW_ORG:
		if (Arrows) for (i = 0; i < nArrow; i++) {
			if (Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
		}
		dirty = true;
		return true;
	case CMD_DELOBJ:
		dirty = true;
		if (parent && tmpl && o) return ForEach(FE_DELOBJ, tmpl, o);
		break;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
	case CMD_SAVE_BARS:
		return SavVarObs((GraphObj **)Bars, nBars, 0L);
	case CMD_SAVE_ERRS:
		return SavVarObs((GraphObj **)Errors, nErrs, 0L);
	case CMD_SAVE_ARROWS:
		return SavVarObs((GraphObj **)Arrows, nArrow, 0L);
	case CMD_SAVE_DROPLINES:
		return SavVarObs((GraphObj **)DropLines, nDrops, 0L);
	case CMD_SAVE_LABELS:
		return SavVarObs((GraphObj **)Labels, nLabel, 0L);
	case CMD_COPY:
		if (parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_TAB:	case CMD_SHTAB:
		go = (GraphObj*)tmpl;
		if (go->Id == GO_DATAPOLYGON || go->Id == GO_DATALINE || go->Id == GO_ERRORPOLYGON) return true;
		if (go && go->parent && go->parent == this) return ForEach(cmd, 0L, o);
		return false;
		}
	return false;
}

void *
PlotScatt::ObjThere(int x, int y)
{
	long i;
	void *ptr;

	if (Symbols) for (i = 0; i < nPoints; i++){
		if (Symbols[i] && (ptr = Symbols[i]->ObjThere(x, y))) return ptr;;
	}
	return 0L;
}

bool
PlotScatt::ForEach(int cmd, void *tmp, anyOutput *o)
{
	int i, j;
	GraphObj **obs[] = { (GraphObj**)Arrows, (GraphObj**)Symbols, (GraphObj**)Errors,
		(GraphObj**)DropLines, (GraphObj**)Labels, (GraphObj**)Bars };
	long c_obs[] = { nArrow, nPoints, nErrs, nDrops, nLabel, nBars };
	long no = 0;
	GraphObj ***go = 0L;
	GraphObj **tmpPlots;
	bool bRedraw;

	switch (cmd) {
	case FE_MUTATE:
	case FE_REPLGO:
		if ((tmpPlots = (GraphObj **)tmp) && tmpPlots[0] && tmpPlots[1]) {
			for (j = 0; j < 6; j++){
				if (obs[j]) for (i = 0; i < c_obs[j]; i++){
					if (obs[j][i] && obs[j][i] == tmpPlots[0]) {
						if (cmd == FE_REPLGO) return ReplaceGO(&obs[j][i], tmpPlots);
						else {
							Undo.MutateGO(&obs[j][i], tmpPlots[1], 0L, o);
							return true;
							}
						}
					}
				}
			if (TheLine == tmpPlots[0]){
				if (cmd == FE_REPLGO) return ReplaceGO((GraphObj**)&TheLine, tmpPlots);
				else {
					Undo.MutateGO((GraphObj**)&TheLine, tmpPlots[1], 0L, o);
					return true;
					}
				}
			}
		return false;
	case FE_PARENT:
		for (j = 0; j < 6; j++){
			if (obs[j]) for (i = 0; i < c_obs[j]; i++){
				if (obs[j][i]) obs[j][i]->parent = this;
				}
			}
		if (TheLine) TheLine->parent = this;
		if (ErrPg) ErrPg->parent = this;
		return true;
	case CMD_UPDATE:	case CMD_SET_DATAOBJ:	case CMD_AUTOSCALE:		case CMD_SCALE:
		for (j = 0; j < 6; j++){
			if (obs[j]) for (i = 0; i < c_obs[j]; i++){
				if (obs[j][i]) obs[j][i]->Command(cmd, tmp, o);
				}
			}
		if (TheLine) TheLine->Command(cmd, tmp, o);
		if (ErrPg) ErrPg->Command(cmd, tmp, o);
		return true;
	case CMD_TAB:
		if (!CurrGO) return false;
		switch (CurrGO->Id) {
		case GO_ARROW:			j = 0;				break;
		case GO_SYMBOL:			j = 1;				break;
		case GO_ERRBAR:			j = 2;				break;
		case GO_DROPLINE:		j = 3;				break;
		case GO_LABEL:			j = 4;				break;
		case GO_BAR:			j = 5;				break;
		default:				return false;
			}
		for (i = 0; i < c_obs[j]; i++){
			if (obs[j][i] && obs[j][i] == CurrGO){
				while (!obs[j][i + 1] && i < c_obs[j]-1) i++;
				if (i < c_obs[j]-1) obs[j][i + 1]->Command(CMD_SELECT, 0L, o);
				return true;
				}
			}
		return false;
	case CMD_SHTAB:
		if (!CurrGO) return false;
		switch (CurrGO->Id) {
		case GO_ARROW:			j = 0;				break;
		case GO_SYMBOL:			j = 1;				break;
		case GO_ERRBAR:			j = 2;				break;
		case GO_DROPLINE:		j = 3;				break;
		case GO_LABEL:			j = 4;				break;
		case GO_BAR:			j = 5;				break;
		default:				return false;
			}
		for (i = 1; i < c_obs[j]; i++){
			if (obs[j][i] && obs[j][i] == CurrGO){
				while (!obs[j][i -1] && i >= 0) i--;
				if (i > 0) obs[j][i-1]->Command(CMD_SELECT, 0L, o);
				return true;
				}
			}
		if (i >= c_obs[j] - 1) return true;
		return false;
	case FE_PLOT:
		if (TheLine) TheLine->DoPlot(o);
		if (ErrPg) ErrPg->DoPlot(o);
		for(j = 5; j >= 0; j--){
			if(obs[j]) for(i = 0; i < c_obs[j]; i++){
				if(obs[j][i]) obs[j][i]->DoPlot(o);
				}
			}
		return true;
	case FE_FLUSH:
		for(j = 0; j < 6; j++){
			if(obs[j]) {
				for(i = 0; i < c_obs[j]; i++) if(obs[j][i]) DeleteGO(obs[j][i]);
				free(obs[j]);	obs[j] = 0L;
				}
			}
		if(ErrRange) free(ErrRange);
		if(yRange) free(yRange);
		if(xRange) free(xRange);
		if(LbRange) free(LbRange);
		ErrRange = yRange = xRange = LbRange = 0L;
		if(TheLine) DeleteGO(TheLine);
		if (ErrPg) DeleteGO(ErrPg);
		Bars = 0L;		Symbols = 0L;		Errors = 0L;	ErrPg = 0L;
		Arrows = 0L;	DropLines = 0L;		Labels = 0L;	TheLine = 0L;
		return true;
	case FE_DELOBJ:
		if(!o) return false;
		for(j = 0, bRedraw = false, go = 0L; j < 6 && !bRedraw; j++) {
			if(obs[j]) for(i = 0; i < c_obs[j]; i++){
				if(obs[j][i]){
					if(tmp == (void*)obs[j][i]) {
						o->MrkMode = MRK_NONE;
						o->MouseCursor(MC_WAIT, true);
						Undo.DeleteGO(&obs[j][i], 0L, o);
						switch(j) {
						case 0: go = (GraphObj***)&Symbols;		no = nPoints;	break;
						case 1: go = (GraphObj***)&Errors;		no = nErrs;		break;
						case 2: go = (GraphObj***)&Arrows;		no = nArrow;	break;
						case 3: go = (GraphObj***)&DropLines;	no = nDrops;	break;
						case 4: go = (GraphObj***)&Labels;		no = nLabel;	break;
						case 5: go = (GraphObj***)&Bars;		no = nBars;		break;
						default:	go = 0L;					no = 0;			break;
							}
						bRedraw = true;
						break;
						}
					}
				}
			}
		if(!bRedraw && TheLine && tmp == (void *) TheLine) {
			o->MrkMode = MRK_NONE;
			Undo.DeleteGO((GraphObj**)(&TheLine), 0L, o);
			bRedraw = true;
			}
		else if (!bRedraw && ErrPg && tmp == (void *)ErrPg) {
			o->MrkMode = MRK_NONE;
			Undo.DeleteGO((GraphObj**)(&ErrPg), 0L, o);
			bRedraw = true;
			}
		if (bRedraw && go) for (i = j = 0; i < no; i++) if (go[0] && go[0][i]) j++;
		if(!j) Undo.DropMemory(this, (void**)go, UNDO_CONTINUE);
		if(bRedraw && dirty) Command(CMD_AUTOSCALE, 0L, o); 
		if(!Bars && !Symbols && !Errors && !Arrows && !TheLine && !DropLines
			&& !Labels) parent->Command(CMD_DELOBJ_CONT, this, o);
		else if(bRedraw) parent->Command(CMD_REDRAW, NULL, o);
		return bRedraw;
	default:							//pass command to all objects
		for(j = 0; j < 6; j++){
			if(obs[j]) for(i = 0; i < c_obs[j]; i++){
				if(obs[j][i]) if(obs[j][i]->Command(cmd, tmp, o)) return true;
				}
			}
		if (ErrPg && ErrPg->Command(cmd, tmp, o)) return true;
		if(TheLine) return (TheLine->Command(cmd, tmp, o));
		return false;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// xyStat is based on scatterplot
xyStat::xyStat(GraphObj *par, DataObj *d):PlotScatt(par, d, 0L)
{
	FileIO(INIT_VARS);
	Id = GO_XYSTAT;
}

xyStat::xyStat(int src):PlotScatt(0)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

xyStat::~xyStat()
{
	ForEach(FE_FLUSH, 0L, 0L);
	if(curr_data) delete curr_data;
	curr_data = 0L;							if(case_prefix) free(case_prefix);
	case_prefix = 0L;						if(yRange) free(yRange);
	yRange = 0L;							if(xRange) free(xRange);
	xRange = 0L;							if(name) free(name);
	name=0L;								if(x_tv) delete(x_tv);
	x_tv = 0;								if (y_tv) delete(y_tv);
	y_tv = 0;
	Undo.InvalidGO(this);
}

bool
xyStat::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_UPDATE:
		if (Symbols) SavVarObs((GraphObj **)Symbols, nPoints, UNDO_CONTINUE);
		if (Bars) SavVarObs((GraphObj **)Bars, nBars, UNDO_CONTINUE);
		if (Errors) SavVarObs((GraphObj **)Errors, nErrs, UNDO_CONTINUE);
		if (Labels) SavVarObs((GraphObj **)Labels, nLabel, UNDO_CONTINUE);
		CreateData();
		ForEach(CMD_SET_DATAOBJ, curr_data, o);
		ForEach(CMD_UPDATE, tmpl, o);
		return dirty = true;
	case CMD_CPY_DATA:
#ifdef _WINDOWS
		extern HWND MainWnd;
		EmptyClipboard();
		OpenClipboard(MainWnd);
		CopyData(this, CF_SYLK);
		CloseClipboard();
		return true;
#else
		return false;
#endif
	case CMD_SET_DATAOBJ:
		if(cmd == CMD_SET_DATAOBJ) {
			Id = GO_XYSTAT;
			if(data && data == (DataObj *) tmpl) return true;
			if(curr_data) delete curr_data;	
			curr_data = 0L;
			data = (DataObj *)tmpl;
			if(data && !curr_data) CreateData();
			tmpl = curr_data;
			}
		ForEach(cmd, tmpl, o);
		return true;
	default:
		return PlotScatt::Command(cmd, tmpl, o);
		}
	return false;
}

void
xyStat::CreateData()
{
	long i, j, k, l, m, n;
	int *ny, c_num, c_txt, c_dattim;
	double y, ss, d, lo, hi, **ay, *ax, *tay, *q1, *q2, *q3;
	lfPOINT *xy;
	AccRange *rX, *rY;
	anyResult x_res, y_res;

	if(!data || !xRange || !yRange || !xRange[0] || !yRange[0]) return;
	if(!(rX = new AccRange(xRange)) || !(rY = new AccRange(yRange))) return;
	if(!x_info) x_info = rlp_strdup(rX->RangeDesc(data, 3));
	if(!y_info) y_info = rlp_strdup(rY->RangeDesc(data, 3));
	m = rX->CountItems();	n = 0;
	if(m < 2 || !(xy = (lfPOINT*) malloc(m * sizeof(lfPOINT)))) {
		delete rX;	delete rY;
		return;
		}
	if(x_tv) delete x_tv;
	x_tv = 0L;
	ny = (int*) calloc(m, sizeof(int));
	ay = (double**) calloc(m, sizeof(double*));
	ax = (double*) calloc(m, sizeof(double));
	tay = (double*)malloc(m * sizeof(double));
	if(!ny || !ay || !ax || !tay) {
		if(ny) free(ny);
		if(ay) free(ay);
		if(ax) free(ax);
		if(tay) free(tay);
		delete rX;	delete rY;
		return;
		}
	rX->DataTypes(data, &c_num, &c_txt, &c_dattim);
	if(c_num < 5 && (c_txt + c_dattim) > 5) {
		x_tv = new TextValue();	
		}
	rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);	dirty = true;
	rX->GetNext(&i, &j);	rY->GetNext(&k, &l);	n=0;
	do {
		if(data->GetResult(&x_res, j, i, false) && data->GetResult(&y_res, l, k, false) && y_res.type == ET_VALUE) {
			xy[n].fy = y_res.value;
			if(x_tv){ 
				switch(x_res.type) {
				case ET_TEXT:
					xy[n++].fx = x_tv->GetValue(x_res.text);
					break;
				case ET_VALUE:	case ET_BOOL:	case ET_DATE:	case ET_TIME:	case ET_DATETIME:
					TranslateResult(&x_res);
					xy[n++].fx = x_tv->GetValue(x_res.text);
					break;
					}
				}
			else if(x_res.type == ET_VALUE) xy[n++].fx = x_res.value;
			}
		}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
	delete rX;			delete rY;
	if(!n) {
		if(ny) free(ny);
		if(ay) free(ay);
		if(ax) free(ax);
		if(tay) free(tay);
		return;
		}
	SortFpArray(n, xy);
	for(i = j = 0; i < (n-1); i++, j++) {
		ax[j] = xy[i].fx;		tay[0] = xy[i].fy;
		ny[j] = 1;
		for(k = 1; xy[i+1].fx == xy[i].fx; k++) {
			tay[k] = xy[i+1].fy;
			i++;		ny[j]++;
			}
		ay[j] = (double*)memdup(tay, k * sizeof(double), 0);
		}
	if(xy[i].fx > xy[i-1].fx) {
		ax[j] = xy[i].fx;		tay[0] = xy[i].fy;
		ny[j] = 1;
		ay[j++] = (double*)memdup(tay, sizeof(double), 0);
		}
	if(type & 0x0480) {		//medians and/or percentiles required
		q1 = (double *)malloc(j * sizeof(double));
		q2 = (double *)malloc(j * sizeof(double));
		q3 = (double *)malloc(j * sizeof(double));
		if(q1 && q2 && q3) {
			for(i = 0; i < j; i++) {
				if(ny[i] > 1) d_quartile(ny[i], ay[i], q1+i, q2+i, q3+i);
				else q1[i] = q2[i] = q3[i] = *ay[i];
				}
			}
		else type &= (~0x0480);
		}
	else q1 = q2 = q3 = 0L;
	if((curr_data = curr_data ? curr_data : new DataObj()) && curr_data->Init(j, 6)) {
		for(i = 0; i < j; i++) curr_data->SetValue(i,0,ax[i]);	// set x-values
		for(i = 0; i < j; i++) {								// set y-values
			if(ny[i] > 1) switch(type & 0x00f0) {
				case 0x0010:	default:
					curr_data->SetValue(i, 1, y=d_amean(ny[i], ay[i]));
					break;
				case 0x0020:
					curr_data->SetValue(i, 1, y=d_gmean(ny[i], ay[i]));
					break;
				case 0x0040:
					curr_data->SetValue(i, 1, y=d_hmean(ny[i], ay[i]));
					break;
				case 0x0080:
					curr_data->SetValue(i, 1, y=q2[i]);
					break;
				}
			else curr_data->SetValue(i, 1, y= *ay[i]);
			curr_data->SetValue(i, 4, y);
			}
		for(i = 0; i < j; i++) {								// set errors
			switch(type & 0x1f00) {
			case 0x0100:	case 0x0200:	case 0x1000:	//SD, SEM, conf. int.
				if(ny[i] > 1) {
					ss = d_variance(ny[i], ay[i], &y);
					switch(type & 0x1f00) {
					case 0x0100:
						curr_data->SetValue(i, 2, sqrt(ss));
						break;
					case 0x0200:
						curr_data->SetValue(i, 2, sqrt(ss)/sqrt((double)ny[i]));
						break;
					case 0x1000:
						d = distinv(t_dist, ny[i]-1, 1, 1.0-(ci/100.0), 2.0);
						curr_data->SetValue(i, 2, d * sqrt(ss)/sqrt((double)ny[i]));
						break;
						}
					}
				else curr_data->SetValue(i, 2, 0.0);
				if(curr_data->GetValue(i, 1, &y) && curr_data->GetValue(i, 2, &hi))
					curr_data->SetValue(i, 4, hi+y);
				break;
			case 0x0400:								//percentiles
				curr_data->SetValue(i, 2, q1[i]);	curr_data->SetValue(i, 3, q3[i]);
				curr_data->SetValue(i, 4, q3[i]);
				break;
			case 0x0800:								//min-max
				lo = hi = *ay[i];
				for(k = 1; k < ny[i]; k++) {
					if(ay[i][k] < lo) lo = ay[i][k];
					if(ay[i][k] > hi) hi = ay[i][k];
					}
				curr_data->SetValue(i, 2, lo);		curr_data->SetValue(i, 3, hi);
				curr_data->SetValue(i, 4, hi);
				break;
				}
			}
		if(type & 0x6000) for(i = 0; i < j; i++) {				// number of cases
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, "%s%d", case_prefix ? case_prefix : "", ny[i]);
#else
			sprintf(TmpTxt, "%s%d", case_prefix ? case_prefix : "", ny[i]);
#endif
			curr_data->SetText(i, 5, TmpTxt);
			}
		}
	if(q1) free(q1);
	if(q2) free(q2);
	if(q3) free(q3);
	for(i = 0; i < m; i++) if(ay[i]) free(ay[i]);
	free(tay);	free(ay);	free(ax);	free(ny);	free(xy);
}

unsigned char *
xyStat::CopySylk()
{
	long cbd = 0, size, nc, nl, i;
	unsigned char *ptr;

	if (!(ptr = (unsigned char *)malloc(size = 10000))) return 0L;
	cbd = rlp_strcpy((char*)ptr, size, (char*)"ID;PWXL;N;E\n"
		"P;Pdd/mm/yyyy\nP;Phh:mm:ss\nP;Pdd/mm/yyyy hh:mm:ss\n");
	nl = 0;	 nc = 0;
	sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
	add_to_buff((char**)&ptr, &cbd, &size, (char*)"Breakdown XY statistics\"\n", 0);
	if (Symbols && nPoints) {
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"Symbols\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 1);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"y\"\n", 0);
		for (i = 0; i < nPoints; i++) {
			if (Symbols[i]) {
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Symbols[i]->GetSize(SIZE_XPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Symbols[i]->GetSize(SIZE_YPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				}
			nl++;
			}
		nc += 2;	nl = 1;
		}
	if (Bars && nBars) {
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"Bars\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 1);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"y\"\n", 0);
		for (i = 0; i < nBars; i++) {
			if (Bars[i]) {
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Bars[i]->GetSize(SIZE_XPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Bars[i]->GetSize(SIZE_YPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				}
			nl++;
			}
		nc += 2;	nl = 1;
		}
	if (Errors && nErrs && Errors[0] && Errors[0]->Id == GO_ERRBAR) {
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"Errors\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 1);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"y\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 2);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"err\"\n", 0);
		for (i = 0; i < nErrs; i++) {
			if (Errors[i]) {
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Errors[i]->GetSize(SIZE_XPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Errors[i]->GetSize(SIZE_YPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 2);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Errors[i]->GetSize(SIZE_ERRBAR_VALUE), false);
				}
			nl++;
			}
		nc += 3;	nl = 1;
		}
	else if (Errors && nErrs && Errors[0] && Errors[0]->Id == GO_WHISKER) {
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"Whiskers\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 1);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"y1\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 2);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"y2\"\n", 0);
		for (i = 0; i < nErrs; i++) {
			if (Errors[i]) {
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Errors[i]->GetSize(SIZE_XPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Errors[i]->GetSize(SIZE_YPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 2);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Errors[i]->GetSize(SIZE_YPOS + 1), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				}
			nl++;
			}
		nc += 3;	nl = 1;
		}
	if (Labels && nLabel) {
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"Labels\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 1);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"y\"\n", 0);
		sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 2);
		add_to_buff((char**)&ptr, &cbd, &size, (char*)"text\"\n", 0);
		for (i = 0; i < nLabel; i++) {
			if (Labels[i]) {
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Labels[i]->GetSize(SIZE_XPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff((char**)&ptr, &cbd, &size, Labels[i]->GetSize(SIZE_YPOS), false);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\n", 1);
				Labels[i]->Command(CMD_GETTEXT, TmpTxt, 0L);
				sylk_cell_ref((char**)&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 2);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)TmpTxt, 0);
				add_to_buff((char**)&ptr, &cbd, &size, (char*)"\"\n", 2);
				}
			nl++;
			}
		nc += 3;	nl = 1;
		}
	add_to_buff((char**)&ptr, &cbd, &size, (char*)"E\n", 2);
	return ptr;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// BarChart is based on scatterplot
BarChart::BarChart(GraphObj *par, DataObj *d):PlotScatt(par, d, 0L)
{
	Id = GO_BARCHART;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Frequenc distribution: bar chart with function
FreqDist::FreqDist(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_FREQDIST;
}

FreqDist::FreqDist(GraphObj *par, DataObj *d, char* range, bool bOnce):Plot(par, d)
{
	FileIO(INIT_VARS);
	if(range && range[0]) {
		ssRef = rlp_strdup(range);
		plots = (GraphObj**)calloc(nPlots=3, sizeof(GraphObj*));
		ProcData(-1);
		if(bOnce && ssRef) {
			free(ssRef);		ssRef = 0L;
			}
		}
	Id = GO_FREQDIST;
}

FreqDist::FreqDist(GraphObj *par, DataObj *d, double *vals, int nvals, int nclasses):Plot(par, d)
{
	int i, j;
	int *cl_data;
	Bar **bars;

	FileIO(INIT_VARS);
	ssRef = 0L;
	plots = (GraphObj**)calloc(nPlots=3, sizeof(GraphObj*));
	for(i = 0, dmin = HUGE_VAL, dmax = -HUGE_VAL; i < nvals; i++) {
		if(vals[i] < dmin) dmin = vals[i];
		if(vals[i] > dmax) dmax = vals[i];
		}
	start = dmin;		step = 1.00001*(dmax-dmin)/((double)nclasses);
	if(!(cl_data = (int*)calloc(nclasses+1, sizeof(int)))) return;
	for(i = 0; i < nvals; i++) {
		j = (int)(floor((vals[i] - start)/step));
		if(j >= 0 && j <= nclasses) cl_data[j]++;
		}
	if(cl_data[nclasses]) nclasses++;
	bars = (Bar**)calloc(nclasses, sizeof(Bar*));
	if(bars) for(i = 0; i < nclasses; i++) {
		bars[i] = new Bar(this, 0L, i*step + start, (double)cl_data[i], BAR_VERTB | BAR_RELWIDTH,
			-1, -1, -1, -1, (char*)"Count");
		if(bars[i]) bars[i]->SetSize(SIZE_BAR, 100.0);
		}
	//create bar chart
	if(bars && (plots[0] = new PlotScatt(this, data, nclasses, bars, 0L))){
		plots[0]->Command(CMD_BAR_FILL, &BarFill, 0L);
		plots[0]->SetColor(COL_BAR_LINE, BarLine.color);
		plots[0]->SetSize(SIZE_BAR_LINE, BarLine.width);
		}
	if(plots[0]){
		Bounds.Xmin = dmin;		Bounds.Xmax = dmax;
		plots[0]->Command(CMD_AUTOSCALE, 0L, 0L);
		}
	free(cl_data);
	Id = GO_FREQDIST;
}

FreqDist::FreqDist(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

FreqDist::~FreqDist()
{
	int i;

	if (plots) {
		for (i = 0; i < nPlots; i++) if (plots[i]) DeleteGO(plots[i]);
		free(plots);						plots = 0L;
		}
	if (curr_data) delete(curr_data);
	curr_data = 0L;							if(ssRef) free(ssRef);
	ssRef = 0L;								if(name) free(name);
	name=0L;								if(x_tv) delete x_tv;
	x_tv = 0L;
}

void
FreqDist::DoPlot(anyOutput *o)
{
	int i;

	if(!plots || !o || !data || !parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	for(i = 0; i < nPlots; i++) {
		if(plots[i]) {
			if(plots[i]->Id >= GO_PLOT && plots[i]->Id < GO_GRAPH){
				if(((Plot*)plots[i])->hidden == 0) plots[i]->DoPlot(o);
				}
			else plots[i]->DoPlot(o);
			}
		}
	if(use_xaxis || use_yaxis) parent->Command(CMD_AXIS, 0L, o);
}

bool
FreqDist::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	static MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if (hidden) return false;
		mev = (MouseEvent *)tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			for (i = 0; i < nPlots; i++) if (plots[i]){
				if (plots[i]->Command(cmd, tmpl, o)) return true;
				}
			break;
		}
		return false;
	case CMD_TAB:			case CMD_SHTAB:
		if (nPlots && plots && plots[0]){
			}
		return false;
	case CMD_LEGEND:
		if (cmd == CMD_MOUSE_EVENT) {
			if (hidden || ((MouseEvent*)tmpl)->Action != MOUSE_LBUP || CurrGO) return false;
			}
		if(plots) for(i = 0; i < nPlots; i++) {
			if(plots[i]){
				if(plots[i]->Id >= GO_PLOT && plots[i]->Id < GO_GRAPH){
					if(((Plot*)plots[i])->hidden == 0) plots[i]->Command(cmd, tmpl, o);
					}
				else plots[i]->Command(cmd, tmpl, o);
				}
			}
		return false;
	case CMD_SCALE:
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(plots && tmpl && parent) for(i = 0; i < nPlots; i++) if(plots[i]){
			if(tmpl == (void*)plots[i]) {
				DeleteGO(plots[i]);	plots[i]=0L;
				if(i == 1) type=0;
				parent->Command(CMD_REDRAW, 0L, o);
				return true;
				}
			}
		return false;
	case CMD_UPDATE:
		if(data && parent && plots) {
			ProcData(0);
			if(!curr_data) return false;
			for(i = 0; i < nPlots; i++) if(plots[i]) {
				plots[i]->Command(CMD_SET_DATAOBJ, curr_data, o);
				plots[i]->Command(CMD_UPDATE, 0L, o);
				}
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_FREQDIST;
		data = (DataObj *)tmpl;
		return true;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		return UseAxis(*((int*)tmpl));
	case CMD_AUTOSCALE:
		if(hidden) return false;
		if(dirty){
			Bounds.Xmin = HUGE_VAL;		Bounds.Ymin = 0;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			if(dmax > dmin) {
				Bounds.Xmin = dmin;		Bounds.Xmax = dmax;
				}
			}
		else{
			if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH && 
				Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) {
				((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
				((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
				return true;
				}
			}
		dirty = false;
		if(plots) for(i = 0; i < nPlots; i++) {
			if(plots[i]){
				if(plots[i]->Id >= GO_PLOT && plots[i]->Id < GO_GRAPH){
					if(((Plot*)plots[i])->hidden == 0) plots[i]->Command(cmd, tmpl, o);
					}
				else plots[i]->Command(cmd, tmpl, o);
				}
			}
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH && 
			Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) {
			((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
			((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
			return true;
			}
		return false;
	case CMD_SET_LINE:
		if (tmpl) {
			Undo.Line(this, &FuncLine, UNDO_CONTINUE);
			memcpy(&FuncLine, tmpl, sizeof(LineDEF));
			return true;
			}
		return false;
	case CMD_DROP_PLOT:		case CMD_DROP_GRAPH:
		if(tmpl && plots && nPlots >1) {
			if(plots[1]) DeleteGO(plots[1]);
			plots[1] = (GraphObj*)tmpl;		dirty = true;		plots[1]->parent = this;
			plots[1]->Command(CMD_SET_DATAOBJ, curr_data, o);
			if (plots[1]->Id == GO_FUNCTION) plots[1]->Command(CMD_SET_LINE, &FuncLine, 0L);
			Command(CMD_AUTOSCALE, 0L, o);
			return true;
			}
		return false;
		}
	return false;
}

void
FreqDist::ProcData(int sel)
{
	AccRange *ar;
	long nv, i, j, r, c, ncl;
	int *f_data, cb_f, c_num, c_txt, c_dattim;
	long size_fo, pos_fo;
	double min, max, mean, sd, tmp, *s_data, *t_data, lstep;
	double chi2, df, x, y;
	anyResult *result;
	bool bValid = false;
	Bar **bars = 0L;
	anyResult res;
	TextDEF td;
	char *fo, *fdesc = 0L, formula[500];

	if(!parent || !data || !ssRef || !plots) return;
	if(curr_data) delete(curr_data);
	curr_data = 0L;
	if(x_tv) delete x_tv;
	x_tv = 0L;
	if((curr_data = new DataObj()) && (ar = new AccRange(ssRef))) {
		if(!y_info && (y_info = (char*)malloc(25*sizeof(char)))) rlp_strcpy(y_info, 25, (char*)"No. of observations");
		dmin = HUGE_VAL, dmax = -HUGE_VAL;
		ar->DataTypes(data, &c_num, &c_txt, &c_dattim);
		if(c_num < 5 && (c_txt + c_dattim) > 5) {
			x_tv = new TextValue();		dmin = 0.0;
			}
		//copy spreadsheet data into array
		nv = ar->CountItems();			ar->GetFirst(&c, &r);
		if(!(s_data = (double*)malloc(nv * sizeof(double))) 
			|| !(t_data = (double*)malloc(nv * sizeof(double)))) {
			delete(ar);					return;
			}
		for(nv = 0; ar->GetNext(&c, &r); ) if(data->GetResult(&res, r, c, false)) {
			if(x_tv){
				switch(res.type) {
				case ET_TEXT:
					if((tmp = x_tv->GetValue(res.text))> 0.0)bValid = true;
					else bValid = false;
					break;
				case ET_VALUE:	case ET_DATE:	case ET_TIME:	case ET_DATETIME:	case ET_BOOL:
					TranslateResult(&res);
					if((tmp = x_tv->GetValue(res.text))> 0.0)bValid = true;
					else bValid = false;
					break;
				default: 
					bValid = false;	
					break;
					}
				}
			else {
				if(res.type == ET_VALUE) {
					tmp = res.value;		bValid = true;
					}
				else bValid = false;
				}
			if(bValid) {
				if(tmp > dmax) dmax = tmp;
				if(tmp < dmin) dmin = tmp;
				s_data[nv] = tmp;
				switch (type & 0xff){
				case 2:
					if(tmp > 0.0) t_data[nv] = log(tmp);
					else nv--;
					break;
				default:	t_data[nv] = tmp;		break;
					}
				nv++;
				}
			}
		delete(ar);
		if(!nv || dmin >= dmax) {
			free(s_data);		s_data = 0L;		free(t_data);		t_data = 0L;
			return;
			}
		min = dmin;		max = dmax;
		lstep = (max-min)/100.0;
		d_variance(nv, t_data, &mean, &sd);
		sd = sqrt(sd/((double)(nv-1)));
		step = fabs(step);
		if(x_tv) {
			start = 0.5;	step = 1.0;		max+= 0.5;
			}
		else if(sel == -1) {
			start = min;	step = (max - min)/(step != 0.0 ? step : 7.0);
			}
		else if(sel == -2) {
			min = start;
			}
		ncl = (int)(floor((max-start)/step))+1;
		if(plots[0] &&	(max > (Bounds.Xmax+step/2.0) || min < (Bounds.Xmin-step/2.0))) {
			DeleteGO(plots[0]);		plots[0] = 0L;
			}
		if(!plots[0])bars = (Bar**)calloc(ncl, sizeof(Bar*));
		f_data = (int*)calloc(ncl+1, sizeof(int));
		for(i = 0; i < nv; i++) {
			j = (int)(floor((s_data[i] - start)/step));
			if(j >= 0 && j < ncl) f_data[j]++;
			else if(s_data[i] == max) f_data[j-1]++;
			}
		if(f_data[ncl]) ncl++;
		curr_data->Init(ncl, 2);
		//create data object containg the counts / bin and bars
		for(i = 0; i< ncl; i++) {
			tmp = start + i * step + step * .5;
			curr_data->SetValue(i, 0, tmp);
			curr_data->SetValue(i, 1, (double)f_data[i]);
			if(bars) {
				bars[i] = new Bar(this, 0L, tmp, (double)f_data[i], BAR_VERTB | BAR_RELWIDTH,
					0, i, 1, i, (char*)"Count");
				if (bars[i]) bars[i]->SetSize(SIZE_BAR, 100.0);
				}
			}
		free(s_data);		free(t_data);		free(f_data);
		//create bar chart
		if(bars && (plots[0] = new PlotScatt(this, data, ncl, bars, 0L))){
			plots[0]->Command(CMD_BAR_FILL, &BarFill, 0L);
			plots[0]->SetColor(COL_BAR_LINE, BarLine.color);
			plots[0]->SetSize(SIZE_BAR_LINE, BarLine.width);
			}
		if(plots[0]){
			Bounds.Xmin = dmin;		Bounds.Xmax = dmax;
			plots[0]->Command(CMD_SET_DATAOBJ, curr_data, 0L);
			plots[0]->Command(CMD_AUTOSCALE, 0L, 0L);
			}
		//create function
		if((type & 0xff) && (fo = (char*)malloc(size_fo = 1000))) {
			pos_fo = rlp_strcpy(fo, 1000, (char*) "[1=Function]\nx1=");
			add_dbl_to_buff(&fo, &pos_fo, &size_fo, min-step*2.0, true);
			add_to_buff(&fo, &pos_fo, &size_fo,(char*)"\nx2=", 4);
			add_dbl_to_buff(&fo, &pos_fo, &size_fo, max+step*2.0, true);
			add_to_buff(&fo, &pos_fo, &size_fo,(char*)"\nxstep=", 7);
			add_dbl_to_buff(&fo, &pos_fo, &size_fo, lstep, true);
			add_to_buff(&fo, &pos_fo, &size_fo,(char*)"\nLine=", 6);
			add_dbl_to_buff(&fo, &pos_fo, &size_fo, DefSize(SIZE_DATA_LINE), true);
			add_to_buff(&fo, &pos_fo, &size_fo,(char*)" 6 0x000000ff 0x0\n", 18);
			cb_f = 0;
			switch (type & 0xff){
			case 2:				//lognormal
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*lognormfreq(x,%g,%g)",nv*step, mean, sd);
#else
				cb_f = sprintf(formula,"%g*lognormfreq(x,%g,%g)",nv*step, mean, sd);
#endif
				fdesc = (char*)"Desc=\"Lognormal Dist.\"\n";
				break;
			case 3:				//exponential
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*expfreq(x,%g)",nv*step, 1.0/mean);
#else
				cb_f = sprintf(formula,"%g*expfreq(x,%g)",nv*step, 1.0/mean);
#endif
				fdesc = (char*)"Desc=\"Exponential Dist.\"\n";
				break;
			case 4:				//rectangular
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*%g*(x>=%g&&x<=%g)",nv*step, 1.0/(dmax-dmin), dmin, dmax);
#else
				cb_f = sprintf(formula,"%g*%g*(x>=%g&&x<=%g)",nv*step, 1.0/(dmax-dmin), dmin, dmax);
#endif
				fdesc = (char*)"Desc=\"Rectangular Dist.\"\n";
				break;
			case 5:				//chi-square
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*chifreq(x,%g)",nv*step, mean);
#else
				cb_f = sprintf(formula,"%g*chifreq(x,%g)",nv*step, mean);
#endif
				fdesc = (char*)"Desc=\"Chi<sup>2</sup> Dist.\"\n";
				break;
			case 10:			//binomial
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*binomfreq(x,%g,%g)*(x>0)",nv*step, dmax, mean/dmax);
#else
				cb_f = sprintf(formula,"%g*binomfreq(x,%g,%g)*(x>0)",nv*step, dmax, mean/dmax);
#endif
				fdesc = (char*)"Desc=\"Binomial Dist.\"\n";
				break;
			case 11:			//poisson
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*poisfreq(x,%g)*(x>0)",nv*step, mean);
#else
				cb_f = sprintf(formula,"%g*poisfreq(x,%g)*(x>0)",nv*step, mean);
#endif
				fdesc = (char*)"Desc=\"Poisson Dist.\"\n";
				break;
			default:			//normal
#ifdef USE_WIN_SECURE
				cb_f = sprintf_s(formula, 500, "%g*normfreq(x,%g,%g)",nv*step, mean, sd);
#else
				cb_f = sprintf(formula,"%g*normfreq(x,%g,%g)",nv*step, mean, sd);
#endif
				fdesc = (char*)"Desc=\"Normal Dist.\"\n";
				break;
				}
			if(cb_f) {
				add_to_buff(&fo, &pos_fo, &size_fo, (char*)"f_xy=\"y=" , 8);
				add_to_buff(&fo, &pos_fo, &size_fo, formula , cb_f);
				add_to_buff(&fo, &pos_fo, &size_fo, (char*)"\\n\"\n" , 4);
				}
			if(fdesc)add_to_buff(&fo, &pos_fo, &size_fo, fdesc, 0);
			OpenGraph(this, 0L, (unsigned char *)fo, false);
			free(fo);							chi2 = df = 0.0;
			//calculate chi-square test of fit
			if(curr_data) for(i = 0; i< ncl; i++) {
				if(curr_data->GetValue(i,0, &x) && curr_data->GetValue(i,1, &y)){
#ifdef USE_WIN_SECURE
					sprintf_s(TmpTxt, TMP_TXT_SIZE, "x=%g;%s", x, formula);
#else
					sprintf(TmpTxt, "x=%g;%s", x, formula);
#endif
					result = do_formula(curr_data, TmpTxt);
					if(result->type == ET_VALUE && fabs(result->value) > 0.0) {
						tmp = y-result->value;	
						tmp = (tmp*tmp)/result->value;
						chi2 += tmp;			df += 1.0;
						}
					}
				}
			//report result of the chi-square test
			if(chi2 > 0.0 && parent && (fo = (char*)malloc(size_fo = 1000))) {
				tmp = chi_dist(chi2, df-1.0, 1.0);
				pos_fo = rlp_strcpy(fo, 1000, (char*)"chi<sup> 2</sup> =");
				add_dbl_to_buff(&fo, &pos_fo, &size_fo, chi2, true, (char*)"%.4g");
				add_to_buff(&fo, &pos_fo, &size_fo, (char*)", n =", 5);
				add_dbl_to_buff(&fo, &pos_fo, &size_fo, df, true);
				add_to_buff(&fo, &pos_fo, &size_fo, (char*)", df =", 6);
				add_dbl_to_buff(&fo, &pos_fo, &size_fo, df-1.0, true);
				add_to_buff(&fo, &pos_fo, &size_fo, (char*)", p =", 5);
				if(tmp < 0.0001) {
					pos_fo--;	add_to_buff(&fo, &pos_fo, &size_fo, (char*)"< 0.0001", 8);
					}
				else add_dbl_to_buff(&fo, &pos_fo, &size_fo, tmp, true, (char*)"%.4g");
				if(!plots[2]) {
					x = (parent->GetSize(SIZE_GRECT_RIGHT) - parent->GetSize(SIZE_GRECT_LEFT))/2.0;
					y = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_DRECT_TOP) + DefSize(SIZE_TEXT);
//					y = parent->GetSize(SIZE_GRECT_BOTTOM) - DefSize(SIZE_TEXT)*5;
//					y -= parent->GetSize(SIZE_DRECT_TOP);
					td.Align = TXA_VTOP | TXA_HCENTER;			td.ColBg = 0x00ffffffL;
					td.ColTxt = 0x00ff0000L;					td.Font = FONT_HELVETICA;
					td.fSize = DefSize(SIZE_TEXT);			td.iSize = 0;
					td.Mode = TXM_TRANSPARENT;					td.RotBL = td.RotCHAR = 0.0;
					td.Style = TXS_NORMAL;						td.text = 0L;
					plots[2] = new Label(this, data, x, y, &td, 0x0L, 0L);
					plots[2]->moveable = 1;
					}
				plots[2]->Command(CMD_SETTEXT, fo, 0L);			free(fo);
				}
			}
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Regression line and symbols
Regression::Regression(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_REGRESSION;
}

Regression::Regression(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	Id = GO_REGRESSION;
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(rLine) rLine->parent = this;
		if(sde) sde->parent = this;
		if(Symbols)
			for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->parent = this;
		}
}

Regression::~Regression()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

double
Regression::GetSize(int select)
{
	return Plot::GetSize(select);
}

bool
Regression::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:
	case SIZE_SYM_LINE:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
Regression::SetColor(int select, DWORD col)
{
	int i;
	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:
		if(Symbols) for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
	}
	return false;
}

void
Regression::DoPlot(anyOutput *o)
{
	int i;
	

	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	if(sde){
		if(rLine && sde->Command(CMD_DROP_OBJECT, rLine, o)) rLine = 0L;
		sde->DoPlot(o);
		}
	if(rLine) rLine->DoPlot(o);
	if(Symbols)	for(i = 0; i < nPoints; i++) 
		if(Symbols[i]) Symbols[i]->DoPlot(o);
	if(use_xaxis || use_yaxis) parent->Command(CMD_AXIS, 0L, o);
	dirty = false;
}

bool
Regression::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j;
	static MouseEvent *mev;
	bool bEmpty,bRedraw = false;
	LineDEF *ld;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(Symbols) for (i = nPoints-1; i >= 0; i--)
				if(Symbols[i] && Symbols[i]->Command(cmd, tmpl, o))return true;
			if(rLine && rLine->Command(cmd, tmpl, o)) return true;
			if(sde && sde->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND && rLine && rLine->Id == GO_REGLINE) {
			ld = rLine->GetLine();
			if(Symbols) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(ld, Symbols[i], (char*)"Regression");
				}
			else ((Legend*)tmpl)->HasFill(ld, 0L, (char*)"Regression");
			return true;
			}
		return false;
	case CMD_SETSCROLL:			case CMD_REDRAW: 		case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) {
			if (rLine || sde) Recalc();
			dirty = true;
			}
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_DROP_OBJECT:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_REGLINE && !rLine) {
			rLine = (RegLine *)tmpl;
			rLine->parent = this;
			return true;
			}
		break;
	case CMD_FLUSH:
		if(yRange) free(yRange);
		if(xRange) free(xRange);
		yRange = xRange = 0L;
		if(rLine) DeleteGO(rLine);
		if(sde) DeleteGO(sde);
		rLine = 0L;		sde = 0L;
		if(Symbols) for (i = nPoints-1; i >= 0; i--)
			if(Symbols[i]) DeleteGO(Symbols[i]);
		if(Symbols) free(Symbols);
		Symbols = 0L;
		return true;
	case CMD_SET_DATAOBJ: case CMD_AUTOSCALE:
		if (cmd == CMD_AUTOSCALE){
			if (dirty){
				Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
				Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
				}
			else return true;
			dirty = false;
			}
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_REGRESSION;
			data = (DataObj *)tmpl;	
			}
		if(rLine) rLine->Command(cmd, tmpl, o);
		if (Symbols) for (i = 0; i < nPoints; i++) {
			if (Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_SYMTEXT:		case CMD_SYM_RANGETEXT:		case CMD_SYMTEXTDEF:
	case CMD_SYM_TYPE:		case CMD_SCALE:
		if (cmd == CMD_SCALE) {
			if (rLine) rLine->Command(cmd, tmpl, o);
			if (sde) sde->Command(cmd, tmpl, o);
			}
		if(Symbols && nPoints) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		if (Symbols && nPoints) return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
		return false;
	case CMD_UPDATE:
		if(Symbols) {
			SavVarObs((GraphObj**)Symbols, nPoints, UNDO_CONTINUE);
			for(i = 0; i < nPoints; i++)
				if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			if(rLine || sde) Recalc();
			return true;
			}
		return false;
	case CMD_DELOBJ:
		if(!parent || !o) return false;
		dirty = bEmpty = bRedraw = false;
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i] && (void*)Symbols[i] == tmpl) {
				bRedraw = true;
				o->HideMark();
				Undo.DeleteGO((GraphObj**)(&Symbols[i]), 0L, o);
				for(j = 0, bEmpty = true; j < nPoints; j++) {
					if(Symbols[j]) {
						bEmpty = false;
						break;
						}
					}
				if(!bEmpty && dirty) Command(CMD_AUTOSCALE, 0L, o);
				break;
				}
		if(rLine && (void*)rLine == tmpl) {
			Undo.DeleteGO((GraphObj**)(&rLine), 0L, o);
			if(!Symbols && !sde) parent->Command(CMD_DELOBJ_CONT, this, o);
			else bRedraw = true;
			}
		if(sde && (void*)sde == tmpl) {
			sde->Command(CMD_RMU, 0L, 0L);
			Undo.DeleteGO((GraphObj**)(&sde), 0L, o);
			if(!Symbols && !rLine) parent->Command(CMD_DELOBJ_CONT, this, o);
			else bRedraw = true;
			}
		if(bEmpty && Symbols) {
			Undo.DropMemory(this, (void**)(&Symbols), UNDO_CONTINUE);
			bRedraw = false;
			if(!rLine && !sde) parent->Command(CMD_DELOBJ_CONT, this, o);
			else bRedraw = true;
			}
		if(bRedraw)parent->Command(CMD_REDRAW, 0L, o);
		return bRedraw;
		}
	return false;
}

void
Regression::Recalc()
{
	int i, j;
	long n;
	bool dValid;
	lfPOINT *val;

	if (nPoints < 2 || !Symbols) return;
	val = (lfPOINT*)calloc(nPoints, sizeof(lfPOINT));
	if (!val) return;
	for(i = 0, n = 0; i < nPoints; i++){
		if(Symbols[i] && Symbols[i]->Id == GO_SYMBOL) {
			j = n;
			val[j].fx = Symbols[i]->GetSize(SIZE_XPOS);
			val[j].fy = Symbols[i]->GetSize(SIZE_YPOS);
			dValid = true;
			switch(type & 0x700) {
			case 0x100:					//logarithmic x
				dValid = (val[j].fx > defs.min4log);
				if (dValid) val[j].fx = log10(val[j].fx);
				break;
			case 0x200:					//reciprocal x
				dValid = (fabs(val[j].fx) > defs.min4log);
				if (dValid) val[j].fx = 1.0 / val[j].fx;
				break;
			case 0x300:					//square root x
				dValid = (fabs(val[j].fx) > defs.min4log);
				if (dValid) val[j].fx = sqrt(val[j].fx);
				break;
				}
			if(dValid) switch(type & 0x7000) {
			case 0x1000:				//logarithmic y
				dValid = (val[j].fy > defs.min4log);
				if (dValid ) val[j].fy = log10(val[j].fy);
				break;
			case 0x2000:				//reciprocal y
				dValid = (fabs(val[j].fy) > defs.min4log);
				if(dValid) val[j].fy = 1.0/val[j].fy;
				break;
			case 0x3000:				//square root y
				dValid = (fabs(val[j].fy) > defs.min4log);
				if (dValid) val[j].fy = sqrt(val[j].fy);
				break;
				}
			if(dValid) n++;
			}
		}
	if(sde && sde->Id == GO_SDELLIPSE) sde->Recalc(val, n);
	if(rLine && rLine->Id == GO_REGLINE) rLine->Recalc(val, n);
	free(val);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// myRegression line and symbols: Linear regression with multiple y for each x
myRegression::myRegression(GraphObj *par, DataObj *d) :Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_MYREGR;
	PrevSym = new Symbol(this, 0L, 0.0, 0.0, SYM_CIRCLE);
	PrevErr = new ErrorBar(this, 0L, 0.0, 0.0, 0.0, ERRBAR_VSYM);
}

myRegression::myRegression(int src) : Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	Id = GO_MYREGR;
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if (reg_res) reg_res->parent = this;
		if (func) func->parent = this;
		if (ci_line1) ci_line1->parent = this;
		if (ci_line2) ci_line2->parent = this;
		if (Symbols) for (i = 0; i < nPoints; i++) if (Symbols[i]) Symbols[i]->parent = this;
		if (Errors) for (i = 0; i < nErrs; i++) if (Errors[i]) Errors[i]->parent = this;
		}
}


myRegression::~myRegression()
{
	long i;

	if (func) delete(Function *)func;
	func = 0L;								if (ci_line1) delete(Function *)ci_line1;
	ci_line1 = 0L;							if (ci_line2) delete(Function *)ci_line2;
	ci_line2 = 0L;							if (reg_res) delete reg_res;
	reg_res = 0L;							if (PrevSym) delete PrevSym;
	PrevSym = 0L;							if (PrevErr) delete PrevErr;
	PrevErr = 0L;
	if (Symbols && nPoints) {
		for (i = 0; i < nPoints; i++){
			if (Symbols[i]) delete Symbols[i];
			}
		free(Symbols);		Symbols = 0L;	nPoints = 0;
		}
	if (Errors && nErrs) {
		for (i = 0; i < nErrs; i++){
			if (Errors[i]) delete Errors[i];
			}
		free(Errors);		Errors = 0L;	nErrs = 0;
		}
}

bool
myRegression::SetSize(int select, double value)
{
	int i;

	switch (select & 0xfff){
	case SIZE_SYMBOL:	case SIZE_SYM_LINE:
		if (PrevSym) PrevSym->SetSize(select, value);
		if (Symbols)	for (i = 0; i < nPoints; i++)
			if (Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	case SIZE_ERRBAR:	case SIZE_ERRBAR_LINE:
		if (PrevErr) PrevErr->SetSize(select, value);
		if (Errors)	for (i = 0; i < nErrs; i++)
			if (Errors[i]) Errors[i]->SetSize(select, value);
		return true;
		}
	return false;
}

bool
myRegression::SetColor(int select, DWORD col)
{
	int i;
	switch (select) {
	case COL_SYM_LINE:		case COL_SYM_FILL:
		if (PrevSym) PrevSym->SetColor(select, col);
		if (Symbols) for (i = 0; i < nPoints; i++)
			if (Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
	case COL_ERROR_LINE:
		if (PrevErr) PrevErr->SetColor(select, col);
		if (Errors) for (i = 0; i < nErrs; i++)
			if (Errors[i]) Errors[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
myRegression::DoPlot(anyOutput *o)
{
	int i;

	if (!o) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if (use_xaxis || use_yaxis) ApplyAxes(o);
	if (func)((Function*)func)->DoPlot(o);
	if (ci_line1)((Function*)ci_line1)->DoPlot(o);
	if (ci_line2)((Function*)ci_line2)->DoPlot(o);
	if (Errors)	for (i = 0; i < nErrs; i++){
		if (Errors[i]) Errors[i]->DoPlot(o);
		}
	if (Symbols)	for (i = 0; i < nPoints; i++){
		if (Symbols[i]) Symbols[i]->DoPlot(o);
		}
	if (reg_res) reg_res->DoPlot(o);
	if (use_xaxis || use_yaxis) parent->Command(CMD_AXIS, 0L, o);
	dirty = false;
}

bool
myRegression::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	bool bRedraw;
	static MouseEvent *mev;
	static long flags = 0;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if (hidden) return false;
		mev = (MouseEvent *)tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if (func && ((Function*)func)->Command(cmd, tmpl, o)) return true;
			if (ci_line1 && ((Function*)ci_line1)->Command(cmd, tmpl, o)) return true;
			if (ci_line2 && ((Function*)ci_line2)->Command(cmd, tmpl, o)) return true;
			if (Errors) for (i = nErrs - 1; i >= 0; i--)
				if (Errors[i] && Errors[i]->Command(cmd, tmpl, o))return true;
			if (Symbols) for (i = nPoints - 1; i >= 0; i--)
				if (Symbols[i] && Symbols[i]->Command(cmd, tmpl, o))return true;
			if (reg_res && reg_res->Command(cmd, tmpl, o)) return true;
			break;
		}
		break;
	case CMD_AUTOSCALE:
		if (func) ((Function*)func)->Command(cmd, tmpl, o);
		if (ci_line1) ((Function*)ci_line1)->Command(cmd, tmpl, o);
		if (ci_line2) ((Function*)ci_line2)->Command(cmd, tmpl, o);
		CheckBounds(Bounds.Xmin, Bounds.Ymin);				//assume Bounds being set by Recalc()
		CheckBounds(Bounds.Xmax, Bounds.Ymax);
		return true;
	case CMD_OBJTREE:
		if(func)((ObjTree*)tmpl)->Command(CMD_UPDATE, func, 0L);
		if (ci_line1)((ObjTree*)tmpl)->Command(CMD_UPDATE, ci_line1, 0L);
		if (ci_line2)((ObjTree*)tmpl)->Command(CMD_UPDATE, ci_line2, 0L);
		break;
	case CMD_LEGEND:
		if (tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND) {
			if (func) func->Command(cmd, tmpl, o);
			if (ci_line1) ci_line1->Command(cmd, tmpl, o);
			if (ci_line2) ci_line2->Command(cmd, tmpl, o);
			if (Symbols) {
				for (i = 0; i < nPoints && i < 100; i++)
					if (Symbols[i]) ((Legend*)tmpl)->HasSym(0L, Symbols[i], 0L);
			}
			if (Errors) {
				for (i = 0; i < nErrs && i < 100; i++)
					if (Errors[i]) Errors[i]->Command(cmd, tmpl, o);;
			}
			return true;
		}
		return false;
	case CMD_SETSCROLL:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if (parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_REDRAW:
		// we come here only upon UNDO of the functions:
		//    no need to cause several redraw's by the parent graph!
		return false;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_MYREGR;
		data = (DataObj *)tmpl;
		if (func) func->Command(cmd, tmpl, o);
		if (ci_line1) ci_line1->Command(cmd, tmpl, o);
		if (ci_line2) ci_line2->Command(cmd, tmpl, o);
		if (PrevSym) PrevSym->Command(cmd, 0L, o);
		if (reg_res) reg_res->Command(cmd, tmpl, o);
		if (Symbols) for (i = 0; i < nPoints; i++)
			if (Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		if (Errors && nErrs) for (i = 0; i < nErrs; i++)
			if (Errors[i]) Errors[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SCALE:
		if (func) func->Command(cmd, tmpl, o);
		if (ci_line1) ci_line1->Command(cmd, tmpl, o);
		if (ci_line2) ci_line2->Command(cmd, tmpl, o);
		if (reg_res) reg_res->Command(cmd, tmpl, o);
		if (PrevSym) PrevSym->Command(cmd, tmpl, o);
		if (PrevErr) PrevErr->Command(cmd, tmpl, o);
		if (Symbols && nPoints) for (i = 0; i < nPoints; i++)
			if (Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		if (Errors && nErrs) for (i = 0; i < nErrs; i++)
			if (Errors[i]) Errors[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_TAB:			case CMD_SHTAB:
		if (CurrGO && CurrGO->parent && CurrGO->parent == this) return true;
		return false;
	case CMD_SYMTEXT:		case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if (PrevSym) PrevSym->Command(cmd, tmpl, o);
		if (Symbols && nPoints) for (i = 0; i < nPoints; i++)
			if (Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_ERR_TYPE:
		if (PrevErr) PrevErr->Command(cmd, tmpl, o);
		if (Errors && nErrs) for (i = 0; i < nErrs; i++)
			if (Errors[i]) Errors[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		if (Symbols && nPoints) return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
		return false;
	case CMD_SAVE_ERRS:
		if (Errors && nErrs) return SavVarObs((GraphObj **)Errors, nErrs, 0L);
		return false;
	case CMD_UPDATE:
		Undo.ValLong(parent, &flags, 0L);		//a dummy: force redraw by parent upon undo
		if (func) Undo.String(func, &(((Function*)func)->cmdxy), UNDO_CONTINUE);
		if (ci_line1)Undo.String(ci_line1, &(((Function*)ci_line1)->cmdxy), UNDO_CONTINUE);
		if (ci_line2)Undo.String(ci_line2, &(((Function*)ci_line2)->cmdxy), UNDO_CONTINUE);
		if (Symbols && nPoints) Undo.DropListGO(this, (GraphObj***)&Symbols, &nPoints, UNDO_CONTINUE);
		if (Errors && nErrs) Undo.DropListGO(this, (GraphObj***)&Errors, &nErrs, UNDO_CONTINUE);
		Recalc();
		if (func) ((Function*)func)->Update(o, 0L);
		if (ci_line1) ((Function*)ci_line1)->Update(o, 0L);
		if (ci_line2) ((Function*)ci_line2)->Update(o, 0L);
		dirty = true;
		break;
	case CMD_DELOBJ:
		if (!o && parent && parent->Id == GO_GRAPH) {
			o = ((Graph*)parent)->getDisp();
			}
		if (!parent || !o) return false;
		bRedraw = false;
		if (Symbols) for (i = 0; i < nPoints && !bRedraw; i++) {
			if (Symbols[i] && (void*)Symbols[i] == tmpl) {
				o->HideMark();
				Undo.DeleteGO((GraphObj**)(&Symbols[i]), 0L, o);
				bRedraw = true;		break;
				}
			}
		if (Errors && !bRedraw) for (i = 0; i < nErrs && !bRedraw; i++) {
			if (Errors[i] && (void*)Errors[i] == tmpl) {
				o->HideMark();
				Undo.DeleteGO((GraphObj**)(&Errors[i]), 0L, o);
				bRedraw = true;		break;
				}
			}	
		if (!bRedraw && func && (void*)func == tmpl) {
			o->HideMark();		CurrGO = 0L;
			Undo.DeleteGO((GraphObj**)(&func), 0L, o);
			bRedraw = true;
			}
		else if (!bRedraw && ci_line1 && (void*)ci_line1 == tmpl) {
			o->HideMark();		CurrGO = 0L;
			Undo.DeleteGO((GraphObj**)(&ci_line1), 0L, o);
			bRedraw = true;
			}
		else if (!bRedraw && ci_line2 && (void*)ci_line2 == tmpl) {
			o->HideMark();		CurrGO = 0L;
			Undo.DeleteGO((GraphObj**)(&ci_line2), 0L, o);
			bRedraw = true;
			}
		else if (!bRedraw && reg_res && (void*)reg_res == tmpl) {
			o->HideMark();		CurrGO = 0L;
			Undo.DeleteGO((GraphObj**)(&reg_res), 0L, o);
			bRedraw = true;	
			}
		if (bRedraw)parent->Command(CMD_REDRAW, 0L, o);
		return bRedraw;
		}
	return false;
}

#ifndef _WINDOWS
#pragma GCC diagnostic ignored "-Wmaybe-uninitialized"
#endif
bool
myRegression::Recalc()
{
	char **ranges = 0L;
	int nyr, i1, j1, npts, nx, cnt;
	long i, j, k, l, n;
	bool bRet = false;
	AccRange *rX = 0L, *rY = 0L, **rrY = 0L;
	Function *curr_func;
	char *cmdxy = 0L;
	double x, y, sx, sy, mx, my, sxy, sxx, syy;
	double dx, dy, a, b, xmin, xmax, meany, stdev, sterr;
	double *tmp_data = 0L;
	double t, df, sdy;

	if (!xRange || !yRange) return false;
	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	ranges = split(yRange, '|', &nyr);
	rX = new AccRange(xRange);		npts = (nx = rX->CountItems()) * nyr;
	sx = sy = 0.0;
	//do linear regression
	for (i1 = cnt = 0, nPoints = 0; i1 < nyr; i1++) {
		rY = new AccRange(ranges[i1]);
		rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
		for (n = 0; n < nx; n++) {
			if (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
				sx += x;		sy += y;	cnt++;	nPoints++;
				CheckBounds(x, y);
				}
			}
		delete rY;		rY = 0L;
		}
	if (!cnt) return false;
	mx = sx / ((double)nPoints);	my = sy / ((double)nPoints);
	sxy = sxx = syy = 0.0;
	xmax = -HUGE_VAL;		xmin = HUGE_VAL;
	if (type & 0x01) {								//do regression
		for (i1 = cnt = 0; i1 < nyr; i1++) {
			rY = new AccRange(ranges[i1]);
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			for (n = 0; n < nx; n++) {
				if (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
					if (x < xmin) xmin = x;
					if (x > xmax) xmax = x;
					dx = mx - x;	dy = my - y;
					sxx += (dx*dx);	syy += (dy*dy);	sxy += (dx*dy);
					cnt++;
					}
				}
			delete rY;			rY = 0L;
			}
		b = sxy / sxx;			a = my - (sxy / sxx) * mx;
		delete rX;				rX = 0L;
#ifdef USE_WIN_SECURE
		sprintf_s(TmpTxt, TMP_TXT_SIZE, "y = %g + %g * x", a, b);
#else
		sprintf(TmpTxt, "y = %g + %g * x", a, b);
#endif
		cmdxy = rlp_strdup(TmpTxt);
		if (!func) func = new Function(this, data, (char*)"Regression Line");
		curr_func = (Function*)func;
		curr_func->Command(CMD_SETFUNC, cmdxy, 0L);
		curr_func->SetSize(SIZE_MIN_X, xmin);		curr_func->SetSize(SIZE_MAX_X, xmax);
		curr_func->SetSize(SIZE_XSTEP, (xmax - xmin) / 20.0);
		curr_func->Command(CMD_SET_LINE, &PrevLine, 0L);
		func = curr_func;
		if (!reg_res) {
			x = (parent->GetSize(SIZE_GRECT_LEFT) + (parent->GetSize(SIZE_DRECT_LEFT) + parent->GetSize(SIZE_DRECT_RIGHT)) / 2.0);
			y = parent->GetSize(SIZE_GRECT_TOP) + parent->GetSize(SIZE_DRECT_TOP) + defs.GetSize(SIZE_TEXT) * 2.0;
			reg_res = new Label(this, data, x, y, 0L, 0L, 0L);
			reg_res->SetSize(SIZE_TEXT, defs.GetSize(SIZE_TEXT) *1.2);
			reg_res->moveable = 1;
			}
		reg_res->Command(CMD_SETTEXT, cmdxy, 0L);
		free(cmdxy);				cmdxy = 0L;
		}
	if ((type & 0x70) == 0x10) {					//create a symbol for each value pair
		if (!rX) rX = new AccRange(xRange);
		npts = (nx = rX->CountItems()) * (nyr + 1);
		Symbols = (Symbol**)calloc(npts + 2, sizeof(Symbol*));
		nPoints = 0;
		for (i1 = j1 = 0; i1 <= nyr; i1++) {
			rY = new AccRange(ranges[i1]);
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			if (rY->CountItems()) for (n = 0; n < nx; n++) {
				if (rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
					CheckBounds(x, y);
					Symbols[nPoints] = new Symbol(this, data, x, y, SYM_CIRCLE, i, j, k, l);
					Symbols[nPoints]->Command(CMD_PREVSYM, PrevSym, 0L);
					Symbols[nPoints]->name = rlp_strdup((char*)"Values");
					nPoints++;
					}
				}
			if (rY) delete rY;
			rY = 0L;
			}
		if (rX) delete rX;
		rX = 0L;
		}
	else if (type & 0x70) {					//sample statistics for each x
		rX = new AccRange(xRange);		npts = (nx = rX->CountItems()) * (nyr + 1);
		rrY = (AccRange**)calloc(nyr + 1, sizeof(AccRange*));
		for (i = 0; i < nyr; i++) {
			rrY[i] = new AccRange(ranges[i]);
			rrY[i]->GetFirst(&k, &l);
			}
		Symbols = (Symbol**)calloc(nx + 2, sizeof(Symbol*));
		Errors = (ErrorBar**)calloc(nx + 2, sizeof(ErrorBar*));
		tmp_data = (double*)calloc(nx*nyr + 2, sizeof(double));
		rX->GetFirst(&i, &j);
		for (i1 = 0, nPoints = 0; i1 < nx; i1++){
			if (rX->GetNext(&i, &j) && data->GetValue(j, i, &x)) {
				for (j = cnt = 0; j < nyr; j++){
					if (rrY[j]->GetNext(&k, &l) && data->GetValue(l, k, &y)) {
						CheckBounds(x, y);
						tmp_data[cnt++] = y;
						}
					}
				if (cnt) {
					stdev = sqrt((d_variance(cnt, tmp_data, &meany)));
					sterr = stdev / sqrt((double)cnt);
					Symbols[nPoints] = new Symbol(this, data, x, meany, SYM_CIRCLE);
					Symbols[nPoints]->Command(CMD_PREVSYM, PrevSym, 0L);
					Symbols[nPoints]->name = rlp_strdup((char*)"Means");
					nPoints++;
					if ((type & 0x70) == 0x20 && cnt > 1 && Errors) {
						Errors[nErrs] = new ErrorBar(this, data, x, meany, stdev, 0);
						Errors[nErrs]->name = rlp_strdup((char*)"Std. Deviation");
						Errors[nErrs]->Command(CMD_PREVSYM, PrevErr, 0L);
						CheckBounds(x, meany + stdev);			CheckBounds(x, meany - stdev);
						nErrs++;
						}
					if ((type & 0x70) == 0x30 && cnt > 1 && Errors) {
						Errors[nErrs] = new ErrorBar(this, data, x, meany, sterr, 0);
						Errors[nErrs]->name = rlp_strdup((char*)"Std. Error");
						Errors[nErrs]->Command(CMD_PREVSYM, PrevErr, 0L);
						CheckBounds(x, meany + sterr);			CheckBounds(x, meany - sterr);
						nErrs++;
						}
					}
				}
			else {						//invalid reference for x
				for (j = cnt = 0; j < nyr; j++)	rrY[j]->GetNext(&k, &l);
				}
			}
		if (tmp_data)free(tmp_data);
		tmp_data = 0L;
		if (!nErrs && Errors) {
			free(Errors);					Errors = 0L;
			}
		delete rX;							rX = 0L;
		}
	if ((type & 0x81) == 0x81 && nyr){
		if (!rrY) {
			rrY = (AccRange**)calloc(nyr + 1, sizeof(AccRange*));
			for (i = 0; i < nyr; i++) {
				rrY[i] = new AccRange(ranges[i]);			rrY[i]->GetFirst(&k, &l);
				}
			}
		df = nPoints - 2;
		t = distinv(t_dist, df, 1.0, (100.0 - ci) / 100.0, 2.0);
		rX = new AccRange(xRange);		npts = (nx = rX->CountItems()) * (nyr + 1);
		rX->GetFirst(&i, &j);
		for (i = 0; i < nyr; i++) if (rrY[i]) rrY[i]->GetFirst(&k, &l);
		for (i1 = 0, sdy = 0.0; i1 < nx; i1++){
			if (rX->GetNext(&i, &j) && data->GetValue(j, i, &x)) {
				for (j = 0; j < nyr; j++){
					if (rrY[j]->GetNext(&k, &l) && data->GetValue(l, k, &y)) {
						dy = y - (a + x * b);	sdy += (dy * dy);
						}
					}
				}
			else {						//invalid reference for x
				for (j = cnt = 0; j < nyr; j++)	rrY[j]->GetNext(&k, &l);
				}
			}
		sdy = sdy / df;
		delete rX;			rX = 0L;
#ifdef USE_WIN_SECURE
		sprintf_s(TmpTxt, TMP_TXT_SIZE - i, "y=%g+x*%g;\nts=sqrt((((x-(%g))^2)/%g+%g)*%g);\ny=y+ts*%g\n",
			a, b, mx, sxx, (1.0 / (double)nPoints), sdy, t);
		cmdxy = rlp_strdup(TmpTxt);
		sprintf_s(TmpTxt, 50, "%g%% Conf. Intervall", ci);
#else
		sprintf(TmpTxt, "y=%g+x*%g;\nts=sqrt((((x-(%g))^2)/%g+%g)*%g);\ny=y+ts*%g\n",
			a, b, mx, sxx, (1.0 / (double)nPoints), sdy, t);
		cmdxy = rlp_strdup(TmpTxt);
		sprintf(TmpTxt, "%g%% Conf. Intervall", ci);
#endif
		if (!ci_line1) ci_line1 = new Function(this, data, TmpTxt);
		curr_func = (Function*)ci_line1;						curr_func->Command(CMD_SETFUNC, cmdxy, 0L);
		curr_func->SetSize(SIZE_MIN_X, xmin);					curr_func->SetSize(SIZE_MAX_X, xmax);
		curr_func->SetSize(SIZE_XSTEP, (xmax - xmin) / 20.0);
		curr_func->Command(CMD_SET_LINE, &PrevCiLine, 0L);		ci_line1 = curr_func;
		free(cmdxy);			cmdxy = 0L;
#ifdef USE_WIN_SECURE
		sprintf_s(TmpTxt, TMP_TXT_SIZE - i, "y=%g+x*%g;\nts=sqrt((((x-(%g))^2)/%g+%g)*%g);\ny=y-ts*%g\n",
			a, b, mx, sxx, (1.0 / (double)nPoints), sdy, t);
		cmdxy = rlp_strdup(TmpTxt);
		sprintf_s(TmpTxt, 50, "%g%% Conf. Intervall", ci);
#else
		sprintf(TmpTxt, "y=%g+x*%g;\nts=sqrt((((x-(%g))^2)/%g+%g)*%g);\ny=y-ts*%g\n",
			a, b, mx, sxx, (1.0 / (double)nPoints), sdy, t);
		cmdxy = rlp_strdup(TmpTxt);
		sprintf(TmpTxt, "%g%% Conf. Intervall", ci);
#endif
		if (!ci_line2) ci_line2 = new Function(this, data, TmpTxt);
		curr_func = (Function*)ci_line2;						curr_func->Command(CMD_SETFUNC, cmdxy, 0L);
		curr_func->SetSize(SIZE_MIN_X, xmin);					curr_func->SetSize(SIZE_MAX_X, xmax);
		curr_func->SetSize(SIZE_XSTEP, (xmax - xmin) / 20.0);
		curr_func->Command(CMD_SET_LINE, &PrevCiLine, 0L);		ci_line2 = curr_func;
		free(cmdxy);			cmdxy = 0L;
		}
	if (ranges) {
		for (i = 0; i < nyr; i++) if (ranges[i]) free(ranges[i]);
		free(ranges);						ranges = 0L;
		}
	if (rrY) {
		for (i = 0; i < nyr; i++) if (rrY[i]) delete rrY[i];
		free(rrY);							rrY = 0L;
		}
	if (tmp_data)free(tmp_data);
	tmp_data = 0L;							if (rX) delete rX;
	rX = 0L;								if (rY) delete rY;
	rY = 0L;
	bRet = (Bounds.Xmin < Bounds.Xmax && Bounds.Ymin < Bounds.Ymax);
	if (bRet) Command(CMD_AUTOSCALE, 0L, 0L);
	return bRet;
}
#ifndef _WINDOWS
#pragma GCC diagnostic warning "-Wmaybe-uninitialized"
#endif

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//myBarPlot is a Plot-Class: Bar chart with many y but a single x-range
myBarPlot::myBarPlot(GraphObj *par, DataObj *d)
	:PlotScatt(par, d)
{
	FileIO(INIT_VARS);
	bar_dx = 0.0;		 Id = GO_MYBARS;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
	PrevSym = new Symbol(0L, 0L, 0.0, 0.0, SYM_CIRCLE);
	PrevErr = new ErrorBar(0L, 0L, 0.0, 0.0, 0.0, ERRBAR_VSYM);
	PrevBar = new Bar(0L, data, 0.0, 0.0, BAR_RELWIDTH | BAR_VERTB);
	memcpy(&PrevLine, defs.GetLine(), sizeof(LineDEF));
}

myBarPlot::myBarPlot(int src) :PlotScatt(0L, 0L)
{
	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_MYBARS;
}

myBarPlot::~myBarPlot()
{
	if (PrevSym) delete PrevSym;
	PrevSym = 0L;							if (PrevErr) delete PrevErr;
	PrevErr = 0L;							if (PrevBar) delete PrevBar;
	PrevBar = 0L;							if (ErrPg) delete ErrPg;
	ErrPg = 0L;								if (xRange) free(xRange);
	xRange = 0L;							if (yRange) free(yRange);
	yRange = 0L;
}

double
myBarPlot::GetSize(int select)
{
	return PlotScatt::GetSize(select);
}

bool
myBarPlot::SetSize(int select, double value)
{
	return PlotScatt::SetSize(select, value);
}

bool
myBarPlot::SetColor(int select, DWORD col)
{
	return PlotScatt::SetColor(select, col);
}

bool
myBarPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	DWORD flags = 0;

	switch (cmd){
	case CMD_MOUSE_EVENT:
		if(PlotScatt::Command(cmd, tmpl, o))return true;
		if (ErrPg && ErrPg->Command(cmd, tmpl, o)) return true;
		return false;
	case CMD_AUTOSCALE:
		if (ErrPg && ErrPg->Command(cmd, tmpl, o)) return true;
		return PlotScatt::Command(cmd, tmpl, o);
	case CMD_SET_DATAOBJ:
		PlotScatt::Command(cmd, tmpl, o);
		Id = GO_MYBARS;
		data = (DataObj *)tmpl;
		if (PrevSym) PrevSym->Command(cmd, 0L, o);
		if (PrevBar) PrevBar->Command(cmd, 0L, o);
		if (PrevErr) PrevErr->Command(cmd, tmpl, o);
		if (ErrPg) ErrPg->Command(cmd, tmpl, o);
		return true;
	case CMD_SCALE:
		PlotScatt::Command(cmd, tmpl, o);
		if (PrevSym) PrevSym->Command(cmd, tmpl, o);
		if (PrevErr) PrevErr->Command(cmd, tmpl, o);
		if (PrevBar) PrevBar->Command(cmd, 0L, o);
		PrevLine.width *= ((scaleINFO*)tmpl)->sy.fy;		PrevLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		break;
		return true;
	case CMD_UPDATE:
		if (Symbols && nPoints) {
			Undo.DropListGO(this, (GraphObj***)&Symbols, &nPoints, flags);
			flags |= UNDO_CONTINUE;
			}
		if (Errors && nErrs){
			Undo.DropListGO(this, (GraphObj***)&Errors, &nErrs, flags);
			flags |= UNDO_CONTINUE;
			}
		if (Bars && nBars){
			Undo.DropListGO(this, (GraphObj***)&Bars, &nBars, flags);
			flags |= UNDO_CONTINUE;
			}
		Recalc();
		break;
	case CMD_DELOBJ:
		if (ErrPg && parent && (void*)ErrPg == tmpl) {
			Undo.DeleteGO((GraphObj**)&ErrPg, 0L, o);
			parent->Command(CMD_REDRAW, tmpl, o);
			return true;
			}
		return PlotScatt::Command(cmd, tmpl, o);
	case CMD_LEGEND:
		PlotScatt::Command(cmd, tmpl, o);
		if (ErrPg) ErrPg->Command(cmd, tmpl, o);
		return true;
	default:
		return PlotScatt::Command(cmd, tmpl, o);
		}
	return false;
}

bool
myBarPlot::Recalc()
{
	char **ranges = 0L;
	int nyr, cnt;
	long i, j, k, l, i1, nli, nx;
	AccRange *rX = 0L, **rrY = 0L;
	anyResult xRes;
	double x, y;
	double meany, stdev, sterr;
	double *tmp_data = 0L;
	lfPOINT *line_data = 0L;
	lfPOINT *loerr = 0L, *hierr = 0L;
	int nerr;

	if (!xRange || !yRange) return false;
	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	ranges = split(yRange, '|', &nyr);
	rX = new AccRange(xRange);					nx = rX->CountItems();
	if (type & 0x10) PrevErr->type |= 0x300;			//if there are bars allso tell error bars
	else PrevErr->type &= (~0x300);
	if (true) {					//sample statistics for each x
		rrY = (AccRange**)calloc(nyr + 1, sizeof(AccRange*));
		for (i = 0; i < nyr; i++) {
			rrY[i] = new AccRange(ranges[i]);			rrY[i]->GetFirst(&k, &l);
			}
		Symbols = (Symbol**)calloc(nx + 2, sizeof(Symbol*));
		Bars = (Bar**)calloc(nx + 2, sizeof(Symbol*));
		Errors = (ErrorBar**)calloc(nx + 2, sizeof(ErrorBar*));
		tmp_data = (double*)calloc(nx*nyr + 2, sizeof(double));
		line_data = (lfPOINT*)calloc(nx*nyr + 2, sizeof(lfPOINT));
		if ((type & 0x103) == 0x101 || (type & 0x102) == 0x102){
			loerr = (lfPOINT*)calloc((nx+1)<<1, sizeof(lfPOINT));
			hierr = (lfPOINT*)calloc((nx+1)<<1, sizeof(lfPOINT));
			nerr = 0;
			}
		rX->GetFirst(&i, &j);		nli = 0;
		for (i1 = 0, nPoints = 0; i1 < nx; i1++){
			if (rX->GetNext(&i, &j) && data->GetResult(&xRes, j, i)) {
				if (x_tv) {
					if (xRes.type == ET_TEXT) x = x_tv->GetValue(xRes.text);
					else x = xRes.value;
					}
				else x = xRes.value;
				for (j = cnt = 0; j < nyr; j++){
					if (rrY[j]->GetNext(&k, &l) && data->GetValue(l, k, &y)) {
						CheckBounds(x, y);		tmp_data[cnt++] = y;
						}
					}
				if (cnt) {
					stdev = sqrt((d_variance(cnt, tmp_data, &meany)));
					sterr = stdev / sqrt((double)cnt);
					line_data[nli].fx = x;		line_data[nli++].fy = meany;
					if (type & 0x10) {
						Bars[nBars] = new Bar(this, data, x, meany, BAR_VERTB | BAR_RELWIDTH);
						Bars[nBars]->Command(CMD_PREVSYM, PrevBar, 0L);
						Bars[nBars]->name = rlp_strdup((char*)"Means");
						nBars++;
						}
					if (type & 0x20) {
						Symbols[nPoints] = new Symbol(this, data, x, meany, SYM_CIRCLE);
						Symbols[nPoints]->Command(CMD_PREVSYM, PrevSym, 0L);
						Symbols[nPoints]->name = rlp_strdup((char*)"Means");
						nPoints++;
						}
					if ((type & 0x43) == 0x41 && cnt > 1 && Errors) {
						if (loerr && hierr){
							loerr[nerr].fx = hierr[nerr].fx = x;
							loerr[nerr].fy = meany - stdev;
							hierr[nerr++].fy = meany + stdev;
							}
						Errors[nErrs] = new ErrorBar(this, data, x, meany, stdev, 0);
						Errors[nErrs]->name = rlp_strdup((char*)"Std. Deviation");
						Errors[nErrs]->Command(CMD_PREVSYM, PrevErr, 0L);
						CheckBounds(x, meany + stdev);			CheckBounds(x, meany - stdev);
						nErrs++;
						}
					if ((type & 0x43) == 0x42 && cnt > 1 && Errors) {
						Errors[nErrs] = new ErrorBar(this, data, x, meany, sterr, 0);
						Errors[nErrs]->name = rlp_strdup((char*)"Std. Error");
						Errors[nErrs]->Command(CMD_PREVSYM, PrevErr, 0L);
						CheckBounds(x, meany + sterr);			CheckBounds(x, meany - sterr);
						nErrs++;
						}
					if ((type & 0x103) == 0x101 && cnt > 1 ) {
						if (loerr && hierr){
							loerr[nerr].fx = hierr[nerr].fx = x;
							loerr[nerr].fy = meany - stdev;						hierr[nerr++].fy = meany + stdev;
							}
						}
					if ((type & 0x103) == 0x102 && cnt > 1 ) {
						if (loerr && hierr){
							loerr[nerr].fx = hierr[nerr].fx = x;
							loerr[nerr].fy = meany - sterr;						hierr[nerr++].fy = meany + sterr;
							}
						}
					}
				}
			else {						//invalid reference for x
				for (j = cnt = 0; j < nyr; j++)	rrY[j]->GetNext(&k, &l);
				}
			}
		if (nli && (type & 0x80)) {
			if (TheLine) {
				TheLine->LineData(line_data, nli);
				line_data = 0L;				nli = 0;
				}
			else {
				TheLine = new DataLine(this, data, line_data, nli, (char*)"Mean Line");
				free(line_data);				line_data = 0L;				nli = 0;
				}
			}
		}
	if (loerr && hierr) {
		if (!ErrPg) ErrPg = new ErrorPolygon(this, data, nerr, loerr, nerr, hierr);
		else ErrPg->SetValues(nerr, loerr, nerr, hierr);
		if (ErrPg->name) free(ErrPg->name);
		ErrPg->name = 0L;
		if ((type & 0x01) == 0x01) ErrPg->name = rlp_strdup((char*)"S.D. Polygon");
		else if ((type & 0x02) == 0x02) ErrPg->name = rlp_strdup((char*)"S.E. Polygon");
		free(loerr);						loerr = 0L;
		free(hierr);						hierr = 0L;
		}
	if (!nPoints && Symbols){
		free(Symbols);						Symbols = 0L;
		}
	if (!nBars && Bars){
		free(Bars);							Bars = 0L;
		}
	if (!nErrs && Errors) {
		free(Errors);						Errors = 0L;
		}
	if (ranges) {
		for (i = 0; i < nyr; i++) if (ranges[i]) free(ranges[i]);
		free(ranges);						ranges = 0L;
		}
	if (rrY) {
		for (i = 0; i < nyr; i++) if (rrY[i]) delete rrY[i];
		free(rrY);							rrY = 0L;
		}
	if (tmp_data) free(tmp_data);
	tmp_data = 0L;						if (line_data) free(line_data);
	line_data = 0L;						if (rX) delete rX;	
	rX = 0L;
	return (Bounds.Xmin < Bounds.Xmax && Bounds.Ymin < Bounds.Ymax);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// BubblePlot is a Plot-Class
BubblePlot::BubblePlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_BUBBLEPLOT;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

BubblePlot::BubblePlot(int src):Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		BubbleFill.hatch = &BubbleFillLine;
		if(Bubbles)for(i = 0; i< nPoints; i++) {
			if(Bubbles[i])Bubbles[i]->parent = this;
			}
		}
	Id = GO_BUBBLEPLOT;
}

BubblePlot::~BubblePlot()
{
	int i;

	if(Bubbles) {
		for(i = 0; i < nPoints; i++) if(Bubbles[i]) DeleteGO(Bubbles[i]);
		free (Bubbles);
		}
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

DWORD
BubblePlot::GetColor(int select)
{
	switch(select) {
	case COL_BUBBLE_FILL:			return BubbleFill.color;
	case COL_BUBBLE_LINE:			return BubbleLine.color;
	case COL_BUBBLE_FILLLINE:	   	return BubbleFillLine.color;
	default:
		return Plot::GetColor(select);
	}
}

void
BubblePlot::DoPlot(anyOutput *o)
{
	int i;

	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) {
		ApplyAxes(o);
		if(Bubbles) for(i = 0; i < nPoints; i++) 
			if(Bubbles[i]) Bubbles[i]->DoPlot(o);
		parent->Command(CMD_AXIS, 0L, o);
		}
	else {
		if(Bubbles) for(i = 0; i < nPoints; i++) 
			if(Bubbles[i]) Bubbles[i]->DoPlot(o);
		}
	dirty = false;
}

bool
BubblePlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	static MouseEvent *mev;
	GraphObj **tmpPlots;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Bubbles && !CurrGO) for(i = nPoints-1; i >=0; i--)
				if(Bubbles[i]) if(Bubbles[i]->Command(cmd, tmpl, o))break;
			break;
			}
		break;
	case CMD_REPL_GO:
		if((tmpPlots = (GraphObj **)tmpl) && tmpPlots[0] && tmpPlots[1] && Bubbles) {
			for(i = 0; i < nPoints; i++) if(Bubbles[i] && Bubbles[i] == tmpPlots[0]) { 
				return ReplaceGO((GraphObj**)&Bubbles[i], tmpPlots);
				}
			}
		return false;
	case CMD_LEGEND:
		if(((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(Bubbles) for (i = 0; i < nPoints; i++)
			if(Bubbles[i]) Bubbles[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SCALE:
		if(!tmpl) return false;
		BubbleLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleFillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleFillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		BubbleFill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		if(Bubbles) for(i = 0; i < nPoints; i++)
			if(Bubbles[i]) Bubbles[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SETSCROLL:		case CMD_REDRAW:		case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_BUBBLE_ATTRIB:		case CMD_BUBBLE_TYPE:		case CMD_BUBBLE_FILL:
	case CMD_BUBBLE_LINE:		case CMD_SET_DATAOBJ:		case CMD_UPDATE:
	case CMD_AUTOSCALE:
		if (cmd == CMD_SET_DATAOBJ){
			Id = GO_BUBBLEPLOT;
			data = (DataObj *)tmpl;
			}
		if (cmd == CMD_UPDATE && Bubbles) {
			SavVarObs((GraphObj **)Bubbles, nPoints, UNDO_CONTINUE);
			}
		if (cmd == CMD_AUTOSCALE && Bubbles) {
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			}
		if (Bubbles) for (i = 0; i < nPoints; i++)
			if(Bubbles[i]) Bubbles[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(Bubbles && parent) for(i = 0; i < nPoints; i++) {
			o->HideMark();
			if(Bubbles[i] && tmpl == (void *)Bubbles[i]) {
				Undo.DeleteGO((GraphObj**)(&Bubbles[i]), 0L, o);
				parent->Command(CMD_REDRAW, 0L, o);
				return true;
				}
			}
		break;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Bubbles, nPoints, 0L);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// PolarPlot is a Plot-Class
PolarPlot::PolarPlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_POLARPLOT;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

PolarPlot::PolarPlot(int src):Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		Fill.hatch = &FillLine;
		//now set parent in all children
		if(Plots) 
			for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->parent = this;
		if(Axes) 
			for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->parent = this;
		}
}

PolarPlot::~PolarPlot()
{
	long i;

	if(Plots){
		for(i = 0; i < nPlots; i++) if(Plots[i]) DeleteGO(Plots[i]);
		free(Plots);		Plots = 0L;
		}
	if(Axes){
		for(i = 0; i < nAxes; i++) if(Axes[i]) DeleteGO(Axes[i]);
		free(Axes);			Axes = 0L;
		}
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

double
PolarPlot::GetSize(int select)
{
	switch(select) {
	case SIZE_BOUNDS_XMIN:		return Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		return Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		return Bounds.Ymin;
	case SIZE_BOUNDS_YMAX:		return Bounds.Ymax;
	case SIZE_BOUNDS_LEFT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->min:0.0;
	case SIZE_BOUNDS_RIGHT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_TOP:		return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_BOTTOM:	return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->min:0.0;
	case SIZE_YAXISX:
		if(!CurrDisp) return 0.0;
		if((((Axis*)Axes[1])->GetAxis())->flags & AXIS_X_DATA) 
			return CurrDisp->fx2fix((((Axis*)Axes[1])->GetAxis())->loc[0].fx);
		else return CurrDisp->co2fix((((Axis*)Axes[1])->GetAxis())->loc[0].fx);
	case SIZE_XAXISY:
		if(!CurrDisp) return 0.0;
		if((((Axis*)Axes[0])->GetAxis())->flags & AXIS_Y_DATA) 
			return CurrDisp->fy2fiy((((Axis*)Axes[0])->GetAxis())->loc[0].fy);
		else return CurrDisp->co2fiy((((Axis*)Axes[0])->GetAxis())->loc[0].fy);
	case SIZE_XCENTER:			return (((Axis*)Axes[0])->GetAxis())->Center.fx;
	case SIZE_YCENTER:			return (((Axis*)Axes[0])->GetAxis())->Center.fy;
	default:
		if(parent) return parent->GetSize(select);
		}
	return DefSize(select);
}

void
PolarPlot::DoPlot(anyOutput *o)
{
	long i;

	if(o) CurrDisp = o;
	else return;
	if(!parent) return;
	CurrRect.Xmin = CurrRect.Xmax = (((Axis*)Axes[1])->GetAxis())->Center.fx + 
		parent->GetSize(SIZE_GRECT_LEFT);
	CurrRect.Xmin -= (((Axis*)Axes[0])->GetAxis())->Radius;	
	CurrRect.Xmax += (((Axis*)Axes[0])->GetAxis())->Radius;
	CurrRect.Ymin = CurrRect.Ymax = (((Axis*)Axes[0])->GetAxis())->Center.fy +
		parent->GetSize(SIZE_GRECT_TOP);
	CurrRect.Ymin -= (((Axis*)Axes[0])->GetAxis())->Radius;
	CurrRect.Ymax += (((Axis*)Axes[0])->GetAxis())->Radius;
	(((Axis*)Axes[0])->GetAxis())->Start = ((((Axis*)Axes[0])->GetAxis())->flags & AXIS_INVERT) ? -offs : offs;
	o->SetRect(CurrRect, defs.dUnits, ((Axis*)Axes[0])->GetAxis(), ((Axis*)Axes[1])->GetAxis());
	o->SetFill(&Fill);
	if(Axes) for(i = 0; i < nAxes; i++) {
		if(i == 1) {
			if(!(type & 0x01) && Axes[i]) Axes[i]->DoPlot(o);
			}
		else if(Axes[i]) Axes[i]->DoPlot(o);
		}
	if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) {
		if(Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH) {
			if(((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(o);
			}
		else Plots[i]->DoPlot(o);
		}
	rDims.left = o->co2ix(CurrRect.Xmin);	rDims.right = o->co2ix(CurrRect.Xmax);
	rDims.top = o->co2iy(CurrRect.Ymin);	rDims.bottom = o->co2iy(CurrRect.Ymax);
	if(parent) parent->Command(CMD_AXIS, 0L, o);
}

void
PolarPlot::DoMark(anyOutput *o, bool mark)
{
	FillDEF fill = { 0, 0xb0808080, 1.0, 0L, 0x00ffffff };
	LineDEF line = { defs.GetSize(SIZE_HAIRLINE), defs.GetSize(SIZE_CELLTEXT), 0xff808080, 0L };

	if (!o) return;
	if (parent && parent->Id == GO_TERNARYXYZ) {
		parent->DoMark(o, mark);				return;
	}
	if (mark) {
		o->SetLine(&line);			o->SetFill(&fill);
		o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		defs.UpdRect(o, rDims.left, rDims.top, rDims.right, rDims.bottom);
	}
	else {
		Command(CMD_REDRAW, 0L, o);
	}
}

bool
PolarPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;
	long i;
	AxisDEF *ad0, *ad1;
	double tmp;

	switch (cmd) {
	case CMD_CONFIG:
		Config();
		return true;
	case CMD_SCALE:
		FillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		FillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_LEGEND:	case CMD_SET_DATAOBJ:		case CMD_UPDATE:
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_POLARPLOT;
			data = (DataObj *)tmpl;
			}
		if (cmd == CMD_SET_DATAOBJ || cmd == CMD_UPDATE) {
			if (Axes) for (i = 0; i < nAxes; i++) {
				if (Axes[i]) Axes[i]->Command(cmd, tmpl, o);
				}
			}
		if(Plots) for(i = 0; i < nPlots; i++){
			if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			}
		return false;
	case CMD_OBJTREE:
		if(!tmpl) return false;
		for (i = 0; i < nAxes; i++) if (Axes[i]) {
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Axes[i], 0L);
			}
		if (Plots) for (i = 0; i < nPlots; i++) if (Plots[i])
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Plots[i], 0L);
		return true;
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		if(o) switch(((MouseEvent*)tmpl)->Action) {
		case MOUSE_LBUP:
			if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i])
				if(Axes[i]->Command(cmd, tmpl, o))return true;
			if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) 
				Plots[i]->Command(cmd, tmpl, o);
			if(!CurrGO && IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)){
				CurrGO = this;
				o->ShowMark(this, MRK_GODRAW);
				return true;
				}
			break;
			}
		return false;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i] && Axes[i] == tmpPlots[0]){
			return ReplaceGO(&Axes[i], tmpPlots);
			}
		break;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_DELOBJ_CONT:	case CMD_DELOBJ:
		if(Plots && nPlots) for(i = 0; i < nPlots; i++) if(tmpl == (void*)Plots[i]) {
			Undo.DeleteGO((GraphObj**)(&Plots[i]), cmd == CMD_DELOBJ_CONT ? UNDO_CONTINUE : 0L, o);
			if(parent)parent->Command(CMD_REDRAW, NULL, o);
			return true;
			}
		if(Axes && nAxes > 1 && (tmpl == (void*)Axes[0] || tmpl == (void*)Axes[1]))
			InfoBox((char*)"Axes required for scaling\ncannot be deleted.");
		break;
	case CMD_AXIS:		//axis changed: reconstruct corresponding axis
		if(Axes && nAxes >1 && tmpl && Axes[0] && Axes[1]) {
			ad0 = ((Axis*)Axes[0])->GetAxis();		ad1 = ((Axis*)Axes[1])->GetAxis();
			if(tmpl == Axes[0]) {
				CheckNewFloat(&ad1->loc[1].fy, ad1->loc[1].fy, ad0->Center.fy, this, UNDO_CONTINUE);
				tmp = ad1->loc[1].fy - ad0->Radius;
				CheckNewFloat(&ad1->loc[0].fy, ad1->loc[0].fy, tmp, this, UNDO_CONTINUE);
				}
			if(tmpl == Axes[1]) {
				CheckNewFloat(&ad0->Center.fy, ad0->Center.fy, ad1->loc[1].fy, this, UNDO_CONTINUE);
				tmp = fabs(ad1->loc[1].fy - ad1->loc[0].fy);
				CheckNewFloat(&ad0->Radius, ad0->Radius, tmp, this, UNDO_CONTINUE);
				}
			}
		break;	
		}
	return false;
}

void *
PolarPlot::ObjThere(int x, int y)
{
	long i;
	void *ptr;

	//do all axes
	for (i = 0; Axes && i < nAxes; i++){
		if (Axes[i] && (ptr = Axes[i]->ObjThere(x, y))) return ptr;;
	}
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// TernaryPlot (triangle plot) is a Plot-Class
TernaryPlot::TernaryPlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_TERNARY;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

TernaryPlot::TernaryPlot(int src):Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	Id = GO_TERNARY;
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		Fill.hatch = &FillLine;
		//now set parent in all children
		if(Plots) 
			for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->parent = this;
		if(Axes) 
			for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->parent = this;
		}
}

TernaryPlot::~TernaryPlot()
{
	long i;

	if(Plots){
		for(i = 0; i < nPlots; i++) if(Plots[i]) DeleteGO(Plots[i]);
		free(Plots);
		Plots = 0L;
		}
	if(Axes){
		for(i = 0; i < nAxes; i++) if(Axes[i]) DeleteGO(Axes[i]);
		free(Axes);	
		Axes = 0L;
		}
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

double
TernaryPlot::GetSize(int select)
{
	switch(select) {
	case SIZE_BOUNDS_XMIN:		return Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		return Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		return Bounds.Ymin;
	case SIZE_BOUNDS_YMAX:		return Bounds.Ymax;
	case SIZE_BOUNDS_LEFT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->min:0.0;
	case SIZE_BOUNDS_RIGHT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_TOP:		return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_BOTTOM:	return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->min:0.0;
//	case SIZE_ANG_DISP:			return ang_disp;
	default:
		if(parent) return parent->GetSize(select);
		}
	return DefSize(select);
}

void
TernaryPlot::DoPlot(anyOutput *o)
{
	long i;
	AxisDEF *ax;
	double dx, dy;

	dx = o->co2fix(parent->GetSize(SIZE_GRECT_LEFT));
	dy = o->co2fiy(parent->GetSize(SIZE_GRECT_TOP));
	if(o) CurrDisp = o;
	else return;
	if(!parent || !Axes || !Axes[0] || !Axes[1] || !Axes[2]) return;
	CurrRect.Xmin = CurrRect.Ymin = HUGE_VAL;
	CurrRect.Xmax = CurrRect.Ymax = -HUGE_VAL;
	for(i = 0; i< nAxes; i++) {
		ax = ((Axis*)Axes[i])->GetAxis();
		if(CurrRect.Xmin > ax->loc[0].fx) CurrRect.Xmin = ax->loc[0].fx;
		if(CurrRect.Xmax < ax->loc[0].fx) CurrRect.Xmax = ax->loc[0].fx;
		if(CurrRect.Xmin > ax->loc[1].fx) CurrRect.Xmin = ax->loc[1].fx;
		if(CurrRect.Xmax < ax->loc[1].fx) CurrRect.Xmax = ax->loc[1].fx;
		if(CurrRect.Ymin > ax->loc[0].fy) CurrRect.Ymin = ax->loc[0].fy;
		if(CurrRect.Ymax < ax->loc[0].fy) CurrRect.Ymax = ax->loc[0].fy;
		if(CurrRect.Ymin > ax->loc[1].fy) CurrRect.Ymin = ax->loc[1].fy;
		if(CurrRect.Ymax < ax->loc[1].fy) CurrRect.Ymax = ax->loc[1].fy;
		}
	o->UseAxis(((Axis*)Axes[0])->GetAxis(), Axes[0]->type, dx, dy);
	o->UseAxis(((Axis*)Axes[1])->GetAxis(), Axes[1]->type, dx, dy);
	if(Axes) for(i = 0; i < nAxes; i++) {
		if(Axes[i]) Axes[i]->DoPlot(o);
		}
	if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) {
		if(Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH) {
			if(((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(o);
			}
		else Plots[i]->DoPlot(o);
		}
	rDims.left = o->co2ix(CurrRect.Xmin);	rDims.right = o->co2ix(CurrRect.Xmax);
	rDims.top = o->co2iy(CurrRect.Ymin);	rDims.bottom = o->co2iy(CurrRect.Ymax);
	if(parent) parent->Command(CMD_AXIS, 0L, o);
}

bool
TernaryPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;
	long i;

	switch (cmd) {
	case CMD_CONFIG:
//		Config();
		return true;
	case CMD_SCALE:
		FillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		FillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_LEGEND:		case CMD_SET_DATAOBJ: case CMD_UPDATE:
		if (cmd == CMD_SET_DATAOBJ){
			Id = GO_TERNARY;
			data = (DataObj *)tmpl;
			}
		if (cmd == CMD_SET_DATAOBJ || cmd == CMD_UPDATE) {
			if (Axes) for (i = 0; i < nAxes; i++){
				if (Axes[i]) Axes[i]->Command(cmd, tmpl, o);
				}
			}
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		return false;
	case CMD_OBJTREE:
		if(!tmpl) return false;
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) 
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Plots[i], 0L);
		return true;
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		if(o) switch(((MouseEvent*)tmpl)->Action) {
		case MOUSE_LBUP:
			if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i])
				if(Axes[i]->Command(cmd, tmpl, o))return true;
			if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) 
				Plots[i]->Command(cmd, tmpl, o);
			if(!CurrGO && IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)){
				CurrGO = this;
				o->ShowMark(&rDims, MRK_INVERT);
				return true;
				}
			break;
			}
		return false;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i] && Axes[i] == tmpPlots[0]){
			return ReplaceGO(&Axes[i], tmpPlots);
			}
		break;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_DELOBJ_CONT:	case CMD_DELOBJ:
		if(Plots && nPlots) for(i = 0; i < nPlots; i++) if(tmpl == (void*)Plots[i]) {
			Undo.DeleteGO((GraphObj**)(&Plots[i]), cmd == CMD_DELOBJ_CONT ? UNDO_CONTINUE : 0L, o);
			if(parent)parent->Command(CMD_REDRAW, NULL, o);
			return true;
			}
		//DEBUG:
		InfoBox((char*)"Axes are required for scaling:\nthey cannot be deleted!");
		break;
		}
	return false;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// TernaryPlot (triangle plot) is a Plot-Class
TernaryXYZ::TernaryXYZ(GraphObj *par, DataObj *d):PlotScatt(par, d, (DWORD)NULL)
{
	FileIO(INIT_VARS);
	Id = GO_TERNARYXYZ;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, (anyOutput*)NULL);
}

TernaryXYZ::TernaryXYZ(GraphObj *par, DataObj *d, Symbol **syms, long nsyms):PlotScatt(par, d, (DWORD)NULL)
{
	long i;

	FileIO(INIT_VARS);
	Symbols = syms;			nPoints = nsyms;
	if(Symbols)for (i = 0; i < nsyms; i++){
		if (Symbols[i])Symbols[i]->parent = this;
		}
	PlotScatt::Command(CMD_AUTOSCALE, 0L, 0L);
	Id = GO_TERNARYXYZ;
}

TernaryXYZ::TernaryXYZ(int src):PlotScatt(0L, 0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	Id = GO_TERNARYXYZ;
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		Fill.hatch = &FillLine;
		//now set parent in all children
		if(Plots) for (i = 0; i < nPlots; i++) {
				if (Plots[i]) Plots[i]->parent = this;
				}
		if (Axes) for (i = 0; i < nAxes; i++) {
			if (Axes[i]) Axes[i]->parent = this;
			}
		if (Symbols) for (i = 0; i < nPoints; i++) {
			if (Symbols[i]) Symbols[i]->parent = this;
			}
		if (Arrows) for (i = 0; i < nArrow; i++) {
			if (Arrows[i]) Arrows[i]->parent = this;
			}
		if (TheLine) TheLine->parent = this;
		}
}

TernaryXYZ::~TernaryXYZ()
{
	long i;

	if(Plots){
		for (i = 0; i < nPlots; i++) {
			if (Plots[i]) DeleteGO(Plots[i]);
			}
		free(Plots);		Plots = 0L;
		}
	if(Symbols) {
		for (i = 0; i < nPoints; i++) {
			if (Symbols[i]) DeleteGO(Symbols[i]);
			}
		free(Symbols);			Symbols = 0L;
		}
	if (Arrows) {
		for (i = 0; i < nArrow; i++){
			if (Arrows[i]) DeleteGO(Arrows[i]);
			}
		free(Arrows);			Arrows = 0L;
		}
	if (TheLine) delete TheLine;
	TheLine = 0L;
	if(Axes){
		for (i = 0; i < nAxes; i++) {
			if (Axes[i]) DeleteGO(Axes[i]);
			}
		free(Axes);			Axes = 0L;
		}
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

double
TernaryXYZ::GetSize(int select)
{
	switch(select) {
	case SIZE_BOUNDS_XMIN:		return Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		return Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		return Bounds.Ymin;
	case SIZE_BOUNDS_YMAX:		return Bounds.Ymax;
	case SIZE_BOUNDS_LEFT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->min:0.0;
	case SIZE_BOUNDS_RIGHT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_TOP:		return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_BOTTOM:	return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->min:0.0;
	case SIZE_ANG_DISP:			return ang_disp;
	case SIZE_SYMBOL:			return defs.GetSize(select);
	default:
		if(parent) return parent->GetSize(select);
		}
	return DefSize(select);
}

void
TernaryXYZ::DoPlot(anyOutput *o)
{
	int i;
	AxisDEF *ax;
	double dx, dy;

	dx = o->co2fix(parent->GetSize(SIZE_GRECT_LEFT));
	dy = o->co2fiy(parent->GetSize(SIZE_GRECT_TOP));
	if (o) CurrDisp = o;
	else return;
	if(!parent) return;
	if(parent->Id != GO_TERNARYXYZ && (!Axes || !Axes[0] || !Axes[1] || !Axes[2])) return;
	CurrRect.Xmin = CurrRect.Ymin = HUGE_VAL;		CurrRect.Xmax = CurrRect.Ymax = -HUGE_VAL;
	for(i = 0; i< nAxes; i++) {
		ax = ((Axis*)Axes[i])->GetAxis();
		if(i == 1) {
			ang_disp = (ax->loc[1].fx - ax->loc[0].fx)/(ax->loc[1].fy - ax->loc[0].fy);
			}
		if(CurrRect.Xmin > ax->loc[0].fx) CurrRect.Xmin = ax->loc[0].fx;
		if(CurrRect.Xmax < ax->loc[0].fx) CurrRect.Xmax = ax->loc[0].fx;
		if(CurrRect.Xmin > ax->loc[1].fx) CurrRect.Xmin = ax->loc[1].fx;
		if(CurrRect.Xmax < ax->loc[1].fx) CurrRect.Xmax = ax->loc[1].fx;
		if(CurrRect.Ymin > ax->loc[0].fy) CurrRect.Ymin = ax->loc[0].fy;
		if(CurrRect.Ymax < ax->loc[0].fy) CurrRect.Ymax = ax->loc[0].fy;
		if(CurrRect.Ymin > ax->loc[1].fy) CurrRect.Ymin = ax->loc[1].fy;
		if(CurrRect.Ymax < ax->loc[1].fy) CurrRect.Ymax = ax->loc[1].fy;
		}
	if(Axes && Axes[0])o->UseAxis(((Axis*)Axes[0])->GetAxis(), Axes[0]->type, dx, dy);
	if(Axes && Axes[1])o->UseAxis(((Axis*)Axes[1])->GetAxis(), Axes[1]->type, dx, dy);
	if(Axes) for(i = 0; i < nAxes; i++) {
		if(Axes[i]) Axes[i]->DoPlot(o);
		}
	if (TheLine) {
		if (TheLine->Id == GO_TERNLINE)((LineTernary*)TheLine)->doPG = doPG;
		TheLine->DoPlot(o);
		}
	if (Symbols) for (i = 0; i < nPoints; i++) {
		if(Symbols[i]) Symbols[i]->DoPlot(o);
		}
	if (Arrows) for (i = 0; i < nArrow; i++) {
		if (Arrows[i]) Arrows[i]->DoPlot(o);
		}
	if (Plots) for (i = 0; i < nPlots; i++) if (Plots[i]) {
		Plots[i]->DoPlot(o);
		}
	rDims.left = iround(o->co2fix(cent.fx - rad_l) + o->un2fix(GetSize(SIZE_GRECT_LEFT)));
	rDims.right = iround(o->co2fix(cent.fx + rad_l) + o->un2fix(GetSize(SIZE_GRECT_LEFT)));
	rDims.top = iround(o->co2fiy(cent.fy - rad_l) + o->un2fix(GetSize(SIZE_GRECT_TOP)));
	rDims.bottom = iround(o->co2fiy(cent.fy + rad_s) + o->un2fix(GetSize(SIZE_GRECT_TOP)));
	if (parent) parent->Command(CMD_AXIS, 0L, o);
}

void
TernaryXYZ::DoMark(anyOutput *o, bool mark)
{
	FillDEF fill = { 0, 0xb0808080, 1.0, 0L, 0x00ffffff };
	LineDEF line = { defs.GetSize(SIZE_HAIRLINE), defs.GetSize(SIZE_CELLTEXT), 0xff808080, 0L };

	if (!o) return;
	if (parent && parent->Id == GO_TERNARYXYZ) {
		parent->DoMark(o, mark);				return;
		}
	if (mark) {
		o->SetLine(&line);			o->SetFill(&fill);
		o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		defs.UpdRect(o, rDims.left, rDims.top, rDims.right, rDims.bottom);
		}
	else {
		Command(CMD_REDRAW, 0L, o);
		}
}

bool
TernaryXYZ::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;
	long i;

	if(o)Undo.SetDisp(o);
	switch (cmd) {
	case CMD_CONFIG:
//		Config();
		return true;
	case CMD_SCALE:
		FillLine.width *= ((scaleINFO*)tmpl)->sy.fy;
		FillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		Fill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_TERNARYXYZ;
		if(tmpl)data = (DataObj *)tmpl;
		if (Axes)for (i = 0; i< nAxes; i++){
			if (Axes[i]) {
				Axes[i]->parent = this;
				Axes[i]->Command(cmd, tmpl, o);
				}
			}
		if(Symbols && nPoints) for(i = 0; i<nPoints; i++) {
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			}
		if (Arrows && nArrow) for (i = 0; i < nArrow; i++) {
			if (Arrows[i]){
				Arrows[i]->parent = this;
				Arrows[i]->Command(cmd, tmpl, o);
				}
			}
		if (TheLine) TheLine->Command(cmd, tmpl, o);
		if (Plots){
			for (i = 0; i < nPlots; i++) if (Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_UPDATE:
		Undo.ValLong(this, &nAxes, 0L);			//dummy arg to start with no UNDO_CONTINUE
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		if(Symbols){
			SavVarObs((GraphObj **)Symbols, nPoints, UNDO_CONTINUE);
			for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			}
		if (Arrows){
			SavVarObs((GraphObj **)Arrows, nArrow, UNDO_CONTINUE);
			for (i = 0; i < nArrow; i++) if (Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
			}
		if (TheLine) TheLine->Command(cmd, tmpl, o);
		if (Plots) for (i = 0; i < nPlots; i++) {
			if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_LEGEND:
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		if (Symbols) {
			if (TheLine && (TheLine->Id == GO_DATALINE || (TheLine->Id == GO_TERNLINE && ((LineTernary*)TheLine)->doPG == 0))) {
				for (i = 0; i < nPoints && i < 100; i++)
					if (Symbols[i]) ((Legend*)tmpl)->HasSym(&((LineTernary*)TheLine)->LineDef, Symbols[i], 0L);
				}
			else {
				for (i = 0; i < nPoints && i < 100; i++)
					if (Symbols[i]) ((Legend*)tmpl)->HasSym(0L, Symbols[i], 0L);
				}
			}
		if (TheLine && (TheLine->Id == GO_DATAPOLYGON || (TheLine->Id == GO_TERNLINE && ((LineTernary*)TheLine)->doPG == 1)))
			TheLine->Command(cmd, tmpl, o);
		return false;
	case CMD_OBJTREE:
		if(!tmpl) return false;
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) 
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Plots[i], 0L);
		return true;
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		if(o) switch(((MouseEvent*)tmpl)->Action) {
		case MOUSE_LBUP:
			if (Plots) for (i = nPlots-1; i >= 0; i--) if (Plots[i]){
				if (Plots[i]->Command(cmd, tmpl, o)) return true;
				}
			if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) 
				if(Symbols[i]->Command(cmd, tmpl, o))return true;
			if (Arrows) for (i = 0; i < nArrow; i++) if (Arrows[i])
				if (Arrows[i]->Command(cmd, tmpl, o))return true;
			if (TheLine) if (TheLine->Command(cmd, tmpl, o)) return true;
			if (Axes) for (i = 0; i< nAxes; i++) if (Axes[i])
				if (Axes[i]->Command(cmd, tmpl, o))return true;
			if (rDims.left == rDims.right || rDims.top == rDims.bottom){
				rDims.left = iround(o->co2fix(cent.fx - rad_l) + o->un2fix(GetSize(SIZE_GRECT_LEFT)));
				rDims.right = iround(o->co2fix(cent.fx + rad_l) + o->un2fix(GetSize(SIZE_GRECT_LEFT)));
				rDims.top = iround(o->co2fiy(cent.fy - rad_l) + o->un2fix(GetSize(SIZE_GRECT_TOP)));
				rDims.bottom = iround(o->co2fiy(cent.fy + rad_s) + o->un2fix(GetSize(SIZE_GRECT_TOP)));
				}
			if (IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)){
				CurrGO = this;
				o->ShowMark(this, MRK_GODRAW);
				return true;
				}
			break;
		default:
			return PlotScatt::Command(cmd, tmpl, o);
			}
		return false;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i] && Axes[i] == tmpPlots[0]){
			return ReplaceGO(&Axes[i], tmpPlots);
			}
		break;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
		break;
	case CMD_DELOBJ:
		if ((((GraphObj*)tmpl))->parent == this) {
			PlotScatt::Command(cmd, tmpl, o);
			if (parent)parent->Command(CMD_REDRAW, NULL, o);
			return true;
			}
		else if (Plots && nPlots) {
			for (i = 0; i < nPlots; i++) if (Plots[i]) {
				Plots[i]->Command(cmd, tmpl, o);
				}
			return true;
			}
		return false;
	case CMD_SYMTEXT:		case CMD_SYMTEXT_UNDO:	case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:		case CMD_DELOBJ_CONT:
		PlotScatt::Command(cmd, tmpl, o);
		if(parent)parent->Command(CMD_REDRAW, NULL, o);
		return true;
	case CMD_SET_LINE:
		memcpy(&Outline, tmpl, sizeof(LineDEF));
		if (TheLine) TheLine->Command(cmd, tmpl, o);
		return true;
	case CMD_PG_FILL:
		memcpy(&Fill, tmpl, sizeof(FillDEF));
		if (Fill.hatch)memcpy(&FillLine, Fill.hatch, sizeof(LineDEF));
		Fill.hatch = &FillLine;
		if (TheLine) TheLine->Command(cmd, tmpl, o);
		return true;
	case CMD_DROP_PLOT:
		if(!tmpl) return false;
		Plots = (Plot**)realloc(Plots, sizeof(Plot*)*((nPlots += 1) + 2));
		if (Plots){
			Undo.SetGO(this, (GraphObj**)&Plots[nPlots - 1], (GraphObj*)tmpl, (DWORD)NULL);
			Undo.ValLong(this, &nPlots, UNDO_CONTINUE);
			Plots[nPlots] = NULL;
			Plots[nPlots - 1]->parent = this;
			Plots[nPlots - 1]->Command(CMD_SET_DATAOBJ, data, 0L);
			return true;
			}
		return false;
	case CMD_SAVE_TICKS:		case CMD_SET_GRIDLINE:
		//do all axes
		for (i = 0; Axes && i < nAxes; i++){
			Axes[i] && Axes[i]->Command(cmd, tmpl, o);
			}
		return true;
		}
	return false;
}

void *
TernaryXYZ::ObjThere(int x, int y)
{
	long i;
	void *ptr;

	//do all axes
	if(Axes && nAxes) for (i = 0; Axes && i < nAxes; i++){
		if (Axes[i] && (ptr = Axes[i]->ObjThere(x, y))) return ptr;;
		}
	else if (parent && parent->Id == GO_TERNARYXYZ) return parent->ObjThere(x, y);
	return PlotScatt::ObjThere(x, y);
}


bool
TernaryXYZ::TransformVal(anyOutput *o, fPOINT3D *lfp, double *fix, double *fiy)
{
	double sum, tmp;
	lfPOINT fp, fip;

	if(parent && parent->Id == GO_TERNARYXYZ) 
		return ((TernaryXYZ*)parent)->TransformVal(o, lfp, fix, fiy);
	if (parent->Id != GO_GRAPH) return false;
	if(!o || !lfp || !fix || !fiy || !Axes[0]) return false;
	sum = lfp->fx + lfp->fy + lfp->fz;
	fp.fx = lfp->fx/sum * 100.0;
	fp.fy = lfp->fy/sum * 100.0;
	o->fp2fip(&fp, &fip);
	tmp = o->co2fiy(o->xAxis.loc[0].fy);
	if (((Axis*)Axes[0])->axis->flags & AXIS_INVERT) {
		fip.fx -= ((tmp - fip.fy + o->co2fiy(parent->GetSize(SIZE_GRECT_TOP))) * ang_disp);
		}
	else fip.fx += ((tmp - fip.fy + o->co2fiy(parent->GetSize(SIZE_GRECT_TOP))) * ang_disp);
	fip.fx -= o->getVPorgX();		fip.fy -= o->getVPorgY();
	*fix = iround(fip.fx);	*fiy = iround(fip.fy);
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// BoxPlot is a Plot-Class
BoxPlot::BoxPlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_BOXPLOT;
}

BoxPlot::BoxPlot(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Boxes) 
			for(i = 0; i < nBoxes; i++) if(Boxes[i]) Boxes[i]->parent = this;
		if(Whiskers) 
			for(i = 0; i < nWhiskers; i++) if(Whiskers[i]) Whiskers[i]->parent = this;
		if(Symbols) 
			for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->parent = this;
		if(Labels) 
			for(i = 0; i < nLabel; i++) if(Labels[i]) Labels[i]->parent = this;
		if(TheLine) TheLine->parent = this;
		}
}

BoxPlot::BoxPlot(GraphObj *par, DataObj *dt, int mode, int c1, int c2, int c3, char *box_name):Plot(par, dt)
{
	long i, nr;
	lfPOINT fp;

	FileIO(INIT_VARS);		Id = GO_BOXPLOT;	fp.fx = fp.fy = 0.0;
	if(data && data->GetSize(&i, &nr)) {
		nPoints = nBoxes = nr;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		Boxes = (Box**)calloc(nr, sizeof(Box*));
		if (Boxes) for (i = 0; i < nr; i++) {
			if(mode == 1) Boxes[i] = new Box(this, data, fp, fp, BAR_RELWIDTH, c1, i, c2, i, c1, i, c3, i);
			else Boxes[i] = new Box(this, data, fp, fp, BAR_RELWIDTH, c2, i, c1, i, c3, i, c1, i);
			if(box_name && box_name[0] && Boxes[i]) Boxes[i]->name = rlp_strdup(box_name);
			}
		}
}

BoxPlot::~BoxPlot()
{
	int i;

	if(Whiskers) {
		for(i = 0; i < nWhiskers; i++) if(Whiskers[i]) DeleteGO(Whiskers[i]);
		free (Whiskers);
		}
	if(Boxes) {
		for(i = 0; i < nBoxes; i++) if(Boxes[i]) DeleteGO(Boxes[i]);
		free (Boxes);
		}
	if(Symbols) {
		for(i = 0; i < nPoints; i++) if(Symbols[i]) DeleteGO(Symbols[i]);
		free (Symbols);
		}
	if(Labels) {
		for(i = 0; i < nLabel; i++) if(Labels[i]) DeleteGO(Labels[i]);
		free (Labels);
		}
	if (CurrPlot == this) CurrPlot = NULL;
	if(TheLine) DeleteGO(TheLine);
	if (ErrPg) DeleteGO(ErrPg);
	if(curr_data) delete curr_data;
	curr_data = 0L;						if(xRange) free(xRange);
	xRange = 0L;						if(yRange) free(yRange);
	yRange = 0L;						if(case_prefix) free(case_prefix);
	case_prefix = 0L;					if(name) free(name);
	name = 0L;							if(x_tv) delete(x_tv);
	x_tv = 0;							if(y_tv) delete(y_tv);
	y_tv = 0;
	Undo.InvalidGO(this);
}

double
BoxPlot::GetSize(int select)
{
	int i;
	double ft1, ft2, d;

	switch(select){
	case SIZE_BOXMINX:
		if(BoxDist.fx >= 0.0001) return BoxDist.fx;
		if((!Boxes) || (nPoints < 2)) return BoxDist.fx = 1.0;
		ft1 = -HUGE_VAL;	ft2 = HUGE_VAL;		BoxDist.fx= HUGE_VAL;
		for(i = 0; i < nPoints; i++) {
			if(Boxes[i]) {
				ft2 = Boxes[i]->GetSize(SIZE_XPOS);
				d = fabs(ft2-ft1);
				if(d != 0.0 && d < BoxDist.fx) BoxDist.fx = d;
				}
			ft1 = ft2;
			}
		return BoxDist.fx = BoxDist.fx > 0.0001 && BoxDist.fx != HUGE_VAL  ? BoxDist.fx : 1.0;
	case SIZE_BOXMINY:
		if(BoxDist.fy >= 0.0001) return BoxDist.fy;
		if((!Boxes) || (nPoints < 2)) return BoxDist.fy = 1.0;
		ft1 = -HUGE_VAL;	ft2 = HUGE_VAL;		BoxDist.fy= HUGE_VAL;
		for(i = 0; i < nPoints; i++) {
			if(Boxes[i]) {
				ft2 = Boxes[i]->GetSize(SIZE_YPOS);
				d = fabs(ft2-ft1);
				if(d != 0.0 && d < BoxDist.fy) BoxDist.fy = d;
				}
			ft1 = ft2;
			}
		return BoxDist.fy = BoxDist.fy > 0.0001 && BoxDist.fy != HUGE_VAL  ? BoxDist.fy : 1.0;
	default:
		return Plot::GetSize(select);
		}
}

bool
BoxPlot::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:		case SIZE_SYM_LINE:
		if(Symbols) for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	case SIZE_WHISKER:		case SIZE_WHISKER_LINE:
		if(Whiskers) for(i = 0; i < nWhiskers; i++) 
			if(Whiskers[i]) Whiskers[i]->SetSize(select, value);
		return true;
	case SIZE_BOX:			case SIZE_BOX_LINE:
		if(Boxes) for(i = 0; i < nBoxes; i++) 
			if(Boxes[i]) Boxes[i]->SetSize(select, value);
		return true;
	case SIZE_LB_XDIST:		case SIZE_LB_YDIST:
		if(Labels) for(i = 0; i < nLabel; i++)
			if(Labels[i]) Labels[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
BoxPlot::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_SYM_LINE:		case COL_SYM_FILL:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
	case COL_WHISKER:
		if(Whiskers) for(i = 0; i < nWhiskers; i++)
			if(Whiskers[i]) Whiskers[i]->SetColor(select, col);
		return true;
	case COL_BOX_LINE:
		if(Boxes) for(i = 0; i < nBoxes; i++)
			if(Boxes[i]) Boxes[i]->SetColor(select, col);
		return true;
	default:
		return false;
		}
}

void
BoxPlot::DoPlot(anyOutput *o)
{
	if(!parent || !o) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	ForEach(FE_PLOT, 0L, o);
	dirty = false;
	if(use_xaxis || use_yaxis)parent->Command(CMD_AXIS, 0L, o);
}

bool
BoxPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	GraphObj *go;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		if(!CurrGO && ((MouseEvent*)tmpl)->Action == MOUSE_LBUP) return ForEach(cmd, tmpl, o);
		return false;
	case CMD_TEXTTHERE:
		if(Labels) for(i = 0; i < nPoints; i++)	if(Labels[i] &&  Labels[i]->Command(cmd, tmpl, o))	return true;
		return false;
	case CMD_COPY:
		CopyData1();
		return true;
	case CMD_CPY_DATA:
		CopyData(this, CF_SYLK);
		return true;
	case CMD_LEGEND:
		if(((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(Symbols) {
			if(TheLine && TheLine->Id == GO_DATALINE) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(&TheLine->LineDef, Symbols[i], 0L);
				}
			else {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(0L, Symbols[i], 0L);
				}
			if(TheLine && TheLine->Id == GO_DATAPOLYGON) TheLine->Command(cmd, tmpl, o);
			}
		else if(TheLine) TheLine->Command(cmd, tmpl, o);
		if(Boxes) for (i = 0; i < nBoxes; i++)	if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		if(Whiskers) for (i = 0; i < nWhiskers; i++)	if(Whiskers[i]) Whiskers[i]->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_BOXPLOT;		data = (DataObj *)tmpl;		dirty = true;
		if(type && xRange && yRange && data) {		//Stat. - Plot ?
			CreateData();
			}
		return ForEach(CMD_SET_DATAOBJ, curr_data ? curr_data : tmpl, o);
	case CMD_SCALE:
		return ForEach(cmd, tmpl, o);
	case CMD_AUTOSCALE:
		if(hidden) return false;
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		else{
			if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH && 
				Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) {
				((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
				((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
				return true;
				}
			}
		dirty = false;
		ForEach(cmd, tmpl, o);
		if(x_tv) {
			Bounds.Xmin = 0.5;		Bounds.Xmax = ((double)x_tv->Count())+0.5;
			}
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH
			&& Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin){
			((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
			((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
			}
		return true;
	case CMD_UPDATE:
		if(parent) {
			if(Boxes) SavVarObs((GraphObj **)Boxes, nBoxes, UNDO_CONTINUE);
			if(Whiskers) SavVarObs((GraphObj **)Whiskers, nWhiskers, UNDO_CONTINUE);
			if(Symbols) SavVarObs((GraphObj **)Symbols, nPoints, UNDO_CONTINUE);
			if(Labels) SavVarObs((GraphObj **)Labels, nLabel, UNDO_CONTINUE); 
			}
		if(type && xRange && yRange) {		//Stat. - Plot ?
			CreateData();
			ForEach(CMD_SET_DATAOBJ, curr_data, o);
			}
		ForEach(cmd, tmpl, o);
		return dirty = true;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_SETSCROLL:		case CMD_REDRAW:		case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_SETTEXTDEF:
		if(Labels) for(i = 0; i < nLabel; i++)
			if(Labels[i]) Labels[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(ForEach(FE_DELOBJ, tmpl, o)) {
			parent->Command(CMD_REDRAW, 0L, o);
			return true;
			}
		return false;
	case CMD_SYMTEXT:		case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
	case CMD_SAVE_BARS:
		return SavVarObs((GraphObj **)Boxes, nBoxes, 0L);
	case CMD_SAVE_BARS_CONT:
		return SavVarObs((GraphObj **)Boxes, nBoxes, UNDO_CONTINUE);
	case CMD_SAVE_ERRS:
		return SavVarObs((GraphObj **)Whiskers, nWhiskers, 0L);
	case CMD_SAVE_LABELS:
		return SavVarObs((GraphObj **)Labels, nLabel, 0L);
	case CMD_BOX_FILL:			case CMD_BOX_TYPE:
		if (cmd == CMD_BOX_TYPE) BoxDist.fy = BoxDist.fx = 0.0;
		if(Boxes) for (i = 0; i < nBoxes; i++)
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_WHISKER_STYLE:		case CMD_ERRDESC:
		if(Whiskers) for (i = 0; i < nWhiskers; i++)
			if(Whiskers[i]) Whiskers[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_TAB:	case CMD_SHTAB:
		go = (GraphObj*)tmpl;
		if (go && go->parent && go->parent == this) return ForEach(cmd, 0L, o);
		return false;
		}
	return false;
}

bool
BoxPlot::ForEach(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj ***pobs[] = {(GraphObj***)&Boxes, (GraphObj***)&Whiskers, 
		(GraphObj***)&Symbols, (GraphObj***)&Labels};
	long n_obs[] = {nBoxes, nWhiskers, nPoints, nLabel};
	GraphObj **p;
	int i, j;
	bool bRet;

	switch(cmd) {
	case FE_DELOBJ:
		if(!o || !parent || !tmpl) return false;
		for(i = 0; i < 4; i++) {
			if(DeleteGOL(pobs[i], n_obs[i], (GraphObj*) tmpl, o)) return true;
			}
		if (ErrPg && tmpl == (void *)ErrPg) {
			Undo.DeleteGO((GraphObj**)(&ErrPg), 0L, o);
			return true;
			}
		if(TheLine && tmpl == (void *) TheLine) {
			Undo.DeleteGO((GraphObj**)(&TheLine), 0L, o);
			return true;
			}
		break;
	case CMD_TAB:
		if (!CurrGO) return false;
		switch (CurrGO->Id) {
		case GO_BOX:			j = 0;				break;
		case GO_WHISKER:		j = 1;				break;
		case GO_SYMBOL:			j = 2;				break;
		case GO_LABEL:			j = 3;				break;
		default:				return false;
			}
		p = *pobs[j];
		for (i = 0; i < n_obs[j]; i++){
			if (p[i] && p[i] == CurrGO){
				while (!p[i + 1] && i < n_obs[j] - 1) i++;
				if (i < n_obs[j] - 1) p[i + 1]->Command(CMD_SELECT, 0L, o);
				return true;
				}
			}
		return false;
	case CMD_SHTAB:
		if (!CurrGO) return false;
		switch (CurrGO->Id) {
		case GO_BOX:			j = 0;				break;
		case GO_WHISKER:		j = 1;				break;
		case GO_SYMBOL:			j = 2;				break;
		case GO_LABEL:			j = 3;				break;
		default:				return false;
			}
		p = *pobs[j];
		for (i = 1; i < n_obs[j]; i++){
			if (p[i] && p[i] == CurrGO){
				while (!p[i - 1] && i >= 0) i--;
				if (i > 0) p[i - 1]->Command(CMD_SELECT, 0L, o);
				return true;
				}
			}
		if (i >= n_obs[j] - 1) return true;
		return false;
	case FE_PLOT:
		if (ErrPg) ErrPg->DoPlot(o);
		if(TheLine) TheLine->DoPlot(o);
		for(i = 0; i < 4; i++){
			if((p= *pobs[i])) for(j = 0; j < n_obs[i]; j++) {
				if(p[j]) p[j]->DoPlot(o);
				}
			}
		return true;
	case CMD_MOUSE_EVENT:				//invers to plot order
		for(i = 3; i >= 0; i--){
			if((p= *pobs[i])) for(j = n_obs[i]-1; j >= 0; j--) {
				if(p[j]) {
					bRet = p[j]->Command(cmd, tmpl, o);
					if(bRet && cmd == CMD_MOUSE_EVENT) return true;
					}
				}
			}
		if (TheLine && TheLine->Command(cmd, tmpl, o)) return true;
		if (ErrPg && ErrPg->Command(cmd, tmpl, o)) return true;
		return false;
	default:							//pass command to all objects
		for(i = 0; i < 4; i++){
			p = *pobs[i];
			if(p) for(j = 0; j < n_obs[i]; j++) {
				if(p[j]) {
					bRet = p[j]->Command(cmd, tmpl, o);
					p[j]->parent = this;
					}
				}
			}
		if (TheLine) {
			TheLine->Command(cmd, tmpl, o);
			TheLine->parent = this;
			}
		if (ErrPg){
			ErrPg->Command(cmd, tmpl, o);
			ErrPg->parent = this;
			}
		return true;
		}
	return false;
}

void
BoxPlot::CreateData()
{
	long i, j, k, l, m, n;
	int *ny, c_num, c_txt, c_dattim;
	double y, ss, d, lo, hi, **ay, *ax, *tay, *q1, *q2, *q3;
	lfPOINT *xy;
	AccRange *rX, *rY;
	anyResult x_res, y_res;

	if(curr_data) delete curr_data;
	curr_data = 0L;
	if(!data || !xRange || !yRange || !xRange[0] || !yRange[0]) return;
	if(!(rX = new AccRange(xRange)) || !(rY = new AccRange(yRange))) return;
	m = rX->CountItems();	n = 0;
	xy = (lfPOINT*)malloc(m * sizeof(lfPOINT));
		if (m < 2 || !(xy)) {
		delete rX;	delete rY;
		return;
		}
	if(x_tv) delete x_tv;
	x_tv = 0L;
	ny = (int*) calloc(m, sizeof(int));
	ay = (double**) calloc(m, sizeof(double*));
	ax = (double*) calloc(m, sizeof(double));
	tay = (double*)malloc(m * sizeof(double));
	if(!ny || !ay || !ax || !tay) {
		if(ny) free(ny);
		if(ay) free(ay);
		if(ax) free(ax);
		if(tay) free(tay);
		delete rX;	delete rY;
		return;
		}
	rX->DataTypes(data, &c_num, &c_txt, &c_dattim);
	if(c_num < 5 && (c_txt + c_dattim) > 5) {
		x_tv = new TextValue();	
		}
	rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
	rX->GetNext(&i, &j);	rY->GetNext(&k, &l);	n=0;
	do {
		if(data->GetResult(&x_res, j, i, false) && data->GetResult(&y_res, l, k, false) && y_res.type == ET_VALUE) {
			xy[n].fy = y_res.value;
			if(x_tv){ 
				switch(x_res.type) {
				case ET_TEXT:
					xy[n++].fx = x_tv->GetValue(x_res.text);
					break;
				case ET_VALUE:	case ET_BOOL:	case ET_DATE:	case ET_TIME:	case ET_DATETIME:
					TranslateResult(&x_res);
					xy[n++].fx = x_tv->GetValue(x_res.text);
					break;
					}
				}
			else if(x_res.type == ET_VALUE) xy[n++].fx = x_res.value;
			}
		}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
	delete rX;			delete rY;
	if(!n) {
		if(ny) free(ny);
		if(ay) free(ay);
		if(ax) free(ax);
		if(tay) free(tay);
		return;
		}
	SortFpArray(n, xy);
	for(i = j = 0; i < (n-1); i++, j++) {
		ax[j] = xy[i].fx;		tay[0] = xy[i].fy;
		ny[j] = 1;
		for(k = 1; xy[i+1].fx == xy[i].fx; k++) {
			tay[k] = xy[i+1].fy;
			i++;		ny[j]++;
			}
		ay[j] = (double*)memdup(tay, k * sizeof(double), 0);
		}
	if(xy[i].fx > xy[i-1].fx) {
		ax[j] = xy[i].fx;		tay[0] = xy[i].fy;
		ny[j] = 1;
		ay[j++] = (double*)memdup(tay, sizeof(double), 0);
		}
	if((type & 0x0004) == 0x0004 || (type & 0x0030) == 0x0030 || (type & 0x0300) == 0x0300) {
		//medians and/or percentiles required
		q1 = (double *)malloc(j * sizeof(double));
		q2 = (double *)malloc(j * sizeof(double));
		q3 = (double *)malloc(j * sizeof(double));
		if(q1 && q2 && q3) {
			for(i = 0; i < j; i++) {
				if(ny[i] > 1) d_quartile(ny[i], ay[i], q1+i, q2+i, q3+i);
				else q1[i] = q2[i] = q3[i] = *ay[i];
				}
			}
		else type = 0;
		}
	else q1 = q2 = q3 = 0L;
	if(type && (curr_data = new DataObj()) && curr_data->Init(j, 8)) {
		for(i = 0; i < j; i++) curr_data->SetValue(i,0,ax[i]);	// set x-values
		for(i = 0; i < j; i++) {								// set means
			if(ny[i] > 1) switch(type & 0x000f) {
				case 0x0001:	default:
					curr_data->SetValue(i, 1, y=d_amean(ny[i], ay[i]));
					break;
				case 0x0002:
					curr_data->SetValue(i, 1, y=d_gmean(ny[i], ay[i]));
					break;
				case 0x0003:
					curr_data->SetValue(i, 1, y=d_hmean(ny[i], ay[i]));
					break;
				case 0x0004:
					curr_data->SetValue(i, 1, y=q2[i]);
					break;
				}
			else curr_data->SetValue(i, 1, y= *ay[i]);
			curr_data->SetValue(i, 6, y);						//label's y
			}
		if((type & 0x00f0) == 0x0010 || (type & 0x00f0) == 0x0020 || (type & 0x00f0) == 0x0050
			|| (type & 0x0f0f) == 0x0201 || (type & 0x0f0f) == 0x0501) for(i = 0; i < j; i++) {
			// set SD, SE, Conf. Intervall
			if(ny[i] > 1) {
				ss = sqrt(d_variance(ny[i], ay[i], &y));
				}
			else {
				y = *ay[i];		ss = 0.0;
				}
			//Box info is in cols 2 & 3
			if((type & 0x00f0) == 0x0010) {
				curr_data->SetValue(i, 2, y - ss);	curr_data->SetValue(i, 3, y + ss);
				}
			else if((type & 0x00f0) == 0x0020) {
				curr_data->SetValue(i, 2, y - ss/sqrt((double)ny[i]));	
				curr_data->SetValue(i, 3, y + ss/sqrt((double)ny[i]));
				}
			else if((type & 0x00f0) == 0x0050) {
				d = ny[i] > 1 ? distinv(t_dist, ny[i]-1, 1, 1.0-(ci_box/100.0), 2.0) : 0;
				curr_data->SetValue(i, 2, y - d*ss/(double)sqrt((double)ny[i]));	
				curr_data->SetValue(i, 3, y + d*ss/(double)sqrt((double)ny[i]));
				}
			//Whisker info is in cols 4 & 5
			if((type & 0x0f0f) == 0x0101) {
				curr_data->SetValue(i, 4, y - ss);	curr_data->SetValue(i, 5, y + ss);
				}
			else if((type & 0x0f0f) == 0x0201) {
				curr_data->SetValue(i, 4, y - ss/sqrt((double)ny[i]));
				curr_data->SetValue(i, 5, y + ss/sqrt((double)ny[i]));
				}
			else if((type & 0x0f0f) == 0x0501) {
				d = ny[i] > 1 ? distinv(t_dist, ny[i]-1, 1, 1.0-(ci_err/100.0), 2.0) : 0;
				curr_data->SetValue(i, 4, y - d*ss/sqrt((double)ny[i]));
				curr_data->SetValue(i, 5, y + d*ss/sqrt((double)ny[i]));
				}
			}
		if((type & 0x00f0) == 0x0040 || (type & 0x0f00) == 0x0400) for(i = 0; i < j; i++) {
			// set min and max
			lo = hi = *ay[i];
 			if(ny[i] > 1) {
				for(k = 1; k < ny[i]; k++) {
					if(ay[i][k] < lo) lo = ay[i][k];
					if(ay[i][k] > hi) hi = ay[i][k];
					}
				}
			if((type & 0x00f0) == 0x0040) {
				curr_data->SetValue(i, 2, lo);	curr_data->SetValue(i, 3, hi);
				}
			if((type & 0x0f00) == 0x0400) {
				curr_data->SetValue(i, 4, lo);	curr_data->SetValue(i, 5, hi);
				}
			}
		if(q1 && q3 && ((type & 0x00f0) == 0x0030 || (type & 0x0f00) == 0x0300)) for(i = 0; i < j; i++) {
			// percentiles ....
			if((type & 0x00f0) == 0x0030) {
				curr_data->SetValue(i, 2, q1[i]);	curr_data->SetValue(i, 3, q3[i]);
				}
			if((type & 0x0f00) == 0x0300) {
				curr_data->SetValue(i, 4, q1[i]);	curr_data->SetValue(i, 5, q3[i]);
				}
			}
		if(type & 0xc000) for(i = 0; i < j; i++) {
			//labels ...
			if((type & 0x4000) && curr_data->GetValue(i, 5, &y)) curr_data->SetValue(i, 6, y);
#ifdef USE_WIN_SECURE
			sprintf_s(TmpTxt, TMP_TXT_SIZE, "%s%d", case_prefix ? case_prefix : "", ny[i]);
#else
			sprintf(TmpTxt, "%s%d", case_prefix ? case_prefix : "", ny[i]);
#endif
			curr_data->SetText(i, 7, TmpTxt);
			}
		}
	else {
		if(curr_data) delete curr_data;
		curr_data = 0L;
		}
	if(q1) free(q1);
	if(q2) free(q2);
	if(q3) free(q3);
	for (i = 0; i < m; i++){
		if (ay[i]) free(ay[i]);
		}
	free(tay);	free(ay);	free(ax);	free(ny);	free(xy);
}

void
BoxPlot::CopyData1()
{
	char *ptr;
	long cbOut, sizeOut;
	char tmptxt[200];
	int cb;
	long i;

	ptr = (char *)malloc(sizeOut = 2000);
	cbOut = 0;
	if (ptr && Symbols && Whiskers &&nPoints && nPoints == nWhiskers && (Symbols[0] || Symbols[1]) && (Whiskers[0] || Whiskers[1])){
#ifdef USE_WIN_SECURE
		cbOut = sprintf_s(ptr, 80, "x\t%s\t%s\n", Symbols[0] ? Symbols[0]->name : Symbols[1]->name, Whiskers[0] ? Whiskers[0]->name : Whiskers[1]->name);
#else
		cbOut = sprintf(ptr, "x\t%s\t%s\n", Symbols[0] ? Symbols[0]->name : Symbols[1]->name, Whiskers[0] ? Whiskers[0]->name : Whiskers[1]->name);
#endif
		for (i = 0; i < nPoints; i++) if(Symbols[i] && Whiskers[i]) {
#ifdef USE_WIN_SECURE
			cb = sprintf_s(tmptxt, 180, "%g\t%g\t%g\t%g\n", Symbols[i]->GetSize(SIZE_XPOS), Symbols[i]->GetSize(SIZE_YPOS),
				Whiskers[i]->GetSize(SIZE_YPOS), Whiskers[i]->GetSize(SIZE_YPOS+1));
#else
			cb = sprintf(tmptxt, "%g\t%g\t%g\t%g\n", Symbols[i]->GetSize(SIZE_XPOS), Symbols[i]->GetSize(SIZE_YPOS),
				Whiskers[i]->GetSize(SIZE_YPOS), Whiskers[i]->GetSize(SIZE_YPOS + 1));
#endif
			Int2Nat(tmptxt);
			add_to_buff(&ptr, &cbOut, &sizeOut, tmptxt, cb);
			}
		CopyText(ptr, 0, CF_TEXT);
		}
	free(ptr);
}

char*
BoxPlot::CopySylk()
{
	long cbd = 0, size, nc, nl, i;
	static char *ptr;

	if (!(ptr = (char *)malloc((size = 10000)*sizeof(char)))) return 0L;
	 cbd = rlp_strcpy(ptr, size, (char*)"ID;PWXL;N;E\n"
		"P;Pdd/mm/yyyy\nP;Phh:mm:ss\nP;Pdd/mm/yyyy hh:mm:ss\n");
	 nl = nc = 0;
	 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
	 add_to_buff(&ptr, &cbd, &size, (char*)"Box Plot Data\"\n", 0);
	 if (Symbols && nPoints) {
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff(&ptr, &cbd, &size, (char*)"Symbols\"\n", 0);
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff(&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc+1);
		add_to_buff(&ptr, &cbd, &size, (char*)"y\"\n", 0);
		for (i = 0; i < nPoints; i++) {
			if (Symbols[i]) {
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff(&ptr, &cbd, &size, Symbols[i]->GetSize(SIZE_XPOS), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff(&ptr, &cbd, &size, Symbols[i]->GetSize(SIZE_YPOS), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				}
			nl++;
			}
		nc += 2;	nl = 1;
		}
	 if (Boxes && nBoxes) {
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		 add_to_buff(&ptr, &cbd, &size, (char*)"Boxes\"\n", 0);
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		 add_to_buff(&ptr, &cbd, &size, (char*)"x\"\n", 0);
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 1);
		 add_to_buff(&ptr, &cbd, &size, (char*)"y1\"\n", 0);
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 2);
		 add_to_buff(&ptr, &cbd, &size, (char*)"y2\"\n", 0);
		 for (i = 0; i < nBoxes; i++) {
			 if (Boxes[i]) {
				 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				 add_dbl_to_buff(&ptr, &cbd, &size, Boxes[i]->GetSize(SIZE_XPOS), false);
				 add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				 add_dbl_to_buff(&ptr, &cbd, &size, Boxes[i]->GetSize(SIZE_YPOS + 1), false);
				 add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 2);
				 add_dbl_to_buff(&ptr, &cbd, &size, Boxes[i]->GetSize(SIZE_YPOS + 2), false);
				 add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				}
			 nl++;
			}
		 nc += 3;	nl = 1;
		}
	 if (Whiskers && nWhiskers) {
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		add_to_buff(&ptr, &cbd, &size, (char*)"Whiskers\"\n", 0);
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		add_to_buff(&ptr, &cbd, &size, (char*)"x\"\n", 0);
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 1);
		add_to_buff(&ptr, &cbd, &size, (char*)"y1\"\n", 0);
		sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 2);
		add_to_buff(&ptr, &cbd, &size, (char*)"y2\"\n", 0);
		for (i = 0; i < nWhiskers; i++) {
			if (Whiskers[i]) {
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff(&ptr, &cbd, &size, Whiskers[i]->GetSize(SIZE_XPOS), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff(&ptr, &cbd, &size, Whiskers[i]->GetSize(SIZE_YPOS), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 2);
				add_dbl_to_buff(&ptr, &cbd, &size, Whiskers[i]->GetSize(SIZE_YPOS + 1), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				}
			nl++;
			}
		nc += 3;	nl = 0;
		}
	 if (Labels && nLabel) {
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc);
		 add_to_buff(&ptr, &cbd, &size, (char*)"Labels\"\n", 0);
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc);
		 add_to_buff(&ptr, &cbd, &size, (char*)"x\"\n", 0);
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 1);
		 add_to_buff(&ptr, &cbd, &size, (char*)"y\"\n", 0);
		 sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl++, nc + 2);
		 add_to_buff(&ptr, &cbd, &size, (char*)"text\"\n", 0);
		 for (i = 0; i < nLabel; i++) {
			if (Labels[i]) {
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc);
				add_dbl_to_buff(&ptr, &cbd, &size, Labels[i]->GetSize(SIZE_XPOS), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K", nl, nc + 1);
				add_dbl_to_buff(&ptr, &cbd, &size, Labels[i]->GetSize(SIZE_YPOS), false);
				add_to_buff(&ptr, &cbd, &size, (char*)"\n", 1);
				Labels[i]->Command(CMD_GETTEXT, TmpTxt, 0L);
				sylk_cell_ref(&ptr, &cbd, &size, (char*)"C;", (char*)";K\"", nl, nc + 2);
				add_to_buff(&ptr, &cbd, &size, (char*)TmpTxt, 0);
				add_to_buff(&ptr, &cbd, &size, (char*)"\"\n", 2);
				}
			nl++;
			}
		}
	add_to_buff(&ptr, &cbd, &size, (char*)"E\n", 2);
	return ptr;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Density distribution plot
DensDisp::DensDisp(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_DENSDISP;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

DensDisp::DensDisp(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Boxes) 
			for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->parent = this;
		}
}

DensDisp::~DensDisp()
{
	int i;

	if(Boxes) {
		for(i = 0; i < nPoints; i++) if(Boxes[i]) DeleteGO(Boxes[i]);
		free (Boxes);
		}
	if(yRange) free(yRange);
	if(xRange) free(xRange);
	yRange = xRange = 0L;
	if(name) free(name);
	name=0L;						Undo.InvalidGO(this);
}

bool
DensDisp::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_BOX_LINE:
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
DensDisp::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_BOX_LINE:
		if(Boxes) for(i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->SetColor(select, col);
		return true;
	default:
		return false;
		}
}

void
DensDisp::DoPlot(anyOutput *o)
{
	int i;

	if(!parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) {
		ApplyAxes(o);
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->DoPlot(o);
		parent->Command(CMD_AXIS, 0L, o);
		}
	else {
		if (Boxes) for (i = 0; i < nPoints; i++) {
			if (Boxes[i]) Boxes[i]->DoPlot(o);
			}
		}
	dirty = false;
}

bool
DensDisp::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_SETSCROLL:		case CMD_REDRAW:		case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_SCALE:
		DefLine.width *= ((scaleINFO*)tmpl)->sy.fy;		DefLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		DefFillLine.width *= ((scaleINFO*)tmpl)->sy.fy;	DefFillLine.patlength *= ((scaleINFO*)tmpl)->sy.fy;
		DefFill.scale *= ((scaleINFO*)tmpl)->sy.fy;
		for(i = 0; i < nPoints; i++){
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			for(i = nPoints-1; i >= 0 && !CurrGO; i--) {
				if(Boxes[i] && Boxes[i]->Command(cmd, tmpl, o))return true;
				}
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		for (i = 0; i < nPoints; i++) {
			if (Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
			}
		Id = GO_DENSDISP;
		data = (DataObj *)tmpl;
		return true;
	case CMD_DELOBJ:
		if(!parent || !o) return false;
		if(DeleteGOL((GraphObj***)&Boxes, nPoints, (GraphObj*)tmpl, o)) 
			return parent->Command(CMD_REDRAW, 0L, o);
		break;
	case CMD_AUTOSCALE:
		if(dirty){
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		else return true;
		dirty = false;
		if (Boxes) for (i = 0; i < nPoints; i++)
			if (Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_BOX_TYPE:		case CMD_BOX_FILL:
		if (cmd == CMD_AUTOSCALE) {
			if (dirty){
				Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
				Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
				}
			else return true;
			dirty = false;
			}
		if(Boxes) for (i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_UPDATE:
		if(Boxes) SavVarObs((GraphObj **)Boxes, nPoints, UNDO_CONTINUE);
		DoUpdate();
		return true;
		}
	return false;
}

void
DensDisp::DoUpdate()
{
	AccRange *rX, *rY;
	long i, j, k, l;
	int ic, n;
	double v, w;
	lfPOINT fp1, fp2;
	Box **op = Boxes;

	if(xRange && yRange && (rX = new AccRange(xRange)) && (rY = new AccRange(yRange))) {
		if((n=rX->CountItems()) == rY->CountItems()) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			if(!(Boxes = (Box**)realloc(Boxes, n * sizeof(Box*)))) return;
			if(op && op != Boxes) Undo.InvalidGO(this);
			for(i = nPoints; i < n; i++) {
				Boxes[i] = 0L;
				}
			nPoints = n;
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			for(ic = 0; ic < n && !data->GetValue(j, i, &v); ic++) {
				rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
				}
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			if(type & 0x10){			//vertical ?
				fp2.fx = 0;		fp2.fy = v;
				}
			else {
				fp2.fx = v;		fp2.fy = 0.0;
				}
			ic = 0;
			do {
				if(data->GetValue(j, i, &v) && data->GetValue(l, k, &w)){
					fp1.fx = fp2.fx;	fp1.fy = fp2.fy;
					if(type & 0x10) {
						CheckBounds(w, fp1.fy);			CheckBounds(-w, v);
						fp2.fy = v;		
						switch(type & 0x3) {
						case 1:		fp1.fx = fp2.fx = w/2.0;		break;
						case 2:		fp1.fx = fp2.fx = -w/2.0;		break;
						default:	fp2.fx = 0.0;					break;
							}
						}
					else {
						CheckBounds(fp1.fx, w);			CheckBounds(v, -w);
						fp2.fx = v;
						switch(type & 0x3) {
						case 1:		fp1.fy = fp2.fy = w/2.0;		break;
						case 2:		fp1.fy = fp2.fy = -w/2.0;		break;
						default:	fp2.fy = 0.0;					break;
							}
						}
					if(op && Boxes[ic]) {
						Boxes[ic]->SetSize(SIZE_XPOS, fp1.fx);	Boxes[ic]->SetSize(SIZE_XPOS+1, fp2.fx);
						Boxes[ic]->SetSize(SIZE_YPOS, fp1.fy);	Boxes[ic]->SetSize(SIZE_YPOS+1, fp2.fy);
						Boxes[ic]->SetSize(SIZE_BOX, (type &0x03) ? w/2.0 : w);
						}
					else if(!op && (Boxes[ic] = new Box(this, data, fp1, fp2, BAR_WIDTHDATA)))
						Boxes[ic]->SetSize(SIZE_BOX, (type &0x03) ? w/2.0 : w);
					}
				ic++;
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			if(!op) {
				SetSize(SIZE_BOX_LINE, DefLine.width);
				SetColor(COL_BOX_LINE, DefLine.color);
				Command(CMD_BOX_FILL, (void*)&DefFill, 0L);
				}
			}
		delete(rX);		delete(rY);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Stacked bars consist of several box-plots
StackBar::StackBar(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_STACKBAR;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

StackBar::StackBar(int src):Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Boxes) for(i = 0; i < numPlots; i++)
			if(Boxes[i]) Boxes[i]->parent = this;
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->parent = this;
		if(Polygons) for(i = 0; i < numPG; i++)
			if(Polygons[i]) Polygons[i]->parent = this;
		if(Lines) for(i = 0; i < numPL; i++)
			if(Lines[i]) Lines[i]->parent = this;
		}
}

StackBar::~StackBar()
{
	long i;

	if(Boxes){
		for(i = 0; i < numPlots; i++) if(Boxes[i]) DeleteGO(Boxes[i]);
		free(Boxes);
		}
	if(xyPlots){
		for(i = 0; i < numXY; i++) if(xyPlots[i]) DeleteGO(xyPlots[i]);
		free(xyPlots);
		}
	if(Polygons) {
		for(i = 0; i < numPG; i++) if(Polygons[i]) DeleteGO(Polygons[i]);
		free(Polygons);
		}
	if(Lines) {
		for(i = 0; i < numPL; i++) if(Lines[i]) DeleteGO(Lines[i]);
		free(Lines);
		}
	if(ssXrange) free(ssXrange);
	if(ssYrange) free(ssYrange);
	if(CumData) delete CumData;
	CumData = 0L;					if(name) free(name);
	name=0L;						if(x_tv) delete x_tv;
	x_tv = 0L;						if(y_tv) delete y_tv;
	y_tv = 0L;
	Undo.InvalidGO(this);
}

bool
StackBar::SetSize(int select, double value)
{
	long i;

	switch(select & 0xfff){
	case SIZE_BAR:
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->SetSize(select, value);
		return true;
	case SIZE_BOX:		case SIZE_BOX_LINE:
		if(Boxes) for(i = 0; i < numPlots; i++) 
			if(Boxes[i]) Boxes[i]->SetSize(select, value);
		return true;
	}
	return false;
}

void
StackBar::DoPlot(anyOutput *o)
{
	long i;
	double dx, dy;
	fRECT oldREC;

	if(!o || !parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	dx = o->un2fix(dspm.fx);		dy = o->un2fiy(dspm.fy);
	memcpy(&oldREC, &o->Box1, sizeof(fRECT));
	if(Boxes) for(i = 0; i < numPlots; i++) if(Boxes[i]) {
		if(Boxes[i]->Id >= GO_PLOT && Boxes[i]->Id < GO_GRAPH) {
			if(((Plot*)Boxes[i])->hidden == 0) Boxes[i]->DoPlot(o);
			}
		else Boxes[i]->DoPlot(o);
		}
	if(xyPlots) for(i = numXY-1; i >= 0 ; i--) if(xyPlots[i]) {
		o->Box1.Xmin = oldREC.Xmin + dx*i;
		o->Box1.Ymin = oldREC.Ymin + dy*i;
		o->Box1.Xmax = oldREC.Xmax + dx*i;
		o->Box1.Ymax = oldREC.Ymax + dy*i;
		if(xyPlots[i]->Id >= GO_PLOT && xyPlots[i]->Id < GO_GRAPH) {
			if(((Plot*)xyPlots[i])->hidden == 0) xyPlots[i]->DoPlot(o);
			}
		else xyPlots[i]->DoPlot(o);
		}
	if(Polygons) for(i = 0; i < numPG; i++)
		if(Polygons[i]) Polygons[i]->DoPlot(o);
	if(Lines) for(i = numPL-1; i >= 0; i--){
		o->Box1.Xmin = oldREC.Xmin + dx*i;
		o->Box1.Ymin = oldREC.Ymin + dy*i;
		o->Box1.Xmax = oldREC.Xmax + dx*i;
		o->Box1.Ymax = oldREC.Ymax + dy*i;
		if(Lines[i]) Lines[i]->DoPlot(o);
		}
	dirty = false;
	memcpy(&o->Box1, &oldREC, sizeof(fRECT));
	if(use_xaxis || use_yaxis) parent->Command(CMD_AXIS, 0L, o);
}

bool
StackBar::Command(int cmd, void *tmpl, anyOutput *o)
{
	long i;
	static MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if (Boxes && !CurrGO) for (i = 0; i < numPlots; i++) {
				if (Boxes[i] && Boxes[i]->Command(cmd, tmpl, o))return true;
				}
			if (xyPlots && !CurrGO) for (i = 0; i < numXY; i++) {
				if (xyPlots[i] && xyPlots[i]->Command(cmd, tmpl, o))return true;
				}
			if (Polygons && !CurrGO) for (i = 0; i < numPG; i++) {
				if (Polygons[i] && Polygons[i]->Command(cmd, tmpl, o))return true;
				}
			if (Lines && !CurrGO) for (i = 0; i < numPL; i++) {
				if (Lines[i] && Lines[i]->Command(cmd, tmpl, o))return true;
				}
			break;
			}
		break;
	case CMD_OBJTREE:
		if(!tmpl) return false;
		if (Boxes) for (i = 0; i < numPlots; i++) if (Boxes[i]) {
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Boxes[i], 0L);
			}
		if (xyPlots) for (i = 0; i < numXY; i++) if (xyPlots[i]) {
			((ObjTree*)tmpl)->Command(CMD_UPDATE, xyPlots[i], 0L);
			}
		return true;
	case CMD_LEGEND:		case CMD_SCALE:
		if (cmd == CMD_SCALE) {
			dspm.fx *= ((scaleINFO*)tmpl)->sx.fy;
			dspm.fy *= ((scaleINFO*)tmpl)->sy.fy;
			}
		if (Boxes) for (i = 0; i < numPlots; i++){
			if (Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
			}
		if (Polygons) for (i = numPG - 1; i >= 0; i--) {
			if (Polygons[i]) Polygons[i]->Command(cmd, tmpl, o);
			}
		if (Lines) for (i = numPL - 1; i >= 0; i--) {
			if (Lines[i]) Lines[i]->Command(cmd, tmpl, o);
			}
		if (xyPlots) for (i = 0; i < numXY; i++) {
			if (xyPlots[i]) xyPlots[i]->Command(cmd, tmpl, o);
			}
		break;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_SETSCROLL:		case CMD_REDRAW:	case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_BAR_TYPE:
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_BARS:		case CMD_SAVE_BARS_CONT:
		if(Boxes) for(i = 0; i < numPlots; i++)
			if(Boxes[i])Boxes[i]->Command(CMD_SAVE_BARS_CONT, tmpl, o);
		return true;
	case CMD_AUTOSCALE:		case CMD_UPDATE:	case CMD_SET_DATAOBJ:
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_STACKBAR;
			if (data == tmpl) return true;
			data = (DataObj *)tmpl;
			}
		if(cmd == CMD_AUTOSCALE) {
			if(hidden) return false;
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			dirty = false;
			}
		if(cum_data_mode){
			if(CumData) delete CumData;
			CumData = CreaCumData(ssXrange, ssYrange, cum_data_mode, StartVal);
			if(CumData) {
				if (Polygons) for (i = 0; i < numPG; i++){
					if (Polygons[i]) Polygons[i]->Command(CMD_SET_DATAOBJ, CumData, o);
					}
				if (Lines) for (i = 0; i < numPL; i++) {
					if (Lines[i]) Lines[i]->Command(CMD_SET_DATAOBJ, CumData, o);
					}
				if (xyPlots) for (i = 0; i < numXY; i++) {
					if (xyPlots[i]) xyPlots[i]->Command(CMD_SET_DATAOBJ, CumData, o);
					}
				if (Boxes) for (i = 0; i < numPlots; i++) {
					if (Boxes[i]) Boxes[i]->Command(CMD_SET_DATAOBJ, CumData, o);
					}
				}
			}
		if(cmd == CMD_SET_DATAOBJ) tmpl = (void*) CumData;
		if(Polygons) for(i = 0; i < numPG; i++)
			if(Polygons[i]) Polygons[i]->Command(cmd, tmpl, o);
		if(Lines) for(i = 0; i < numPL; i++){
			if(Lines[i]) Lines[i]->Command(cmd, tmpl, o);
			}
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) {
				if(cmd == CMD_UPDATE) xyPlots[i]->Command(CMD_SET_DATAOBJ, data, o);
				if(cmd == CMD_AUTOSCALE && xyPlots[i]->Id >= GO_PLOT && xyPlots[i]->Id < GO_GRAPH) {
					if(((Plot*)xyPlots[i])->hidden == 0) xyPlots[i]->Command(cmd, tmpl, o);
					}
				else xyPlots[i]->Command(cmd, tmpl, o);
				}
		if (Boxes) for (i = 0; i < numPlots; i++){
			if (Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_BOX_TYPE:
		if (Boxes) for (i = 0; i < numPlots; i++) {
			if (Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_DELOBJ:
		if(o) o->HideMark();
		if(!tmpl || !parent) return false;
		if(DeleteGOL((GraphObj***)&Polygons, numPG, (GraphObj*)tmpl, o)) 
			return parent->Command(CMD_REDRAW, 0L, o);
		if(DeleteGOL((GraphObj***)&Lines, numPL, (GraphObj*)tmpl, o)) 
			return parent->Command(CMD_REDRAW, 0L, o);
		if(DeleteGOL((GraphObj***)&xyPlots, numXY, (GraphObj*)tmpl, o)) 
			return parent->Command(CMD_REDRAW, 0L, o);
		if(DeleteGOL((GraphObj***)&Boxes, numPlots, (GraphObj*)tmpl, o)) 
			return parent->Command(CMD_REDRAW, 0L, o);
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i] && xyPlots[i]->Command(cmd, tmpl, o)) return true;
		if(Boxes) for(i = 0; i < numPlots; i++)
			if(Boxes[i] && Boxes[i]->Command(cmd, tmpl, o)) return true;
		return false;
	case CMD_TAB:	case CMD_SHTAB:
		return false;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Phenology compares different indicators along a time axis
Phenology::Phenology(GraphObj *par, DataObj *d) :Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_PHENOLOG;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

Phenology::Phenology(int src) :Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	if (src == FILE_READ) {
		FileIO(FILE_READ);
		if (Plots) for (i = 0; i < nPlots; i++)	if (Plots[i]) Plots[i]->parent = this;
		}
	Id = GO_PHENOLOG;
}

Phenology::~Phenology()
{
	long i;

	if (ssXrange) free(ssXrange);
	if (ssYrange) free(ssYrange);
	if (Plots) {
		for (i = 0; i < nPlots; i++){
			if (Plots[i]) delete Plots[i];
			}
		free(Plots);	Plots = 0L;		nPlots = 0;
		}
	if (name) free(name);
	name = 0L;
	selRC_size = 0;
	if (selRC_rec) free(selRC_rec);
	selRC_rec = 0L;		selRC_curr = -1;	selRC_ref = -1;
}
void
Phenology::DoPlot(anyOutput *o)
{
	long i;

	if (parent && parent->Id == GO_GRAPH) {
		selRC_size = ((Graph*)parent)->NumAxes;
		if (selRC_rec) free(selRC_rec);
		selRC_rec = (RECT*)calloc(selRC_size, sizeof(RECT));
		}
	if (Plots) {
		for (i = 0; i < nPlots; i++) if (Plots[i]){
			Plots[i]->DoPlot(o);
			if (Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH && Plots[i]->use_yaxis){
				selRC_rec[i + 1].top = o->co2iy(((Graph*)parent)->Axes[Plots[i]->use_yaxis]->GetSize(SIZE_YPOS));
				selRC_rec[i + 1].bottom = o->co2iy(((Graph*)parent)->Axes[Plots[i]->use_yaxis]->GetSize(SIZE_YPOS + 1));
				selRC_rec[i + 1].left = o->co2ix(((Graph*)parent)->Axes[0]->GetSize(SIZE_XPOS));
				selRC_rec[i + 1].right = o->co2ix(((Graph*)parent)->Axes[0]->GetSize(SIZE_XPOS+1));
				}
			}
		}
}

bool
Phenology::Command(int cmd, void *tmpl, anyOutput *o) {
	long i;
	static MouseEvent *mev;
	static FillDEF bRCfill = { 0, 0x00f0f0f0, 1.0, 0L, 0x00ffffff };

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if (hidden) return false;
		mev = (MouseEvent *)tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if (Plots){
				if (selRC_curr > 0) parent->Command(CMD_REDRAW, 0L, o);
				selRC_curr = -1;	selRC_ref = -1;
				for (i = 0; i < nPlots; i++){
					if ((IsInRect(selRC_rec[i + 1], mev->x, mev->y)) && selRC_curr <1) {
						selRC_curr = i + 1;
						o->SetFill(&bRCfill);
						o->oSolidRectangle(selRC_rec[i + 1].left+4, selRC_rec[i + 1].top, selRC_rec[i + 1].right -2, selRC_rec[i + 1].bottom-4);
						Plots[i]->DoPlot(o);
						defs.Idle(CMD_UPDATE);
						}
					if (Plots[i] && Plots[i]->Command(cmd, tmpl, o))return true;
					}
				}
			break;
			}
		break;
	case CMD_REG_AXISPLOT:	//notification: plot can handle its own axes
		return parent->Command(cmd, tmpl, o);
	case CMD_AXIS:			//revert from own axis plot
		return parent->Command(cmd, tmpl, o);
	case CMD_AUTOSCALE:
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		if (Plots) for (i = 0; i < nPlots; i++)
			if (Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_OBJTREE:
		if (!tmpl) return false;
		if (Plots) for (i = 0; i < nPlots; i++) if (Plots[i])
			((ObjTree*)tmpl)->Command(CMD_UPDATE, Plots[i], 0L);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_PHENOLOG;
		data = (DataObj *)tmpl;
		((Graph*)parent)->Axes[1]->hidden = 1;				//hide master y-Axis
		if (Plots) for (i = 0; i < nPlots; i++) {
			if (Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_SCALE:
		if (Plots) for (i = 0; i < nPlots; i++) {
			if (Plots[i]) Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_TAB:		case CMD_SHTAB:
		return true;
	case CMD_MRK_DIRTY:
		dirty = true;
		return parent->Command(cmd, tmpl, o);
	case CMD_REDRAW:
		return parent->Command(cmd, tmpl, o);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Stacked polygons is based on stacked bar
StackPG::StackPG(GraphObj *par, DataObj *d):StackBar(par, d)
{
	Id = GO_STACKPG;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// waterfall plot is based on stacked bar
Waterfall::Waterfall(GraphObj *par, DataObj *d):StackBar(par, d)
{
	Id = GO_WATERFALL;
	dspm.fx = dspm.fy = 0.0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// multi data line plot is based on stacked bar
MultiLines::MultiLines(GraphObj *par, DataObj *d):StackBar(par, d)
{
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// simple pie chart
PieChart::PieChart(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_PIECHART;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

PieChart::PieChart(int src):Plot(0L, 0L)
{
	int i;
	
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in  children
		if(Segments) 
			for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->parent = this;
		}
}

PieChart::~PieChart()
{
	int i;

	if(Segments) {
		for(i = 0; i < nPts; i++) if(Segments[i]) DeleteGO(Segments[i]);
		free(Segments);		Segments = 0L;
		}
	if(ssRefA) free(ssRefA);
	if(ssRefR) free(ssRefR);
	ssRefA = ssRefR = 0L;
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

bool 
PieChart::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff) {
	case SIZE_XPOS:
	case SIZE_YPOS:
	case SIZE_RADIUS1:
	case SIZE_RADIUS2:
		if(Segments) for(i = 0; i < nPts; i++) {
			if(Segments[i]) Segments[i]->SetSize(select, value);
			}
		return true;
	default:
		return false;
		}
	return true;
}

void
PieChart::DoPlot(anyOutput *o)
{
	int i;

	if (Segments) for (i = 0; i < nPts; i++){
		if (Segments[i]) Segments[i]->DoPlot(o);
		}
}

bool
PieChart::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Segments && !CurrGO) for(i = nPts-1; i>=0; i--)
				if(Segments[i]) if(Segments[i]->Command(cmd, tmpl, o))break;
			break;
			}
		break;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_UPDATE:
		DoUpdate();
		return true;
	case CMD_SHIFT_OUT:		case CMD_SEG_FILL:		case CMD_SEG_LINE:
	case CMD_SEG_MOVEABLE:	case CMD_LEGEND:		case CMD_SCALE:
	case CMD_SET_DATAOBJ:
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_PIECHART;
			data = (DataObj *)tmpl;
			}
		if (cmd == CMD_SCALE) {
			CtDef.fx *= ((scaleINFO*)tmpl)->sx.fy;	CtDef.fy *= ((scaleINFO*)tmpl)->sx.fy;
			}
		if (Segments) for (i = 0; i < nPts; i++)
			if(Segments[i]) Segments[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		o->HideMark();
		if(Segments && parent) for(i = 0; i < nPts; i++) {
			if(Segments[i] && tmpl == (void *)Segments[i]) {
				Undo.DeleteGO((GraphObj**)(&Segments[i]), 0L, o);
				parent->Command(CMD_REDRAW, NULL, o);
				return true;
				}
			}
		break;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Segments, nPts, 0L);
		}
	return false;
}

void
PieChart::DoUpdate()
{
	AccRange *rY = 0L, *rR = 0L;
	double sum, fv, dang1, dang2;
	long i, ix, iy, rix, riy;

	if(ssRefA && (rY = new AccRange(ssRefA))) {
		SavVarObs((GraphObj **)Segments, nPts, UNDO_CONTINUE);
		if(ssRefR) rR = new AccRange(ssRefR);
		rY->GetFirst(&ix, &iy);				rY->GetNext(&ix, &iy);
		for(i = 0, sum = 0.0; i < nPts; i++){
			if(data->GetValue(iy, ix, &fv)) sum += fv;
			rY->GetNext(&ix, &iy);
			}
		sum /= CtDef.fy;
		dang1 = dang2 = CtDef.fx;
		rY->GetFirst(&ix, &iy);				rY->GetNext(&ix, &iy);
		if(rR) {
			rR->GetFirst(&rix, &riy);		rR->GetNext(&rix, &riy);
			}
		for(i = 0; i < nPts; i++){
			if(data->GetValue(iy, ix, &fv)) {
				dang2 -= (double)fv / sum;
				if(dang2 < 0.0) dang2 += 360.0;
				if(Segments[i]) {
					Segments[i]->SetSize(SIZE_ANGLE1, dang1);
					Segments[i]->SetSize(SIZE_ANGLE2, dang2);
					if(rR && data->GetValue(riy, rix, &fv)){
						fv *= FacRad;
						Segments[i]->SetSize(SIZE_RADIUS2, fv);
						}
					}
				dang1 = dang2;
				}
			rY->GetNext(&ix, &iy);
			if(rR) rR->GetNext(&rix, &riy);
			}
		}
	if(rY) delete rY;
	if(rR) delete rR;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// ring chart is based on piechart
RingChart::RingChart(GraphObj *par, DataObj *d):PieChart(par, d)
{
	Id = GO_RINGCHART;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a GoGroup contains objects NOT referring to data (e.g. drawing objects)
GoGroup::GoGroup(GraphObj *par, DataObj *d):Plot(par, d)
{
	Objects = 0L;
	nObs = 0;
	fPos.fx = fPos.fy = 0.0;
	Id = GO_GROUP;
}

GoGroup::GoGroup(int src):Plot(0L, 0L)
{
	long i;
	
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in  children
		if(Objects) 
			for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->parent = this;
		}
}

GoGroup::~GoGroup()
{
	long i;

	if(Objects && nObs) {
		for(i = 0; i < nObs; i++) if(Objects[i]) DeleteGO(Objects[i]);
		free(Objects);
		}
	if(name) free(name);
	name=0L;
	Undo.InvalidGO(this);
}

double
GoGroup::GetSize(int select)
{
	if(parent) switch(select){
	case SIZE_GRECT_TOP:
	case SIZE_GRECT_BOTTOM:
		return parent->GetSize(select)-fPos.fy;
	case SIZE_GRECT_LEFT:
	case SIZE_GRECT_RIGHT:
		return parent->GetSize(select)-fPos.fx;
	case SIZE_XPOS:
		return fPos.fx;
	case SIZE_YPOS:
		return fPos.fy;
		}
	return 0.0f;
}

void 
GoGroup::DoPlot(anyOutput *o)
{
	long i;
	double dx, dy;
	
	dx = o->un2fix(fPos.fx + (parent ? parent->GetSize(SIZE_GRECT_LEFT) : 0.0));
	dy = o->un2fiy(fPos.fy + (parent ? parent->GetSize(SIZE_GRECT_TOP) : 0.0));
	o->incVPorg(dx, dy);
	for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->DoPlot(o);
	o->incVPorg(-dx, -dy);
}

bool
GoGroup::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmp_go;
	MouseEvent *mev;
	long i;

	switch(cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Objects && !CurrGO) for(i = nObs-1; i>=0; i--)
				if(Objects[i]) if(Objects[i]->Command(cmd, tmpl, o))break;
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_GROUP;
		if(Objects) for(i = 0; i < nObs; i++)
			if(Objects[i]) Objects[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DROP_OBJECT:
		if(!Objects || nObs<1) {
			if((Objects = (GraphObj **)calloc(1, sizeof(GraphObj*)))){
				Objects[0] = (GraphObj *)tmpl;
				nObs = 1;
				return true;
				}
			}
		else if((tmp_go = (GraphObj **)realloc(Objects, (nObs+1)*sizeof(GraphObj*)))) {
			Objects = tmp_go;
			Objects[nObs++] = (GraphObj *)tmpl;
			return true;
			}
		break;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(CMD_REDRAW, tmpl, o);
		return false;
	case CMD_DELOBJ:
		if(Objects && parent) for(i = 0; i < nObs; i++) {
			o->HideMark();
			if(Objects[i] && tmpl == (void *)Objects[i]) {
				Undo.DeleteGO((GraphObj**)(&Objects[i]), 0L, o);
				parent->Command(CMD_REDRAW, NULL, o);
				return true;
				}
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// star chart
StarChart::StarChart(GraphObj *par, DataObj *d):GoGroup(par, d)
{
	Id = GO_STARCHART;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// three dimensional scatterplot
Scatt3D::Scatt3D(GraphObj *par, DataObj *d, DWORD flags):Plot(par, d)
{
	FileIO(INIT_VARS);
	c_flags = flags;
	Id = GO_SCATT3D;
}

Scatt3D::Scatt3D(GraphObj *par, DataObj *d, Brick **cols, long nob):Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	c_flags = 0L;		Id = GO_SCATT3D;
	Columns = cols;		nColumns = nob;
	if (Columns) for (i = 0; i < nColumns; i++) {
		if (Columns[i]) Columns[i]->parent = this;
		}
}

Scatt3D::Scatt3D(GraphObj *par, DataObj *d, Sphere **ba, long nob):Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	c_flags = 0L;		Id = GO_SCATT3D;
	Balls = ba;			nBalls = nob;
	if (Balls) for (i = 0; i < nBalls; i++) {
		if (Balls[i]) Balls[i]->parent = this;
		}
}

Scatt3D::Scatt3D(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Scatt3D::~Scatt3D()
{
	GraphObj **objects[] = {(GraphObj **)Balls, (GraphObj **)Columns, (GraphObj **)DropLines,
		(GraphObj **)Arrows, (GraphObj **)Errors};
	long n_obs[] = {nBalls, nColumns, nDropLines, nArrows, nErrors};
	long i, j;

	if(ssRefX) free(ssRefX);
	if(ssRefY) free(ssRefY);
	if(ssRefZ) free(ssRefZ);
	ssRefX = ssRefY = ssRefZ = 0L;
	Undo.InvalidGO(this);
	for(j = 0; j < 5; j++) {
		if(objects[j]) {
			for (i = 0; i < n_obs[j]; i++) {
				if (objects[j][i]) DeleteGO(objects[j][i]);
				}
			free(objects[j]);
			}
		}
	if(name) free(name);
	name=0L;
	if(data_desc) free(data_desc);
	data_desc = 0L;
}

double
Scatt3D::GetSize(int select)
{
	if(parent) return parent->GetSize(select);
	return 0.0;
}

bool
Scatt3D::SetSize(int select, double value)
{
	long i;

	switch(select & 0xfff) {
	case SIZE_SYM_LINE:		case SIZE_SYMBOL:
		if(Balls) for (i=0; i < nBalls; i++)
			if(Balls[i]) Balls[i]->SetSize(select, value);
		return true;
	case SIZE_BAR_BASE:		case SIZE_BAR_LINE:		case SIZE_BAR:
	case SIZE_BAR_DEPTH:
		if(Columns) for (i=0; i < nColumns; i++)
			if(Columns[i]) Columns[i]->SetSize(select, value);
		return true;
	case SIZE_ARROW_LINE:	case SIZE_ARROW_CAPWIDTH:	case SIZE_ARROW_CAPLENGTH:
		if(Arrows) for (i=0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->SetSize(select, value);
		return true;
	case SIZE_ERRBAR:		case SIZE_ERRBAR_LINE:
		if(Errors) for(i = 0; i < nErrors; i++) 
			if(Errors[i]) Errors[i]->SetSize(select, value);
		return true;
		}
	return false;
}

bool
Scatt3D::SetColor(int select, DWORD col)
{
	long i;

	switch(select) {
	case COL_SYM_LINE:			case COL_SYM_FILL:
		if(Balls) for (i=0; i < nBalls; i++)
			if(Balls[i]) Balls[i]->SetColor(select, col);
		return true;
	case COL_BAR_LINE:			case COL_BAR_FILL:
		if(Columns) for (i=0; i < nColumns; i++)
			if(Columns[i]) Columns[i]->SetColor(select, col);
		return true;
	case COL_ARROW:
		if(Arrows) for (i=0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->SetColor(select, col);
		return true;
	case COL_ERROR_LINE:
		if(Errors) for (i=0; i < nErrors; i++)
			if(Errors[i]) Errors[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
Scatt3D::DoPlot(anyOutput *o)
{
	GraphObj **objects[] = {(GraphObj **)Balls, (GraphObj **)Columns, (GraphObj **)DropLines,
		(GraphObj **)Arrows, (GraphObj **)Errors};
	long n_obs[] = {nBalls, nColumns, nDropLines, nArrows, nErrors};
	long i, j;
	RECT rc;

	if(!o || !parent) return;
	if(use_xaxis || use_yaxis || use_zaxis) ApplyAxes(o);
	o->GetSize(&rc);
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	rDims.left = rc.right;			rDims.right = rc.left;
	rDims.top = rc.bottom;			rDims.bottom = rc.top;
	for(j = 0; j < 5; j++) {
		if(objects[j]) for(i = 0; i < n_obs[j]; i++) {
			if(objects[j][i]) {
				objects[j][i]->DoPlot(o);
				UpdateMinMaxRect(&rDims, objects[j][i]->rDims.right, objects[j][i]->rDims.top);
				UpdateMinMaxRect(&rDims, objects[j][i]->rDims.left, objects[j][i]->rDims.bottom);
		 		}
			}
		}
	if(Line) {
		Line->DoPlot(o);
		UpdateMinMaxRect(&rDims, Line->rDims.right, Line->rDims.top);
		UpdateMinMaxRect(&rDims, Line->rDims.left, Line->rDims.bottom);
		}
	if(rib) {
		rib->DoPlot(o);
		UpdateMinMaxRect(&rDims, rib->rDims.right, rib->rDims.top);
		UpdateMinMaxRect(&rDims, rib->rDims.left, rib->rDims.bottom);
		}
	if(use_xaxis || use_yaxis || use_zaxis)parent->Command(CMD_AXIS, 0L, o);
	dirty = false;
}

bool
Scatt3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj ***objects[] = {(GraphObj ***)&Balls, (GraphObj ***)&Columns, (GraphObj ***)&DropLines,
		(GraphObj ***)&Arrows, (GraphObj ***)&Errors};
	long n_obs[] = {nBalls, nColumns, nDropLines, nArrows, nErrors};
	long i, j;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			for(j = 0; j < 5; j++) {
				if(*(objects[j])) for(i = 0; i < n_obs[j]; i++) {
					if((*objects[j])[i]) if((*objects[j])[i]->Command(cmd, tmpl, o))return true;
					}
				}
			break;
			}
		break;
	case CMD_SET_GO3D:		case CMD_SETSCROLL:		case CMD_REDRAW:
	case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_DL_LINE:		case CMD_DL_TYPE:
		if(DropLines) for(i = 0; i < nDropLines; i++)
			if(DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_ARROW_TYPE:	case CMD_ARROW_ORG3D:
		if(Arrows) for(i = 0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_USEAXIS:
		return UseAxis(*((int*)tmpl));
	case CMD_LEGEND:
		if(!tmpl) return false;
		if(Balls) {
			if(Line && Line->Id == GO_LINE3D) {
				for (i = 0; i < nBalls && i < 100; i++)
					if(Balls[i]) ((Legend*)tmpl)->HasSym(&Line->Line, Balls[i], 0L);
				}
			else {
				for (i = 0; i < nBalls && i < 100; i++) {
					if(Balls[i]) {
						if(Balls[i]->type) Balls[i]->Command(cmd, tmpl, o);
						else ((Legend*)tmpl)->HasSym(0L, Balls[i], 0L);
						}
					}
				}
			}
		else if(Line) Line->Command(cmd, tmpl, o);
		if(Columns) for(i = 0; i < nColumns; i++)
			if(Columns[i]) Columns[i]->Command(cmd, tmpl, o);
		if(rib) rib->Command(cmd, tmpl, o);
		return true;
	case CMD_UPDATE:	case CMD_SCALE:		case CMD_SET_DATAOBJ :
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_SCATT3D;
			data = (DataObj *)tmpl;
			}
		for(j = 0; j < 5; j++) {
			if(*objects[j]) {
				if(cmd == CMD_UPDATE || cmd == CMD_SCALE) SavVarObs(*objects[j], n_obs[j], UNDO_CONTINUE);
				for(i = 0; i < n_obs[j]; i++) {
					if((*objects[j])[i]) (*objects[j])[i]->Command(cmd, tmpl, o);
					}
				}
			}
		if(Line) Line->Command(cmd, tmpl, o);
		if(rib) rib->Command(cmd, tmpl, o);
		return true;
	case CMD_AUTOSCALE:
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			for(j = 0; j < 5; j++) {
				if(*objects[j]) for(i = 0; i < n_obs[j]; i++) {
					if((*objects[j])[i]) (*objects[j])[i]->Command(cmd, tmpl, o);
					}
				}
			if(Line) Line->Command(cmd, tmpl, o);
			if(rib) rib->Command(cmd, tmpl, o);
			}
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH &&
			xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy && zBounds.fx <= zBounds.fy){
			((Plot*)parent)->CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
			((Plot*)parent)->CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
			}
		dirty = false;
		return true;
	case CMD_ERR_TYPE:		case CMD_WHISKER_STYLE:		case CMD_ERRDESC:
		if(Errors) for(i = 0; i < nErrors; i++) {
			if(Errors[i]) Errors[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_DELOBJ:
		if(o) o->HideMark();
		if(!tmpl || !parent) return false;
		if(rib && rib->Command(cmd, tmpl, o)) return true;
		for(j = 0; j < 5; j++) if(DeleteGOL(objects[j], n_obs[j], (GraphObj*)tmpl, o)) 
			return parent->Command(CMD_REDRAW, 0L, o);
		if(Line && Line == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Line), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(rib && rib == tmpl) {
			Undo.DeleteGO((GraphObj**)(&rib), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		return false;
	case CMD_SYM_FILL:
		if(Balls) for(i= 0; i < nBalls; i++) if(Balls[i])Balls[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_BAR_FILL:
		if(Columns) for(i= 0; i < nColumns; i++) if(Columns[i])Columns[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Balls, nBalls, 0L);
	case CMD_SAVE_BARS:
		return SavVarObs((GraphObj **)Columns, nColumns, 0L);
	case CMD_SAVE_ARROWS:
		return SavVarObs((GraphObj **)Arrows, nArrows, 0L);
	case CMD_SAVE_ERRS:
		return SavVarObs((GraphObj **)Errors, nErrors, 0L);
	case CMD_SAVE_DROPLINES:
		return SavVarObs((GraphObj **)DropLines, nDropLines, 0L);
	case CMD_RECALC:
		if (Balls) for (i = 0; i < nBalls; i++) {
			if (Balls[i]) Balls[i]->Command(cmd, tmpl, o);
			}
		if (Columns) for (i = 0; i < nColumns; i++) {
			if (Columns[i]) Columns[i]->Command(cmd, tmpl, o);
			}
		if (DropLines) for (i = 0; i < nDropLines; i++) {
			if (DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
			}
		if (Arrows) for (i = 0; i < nArrows; i++) {
			if (Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
			}
		if (Errors) for (i = 0; i < nErrors; i++) {
			if (Errors[i]) Errors[i]->Command(cmd, tmpl, o);
			}
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// three dimensional ribbon based on list of Plane3D objects
Ribbon::Ribbon(GraphObj *par, DataObj *d, double z, double width, char *xr, char *yr)
	:Plot(par, d)
{
	FileIO(INIT_VARS);		Id = GO_RIBBON;		type = 1;
	if(xr && xr[0]) ssRefX = rlp_strdup(xr);
	if(yr && yr[0]) ssRefY = rlp_strdup(yr);
	z_value = z;	z_width = width;
}

Ribbon::Ribbon(GraphObj *par, DataObj *d, int which, char *xr, char *yr, char *zr)
	:Plot(par, d)
{
	FileIO(INIT_VARS);		Id = GO_RIBBON;			type = which;
	if(xr && xr[0]) ssRefX = rlp_strdup(xr);
	if(yr && yr[0]) ssRefY = rlp_strdup(yr);
	if(zr && zr[0]) ssRefZ = rlp_strdup(zr);
	CreateObs();
}

Ribbon::Ribbon(GraphObj *par, DataObj *d, GraphObj **go, int ngo)
	:Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);		Id = GO_RIBBON;		type = 3;
	planes = (Plane3D**)go;						nPlanes = ngo;
	for(i = 0; i < ngo; i++) planes[i]->parent = this;
}


Ribbon::Ribbon(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_RIBBON;
}

Ribbon::~Ribbon()
{
	int i;

	if(ssRefX) free(ssRefX);
	if(ssRefY) free(ssRefY);
	if(ssRefZ) free(ssRefZ);
	ssRefX = ssRefY = ssRefZ = 0L;
	Undo.InvalidGO(this);
	if(planes) {
		for (i = 0; i < nPlanes; i++) {
			if (planes[i]) DeleteGO(planes[i]);
			}
		free(planes);		planes = 0L;
		}
	if(values) free(values);
	values = 0L;	nVal = 0;		if(name) free(name);
	name=0L;						if(data_desc) free(data_desc);	
	data_desc = 0L;
}

double
Ribbon::GetSize(int select)
{
	switch(select) {
	case SIZE_CELLWIDTH:	return relwidth;
	case SIZE_ZPOS:			return z_value;
		}
	return 0.0;
}

bool
Ribbon::SetSize(int select, double value)
{
	int i;

	switch(select) {
	case SIZE_SYM_LINE:
		if(planes) for (i=0; i < nPlanes; i++)
			if(planes[i]) planes[i]->SetSize(select, value);
		return true;
	case SIZE_CELLWIDTH:
		if(value != relwidth) {
			//assume planes saved already by CMD_SAVE_SYMBOLS
			Undo.ValFloat(this, &relwidth, UNDO_CONTINUE);
			relwidth = value;
			if(planes) UpdateObs(false);
			}
		return true;
	case SIZE_ZPOS:
		if(value != z_value) {
			//assume planes saved already by CMD_SAVE_SYMBOLS
			Undo.ValFloat(this, &z_value, UNDO_CONTINUE);
			z_value = value;
			if(planes) UpdateObs(false);
			}
		return true;
		}
	return false;
}

bool
Ribbon::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_POLYGON:		case COL_POLYLINE:
		if (select == COL_POLYLINE) Line.color = col;
		if(select == COL_POLYGON) Fill.color = col;
		if (planes) for (i = 0; i < nPlanes; i++) {
			if (planes[i]) planes[i]->SetColor(select, col);
			}
		return true;
		}
	return false;
}

void
Ribbon::DoPlot(anyOutput *o)
{
	int i;

	if(!planes) CreateObs();
	if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->DoPlot(o);
	dirty = false;
}

bool
Ribbon::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(planes && !CurrGO) for(i = 0; i < nPlanes; i++)
				if(planes[i]) if(planes[i]->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_SCALE:
		z_value *= ((scaleINFO*)tmpl)->sz.fy;
		z_width *= ((scaleINFO*)tmpl)->sz.fy;
		for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->Command(cmd, tmpl, o);
		break;
	case CMD_SET_GO3D:		case CMD_SETSCROLL:		case CMD_REDRAW:
	case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) dirty = true;
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_RIBBON;
		if(tmpl) data = (DataObj *)tmpl;
		if(planes) for(i = 0; i < nPlanes; i++)
			if(planes[i]) planes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_UPDATE:
		SavVarObs((GraphObj **)planes, nPlanes, UNDO_CONTINUE);
		Undo.DataMem(this, (void**)&values, nVal * sizeof(fPOINT3D), &nVal, UNDO_CONTINUE);
		UpdateObs(dirty = true);
		if(parent) parent->Command(CMD_MRK_DIRTY, tmpl, o);
		return true;
	case CMD_AUTOSCALE:
		if(!planes) CreateObs();
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			if(planes) for(i = 0; i < nPlanes; i++)
				if(planes[i]) planes[i]->Command(cmd, tmpl, o);
			}
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH &&
			xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy && zBounds.fx <= zBounds.fy){
			((Plot*)parent)->CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
			((Plot*)parent)->CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
			}
		return true;
	case CMD_DELOBJ:
		if(!tmpl || !parent) return false;
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&planes[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		return false;
	case CMD_SYM_FILL:		case CMD_LEGEND:
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->Command(cmd, tmpl, o); 
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)planes, nPlanes, 0L);
		}
	return false;
}

void
Ribbon::CreateObs()
{
	long i, j, n, rx, cx, ry, cy, rz, cz;
	double fx, fy, fz, tmp;
	fPOINT3D pg[5];
	AccRange *rX, *rY, *rZ;
	Triangle *trl, *trc, *trn;

	if(planes || !data) return;
	rX = rY = rZ = 0L;
	switch(type) {
	case 1:
		if(!ssRefX || !ssRefY) return;
		if(z_width == 0.0) z_width = 1.0;
		if(relwidth == 0.0) relwidth = 0.6;
		if((rX = new AccRange(ssRefX)) && (rY = new AccRange(ssRefY))) {
			if(!data_desc) data_desc = rlp_strdup(rY->RangeDesc(data, 1));
			tmp = relwidth*z_width/2.0;
			if(!(values = (fPOINT3D*)calloc(i = rX->CountItems(), sizeof(fPOINT3D)))){
				delete rX;	delete rY;	return;
				}
			if(!(planes = (Plane3D**)calloc(i-1, sizeof(Plane3D*)))){
				free(values);	values = 0L;	delete rX;	delete rY;	return;
				}
			for(i = j = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry);
				rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry); i++) {
				if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy)) {
					values[i].fx = fx;	values[i].fy = fy;	values[i].fz = z_value;
					pg[3].fx = pg[2].fx = fx;	pg[3].fy = pg[2].fy = fy;
					pg[2].fz = z_value - tmp;	pg[3].fz = z_value +tmp;
					if(j) {
						pg[4].fx = pg[0].fx;	pg[4].fy = pg[0].fy; pg[4].fz = pg[0].fz;
						planes[i-1] = new Plane3D(this, data, pg, 5);
						if(planes[i-1]) planes[i-1]->Command(CMD_PG_FILL, &Fill, 0L);
						}
					j++;
					pg[0].fx = pg[3].fx;	pg[0].fy = pg[3].fy; pg[0].fz = pg[3].fz;
					pg[1].fx = pg[2].fx;	pg[1].fy = pg[2].fy; pg[1].fz = pg[2].fz;
					}
				}
			nPlanes = i-1;		nVal = i;
			}
		break;
	case 2:
		if(!ssRefX || !ssRefY || !ssRefZ) return;
		if((rX = new AccRange(ssRefX)) && (rY = new AccRange(ssRefY)) && (rZ = new AccRange(ssRefZ))) {
			if(!(values = (fPOINT3D*)calloc(i = rX->CountItems(), sizeof(fPOINT3D)))){
				delete rX;	delete rY;	delete rZ;	return;
				}
			if(!(planes = (Plane3D**)calloc(i-1, sizeof(Plane3D*)))){
				free(values);	values = 0L;	delete rX;	delete rY;	delete rZ;	return;
				}
			if(!data_desc) data_desc = rlp_strdup(rY->RangeDesc(data, 1));
			for(i = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry), rZ->GetFirst(&cz, &rz);
				rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && rZ->GetNext(&cz, &rz); i++) {
				if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy) &&
					data->GetValue(rz, cz, &fz)) {
					values[i].fx = fx;	values[i].fy = fy;	values[i].fz = fz;
					pg[3].fx = pg[2].fx = fx;	pg[2].fz = pg[3].fz = fz;		
					pg[3].fy = 0.0;				pg[2].fy = fy;
					if(i) {
						pg[4].fx = pg[0].fx;	pg[4].fy = pg[0].fy; pg[4].fz = pg[0].fz;
						planes[i-1] = new Plane3D(this, data, pg, 5);
						if(planes[i-1]) planes[i-1]->Command(CMD_PG_FILL, &Fill, 0L);
						}
					pg[0].fx = pg[3].fx;		pg[0].fy = pg[3].fy;	pg[0].fz = pg[3].fz;
					pg[1].fx = pg[2].fx;		pg[1].fy = pg[2].fy;	pg[1].fz = pg[2].fz;
					}
				}
			nPlanes = i-1;		nVal = i;
			}
		break;
	case 3:
		if(!ssRefX || !ssRefY || !ssRefZ) break;
		Undo.InvalidGO(this);
		trl = Triangulate1(ssRefX, ssRefZ, ssRefY, data);
		if (!trl) {
			ErrorBox((char*)SCMS_INVALID_DATA);
			return;
			}
		for(i = 0, trc = trl; trc; i++) trc = trc->next;
		if((n = i) && (planes = (Plane3D**)malloc(n*sizeof(Plane3D*)))) 
			for(i = nPlanes = 0, trc = trl; trc && i < n; i++) {
			for(j = 0; j < 4; j++) {	//swap y and z values;
				tmp = trc->pt[j].fz;	trc->pt[j].fz = trc->pt[j].fy;	trc->pt[j].fy = tmp;
				}
			planes[nPlanes++] = new Plane3D(this, data, trc->pt, 4);
			trn = trc->next;	delete trc;		trc = trn;
			}
		dirty = true;			Command(CMD_AUTOSCALE, 0L, 0L);
		break;
		}
	if(rX) delete rX;
	if(rY) delete rY;
	if(rZ) delete rZ;
}

void
Ribbon::UpdateObs(bool bNewData)
{
	long i, j, k, rx, cx, ry, cy, rz, cz;
	double fx, fy, fz, tmp, da1, da2;
	AccRange *rX, *rY, *rZ;
	int sel_id[] = {SIZE_XPOS, SIZE_YPOS, SIZE_ZPOS};

	if(!planes || (!values && type < 3)) return;
	rX = rY = rZ = 0L;
	switch(type) {
	case 1:
		if(!ssRefX || !ssRefY || !data) return;
		if(z_width == 0.0) z_width = 1.0;
		if(relwidth == 0.0) relwidth = 0.6;
		if((rX = new AccRange(ssRefX)) && (rY = new AccRange(ssRefY))) {
			tmp = relwidth*z_width/2.0;
			for(i = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry);
				rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && i < nVal; i++) {
				if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy)) {
					if(bNewData) {
						values[i].fx = fx;	values[i].fy = fy;	values[i].fz = z_value;
						}
					else {
						fx = values[i].fx;	fy = values[i].fy;	values[i].fz = z_value;
						}
					if(i && planes[i-1]) {
						for(j = 0; j < 3; j++){
							for(k = 0; k <5; k++) {
								switch (j) {
								case 0:	
									da1 = values[i-1].fx;		da2 = values[i].fx;
									break;
								case 1:
									da1 = values[i-1].fy;		da2 = values[i].fy;
									break;
								case 2:
									if(k != 1 && k != 2) da1 = da2 = (values[i].fz + tmp);
									else da1 = da2 = (values[i].fz - tmp);
									break;
									}
								planes[i-1]->SetSize(sel_id[j]+k, (k != 2 && k != 3) ? da1 : da2);
								}
							}
						}
					}
				}
			}
		break;
	case 2:
		if(!ssRefX || !ssRefY || !ssRefZ || !data) return;
		if((rX = new AccRange(ssRefX)) && (rY = new AccRange(ssRefY)) && (rZ = new AccRange(ssRefZ))) {
			for(i = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry), rZ->GetFirst(&cz, &rz);
				rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && rZ->GetNext(&cz, &rz) && i < nVal; i++) {
				if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy) && data->GetValue(rz, cz, &fz)) {
					values[i].fx = fx;	values[i].fy = fy;	values[i].fz = fz;
					if(i && planes[i-1]) {
						planes[i-1]->SetSize(SIZE_XPOS, values[i-1].fx);	planes[i-1]->SetSize(SIZE_XPOS+3, fx);
						planes[i-1]->SetSize(SIZE_XPOS+1, values[i-1].fx);	planes[i-1]->SetSize(SIZE_XPOS+2, fx);
						planes[i-1]->SetSize(SIZE_XPOS+4, values[i-1].fx);
						planes[i-1]->SetSize(SIZE_YPOS+1, values[i-1].fy);	planes[i-1]->SetSize(SIZE_YPOS+2, fy);
						planes[i-1]->SetSize(SIZE_ZPOS, values[i-1].fz);	planes[i-1]->SetSize(SIZE_ZPOS+3, fz);
						planes[i-1]->SetSize(SIZE_ZPOS+1, values[i-1].fz);	planes[i-1]->SetSize(SIZE_ZPOS+2, fz);
						planes[i-1]->SetSize(SIZE_ZPOS+4, values[i-1].fz);
						}
					}
				}
			}
		break;
	case 3:
		if(planes) {
			for(i = 0; i < nPlanes; i++) if(planes[i]) DeleteGO(planes[i]);
			free(planes);		planes = 0L;	nPlanes = 0;
			}
		CreateObs();
		}
	if(rX) delete rX;
	if(rY) delete rY;
	if(rZ) delete rZ;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// draw a 3dimensional grid
Grid3D::Grid3D(GraphObj *par, DataObj *d, int sel, double x1, double xstep, double z1, double zstep)
	:Plot(par, d)
{
	FileIO(INIT_VARS);		Id = GO_GRID3D;
	start.fx = x1;			step.fx = xstep;
	start.fz = z1;			step.fz = zstep;
	type = sel;
}

Grid3D::Grid3D(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_GRID3D;
}

Grid3D::~Grid3D()
{
	int i;

	Undo.InvalidGO(this);
	if(lines) {
		for(i = 0; i < nLines; i++) if(lines[i]) DeleteGO(lines[i]);
		free(lines);		lines = 0L;
		}
	if(planes) {
		for(i = 0; i < nPlanes; i++) if(planes[i]) DeleteGO(planes[i]);
		free(planes);		planes = 0L;
		}
	nLines = nPlanes = 0;
	if(name) free(name);
	name=0L;
}

bool 
Grid3D::SetSize(int select, double value)
{
	int i;

	switch (select) {
	case SIZE_SYM_LINE:
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->SetSize(select, value); 
		return true;
		}
	return false;
}

bool
Grid3D::SetColor(int select, DWORD col)
{
	int i;

	switch (select) {
	case COL_POLYLINE:
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->SetColor(select, col); 
		return true;
		}
	return false;
}

void
Grid3D::DoPlot(anyOutput *o)
{
	long i;

	if(!lines && !planes) CreateObs(false);
	if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->DoPlot(o);
	if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->DoPlot(o);
	dirty = false;
}

bool
Grid3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	if (!o && parent && parent->Id == GO_PLOT3D) o = ((Plot3D*)parent)->getDisp();
	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(!o || hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(lines && !CurrGO) for(i = 0; i < nLines; i++)
				if(lines[i]) if(lines[i]->Command(cmd, tmpl, o)) return true;
			if(planes && !CurrGO) for(i = 0; i < nPlanes; i++)
				if(planes[i]) if(planes[i]->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_SET_LINE:
		if (!o) return false;
		if(tmpl) {
			memcpy(&Line, tmpl, sizeof(LineDEF));
			if(lines) {
				SavVarObs((GraphObj**)lines, nLines, 0L);
				for(i = 0; i < nLines; i++)
					if(lines[i]) lines[i]->Command(cmd, tmpl, o);
				}
			if(planes) {
				SavVarObs((GraphObj**)planes, nPlanes, 0L);
				for(i = 0; i < nPlanes; i++)
					if(planes[i]) planes[i]->Command(cmd, tmpl, o);
				}
			}
		break;
	case CMD_LEGEND:
		if(!hidden) ((Legend*)tmpl)->HasFill(&Line, planes ? &Fill : 0L, 0L);
		break;
	case CMD_CONFIG:
		return Configure();
	case CMD_MRK_DIRTY:
		dirty = true;
		if (o && parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_GO3D:		case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_UPDATE:	case CMD_SET_DATAOBJ:
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_GRID3D;
			tmpl = data;
			}
		if(lines) for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->Command(cmd, tmpl, o);
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->Command(cmd, tmpl, o);
		return dirty = (cmd == CMD_UPDATE);
	case CMD_AUTOSCALE:
		if(!lines && !planes) CreateObs(false);
		Id = GO_GRID3D;
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			if(lines) for(i = 0; i < nLines; i++)
				if(lines[i]) lines[i]->Command(cmd, tmpl, o);
			if(planes) for(i = 0; i < nPlanes; i++)
				if(planes[i]) planes[i]->Command(cmd, tmpl, o);
			dirty = false;
			}
		if(zBounds.fx > zBounds.fy) zBounds.fx = zBounds.fy = 0.0;
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH &&
			xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy){
			((Plot*)parent)->CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
			((Plot*)parent)->CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
			}
		return true;
	case CMD_SYM_FILL:
		if(!tmpl) return false;
		memcpy(&Fill, tmpl, sizeof(FillDEF));
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->Command(cmd, tmpl, o); 
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)planes, nPlanes, 0L);
	case CMD_DELOBJ:
		if(!parent || !o) return false;
		if(DeleteGOL((GraphObj***)&lines,nLines,(GraphObj*)tmpl,o)) return parent->Command(CMD_REDRAW, 0L, o);
		if(DeleteGOL((GraphObj***)&planes,nPlanes,(GraphObj*)tmpl,o)) return parent->Command(CMD_REDRAW, 0L, o);
		break;
		}
	return false;
}

void
Grid3D::CreateObs(bool set_undo)
{
	int i, ir, ic, idx;
	long w, h;
	static fPOINT3D vec[6];

	if(!parent || !data || lines || planes) return;
	dirty = true;
	if(type == 0) {
		data->GetSize(&w, &h);
		if(0 >= (nLines = 2 * w * h - w - h)) return;
		if(!(lines =(Line3D**)calloc(nLines+2, sizeof(Line3D*)))) return;
		vec[0].fx = start.fx;			data->GetValue(0, 0, &vec[0].fy);
		for(ic = 1, idx = 0; ic <= w; ic++) {
			vec[0].fz = start.fz;
			data->GetValue(0, ic-1, &vec[0].fy);
			for(ir = 1; ir <= h; ir++){
				if(ic < w && data->GetValue(ir-1, ic, &vec[1].fy)) {
					vec[1].fz = vec[0].fz;	vec[1].fx = vec[0].fx + step.fx;
					lines[idx++] = new Line3D(this, data, vec, 2, 
						-1, -1, ic-1, ir-1, -1, -1, -1, -1, ic, ir-1, -1, -1);
					}
				if(ir < h && data->GetValue(ir, ic-1, &vec[1].fy)) {
					vec[1].fz = vec[0].fz + step.fz;	vec[1].fx = vec[0].fx;
					lines[idx++] = new Line3D(this, data, vec, 2,
						-1, -1, ic-1, ir-1, -1, -1, -1, -1, ic-1, ir, -1, -1);
					}
				vec[0].fz += step.fz;	vec[0].fy = vec[1].fy;
				}
			vec[0].fx += step.fx;
			}
		for(i = 0; i < nLines; i++) if(lines[i]) lines[i]->Command(CMD_SET_LINE, &Line, 0L);
		}
	else if(type == 1) {
		vec[0].fz = vec[4].fz = vec[1].fz = start.fz;
		vec[3].fz = vec[2].fz = (start.fz +step.fz);
		data->GetSize(&w, &h);
		if(0 >= (nPlanes = w * h)) return;
		if(!(planes =(Plane3D**)calloc(nPlanes, sizeof(Plane3D*)))) return;
		vec[0].fx = vec[3].fx = vec[4].fx = start.fx;
		vec[1].fx = vec[2].fx = (start.fx + step.fx);
		for(ic = 1, idx = 0; ic <= w; ic++) {
			vec[0].fz = vec[4].fz = vec[1].fz = start.fz;
			vec[3].fz = vec[2].fz = (start.fz + step.fz);
			data->GetValue(0, ic-1, &vec[0].fy);	data->GetValue(0, ic, &vec[1].fy);
			vec[4].fy = vec[0].fy;
			for(ir = 1; ir <= h; ir++){
				if(ic < w && ir < h && data->GetValue(ir, ic, &vec[2].fy) 
					&& data->GetValue(ir, ic-1, &vec[3].fy)) {
					planes[idx++] = new Plane3D(this, 0L, vec, 5);
					}
				vec[0].fz += step.fz;		vec[1].fz += step.fz;		vec[2].fz += step.fz;
				vec[3].fz += step.fz;		vec[4].fz += step.fz;
				vec[0].fy = vec[4].fy = vec[3].fy;						vec[1].fy = vec[2].fy;
				}
			
			vec[0].fx += step.fx;			vec[1].fx += step.fx;		vec[2].fx += step.fx;
			vec[3].fx += step.fx;			vec[4].fx += step.fx;
			}
		nPlanes = idx;
		for(i = 0; i < nPlanes; i++) if(planes[i]){
			planes[i]->Command(CMD_SET_LINE, &Line, 0L);
			planes[i]->Command(CMD_SYM_FILL, &Fill, 0L);
			}
		SetSize(SIZE_SYM_LINE, Line.width);		SetColor(COL_POLYLINE, Line.color);
		}
	if(set_undo) {
		if(planes && nPlanes)Undo.StoreListGO(parent, (GraphObj***)&planes, &nPlanes, UNDO_CONTINUE);
		if(lines && nLines)Undo.StoreListGO(parent, (GraphObj***)&lines, &nLines, UNDO_CONTINUE);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define minima and maxima rectangle to be used by graph
Limits::Limits(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Limits::~Limits()
{
	if(name) free(name);
	name=0L;
}

double
Limits::GetSize(int)
{
	return 0.0;
}

bool
Limits::Command(int cmd, void *tmpl, anyOutput *)
{
	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_LIMITS;
		data = (DataObj *)tmpl;	
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a user defined function
Function::Function(GraphObj *par, DataObj *d, char *desc):Plot(par, d)
{
	FileIO(INIT_VARS);		cmdxy = (char*)malloc(20*sizeof(char));
	if(parent && parent->Id == GO_POLARPLOT) {
		x1 = 0.0;			x2 = 360.0;			xstep = 0.5;
		if(cmdxy)rlp_strcpy(cmdxy, 20, (char*)"sin(pi*x/30)+1.1");
		}
	else {
		x1 = 0.0;			x2 = 100.0;			xstep = 0.5;
		if(cmdxy)rlp_strcpy(cmdxy, 20, (char*)"sin(x)/x");
		}
	if(desc) name = rlp_strdup(desc);
	Id = GO_FUNCTION;
}

Function::Function(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_FUNCTION;
}

Function::~Function()
{
	if(cmdxy) free(cmdxy);
	cmdxy = 0L;					if(param) free(param);
	param = 0L;					if(dl) DeleteGO(dl);
	dl = 0L;					if(name) free(name);
	name=0L;
}

bool
Function::SetSize(int select, double value)
{
	switch(select & 0xfff){
	case SIZE_MIN_X:	x1 = value;		return true;
	case SIZE_MAX_X:	x2 = value;		return true;
	case SIZE_XSTEP:	xstep=value;	return true;
		}
	return false;
}

void
Function::DoPlot(anyOutput *o)
{
	if (hidden) return;
	if((!dl || dirty) && cmdxy && cmdxy[0]) Update(o, 0);
	dirty = false;
	if(dl && o) {
		dl->Command(CMD_SET_LINE, &Line, o);
		dl->DoPlot(o);
		}
}

bool
Function::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_CONFIG:
		PropertyDlg();
		break;
	case CMD_LEGEND:	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		if(dl) return dl->Command(cmd, tmpl, o);
		break;
	case CMD_SCALE:
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_DELOBJ:
		if(parent && tmpl && tmpl == dl) return parent->Command(CMD_DELOBJ, this, o);
		break;
	case CMD_MRK_DIRTY:
		return dirty = true;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_LINE:
		if(tmpl) memcpy(&Line, tmpl, sizeof(LineDEF));
		break;
	case CMD_SET_DATAOBJ:
		if(dl) dl->Command(cmd, tmpl, o);
		Id = GO_FUNCTION;
		data = (DataObj *)tmpl;
		return true;
	case CMD_SETPARAM:
		if(tmpl) {
			if(param) free(param);
			param = 0L;
			if(*((char*)tmpl))param = rlp_strdup((char*)tmpl);
			}
		dirty = true;
		return true;
	case CMD_SETFUNC:
		if(tmpl) {
			if(cmdxy) free(cmdxy);
			cmdxy = 0L;
			if(*((char*)tmpl))cmdxy = rlp_strdup((char*)tmpl);
			}
		dirty = true;
		return true;
	case CMD_UPDATE:
		return Update(o, UNDO_CONTINUE);
	case CMD_OBJTREE:
//		((ObjTree*)tmpl)->Command(CMD_UPDATE, this, 0L);
		return true;
	case CMD_AUTOSCALE:
		if(!dl) return Update(o, 0L);
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			if(dl) dl->Command(cmd, tmpl, o);
			dirty = false;
			}
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH
			&& Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) {
			((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
			((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
			}
		return true;
		}
	return false;
}
void
Function::CheckBounds(double x, double y)
{
	//we come here from DataLine only
	if (x < Bounds.Xmin) Bounds.Xmin = x;
	if (x > Bounds.Xmax) Bounds.Xmax = x;
	if (y < Bounds.Ymin) Bounds.Ymin = y;
	if (y > Bounds.Ymax) Bounds.Ymax = y;
}

bool
Function::Update(anyOutput *o, DWORD)
{
	lfPOINT *xydata;
	long ndata;

	if(!parent || !cmdxy ) return false;
	LockData(false, false);
	do_xyfunc(data, x1, x2, xstep, cmdxy, &xydata, &ndata, param);
	LockData(false, false);
	if(xydata && ndata >1) {
		if(!dl) dl = new DataLine(this, data, xydata, ndata, name);
		else dl->LineData(xydata, ndata);
		dirty = true;
		Command(CMD_AUTOSCALE, 0L, o);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a user defined function
static char *lastFunc2D = 0L, *lastParam2D=0L;
FitFunc::FitFunc(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	x1 = 0.0;			x2 = 100.0;					xstep = 0.5;		dl = 0L;
	if(lastFunc2D && lastFunc2D[0] && lastParam2D && lastParam2D[0]) {
		cmdxy = rlp_strdup(lastFunc2D);
		parxy = rlp_strdup(lastParam2D);
		}
	if(!cmdxy || !parxy) {
		cmdxy = (char*)malloc(20*sizeof(char));		parxy = (char*)malloc(20*sizeof(char));
		if (cmdxy) rlp_strcpy(cmdxy, 20, (char*)"(a*x) / (b + x)");
		if (parxy) rlp_strcpy(parxy, 20, (char*)"a = 0.2; b = 3;");
		}
	x_info = 0L;		y_info = 0L;
	Id = GO_FITFUNC;
}


FitFunc::FitFunc(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_FITFUNC;
}

FitFunc::~FitFunc()
{
	int i;

	if(Symbols) {
		for(i = 0; i< nPoints; i++) if(Symbols[i]) DeleteGO(Symbols[i]);
		free(Symbols);
		}
	if(cmdxy) free(cmdxy);
	cmdxy = 0L;					if(parxy) free(parxy);
	parxy = 0L;					if(ssXref) free(ssXref);
	ssXref = 0L;				if(ssYref) free(ssYref);
	ssYref = 0L;				if(dl) DeleteGO(dl);
	dl = 0L;					if(name) free(name);
	name = 0L;					Undo.InvalidGO(this);
}

bool
FitFunc::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:
	case SIZE_SYM_LINE:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
		}
	return false;
}

bool
FitFunc::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
FitFunc::DoPlot(anyOutput *o)
{
	int i;

	if(!data || x1 >= x2) return;
	dirty = false;
	if(!dl && (dl = new Function(this, data, (char*)"Fitted function"))) {
		dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
		dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
		dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
		dl->Update(o, UNDO_CONTINUE);
		}
	if(dl && o) {
		dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
		dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
		dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
		dl->DoPlot(o);
		}
	if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->DoPlot(o);
}

bool 
FitFunc::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;
	LineDEF *ld;

	switch(cmd) {
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND && dl && dl->Id == GO_FUNCTION) {
			ld = dl->GetLine();
			if(Symbols) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(ld, Symbols[i], (char*)"Fitted function");
				}
			else ((Legend*)tmpl)->HasFill(ld, 0L, dl->name);
			return true;
			}
		return false;
	case CMD_ENDDIALOG:
		if(!cmdxy || !parxy) return false;
		if((i = rlp_strlen(cmdxy))) {
			lastFunc2D = (char*)realloc(lastFunc2D, i + 2);
			if (lastFunc2D) rlp_strcpy(lastFunc2D, i+1, cmdxy);
			}
		if((i = rlp_strlen(parxy))) {
			lastParam2D = (char*)realloc(lastParam2D, i + 2);
			if (lastParam2D) rlp_strcpy(lastParam2D, i+1, parxy);
			}
		return true;
	case CMD_SCALE:
		if(dl) return dl->Command(cmd, tmpl, o);
		if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		Line.patlength *= ((scaleINFO*)tmpl)->sy.fy;		Line.width *= ((scaleINFO*)tmpl)->sy.fy;
		return true;
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Symbols && !CurrGO) for(i = nPoints-1; i >=0; i--)
				if(Symbols[i] && Symbols[i]->Command(cmd, tmpl, o))return true;
			break;
			}
		if(dl) return dl->Command(cmd, tmpl, o);
		return false;
	case CMD_AUTOSCALE:
		if(dirty) {
			if(!dl && (dl = new Function(this, data, (char*)"Fitted function"))) {
				dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
				dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
				dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
				dl->Update(o, UNDO_CONTINUE);
				}
			if(dl) {
				dl->Command(cmd, tmpl, o);
				memcpy(&Bounds, &dl->Bounds, sizeof(fRECT));
				}
			if (Symbols) for (i = 0; i < nPoints; i++) {
				if (Symbols[i])Symbols[i]->Command(cmd, tmpl, o);
				}
			dirty = false;
			}
		return true;
	case CMD_UPDATE:
		if(Symbols) {
			SavVarObs((GraphObj**)Symbols, nPoints, UNDO_CONTINUE);
			for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			}
		if (PropertyDlg()){
			if (!dl) dl = new Function(this, data, (char*)"Fitted function");
			if (dl){
				dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
				dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
				dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
				dl->Update(o, UNDO_CONTINUE);
				}
			dirty = true;
			if (parent) parent->Command(CMD_MRK_DIRTY, 0L, o);
			}
		return true;
	case CMD_DELOBJ:
		if(!parent) return false;
		if(tmpl && tmpl == dl) return parent->Command(CMD_DELOBJ, this, o);
		else if(DeleteGOL((GraphObj***)&Symbols,nPoints,(GraphObj*)tmpl,o)) return parent->Command(CMD_REDRAW,0L,o);
		else if(dl) return dl->Command(cmd, tmpl, o);
		return false;
	case CMD_SET_DATAOBJ:		case CMD_REDRAW:	case CMD_MRK_DIRTY:
		if (cmd == CMD_REDRAW) {
			if (parent) return parent->Command(cmd, tmpl, o);
			}
		if (cmd == CMD_MRK_DIRTY) {
			dirty = true;
			if (dl){
				dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
				dl->SetSize(SIZE_XSTEP, xstep);
				}
			}
		if (cmd == CMD_SET_DATAOBJ) {
			Id = GO_FITFUNC;
			if (tmpl) data = (DataObj *)tmpl;
			}
		if(dl) dl->Command(cmd, tmpl, o);
		if (Symbols) for (i = 0; i < nPoints; i++) {
			if (Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_SYMTEXT:		case CMD_SYMTEXT_UNDO:	case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// normal quantile plot and derivates
NormQuant::NormQuant(GraphObj *par, DataObj *d, char* range)
	:Plot(par, d)
{
	FileIO(INIT_VARS);
	if(range && range[0]) ssRef = rlp_strdup(range);
	if (ssRef) ProcessData();
	else ssRef = 0L;
	Id = GO_NORMQUANT;
}

NormQuant::NormQuant(GraphObj *par, DataObj *d, double *val, int nval, char *Ref)
	:Plot(par, d)
{
	FileIO(INIT_VARS);		ssRef = 0L;
	if(val && nval) {
		src_data = (double*)memdup(val, nval*sizeof(double), 0);
		SortArray(nData = nval, src_data);		ProcessData();
		}
	if (Ref && Ref[0]) ssRef = rlp_strdup(Ref);
	Id = GO_NORMQUANT;
}

NormQuant::NormQuant(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		if(nData)SortArray(nData, src_data);
		ProcessData();
		}
}

NormQuant::~NormQuant()
{
	if(ssRef) free(ssRef);
	ssRef = 0L;					if(x_vals) free(x_vals);
	x_vals = 0L;				if(y_vals) free(y_vals);
	y_vals = 0L;				if(src_data)free(src_data);
	src_data = 0L;				if(sy)delete(sy);
}

void
NormQuant::DoPlot(anyOutput *o)
{
	int i;

	if (!sy) {
		sy = new Symbol(this, data, 0.0, 0.0, SYM_CIRCLE);
		sy->SetSize(SIZE_SYMBOL, NiceValue(defs.GetSize(SIZE_SYMBOL)*0.75));
		}
	//draw symbols
	if(sy && y_vals && src_data && y_vals && nValidData) {
		for(i = 0; i < nValidData; i++) {
			sy->SetSize(SIZE_XPOS, x_vals[i]);
			sy->SetSize(SIZE_YPOS, y_vals[i]);
			sy->DoPlot(o);
			}
		}
}

bool
NormQuant::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch(cmd) {
	case CMD_LEGEND:
		if (sy) ((Legend*)tmpl)->HasSym(0L, sy, x_info ? x_info : (char*)SDLG_NQUANT_LEGEND);
		return true;
	case CMD_SCALE:
		return true;
	case CMD_UPDATE:
		return true;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_NORMQUANT;
		if(sy) sy->Command(cmd, tmpl, o);
		data = (DataObj *)tmpl;
		return true;
		}
	return false;
}

bool
NormQuant::ProcessData()
{
	long i, r, c, n;
	AccRange *rD;
	double y, dtmp, sum;

	if(data && ssRef && ssRef[0] && (rD = new AccRange(ssRef))) {
		if((n = rD->CountItems()) && (src_data = (double*)realloc(src_data, n * sizeof(double)))){
			for(nData = 0, rD->GetFirst(&c, &r); rD->GetNext(&c, &r); ) {
				if(data->GetValue(r, c, &dtmp)) src_data[nData++] = dtmp;
				}
			if(nData)SortArray(nData, src_data);
			}
		y_info = (char*)malloc(20);
		if (y_info){
			rlp_strcpy(y_info, 20, (char*)SDLG_NQUANT_YAXIS);
			}
		x_info = rlp_strdup(rD->RangeDesc(data, 2));
		delete rD;
		}
	if(src_data && nData) {
		Bounds.Ymin = HUGE_VAL;			Bounds.Ymax = -HUGE_VAL;
		if(!(x_vals = (double*)realloc(x_vals, nData * sizeof(double))))return false;
		if(!(y_vals = (double*)realloc(y_vals, nData * sizeof(double))))return false;
		for(n = i = 0, sum = dtmp = 1.0/((double)nData); i < (nData-1); i++ ) {
			y = distinv(norm_dist, 0.0, 1.0, sum, 0.5);
			if(y > -HUGE_VAL && y < HUGE_VAL) {
				y_vals[n] = y;			x_vals[n] = src_data[i];
				if(y < Bounds.Ymin) Bounds.Ymin = y;
				if(y > Bounds.Ymax) Bounds.Ymax = y;
				n++;
				}
			sum += dtmp;
			}
		Bounds.Xmax = src_data[nData-1];	Bounds.Xmin = src_data[0];
		if(Bounds.Ymax <= Bounds.Ymin) {
			Bounds.Ymin = -5.0;		Bounds.Ymax = 5.0;
			}
		nValidData = n;
		return (n > 3);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Contour Plot
ContourPlot::ContourPlot(GraphObj *par, DataObj *d)
	:Plot(par, d)
{
	FileIO(INIT_VARS);		Id = GO_CONTOUR;
}

ContourPlot::ContourPlot(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	Id = GO_CONTOUR;
}

ContourPlot::~ContourPlot()
{
	int i;

	if(Symbols) {
		for (i = 0; i < nSym; i++) {
			if (Symbols[i]) DeleteGO(Symbols[i]);
			}
		free(Symbols);
		Symbols = 0L;		nSym = 0;
		}
	if(Labels) {
		for (i = 0; i < nLab; i++) {
			if (Labels[i]) DeleteGO(Labels[i]);
			}
		free(Labels);
		Labels = 0L;		nLab = 0;
		}
	if(zAxis)			DeleteGO(zAxis);
	if(ssRefX) free(ssRefX);
	if(ssRefY) free(ssRefY);
	if(ssRefZ) free(ssRefZ);
	ssRefX = ssRefY = ssRefZ = 0L;
	Undo.InvalidGO(this);
	if(name) free(name);
	name=0L;			if(val) free(val);
	val = 0L;
}

bool
ContourPlot::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:		case SIZE_SYM_LINE:
		if(Symbols) for(i = 0; i < nSym; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	case SIZE_LB_XDIST:		case SIZE_LB_YDIST:
		if(Labels) for(i = 0; i < nLab; i++) 
			if(Labels[i]) Labels[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
ContourPlot::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_SYM_LINE:	case COL_SYM_FILL:
		if (Symbols) for (i = 0; i < nSym; i++) {
			if (Symbols[i]) Symbols[i]->SetColor(select, col);
			}
		return true;
		}
	return false;
}

void 
ContourPlot::DoPlot(anyOutput *o)
{
	FillDEF bg_fill={0, 0x0L, 1.0, 0L, 0x0L};
	LineDEF bg_line = {0.0, 1.0, 0x0L, 0x0L};
	POINT clp[4];
	int i;

	if(!zAxis){
		DoAxis(o);
		DoTriangulate();
		}
	if(zAxis) {
		clp[0].x = clp[3].x = iround(o->Box1.Xmin);	clp[0].y = clp[1].y = iround(o->Box1.Ymin);
		clp[1].x = clp[2].x = iround(o->Box1.Xmax);	clp[2].y = clp[3].y = iround(o->Box1.Ymax);
		ClipBezier(0L, 0L, clp[0], clp[0], clp[0], clp[0], &clp[0], &clp[2]);	//set clipping rectangle
		bg_fill.color = bg_fill.color2 = bg_line.color = zAxis->GradColor(z_axis.min);
		o->SetFill(&bg_fill);				o->SetLine(&bg_line);
		o->oPolygon(clp, 4, 0L);
		zAxis->Command(CMD_DRAWPG, 0L, o);
		if(Symbols) {
			for (i = 0; i < nSym; i++) if (Symbols[i]) if (!zAxis->Command(CMD_SCHED_PG, Symbols[i],o)) Symbols[i]->DoPlot(o);
			}
		if(Labels) {
			for (i = 0; i < nLab; i++) if (Labels[i]) if (!zAxis->Command(CMD_SCHED_PG, Labels[i], o))Labels[i]->DoPlot(o);
			}
		zAxis->DoPlot(o);
		}
}

bool
ContourPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(hidden) return false;
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if (Symbols) for (i = nSym - 1; i >= 0; i--){
				if (Symbols[i] && Symbols[i]->Command(cmd, tmpl, o)) return true;
				}
			if (Labels) for (i = nLab - 1; i >= 0; i--) {
				if (Labels[i] && Labels[i]->Command(cmd, tmpl, o)) return true;
				}
			if(zAxis && zAxis->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_OBJTREE:
		((ObjTree*)tmpl)->Command(CMD_UPDATE, zAxis, 0L);
		return true;
	case CMD_SETTEXTDEF:
		if(Labels) for (i = nLab-1; i >= 0; i--)
			if(Labels[i]) Labels[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SCALE:
		if(Symbols) for (i = nSym-1; i >= 0; i--)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		if(Labels) for (i = nLab-1; i >= 0; i--)
			if(Labels[i]) Labels[i]->Command(cmd, tmpl, o);
		if(zAxis) zAxis->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(parent && DeleteGOL((GraphObj***)&Symbols, nSym, (GraphObj*) tmpl, o))
			return parent->Command(CMD_REDRAW,0L,o);
		if(parent && DeleteGOL((GraphObj***)&Labels, nLab, (GraphObj*) tmpl, o))
			return parent->Command(CMD_REDRAW,0L,o);
		return false;
	case CMD_UPDATE:
		if(Symbols) Undo.DropListGO(this, (GraphObj***)&Symbols, &nSym, UNDO_CONTINUE);
		if(Labels) Undo.DropListGO(this, (GraphObj***)&Labels, &nLab, UNDO_CONTINUE);
		LoadData(ssRefX, ssRefY, ssRefZ);
//		if(zAxis) zAxis->Command(CMD_RECALC, tmpl, o);
//		else DoAxis(o);
		DoAxis(o);
		if(zAxis) DoTriangulate();
		return true;
	case CMD_RECALC:
		if(zAxis) zAxis->Command(cmd, tmpl, o);
		else DoAxis(o);
		if(zAxis) DoTriangulate();
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_CONTOUR;
		if(Symbols) for (i = nSym-1; i >= 0; i--)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		if(Labels) for (i = nLab-1; i >= 0; i--)
			if(Labels[i]) Labels[i]->Command(cmd, tmpl, o);
		if(zAxis) zAxis->Command(cmd, tmpl, o);
		break;
	case CMD_SETSCROLL:		case CMD_REDRAW:	case CMD_MRK_DIRTY:
		if (cmd == CMD_MRK_DIRTY) {
			dirty = true;
			}
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		return UseAxis(*((int*)tmpl));
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nSym, 0L);
	case CMD_SAVE_LABELS:
		return SavVarObs((GraphObj **)Labels, nLab, 0L);
	case CMD_SYMTEXT:	case CMD_SYMTEXT_UNDO:		case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nSym; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
		}
	return false;
}

bool
ContourPlot::LoadData(char *xref, char *yref, char *zref)
{
	AccRange *rX, *rY, *rZ;
	int i, n;
	long cx, cy, cz, rx, ry, rz;
	int nVals, nTxt, nTime;
	bool bValid;
	anyResult arx, ary, arz;

	if(!data || !xref || !xref[0] || !yref || !yref[0] || !zref || !zref[0]) return false;
	if(!(rX = new AccRange(xref)) || !(rY = new AccRange(yref)) || !(rZ = new AccRange(zref))) return false;
	if(val) free(val);
	val = 0L;		nval = 0;
	if(3 < (n = rX->CountItems())) val = (fPOINT3D*) malloc((n+1)*sizeof(fPOINT3D));
	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
	xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
	if(rX->DataTypes(data, &nVals, &nTxt, &nTime)){
		if(!nVals && nTime > 1 && nTime > nTxt) x_dtype = ET_DATETIME;
		else x_dtype = 0;
		}
	if(val && rX->GetFirst(&cx, &rx) && rY->GetFirst(&cy, &ry) && rZ->GetFirst(&cz, &rz)) {
		i = 0;
		while(rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && rZ->GetNext(&cz, &rz)) {
			if(data->GetResult(&arx, rx, cx) && data->GetResult(&ary, ry, cy) && data->GetResult(&arz, rz, cz) 
				&& ary.type == ET_VALUE && arz.type == ET_VALUE) {
				bValid = false;
				if(x_dtype == ET_DATETIME && (arx.type == ET_DATE || arx.type == ET_TIME || arx.type == ET_DATETIME))
					bValid = true;
				else if(!x_dtype && arx.type == ET_VALUE) bValid = true;
				if(bValid) {
					val[i].fx = arx.value;			val[i].fy = ary.value;
					val[i].fz = arz.value;			i++;
					if(arx.value < Bounds.Xmin) Bounds.Xmin = arx.value;
					if(arx.value > Bounds.Xmax) Bounds.Xmax = arx.value;
					if(ary.value < Bounds.Ymin) Bounds.Ymin = ary.value;
					if(ary.value > Bounds.Ymax) Bounds.Ymax = ary.value;
					if(arz.value < zBounds.fx) zBounds.fx = arz.value;
					if(arz.value > zBounds.fy) zBounds.fy = arz.value;
					}
				}
			}
		xBounds.fx = Bounds.Xmin;		xBounds.fy = Bounds.Xmax;
		yBounds.fx = Bounds.Ymin;		yBounds.fy = Bounds.Ymax;
		nval = i;
		}
	delete	rX;		delete rY;		delete rZ;
	return (nval >3);
}

/*
// the following code is temporal and not complete
#define SIM_TOL 1.0e-12
bool values_similar(double v1, double v2)
{
	if (fabs(v2 - v1) < SIM_TOL) return true;
	return false;
}
#undef SIM_TOL

bool find_rectangles(Triangle *trl)
{
	Triangle *trn, *trnn, *trc;
	Triangle *list1 = 0L;
	double sq[3], ssq, dx, dy;
	DWORD flags;

	if (!trl) return false;
	list1 = new Triangle(&trl->pt[0], &trl->pt[1], &trl->pt[2]);
	trc = trl;				trn = 0L;
	if (trc) trn = trc->next;
	if (trc && trl && trn) do {
		dx = trc->pt[1].fx - trc->pt[0].fx;		dy = trc->pt[1].fy - trc->pt[0].fy;
		sq[0] = dx * dx + dy * dy;
		dx = trc->pt[2].fx - trc->pt[1].fx;		dy = trc->pt[2].fy - trc->pt[1].fy;
		sq[1] = dx * dx + dy * dy;
		dx = trc->pt[2].fx - trc->pt[0].fx;		dy = trc->pt[2].fy - trc->pt[0].fy;
		sq[2] = dx * dx + dy * dy;				flags = 0x0L;
		if (sq[0] > sq[1] && sq[0] > sq[2]) {
			ssq = sq[1] + sq[2];
			if (values_similar(ssq, sq[0])) flags = 0x02;
			}
		if (sq[1] > sq[0] && sq[1] > sq[2]) {
			ssq = sq[0] + sq[2];
			if (values_similar(ssq, sq[0])) flags = 0x06;
			}
		if (sq[2] > sq[0] && sq[2] > sq[1]) {
			ssq = sq[0] + sq[2];
			if (values_similar(ssq, sq[0])) flags = 0x0a;
			}
		if (flags) trc->flags |= flags;							//flags indicate an orthogonal triangle
		trnn = new Triangle(&trn->pt[0], &trn->pt[1], &trn->pt[2]);
		trc = trn;			trn = trc->next;
		} while (trn);
}
*/

extern fPOINT3D super_dim[4];				//defined in rlp_math.cpp

bool
ContourPlot::DoTriangulate()
{
	int i;
	double srz, zsum = 0.0, dx, dy/*, dz*/;
	Triangle *trl, *trn, *trnn, *trc;
	Triangulate *tria;
	fPOINT3D p1, p2, p3;

	//check minima and maxima
	if(!val || nval < 4) return false;
	if((flags & 0x03) == 0x02 || Bounds.Xmax <= Bounds.Xmin || Bounds.Ymax <= Bounds.Ymin || zBounds.fy <= zBounds.fx) {
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		for(i = 0; i < nval; i++) {
			if(val[i].fx < Bounds.Xmin) Bounds.Xmin = val[i].fx;
			if(val[i].fx > Bounds.Xmax) Bounds.Xmax = val[i].fx;
			if(val[i].fy < Bounds.Ymin) Bounds.Ymin = val[i].fy;
			if(val[i].fy > Bounds.Ymax) Bounds.Ymax = val[i].fy;
			if(val[i].fz < zBounds.fx) zBounds.fx = val[i].fz;
			if(val[i].fz > zBounds.fy) zBounds.fy = val[i].fz;
			zsum += val[i].fz;
			}
		xBounds.fx = Bounds.Xmin;		xBounds.fy = Bounds.Xmax;
		yBounds.fx = Bounds.Ymin;		yBounds.fy = Bounds.Ymax;
		}
	if(Bounds.Xmax <= Bounds.Xmin || Bounds.Ymax <= Bounds.Ymin || zBounds.fy <= zBounds.fx) return false;
	//setup two super triangles
	dx = (Bounds.Xmax - Bounds.Xmin) / 100.0; 		dy = (Bounds.Ymax - Bounds.Ymin) / 100.0;
//	dz = (zBounds.fy - zBounds.fx) / 100.0;
	super_dim[0].fx = super_dim[2].fx = Bounds.Xmin - dx;		super_dim[0].fy = Bounds.Ymin - dy;
	super_dim[1].fx = Bounds.Xmax + dx;		super_dim[1].fy = super_dim[2].fy = Bounds.Ymax + dy;
	super_dim[3].fx = Bounds.Xmax + dx;		super_dim[3].fy = Bounds.Ymin - dy;
	switch (flags & 0x03) {
		case 0:		default:
			srz = zBounds.fx;				break;
		case 1:
			srz = zBounds.fy;				break;
		case 2:
			srz = (zsum/((double)nval));	break;
		case 3:
			srz = sr_zval;					break;
			}
	super_dim[0].fz = super_dim[1].fz = super_dim[2].fz = super_dim[3].fz = srz;
	p1.fz = p2.fz = p3.fz = super_dim[0].fz;
	p1.fx = p3.fx = super_dim[0].fx;				p2.fx = super_dim[1].fx;
	p1.fy = super_dim[0].fy;						p3.fy = p2.fy = super_dim[1].fy;
	trl = new Triangle(&p1, &p2, &p3);
	p1.fx = super_dim[0].fx;						p2.fx = p3.fx = super_dim[1].fx;
	p1.fy = p2.fy = super_dim[0].fy;				p3.fy = super_dim[1].fy;
	trn = new Triangle(&p1, &p2, &p3);
	trl->flags = trn->flags = 0x01;					//these triangles reference the super triangles
	trl->next = trn;
	//do triangulation
	if(!(tria = new Triangulate(trl))) {
		delete tria;		delete trl;
		return false;
		}
	for(i = 0; i < nval; i++) {
		tria->AddVertex(&val[i]);
		}
	trl = tria->trl;		delete tria;		tria = 0L;
	//remove references to superrectangle 
	if (true) {
		trn = trl->next;
		while (trn && trl && (trl->flags & 0x01)) {
			trn = trl->next;		delete trl;				trl = trn->next;
			}
		trc = trl;
		if (trc)trn = trc->next;
		if (trc && trl && trn) do {
			trnn = trn->next;
			if (trn->flags & 0x01) {
				trc->next = trnn;		delete trn;			trn = trnn;
				}
			else {
				trc = trn;			trn = trc->next;
				}
			} while (trn);
		}
	//cut surface: create isopleths
	if(!zAxis) DoAxis(0L);
	if(zAxis && !zAxis->NumTicks) zAxis->CreateTicks();
	if(zAxis) for(i = 0; i < zAxis->NumTicks; i++) {
		trc = trl;
		while(trc) {
			trn = trc->next;
			if(zAxis->Ticks[i]) trc->IsoLine(zAxis->Ticks[i]->GetSize(SIZE_MINE), zAxis->Ticks[i]);
			trc = trn;	
			}
		if(zAxis->Ticks[i]) zAxis->Ticks[i]->ProcSeg();
		}
	//create symbols
	if(flags & 0x30) DoSymbols(trl);
	//free triangle list
	trc = trl;
	while(trc) {
		trn = trc->next; 		delete trc;			trc = trn;	
		}
	return false;
}

bool
ContourPlot::DoAxis(anyOutput *o)
{
	TextDEF tlbdef;

	if(!zAxis) {
		z_axis.min = zBounds.fx;		z_axis.max = zBounds.fy;
		NiceAxis(&z_axis, 5);
		z_axis.flags = AXIS_AUTOSCALE | AXIS_POSTICKS;
		z_axis.loc[0].fx = z_axis.loc[1].fx = defs.GetSize(SIZE_DRECT_RIGHT) + defs.GetSize(SIZE_TEXT)*1.5;
		z_axis.loc[0].fy = defs.GetSize(SIZE_DRECT_TOP);
		z_axis.loc[1].fy = defs.GetSize(SIZE_DRECT_BOTTOM)	 /*+ defs.GetSize(SIZE_TEXT, defs.dUnits)*10.0*/;
		if(!(zAxis = new Axis(this, data, &z_axis, AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS)))return false;
		zAxis->SetSize(SIZE_LB_XDIST, -NiceValue(DefSize(SIZE_AXIS_TICKS)*3.0)); 
		zAxis->SetSize(SIZE_TLB_XDIST, NiceValue(DefSize(SIZE_AXIS_TICKS)*2.0)); 
		tlbdef.ColTxt = defs.Color(COL_AXIS);				tlbdef.ColBg = 0x00ffffffL;
		tlbdef.RotBL = tlbdef.RotCHAR = 0.0;				tlbdef.iSize = 0;
		tlbdef.fSize = DefSize(SIZE_TICK_LABELS);			tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
		tlbdef.Style = TXS_NORMAL;					tlbdef.Mode = TXM_TRANSPARENT;
		tlbdef.Font = FONT_HELVETICA;					tlbdef.text = 0L;
		zAxis->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);		zAxis->type = 4;
		zAxis->CreateTicks();						zAxis->moveable = 1;
		}
	else if(z_axis.flags & AXIS_AUTOSCALE) {
		z_axis.min = zBounds.fx;		z_axis.max = zBounds.fy;
		NiceAxis(&z_axis, 4);
		if(zAxis) {
			zAxis->Command(CMD_AUTOSCALE, &z_axis, o);
			zAxis->Command(CMD_RECALC, 0L, o);
			}
		}
	return true;
}

void
ContourPlot::DoSymbols(Triangle *trl)
{
	int i;
	Triangle *trc;
	fPOINT3D *vx;
	int isValid;
	char lb_buff[80];
	TextDEF td = { 0x0L, 0x00ffffffL, defs.GetSize(SIZE_SYMBOL), 0.0, 0.0, 0, TXA_HLEFT | TXA_VCENTER, TXM_TRANSPARENT,
		TXS_NORMAL, FONT_HELVETICA, (unsigned char*)lb_buff+1};

	if(!val || nval < 4 || !trl || Symbols || Labels || !(flags & 0x30)) return;
	if(!(vx = (fPOINT3D*)malloc((nval +1) * sizeof(fPOINT3D)))) return;
	switch(flags & 0x30) {
	case 0x10:			// at minima
		for(i = 0; i < nval; i++) {
			trc = trl;	isValid = 1;
			do {
				if (trc->pt[trc->order[0]].fx == val[i].fx && trc->pt[trc->order[0]].fy == val[i].fy && 
					trc->pt[trc->order[0]].fz == val[i].fz) isValid++;
				if ((trc->pt[trc->order[1]].fx == val[i].fx && trc->pt[trc->order[1]].fy == val[i].fy
					&& trc->pt[trc->order[1]].fz == val[i].fz) || (trc->pt[trc->order[2]].fx == val[i].fx
					&& trc->pt[trc->order[2]].fy == val[i].fy && trc->pt[trc->order[2]].fz == val[i].fz))
					isValid = 0;
				trc = trc->next;
				} while(trc && isValid);
			if(isValid > 1) {
				vx[nSym].fx = val[i].fx;		vx[nSym].fy = val[i].fy;
				vx[nSym].fz = val[i].fz;		nSym++;
				}
			}
		break;
	case 0x20:			// at maxima
		for(i = 0; i < nval; i++) {
			trc = trl;	isValid = 1;
			do {
				if (trc->pt[trc->order[2]].fx == val[i].fx && trc->pt[trc->order[2]].fy == val[i].fy &&
					trc->pt[trc->order[2]].fz == val[i].fz) isValid++;
				if ((trc->pt[trc->order[0]].fx == val[i].fx && trc->pt[trc->order[0]].fy == val[i].fy
					&& trc->pt[trc->order[0]].fz == val[i].fz) || (trc->pt[trc->order[1]].fx == val[i].fx 
					&& trc->pt[trc->order[1]].fy == val[i].fy && trc->pt[trc->order[1]].fz == val[i].fz))
					isValid = 0;
				trc = trc->next;
				} while(trc && isValid);
			if(isValid > 1) {
				vx[nSym].fx = val[i].fx;		vx[nSym].fy = val[i].fy;
				vx[nSym].fz = val[i].fz;		nSym++;
				}
			}
		break;
	case 0x30:			// all values
		for(nSym = 0; nSym < nval; nSym++) {
			vx[nSym].fx = val[nSym].fx;		vx[nSym].fy = val[nSym].fy;
			vx[nSym].fz = val[nSym].fz;
			}
		break;
	}
	// create symbols
	if(nSym && (flags & 0x30) && (Symbols = (Symbol**)malloc(nSym * sizeof(Symbol*)))) {
		for(i =0; i < nSym; i++) {
			Symbols[i] = new Symbol(this, data, vx[i].fx, vx[i].fy, 0);
			}
		}
	// add labels to symbols?
	if(nSym && (flags & 0x40) && (Labels = (Label**)malloc(nSym * sizeof(Label*)))) {
		for(nLab = 0; nLab < nSym; nLab++) {
			WriteNatFloatToBuff(lb_buff, vx[nLab].fz, (char*)"%g");
			Labels[nLab] = new Label(this, data, vx[nLab].fx, vx[nLab].fy, &td, LB_X_DATA | LB_Y_DATA, 0L);
			if (Labels[nLab]){
				Labels[nLab]->SetSize(SIZE_LB_XDIST, defs.GetSize(SIZE_SYMBOL));
				}
			}
		}
	free(vx);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// three dimensional graph
Plot3D::Plot3D(GraphObj *par, DataObj *d, DWORD flags):Plot(par, d)
{
	RotDef = (double*)malloc(6 *sizeof(double));
	FileIO(INIT_VARS);
	if (par) par->SetSize(SIZE_GRECT_BOTTOM, par->GetSize(SIZE_GRECT_BOTTOM) + (par->GetSize(SIZE_GRECT_BOTTOM) -
		par->GetSize(SIZE_GRECT_TOP)) / 5.0);
	Id = GO_PLOT3D;			crea_flags = flags;
	xBounds.fx = yBounds.fx = zBounds.fx = Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
	xBounds.fy = yBounds.fy = zBounds.fy = Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	LastDirect = 0;		CurrDisp = 0L;
}

Plot3D::Plot3D(int src):Plot(0L, 0L)
{
	int i;

	RotDef = (double*)malloc(6 *sizeof(double));
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if (Axes) for (i = 0; i < nAxes; i++) {
			if (Axes[i]) Axes[i]->parent = this;
			}
		if (plots) for (i = 0; i < nPlots; i++) {
			if (plots[i]) plots[i]->parent = this;
			}
		}
	LastDirect = 0;		CurrDisp = 0L;		Id = GO_PLOT3D;		//int children!
	Command(CMD_MRK_DIRTY, 0L, 0L);		Command(CMD_AUTOSCALE, 0L, 0L);
}

Plot3D::~Plot3D()
{
	int i;

	if(plots) {
		for (i = 0; i < nPlots; i++) {
			if (plots[i] && plots[i] != this) DeleteGO(plots[i]);
			}
		free(plots);
		}
	if(Axes) {
		for (i = 0; i < nAxes; i++) {
			if (Axes[i]) DeleteGO(Axes[i]);
			}
		free(Axes);
		}
	plots = 0L;		nPlots = nAxes = 0;
	if(nscp > 0 && nscp <= nPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;			Sc_Plots = 0L;
	if(drag) DeleteGO(drag);
	drag = 0L;			if(dispObs) free(dispObs);
	free(RotDef);		if(name) free(name);
	name=0L;			dispObs = 0L;
	if (mo) DelBitmapClass(mo);
	mo = 0L;	mrc.left = mrc.top = mrc.right = mrc.bottom = 0;
	Undo.InvalidGO(this);
}

double
Plot3D::GetSize(int select)
{
	AxisDEF *ax;

	switch(select){
	//The Bounds values must be returned by every plot:
	//   they are necessary for scaling !
	case SIZE_BOUNDS_XMIN:
		if(Axes && nAxes >2 && Axes[0] && (ax = Axes[0]->GetAxis()))
			return (ax->flags & AXIS_INVERT) ? ax->max : ax->min;
		return 0.0;
	case SIZE_BOUNDS_XMAX:
		if(Axes && nAxes >2 && Axes[0] && (ax = Axes[0]->GetAxis()))
			return (ax->flags & AXIS_INVERT) ? ax->min : ax->max;
		return 0.0;
	case SIZE_BOUNDS_YMIN:
		if(Axes && nAxes >2 && Axes[1] && (ax = Axes[1]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->max : ax->min;
		return 0.0;
	case SIZE_BOUNDS_YMAX:
		if(Axes && nAxes >2 && Axes[1] && (ax = Axes[1]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->min : ax->max;
		return 0.0;
	case SIZE_BOUNDS_ZMIN:
		if(Axes && nAxes >2 && Axes[2] && (ax = Axes[2]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->max : ax->min;
		return 0.0;
	case SIZE_BOUNDS_ZMAX:
		if(Axes && nAxes >2 && Axes[2] && (ax = Axes[2]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->min : ax->max;
		return 0.0;
	case SIZE_XPOS:		case SIZE_XPOS+4:		return cu1.fx;
	case SIZE_XPOS+1:	case SIZE_XPOS+5:		return cu2.fx;
	case SIZE_XPOS+2:	case SIZE_XPOS+6:		return cu2.fx;
	case SIZE_XPOS+3:	case SIZE_XPOS+7:		return cu1.fx;
	case SIZE_YPOS:		case SIZE_YPOS+1:		case SIZE_YPOS+2:
	case SIZE_YPOS+3:			return cu1.fy;
	case SIZE_YPOS+4:	case SIZE_YPOS+5:		case SIZE_YPOS+6:
	case SIZE_YPOS+7:			return cu2.fy;
	case SIZE_ZPOS:		case SIZE_ZPOS+1:		case SIZE_ZPOS+4:
	case SIZE_ZPOS+5:			return cu1.fz;
	case SIZE_ZPOS+2:	case SIZE_ZPOS+3:		case SIZE_ZPOS+6:
	case SIZE_ZPOS+7:			return cu2.fz;
	case SIZE_XCENTER:	return rotC.fx;
	case SIZE_YCENTER:	return rotC.fy;
	case SIZE_ZCENTER:	return rotC.fz;
	default:
		return DefSize(select);
		}
}

bool
Plot3D::SetColor(int select, DWORD col)
{
	int i;

	switch(select & 0xfff) {
	case COL_AXIS:
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
Plot3D::DoPlot(anyOutput *o)
{
	long i, j;

	nObs = 0;
	if (o) CurrDisp = o;
	else o = CurrDisp;
	if(!parent || !o) return;
	if(nscp > 0 && nscp <= nPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;			Sc_Plots = 0L;		o->MouseCursor(MC_WAIT, true);
	if(dirty) DoAutoscale();
	o->HiVal = yBounds.fy;		o->LoVal = yBounds.fx;
	if(Axes && nAxes >2) {		//if no axes then parent is another Plot3D ...
		o->LightSource(&light_source);
		CurrAxes = Axes;		NumCurrAxes = nAxes;
		cu1.fx = cub1.fx;		cu1.fy = cub1.fy;		cu1.fz = cub1.fz;
		cu2.fx = cub2.fx;		cu2.fy = cub2.fy;		cu2.fz = cub2.fz;
		rc.fx = rotC.fx;		rc.fy = rotC.fy;		rc.fz = rotC.fz;
		o->SetSpace(&cu1, &cu2, defs.dUnits, RotDef, &rc, Axes[0]->GetAxis(),
			Axes[1]->GetAxis(), Axes[2]->GetAxis());
		for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->DoPlot(o);
		}
	else if(IsPlot3D(parent)) {
		if (use_xaxis || use_yaxis || use_zaxis)ApplyAxes(o);
		parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
		}
	else CurrAxes = 0L;
	if (!plots && Id == GO_FUNC3D && ((Func3D*)this)->gob) {
		(((Func3D*)this)->gob)->DoPlot(o);
		}
	if(plots) for(i = 0; i < nPlots; i++) if(plots[i]){
		if(plots[i]->Id >= GO_PLOT && plots[i]->Id < GO_GRAPH) {
			if(((Plot*)plots[i])->hidden == 0) plots[i]->DoPlot(o);
			}
		else plots[i]->DoPlot(o);
		if(i) {
			UpdateMinMaxRect(&rDims, plots[i]->rDims.right, plots[i]->rDims.top);
			UpdateMinMaxRect(&rDims, plots[i]->rDims.left, plots[i]->rDims.bottom);
			}
		else memcpy(&rDims, &plots[i]->rDims, sizeof(RECT));
			}
	for(i = 0; i< nAxes; i++) if(Axes[i]){
		UpdateMinMaxRect(&rDims, Axes[i]->rDims.right, Axes[i]->rDims.top);
		UpdateMinMaxRect(&rDims, Axes[i]->rDims.left, Axes[i]->rDims.bottom);
		}
	for (i = j = 0; i < nObs; i++) if (dispObs[i].go) {
		dispObs[j].Zmax = dispObs[i].Zmax;		dispObs[j].Zmin = dispObs[i].Zmin;
		dispObs[j++].go = dispObs[i].go;
		}
	nObs = j;
	if(nObs >1  && dispObs){
		SortObj();
		if(o->OC_type != OC_GRIDVIEW) for (i = 0; i < nObs; i++){
			for(j = 0; j < nObs; j++) {
				if(dispObs[j].go && dispObs[i].go && dispObs[j].go->Id != GO_LINESEG && i != j) switch(dispObs[i].go->Id) {
				case GO_LINESEG:
				case GO_SPHERE:
				case GO_PLANE:
					dispObs[i].go->Command(CMD_CLIP, dispObs[j].go, o);
					break;
					}
				}
			}
		for (i = 0; i < nObs; i++)	if (dispObs[i].go) dispObs[i].go->Command(CMD_REDRAW, 0L, o);
		}
	if(IsPlot3D(parent) && (use_xaxis || use_yaxis || use_zaxis))parent->Command(CMD_AXIS, 0L, o);
	dirty = false;
	o->MouseCursor(MC_ARROW, true);
}

void
Plot3D::DoMark(anyOutput *o, bool mark)
{
	if (mo) DelBitmapClass(mo);
	mo = 0L;	mrc.left = mrc.top = mrc.right = mrc.bottom = 0;
	if (!drag) drag = new Drag3D(this);
	if(mark && drag) drag->DoPlot(o);
	else parent->Command(CMD_REDRAW, 0L, 0L);
}

bool
Plot3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	long i;
	GraphObj **tmpPlots;

	if (o) CurrDisp = o;
	else o = CurrDisp;
	switch (cmd) {
	case CMD_SCALE:
		cub1.fx *= ((scaleINFO*)tmpl)->sx.fy;		cub1.fy *= ((scaleINFO*)tmpl)->sy.fy;
		cub1.fz *= ((scaleINFO*)tmpl)->sz.fy;		cub2.fx *= ((scaleINFO*)tmpl)->sx.fy;
		cub2.fy *= ((scaleINFO*)tmpl)->sy.fy;		cub2.fz *= ((scaleINFO*)tmpl)->sz.fy;
		rotC.fx *= ((scaleINFO*)tmpl)->sx.fy;		rotC.fy *= ((scaleINFO*)tmpl)->sy.fy;
		rotC.fz *= ((scaleINFO*)tmpl)->sz.fy;
		if(plots) for(i = 0; i < nPlots; i++)
			if(plots[i]) plots[i]->Command(cmd, tmpl, o);
		if(Axes) for(i = 0; i < nAxes; i++)
			if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_MOUSE_EVENT:
		if(hidden || ((MouseEvent*)tmpl)->Action != MOUSE_LBUP || CurrGO) return false;
		if(dispObs) for (i = nObs-1; i >= 0; i--)	{
			if(dispObs[i].go) dispObs[i].go->Command(cmd, tmpl, o);
			if(CurrGO) return true;
			}
		if(IsInRect(rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) {
			o->ShowMark(CurrGO = this, MRK_GODRAW);
			return true;
			}
		break;
	case CMD_USEAXIS:
		if(IsPlot3D(parent)) return UseAxis(*((int*)tmpl));
		break;
	case CMD_REG_AXISPLOT:	//notification: plot can handle its own axes
		if(nscp > 0 && nscp <= nPlots && Sc_Plots)  {
			for(i = 0; i < nscp; i++)
				if(Sc_Plots[i] == (GraphObj*)tmpl) return true;
			tmpPlots = (GraphObj**)realloc(Sc_Plots, (nscp + 1)*sizeof(GraphObj*));
			if (tmpPlots){
				tmpPlots[nscp++] = (GraphObj *)tmpl;
				Sc_Plots = tmpPlots;
				}
			else {		//memory allocation error
				nscp = 0;				Sc_Plots = 0L;
				}
			}
		else {
			Sc_Plots = (GraphObj **)calloc(1, sizeof(GraphObj*));
			if (Sc_Plots){
				Sc_Plots[0] = (GraphObj *)tmpl;
				nscp = 1;
				}
			else nscp = 0;
			}
		return true;
	case CMD_AXIS:			//one of the plots has changed scaling: reset
		CurrAxes = Axes;	NumCurrAxes = (int)nAxes;
		if(o) o->SetSpace(&cu1, &cu2, defs.dUnits, RotDef, &rc, Axes[0]->GetAxis(),
			Axes[1]->GetAxis(), Axes[2]->GetAxis());
		return true;
	case CMD_OBJTREE:
		if(!tmpl || !plots) return false;
		if (Axes) for (i = 0; i < nAxes; i++) {
			if (Axes[i]) ((ObjTree*)tmpl)->Command(CMD_UPDATE, Axes[i], 0L);
			}
		for(i = 0; i < nPlots; i++) if(plots[i]) {
			((ObjTree*)tmpl)->Command(CMD_UPDATE, plots[i], 0L);
			if(plots[i]->Id > GO_PLOT && plots[i]->Id < GO_GRAPH) plots[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i] && plots[i] == tmpPlots[0]){
			return dirty = ReplaceGO((GraphObj**)&plots[i], tmpPlots);
			}
		return false;
	case CMD_MRK_DIRTY:
		if(IsPlot3D(parent) && o) return parent->Command(cmd, tmpl, o);
		return dirty = true;
	case CMD_ADDAXIS:
		if(AddAxis()){
			if(parent && o) return parent->Command(CMD_REDRAW, tmpl, o);
			}
		return false;
	case CMD_SET_GO3D:
		if(IsPlot3D(parent) && o) return parent->Command(CMD_SET_GO3D, tmpl, o);
		return AcceptObj((GraphObj *)tmpl);
	case CMD_SETSCROLL:				case CMD_REDRAW:
		if(parent && o) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SHIFTLEFT:
		return Rotate(-0.017452406, 0.0, 0.0, o, true, 'L');
	case CMD_SHIFTRIGHT:
		return Rotate(0.017452406, 0.0, 0.0, o, true, 'R');
	case CMD_SHIFTUP:
		return Rotate(0.0, 0.017452406, 0.0, o, true,'U');
	case CMD_SHIFTDOWN:
		return Rotate(0.0, -0.017452406, 0.0, o, true,'D');
	case CMD_CURRIGHT:
		return Rotate(0.087155742, 0.0, 0.0, o, true, 'R');
	case CMD_CURRLEFT:
		return Rotate(-0.087155742, 0.0, 0.0, o, true, 'L');
	case CMD_CURRUP:
		return Rotate(0.0, 0.087155742, 0.0, o, true, 'U');
	case CMD_CURRDOWN:
		return Rotate(0.0, -0.087155742, 0.0, o, true, 'D');
	case CMD_ADDCHAR:
		if(tmpl && *((int*)tmpl) == 'r') return Rotate(0.0, 0.0, 0.087155742, o, true, 'R');
		if(tmpl && *((int*)tmpl) == 'l') return Rotate(0.0, 0.0, -0.087155742, o, true, 'L');
		if(tmpl && *((int*)tmpl) == 'R') return Rotate(0.0, 0.0, 0.017452406, o, true, 'R');
		if(tmpl && *((int*)tmpl) == 'L') return Rotate(0.0, 0.0, -0.017452406, o, true, 'L');
		return false;
	case CMD_LEGEND:
		if(plots) for(i = 0; i < nPlots; i++)
			if(plots[i]) plots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_UPDATE:	case CMD_SET_DATAOBJ:
		Id = GO_PLOT3D;
		if (cmd == CMD_SET_DATAOBJ) {
			data = (DataObj *)tmpl;
			}
		if(plots) for(i = 0; i < nPlots; i++)
			if(plots[i] && plots[i] != this && plots[i]->parent != this) plots[i]->Command(cmd, tmpl, o);
		if(Axes) for(i = 0; i < nAxes; i++)
			if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(DeleteGOL((GraphObj***)&plots,nPlots,(GraphObj*)tmpl,o)) return parent->Command(CMD_REDRAW,0L,o);
		return false;
	case CMD_MOVE:
		if (mo) DelBitmapClass(mo);
		mo = 0L;	mrc.left = mrc.top = mrc.right = mrc.bottom = 0;
		if(CurrGO && CurrGO->Id == GO_DRAGHANDLE) {
			CalcRotation(((lfPOINT*)tmpl)[0].fx, ((lfPOINT*)tmpl)[0].fy, o, true);
			if(parent) return parent->Command(CMD_REDRAW, 0L, 0L);
			}
		return true;
	case CMD_CONFIG:
		this->Configure();			return true;
	case CMD_AUTOSCALE:
		if(dirty) {
			DoAutoscale();
			dirty = false;
			}
		return true;
	case CMD_DROP_PLOT:
		if(!parent || !tmpl || ((GraphObj*)tmpl)->Id < GO_PLOT) return false;
		if(IsPlot3D(parent)) return parent->Command(cmd, tmpl, o);
		if(!nPlots) {
			plots = (GraphObj**)calloc(2, sizeof(GraphObj*));
			if(plots) {
				nPlots = 1;					plots[0] = (Plot *)tmpl;
				plots[0]->parent = this;	CreateAxes();
				return dirty = parent->Command(CMD_REDRAW, 0L, 0L);
				}
			}
		else {
			((Plot *)tmpl)->parent = this;
			tmpPlots = (GraphObj**)memdup(plots, sizeof(GraphObj*) * (nPlots+2), 0);
			Undo.ListGOmoved(plots, tmpPlots, nPlots);
			Undo.SetGO(this, &tmpPlots[nPlots++], (Plot *)tmpl, 0L);
			free(plots);			plots = tmpPlots;
			return dirty = parent->Command(CMD_REDRAW, 0L, 0L);
			}
		return false;
	case CMD_ADDPLOT:
		return AddPlot(0x0);
	case CMD_RECALC:
		if (plots) for (i = 0; i < nPlots; i++) if (plots[i]){
			plots[i]->Command(cmd, tmpl, o);
			}
		return false;
		}
	return false;
}

void *
Plot3D::ObjThere(int x, int y)
{
	if(drag) return drag->ObjThere(x, y);
	return 0L;
}

void
Plot3D::Track(POINT *p, anyOutput *o, bool)
{
	fPOINT3D v, iv;
	POINT pts[5];
	RECT upd_rc;

	if(!o) return;
	CalcRotation(((lfPOINT*)p)->fx, ((lfPOINT*)p)->fy, o, false);
	memcpy(&upd_rc, &rDims, sizeof(RECT));
	IncrementMinMaxRect(&rDims, 3);
	o->UpdateRect(&upd_rc, true);
	if (mo && mrc.bottom != mrc.top && mrc.left != mrc.right) {
		o->CopyBitmap(mrc.left, mrc.top + o->MenuHeight(), mo, 0, 0, mrc.right - mrc.left,
			mrc.bottom - mrc.top, false);
		}
	if (!mo) {
		mrc.bottom = o->un2ix(parent->GetSize(SIZE_GRECT_BOTTOM));
		mrc.top = o->un2ix(parent->GetSize(SIZE_GRECT_TOP));
		mrc.left = o->un2ix(parent->GetSize(SIZE_GRECT_LEFT));
		mrc.right = o->un2ix(parent->GetSize(SIZE_GRECT_RIGHT));
		mo = GetRectBitmap(&mrc, o);
		}
	o->SetLine(&track_line);
	memcpy(&v, &cu2, sizeof(fPOINT3D));
	o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[0].x, pts[0].y);
	v.fx = cu1.fx;					o->cvec2ivec(&v, &iv); 
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[1].x, pts[1].y);
	v.fy = cu1.fy;					o->cvec2ivec(&v, &iv); 
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[2].x, pts[2].y);
	v.fz = cu1.fz;					o->cvec2ivec(&v, &iv); 
	pts[3].x = iround(iv.fx);		pts[3].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[3].x, pts[3].y);
	v.fy = cu2.fy;					o->cvec2ivec(&v, &iv); 
	pts[4].x = iround(iv.fx);		pts[4].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	o->oPolyline(pts, 5);
	v.fz = cu2.fz;					o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	v.fz = cu1.fz;					o->cvec2ivec(&v, &iv); 
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	v.fx = cu2.fx;					o->cvec2ivec(&v, &iv);
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	v.fz = cu2.fz;					o->cvec2ivec(&v, &iv);
	pts[3].x = iround(iv.fx);		pts[3].y = iround(iv.fy);
	v.fy = cu1.fy;					o->cvec2ivec(&v, &iv);
	pts[4].x = iround(iv.fx);		pts[4].y = iround(iv.fy);
	o->oPolyline(pts, 5);
	v.fy = cu2.fy;	v.fz = cu1.fz;		o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	v.fy = cu1.fy;					o->cvec2ivec(&v, &iv);
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	v.fx = cu1.fx;					o->cvec2ivec(&v, &iv);
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	o->oPolyline(pts, 3);
	v.fz = cu2.fz;					o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	v.fx = cu2.fx;					o->cvec2ivec(&v, &iv);
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	v.fz = cu1.fz;					o->cvec2ivec(&v, &iv);
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	o->oPolyline(pts, 3);
}

#ifndef _WINDOWS
#pragma GCC diagnostic ignored "-Wmissing-field-initializers"
#endif
void
Plot3D::CreateAxes()
{
	typedef struct {
		double x1, y1, z1, x2, y2, z2;
		DWORD flags;
		int a_type, t_type;
		double lb_x, lb_y, tlb_x, tlb_y;
		int txa;
		}Axis3Ddef;
	AxisDEF tmp_axis;
	double ts = DefSize(SIZE_AXIS_TICKS);
	int i;
	if(Axes || !parent)return;
	TextDEF tlbdef = {parent->GetColor(COL_AXIS), 0x00ffffffL, DefSize(SIZE_TICK_LABELS),
		0.0, 0.0, 0, TXA_HLEFT | TXA_VCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	Axis3Ddef *at = 0L;
	Axis3Ddef at1[] = {
		{cub1.fx, cub1.fy, 0.0, cub2.fx, cub1.fy, 0.0,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 1, 3,
			0.0, NiceValue((ts+DefSize(SIZE_AXIS_TICKS))*2.0), 0.0,
			NiceValue(ts * 2.0), TXA_HCENTER | TXA_VTOP},
		{cub1.fx, cub1.fy, 0.0, cub1.fx, cub2.fy, 0.0,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 2, 2,
			-NiceValue((ts+DefSize(SIZE_AXIS_TICKS))*3.0), 0.0,
			-NiceValue(ts * 2.0), 0.0, TXA_HRIGHT | TXA_VCENTER},
		{cub1.fx, cub1.fy, 0.0, cub1.fx, cub1.fy, cub2.fz,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 3, 2,
			-NiceValue((ts+DefSize(SIZE_AXIS_TICKS))*3.0), 0.0,
			-NiceValue(ts * 2.0), 0.0, TXA_HRIGHT | TXA_VCENTER}};
	Axis3Ddef at2[] = {
		{at1[0].x1, at1[0].y1, at1[2].z2, at1[0].x2, at1[0].y2, at1[2].z2,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 1, 3,
			0.0, at1[0].lb_y, 0.0,	at1[0].tlb_y, TXA_HCENTER | TXA_VTOP},
		{at1[0].x2, at1[1].y1, 0.0, at1[0].x2, at1[1].y2, 0.0,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 2, 2,
			-at1[1].lb_x, 0.0, -at1[1].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER},
		{at1[0].x2, at1[0].y1, 0.0, at1[0].x2, at1[0].y1, at1[2].z2,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 3, 2,
			-at1[2].lb_x, 0.0, -at1[2].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER},
		{at1[0].x1, at1[0].y1, 0.0, at1[0].x2, at1[0].y2, 0.0,
			AXIS_3D, 1, 3, 0.0, at1[0].lb_y, 0.0, at1[0].tlb_y, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y2, 0.0, at1[0].x2, at1[1].y2, 0.0,
			AXIS_3D, 1, 3, 0.0, at1[0].lb_y, 0.0, at1[0].tlb_y, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y1, 0.0, at1[0].x1, at1[1].y2, 0.0,
			AXIS_3D, 2, 2, at1[1].lb_x, 0.0, at1[1].tlb_x, 0.0, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y1, at1[2].z2, at1[0].x1, at1[1].y2, at1[2].z2,
			AXIS_3D, 2, 2, at1[1].lb_x, 0.0, at1[1].tlb_x, 0.0, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[0].y1, 0.0, at1[0].x1, at1[0].y1, at1[2].z2,
			AXIS_3D, 3, 2, at1[2].lb_x, 0.0, at1[2].tlb_x, 0.0, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y2, 0.0, at1[0].x1, at1[1].y2, at1[2].z2,
			AXIS_3D, 3, 2, at1[2].lb_x, 0.0, at1[2].tlb_x, 0.0}, TXA_HCENTER | TXA_VCENTER};
	Axis3Ddef at3[] = {
		{at1[0].x1, (at1[1].y1+at1[1].y2)/2.0, at1[2].z2/2.0, at1[0].x2, 
			(at1[1].y1+at1[1].y2)/2.0, at1[2].z2/2.0,
			AXIS_3D | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 1, 3,
			0.0, at1[0].lb_y, 0.0,	at1[0].tlb_y, TXA_HCENTER | TXA_VTOP},
		{(at1[0].x1 + at1[0].x2)/2.0, at1[1].y1, at1[2].z2/2.0, 
			(at1[0].x1 + at1[0].x2)/2.0, at1[1].y2, at1[2].z2/2.0,
			AXIS_3D | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 2, 2,
			-at1[1].lb_x, 0.0, -at1[1].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER},
		{(at1[0].x1 + at1[0].x2)/2.0, (at1[1].y1+at1[1].y2)/2.0, 0.0,
			(at1[0].x1 + at1[0].x2)/2.0, (at1[1].y1+at1[1].y2)/2.0, at1[2].z2,
			AXIS_3D | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 3, 2,
			-at1[2].lb_x, 0.0, -at1[2].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER}};

	tmp_axis.min = 0.0;			tmp_axis.max = 100.0;
	tmp_axis.Start = 0.0;		tmp_axis.Step = 20.0;
	tmp_axis.Center.fx = tmp_axis.Center.fy = 0.0;
	tmp_axis.Radius = 0.0;		tmp_axis.nBreaks = 0;
	tmp_axis.breaks = 0L;		tmp_axis.owner = 0L;
	switch(AxisTempl3D){
	case 0:		at = at1;		nAxes = 3;		break;
	case 1:		at = at2;		nAxes = 9;		break;
	case 2:		at = at3;		nAxes = 3;		break;
		}
	if(!(Axes = (Axis**)calloc(nAxes, sizeof(Axis *))))return;
	if(at && nAxes) for(i = 0; i < nAxes; i++) {
		tmp_axis.loc[0].fx = at[i].x1;		tmp_axis.loc[0].fy = at[i].y1;
		tmp_axis.loc[0].fz = at[i].z1;		tmp_axis.loc[1].fx = at[i].x2;
		tmp_axis.loc[1].fy = at[i].y2;		tmp_axis.loc[1].fz = at[i].z2;
		tlbdef.Align = at[i].txa;
		if((Axes[i] = new Axis(this, data, &tmp_axis, at[i].flags))){
			Axes[i]->type = at[i].a_type;
			Axes[i]->SetSize(SIZE_LB_YDIST, at[i].lb_y);
			Axes[i]->SetSize(SIZE_LB_XDIST, at[i].lb_x);
			Axes[i]->SetSize(SIZE_TLB_YDIST, at[i].tlb_y);
			Axes[i]->SetSize(SIZE_TLB_XDIST, at[i].tlb_x);
			Axes[i]->Command(CMD_TICK_TYPE, &at[i].t_type, 0L);
			Axes[i]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			switch (i){
			case 0:
				if (!Axes[i]->name)Axes[i]->name = rlp_strdup((char*)"x-axis");
				break;
			case 1:
				if (!Axes[i]->name)Axes[i]->name = rlp_strdup((char*)"y-axis");
				break;
			case 2:
				if (!Axes[i]->name)Axes[i]->name = rlp_strdup((char*)"z-axis");
				break;
				}
			}
		}
}

#ifndef _WINDOWS
#pragma GCC diagnostic warning "-Wmissing-field-initializers"
#endif
void
Plot3D::DoAutoscale()
{
	int i;
	AxisDEF *ad;
	GraphObj **plt;
	long nplt;

	if (Id == GO_FUNC3D && ((Func3D*)this)->gob){		//we come here upoun reading a file
		plt = (GraphObj **) &((Func3D*)this)->gob;
		nplt = 1;
		}
	else {
		plt = plots;	nplt = nPlots;
		}
	if(!plt) return;
	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
	xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
	for (i = 0; i < nplt; i++) {
		if(plt[i] && plt[i] != this) {
			if (!plt[i]->Id) plt[i]->Command(CMD_SET_DATAOBJ, 0L, 0L);
			if(plt[i]->Id >= GO_PLOT && plt[i]->Id < GO_GRAPH) {
				if(!((Plot*)plt[i])->hidden) plt[i]->Command(CMD_AUTOSCALE, 0L, 0L);
				}
			else plt[i]->Command(CMD_AUTOSCALE, 0L, 0L);
			}
		}
	if(xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy && zBounds.fx <= zBounds.fy){
		if(Axes)for(i = 0; i < 3; i++) {
			if (Axes[i]){
				ad = Axes[i]->axis;
				if (ad->flags & AXIS_AUTOSCALE) {
					switch (i) {
					case 0:
						if (xBounds.fx == xBounds.fy) {
							xBounds.fx -= 1.0;	xBounds.fy += 1.0;
							}
						ad->min = xBounds.fx;	ad->max = xBounds.fy;	break;
					case 1:
						if (yBounds.fx == yBounds.fy) {
							yBounds.fx -= 1.0;	yBounds.fy += 1.0;
							}
						ad->min = yBounds.fx;	ad->max = yBounds.fy;	break;
					case 2:
						if (zBounds.fx == zBounds.fy) {
							zBounds.fx -= 1.0;	zBounds.fy += 1.0;
							}
						ad->min = zBounds.fx;	ad->max = zBounds.fy;	break;
						}
					NiceAxis(ad, 4);
					if (ad->min <= 0.0 && ((ad->flags & 0xf000) == AXIS_LOG ||
						(ad->flags & 0xf000) == AXIS_RECI)) {
						ad->min = base4log(ad, i);
						}
					Axes[i]->Command(CMD_AUTOSCALE, ad, 0L);
					}
				}
			}
		else if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
			((Plot*)parent)->CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
			}
		}
}

//Implement some kind of virtual trackball
//see: J. Hultquist: A Virtual Trackball
//Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
//ISBN 0-12-286165-5, pp. 462-463
void
Plot3D::CalcRotation(double dx, double dy, anyOutput *o, bool accept)
{
	fPOINT3D V0, V1, A;
	double R, R2, si, dp, NewRot[6];
	double rotM[3][3], newM[3][3];			//rotation matrices

	if(!CurrGO || CurrGO->Id != GO_DRAGHANDLE || CurrGO->type < DH_18 ||
		CurrGO->type > DH_88) return;
	//get coordinates for last accepted rotation
	V0.fx = GetSize(SIZE_XPOS + CurrGO->type - DH_18);
	V0.fy = GetSize(SIZE_YPOS + CurrGO->type - DH_18);
	V0.fz = GetSize(SIZE_ZPOS + CurrGO->type - DH_18);
	//revert to last matrix	
	o->SetSpace(&cu1, &cu2, defs.dUnits, RotDef, &rc, Axes[0]->GetAxis(),
		Axes[1]->GetAxis(), Axes[2]->GetAxis());
	o->cvec2ivec(&V0, &V1);
	memcpy(&V0, &V1, sizeof(fPOINT3D));
	V1.fx += o->un2fix(dx);		V1.fy += o->un2fiy(dy);
	V0.fx -= o->rotC.fx;		V0.fy -= o->rotC.fy;		V0.fz -= o->rotC.fz;
	V1.fx -= o->rotC.fx;		V1.fy -= o->rotC.fy;		V1.fz -= o->rotC.fz;
	R = sqrt(R2 = V0.fx * V0.fx + V0.fy * V0.fy + V0.fz * V0.fz);
	R2 -= (V1.fx * V1.fx + V1.fy * V1.fy);
	if (R2 <= 1.0) return;
	V1.fz = V1.fz > 0.0 ? sqrt(R2) : -sqrt(R2);
	V0.fx /= R;			V0.fy /= R;			V0.fz /= R;
	V1.fx /= R;			V1.fy /= R;			V1.fz /= R;
	A.fx = (V1.fy * V0.fz) - (V1.fz * V0.fy);
	A.fy = (V1.fz * V0.fx) - (V1.fx * V0.fz);
	A.fz = (V1.fx * V0.fy) - (V1.fy * V0.fx);

	si = sqrt(A.fx * A.fx + A.fy * A.fy + A.fz * A.fz);
	if(si > 0.001) {
		NewRot[0] = A.fx;	NewRot[1] = A.fy;	NewRot[2] = A.fz;
		NewRot[3] = si;		NewRot[4] = sqrt(1.0-si*si);	NewRot[5] = 1.0-NewRot[4];
		//normalize vector part of NewRot
		dp = sqrt(NewRot[0]*NewRot[0] + NewRot[1]*NewRot[1] + NewRot[2]*NewRot[2]);
		NewRot[0] /= dp;	NewRot[1] /= dp;	NewRot[2] /= dp;
		//set up rotation matrix from quaternion
		//see: Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
		//M.E. Pique: Rotation Tools
		// ISBN 0-12-286165-5, p.466
		rotM[0][0] = NewRot[5]*NewRot[0]*NewRot[0] + NewRot[4];
		rotM[0][1] = NewRot[5]*NewRot[0]*NewRot[1] + NewRot[3]*NewRot[2];
		rotM[0][2] = NewRot[5]*NewRot[0]*NewRot[2] - NewRot[3]*NewRot[1];
		rotM[1][0] = NewRot[5]*NewRot[0]*NewRot[1] - NewRot[3]*NewRot[2];
		rotM[1][1] = NewRot[5]*NewRot[1]*NewRot[1] + NewRot[4];
		rotM[1][2] = NewRot[5]*NewRot[1]*NewRot[2] + NewRot[3]*NewRot[0];
		rotM[2][0] = NewRot[5]*NewRot[0]*NewRot[2] + NewRot[3]*NewRot[1];
		rotM[2][1] = NewRot[5]*NewRot[1]*NewRot[2] - NewRot[3]*NewRot[0];
		rotM[2][2] = NewRot[5]*NewRot[2]*NewRot[2] + NewRot[4];
		//rotate rotation matrix
		if(MatMul(o->rotM, rotM, newM)) memcpy(&o->rotM, &newM, sizeof(newM));
		}
	if(accept) {
		//create new quaternion in RotDef from rotation matrix of output class
		Undo.RotDef(this, &RotDef, 0L);
		RotDef[4] = (o->rotM[0][0] + o->rotM[1][1] + o->rotM[2][2] -1)/2.0;
		RotDef[3] = sqrt(1.0-RotDef[4]*RotDef[4]);
		RotDef[0] = (o->rotM[1][2] - o->rotM[2][1])/(2.0 * RotDef[3]);
		RotDef[1] = (o->rotM[2][0] - o->rotM[0][2])/(2.0 * RotDef[3]);
		RotDef[2] = (o->rotM[0][1] - o->rotM[1][0])/(2.0 * RotDef[3]);
		RotDef[5] = 1.0 - RotDef[4];			LastDirect = 0;
		}
}

bool
Plot3D::AcceptObj(GraphObj *go)
{
	if (!dispObs) {
		nmaxObs = 2048;
		if (!(dispObs = (obj_desc*)calloc(nmaxObs, sizeof(obj_desc))))return false;
		nObs = 0;
		}
	else if ((nObs + 4) >= nmaxObs) {
		nmaxObs += 1024;
		dispObs = (obj_desc*)realloc(dispObs, nmaxObs * sizeof(obj_desc));
		if (!dispObs) return false;
		}
	if (CurrDisp) switch (go->Id){
	case GO_PLANE:	case GO_LINESEG:	case GO_SPHERE:
		go->Command(CMD_RECALC, 0L, CurrDisp);
		break;
	case GO_LABEL:
		break;
	default:
		return false;			//invalid object;
		}
	dispObs[nObs].Zmin = go->GetSize(SIZE_MIN_Z);
	dispObs[nObs].Zmax = go->GetSize(SIZE_MAX_Z);
	dispObs[nObs++].go = go;
	return true;
}

void
Plot3D::SortObj()
{
	long i, j;
	double *vals, *idx;
	obj_desc *new_obs;

	if (!dispObs || nObs < 2) return;
	vals = (double*)calloc(nObs+2, sizeof(double));
	idx = (double*)calloc(nObs+2, sizeof(double));
	for (i = 0; i < nObs; i++) {
		if (dispObs[i].go) {
			switch (dispObs[i].go->Id){
			case GO_PLANE:	case GO_LINESEG:	case GO_SPHERE:
			case GO_LABEL:
				vals[i] = (dispObs[i].go)->GetSize(SIZE_MIN_Z);
				dispObs[i].Zmax = vals[i];
				dispObs[i].Zmin = (dispObs[i].go)->GetSize(SIZE_MIN_Z);
				}
			}
		else vals[i] = 0.0;
		idx[i] = (double)i;
		}
	//sorts array 'vals' making the corresponding rearrangement of 'idx'
	SortArray2(nObs, vals, idx);
	if (!(new_obs = (obj_desc*)calloc(nObs + 20, sizeof(obj_desc))))return;
	for (i = 0; i < nObs; i++) {
		j = (long)idx[i];
		if (j >=0 && j < nObs && dispObs[j].go) {
			new_obs[i].Zmax = dispObs[j].Zmax;
			new_obs[i].Zmin = dispObs[j].Zmin;
			new_obs[i].go = dispObs[j].go;
			}
		else {
			new_obs[i].Zmax = new_obs[i].Zmin = 0.0;
			new_obs[i].go = 0L;
			}
		}
	//keep dispObs array in original position
	for (i = 0; i < nObs; i++) {
		dispObs[i].Zmax = new_obs[i].Zmax;
		dispObs[i].Zmin = new_obs[i].Zmin;
		dispObs[i].go = new_obs[i].go;
		}
	free(new_obs);
	return;
}

bool
Plot3D::Rotate(double dx, double dy, double dz, anyOutput *o, bool accept, int direction)
{
	int i;
	double si, NewRot[6];
	double rotM[3][3], newM[3][3];			//rotation matrices
	bool bRet = true;
	DWORD undo_flags;

	if (!o) return false;
	HideTextCursor();
	o->SetSpace(&cu1, &cu2, defs.dUnits, RotDef, &rc, Axes[0]->GetAxis(),
		Axes[1]->GetAxis(), Axes[2]->GetAxis());
	Undo.SetDisp(o);
	if (direction == LastDirect) undo_flags = UNDO_CONTINUE;
	else undo_flags = 0L;
	LastDirect = direction;
	for(i = 0; i < 3; i++) {
		switch (i){
		case 0:
			if(dx > 0.0) {
				NewRot[0] = -o->rotM[1][0];			NewRot[1] = -o->rotM[1][1];
				NewRot[2] = -o->rotM[1][2];
				NewRot[3] = si = dx;
				}
			else {
				NewRot[0] = o->rotM[1][0];			NewRot[1] = o->rotM[1][1];
				NewRot[2] = o->rotM[1][2];
				NewRot[3] = si = -dx;
				}
			break;
		case 1:
			if(dy > 0.0) {
				NewRot[0] = -o->rotM[0][0];			NewRot[1] = -o->rotM[0][1];
				NewRot[2] = -o->rotM[0][2];
				NewRot[3] = si = dy;
				}
			else {
				NewRot[0] = o->rotM[0][0];			NewRot[1] = o->rotM[0][1];
				NewRot[2] = o->rotM[0][2];
				NewRot[3] = si = -dy;
				}
			break;
		case 2:
			if(dz > 0.0) {
				NewRot[0] = -o->rotM[2][0];			NewRot[1] = -o->rotM[2][1];
				NewRot[2] = -o->rotM[2][2];
				NewRot[3] = si = dz;
				}
			else {
				NewRot[0] = o->rotM[2][0];			NewRot[1] = o->rotM[2][1];
				NewRot[2] = o->rotM[2][2];
				NewRot[3] = si = -dz;
				}
			break;
			}
		if(si > 0.0) {
			NewRot[4] = sqrt(1.0-si*si);	NewRot[5] = 1.0-NewRot[4];
			//set up rotation matrix from quaternion
			//see: Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
			//M.E. Pique: Rotation Tools
			// ISBN 0-12-286165-5, p.466
			rotM[0][0] = NewRot[5]*NewRot[0]*NewRot[0] + NewRot[4];
			rotM[0][1] = NewRot[5]*NewRot[0]*NewRot[1] + NewRot[3]*NewRot[2];
			rotM[0][2] = NewRot[5]*NewRot[0]*NewRot[2] - NewRot[3]*NewRot[1];
			rotM[1][0] = NewRot[5]*NewRot[0]*NewRot[1] - NewRot[3]*NewRot[2];
			rotM[1][1] = NewRot[5]*NewRot[1]*NewRot[1] + NewRot[4];
			rotM[1][2] = NewRot[5]*NewRot[1]*NewRot[2] + NewRot[3]*NewRot[0];
			rotM[2][0] = NewRot[5]*NewRot[0]*NewRot[2] + NewRot[3]*NewRot[1];
			rotM[2][1] = NewRot[5]*NewRot[1]*NewRot[2] - NewRot[3]*NewRot[0];
			rotM[2][2] = NewRot[5]*NewRot[2]*NewRot[2] + NewRot[4];
			if(MatMul(o->rotM, rotM, newM))	memcpy(&o->rotM, &newM, sizeof(newM));
			else accept = bRet = false;
			}
		}
	if(accept && bRet) {
		//create new quaternion in RotDef from rotation matrix of output class
		Undo.RotDef(this, &RotDef, undo_flags);
		RotDef[4] = (o->rotM[0][0] + o->rotM[1][1] + o->rotM[2][2] -1)/2.0;
		RotDef[3] = sqrt(1.0-RotDef[4]*RotDef[4]);
		RotDef[0] = (o->rotM[1][2] - o->rotM[2][1])/(2.0 * RotDef[3]);
		RotDef[1] = (o->rotM[2][0] - o->rotM[0][2])/(2.0 * RotDef[3]);
		RotDef[2] = (o->rotM[0][1] - o->rotM[1][0])/(2.0 * RotDef[3]);
		RotDef[5] = 1.0-RotDef[4];
		if(parent)return parent->Command(CMD_REDRAW, 0L, o);
		}
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// use Plot3D to create a 2.5 dimensional chart
Chart25D::Chart25D(GraphObj *par, DataObj *d, DWORD flags)
	:Plot3D(par, d, flags)
{
	dspm.fx = dspm.fy = dspm.fz = 1.0;
}

Chart25D::~Chart25D()
{
	if(name) free(name);
	name=0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// use Plot3D to create a 2.5 dimensional ribbon chart
Ribbon25D::Ribbon25D(GraphObj *par, DataObj *d, DWORD flags)
	:Plot3D(par, d, flags)
{
	dspm.fx = dspm.fy = dspm.fz = 1.0;
}

Ribbon25D::~Ribbon25D()
{
	if(name) free(name);
	name=0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// use Plot3D to create a 3 dimensional bubble plot
BubblePlot3D::BubblePlot3D(GraphObj *par, DataObj *d)
	:Plot3D(par, d, 0x0L)
{
}

BubblePlot3D::~BubblePlot3D()
{
	if(name) free(name);
	name=0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// 3D function plotter
Func3D::Func3D(GraphObj *par, DataObj *d)
	:Plot3D(par, d, 0x0L)
{
	FileIO(INIT_VARS);
	cmdxy = (char*)malloc(1024 * sizeof(char));
	if(cmdxy) rlp_strcpy(cmdxy, 1024, (char*)"r=sqrt(x*x+z*z)\ny=1-exp(-8/(r+1))");
	Id = GO_FUNC3D;
}

Func3D::Func3D(GraphObj *par, DataObj *d, double x_1, double x_2, double stepx, double z_1, double z_2, 
	double stepz, char *cmd, char *pm):Plot3D(par, d, 0x0L)
{
	size_t cb;

	FileIO(INIT_VARS);
	if(cmd && (cb = strlen(cmd)+2) && (cmdxy = (char*)malloc(cb*(sizeof(char)))))
		rlp_strcpy(cmdxy, (int)cb, cmd);
	if(pm && (cb = strlen(pm)) && (param = (char*)malloc(cb*(sizeof(char)))))
		rlp_strcpy(param, (int)cb, pm);
	x1 = x_1;	x2 = x_2;	xstep = stepx;
	z1 = z_1;	z2 = z_2;	zstep = stepz;
	Id = GO_FUNC3D;
}

Func3D::Func3D(int src):Plot3D(0)
{
	int i;

	RotDef = (double*)calloc(6, sizeof(double));
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Axes) for(i = 0; i < nAxes; i++)
			if(Axes[i]) Axes[i]->parent = this;
		if (gob) {
			gob->parent = this;
			gob->Command(CMD_SET_DATAOBJ, 0L, 0L);
			}
		}
	Id = GO_FUNC3D;
}

Func3D::~Func3D()
{
	if(param) free(param);
	param = 0L;					if(cmdxy) free(cmdxy);
	cmdxy = 0L;					if(gda) delete(gda);
	gda = 0L;					if(name) free(name);
	name = 0L;					if (RotDef) free(RotDef);
	RotDef = 0L;
}
void
Func3D::DoPlot(anyOutput *o)
{
	if (dirty && gob && parent && parent->Id >= GO_PLOT && parent->Id <= GO_GRAPH) {
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
		xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
		gob->Command(CMD_AUTOSCALE, 0L, 0L);
		DoAutoscale();
		CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
		CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
		gob->DoPlot(o);		dirty = false;
		}
	else Plot3D::DoPlot(o);
}

bool
Func3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	long i;
	GraphObj *go = (GraphObj*)tmpl;

	switch(cmd) {
	case CMD_SET_DATAOBJ:
		Plot3D::Command(cmd, tmpl, o);
		if(gob) gob->Command(cmd, gda, o);
		else if(!gob) Update();
		Id = GO_FUNC3D;
		return true;
	case CMD_UPDATE:
		return Update();
	case CMD_SET_GO3D:
		switch (go->Id){
		case GO_LINESEG:
			return false;
		case GO_PLANE:
			return Plot3D::Command(cmd, go, o);
		default:
			break;
			}
		break;
	case CMD_MOUSE_EVENT:
		if (CurrGO) return false;
		mev = (MouseEvent *)tmpl;
		//do all axes
		switch (mev->Action) {
		case MOUSE_RBUP:	case MOUSE_LBUP:
			if (Axes) for (i = 0; !CurrGO && i < nAxes; i++)
				if (Axes[i]) Axes[i]->Command(cmd, tmpl, o);
			break;
			}
		if (!CurrGO) return Plot3D::Command(cmd, go, o);
		return CurrGO ? true : false;
		}
	return Plot3D::Command(cmd, go, o);
}

bool
Func3D::Update()
{
	if(cmdxy) {
		dirty = true;
		if(xstep == 0.0) xstep = 1.0;
		if(zstep == 0.0) zstep = 1.0;
		if(!gda) gda = new DataObj();
		if(gda && do_func3D(gda, x1, x2, xstep, z1, z2, zstep, cmdxy, param)) {
			gob = new Grid3D(this, gda, type, x1, xstep, z1, zstep);
			if(gob) {
				gob->Command(CMD_SET_LINE, &Line, 0L);
				gob->Command(CMD_SYM_FILL, &Fill, 0L);
				if(!plots && (plots = (GraphObj**)calloc(2, sizeof(GraphObj*)))) {
					nPlots = 1;				plots[0] = (Plot *)gob;
					if(parent->Id == GO_GRAPH) CreateAxes();
					return dirty = true;
					}
				else if(plots && nPlots && plots[0]->Id == GO_GRID3D) {
					Undo.DeleteGO(&plots[0], UNDO_CONTINUE, 0L);
					Undo.SetGO(this, &plots[0], gob, UNDO_CONTINUE);
					return true;
					}
				else {
					DeleteGO(gob);		gob=0L;
					}
				}
			}
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// fit 3D function to data
static char *lastFunc3D = 0L, *lastParam3D=0L;
FitFunc3D::FitFunc3D(GraphObj *par, DataObj *d)
	:Plot3D(par, d, 0x0L)
{
	FileIO(INIT_VARS);
	if(lastFunc3D && lastFunc3D[0] && lastParam3D && lastParam3D[0]) {
		cmdxy = rlp_strdup(lastFunc3D);
		param = rlp_strdup(lastParam3D);
		}
	if(!cmdxy || !param) {
		cmdxy = (char*)malloc(80*sizeof(char));		param = (char*)malloc(80*sizeof(char));
		if(cmdxy) rlp_strcpy(cmdxy, 20, (char*)"a+b*x^c");
//		if(cmdxy) rlp_strcpy(cmdxy, 20, (char*)"a+b*x+c*z");
		if (param) rlp_strcpy(param, 20, (char*)"a=1; b=1; c=1.1;");
		}
	Id = GO_FITFUNC3D;
}

FitFunc3D::FitFunc3D(int src):Plot3D(0)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if (Axes) for (i = 0; i < nAxes; i++) {
			if (Axes[i]) Axes[i]->parent = this;
			}
		if (plots) for (i = 0; i < nPlots; i++) {
			if (plots[i]) plots[i]->parent = this;
			}
		}
}

FitFunc3D::~FitFunc3D()
{
	if(param) free(param);
	param = 0L;					if(cmdxy) free(cmdxy);
	cmdxy = 0L;					if(gda) delete(gda);
	gda = 0L;					if(name) free(name);
	name=0L;
}

bool
FitFunc3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;

	switch(cmd) {
	case CMD_ENDDIALOG:
		if(!cmdxy || !param) return false;
		i = rlp_strlen(cmdxy);
		if(i) {
			lastFunc3D = (char*)realloc(lastFunc3D, i + 2);
			if (lastFunc3D) rlp_strcpy(lastFunc3D, i + 1, cmdxy);
			}
		if((i = rlp_strlen(param))) {
			lastParam3D = (char*)realloc(lastParam3D, i + 2);
			if (lastParam3D) rlp_strcpy(lastParam3D, i + 1, param);
			}
		return true;
	case CMD_SET_DATAOBJ:
		Plot3D::Command(cmd, tmpl, o);
		if(gob && gda) gob->Command(cmd, gda, o);
		else if (!gob) Update();
		Id = GO_FITFUNC3D;
		return true;
	case CMD_UPDATE:
		return Update();
	case CMD_CONFIG:
		return PropertyDlg();
		}
	return Plot3D::Command(cmd, tmpl, o);
}

bool
FitFunc3D::Update()
{
	if(cmdxy) {
		dirty = true;
		if(xstep == 0.0) xstep = 1.0;
		if(zstep == 0.0) zstep = 1.0;
		if(!gda) gda = new DataObj();
		if(gda && do_func3D(gda, x1, x2, xstep, z1, z2, zstep, cmdxy, param)) {
			gob = new Grid3D(this, gda, type, x1, xstep, z1, zstep);
			if(gob) {
				gob->Command(CMD_SET_LINE, &Line, 0L);
				gob->Command(CMD_SYM_FILL, &Fill, 0L);
				if(!plots && (plots = (GraphObj**)calloc(3, sizeof(GraphObj*)))) {
					nPlots = 1;				plots[0] = (Plot *)gob;
					if(parent->Id == GO_GRAPH) CreateAxes();
					return dirty = parent->Command(CMD_REDRAW, 0L, 0L);
					}
				else if(plots && nPlots && plots[0]->Id == GO_GRID3D) {
					Undo.DeleteGO(&plots[0], UNDO_CONTINUE, 0L);
					Undo.SetGO(this, &plots[0], gob, UNDO_CONTINUE);
					return true;
					}
				else {
					DeleteGO(gob);		gob=0L;
					}
				}
			}
		}
	return false;
}
