//Output.cpp, Copyright (c) 2000-2025 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags
#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif
#include "rlp_strings.h"

tag_Units Units[] = {{0, (char*)"mm", 1.0f}, {1, (char*)"cm", 10.0f}, {2, (char*)"inch", 25.4f},
	};

extern def_vars defs;
extern GraphObj *CurrGO, *TrackGO;		//Selected Graphic Objects
extern Label *CurrLabel;
extern Graph *CurrGraph;
extern dragHandle *CurrHandle;
extern tagProgBarDlg *ProgressBarDlg;
extern void *prog_bar_ptr;
extern long prog_bar_max, *prog_bar_current;
extern DWORD prog_bar_color;
extern notary *Notary;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Output base class
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
anyOutput::anyOutput()
{
	minLW = 1.0;
	SetFill(defs.GetFill(defs.dUnits));
	dBgCol = dFillCol = defs.Color(COL_BG);
	xAxis.owner = yAxis.owner = zAxis.owner = (void *)this;
	xAxis.flags = yAxis.flags = zAxis.flags = 0L;
	xAxis.min = yAxis.min = zAxis.min = 0.0;			xAxis.max = yAxis.max = zAxis.max = 1.0;
	xAxis.nBreaks = yAxis.nBreaks = zAxis.nBreaks = 0;	xAxis.breaks = yAxis.breaks = zAxis.breaks = 0L;
	ddx = ddy = ddz = 1.0;
	RLP.finc = 1.0f;
	RLP.fp = 0.0f;
	dPattern = 0xffffffffL;			//impossible (invisible)line pattern to start with
	dBgCol = defs.Color(COL_BG);	OC_type = OC_UNKNOWN;
	MrkMode = MRK_NONE;				MrkRect = 0L;		MrkBitmap = 0L;
	VPorg.fx = 0.0;		VPorg.fy = 0.0;		 VPscale = 1.0;
	cCursor = MC_ARROW;
	rotM[0][0] = rotM[1][1] = rotM[2][2] = 1.0;
	rotM[0][1] = rotM[0][2] = rotM[1][0] = rotM[1][2] = rotM[2][0] = rotM[2][1] = 0.0;
	hasHistMenu = false;			HistMenuSize = 0;
	light_source.fx = light_source.fy = 0.0;
	x_owner = y_owner = z_owner = 0L;
	ClipRC.left = ClipRC.right = ClipRC.top = ClipRC.bottom = 0;
	isDialog = hasClip = false;
	HiVal = LoVal = 0.0;
}

void
anyOutput::SetRect(fRECT rec, int u, AxisDEF *x_ax, AxisDEF *y_ax)
{
	double spx, spy;

	if (u >= 0 && u < NUM_UNITS) defs.dUnits = u;
	spx = rec.Xmax - rec.Xmin;	spy = rec.Ymin -rec.Ymax;
	MrkMode = MRK_NONE;
	Box1.Xmin = co2fix(rec.Xmin);	Box1.Ymin = co2fiy(rec.Ymax);
	Box1.Xmax = co2fix(rec.Xmax);	Box1.Ymax = co2fiy(rec.Ymin);
	if(!x_ax || !y_ax) return;
	if(x_ax->flags & AXIS_DEFRECT) {
		Box1.Xmin = co2fix(x_ax->loc[0].fx);
		Box1.Xmax = co2fix(x_ax->loc[1].fx);
		spx = x_ax->loc[1].fx - x_ax->loc[0].fx;
		}
	if(y_ax->flags & AXIS_DEFRECT) {
		Box1.Ymin = co2fiy(y_ax->loc[0].fy);
		Box1.Ymax = co2fiy(y_ax->loc[1].fy);
		spy = y_ax->loc[1].fy - y_ax->loc[0].fy;
		}
	memcpy(&xAxis, x_ax, sizeof(AxisDEF));
	memcpy(&yAxis, y_ax, sizeof(AxisDEF));
	ddy = GetAxisFac(&yAxis, un2fiy(spy), 1);
	ddx = GetAxisFac(&xAxis, un2fix(spx), 0);
	x_owner = xAxis.owner;		y_owner = yAxis.owner;
	xAxis.owner = yAxis.owner = this;
}

void
anyOutput::UseAxis(AxisDEF *ax, int type, double dx, double dy)
{
	AxisDEF *cax;

	MrkMode = MRK_NONE;
	if(!ax) return;
	switch (type) {
		case 1:			//x-axis
		case 5:
			memcpy(cax = &xAxis, ax, sizeof(AxisDEF));
			x_owner = xAxis.owner;
			if (xAxis.flags & AXIS_DEFRECT) {
				Box1.Xmin = co2fix(ax->loc[0].fx);					Box1.Xmax = co2fix(ax->loc[1].fx);
				}
			else {
				Box1.Xmin = co2fix(ax->loc[0].fx) + dx;				Box1.Xmax = co2fix(ax->loc[1].fx) + dx;
				}
			ddx = GetAxisFac(&xAxis, Box1.Xmax - Box1.Xmin, 0);
			break;
		case 2:			//y-axis
		case 6:		case 7:
			memcpy(cax = &yAxis, ax, sizeof(AxisDEF));
			y_owner = yAxis.owner;
			if(ax->flags & AXIS_3D) {
				Box1.Ymax = co2fiy(ax->loc[0].fy)+dy;				Box1.Ymin = co2fiy(ax->loc[1].fy)+dy;
				}
			else if (yAxis.flags & AXIS_DEFRECT) {
				Box1.Ymin = co2fiy(yAxis.loc[0].fy)+dy - VPorg.fy;	Box1.Ymax = co2fiy(yAxis.loc[1].fy)+dy - VPorg.fy;
				}
			else {
				Box1.Ymin = co2fiy(ax->loc[0].fy)+dy;				Box1.Ymax = co2fiy(ax->loc[1].fy)+dy;
				}
			ddy = GetAxisFac(&yAxis, Box1.Ymax - Box1.Ymin, 1);
			break;
		case 3:			//z-axis
			memcpy(cax = &zAxis, ax, sizeof(AxisDEF));				z_owner = zAxis.owner;
			Box1z.fx = un2fiz(ax->loc[0].fz);						Box1z.fy = un2fiz(ax->loc[1].fz);
			ddz = GetAxisFac(&zAxis, Box1z.fy - Box1z.fx, 2);
			break;
		default:		//unnknown direction
			return;
		}
	cax->owner = this;
}

void
anyOutput::SetSpace(fPOINT3D *cub1, fPOINT3D *cub2, int u, double *rot, 
	fPOINT3D *cent, AxisDEF *x_ax, AxisDEF *y_ax, AxisDEF *z_ax)
{
	double rotQ[6];		//rotation definition:
						//  unit vector x
						//              y
						//              z
						//  sin(phi)
						//  cos(phi)
						//  1.0 -cos(phi)
	double dp;

	if (u >= 0 && u < NUM_UNITS) defs.dUnits = u;
	MrkMode = MRK_NONE;									HideTextCursor();
	memcpy(&xAxis, x_ax, sizeof(AxisDEF));				memcpy(&yAxis, y_ax, sizeof(AxisDEF));
	memcpy(&zAxis, z_ax, sizeof(AxisDEF));				xAxis.owner = yAxis.owner = zAxis.owner = this;
	//assume resolution equal in all directions: use un2fix() for
	//   all coordinates
	Box1.Xmin = co2fix(cub1->fx);		Box1.Ymin = co2fiy(cub2->fy);
	Box1.Xmax = co2fix(cub2->fx);		Box1.Ymax = co2fiy(cub1->fy);
	Box1z.fx = un2fiz(cub1->fz);		Box1z.fy = un2fiz(cub2->fz);
	if(x_ax->flags & AXIS_DEFRECT) {
		Box1.Xmin = co2fix(x_ax->loc[0].fx);	Box1.Xmax = co2fix(x_ax->loc[1].fx);
		}
	if(y_ax->flags & AXIS_DEFRECT) {
		Box1.Ymax = co2fiy(y_ax->loc[0].fy);	Box1.Ymin = co2fiy(y_ax->loc[1].fy);
		}
	if(z_ax->flags & AXIS_DEFRECT) {
		Box1z.fx = un2fiz(z_ax->loc[0].fz);		Box1z.fy = un2fiz(z_ax->loc[1].fz);
		}
	ddx = GetAxisFac(&xAxis, Box1.Xmax-Box1.Xmin, 0);
	ddy = GetAxisFac(&yAxis, Box1.Ymax-Box1.Ymin, 1);
	ddz = GetAxisFac(&zAxis, Box1z.fy - Box1z.fx, 2);
	rotC.fx = un2fix(cent->fx)+ VPorg.fx;
	rotC.fy = un2fiy(cent->fy)+ VPorg.fy;	
	rotC.fz = un2fiz(cent->fz);
	memcpy(rotQ, rot, sizeof(rotQ));
	//normalize vector part of rotQ
	dp = sqrt(rotQ[0]*rotQ[0] + rotQ[1]*rotQ[1] + rotQ[2]*rotQ[2]);
	rotQ[0] /= dp;		rotQ[1] /= dp;		rotQ[2] /= dp;
	dp = sqrt(rotQ[0]*rotQ[0] + rotQ[1]*rotQ[1] + rotQ[2]*rotQ[2]);
	//set up rotation matrix from quaternion
	//see: Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
	//M.E. Pique: Rotation Tools
	// ISBN 0-12-286165-5, p. 466
	rotM[0][0] = rotQ[5]*rotQ[0]*rotQ[0] + rotQ[4];
	rotM[0][1] = rotQ[5]*rotQ[0]*rotQ[1] + rotQ[3]*rotQ[2];
	rotM[0][2] = rotQ[5]*rotQ[0]*rotQ[2] - rotQ[3]*rotQ[1];
	rotM[1][0] = rotQ[5]*rotQ[0]*rotQ[1] - rotQ[3]*rotQ[2];
	rotM[1][1] = rotQ[5]*rotQ[1]*rotQ[1] + rotQ[4];
	rotM[1][2] = rotQ[5]*rotQ[1]*rotQ[2] + rotQ[3]*rotQ[0];
	rotM[2][0] = rotQ[5]*rotQ[0]*rotQ[2] + rotQ[3]*rotQ[1];
	rotM[2][1] = rotQ[5]*rotQ[1]*rotQ[2] - rotQ[3]*rotQ[0];
	rotM[2][2] = rotQ[5]*rotQ[2]*rotQ[2] + rotQ[4];
}

void
anyOutput::LightSource(lfPOINT *ls)
{
	int i, j, m;
	double angx, angy;
	double a[3][3], b[3][3];

	if(light_source.fx == 0.0 || light_source.fy == 0.0 ||
		ls->fx != light_source.fx || ls->fy != light_source.fy) {
		light_source.fx = ls->fx;		light_source.fy = ls->fy;
		angx = ls->fx * 0.017453292;		angy = ls->fy * 0.017453292;
		for (i = 0; i < 3; i++)	for(j = 0; j < 3; j++) {
			a[i][j] = b[i][j] = 0.0;
			}
		//first axis
		a[0][0] = 1.0;			a[1][1] = cos(angx);		a[1][2] = -sin(angx);
		a[2][1] = -a[1][2];		a[2][2] = a[1][1];
		//second axis
		b[0][0] = cos(angy);	b[0][1] = -sin(angy);		b[1][0] = -b[0][1];
		b[1][1] = b[0][0];		b[2][2] = 1.0;
		//combine the two rotations
		for (i = 0; i < 3; i++) for(j = 0; j < 3; j++){
			light_vec[i][j] = 0.0;
			for(m = 0; m < 3; m++) light_vec[i][j] += (a[i][m] * b[m][j]);
			}
		}
}

DWORD
anyOutput::VecColor(double *plane_vec, DWORD color1, DWORD color2)
{
	double v[3], vec[3], vlength;
	int i, j;

	//rotate vector towards the light source
	if(!plane_vec) return color1;
	v[0] = plane_vec[0];		v[1] = plane_vec[2];	v[2] = plane_vec[1];
	for (i = 0; i < 3; i++) for(j = 0, vec[i] = 0.0; j < 3; j++)
		vec[i] += (light_vec[i][j] * v[j]);
	//normalize vec: both vector should have unit length but make sure
	vlength = sqrt(vec[0]*vec[0]+vec[1]*vec[1]+vec[2]*vec[2]);
	if(vlength < 0.9) return color1;
	vec[0] /= vlength;	vec[1] /= vlength;	vec[2] /= vlength;
	//calc color
	return IpolCol(color1, color2, fabs(vec[1]));
}

bool
anyOutput::GetSize(RECT *rc)
{
	memcpy(rc, &DeskRect, sizeof(RECT));
	return true;
}

double
anyOutput::fix2fx(double x)
{
	double temp;

	//assume no transforms;
	temp = (x - Box1.Xmin) / ddx + xAxis.min;
	return temp;
}

double
anyOutput::fiy2fy(double y)
{
	double temp;

	//assume no transformation
	temp = (Box1.Ymax - y) / ddy + yAxis.min;
	return temp;
}

bool
anyOutput::ActualSize(RECT *rc) {
	return GetSize(rc);
}

double
anyOutput::fx2fix(double x)
{
	double temp;

	x = TransformValue(&xAxis, x, true);
	temp = (x - xAxis.min)*ddx;
	if(0 == (xAxis.flags & AXIS_INVERT)) return temp + Box1.Xmin;
	return Box1.Xmax-temp;
}

long 
anyOutput::fx2ix(double x){
	return (long)(0.5 + fx2fix(x));
}

double
anyOutput::fy2fiy(double y)
{
	double temp;

	y = TransformValue(&yAxis, y, true);
	temp = (y - yAxis.min)*ddy;
	if(AXIS_INVERT == (yAxis.flags & AXIS_INVERT)) return temp + Box1.Ymin;
	return Box1.Ymax-temp;
}

long
anyOutput::fy2iy(double y){
	return (long)(0.5 + fy2fiy(y));
}

double
anyOutput::fz2fiz(double z)
{
	double temp;

	z = TransformValue(&zAxis, z, true);
	temp = (z - zAxis.min)*ddz;
	if(0 == (zAxis.flags & AXIS_INVERT)) return temp + Box1z.fx;
	return Box1z.fy-temp;
}

bool 
anyOutput::fp2fip(lfPOINT *fdp, lfPOINT *fip)
{
	double x, y, si, csi, temp, a1, a2, b1, b2, yx, yy, xx, xy;

	if(!fdp || !fip) return false;
	si = 1.0;		csi = 0.0;
	if((xAxis.flags & AXIS_ANGULAR) && (yAxis.flags & AXIS_RADIAL)) {
		x = 6.283185307 * TransformValue(&xAxis, fdp->fx + xAxis.Start, true)/(xAxis.max-xAxis.min);
		si = sin(x);					csi = cos(x);
		y = TransformValue(&yAxis, fdp->fy, true);
		temp = (y - yAxis.min)*ddy;
		if(yAxis.flags & AXIS_INVERT) temp = Box1.Ymax - Box1.Ymin - temp;
		fip->fx = ((Box1.Xmin + Box1.Xmax)/2.0) + csi * temp;
		if(xAxis.flags & AXIS_INVERT) fip->fy = Box1.Ymax + si * temp;
		else fip->fy = Box1.Ymax - si * temp;
		fip->fy += disp_y;
		return true;
		}
	else if((xAxis.flags & AXIS_TRIA) && (yAxis.flags & AXIS_TRIA) && x_owner && y_owner) {
		if(((GraphObj*)y_owner)->Id == GO_AXIS) {
			((Axis*)y_owner)->GetValuePos(fdp->fy, &yx, &yy, &temp, this);
			((Axis*)x_owner)->GetValuePos(fdp->fx, &xx, &xy, &temp, this);
			}
		else {
			fip->fx = co2fix(yAxis.loc[1].fx);
			fip->fx = co2fix(yAxis.loc[1].fx);
			return false;
			}
		fip->fx = un2fix(yAxis.loc[1].fx) - un2fix(yAxis.loc[0].fx);
		fip->fy = un2fiy(yAxis.loc[1].fy) - un2fiy(yAxis.loc[0].fy);
		a1 = fip->fy / fip->fx;
		a2 = -a1;
		x = fx2fix(fdp->fx);				y = fy2fiy(fdp->fy);
		if(yAxis.loc[0].fx < yAxis.loc[1].fx) {
			b1 = xy - xx * a1;				b2 = yy - yx * a2;
			fip->fx = (b2 - b1)/(a1-a2);	fip->fy = fip->fx * a1 + b1;
			}
		else {
			fip->fx = x - TransformValue(&yAxis, fdp->fy, true) * csi;
			fip->fy = y + TransformValue(&xAxis, fdp->fx, true) * si;
			}
		return true;
		}
	else {
		fip->fx = fx2fix(fdp->fx);		fip->fy = fy2fiy(fdp->fy);
		return true;
		}
	return false;
}

bool 
anyOutput::fvec2ivec(fPOINT3D *v, fPOINT3D *iv)
{
	double x, y, z;
	
	if(!v || !iv) return false;
	x = fx2fix(v->fx)-rotC.fx;
	y = fy2fiy(v->fy)-rotC.fy;
	z = fz2fiz(v->fz)-rotC.fz;
	iv->fx = x * rotM[0][0] + y * rotM[0][1] + z * rotM[0][2] + rotC.fx;
	iv->fy = x * rotM[1][0] + y * rotM[1][1] + z * rotM[1][2] + rotC.fy;
	iv->fz = x * rotM[2][0] + y * rotM[2][1] + z * rotM[2][2] + rotC.fz;
	iv->fx += disp_x;			iv->fy += disp_y;
	return true;
}

bool
anyOutput::cvec2ivec(fPOINT3D *v, fPOINT3D *iv)
{
	double x, y, z;
	
	if(!v || !iv) return false;
	x = co2fix(v->fx)-rotC.fx;
	y = co2fiy(v->fy)-rotC.fy;
	z = un2fiz(v->fz)-rotC.fz;
	iv->fx = x * rotM[0][0] + y * rotM[0][1] + z * rotM[0][2] + rotC.fx;
	iv->fy = x * rotM[1][0] + y * rotM[1][1] + z * rotM[1][2] + rotC.fy;
	iv->fz = x * rotM[2][0] + y * rotM[2][1] + z * rotM[2][2] + rotC.fz;
	iv->fx += disp_x;			iv->fy += disp_y;
	return true;
}

bool
anyOutput::uvec2ivec(fPOINT3D *v, fPOINT3D *iv)
{
	double x, y, z;
	
	if(!v || !iv) return false;
	x = un2fix(v->fx);
	y = un2fiy(v->fy);
	z = un2fiz(v->fz);
	iv->fx = x * rotM[0][0] + y * rotM[0][1] + z * rotM[0][2];
	iv->fy = x * rotM[1][0] + y * rotM[1][1] + z * rotM[1][2];
	iv->fz = x * rotM[2][0] + y * rotM[2][1] + z * rotM[2][2];
	return true;
}

double
anyOutput::un2fix(double x)
{
	return (x * VPscale * hres*Units[defs.dUnits].convert/25.4);
}

long
anyOutput::un2ix(double x) {
	return (long)(0.5 + un2fix(x));
}

long
anyOutput::co2ix(double x) {
	return un2ix(x) + iround(VPorg.fx);
}

double
anyOutput::co2fix(double x) {
	return un2fix(x) + VPorg.fx;
}

double
anyOutput::un2fiy(double y)
{
	return (y * VPscale * vres*Units[defs.dUnits].convert/25.4);
}

long
anyOutput::un2iy(double y) {
	return (long)(0.5 + un2fiy(y));
}

long
anyOutput::co2iy(double y) {
	return un2iy(y) + iround(VPorg.fy);
}

double
anyOutput::co2fiy(double y) {
	return un2fiy(y) + VPorg.fy;
}

double
anyOutput::un2fiz(double z)
{
	return (z * VPscale * hres*Units[defs.dUnits].convert/25.4);
}

double
anyOutput::fix2un(double fix)
{
	return (fix/Units[defs.dUnits].convert*25.4/hres)/VPscale;
}

double
anyOutput::fiy2un(double fiy)
{
	return (fiy/Units[defs.dUnits].convert*25.4/vres)/VPscale;
}

bool
anyOutput::SetLine(LineDEF *nl)
{
	memcpy(&oLine, nl, sizeof(LineDEF));
	LineWidth = nl->width;			iLine = iround(un2fix(nl->width));
	dLineCol = nl->color;			dPattern = nl->pattern;
	RLP.finc = 256.0 / un2fix(nl->patlength*8.0);
	RLP.fp = 0.0;
	return true;
}

bool
anyOutput::GetLine(LineDEF *lDef)
{
	if(lDef) {
		lDef->width = LineWidth;		lDef->color = dLineCol;
		lDef->pattern = dPattern;
		return true;
		}
	return false;
}

bool
anyOutput::GetFill(FillDEF *fDef)
{
	if (fDef){
		memcpy(fDef, &FillDef, sizeof(FillDEF));
		return true;
		}
	return false;
}

bool
anyOutput::ClipRect(RECT *)
{
	return false;
}

//Test if a rectangle is clipped
//return values:
//    0    no clipping
//    1    horizontal clipping
//    2    vertical clipping
//    3    horizontal and vertical clipping
//    7    big object greater clipping rectangle?
//    8    outside range (hidden)
int 
anyOutput::ClipStatus(RECT *rec)
{
	int clip;
	long y1, y2;

	clip = 0;
	y1 = rec->top + MenuHeight();	y2 = rec->bottom + MenuHeight();
	if (ClipRC.left == ClipRC.right || ClipRC.top == ClipRC.bottom) return 0;
	if (rec->left >= ClipRC.left && rec->right <= ClipRC.right && y1 >= ClipRC.top
		&& y2 <= ClipRC.bottom) return 0;
	if (rec->left < ClipRC.left && rec->right < ClipRC.left) return 8;
	if (rec->left > ClipRC.right && rec->right > ClipRC.right) return 8;
	if (y1 < ClipRC.top && y2 < ClipRC.top) return 8;
	if (y1 > ClipRC.bottom && y2 > ClipRC.bottom) return 8;
	if (rec->left < ClipRC.left && rec->right > ClipRC.left && rec->right < ClipRC.right) clip |= 0x01;
	if (rec->left > ClipRC.left && rec->right > ClipRC.right && rec->left < ClipRC.right) clip |= 0x01;
	if (y1 < ClipRC.top && y2 >= ClipRC.top && y2 < ClipRC.bottom) clip |= 0x02;
	if (y1 >= ClipRC.bottom && y2 > ClipRC.bottom && y1 < ClipRC.top) clip |= 0x02;
	if (clip) return clip;
	//opbject probably spanning over the whole clipping range
	return 7;
}

bool
anyOutput::GetClipRec(RECT *rec)
{
	if (ClipRC.right > ClipRC.left && ClipRC.bottom > ClipRC.top && rec) {
		rec->bottom = ClipRC.bottom - MenuHeight();
		rec->top = ClipRC.top - MenuHeight();
		rec->left = ClipRC.left;
		rec->right = ClipRC.right;
		return true;
		}
	return false;
}

void
anyOutput::Focus() {
	return;
}

void
anyOutput::Caption(char *, bool){
	return;
}

void
anyOutput::MouseCursor(int, bool){
	return;
}

bool
anyOutput::SetScroll(bool, int, int, int, int){
	return false;
}

bool
anyOutput::SetFill(FillDEF *fill)
{
	if (fill){
		memcpy(&FillDef, fill, sizeof(FillDEF));
		if (fill->hatch) {
			memcpy(&FillLine, fill->hatch, sizeof(LineDEF));
			}
		FillDef.hatch = &FillLine;
		dFillCol = FillDef.color;		dFillCol2 = FillDef.color2;
		}
	return false;
}

bool
anyOutput::SetTextSpec(TextDEF *set)
{
	memcpy(&TxtSet, set, sizeof(TextDEF));
	TxtSet.text = 0L;
	return true;
}

bool
anyOutput::Erase(DWORD  color)
{
	dFillCol = color;
	return false;
}

bool
anyOutput::StartPage(){
	return false;
}

bool
anyOutput::EndPage(){
	return false;
}

bool
anyOutput::Eject(){		//printers only
	return false;
}

bool
anyOutput::UpdateRect(RECT *, bool){
	return false;
}

bool
anyOutput::CopyBitmap(int, int, anyOutput*, int, int, int, int, bool){
	return false;
}

bool
anyOutput::CopyObject(void *){
	return false;
}

void
anyOutput::ShowBitmap(int, int, anyOutput*){
	return;
}

bool
anyOutput::ShowMark(void *src, int Mode)
{
	GraphObj *go;

	if(MrkMode == MRK_GODRAW && Mode == MRK_GODRAW && src == MrkRect) return true;
	HideCopyMark(false);
	switch (Mode) {
		case MRK_INVERT:
			if(MrkBitmap && MrkRect && MrkMode == MRK_INVERT && ((RECT*)MrkRect)->left ==  ((RECT*)src)->left && 
				((RECT*)MrkRect)->top ==  ((RECT*)src)->top && ((RECT*)MrkRect)->right ==  ((RECT*)src)->right &&
				((RECT*)MrkRect)->bottom ==  ((RECT*)src)->bottom) return true;
			if(MrkMode != MRK_NONE) HideMark();
			MrkMode = Mode;			MrkRect = src;
			if(MrkBitmap != NULL) DelBitmapClass((anyOutput*)MrkBitmap);
			MrkBitmap = 0L;
			MrkBitmap = GetRectBitmap((RECT*)src, this);
			CopyBitmap(((RECT*)MrkRect)->left, ((RECT*)MrkRect)->top, this, 0, 0,((RECT*)MrkRect)->right - ((RECT*)MrkRect)->left, 
				((RECT*)MrkRect)->bottom - ((RECT*)MrkRect)->top, true);
			return UpdateRect((RECT*)src, true);
		case MRK_GODRAW:	case MRK_SSB_DRAW:
			if(MrkMode != MRK_NONE) HideMark();
			MrkMode = Mode;										MrkRect = src;
			go = (GraphObj *) src;								go->DoMark(this, true);
			CurrGO = go;
			if(CurrLabel && CurrLabel != CurrGO) {
				HideTextCursor();								CurrLabel = 0L;
				}
			return true;
		}
	return false;
}

bool
anyOutput::HideMark()
{
	switch(MrkMode) {
		case MRK_NONE:
			return true;
		case MRK_INVERT:
			MrkMode = MRK_NONE;
			if(!MrkBitmap || !MrkRect) return false;
			RestoreRectBitmap((anyOutput**)&MrkBitmap, (RECT*)MrkRect, this);
			MrkBitmap = 0L;						MrkRect = 0L;
			return true;
		case MRK_GODRAW:
			MrkMode = MRK_NONE;					//inhibit reentrance
			if(CurrLabel && CurrLabel->Command(CMD_HIDEMARK, 0L, this)){
				if(CurrGraph) CurrGraph->Command(CMD_REDRAW, 0L, this);
				}
			else if(MrkRect)((GraphObj*)MrkRect)->DoMark(this, false);
			else if(CurrGraph) CurrGraph->Command(CMD_REDRAW, 0L, this);
			return true;
		case MRK_SSB_DRAW:
			MrkMode = MRK_NONE;
			if (MrkRect) ((GraphObj*)MrkRect)->DoMark(this, false);
			return true;
		}
	return false;
}

int
anyOutput::CalcCursorPos(unsigned char *txt, POINT p, POINT *fit)
{
	int i, d, CurrPos;
	double w, h;

	d = TxtSet.iSize >>2;
	if(!txt || !fit) return 0;
	if (!(i = (int)strlen((char*)txt)))return 0;
	//right justified text
	if(TXA_HRIGHT == (TxtSet.Align & TXA_HRIGHT)){
		if((p.x - fit->x) < d) return i;
		for (CurrPos = i-1; CurrPos >= 0; CurrPos--) {
			if(!oGetTextExtent(txt+CurrPos, i-CurrPos, &w, &h)) return 0;
			if((w = p.x - w - d) <= fit->x) return CurrPos;
			}
		return 0;
		}
	//left justified text
	else {
		if((fit->x - p.x) < d) return 0;
		for (CurrPos = i; CurrPos >= 0; CurrPos--) {
			if(!oGetTextExtent(txt, CurrPos, &w, &h)) return 0;
			if((w = p.x + w - d) <= fit->x) return CurrPos;
			}
		}
	return 0;
}

bool
anyOutput::TextCursor(unsigned char *txt, POINT p, POINT *fit, int *pos, int dx)
{
	int i, CurrPos;
	double w, h;
	RECT disp;

	if(fit) CurrPos = CalcCursorPos(txt, p, fit);
	//recalculate caret position
	if(txt && pos && !fit){
		if(TxtSet.Align & TXA_HRIGHT) {		//right justfied text
			if((i = (int)strlen((char*)txt)-(*pos))){
				if(!oGetTextExtent(txt+(*pos), i, &w, &h)) return false;
				w = p.x - w;
				}
			else w = p.x-1;
			}
		else {								//left justified text
			if(!(*pos)) w = 0;
			else if(!oGetTextExtent(txt, *pos, &w, &h))return false;
			w += p.x;
			}
		}
	else if(!fit)return false;
	//right justified text: search caret and cursor position
	else if(txt && (TxtSet.Align & TXA_HRIGHT)){
		i = (int)strlen((char*)txt);
		if(i == CurrPos) w = 1;
		else if(!oGetTextExtent(txt+CurrPos, i-CurrPos, &w, &h)) return false;
		w = p.x - w;
		}
	//left justified text: search caret and cursor position
	else if(txt && fit) {
		if (!CurrPos) w = 0;
		else if(!oGetTextExtent(txt, CurrPos, & w, &h)) return false;
		w += p.x;
		}
	if(fit && pos) *pos = CurrPos;
	disp.left = disp.right = iround(w)+dx;
	disp.top = p.y;
	if(TxtSet.Align & TXA_VCENTER) {
		disp.top -= (TxtSet.iSize>>1);
		}
#ifdef _WINDOWS
	disp.bottom = disp.top + TxtSet.iSize-1;
#else
	disp.top -= 1;
	disp.bottom = disp.top + iround(TxtSet.iSize*1.25)-2;
#endif
	ShowTextCursor(this, &disp, 0x0L);
	return true;
}

//we need our own implementation of Bresenham's line drawing algorithm to draw
//   a line with variable pattern sizes.
//Ref: P.S. Heckbert (1990) "Digital Line Drawing", in: Graphic Gems
//   (A.S. Glassner, ed.); Academic Press, Inc.,
//   ISBN 0-12-286165-5
bool
anyOutput::PatLine(POINT p1, POINT p2)
{
	int d, ax, ay, sx, sy, dx, dy;
	double fInc2;
	bool bPen;
	POINT tr[2];

	dx = p2.x - p1.x;
	fInc2 = RLP.finc * 0.414213562;			//increment by sqrt(2) if 45� slope
	if ( p2.x < p1.x) { 	ax = (-dx)<<1;		sx = -1;		}
	else {					ax = dx <<1;		sx = 1;		}
	dy = p2.y - p1.y;
	if (p2.y < p1.y) {	ay = (-dy)<<1;			sy = -1;		}
	else {					ay = dy<<1;			sy = 1;		}
	tr[0].x = tr[1].x = p1.x;							tr[0].y = tr[1].y = p1.y;
	if(dPattern &(1 << ((int)RLP.fp))) bPen = false;
	else bPen = true;
	if (ax > ay) {													// x dominant
		d = ay - (ax >>1);
		for ( ; ; ) {
			RLP.fp += RLP.finc;
			if(RLP.fp >= 32.0f) RLP.fp -= 32.0f;
			if(bPen) {
				if(tr[1].x == p2.x) return oSolidLine(tr);
				if (d >= 0) {tr[1].y += sy;	d -= ax; RLP.fp += fInc2;}
				tr[1].x += sx;
				if(dPattern &(1 << ((int)RLP.fp))) {
					bPen = false;
					oSolidLine(tr);
					tr[0].x = tr[1].x; tr[0].y = tr[1].y;
					}
				}
			else {
				if(tr[0].x == p2.x) return true;
				if (d >= 0) {tr[0].y += sy;	d -= ax; RLP.fp += fInc2;}
				tr[0].x += sx;
				if(!(dPattern &(1 << ((int)RLP.fp)))) {
					bPen = true;
					tr[1].x = tr[0].x; tr[1].y = tr[0].y;
					}
				}
			d += ay;
			}
		}
	else {															// y dominant
		d = ax - (ay >>1);
		for ( ; ; ) {
			RLP.fp += RLP.finc;
			if(RLP.fp >= 32.0f) RLP.fp -= 32.0f;
			if (bPen){
				if (tr[1].y == p2.y) return oSolidLine(tr);
				if (d >= 0) {tr[1].x += sx; d -= ay; RLP.fp += fInc2;}
				tr[1].y += sy;
				if(dPattern &(1 << ((int)RLP.fp))) {
					bPen = false;
					oSolidLine(tr);
					tr[0].x = tr[1].x; tr[0].y = tr[1].y;
					}
				}
			else {
				if (tr[0].y == p2.y) return true;
				if (d >= 0) {tr[0].x += sx; d -= ay; RLP.fp += fInc2;}
				tr[0].y += sy;
				if(!(dPattern &(1 << ((int)RLP.fp)))) {
					bPen = true;
					tr[1].x = tr[0].x; tr[1].y = tr[0].y;
					}
				}
			d += ax;
			}
		}
}

bool
anyOutput::PatLine(lfPOINT p1, lfPOINT p2)
{
	POINT ip1, ip2;

	ip1.x = (long)p1.fx;	ip1.y = (long)p1.fy;
	ip2.x = (long)p2.fx;	ip2.y = (long)p2.fy;
	return PatLine(ip1, ip2);
}

void
anyOutput::ShowLine(POINT *, int, DWORD, bool){
	return;
}

void
anyOutput::ShowEllipse(POINT, POINT, DWORD, bool){
	return;
}

bool
anyOutput::oPolyline(POINT *, int){
	return false;
}

bool
anyOutput::foPolyline(lfPOINT *, int){
	return false;
}

bool
anyOutput::oBezier(POINT *spts, int scp, POINT **bpts, long *bcp, bool dodraw)
{
	long i, nPts;
	POINT *pts;

	pts = (POINT*)malloc(scp * 64 * sizeof(POINT));
	for(i= nPts = 0; i< (scp-2); i += 3) {
		DrawBezier(&nPts, pts, spts[i], spts[i+1], spts[i+2], spts[i+3], 0);
		}
	if(dodraw)oPolyline(pts, nPts);
	if(bpts && bcp) {
		*bpts = pts;		*bcp = nPts;
		}
	else free(pts);
	return true;
}

bool
anyOutput::oRectangle(int, int, int, int, char *)
{
	return false;
}

bool
anyOutput::oCircle(int, int, int, int, char *)
{
	return false;
}

bool 
anyOutput::foCircle(double x1, double y1, double x2, double y2, char *nam)
{
	return oCircle(iround(x1), iround(y1), iround(x2), iround(y2), nam);
}

bool
anyOutput::oCircArc(int x1, int y1, int x2, int y2)
//code thanks to kjhughes on Stack Overflow
{
	int w, h, xe, ye;
	double kappa, ox, oy, xm, ym;
	POINT *pts, p1, p2, p3, p4;
	long nPts = 0, pt_size;

	pt_size = (x2 - x1) * 4;
	pts = (POINT*)malloc(pt_size * sizeof(POINT));
	w = (x2 - x1);	h = (y2 - y1);
	kappa = 0.5522848;
	ox = (w / 2.0) * kappa;		// control point offset horizontal
	oy = (h / 2.0) * kappa;		// control point offset vertical
	xe = x1 + w;				// x-end
	ye = y1 + h;				// y-end
	xm = x1 + w / 2.0;			// x-middle
	ym = y1 + h / 2.0;			// y-middle
	p1.x = x1;					p1.y = iround(ym);
	p2.x = x1;					p2.y = iround(ym - oy);
	p3.x = iround(xm - ox);		p3.y = y1;
	p4.x = iround(xm);			p4.y = y1;
	DrawBezier(&nPts, pts, p1, p2, p3, p4, 0);
	p1.x = p4.x;				p1.y = p4.y;
	p2.x = iround(xm + ox);		p2.y = y1;
	p3.x = iround(xe);			p3.y = iround(ym - oy);
	p4.x = iround(xe);			p4.y = iround(ym);
	DrawBezier(&nPts, pts, p1, p2, p3, p4, 0);
	p1.x = p4.x;				p1.y = p4.y;
	p2.x = iround(xe);			p2.y = iround(ym + oy);
	p3.x = iround(xm + ox);		p3.y = iround(ye);
	p4.x = iround(xm);			p4.y = iround(ye);
	DrawBezier(&nPts, pts, p1, p2, p3, p4, 0);
	p1.x = p4.x;				p1.y = p4.y;
	p2.x = iround(xm - ox);		p2.y = iround(ye);
	p3.x = x1;					p3.y = iround(ym + oy);
	p4.x = x1;					p4.y = iround(ym);
	DrawBezier(&nPts, pts, p1, p2, p3, p4, 0);
	if (pts[0].x != pts[nPts - 1].x || pts[0].y != pts[nPts - 1].y) {
		pts[nPts].x = pts[0].x;	pts[nPts++].y = pts[0].y;		//close polygon
		}
	oPolyline(pts, nPts);
	free(pts);
	return true;
}

bool
anyOutput::oSolidRectangle(int x1, int y1, int x2, int y2)
{
	return oRectangle(x1, y1, x2, y2);
}

bool
anyOutput::foSolidRectangle(double x1, double y1, double x2, double y2)
{
	return oRectangle(iround(x1), iround(y1), iround(x2), iround(y2));
}

bool
anyOutput::oFillRect(RECT *rec, DWORD color)
{
	static LineDEF line = { defs.GetSize(SIZE_HAIRLINE), 1.0, 0x00f0f0f0L, 0L };
	static FillDEF fill = { FILL_NONE, 0x0L, 1.0, NULL, 0x00ffffffL };

	fill.color = color;		line.color = color;
	SetLine(&line);			SetFill(&fill);
	oRectangle(rec->left, rec->top, rec->right, rec->bottom);
	return true;
}

bool
anyOutput::foRectangle(double x1, double y1, double x2, double y2, char *nam)
{
	return oRectangle(iround(x1), iround(y1), iround(x2), iround(y2), nam);
}

bool
anyOutput::oSolidLine(POINT *){
	return false;
}

bool
anyOutput::foSolidLine(lfPOINT *p)
{
	POINT pts[2];

	pts[0].x = iround(p[0].fx);			pts[0].y = iround(p[0].fy);
	pts[1].x = iround(p[1].fx);			pts[1].y = iround(p[1].fy);
	return oSolidLine(pts);
}

bool
anyOutput::SetMenu(int){
	return false;
}

void
anyOutput::CheckMenu(int, bool){
	return;
}

void
anyOutput::FileHistory(){
	return;
}

bool 
anyOutput::oGetPix(int, int, DWORD *){
	return false;
}

bool
anyOutput::oDrawIcon(int, int, int) {
	return false;
}

//The following functions are a stub: They should be implemented by every output class
bool
anyOutput::oGetTextExtent(unsigned char *text, int, double *width, double *height)
{
	*width = un2fix(((double)rlp_strlen(text)) * 35.0 * TxtSet.fSize/52.0);
	*height = un2fiy(TxtSet.fSize);
	return true;
}

bool
anyOutput::oGetTextExtentW(w_char *text, int cb, double *width, double *height)
{
	if(cb < 1) for(cb = 0; text[cb]; cb++);
	*width = (TxtSet.iSize * cb) /2.0;
	*height = TxtSet.iSize;
	return true;
}

bool 
anyOutput::oSphere(int cx, int cy, int r, POINT *pts, int cp, char *nam)
{
	FillDEF fd;
	HatchOut *ho;
	int i, j;
	double mlw;

	if(pts && cp) oPolygon(pts, cp, nam);
	else oCircle(cx - r, cy - r, cx + r + 1, cy + r + 1, nam);
	fd.color = dFillCol;		fd.color2 = dFillCol2;
	fd.hatch = 0L;			fd.scale = 1.0;
	fd.type = FILL_NONE;
	ho = new HatchOut(this);
	if(ho != NULL) {
		ho->SetFill(&fd);
		mlw = minLW;		minLW = ho->minLW = 2.0;
		ho->light_source.fx = light_source.fx;
		ho->light_source.fy = light_source.fy;
		for(i = 0; i < 3; i++) for (j = 0; j < 3; j++) {
			ho->light_vec[i][j] = light_vec[i][j];
			}
		ho->oSphere(cx, cy, r-iLine, pts, cp, 0L);
		delete(ho);	
		minLW = mlw;
		return true;
		}
	return false;
}

bool
anyOutput::oTextOut(int, int, unsigned char *, int){
	return false;
}

bool
anyOutput::oTextOutW(int, int, w_char *, int){
	return false;
}

bool
anyOutput::oPolygon(POINT *, int, char *){ 
	return false;
}

bool
anyOutput::foPolygon(lfPOINT *pts, int np, char *nam)
{
	POINT *pt;
	int i;

	pt = (POINT*)malloc(np * sizeof(POINT));
	for (i = 0; i < np; i++) {
		pt[i].x = iround(pts[i].fx);		pt[i].y = iround(pts[i].fy);
		}
	oPolygon(pt, np, nam);
	free(pt);
	return false;
}

bool
anyOutput::oSolidPolygon(POINT *, int){
	return false;
}

bool 
anyOutput::Command(int cmd, void *data)
{
	switch (cmd) {
	case CMD_CPY_DATA:				//coming here from the layers dialog
		((Plot*)data)->Command(cmd, 0L, this);
		return false;
		}
	return false;
}

bool
anyOutput::setVPorg(double x, double y, double scale)
{
	if (fabs(x) < 10000.0 && fabs(y) < 10000.0 && scale > 0.1 && scale < 1000.0) {
		if(VPorg.fx == x && VPorg.fy == y && VPscale == scale) return false;
		VPorg.fx = x;	VPorg.fy = y;	VPscale = scale;
		return true;
		}
	return false;
}

bool 
anyOutput::incVPorg(double dx, double dy)
{
	if (fabs(dx) >= 0.0 && fabs(dy) >= dy && fabs(dx) < 1000.0 && fabs(dy) < 1000.0) {
		VPorg.fx += dx;		VPorg.fy += dy;
		return true;
		}
	return false;
}

bool 
anyOutput::setVPorgScale(double scale)
{
	if (scale > 0.1 && scale < 1000) {
		VPscale = scale;
		return true;
		}
	return false;
}

// a stub for output classes that do not support color gradients
bool 
anyOutput::GradPG(fPOINT3D *pt, long n, double, double, double, double, lfPOINT *)
{
	lfPOINT *pts;
	long i;

	pts = (lfPOINT*)malloc(n * sizeof(lfPOINT));
	for (i = 0; i < n; i++) {
		pts[i].fx = pt[i].fx;		pts[i].fy = pt[i].fy;
		}
	foPolygon(pts, (int)n);
	free(pts);
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Temporarily display a graph as grid model
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
GridView::GridView(anyOutput *parent):anyOutput()
{
	out = parent;
	if(out){
		out->GetSize(&DeskRect);
		hres = out->hres;		vres = out->vres;		VPscale = out->getVPorgScale();
		VPorg.fx = out->getVPorgX();		VPorg.fy = out->getVPorgY();
		}
	else {
		DeskRect.left = DeskRect.right = DeskRect.top = DeskRect.bottom = 0;
		hres = vres = 100;
		VPorg.fx = VPorg.fy = 0.0;
		}
	minLW = 1.0;		dPattern = 0L;
	dLineCol = 0x00cbcbcbL;					//current color of line;
	OC_type = OC_GRIDVIEW;					Notary->ValPtr(this, true);
}

GridView::~GridView()
{
	Notary->ValPtr(this, false);
}

bool
GridView::SetLine(LineDEF *lDef)
{
	dLineCol = (lDef->color & 0x00ffffffL);
	dLineCol |=  0x00808080;
	return true;
}

bool 
GridView::SetTextSpec(TextDEF *set)
{
	anyOutput::SetTextSpec(set);
	return out->SetTextSpec(set);
}


bool
GridView::SetFill(FillDEF *)
{
	return true;
}

bool
GridView::oGetTextExtent(unsigned char *text, int cb, double *width, double *height)
{
	return out->oGetTextExtent(text, cb, width, height);
}

bool
GridView::oGetTextExtentW(w_char *text, int cb, double *width, double *height)
{
	return out->oGetTextExtentW(text, cb, width, height);
}

bool
GridView::oCircle(int x1, int y1, int x2, int y2, char *)
{
	if (out) out->oCircArc(x1, y1, x2, y2);
	return true;
	}

bool
GridView::oPolyline(POINT *pts, int cp)
{
	if (out) out->oPolyline(pts, cp);
	return true;
}

bool
GridView::foPolyline(lfPOINT *pts, int cp)
{
	if (out) out->foPolyline(pts, cp);
	return true;
}

bool
GridView::oRectangle(int x1, int y1, int x2, int y2, char *)
{
	POINT pts[5];

	pts[0].x = pts[3].x = pts[4].x = x1;
	pts[1].x = pts[2].x = x2;
	pts[0].y = pts[1].y = pts[4].y = y1;
	pts[2].y = pts[3].y = y2;
	return oPolyline(pts, 5);
}

bool
GridView::oSolidLine(POINT *p)
{
	if (out) out->oSolidLine(p);
	return true;
}

bool
GridView::oTextOut(int x, int y, unsigned char *txt, int cb)
{
	double width, height;

	if (!txt || !txt[0]) return false;
	out->oGetTextExtent(txt, cb, &width, &height);
	return ShowTxtRec(x, y, iround(width), iround(height));
}

bool
GridView::oTextOutW(int x, int y, w_char *txt, int cb)
{
	double width, height;

	if (!txt || !txt[0]) return false;
	out->oGetTextExtentW(txt, cb, &width, &height);
	return ShowTxtRec(x, y, iround(width), iround(height));
}

bool
GridView::oPolygon(POINT *pts, int cp, char *)
{
	return oPolyline(pts, cp);
}

bool
GridView::ShowTxtRec(int x, int y, int width, int height)
{
	int ix = 0, iy = 0;
	double si, csi;
	fRECT rc;
	POINT pts[5];

	if (out->TxtSet.RotBL == 0.0){
		if ((out->TxtSet.Align & TXA_HCENTER) == TXA_HCENTER) ix -= (width >> 1);
		else if ((out->TxtSet.Align & TXA_HRIGHT) == TXA_HRIGHT) ix -= width;
		if ((out->TxtSet.Align & TXA_VCENTER) == TXA_VCENTER) iy -= (height >> 1);
		else if ((out->TxtSet.Align & TXA_VBOTTOM) == TXA_VBOTTOM) iy -= height;
		pts[0].x = pts[3].x = pts[4].x = ix + x;			pts[1].x = pts[2].x = ix + x + width;
		pts[0].y = pts[1].y = pts[4].y = iy + y;			pts[2].y = pts[3].y = iy + y + height;
		oPolyline(pts, 5);
		return true;
		}
	else if (out->TxtSet.RotBL == 90.0){
		if ((out->TxtSet.Align & TXA_HCENTER) == TXA_HCENTER) iy -= (width >> 1);
		else if ((out->TxtSet.Align & TXA_HRIGHT) == TXA_HRIGHT) iy -= width;
		if ((out->TxtSet.Align & TXA_VCENTER) == TXA_VCENTER) ix -= (height >> 1);
		else if ((out->TxtSet.Align & TXA_VBOTTOM) == TXA_VBOTTOM) ix -= height;
		pts[0].x = pts[3].x = pts[4].x = ix + x;			pts[1].x = pts[2].x = ix + x + height;
		pts[0].y = pts[1].y = pts[4].y = iy + y;			pts[2].y = pts[3].y = iy + y + width;
		oPolyline(pts, 5);
		return true;
		}
	else {
		si = sin(out->TxtSet.RotBL *0.01745329252);	csi = cos(out->TxtSet.RotBL *0.01745329252);
		rc.Xmin = -2.0;		rc.Ymin = 0.0;		rc.Xmax = width;		rc.Ymax = height;
		if (TxtSet.Align & TXA_HCENTER) {
			rc.Xmin -= width / 2.0 - 1.0;		rc.Xmax -= width / 2.0 - 1.0;
			}
		else if (TxtSet.Align & TXA_HRIGHT) {
			rc.Xmin -= width - 2.0;				rc.Xmax -= width - 2.0;
			}
		if (TxtSet.Align & TXA_VCENTER) {
			rc.Ymin -= height / 2.0;			rc.Ymax -= height / 2.0;
			}
		else if (TxtSet.Align & TXA_VBOTTOM) {
			rc.Ymin -= height;					rc.Ymax -= height;
			}
		pts[0].x = iround(rc.Xmin*csi + rc.Ymin*si) + x;		pts[0].y = iround(rc.Ymin*csi - rc.Xmin*si) + y;
		pts[1].x = iround(rc.Xmax*csi + rc.Ymin*si) + x;		pts[1].y = iround(rc.Ymin*csi - rc.Xmax*si) + y;
		pts[2].x = iround(rc.Xmax*csi + rc.Ymax*si) + x;		pts[2].y = iround(rc.Ymax*csi - rc.Xmax*si) + y;
		pts[3].x = iround(rc.Xmin*csi + rc.Ymax*si) + x;		pts[3].y = iround(rc.Ymax*csi - rc.Xmin*si) + y;
		pts[4].x = pts[0].x;	pts[4].y = pts[0].y;
		oPolyline(pts, 5);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process hatch patterns in an output class
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
enum {HO_NONE, HO_RECT, HO_CIRCLE, HO_ELLIPSE, HO_BIGELLYPSE,
	HO_POLYGON};

static struct _HatchDef{
	struct {
		RECT rec;
	}rec;
	struct {
		POINT centre;
		long sr;
	}cir;
	struct {
		POINT centre;
		long ix, iy;
		unsigned long sab;
	}ell;
	struct {
		POINT fo[2];
		unsigned int rsab;
		long a, b;
	}bell;
	struct {
		POINT *pts;
		long cp;
	}plg;
}HatchDef;

HatchOut::HatchOut(anyOutput *Parent):anyOutput()
{
	ParInit = false;	ho = HO_NONE;	ht = FILL_COMBS;
	out = Notary->IsValidPtr(Parent) ? Parent : 0L;
	OC_type = 0;		xbase = ybase = 1.5;
	if (out) out->GetSize(&DeskRect);
	else DeskRect.left = DeskRect.right = DeskRect.top = DeskRect.bottom = 0;
	MyLineDef.width = 0.0f;								MyLineDef.patlength = 10.0f;
	MyLineDef.color = 0x00ff0000L;						MyLineDef.pattern = 0x00000000L;
	MyFillDef.color = MyFillDef.color2 = 0x00ffffffL;	MyFillDef.type = 0;
	MyFillDef.scale = 1.0;								MyFillDef.hatch = 0L;
	pClipRC.left = pClipRC.right = pClipRC.top = pClipRC.bottom = 0;
}

bool
HatchOut::SetFill(FillDEF *fill)
{
	if(!fill) return false;
	ht = (fill->type & 0xff);
	if (fill->hatch) {
		memcpy(&MyLineDef, fill->hatch, sizeof(LineDEF));
		MyFillDef.color = MyFillDef.color2 = MyLineDef.color;
		MyFillDef.type = FILL_KEEP_ALIVE;					//keep Hatchout class alive when setfill to parent
		MyFillDef.scale = fill->scale;
		MyFillDef.hatch = 0L;
		}
	//we assume that all operations are at a 1:1 pixel relation to the parent,
	//  but we use un2fix and un2fiy from the parent: correct for zoom ....
	if(out) switch (defs.dUnits) {
	case 0:						//parent uses mm
		xbase = out->un2fix(1.5);		ybase = out->un2fiy(1.5);
		break;
	case 1:						//parent uses cm
		xbase = out->un2fix(0.15);		ybase = out->un2fiy(0.15);
		break;
	case 2:						//parent uses inches
		xbase = out->un2fix(0.059);		ybase = out->un2fiy(0.059);
		break;
		}
	if(fill->scale >0.05f && fill->scale < 20.0) {
		xbase *= fill->scale;		ybase *= fill->scale;
		}
	dFillCol = fill->color;
	dFillCol2 = fill->color2;
	return true;
}

bool
HatchOut::StartPage()
{
	if (out) out->GetSize(&DeskRect);
	return true;
}

bool
HatchOut::oCircle(int x1, int y1, int x2, int y2, char *)
{
	long tmp;

	if(x1 < x2) {		UseRect.left = x1;		UseRect.right = x2;		}
	else {				UseRect.left = x2;		UseRect.right = x1;		}
	if(y1 < y2) {		UseRect.top = y1;			UseRect.bottom = y2;		}
	else {				UseRect.top = y2;			UseRect.bottom = y1;		}
	if((UseRect.right -UseRect.left)==(UseRect.bottom-UseRect.top)) {
		HatchDef.cir.centre.x = (UseRect.right + UseRect.left)/2;
		HatchDef.cir.centre.y = (UseRect.bottom + UseRect.top)/2;
		tmp = (UseRect.right - UseRect.left)/2;
		HatchDef.cir.sr = (long)tmp * (long)tmp-1;
		if(HatchDef.cir.sr >9) HatchDef.cir.sr -= tmp;	//stay inside circle
		ho = HO_CIRCLE;
		PrepareParent(false);
		return DoHatch();
		}
	//for small ellipses use the centered equation
	if((UseRect.right -UseRect.left) <512 && (UseRect.bottom-UseRect.top)<512) {
		ho = HO_ELLIPSE;
		HatchDef.ell.centre.x = (UseRect.right + UseRect.left)/2;
		HatchDef.ell.centre.y = (UseRect.bottom + UseRect.top)/2;
		HatchDef.ell.ix = tmp = (UseRect.right - UseRect.left)/2;
		HatchDef.ell.sab = tmp * tmp;
		HatchDef.ell.iy = tmp = (UseRect.bottom - UseRect.top)/2;
		HatchDef.ell.sab *= (tmp * tmp);
		PrepareParent(false);
		return DoHatch();
		}
	//for bigger ellipses we use the focuses to describe the ellipse
	//  this reduces numerical problems
	ho = HO_BIGELLYPSE;
	HatchDef.bell.fo[0].x= HatchDef.bell.fo[1].x= (UseRect.right+UseRect.left)/2;
	HatchDef.bell.fo[0].y= HatchDef.bell.fo[1].y= (UseRect.bottom+UseRect.top)/2;
	if((UseRect.right -UseRect.left) >(UseRect.bottom-UseRect.top)){
		HatchDef.bell.rsab =	UseRect.right - UseRect.left;
		HatchDef.bell.a = (UseRect.right - UseRect.left)/2;
		HatchDef.bell.b = (UseRect.bottom - UseRect.top)/2;
		tmp = isqr(HatchDef.bell.a*HatchDef.bell.a -
			HatchDef.bell.b*HatchDef.bell.b);
		HatchDef.bell.fo[0].x -= tmp;
		HatchDef.bell.fo[1].x += tmp;
		}
	else {
		HatchDef.bell.rsab =	UseRect.bottom - UseRect.top;
		HatchDef.bell.b = (UseRect.right - UseRect.left)/2;
		HatchDef.bell.a = (UseRect.bottom - UseRect.top)/2;
		tmp = isqr(HatchDef.bell.a*HatchDef.bell.a -
			HatchDef.bell.b*HatchDef.bell.b);
		HatchDef.bell.fo[0].y -= tmp;
		HatchDef.bell.fo[1].y += tmp;
		}
	PrepareParent(false);
	return DoHatch();
}

bool
HatchOut::oSphere(int cx, int cy, int r, POINT *pts, int cp, char *)
{
	double v[3], vec[3];
	int i, j;

	ht = FILL_LIGHT3D;

	v[0] = 0.0;		v[1] = 1.0;		v[2] = 0.0;
	for (i = 0; i < 3; i++) for(j = 0, vec[i] = 0.0; j < 3; j++)
		vec[i] += (light_vec[i][j] * v[j]);
	circ_grad.cx = cx + iround(vec[0]*((double)r));
	circ_grad.cy = cy - iround(vec[2]*((double)r));
	circ_grad.r = r;
	if(pts && cp) return oPolygon(pts, cp);
	return oCircle(cx - r, cy - r, cx + r + 1, cy + r + 1);
}

bool
HatchOut::oRectangle(int x1, int y1, int x2, int y2, char *)
{
	if(x1 < x2) {
		HatchDef.rec.rec.left = UseRect.left = x1;
		HatchDef.rec.rec.right = UseRect.right = x2;
		}
	else {
		HatchDef.rec.rec.left = UseRect.left = x2;
		HatchDef.rec.rec.right = UseRect.right = x1;
		}
	if(y1 < y2) {
		HatchDef.rec.rec.top = UseRect.top = y1;
		HatchDef.rec.rec.bottom = UseRect.bottom = y2;
		}
	else {
		HatchDef.rec.rec.top = UseRect.top = y2;
		HatchDef.rec.rec.bottom = UseRect.bottom = y1;
		}
	ho = HO_RECT;
	PrepareParent(false);
	return DoHatch();
}

bool
HatchOut::oPolygon(POINT *pts, int cp, char *)
{
	int i;
	POINT *p;

	p = (POINT*)malloc((cp+2)*(sizeof(POINT)));
	HatchDef.plg.pts = p;
	if(!p || cp < 3)return false;
	HatchDef.plg.pts[0].x = UseRect.left = UseRect.right = pts[0].x;
	HatchDef.plg.pts[0].y = UseRect.top = UseRect.bottom = pts[0].y;
	for(i = 1; i < cp; i++){
		UseRect.left = UseRect.left < pts[i].x ? UseRect.left : pts[i].x;
		UseRect.right = UseRect.right > pts[i].x ? UseRect.right : pts[i].x;
		UseRect.top = UseRect.top < pts[i].y ? UseRect.top : pts[i].y;
		UseRect.bottom = UseRect.bottom > pts[i].y ? UseRect.bottom : pts[i].y;
		p[i].x = pts[i].x;		p[i].y = pts[i].y;
		}
	i--;
	if(p[i].x != pts[0].x || p[i].y != pts[0].y) {
		i++;
		p[i].x = pts[0].x;		p[i].y = pts[0].y;
		}
	HatchDef.plg.cp = i+1;
	ho= HO_POLYGON;
	PrepareParent(false);
	return DoHatch();
}

bool
HatchOut::PrepareParent(bool Restore)
{
	if (Restore){
		if (out && ParInit){
			out->SetLine(&ParLineDef);
			out->SetFill(&ParFillDef);
			}
		ParInit = false;
		}
	else if (out) {
		out->GetLine(&ParLineDef);					out->GetFill(&ParFillDef);
		ParInit = out->SetLine(&MyLineDef);
		out->SetFill(&MyFillDef);
		out->CopyClipRC(&pClipRC);
		if (pClipRC.bottom > pClipRC.top) {
			pClipRC.top -= out->MenuHeight();		pClipRC.bottom -= out->MenuHeight();
			}
		}
	return true;
}

bool
HatchOut::DoHatch()
{
	MkPolyLine(NULL, NULL);
	if(out) switch(ht){
	case FILL_NONE:											break;
	case FILL_HLINES:		Lines000();						break;
	case FILL_VLINES:		Lines090();						break;
	case FILL_HVCROSS:		Lines000();		Lines090();		break;
	case FILL_DLINEU:		Lines045();						break;
	case FILL_DLINED:		Lines315();						break;
	case FILL_DCROSS:		Lines045();		Lines315();		break;
	case FILL_STIPPLE1:		case FILL_STIPPLE2:		
	case FILL_STIPPLE3:		case FILL_STIPPLE4:
	case FILL_STIPPLE5:		Stipple(ht);					break;
	case FILL_ZIGZAG:		Zigzag();						break;
	case FILL_COMBS:		Combs();						break;
	case FILL_BRICKH:		BricksH();						break;
	case FILL_BRICKV:		BricksV();						break;
	case FILL_BRICKDU:		Bricks045();					break;
	case FILL_BRICKDD:		Bricks315();					break;
	case FILL_TEXTURE1:		Texture1();						break;
	case FILL_TEXTURE2:		Texture2();						break;
	case FILL_WAVES1:		Arcs(FILL_WAVES1);				break;
	case FILL_SCALES:		Arcs(FILL_SCALES);				break;
	case FILL_SHINGLES:		Arcs(FILL_SHINGLES);			break;
	case FILL_WAVES2:		Waves2(0);						break;
	case FILL_HERRING:		Herringbone();					break;
	case FILL_CIRCLES:		Circles();						break;
	case FILL_GRASS:		Grass();						break;
	case FILL_FOAM:			Foam();							break;
	case FILL_RECS:			Recs();							break;
	case FILL_HASH:			Hash();							break;
	case FILL_WATER:		Waves2(1);						break;
	case FILL_CHESS:		Chess();						break;
	case FILL_LIGHT3D:		CircGrad();						break;
		}

	//clean up
	if(ho == HO_POLYGON) {
		if(HatchDef.plg.pts) free(HatchDef.plg.pts);
		HatchDef.plg.pts = NULL;
		}
	ho = HO_NONE;
	MkPolyLine(NULL, out);
	return PrepareParent(true);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// collect line segments to a polyline command
#define MK_PL_MAX 500
bool
HatchOut::MkPolyLine(POINT *p, anyOutput *o, bool bFill)
{
	static POINT pl[MK_PL_MAX+2];
	static long npt = 0;

	if(p && o) {
		if(!npt) {
			memcpy(pl, p, 2*sizeof(POINT));
			npt = 2;
			return true;
			}
		if(p[0].x != pl[npt-1].x || p[0].y != pl[npt-1].y) {
			if (bFill && npt >2) {
				OC_type = OC_HATCH;
				o->oSolidPolygon(pl, npt);
				OC_type = 0;
				}
			else o->oPolyline(pl, npt);
			memcpy(pl, p, 2 * sizeof(POINT));
			npt = 2;
			return true;
			}
		AddToPolygon(&npt, pl, p+1);
		if(npt > (MK_PL_MAX-2)) {
			npt = 0;
			return o->oPolyline(pl, MK_PL_MAX-1);
			}
		if (pl[npt - 1].x != (p + 1)->x || pl[npt - 1].y != (p + 1)->y) {
			pl[npt].x = (p + 1)->x;		pl[npt].y = (p + 1)->y;		npt++;
			}
		return true;
		}
	if(!p && !o) {
		npt = 0;
		return true;
		}
	if(!p && o && npt) {
		if (bFill && npt >2) {
			OC_type = OC_HATCH;
			o->oSolidPolygon(pl, npt);
			OC_type = 0;
			}
		else if (npt == 2) o->oSolidLine(pl);
		else o->oPolyline(pl, npt);
		npt = 0;
		return true;
      }
	return false;
}

//use Bresenham's algorithm to draw lines
//Ref: P.S. Heckbert (1990) "Digital Line Drawing", in: Graphic Gems
//   (A.S. Glassner, ed.); Academic Press, Inc.,
//   ISBN 0-12-286165-5
bool
HatchOut::HatchLine(POINT p1, POINT p2, bool bFill)
{
	long d;
	bool bPen;
	POINT tr[2], dxy, sxy, axy;

	dxy.x = p2.x - p1.x;
	if ( p2.x < p1.x) { 	axy.x = (-dxy.x)<<1;		sxy.x = -1;		}
	else {					axy.x = dxy.x <<1;			sxy.x = 1;		}
	dxy.y = p2.y - p1.y;
	if (p2.y < p1.y) {		axy.y = (-dxy.y)<<1;		sxy.y = -1;		}
	else {					axy.y = dxy.y<<1;			sxy.y = 1;		}
	tr[0].x = tr[1].x = p1.x;							tr[0].y = tr[1].y = p1.y;
	if(IsInside(p1)) bPen = true;
	else bPen = false;
	if (axy.x > axy.y) {										// x dominant
		d = axy.y - (axy.x >>1);
		for ( ; ; ) {
			if(bPen) {
				if (tr[1].x == p2.x) return MkPolyLine(tr, out, bFill);
				if (d >= 0) {tr[1].y += sxy.y;	d -= axy.x;}
				tr[1].x += sxy.x;
				if(!IsInside(tr[1])) {
					bPen = false;
					MkPolyLine(tr, out, bFill);
					tr[0].x = tr[1].x;	tr[0].y = tr[1].y;
					}
				}
			else {
				if(tr[0].x == p2.x) return true;
				if (d >= 0) {tr[0].y += sxy.y;	d -= axy.x;}
				tr[0].x += sxy.x;
				if(IsInside(tr[0])){
					bPen = true;
					tr[1].x = tr[0].x;	tr[1].y = tr[0].y;
					}
				}
			d += axy.y;
			}
		}
	else {													// y dominant
		d = axy.x - (axy.y >>1);
		for ( ; ; ) {
			if (bPen){
				if (tr[1].y == p2.y) return MkPolyLine(tr, out, bFill);
				if (d >= 0) {tr[1].x += sxy.x; d -= axy.y;}
				tr[1].y += sxy.y;
				if(!IsInside(tr[1])) {
					bPen = false;
					MkPolyLine(tr, out, bFill);
					tr[0].x = tr[1].x;	tr[0].y = tr[1].y;
					}
				}
			else {
				if (tr[0].y == p2.y) return true;
				if (d >= 0) {tr[0].x += sxy.x; d -= axy.y;}
				tr[0].y += sxy.y;
				if(IsInside(tr[0])) {
					bPen = true;
					tr[1].x = tr[0].x;	tr[1].y = tr[0].y;
					}
				}
			d += axy.x;
			}
		}
}

//use circular Bresenham's algorithm to draw arcs
//Ref: C. Montani, R. Scopigno (1990) "Speres-To-Voxel Conversion", in:
//   Graphic Gems (A.S. Glassner ed.) Academic Press, Inc.; 
//   ISBN 0-12-288165-5 
bool
HatchOut::HatchArc(long ix, long iy, long r, int qad, bool start)
{
	long x, y, q, di, de, lim;
	bool bInside;
	static POINT tr[2];

	if (ix < 0 || ix > 32000 || iy < 0 || iy > 32000 || r < 0 || r > 32000) {
		return false;
		}
	if(r < 1) r = 1;
	if(start) {
		tr[1].x = ix-r;		tr[1].y = iy;
		}
	for(q = 0; q < qad; q++) {
		x = lim = 0;	y = r;	di = 2*(1-r);
		bInside = IsInside(tr[1]);
		while (y >= lim){
			if(di < 0) {
				de = 2*di + 2*y -1;
				if(de > 0) {
					x++;	y--;	di += (2*x -2*y +2);
					}
				else {
					x++;	di += (2*x +1);
					}
				}
			else {
				de = 2*di -2*x -1;
				if(de > 0) {
					y--;	di += (-2*y +1);
					}
				else {
					x++;	y--;	di += (2*x -2*y +2);
					}
				}
			tr[0].x = tr[1].x;		tr[0].y = tr[1].y;
			switch(q) {
			case 0:	tr[1].x = ix-y;		tr[1].y = iy+x;	break;
			case 1: tr[1].x = ix+x;		tr[1].y = iy+y;	break;
			case 2: tr[1].x = ix+y;		tr[1].y = iy-x;	break;
			case 3:	tr[1].x = ix-x;		tr[1].y = iy-y;	break;
				}
			if(IsInside(tr[1])){
				if (bInside)MkPolyLine(tr, out, false);
				bInside = true;
				}
			else {
				if (bInside) MkPolyLine(0L, out, false);
				bInside = false;
				}
			}
		}
	return true;
}

bool
HatchOut::IsInside(POINT p)
{
	long tmp1, tmp2, tmp3, tmp4;

	if (!out) return false;
	if ((p.x < (DeskRect.left - 50) || p.x >(DeskRect.right + 50)
			|| p.y < (DeskRect.top - 50) || p.y >(DeskRect.bottom + 50))) return false;
	if (p.x < 0 || p.x > 32000 || p.y < 0 || p.y > 32000) {
		return false;
		}
	if (pClipRC.right > pClipRC.left && pClipRC.bottom > pClipRC.top) {
		if (!IsInRect(pClipRC, p.x, p.y)) return false;
		}
	switch(ho){
	case HO_RECT:
		if(p.x > HatchDef.rec.rec.left && p.x < HatchDef.rec.rec.right &&
			p.y > HatchDef.rec.rec.top && p.y < HatchDef.rec.rec.bottom)
			return true;
		return false;
	case HO_CIRCLE:
		tmp1 = p.x-HatchDef.cir.centre.x;
		tmp2 = p.y-HatchDef.cir.centre.y;
		if((tmp1 * tmp1 + tmp2 * tmp2) < HatchDef.cir.sr) return true;
		return false;
	case HO_ELLIPSE:
		tmp1 = p.x-HatchDef.ell.centre.x;
		tmp2 = p.y-HatchDef.ell.centre.y;
		tmp3 = HatchDef.ell.iy;
		tmp4 = HatchDef.ell.ix;
		if((unsigned long)(tmp1 * tmp1 * tmp3 * tmp3 + tmp2 * tmp2 * tmp4 * tmp4) < HatchDef.ell.sab)
			return true;
		return false;
	case HO_BIGELLYPSE:
		tmp1 = HatchDef.bell.fo[0].x - p.x;			tmp1 *= tmp1;
		tmp2 = HatchDef.bell.fo[0].y - p.y;			tmp2 *= tmp2;
		tmp3 = HatchDef.bell.fo[1].x - p.x;			tmp3 *= tmp3;
		tmp4 = HatchDef.bell.fo[1].y - p.y;			tmp4 *= tmp4;
		return (isqr(tmp1+tmp2)+isqr(tmp3+tmp4)) < HatchDef.bell.rsab;
	case HO_POLYGON:
		return IsInPolygon(&p, HatchDef.plg.pts, HatchDef.plg.cp, 0.5);
		}
	return false;
}

void
HatchOut::Lines000()
{
	long y, yinc, y1, y2;
	POINT Line[2];

	if(2>(yinc = iround(ybase*.8)))yinc = 2;
	Line[0].x = UseRect.left;	Line[1].x = UseRect.right;
	y1 = UseRect.top;			y2 = UseRect.bottom;
	for(y = y1; y < y2; y += yinc) {
		Line[0].y = Line[1].y = y;
		HatchLine(Line[0], Line[1]);
		}
}

void
HatchOut::Lines090()
{
	long x, xinc, x1, x2;
	POINT Line[2];

	if(2>(xinc = iround(xbase*.8)))xinc = 2;
	Line[0].y = UseRect.top;	Line[1].y = UseRect.bottom;
	x1 = UseRect.left;			x2 = UseRect.right;
	for(x = x1; x < x2; x += xinc) {
		Line[0].x = Line[1].x = x;
		HatchLine(Line[0], Line[1]);
		}
}

void
HatchOut::Lines045()
{
	long x, y, x1, y1, y2, xinc, yinc;
	POINT Line[2];

	if(3>(xinc=iround(xbase*1.2)))xinc=3;
	if(3>(yinc=iround(ybase*1.2)))yinc=3;
	Line[1].x = x = UseRect.right;				Line[0].y = Line[1].y = y = UseRect.bottom;
	x1 = UseRect.left;							y2 = UseRect.top;
	while(x > x1) {
		Line[0].x = x = x-xinc;
		if(y > y2) Line[1].y = y = y-yinc;
		else Line[1].x -= xinc;
		HatchLine(Line[0], Line[1]);
		}
	y1 = Line[0].y;
	while(y1 > UseRect.top) {
		Line[0].y = y1 = y1-yinc;
		if(y > y2) Line[1].y = y = y-yinc;
		else Line[1].x -= xinc;
		HatchLine(Line[0], Line[1]);
		}
}

void
HatchOut::Lines315()
{
	long x, y, x1, y1, y2, xinc, yinc;
	POINT Line[2];

	xinc = iround(xbase*1.2);				yinc = iround(ybase*1.2);
	if(3 > xinc) xinc = 3;	
	if(3 > yinc) yinc = 3;
	Line[1].x = x = UseRect.right;			Line[0].y = Line[1].y = y = UseRect.top;
	x1 = UseRect.left;						y2 = UseRect.bottom;
	while (x > x1) {
		Line[0].x = x = x-xinc;				if(y < y2) Line[1].y = y = y+yinc;
		else Line[1].x -= xinc;	
		HatchLine(Line[0], Line[1]);
		}
	y1 = Line[0].y;
	while(y1 < y2) {
		Line[0].y = y1 = y1+yinc;
		if(y < y2) Line[1].y = y = y+yinc;
		else Line[1].x -= xinc;
		HatchLine(Line[0], Line[1]);
		}
}

void
HatchOut::Stipple(int type)
{
	long x, x1, x2, y, y1, y2, xinc, yinc, level, xspac, yspac;
	POINT Line[2];
	bool doFill = false;

	xinc = iround(xbase*0.48);						yinc = iround(ybase*0.48);
	if(!xinc) xinc = 1;	
	if(!yinc) yinc = 1;
	xspac = iround(xbase*0.56);						yspac = iround(ybase*0.56);
	if(!xspac)	xspac = 1;
	if(!yspac) yspac = 1;
	level = 0;			x1 = UseRect.left;			x2 = UseRect.right;
	y1 = UseRect.top;	y2 = UseRect.bottom;
	for(x = x1; x < x2; x += xspac*2) {
		level &= 0x1;
		for(y = y1; y < y2; y += yspac*4) {
			if(type < FILL_STIPPLE3) {
				Line[0].x = x;
				Line[0].y = level? y+yinc+yspac*2 : y+yinc;
				Line[1].x = Line[0].x+xinc;		Line[1].y = Line[0].y-yinc;
				if(type == FILL_STIPPLE1)HatchLine(Line[0], Line[1], doFill);
				Line[0].x = Line[1].x;	Line[0].y = Line[1].y;
				Line[1].x += xinc;		Line[1].y += yinc;
				if(type == FILL_STIPPLE1)HatchLine(Line[0], Line[1], doFill);
				Line[0].x = Line[1].x;	Line[0].y = Line[1].y;
				Line[1].x -= xinc;		Line[1].y += yinc;
				HatchLine(Line[0], Line[1], doFill);
				Line[0].x = Line[1].x;	Line[0].y = Line[1].y;
				Line[1].x -= xinc;				Line[1].y -= yinc;
				HatchLine(Line[0], Line[1], doFill);
				}
			else if(type <= FILL_STIPPLE5) {
				Line[0].x = x;
				Line[0].y =Line[1].y = level? y+yinc+yspac*2 : y+yinc;
				Line[1].x = Line[0].x+xinc*2;
				if(type == FILL_STIPPLE3 || type == FILL_STIPPLE4)HatchLine(Line[0], Line[1]);
				Line[0].x = Line[1].x = x+xinc;	
				Line[0].y = Line[1].y -yinc;
				Line[1].y += yinc;
				if(type == FILL_STIPPLE3 || type == FILL_STIPPLE5)HatchLine(Line[0], Line[1]);
				}
			}
		level++;
		}
}

void
HatchOut::Zigzag()
{
	long yinc, x1, x2, ix, y1, iy;
	POINT Line[2];

	if(3>(yinc = iround(ybase)))yinc = 3;
	if(2>(iy = iround(ybase*.8)))iy = 2;
	if(2>(ix = iround(xbase*.8)))ix=2;
	Line[0].x = Line[1].x = UseRect.left;	Line[0].y = Line[1].y = UseRect.top;
	y1 = UseRect.bottom;		x1 = UseRect.right;		x2 = UseRect.left;
	while(Line[0].y < y1 +iy) {
		while(Line[1].x < x1) {
			Line[1].y -= iy;				Line[1].x += ix;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[1].x += ix;
			Line[0].y = Line[1].y;			Line[1].y += iy;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[0].y = Line[1].y;
			}
		Line[0].x = Line[1].x = x2;			Line[0].y = Line[1].y += yinc;
		}
}

void
HatchOut::Combs()
{
	long x, x1, x2, y, y1, y2, xinc, yinc;
	POINT Line[2], Next;

	if(!(yinc = iround(ybase*.4)))yinc = 1;
	if(2 >(xinc = iround(xbase*.69282))) xinc = 2;		//exact yinc *sin(60�)*2
	y = UseRect.top + yinc;		y1 = UseRect.bottom;	x1 = UseRect.left;
	x2 = UseRect.right;			y2 = UseRect.top;
	while(y < y1 + yinc*2) {
		Line[0].y = Line[1].y = y;						Line[0].x = Line[1].x = x1-xinc;
		while(Line[1].x < x2 && Line[1].y >= y2){
			Line[1].y -= (yinc*2);
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[1].x += xinc;		Line[0].y = Line[1].y;		Line[1].y -= yinc;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[0].y = Line[1].y;	Line[1].x += xinc;			Line[1].y += yinc;
			HatchLine(Line[0], Line[1]);
			Line[1].x -= xinc;				Line[1].y -= yinc;
			}
		y += yinc*6;
		}
	Next.x = x = x1-xinc;					Next.y = y;
	while(x < x2) {
		Line[0].y = Line[1].y = Next.y;		Line[0].x = Line[1].x = x;
		while(Line[1].x < x2 && Line[1].y >= y2){
			Line[1].y -= (yinc*2);
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[1].x += xinc;		Line[0].y = Line[1].y;		Line[1].y -= yinc;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[0].y = Line[1].y;	Line[1].x += xinc;			Line[1].y += yinc;
			HatchLine(Line[0], Line[1]);
			Line[1].x -= xinc;				Line[1].y -= yinc;
			}
		x += xinc*2;
		}
}

void
HatchOut::BricksH()
{
	long i, j, x1, x2, y, y1, y2, yinc, xinc;
	POINT Line[2];

	if(4>(xinc=iround(xbase*1.6)))xinc=4;
	if(2>(yinc=iround(ybase*0.8)))yinc=2;
	y1 = UseRect.top;		y2 = UseRect.bottom;
	x1 = UseRect.left;		x2 = UseRect.right;
	for(y = y1, j = 0; y < y2; y += yinc, j++) {
		Line[0].x = x1;		Line[1].x = x2;
		Line[0].y = Line[1].y = y;
 		HatchLine(Line[0], Line[1]);
		Line[0].y ++;								Line[1].y += yinc;
		for (i = (j&1)?x1:x1+(xinc>>1); i<x2; i+=xinc){
			Line[0].x = Line[1].x = i;
			HatchLine(Line[0], Line[1]);
			}
		}
}

void
HatchOut::BricksV()
{
	long i, j, x, x1, x2, y1, y2, yinc, xinc;
	POINT Line[2];

	xinc = iround(xbase*0.8);				yinc = iround(ybase*1.6);
	if(2 > xinc) xinc=2;
	if(4 > yinc) yinc=4;
	x1 = UseRect.left;						x2 = UseRect.right;
	y1 = UseRect.top;						y2 = UseRect.bottom;
	for(x = x1, j= 0; x < x2; x += xinc, j++) {
		Line[0].y = y1;		Line[1].y = y2;
		Line[0].x = Line[1].x = x;
		HatchLine(Line[0], Line[1]);
		Line[0].x ++;			Line[1].x += xinc;
		for (i = (j & 1) ? y1 : y1 + (yinc >> 1); i < y2; i += yinc){
			Line[0].y = Line[1].y = i;
			HatchLine(Line[0], Line[1]);
			}
		}
}

void
HatchOut::Bricks045()
{
	long xinc, yinc, x1, x2, y1, bwx, bwy;
	POINT Line[2];

	xinc = iround(xbase*.7);				yinc = iround(ybase*.7);
	if(2 > xinc) xinc = 2;
	if(2 > yinc) yinc = 2;
	bwx = xinc *2;							bwy = yinc *2;
	Line[0].x = UseRect.left - xinc;		Line[0].y = UseRect.top;
	y1 = UseRect.bottom;					x1 = UseRect.left;
	x2 = UseRect.right;
	while((Line[0].y < y1 + bwy)) {
		while(Line[0].x < x2) {
			Line[1].x = Line[0].x + bwx;	Line[1].y = Line[0].y - bwy;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[0].y = Line[1].y;
			Line[1].x = Line[0].x + xinc;	Line[1].y = Line[0].y + yinc;
			HatchLine(Line[0], Line[1]);
			Line[0].y += bwy;
			}
		Line[0].y += bwy;					Line[0].x = x1 - xinc;
		}
}

void
HatchOut::Bricks315()
{
	long xinc, yinc, x1, x2, y1, bwx, bwy;
	POINT Line[2];

	xinc = iround(xbase*.7);				yinc = iround(ybase*.7);
	if(2 > xinc) xinc = 2;
	if(2 > yinc) yinc = 2;
	bwx = xinc *2;							bwy = yinc *2;
	Line[0].x = UseRect.left - xinc;		Line[0].y = UseRect.top -bwy;
	y1 = UseRect.bottom;					x1 = UseRect.right;
	x2 = UseRect.left;
	while((Line[0].y < y1)) {
		while(Line[0].x < x1) {
			Line[1].x = Line[0].x + bwx;	Line[1].y = Line[0].y + bwy;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[0].y = Line[1].y;
			Line[1].x = Line[0].x + xinc;	Line[1].y = Line[0].y - yinc;
			HatchLine(Line[0], Line[1]);
			Line[0].y -= bwy;
			}
		Line[0].y += bwy;					Line[0].x = x2 - xinc;
		}
}

void
HatchOut::Texture1()
{
	long j, xinc, x1, x2, y1, yinc;
	POINT Line[2];

	xinc = iround(xbase*0.4);					yinc = iround(ybase*0.4);
	if(!xinc) xinc = 1;	
	if(!yinc) yinc = 1;
	Line[0].x = UseRect.left - xinc;			Line[0].y = UseRect.top - yinc;
	y1 = UseRect.bottom;						x1 = UseRect.right;
	x2 = UseRect.left;							j = 0;
	while((Line[0].y < y1)) {
		while(Line[0].x < x1) {
			Line[1].x = Line[0].x;				Line[1].y = Line[0].y + (yinc *2);
			HatchLine(Line[0], Line[1]);
			Line[0].x += xinc;					Line[0].y += yinc;
			Line[1].x = Line[0].x + (xinc *2);	Line[1].y = Line[0].y;
			HatchLine(Line[0], Line[1]);
			Line[0].x += xinc*3;				Line[0].y -= yinc;
			}
		j++;
		Line[0].y += (yinc<<1);					Line[0].x = x2 - xinc;
		if(j &0x01) Line[0].x += (xinc<<1);
		}
}


void
HatchOut::Texture2()
{
	long j, x1, x2, y1, xinc, yinc;
	POINT Line[2];

	xinc = iround(xbase*.6);				yinc = iround(ybase*.6);
	if(2 > xinc) xinc = 2;
	if(2 > yinc) yinc = 2;
	Line[0].x = UseRect.left - (xinc <<1);	Line[0].y = UseRect.top - (yinc <<1);
	y1 = UseRect.bottom;					x1 = UseRect.right;
	x2 = UseRect.left;						j = 0;
	while((Line[0].y < y1)) {
		while(Line[0].x < x1) {
			Line[1].x = Line[0].x;			Line[1].y = Line[0].y + (yinc *3);
			HatchLine(Line[0], Line[1]);
			Line[0].x += xinc;				Line[1].x = Line[0].x;
			HatchLine(Line[0], Line[1]);
			Line[0].y += yinc;				Line[1].x = Line[0].x + (xinc *3);
			Line[1].y = Line[0].y;
			HatchLine(Line[0], Line[1]);
			Line[0].y += yinc;				Line[1].y = Line[0].y;
			HatchLine(Line[0], Line[1]);
			Line[0].x += xinc*3;			Line[0].y -= yinc*2;
			}
		j++;
		Line[0].y += (yinc <<1);		Line[0].x = x2 - (xinc <<1);
		if(j &0x01) Line[0].x += (xinc <<1);
		}
}

void
HatchOut::Arcs(int type)
{
	long i, j, level, x1, x2, y1, y2, ix, iy;

	if(type == FILL_SHINGLES) {
		iy = iround(ybase*1.6);			ix = iround(xbase*.8);
		}
	else {
		iy = iround(ybase*.8);			ix = iround(xbase*.8);
		}
	if(iy < 2) iy = 2;
	if(ix < 2) ix = 2;
	x1 = UseRect.right;					x2 = UseRect.left;
	y1 = UseRect.top;					y2 = UseRect.bottom;
	x1 += ix;		y1 -= (iy<<1);		UseRect.bottom += (iy<<1);
	for(i = y1, level = 0; i < y2; i+= iy, level++) {
		if(type == FILL_WAVES1) {
			HatchArc(x2, i, ix, 0, true);
			for(j = x2; j < x1; j += (ix <<1)) HatchArc(j, i, ix, 2, false);
			i += iy/3;
			}
		else if(type == FILL_SCALES) {
			HatchArc(x2, i, ix, 0, true);
			for (j = x2; j < x1; j += (ix <<1))
				HatchArc((level &1) ? j+ix:j, i, ix, 2, false);
			i++;
			}
		else {
			for(j = x2; j < x1; j += ix*2){
				HatchArc((level &1) ? j+ix:j, i-1, ix, 0, true);
				HatchArc((level &1) ? j+ix:j, i + iy/2, ix, 2, false);
				}
			i++;
			}
		}
}

void
HatchOut::Waves2(int type)					//hatch using sine waves
{
	long i, j, level, x1, x2, y, y1, y2, ix, yinc, *pts;
	POINT Line[2];
	double dtmp;

	yinc = iround(type ? ybase*.8 : ybase*1.2);
	if (3 > yinc) yinc = 3;
	ix = iround(xbase*2.5);
	if(type == 0 && 14 > ix) ix = 14;
	else if(type == 1 && (7>(ix = iround(xbase*1.2)))) ix = 7;
	if(!(pts = (long *)malloc(ix * sizeof(long))))return;
	y1 = UseRect.bottom;							y2 = UseRect.top;
	x1 = UseRect.left;								x2 = UseRect.right;
	for(i = 0; i < ix; i++) {
		dtmp = sin(6.283185307/((double)ix/(double)i));
		if(type == 1) dtmp /= 2.0;
		pts[i] = dtmp > 0.0 ? iround(0.3*ybase*dtmp) : iround(0.3*ybase*dtmp);
		}
	y1 += yinc;					x2++;
	for(y = y2, level = 0; y <= y1; y += yinc, level++){
		Line[0].x = x1;			Line[1].x = x1+1;
		Line[0].y = y;						Line[1].y = y+pts[1];
		if(type == 0) for(j = 2; Line[0].x < x2; j++) {
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x;			Line[1].x++;
			Line[0].y = Line[1].y;			Line[1].y = y + pts[j%ix];
			}
		else if(type == 1) {
			if(level & 1) {
				Line[0].x += (ix + (ix>>1));		Line[1].x = Line[0].x +1;
				}
			for(j = 2; Line[0].x < UseRect.right; j++){
				HatchLine(Line[0], Line[1]);
				Line[0].x = Line[1].x;
				Line[0].y = Line[1].y;				Line[1].y = y + pts[j%ix];
				if((j-1) % ix) Line[1].x++;
				else {
					HatchLine(Line[0], Line[1]);
					Line[1].x += (ix << 1);			Line[0].x = Line[1].x -1;
					}
				}
			}
		}
	free(pts);
}

void
HatchOut::Herringbone()
{
	long ix1, ix2, iy1, iy2, y, x1, x2, y1, y2;
	POINT Line[2];

	if(2>(ix1 = iround(xbase*.6)))ix1 = 2;
	ix2 = ix1*4;
	if(2>(iy1 = iround(ybase*.6)))iy1 = 2;
	iy2 = iy1*4;
	y1 = UseRect.top;							y2 = UseRect.bottom;
	x1 = UseRect.left;							x2 = UseRect.right;
	for(y = y1; y <= y2 + iy2; y += iy1*2) {
		Line[0].x = x1-ix2;		Line[1].x = Line[0].x + ix2;
		Line[0].y = y;						Line[1].y = Line[0].y - iy2;
		do {
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x - ix1;	Line[1].x = Line[0].x + ix2;
			Line[0].y = Line[1].y - iy1;	Line[1].y = Line[0].y + iy2;
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x - ix1;	Line[1].x = Line[0].x + ix2;
			Line[0].y = Line[1].y + iy1;	Line[1].y = Line[0].y - iy2;
			} while(Line[0].x <= x2);
		}
}

void
HatchOut::Circles()
{
	long x, x1, x2, y, y1, y2, r, level, xspac, yspac;

	xspac = iround(xbase*.8);			yspac = iround(ybase*1.38564);
	if(2 > xspac) xspac = 2;
	if(4 > yspac) yspac = 4;
	r = iround(xbase*.4);				if(1 > r) r = 1;
	y1 = UseRect.bottom;				y2 = UseRect.top;
	x1 = UseRect.left;					x2 = UseRect.right;
	level = 0;
	y1 += yspac;						x2 += r;
	for(x = x1; x < x2; x += xspac) {
		level &= 0x1;
		for(y = y2; y < y1; y += yspac*2) {
			HatchArc(x, level? y+yspac : y, r, 4, true);
			}
		level++;
		}
}

void
HatchOut::Grass()
{
	long i, x1, x2, y1, y2, count, dh, dw;
	double xsize, ysize;
	long idum = -1;
	POINT pts[2];

	xsize = xbase * 2.0;				ysize = ybase * 2.0;
	if(xsize < 4.0) xsize = 4.0;
	if(ysize < 4.0) ysize = 4.0;
	IncrementMinMaxRect(&UseRect, (long)xsize);
	x1 = UseRect.left;					x2 = UseRect.right;
	y1 = UseRect.top;					y2 = UseRect.bottom;
	count = (x2 - x1)*(y2-y1);
	i = (long)(xsize*ysize*0.15);		if(i) count /= i;
	dh = y2-y1;	dw = x2-x1;
	for(i = 0; i < count; i++) {
		pts[0].x = x1+(long)(dw*ran2(&idum));					pts[0].y = y1+(long)(dh*ran2(&idum));
		pts[1].x = pts[0].x + (long)(ran2(&idum)*xsize);		pts[1].y = pts[0].y + (long)(ran2(&idum)*ysize);
		if(pts[0].x != pts[1].x || pts[0].y != pts[1].y) HatchLine(pts[0], pts[1]);
		}
}

void
HatchOut::Foam()
{
	long i, x1, x2, y1, y2, count, dh, dw;
	double xsize;
	long idum = -1;

	xsize = xbase *0.9;					if(xsize < 2.0) xsize = 2.0;
	IncrementMinMaxRect(&UseRect, (long)xsize);
	x1 = UseRect.left;					x2 = UseRect.right;
	y1 = UseRect.top;					y2 = UseRect.bottom;
	count = (x2 - x1)*(y2 - y1);		count /= (long)(xsize*xsize);
	dh = y2 - y1;						dw = x2 - x1;
	for(i = 0; i < count; i++) {
		HatchArc(x1+(long)(dw*ran2(&idum)), y1+(long)(dh*ran2(&idum)),
			(long)(ran2(&idum)*ran2(&idum)*xsize +xsize*.2), 4, true);
		}
}

void
HatchOut::Recs()
{
	long i, x1, x2, y1, y2, count, dh, dw;
	double xsize, ysize;
	long idum = -1;
	POINT Line[5];

	xsize = xbase *2.8;					ysize = ybase *2.8;
	if(xsize < 4.0) xsize = 4.0;
	if(ysize < 4.0) ysize = 4.0;
	IncrementMinMaxRect(&UseRect, (int)xsize);
	x1 = UseRect.left;					x2 = UseRect.right;
	y1 = UseRect.top;					y2 = UseRect.bottom;
	count = (x2 - x1)*(y2 - y1);
	i = (long)(floor(xsize*ysize*.4));	if(i) count /= i;
	dh = y2 - y1;						dw = x2 - x1;
	for(i = 0; i < count; i++) {
		Line[0].x = Line[3].x = Line[4].x = x1 + (long)(dw*ran2(&idum));
		Line[0].y = Line[1].y = Line[4].y = y1 + (long)(dh*ran2(&idum));
		Line[1].x = Line[2].x = Line[0].x + (long)(ran2(&idum)*xsize +xsize*.2);
		Line[2].y = Line[3].y = Line[0].y + (long)(ran2(&idum)*ysize +ysize*.2);
		HatchLine(Line[0], Line[1]);			HatchLine(Line[1], Line[2]);
		HatchLine(Line[2], Line[3]);			HatchLine(Line[3], Line[4]);
		}
}

void
HatchOut::Hash()
{
	long i, dh, dw, x1, x2, y1, y2, cx, cy, xinc, yinc;
	double xsize, ysize, mix, miy;
	long idum = -1;
	POINT Line[5];

	xsize = xbase * 0.9;						ysize = ybase * 0.9;
	xinc = iround(xsize * 3.3);					yinc = iround(ysize * 2.2);
	mix = xbase *.5;							miy = ybase *.5;
	dw = iround(xbase > 5 ? xbase/2.0 : 2.0);	dh = iround(ybase > 5 ? xbase/2.0 : 2.0);
	if(xsize < 2.0) xsize = 2.0;
	if(ysize < 2.0) ysize = 2.0;
	IncrementMinMaxRect(&UseRect, (int)xsize);
	x1 = UseRect.left;							x2 = UseRect.right;
	y1 = UseRect.top;							y2 = UseRect.bottom;
	for(i = 0, cy = y1; cy < y2; i++, cy += yinc) {
		for(cx = (i & 1) ? x1 : x1 + (xinc >>1); cx < x2; cx += xinc) { 
			Line[0].x = Line[1].x = cx;
			Line[0].y = cy - iround(ran2(&idum) * ysize + miy);
			Line[1].y = cy + iround(ran2(&idum) * ysize + miy);
			HatchLine(Line[0], Line[1]);
			Line[0].x = Line[1].x = cx + dw;
			Line[0].y = cy - iround(ran2(&idum) * ysize + miy);
			Line[1].y = cy + iround(ran2(&idum) * ysize + miy);
			HatchLine(Line[0], Line[1]);
			Line[0].y = Line[1].y = cy;
			Line[0].x = cx - iround(ran2(&idum) * xsize + mix);
			Line[1].x = cx + iround(ran2(&idum) * xsize + mix);
			HatchLine(Line[0], Line[1]);
			Line[0].y = Line[1].y = cy + dh;
			Line[0].x = cx - iround(ran2(&idum) * xsize + mix);
			Line[1].x = cx + iround(ran2(&idum) * xsize + mix);
			HatchLine(Line[0], Line[1]);
			}
		}
}

void
HatchOut::Chess()
{
	long x, x1, x2, y, y0, y1, y2, xinc, yinc, level;
	long cp, size_pts = 1000;
	bool isRect = false;
	POINT Line[2], *pts, np;

	if (!Notary->IsValidPtr(out)) return;
	xinc = iround(xbase*0.99);					yinc = iround(ybase*0.99);
	if (!xinc) xinc = 2;
	if (!yinc) yinc = 2;
	x1 = UseRect.left;							x2 = UseRect.right;
	y1 = UseRect.top;							y2 = UseRect.bottom;
	if (ho == HO_POLYGON && HatchDef.plg.cp == 5) {		//test if polygon is in fact a rectangle
		if (HatchDef.plg.pts[0].x == x1 && HatchDef.plg.pts[0].y == y1 && HatchDef.plg.pts[1].x == x2
			&& HatchDef.plg.pts[1].y == y1 && HatchDef.plg.pts[2].x == x2 && HatchDef.plg.pts[2].y == y2
			&& HatchDef.plg.pts[3].x == x1 && HatchDef.plg.pts[3].y == y2 && HatchDef.plg.pts[4].x == x1
			&& HatchDef.plg.pts[4].y == y1) isRect = true;
		}
	level = 0;		if (ho == HO_RECT) isRect = true;
	pts = (POINT*)malloc(size_pts * sizeof(POINT));
	for (x = x1 - xinc; x < x2; x += xinc) {
		level &= 0x1;		cp = 0;
		for (y = y1 - yinc; y <= y2; y += (yinc<<1)) {
			if (isRect) {
				y0 = level ? y + (yinc << 1) - 1 : y + yinc - 1;
				if (x < x2 && (x + xinc) > x1 && y0 < y2 && (y0 + yinc) > y1){
					out->oSolidRectangle(x > x1 ? x : x1, y0 > y1 ? y0 : y1,
						(x + xinc - 1) < x2 ? x + xinc - 1: x2, (y0 + yinc) < y2 ? y0 + yinc : y2);
					}
				}
			else {
				if (cp > (size_pts - 6)) {
					size_pts += 1000;
					pts = (POINT*)realloc(pts, size_pts * sizeof(POINT));
					}
				Line[0].x = x;						Line[0].y = level ? y + (yinc << 1) - 1 : y + yinc - 1;
				Line[1].x = Line[0].x + xinc - 1;	Line[1].y = Line[0].y;
				np.y = Line[0].y;
				for (np.x = Line[0].x; np.x <= Line[1].x; np.x++){
					if(IsInside(np)) AddToPolygon(&cp, pts, &np);
					}
				Line[0].y = Line[1].y;				Line[0].x = Line[1].x;
				Line[1].y += yinc;
				np.x = Line[0].x;
				for (np.y = Line[0].y; np.y <= Line[1].y; np.y++){
					if (IsInside(np)) AddToPolygon(&cp, pts, &np);
					}
				Line[0].y = Line[1].y;				Line[0].x = Line[1].x;
				Line[1].x = x;
				np.y = Line[0].y;
				for (np.x = Line[0].x; np.x >= Line[1].x; np.x--){
					if (IsInside(np)) AddToPolygon(&cp, pts, &np);
					}
				Line[0].y = Line[1].y;				Line[0].x = Line[1].x;
				Line[1].y = level ? y + yinc * 2 : y + yinc;
				np.x = Line[0].x;
				for (np.y = Line[0].y; np.y >= Line[1].y; np.y--){
					if (IsInside(np)) AddToPolygon(&cp, pts, &np);
					}
				OC_type = OC_HATCH;
				if (cp > 2) {
					if (IsInside(np) && pts[cp].x != pts[cp - 1].x && pts[cp].y != pts[cp - 1].y){
						pts[cp].x = np.x;	pts[cp].y = np.y;
						cp++;
						}
					out->oSolidPolygon(pts, cp);
					}
				OC_type = 0;						cp = 0;
				}
			}
		level++;
		}
	if(pts) free(pts);
}

void
HatchOut::CircGrad()
{
	long i;
	double f;
	LineDEF ld = {0.0, 1.0, 0x0, 0x0};

	for(i = 1; i < circ_grad.r; i++) {
		f = (((double)i)/((double)circ_grad.r));
		f = f*f;
		ld.color = IpolCol(dFillCol, dFillCol2, f);
		out->SetLine(&ld);
		HatchArc(circ_grad.cx, circ_grad.cy, i, 4, true);
		}
}
