function [G,target,repl] = ISAOL(Niter,Y,l,num,G,al,thres)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Implements the implicit AOL Gradient Descent algorithm using Niter
% iterations starting from an operator G0 with measurement data y, 
% for which we have that Op*y is sparse. 
% We want to find Op.
% In addition, we use a replacement strategy for the operator G, so if two
% rows have a larger overlap than thresRepl, one of the rows is
% reinitialized.
%
% Input:
%   -Niter  	... Number of iterations
%   -Y          ... Signal data
%   -l          ... Cosparsity
%   -num        ... Number of sampled signals in each step
%   -G          ... Starting operator (default: random, size 2dxd)
%   -al         ... stepsize (default: 1)
%   -thres      ... replacement threshold (default: 1 = no replacement)
% Output:
%   -G          ... the learned operator
%   -target     ... the value of the target function for all iterations
%   -repl       ... number of replacements in each of the iterations
%
% (c) 03.04.2016 Michael Sandbichler
%
%
% This software is a free software distributed under the terms of the GNU 
% Public License version 3 (http://www.gnu.org/licenses/gpl.txt). You can 
% redistribute it and/or modify it under the terms of this licence, for 
% personal and non-commercial use and research purpose. 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% initializations and error catches
[d,N] = size(Y);

if nargin < 4
    err('Too few input arguments.');
    return;
end

if nargin < 5
    G= randn(2*d,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
else
[n,d1] = size(G);
if d1~=d
   disp('Input operator G has the wrong size. Choosing a random operator instead.');
    G= randn(n,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
end
end

if nargin<6
    %default: stepsize = 1
    al = 1;
end

if nargin < 7
    %default: no replacement
    thres= 1;
end

if num > N
    num = N;
end

target(1:Niter) = 0;
repl(1:Niter) = 0;

%% the algorithm
for iter =1:Niter
    
   %draw a sample of size num from the data Y
    sample = randperm(N);
    sample = sample(1:num);
    ySub = Y(:,sample);
    S = (G*ySub);
    
    %find the l smallest entries in each column
    [~,I] = sort(abs(S),1);
    I=I(1:l,:);
    
    %calculate the value of the target function on the training data
    target(iter) = 1/num*sum(sum(S(I).^2,1));
    
    for k= 1:n
    %perform the gradient step for each of the rows of G
        
    J = logical(sum(I==k,1));
                
    if sum(J)==0 %if J is empty, we rethrow the dice
    G(k,:) = randn(1,d);
    else
        
    Ys = ySub(:,J)*ySub(:,J)';
    M = eye(d) + al*Ys;
    g = M\(G(k,:)'); 
    G(k,:) = g';
    end
    

    if(norm(G(k,:))~=0)
    G(k,:) = G(k,:)/norm(G(k,:));
    end


    %replace duplicate rows   
    
    
    aux = find((abs(G*G'-eye(n))')>thres);
    if ~isempty(mod(aux,n+1)) 
        doubleRows = [ceil(aux/n),mod(aux-1,n)+1];
        while(~isempty(doubleRows))
        %redraw the duplicate rows
        G(doubleRows(1,1),:) = randn(1,d);  
        G(doubleRows(1,1),:) = G(doubleRows(1,1),:)/norm(G(doubleRows(1,1),:));
        aux = find((abs(G*G'-eye(n))')>thres);
        doubleRows = [ceil(aux/n),mod(aux,n)];
        display('Duplicate row replaced');
        repl(iter) = repl(iter)+1;
        end
    end 

    end
end
return;