function dico = ictkm_simplified(data,K,S,m,maxit,dinit)
% -------------------------------------------------------------------------
% Perform dictionary learning via IcTKM. This is a simplified version of
% the IcTKM algorithm. For the complete, more general implementation refer
% to the script IcTKM.m and the toolbox documentation.
% -------------------------------------------------------------------------
% Input:
% -------------------------------------------------------------------------
%        - 'data': sparse training data matrix with [d,N]=size(data),
%                  generated by calling the script make_sparse_signal.m
%        - 'K': number of atoms to learn
%        - 'S': sparsity level of the training data
%        - 'm': Johnson-Lindentrauss embedding dimension with m < d
%        - 'maxit': maximum number of iterations
%        - 'dinit': initialization dictionary with [d,K]=size(dinit)
% -------------------------------------------------------------------------
% Output:
% -------------------------------------------------------------------------
%        - 'dico': dictionary learned from the training data.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compressed Dictionary Learning Toolbox
% Copyright (C) Flavio Teixeira 2018
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License
% as published by the Free Software Foundation; either version 2
% of the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor,
% Boston, MA  02110-1301, USA.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%-------------------------------------------------------------------------
% learning problem dimension
%-------------------------------------------------------------------------
[d,N]=size(data);
dold=dinit;
dnew=zeros(d,K);
for it=1:maxit
    %----------------------------------------------------------------------
    % draw JL embedding
    %----------------------------------------------------------------------
    randsigncol = sign(randn(d,1));
    partial = randperm(d);
    %----------------------------------------------------------------------
    % compress dictionary and data
    %----------------------------------------------------------------------
    sktchdold = fft((randsigncol*ones(1,K)).*dold);
    sktchdold = sktchdold(partial(1:m),:);
    sktchdata = fft((randsigncol*ones(1,N)).*data);
    sktchdata = sktchdata(partial(1:m),:);
    %----------------------------------------------------------------------
    % compressed thresholding
    %----------------------------------------------------------------------
    sktchip=sktchdold'*sktchdata;
    abssktchip=abs(sktchip);
    [~,I] = sort(abssktchip,1,'descend');
    % precompute gram matrix
    gram=dold'*dold;
    for n=1:N
        %------------------------------------------------------------------
        % Recompute S uncompressed inner-products
        %------------------------------------------------------------------
        In = I(1:S,n);
        ipn = dold(:,In)'*data(:,n);
        %------------------------------------------------------------------
        % compute k-residual means
        %------------------------------------------------------------------
        res = data(:,n)-dold(:,In)*(gram(In,In)\ipn); 
        % Use the line below when the one above gives a warning. The above 
        % computation is faster but less accurate.
        %res = data(:,n)-dold(:,In)*pinv(gram(In,In))*ipn;
        %------------------------------------------------------------------
        % update dictionary
        %------------------------------------------------------------------
        dnew(:,In)=dnew(:,In)+real(res*sign(ipn)');
        dnew(:,In)=dnew(:,In)+dold(:,In)*diag(abs(ipn));
    end
    %----------------------------------------------------------------------
    % scaling
    %----------------------------------------------------------------------
    scale=sum(dnew.*dnew);
    iszero=find(scale < 1e-3);
    %----------------------------------------------------------------------
    % redraw atoms that are not used
    %----------------------------------------------------------------------
    dnew(:,iszero)=randn(d,length(iszero));
    scale(iszero)=sum(dnew(:,iszero).*dnew(:,iszero));
    %----------------------------------------------------------------------
    % normalization
    %----------------------------------------------------------------------
    dnew = dnew*diag(1./sqrt(scale));
    dold = dnew;    
end
% save recovered dictionary
dico=dold;
end
