function [G,nrep] = replaceDuplicates(G,thres,active,maxRep)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% replaces duplicate rows of a matrix G if their overlap
% exceeds 'thres' with the method described in the article
% 'Online and Stable Learning of Analysis Operators' 
% M Sandbichler, K Schnass
%
% input: 
%       - G         ... Kxd matrix
%       - thres     ... replacement threshold
%       - active    ... counter variable, how often each of rows of G was
%                       used on the data
%       - maxRep    ... maximum number of replacements (default: d/20)
%
% output:
%       - G         ... decorrelated matrix
%       - nrep      ... number of performed replacements
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargin<3
    disp('too few input arguments for replacement');
    return;
end

[~,d] = size(G);

if nargin==4
    maxRep = floor(d/20);
end

if nargin>4
    disp('too many input arguments for replacement');
    return;
end

Gram = G*G';
[n,~] = size(Gram);

%check which elements in the hollow gram matrix exceed the given threshold
aux = find((abs(Gram'-eye(n))')>thres);
nrep = 0;

if ~isempty(mod(aux,n+1)) 
     doubleRows = [ceil(aux/n),mod(aux-1,n)+1];
     doubleRows(doubleRows==0) = n;
     
     while(~isempty(doubleRows) && nrep < maxRep)
        %exchange the duplicate rows        
        nrep = nrep+1;     
        %perform replacement for the first pair of rows
        doub = doubleRows(1,:);
        %check how often each of analysers is used 
        actInd = [active(doub(1)),active(doub(2))];
        posMin = doub(actInd == min(actInd(1),actInd(2)));
        posMax = doub(actInd == max(actInd(1),actInd(2)));
        %posMin now stores the position of the row, which is used less
        %frequently
        posMin = posMin(1);
        posMax = posMax(1);
                
        G(posMin,:) = G(posMin,:) - (G(posMax,:)*G(posMin,:)')*G(posMax,:);
        
        %normalise or, if the conditioning of this operation is too bad,
        %redraw and normalise
        if norm(G(posMin,:))>1e-14
            G(posMin,:) = G(posMin,:)/norm(G(posMin,:));
        else
            G(posMin,:) = randn(1,d);
            G(posMin,:) = G(posMin,:)/norm(G(posMin,:));
        end
        
        G(posMin,:) = G(posMin,:)/norm(G(posMin,:));
        %redo if there are still double rows
        aux = find((abs(G*G'-eye(n))')>thres);
        doubleRows = [ceil(aux/n),mod(aux,n)];        
     end
end
