function G = SVAOL(Niter,y,l,num,G,svIter,thres,display)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Implements the AOL Gradient Descent algorithm using Niter iterations
% starting from an operator G0 with measurement data y, for which we have
% that Op*y is sparse. We want to find Op.
% Input:
%   -Niter  ... Number of iterations
%   -y      ... Signal data
%   -l      ... Cosparsity
%   -num    ... Number of sampled signals in each step (default: all)
%   -G      ... Starting operator (default: random 2d x d)
%   -svIter ... Number of iterations to compute the smallest singular
%               vector. (default: 15)
%   -thres  ... replacement threshold (default: 1 - no replacement)
%   -display ...output warnings etc.
%
% Output:
%   -G      ... the learned operator
% 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Input handling

if nargin < 3
    disp('Too few input arguments.');
    return
end
[d,N] = size(y);

if nargin < 4
    num = N;
end

if nargin < 5
    G = randn(2*d,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
end

[n,dG] = size(G);
if dG~=d
    disp('Size of input operator does not match the data. Choosing a random operator instead.');
    G = randn(2*d,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
    n = 2*d;
end

if nargin <6
    svIter = 15;
end

if nargin <7
    thres = 1;
end

if nargin < 8
    display = 1;
end

if num > N
    num = N;
end


%%%%

%%%% perform the iterations

for iter =1:Niter
    
    %choose a sample of size 'num' from the given data 'y'
    sample = randperm(N);
    sample = sample(1:num);
    active(1:n) = 0; %keeps track of the use of the rows
    ySub = y(:,sample);
    
    % perform the thresholding
    S = (G*ySub);
    [~,I] = sort(abs(S),1);
    I=I(1:l,:);    
    
    % update all rows
    for k= 1:n
    J = logical(sum(I==k,1));
    active(k) = sum(J);
                
    if sum(J)==0
    G(k,:) = randn(1,d);
    else
    
    % set up the data matrix
    Y = ySub(:,J)*ySub(:,J)';
    
    [g,s] = SSVIICG(Y,svIter);

    G(k,:) = g;
    end
    
    %renormalize G
    if(norm(G(k,:))>1e-12)
    G(k,:) = G(k,:)/norm(G(k,:));
    end
        
    end
    
    %perform replacement
    [G, breakCounter] = replaceDuplicates(G,thres,active,10); 
     if display && (breakCounter > 0)
        disp(strcat('SVAOL: ',num2str(breakCounter),' duplicate rows replaced (max 10)'));
     end
     if display && (breakCounter >= 10)
        disp(strcat('max. number of replacement reached, check threshold'));
     end
    
end
return;