function G = IAOL(Niter,Y,l,num,G,thres,display)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Implements the implicit AOL Gradient Descent algorithm using Niter
% iterations starting from an operator G with measurement data y, 
% for which we have that Op*y is sparse. 
% We want to find Op.
% In addition, we use a replacement strategy for the operator G, so if two
% rows have a larger overlap than thres, one of the rows is
% replaced.
%
% Input:
%   -Niter  	... Number of iterations
%   -Y          ... Signal data
%   -l          ... Cosparsity
%   -num        ... Number of sampled signals in each step
%   -G          ... Starting operator (default: random, size 2dxd)
%   -thres      ... replacement threshold (default: 1 = no replacement)
%   -display    ...output warnings etc.
%
% Output:
%   -G          ... the learned operator
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% initializations and error catches
[d,N] = size(Y);

if nargin < 3
    err('Too few input arguments.');
    return;
end

if nargin <4
    num = N;
end

if nargin < 5
    G= randn(2*d,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
else
[n,d1] = size(G);
if d1~=d
   disp('Input operator G has the wrong size. Choosing a random operator instead.');
    G= randn(n,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
end
end

if nargin < 6
    thres= 1;
end

if nargin < 7
    display = 1;
end

if num > N
    num = N;
end

%% the algorithm
for iter =1:Niter
    
   %draw a sample of size num from the data Y
    sample = randperm(N);
    sample = sample(1:num);
    ySub = Y(:,sample);
    S = (G*ySub);
    
    %find the l smallest entries in each column
    [~,I] = sort(abs(S),1);
    I=I(1:l,:);
    
    X=S;
    I = I+repmat(0:n:(num*n-1),l,1);
    X(I)=0;
    
    U = -(X-S);
    U = (abs(U)>0);
    active = sum(U,2);
    
    
    for k= 1:n
    %perform the gradient step for each of the rows of G
        
        J = U(k,:);
    
        if sum(J)==0 %if J is empty, we rethrow the dice
            G(k,:) = randn(1,d);
        else

            Ys = ySub(:,J)*ySub(:,J)';
            M = eye(d) + Ys;
            g = M\(G(k,:)'); 
            G(k,:) = g';
        end


        if(norm(G(k,:))>1e-13)
            G(k,:) = G(k,:)/norm(G(k,:));
        end

    
    end
     
    
    [G, breakCounter] = replaceDuplicates(G,thres,active,10);
    
    if display && (breakCounter > 0)
        disp(strcat('IAOL: ',num2str(breakCounter),' duplicate rows replaced (max 10)'));
    end
    
    if display && (breakCounter >= 10)
        disp(strcat('max. number of replacement reached, check threshold'));
    end

    
end
return;