function G = FAOL(Niter,Y,l,num,G,thres,display)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Implements the AOL Gradient Descent algorithm using Niter iterations
% starting from an operator G with measurement data Y, for which we have
% that Op*y is sparse. We want to find Op.
% In addition, we use a replacement strategy for the operator G, so if two
% rows have a larger overlap than thresRepl, one of the rows is
% reinitialized.
%
% Input:
%   -Niter  ... Number of iterations
%   -Y      ... Signal data
%   -l      ... Cosparsity
%   -num    ... Number of sampled signals in each step
%   -G      ... Starting operator (default: random, size 2dxd)
%   -thres  ... replacement threshold (default: 1 = no replacement)
%   -display ...output warnings etc.
%
% Output:
%   -G      ... the learned operator
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% initializations and error catches
[d,N] = size(Y);

if nargin < 3
    err('Too few input arguments.');
    return;
end

if nargin <4
    num = N;
end

if nargin < 5
    G= randn(2*d,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
else
[n,d1] = size(G);
if d1~=d
   disp('Input operator G has the wrong size. Choosing a random operator instead.');
    G= randn(n,d);
    scale = diag(G*G');
    G = diag(1./sqrt(scale))*G;
end
end

if nargin < 6
    thres = 1;
end

if nargin < 7
    display = 1;
end

if num > N
    num = N;
end

%% the algorithm
for iter =1:Niter
    
    %draw a sample of size num from the data Y
    
    sample = randperm(N);
    sample = sample(1:num);
    ySub = Y(:,sample);
    
    S = (G*ySub);
    
    
    %find the l smallest entries in each column
    [~,I] = sort(abs(S),1);
    
    %find the l smallest entries in each column
    I=I(1:l,:);
    
    X=S;
    I = I+repmat(0:n:(num*n-1),l,1);
    X(I)=0;
    
    U = -(X-S);
    
    Gr = U*ySub';
    
    U = (abs(U)>0);
    active = sum(U,2);
    
    %compute the optimal stepsize
    a = sum(Gr.*G,2);
    b = sum(Gr.*Gr,2);
    aux = ySub'*Gr';
    aux = aux'.*U;
    c = sum(aux.*aux,2);
    
    t = ((a.*b - c)+sqrt((c-a.*b).^2 - 4*(b.^2-a.*c).*(a.^2-b)))./(2*(b.^2-a.*c));
    
    ind = (b.^2-a.*c==0);
    t(ind) = a(ind)./b(ind);
    t(b==0) = 0;
    
    G = G-repmat(t,1,d).*Gr;
    
    scale = sum(G.*G,2);
    G(scale<=1e-13,:) = randn(sum(scale<=1e-13),d);
    scale = sum(G.*G,2);
    G= diag(1./sqrt(scale))*G;
    
    %perform replacements
    [G, breakCounter] = replaceDuplicates(G,thres,active,10);
     if display && (breakCounter > 0)
        disp(strcat('FAOL: ',num2str(breakCounter),' duplicate rows replaced (max 10)'));
     end
     if display && (breakCounter >= 10)
        disp(strcat('max. number of replacement reached, check threshold'));
     end

     
end
